#!/bin/bash

###########################################################
## Configuration Generator for EM2RS devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

#################################
### EM2RS devices config generator
#################################

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg";     # hardware configuration
readonly simcfg="${scriptbase}_sim.cfg";     # simulator configuration
readonly proxycfg="${scriptbase}_proxy.cfg"; # ModBus proxy configuration
readonly diccfg="${scriptbase}_dic.cfg";     # DIM client configuration
readonly discfg="${scriptbase}_dis.cfg";     # DIM server configuration
readonly devmsg="devPostMsg";                # Method to send/post messages
readonly tabcrc="../circuits/${scriptbase}_link.crc";
declare -i numhdwcfg=0;                      # Flag to include hardware config
declare -i CTRL_TOP=0;                       # Top  of main control window
declare -i CTRL_LEFT=550;                    # Left of main control window
declare -i CTRL_LEFT_P=980;                  # Left of param control window
declare -i CTRL_TOP_STEP=150;                # Step for
declare -i TABCRC_TOP=0;                     # Step for
declare -i TABCRC_STEP=35;                   # Step for
declare fileprefix="";                       # Facility name
declare mainprefix="";                       # Device identifier name


#######################
# generate EM2RS devices
#######################
function em2rs_devices(){
 delete_files $hdwcfg $simcfg $proxycfg $tabcrc $diccfg $discfg;
 EM2RS_PROXY_INIT "$@" >> $proxycfg;
#Device     Connect DevLabel         Facility ID    ComPort ComBaudrate     ModbusPort UnitId Polling SpeedUnit PosUnit MovUnit
# EM2RS_DEV   COM     Test             EM2RS    D1    4001    115200          1          1      10      мм/с      мм      мм;
# EM2RS_DEV  COM     Поворотный_стол  EM2RS    D2    1       115200          2          2      10      гр/с      гр      гр;
 EM2RS_DEV   TCP     Test            EM2RS    D1    4001    172.21.4.252    1          1      10      гр/с      гр      гр;
 EM2RS_PROXY_END "$@" >> $proxycfg;
 cat $hdwcfg;
 cat $simcfg;
};


########################
# generate EM2RS device
########################
function EM2RS_DEV(){
 EM2RS_SET "$@";
 EM2RS_HDW "$@" >> $hdwcfg;
 EM2RS_SIM "$@" >> $simcfg;
 if [ -f $calibr ]; then echo "Calibration file for $IDEN already exist. Skip"; else EM2RS_CAL "$@" >> $calibr; fi;
 if [ -f $calibr ]; then
  if unix textmetadata -c $calibr > /dev/null 2>&1; then
   true; # checksum is valid, so nothing to do
  else
   unix textmetadata -a $calibr > /dev/null 2>&1;
  fi;
 fi;
 EM2RS_DIC "$@" >> $diccfg;
 EM2RS_DIS "$@" >> $discfg;
 if [ "$ConnectionType" = "COM" ]; then EM2RS_PROXY_COM "$@" >> $proxycfg; fi;
 if [ "$ConnectionType" = "TCP" ]; then EM2RS_PROXY_TCP "$@" >> $proxycfg; fi;
 EM2RS_TABCRC "$@" >> $tabcrc;
 let CTRL_TOP+=$CTRL_TOP_STEP;
};

######################
# set EM2RS variables
######################
function EM2RS_SET(){
 ConnectionType="$(upper_case "$1")"; shift; # Connection Type
 DevLabel="$1"; shift;                       # Device label
 FACN="$(upper_case "$1")"; shift;           # Facility name
 IDEN="$(upper_case "$1")"; shift;           # Device identifier name
 ComPort="$1"; TCPPort="$1"; shift;          # Physical COM Port number
 ComBaudrate="$1"; IP="$1"; shift;           # Physical COM Port baudrate or IP
 ModbusPort="$1";           shift;           # Modbus Port number
 ModbusUnitId="$1";         shift;           # Modbus Unit Id
 ModbusPolling="$1";        shift;           # Modbus Polling period, ms
 SpeedUnit="$1";            shift;           # Speed measurement unit
 PosUnit="$1";              shift;           # Position measurement unit
 MovUnit="$1";              shift;           # Moving measurement unit
 FACP="$(echo $FACN | tr '.' '/')";          # Facility DIM  path, i.e. FACN name with . to / replacement
 IDNP="$(echo $IDEN | tr '.' '/')";          # Identifier file name, i.e. IDEN name with . to / replacement
 FACF="$(echo $FACN | tr '.' '_')";          # Facility file name, i.e. FACN name with . to _ replacement
 IDNF="$(echo $IDNP | tr '.' '_')";          # Identifier file name, i.e. IDEN name with . to _ replacement
 FACP="$(upper_case "$FACP")";               # DIM path  in upper case
 FACF="$(lower_case "$FACF")";               # Filenames in lower case
 calibr="$(echo "../calibr/${FACF}_$IDNF.cal" | tr '[:upper:]' '[:lower:]')";
};


######################
# set EM2RS proxy initialization
######################
function EM2RS_PROXY_INIT(){
 echo "[TagList]";
 echo "MODBUSPROXY.BYTERATE.RX  = Real 0";
 echo "MODBUSPROXY.BYTERATE.TX  = Real 0";
 echo "MODBUSPROXY.POLLRATE.RX  = Real 0";
 echo "MODBUSPROXY.POLLRATE.TX  = Real 0";
 echo "MODBUSPROXY.BUGSRATE.RX  = Real 0";
 echo "MODBUSPROXY.BUGSRATE.TX  = Real 0";
 echo "MODBUSPROXY.BUGSRATE.EX  = Real 0";
 echo "MODBUSPROXY.BYTECOUNT.RX = Real 0";
 echo "MODBUSPROXY.BYTECOUNT.TX = Real 0";
 echo "MODBUSPROXY.POLLCOUNT.RX = Real 0";
 echo "MODBUSPROXY.POLLCOUNT.TX = Real 0";
 echo "MODBUSPROXY.BUGSCOUNT.RX = Real 0";
 echo "MODBUSPROXY.BUGSCOUNT.TX = Real 0";
 echo "MODBUSPROXY.BUGSCOUNT.EX = Real 0";
 echo "[]";
 echo "";
 echo "[&ModbusProxy.StartupScript]";
 echo "@Reset";
 echo ";--- Задание таблицы портов (порт,протокол,описание)";
 echo ";--- Задание счетчиков приема/передачи/ошибок";
};

function EM2RS_PROXY_COM(){
 echo ";--- Логический порт $ModbusPort";
 echo "@Port $ModbusPort RTU com port $ComPort baudrate $ComBaudrate parity none databits 8 stopbits 1";
 echo "@PortByteCounters $ModbusPort MODBUSPROXY.BYTECOUNT.RX, MODBUSPROXY.BYTECOUNT.TX";
 echo "@PortPollCounters $ModbusPort MODBUSPROXY.POLLCOUNT.RX, MODBUSPROXY.POLLCOUNT.TX";
 echo "@PortErrorCounters $ModbusPort MODBUSPROXY.BUGSCOUNT.RX, MODBUSPROXY.BUGSCOUNT.TX, MODBUSPROXY.BUGSCOUNT.EX";
 echo "@PortByteRates $ModbusPort MODBUSPROXY.BYTERATE.RX, MODBUSPROXY.BYTERATE.TX";
 echo "@PortPollRates $ModbusPort MODBUSPROXY.POLLRATE.RX, MODBUSPROXY.POLLRATE.TX";
 echo "@PortErrorRates $ModbusPort MODBUSPROXY.BUGSRATE.RX, MODBUSPROXY.BUGSRATE.TX, MODBUSPROXY.BUGSRATE.EX";
 echo "@TimeGap $ModbusPort 10";
 echo "@CheckOpt $ModbusPort *";
};

function EM2RS_PROXY_TCP(){
 echo ";--- Логический порт $ModbusPort";
 echo "@Port $ModbusPort RTU tcp port $TCPPort client $IP";
 echo "@PortByteCounters $ModbusPort MODBUSPROXY.BYTECOUNT.RX, MODBUSPROXY.BYTECOUNT.TX";
 echo "@PortPollCounters $ModbusPort MODBUSPROXY.POLLCOUNT.RX, MODBUSPROXY.POLLCOUNT.TX";
 echo "@PortErrorCounters $ModbusPort MODBUSPROXY.BUGSCOUNT.RX, MODBUSPROXY.BUGSCOUNT.TX, MODBUSPROXY.BUGSCOUNT.EX";
 echo "@PortByteRates $ModbusPort MODBUSPROXY.BYTERATE.RX, MODBUSPROXY.BYTERATE.TX";
 echo "@PortPollRates $ModbusPort MODBUSPROXY.POLLRATE.RX, MODBUSPROXY.POLLRATE.TX";
 echo "@PortErrorRates $ModbusPort MODBUSPROXY.BUGSRATE.RX, MODBUSPROXY.BUGSRATE.TX, MODBUSPROXY.BUGSRATE.EX";
 echo "@TimeGap $ModbusPort 10";
 echo "@CheckOpt $ModbusPort *";
};

function EM2RS_PROXY_END(){
 echo ";---";
 echo "@ZeroPortCounters";
 echo "@View";
 echo "@CpuProfiler start 1 60 7";
 echo "[]";
};

########################
# generate EM2RS hardware
########################
function EM2RS_HDW(){
 echo ";**************************";
 echo "; EM2RS Driver for $FACN.$IDEN";
 echo ";**************************";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.DRV = device software program";
 echo "[&$FACN.$IDEN.DRV]";
 echo "Comment = Modbus RTU Driver for EM2RS controller.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpHigher";
 echo "ProgramSource  = ..\DaqPas\\${fileprefix}em2rs_drv.pas";
 echo "DigitalFifo    = 1024";
 echo "AnalogFifo     = 1024";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 3";
 echo "Simulator      = 0";
 echo "ModbusPort     = $ModbusPort";
 echo "ModbusUnitId   = $ModbusUnitId";
 echo "ModbusTimeout  = 250";
 echo "ModbusDeadline = 60000";
 echo "ModbusPolling  = $ModbusPolling";
 echo "DelayOnStart   = 1000";
 echo "tagPrefix      = $FACN.$IDEN";
 echo "EncryptMethod  = RC6";
 echo "TrustedUsers   = [TrustedUsers]";
 echo "DevLabel       = $DevLabel ; Описание устройства";
 echo "SpeedUnit      = $SpeedUnit ; Физическая величина скорости";
 echo "PosUnit        = $PosUnit ; Физическая величина позиции";
 echo "MovUnit        = $MovUnit ; Физическая величина требуемой позиции";
 echo "AnalogOutputs  = 63";
 echo "Link AnalogOutput 0  with curve $FACN.$IDEN.PULSE        history 1000 ; \$0001 Pulse/Revolution";
 echo "Link AnalogOutput 1  with curve $FACN.$IDEN.CTRLMODE     history 1000 ; \$0003 Control Mode (CS2RS)";
 echo "Link AnalogOutput 2  with curve $FACN.$IDEN.MOTDIRECT    history 1000 ; \$0007 Motor Direction";
 echo "Link AnalogOutput 3  with curve $FACN.$IDEN.POSERROR     history 1000 ; \$000B Allowed max pos follow error *";
 echo "Link AnalogOutput 4  with curve $FACN.$IDEN.FORCED       history 1000 ; \$000F Forced enable by software *";
 echo "Link AnalogOutput 5  with curve $FACN.$IDEN.POSLOOPKP    history 1000 ; \$0051 Position loop Kp *";
 echo "Link AnalogOutput 6  with curve $FACN.$IDEN.VELLOOPKI    history 1000 ; \$0053 Velocity loop Ki *";
 echo "Link AnalogOutput 7  with curve $FACN.$IDEN.VELLOOPKP    history 1000 ; \$0055 Velocity loop Kp *";
 echo "Link AnalogOutput 8  with curve $FACN.$IDEN.POSLOOPKPH   history 1000 ; \$0065 Position loop KpH *";
 echo "Link AnalogOutput 9  with curve $FACN.$IDEN.CMDTIME      history 1000 ; \$00A1 Command filter time *";
 echo "Link AnalogOutput 10 with curve $FACN.$IDEN.VELOTOC      history 1000 ; \$00A3 Vel swtch pnt: O to C loop *";
 echo "Link AnalogOutput 11 with curve $FACN.$IDEN.VELCTOO      history 1000 ; \$00A5 Vel swtch pnt: C to O loop *";
 echo "Link AnalogOutput 12 with curve $FACN.$IDEN.DELAYOTOC    history 1000 ; \$00A7 Delay of O to C loop *";
 echo "Link AnalogOutput 13 with curve $FACN.$IDEN.DELAYCTOO    history 1000 ; \$00A9 Delay of C to O loop *";
 echo "Link AnalogOutput 14 with curve $FACN.$IDEN.CFGINPUT1    history 1000 ; \$0145 Input port DI1";
 echo "Link AnalogOutput 15 with curve $FACN.$IDEN.CFGINPUT2    history 1000 ; \$0147 Input port DI2";
 echo "Link AnalogOutput 16 with curve $FACN.$IDEN.CFGINPUT3    history 1000 ; \$0149 Input port DI3";
 echo "Link AnalogOutput 17 with curve $FACN.$IDEN.CFGINPUT4    history 1000 ; \$014B Input port DI4";
 echo "Link AnalogOutput 18 with curve $FACN.$IDEN.CFGINPUT5    history 1000 ; \$014D Input port DI5";
 echo "Link AnalogOutput 19 with curve $FACN.$IDEN.CFGINPUT6    history 1000 ; \$014F Input port DI6";
 echo "Link AnalogOutput 20 with curve $FACN.$IDEN.CFGINPUT7    history 1000 ; \$0151 Input port DI7";
 echo "Link AnalogOutput 21 with curve $FACN.$IDEN.CFGOUTPUT1   history 1000 ; \$0157 Output port DO1";
 echo "Link AnalogOutput 22 with curve $FACN.$IDEN.CFGOUTPUT2   history 1000 ; \$0159 Output port DO2";
 echo "Link AnalogOutput 23 with curve $FACN.$IDEN.CFGOUTPUT3   history 1000 ; \$015B Output port DO3";
 echo "Link AnalogOutput 24 with curve $FACN.$IDEN.ALRMDETECT   history 1000 ; \$016D Alarm detection selection";
 echo "Link AnalogOutput 25 with curve $FACN.$IDEN.DIST2SEND    history 1000 ; \$0171 Dist. 2send \"In Pos\" Out sign";
 echo "Link AnalogOutput 26 with curve $FACN.$IDEN.BUSVOLT      history 1000 ; \$0177 Bus Voltage";
 echo "Link AnalogOutput 27 with curve $FACN.$IDEN.DISTATUS     history 1000 ; \$0179 Digital Input Status";
 echo "Link AnalogOutput 28 with curve $FACN.$IDEN.DOSTATUS     history 1000 ; \$017B Digital Output Status";
 echo "Link AnalogOutput 29 with curve $FACN.$IDEN.PEAKCUR      history 1000 ; \$0191 Peak Current";
 echo "Link AnalogOutput 30 with curve $FACN.$IDEN.HOLDINCLSD   history 1000 ; \$0193 % of hold curr in closed-loop";
 echo "Link AnalogOutput 31 with curve $FACN.$IDEN.HOLDINOPEN   history 1000 ; \$0195 % of hold curr in open-loop";
 echo "Link AnalogOutput 32 with curve $FACN.$IDEN.BAUDRATE     history 1000 ; \$01BD Baudrate";
 echo "Link AnalogOutput 33 with curve $FACN.$IDEN.ADDRESS      history 1000 ; \$01BF RS485 ID";
 echo "Link AnalogOutput 34 with curve $FACN.$IDEN.RS485TYPE    history 1000 ; \$01С1 RS485 Data type";
 echo "Link AnalogOutput 35 with curve $FACN.$IDEN.STBCURPER    history 1000 ; \$01D3 Standby Current Percentage";
 echo "Link AnalogOutput 36 with curve $FACN.$IDEN.JOGVEL       history 1000 ; \$01E1 JOG Velocity";
 echo "Link AnalogOutput 37 with curve $FACN.$IDEN.JOGINTER     history 1000 ; \$01E3 Interval";
 echo "Link AnalogOutput 38 with curve $FACN.$IDEN.JOGRUNTIME   history 1000 ; \$01E5 Running times";
 echo "Link AnalogOutput 39 with curve $FACN.$IDEN.JOGACCDEC    history 1000 ; \$01E7 Acc/Dec. time";
 echo "Link AnalogOutput 40 with curve $FACN.$IDEN.ENCODERES    history 1000 ; \$0233 Encoder resolution";
 echo "Link AnalogOutput 41 with curve $FACN.$IDEN.MOTSTATUS    history 1000 ; \$1003 Motion Status";
 echo "Link AnalogOutput 42 with curve $FACN.$IDEN.SAVEWORD     history 1000 ; \$1901 Save parameter status word";
 echo "Link AnalogOutput 43 with curve $FACN.$IDEN.CURALARM     history 1000 ; \$2203 Current alarm";
 echo "Link AnalogOutput 44 with curve $FACN.$IDEN.PRCONTROL    history 1000 ; \$6000 PR control register";
 echo "Link AnalogOutput 45 with curve $FACN.$IDEN.TRIGGER      history 1000 ; \$6002 Trigger register";
 echo "Link AnalogOutput 46 with curve $FACN.$IDEN.SOFTLIMPOS   history 1000 ; \$6006+\$6007 Soft limit positive";
 echo "Link AnalogOutput 47 with curve $FACN.$IDEN.SOFTLIMNEG   history 1000 ; \$6008+\$6009 Soft limit negative";
 echo "Link AnalogOutput 48 with curve $FACN.$IDEN.HOMINGMODE   history 1000 ; \$600A Homing mode";
 echo "Link AnalogOutput 49 with curve $FACN.$IDEN.AFTERHOMING  history 1000 ; \$600D+\$600E Move to specified location after homing";
 echo "Link AnalogOutput 50 with curve $FACN.$IDEN.HOMHIGHVEL   history 1000 ; \$600F Homing hight velocity";
 echo "Link AnalogOutput 51 with curve $FACN.$IDEN.HOMLOWVEL    history 1000 ; \$6010 Homing low velocity";
 echo "Link AnalogOutput 52 with curve $FACN.$IDEN.HOMINGACC    history 1000 ; \$6011 Homing Acc";
 echo "Link AnalogOutput 53 with curve $FACN.$IDEN.HOMINGDEC    history 1000 ; \$6012 Homing Dec";
 echo "Link AnalogOutput 54 with curve $FACN.$IDEN.LIMSTOPTIME  history 1000 ; \$6016 Limit stop time";
 echo "Link AnalogOutput 55 with curve $FACN.$IDEN.ESTOPTIME    history 1000 ; \$6017 E-STOP time";
 echo "Link AnalogOutput 56 with curve $FACN.$IDEN.ACTPOS       history 1000 ; \$602C+\$602D Actual position";
 echo "Link AnalogOutput 57 with curve $FACN.$IDEN.MOTPATH0     history 1000 ; \$6200 Motion of path 0";
 echo "Link AnalogOutput 58 with curve $FACN.$IDEN.POSITION     history 1000 ; \$6201+\$6202 Position";
 echo "Link AnalogOutput 59 with curve $FACN.$IDEN.SPEED        history 1000 ; \$6203 Velocity";
 echo "Link AnalogOutput 60 with curve $FACN.$IDEN.ACCELERAT    history 1000 ; \$6204 Acc";
 echo "Link AnalogOutput 61 with curve $FACN.$IDEN.DECELERAT    history 1000 ; \$6205 Dec";
 echo "Link AnalogOutput 62 with curve $FACN.$IDEN.GUI.POSITION history 1000 ;";
 echo "DigitalOutputs = 8";
 echo "Link DigitalOutput 0 with curve $FACN.$IDEN.STATE.INDIC history 1000";
 echo "Link DigitalOutput 1 with curve $FACN.$IDEN.POLLRATE.RX history 1000";
 echo "Link DigitalOutput 2 with curve $FACN.$IDEN.POLLRATE.TX history 1000";
 echo "Link DigitalOutput 3 with curve $FACN.$IDEN.POLLRATE.EX history 1000";
 echo "Link DigitalOutput 4 with curve $FACN.$IDEN.ERROR.COUNT history 1000";
 echo "Link DigitalOutput 5 with curve $FACN.$IDEN.POLLSUMM.RX history 1000";
 echo "Link DigitalOutput 6 with curve $FACN.$IDEN.POLLSUMM.TX history 1000";
 echo "Link DigitalOutput 7 with curve $FACN.$IDEN.POLLSUMM.EX history 1000";
 echo "Calibrations = 1";
 echo "Calibration#0  = ..\Calibr\\${FACF}_${IDNF}.CAL Step(st) Position(mm) Z SubZ Line 0 100";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$FACN.$IDEN.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ..\data\\${FACF}_${IDNF}_custom.ini";
 echo "CustomIniSection  = [$FACN.$IDEN.CustomParameters]";
 echo "CustomIniBackups  = ..\..\\${mainprefix}_data\\$FACF\custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "[ConfigFileList]";
 echo "ConfigFile = ..\data\\${FACF}_${IDNF}_custom.ini";
 echo "[]";
 echo "";
 echo "[TagList]";
 echo "$FACN.$IDEN.POLL.ENABLE     = Integer 1 ; Enable poll cycle";
 echo "$FACN.$IDEN.DIMGUICLICK     = String  ? ; The DIM click sensor data";
 echo "$FACN.$IDEN.CLOCK           = string  ? ; Host clock Date-Time";
 echo "$FACN.$IDEN.SERVID          = string  ? ; ServerId: pid@hostname";
 echo "$FACN.$IDEN.PULSE           = Integer 0 ; \$0001 Pulse/Revolution";
 echo "$FACN.$IDEN.CTRLMODE        = Integer 0 ; \$0003 Control Mode (CS2RS)";
 echo "$FACN.$IDEN.MOTDIRECT       = Integer 0 ; \$0007 Motor Direction";
 echo "$FACN.$IDEN.POSERROR        = Integer 0 ; \$000B Allowed max pos follow error *";
 echo "$FACN.$IDEN.FORCED          = Integer 0 ; \$000F Forced enable by software *";
 echo "$FACN.$IDEN.POSLOOPKP       = Integer 0 ; \$0051 Position loop Kp *";
 echo "$FACN.$IDEN.VELLOOPKI       = Integer 0 ; \$0053 Velocity loop Ki *";
 echo "$FACN.$IDEN.VELLOOPKP       = Integer 0 ; \$0055 Velocity loop Kp *";
 echo "$FACN.$IDEN.POSLOOPKPH      = Integer 0 ; \$0065 Position loop KpH *";
 echo "$FACN.$IDEN.CMDTIME         = Integer 0 ; \$00A1 Command filter time *";
 echo "$FACN.$IDEN.VELOTOC         = Integer 0 ; \$00A3 Vel swtch pnt: O to C loop *";
 echo "$FACN.$IDEN.VELCTOO         = Integer 0 ; \$00A5 Vel swtch pnt: C to O loop *";
 echo "$FACN.$IDEN.DELAYOTOC       = Integer 0 ; \$00A7 Delay of O to C loop *";
 echo "$FACN.$IDEN.DELAYCTOO       = Integer 0 ; \$00A9 Delay of C to O loop *";
 echo "$FACN.$IDEN.CFGINPUT1       = Integer 0 ; \$0145 Input port DI1";
 echo "$FACN.$IDEN.CFGINPUT1.NC    = Integer 0 ; \$0145 Input port DI1";
 echo "$FACN.$IDEN.CFGINPUT2       = Integer 0 ; \$0147 Input port DI2";
 echo "$FACN.$IDEN.CFGINPUT2.NC    = Integer 0 ; \$0147 Input port DI2";
 echo "$FACN.$IDEN.CFGINPUT3       = Integer 0 ; \$0149 Input port DI3";
 echo "$FACN.$IDEN.CFGINPUT3.NC    = Integer 0 ; \$0149 Input port DI3";
 echo "$FACN.$IDEN.CFGINPUT4       = Integer 0 ; \$014B Input port DI4";
 echo "$FACN.$IDEN.CFGINPUT4.NC    = Integer 0 ; \$014B Input port DI4";
 echo "$FACN.$IDEN.CFGINPUT5       = Integer 0 ; \$014D Input port DI5";
 echo "$FACN.$IDEN.CFGINPUT5.NC    = Integer 0 ; \$014D Input port DI5";
 echo "$FACN.$IDEN.CFGINPUT6       = Integer 0 ; \$014F Input port DI6";
 echo "$FACN.$IDEN.CFGINPUT6.NC    = Integer 0 ; \$014F Input port DI6";
 echo "$FACN.$IDEN.CFGINPUT7       = Integer 0 ; \$0151 Input port DI7";
 echo "$FACN.$IDEN.CFGINPUT7.NC    = Integer 0 ; \$0151 Input port DI7";
 echo "$FACN.$IDEN.CFGOUTPUT1      = Integer 0 ; \$0157 Output port DO1";
 echo "$FACN.$IDEN.CFGOUTPUT1.NC   = Integer 0 ; \$0157 Output port DO1";
 echo "$FACN.$IDEN.CFGOUTPUT2      = Integer 0 ; \$0159 Output port DO2";
 echo "$FACN.$IDEN.CFGOUTPUT2.NC   = Integer 0 ; \$0159 Output port DO2";
 echo "$FACN.$IDEN.CFGOUTPUT3      = Integer 0 ; \$015B Output port DO3";
 echo "$FACN.$IDEN.CFGOUTPUT3.NC   = Integer 0 ; \$015B Output port DO3";
 echo "$FACN.$IDEN.ALRMDETECT      = Integer 0 ; \$016D Alarm detection selection";
 echo "$FACN.$IDEN.DIST2SEND       = Integer 0 ; \$0171 Dist. 2send \"In Pos\" Out sign";
 echo "$FACN.$IDEN.BUSVOLT         = Real 0    ; \$0177 Bus Voltage";
 echo "$FACN.$IDEN.DISTATUS        = Integer 0 ; \$0179 Digital Input Status";
 echo "$FACN.$IDEN.DOSTATUS        = Integer 0 ; \$017B Digital Output Status";
 echo "$FACN.$IDEN.PEAKCUR         = Real 0    ; \$0191 Peak Current";
 echo "$FACN.$IDEN.HOLDINCLSD      = Integer 0 ; \$0193 % of hold curr in closed-loop";
 echo "$FACN.$IDEN.HOLDINOPEN      = Integer 0 ; \$0195 % of hold curr in open-loop";
 echo "$FACN.$IDEN.BAUDRATE        = Integer 0 ; \$01BD Baudrate";
 echo "$FACN.$IDEN.ADDRESS         = Integer 0 ; \$01BF RS485 ID";
 echo "$FACN.$IDEN.RS485TYPE       = Integer 0 ; \$01C1 Data type selection";
 echo "$FACN.$IDEN.STBCURPER       = Integer 0 ; \$01D3 Standby Current Percentage";
 echo "$FACN.$IDEN.JOGVEL          = Integer 0 ; \$01E1 JOG Velocity";
 echo "$FACN.$IDEN.JOGINTER        = Integer 0 ; \$01E3 Interval";
 echo "$FACN.$IDEN.JOGRUNTIME      = Integer 0 ; \$01E5 Running times";
 echo "$FACN.$IDEN.JOGACCDEC       = Integer 0 ; \$01E7 Acc/Dec. time";
 echo "$FACN.$IDEN.SOFTWARE        = Integer 0 ; \$01FF Version Information";
 echo "$FACN.$IDEN.FIRMWARE        = Integer 0 ; \$0201 Firmare Version";
 echo "$FACN.$IDEN.ENCODERES       = Integer 0 ; \$0233 Encoder resolution";
 echo "$FACN.$IDEN.MOTSTATUS       = Integer 0 ; \$1003 Motion Status";
 echo "$FACN.$IDEN.CTRLWORD        = Integer 0 ; \$1801 Control Word";
 echo "$FACN.$IDEN.SAVEWORD        = Integer 0 ; \$1901 Save parameter status word";
 echo "$FACN.$IDEN.CURALARM        = Integer 0 ; \$2203 Current alarm";
 echo "$FACN.$IDEN.PRCONTROL       = Integer 0 ; \$6000 PR control register";
 echo "$FACN.$IDEN.TRIGGER         = Integer 0 ; \$6002 Trigger register";
 echo "$FACN.$IDEN.SOFTLIMPOS      = Integer 0 ; \$6006+\$6007 Software limit positive";
 echo "$FACN.$IDEN.SOFTLIMNEG      = Integer 0 ; \$6008+\$6009 Software limit positive";
 echo "$FACN.$IDEN.HOMINGMODE      = Integer 0 ; \$600A Homing mode";
 echo "$FACN.$IDEN.AFTERHOMING     = Integer 0 ; \$600D+\$600E Move to specified location after homing";
 echo "$FACN.$IDEN.HOMHIGHVEL      = Integer 0 ; \$600F Homing hight velocity";
 echo "$FACN.$IDEN.HOMLOWVEL       = Integer 0 ; \$6010 Homing low velocity";
 echo "$FACN.$IDEN.HOMINGACC       = Integer 0 ; \$6011 Homing Acc";
 echo "$FACN.$IDEN.HOMINGDEC       = Integer 0 ; \$6012 Homing Dec";
 echo "$FACN.$IDEN.LIMSTOPTIME     = Integer 0 ; \$6016 Limit stop time";
 echo "$FACN.$IDEN.ESTOPTIME       = Integer 0 ; \$6017 E-STOP time";
 echo "$FACN.$IDEN.ACTPOS          = Integer 0 ; \$602C+\$602D Actual position";
 echo "$FACN.$IDEN.MOTPATH0        = Integer 0 ; \$6200 Motion of path 0";
 echo "$FACN.$IDEN.POSITION        = Integer 0 ; \$6201+\$6202 Position";
 echo "$FACN.$IDEN.SPEED           = Integer 0 ; \$6203 Velocity";
 echo "$FACN.$IDEN.ACCELERAT       = Integer 0 ; \$6204 Acc";
 echo "$FACN.$IDEN.DECELERAT       = Integer 0 ; \$6205 Dec";
 echo "$FACN.$IDEN.GUI.DEVLABEL    = String  ? ;";
 echo "$FACN.$IDEN.GUI.SPEEDUNIT   = String  ? ;";
 echo "$FACN.$IDEN.GUI.POSUNIT     = String  ? ;";
 echo "$FACN.$IDEN.GUI.MOVUNIT     = String  ? ;";
 echo "$FACN.$IDEN.GUI.POSITION    = Real    0 ;";
 echo "$FACN.$IDEN.GUI.SPEED       = Real    0 ;";
 echo "$FACN.$IDEN.GUI.DESTINATION = Real    0 ;";
 echo "$FACN.$IDEN.GUI.DIRECTFWD   = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.DIRECTBWD   = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.HOMING      = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.MOVE        = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.ESTOP       = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.SETZERO     = Integer 0 ;";
 echo "$FACN.$IDEN.GUI.POSLIMIT    = Integer 0 ; Limit swith positive";
 echo "$FACN.$IDEN.GUI.NEGLIMIT    = Integer 0 ; Limit swith negative";
 echo "$FACN.$IDEN.STATE           = Integer 0 ; Status indicator";
 echo "$FACN.$IDEN.POLLRATE.RX     = Real    0 ; Poll rate, receiver    (per sec)";
 echo "$FACN.$IDEN.POLLRATE.TX     = Real    0 ; Poll rate, transmitter (per sec)";
 echo "$FACN.$IDEN.POLLRATE.EX     = Real    0 ; Poll rate, errors      (per sec)";
 echo "$FACN.$IDEN.ERROR.COUNT     = Real    0 ; Error counter          (counter)";
 echo "$FACN.$IDEN.POLLSUMM.RX     = Real    0 ; Poll summ, receiver    (counter)";
 echo "$FACN.$IDEN.POLLSUMM.TX     = Real    0 ; Poll summ, transmitter (counter)";
 echo "$FACN.$IDEN.POLLSUMM.EX     = Real    0 ; Poll summ, errors      (counter)";
 echo "[]";
 echo "";
 echo "[DataStorage]";
 echo "$FACN.$IDEN.PULSE        = Curve 0 10 Black 15 1 ; \$0001 Pulse/Revolution";
 echo "$FACN.$IDEN.CTRLMODE     = Curve 0 10 Black 15 1 ; \$0003 Control Mode (CS2RS)";
 echo "$FACN.$IDEN.MOTDIRECT    = Curve 0 10 Black 15 1 ; \$0007 Motor Direction";
 echo "$FACN.$IDEN.POSERROR     = Curve 0 10 Black 15 1 ; \$000B Allowed max pos follow error *";
 echo "$FACN.$IDEN.FORCED       = Curve 0 10 Black 15 1 ; \$000F Forced enable by software *";
 echo "$FACN.$IDEN.POSLOOPKP    = Curve 0 10 Black 15 1 ; \$0051 Position loop Kp *";
 echo "$FACN.$IDEN.VELLOOPKI    = Curve 0 10 Black 15 1 ; \$0053 Velocity loop Ki *";
 echo "$FACN.$IDEN.VELLOOPKP    = Curve 0 10 Black 15 1 ; \$0055 Velocity loop Kp *";
 echo "$FACN.$IDEN.POSLOOPKPH   = Curve 0 10 Black 15 1 ; \$0065 Position loop KpH *";
 echo "$FACN.$IDEN.CMDTIME      = Curve 0 10 Black 15 1 ; \$00A1 Command filter time *";
 echo "$FACN.$IDEN.VELOTOC      = Curve 0 10 Black 15 1 ; \$00A3 Vel swtch pnt: O to C loop *";
 echo "$FACN.$IDEN.VELCTOO      = Curve 0 10 Black 15 1 ; \$00A5 Vel swtch pnt: C to O loop *";
 echo "$FACN.$IDEN.DELAYOTOC    = Curve 0 10 Black 15 1 ; \$00A7 Delay of O to C loop *";
 echo "$FACN.$IDEN.DELAYCTOO    = Curve 0 10 Black 15 1 ; \$00A9 Delay of C to O loop *";
 echo "$FACN.$IDEN.CFGINPUT1    = Curve 0 10 Black 15 1 ; \$0145 Input port DI1";
 echo "$FACN.$IDEN.CFGINPUT2    = Curve 0 10 Black 15 1 ; \$0147 Input port DI2";
 echo "$FACN.$IDEN.CFGINPUT3    = Curve 0 10 Black 15 1 ; \$0149 Input port DI3";
 echo "$FACN.$IDEN.CFGINPUT4    = Curve 0 10 Black 15 1 ; \$014B Input port DI4";
 echo "$FACN.$IDEN.CFGINPUT5    = Curve 0 10 Black 15 1 ; \$014D Input port DI5";
 echo "$FACN.$IDEN.CFGINPUT6    = Curve 0 10 Black 15 1 ; \$014F Input port DI6";
 echo "$FACN.$IDEN.CFGINPUT7    = Curve 0 10 Black 15 1 ; \$0151 Input port DI7";
 echo "$FACN.$IDEN.CFGOUTPUT1   = Curve 0 10 Black 15 1 ; \$0157 Output port DO1";
 echo "$FACN.$IDEN.CFGOUTPUT2   = Curve 0 10 Black 15 1 ; \$0159 Output port DO2";
 echo "$FACN.$IDEN.CFGOUTPUT3   = Curve 0 10 Black 15 1 ; \$015B Output port DO3";
 echo "$FACN.$IDEN.ALRMDETECT   = Curve 0 10 Black 15 1 ; \$016D Alarm detection selection";
 echo "$FACN.$IDEN.DIST2SEND    = Curve 0 10 Black 15 1 ; \$0171 Dist. 2send \"In Pos\" Out sign";
 echo "$FACN.$IDEN.BUSVOLT      = Curve 0 10 Black 15 1 ; \$0177 Bus Voltage";
 echo "$FACN.$IDEN.DISTATUS     = Curve 0 10 Black 15 1 ; \$0179 Digital Input Status";
 echo "$FACN.$IDEN.DOSTATUS     = Curve 0 10 Black 15 1 ; \$017B Digital Output Status";
 echo "$FACN.$IDEN.PEAKCUR      = Curve 0 10 Black 15 1 ; \$0191 Peak Current";
 echo "$FACN.$IDEN.HOLDINCLSD   = Curve 0 10 Black 15 1 ; \$0193 % of hold curr in closed-loop";
 echo "$FACN.$IDEN.HOLDINOPEN   = Curve 0 10 Black 15 1 ; \$0195 % of hold curr in open-loop";
 echo "$FACN.$IDEN.BAUDRATE     = Curve 0 10 Black 15 1 ; \$01BD Baudrate";
 echo "$FACN.$IDEN.ADDRESS      = Curve 0 10 Black 15 1 ; \$01BF RS485 ID";
 echo "$FACN.$IDEN.RS485TYPE    = Curve 0 10 Black 15 1 ; \$01С1 RS485 Data type selection";
 echo "$FACN.$IDEN.STBCURPER    = Curve 0 10 Black 15 1 ; \$01D3 Standby Current Percentage";
 echo "$FACN.$IDEN.JOGVEL       = Curve 0 10 Black 15 1 ; \$01E1 JOG Velocity";
 echo "$FACN.$IDEN.JOGINTER     = Curve 0 10 Black 15 1 ; \$01E3 Interval";
 echo "$FACN.$IDEN.JOGRUNTIME   = Curve 0 10 Black 15 1 ; \$01E5 Running times";
 echo "$FACN.$IDEN.JOGACCDEC    = Curve 0 10 Black 15 1 ; \$01E7 Acc/Dec. time";
 echo "$FACN.$IDEN.ENCODERES    = Curve 0 10 Black 15 1 ; \$0233 Encoder resolution";
 echo "$FACN.$IDEN.MOTSTATUS    = Curve 0 10 Black 15 1 ; \$1003 Motion Status";
 echo "$FACN.$IDEN.SAVEWORD     = Curve 0 10 Black 15 1 ; \$1901 Save parameter status word";
 echo "$FACN.$IDEN.CURALARM     = Curve 0 10 Black 15 1 ; \$2203 Current alarm";
 echo "$FACN.$IDEN.PRCONTROL    = Curve 0 10 Black 15 1 ; \$6000 PR control register";
 echo "$FACN.$IDEN.TRIGGER      = Curve 0 10 Black 15 1 ; \$6002 Trigger register";
 echo "$FACN.$IDEN.SOFTLIMPOS   = Curve 0 10 Black 15 1 ; \$6006+\$6007 Software limit positive";
 echo "$FACN.$IDEN.SOFTLIMNEG   = Curve 0 10 Black 15 1 ; \$6008+\$6009 Software limit positive";
 echo "$FACN.$IDEN.HOMINGMODE   = Curve 0 10 Black 15 1 ; \$600A Homing mode";
 echo "$FACN.$IDEN.AFTERHOMING  = Curve 0 10 Black 15 1 ; \$600D+\$600E Move to specified location after homing";
 echo "$FACN.$IDEN.HOMHIGHVEL   = Curve 0 10 Black 15 1 ; \$600F Homing hight velocity";
 echo "$FACN.$IDEN.HOMLOWVEL    = Curve 0 10 Black 15 1 ; \$6010 Homing low velocity";
 echo "$FACN.$IDEN.HOMINGACC    = Curve 0 10 Black 15 1 ; \$6011 Homing Acc";
 echo "$FACN.$IDEN.HOMINGDEC    = Curve 0 10 Black 15 1 ; \$6012 Homing Dec";
 echo "$FACN.$IDEN.LIMSTOPTIME  = Curve 0 10 Black 15 1 ; \$6016 Limit stop time";
 echo "$FACN.$IDEN.ESTOPTIME    = Curve 0 10 Black 15 1 ; \$6017 E-STOP time";
 echo "$FACN.$IDEN.ACTPOS       = Curve 0 10 Black 15 1 ; \$602C+\$602D Actual position";
 echo "$FACN.$IDEN.MOTPATH0     = Curve 0 10 Black 15 1 ; \$6200 Motion of path 0";
 echo "$FACN.$IDEN.POSITION     = Curve 0 10 Black 15 1 ; \$6201+\$6202";
 echo "$FACN.$IDEN.SPEED        = Curve 0 10 Black 15 1 ; \$6203 Velocity";
 echo "$FACN.$IDEN.ACCELERAT    = Curve 0 10 Black 15 1 ; \$6204 Acc";
 echo "$FACN.$IDEN.DECELERAT    = Curve 0 10 Black 15 1 ; \$6205 Dec";
 echo "$FACN.$IDEN.GUI.POSITION = Curve 0 10 Black 15 1 ;";
 echo "$FACN.$IDEN.STATE.INDIC  = Curve 0 10 Black 15 1 ;";
 echo "$FACN.$IDEN.POLLRATE.RX  = Curve 0 10 Black 15 1 ; Poll rate, receiver    (per sec)";
 echo "$FACN.$IDEN.POLLRATE.TX  = Curve 0 10 Black 15 1 ; Poll rate, transmitter (per sec)";
 echo "$FACN.$IDEN.POLLRATE.EX  = Curve 0 10 Black 15 1 ; Poll rate, errors      (per sec)";
 echo "$FACN.$IDEN.ERROR.COUNT  = Curve 0 10 Black 15 1 ; Error counter          (counter)";
 echo "$FACN.$IDEN.POLLSUMM.RX  = Curve 0 10 Black 15 1 ; Poll summ, receiver    (counter)";
 echo "$FACN.$IDEN.POLLSUMM.TX  = Curve 0 10 Black 15 1 ; Poll summ, transmitter (counter)";
 echo "$FACN.$IDEN.POLLSUMM.EX  = Curve 0 10 Black 15 1 ; Poll summ, errors      (counter)";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.CTRL = Circuit_Window";
 echo "[$FACN.$IDEN.CTRL]";
 echo "Circuit = ..\Circuits\\${fileprefix}em2rs_ctrl.crc";
 echo "Link sensor EM2RS.TOOLS            with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PARAMS           with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SPEED            with tag   $FACN.$IDEN.GUI.SPEED       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.STATE.INDIC      with curve $FACN.$IDEN.STATE.INDIC     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.STATE.MOTSTATUS  with curve $FACN.$IDEN.MOTSTATUS       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.NEGLIMIT         with tag   $FACN.$IDEN.GUI.NEGLIMIT    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POSLIMIT         with tag   $FACN.$IDEN.GUI.POSLIMIT    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.MODE             with tag   $FACN.$IDEN.MOTPATH0        device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.DESTINATION      with tag   $FACN.$IDEN.GUI.DESTINATION device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.EMERG.STOP       with tag   $FACN.$IDEN.GUI.ESTOP       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.HOMING           with tag   $FACN.$IDEN.GUI.HOMING      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.DIRECTBWD        with tag   $FACN.$IDEN.GUI.DIRECTBWD   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.MOVE             with tag   $FACN.$IDEN.GUI.MOVE        device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.DIRECTFWD        with tag   $FACN.$IDEN.GUI.DIRECTFWD   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POSITION         with curve $FACN.$IDEN.GUI.POSITION    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SETZERO          with tag   $FACN.$IDEN.GUI.SETZERO     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SPEED.UNIT       with tag   $FACN.$IDEN.GUI.SPEEDUNIT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.DESTINATION.UNIT with tag   $FACN.$IDEN.GUI.MOVUNIT     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POSITION.UNIT    with tag   $FACN.$IDEN.GUI.POSUNIT     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.DEVLABEL         with tag   $FACN.$IDEN.GUI.DEVLABEL    device &${FACN}.$IDEN.DRV";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.CTRL.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=430|Height=315";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.CTRL.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SHOW @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HIDE @WinHide $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SELECT @WinSelect $FACN.$IDEN.CTRL";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HOME";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HIDE";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.PARAM = Circuit_Window";
 echo "[$FACN.$IDEN.PARAM]";
 echo "Circuit = ..\Circuits\\${fileprefix}em2rs_parm.crc";
 echo "Link sensor EM2RS.POLL_ENABLE         with tag   $FACN.$IDEN.POLL.ENABLE device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SERVID              with tag   $FACN.$IDEN.SERVID      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.CLOCK               with tag   $FACN.$IDEN.CLOCK       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.PULSE           with curve $FACN.$IDEN.PULSE       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.MOTDIRECT       with curve $FACN.$IDEN.MOTDIRECT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.POSERROR        with curve $FACN.$IDEN.POSERROR    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.BUSVOLT         with curve $FACN.$IDEN.BUSVOLT     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.PEAKCUR         with curve $FACN.$IDEN.PEAKCUR     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.BAUDRATE        with curve $FACN.$IDEN.BAUDRATE    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.ADDRESS         with curve $FACN.$IDEN.ADDRESS     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.RS485TYPE       with curve $FACN.$IDEN.RS485TYPE   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.STBCURPER       with curve $FACN.$IDEN.STBCURPER   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOLDINCLSD      with curve $FACN.$IDEN.HOLDINCLSD  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOLDINOPEN      with curve $FACN.$IDEN.HOLDINOPEN  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.JOGVEL          with curve $FACN.$IDEN.JOGVEL      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.JOGINTER        with curve $FACN.$IDEN.JOGINTER    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.JOGRUNTIME      with curve $FACN.$IDEN.JOGRUNTIME  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.JOGACCDEC       with curve $FACN.$IDEN.JOGACCDEC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.ENCODERES       with curve $FACN.$IDEN.ENCODERES   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.CTRLWORD        with tag   $FACN.$IDEN.CTRLWORD    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.TRIGGER         with curve $FACN.$IDEN.TRIGGER     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOMINGMODE      with curve $FACN.$IDEN.HOMINGMODE  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOMHIGHVEL      with curve $FACN.$IDEN.HOMHIGHVEL  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOMLOWVEL       with curve $FACN.$IDEN.HOMLOWVEL   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOMINGACC       with curve $FACN.$IDEN.HOMINGACC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.HOMINGDEC       with curve $FACN.$IDEN.HOMINGDEC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.ACTPOS          with curve $FACN.$IDEN.ACTPOS      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.MOTPATH0        with curve $FACN.$IDEN.MOTPATH0    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.POSITION        with curve $FACN.$IDEN.POSITION    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.SPEED           with curve $FACN.$IDEN.SPEED       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.ACCELERAT       with curve $FACN.$IDEN.ACCELERAT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.SET.DECELERAT       with curve $FACN.$IDEN.DECELERAT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SOFTWARE        with tag   $FACN.$IDEN.SOFTWARE    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.FIRMWARE        with tag   $FACN.$IDEN.FIRMWARE    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.PULSE           with tag   $FACN.$IDEN.PULSE       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CTRLMODE        with tag   $FACN.$IDEN.CTRLMODE    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTDIRECT       with tag   $FACN.$IDEN.MOTDIRECT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.POSERROR        with tag   $FACN.$IDEN.POSERROR    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.BUSVOLT         with tag   $FACN.$IDEN.BUSVOLT     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.PEAKCUR         with tag   $FACN.$IDEN.PEAKCUR     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.BAUDRATE        with tag   $FACN.$IDEN.BAUDRATE    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.ADDRESS         with tag   $FACN.$IDEN.ADDRESS     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.RS485TYPE       with tag   $FACN.$IDEN.RS485TYPE   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.STBCURPER       with tag   $FACN.$IDEN.STBCURPER   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOLDINCLSD      with tag   $FACN.$IDEN.HOLDINCLSD  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOLDINOPEN      with tag   $FACN.$IDEN.HOLDINOPEN  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGVEL          with tag   $FACN.$IDEN.JOGVEL      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGINTER        with tag   $FACN.$IDEN.JOGINTER    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGRUNTIME      with tag   $FACN.$IDEN.JOGRUNTIME  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGACCDEC       with tag   $FACN.$IDEN.JOGACCDEC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.ENCODERES       with tag   $FACN.$IDEN.ENCODERES   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SAVEWORD        with curve $FACN.$IDEN.SAVEWORD    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CURALARM        with curve $FACN.$IDEN.CURALARM    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.TRIGGER         with tag   $FACN.$IDEN.TRIGGER     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SOFTLIMPOS      with tag   $FACN.$IDEN.SOFTLIMPOS  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SOFTLIMNEG      with tag   $FACN.$IDEN.SOFTLIMNEG  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMINGMODE      with tag   $FACN.$IDEN.HOMINGMODE  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.AFTERHOMING     with tag   $FACN.$IDEN.AFTERHOMING device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMHIGHVEL      with tag   $FACN.$IDEN.HOMHIGHVEL  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMLOWVEL       with tag   $FACN.$IDEN.HOMLOWVEL   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMINGACC       with tag   $FACN.$IDEN.HOMINGACC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMINGDEC       with tag   $FACN.$IDEN.HOMINGDEC   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.LIMSTOPTIME     with tag   $FACN.$IDEN.LIMSTOPTIME device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.ESTOPTIME       with tag   $FACN.$IDEN.ESTOPTIME   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.ACTPOS          with curve $FACN.$IDEN.ACTPOS      device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTPATH0        with tag   $FACN.$IDEN.MOTPATH0    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.POSITION        with tag   $FACN.$IDEN.POSITION    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SPEED           with tag   $FACN.$IDEN.SPEED       device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.ACCELERAT       with tag   $FACN.$IDEN.ACCELERAT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DECELERAT       with tag   $FACN.$IDEN.DECELERAT   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DOSTATUS.0      with curve $FACN.$IDEN.DOSTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DOSTATUS.1      with curve $FACN.$IDEN.DOSTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DOSTATUS.2      with curve $FACN.$IDEN.DOSTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.0      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.1      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.2      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.3      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.4      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.5      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.DISTATUS.6      with curve $FACN.$IDEN.DISTATUS    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.0     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.1     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.2     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.3     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.4     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.MOTSTATUS.5     with curve $FACN.$IDEN.MOTSTATUS   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.SOFTLIM.ENB     with tag   $FACN.$IDEN.PRCONTROL   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMINGONSTART   with tag   $FACN.$IDEN.PRCONTROL   device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.AFTERHOMING.ENB with tag   $FACN.$IDEN.HOMINGMODE  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.HOMINGDIR       with tag   $FACN.$IDEN.HOMINGMODE  device &${FACN}.$IDEN.DRV";
 echo "; I/O config";
 echo "Link sensor EM2RS.PAR.CFGINPUT1     with tag $FACN.$IDEN.CFGINPUT1     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT1.NC  with tag $FACN.$IDEN.CFGINPUT1.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT2     with tag $FACN.$IDEN.CFGINPUT2     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT2.NC  with tag $FACN.$IDEN.CFGINPUT2.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT3     with tag $FACN.$IDEN.CFGINPUT3     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT3.NC  with tag $FACN.$IDEN.CFGINPUT3.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT4     with tag $FACN.$IDEN.CFGINPUT4     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT4.NC  with tag $FACN.$IDEN.CFGINPUT4.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT5     with tag $FACN.$IDEN.CFGINPUT5     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT5.NC  with tag $FACN.$IDEN.CFGINPUT5.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT6     with tag $FACN.$IDEN.CFGINPUT6     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT6.NC  with tag $FACN.$IDEN.CFGINPUT6.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT7     with tag $FACN.$IDEN.CFGINPUT7     device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGINPUT7.NC  with tag $FACN.$IDEN.CFGINPUT7.NC  device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT1    with tag $FACN.$IDEN.CFGOUTPUT1    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT1.NC with tag $FACN.$IDEN.CFGOUTPUT1.NC device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT2    with tag $FACN.$IDEN.CFGOUTPUT2    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT2.NC with tag $FACN.$IDEN.CFGOUTPUT2.NC device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT3    with tag $FACN.$IDEN.CFGOUTPUT3    device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.CFGOUTPUT3.NC with tag $FACN.$IDEN.CFGOUTPUT3.NC device &${FACN}.$IDEN.DRV";
 echo "; Fast PRCONTROL";
 echo "Link sensor EM2RS.PAR.PRSTART  with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.PRHOME   with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.PRESTOP  with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.PRSETBY0 with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGCW    with device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.PAR.JOGCCW   with device &${FACN}.$IDEN.DRV";
 echo "; ModBus PollRate";
 echo "Link sensor EM2RS.POLLRATE.RX with curve $FACN.$IDEN.POLLRATE.RX device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POLLRATE.TX with curve $FACN.$IDEN.POLLRATE.TX device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POLLRATE.EX with curve $FACN.$IDEN.POLLRATE.EX device &${FACN}.$IDEN.DRV";
 echo "Link sensor EM2RS.POLLSUMM.EX with curve $FACN.$IDEN.POLLSUMM.EX device &${FACN}.$IDEN.DRV";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.PARAM.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.HOME @WinShow $FACN.$IDEN.PARAM";
 echo "@cron.job $FACN.$IDEN.PARAM.HOME @WinDraw $FACN.$IDEN.PARAM|Left=$CTRL_LEFT_P|Top=$CTRL_TOP|Width=950|Height=580";
 echo "@cron.job $FACN.$IDEN.PARAM.HOME @WinDraw $FACN.$IDEN.PARAM|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.PARAM.HOME @WinDraw $FACN.$IDEN.PARAM|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.PARAM.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.SHOW @WinShow $FACN.$IDEN.PARAM";
 echo "@cron.tab $FACN.$IDEN.PARAM.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.HIDE @WinHide $FACN.$IDEN.PARAM";
 echo "@cron.tab $FACN.$IDEN.PARAM.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.SELECT @WinSelect $FACN.$IDEN.PARAM";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.HOME";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.HIDE";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.PARAM.PLOT = Curve_Window";
 echo "[$FACN.$IDEN.PARAM.PLOT]";
 echo "AxisX = ^R{Часов}___^N^CВремя, 0, 0.25";
 echo "AxisY = ^CСостояние_каналов^N^L___{Статус}, 0, 1000";
 echo "CurveList = $FACN.$IDEN.PULSE";
 echo "CurveList = $FACN.$IDEN.CTRLMODE";
 echo "CurveList = $FACN.$IDEN.MOTDIRECT";
 echo "CurveList = $FACN.$IDEN.POSERROR";
 echo "CurveList = $FACN.$IDEN.FORCED";
 echo "CurveList = $FACN.$IDEN.POSLOOPKP";
 echo "CurveList = $FACN.$IDEN.VELLOOPKI";
 echo "CurveList = $FACN.$IDEN.VELLOOPKP";
 echo "CurveList = $FACN.$IDEN.POSLOOPKPH";
 echo "CurveList = $FACN.$IDEN.CMDTIME";
 echo "CurveList = $FACN.$IDEN.VELOTOC";
 echo "CurveList = $FACN.$IDEN.VELCTOO";
 echo "CurveList = $FACN.$IDEN.DELAYOTOC";
 echo "CurveList = $FACN.$IDEN.DELAYCTOO";
 echo "CurveList = $FACN.$IDEN.ALRMDETECT";
 echo "CurveList = $FACN.$IDEN.DIST2SEND";
 echo "CurveList = $FACN.$IDEN.BUSVOLT";
 echo "CurveList = $FACN.$IDEN.DISTATUS";
 echo "CurveList = $FACN.$IDEN.DOSTATUS";
 echo "CurveList = $FACN.$IDEN.PEAKCUR";
 echo "CurveList = $FACN.$IDEN.HOLDINCLSD";
 echo "CurveList = $FACN.$IDEN.HOLDINOPEN";
 echo "CurveList = $FACN.$IDEN.BAUDRATE";
 echo "CurveList = $FACN.$IDEN.ADDRESS";
 echo "CurveList = $FACN.$IDEN.RS485TYPE";
 echo "CurveList = $FACN.$IDEN.STBCURPER";
 echo "CurveList = $FACN.$IDEN.JOGVEL";
 echo "CurveList = $FACN.$IDEN.JOGINTER";
 echo "CurveList = $FACN.$IDEN.JOGRUNTIME";
 echo "CurveList = $FACN.$IDEN.JOGACCDEC";
 echo "CurveList = $FACN.$IDEN.ENCODERES";
 echo "CurveList = $FACN.$IDEN.MOTSTATUS";
 echo "CurveList = $FACN.$IDEN.SAVEWORD";
 echo "CurveList = $FACN.$IDEN.CURALARM";
 echo "CurveList = $FACN.$IDEN.PRCONTROL";
 echo "CurveList = $FACN.$IDEN.TRIGGER";
 echo "CurveList = $FACN.$IDEN.HOMINGMODE";
 echo "CurveList = $FACN.$IDEN.HOMHIGHVEL";
 echo "CurveList = $FACN.$IDEN.HOMLOWVEL";
 echo "CurveList = $FACN.$IDEN.HOMINGACC";
 echo "CurveList = $FACN.$IDEN.HOMINGDEC";
 echo "CurveList = $FACN.$IDEN.ACTPOS";
 echo "CurveList = $FACN.$IDEN.MOTPATH0";
 echo "CurveList = $FACN.$IDEN.POSITION";
 echo "CurveList = $FACN.$IDEN.GUI.POSITION";
 echo "CurveList = $FACN.$IDEN.POLLRATE.RX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.TX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.EX";
 echo "CurveList = $FACN.$IDEN.STATE.INDIC";
 echo "CurveList = $FACN.$IDEN.ERROR.COUNT";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.RX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.TX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.EX";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Left=167|Top=120|Height=600|Width=1000";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Options=-Min,-Max,+Close,+VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SHOW @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HIDE @WinHide $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SELECT @WinSelect $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HOME";
 echo "@cron.job $FACN.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HIDE";
 echo "[]";
 echo "";
 echo "[&DatSrv]";
 echo "CurveList = $FACN.$IDEN.PULSE";
 echo "CurveList = $FACN.$IDEN.CTRLMODE";
 echo "CurveList = $FACN.$IDEN.MOTDIRECT";
 echo "CurveList = $FACN.$IDEN.POSERROR";
 echo "CurveList = $FACN.$IDEN.FORCED";
 echo "CurveList = $FACN.$IDEN.POSLOOPKP";
 echo "CurveList = $FACN.$IDEN.VELLOOPKI";
 echo "CurveList = $FACN.$IDEN.VELLOOPKP";
 echo "CurveList = $FACN.$IDEN.POSLOOPKPH";
 echo "CurveList = $FACN.$IDEN.CMDTIME";
 echo "CurveList = $FACN.$IDEN.VELOTOC";
 echo "CurveList = $FACN.$IDEN.VELCTOO";
 echo "CurveList = $FACN.$IDEN.DELAYOTOC";
 echo "CurveList = $FACN.$IDEN.DELAYCTOO";
 echo "CurveList = $FACN.$IDEN.CFGINPUT1";
 echo "CurveList = $FACN.$IDEN.CFGINPUT2";
 echo "CurveList = $FACN.$IDEN.CFGINPUT3";
 echo "CurveList = $FACN.$IDEN.CFGINPUT4";
 echo "CurveList = $FACN.$IDEN.CFGINPUT5";
 echo "CurveList = $FACN.$IDEN.CFGINPUT6";
 echo "CurveList = $FACN.$IDEN.CFGINPUT7";
 echo "CurveList = $FACN.$IDEN.CFGOUTPUT1";
 echo "CurveList = $FACN.$IDEN.CFGOUTPUT2";
 echo "CurveList = $FACN.$IDEN.CFGOUTPUT3";
 echo "CurveList = $FACN.$IDEN.ALRMDETECT";
 echo "CurveList = $FACN.$IDEN.DIST2SEND";
 echo "CurveList = $FACN.$IDEN.BUSVOLT";
 echo "CurveList = $FACN.$IDEN.DISTATUS";
 echo "CurveList = $FACN.$IDEN.DOSTATUS";
 echo "CurveList = $FACN.$IDEN.PEAKCUR";
 echo "CurveList = $FACN.$IDEN.HOLDINCLSD";
 echo "CurveList = $FACN.$IDEN.HOLDINOPEN";
 echo "CurveList = $FACN.$IDEN.BAUDRATE";
 echo "CurveList = $FACN.$IDEN.ADDRESS";
 echo "CurveList = $FACN.$IDEN.RS485TYPE";
 echo "CurveList = $FACN.$IDEN.STBCURPER";
 echo "CurveList = $FACN.$IDEN.JOGVEL";
 echo "CurveList = $FACN.$IDEN.JOGINTER";
 echo "CurveList = $FACN.$IDEN.JOGRUNTIME";
 echo "CurveList = $FACN.$IDEN.JOGACCDEC";
 echo "CurveList = $FACN.$IDEN.ENCODERES";
 echo "CurveList = $FACN.$IDEN.MOTSTATUS";
 echo "CurveList = $FACN.$IDEN.SAVEWORD";
 echo "CurveList = $FACN.$IDEN.CURALARM";
 echo "CurveList = $FACN.$IDEN.PRCONTROL";
 echo "CurveList = $FACN.$IDEN.TRIGGER";
 echo "CurveList = $FACN.$IDEN.SOFTLIMPOS";
 echo "CurveList = $FACN.$IDEN.SOFTLIMNEG";
 echo "CurveList = $FACN.$IDEN.HOMINGMODE";
 echo "CurveList = $FACN.$IDEN.AFTERHOMING";
 echo "CurveList = $FACN.$IDEN.HOMHIGHVEL";
 echo "CurveList = $FACN.$IDEN.HOMLOWVEL";
 echo "CurveList = $FACN.$IDEN.HOMINGACC";
 echo "CurveList = $FACN.$IDEN.HOMINGDEC";
 echo "CurveList = $FACN.$IDEN.LIMSTOPTIME";
 echo "CurveList = $FACN.$IDEN.ESTOPTIME";
 echo "CurveList = $FACN.$IDEN.ACTPOS";
 echo "CurveList = $FACN.$IDEN.MOTPATH0";
 echo "CurveList = $FACN.$IDEN.POSITION";
 echo "CurveList = $FACN.$IDEN.SPEED";
 echo "CurveList = $FACN.$IDEN.ACCELERAT";
 echo "CurveList = $FACN.$IDEN.DECELERAT";
 echo "CurveList = $FACN.$IDEN.POLLRATE.RX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.TX";
 echo "CurveList = $FACN.$IDEN.POLLRATE.EX";
 echo "CurveList = $FACN.$IDEN.STATE.INDIC";
 echo "CurveList = $FACN.$IDEN.ERROR.COUNT";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.RX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.TX";
 echo "CurveList = $FACN.$IDEN.POLLSUMM.EX";
 echo "[]";
 echo "";
 echo "[$FACN.$IDEN.CustomParameters.TagList]";
 echo "TagList = $FACN.$IDEN.PULSE";
 echo "TagList = $FACN.$IDEN.CTRLMODE";
 echo "TagList = $FACN.$IDEN.MOTDIRECT";
 echo "TagList = $FACN.$IDEN.POSERROR";
 echo "TagList = $FACN.$IDEN.FORCED";
 echo "TagList = $FACN.$IDEN.POSLOOPKP";
 echo "TagList = $FACN.$IDEN.VELLOOPKI";
 echo "TagList = $FACN.$IDEN.VELLOOPKP";
 echo "TagList = $FACN.$IDEN.POSLOOPKPH";
 echo "TagList = $FACN.$IDEN.CMDTIME";
 echo "TagList = $FACN.$IDEN.VELOTOC";
 echo "TagList = $FACN.$IDEN.VELCTOO";
 echo "TagList = $FACN.$IDEN.DELAYOTOC";
 echo "TagList = $FACN.$IDEN.DELAYCTOO";
 echo "TagList = $FACN.$IDEN.CFGINPUT1";
 echo "TagList = $FACN.$IDEN.CFGINPUT2";
 echo "TagList = $FACN.$IDEN.CFGINPUT3";
 echo "TagList = $FACN.$IDEN.CFGINPUT4";
 echo "TagList = $FACN.$IDEN.CFGINPUT5";
 echo "TagList = $FACN.$IDEN.CFGINPUT6";
 echo "TagList = $FACN.$IDEN.CFGINPUT7";
 echo "TagList = $FACN.$IDEN.CFGOUTPUT1";
 echo "TagList = $FACN.$IDEN.CFGOUTPUT2";
 echo "TagList = $FACN.$IDEN.CFGOUTPUT3";
 echo "TagList = $FACN.$IDEN.ALRMDETECT";
 echo "TagList = $FACN.$IDEN.DIST2SEND";
 echo "TagList = $FACN.$IDEN.BUSVOLT";
 echo "TagList = $FACN.$IDEN.DISTATUS";
 echo "TagList = $FACN.$IDEN.DOSTATUS";
 echo "TagList = $FACN.$IDEN.PEAKCUR";
 echo "TagList = $FACN.$IDEN.HOLDINCLSD";
 echo "TagList = $FACN.$IDEN.HOLDINOPEN";
 echo "TagList = $FACN.$IDEN.BAUDRATE";
 echo "TagList = $FACN.$IDEN.ADDRESS";
 echo "TagList = $FACN.$IDEN.RS485TYPE";
 echo "TagList = $FACN.$IDEN.STBCURPER";
 echo "TagList = $FACN.$IDEN.JOGVEL";
 echo "TagList = $FACN.$IDEN.JOGINTER";
 echo "TagList = $FACN.$IDEN.JOGRUNTIME";
 echo "TagList = $FACN.$IDEN.JOGACCDEC";
 echo "TagList = $FACN.$IDEN.ENCODERES";
 echo "TagList = $FACN.$IDEN.SOFTWARE";
 echo "TagList = $FACN.$IDEN.FIRMWARE";
 echo "TagList = $FACN.$IDEN.MOTSTATUS";
 echo "TagList = $FACN.$IDEN.SAVEWORD";
 echo "TagList = $FACN.$IDEN.CTRLWORD";
 echo "TagList = $FACN.$IDEN.CURALARM";
 echo "TagList = $FACN.$IDEN.PRCONTROL";
 echo "TagList = $FACN.$IDEN.TRIGGER";
 echo "TagList = $FACN.$IDEN.SOFTLIMPOS";
 echo "TagList = $FACN.$IDEN.SOFTLIMNEG";
 echo "TagList = $FACN.$IDEN.HOMINGMODE";
 echo "TagList = $FACN.$IDEN.AFTERHOMING";
 echo "TagList = $FACN.$IDEN.HOMHIGHVEL";
 echo "TagList = $FACN.$IDEN.HOMLOWVEL";
 echo "TagList = $FACN.$IDEN.HOMINGACC";
 echo "TagList = $FACN.$IDEN.HOMINGDEC";
 echo "TagList = $FACN.$IDEN.LIMSTOPTIME";
 echo "TagList = $FACN.$IDEN.ESTOPTIME";
 echo "TagList = $FACN.$IDEN.ACTPOS";
 echo "TagList = $FACN.$IDEN.MOTPATH0";
 echo "TagList = $FACN.$IDEN.POSITION";
 echo "TagList = $FACN.$IDEN.SPEED";
 echo "TagList = $FACN.$IDEN.ACCELERAT";
 echo "TagList = $FACN.$IDEN.DECELERAT";
 echo "[]";
 echo "";
 echo "[$FACN.MAIN.CTRL]";
 echo "Link sensor @CRON+@CRON.RUN+$FACN.$IDEN.CTRL.SELECT with tag $FACN.$IDEN.GUI.DEVLABEL   device &$FACN.MAIN.CTRL";
 echo "Link sensor $FACN.$IDEN.VALUE                       with curve $FACN.$IDEN.GUI.POSITION device &$FACN.$IDEN.DRV";
 echo "[]";
 echo "";
};

##########################
# generate EM2RS simulator
##########################
function EM2RS_SIM(){
 echo "[&$FACN.$IDEN.DRV]";
 echo "Simulator = 1";
 echo "[]";
 # Include hardware config if one not done yet
 if [ "$numhdwcfg" = "0" ]; then let numhdwcfg=$numhdwcfg+1; else return 0; fi;
 echo "[ConfigFileList]";
 echo "ConfigFile = $hdwcfg";
 echo "[]";
};

function EM2RS_CAL(){
 echo "[Step(st)-Position(mm) calibration]";
 echo "FitMethod = Polynom";
 echo "TransformX = Line";
 echo "TransformY = Line";
 echo "Power = 1";
 echo "Center = 0";
 echo "Scale = 1";
 echo "Fixed = 0";
 echo "Coeff[0] = 0";
 echo "Coeff[1] = 0.025";
 echo "Bounds = -10000, 10000";
 echo "Data Step(st)  Position(mm)  Weight  Z";
 echo "     0         0             1       0";
 echo "     400       10            1       0";
 echo "End Data";
 echo "[MetaData]";
 echo "@CheckSum=2A2A9F19B48D74D5C00482C6190BB40C";
 echo "@DateTime=2022.11.09-08:42:51";
 echo "@TimeStamp=63803580171730";
 echo "[]";
};

function EM2RS_TABCRC(){
 let YPOS=${TABCRC_TOP}+60;
 echo "[SensorList]";
 echo "Sensor = @cron+@cron.run+$FACN.$IDEN.CTRL.SELECT";
 echo "[@cron+@cron.run+$FACN.$IDEN.CTRL.SELECT]";
 echo "Pos = 10, $YPOS";
 echo "Painter(v) = (glBorderBevel=2)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "LED = 1, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:14";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_16_21_24_lightsteelblue1_ptmono.bmp";
 echo "Hint = $FACN.$IDEN.DRV: Открыть окно управления";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.VALUE";
 echo "[$FACN.$IDEN.VALUE]";
 echo "Pos = 290, $YPOS";
 echo "Painter(v) = (glBorderBevel=2)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "LED = 8, 0, 0, %10.3f, Name:PT_Mono\Style:[Bold]\Size:20";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_12_21_24_lightsteelblue1_ptmono.bmp";
 echo "Hint = $FACN.$IDEN.VALUE: Текущая позиция двигателя";
 echo "[]";
 let TABCRC_TOP+=$TABCRC_STEP;
};

###################################
# generate EM2RS DIM server config
###################################
function EM2RS_DIS(){
 unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimServerMode = 1 \
  -n end \
  -n dis_cmnd   $FACP/$IDNP/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n $devmsg "&$FACN.$IDEN.DRV @DIMGUICLICK=%**" \
  -n end \
  -n dic_cmnd   $FACP/$IDNP/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dis_info   $FACP/$IDNP/POLL/ENABLE \
  -n tag        $FACN.$IDEN.POLL.ENABLE \
  -n end \
  -n dis_info   $FACP/$IDNP/CLOCK \
  -n tag        $FACN.$IDEN.CLOCK \
  -n end \
  -n dis_info   $FACP/$IDNP/SERVID \
  -n tag        $FACN.$IDEN.SERVID \
  -n end \
  -n dis_info   $FACP/$IDNP/PULSE \
  -n tag        $FACN.$IDEN.PULSE \
  -n end \
  -n dis_info   $FACP/$IDNP/CTRLMODE \
  -n tag        $FACN.$IDEN.CTRLMODE \
  -n end \
  -n dis_info   $FACP/$IDNP/MOTDIRECT \
  -n tag        $FACN.$IDEN.MOTDIRECT \
  -n end \
  -n dis_info   $FACP/$IDNP/POSERROR \
  -n tag        $FACN.$IDEN.POSERROR \
  -n end \
  -n dis_info   $FACP/$IDNP/FORCED \
  -n tag        $FACN.$IDEN.FORCED \
  -n end \
  -n dis_info   $FACP/$IDNP/POSLOOPKP \
  -n tag        $FACN.$IDEN.POSLOOPKP \
  -n end \
  -n dis_info   $FACP/$IDNP/VELLOOPKI \
  -n tag        $FACN.$IDEN.VELLOOPKI \
  -n end \
  -n dis_info   $FACP/$IDNP/VELLOOPKP \
  -n tag        $FACN.$IDEN.VELLOOPKP \
  -n end \
  -n dis_info   $FACP/$IDNP/POSLOOPKPH \
  -n tag        $FACN.$IDEN.POSLOOPKPH \
  -n end \
  -n dis_info   $FACP/$IDNP/CMDTIME \
  -n tag        $FACN.$IDEN.CMDTIME \
  -n end \
  -n dis_info   $FACP/$IDNP/VELOTOC \
  -n tag        $FACN.$IDEN.VELOTOC \
  -n end \
  -n dis_info   $FACP/$IDNP/VELCTOO \
  -n tag        $FACN.$IDEN.VELCTOO \
  -n end \
  -n dis_info   $FACP/$IDNP/DELAYOTOC \
  -n tag        $FACN.$IDEN.DELAYOTOC \
  -n end \
  -n dis_info   $FACP/$IDNP/DELAYCTOO \
  -n tag        $FACN.$IDEN.DELAYCTOO \
  -n end \
  -n dis_info   $FACP/$IDNP/CFGINPUT1+CFGINPUT2+CFGINPUT3+CFGINPUT4+CFGINPUT5+CFGINPUT6+CFGINPUT7 \
  -n tag        $FACN.$IDEN.CFGINPUT1 \
  -n tag        $FACN.$IDEN.CFGINPUT2 \
  -n tag        $FACN.$IDEN.CFGINPUT3 \
  -n tag        $FACN.$IDEN.CFGINPUT4 \
  -n tag        $FACN.$IDEN.CFGINPUT5 \
  -n tag        $FACN.$IDEN.CFGINPUT6 \
  -n tag        $FACN.$IDEN.CFGINPUT7 \
  -n end \
  -n dis_info   $FACP/$IDNP/CFGINPUT1_NC+CFGINPUT2_NC+CFGINPUT3_NC+CFGINPUT4_NC+CFGINPUT5_NC+CFGINPUT6_NC+CFGINPUT7_NC \
  -n tag        $FACN.$IDEN.CFGINPUT1.NC \
  -n tag        $FACN.$IDEN.CFGINPUT2.NC \
  -n tag        $FACN.$IDEN.CFGINPUT3.NC \
  -n tag        $FACN.$IDEN.CFGINPUT4.NC \
  -n tag        $FACN.$IDEN.CFGINPUT5.NC \
  -n tag        $FACN.$IDEN.CFGINPUT6.NC \
  -n tag        $FACN.$IDEN.CFGINPUT7.NC \
  -n end \
  -n dis_info   $FACP/$IDNP/CFGOUTPUT1+CFGOUTPUT2+CFGOUTPUT3 \
  -n tag        $FACN.$IDEN.CFGOUTPUT1 \
  -n tag        $FACN.$IDEN.CFGOUTPUT2 \
  -n tag        $FACN.$IDEN.CFGOUTPUT3 \
  -n end \
  -n dis_info   $FACP/$IDNP/CFGOUTPUT1_NC+CFGOUTPUT2_NC+CFGOUTPUT3_NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT1.NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT2.NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT3.NC \
  -n end \
  -n dis_info   $FACP/$IDNP/ALRMDETECT \
  -n tag        $FACN.$IDEN.ALRMDETECT \
  -n end \
  -n dis_info   $FACP/$IDNP/DIST2SEND \
  -n tag        $FACN.$IDEN.DIST2SEND \
  -n end \
  -n dis_info   $FACP/$IDNP/BUSVOLT \
  -n tag        $FACN.$IDEN.BUSVOLT \
  -n end \
  -n dis_info   $FACP/$IDNP/DISTATUS \
  -n tag        $FACN.$IDEN.DISTATUS \
  -n end \
  -n dis_info   $FACP/$IDNP/DOSTATUS \
  -n tag        $FACN.$IDEN.DOSTATUS \
  -n end \
  -n dis_info   $FACP/$IDNP/PEAKCUR \
  -n tag        $FACN.$IDEN.PEAKCUR \
  -n end \
  -n dis_info   $FACP/$IDNP/HOLDINCLSD \
  -n tag        $FACN.$IDEN.HOLDINCLSD \
  -n end \
  -n dis_info   $FACP/$IDNP/HOLDINOPEN \
  -n tag        $FACN.$IDEN.HOLDINOPEN \
  -n end \
  -n dis_info   $FACP/$IDNP/BAUDRATE \
  -n tag        $FACN.$IDEN.BAUDRATE \
  -n end \
  -n dis_info   $FACP/$IDNP/ADDRESS \
  -n tag        $FACN.$IDEN.ADDRESS \
  -n end \
  -n dis_info   $FACP/$IDNP/RS485TYPE \
  -n tag        $FACN.$IDEN.RS485TYPE \
  -n end \
  -n dis_info   $FACP/$IDNP/STBCURPER \
  -n tag        $FACN.$IDEN.STBCURPER \
  -n end \
  -n dis_info   $FACP/$IDNP/JOGVEL \
  -n tag        $FACN.$IDEN.JOGVEL \
  -n end \
  -n dis_info   $FACP/$IDNP/JOGINTER \
  -n tag        $FACN.$IDEN.JOGINTER \
  -n end \
  -n dis_info   $FACP/$IDNP/JOGRUNTIME \
  -n tag        $FACN.$IDEN.JOGRUNTIME \
  -n end \
  -n dis_info   $FACP/$IDNP/JOGACCDEC \
  -n tag        $FACN.$IDEN.JOGACCDEC \
  -n end \
  -n dis_info   $FACP/$IDNP/ENCODERES \
  -n tag        $FACN.$IDEN.ENCODERES \
  -n end \
  -n
 unix dimcfg \
  -n dis_info   $FACP/$IDNP/SOFTWARE+FIRMWARE \
  -n tag        $FACN.$IDEN.SOFTWARE \
  -n tag        $FACN.$IDEN.FIRMWARE \
  -n end \
  -n dis_info   $FACP/$IDNP/MOTSTATUS \
  -n tag        $FACN.$IDEN.MOTSTATUS \
  -n end \
  -n dis_info   $FACP/$IDNP/CTRLWORD \
  -n tag        $FACN.$IDEN.CTRLWORD \
  -n end \
  -n dis_info   $FACP/$IDNP/SAVEWORD \
  -n tag        $FACN.$IDEN.SAVEWORD \
  -n end \
  -n dis_info   $FACP/$IDNP/CURALARM \
  -n tag        $FACN.$IDEN.CURALARM \
  -n end \
  -n dis_info   $FACP/$IDNP/PRCONTROL \
  -n tag        $FACN.$IDEN.PRCONTROL \
  -n end \
  -n dis_info   $FACP/$IDNP/TRIGGER \
  -n tag        $FACN.$IDEN.TRIGGER \
  -n end \
  -n dis_info   $FACP/$IDNP/SOFTLIMPOS \
  -n tag        $FACN.$IDEN.SOFTLIMPOS \
  -n end \
  -n dis_info   $FACP/$IDNP/SOFTLIMNEG \
  -n tag        $FACN.$IDEN.SOFTLIMNEG \
  -n end \
  -n dis_info   $FACP/$IDNP/HOMINGMODE \
  -n tag        $FACN.$IDEN.HOMINGMODE \
  -n end \
  -n dis_info   $FACP/$IDNP/AFTERHOMING \
  -n tag        $FACN.$IDEN.AFTERHOMING \
  -n end \
  -n dis_info   $FACP/$IDNP/HOMHIGHVEL \
  -n tag        $FACN.$IDEN.HOMHIGHVEL \
  -n end \
  -n dis_info   $FACP/$IDNP/HOMLOWVEL \
  -n tag        $FACN.$IDEN.HOMLOWVEL \
  -n end \
  -n dis_info   $FACP/$IDNP/HOMINGACC \
  -n tag        $FACN.$IDEN.HOMINGACC \
  -n end \
  -n dis_info   $FACP/$IDNP/HOMINGDEC \
  -n tag        $FACN.$IDEN.HOMINGDEC \
  -n end \
  -n dis_info   $FACP/$IDNP/LIMSTOPTIME \
  -n tag        $FACN.$IDEN.LIMSTOPTIME \
  -n end \
  -n dis_info   $FACP/$IDNP/ESTOPTIME \
  -n tag        $FACN.$IDEN.ESTOPTIME \
  -n end \
  -n dis_info   $FACP/$IDNP/ACTPOS \
  -n tag        $FACN.$IDEN.ACTPOS \
  -n end \
  -n dis_info   $FACP/$IDNP/MOTPATH0 \
  -n tag        $FACN.$IDEN.MOTPATH0 \
  -n end \
  -n dis_info   $FACP/$IDNP/POSITION \
  -n tag        $FACN.$IDEN.POSITION \
  -n end \
  -n dis_info   $FACP/$IDNP/SPEED \
  -n tag        $FACN.$IDEN.SPEED \
  -n end \
  -n dis_info   $FACP/$IDNP/ACCELERAT \
  -n tag        $FACN.$IDEN.ACCELERAT \
  -n end \
  -n dis_info   $FACP/$IDNP/DECELERAT \
  -n tag        $FACN.$IDEN.DECELERAT \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/SPEEDUNIT+POSUNIT+MOVUNIT+DEVLABEL \
  -n tag        $FACN.$IDEN.GUI.SPEEDUNIT \
  -n tag        $FACN.$IDEN.GUI.POSUNIT \
  -n tag        $FACN.$IDEN.GUI.MOVUNIT \
  -n tag        $FACN.$IDEN.GUI.DEVLABEL \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/POSITION \
  -n tag        $FACN.$IDEN.GUI.POSITION \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/SPEED \
  -n tag        $FACN.$IDEN.GUI.SPEED \
  -n end \
  -n
 unix dimcfg \
  -n dis_info   $FACP/$IDNP/GUI/DESTINATION \
  -n tag        $FACN.$IDEN.GUI.DESTINATION \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/DIRECTFWD \
  -n tag        $FACN.$IDEN.GUI.DIRECTFWD \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/DIRECTBWD \
  -n tag        $FACN.$IDEN.GUI.DIRECTBWD \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/HOMING \
  -n tag        $FACN.$IDEN.GUI.HOMING \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/MOVE \
  -n tag        $FACN.$IDEN.GUI.MOVE \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/ESTOP \
  -n tag        $FACN.$IDEN.GUI.ESTOP \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/SETZERO \
  -n tag        $FACN.$IDEN.GUI.SETZERO \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/POSLIMIT \
  -n tag        $FACN.$IDEN.GUI.POSLIMIT \
  -n end \
  -n dis_info   $FACP/$IDNP/GUI/NEGLIMIT \
  -n tag        $FACN.$IDEN.GUI.NEGLIMIT \
  -n end \
  -n dis_info   $FACP/$IDNP/STATE \
  -n tag        $FACN.$IDEN.STATE \
  -n end \
  -n dis_info   $FACP/$IDNP/POLLRATE/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLRATE.RX \
  -n tag        $FACN.$IDEN.POLLRATE.TX \
  -n tag        $FACN.$IDEN.POLLRATE.EX \
  -n end \
  -n dis_info   $FACP/$IDNP/POLLSUMM/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLSUMM.RX \
  -n tag        $FACN.$IDEN.POLLSUMM.TX \
  -n tag        $FACN.$IDEN.POLLSUMM.EX \
  -n end \
  -n dis_info   $FACP/$IDNP/ERROR/COUNT \
  -n tag        $FACN.$IDEN.ERROR.COUNT \
  -n end \
  -n
 echo "";
};

function EM2RS_DIC(){
 unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimClientMode = 1 \
  -n end \
  -n dic_cmnd   $FACP/$IDNP/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dic_info   $FACP/$IDNP/POLL/ENABLE \
  -n tag        $FACN.$IDEN.POLL.ENABLE \
  -n end \
  -n dic_info   $FACP/$IDNP/PULSE \
  -n tag        $FACN.$IDEN.PULSE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.PULSE" \
  -n end \
  -n dic_info   $FACP/$IDNP/CTRLMODE \
  -n tag        $FACN.$IDEN.CTRLMODE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CTRLMODE" \
  -n end \
  -n dic_info   $FACP/$IDNP/CLOCK \
  -n tag        $FACN.$IDEN.CLOCK \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CLOCK" \
  -n end \
  -n dic_info   $FACP/$IDNP/SERVID \
  -n tag        $FACN.$IDEN.SERVID \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.SERVID" \
  -n end \
  -n dic_info   $FACP/$IDNP/MOTDIRECT \
  -n tag        $FACN.$IDEN.MOTDIRECT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.MOTDIRECT" \
  -n end \
  -n dic_info   $FACP/$IDNP/POSERROR \
  -n tag        $FACN.$IDEN.POSERROR \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POSERROR" \
  -n end \
  -n dic_info   $FACP/$IDNP/FORCED \
  -n tag        $FACN.$IDEN.FORCED \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.FORCED" \
  -n end \
  -n dic_info   $FACP/$IDNP/POSLOOPKP \
  -n tag        $FACN.$IDEN.POSLOOPKP \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POSLOOPKP" \
  -n end \
  -n dic_info   $FACP/$IDNP/VELLOOPKI \
  -n tag        $FACN.$IDEN.VELLOOPKI \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VELLOOPKI" \
  -n end \
  -n dic_info   $FACP/$IDNP/VELLOOPKP \
  -n tag        $FACN.$IDEN.VELLOOPKP \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VELLOOPKP" \
  -n end \
  -n dic_info   $FACP/$IDNP/POSLOOPKPH \
  -n tag        $FACN.$IDEN.POSLOOPKPH \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POSLOOPKPH" \
  -n end \
  -n dic_info   $FACP/$IDNP/CMDTIME \
  -n tag        $FACN.$IDEN.CMDTIME \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CMDTIME" \
  -n end \
  -n dic_info   $FACP/$IDNP/VELOTOC \
  -n tag        $FACN.$IDEN.VELOTOC \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VELOTOC" \
  -n end \
  -n dic_info   $FACP/$IDNP/VELCTOO \
  -n tag        $FACN.$IDEN.VELCTOO \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VELCTOO" \
  -n end \
  -n dic_info   $FACP/$IDNP/DELAYOTOC \
  -n tag        $FACN.$IDEN.DELAYOTOC \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DELAYOTOC" \
  -n end \
  -n dic_info   $FACP/$IDNP/DELAYCTOO \
  -n tag        $FACN.$IDEN.DELAYCTOO \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DELAYCTOO" \
  -n end \
  -n dic_info   $FACP/$IDNP/CFGINPUT1+CFGINPUT2+CFGINPUT3+CFGINPUT4+CFGINPUT5+CFGINPUT6+CFGINPUT7 \
  -n tag        $FACN.$IDEN.CFGINPUT1 \
  -n tag        $FACN.$IDEN.CFGINPUT2 \
  -n tag        $FACN.$IDEN.CFGINPUT3 \
  -n tag        $FACN.$IDEN.CFGINPUT4 \
  -n tag        $FACN.$IDEN.CFGINPUT5 \
  -n tag        $FACN.$IDEN.CFGINPUT6 \
  -n tag        $FACN.$IDEN.CFGINPUT7 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT1" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT2" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT3" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT4" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT5" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT6" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGINPUT7" \
  -n end \
  -n dic_info   $FACP/$IDNP/CFGINPUT1_NC+CFGINPUT2_NC+CFGINPUT3_NC+CFGINPUT4_NC+CFGINPUT5_NC+CFGINPUT6_NC+CFGINPUT7_NC \
  -n tag        $FACN.$IDEN.CFGINPUT1.NC \
  -n tag        $FACN.$IDEN.CFGINPUT2.NC \
  -n tag        $FACN.$IDEN.CFGINPUT3.NC \
  -n tag        $FACN.$IDEN.CFGINPUT4.NC \
  -n tag        $FACN.$IDEN.CFGINPUT5.NC \
  -n tag        $FACN.$IDEN.CFGINPUT6.NC \
  -n tag        $FACN.$IDEN.CFGINPUT7.NC \
  -n end \
  -n dic_info   $FACP/$IDNP/CFGOUTPUT1+CFGOUTPUT2+CFGOUTPUT3 \
  -n tag        $FACN.$IDEN.CFGOUTPUT1 \
  -n tag        $FACN.$IDEN.CFGOUTPUT2 \
  -n tag        $FACN.$IDEN.CFGOUTPUT3 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGOUTPUT1" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGOUTPUT2" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CFGOUTPUT3" \
  -n end \
  -n dic_info   $FACP/$IDNP/CFGOUTPUT1_NC+CFGOUTPUT2_NC+CFGOUTPUT3_NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT1.NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT2.NC \
  -n tag        $FACN.$IDEN.CFGOUTPUT3.NC \
  -n end \
  -n dic_info   $FACP/$IDNP/ALRMDETECT \
  -n tag        $FACN.$IDEN.ALRMDETECT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ALRMDETECT" \
  -n end \
  -n dic_info   $FACP/$IDNP/DIST2SEND \
  -n tag        $FACN.$IDEN.DIST2SEND \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DIST2SEND" \
  -n end \
  -n dic_info   $FACP/$IDNP/BUSVOLT \
  -n tag        $FACN.$IDEN.BUSVOLT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.BUSVOLT" \
  -n end \
  -n dic_info   $FACP/$IDNP/DISTATUS \
  -n tag        $FACN.$IDEN.DISTATUS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DISTATUS" \
  -n end \
  -n dic_info   $FACP/$IDNP/DOSTATUS \
  -n tag        $FACN.$IDEN.DOSTATUS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DOSTATUS" \
  -n end \
  -n dic_info   $FACP/$IDNP/PEAKCUR \
  -n tag        $FACN.$IDEN.PEAKCUR \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.PEAKCUR" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOLDINCLSD \
  -n tag        $FACN.$IDEN.HOLDINCLSD \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOLDINCLSD" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOLDINOPEN \
  -n tag        $FACN.$IDEN.HOLDINOPEN \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOLDINOPEN" \
  -n end \
  -n dic_info   $FACP/$IDNP/BAUDRATE \
  -n tag        $FACN.$IDEN.BAUDRATE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.BAUDRATE" \
  -n end \
  -n dic_info   $FACP/$IDNP/ADDRESS \
  -n tag        $FACN.$IDEN.ADDRESS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ADDRESS" \
  -n end \
  -n dic_info   $FACP/$IDNP/RS485TYPE \
  -n tag        $FACN.$IDEN.RS485TYPE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.RS485TYPE" \
  -n end \
  -n dic_info   $FACP/$IDNP/STBCURPER \
  -n tag        $FACN.$IDEN.STBCURPER \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.STBCURPER" \
  -n end \
  -n dic_info   $FACP/$IDNP/JOGVEL \
  -n tag        $FACN.$IDEN.JOGVEL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.JOGVEL" \
  -n end \
  -n dic_info   $FACP/$IDNP/JOGINTER \
  -n tag        $FACN.$IDEN.JOGINTER \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.JOGINTER" \
  -n end \
  -n dic_info   $FACP/$IDNP/JOGRUNTIME \
  -n tag        $FACN.$IDEN.JOGRUNTIME \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.JOGRUNTIME" \
  -n end \
  -n dic_info   $FACP/$IDNP/JOGACCDEC \
  -n tag        $FACN.$IDEN.JOGACCDEC \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.JOGACCDEC" \
  -n end \
  -n dic_info   $FACP/$IDNP/ENCODERES \
  -n tag        $FACN.$IDEN.ENCODERES \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ENCODERES" \
  -n end \
  -n dic_info   $FACP/$IDNP/SOFTWARE+FIRMWARE \
  -n tag        $FACN.$IDEN.SOFTWARE \
  -n tag        $FACN.$IDEN.FIRMWARE \
  -n end \
  -n dic_info   $FACP/$IDNP/MOTSTATUS \
  -n tag        $FACN.$IDEN.MOTSTATUS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.MOTSTATUS" \
  -n end \
  -n dic_info   $FACP/$IDNP/CTRLWORD \
  -n tag        $FACN.$IDEN.CTRLWORD \
  -n end \
  -n
 unix dimcfg \
  -n dic_info   $FACP/$IDNP/SAVEWORD \
  -n tag        $FACN.$IDEN.SAVEWORD \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.SAVEWORD" \
  -n end \
  -n dic_info   $FACP/$IDNP/CURALARM \
  -n tag        $FACN.$IDEN.CURALARM \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CURALARM" \
  -n end \
  -n dic_info   $FACP/$IDNP/PRCONTROL \
  -n tag        $FACN.$IDEN.PRCONTROL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.PRCONTROL" \
  -n end \
  -n dic_info   $FACP/$IDNP/TRIGGER \
  -n tag        $FACN.$IDEN.TRIGGER \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.TRIGGER" \
  -n end \
  -n dic_info   $FACP/$IDNP/SOFTLIMPOS \
  -n tag        $FACN.$IDEN.SOFTLIMPOS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.SOFTLIMPOS" \
  -n end \
  -n dic_info   $FACP/$IDNP/SOFTLIMNEG \
  -n tag        $FACN.$IDEN.SOFTLIMNEG \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.SOFTLIMNEG" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOMINGMODE \
  -n tag        $FACN.$IDEN.HOMINGMODE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOMINGMODE" \
  -n end \
  -n dic_info   $FACP/$IDNP/AFTERHOMING \
  -n tag        $FACN.$IDEN.AFTERHOMING \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.AFTERHOMING" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOMHIGHVEL \
  -n tag        $FACN.$IDEN.HOMHIGHVEL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOMHIGHVEL" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOMLOWVEL \
  -n tag        $FACN.$IDEN.HOMLOWVEL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOMLOWVEL" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOMINGACC \
  -n tag        $FACN.$IDEN.HOMINGACC \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOMINGACC" \
  -n end \
  -n dic_info   $FACP/$IDNP/HOMINGDEC \
  -n tag        $FACN.$IDEN.HOMINGDEC \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.HOMINGDEC" \
  -n end \
  -n dic_info   $FACP/$IDNP/LIMSTOPTIME \
  -n tag        $FACN.$IDEN.LIMSTOPTIME \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.LIMSTOPTIME" \
  -n end \
  -n dic_info   $FACP/$IDNP/ESTOPTIME \
  -n tag        $FACN.$IDEN.ESTOPTIME \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ESTOPTIME" \
  -n end \
  -n dic_info   $FACP/$IDNP/ACTPOS \
  -n tag        $FACN.$IDEN.ACTPOS \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ACTPOS" \
  -n end \
  -n dic_info   $FACP/$IDNP/MOTPATH0 \
  -n tag        $FACN.$IDEN.MOTPATH0 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.MOTPATH0" \
  -n end \
  -n dic_info   $FACP/$IDNP/POSITION \
  -n tag        $FACN.$IDEN.POSITION \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POSITION" \
  -n end \
  -n dic_info   $FACP/$IDNP/SPEED \
  -n tag        $FACN.$IDEN.SPEED \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.SPEED" \
  -n end \
  -n dic_info   $FACP/$IDNP/ACCELERAT \
  -n tag        $FACN.$IDEN.ACCELERAT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ACCELERAT" \
  -n end \
  -n dic_info   $FACP/$IDNP/DECELERAT \
  -n tag        $FACN.$IDEN.DECELERAT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.DECELERAT" \
  -n end \
  -n
 unix dimcfg \
  -n dic_info   $FACP/$IDNP/GUI/SPEEDUNIT+POSUNIT+MOVUNIT+DEVLABEL \
  -n tag        $FACN.$IDEN.GUI.SPEEDUNIT \
  -n tag        $FACN.$IDEN.GUI.POSUNIT \
  -n tag        $FACN.$IDEN.GUI.MOVUNIT \
  -n tag        $FACN.$IDEN.GUI.DEVLABEL \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/POSITION \
  -n tag        $FACN.$IDEN.GUI.POSITION \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.GUI.POSITION" \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/SPEED \
  -n tag        $FACN.$IDEN.GUI.SPEED \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/DESTINATION \
  -n tag        $FACN.$IDEN.GUI.DESTINATION \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/DIRECTFWD \
  -n tag        $FACN.$IDEN.GUI.DIRECTFWD \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/DIRECTBWD \
  -n tag        $FACN.$IDEN.GUI.DIRECTBWD \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/HOMING \
  -n tag        $FACN.$IDEN.GUI.HOMING \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/MOVE \
  -n tag        $FACN.$IDEN.GUI.MOVE \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/ESTOP \
  -n tag        $FACN.$IDEN.GUI.ESTOP \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/SETZERO \
  -n tag        $FACN.$IDEN.GUI.SETZERO \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/POSLIMIT \
  -n tag        $FACN.$IDEN.GUI.POSLIMIT \
  -n end \
  -n dic_info   $FACP/$IDNP/GUI/NEGLIMIT \
  -n tag        $FACN.$IDEN.GUI.NEGLIMIT \
  -n end \
  -n dic_info   $FACP/$IDNP/STATE \
  -n tag        $FACN.$IDEN.STATE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.STATE" \
  -n end \
  -n dic_info   $FACP/$IDNP/POLLRATE/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLRATE.RX \
  -n tag        $FACN.$IDEN.POLLRATE.TX \
  -n tag        $FACN.$IDEN.POLLRATE.EX \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.RX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.TX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLRATE.EX" \
  -n end \
  -n dic_info   $FACP/$IDNP/POLLSUMM/RX+TX+EX \
  -n tag        $FACN.$IDEN.POLLSUMM.RX \
  -n tag        $FACN.$IDEN.POLLSUMM.TX \
  -n tag        $FACN.$IDEN.POLLSUMM.EX \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.RX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.TX" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLLSUMM.EX" \
  -n end \
  -n dic_info   $FACP/$IDNP/ERROR/COUNT \
  -n tag        $FACN.$IDEN.ERROR.COUNT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ERROR.COUNT" \
  -n end \
  -n
 echo "";
}

##############
# delete files
##############
function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scripthome >/dev/null 2>&1; then
  em2rs_devices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
