////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// L-CARD E14-140 driver for CRW-DAQ: main driver class                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20101112 - Created by A.K.                                                 //
// 20241208 - Modified for FPC.                                               //
////////////////////////////////////////////////////////////////////////////////

unit _crw_e140_api; //  E140 main driver class.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 _crw_alloc, _crw_ef, _crw_str, _crw_fio, _crw_rtc,
 _crw_ascio, _crw_sharm, _crw_proc,
 _crw_lcard, _crw_e140_shm;

 {
 List of possible LDevDriverStatus names.
 }
const
 LDevDriverStatusNames = 'INVALID,STOPPED,STARTED';

 {
 Get LCARD driver status: INVALID,STOPPED,STARTED.
 }
function LDevDriverStatus:LongString;

 {
 Start LCARD driver if one not started.
 }
function LDevDriverStart:Boolean;

 {
 Get list of LCARD devices.
 }
function LDevListDevices(Reset:Boolean=false):LongString;

 {
 Find slot number by serial number or return -1.
 }
function LPlataSlotBySerNum(const SerNum:LongString; Id:LongString='31:E140'):Integer;

 {
 For simulation.
 }
function TestE140AdcPars(nChan:Integer; nRange:Integer; Freq:Double):ADC_PAR;

 //////////////////////////////
 // Ring buffer for ADC polling
 //////////////////////////////
type
 TAdcPollRing = record   // For ADC polling
  Sync        : PULONG;  // Synchronization variable
  Buff        : PSmallIntArray; // Data buffer
  Size        : ULONG;   // Buffer size for ADC
  HalfSize    : ULONG;   // Half FIFO size
  HalfFlag    : ULONG;   // 0/1 halfs of buffer
  HalfStep    : Int64;   // Step of half buffers
  DataOffs    : Int64;   // Data offset (flat)
 public
  procedure Reset;
  function  GetSync:ULONG;
  function  Valid:Boolean;
  function  Init(const Adc:ADC_PAR):Boolean;
  function  HalfBuffHasCome:Boolean;
  procedure NextHalf;
 end;
 ///////////////////////
 // E14-140 driver class
 ///////////////////////
type
 E140_FAILURE = class(Exception);
 TLDev_Reporter = procedure(const Msg:LongString); // Procedure to report errors
 TE140 = class(TMasterObject)
 private
  mySlot      : Integer;                   // Slot number or -1
  myHand      : THandle;                   // LCARD Device handle 
  myApi       : IDaqLDevice;	           // LCARD device interface
  myBuffSize  : ULONG;                     // LCARD ADC buffer size
  mySp        : SLOT_PAR;                  // Slot parameters
  myPd        : PLATA_DESCR_U2;            // Plata description
  myAp        : ASYNC_PAR;                 // Uses for IoAsync
  myName      : LongString;                // Module name, must be E140
  mySharm     : TSharedMemory;             // Shared memory instance
  myShared    : PE140_SHARED_BUFFER;       // Shared memory pointer
  mySerNum    : LongString;                // L-CARD Serial number (wanted)
  mySerial    : LongString;                // L-CARD Serial number (readback)
  myRequest   : record                     // Data for I/O requests
   Number     : Integer;                   // 0..E140NumBuffers-1
   Blocks     : Int64;                     // Total number of blocks
   Samples    : Int64;                     // Total number of samples
   Index      : array[0..E140NumBuffers-1] of Integer;    // Shared memory index
   Offset     : array[0..E140NumBuffers-1] of Integer;    // Shared memory offset
  end;
  myReporter  : TLDev_Reporter;            // Error report routine
  myBuffTime  : Integer;                   // Buffer capacity in time
  myAdcRing   : TAdcPollRing;              // Ring buffer for ADC polling
 private
  function  GetSlot:Integer;
  procedure SetReporter(aReporter:TLDev_Reporter);
  procedure PrintReport(const msg:LongString);
  procedure Success(const s:LongString); 
  procedure Failure(const s:LongString);
  procedure BugReport(const E:Exception; const msg:LongString='');
  procedure PrintDescr(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2);
  procedure PrintADC(const aAdc:ADC_PAR);
  function  FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PAR):Integer;
  function  GetSharedBuff:PE140_SHARED_BUFFER;
  function  GetSharedName:LongString;
  function  GetSerNum:LongString;
  procedure SetSerNum(const aSerNum:LongString);
  function  GetSerial:LongString;
 protected
  function  ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD; virtual;
 public
  constructor Create(aShare:LongString; aReporter:TLDev_Reporter=nil;
                     aSlot:Integer=-1; aAdc:pADC_PAR=nil; aBuffTime:Integer=E140DefBufTime);
  destructor  Destroy; override;
 public
  property  Reporter     : TLDev_Reporter      write SetReporter;
  property  Report       : LongString          write PrintReport;
  property  SharedBuff   : PE140_SHARED_BUFFER read GetSharedBuff;
  property  SharedName   : LongString          read GetSharedName;
  property  SerNum       : LongString          read GetSerNum write SetSerNum;
  property  Serial       : LongString          read GetSerial;
  property  Slot         : Integer             read GetSlot;
 public
  function  ReadSerial(Save:Boolean=True):LongString;
  function  Open(aSlot:Integer; aAdc:pADC_PAR=nil):Boolean;
  function  Opened:Boolean;
  procedure Close;
  function  Start(aAdc:pADC_PAR=nil):Double;
  function  Started:Double;
  function  Stop:Double;
  function  Polling:Int64;
  function  PollAdcHandler(const Adc:ADC_PAR):Int64;
  function  DAC_SAMPLE(DacData:ULONG; DacChannel:WORD):Boolean;
  function  TTL_OUT(TtlOut:ULONG):Boolean;
  function  TTL_IN(var TtlIn:ULONG):Boolean;
  function  GetLastErrorInfo:LongString;
 public
  function  AdcRingSize:ULONG;
  function  RequestAdcRing(aSize:ULONG):ULONG;
 end;

function NewE140(aShare:LongString; aReporter:TLDev_Reporter=nil; aSlot:Integer=-1; aAdc:pADC_PAR=nil; aBuffTime:Integer=E140DefBufTime):TE140;
procedure Kill(var TheObject:TE140); overload;

const // индексы доступных диапазонов входного напряжения модуля E14-440
 ADC_INPUT_RANGE_10000mV_E140 = ($00);
 ADC_INPUT_RANGE_2500mV_E140  = ($01);
 ADC_INPUT_RANGE_625mV_E140   = ($02);
 ADC_INPUT_RANGE_156mV_E140   = ($03);
 INVALID_ADC_INPUT_RANGE_E140 = ($04);

const
 LcardBigBufferSize = 1024*1024*10;
 LcardTimeStepMs    : Integer = 50;

implementation

 ////////////////////
 // Utility functions
 ////////////////////

function LDevDriverStatus:LongString;
var s:LongString;
begin
 Result:='INVALID'; s:='';
 if RunCommand('unix ldev-cpl status',s) then
 if StartsText('ldev status:',s) then s:=ExtractWord(3,s,ScanSpaces);
 if (WordIndex(s,LDevDriverStatusNames,ScanSpaces)>0) then Result:=s;
end;

function LDevDriverStart:Boolean;
var status:LongString;
begin
 Result:=False;
 status:=LDevDriverStatus;
 if SameText(status,'INVALID') then Exit;
 if SameText(status,'STARTED') then Exit(True);
 if SameText(status,'STOPPED') then begin
  if RunCommand('sudo -n unix ldev-cpl start',status) then begin
   status:=LDevDriverStatus;
   if SameText(status,'STARTED') then Exit(true);
  end;
 end;
end;

const
 lsldevOut:LongString='';

function LDevListDevices(Reset:Boolean=false):LongString;
begin
 Result:='';
 if Reset then lsldevOut:='';
 if (lsldevOut='') then
 if RunCommand('unix lsldev',lsldevOut)
 then lsldevOut:=ValidateEol(lsldevOut)
 else lsldevOut:='';
 Result:=lsldevOut;
end;

function LPlataSlotBySerNum(const SerNum:LongString; Id:LongString='31:E140'):Integer;
var il,iw,nl,nw,slot:Integer; Lines,Line,Iden,Sern:LongString;
begin
 Result:=-1;
 Lines:=LDevListDevices;
 for il:=1 to WordCount(Lines,EolnDelims) do begin
  Line:=ExtractWord(il,Lines,EolnDelims);
  Iden:=ExtractWord(5,Line,ScanSpaces);
  Sern:=ExtractWord(8,Line,ScanSpaces);
  if not SameText(Iden,Id) then Continue;
  Slot:=StrToIntDef(ExtractWord(2,Line,ScanSpaces),-1);
  if (Slot<0) then Continue;
  if (SerNum='') then Exit(Slot);
  if SameText(SerNum,Sern) then Exit(Slot);
 end;
end;

function TestE140AdcPars(nChan:Integer; nRange:Integer; Freq:Double):ADC_PAR;
const Adc:ADC_PAR=(); var i:Integer;
begin
 Adc:=Default(ADC_PAR);
 with Adc.t1 do begin
  s_Type           := L_ADC_PARAM;
  AutoInit         := 1;
  FIFO             := 0;               // размер половины аппаратного буфера FIFO на плате
  IrqStep          := 0;               // шаг генерации прерываний
  Pages            := 0;               // размер кольцевого буфера в шагах прерываний
//ClkSource        := INT_ADC_CLOCK_E140;
//EnableClkOutput  := ADC_CLOCK_TRANS_DISABLED_E140;
//InputMode        := NO_SYNC_E140;
  SynchroType      := 0;
  SynchroMode      := 0;
  AdChannel        := 0;
  AdPorog          := 0;
  NCh              := nChan;
  for i:=0 to nChan-1 do Chn[i]:=(i and 31) or (nRange shl $6);
  dRate            := Freq;
  FPDelay          := 0;
  dKadr            := 0;
 end; 
 Result:=Adc;
end;

 //////////////////////////////
 // TAdcPollRing implementation
 //////////////////////////////
procedure TAdcPollRing.Reset;
begin
 Size:=0;
 Buff:=nil;
 Sync:=nil;
 HalfSize:=0;
 HalfFlag:=0;
 HalfStep:=0;
 DataOffs:=0;
end;

function TAdcPollRing.GetSync:ULONG;
const z=High(ULONG);
begin
 if Assigned(Sync)
 then Result:=InterlockedCompareExchange(Sync^,z,z)
 else Result:=0;
end;

function TAdcPollRing.Valid:Boolean;
begin
 if not Assigned(Sync) then Exit(False);
 if (HalfSize<=0) then Exit(False);
 Result:=true;
end;

function TAdcPollRing.Init(const Adc:ADC_PAR):Boolean;
begin
 HalfSize:=((Adc.t1.IrqStep*Adc.t1.Pages) div 2);
 Result:=Valid;
end;

function TAdcPollRing.HalfBuffHasCome:Boolean;
var HFlag:ULONG;
begin
 HFlag:=IfThen((GetSync<=HalfSize),0,1);
 Result:=(HFlag=HalfFlag)
end;

procedure TAdcPollRing.NextHalf;
begin
 HalfFlag:=IfThen((HalfFlag<>0),0,1);
 Inc(DataOffs,HalfSize);
 Inc(HalfStep);
end;

 //////////////////////
 // TE140 help routines
 //////////////////////

function LSucceed(Code:ULONG):Boolean;
begin
 Result:=(Code=L_SUCCESS);
end;

function ValidHandle(h:THandle):Boolean;
begin
 Result:=(h<>0) and (h<>INVALID_HANDLE_VALUE);
end;

procedure DefaultReporter(const Msg:LongString);
begin
 StdOut.Put:=Msg;
end;

procedure FillTestAdcPar(var adcPar: ADC_PAR);
begin
 adcPar.t1.s_Type := L_ADC_PARAM;
 adcPar.t1.AutoInit := 1;
 adcPar.t1.dRate := 100.0;
 adcPar.t1.dKadr := 0;
 adcPar.t1.dScale := 0;
 adcPar.t1.SynchroType := 0;
 adcPar.t1.SynchroSensitivity := 0;
 adcPar.t1.SynchroMode := 0;
 adcPar.t1.AdChannel := 0;
 adcPar.t1.AdPorog := 0;
 adcPar.t1.NCh := 4;
 adcPar.t1.Chn[0] := $0;
 adcPar.t1.Chn[1] := $1;
 adcPar.t1.Chn[2] := $2;
 adcPar.t1.Chn[3] := $3;
 adcPar.t1.FIFO := 4096;
 adcPar.t1.IrqStep := 4096;
 adcPar.t1.Pages := 32;
 adcPar.t1.IrqEna := 1;
 adcPar.t1.AdcEna := 1;
end;

 ///////////////////////
 // TE140 implementation
 ///////////////////////

procedure TE140.SetReporter(aReporter:TLDev_Reporter);
begin
 if Assigned(Self) then myReporter:=aReporter;
end;

procedure TE140.PrintReport(const msg:LongString);
begin
 if Assigned(Self) then
 if Assigned(myReporter) then
 try
  myReporter(msg);
 except
  on E:Exception do LockedInc(StdIoErrorCount);
 end;
end;

function TE140.GetSharedBuff:PE140_SHARED_BUFFER;
begin
 if Assigned(Self)
 then Result:=myShared
 else Result:=nil;
end;

function TE140.GetSlot:Integer;
begin
 if Assigned(Self)
 then Result:=mySlot
 else Result:=-1;
end;

function TE140.GetSharedName:LongString;
begin
 if Assigned(Self)
 then Result:=mySharm.Name
 else Result:='';
end;

function TE140.GetSerNum:LongString;
begin
 if Assigned(Self)
 then Result:=mySerNum
 else Result:='';
end;

procedure TE140.SetSerNum(const aSerNum:LongString);
begin
 if Assigned(Self) then mySerNum:=Trim(aSerNum);
end;

function TE140.GetSerial:LongString;
begin
 if Assigned(Self)
 then Result:=mySerial
 else Result:='';
end;

function TE140.Started:Double;
begin
 if Assigned(Self) and Assigned(myShared)
 then Result:=myShared.StartTime
 else Result:=0;
end;

function TE140.Opened:Boolean;
begin
 if Assigned(Self)
 then Result:=Assigned(myApi) and ValidHandle(myHand)
 else Result:=False;
end;

function TE140.ReadSerial(Save:Boolean=True):LongString;
var sp:SLOT_PAR; pd:PLATA_DESCR_U2;
begin
 Result:='';
 if Assigned(Self) then
 try
  if Opened then
  if Assigned(myApi) then
  if LSucceed(myApi.GetSlotParam(sp)) then
  if LSucceed(myApi.ReadPlataDescr(pd)) then begin
   if Save then begin mySp:=sp; myPd:=pd; end;
   Result:=LPlataSerNum(sp,pd);
  end;
 except
  on E:Exception do BugReport(E,'ReadSerial');
 end;
end;

procedure TE140.Success(const s:LongString);
begin
 Report:='@E140.OnSuccess='+s;
end;
 
procedure TE140.Failure(const s:LongString);
begin
 Raise E140_Failure.Create(s);
end; 

procedure TE140.BugReport(const E:Exception; const msg:LongString);
begin
 if Assigned(E) then begin
  if Length(msg)>0
  then Report:=Format('@E140.OnException=%s,%s,"%s"',[E.ClassName,E.Message,msg])
  else Report:=Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]);
 end;
 LockedInc(StdIoErrorCount);
end;

function TE140.Open(aSlot:Integer; aAdc:pADC_PAR=nil):Boolean;
var lnk:LongString; ifac:LUnknown; icod,lcod:Integer;
begin
 Result:=false;
 try
  //
  // Check if one already started or opened
  //
  if Started>0 then Stop;
  if Opened then Close;
  //
  // Check device exists.
  //
  lnk:=LDeviceLink(aSlot);
  if LDeviceExists(lnk)
  then Success('Device exist '+lnk)
  else Failure('Error: Device is not exists '+lnk);
  //
  // Create Instance and query interface.
  //
  ifac:=LCreateInstance(aSlot);
  icod:=GetLastOsError;
  if Assigned(ifac)
  then Success(Format('Slot %d',[aSlot]))
  else Failure('Error: '+LErrorCodeName(icod));
  lcod:=ifac.QueryInterface(IID_ILDEV,myApi);
  if LSucceed(lcod) and Assigned(myApi)
  then Success('Interface '+GuidToString(IID_ILDEV))
  else Failure('Error: '+LErrorCodeName(lcod));
  //
  // Test plata load succeed.
  //
  if LSucceed(myApi.PlataTest)
  then Success('PlataTest')
  else Failure('Error: PlataTest ');
  //
  // Open device.
  //
  myHand:=myApi.OpenLDevice; 
  if ValidHandle(myHand)
  then Success('OpenLDevice')
  else Failure('Error: OpenLDevice '+LErrorCodeName(GetLastOsError)); 
  //
  // Read board name, serial number and other params.
  //
  mySerial:=ReadSerial;
  myName:=LBoardName(mySp);
  if SameText(myName,'E140')
  then Success('LBoardName '+myName+' ReadSerial '+mySerial)
  else Failure('Error: Device is not E14-140.');
  //
  // Device reset
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice'); 
  //
  // Check shared memory
  //
  if Assigned(myShared)
  then Success('SharedMemory = '+IntToStr(mySharm.Size))
  else Failure('SharedMemory');
  ZeroMemory(myShared,mySharm.Size);
  //
  // Read device driver parameters
  //
  if LSucceed(myApi.GetSlotParam(myShared.SlotPar))
  then Success('GetSlotParam')
  else Failure('GetSlotParam');
  if LSucceed(myApi.ReadPlataDescr(myShared.PlataDescr))
  then Success('GetPlataDescr')
  else Failure('GetPlataDescr');
  PrintDescr(myShared.SlotPar,myShared.PlataDescr);
  //
  // Request ADC buffer
  //
  myBuffSize:=LcardBigBufferSize;
  if LSucceed(myApi.RequestBufferStream(myBuffSize,L_STREAM_ADC))
  then Success('RequestBufferStream '+IntToStr(myBuffSize)+' bytes')
  else Failure('RequestBufferStream');
  //
  // Set/Get current ADC settings
  //
  if Assigned(aAdc) then begin
   myShared.AdcInited:=aAdc^;
   myShared.AdcWanted:=aAdc^;
   myShared.AdcActual:=aAdc^;
   PrintAdc(myShared.AdcWanted);
   //
   // Read actual (accepted) ADC settings
   //
   if not LSucceed(myApi.FillDAQparameters(PDAQ_PAR(@myShared.AdcActual)^))
   then Failure('FillDAQparameters')
   else Success('FillDAQparameters');
   PrintAdc(myShared.AdcActual);
  end;
  //
  // Done.
  //
  Result:=Opened;
  if Opened
  then Report:='Opened Slot '+IntToStr(aSlot)
  else Report:='Failed Slot '+IntToStr(aSlot);
 except
  on E:Exception do begin
   BugReport(E,'OpenSlot');
   Close;
  end;
 end;
end;

constructor TE140.Create(aShare:LongString; aReporter:TLDev_Reporter;
                         aSlot:Integer; aAdc:pADC_PAR; aBuffTime:Integer);
begin
 Inherited Create;
 mySlot:=-1;
 myApi:=nil;
 myAdcRing.Reset;
 Reporter:=aReporter;
 myBuffTime:=aBuffTime;
 mySp:=Default(SLOT_PAR);
 myPd:=Default(PLATA_DESCR_U2); 
 mySharm:=NewSharedMemory(aShare,Sizeof(myShared^));
 mySharm.Master:=@mySharm;
 myShared:=mySharm[0];
 if (aSlot>=0) then Open(aSlot,aAdc);
end;

destructor TE140.Destroy;
begin
 if Started>0 then Stop;
 if Opened then Close;
 Kill(mySharm);
 mySlot:=-1;
 myApi:=nil;
 myName:='';
 mySerNum:='';
 mySerial:='';
 Inherited Destroy;
end;

procedure TE140.PrintDescr(const sp:SLOT_PAR; const pd:PLATA_DESCR_U2);
begin
 if Assigned(Self) then
 try
  Report:=GetE140Description(sp,pd,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E,'PrintDescr');
 end;
end;

procedure TE140.PrintADC(const aAdc:ADC_PAR);
begin
 if Assigned(Self) then
 try
  Report:=GetE140AdcPars(aAdc,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E,'PrintAdc');
 end;
end;

function TE140.FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PAR):Integer;
var Quantity,AdjustFactor,MaxBuffLen:Integer; BuffRate:Double;
begin
 Result:=0;
 if Assigned(Self) then begin
  if (aBuffTime<=0) then Result:=E140DefBuffLen else with aAdc.t1 do begin
   MaxBuffLen:=SizeOf(myShared.Buffer) div SizeOf(myShared.Buffer[0]);
   BuffRate:=Max(dRate,dKadr*Nch);
   AdjustFactor:=Max(Nch,1);
   while (AdjustFactor mod 2) = 0 do
   AdjustFactor:=AdjustFactor div 2;
   MaxBuffLen:=MaxBuffLen div E140NumBuffers;
   Quantity:=E140BuffAdjust*AdjustFactor;
   MaxBuffLen:=MaxBuffLen - Quantity;
   Result:=Round(aBuffTime*BuffRate);
   Result:=Min(MaxBuffLen,Result);
   Result:=Max(E140MinBuffLen,Result);
   Result:=AdjustBufferSize(Result,Quantity);
   if Result*E140NumBuffers*SizeOf(myShared.Buffer[0])>SizeOf(myShared.Buffer)
   then Failure(Format('Invalid buffer length %d',[Result]));
  end;
 end;
end;

const
 IsClearHeader = False;
 IsClearBuffer = True;

procedure TE140.Close;
begin
 if Assigned(Self) then
 try
  if Started>0 then Stop;
  if Assigned(myApi) then begin
   if ValidHandle(myHand) then begin
    myApi.StopLDevice;
    myApi.CloseLDevice;
   end;
   myApi:=nil;
  end;
  mySlot:=-1;
  myName:=''; mySerial:='';
  myHand:=INVALID_HANDLE_VALUE;
  if Assigned(myShared) then begin
   if IsClearHeader then ZeroMemory(@myShared.Header,SizeOf(myShared.Header));
   if IsClearBuffer then ZeroMemory(@myShared.Buffer,SizeOf(myShared.Buffer));
  end;
 except
  on E:Exception do BugReport(E,'Close');
 end;
end;

function TE140.Stop:Double;
begin
 Result:=Started;
 if Result>0 then
 try
  //
  // Check interface
  //
  if Assigned(myApi)
  then Success('STOP')
  else Failure('STOP'); 
  //
  // Reset device
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice');
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Start(aAdc:pADC_PAR):Double;
var i,Leng:Integer; Rate:Double; aTestAdcPar:ADC_PAR;
begin
 Result:=0;
 if Opened then
 try
  //
  // Stop if one already started
  //
  if Started>0 then Stop;
  //
  // Check interface
  //
  if Assigned(myApi)
  then Success('START')
  else Failure('START'); 
  //
  // Check shared memory
  //
  if Assigned(myShared)
  then Success('SharedMemory = '+IntToStr(mySharm.Size))
  else Failure('SharedMemory');
  //
  // Reset device
  //
  if LSucceed(myApi.StopLDevice)
  then Success('StopLDevice')
  else Failure('StopLDevice');
  //
  //
  //
  FillTestAdcPar(aTestAdcPar);
  if (aAdc=nil) then aAdc:=@aTestAdcPar;
  //
  // Write wanted ADC settings if one specified
  //
  if Assigned(aAdc) then begin
   myShared.AdcWanted:=aAdc^;
   PrintAdc(myShared.AdcWanted);
   myShared.AdcActual:=myShared.AdcWanted;
   if not LSucceed(myApi.FillDaqParameters(PDAQ_PAR(@myShared.AdcActual)^))
   then Failure('FillDaqParameters')
   else Success('FillDaqParameters');
   PrintAdc(myShared.AdcActual);
  end;
  //
  // Calculate wanted buffer size
  // It's near BuffTime ms of data taking
  //
  Leng:=FindBufferLength(myBuffTime,myShared.AdcActual);
  Success(Format('BuffTime = %d',[myBuffTime]));
  Success(Format('BuffLeng = %d',[Leng]));
  //
  // SetParametersStream
  //
  myAdcRing.Reset;
  myAdcRing.Size:=myBuffSize;
  myAdcRing.Init(myShared.AdcActual);
  if LSucceed(myApi.SetParametersStream(PDAQ_PAR(@myShared.AdcActual)^, myAdcRing.Size, myAdcRing.Buff, myAdcRing.Sync,L_STREAM_ADC))
  then Success('SetParametersStream')
  else Failure('SetParametersStream');
  
  {
  //
  // Prepare asynchronous I/O buffers
  //
  for i:=0 to E140NumBuffers-1 do begin
   // Initialize OVERLAPPED, asynchronous I/O event
   ZeroMemory(@myRequest.ReadOv[i],SizeOf(myRequest.ReadOv[i]));
   myRequest.ReadOv[i].hEvent := CreateEvent(nil, FALSE , FALSE, nil);
   // Prepare IoReq struct for ReadData requests
   ZeroMemory(@myRequest.IoReq[i],SizeOf(myRequest.IoReq[i]));
   myRequest.IoReq[i].Buffer := @myShared.Buffer[i*Leng];
   myRequest.IoReq[i].NumberOfWordsToPass := Leng;
   myRequest.IoReq[i].NumberOfWordsPassed := 0;
   myRequest.IoReq[i].Overlapped := @myRequest.ReadOv[i];
   Rate:=Max(myShared.AdcActual.AdcRate,myShared.AdcActual.KadrRate*myShared.AdcActual.ChannelsQuantity);
   myRequest.IoReq[i].TimeOut := Round(Int(Leng/Rate)) + Max(E140DefTimeout,myBuffTime);
   myRequest.Offset[i]:=SizeOf(myShared.Header)+i*Leng*SizeOf(myShared.Buffer[0]);
   myRequest.Index[i]:=i*Leng;
  end;
  // 
  // Start first asynchronous I/O request queue
  //
  myRequest.Number:=0;
  for i:=0 to E140NumBuffers-1 do
  if not myApi.ReadData(@myRequest.IoReq[i])
  then Failure(Format('ReadData %d',[i]))
  else Success(Format('ReadData %d',[i]));
  //
  // Now ready to start data acquisition
  //
  if not myApi.START_ADC
  then Failure('START_ADC')
  else Success('START_ADC');
  }
  //
  // Start device
  //
  if LSucceed(myApi.StartLDevice)
  then Success('StartLDevice')
  else Failure('StartLDevice');
  //
  myShared.StartTime:=MSecNow;
  myRequest.Samples:=0;
  myRequest.Blocks:=0;
  Result:=Started;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.AdcRingSize:ULONG;
begin
 Result:=0;
 if Assigned(Self) then Result:=myAdcRing.Size;
end;

function TE140.RequestAdcRing(aSize:ULONG):ULONG;
begin
 Result:=0;
 if Assigned(Self) then
 if Assigned(myApi) then begin
  if LSucceed(myApi.RequestBufferStream(aSize,L_STREAM_ADC)) then begin
   Report:='Allocated ADC Buffer : '+IntToStr(aSize);
   myAdcRing.Size:=aSize;
  end;
 end;
end;

function TE140.PollAdcHandler(const Adc:ADC_PAR):Int64;
var nBytes,nWords:DWORD; dt:Double;
begin
 Result:=0;
 if Assigned(Self) then
 if Assigned(myApi) then
 if Assigned(myShared) then
 try
  if (Started>0) then begin
   if not myAdcRing.Valid then Failure('Error: invalid AdcRing');
   if myAdcRing.HalfBuffHasCome then begin
    dt:=(mSecNow-myShared.StartTime)*1e-3;
    StdOut.Put:=Format('HalfStep=%d; DataOffs=%d; Elapsed=%1.3f',
                 [myAdcRing.HalfStep,myAdcRing.DataOffs,dt]);
    myAdcRing.NextHalf;
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Polling:Int64;
var nBytes,nWords:DWORD;
begin
 Result:=0;
 if Assigned(Self) then
 if Assigned(myApi) then
 if Assigned(myShared) then
 try
 if Started>0 then begin
   PollAdcHandler(myShared.AdcActual);
   {
   Result:=myRequest.Blocks;
   if GetOverlappedResult(myHand,myRequest.IoReq[myRequest.Number].Overlapped^,nBytes,False) then begin
    nWords:=nBytes div SizeOf(myShared.Buffer[0]);
    if nBytes mod SizeOf(myShared.Buffer[0]) <> 0
    then Failure(Format('Invalid nBytes = %d',[nBytes]));
    ProcessBuffer(nWords,myRequest.Number,myRequest.Samples,myRequest.Blocks);
    Inc(myRequest.Blocks); 
    Result:=myRequest.Blocks;
    Inc(myRequest.Samples,nWords);
    if not myApi.ReadData(@myRequest.IoReq[myRequest.Number])
    then Failure(Format('ReadData %d',[myRequest.Number]))
    else Success(Format('ReadData %d',[myRequest.Number]));
    myRequest.Number:=(myRequest.Number+1) mod E140NumBuffers;
   end else
   if GetLastError <> ERROR_IO_INCOMPLETE then begin
    Failure(Format('GetOverlappedResult=%d',[GetLastError]));
   end;
   }
  end;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD;
var sn:LongString; nw:DWORD; nr,ri,ro:Integer; nb,ns:Int64;
begin
 Result:=0;
 if Assigned(Self) then
 try
  sn:=mySharm.Name; nw:=nWords;
  nr:=nRequest; nb:=nBlocks; ns:=nSamples;
  ro:=myRequest.Offset[nr]; ri:=myRequest.Index[nr];
  Report:=Format('@E140.OnReadout=%s,%d,%d,%d,%d,%d,%d',
                [sn, nw, ro, ri, nr, nb, ns]);
  Result:=nw;
 except
  on E:Exception do BugReport(E,'ProcessBuffer');
 end;
end;

function TE140.DAC_SAMPLE(DacData:ULONG; DacChannel:WORD):Boolean;
begin
 Result:=False;
 if Opened then
 try
  myAp:=Default(ASYNC_PAR);
  myAp.s_Type:=L_ASYNC_DAC_OUT;
  myAp.Mode:=DacChannel;
  myAp.Data[0]:=DacData;
  Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myAp)^));
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.TTL_OUT(TtlOut:ULONG):Boolean;
begin
 Result:=False;
 if Opened then
 try
  myAp:=Default(ASYNC_PAR);
  myAp.s_Type:=L_ASYNC_TTL_OUT;
  myAp.Data[0]:=TtlOut;
  Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myAp)^));
 except
  on E:Exception do BugReport(E,'TTL_OUT');
 end;
end;

function TE140.TTL_IN(var TtlIn:ULONG):Boolean;
begin
 Result:=False;
 if Opened then
 try
  myAp:=Default(ASYNC_PAR);
  myAp.s_Type:=L_ASYNC_TTL_INP;
  Result:=LSucceed(myApi.IoAsync(PDAQ_PAR(@myAp)^));
  TtlIn:=myAp.Data[0];
 except
  on E:Exception do BugReport(E,'TTL_IN');
 end;
end;
 
function TE140.GetLastErrorInfo:LongString;
begin
 Result:='E140 error: NIL reference';
 if Assigned(Self) then
 if Assigned(myShared) then
 try
  if Assigned(myApi) then
  Result:=Format('E140 error %d',[myShared.LastError]);
 except
  on E:Exception do Result:=E.Message;
 end;
end;

function NewE140(aShare:LongString; aReporter:TLDev_Reporter; aSlot:Integer; aAdc:pADC_PAR; aBuffTime:Integer):TE140;
begin
 Result:=nil;
 try
  Result:=TE140.Create(aShare,aReporter,aSlot,aAdc,aBuffTime);
  if not Assigned(Result.myShared) then Raise E140_FAILURE.Create('Shared memory fails!');
 except
  on E:Exception do begin
   Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
   Kill(Result);
  end;
 end; 
end;

procedure Kill(var TheObject:TE140); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
 end; 
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_e140_api;
begin
 lsldevOut:='';
end;

procedure Free_crw_e140_api;
begin
 lsldevOut:='';
end;

initialization

 Init_crw_e140_api;

finalization

 Free_crw_e140_api;

end.

//////////////
// END OF FILE
//////////////

