 {
 ***********************************************************************
 Daq Pascal application program ZUPDC_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @PrintCmdTable - Print Command Table state.
| @Simulate id v - Set simulation data v to id=(AV,SV,AA,
|                  SA,RM,OT,OP,UP,AS,OS,AL,PS,OHM,NLP).
| @Setting id v  - Send command id with data v, id=(ADRn,
|                  RMTn,VOLn,CURn,OVPn,UVPn,FLDn,OUTn,ASTn,
|                  RM,SV,SA,OP,UP,FD,AS,OT,OS,AL,PS,DCL,MDL,
|                  RATE_A,RATE_V).
| @Ramping id v  - Start ramping id=(VOLn,CURn) to v value.
| @Remote @cmd   - run @cmd remotely: send to DIM server
| @Edit id       - edit id
| @MenuToolsOpen - open Menu Tools dialog.
| @LoadIni       - load params from INI file.
| @SaveIni       - save params to   INI file.
| @DimGuiClick   - handle remote click from DIM
|********************************************************
[]
 }
program ZUPDC_DRV;
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 st_NoReq           = 0;         { No request sent                  }
 st_WaitAns         = 1;         { Waiting answer                   }
 st_WaitGap         = 2;         { Waiting time gap                 }
 st_TimeOut         = 3;         { TimeOut found                    }
 cm_ADRn            = 1;         { :ADRn;                           }
 cm_DCL             = 2;         { :DCL;                            }
 cm_RMTn            = 3;         { :RMTn;                           }
 cm_RMTa            = 4;         { :RMT?;                           }
 cm_MDLa            = 5;         { :MDL?;                           }
 cm_REVa            = 6;         { :REV?;                           }
 cm_VOLn            = 7;         { :VOLn;                           }
 cm_VOLs            = 8;         { :VOL!;                           }
 cm_VOLa            = 9;         { :VOL?;                           }
 cm_CURn            = 10;        { :CURn;                           }
 cm_CURs            = 11;        { :CUR!;                           }
 cm_CURa            = 12;        { :CUR?;                           }
 cm_OUTn            = 13;        { :OUTn;                           }
 cm_OUTa            = 14;        { :OUT?;                           }
 cm_FLDn            = 15;        { :FLDn;                           }
 cm_FLDa            = 16;        { :FLD?;                           }
 cm_OVPn            = 17;        { :OVPn;                           }
 cm_OVPa            = 18;        { :OVP?;                           }
 cm_UVPn            = 19;        { :UVPn;                           }
 cm_UVPa            = 20;        { :UVP?;                           }
 cm_ASTn            = 21;        { :ASTn;                           }
 cm_ASTa            = 22;        { :AST?;                           }
 cm_STAa            = 23;        { :STA?;                           }
 cm_ALMa            = 24;        { :ALM?;                           }
 cm_STPa            = 25;        { :STP?;                           }
 cm_STTa            = 26;        { :STT?;                           }
 MaxCmdNum          = 26;        { Max command id number            }
 cf_RO              = 0;         { Read Only                        }
 cf_RW              = 1;         { Read/Write                       }
 ao_POLL_RATE       = 0;         { Analog outputs...                }
 ao_ERROR_CNT       = 1;         {                                  }
 ao_PAR_AA          = 2;         {                                  }
 ao_PAR_AV          = 3;         {                                  }
 ao_PAR_AW          = 4;         {                                  }
 ao_PAR_SA          = 5;         {                                  }
 ao_PAR_SV          = 6;         {                                  }
 ao_PAR_OS          = 7;         {                                  }
 ao_PAR_AL          = 8;         {                                  }
 ao_PAR_PS          = 9;         {                                  }
 ao_PAR_OP          = 10;        {                                  }
 ao_PAR_UP          = 11;        {                                  }
 si_OFFLINE         = 'OFFLINE'; { To identify offline state        }
 RS_232             = 232;       {                                  }
 RS_422             = 422;       {                                  }
 RS_485             = 485;       {                                  }
 DelayOnStart       = 1000;      { Delay before start polling       }
 EnforceCRLF        = false;     { Enforce CRLF    on each send     }
 UseCmdHash         = true;      { Use CmdHash for fast search      }

type
 TTagRef = record tag,nai,ndi,nao,ndo:Integer; val:Real; end;
 TRamping = record t0,t1,v0,v1:real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 ZUPDC             : record      { ZUP DC data                      }
  Simulator        : Boolean;    { Simulator mode                   }
  ecTimeOut        : Integer;    { Error code: TimeOut              }
  Com              : record      { COM port data                    }
   Port            : Integer;    {  Port number                     }
   Addr            : Integer;    {  Address                         }
   Status          : Integer;    {  Status flag                     }
   TimeOut         : Integer;    {  TimeOut, ms                     }
   TimeGap         : Integer;    {  Time delay after poll, ms       }
   PollPeriod      : Integer;    { Poll period, ms                  }
   Req,Ans,Buf     : String;     {  Request, Answer                 }
   ReqTime,AnsTime : Real;       {  Request, Answer time, ms        }       
   LastPoll        : Real;       { Time of last poll, ms            }
   PollRate        : Integer;    { Poll rate, poll/sec              }
   Interface       : Integer;    { 232/422/485                      }
  end;                           {                                  }
  Cmd              : record      {  Command cycle data              }
   Num             : Integer;    {   Current command number         }
   Enabled         : array [1..MaxCmdNum] of Boolean; {             }
   Acronym         : array [1..MaxCmdNum] of String;  {             }
   AnsPref         : array [1..MaxCmdNum] of String;  {             }
   DataFmt         : array [1..MaxCmdNum] of String;  {             }
   Comment         : array [1..MaxCmdNum] of String;  {             }
   OpData          : array [1..MaxCmdNum] of Real;    {             }
   OpBuff          : array [1..MaxCmdNum] of Real;    {             }
   SimDat          : array [1..MaxCmdNum] of Real;    {             }
   AoNum           : array [1..MaxCmdNum] of Integer; {             }
   Flags           : array [1..MaxCmdNum] of Integer; {             }
   Tag             : array [1..MaxCmdNum] of Integer; {             }
   Min             : array [1..MaxCmdNum] of Real;    {             }
   Max             : array [1..MaxCmdNum] of Real;    {             }
  end;                           {                                  }
  CmdHash          : Integer;    { Hash list ref. for fast search   }
  POLL_ENABLE      : TTagRef;    {                                  }
  POLL_RATE        : TTagRef;    {                                  }
  ERROR_CNT        : TTagRef;    {                                  }
  ID_NAME          : TTagRef;    {                                  }
  ID_MDL           : TTagRef;    {                                  }
  ID_REV           : TTagRef;    {                                  }
  PAR_AA           : TTagRef;    {                                  }
  PAR_AV           : TTagRef;    {                                  }
  PAR_AW           : TTagRef;    {                                  }
  PAR_SA           : TTagRef;    {                                  }
  PAR_SV           : TTagRef;    {                                  }
  PAR_OS           : TTagRef;    {                                  }
  PAR_AL           : TTagRef;    {                                  }
  PAR_PS           : TTagRef;    {                                  }
  PAR_OP           : TTagRef;    {                                  }
  PAR_UP           : TTagRef;    {                                  }
  SET_FD           : TTagRef;    {                                  }
  SET_AS           : TTagRef;    {                                  }
  SET_OT           : TTagRef;    {                                  }
  RATE_A           : TTagRef;    {                                  }
  RATE_V           : TTagRef;    {                                  }
  RAMP_A           : TTagRef;    {                                  } 
  RAMP_V           : TTagRef;    {                                  } 
  SET_FD_GUI       : TTagRef;    {                                  }
  SET_AS_GUI       : TTagRef;    {                                  }
  SET_OT_GUI       : TTagRef;    {                                  }
  RampingA         : TRamping;   {                                  }
  RampingV         : TRamping;   {                                  }
  RampingTime      : Real;       {                                  }
  ModelCode        : Integer;    { Encoded model ranges             }
  ModelName        : String;     { Nemic-Lambda ZUP(6V-33A)         }
  SimOhm           : Real;       { Simulate output resistance, Ohm  }
  SimNLP           : Real;       { Simulate noise level, percent    }
 end;                            {                                  }
 cmd_DimTagUpdate  : Integer;    { @DimTagUpdate                    }
 cmd_Simulate      : Integer;    { @Simulate                        }
 cmd_PrintCmdTable : Integer;    { @PrintCmdTable                   }
 cmd_Ramping       : Integer;    { @Ramping                         }
 cmd_Setting       : Integer;    { @Setting                         }
 cmd_Remote        : Integer;    { @Remote                          }
 cmd_Edit          : Integer;    { @Edit                            }
 cmd_MenuToolsOpen : Integer;    { @MenuToolsOpen                   }
 cmd_LoadIni       : Integer;    { @LoadIni                         }
 cmd_SaveIni       : Integer;    { @SaveIni                         }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 //
 // Ramping routines.
 //
 procedure RampingClear(var R:TRamping);
 begin
  R.t0:=0; R.t1:=0; R.v0:=0; R.v1:=0;
 end;
 procedure RampingStart(var R:TRamping; v0,v1,rate:Real);
 begin
  RampingClear(R);
  R.t0:=mSecNow; R.t1:=R.t0; R.v0:=v0; R.v1:=v1;
  if (rate>0) then R.t1:=R.t1+1000*abs((v1-v0)/rate);
 end;
 function RampingGoing(var R:TRamping):Boolean;
 begin
  RampingGoing:=(R.t0>0);
 end;
 function RampingComplete(var R:TRamping):Boolean;
 begin
  RampingComplete:=(mSecNow>R.t1);
 end;
 function RampingValue(var R:TRamping):Real;
 var v,ms:Real;
 begin
  ms:=mSecNow;
  if (ms>R.t1) or (R.t0=0) then v:=R.v1
  else v:=R.v0+(R.v1-R.v0)*(ms-R.t0)/(R.t1-R.t0);
  RampingValue:=v;
 end;
 //
 // Get list of available Volt/Amper ranges, see ZUP manual.
 // SV  - Setting Volt                SA  - Setting Amper
 // OP  - Overvoltage Protection   UP  - Undervoltage Protection
 // OPL - OP lower  range          OPH - OP Higher range
 //
 function ListRangeSV:String;
 begin
  ListRangeSV:='6.000 10.000 20.000 36.00 60.00 80.00 120.00';
 end;
 function ListRangeOPL:String;
 begin
  ListRangeOPL:='0.20 00.5 01.0 01.8 03.0 04.0 006.0';
 end;
 function ListRangeOPH:String;
 begin
  ListRangeOPH:='7.50 13.0 24.0 40.0 66.0 88.0 132.0';
 end;
 function ListRangeUP:String;
 begin
  ListRangeUP:='5.98 9.97 19.9 35.9 59.8 79.8 119.8';
 end;
 function ListRangeSA:String;
 begin
  ListRangeSA:='1.8000 2.5000 3.500 3.600 5.000 6.000 7.000 '
  +'10.000 12.000 14.000 20.000 24.000 33.00 40.00 66.00 80.00 132.00';
 end;
 //
 // Get range by sample: 60.00 -> 60
 //
 function GetRangeBySample(Sample:String):Real;
 begin
  GetRangeBySample:=rVal(Sample);
 end;
 //
 // Get format by sample: 60.00 -> %5.2f
 //
 function GetFormatBySample(Sample:String):String;
 var w,d:Integer;
 begin
  w:=Length(Sample); d:=w-Pos('.',Sample);
  GetFormatBySample:='%'+Str(w)+'.'+Str(d)+'f';
 end;
 //
 // Print list of ranges with formats.
 //
 procedure PrintRangeList(prefix,list:String);
 var i:Integer;
 begin
  for i:=1 to WordCount(list) do
  Success(prefix+StrFmt('[%d] = ',i)
                +StrFmt(' %6s',ExtractWord(i,list))
                +StrFmt(' %6g',GetRangeBySample(ExtractWord(i,list)))
                +StrFmt(' %6s',GetFormatBySample(ExtractWord(i,list))));
 end;
 //
 // Find index of arg in list of available ranges.
 //
 function FindRangeIndex(arg,ListRanges:String):Integer;
 var i,n:Integer; r:Real;
 begin
  n:=0;  r:=rVal(Trim(arg));  if not IsNaN(r) then
  for i:=1 to WordCount(ListRanges) do if n=0 then
  if (r=rVal(ExtractWord(i,ListRanges))) then n:=i;
  FindRangeIndex:=n;
 end;
 //
 // Encode/Decode Volt/Amper Range Index by ModelCode.
 //
 function ModelCodeFactor:Integer;
 begin
  ModelCodeFactor:=256;
 end;
 function EncodeModelRanges(VolRange,CurRange:Integer):Integer;
 begin
  EncodeModelRanges:=VolRange+CurRange*ModelCodeFactor;
 end;
 function DecodeVolRange(ModelCode:Integer):Integer;
 begin
  DecodeVolRange:=ModelCode mod ModelCodeFactor;
 end;
 function DecodeCurRange(ModelCode:Integer):Integer;
 begin
  DecodeCurRange:=ModelCode div ModelCodeFactor;
 end;
 function CheckModelCode(ModelCode:Integer):Integer;
 begin
  if (ModelCode<=0) then ModelCode:=0 else
  if (DecodeVolRange(ModelCode)=0) then ModelCode:=0 else
  if (DecodeCurRange(ModelCode)=0) then ModelCode:=0;
  CheckModelCode:=ModelCode;
 end;
 //
 // Get model Volt/Amper range by ModelCode.
 //
 function ModelRangeSV(ModelCode:Integer):Real;
 begin
  ModelRangeSV:=GetRangeBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeSV));
 end;
 function ModelRangeOPL(ModelCode:Integer):Real;
 begin
  ModelRangeOPL:=GetRangeBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeOPL));
 end;
 function ModelRangeOPH(ModelCode:Integer):Real;
 begin
  ModelRangeOPH:=GetRangeBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeOPH));
 end;
 function ModelRangeUP(ModelCode:Integer):Real;
 begin
  ModelRangeUP:=GetRangeBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeUP));
 end;
 function ModelRangeSA(ModelCode:Integer):Real;
 begin
  ModelRangeSA:=GetRangeBySample(ExtractWord(DecodeCurRange(ModelCode),ListRangeSA));
 end;
 //
 // Get model Volt/Amper format by ModelCode.
 //
 function ModelFormatSV(ModelCode:Integer):String;
 begin
  ModelFormatSV:=GetFormatBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeSV));
 end;
 function ModelFormatOP(ModelCode:Integer):String;
 begin
  ModelFormatOP:=GetFormatBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeOPH));
 end;
 function ModelFormatUP(ModelCode:Integer):String;
 begin
  ModelFormatUP:=GetFormatBySample(ExtractWord(DecodeVolRange(ModelCode),ListRangeUP));
 end;
 function ModelFormatSA(ModelCode:Integer):String;
 begin
  ModelFormatSA:=GetFormatBySample(ExtractWord(DecodeCurRange(ModelCode),ListRangeSA));
 end;
 //
 // Check & return ZUPDC ModelCode, nonzero if MDL detection was successful.
 //
 function ZUPDC_ModelCode:Integer;
 begin
  ZUPDC_ModelCode:=CheckModelCode(ZUPDC.ModelCode);
 end;
 //
 // Parse string like 'Nemic-Lambda ZUP(6V-33A)', extract volt (6) and amper (33).
 // Then find range index of Volt and Amper scale and return ModelCode.
 //
 function ZUPDC_ParseMDL(s:String; silent:Boolean):Integer;
 var ModelCode,VolRangeIndex,CurRangeIndex,pv,pa,cv,ca:Integer; rsv,rsa:Real;
  procedure TailAfter(var s:String; id:String);
  var p:Integer;
  begin
   p:=Pos(id,s); if (p>0) then s:=Copy(s,p+Length(id));
  end;
 begin
  s:=Trim(s);
  ModelCode:=0;
  VolRangeIndex:=0;
  CurRangeIndex:=0;
  TailAfter(s,'Nemic');
  TailAfter(s,'Lambda');
  TailAfter(s,'ZUP');
  pv:=Pos('V',s); cv:=0;
  pa:=Pos('A',s); ca:=0;
  if (pv>0) and (pa>0) then begin
   while Pos(StrFetch(s,pv-1),'0123456789.')>0 do begin pv:=pv-1; cv:=cv+1; end;
   while Pos(StrFetch(s,pa-1),'0123456789.')>0 do begin pa:=pa-1; ca:=ca+1; end;
   if (cv>0) and (ca>0) then begin
    VolRangeIndex:=FindRangeIndex(Trim(Copy(s,pv,cv)),ListRangeSV);
    CurRangeIndex:=FindRangeIndex(Trim(Copy(s,pa,ca)),ListRangeSA);
    ModelCode:=CheckModelCode(EncodeModelRanges(VolRangeIndex,CurRangeIndex));
   end;
  end;
  if not silent then
  if ModelCode>0 then begin
   rsv:=ModelRangeSV(ModelCode);
   rsa:=ModelRangeSA(ModelCode);
   Success('Detected MDL: '+Str(rsv)+'V-'+Str(rsa)+'A');
  end else begin
   Problem('Fail detect MDL: '+s);
  end;
  ZUPDC_ParseMDL:=ModelCode;
 end;
 //
 // Extract ZUP data by prefix (AV,SV,AA,SA,OS,AL,PS,RM,OT,FD,OP,UP)
 // Writeln(ZUPDC_ExtractData('AV5.010SV5.010AA00.00SA24.31OS00010000AL00000PS00000','SV')); -> 5.010
 //
 function ZUPDC_ExtractData(Data,Prefix:String):String;
 var p:Integer;
 begin
  p:=Pos(Prefix,Data);
  if (p=0) then Data:='' else begin
   Data:=Copy(Data,p+Length(Prefix));
   p:=1; while Pos(StrFetch(Data,p),'0123456789.-+')>0 do p:=p+1;
   Data:=Copy(Data,1,p-1);
  end;
  ZUPDC_ExtractData:=Data;
 end;
 //
 // Convert binary string to integer.
 // Writeln(ZUPDC_ValBin('001')); -> 4
 //
 function ZUPDC_ValBin(Data:String):Integer;
 var i,n,e:Integer; c:Char;
 begin
  Data:=Trim(Data); n:=0; e:=0;
  for i:=1 to Length(Data) do begin
   c:=StrFetch(Data,i);
   if (c='0') or (c='1')
   then n:=iSetBit(n,i-1,Ord(c='1'))
   else e:=e+1;
  end;
  if (e>0) then n:=0;
  ZUPDC_ValBin:=n;
 end;
 //
 // Format ZUPDC data: ZUPDC_Fmt('%6.2f',pi); -> 003.14
 // Format ZUPDC data: ZUPDC_Bin('5-bit',4);  -> 00100
 //
 function ZUPDC_Fmt(Fmt:String; Data:Real):String;
 var i,n:Integer;
 begin
  if Pos('%',Fmt)>0 then begin
   Fmt:=StrReplace(StrFmt(Fmt,Data),Dump(' '),Dump('0'),3);
  end else begin
   n:=Val(Copy(Fmt,1,1)); Fmt:='';
   for i:=0 to n-1 do Fmt:=Fmt+Str(Ord(IsBit(Data,i)));
  end;
  ZUPDC_Fmt:=Fmt;
 end;
 //
 // Find command by id (Acronym/AnsPref).
 //
 function ZUPDC_FindCmd(id:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(id)>0) then
  if UseCmdHash and (ZUPDC.CmdHash<>0) then begin
   n:=HashList_GetLink(ZUPDC.CmdHash,id);
  end else begin
   for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(id,ZUPDC.Cmd.Acronym[i])
   or IsSameText(id,ZUPDC.Cmd.AnsPref[i])
   then n:=i;
  end;
  ZUPDC_FindCmd:=n;
 end;
 //
 // Nice tooltip notifier.
 //
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  if Length(aText)>0 then begin
   ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
  end;
 end;
 //
 // Post command to local/remote server.
 //
 procedure PostCmdLocal(cmd:String);
 begin
  DevPostCmdLocal(cmd);
 end;
 procedure PostCmdRemote(cmd:String);
 begin
  Dim_GuiConsoleSend(DevName,cmd);
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс Регистров Устройства   (DCL)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting DCL');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Включить   Защиту (foldback) (FLD)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting FLDn 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Защиту (foldback) (FLD)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting FLDn 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Включить   Авто-(ре)Старт    (AST)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting ASTn 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Авто-(ре)Старт    (AST)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting ASTn 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Подключить Выход             (OUT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting OUTn 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Выход             (OUT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting OUTn 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка по Току, А           (CUR)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit CURn');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка по Напряжению, В     (VOL)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit VOLn');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Защита по переНапряжению, В  (UVP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit OVPn');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Защита по недоНапряжению, В  (UVP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit UVPn');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Печать состояния таблицы команд');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@PrintCmdTable');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 450 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  235 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 //
 // Initialize tag refreshment value.
 //
 procedure ZUPDC_FillTag(tag:Integer; InitVal:Real);
  procedure Process(var R:TTagRef);
  begin
   if (R.tag=tag) then bNul(ShouldRefresh(R.val,InitVal)>0);
  end;
 begin
  if (TypeTag(tag)>0) then begin
   Process(ZUPDC.POLL_ENABLE);
   Process(ZUPDC.POLL_RATE);
   Process(ZUPDC.ERROR_CNT);
   Process(ZUPDC.ID_NAME);
   Process(ZUPDC.ID_MDL);
   Process(ZUPDC.ID_REV);
   Process(ZUPDC.PAR_AA);
   Process(ZUPDC.PAR_AV);
   Process(ZUPDC.PAR_AW);
   Process(ZUPDC.PAR_SA);
   Process(ZUPDC.PAR_SV);
   Process(ZUPDC.PAR_OP);
   Process(ZUPDC.PAR_UP);
   Process(ZUPDC.PAR_OS);
   Process(ZUPDC.PAR_AL);
   Process(ZUPDC.PAR_PS);
   Process(ZUPDC.SET_FD);
   Process(ZUPDC.SET_AS);
   Process(ZUPDC.SET_OT);
   Process(ZUPDC.RATE_A);
   Process(ZUPDC.RATE_V);
   Process(ZUPDC.RAMP_A);
   Process(ZUPDC.RAMP_V);
  end;
 end;
 //
 // Initialize tags.
 //
 procedure ZUPDC_FillTags(InitVal:Real);
 begin
  ZUPDC.POLL_ENABLE.val:=InitVal;
  ZUPDC.POLL_RATE.val:=InitVal;
  ZUPDC.ERROR_CNT.val:=InitVal;
  ZUPDC.ID_NAME.val:=InitVal;
  ZUPDC.ID_MDL.val:=InitVal;
  ZUPDC.ID_REV.val:=InitVal;
  ZUPDC.PAR_AA.val:=InitVal;
  ZUPDC.PAR_AV.val:=InitVal;
  ZUPDC.PAR_AW.val:=InitVal;
  ZUPDC.PAR_SA.val:=InitVal;
  ZUPDC.PAR_SV.val:=InitVal;
  ZUPDC.PAR_OP.val:=InitVal;
  ZUPDC.PAR_UP.val:=InitVal;
  ZUPDC.PAR_OS.val:=InitVal;
  ZUPDC.PAR_AL.val:=InitVal;
  ZUPDC.PAR_PS.val:=InitVal;
  ZUPDC.SET_FD.val:=InitVal;
  ZUPDC.SET_AS.val:=InitVal;
  ZUPDC.SET_OT.val:=InitVal;
  ZUPDC.RATE_A.val:=InitVal;
  ZUPDC.RATE_V.val:=InitVal;
  ZUPDC.RAMP_A.val:=InitVal;
  ZUPDC.RAMP_V.val:=InitVal;
 end;
 procedure ZUPDC_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
   InitTag(ZUPDC.POLL_ENABLE.tag, Prefix+'.POLL_ENABLE',    1); 
   InitTag(ZUPDC.POLL_RATE.tag,   Prefix+'.POLL_RATE',      2);
   InitTag(ZUPDC.ERROR_CNT.tag,   Prefix+'.ERROR_CNT',      2);
   InitTag(ZUPDC.ID_NAME.tag,     Prefix+'.ID_NAME',        3);
   InitTag(ZUPDC.ID_MDL.tag,      Prefix+'.ID_MDL',         3);
   InitTag(ZUPDC.ID_REV.tag,      Prefix+'.ID_REV',         3);
   InitTag(ZUPDC.PAR_AA.tag,      Prefix+'.PAR_AA',         2);
   InitTag(ZUPDC.PAR_AV.tag,      Prefix+'.PAR_AV',         2);
   InitTag(ZUPDC.PAR_AW.tag,      Prefix+'.PAR_AW',         2);
   InitTag(ZUPDC.PAR_SA.tag,      Prefix+'.PAR_SA',         2);
   InitTag(ZUPDC.PAR_SV.tag,      Prefix+'.PAR_SV',         2);
   InitTag(ZUPDC.PAR_OP.tag,      Prefix+'.PAR_OP',         2);
   InitTag(ZUPDC.PAR_UP.tag,      Prefix+'.PAR_UP',         2);
   InitTag(ZUPDC.PAR_OS.tag,      Prefix+'.PAR_OS',         1);
   InitTag(ZUPDC.PAR_AL.tag,      Prefix+'.PAR_AL',         1);
   InitTag(ZUPDC.PAR_PS.tag,      Prefix+'.PAR_PS',         1);
   InitTag(ZUPDC.SET_FD.tag,      Prefix+'.SET_FD',         1);
   InitTag(ZUPDC.SET_AS.tag,      Prefix+'.SET_AS',         1);
   InitTag(ZUPDC.SET_OT.tag,      Prefix+'.SET_OT',         1);
   InitTag(ZUPDC.RATE_A.tag,      Prefix+'.RATE_A',         2);
   InitTag(ZUPDC.RATE_V.tag,      Prefix+'.RATE_V',         2);
   InitTag(ZUPDC.RAMP_A.tag,      Prefix+'.RAMP_A',         1);
   InitTag(ZUPDC.RAMP_V.tag,      Prefix+'.RAMP_V',         1);
   ZUPDC.SET_FD_GUI.tag:=ZUPDC.SET_FD.tag; ZUPDC.SET_FD_GUI.val:=iGetTag(ZUPDC.SET_FD.tag);
   ZUPDC.SET_AS_GUI.tag:=ZUPDC.SET_AS.tag; ZUPDC.SET_AS_GUI.val:=iGetTag(ZUPDC.SET_AS.tag);
   ZUPDC.SET_OT_GUI.tag:=ZUPDC.SET_OT.tag; ZUPDC.SET_OT_GUI.val:=iGetTag(ZUPDC.SET_OT.tag);
   ZUPDC_FillTags(-MaxReal);
  end;
 end;
 //
 // Update DIM tag in server mode.
 //
 procedure UpdateDimTag(tag:Integer);
 var v:Real;
 begin
  if DIM_IsServerMode then
  if (TypeTag(tag)>0) then begin
   v:=GetStampOfTag(tag,_NaN);
   if not IsNan(v) then ZUPDC_FillTag(tag,v);
   DIM_UpdateTag(tag,'');
  end;
 end;
 //
 // Check if COM port opened.
 //
 function IsComPortOpened:Boolean;
 begin
  IsComPortOpened:=(ComSpace>=0);
 end;
 //
 // Check command number is valid. 
 //
 function IsValidCmdNum(n:Integer):Boolean;
 begin
  IsValidCmdNum:=(n>=1) and (n<=MaxCmdNum);
 end;
 //
 // Check if command (cmd) is query (?,!), i.e. should wait reply.
 //
 function IsQueryCmd(cmd:String):Boolean;
 begin
  IsQueryCmd:=(Pos('?',cmd)+Pos('!',cmd)>0);
 end;
 function IsQueryCmdNum(n:Integer):Boolean;
 begin
  if IsValidCmdNum(n)
  then IsQueryCmdNum:=IsQueryCmd(ZUPDC.Cmd.Acronym[n])
  else IsQueryCmdNum:=false;
 end;
 //
 // Enable/disable command n.
 //
 procedure EnableCmdNum(n:Integer; Enabled:Boolean);
 begin
  if IsValidCmdNum(n) then ZUPDC.Cmd.Enabled[n]:=Enabled;
 end;
 //
 // Check if ZUPDC should use ADRn, i.e. has ComInterface RS422 or RS485.
 //
 function ZUPDC_UsesADRn:Boolean;
 begin
  ZUPDC_UsesADRn:=(abs(ZUPDC.Com.Interface)>RS_232);
 end;
 //
 // Check if ZUPDC polling enabled, with warning or not.
 //
 function ZUPDC_CheckPolling(Warn:Integer):Boolean;
 var flag:Boolean; msg:String;
 begin
  msg:='';
  flag:=iGetTag(ZUPDC.POLL_ENABLE.tag)>0;
  if (Warn<>0) and not flag then begin
   msg:=StrFmt('Для изменения параметров %s надо включить Опрос.',sGetTag(ZUPDC.ID_NAME.tag));
   if iAnd(Warn,1)>0 then NiceNotify(msg,15000);
   if iAnd(Warn,2)>0 then UnixWBox('Предупреждение',msg,600,20,15);
   if iAnd(Warn,4)>0 then Cron('@run /sw7 unix messagebox "Предупреждение" "'+msg+'" -t 15');
   if iAnd(Warn,8)>0 then Warning(msg);
  end;
  ZUPDC_CheckPolling:=flag;
  msg:='';
 end;
 //
 // Clear commands.
 //
 procedure ZUPDC_Clear_Cmd;
 var i:Integer;
 begin
  for i:=1 to MaxCmdNum do begin
   ZUPDC.Cmd.Acronym[i]:='';
   ZUPDC.Cmd.AnsPref[i]:='';
   ZUPDC.Cmd.DataFmt[i]:='';
   ZUPDC.Cmd.Comment[i]:='';
   ZUPDC.Cmd.Enabled[i]:=False;
   ZUPDC.Cmd.OpData[i]:=_NAN;
   ZUPDC.Cmd.OpBuff[i]:=_NAN;
   ZUPDC.Cmd.SimDat[i]:=_NAN;
   ZUPDC.Cmd.AoNum[i]:=-1;
   ZUPDC.Cmd.Flags[i]:=0;
   ZUPDC.Cmd.Tag[i]:=0;
   ZUPDC.Cmd.Min[i]:=0;
   ZUPDC.Cmd.Max[i]:=0;
  end;
 end;
 //
 // Initialize commands.
 //
 procedure ZUPDC_Init_Cmd;
  procedure AddHash(n:Integer; key:String);
  begin
   key:=Trim(key);
   if Length(key)>0 then
   if IsValidCmdNum(n) then
   if ZUPDC.CmdHash<>0 then
   bNul(HashList_SetLink(ZUPDC.CmdHash,key,n));
  end;
  procedure Add(n,AoNum:Integer; Info:String);
  begin
   if IsValidCmdNum(n) then begin
    ZUPDC.Cmd.Acronym[n]:=ExtractWord(1,Info);
    ZUPDC.Cmd.AnsPref[n]:=StrReplace(ExtractWord(2,Info),Dump('*'),'',3);
    ZUPDC.Cmd.DataFmt[n]:=StrReplace(ExtractWord(3,Info),Dump('*'),'',3);
    ZUPDC.Cmd.Comment[n]:=ExtractWord(4,Info);
    ZUPDC.Cmd.Enabled[n]:=Val(ExtractWord(5,Info))>0;
    ZUPDC.Cmd.OpData[n]:=rVal(ExtractWord(6,Info));
    ZUPDC.Cmd.SimDat[n]:=rVal(ExtractWord(7,Info));
    ZUPDC.Cmd.AoNum[n]:=AoNum;
    ZUPDC.Cmd.Tag[n]:=FindTag(CrvName(RefAo(AoNum)));
    ZUPDC.Cmd.Min[n]:=rVal(ExtractWord(8,Info));
    ZUPDC.Cmd.Max[n]:=rVal(ExtractWord(9,Info));
    AddHash(n,ZUPDC.Cmd.Acronym[n]);
    AddHash(n,ZUPDC.Cmd.AnsPref[n]);
   end;
  end;
 begin
  ZUPDC.Cmd.Num:=1;
  //  Command  AoNum      Acron Ans Fmt   Comment          En OpData SimDat  Min Max
  Add(cm_ADRn, -1,        'ADRn **  %2.0f Set_Address       1      1      1    1  31');
  Add(cm_DCL,  -1,        'DCL  **  ***** Device_Clear      0      0      0    0   0');
  Add(cm_RMTn, -1,        'RMTn **  %1.0f Set_Remote_Mode   0      1      1    0   2');
  Add(cm_RMTa, -1,        'RMT? RM  ***** Get_Remote_Mode   0      1      1    1   2');
  Add(cm_MDLa, -1,        'MDL? **  ***** Get_Model_Ident   1      0      0    0   0');
  Add(cm_REVa, -1,        'REV? **  ***** Get_Software_Rev  1      0      0    0   0');
  Add(cm_VOLn, ao_PAR_SV, 'VOLn **  %6.2f Set_Setting_Volt  0      0      0    0 120');
  Add(cm_VOLs, ao_PAR_SV, 'VOL! SV  ***** Get_Setting_Volt  0      0    6.0    0 120');
  Add(cm_VOLa, ao_PAR_AV, 'VOL? AV  ***** Get_Actual_Volt   0      0    5.9    0 120');
  Add(cm_CURn, ao_PAR_SA, 'CURn **  %6.2f Set_Setting_Amper 0      0      0    0 132');
  Add(cm_CURs, ao_PAR_SA, 'CUR! SA  ***** Get_Setting_Amper 0      0   33.0    0 132');
  Add(cm_CURa, ao_PAR_AA, 'CUR? AA  ***** Get_Actual_Amper  0      0   32.9    0 132');
  Add(cm_OUTn, -1,        'OUTn **  %1.0f Set_Output_State  0      0      0    0   1');
  Add(cm_OUTa, -1,        'OUT? OT  ***** Get_Output_State  0      0      0    0   1');
  Add(cm_FLDn, -1,        'FLDn **  %1.0f Set_Foldback_Prot 0      0      0    0   2');
  Add(cm_FLDa, -1,        'FLD? FD  ***** Get_Foldback_Prot 0      0      0    0   1');
  Add(cm_OVPn, ao_PAR_OP, 'OVPn **  %4.1f Set_OverVolt_Prot 0      0      0    0 132');
  Add(cm_OVPa, ao_PAR_OP, 'OVP? OP  ***** Get_OverVolt_Prot 1      0      0    0 132');
  Add(cm_UVPn, ao_PAR_UP, 'UVPn **  %4.1f Set_UnderVolt_Pro 0      0      0    0 120');
  Add(cm_UVPa, ao_PAR_UP, 'UVP? UP  ***** Get_UnderVolt_Pro 1      0      0    0 120');
  Add(cm_ASTn, -1,        'ASTn **  %1.0f Set_AutoStartMode 0      0      0    0   1');
  Add(cm_ASTa, -1,        'AST? AS  ***** Get_AutoStartMode 0      0      0    0   1');
  Add(cm_STAa, ao_PAR_OS, 'STA? OS  8-bit Get_Oper-n_Status 0      0      0    0 255');
  Add(cm_ALMa, ao_PAR_AL, 'ALM? AL  5-bit Get_Alarm_Status  0      0      0    0  31');
  Add(cm_STPa, ao_PAR_PS, 'STP? PS  5-bit Get_ProgramStatus 0      0      0    0  31');
  Add(cm_STTa, -1,        'STT? **  ***** Get_Status_Total  1      0      0    0   0');
  ZUPDC.Cmd.OpData[cm_ADRn]:=ZUPDC.Com.Addr;
 end;
 //
 // Print command table
 //
 procedure PrintCmdTable;
 var n:Integer;
 begin
  Success('Command table:');
  Success(StrFmt('%-2s ','Cm')
         +StrFmt('%-2s ','Ao')
         +StrFmt('%-4s ','Acro')
         +StrFmt('%-3s ','Ans')
         +StrFmt('%-5s ','Fmt')
         +StrFmt('%-17s ','Comment')
         +StrFmt('%-2s ','En')
         +StrFmt('%6s ','OpData')
         +StrFmt('%6s ','SimDat')
         +StrFmt('%6s ','Min')
         +StrFmt('%6s ','Max')
         );
  for n:=1 to MaxCmdNum do
  Success(StrFmt('%-2d ',n)
         +StrFmt('%-2d ',ZUPDC.Cmd.AoNum[n])
         +StrFmt('%-4s ',ZUPDC.Cmd.Acronym[n])
         +StrFmt('%-3s ',ZUPDC.Cmd.AnsPref[n])
         +StrFmt('%-5s ',ZUPDC.Cmd.DataFmt[n])
         +StrFmt('%-17.17s ',ZUPDC.Cmd.Comment[n])
         +StrFmt('%-2d ',Ord(ZUPDC.Cmd.Enabled[n]))
         +StrFmt('%6g ',ZUPDC.Cmd.OpData[n])
         +StrFmt('%6g ',ZUPDC.Cmd.SimDat[n])
         +StrFmt('%6g ',ZUPDC.Cmd.Min[n])
         +StrFmt('%6g ',ZUPDC.Cmd.Max[n])
         );
 end;
 //
 // Update ranges/formats by ModelCode.
 //
 function ZUPDC_UpdateModel(ModelCode:Integer):Integer;
 begin
  ModelCode:=CheckModelCode(ModelCode);
  if ModelCode>0 then begin
   ZUPDC.Cmd.DataFmt[cm_VOLn]:=ModelFormatSV(ModelCode);
   ZUPDC.Cmd.Max[cm_VOLn]:=ModelRangeSV(ModelCode);
   ZUPDC.Cmd.Max[cm_VOLs]:=ModelRangeSV(ModelCode);
   ZUPDC.Cmd.Max[cm_VOLa]:=ModelRangeSV(ModelCode);
   ZUPDC.Cmd.DataFmt[cm_CURn]:=ModelFormatSA(ModelCode);
   ZUPDC.Cmd.Max[cm_CURn]:=ModelRangeSA(ModelCode);
   ZUPDC.Cmd.Max[cm_CURs]:=ModelRangeSA(ModelCode);
   ZUPDC.Cmd.Max[cm_CURa]:=ModelRangeSA(ModelCode);
   ZUPDC.Cmd.DataFmt[cm_OVPn]:=ModelFormatOP(ModelCode);
   ZUPDC.Cmd.Max[cm_OVPn]:=ModelRangeOPH(ModelCode);
   ZUPDC.Cmd.Max[cm_OVPa]:=ModelRangeOPH(ModelCode);
   ZUPDC.Cmd.Min[cm_OVPn]:=ModelRangeOPL(ModelCode);
   ZUPDC.Cmd.Min[cm_OVPa]:=ModelRangeOPL(ModelCode);
   ZUPDC.Cmd.DataFmt[cm_UVPn]:=ModelFormatUP(ModelCode);
   ZUPDC.Cmd.Max[cm_UVPn]:=ModelRangeUP(ModelCode);
   ZUPDC.Cmd.Max[cm_UVPa]:=ModelRangeUP(ModelCode);
  end;
  ZUPDC_UpdateModel:=ModelCode;
 end;
 //
 // Main command cycle.
 //
 procedure ZUPDC_CMD_CYCLE;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if IsValidCmdNum(Num)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if ZUPDC.Cmd.Enabled[Num]
    then i:=MaxCmdNum
    else i:=i+1;
   end; 
   NextCmdNum:=Num;
  end;
  procedure ClearRequest(Num,Status:Integer);
  begin
   ZUPDC.Com.Req:='';
   ZUPDC.Com.Ans:='';
   ZUPDC.Com.Buf:='';
   ZUPDC.Com.ReqTime:=0;
   ZUPDC.Com.AnsTime:=0;
   ZUPDC.Com.Status:=Status;
   ZUPDC.Cmd.Num:=ValidateCmdNum(Num);
  end;
  function ReqCmd(n:Integer):String;
  var s:String; OpData:Real;
  begin
   s:='';
   if IsValidCmdNum(n) then begin
    s:=ZUPDC.Cmd.Acronym[n];
    if (Length(s)>0) and not IsQueryCmd(s) then begin
     OpData:=ZUPDC.Cmd.OpData[n];
     if IsNan(OpData) then s:='' else
     s:=Copy(s,1,3)+ZUPDC_Fmt(ZUPDC.Cmd.DataFmt[n],OpData);
    end;
    if (Length(s)>0) then s:=StrFmt(':%s;',s);
   end;
   ReqCmd:=s;
   s:='';
  end;
  procedure PrepareRequest(n:Integer);
  begin
   if IsValidCmdNum(n) then begin
    ZUPDC.Com.Req:=ReqCmd(n);
    if (Length(ZUPDC.Com.Req)>0) then begin
     if (n<>cm_ADRn) and ZUPDC_UsesADRn then ZUPDC.Com.Req:=ReqCmd(cm_ADRn)+ZUPDC.Com.Req;
     if EnforceCRLF then ZUPDC.Com.Req:=ZUPDC.Com.Req+CRLF;
    end;
   end;
  end;
  procedure UpdateMDL(data:String);
  begin
   if not IsEmptyStr(data) then begin
    ZUPDC.ModelCode:=ZUPDC_UpdateModel(ZUPDC_ParseMDL(data,iGetTag(ZUPDC.POLL_ENABLE.tag)=0));
    if (ZUPDC_ModelCode=0) then data:=si_OFFLINE;
    if IsEmptyStr(data) then data:=si_OFFLINE;
    bNul(sSetTag(ZUPDC.ID_MDL.tag,data));
    if ShouldRefresh(ZUPDC.ID_MDL.val,GetStampOfTag(ZUPDC.ID_MDL.tag,_NaN))>0
    then UpdateDimTag(ZUPDC.ID_MDL.tag);
   end;
  end;
  procedure UpdateCmdValue(n:Integer; value:Real);
  var aonum,tag:Integer;
  begin
   if IsValidCmdNum(n) and not IsNan(value) and not IsInf(value) then begin
    tag:=ZUPDC.Cmd.Tag[n]; aonum:=ZUPDC.Cmd.AoNum[n];
    if TypeTag(tag)=1 then begin
      bNul(iSetTag(tag,Round(value)));
      UpdateDimTag(tag);
    end;
    if TypeTag(tag)=2 then begin
     bNul(rSetTag(tag,value));
     UpdateDimTag(tag);
    end;
    if aonum>=0 then UpdateAo(aonum,time,value);
    ZUPDC.Cmd.OpData[n]:=value;
   end;
  end;  
  procedure Update_PAR_AW;
  var volt,amper,watt:Real;
  begin
   volt:=ZUPDC.Cmd.OpData[cm_VOLa];
   amper:=ZUPDC.Cmd.OpData[cm_CURa];
   watt:=volt*amper;
   bNul(rSetTag(ZUPDC.PAR_AW.tag,watt));
   UpdateDimTag(ZUPDC.PAR_AW.tag);
   UpdateAo(ao_PAR_AW,time,watt);
  end;
  procedure Update_PAR_BIT(n:Integer; bit:Integer; value:Real);
  begin
  if IsValidCmdNum(n) then
   UpdateCmdValue(n,iSetBit(Round(ZUPDC.Cmd.OpData[n]),bit,Round(value)));
  end;
  procedure UpdateData(n:Integer; data:String; Bin:Boolean);
  var value:Real;
  begin
   if Length(data)>0 then
   if IsValidCmdNum(n) then begin
    data:=ZUPDC_ExtractData(data,ZUPDC.Cmd.AnsPref[n]);
    if Length(data)>0 then begin
     if Bin then value:=ZUPDC_ValBin(data) else value:=rVal(data);
     if not IsNan(value) and not IsInf(value) then begin
      UpdateCmdValue(n,value);
      if (n=cm_CURa) then Update_PAR_AW; // Update Watts
      if (n=cm_FLDa) then Update_PAR_BIT(cm_STAa,1,value);
      if (n=cm_ASTa) then Update_PAR_BIT(cm_STAa,2,value);
      if (n=cm_OUTa) then Update_PAR_BIT(cm_STAa,3,value);
     end;
    end;
   end;
  end;
  procedure DecodeData(n:Integer; data:String);
  var value:Real; apl:Integer;
  begin
   if IsValidCmdNum(n) then begin
    data:=Trim(data);
    if Length(data)>0 then begin
     if (n=cm_MDLa) then begin
      UpdateMDL(data);
      if ZUPDC_ModelCode>0 then ZUPDC.Cmd.Enabled[n]:=false;
     end else
     if (n=cm_REVa) then begin
      bNul(sSetTag(ZUPDC.ID_REV.tag,data));
      UpdateDimTag(ZUPDC.ID_REV.tag);
      ZUPDC.Cmd.Enabled[n]:=false;
     end else begin
      UpdateData(cm_CURa,data,false);
      UpdateData(cm_VOLa,data,false);
      UpdateData(cm_VOLs,data,false);
      UpdateData(cm_CURs,data,false);
      UpdateData(cm_OVPa,data,false);
      UpdateData(cm_UVPa,data,false);
      UpdateData(cm_FLDa,data,false);
      UpdateData(cm_ASTa,data,false);
      UpdateData(cm_OUTa,data,false);
      UpdateData(cm_STAa,data,true);
      UpdateData(cm_ALMa,data,true);
      UpdateData(cm_STPa,data,true);
     end;
    end;
   end;
  end;
  function HandleRequest(n:Integer; Ans:String):Boolean;
  var Handled:Boolean;
  begin
   Handled:=false;
   if IsValidCmdNum(n) and (Length(Ans)>0) then begin
    ViewImp('COM < '+Ans);
    DecodeData(n,Ans);
    Handled:=True;
   end;
   HandleRequest:=Handled;
  end;
  procedure PrintDetails;
  begin
   if iAnd(DebugFlags,dfDetails)>0 then Details('Run '+Str(RunCount)
    +', Cmd '+ZUPDC.Cmd.Acronym[ZUPDC.Cmd.Num]
    +', Enb '+Str(Ord(ZUPDC.Cmd.Enabled[ZUPDC.Cmd.Num]))
    +', Sta '+Str(ZUPDC.Com.Status)
    +', Req '+Trim(ZUPDC.Com.Req)
    +', Ans '+Trim(ZUPDC.Com.Ans)
    +', Buf '+Trim(ZUPDC.Com.Buf)
    );
  end;
  procedure RampingHandler;
  var v:Real;
  begin
   if mSecNow>ZUPDC.RampingTime+ZUPDC.Com.PollPeriod then begin
    if RampingGoing(ZUPDC.RampingA) then begin
     v:=RampingValue(ZUPDC.RampingA);
     DevPostCmdLocal('@Setting '+ZUPDC.Cmd.Acronym[cm_CURn]+' '+Str(v));
     if RampingComplete(ZUPDC.RampingA) then RampingClear(ZUPDC.RampingA);
    end;
    if RampingGoing(ZUPDC.RampingV) then begin
     v:=RampingValue(ZUPDC.RampingV);
     DevPostCmdLocal('@Setting '+ZUPDC.Cmd.Acronym[cm_VOLn]+' '+Str(v));
     if RampingComplete(ZUPDC.RampingV) then RampingClear(ZUPDC.RampingV);
    end;
    bNul(iSetTag(ZUPDC.RAMP_A.tag,Ord(RampingGoing(ZUPDC.RampingA))));
    bNul(iSetTag(ZUPDC.RAMP_V.tag,Ord(RampingGoing(ZUPDC.RampingV))));
    ZUPDC.RampingTime:=mSecNow;
   end;
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsNan(ZUPDC.Cmd.OpBuff[n]) then begin
    if not IsQueryCmdNum(n) then begin
     ZUPDC.Cmd.OpData[n]:=ZUPDC.Cmd.OpBuff[n];
     EnableCmdNum(n,true);
    end;
    ZUPDC.Cmd.OpBuff[n]:=_NAN;
   end;
  end;
  procedure EnableMDL;
  begin
   EnableCmdNum(cm_ADRn,true);
   EnableCmdNum(cm_MDLa,true);
   EnableCmdNum(cm_REVa,true);
  end;
  procedure ExecuteIdleActions;
  begin
   UpdateOpData;
   RampingHandler;
   if (ZUPDC_ModelCode=0) then EnableMDL;
  end;
  procedure DimRefreshTag(var R:TTagRef);
  var tag:Integer; v:Real;
  begin
   tag:=R.tag; v:=GetStampOfTag(tag,_NaN);
   if not IsNaN(v) then if ShouldRefresh(R.val,v)>0 then UpdateDimTag(tag);
  end;
 begin
  //
  // COM port communication handler
  //
  if not DIM_IsClientMode then begin
   if IsComPortOpened then begin
    if iGetTag(ZUPDC.POLL_ENABLE.tag)>0 then begin
     ZUPDC.Cmd.Num:=ValidateCmdNum(ZUPDC.Cmd.Num);
     if iAnd(DebugFlags,dfDetails)>0 then PrintDetails;
     if ZUPDC.Com.Status=st_NoReq then begin
      ExecuteIdleActions;
      ClearRequest(ZUPDC.Cmd.Num,st_NoReq);
      if mSecNow>=ZUPDC.Com.LastPoll+ZUPDC.Com.PollPeriod then begin
       if ZUPDC.Cmd.Enabled[ZUPDC.Cmd.Num] then PrepareRequest(ZUPDC.Cmd.Num);
       if Length(ZUPDC.Com.Req)>0 then begin
        PurgeComPort;
        if ComWrite(ZUPDC.Com.Req) then begin
         ViewExp('COM > '+Trim(ZUPDC.Com.Req));
         if IsQueryCmdNum(ZUPDC.Cmd.Num) then begin
          ZUPDC.Com.Status:=st_WaitAns;
          ZUPDC.Com.ReqTime:=mSecNow;
         end else begin
          ZUPDC.Com.Status:=st_WaitGap;
          ZUPDC.Com.ReqTime:=mSecNow;
          ZUPDC.Com.AnsTime:=mSecNow;
         end;
        end else begin
         Trouble('Could not send request '+Trim(ZUPDC.Com.Req));
         ClearRequest(NextCmdNum(ZUPDC.Cmd.Num),st_NoReq);
        end;
        if ZUPDC.Com.Status=st_WaitAns
        then ZUPDC.Com.LastPoll:=mSecNow;
       end else ClearRequest(NextCmdNum(ZUPDC.Cmd.Num),st_NoReq);
      end;
     end else
     if ZUPDC.Com.Status=st_WaitAns then begin
      if Com_Readln(ZUPDC.Com.Ans,ZUPDC.Com.Buf) then begin
       bNul(HandleRequest(ZUPDC.Cmd.Num,Trim(ZUPDC.Com.Ans)));
       ZUPDC.Com.PollRate:=ZUPDC.Com.PollRate+1;
       ZUPDC.Com.Status:=st_WaitGap;
       ZUPDC.Com.AnsTime:=mSecNow;
      end else
      if mSecNow>ZUPDC.Com.ReqTime+ZUPDC.Com.TimeOut then begin
       ZUPDC.Com.Status:=st_TimeOut;
      end;
     end;
     if ZUPDC.Com.Status=st_WaitGap then begin
      if mSecNow>=ZUPDC.Com.AnsTime+ZUPDC.Com.TimeGap then begin
       if not IsQueryCmdNum(ZUPDC.Cmd.Num) then EnableCmdNum(ZUPDC.Cmd.Num,false);
       ClearRequest(NextCmdNum(ZUPDC.Cmd.Num),st_NoReq);
      end;
     end;
     if ZUPDC.Com.Status=st_TimeOut then begin
      if ZUPDC.Cmd.Num=cm_MDLa then UpdateMDL(si_OFFLINE);
      Failure(ZUPDC.ecTimeOut,'TimeOut on command '+ZUPDC.Cmd.Acronym[ZUPDC.Cmd.Num]+' request '+Trim(ZUPDC.Com.Req));
      ClearRequest(NextCmdNum(ZUPDC.Cmd.Num),st_NoReq);
      EnableMDL;
     end;
    end else begin
     ClearRequest(NextCmdNum(0),st_NoReq);
     UpdateMDL(si_OFFLINE);
     EnableMDL;
    end;
   end;
  end;
  //
  // Update DIM services
  //
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then ZUPDC_FillTags(-MaxReal);
   DimRefreshTag(ZUPDC.POLL_ENABLE);
   DimRefreshTag(ZUPDC.POLL_RATE);
   DimRefreshTag(ZUPDC.ERROR_CNT);
   DimRefreshTag(ZUPDC.ID_MDL);
   DimRefreshTag(ZUPDC.ID_REV);
   DimRefreshTag(ZUPDC.PAR_AA);
   DimRefreshTag(ZUPDC.PAR_AV);
   DimRefreshTag(ZUPDC.PAR_AW);
   DimRefreshTag(ZUPDC.PAR_SA);
   DimRefreshTag(ZUPDC.PAR_SV);
   DimRefreshTag(ZUPDC.PAR_OP);
   DimRefreshTag(ZUPDC.PAR_UP);
   DimRefreshTag(ZUPDC.PAR_OS);
   DimRefreshTag(ZUPDC.PAR_AL);
   DimRefreshTag(ZUPDC.PAR_PS);
   DimRefreshTag(ZUPDC.RATE_A);
   DimRefreshTag(ZUPDC.RATE_V);
   DimRefreshTag(ZUPDC.RAMP_A);
   DimRefreshTag(ZUPDC.RAMP_V);
   DimRefreshTag(ZUPDC.SET_FD);
   DimRefreshTag(ZUPDC.SET_AS);
   DimRefreshTag(ZUPDC.SET_OT);
  end;
 end;
 //
 // Xor bit on click (local version).
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // Xor bit on click (remote version).
 //
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DIM_GuiClickSend(DIM_GuiClickBuff+'NewValue='+Str(iXor(iGetTag(tag),XorMask)));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // GUI Handler to process user input...
 //
 procedure ZUPDC_GUI_POLL;
 var ClickCurve:Integer; s,slb:String;
  procedure ClickEditZupdcCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if ClickTag=ZUPDC.Cmd.Tag[n] then begin
    DevPostCmdLocal('@Edit '+ZUPDC.Cmd.Acronym[n]+' '+mime_encode(slb));
    bNul(Voice(snd_Click));
   end;
  end;
  procedure DevSendZupdcCmd(n:Integer; prefix,data:String; min,max:Real);
  var v:Real;
  begin
   if IsValidCmdNum(n) then
   if not IsEmptyStr(data) then
   if TypeTag(ZUPDC.Cmd.Tag[n])=1 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmdLocal(prefix+ZUPDC.Cmd.Acronym[n]+' '+Str(Round(v)));
   end else
   if TypeTag(ZUPDC.Cmd.Tag[n])=2 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmdLocal(prefix+ZUPDC.Cmd.Acronym[n]+' '+Str(v));
   end else
   if TypeTag(ZUPDC.Cmd.Tag[n])=3 then begin
    DevPostCmdLocal(prefix+ZUPDC.Cmd.Acronym[n]+' '+Trim(data));
   end;
  end;
  procedure CheckEditZupdcCmd(n:Integer; prefix:String);
  var s:String;
  begin
   s:='';
   if IsValidCmdNum(n) then
   if CheckEditTag(ZUPDC.Cmd.Tag[n],s) then begin
    if ((n=cm_VOLn) and (rGetTag(ZUPDC.RATE_V.tag)>0))
    or ((n=cm_CURn) and (rGetTag(ZUPDC.RATE_A.tag)>0))
    then DevSendZupdcCmd(n,prefix+'@Ramping ',s,ZUPDC.Cmd.Min[n],ZUPDC.Cmd.Max[n])
    else DevSendZupdcCmd(n,prefix+'@Setting ',s,ZUPDC.Cmd.Min[n],ZUPDC.Cmd.Max[n]);
    EditReset;
   end;
   s:='';
  end;
  procedure RefreshSetting(n:Integer; var S:TTagRef);
  begin
   if IsValidCmdNum(n) then begin
    if TypeTag(S.tag)=1 then
    if ShouldRefresh(S.val,iGetTag(S.tag))>0 then
    DevPostCmdLocal('@Setting '+ZUPDC.Cmd.Acronym[n]+' '+Str(iGetTag(S.tag)));
    if TypeTag(S.tag)=2 then
    if ShouldRefresh(S.val,rGetTag(S.tag))>0 then
    DevPostCmdLocal('@Setting '+ZUPDC.Cmd.Acronym[n]+' '+Str(rGetTag(S.tag)));
   end;
  end;
 begin
  s:=''; slb:='';
  DIM_GuiClickBuff:='';
  //
  // Handle commands...
  //
  if DIM_IsServerMode then begin
   RefreshSetting(cm_FLDn,ZUPDC.SET_FD_GUI);
   RefreshSetting(cm_ASTn,ZUPDC.SET_AS_GUI);
   RefreshSetting(cm_OUTn,ZUPDC.SET_OT_GUI);
  end;
  //
  // Handle user mouse/keyboard clicks...
  // ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,KeyUp)
  // ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  //
  if ClickWhat<>0 then
  repeat
   //
   // Copy GUI click to DIM buffer for remote execution.
   //
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   //
   // Handle MouseDown/KeyDown
   //
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    //
    // Handle Left mouse button click
    //
    if (ClickButton=VK_LBUTTON) then begin
     //
     // Handle local clicks
     //
     if ClickIsLocal then begin
      // Sensor left bottom setting for editing
      slb:=SetFormUnderSensorLeftBottom(ClickParams(''));
      //
      // Handle sensor clicks...
      //
      if IsSameText(ClickSensor,'HELP') then begin
       Cron('@Browse '+DaqFileRef(AdaptFileName(ReadIni('[DAQ] HelpFile')),'.htm'));
       bNul(Voice(snd_Click));
      end;
      if (ClickTag<>0) then begin
       if (ClickTag=ZUPDC.ID_MDL.tag) then begin
        DevPostCmdLocal('@Remote @Setting MDL');
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=DIM_GuiClickFallBackModeTag) then begin
        if DIM_IsServerMode
        then ClickTagXorLocal(DIM_GuiClickFallBackModeTag,1)
        else ShowTooltip('text "Режим DIM FallBack применим только к серверу."  preset stdNotify delay 15000');
       end;
       ClickTagXorRemote(ZUPDC.POLL_ENABLE.tag,1);
       if (ClickTag=ZUPDC.RATE_A.tag) then begin
        StartEditTagEx(ZUPDC.RATE_A.tag,'Темп роста Тока, Ампер/сек',slb);
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=ZUPDC.RATE_V.tag) then begin
        StartEditTagEx(ZUPDC.RATE_V.tag,'Темп роста Напряжения, Вольт/сек',slb);
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=ZUPDC.SET_FD.tag)
       or (ClickTag=ZUPDC.SET_AS.tag)
       or (ClickTag=ZUPDC.SET_OT.tag)
       or (ClickTag=ZUPDC.Cmd.Tag[cm_CURn])
       or (ClickTag=ZUPDC.Cmd.Tag[cm_VOLn])
       or (ClickTag=ZUPDC.Cmd.Tag[cm_OVPn])
       or (ClickTag=ZUPDC.Cmd.Tag[cm_UVPn])
       then begin
        if ZUPDC_CheckPolling(1) then begin
         ClickTagXorRemote(ZUPDC.SET_FD.tag,1);
         ClickTagXorRemote(ZUPDC.SET_AS.tag,1);
         ClickTagXorRemote(ZUPDC.SET_OT.tag,1);
         ClickEditZupdcCmd(cm_CURn);
         ClickEditZupdcCmd(cm_VOLn);
         ClickEditZupdcCmd(cm_OVPn);
         ClickEditZupdcCmd(cm_UVPn);
        end;
       end;
      end;
      //
      // Select Plot & Tab windows by curve...
      //
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      //
      // Console commands: @url_encoded_sensor ...
      //
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     //
     // Handle remote clicks comes from DIM via @DimGuiClick message.
     // @DimGuiClick default handler decode and write events to FIFO,
     // so we can find it as clicks and can handle it in usual way.
     //
     if ClickIsRemote then begin
      //
      // Show time difference.
      //
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      //
      // Handle remote console commands...
      //
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevPostCmdLocal(s);
      //
      // Handle remote sensor clicks...
      //
      if TypeTag(ClickTag)>0 then begin
       s:=ClickParams('NewValue');
       if Length(s)>0 then begin
        if ClickTag=ZUPDC.POLL_ENABLE.tag then UpdateTag(ClickTag,s,0,1);
        if ClickTag=ZUPDC.SET_FD.tag then UpdateTag(ClickTag,s,0,1);
        if ClickTag=ZUPDC.SET_AS.tag then UpdateTag(ClickTag,s,0,1);
        if ClickTag=ZUPDC.SET_OT.tag then UpdateTag(ClickTag,s,0,1);
       end;
      end;
     end;
    end;
   end;
  until (ClickRead=0);
  //
  // Edit handling...
  //
  if EditStateDone then begin
   if CheckEditTag(ZUPDC.RATE_A.tag,s) then DevPostCmdLocal('@Remote @Setting RATE_A '+s);
   if CheckEditTag(ZUPDC.RATE_V.tag,s) then DevPostCmdLocal('@Remote @Setting RATE_V '+s);
   CheckEditZupdcCmd(cm_CURn,'@Remote ');
   CheckEditZupdcCmd(cm_VOLn,'@Remote ');
   CheckEditZupdcCmd(cm_OVPn,'@Remote ');
   CheckEditZupdcCmd(cm_UVPn,'@Remote ');
   //
   // Menu TOOLS.
   //
   MenuToolsHandler;
   //
   // Warning,Information.
   //
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  DIM_GuiClickBuff:='';
  s:=''; slb:='';
 end;
 //
 // Handle data requests in simulation mode. See ZUP manual.
 //
 procedure ZUPDC_Sim(Req:String);
 var v,rndv,rnda,rm,sv,sa,av,aa,op,up,os,al,ps,ot,as,fd,ohm,nlp:Real;
  procedure CopyLocal(Local:Boolean);
  begin
   if Local then begin
    sv:=ZUPDC.Cmd.SimDat[cm_VOLs]; sa:=ZUPDC.Cmd.SimDat[cm_CURs];
    av:=ZUPDC.Cmd.SimDat[cm_VOLa]; aa:=ZUPDC.Cmd.SimDat[cm_CURa];
    op:=ZUPDC.Cmd.SimDat[cm_OVPa]; up:=ZUPDC.Cmd.SimDat[cm_UVPa];
    os:=ZUPDC.Cmd.SimDat[cm_STAa]; al:=ZUPDC.Cmd.SimDat[cm_ALMa];
    ps:=ZUPDC.Cmd.SimDat[cm_STPa]; rm:=ZUPDC.Cmd.SimDat[cm_RMTa];
    fd:=Ord(IsBit(os,1)); as:=Ord(IsBit(os,2)); ot:=Ord(IsBit(os,3));
    ohm:=ZUPDC.SimOhm; nlp:=ZUPDC.SimNLP;
   end else begin
    ZUPDC.Cmd.SimDat[cm_VOLs]:=sv; ZUPDC.Cmd.SimDat[cm_CURs]:=sa;
    ZUPDC.Cmd.SimDat[cm_VOLa]:=av; ZUPDC.Cmd.SimDat[cm_CURa]:=aa;
    ZUPDC.Cmd.SimDat[cm_OVPa]:=op; ZUPDC.Cmd.SimDat[cm_UVPa]:=up;
    ZUPDC.Cmd.SimDat[cm_STAa]:=os; ZUPDC.Cmd.SimDat[cm_ALMa]:=al;
    ZUPDC.Cmd.SimDat[cm_STPa]:=ps; ZUPDC.Cmd.SimDat[cm_RMTa]:=rm;
   end;
  end;
 begin
  if Length(Req)>0 then begin
   ViewImp('COM < '+Trim(Req));
   if (Pos(';',Req)>0) then Req:=Copy(Req,1,Pos(';',Req)-1) else Req:='';
   if (Pos(':',Req)>0) then Req:=Copy(Req,Pos(':',Req)+1) else Req:='';
   if Length(Req)>0 then begin
    v:=rVal(Copy(Req,4));
    CopyLocal(true);
    rndv:=nlp/100*Random(-1,1)*sv;
    rnda:=nlp/100*Random(-1,1)*sa;
    if (ohm>0) then begin
     av:=sv; aa:=sa*ot;
     if av/ohm*ot<sa then begin
      os:=iSetBit(Round(os),0,0);
      aa:=av/ohm*ot;
     end else begin
      os:=iSetBit(Round(os),0,1);
      av:=sa*ohm;
     end;
     if (fd>0) and (aa>=sa) then begin
      os:=iSetBit(Round(os),3,0);
      al:=iSetBit(Round(al),3,1);
     end;
     if (al>0) then begin
      os:=iSetBit(Round(os),7,1);
     end;
    end;
    if IsSameText('ADR',Copy(Req,1,3)) then begin
     if not IsNan(v) then ZUPDC.Cmd.SimDat[cm_ADRn]:=v;
    end;
    if IsSameText('DCL',Copy(Req,1,3)) then begin
     os:=0;
     al:=0;
     ps:=0;
    end;
    if IsSameText('RMT',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='RM'+ZUPDC_Fmt('%1.0f',rm);
     if not IsNan(v) then ZUPDC.Cmd.SimDat[cm_RMTa]:=v;
    end;
    if IsSameText('MDL',Copy(Req,1,3)) then begin
     if IsEmptyStr(ZUPDC.ModelName) then ZUPDC.ModelName:='Nemic-Lambda ZUP(6V-33A)';
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:=ZUPDC.ModelName+' SIM';
    end;
    if IsSameText('REV',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='Ver 6-33 1.0';
    end;
    if IsSameText('VOL',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='AV'+ZUPDC_Fmt('%6.2f',av+rndv);
     if IsSameText('!',Copy(Req,4)) then ZUPDC.Com.Ans:='SV'+ZUPDC_Fmt('%6.2f',sv);
     if not IsNan(v) then sv:=v;
     if not IsNan(v) then av:=v;
    end;
    if IsSameText('CUR',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='AA'+ZUPDC_Fmt('%6.2f',aa+rnda);
     if IsSameText('!',Copy(Req,4)) then ZUPDC.Com.Ans:='SA'+ZUPDC_Fmt('%6.2f',sa);
     if not IsNan(v) then sa:=v;
     if not IsNan(v) then aa:=v;
    end;
    if IsSameText('OUT',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='OT'+ZUPDC_Fmt('%1.0f',ot);
     if not IsNan(v) then os:=iSetBitState(Round(os),3,IsBit(v,0));
     if not IsNan(v) then ot:=Ord(IsBit(os,3));
    end;
    if IsSameText('FLD',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='FD'+ZUPDC_Fmt('%1.0f',fd);
     if not IsNan(v) then os:=iSetBitState(Round(os),1,IsBit(v,0));
     if not IsNan(v) then fd:=Ord(IsBit(os,1));
    end;
    if IsSameText('AST',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='AS'+ZUPDC_Fmt('%1.0f',as);
     if not IsNan(v) then os:=iSetBitState(Round(os),2,IsBit(v,0));
     if not IsNan(v) then as:=Ord(IsBit(os,2));
    end;
    if IsSameText('OVP',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='OP'+ZUPDC_Fmt('%6.2f',op);
     if not IsNan(v) then op:=v;
    end;
    if IsSameText('UVP',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='UP'+ZUPDC_Fmt('%6.2f',up);
     if not IsNan(v) then up:=v;
    end;
    if IsSameText('STA',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='OS'+ZUPDC_Fmt('8-bit',os);
    end;
    if IsSameText('ALM',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='AL'+ZUPDC_Fmt('5-bit',al);
    end;
    if IsSameText('STP',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='PS'+ZUPDC_Fmt('5-bit',ps);
    end;
    if IsSameText('STT',Copy(Req,1,3)) then begin
     if IsSameText('?',Copy(Req,4)) then ZUPDC.Com.Ans:='AV'+ZUPDC_Fmt('%6.2f',av+rndv)
                                                       +'SV'+ZUPDC_Fmt('%6.2f',sv)
                                                       +'AA'+ZUPDC_Fmt('%6.2f',aa+rnda)
                                                       +'SA'+ZUPDC_Fmt('%6.2f',sa)
                                                       +'OS'+ZUPDC_Fmt('8-bit',os)
                                                       +'AL'+ZUPDC_Fmt('5-bit',al)
                                                       +'PS'+ZUPDC_Fmt('5-bit',ps);
    end;
    CopyLocal(false);
    if Length(ZUPDC.Com.Ans)>0 then begin
     if ComWrite(ZUPDC.Com.Ans+CRLF)
     then ViewExp('COM > '+Trim(ZUPDC.Com.Ans))
     else Trouble('Could not send '+Trim(ZUPDC.Com.Ans));
    end;
   end;
  end;
  ZUPDC.Com.Req:='';
  ZUPDC.Com.Ans:='';
 end;
 //
 // Self Test procedures.
 //
 procedure ZUPDC_SelfTest;
 begin
  bNul(Echo(StrFmt('pi/10=%6.3f',pi/10)));
  Writeln(ZUPDC_ExtractData('AV5.010SV5.010AA00.00SA24.31OS00010000AL00000PS00000','SV'));
  Writeln(ZUPDC_ValBin('001'));
 end;
 //
 // ZUPDC cleanup.
 //
 procedure ZUPDC_Clear;
 begin
  ZUPDC_Clear_Cmd;
  ZUPDC.CmdHash:=0;
  ZUPDC.Com.Req:='';
  ZUPDC.Com.Ans:='';
  ZUPDC.Com.Buf:='';
  ZUPDC.ModelCode:=0;
  ZUPDC.ModelName:='';
  ZUPDC.RampingTime:=0;
  RampingClear(ZUPDC.RampingA);
  RampingClear(ZUPDC.RampingV);
  ZUPDC.SimOhm:=0;
  ZUPDC.SimNLP:=0;
 end;
 //
 // ZUPDC initialization.
 //
 procedure ZUPDC_Init;
 var i:Integer;
 begin
  //
  // Register error codes
  //
  ZUPDC.ecTimeOut:=RegisterErr(progname+': TimeOut');
  //
  // Initialize variables
  //
  ZUPDC.Com.ReqTime:=0;
  ZUPDC.Com.AnsTime:=0;
  ZUPDC.Com.LastPoll:=0;
  ZUPDC.Com.Status:=st_NoReq;
  ZUPDC.Com.Interface:=RS_232;
  //
  // Initialize command table.
  //
  ZUPDC.CmdHash:=HashList_Init(0);
  ZUPDC_Init_Cmd;
  //
  // Read ini file variables
  //
  ZUPDC.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  if ZUPDC.Simulator
  then Success('Run simulator mode.')
  else Success('Run as driver mode.');
  ZUPDC.Com.Port:=iValDef(ReadIni('ComPort'),0);
  Success('ComPort = '+Str(ZUPDC.Com.Port));
  ZUPDC.Com.Addr:=iValDef(ReadIni('ComAddr'),1);
  Success('ComAddr = '+Str(ZUPDC.Com.Addr));
  ZUPDC.Com.TimeOut:=iValDef(ReadIni('ComTimeOut'),0);
  Success('ComTimeOut = '+Str(ZUPDC.Com.TimeOut));
  ZUPDC.Com.TimeGap:=iValDef(ReadIni('ComTimeGap'),0);
  Success('ComTimeGap = '+Str(ZUPDC.Com.TimeGap));
  ZUPDC.Com.PollPeriod:=iValDef(ReadIni('PollPeriod'),100);
  Success('PollPeriod = '+Str(ZUPDC.Com.PollPeriod));
  ZUPDC.ModelName:=ReadIniVar('ModelName',28);
  Success('ModelName = '+ZUPDC.ModelName);
  ZUPDC.ModelCode:=ZUPDC_UpdateModel(ZUPDC_ParseMDL(ZUPDC.ModelName,false));
  ZUPDC.Cmd.OpData[cm_ADRn]:=ZUPDC.Com.Addr;
  ZUPDC.Com.Interface:=iValDef(ZUPDC_ExtractData(ReadIni('ComInterface'),'RS'),RS_232);
  Success('ComInterface = RS'+Str(ZUPDC.Com.Interface));
  //
  // Initialize tags...
  //
  if not ZUPDC.Simulator then begin
   ZUPDC_InitTags(ReadIni('tagPrefix'));
   bNul(sSetTag(ZUPDC.ID_MDL.tag,ZUPDC.ModelName));
  end;
  //
  // Initialize COM port
  //
  if not DIM_IsClientMode then
  if ComOpen('[SerialPort-COM'+Str(ZUPDC.Com.Port)+']')
  then Success('COM port initialized.')
  else Trouble('COM port failed.');
  //
  // List/find available Volt/Amper ranges
  //
  PrintRangeList('RangeSV',ListRangeSV);
  PrintRangeList('RangeSA',ListRangeSA);
 end;
 //
 // ZUPDC finalization.
 //
 procedure ZUPDC_Free;
 begin
  bNul(ComClose);
  if (ZUPDC.CmdHash<>0) then begin
   bNul(HashList_Free(ZUPDC.CmdHash));
   ZUPDC.CmdHash:=0;
  end;
 end;
 //
 // ZUPDC polling.
 //
 procedure ZUPDC_Poll;
 begin
  if ZUPDC.Simulator then begin
   if IsComPortOpened then begin
    if (Pos(';',ZUPDC.Com.Buf)>0) and (Pos(Copy(CRLF,1,1),ZUPDC.Com.Buf)=0)
    then ZUPDC.Com.Buf:=StrReplace(ZUPDC.Com.Buf,Dump(';'),';'+CRLF,3);
    while Com_Readln(ZUPDC.Com.Req,ZUPDC.Com.Buf) do ZUPDC_Sim(ZUPDC.Com.Req);
   end;
  end else begin
   ZUPDC_GUI_POLL;
   if not DIM_IsClientMode then begin
    if SysTimer_Pulse(1000)>0 then begin
     bNul(rSetTag(ZUPDC.ERROR_CNT.tag,GetErrCount(-1)));
     UpdateDimTag(ZUPDC.ERROR_CNT.tag);
     UpdateAo(ao_ERROR_CNT,time,GetErrCount(-1));
     bNul(rSetTag(ZUPDC.POLL_RATE.tag,ZUPDC.Com.PollRate));
     UpdateDimTag(ZUPDC.POLL_RATE.tag);
     UpdateAo(ao_POLL_RATE,time,ZUPDC.Com.PollRate);
     ZUPDC.Com.PollRate:=0;
    end;
    if mSecNow-FixmSecNow>DelayOnStart then
    ZUPDC_CMD_CYCLE;
   end;
  end;
 end;
 //
 // Clear user application strings...
 //
 procedure ClearApplication;
 begin
  ZUPDC_Clear;
 end;
 //
 // User application Initialization...
 //
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  ZUPDC_Init;
  RunStartupScript;
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  cmd_DimTagUpdate  := RegisterStdInCmd('@DimTagUpdate',  '');
  cmd_Simulate      := RegisterStdInCmd('@Simulate',      '');
  cmd_PrintCmdTable := RegisterStdInCmd('@PrintCmdTable', '');
  cmd_Ramping       := RegisterStdInCmd('@Ramping',       '');
  cmd_Setting       := RegisterStdInCmd('@Setting',       '');
  cmd_Remote        := RegisterStdInCmd('@Remote',        '');
  cmd_Edit          := RegisterStdInCmd('@Edit',          '');
  cmd_MenuToolsOpen := RegisterStdInCmd('@MenuToolsOpen', '');
  cmd_LoadIni       := RegisterStdInCmd('@LoadIni',       '');
  cmd_SaveIni       := RegisterStdInCmd('@SaveIni',       '');
  ZUPDC_SelfTest;
 end;
 //
 // User application Finalization...
 //
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  ZUPDC_Free;
 end;
 //
 // User application Polling...
 //
 procedure PollApplication;
 begin
  ZUPDC_Poll;
 end;
 //
 // Process data coming from standard input...
 //
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid,tag,n:Integer; v:Real; slb:String;
  procedure SettingCmd(n:Integer; v:Real);
  begin
   if not IsNan(v) then
   if IsValidCmdNum(n) then begin
    Success(cmd+' '+ZUPDC.Cmd.Acronym[n]+' '+Str(v));
    ZUPDC.Cmd.OpBuff[n]:=v;
    EnableCmdNum(n,true);
    if (n=cm_FLDn) then bNul(iSetTag(ZUPDC.SET_FD.tag,Ord(v<>0)));
    if (n=cm_ASTn) then bNul(iSetTag(ZUPDC.SET_AS.tag,Ord(v<>0)));
    if (n=cm_OUTn) then bNul(iSetTag(ZUPDC.SET_OT.tag,Ord(v<>0)));
   end;
  end;
  procedure CheckEditCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(ZUPDC.Cmd.Tag[n])>=1 then
   if TypeTag(ZUPDC.Cmd.Tag[n])<=2 then
   if IsSameText(ExtractWord(1,arg),ZUPDC.Cmd.Acronym[n])
   or IsSameText(ExtractWord(1,arg),ZUPDC.Cmd.AnsPref[n]) then
   StartEditTagEx(ZUPDC.Cmd.Tag[n],ZUPDC.Cmd.Acronym[n]+': '+ZUPDC.Cmd.Comment[n],slb);
  end;
  procedure UpdateZupCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(ZUPDC.Cmd.Tag[n])>=1 then
   if TypeTag(ZUPDC.Cmd.Tag[n])<=2 then
   DevPostCmdLocal('@Setting '+ZUPDC.Cmd.Acronym[n]+' '+ TagAsText(ZUPDC.Cmd.Tag[n]));
  end;
  procedure UpdateAoByTag(AoNum,tag:Integer);
  begin
   if (AoNum>=0) then begin
    if TypeTag(tag)=1 then UpdateAo(AoNum,time,iGetTag(tag));
    if TypeTag(tag)=2 then UpdateAo(AoNum,time,rGetTag(tag));
   end;
  end;
  procedure UpdateSettingsAfterLoadIni;
  begin
   if (iGetTag(ZUPDC.POLL_ENABLE.tag)<>0) then begin
    DevPostCmdLocal('@Setting FD '+TagAsText(ZUPDC.SET_FD.tag));
    DevPostCmdLocal('@Setting AS '+TagAsText(ZUPDC.SET_AS.tag));
    DevPostCmdLocal('@Setting OT '+TagAsText(ZUPDC.SET_OT.tag));
    DevPostCmdLocal('@Setting OP '+TagAsText(ZUPDC.PAR_OP.tag));
    DevPostCmdLocal('@Setting UP '+TagAsText(ZUPDC.PAR_UP.tag));
    DevPostCmdLocal('@Ramping SV '+TagAsText(ZUPDC.PAR_SV.tag));
    DevPostCmdLocal('@Ramping SA '+TagAsText(ZUPDC.PAR_SA.tag));
   end;
  end;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  slb:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Remote
   }
   if (cmdid=cmd_Remote) then begin
    if not IsEmptyStr(arg) then PostCmdRemote(Trim(arg));
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if TypeTag(tag)>0 then begin
      if tag=ZUPDC.POLL_RATE.tag then UpdateAoByTag(ao_POLL_RATE,tag);
      if tag=ZUPDC.ERROR_CNT.tag then UpdateAoByTag(ao_ERROR_CNT,tag);
      if tag=ZUPDC.PAR_AA.tag    then UpdateAoByTag(ao_PAR_AA,tag);
      if tag=ZUPDC.PAR_AV.tag    then UpdateAoByTag(ao_PAR_AV,tag);
      if tag=ZUPDC.PAR_AW.tag    then UpdateAoByTag(ao_PAR_AW,tag);
      if tag=ZUPDC.PAR_SA.tag    then UpdateAoByTag(ao_PAR_SA,tag);
      if tag=ZUPDC.PAR_SV.tag    then UpdateAoByTag(ao_PAR_SV,tag);
      if tag=ZUPDC.PAR_OP.tag    then UpdateAoByTag(ao_PAR_OP,tag);
      if tag=ZUPDC.PAR_UP.tag    then UpdateAoByTag(ao_PAR_UP,tag);
      if tag=ZUPDC.PAR_OS.tag    then UpdateAoByTag(ao_PAR_OS,tag);
      if tag=ZUPDC.PAR_AL.tag    then UpdateAoByTag(ao_PAR_AL,tag);
      if tag=ZUPDC.PAR_PS.tag    then UpdateAoByTag(ao_PAR_PS,tag);
     end;
    end;
   end else
   {
   @Setting VOLn 1.0    @Setting SV 1.0
   @Setting CURn 2.0    @Setting SA 2.0
   @Setting OVPn 3.0    @Setting OP 3.0
   @Setting UVPn 4.0    @Setting UP 4.0
   @Setting OUTn 1      @Setting OT 1
   @Setting FLDn 1      @Setting FD 1
   @Setting ASTn 1      @Setting AS 1
   @Setting DCL         @Setting MDL
   @Setting RATE_A 1.0  @Setting RATE_V 1.0
   }
   if (cmdid=cmd_Setting) then begin
    if not ZUPDC.Simulator and not DIM_IsClientMode then begin
     n:=ZUPDC_FindCmd(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=rVal(ExtractWord(2,arg));
      if (n=cm_VOLs) then n:=cm_VOLn;
      if (n=cm_CURs) then n:=cm_CURn;
      if (n=cm_OVPa) then n:=cm_OVPn;
      if (n=cm_UVPa) then n:=cm_UVPn;
      if (n=cm_OUTa) then n:=cm_OUTn;
      if (n=cm_ASTa) then n:=cm_ASTn;
      if (n=cm_FLDa) then n:=cm_FLDn;
      if (n=cm_VOLn) then SettingCmd(n,v);
      if (n=cm_CURn) then SettingCmd(n,v);
      if (n=cm_OVPn) then SettingCmd(n,v);
      if (n=cm_UVPn) then SettingCmd(n,v);
      if (n=cm_OUTn) then SettingCmd(n,v);
      if (n=cm_OUTn) then SettingCmd(n,v);
      if (n=cm_FLDn) then SettingCmd(n,v);
      if (n=cm_ASTn) then SettingCmd(n,v);
      if (n=cm_DCL)  then SettingCmd(n,0);
      if (n=cm_DCL)  then bNul(iSetTag(ZUPDC.SET_FD.tag,0));
      if (n=cm_DCL)  then bNul(iSetTag(ZUPDC.SET_AS.tag,0));
      if (n=cm_DCL)  then bNul(iSetTag(ZUPDC.SET_OT.tag,0));
     end else
     if IsSameText(ExtractWord(1,arg),'RATE_A') then begin
      UpdateTag(ZUPDC.RATE_A.tag,ExtractWord(2,arg),0,100);
     end else
     if IsSameText(ExtractWord(1,arg),'RATE_V') then begin
      UpdateTag(ZUPDC.RATE_V.tag,ExtractWord(2,arg),0,100);
     end else
     if IsSameText(ExtractWord(1,arg),'MDL') then begin
      bNul(sSetTag(ZUPDC.ID_MDL.tag,'?'));
      UpdateDimTag(ZUPDC.ID_MDL.tag);
      EnableCmdNum(cm_MDLa,true);
     end;
    end;
    Data:='';
   end else 
   {
   @Ramping VOLn 1.0    @Ramping SV 1.0
   @Ramping CURn 2.0    @Ramping SA 2.0
   }
   if (cmdid=cmd_Ramping) then begin
    if not ZUPDC.Simulator and not DIM_IsClientMode then begin
     n:=ZUPDC_FindCmd(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=rVal(ExtractWord(2,arg));
      if (n=cm_VOLs) then n:=cm_VOLn;
      if (n=cm_CURs) then n:=cm_CURn;
      if (n=cm_VOLn) then begin
       if (v>=ZUPDC.Cmd.Min[cm_VOLn]) and (v<=ZUPDC.Cmd.Max[cm_VOLn]) then begin
        RampingStart(ZUPDC.RampingV,ZUPDC.Cmd.OpData[cm_VOLs],v,rGetTag(ZUPDC.RATE_V.tag));
        Success(cmd+' '+ZUPDC.Cmd.Acronym[cm_VOLn]+' '+Str(v));
       end else Problem('VOL ramping value out of range.');
      end;
      if (n=cm_CURn) then begin
       if (v>=ZUPDC.Cmd.Min[cm_CURn]) and (v<=ZUPDC.Cmd.Max[cm_CURn]) then begin
        RampingStart(ZUPDC.RampingA,ZUPDC.Cmd.OpData[cm_CURs],v,rGetTag(ZUPDC.RATE_A.tag));
        Success(cmd+' '+ZUPDC.Cmd.Acronym[cm_CURn]+' '+Str(v));
       end else Problem('CUR ramping value out of range.');
      end;
     end;
    end;
    Data:='';
   end else 
   {
   @Edit VOLn <mime_encoded_sensor_left_bottom_setting>
   @Edit CURn
   @Edit OVPn
   @Edit UVPn
   }
   if (cmdid=cmd_Edit) then begin
    if not ZUPDC.Simulator then begin
     if ZUPDC_CheckPolling(1) then begin
      slb:=mime_decode(ExtractWord(2,arg));
      CheckEditCmd(cm_CURn);
      CheckEditCmd(cm_VOLn);
      CheckEditCmd(cm_OVPn);
      CheckEditCmd(cm_UVPn);
     end;
    end;
    Data:='';
   end else
   {
   @MenuToolsOpen
   }
   if (cmdid=cmd_MenuToolsOpen) then begin
    MenuToolsStarter;
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    if not ZUPDC.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
     UpdateSettingsAfterLoadIni;
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    if not ZUPDC.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   @Simulate VOL! 1.0     @Simulate SV 1.0
   @Simulate VOL? 2.0     @Simulate AV 2.0
   @Simulate CUR! 3.0     @Simulate SA 3.0
   @Simulate CUR? 4.0     @Simulate AA 4.0
   @Simulate OUT? 1       @Simulate OT 1
   @Simulate RMT? 1       @Simulate RM 1
   @Simulate FLD? 1       @Simulate FD 1
   @Simulate OVP? 1       @Simulate OP 1
   @Simulate UVP? 1       @Simulate UP 1
   @Simulate AST? 1       @Simulate AS 1
   @Simulate STA? 1       @Simulate OS 1
   @Simulate ALM? 1       @Sumulate AL 1
   @Simulate STP? 1       @Simulate PS 1
   @Simulate OHM  1.5
   @Simulate NLP  0.1
   }
   if (cmdid=cmd_Simulate) then begin
    if ZUPDC.Simulator then begin
     n:=ZUPDC_FindCmd(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=ZUPDC.Cmd.SimDat[n];
      v:=rValDef(ExtractWord(2,arg),v);
      ZUPDC.Cmd.SimDat[n]:=v;
      Success(cmd+' '+ZUPDC.Cmd.Acronym[n]+' '+Str(v));
      if (n=cm_FLDa) then ZUPDC.Cmd.SimDat[cm_STAa]:=iSetBit(Round(ZUPDC.Cmd.SimDat[cm_STAa]),1,Round(v));
      if (n=cm_ASTa) then ZUPDC.Cmd.SimDat[cm_STAa]:=iSetBit(Round(ZUPDC.Cmd.SimDat[cm_STAa]),2,Round(v));
      if (n=cm_OUTa) then ZUPDC.Cmd.SimDat[cm_STAa]:=iSetBit(Round(ZUPDC.Cmd.SimDat[cm_STAa]),3,Round(v));
     end;
     if IsSameText(ExtractWord(1,arg),'OHM') then begin
      ZUPDC.SimOhm:=rValDef(ExtractWord(2,arg),ZUPDC.SimOhm);
      Success(cmd+' OHM '+Str(ZUPDC.SimOhm));
     end;
     if IsSameText(ExtractWord(1,arg),'NLP') then begin
      ZUPDC.SimNLP:=rValDef(ExtractWord(2,arg),ZUPDC.SimNLP);
      Success(cmd+' NLP '+Str(ZUPDC.SimNLP));
     end;
    end;
    Data:='';
   end else
   {
   @PrintCmdTable
   }
   if (cmdid=cmd_PrintCmdTable) then begin
    PrintCmdTable;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  slb:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
