#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for ZUPDC devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

#################################
### DOZA devices config generator
### ZUPDC is ZUP DC by TDK-Lambda
#################################

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg"; # hardware   configuration
readonly simcfg="${scriptbase}_sim.cfg"; # simulator  configuration
readonly diccfg="${scriptbase}_dic.cfg"; # DIM client configuration
readonly discfg="${scriptbase}_dis.cfg"; # DIM server configuration
readonly impcfg="${scriptbase}_imp.cfg"; # DIM import configuration
readonly devmsg="devPostMsg";            # Method to send/post messages
readonly MAIN_CTRL="ZUPDC.MAIN.CTRL";    # Name of main control window
readonly PARAM_PLOT="ZUPDC.PARAM.PLOT";  # Name of Param plot window
readonly STATE_PLOT="ZUPDC.STATE.PLOT";  # Name of State plot window
declare -i numhdwcfg=0;                  # Flag to include hardware config
declare -i CTRL_TOP=0;                   # Top  of main control window
declare -i CTRL_LEFT=450;                # Left of main control window
declare -i CTRL_TOP_STEP=50;
declare -i MAIN_HEIGHT=160;
declare -i MAIN_H_STEP=50;
declare -i TABCRC_TOP=0;
declare -i TABCRC_STEP=50;
readonly tabcrc="../circuits/${scriptbase}.crc";
declare FACN="";                         # Facility name
declare IDEN="";                         # Device identifier name
declare FACP="";                         # Facility DIM  path
declare FACF="";                         # Facility file name
declare IDEN_LO="";                      # Device identifier name (lower case)

#########################
# Generate ZUPDC devices.
#########################
function zupdc_devices(){
 delete_files $hdwcfg $simcfg $diccfg $discfg $tabcrc;
 ####################################################################################
 ### Device  Facility    Identifier PortNum Address Polling  Model
 ####################################################################################
 zupdc_dev   DEMO.ZUPDC  ZUP1       1       1       50       "Nemic-Lambda ZUP(6V-33A)";
 zupdc_dev   DEMO.ZUPDC  ZUP2       2       1       50       "Nemic-Lambda ZUP(6V-66A)";
 zupdc_dev   DEMO.ZUPDC  ZUP3       3       1       50       "Nemic-Lambda ZUP(10V-20A)";
#zupdc_dev   DEMO.ZUPDC  ZUP4       4       1       50       "Nemic-Lambda ZUP(10V-40A)";
#zupdc_dev   DEMO.ZUPDC  ZUP5       5       1       50       "Nemic-Lambda ZUP(20V-10A)";
#zupdc_dev   DEMO.ZUPDC  ZUP6       6       1       50       "Nemic-Lambda ZUP(20V-20A)";
#zupdc_dev   DEMO.ZUPDC  ZUP7       7       1       50       "Nemic-Lambda ZUP(36V-6A)";
#zupdc_dev   DEMO.ZUPDC  ZUP8       8       1       50       "Nemic-Lambda ZUP(36V-12A)";
#zupdc_dev   DEMO.ZUPDC  ZUP9       9       1       50       "Nemic-Lambda ZUP(60V-3.5A)";
 ####################################################################################
 zupdc_done >> $hdwcfg;
 cat $hdwcfg;
 cat $simcfg;
};

########################
# Generate final config.
########################
function zupdc_done(){
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.job $MAIN_CTRL.HOME @async @async @WinDraw $MAIN_CTRL|Height=$MAIN_HEIGHT";
 echo "[]";
 echo "";
};

########################
# Generate ZUPDC device.
########################
function zupdc_dev(){
 zupdc_hdw "$@" >> $hdwcfg;
 zupdc_sim "$@" >> $simcfg;
 zupdc_dic "$@" >> $diccfg;
 zupdc_dis "$@" >> $discfg;
 zupdc_tabcrc   >> $tabcrc;
 let "CTRL_TOP+=CTRL_TOP_STEP";
 let "MAIN_HEIGHT+=MAIN_H_STEP";
};

#################################
# Generate ZUPDC hardware config.
#################################
function zupdc_hdw(){
 FACN="$1"       ; shift    ; ## Facility name
 IDEN="$1"       ; shift    ; ## Device identifier name
 ComPort="$1"    ; shift    ; ## COM port number
 ComAddr="$1"    ; shift    ; ## COM port device address
 ComPolling="$1" ; shift    ; ## COM port polling period, ms
 ModelName="$1"  ; shift    ; ## Device model name
 FACP="$(echo "$FACN" | tr '.' '/')"; ## Facility DIM  path, i.e. FACN name with . to / replacement
 FACF="$(echo "$FACN" | tr '.' '_')"; ## Facility file name, i.e. FACN name with . to _ replacement
 FACF="$(echo "$FACF" | tr '[:upper:]' '[:lower:]')"; ## Facility file name to lower case
 IDEN_LO="$(echo "$IDEN" | tr '[:upper:]' '[:lower:]')";
 echo ";**********************";
 echo "; ZUPDC Driver for $IDEN";
 echo ";**********************";
 echo "";
 echo "[SerialPort-COM$ComPort]";
 echo "Port     = COM$ComPort";
 echo "BaudRate = 9600";
 echo "Parity   = NONE";
 echo "DataBits = 8";
 echo "StopBits = 1";
 echo "[]";
 echo "";
 echo "[TagList]";
 echo "$FACN.$IDEN.POLL_ENABLE = integer 0               ; Enable polling flag";
 echo "$FACN.$IDEN.DIMGUICLICK = string  ?               ; The DIM click sensor data";
 echo "$FACN.$IDEN.POLL_RATE   = real    0               ; Polling rate, poll/sec";
 echo "$FACN.$IDEN.ERROR_CNT   = real    0               ; Device error counter";
 echo "$FACN.$IDEN.ID_NAME     = string  $IDEN           ; Device identifier string";
 echo "$FACN.$IDEN.ID_MDL      = string  ?               ; Device model, e.g. Nemic-Lambda ZUP(6V-33A)";
 echo "$FACN.$IDEN.ID_REV      = string  ?               ; Software revision, e.g. Ver 6-33 1.0";
 echo "$FACN.$IDEN.PAR_SV      = real    0               ; Setting   Volt";
 echo "$FACN.$IDEN.PAR_AV      = real    0               ; Actual    Volt";
 echo "$FACN.$IDEN.PAR_AW      = real    0               ; Actual    Watts (calculated as AV*AA)";
 echo "$FACN.$IDEN.PAR_SA      = real    0               ; Setting   Amper";
 echo "$FACN.$IDEN.PAR_AA      = real    0               ; Actual    Amper";
 echo "$FACN.$IDEN.PAR_OS      = integer 0               ; Operation Status Register";
 echo "$FACN.$IDEN.PAR_AL      = integer 0               ; Alarm     Status Register";
 echo "$FACN.$IDEN.PAR_PS      = integer 0               ; Program   Status Register";
 echo "$FACN.$IDEN.PAR_OP      = real    0               ; Setting   OverVoltage  Protection";
 echo "$FACN.$IDEN.PAR_UP      = real    0               ; Setting   UnderVoltage Protection";
 echo "$FACN.$IDEN.SET_FD      = integer 0               ; Button to set foldback protection";
 echo "$FACN.$IDEN.SET_AS      = integer 0               ; Button to set autostart mode";
 echo "$FACN.$IDEN.SET_OT      = integer 0               ; Button to set output state";
 echo "$FACN.$IDEN.RATE_A      = real    1               ; Current ramping rate, Amper/sec";
 echo "$FACN.$IDEN.RATE_V      = real    1               ; Voltage ramping rate, Volt/sec";
 echo "$FACN.$IDEN.RAMP_A      = integer 0               ; Current ramping state";
 echo "$FACN.$IDEN.RAMP_V      = integer 0               ; Voltage ramping state";
 echo "[]";
 echo "";
 echo "[DataStorage]";
 echo "$FACN.$IDEN.POLL_RATE   = Curve 0 10 Black 15 1   ; Polling rate, poll/sec";
 echo "$FACN.$IDEN.ERROR_CNT   = Curve 0 10 Black 15 1   ; Device error counter";
 echo "$FACN.$IDEN.PAR_SV      = Curve 0 10 Black 15 1   ; Setting   Volt";
 echo "$FACN.$IDEN.PAR_AV      = Curve 0 10 Black 15 1   ; Actual    Volt";
 echo "$FACN.$IDEN.PAR_SA      = Curve 0 10 Black 15 1   ; Setting   Amper";
 echo "$FACN.$IDEN.PAR_AA      = Curve 0 10 Black 15 1   ; Actual    Amper";
 echo "$FACN.$IDEN.PAR_AW      = Curve 0 10 Black 15 1   ; Actual    Watts (calculated as AV*AA)";
 echo "$FACN.$IDEN.PAR_OS      = Curve 0 10 Black 15 1   ; Operation Status Register";
 echo "$FACN.$IDEN.PAR_AL      = Curve 0 10 Black 15 1   ; Alarm     Status Register";
 echo "$FACN.$IDEN.PAR_PS      = Curve 0 10 Black 15 1   ; Program   Status Register";
 echo "$FACN.$IDEN.PAR_OP      = Curve 0 10 Black 15 1   ; Setting   OverVoltage  Protection";
 echo "$FACN.$IDEN.PAR_UP      = Curve 0 10 Black 15 1   ; Setting   UnderVoltage Protection";
 echo "[]";
 echo "";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.CTRL = device software program";
 echo "[&$FACN.$IDEN.CTRL]";
 echo "Comment        = Hardware driver for ZUP DC power supply $IDEN.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpHigher";
 echo "ProgramSource  = ../daqpas/zupdc_drv.pas";
 echo "StartupScript  = [&$FACN.$IDEN.CTRL.StartupScript]";
 echo "FinallyScript  = [&$FACN.$IDEN.CTRL.FinallyScript]";
 echo "DigitalFifo    = 1024";
 echo "AnalogFifo     = 1024";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "Simulator      = 0";
 echo "ComPort        = $ComPort";
 echo "ComAddr        = $ComAddr";
 echo "ComTimeOut     = 250";
 echo "ComTimeGap     = 20";
 echo "ComInterface   = RS232";
 echo "PollPeriod     = $ComPolling";
 echo "ModelName      = $ModelName";
 echo "tagPrefix      = $FACN.$IDEN";
 echo "HelpFile       = ../help/zupdc.htm";
 echo "EncryptMethod  = RC6";
 echo "TrustedUsers   = [TrustedUsers]";
 echo "AnalogOutputs  = 12";
 echo "Link AnalogOutput  0 with curve $FACN.$IDEN.POLL_RATE    history 1000";
 echo "Link AnalogOutput  1 with curve $FACN.$IDEN.ERROR_CNT    history 1000";
 echo "Link AnalogOutput  2 with curve $FACN.$IDEN.PAR_AA       history 1000";
 echo "Link AnalogOutput  3 with curve $FACN.$IDEN.PAR_AV       history 1000";
 echo "Link AnalogOutput  4 with curve $FACN.$IDEN.PAR_AW       history 1000";
 echo "Link AnalogOutput  5 with curve $FACN.$IDEN.PAR_SA       history 1000";
 echo "Link AnalogOutput  6 with curve $FACN.$IDEN.PAR_SV       history 1000";
 echo "Link AnalogOutput  7 with curve $FACN.$IDEN.PAR_OS       history 1000";
 echo "Link AnalogOutput  8 with curve $FACN.$IDEN.PAR_AL       history 1000";
 echo "Link AnalogOutput  9 with curve $FACN.$IDEN.PAR_PS       history 1000";
 echo "Link AnalogOutput 10 with curve $FACN.$IDEN.PAR_OP       history 1000";
 echo "Link AnalogOutput 11 with curve $FACN.$IDEN.PAR_UP       history 1000";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$FACN.$IDEN.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ../data/${FACF}_${IDEN_LO}_custom.ini";
 echo "CustomIniSection  = [$FACN.$IDEN.CustomParameters]";
 echo "CustomIniBackups  = ../data/custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "[ConfigFileList]";
 echo "ConfigFile = ../data/${FACF}_${IDEN_LO}_custom.ini";
 echo "[]";
 echo "[$FACN.$IDEN.CustomParameters.TagList]";
 echo "TagList = $FACN.$IDEN.POLL_ENABLE";
 echo "TagList = $FACN.$IDEN.PAR_SA";
 echo "TagList = $FACN.$IDEN.PAR_SV";
 echo "TagList = $FACN.$IDEN.PAR_OP";
 echo "TagList = $FACN.$IDEN.PAR_UP";
 echo "TagList = $FACN.$IDEN.SET_FD";
 echo "TagList = $FACN.$IDEN.SET_AS";
 echo "TagList = $FACN.$IDEN.SET_OT";
 echo "TagList = $FACN.$IDEN.RATE_A";
 echo "TagList = $FACN.$IDEN.RATE_V";
 echo "[]";
 echo "[&$FACN.$IDEN.CTRL.StartupScript]";
 echo "@cron @cron.tab $FACN.$IDEN.CTRL.StartupLoadIni 0 0 0";
 echo "@cron @cron.pul $FACN.$IDEN.CTRL.StartupLoadIni 3000 1";
 echo "@cron @cron.job $FACN.$IDEN.CTRL.StartupLoadIni @DevPostMsg &$FACN.$IDEN.CTRL @LoadIni";
 echo "@cron @cron.job $FACN.$IDEN.CTRL.StartupLoadIni @cron.del $FACN.$IDEN.CTRL.StartupLoadIni";
 echo "[]";
 echo "[&$FACN.$IDEN.CTRL.FinallyScript]";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.CTRL = Circuit_Window ";
 echo "[$FACN.$IDEN.CTRL]";
 echo "Circuit = ../circuits/zupdc_ctrl.crc";
 echo "Link sensor POLL_ENABLE       with tag   $FACN.$IDEN.POLL_ENABLE device &$FACN.$IDEN.CTRL";
 echo "Link sensor POLL_MEMO         with tag   $FACN.$IDEN.POLL_ENABLE device &$FACN.$IDEN.CTRL";
 echo "Link sensor ID_NAME           with tag   $FACN.$IDEN.ID_NAME     device &$FACN.$IDEN.CTRL";
 echo "Link sensor ID_MDL            with tag   $FACN.$IDEN.ID_MDL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor ID_REV            with tag   $FACN.$IDEN.ID_REV      device &$FACN.$IDEN.CTRL";
 echo "Link sensor POLL_RATE         with curve $FACN.$IDEN.POLL_RATE   device &$FACN.$IDEN.CTRL";
 echo "Link sensor ERROR_CNT         with curve $FACN.$IDEN.ERROR_CNT   device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AA            with curve $FACN.$IDEN.PAR_AA      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AV            with curve $FACN.$IDEN.PAR_AV      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AW            with curve $FACN.$IDEN.PAR_AW      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_SA            with tag   $FACN.$IDEN.PAR_SA      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_SV            with tag   $FACN.$IDEN.PAR_SV      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS            with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.0          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.1          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.2          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.3          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.4          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.5          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.6          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OS.7          with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL            with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.0          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.1          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.2          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.3          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.4          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.5          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.6          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_AL.7          with curve $FACN.$IDEN.PAR_AL      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS            with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.0          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.1          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.2          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.3          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.4          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.5          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.6          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_PS.7          with curve $FACN.$IDEN.PAR_PS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_OP            with tag   $FACN.$IDEN.PAR_OP      device &$FACN.$IDEN.CTRL";
 echo "Link sensor PAR_UP            with tag   $FACN.$IDEN.PAR_UP      device &$FACN.$IDEN.CTRL";
 echo "Link sensor SET_FD            with tag   $FACN.$IDEN.SET_FD      device &$FACN.$IDEN.CTRL";
 echo "Link sensor SET_AS            with tag   $FACN.$IDEN.SET_AS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor SET_OT            with tag   $FACN.$IDEN.SET_OT      device &$FACN.$IDEN.CTRL";
 echo "Link sensor RATE_A            with tag   $FACN.$IDEN.RATE_A      device &$FACN.$IDEN.CTRL";
 echo "Link sensor RATE_V            with tag   $FACN.$IDEN.RATE_V      device &$FACN.$IDEN.CTRL";
 echo "Link sensor RAMP_A            with tag   $FACN.$IDEN.RAMP_A      device &$FACN.$IDEN.CTRL";
 echo "Link sensor RAMP_V            with tag   $FACN.$IDEN.RAMP_V      device &$FACN.$IDEN.CTRL";
 echo "Link sensor @BrowseHelp                                       with device &$FACN.$IDEN.CTRL";
 echo "Link sensor @Remote+@LoadIni                                  with device &$FACN.$IDEN.CTRL";
 echo "Link sensor @Remote+@SaveIni                                  with device &$FACN.$IDEN.CTRL";
 echo "Link sensor @OpenConsole                                      with device &$FACN.$IDEN.CTRL";
 echo "Link sensor @MenuToolsOpen                                    with device &$FACN.$IDEN.CTRL";
 echo "Link sensor @Remote+@Setting+DCL                              with device &$FACN.$IDEN.CTRL";
 echo "Link sensor DIM+FallBack+Mode with tag   &DimSrv.FallBackMode      device &$FACN.$IDEN.CTRL";
 echo "[]";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.CTRL.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=670|Height=312";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.CTRL.HOME @WinDraw $FACN.$IDEN.CTRL|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.CTRL.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SHOW @WinShow $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.HIDE @WinHide $FACN.$IDEN.CTRL";
 echo "@cron.tab $FACN.$IDEN.CTRL.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.CTRL.SELECT @WinSelect $FACN.$IDEN.CTRL";
 echo "@cron.job $MAIN_CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HOME";
 echo "@cron.job $MAIN_CTRL.HOME @cron.run $FACN.$IDEN.CTRL.HIDE";
 echo "[]";
 echo "";
 echo "[$MAIN_CTRL]";
 echo "Link sensor $IDEN.CTRL        with                                 device &$MAIN_CTRL";
 echo "Link sensor $IDEN.POLL_ENABLE with tag   $FACN.$IDEN.POLL_ENABLE device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.POLL_RATE   with curve $FACN.$IDEN.POLL_RATE   device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_AA      with curve $FACN.$IDEN.PAR_AA      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_AV      with curve $FACN.$IDEN.PAR_AV      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_AW      with curve $FACN.$IDEN.PAR_AW      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_SA      with tag   $FACN.$IDEN.PAR_SA      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_SV      with tag   $FACN.$IDEN.PAR_SV      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.RAMP_A      with tag   $FACN.$IDEN.RAMP_A      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.RAMP_V      with tag   $FACN.$IDEN.RAMP_V      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_OS.0    with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.PAR_OS.7    with curve $FACN.$IDEN.PAR_OS      device &$FACN.$IDEN.CTRL";
 echo "Link sensor $IDEN.SET_OT      with tag   $FACN.$IDEN.SET_OT      device &$FACN.$IDEN.CTRL";
 echo "[]";
 echo "";
 echo "[$PARAM_PLOT]";
 echo "CurveList = $FACN.$IDEN.PAR_AA";
 echo "CurveList = $FACN.$IDEN.PAR_AV";
 echo "CurveList = $FACN.$IDEN.PAR_AW";
 echo "CurveList = $FACN.$IDEN.PAR_SA";
 echo "CurveList = $FACN.$IDEN.PAR_SV";
 echo "CurveList = $FACN.$IDEN.PAR_OP";
 echo "CurveList = $FACN.$IDEN.PAR_UP";
 echo "[]";
 echo "";
 echo "[$STATE_PLOT]";
 echo "CurveList = $FACN.$IDEN.POLL_RATE";
 echo "CurveList = $FACN.$IDEN.ERROR_CNT";
 echo "CurveList = $FACN.$IDEN.PAR_OS";
 echo "CurveList = $FACN.$IDEN.PAR_AL";
 echo "CurveList = $FACN.$IDEN.PAR_PS";
 echo "[]";
 echo "";
 echo "[&DATSRV]";
 echo "CurveList = $FACN.$IDEN.PAR_AA";
 echo "CurveList = $FACN.$IDEN.PAR_AV";
 echo "CurveList = $FACN.$IDEN.PAR_AW";
 echo "CurveList = $FACN.$IDEN.PAR_SA";
 echo "CurveList = $FACN.$IDEN.PAR_SV";
 echo "CurveList = $FACN.$IDEN.PAR_OP";
 echo "CurveList = $FACN.$IDEN.PAR_UP";
 echo "CurveList = $FACN.$IDEN.POLL_RATE";
 echo "CurveList = $FACN.$IDEN.ERROR_CNT";
 echo "CurveList = $FACN.$IDEN.PAR_OS";
 echo "CurveList = $FACN.$IDEN.PAR_AL";
 echo "CurveList = $FACN.$IDEN.PAR_PS";
 echo "[]";
};

function zupdc_sim(){
 if [ $numhdwcfg -eq 0 ]; then zupdc_sim_once;  let "numhdwcfg+=1"; fi;
 let "ComPair=ComPort+10";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.SIM = device software program";
 echo "[&$FACN.$IDEN.SIM]";
 echo "Comment        = Simulation for ZUP DC power supply $IDEN.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpNormal";
 echo "ProgramSource  = ../daqpas/zupdc_drv.pas";
 echo "StartupScript  = [&$FACN.$IDEN.SIM.StartupScript]";
 echo "FinallyScript  = [&$FACN.$IDEN.SIM.FinallyScript]";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "Simulator      = 1";
 echo "ComPort        = $ComPair";
 echo "ComAddr        = $ComAddr";
 echo "ComTimeOut     = 250";
 echo "ComTimeGap     = 20";
 echo "PollPeriod     = $ComPolling";
 echo "ModelName      = $ModelName";
 echo "tagPrefix      = $FACN.$IDEN";
 echo "[]";
 echo "[&$FACN.$IDEN.SIM.StartupScript]";
 echo "@Simulate OHM 1.0 ; Simulate Output Resistance, Ohm";
 echo "@Simulate NLP 0.1 ; Simulate Noise Level in Percent";
 echo "[]";
 echo "[&$FACN.$IDEN.SIM.FinallyScript]";
 echo "[]";
 echo "";
 echo "[SerialPort-COM$ComPort]";
 echo "PipeName    = $IDEN.com$ComPort.sim";
 echo "PipePolling = 4, tpHighest";
 echo "TimeOut     = 1000";
 echo "FifoSize    = 16";
 echo "[]";
 echo "[SerialPort-COM$ComPair]";
 echo "PipeName    = ./$IDEN.com$ComPort.sim";
 echo "PipePolling = 4, tpHighest";
 echo "TimeOut     = 1000";
 echo "FifoSize    = 16";
 echo "[]";
 echo "";
};
function zupdc_sim_once(){
 echo "[ConfigFileList]";
 echo "ConfigFile = $hdwcfg";
 echo "[]";
 echo "";
};

function zupdc_tabcrc(){
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.CTRL";
 echo "[$IDEN.CTRL]";
 echo "Pos = 10, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:28\Style:[Bold]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_6_28_4_aqua_ptmono.bmp $IDEN";
 echo "Hint = $IDEN.CTRL: Источник питания $IDEN";
 echo "[]";
 echo "";
 echo "[Circuit.StartupScript]";
 echo "${IDEN}_poll_enab = 0";
 echo "${IDEN}_poll_rate = 0";
 echo "${IDEN}_poll_norm = 5";
 echo "${IDEN}_poll_gtrz = 0";
 echo "${IDEN}_poll_good = 0";
 echo "${IDEN}_ramping_a = 0";
 echo "${IDEN}_ramping_v = 0";
 echo "${IDEN}_cur_state = 0";
 echo "${IDEN}_out_state = 0";
 echo "${IDEN}_alm_state = 0";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_SA";
 echo "[$IDEN.PAR_SA]";
 echo "Pos = 150, $YPOS";
 echo "LED = 7, 0, 3, %7.3f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_white_ptmono.bmp";
 echo "Hint = $IDEN.PAR_SA - Уставка тока, Ампер (Setting Amper)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_SV";
 echo "[$IDEN.PAR_SV]";
 echo "Pos = 150, $YPOS";
 echo "LED = 7, 0, 3, %7.3f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_white_ptmono.bmp";
 echo "Hint = $IDEN.PAR_SV - Уставка напряжения, Вольт (Setting Volt)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.RAMP_A";
 echo "[$IDEN.RAMP_A]";
 echo "Pos = 231, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Mono\Size:13\Style:[Bold]\Color:Navy";
 echo "TagEval(v) = ne(${IDEN}_ramping_a=v,0)";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_silver_ptmono.bmp А";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_yellow_ptmono.bmp А";
 echo "Hint = $IDEN.RAMP_A - статус перехода по току";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.RAMP_V";
 echo "[$IDEN.RAMP_V]";
 echo "Pos = 231, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Mono\Size:13\Style:[Bold]\Color:Navy";
 echo "TagEval(v) = ne(${IDEN}_ramping_v=v,0)";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_silver_ptmono.bmp В";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_yellow_ptmono.bmp В";
 echo "Hint = $IDEN.RAMP_V - статус перехода по напряжению";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_AA";
 echo "[$IDEN.PAR_AA]";
 echo "Pos = 255, $YPOS";
 echo "LED = 7, 0, 3, %7.3f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "TagEval(v) = ${IDEN}_poll_good";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_silver_ptmono.bmp";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_aqua_ptmono.bmp";
 echo "Hint = $IDEN.PAR_AA - Измерение тока, Ампер (Actual Amper)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_AV";
 echo "[$IDEN.PAR_AV]";
 echo "Pos = 255, $YPOS";
 echo "LED = 7, 0, 3, %7.3f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "TagEval(v) = ${IDEN}_poll_good";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_silver_ptmono.bmp";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_aqua_ptmono.bmp";
 echo "Hint = $IDEN.PAR_AV - Измерение напряжения, Вольт (Actual Volt)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_AW";
 echo "[$IDEN.PAR_AW]";
 echo "Pos = 340, $YPOS";
 echo "LED = 7, 0, 0, %7.0f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "TagEval(v) = ${IDEN}_poll_good";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_silver_ptmono.bmp";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_aqua_ptmono.bmp";
 echo "Hint = $IDEN.PAR_AW - Измерение мощности, Ватт (Actual Watt)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_OS.0";
 echo "[$IDEN.PAR_OS.0]";
 echo "Pos = 340, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Sans_Narrow\Size:9\Style:[Normal]";
 echo "TagEval(v) = (${IDEN}_cur_state=isbit(v,0))+(${IDEN}_out_state=isbit(v,3))*0";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_silver_ptmono.bmp VOL";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_silver_ptmono.bmp CUR";
 echo "Hint = $IDEN.PAR_OS.0 - Режим стабилизации напряжения/тока (VOL/CUR).";
 echo "[]";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.PAR_OS.7";
 echo "[$IDEN.PAR_OS.7]";
 echo "Pos = 365, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Sans_Narrow\Size:9\Style:[Normal]";
 echo "TagEval(v) = (${IDEN}_alm_state=isbit(v,7))";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_silver_ptmono.bmp ALM";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_2_13_4_red_ptmono.bmp    ALM";
 echo "Hint = $IDEN.PAR_OS.7 - ALARM state.";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.POLL_RATE";
 echo "[$IDEN.POLL_RATE]";
 echo "Pos = 390, $YPOS";
 echo "LED = 2, 0, 0, %2.0f, Name:PT_Mono\Size:13\Style:[Bold]";
 echo "TagEval(v) = ${IDEN}_poll_enab*(1+(${IDEN}_poll_gtrz=gt(${IDEN}_poll_rate=v,0))+(${IDEN}_poll_good=gt(v,${IDEN}_poll_norm)))";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_3_13_4_silver_ptmono.bmp";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_3_13_4_red_ptmono.bmp";
 echo "Tag#3 = 2, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_3_13_4_yellow_ptmono.bmp";
 echo "Tag#4 = 3, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_3_13_4_lime_ptmono.bmp";
 echo "Hint = $IDEN.POLL_RATE - Частота опроса, Герц";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = $IDEN.SET_OT";
 echo "[$IDEN.SET_OT]";
 echo "Pos = 455, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "TagEval(v) = isbit(v,0)+2*${IDEN}_out_state";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)";
 echo "Painter(v) = (glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_silver_ptmono.bmp Нагрузка";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_yellow_ptmono.bmp Нагрузка";
 echo "Tag#3 = 2, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_yellow_ptmono.bmp Нагрузка";
 echo "Tag#4 = 3, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_lime_ptmono.bmp   Нагрузка";
 echo "Hint = $IDEN.SET_OT - OutpuT - Включить Нагрузку (Выход)";
 echo "[]";
 echo "";
 let "YPOS=TABCRC_TOP+83";
 echo "[SensorList]";
 echo "Sensor = $IDEN.POLL_ENABLE";
 echo "[$IDEN.POLL_ENABLE]";
 echo "Pos = 455, $YPOS";
 echo "LED = 0, 0, 0, %s, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "TagEval(v) = (${IDEN}_poll_enab=isbit(v,0))*(1+${IDEN}_poll_gtrz+${IDEN}_poll_good)";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)";
 echo "Painter(v) = (glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_silver_ptmono.bmp Опрос";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_red_ptmono.bmp    Опрос";
 echo "Tag#3 = 2, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_yellow_ptmono.bmp Опрос";
 echo "Tag#4 = 3, ~~\Resource\DaqSite\StdLib\Bitmaps\ledbmp_8_13_4_lime_ptmono.bmp   Опрос";
 echo "Hint = $IDEN.POLL_ENABLE - Разрешение опроса COM порта";
 echo "[]";
 let "TABCRC_TOP+=TABCRC_STEP";
};

function zupdc_dis(){
 unix dimcfg \
  -n section  "[&$FACN.$IDEN.CTRL]" \
  -n print    DimServerMode = 1 \
  -n end \
  -n dis_cmnd $FACP/$IDEN/DIMGUICLICK \
  -n tag      $FACN.$IDEN.DIMGUICLICK \
  -n $devmsg "&$FACN.$IDEN.CTRL @DIMGUICLICK=%**" \
  -n end \
  -n dic_cmnd $FACP/$IDEN/DIMGUICLICK \
  -n tag      $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dis_info $FACP/$IDEN/POLL_ENABLE \
  -n tag      $FACN.$IDEN.POLL_ENABLE \
  -n end \
  -n dis_info $FACP/$IDEN/POLL_RATE \
  -n tag      $FACN.$IDEN.POLL_RATE \
  -n end \
  -n dis_info $FACP/$IDEN/ERROR_CNT \
  -n tag      $FACN.$IDEN.ERROR_CNT \
  -n end \
  -n dis_info $FACP/$IDEN/ID_MDL \
  -n tag      $FACN.$IDEN.ID_MDL \
  -n end \
  -n dis_info $FACP/$IDEN/ID_REV \
  -n tag      $FACN.$IDEN.ID_REV \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_AA \
  -n tag      $FACN.$IDEN.PAR_AA \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_AV \
  -n tag      $FACN.$IDEN.PAR_AV \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_AW \
  -n tag      $FACN.$IDEN.PAR_AW \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_SA \
  -n tag      $FACN.$IDEN.PAR_SA \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_SV \
  -n tag      $FACN.$IDEN.PAR_SV \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_OP \
  -n tag      $FACN.$IDEN.PAR_OP \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_UP \
  -n tag      $FACN.$IDEN.PAR_UP \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_OS \
  -n tag      $FACN.$IDEN.PAR_OS \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_AL \
  -n tag      $FACN.$IDEN.PAR_AL \
  -n end \
  -n dis_info $FACP/$IDEN/PAR_PS \
  -n tag      $FACN.$IDEN.PAR_PS \
  -n end \
  -n dis_info $FACP/$IDEN/RATE_A \
  -n tag      $FACN.$IDEN.RATE_A \
  -n end \
  -n dis_info $FACP/$IDEN/RATE_V \
  -n tag      $FACN.$IDEN.RATE_V \
  -n end \
  -n dis_info $FACP/$IDEN/RAMP_A \
  -n tag      $FACN.$IDEN.RAMP_A \
  -n end \
  -n dis_info $FACP/$IDEN/RAMP_V \
  -n tag      $FACN.$IDEN.RAMP_V \
  -n end \
  -n dis_info $FACP/$IDEN/SET_FD \
  -n tag      $FACN.$IDEN.SET_FD \
  -n end \
  -n dis_info $FACP/$IDEN/SET_AS \
  -n tag      $FACN.$IDEN.SET_AS \
  -n end \
  -n dis_info $FACP/$IDEN/SET_OT \
  -n tag      $FACN.$IDEN.SET_OT \
  -n end \
  -n ;
};

######################
# DIM client services.
######################
function zupdc_dic(){
 unix dimcfg \
  -n section  "[&$FACN.$IDEN.CTRL]" \
  -n print    DimClientMode = 1 \
  -n end \
  -n dic_cmnd $FACP/$IDEN/DIMGUICLICK \
  -n tag      $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dic_info $FACP/$IDEN/POLL_ENABLE \
  -n tag      $FACN.$IDEN.POLL_ENABLE \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.POLL_ENABLE" \
  -n end \
  -n dic_info $FACP/$IDEN/POLL_RATE \
  -n tag      $FACN.$IDEN.POLL_RATE \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.POLL_RATE" \
  -n end \
  -n dic_info $FACP/$IDEN/ERROR_CNT \
  -n tag      $FACN.$IDEN.ERROR_CNT \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.ERROR_CNT" \
  -n end \
  -n dic_info $FACP/$IDEN/ID_MDL \
  -n tag      $FACN.$IDEN.ID_MDL \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.ID_MDL" \
  -n end \
  -n dic_info $FACP/$IDEN/ID_REV \
  -n tag      $FACN.$IDEN.ID_REV \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.ID_REV" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_AA \
  -n tag      $FACN.$IDEN.PAR_AA \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_AA" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_AV \
  -n tag      $FACN.$IDEN.PAR_AV \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_AV" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_AW \
  -n tag      $FACN.$IDEN.PAR_AW \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_AW" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_SA \
  -n tag      $FACN.$IDEN.PAR_SA \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_SA" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_SV \
  -n tag      $FACN.$IDEN.PAR_SV \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_SV" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_OP \
  -n tag      $FACN.$IDEN.PAR_OP \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_OP" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_UP \
  -n tag      $FACN.$IDEN.PAR_UP \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_UP" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_OS \
  -n tag      $FACN.$IDEN.PAR_OS \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_OS" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_AL \
  -n tag      $FACN.$IDEN.PAR_AL \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_AL" \
  -n end \
  -n dic_info $FACP/$IDEN/PAR_PS \
  -n tag      $FACN.$IDEN.PAR_PS \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.PAR_PS" \
  -n end \
  -n dic_info $FACP/$IDEN/RATE_A \
  -n tag      $FACN.$IDEN.RATE_A \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.RATE_A" \
  -n end \
  -n dic_info $FACP/$IDEN/RATE_V \
  -n tag      $FACN.$IDEN.RATE_V \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.RATE_V" \
  -n end \
  -n dic_info $FACP/$IDEN/RAMP_A \
  -n tag      $FACN.$IDEN.RAMP_A \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.RAMP_A" \
  -n end \
  -n dic_info $FACP/$IDEN/RAMP_V \
  -n tag      $FACN.$IDEN.RAMP_V \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.RAMP_V" \
  -n end \
  -n dic_info $FACP/$IDEN/SET_FD \
  -n tag      $FACN.$IDEN.SET_FD \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.SET_FD" \
  -n end \
  -n dic_info $FACP/$IDEN/SET_AS \
  -n tag      $FACN.$IDEN.SET_AS \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.SET_AS" \
  -n end \
  -n dic_info $FACP/$IDEN/SET_OT \
  -n tag      $FACN.$IDEN.SET_OT \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.SET_OT" \
  -n end \
  -n ;
};

##############
# delete files
##############
function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scriptHOME >/dev/null 2>&1; then
  zupdc_devices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
