 {                                                                            
 ***********************************************************************
 Daq Pascal application program WIKA_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @MenuToolsConfirmation - for internal use.
| @PrintCmdTable - Print Command Table state.
| @Simulate id v - Set simulation data v to id=(MA,ME,PZ,PK,
|                  AZ).
| @Setting id v  - Send command id with data v, id=(AZ).
| @Remote @cmd   - run @cmd remotely: send to DIM server
| @Edit id       - edit id (AZ)
| @BrowseHelp    - open help file in browser.
| @OpenConsole   - open driver console window.
| @MenuToolsOpen - open Menu Tools dialog.
| @LoadIni       - load params from INI file.
| @SaveIni       - save params to   INI file.
| @DimGuiClick   - handle remote click from DIM
|********************************************************
[]
 }
program WIKA_D1X_DRV;            { WIKA D1X driver                  }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 st_NoReq           = 0;         { Status : No request sent         }
 st_WaitAns         = 1;         { Status : Waiting answer          }
 st_WaitGap         = 2;         { Status : Waiting time gap        }
 st_TimeOut         = 3;         { Status : TimeOut found           }
 es_ERROR           = 'ERROR';      { Error string: error detected  }
 es_ERROR_CR        = 'ERROR CR';   { Error string: CR expected     }
 es_ERROR_ARG       = 'ERROR ARG';  { Error string: bad argument    }
 es_ERROR_LEN       = 'ERROR LEN';  { Error string: bad length      }
 es_ERROR_LRC       = 'ERROR LRC';  { Error string: bad checksum    }
 es_ERROR_ETX       = 'ERROR ETX';  { Error string: ETX expected    }
 es_ERROR_EOT       = 'ERROR_EOT';  { Error string: EOT expected    }
 es_ERROR_x50       = 'ERROR x50';  { Error string: 'P' expected    }
 es_ERROR_x54       = 'ERROR x54';  { Error string: 'T' expected    }
 es_ERROR_x6B       = 'ERROR x6B';  { Error string: 'k' expected    }
 es_ERROR_x4B       = 'ERROR x4B';  { Error string: 'K' expected    }
 es_ERROR_x61       = 'ERROR x61';  { Error string: 'a' expected    }
 es_ERROR_x7A       = 'ERROR x7A';  { Error string: 'z' expected    }
 es_ERROR_x73       = 'ERROR x73';  { Error string: 's' expected    }
 es_ERROR_x6F       = 'ERROR x6F';  { Error string: 'o' expected    }
 cm_SO              = 1;         { SO set operation mode            }
 cm_MA              = 2;         { MA read MBA lower limit of range }
 cm_ME              = 3;         { ME read MBE high  limit of range }
 cm_PZ              = 4;         { PZ read pressure as phys. unit   }
 cm_PK              = 5;         { PK read pressure value in digits }
 cm_TW              = 6;         { TW read temperature value        }
 cm_KN              = 7;         { KN read instrument number        }
 cm_AZ              = 8;         { AZ set response time             }
 MaxCmdNum          = 8;         { Max command id number            }
 ao_POLL_RATE       = 0;         { Analog outputs...                }
 ao_ERROR_CNT       = 1;         {                                  }
 ao_PAR_MA          = 2;         {                                  }
 ao_PAR_ME          = 3;         {                                  }
 ao_PAR_PZ          = 4;         {                                  }
 ao_PAR_PZM         = 5;         {                                  }
 ao_PAR_PZU         = 6;         {                                  }
 ao_PAR_PK          = 7;         {                                  }
 ao_PAR_PKS         = 8;         {                                  }
 ao_PAR_TW          = 9;         {                                  }
 ao_PAR_AZ          = 10;        {                                  }
 si_OFFLINE         = 'OFFLINE'; { To identify offline state        }
 list_Measuring     = 'rel abs'; { List of measuring methods        }
 list_PhysUnits     = 'bar psi mmHg inHg mWs inWs'; { Pressure unit }
 DelayOnStart       = 1000;      { Delay before start polling       }
 UseCmdHash         = true;      { Use CmdHash for fast search      }

type
 TTagRef  = record tag,nai,ndi,nao,ndo:Integer; val:Real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 WIKA             : record       { WIKA data                        }
  Simulator        : Boolean;    { Simulator mode                   }
  ecTimeOut        : Integer;    { Error code: TimeOut              }
  Com              : record      { COM port data                    }
   Port            : Integer;    {  Port number                     }
   Status          : Integer;    {  Status flag                     }
   TimeOut         : Integer;    {  TimeOut, ms                     }
   TimeGap         : Integer;    {  Time delay after poll, ms       }
   PollPeriod      : Integer;    { Poll period, ms                  }
   Req,Ans,Buf     : String;     {  Request, Answer                 }
   ReqTime,AnsTime : Real;       {  Request, Answer time, ms        }       
   LastPoll        : Real;       { Time of last poll, ms            }
   PollRate        : Integer;    { Poll rate, poll/sec              }
  end;                           {                                  }
  Cmd              : record      {  Command cycle data              }
   Num             : Integer;    {   Current command number         }
   Enabled         : array [1..MaxCmdNum] of Boolean; {             }
   Acronym         : array [1..MaxCmdNum] of String;  {             }
   ReqHead         : array [1..MaxCmdNum] of String;  {             }
   AnsHead         : array [1..MaxCmdNum] of String;  {             }
   AnsLeng         : array [1..MaxCmdNum] of Integer; {             }
   Comment         : array [1..MaxCmdNum] of String;  {             }
   OpData          : array [1..MaxCmdNum] of Real;    {             }
   OpBuff          : array [1..MaxCmdNum] of Real;    {             }
   SimDat          : array [1..MaxCmdNum] of Real;    {             }
   SimStr          : array [1..MaxCmdNum] of String;  {             }
   AoNum           : array [1..MaxCmdNum] of Integer; {             }
   Tag             : array [1..MaxCmdNum] of Integer; {             }
   Min             : array [1..MaxCmdNum] of Real;    {             }
   Max             : array [1..MaxCmdNum] of Real;    {             }
  end;                           {                                  }
  CmdHash          : Integer;    { Hash list ref. for fast search   }
  POLL_ENABLE      : TTagRef;    {                                  }
  POLL_RATE        : TTagRef;    {                                  }
  ERROR_CNT        : TTagRef;    {                                  }
  ID_NAME          : TTagRef;    {                                  }
  ID_IDN           : TTagRef;    {                                  }
  PAR_MA           : TTagRef;    {                                  }
  PAR_ME           : TTagRef;    {                                  }
  PAR_PZ           : TTagRef;    {                                  }
  PAR_PZMU         : TTagRef;    {                                  }
  PAR_PZM          : TTagRef;    {                                  }
  PAR_PZU          : TTagRef;    {                                  }
  PAR_PK           : TTagRef;    {                                  }
  PAR_PKS          : TTagRef;    {                                  }
  PAR_KN           : TTagRef;    {                                  }
  PAR_TW           : TTagRef;    {                                  }
  PAR_AZ           : TTagRef;    {                                  }
  ModelCode        : Integer;    { Encoded model ranges (Tab index) }
  ModelName        : String;     { Model name like D-10-P           }
 end;                            {                                  }
 cmd_DimTagUpdate  : Integer;    { @DimTagUpdate                    }
 cmd_Simulate      : Integer;    { @Simulate                        }
 cmd_PrintCmdTable : Integer;    { @PrintCmdTable                   }
 cmd_Setting       : Integer;    { @Setting                         }
 cmd_Remote        : Integer;    { @Remote                          }
 cmd_Edit          : Integer;    { @Edit                            }
 cmd_BrowseHelp    : Integer;    { @BrowseHelp                      }
 cmd_OpenConsole   : Integer;    { @OpenConsole                     }
 cmd_MenuToolsOpen : Integer;    { @MenuToolsOpen                   }
 cmd_LoadIni       : Integer;    { @LoadIni                         }
 cmd_SaveIni       : Integer;    { @SaveIni                         }
 cmd_MenuToolsConfirmation:Integer;{ Command @MenuToolsConfirmation }
 MenuToolsSelected : Integer;    { MenuTools selected item index    }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 //
 // Calculate checksum - method=89(ModbusLrc).
 // Uses Longitudinal Redundancy Check.
 //
 function CalcCheckSum(s:String):Integer;
 begin
  CalcCheckSum:=HashIndexOf(s,256,89);
 end;
 {
 function CalcCheckSum(s:String):Integer;
 var i,cs:Integer;
 begin
  cs:=0; for i:=1 to Length(s) do cs:=cs+Ord(StrFetch(s,i));
  cs:=iAnd(cs,255); cs:=iXor(cs,255); cs:=cs+1;
  CalcCheckSum:=iAnd(cs,255);
 end;
 }
 //
 // Append checksum to data string s.
 //
 function AppendCheckSum(s:String):String;
 var cs:Integer;
 begin
  if Length(s)>0 then begin
   cs:=CalcCheckSum(s);
   s:=s+Chr(cs);
  end;
  AppendCheckSum:=s;
 end;
 //
 // Append checksum and CR to data string s.
 //
 function AppendCheckSumCR(s:String):String;
 begin
  AppendCheckSumCR:=AppendCheckSum(s)+Chr(13);
 end;
 //
 // Check if data string s has valid checksum.
 //
 function HasValidCheckSum(s:String):Boolean;
 var l,c1,c2:Integer;
 begin
  c1:=0; c2:=0;
  l:=Length(s);
  if (l>1) then begin
   c1:=CalcCheckSum(Copy(s,1,l-1));
   c2:=Ord(StrFetch(s,l));
  end;
  HasValidCheckSum:=(c1=c2) and (l>1);
 end;
 //
 // Decode lower limit of range (MBA).
 // Return 'Value rel/abs' or 'ERROR XXX'.
 // For example: '2.5 abs' or 'ERROR LRC'.
 //
 function Decode_MA(s:String):String;
 var r:String; m,v:Real; h,l,f,e,a:Integer;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>6) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>Chr(3)) then r:=es_ERROR_ETX else
  if (StrFetch(s,6)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,5)) then r:=es_ERROR_LRC else begin
   h:=Ord(StrFetch(s,2));                                  // MBA hb
   l:=Ord(StrFetch(s,3));                                  // MBA lb
   f:=Ord(StrFetch(s,4));                                  // MB-Factor
   a:=(f div 128);                                         // Measuring method
   m:=h*256+iAnd(l,127);  if iAnd(l,128)>0 then m:=-m;     // Mantissa
   e:=iAnd(f,63);         if iAnd(f,64)>0  then e:=-e;     // Exponent
   v:=m*power(10,e);                                       // Value as number
   r:=StrFmt('%.7g',v);                                    // Value as string
   r:=r+' '+ExtractWord(1+a,list_Measuring);               // Measuring method
  end;
  Decode_MA:=r; r:='';
 end;
 //
 // Decode upper limit of range (MBE).
 // Return 'Value rel/abs' or 'ERROR XXX'.
 // For example: '2.5 abs' or 'ERROR LRC'.
 //
 function Decode_ME(s:String):String;
 var r:String; m,v:Real; h,l,f,e,a:Integer;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>6) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>Chr(4)) then r:=es_ERROR_EOT else
  if (StrFetch(s,6)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,5)) then r:=es_ERROR_LRC else begin
   h:=Ord(StrFetch(s,2));                                  // MBE hb
   l:=Ord(StrFetch(s,3));                                  // MBE lb
   f:=Ord(StrFetch(s,4));                                  // MB-Factor
   a:=(f div 128);                                         // Measuring method
   m:=iAnd(h,127)*256+l;  if iAnd(h,128)>0 then m:=-m;     // Mantissa
   e:=iAnd(f,63);         if iAnd(f,64)>0  then e:=-e;     // Exponent
   v:=m*power(10,e);                                       // Value as number
   r:=StrFmt('%.7g',v);                                    // Value as string
   r:=r+' '+ExtractWord(1+a,list_Measuring);               // Measuring method
  end;
  Decode_ME:=r; r:='';
 end;
 //
 // Decode pressure value in physical units.
 // Return 'Value rel/abs Units' or 'ERROR XXX'.
 // For example: '2.5 abs bar'   or 'ERROR LRC'.
 //
 function Decode_PZ(s:String):String;
 var r:String; m,v:Real; h,l,f,e,a,u:Integer;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>6) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'P') then r:=es_ERROR_x50 else
  if (StrFetch(s,6)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,5)) then r:=es_ERROR_LRC else begin
   h:=Ord(StrFetch(s,2));                                  // hb
   l:=Ord(StrFetch(s,3));                                  // lb
   f:=Ord(StrFetch(s,4));                                  // P-Factor
   a:=(f div 128);                                         // Measuring method
   u:=iAnd(f,7);                                           // Physical Units
   m:=iAnd(h,127)*256+l;  if iAnd(h,128)>0 then m:=-m;     // Mantissa
   e:=iAnd(f div 8,7);    if iAnd(f,64)>0  then e:=-e;     // Exponent
   v:=m*power(10,e);                                       // Value as number
   r:=StrFmt('%.7g',v);                                    // Value as string
   r:=r+' '+ExtractWord(1+a,list_Measuring);               // Measuring method
   r:=r+' '+ExtractWord(1+u,'bar psi mmHg inHg mWs inWs'); // Physical Units as string   
  end;
  Decode_PZ:=r; r:='';
 end;
 //
 // Decode pressure in digits.
 // Return 'Value Status' or 'ERROR XXX'.
 // For example: '2.5 ok' or 'ERROR LRC'.
 //
 function Decode_PK(s:String; MBA,MBE:Real):String;
 var r:String; v:Real; h,l,b:Integer;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>6) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'k') then r:=es_ERROR_x6B else
  if (StrFetch(s,6)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,5)) then r:=es_ERROR_LRC else begin
   h:=Ord(StrFetch(s,2));                                  // hb
   l:=Ord(StrFetch(s,3));                                  // lb
   b:=Ord(StrFetch(s,4));                                  // Status byte
   v:=((h*256+l)-10000)*(MBE-MBA)/50000+MBA;               // Value as number
   r:=StrFmt('%.7g',v);                                    // Value as string
   r:=r+' '+ExtractWord(1+b,'ok fail');                    // Status ok/fail
  end;
  Decode_PK:=r; r:='';
 end;
 //
 // Decode temperature value in Celsius.
 // Return 'Value Units' or 'ERROR XXX'.
 // For example: '25 °C' or 'ERROR LRC'.
 //
 function Decode_TW(s:String):String;
 var r:String; v:Real; h,l,b:Integer;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>6) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'T') then r:=es_ERROR_x54 else
  if (StrFetch(s,6)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,5)) then r:=es_ERROR_LRC else begin
   h:=Ord(StrFetch(s,2));                                  // hb
   l:=Ord(StrFetch(s,3));                                  // lb
   b:=Ord(StrFetch(s,4));                                  // Always zero
   v:=(h*256+l)/2; if iAnd(h,1)>0 then v:=-v;              // Value as number
   r:=StrFmt('%.7g',v);                                    // Value as string
   r:=r+' °C';                                             // Physical Units as string
  end;
  Decode_TW:=r; r:='';
 end;
 //
 // Decode instrument number.
 // Return 'Number'     or 'ERROR XXX'.
 // For example: '1234' or 'ERROR LRC'.
 //
 function Decode_KN(s:String):String;
 var r:String;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>7) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'K') then r:=es_ERROR_x4B else
  if (StrFetch(s,7)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,6)) then r:=es_ERROR_LRC else begin
   r:=Copy(s,2,4);                                         // Value as string
  end;
  Decode_KN:=r; r:='';
 end;
 //
 // Decode response time code.
 // Return 'Code'      or 'ERROR XXX'.
 // For example: '$FF' or 'ERROR LRC'.
 //
 function Decode_AZ(s:String):String;
 var r:String;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>5) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'a') then r:=es_ERROR_x61 else
  if (StrFetch(s,2)<>'z') then r:=es_ERROR_x7A else
  if (StrFetch(s,5)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,4)) then r:=es_ERROR_LRC else begin
   r:='$'+HexB(Ord(StrFetch(s,3)));                        // Value as string
  end;
  Decode_AZ:=r; r:='';
 end;
 //
 // Decode polling mode.
 // Return '$FF'       or 'ERROR XXX'.
 // For example: '$FF' or 'ERROR LRC'.
 //
 function Decode_SO(s:String):String;
 var r:String;
 begin
  r:=es_ERROR_ARG;
  if (Length(s)<>5) then r:=es_ERROR_LEN else
  if (StrFetch(s,1)<>'s') then r:=es_ERROR_x73 else
  if (StrFetch(s,2)<>'o') then r:=es_ERROR_x6F else
  if (StrFetch(s,5)<>Chr(13)) then r:=es_ERROR_CR else
  if not HasValidCheckSum(Copy(s,1,4)) then r:=es_ERROR_LRC else begin
   r:='$'+HexB(Ord(StrFetch(s,3)));                        // Value as string
  end;
  Decode_SO:=r; r:='';
 end;
 //
 // Encode_PZ('1.23 abs bar');
 //
 function Encode_PZ(s:String):String;
 var v:Real; m,u,e,d,h,l,f:Integer;
 begin
  v:=rVal(ExtractWord(1,s));
  m:=WordIndex(ExtractWord(2,s),list_Measuring)-1; if m<0 then m:=0;
  u:=WordIndex(ExtractWord(3,s),list_PhysUnits)-1; if u<0 then u:=0;
  if IsNan(v) then s:='' else begin
   e:=0;
   if abs(v)<32768 then begin
    while (e>-7) and (abs(v)*10<32768) do begin
     e:=e-1; v:=v*10;
    end;
   end else begin
    while (e<7) and (abs(v)>=32768) do begin
     e:=e+1; v:=v/10;
    end;
   end;
   d:=Round(abs(v)); l:=d mod 256; h:=d div 256; if v<0 then h:=iOr(h,128);
   f:=u+abs(e)*8+m*128; if e<0 then f:=iOr(f,64);
   s:='P'; s:=AppendCheckSumCR(s+Chr(h)+Chr(l)+Chr(f));   
  end;
  Encode_PZ:=s;
 end;
 //
 // Encode_TW('23.5');
 //
 function Encode_TW(s:String):String;
 var v:Real; d,h,l:Integer;
 begin
  v:=rVal(ExtractWord(1,s));
  if IsNan(v) then s:='' else begin
   d:=Round(abs(v)*2); l:=d mod 256; h:=d div 256; h:=iSetBitState(h,0,v<0);
   s:='T'; s:=AppendCheckSumCR(s+Chr(h)+Chr(l)+Chr(0));   
  end;
  Encode_TW:=s;
 end;
 //
 // Test check sum and WIKA D1X decoder algorithms.
 // Must be: $72 1 -1 rel 3 rel 0 rel 0.25 rel -1 rel bar 0.125 rel bar 0.1 ok 25 °C 1234 ...
 //
 procedure TestCheckSum;
 begin
  Success('$'+HexB(CalcCheckSum(URL_Decode('MA%00')))+' '
         +Str(Ord(HasValidCheckSum(URL_Decode('MA%00%72'))))+' '
         +Decode_MA(AppendCheckSumCR(URL_Decode('%03%00%8A%41')))+' '
         +Decode_ME(AppendCheckSumCR(URL_Decode('%04%00%1E%41')))+' '
         +Decode_MA(AppendCheckSumCR(URL_Decode('%03%00%00%42')))+' '
         +Decode_ME(AppendCheckSumCR(URL_Decode('%04%00%19%42')))+' '
         +Decode_PZ(AppendCheckSumCR(URL_Decode('P%A7%10%60')))+' '
         +Decode_PZ(AppendCheckSumCR(URL_Decode('P%30%D4%68')))+' '
         +Decode_PK(AppendCheckSumCR(URL_Decode('k%30%D4%00')),0,2)+' '
         +Decode_TW(AppendCheckSumCR(URL_Decode('T%00%32%00')))+' '
         +Decode_KN(AppendCheckSumCR(URL_Decode('K1234')))+' '
         +Decode_PZ(Encode_PZ('3.14 abs psi'))+' '
         +Decode_PZ(Encode_PZ('3.14e5 rel bar'))+' '
         +Decode_PZ(Encode_PZ('-3.14e-5 rel bar'))+' '
  );
 end;
 //
 // Nice tooltip notifier.
 //
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  if Length(aText)>0 then begin
   ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
  end;
 end;
 //
 // Post command to local/remote server.
 //
 procedure PostCmdLocal(cmd:String);
 begin
  DevPostCmd(devMySelf,cmd);
 end;
 procedure PostCmdRemote(cmd:String);
 begin
  Dim_GuiConsoleSend(DevName,cmd);
 end;
 //
 // Check ModelCode is valid or return zero.
 //
 function CheckModelCode(ModelCode:Integer):Integer;
 begin
  if (ModelCode<1) then ModelCode:=0 else
  CheckModelCode:=ModelCode;
 end;
 //
 // Check & return WIKA ModelCode, nonzero if IDN detection was successful.
 //
 function WIKA_ModelCode:Integer;
 begin
  WIKA_ModelCode:=CheckModelCode(WIKA.ModelCode);
 end;
 //
 // Parse IDN string like 'P-10-P', find ModelCode.
 //
 function WIKA_ParseIDN(IDN:String; silent:Boolean):Integer;
 var ModelCode:Integer;
 begin
  ModelCode:=0; IDN:=Trim(IDN);
  if not IsSameText(IDN,si_OFFLINE) then ModelCode:=1;
  if not silent then
  if CheckModelCode(ModelCode)>0 
  then Success('Detected IDN: '+IDN)
  else Problem('Fail detect IDN: '+IDN);
  WIKA_ParseIDN:=ModelCode;
 end;
 //
 // Unique string to identify MenuTools.
 //
 function MenuToolsIdent:String;
 begin
  MenuToolsIdent:=StrFmt('MENU_TOOLS_%s',DevName);
 end;
 //
 // Get text of Tools menu items
 //
 function MenuToolsItem(MenuTools:Integer):String;
 begin
  if MenuTools=0  then MenuToolsItem:='@BrowseHelp             = Вызвать справочное окно (HELP)'   else
  if MenuTools=1  then MenuToolsItem:='@OpenConsole            = Открыть консольное окно драйвера' else
  if MenuTools=2  then MenuToolsItem:='@DebugFlags 3           = Режим отладки: нормальный'        else
  if MenuTools=3  then MenuToolsItem:='@DebugFlags 15          = Режим отладки: ввод-вывод'        else
  if MenuTools=4  then MenuToolsItem:='@DebugFlags 31          = Режим отладки: детальный'         else
  if MenuTools=5  then MenuToolsItem:='@Remote @LoadIni        = Загрузить Параметры из INI файла' else
  if MenuTools=6  then MenuToolsItem:='@Remote @SaveIni        = Сохранить Параметры в  INI файле' else
  if MenuTools=7  then MenuToolsItem:='@Remote @Setting AZ 0   = Response Time = 0  ms'            else
  if MenuTools=8  then MenuToolsItem:='@Remote @Setting AZ $FF = Response Time = 15 ms'            else
  if MenuTools=9  then MenuToolsItem:='@Edit AZ                = Уставка Response Time'            else
  if MenuTools=10 then MenuToolsItem:='@PrintCmdTable          = Печать состояния таблицы команд'  else
  MenuToolsItem:='';
 end;
 //
 // Execute Tools menu commands
 //
 procedure MenuToolsCmnd(MenuTools:Integer);
 var Item:String;
 begin
  Item:=MenuToolsItem(MenuTools);
  if (Pos('@',Item)=1) and (Pos('=',Item)>0) then begin
   Item:=Trim(Copy(Item,1,Pos('=',Item)-1));
   DevPostCmd(DevMySelf,Item);
  end;
  Item:='';
 end;
 //
 // Open MenuTools to edit. Detect number of items automatically.
 //
 procedure MenuToolsOpen;
 var i,err:Integer;
 begin
  i:=0; err:=0;
  if EditState=0 then begin
   err:=err+Pos('?',Edit('(Команда "Инструменты"... '));
   err:=err+Pos('?',Edit(' Что выбираете:'));
   while not IsEmptyStr(MenuToolsItem(i)) do begin
    err:=err+Pos('?',Edit(' '+MenuToolsItem(i)));
    i:=i+1;
   end;
   err:=err+Pos('?',Edit(')MenuList '+MenuToolsIdent));
   if err>0 then Warning('Error initializing MenuList!');
  end else Warning('Cannot edit right now!');
  MenuToolsSelected:=-1;
 end;
 //
 // Find command by id (Acronym).
 //
 function WIKA_Cmd_Find(id:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(id)>0) then
  if UseCmdHash and (WIKA.CmdHash<>0) then begin
   n:=HashList_GetLink(WIKA.CmdHash,id);
  end else begin
   for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(id,WIKA.Cmd.Acronym[i])
   then n:=i;
  end;
  WIKA_Cmd_Find:=n;
 end;
 //
 // Initialize tag refreshment value.
 //
 procedure WIKA_FillTag(tag:Integer; InitVal:Real);
  procedure Process(var R:TTagRef);
  begin
   if (R.tag=tag) then bNul(ShouldRefresh(R.val,InitVal)>0);
  end;
 begin
  if (TypeTag(tag)>0) then begin
   Process(WIKA.POLL_ENABLE);
   Process(WIKA.POLL_RATE);
   Process(WIKA.ERROR_CNT);
   Process(WIKA.ID_NAME);
   Process(WIKA.ID_IDN);
   Process(WIKA.PAR_MA);
   Process(WIKA.PAR_ME);
   Process(WIKA.PAR_PZ);
   Process(WIKA.PAR_PZMU);
   Process(WIKA.PAR_PZM);
   Process(WIKA.PAR_PZU);
   Process(WIKA.PAR_PK);
   Process(WIKA.PAR_PKS);
   Process(WIKA.PAR_KN);
   Process(WIKA.PAR_TW);
   Process(WIKA.PAR_AZ);
  end;
 end;
 //
 // Initialize tags.
 //
 procedure WIKA_FillTags(InitVal:Real);
 begin
  WIKA.POLL_ENABLE.val:=InitVal;
  WIKA.POLL_RATE.val:=InitVal;
  WIKA.ERROR_CNT.val:=InitVal;
  WIKA.ID_NAME.val:=InitVal;
  WIKA.ID_IDN.val:=InitVal;
  WIKA.PAR_MA.val:=InitVal;
  WIKA.PAR_ME.val:=InitVal;
  WIKA.PAR_PZ.val:=InitVal;
  WIKA.PAR_PZMU.val:=InitVal;
  WIKA.PAR_PZM.val:=InitVal;
  WIKA.PAR_PZU.val:=InitVal;
  WIKA.PAR_PK.val:=InitVal;
  WIKA.PAR_PKS.val:=InitVal;
  WIKA.PAR_KN.val:=InitVal;
  WIKA.PAR_TW.val:=InitVal;
  WIKA.PAR_AZ.val:=InitVal;
 end;
 procedure WIKA_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
   InitTag(WIKA.POLL_ENABLE.tag, Prefix+'.POLL_ENABLE',    1); 
   InitTag(WIKA.POLL_RATE.tag,   Prefix+'.POLL_RATE',      2);
   InitTag(WIKA.ERROR_CNT.tag,   Prefix+'.ERROR_CNT',      2);
   InitTag(WIKA.ID_NAME.tag,     Prefix+'.ID_NAME',        3);
   InitTag(WIKA.ID_IDN.tag,      Prefix+'.ID_IDN',         3);
   InitTag(WIKA.PAR_MA.tag,      Prefix+'.PAR_MA',         2);
   InitTag(WIKA.PAR_ME.tag,      Prefix+'.PAR_ME',         2);
   InitTag(WIKA.PAR_PZ.tag,      Prefix+'.PAR_PZ',         2);
   InitTag(WIKA.PAR_PZMU.tag,    Prefix+'.PAR_PZMU',       3);
   InitTag(WIKA.PAR_PZM.tag,     Prefix+'.PAR_PZM',        1);
   InitTag(WIKA.PAR_PZU.tag,     Prefix+'.PAR_PZU',        1);
   InitTag(WIKA.PAR_PK.tag,      Prefix+'.PAR_PK',         2);
   InitTag(WIKA.PAR_PKS.tag,     Prefix+'.PAR_PKS',        1);
   InitTag(WIKA.PAR_KN.tag,      Prefix+'.PAR_KN',         3);
   InitTag(WIKA.PAR_TW.tag,      Prefix+'.PAR_TW',         2);
   InitTag(WIKA.PAR_AZ.tag,      Prefix+'.PAR_AZ',         1);
   WIKA_FillTags(-MaxReal);
  end;
 end;
 //
 // Update DIM tag in server mode.
 //
 procedure UpdateDimTag(tag:Integer);
 var v:Real;
 begin
  if DIM_IsServerMode then
  if (TypeTag(tag)>0) then begin
   v:=GetStampOfTag(tag,_NaN);
   if not IsNan(v) then WIKA_FillTag(tag,v);
   DIM_UpdateTag(tag,'');
  end;
 end;
 //
 // Check if COM port opened.
 //
 function IsComPortOpened:Boolean;
 begin
  IsComPortOpened:=(ComSpace>=0);
 end;
 //
 // Check command number is valid. 
 //
 function IsValidCmdNum(n:Integer):Boolean;
 begin
  IsValidCmdNum:=(n>=1) and (n<=MaxCmdNum);
 end;
 //
 // Check if command (n) is query.
 //
 function IsQueryCmdNum(n:Integer):Boolean;
 begin
  if IsValidCmdNum(n)
  then IsQueryCmdNum:=(n<>cm_AZ)
  else IsQueryCmdNum:=false;
 end;
 //
 // Enable/disable command n.
 //
 procedure EnableCmdNum(n:Integer; Enabled:Boolean);
 begin
  if IsValidCmdNum(n) then WIKA.Cmd.Enabled[n]:=Enabled;
 end;
 //
 // Check if WIKA polling enabled, with warning or not.
 //
 function WIKA_CheckPolling(Warn:Integer):Boolean;
 var flag:Boolean; msg:String;
 begin
  msg:='';
  flag:=iGetTag(WIKA.POLL_ENABLE.tag)>0;
  if (Warn<>0) and not flag then begin
   msg:=StrFmt('Для изменения параметров %s надо включить Опрос.',sGetTag(WIKA.ID_NAME.tag));
   if iAnd(Warn,1)>0 then NiceNotify(msg,30);
   if iAnd(Warn,2)>0 then UnixWBox('Предупреждение',msg,600,20,15);
   if iAnd(Warn,4)>0 then Cron('@run /sw7 unix messagebox "Предупреждение" "'+msg+'" -t 15');
   if iAnd(Warn,8)>0 then Warning(msg);
  end;
  WIKA_CheckPolling:=flag;
  msg:='';
 end;
 //
 // Clear commands.
 //
 procedure WIKA_Clear_Cmd;
 var i:Integer;
 begin
  for i:=1 to MaxCmdNum do begin
   WIKA.Cmd.Acronym[i]:='';
   WIKA.Cmd.ReqHead[i]:='';
   WIKA.Cmd.AnsHead[i]:='';
   WIKA.Cmd.AnsLeng[i]:=0;
   WIKA.Cmd.Comment[i]:='';
   WIKA.Cmd.Enabled[i]:=False;
   WIKA.Cmd.OpData[i]:=_NAN;
   WIKA.Cmd.OpBuff[i]:=_NAN;
   WIKA.Cmd.SimDat[i]:=_NAN;
   WIKA.Cmd.SimStr[i]:='';
   WIKA.Cmd.AoNum[i]:=-1;
   WIKA.Cmd.Tag[i]:=0;
   WIKA.Cmd.Min[i]:=0;
   WIKA.Cmd.Max[i]:=0;
  end;
 end;
 //
 // Initialize commands.
 //
 procedure WIKA_Init_Cmd;
  procedure AddHash(n:Integer; key:String);
  begin
   key:=Trim(key);
   if Length(key)>0 then
   if IsValidCmdNum(n) then
   if WIKA.CmdHash<>0 then
   bNul(HashList_SetLink(WIKA.CmdHash,key,n));
  end;
  procedure Add(n,AoNum:Integer; Info:String);
  begin
   if IsValidCmdNum(n) then begin
    WIKA.Cmd.Acronym[n]:=ExtractWord(1,Info);
    WIKA.Cmd.ReqHead[n]:=URL_Decode(StrReplace(ExtractWord(2,Info),Dump('*'),'',3));
    WIKA.Cmd.AnsHead[n]:=URL_Decode(StrReplace(ExtractWord(3,Info),Dump('*'),'',3));
    WIKA.Cmd.AnsLeng[n]:=Val(ExtractWord(4,Info));
    WIKA.Cmd.Comment[n]:=ExtractWord(5,Info);
    WIKA.Cmd.Enabled[n]:=Val(ExtractWord(6,Info))>0;
    WIKA.Cmd.OpData[n]:=rVal(ExtractWord(7,Info));
    WIKA.Cmd.SimDat[n]:=rVal(ExtractWord(8,Info));
    WIKA.Cmd.Min[n]:=rVal(ExtractWord(9,Info));
    WIKA.Cmd.Max[n]:=rVal(ExtractWord(10,Info));
    WIKA.Cmd.Tag[n]:=FindTag(CrvName(RefAo(AoNum)));
    WIKA.Cmd.AoNum[n]:=AoNum;
    AddHash(n,WIKA.Cmd.Acronym[n]);
   end;
  end;
 begin
  WIKA.Cmd.Num:=1;
  //Command  AoNum      Acronym ReqHead AnsHead Leng Comment             En OpData SimDat Min Max
  Add(cm_SO, -1,        'SO     SO%FF   so%FF   5    SetOperationMode    1  0      0      0   0');
  Add(cm_MA, ao_PAR_MA, 'MA     MA%00   %03     6    GetLoRangeMBA       1  0      0      0   0');
  Add(cm_ME, ao_PAR_ME, 'ME     ME%00   %04     6    GetUpRangeMBE       1  0      0      0   0');
  Add(cm_PZ, ao_PAR_PZ, 'PZ     PZ%00   P       6    GetPressureUnits    1  0      0      0   0');
  Add(cm_PK, ao_PAR_PK, 'PK     PK%00   k       6    GetPressureDigit    1  0      0      0   0');
  Add(cm_TW, ao_PAR_TW, 'TW     TW%00   T       6    GetTemperature      1  0      0      0   0');
  Add(cm_KN, -1,        'KN     KN%00   K       7    GetInstrumentNumber 1  0      0      0   0');
  Add(cm_AZ, ao_PAR_AZ, 'AZ     AZ      a       5    SetResponseTime     1  0      0      0   255');
 end;
 //
 // Print command table
 //
 procedure PrintCmdTable;
 var n:Integer;
 begin
  Success('Command table:');
  Success(StrFmt('%-2s ','Cm')
         +StrFmt('%-2s ','Ao')
         +StrFmt('%-4s ','Acro')
         +StrFmt('%-3s ','Req')
         +StrFmt('%-5s ','Ans')
         +StrFmt('%-5s ','Len')
         +StrFmt('%-17s ','Comment')
         +StrFmt('%-2s ','En')
         +StrFmt('%6s ','OpData')
         +StrFmt('%6s ','SimDat')
         +StrFmt('%6s ','Min')
         +StrFmt('%6s ','Max')
         );
  for n:=1 to MaxCmdNum do
  Success(StrFmt('%-2d ',n)
         +StrFmt('%-2d ',WIKA.Cmd.AoNum[n])
         +StrFmt('%-4s ',WIKA.Cmd.Acronym[n])
         +StrFmt('%-3s ',WIKA.Cmd.ReqHead[n])
         +StrFmt('%-5s ',WIKA.Cmd.AnsHead[n])
         +StrFmt('%-5d ',WIKA.Cmd.AnsLeng[n])
         +StrFmt('%-17.17s ',WIKA.Cmd.Comment[n])
         +StrFmt('%-2d ',Ord(WIKA.Cmd.Enabled[n]))
         +StrFmt('%6g ',WIKA.Cmd.OpData[n])
         +StrFmt('%6g ',WIKA.Cmd.SimDat[n])
         +StrFmt('%6g ',WIKA.Cmd.Min[n])
         +StrFmt('%6g ',WIKA.Cmd.Max[n])
         );
 end;
 //
 // Update ranges/formats by ModelCode.
 //
 function WIKA_UpdateModel(ModelCode:Integer):Integer;
 begin
  ModelCode:=CheckModelCode(ModelCode);
  if ModelCode>0 then begin
  end;
  WIKA_UpdateModel:=ModelCode;
 end;
 //
 // Main command cycle.
 //
 procedure WIKA_CMD_CYCLE;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if IsValidCmdNum(Num)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if WIKA.Cmd.Enabled[Num]
    then i:=MaxCmdNum
    else i:=i+1;
   end; 
   NextCmdNum:=Num;
  end;
  procedure ClearRequest(Num,Status:Integer);
  begin
   WIKA.Com.Req:='';
   WIKA.Com.Ans:='';
   WIKA.Com.Buf:='';
   WIKA.Com.ReqTime:=0;
   WIKA.Com.AnsTime:=0;
   WIKA.Com.Status:=Status;
   WIKA.Cmd.Num:=ValidateCmdNum(Num);
  end;
  function ReqCmd(n:Integer):String;
  var s:String; OpData:Real;
  begin
   s:='';
   if IsValidCmdNum(n) then begin
    s:=WIKA.Cmd.ReqHead[n];
    if (Length(s)>0) then begin
     if (n=cm_AZ) then begin
      OpData:=WIKA.Cmd.OpData[n];
      if IsNan(OpData) then s:='' else s:=s+Chr(Round(OpData));
     end;
    end;
    if (Length(s)>0) then s:=AppendCheckSumCR(s);
   end;
   ReqCmd:=s;
   s:='';
  end;
  procedure PrepareRequest(n:Integer);
  begin
   if IsValidCmdNum(n) then begin
    WIKA.Com.Req:=ReqCmd(n);
   end;
  end;
  procedure UpdateIDN(data:String);
  var v:Real;
  begin
   if not IsEmptyStr(data) then begin
    WIKA.ModelCode:=WIKA_UpdateModel(WIKA_ParseIDN(data,iGetTag(WIKA.POLL_ENABLE.tag)=0));
    if (WIKA_ModelCode=0) then data:=si_OFFLINE;
    if IsEmptyStr(data) then data:=si_OFFLINE;
    bNul(sSetTag(WIKA.ID_IDN.tag,data));
    if ShouldRefresh(WIKA.ID_IDN.val,GetStampOfTag(WIKA.ID_IDN.tag,_NaN))>0
    then UpdateDimTag(WIKA.ID_IDN.tag);
   end;
  end;
  procedure UpdateCmdValue(n:Integer; value:Real);
  var aonum,tag:Integer;
  begin
   if IsValidCmdNum(n) and not IsNan(value) and not IsInf(value) then begin
    tag:=WIKA.Cmd.Tag[n]; aonum:=WIKA.Cmd.AoNum[n];
    if TypeTag(tag)=1 then begin
      bNul(iSetTag(tag,Round(value)));
      UpdateDimTag(tag);
    end;
    if TypeTag(tag)=2 then begin
     bNul(rSetTag(tag,value));
     UpdateDimTag(tag);
    end;
    if aonum>=0 then UpdateAo(aonum,time,value);
    WIKA.Cmd.OpData[n]:=value;
   end;
  end;  
  procedure DecodeData(n:Integer; data:String);
  var ans:String; v:Real; i:Integer;
  begin
   ans:='';
   if IsValidCmdNum(n) then begin
    if Length(data)>0 then begin
     if (n=cm_SO) then begin
      ans:=Decode_SO(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       UpdateIDN(WIKA.ModelName);
       EnableCmdNum(n,false);
      end;
     end else
     if (n=cm_MA) then begin
      ans:=Decode_MA(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then begin
        EnableCmdNum(n,false);
        UpdateCmdValue(n,v);
       end;
      end;
     end else
     if (n=cm_ME) then begin
      ans:=Decode_ME(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then begin
        EnableCmdNum(n,false);
        UpdateCmdValue(n,v);
       end;
      end;
     end else
     if (n=cm_PZ) then begin
      ans:=Decode_PZ(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then begin
        bNul(sSetTag(WIKA.PAR_PZMU.tag,SkipWords(1,ans)));
        bNul(iSetTag(WIKA.PAR_PZM.tag,WordIndex(ExtractWord(2,ans),list_Measuring)-1));
        bNul(iSetTag(WIKA.PAR_PZU.tag,WordIndex(ExtractWord(3,ans),list_PhysUnits)-1));
        UpdateAo(ao_PAR_PZM,time,iGetTag(WIKA.PAR_PZM.tag));
        UpdateAo(ao_PAR_PZU,time,iGetTag(WIKA.PAR_PZU.tag));
        UpdateCmdValue(n,v);
       end;
      end;
     end else
     if (n=cm_PK) then begin
      ans:=Decode_PK(data,rGetTag(WIKA.PAR_MA.tag),rGetTag(WIKA.PAR_ME.tag));
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then begin
        bNul(iSetTag(WIKA.PAR_PKS.tag,Val(ExtractWord(2,ans))));
        UpdateAo(ao_PAR_PKS,time,iGetTag(WIKA.PAR_PKS.tag));
        UpdateCmdValue(n,v);
       end;
      end;
     end else
     if (n=cm_KN) then begin
      ans:=Decode_KN(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       Success('Received KN = '+ans);
       bNul(sSetTag(WIKA.PAR_KN.tag,ans));
       UpdateDimTag(WIKA.PAR_KN.tag);
       EnableCmdNum(n,false);
      end;
     end else
     if (n=cm_TW) then begin
      ans:=Decode_TW(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then begin
        UpdateCmdValue(n,v);
       end;
      end;
     end else
     if (n=cm_AZ) then begin
      ans:=Decode_AZ(data);
      if IsSameText(ExtractWord(1,ans),es_ERROR) then begin
       Trouble(ans);
      end else begin
       v:=rVal(ExtractWord(1,ans));
       if not IsNan(v) then UpdateCmdValue(n,v);
       Success('Received AZ = '+ans);
       EnableCmdNum(n,false);
      end;
     end;
    end;
   end;
   ans:='';
  end;
  function HandleRequest(n:Integer; Ans:String):Boolean;
  var Handled:Boolean;
  begin
   Handled:=false;
   if IsValidCmdNum(n) and (Length(Ans)>0) then begin
    ViewImp('COM < '+URL_Packed(Ans));
    DecodeData(n,Ans);
    Handled:=True;
   end;
   HandleRequest:=Handled;
  end;
  procedure PrintDetails;
  begin
   if iAnd(DebugFlags,dfDetails)>0 then Details('Run '+Str(RunCount)
    +', Cmd '+WIKA.Cmd.Acronym[WIKA.Cmd.Num]
    +', Enb '+Str(Ord(WIKA.Cmd.Enabled[WIKA.Cmd.Num]))
    +', Sta '+Str(WIKA.Com.Status)
    +', Req '+Trim(WIKA.Com.Req)
    +', Ans '+Trim(WIKA.Com.Ans)
    +', Buf '+Trim(WIKA.Com.Buf)
    );
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsNan(WIKA.Cmd.OpBuff[n]) then begin
    if not IsQueryCmdNum(n) then begin
     WIKA.Cmd.OpData[n]:=WIKA.Cmd.OpBuff[n];
     EnableCmdNum(n,true);
    end;
    WIKA.Cmd.OpBuff[n]:=_NAN;
   end;
  end;
  procedure EnableIDN;
  begin
   EnableCmdNum(cm_SO,true);
   EnableCmdNum(cm_MA,true);
   EnableCmdNum(cm_ME,true);
   EnableCmdNum(cm_KN,true);
   EnableCmdNum(cm_AZ,true);
  end;
  procedure ExecuteIdleActions;
  begin
   UpdateOpData;
   if (WIKA_ModelCode=0) then EnableIDN;
  end;
  procedure DimRefreshTag(var R:TTagRef);
  var tag:Integer; v:Real;
  begin
   tag:=R.tag; v:=GetStampOfTag(tag,_NaN);
   if not IsNaN(v) then if ShouldRefresh(R.val,v)>0 then UpdateDimTag(tag);
  end;
 begin
  //
  // COM port communication handler
  //
  if not DIM_IsClientMode then begin
   if IsComPortOpened then begin
    if iGetTag(WIKA.POLL_ENABLE.tag)>0 then begin
     WIKA.Cmd.Num:=ValidateCmdNum(WIKA.Cmd.Num);
     if iAnd(DebugFlags,dfDetails)>0 then PrintDetails;
     if WIKA.Com.Status=st_NoReq then begin
      ExecuteIdleActions;
      ClearRequest(WIKA.Cmd.Num,st_NoReq);
      if mSecNow>=WIKA.Com.LastPoll+WIKA.Com.PollPeriod then begin
       if WIKA.Cmd.Enabled[WIKA.Cmd.Num] then PrepareRequest(WIKA.Cmd.Num);
       if Length(WIKA.Com.Req)>0 then begin
        PurgeComPort;
        if ComWrite(WIKA.Com.Req) then begin
         ViewExp('COM > '+URL_Packed(WIKA.Com.Req));
         WIKA.Com.Status:=st_WaitAns;
         WIKA.Com.ReqTime:=mSecNow;
        end else begin
         Trouble('Could not send request '+URL_Packed(WIKA.Com.Req));
         ClearRequest(NextCmdNum(WIKA.Cmd.Num),st_NoReq);
        end;
        if WIKA.Com.Status=st_WaitAns
        then WIKA.Com.LastPoll:=mSecNow;
       end else ClearRequest(NextCmdNum(WIKA.Cmd.Num),st_NoReq);
      end;
     end else
     if WIKA.Com.Status=st_WaitAns then begin
      WIKA.Com.Ans:=WIKA.Com.Ans+ComRead(255);
      if (Length(WIKA.Com.Ans)>=WIKA.Cmd.AnsLeng[WIKA.Cmd.Num]) then begin
       bNul(HandleRequest(WIKA.Cmd.Num,WIKA.Com.Ans));
       WIKA.Com.PollRate:=WIKA.Com.PollRate+1;
       WIKA.Com.Status:=st_WaitGap;
       WIKA.Com.AnsTime:=mSecNow;
      end else
      if mSecNow>WIKA.Com.ReqTime+WIKA.Com.TimeOut then begin
       WIKA.Com.Status:=st_TimeOut;
      end;
     end;
     if WIKA.Com.Status=st_WaitGap then begin
      if mSecNow>=WIKA.Com.AnsTime+WIKA.Com.TimeGap then begin
       if not IsQueryCmdNum(WIKA.Cmd.Num) then EnableCmdNum(WIKA.Cmd.Num,false);
       ClearRequest(NextCmdNum(WIKA.Cmd.Num),st_NoReq);
      end;
     end;
     if WIKA.Com.Status=st_TimeOut then begin
      if WIKA.Cmd.Num=cm_SO then UpdateIDN(si_OFFLINE);
      Failure(WIKA.ecTimeOut,'TimeOut on command '+WIKA.Cmd.Acronym[WIKA.Cmd.Num]+' request '+Trim(WIKA.Com.Req));
      ClearRequest(NextCmdNum(WIKA.Cmd.Num),st_NoReq);
      EnableIDN;
     end;
    end else begin
     ClearRequest(NextCmdNum(0),st_NoReq);
     UpdateIDN(si_OFFLINE);
     EnableIDN;
    end;
   end;
  end;
  //
  // Update DIM services
  //
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then WIKA_FillTags(-MaxReal);
   DimRefreshTag(WIKA.POLL_ENABLE);
   DimRefreshTag(WIKA.POLL_RATE);
   DimRefreshTag(WIKA.ERROR_CNT);
   DimRefreshTag(WIKA.ID_IDN);
   DimRefreshTag(WIKA.PAR_MA);
   DimRefreshTag(WIKA.PAR_ME);
   DimRefreshTag(WIKA.PAR_PZ);
   DimRefreshTag(WIKA.PAR_PZMU);
   DimRefreshTag(WIKA.PAR_PZM);
   DimRefreshTag(WIKA.PAR_PZU);
   DimRefreshTag(WIKA.PAR_PK);
   DimRefreshTag(WIKA.PAR_PKS);
   DimRefreshTag(WIKA.PAR_KN);
   DimRefreshTag(WIKA.PAR_TW);
   DimRefreshTag(WIKA.PAR_AZ);
  end;
 end;
 //
 // Xor bit on click (local version).
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // Xor bit on click (remote version).
 //
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DIM_GuiClickSend(DIM_GuiClickBuff+'NewValue='+Str(iXor(iGetTag(tag),XorMask)));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // GUI Handler to process user input...
 //
 procedure WIKA_GUI_POLL;
 var ClickCurve:Integer; s:String;
  procedure ClickEditWikaCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if ClickTag=WIKA.Cmd.Tag[n] then begin
    DevPostCmd(DevMySelf,'@Edit '+WIKA.Cmd.Acronym[n]);
    bNul(Voice(snd_Click));
   end;
  end;
  procedure DevSendWikaCmd(n:Integer; prefix,data:String; min,max:Real);
  var v:Real;
  begin
   if IsValidCmdNum(n) then
   if not IsEmptyStr(data) then
   if TypeTag(WIKA.Cmd.Tag[n])=1 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmd(DevMySelf,prefix+WIKA.Cmd.Acronym[n]+' '+Str(Round(v)));
   end else
   if TypeTag(WIKA.Cmd.Tag[n])=2 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmd(DevMySelf,prefix+WIKA.Cmd.Acronym[n]+' '+Str(v));
   end else
   if TypeTag(WIKA.Cmd.Tag[n])=3 then begin
    DevPostCmd(DevMySelf,prefix+WIKA.Cmd.Acronym[n]+' '+Trim(data));
   end;
  end;
  procedure CheckEditWikaCmd(n:Integer; prefix:String);
  var s:String;
  begin
   s:='';
   if IsValidCmdNum(n) then
   if CheckEditTag(WIKA.Cmd.Tag[n],s) then begin
    DevSendWikaCmd(n,prefix+'@Setting ',s,WIKA.Cmd.Min[n],WIKA.Cmd.Max[n]);
    sNul(Edit(''));
   end;
   s:='';
  end;
  procedure RefreshSetting(n:Integer; var S:TTagRef);
  begin
   if IsValidCmdNum(n) then begin
    if TypeTag(S.tag)>0 then
    if ShouldRefresh(S.val,GetStampOfTag(S.tag,0))>0 then
    DevPostCmd(DevMySelf,'@Setting '+WIKA.Cmd.Acronym[n]+' '+TagAsText(S.tag));
   end;
  end;
 begin
  s:='';
  DIM_GuiClickBuff:='';
  //
  // Handle user mouse/keyboard clicks...
  // ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,KeyUp)
  // ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  //
  if ClickWhat<>0 then
  repeat
   //
   // Copy GUI click to DIM buffer for remote execution.
   //
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   //
   // Handle MouseDown/KeyDown
   //
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    //
    // Handle Left mouse button click
    //
    if (ClickButton=VK_LBUTTON) then begin
     //
     // Handle local clicks
     //
     if ClickIsLocal then begin
      //
      // Handle sensor clicks...
      //
      if IsSameText(ClickSensor,'HELP') then begin
       Cron('@Browse '+DaqFileRef(AdaptFileName(ReadIni('[DAQ] HelpFile')),'.htm'));
       bNul(Voice(snd_Click));
      end;
      if (ClickTag<>0) then begin
       if (ClickTag=WIKA.ID_IDN.tag) then begin
        DevPostCmd(DevMySelf,'@Remote @Setting IDN');
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=DIM_GuiClickFallBackModeTag) then begin
        if DIM_IsServerMode
        then ClickTagXorLocal(DIM_GuiClickFallBackModeTag,1)
        else ShowTooltip('text "Режим DIM FallBack применим только к серверу."  preset stdNotify delay 15000');
       end;
       ClickTagXorRemote(WIKA.POLL_ENABLE.tag,1);
       //if (ClickTag=WIKA.PAR_AZ.tag) then StartEditTag(WIKA.PAR_AZ.tag,'Response time, 1..255');
       if (ClickTag=WIKA.Cmd.Tag[cm_AZ])
       then begin
        if WIKA_CheckPolling(1) then begin
         ClickEditWikaCmd(cm_AZ);
        end;
       end;
      end;
      //
      // Select Plot & Tab windows by curve...
      //
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      //
      // Console commands: @url_encoded_sensor ...
      //
      if (strFetch(ClickSensor,1)='@') then begin
       DevPostCmd(devMySelf,url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     //
     // Handle remote clicks comes from DIM via @DimGuiClick message.
     // @DimGuiClick default handler decode and write events to FIFO,
     // so we can find it as clicks and can handle it in usual way.
     //
     if ClickIsRemote then begin
      //
      // Show time difference.
      //
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      //
      // Handle remote console commands...
      //
      s:=Dim_GuiConsoleRecv(DevName,'');
      if (strFetch(s,1)='@') then DevPostCmd(devMySelf,s);
      //
      // Handle remote sensor clicks...
      //
      if TypeTag(ClickTag)>0 then begin
       s:=ClickParams('NewValue');
       if Length(s)>0 then begin
        if ClickTag=WIKA.POLL_ENABLE.tag then UpdateTag(ClickTag,s,0,1);
       end;
      end;
     end;
    end;
   end;
  until (ClickRead=0);
  //
  // Edit handling...
  //
  if EditState=ef_Done then begin
   //if CheckEditTag(WIKA.PAR_AZ.tag,s) then DevPostCmd(devMySelf,'@Remote @Setting AZ '+s);
   CheckEditWikaCmd(cm_AZ,'@Remote ');
   //
   // TOOLS menu.
   //
   if IsSameText(ExtractWord(1,Edit('?ans 0')),MenuToolsIdent) then begin
    if Val(ExtractWord(2,Edit('?ans 0')))=1 then begin
     MenuToolsSelected:=Val(Edit('?ans 1'));
     if MenuToolsSelected>=0 then
     if (MenuToolsSelected>80) then begin
      DevPostCmd(devMySelf,'@MenuToolsConfirmation '+UpCaseStr(StrReplace(StrReplace(
                 MenuToolsItem(MenuToolsSelected),'  ',Dump(' '),3),'  ',Dump(' '),3)));
     end else begin
      MenuToolsCmnd(MenuToolsSelected);
      MenuToolsSelected:=-1;
     end;
    end;
    sNul(Edit(''));
   end;
   //
   // TOOLS menu (after confirmation).
   //
   if IsSameText(ExtractWord(1,Edit('?ans 0')),'YesNo_'+MenuToolsIdent) then begin
    if MenuToolsSelected>=0 then
    if Val(ExtractWord(2,Edit('?ans 0')))=6 then MenuToolsCmnd(MenuToolsSelected);
    MenuToolsSelected:=-1;
    sNul(Edit(''));
   end;
   //
   // Warning, Information dialog completion.
   //
   if IsSameText(ExtractWord(1,Edit('?ans 0')),'Warning') then sNul(Edit(''));
   if IsSameText(ExtractWord(1,Edit('?ans 0')),'Information') then sNul(Edit(''));
  end;
  if EditState=ef_Done then begin
   Problem('Uncompleted edit detected!');
   sNul(Edit(''));
  end else
  if iAnd(EditState,ef_ErrorFound)<>0 then begin
   Problem('Edit error detected!');
   sNul(Edit(''));
  end;
  DIM_GuiClickBuff:='';
  s:='';
 end;
 //
 // Handle data requests in simulation mode. 
 //
 procedure WIKA_Sim(Req:String);
 var cmd,arg:String; rt:Integer;
 begin
  cmd:=''; arg:='';
  if Length(Req)>0 then begin
   ViewImp('COM < '+URL_Encode(Req));
   if Length(Req)>0 then begin
    //
    // Check request checksum and CR
    //
    if (StrFetch(Req,Length(Req))=Chr(13)) then
    if HasValidCheckSum(Copy(Req,1,Length(Req)-1)) then
    if DebugFlagEnabled(dfDetails) then Details('Request CheckSum/CR is valid');
    //
    // Change operating mode to polling mode
    //
    if Req=AppendCheckSumCR(URL_Decode('SO%FF')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('so%FF'));
    end;
    //
    // Read out DMU’s lower limit of range (MBA)
    //
    if Req=AppendCheckSumCR(URL_Decode('MA%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('%03%00%8A%41'));
    end;
    //
    // Read out DMU’s upper limit of range (MBE)
    //
    if Req=AppendCheckSumCR(URL_Decode('ME%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('%04%00%1E%41'));
    end;
    //
    // Read out pressure value as physical unit
    //
    if Req=AppendCheckSumCR(URL_Decode('PZ%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('P%30%D4%68'));
     if (WIKA.Cmd.SimStr[cm_PZ]<>'')
     then WIKA.Com.Ans:=Encode_PZ(WIKA.Cmd.SimStr[cm_PZ]);
    end;
    //
    // Read out pressure value in digits
    //
    if Req=AppendCheckSumCR(URL_Decode('PK%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('k%30%D4%00'));
    end;
    //
    // Read out temperature value
    //
    if Req=AppendCheckSumCR(URL_Decode('TW%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('T%00%32%00'));
     if (WIKA.Cmd.SimStr[cm_TW]<>'')
     then WIKA.Com.Ans:=Encode_TW(WIKA.Cmd.SimStr[cm_TW]);
    end;
    //
    // Read out instrument number
    //
    if Req=AppendCheckSumCR(URL_Decode('KN%00')) then begin
     WIKA.Com.Ans:=AppendCheckSumCR(URL_Decode('K1234'));
     if (WIKA.Cmd.SimStr[cm_KN]<>'')
     then WIKA.Com.Ans:=AppendCheckSumCR(StrFmt('K%4.4s',URL_Decode(WIKA.Cmd.SimStr[cm_KN])));
    end;
    //
    // Response time in polling mode
    //
    if (Length(Req)=5) then if (Copy(Req,1,2)='AZ') then
    if HasValidCheckSum(Copy(Req,1,Length(Req)-1)) then begin
     rt:=Ord(StrFetch(Req,3));
     WIKA.Com.Ans:=AppendCheckSumCR('az'+Chr(rt));
    end;
    //
    // Send reply to requests
    //
    if Length(WIKA.Com.Ans)>0 then begin
     if ComWrite(WIKA.Com.Ans)
     then ViewExp('COM > '+URL_Encode(WIKA.Com.Ans))
     else Trouble('Could not send '+URL_Encode(WIKA.Com.Ans));
    end;
   end;
  end;
  WIKA.Com.Req:='';
  WIKA.Com.Ans:='';
 end;
 //
 // WIKA cleanup.
 //
 procedure WIKA_Clear;
 begin
  WIKA_Clear_Cmd;
  WIKA.CmdHash:=0;
  WIKA.Com.Req:='';
  WIKA.Com.Ans:='';
  WIKA.Com.Buf:='';
  WIKA.ModelCode:=0;
  WIKA.ModelName:='';
 end;
 //
 // WIKA initialization.
 //
 procedure WIKA_Init;
 var i:Integer;
 begin
  //
  // Register error codes
  //
  WIKA.ecTimeOut:=RegisterErr(progname+': TimeOut');
  //
  // Initialize variables
  //
  MenuToolsSelected:=-1;
  WIKA.Com.ReqTime:=0;
  WIKA.Com.AnsTime:=0;
  WIKA.Com.LastPoll:=0;
  WIKA.Com.Status:=st_NoReq;
  //
  // Initialize command table.
  //
  WIKA.CmdHash:=HashList_Init(0);
  WIKA_Init_Cmd;
  //
  // Read ini file variables
  //
  WIKA.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  if WIKA.Simulator
  then Success('Run simulator mode.')
  else Success('Run as driver mode.');
  WIKA.Com.Port:=iValDef(ReadIni('ComPort'),0);
  Success('ComPort = '+Str(WIKA.Com.Port));
  WIKA.Com.TimeOut:=iValDef(ReadIni('ComTimeOut'),250);
  Success('ComTimeOut = '+Str(WIKA.Com.TimeOut));
  WIKA.Com.TimeGap:=iValDef(ReadIni('ComTimeGap'),10);
  Success('ComTimeGap = '+Str(WIKA.Com.TimeGap));
  WIKA.Com.PollPeriod:=iValDef(ReadIni('PollPeriod'),100);
  Success('PollPeriod = '+Str(WIKA.Com.PollPeriod));
  WIKA.ModelName:=ReadIniVar('ModelName',28);
  Success('ModelName = '+WIKA.ModelName);
  WIKA.ModelCode:=WIKA_UpdateModel(WIKA_ParseIDN(WIKA.ModelName,false));
  //
  // Initialize tags...
  //
  if not WIKA.Simulator then begin
   WIKA_InitTags(ReadIni('tagPrefix'));
   bNul(sSetTag(WIKA.ID_IDN.tag,WIKA.ModelName));
  end;
  //
  // Initialize COM port
  //
  if not DIM_IsClientMode then
  if ComOpen('[SerialPort-COM'+Str(WIKA.Com.Port)+']')
  then Success('COM port initialized.')
  else Trouble('COM port failed.');
 end;
 //
 // WIKA finalization.
 //
 procedure WIKA_Free;
 begin
  bNul(ComClose);
  if (WIKA.CmdHash<>0) then begin
   bNul(HashList_Free(WIKA.CmdHash));
   WIKA.CmdHash:=0;
  end;
 end;
 //
 // WIKA polling.
 //
 procedure WIKA_Poll;
 begin
  if WIKA.Simulator then begin
   if IsComPortOpened then begin
    WIKA.Com.Req:=WIKA.Com.Req+ComRead(255);
    if (Length(WIKA.Com.Req)>=5) then
    if (StrFetch(WIKA.Com.Req,Length(WIKA.Com.Req))=Chr(13))
    then WIKA_Sim(WIKA.Com.Req);
   end;
  end else begin
   WIKA_GUI_POLL;
   if not DIM_IsClientMode then begin
    if SysTimer_Pulse(1000)>0 then begin
     bNul(rSetTag(WIKA.ERROR_CNT.tag,GetErrCount(-1)));
     UpdateDimTag(WIKA.ERROR_CNT.tag);
     UpdateAo(ao_ERROR_CNT,time,GetErrCount(-1));
     bNul(rSetTag(WIKA.POLL_RATE.tag,WIKA.Com.PollRate));
     UpdateDimTag(WIKA.POLL_RATE.tag);
     UpdateAo(ao_POLL_RATE,time,WIKA.Com.PollRate);
     WIKA.Com.PollRate:=0;
    end;
    if mSecNow-FixmSecNow>DelayOnStart then
    WIKA_CMD_CYCLE;
   end;
  end;
 end;
 //
 // Clear user application strings...
 //
 procedure ClearApplication;
 begin
  WIKA_Clear;
 end;
 //
 // User application Initialization...
 //
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  WIKA_Init;
  RunStartupScript;
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  cmd_DimTagUpdate          := RegisterStdInCmd('@DimTagUpdate',          '');
  cmd_Simulate              := RegisterStdInCmd('@Simulate',              '');
  cmd_PrintCmdTable         := RegisterStdInCmd('@PrintCmdTable',         '');
  cmd_Setting               := RegisterStdInCmd('@Setting',               '');
  cmd_Remote                := RegisterStdInCmd('@Remote',                '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',                  '');
  cmd_BrowseHelp            := RegisterStdInCmd('@BrowseHelp',            '');
  cmd_OpenConsole           := RegisterStdInCmd('@OpenConsole',           '');
  cmd_MenuToolsOpen         := RegisterStdInCmd('@MenuToolsOpen',         '');
  cmd_LoadIni               := RegisterStdInCmd('@LoadIni',               '');
  cmd_SaveIni               := RegisterStdInCmd('@SaveIni',               '');
  cmd_MenuToolsConfirmation := RegisterStdInCmd('@MenuToolsConfirmation', '');
  TestCheckSum;
 end;
 //
 // User application Finalization...
 //
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  WIKA_Free;
 end;
 //
 // User application Polling...
 //
 procedure PollApplication;
 begin
  WIKA_Poll;
 end;
 //
 // Process data coming from standard input...
 //
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid,tag,n,m:Integer; v:Real;
  procedure SettingCmd(n:Integer; v:Real);
  begin
   if not IsNan(v) then
   if IsValidCmdNum(n) then begin
    Success(cmd+' '+WIKA.Cmd.Acronym[n]+' '+Str(v));
    WIKA.Cmd.OpBuff[n]:=v;
    EnableCmdNum(n,true);
   end;
  end;
  procedure CheckEditCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(WIKA.Cmd.Tag[n])>=1 then
   if TypeTag(WIKA.Cmd.Tag[n])<=2 then
   if IsSameText(ExtractWord(1,arg),WIKA.Cmd.Acronym[n])
   or IsSameText(ExtractWord(1,arg),WIKA.Cmd.AnsHead[n]) then
   StartEditTag(WIKA.Cmd.Tag[n],WIKA.Cmd.Acronym[n]+': '+WIKA.Cmd.Comment[n]);
  end;
  procedure UpdateWikaCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(WIKA.Cmd.Tag[n])>=1 then
   if TypeTag(WIKA.Cmd.Tag[n])<=2 then
   DevPostCmd(DevMySelf,'@Setting '+WIKA.Cmd.Acronym[n]+' '+ TagAsText(WIKA.Cmd.Tag[n]));
  end;
  procedure OpenConsole(s:String);
  begin
   if IsEmptyStr(s) then s:=DevName;
   bNul(WinShow(ParamStr('Console '+s)));
   bNul(WinSelect(ParamStr('Console '+s)));
  end;
  procedure UpdateAoByTag(AoNum,tag:Integer);
  begin
   if (AoNum>=0) then begin
    if TypeTag(tag)=1 then UpdateAo(AoNum,time,iGetTag(tag));
    if TypeTag(tag)=2 then UpdateAo(AoNum,time,rGetTag(tag));
   end;
  end;
  procedure UpdateSettingsAfterLoadIni;
  begin
   if (iGetTag(WIKA.POLL_ENABLE.tag)<>0) then begin
    DevPostCmd(DevMySelf,'@Setting AZ '+TagAsText(WIKA.PAR_AZ.tag));
   end;
  end;
  procedure CheckEdit(var n:Integer; s:String);
  begin
   if Pos('?',s)>0 then n:=n+1;
  end;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if TypeTag(tag)>0 then begin
      if tag=WIKA.POLL_RATE.tag then UpdateAoByTag(ao_POLL_RATE,tag);
      if tag=WIKA.ERROR_CNT.tag then UpdateAoByTag(ao_ERROR_CNT,tag);
      if tag=WIKA.PAR_MA.tag    then UpdateAoByTag(ao_PAR_MA,tag);
      if tag=WIKA.PAR_ME.tag    then UpdateAoByTag(ao_PAR_ME,tag);
      if tag=WIKA.PAR_PZ.tag    then UpdateAoByTag(ao_PAR_PZ,tag);
      if tag=WIKA.PAR_PZM.tag   then UpdateAoByTag(ao_PAR_PZM,tag);
      if tag=WIKA.PAR_PZU.tag   then UpdateAoByTag(ao_PAR_PZU,tag);
      if tag=WIKA.PAR_PK.tag    then UpdateAoByTag(ao_PAR_PK,tag);
      if tag=WIKA.PAR_PKS.tag   then UpdateAoByTag(ao_PAR_PKS,tag);
      if tag=WIKA.PAR_TW.tag    then UpdateAoByTag(ao_PAR_TW,tag);
      if tag=WIKA.PAR_AZ.tag    then UpdateAoByTag(ao_PAR_AZ,tag);
     end;
    end;
   end else
   {
   @Setting AZ 0
   }
   if (cmdid=cmd_Setting) then begin
    if not WIKA.Simulator and not DIM_IsClientMode then begin
     n:=WIKA_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=rVal(ExtractWord(2,arg));
      if (n=cm_AZ) then SettingCmd(n,v);
     end;
    end;
    Data:='';
   end else 
   {
   @Edit AZ
   }
   if (cmdid=cmd_Edit) then begin
    if not WIKA.Simulator then begin
     n:=WIKA_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      if WIKA_CheckPolling(1) then begin
       CheckEditCmd(cm_AZ);
      end;
     end;
    end;
    Data:='';
   end else
   {
   @Simulate PZ 1.5 abs bar
   @Simulate KN 1234
   @Simulate TW 25
   @Simulate AZ 0
   -@Simulate MA 0.0
   -@Simulate ME 2.0
   -@Simulate PK 1.5
   }
   if (cmdid=cmd_Simulate) then begin
    if WIKA.Simulator then begin
     n:=WIKA_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      WIKA.Cmd.SimDat[n]:=rValDef(ExtractWord(2,arg),WIKA.Cmd.SimDat[n]);
      Success(cmd+' '+WIKA.Cmd.Acronym[n]+' '+Str(WIKA.Cmd.SimDat[n]));
      WIKA.Cmd.SimStr[n]:=Trim(SkipWords(1,arg));
     end;
    end;
    Data:='';
   end else
   {
   @MenuToolsOpen
   }
   if (cmdid=cmd_MenuToolsOpen) then begin
    MenuToolsOpen;
    Data:='';
   end else
   {
   @MenuToolsConfirmation
   }
   if (cmdid = cmd_MenuToolsConfirmation) then begin
    if not WIKA.Simulator then begin
     if MenuToolsSelected>=0 then
     if EditState=0 then begin
      if Pos('?',Edit('(Вы действительно хотите:')
                +Edit(' ')
                +Edit(' '+arg)
                +Edit(' ')
                +Edit(' Эта операция может вызвать проблемы!')
                +Edit(')YesNo YesNo_'+MenuToolsIdent))>0
      then Problem('Could not initialize dialog!');
     end else Problem('Could not initialize dialog!');
    end;
    Data:='';
   end else
   {
   @BrowseHelp
   }
   if (cmdid=cmd_BrowseHelp) then begin
    rNul(Eval('@Global @Async @Silent @Run WebBrowser '+DaqFileRef(AdaptFileName(ReadIni('HelpFile')),'.htm')));
    Data:='';
   end else
   {
   @Remote
   }
   if (cmdid=cmd_Remote) then begin
    if not IsEmptyStr(arg) then PostCmdRemote(Trim(arg));
    Data:='';
   end else
   {
   @OpenConsole
   }
   if (cmdid=cmd_OpenConsole) then begin
    OpenConsole(arg);
    Data:='';
   end else
   {
   @PrintCmdTable
   }
   if (cmdid=cmd_PrintCmdTable) then begin
    PrintCmdTable;
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    if not WIKA.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
     UpdateSettingsAfterLoadIni;
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    if not WIKA.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
