#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
## Script to call before/after DAQ config start/stop.   ##
##########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

function preprocessor_action(){
 local nsel="$scriptbase";
 ../utility/daqsys.logon.sh;
 delete_files ${nsel}_selector.txt;
 local head="$(dirname $CRW_DAQ_SYS_TITLE | tr '#' '_')";
 local caption="$head - DUH: SELECT CONFIGURATION …";
 unix grun $SHELL -c "sleep 1; unix setforegroundwindow \"$caption\"; unix movewindow \"$caption\" 167 0;";
 unix zenity --title "$caption" \
      --list --modal --timeout 30 --width 300 --height 300 \
      --text "<big><b>Select number of Heaters for UniHeat.</b></big>" \
      --radiolist --column "Check" --column "Number" --column "Of Heaters" \
      --separator " " --print-column=2 \
      true   "8"  "Heaters" \
      false  "10" "Heaters" \
      false  "16" "Heaters" \
      false  "20" "Heaters" \
      false  "24" "Heaters" \
      false  "30" "Heaters" \
      false  "32" "Heaters" \
      > ${nsel}_selector.txt;
 local item="$(unix cat ${nsel}_selector.txt)";
 if [ -z "$item" ]; then item="8"; fi;
 echo "Selected item = $item";
 export "UH_NumHeaters=$item";
 ../config/duh_simulator_make.sh $item;
 delete_files ${nsel}_selector.txt;
};

function postprocessor_action(){
 ../utility/daqsys.logout.sh;
 ../clear.sh;
 ../config/demo_uniheater_zero.sh;
 delete_files ../config/duh_simulator.cfg;
};

function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -fv "$1"; fi;
  shift;
 done;
};

function daq_preprocessor(){
 local daqcfg="$(basename $CRW_DAQ_CONFIG_FILE)";
 unix tooltip-notifier text "$(date +%Y.%m.%d-%H:%M:%S): PreProcessor is running on $(hostname) …" preset stdInfo delay 7000;
 if [ -n "$CRW_DAQ_CONFIG_HOME_DIR" ]; then
  pushd "$CRW_DAQ_CONFIG_HOME_DIR" && preprocessor_action;
  popd;
 fi;
};

function daq_postprocessor(){
 echo "PostProcessor is running on $(hostname) …";
 unix tooltip-notifier text "$(date +%Y.%m.%d-%H:%M:%S): PostProcessor is running on $(hostname) …" preset stdInfo delay 7000;
 if [ -n "$CRW_DAQ_CONFIG_HOME_DIR" ]; then
  pushd "$CRW_DAQ_CONFIG_HOME_DIR" && postprocessor_action;
  popd;
 fi;
};

function daq_processor(){
 # target hosts to run script
 local wanted_hosts="demo-daq-pc";
 # uncomment next line to run on all hosts
 wanted_hosts="$wanted_hosts $(hostname)";
 if word_is_in_list "$(hostname)" "$wanted_hosts"; then
  case $1 in
   pre)  daq_preprocessor;  ;;
   post) daq_postprocessor; ;;
   *) ;;
  esac;
 else
  echo "Skip $1processor on host $(hostname).";
 fi;
};

function main(){
 if [ -z "$CRW_DAQ_SYS_HOME_DIR" ]; then
  fatal 1 "$(langstr ru "Сценарий вызывается только из CRW-DAQ." en "Script should be called from CRW-DAQ only.").";
 fi;
 if [ -z "$CRW_DAQ_CONFIG_FILE" ]; then
  fatal 1 "$(langstr ru "Сценарий вызывается только из DAQ." en "Script should be called from DAQ only.").";
 fi;
 case $1 in
  pre)  daq_processor pre;  ;;
  post) daq_processor post; ;;
  *) 1>&2 echo -e "\n usage:\n $scriptname (pre|post)\n"; return 1; ;;
 esac;
};

main "$@";

##############
## END OF FILE
##############
