#!/usr/local/bin/crwkit pyvenv

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# The code designed to run in 'crwkit pyvenv' environment.
##########################################################

import sys, os, io, time, math, random
from pycrwkit import *

##########################################################
# Override DaqApplication class for Demo User Application:
##########################################################
class WaveDaqApplication(DaqApplication):
    ######################
    # Override constructor
    ######################
    def __init__(self,title,poll=4):
        super().__init__(title,poll)
        self.Amplitude = 1.0
        self.Frequency = 1.0
        self.Noise = 0.05

    ###############################
    # Override Greetings on startup
    ###############################
    def Greetengs(self):
        super().Greetengs() # call inherited
        print("Welcome to %s program." % self.Title)

    ####################################
    # Override Processing of StdIn lines
    ####################################
    def StdInProcessing(self,line):
        if not isinstance(line,str):
            return False
        # call inherited StdIn processing
        return super().StdInProcessing(line)

    ##################################
    # Override HandleCommand for StdIn
    ##################################
    def HandleCommand(self,cmd,arg):
        if not isinstance(cmd,str):
            return False
        # call inherited command handler
        return super().HandleCommand(cmd,arg)

    ##############################
    # Execute user polling actions
    ##############################
    def ExecPollingActions(self):
        # call inherited polling actions
        return super().ExecPollingActions()

    ######################################
    # Define new handler for command @wave
    # > @wave t          - time(sec)
    # < @wave t sin cos  - result
    ######################################
    def DoWave(self,cmd,arg):
        if not SameText(cmd,"@Wave"):
            return False
        t = StrToFloat(arg)
        if (t is None):
            print("@wave NaN")
            return False
        (si,co) = self.CalcWave(t)
        print("@wave %.15g %.15g %.15g" % (t,si,co))
        return True

    ##########################
    # Calculate wave at time t
    ##########################
    def CalcWave(self,t):
        fr = self.Frequency
        am = self.Amplitude
        ns = self.Noise
        rn = random.uniform(-0.5,0.5)
        si = am*math.sin(fr*t)+ns*rn
        rn = random.uniform(-0.5,0.5)
        co = am*math.cos(fr*t)+ns*rn
        return (si,co)

    ######################################
    # Define new handler for command @para
    # > @para a f n
    # < @para a f n
    ######################################
    def DoPara(self,cmd,arg):
        if not SameText(cmd,"@Para"):
            return False
        spl = arg.split()
        spl.append("")
        spl.append("")
        spl.append("")
        (sa,sf,sn) = (spl[0],spl[1],spl[2])
        va = StrToFloat(sa)
        if isinstance(va,float):
            self.Amplitude = va
        vf = StrToFloat(sf)
        if isinstance(vf,float):
            self.Frequency = vf
        vn = StrToFloat(sn)
        if isinstance(vn,float):
            self.Noise = vn
        a = self.Amplitude
        f = self.Frequency
        n = self.Noise
        print("@Para %.15g %.15g %.15g" % (a,f,n))
        return True

##########################################
# Create the DaqApplication object to run:
##########################################
Application = WaveDaqApplication("wave",4)

###########################################
# Then add new application command handlers
###########################################
Application.AddCommandHandler(Application.DoWave,"@Wave t - calculate wave at time t")
Application.AddCommandHandler(Application.DoPara,"@Para a f n - set ampl, freq, noise")

###################
# Main program body
###################
def execute_main():
    try:
        Application.Welcome()
        Application.RunLoop()
        Application.Goodbye()
        Application.SysExit()
    except Exception as E:
        Application.OnError(E)
        Application.SysExit(1)

######
# MAIN
######

if __name__ == '__main__':
    execute_main()

#############
# END OF FILE
#############
