#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for MAYAK devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

##################################
### MAYAK devices config generator
##################################

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg"; # hardware   configuration
readonly simcfg="${scriptbase}_sim.cfg"; # simulator  configuration
readonly PARAM_PLOT="MAYAK.PARAM.PLOT";  # Name of Param plot window
readonly STATE_PLOT="MAYAK.STATE.PLOT";  # Name of State plot window
readonly MAIN_CTRL="MAYAK.MAIN.CTRL";    # Name of main control window
declare -i numhdwcfg=0;                  # Flag to include hardware config
declare -i CTRL_TOP=0;                   # Top  of main control window
declare -i CTRL_LEFT=450;                # Left of main control window
declare -i CTRL_TOP_STEP=80;             # Step for windows
declare FACN="";                         # Facility name
declare IDEN="";                         # Device identifier name
declare FACP="";                         # Facility DIM  path
declare FACF="";                         # Facility file name
declare ModbusPort="";                   # Modbus Port number
declare ModbusUnitId="";                 # Modbus Unit Id
declare ModbusPolling="";                # Modbus Polling period, ms

########################
# generate MAYAK devices
########################
function mayak_devices(){
 delete_files $hdwcfg $simcfg;
 ###  Device      Facility    Identitier  Port UnitId Polling
      BDGB14I_DEV DEMO.MAYAK  MAYAK01     1    1      100;
 #    BDGB14I_DEV DEMO.MAYAK  MAYAK02     1    2      100;
 #    BDGB14I_DEV DEMO.MAYAK  MAYAK03     1    3      100;
 cat $hdwcfg;
 cat $simcfg;
};

function BDGB14I_DEV(){
 BDGB14I_HDW "$@" >> $hdwcfg;
 BDGB14I_SIM "$@" >> $simcfg;
 let "CTRL_TOP+=CTRL_TOP_STEP";
};

function BDGB14I_HDW(){
 FACN="$(upper_case "$1")"; shift;   # Facility name
 IDEN="$(upper_case "$1")"; shift;   # Device identifier name
 ModbusPort="$1";           shift;   # Modbus Port number
 ModbusUnitId="$1";         shift;   # Modbus Unit Id
 ModbusPolling="$1";        shift;   # Modbus Polling period, ms
 FACP="$(echo $FACN | tr '.' '/')";  # Facility DIM  path, i.e. FACN name with . to / replacement
 FACF="$(echo $FACN | tr '.' '_')";  # Facility file name, i.e. FACN name with . to _ replacement
 FACP="$(upper_case "$FACP")";       # DIM path  in upper case
 FACF="$(lower_case "$FACF")";       # Filenames in lower case
 IDENLO="$(lower_case "$IDEN")";     # Device ID in lower case
 echo ";***************************";
 echo "; BDGB14I Driver for $IDEN";
 echo ";***************************";
 echo "[DeviceList]";
 echo "&$FACN.BDGB14I.$IDEN.CTRL = device software program";
 echo "[&$FACN.BDGB14I.$IDEN.CTRL]";
 echo "Comment        = MAYAK BDGB14I MODBUS RTU DRIVER FOR $IDEN.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpHighest";
 echo "ProgramSource  = ..\DaqPas\mayak_bdgb14i.pas";
 echo "DigitalFifo    = 1024";
 echo "AnalogFifo     = 1024";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "Simulator      = 0";
 echo "ModbusPort     = $ModbusPort";
 echo "ModbusUnitId   = $ModbusUnitId";
 echo "ModbusTimeout  = 250";
 echo "ModbusDeadline = 60000";
 echo "ModbusPolling  = $ModbusPolling";
 echo "DelayOnStart   = 1000";
 echo "tagPrefix      = $FACN.BDGB14I.$IDEN";
 echo "winControl     = $FACN.BDGB14I.$IDEN.CTRL";
 echo "AnalogOutputs  = 14";
 echo "Link AnalogOutput  0 with curve $FACN.BDGB14I.$IDEN.PARAM.CURIA  history 1000";
 echo "Link AnalogOutput  1 with curve $FACN.BDGB14I.$IDEN.PARAM.GASVA  history 10000";
 echo "Link AnalogOutput  2 with curve $FACN.BDGB14I.$IDEN.PARAM.CURTC  history 1000";
 echo "Link AnalogOutput  3 with curve $FACN.BDGB14I.$IDEN.PARAM.CURHV  history 1000";
 echo "Link AnalogOutput  4 with curve $FACN.BDGB14I.$IDEN.PARAM.CURAD  history 1000";
 echo "Link AnalogOutput  5 with curve $FACN.BDGB14I.$IDEN.PARAM.CURU1  history 1000";
 echo "Link AnalogOutput  6 with curve $FACN.BDGB14I.$IDEN.PARAM.CURU2  history 1000";
 echo "Link AnalogOutput  7 with curve $FACN.BDGB14I.$IDEN.PARAM.CURU0  history 1000";
 echo "Link AnalogOutput  8 with curve $FACN.BDGB14I.$IDEN.PARAM.CURR2  history 1000";
 echo "Link AnalogOutput  9 with curve $FACN.BDGB14I.$IDEN.PARAM.CURI0  history 1000";
 echo "Link AnalogOutput 10 with curve $FACN.BDGB14I.$IDEN.PARAM.PREHV  history 100";
 echo "Link AnalogOutput 11 with curve $FACN.BDGB14I.$IDEN.PARAM.DEFHV  history 100";
 echo "Link AnalogOutput 12 with curve $FACN.BDGB14I.$IDEN.PARAM.ALRM1  history 100";
 echo "Link AnalogOutput 13 with curve $FACN.BDGB14I.$IDEN.PARAM.ALRM2  history 100";
 echo "DigitalOutputs = 22";
 echo "Link DigitalOutput  0 with curve $FACN.BDGB14I.$IDEN.STATE.STNET history 1000";
 echo "Link DigitalOutput  1 with curve $FACN.BDGB14I.$IDEN.STATE.STADC history 1000";
 echo "Link DigitalOutput  2 with curve $FACN.BDGB14I.$IDEN.STATE.STREG history 1000";
 echo "Link DigitalOutput  3 with curve $FACN.BDGB14I.$IDEN.STATE.STCON history 1000";
 echo "Link DigitalOutput  4 with curve $FACN.BDGB14I.$IDEN.STATE.RECTL history 100";
 echo "Link DigitalOutput  5 with curve $FACN.BDGB14I.$IDEN.STATE.REREG history 100";
 echo "Link DigitalOutput  6 with curve $FACN.BDGB14I.$IDEN.STATE.FACTN history 10";
 echo "Link DigitalOutput  7 with curve $FACN.BDGB14I.$IDEN.STATE.FACTD history 10";
 echo "Link DigitalOutput  8 with curve $FACN.BDGB14I.$IDEN.STATE.NETAD history 10";
 echo "Link DigitalOutput  9 with curve $FACN.BDGB14I.$IDEN.STATE.NETIF history 10";
 echo "Link DigitalOutput 10 with curve $FACN.BDGB14I.$IDEN.STATE.SOFTV history 10";
 echo "Link DigitalOutput 11 with curve $FACN.BDGB14I.$IDEN.STATE.SOFTD history 10";
 echo "Link DigitalOutput 12 with curve $FACN.BDGB14I.$IDEN.STATE.DEVID history 10";
 echo "Link DigitalOutput 13 with curve $FACN.BDGB14I.$IDEN.STATE.RECON history 100";
 echo "Link DigitalOutput 14 with curve $FACN.BDGB14I.$IDEN.STATE.INDIC history 1000";
 echo "Link DigitalOutput 15 with curve $FACN.BDGB14I.$IDEN.POLLRATE.RX history 1000";
 echo "Link DigitalOutput 16 with curve $FACN.BDGB14I.$IDEN.POLLRATE.TX history 1000";
 echo "Link DigitalOutput 17 with curve $FACN.BDGB14I.$IDEN.POLLRATE.EX history 1000";
 echo "Link DigitalOutput 18 with curve $FACN.BDGB14I.$IDEN.ERROR.COUNT history 1000";
 echo "Link DigitalOutput 19 with curve $FACN.BDGB14I.$IDEN.POLLSUMM.RX history 1000";
 echo "Link DigitalOutput 20 with curve $FACN.BDGB14I.$IDEN.POLLSUMM.TX history 1000";
 echo "Link DigitalOutput 21 with curve $FACN.BDGB14I.$IDEN.POLLSUMM.EX history 1000";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$FACN.BDGB14I.$IDEN.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ..\data\\${FACF}_bdgb14i_${IDENLO}_custom.ini";
 echo "CustomIniSection  = [$FACN.BDGB14I.$IDEN.CustomParameters]";
 echo "CustomIniBackups  = ..\data\custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "[ConfigFileList]";
 echo "ConfigFile = ..\data\\${FACF}_bdgb14i_${IDENLO}_custom.ini";
 echo "[]";
 echo "[$FACN.BDGB14I.$IDEN.CustomParameters.TagList]";
 echo "TagList = $FACN.BDGB14I.$IDEN.PARAM.ALRM1";
 echo "TagList = $FACN.BDGB14I.$IDEN.PARAM.ALRM2";
 echo "[]";
 echo "";
 echo "[TagList]";
 echo "$FACN.BDGB14I.$IDEN.PARAM.IDENT = string  BDGB14I.$IDEN  ; Display device name";
 echo "$FACN.BDGB14I.$IDEN.STATE.FACTN = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.FACTD = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.NETAD = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.NETIF = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.SOFTV = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.SOFTD = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.DEVID = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.RECON = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.DEFHV = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.STNET = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.STADC = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.STREG = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.STCON = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURIA = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.GASVA = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURTC = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURHV = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURAD = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU1 = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU2 = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU0 = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURR2 = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURI0 = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.PREHV = real    0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.RECTL = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.REREG = integer 0               ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.ALRM1 = real    1E9             ; ";
 echo "$FACN.BDGB14I.$IDEN.PARAM.ALRM2 = real    1E10            ; ";
 echo "$FACN.BDGB14I.$IDEN.STATE.FACTS = string ?                ; Factory  Date as string";
 echo "$FACN.BDGB14I.$IDEN.STATE.SOFTS = string ?                ; Software Date as string";
 echo "$FACN.BDGB14I.$IDEN.STATE.INDIC = integer 0               ; Overall status indicator";
 echo "$FACN.BDGB14I.$IDEN.POLL.ENABLE = integer 1               ; Enable poll cycle";
 echo "$FACN.BDGB14I.$IDEN.EDIT.ENABLE = integer 0               ; Enable edit params";
 echo "$FACN.BDGB14I.$IDEN.AUTO.CALIBR = integer 0               ; Auto Calibration";
 echo "$FACN.BDGB14I.$IDEN.NET.RESTART = integer 0               ; Network Restart";
 echo "[]";
 echo "";
 echo "[DataStorage] ";
 echo "$FACN.BDGB14I.$IDEN.STATE.FACTN = Curve 0 10 Black  15 1  ; 0x00 Factory Number,         0..65535";
 echo "$FACN.BDGB14I.$IDEN.STATE.FACTD = Curve 0 10 Black  15 1  ; 0x01 Factory Date,           ?";
 echo "$FACN.BDGB14I.$IDEN.STATE.NETAD = Curve 0 10 Black  15 1  ; 0x02 Network Address,        1..247";
 echo "$FACN.BDGB14I.$IDEN.STATE.NETIF = Curve 0 10 Black  15 1  ; 0x03 Network Interface mode, see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.SOFTV = Curve 0 10 Black  15 1  ; 0x04 Software Version,       see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.SOFTD = Curve 0 10 Black  15 1  ; 0x05 Software Date,          see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.DEVID = Curve 0 10 Black  15 1  ; 0x06 Device Identifier,      see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.RECON = Curve 0 10 Black  15 1  ; 0x07 Register of Converter,  see Ref.";
 echo "$FACN.BDGB14I.$IDEN.PARAM.DEFHV = Curve 0 10 Black  15 1  ; 0x08 Default High Voltage,   V";
 echo "$FACN.BDGB14I.$IDEN.STATE.STNET = Curve 0 10 Black  15 1  ; 0x30 Status of Network,      B[0..7]=(hardware address), B[8]=(uses hardware address)";
 echo "$FACN.BDGB14I.$IDEN.STATE.STADC = Curve 0 10 Black  15 1  ; 0x31 Status of ADC,          B[8]=(ADC not/ready)";
 echo "$FACN.BDGB14I.$IDEN.STATE.STREG = Curve 0 10 Black  15 1  ; 0x32 Status of Regime,       see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.STCON = Curve 0 10 Black  15 1  ; 0x33 Status of Converter,    see Ref";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURIA = Curve 0 10 Black  15 1  ; 0x34 Measured Сurrent I,     A";
 echo "$FACN.BDGB14I.$IDEN.PARAM.GASVA = Curve 0 10 Black  15 1  ; 0x36 Gas volume activity,    Bq/m3";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURTC = Curve 0 10 Black  15 1  ; 0x38 Current Temperature,    °C";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURHV = Curve 0 10 Black  15 1  ; 0x3a Current High Voltage,   V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURAD = Curve 0 10 Black  15 1  ; 0x3c Current ADC,            V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU1 = Curve 0 10 Black  15 1  ; 0x3e Current U1,             V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU2 = Curve 0 10 Black  15 1  ; 0x40 Current U2,             V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURU0 = Curve 0 10 Black  15 1  ; 0x42 Current U0,             V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURR2 = Curve 0 10 Black  15 1  ; 0x44 Current R2,             V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.CURI0 = Curve 0 10 Black  15 1  ; 0x46 Current I0,             V";
 echo "$FACN.BDGB14I.$IDEN.PARAM.PREHV = Curve 0 10 Black  15 1  ; 0x48 Preset  High Voltage,   V";
 echo "$FACN.BDGB14I.$IDEN.STATE.RECTL = Curve 0 10 Black  15 1  ; 0x4a Register of Control,    see Ref.";
 echo "$FACN.BDGB14I.$IDEN.STATE.REREG = Curve 0 10 Black  15 1  ; 0x4b Register of Regime,     see Ref.";
 echo "$FACN.BDGB14I.$IDEN.PARAM.ALRM1 = Curve 0 10 Black  15 1  ; Alarm 1 level of activity,   YELLOW";
 echo "$FACN.BDGB14I.$IDEN.PARAM.ALRM2 = Curve 0 10 Black  15 1  ; Alarm 2 level of activity,   RED";
 echo "$FACN.BDGB14I.$IDEN.STATE.INDIC = Curve 0 10 Black  15 1  ; Overall Status Indicator,    O..8=GREEN/YELLOW/RED/TEST/ERROR/TIMEOUT/REFUSE/SIMULATOR/OFF";
 echo "$FACN.BDGB14I.$IDEN.POLLRATE.RX = Curve 0 10 Black  15 1  ; Poll rate, receiver          (per sec)";
 echo "$FACN.BDGB14I.$IDEN.POLLRATE.TX = Curve 0 10 Black  15 1  ; Poll rate, transmitter       (per sec)";
 echo "$FACN.BDGB14I.$IDEN.POLLRATE.EX = Curve 0 10 Black  15 1  ; Poll rate, errors            (per sec)";
 echo "$FACN.BDGB14I.$IDEN.ERROR.COUNT = Curve 0 10 Black  15 1  ; Error counter                (counter)";
 echo "$FACN.BDGB14I.$IDEN.POLLSUMM.RX = Curve 0 10 Black  15 1  ; Poll summ, receiver          (counter)";
 echo "$FACN.BDGB14I.$IDEN.POLLSUMM.TX = Curve 0 10 Black  15 1  ; Poll summ, transmitter       (counter)";
 echo "$FACN.BDGB14I.$IDEN.POLLSUMM.EX = Curve 0 10 Black  15 1  ; Poll summ, errors            (counter)";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.BDGB14I.$IDEN.CTRL = Circuit_Window ";
 echo "[$FACN.BDGB14I.$IDEN.CTRL]";
 echo "Circuit = ..\circuits\mayak_bdgb14i_ctrl.crc";
 echo "Link sensor @Cron+@Browse+www.po-mayak.ru                                  with device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor @mayak_bdgb14i.htm                                             with device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.POLL.ENABLE      with tag   $FACN.BDGB14I.$IDEN.POLL.ENABLE device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.EDIT.ENABLE      with tag   $FACN.BDGB14I.$IDEN.EDIT.ENABLE device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.AUTO.CALIBR      with tag   $FACN.BDGB14I.$IDEN.AUTO.CALIBR device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.NET.RESTART      with tag   $FACN.BDGB14I.$IDEN.NET.RESTART device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.IDENT      with tag   $FACN.BDGB14I.$IDEN.PARAM.IDENT device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.FACTN      with curve $FACN.BDGB14I.$IDEN.STATE.FACTN device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.FACTD      with tag   $FACN.BDGB14I.$IDEN.STATE.FACTS device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.NETAD.ADDR with curve $FACN.BDGB14I.$IDEN.STATE.NETAD device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.NETIF.BAUD with curve $FACN.BDGB14I.$IDEN.STATE.NETIF device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.NETIF.MODE with curve $FACN.BDGB14I.$IDEN.STATE.NETIF device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.SOFTV.LO   with curve $FACN.BDGB14I.$IDEN.STATE.SOFTV device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.SOFTV.HI   with curve $FACN.BDGB14I.$IDEN.STATE.SOFTV device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.SOFTD      with tag   $FACN.BDGB14I.$IDEN.STATE.SOFTS device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.DEVID      with curve $FACN.BDGB14I.$IDEN.STATE.DEVID device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.RECON.STAT with curve $FACN.BDGB14I.$IDEN.STATE.RECON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.DEFHV      with curve $FACN.BDGB14I.$IDEN.PARAM.DEFHV device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STNET.ADDR with curve $FACN.BDGB14I.$IDEN.STATE.STNET device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STNET.ASRC with curve $FACN.BDGB14I.$IDEN.STATE.STNET device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STNET.STAT with curve $FACN.BDGB14I.$IDEN.STATE.STNET device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STADC.STAT with curve $FACN.BDGB14I.$IDEN.STATE.STADC device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STADC.IND  with curve $FACN.BDGB14I.$IDEN.STATE.STADC device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.TYPE with tag   $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.CODE with tag   $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.TC   with curve $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.HV   with curve $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.U0   with curve $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.I0   with curve $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STREG.R2   with curve $FACN.BDGB14I.$IDEN.STATE.STREG device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.CODE with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.S1   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.S2   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.S3   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.S4   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.AC   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.U1   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.U2   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.U0   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.R2   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.I0   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.IA   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.STCON.UP   with curve $FACN.BDGB14I.$IDEN.STATE.STCON device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURIA      with curve $FACN.BDGB14I.$IDEN.PARAM.CURIA device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.GASVA      with curve $FACN.BDGB14I.$IDEN.PARAM.GASVA device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURTC      with curve $FACN.BDGB14I.$IDEN.PARAM.CURTC device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURHV      with curve $FACN.BDGB14I.$IDEN.PARAM.CURHV device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURAD      with curve $FACN.BDGB14I.$IDEN.PARAM.CURAD device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURU1      with curve $FACN.BDGB14I.$IDEN.PARAM.CURU1 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURU2      with curve $FACN.BDGB14I.$IDEN.PARAM.CURU2 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURU0      with curve $FACN.BDGB14I.$IDEN.PARAM.CURU0 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURR2      with curve $FACN.BDGB14I.$IDEN.PARAM.CURR2 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.CURI0      with curve $FACN.BDGB14I.$IDEN.PARAM.CURI0 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.PREHV      with tag   $FACN.BDGB14I.$IDEN.PARAM.PREHV device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.ALRM1      with tag   $FACN.BDGB14I.$IDEN.PARAM.ALRM1 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.PARAM.ALRM2      with tag   $FACN.BDGB14I.$IDEN.PARAM.ALRM2 device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.STATE.INDIC      with curve $FACN.BDGB14I.$IDEN.STATE.INDIC device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.POLLRATE.RX      with curve $FACN.BDGB14I.$IDEN.POLLRATE.RX device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.POLLRATE.TX      with curve $FACN.BDGB14I.$IDEN.POLLRATE.TX device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.POLLRATE.EX      with curve $FACN.BDGB14I.$IDEN.POLLRATE.EX device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "Link sensor BDGB14I.POLLSUMM.EX      with curve $FACN.BDGB14I.$IDEN.POLLSUMM.EX device &$FACN.BDGB14I.$IDEN.CTRL";
 echo "[]";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.BDGB14I.$IDEN.CTRL.HOME 0 0 0";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.HOME @WinShow $FACN.BDGB14I.$IDEN.CTRL";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.HOME @WinDraw $FACN.BDGB14I.$IDEN.CTRL|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=720|Height=360";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.HOME @WinDraw $FACN.BDGB14I.$IDEN.CTRL|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.HOME @WinDraw $FACN.BDGB14I.$IDEN.CTRL|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.BDGB14I.$IDEN.CTRL.SHOW 0 0 0";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.SHOW @WinShow $FACN.BDGB14I.$IDEN.CTRL";
 echo "@cron.tab $FACN.BDGB14I.$IDEN.CTRL.HIDE 0 0 0";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.HIDE @WinHide $FACN.BDGB14I.$IDEN.CTRL";
 echo "@cron.tab $FACN.BDGB14I.$IDEN.CTRL.SELECT 0 0 0";
 echo "@cron.job $FACN.BDGB14I.$IDEN.CTRL.SELECT @WinSelect $FACN.BDGB14I.$IDEN.CTRL";
 echo "@cron.job $MAIN_CTRL.HOME @cron.run $FACN.BDGB14I.$IDEN.CTRL.HOME";
 echo "@cron.job $MAIN_CTRL.HOME @cron.run $FACN.BDGB14I.$IDEN.CTRL.HIDE";
 echo "[]";
 echo "";
 echo "[$MAIN_CTRL]";
 echo "Link sensor $IDEN.CTRL  with                                       device &$MAIN_CTRL";
 echo "Link sensor $IDEN.VALUE with curve $FACN.BDGB14I.$IDEN.PARAM.GASVA device &$MAIN_CTRL";
 echo "[]";
 echo "";
 echo "[$PARAM_PLOT]";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURIA";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.GASVA";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURTC";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURHV";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURAD";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURU1";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURU2";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURU0";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURR2";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.CURI0";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.PREHV";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.DEFHV";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.ALRM1";
 echo "CurveList = $FACN.BDGB14I.$IDEN.PARAM.ALRM2";
 echo "[]";
 echo "";
 echo "[$STATE_PLOT]";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.FACTN";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.FACTD";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.NETAD";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.NETIF";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.SOFTV";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.SOFTD";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.DEVID";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.RECON";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.STNET";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.STADC";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.STREG";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.STCON";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.RECTL";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.REREG";
 echo "CurveList = $FACN.BDGB14I.$IDEN.STATE.INDIC";
 echo "CurveList = $FACN.BDGB14I.$IDEN.POLLRATE.RX";
 echo "CurveList = $FACN.BDGB14I.$IDEN.POLLRATE.TX";
 echo "CurveList = $FACN.BDGB14I.$IDEN.POLLRATE.EX";
 echo "CurveList = $FACN.BDGB14I.$IDEN.ERROR.COUNT";
 echo "[]";
 echo "";
 echo "";
 echo "[&DATSRV]";
 echo "CurveList = [$PARAM_PLOT]";
 echo "CurveList = [$STATE_PLOT]";
 echo "[]";
};

function BDGB14I_SIM(){
 echo "[&$FACN.BDGB14I.$IDEN.CTRL]";
 echo "Simulator = 1";
 echo "[]";
 echo "";
 ### Include hardware config if one not done yet
 if [ $numhdwcfg -eq 0 ]; then let "numhdwcfg++"; else return; fi;
 echo "[ConfigFileList]";
 echo "ConfigFile = $hdwcfg";
 echo "[]";
 echo "";
};

##############
# delete files
##############
function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scriptHOME >/dev/null 2>&1; then
  mayak_devices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
