 {
 HV power supply driver & simulator for Spellman SL & SR seria, RS-232.
 Last release:
  18.11.2009: vjushin@mail.ru
  23.03.2011: kouriakine@mail.ru
  21.03.2012: kouriakine@mail.ru
 Notes:
  Command example: 02 32 32 2C 70 03
  Answer  example: 02 32 32 2C 30 2C 30 2C 30 2C 5C 03
  Command example: 02 32 36 2C 6C 03
  Answer  example: 02 32 36 2C 1E 7D 01 43 32 20 DC 80 1E 7D 2C 58 03
  On error command reply: 45 23 31 (E#1)
 [@Help]
 |StdIn Command list: "@cmd=arg" or "@cmd arg"
 |******************************************************
 | @Sim.Disabled=n - Disable (1/0) to simulate TimeOut.
 | @Sim.CheckBug=n - Simulate check summ error (n>0).
 | @Sim.CmdIdBug=n - Simulate CmdId format error (n>0).
 | @Sim.InterLk=n  - Simulate Interlock state (1/0).
 | @Sim.FaultSt=n  - Simulate Fault state (1/0).
 |******************************************************
 []
 }
program HVSS_DRV;                { SPELLMAN HV power supply driver  }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 MinCmdLeng        = 6;          { Minimal length of commands       }
 MinAnsLeng        = 8;          { Minimal length of device answers }
 MaxStrLeng        = 16384;      { Maximal string length for safety }
 MaxComLeng        = 255;        { Maximal data length for ComRead  }
 SimRLoad          = 1.0E3;      { Simulation resistance, kOhm      }
 MaxCmdNum         = 32;         { Maximal command number           }
 TimeOutMin        = 100;        { Minimal timeout range            }
 TimeOutMax        = 60000;      { Maximal timeout range            }
 TimeOutDef        = 10000;      { Default timeout value            }
 TimeOutFac        = 2;          { TimeOut factor                   }
 TimeGapDef        = 0;          { Default TimeGap                  }
 {-------------------------------- Device specific command codes    }
 cm_SetVoltage     = 10;         { Set DAC code for Voltage         }
 cm_SetCurrent     = 11;         { Set DAC code for Current         }
 cm_ReadAnalog     = 20;         { Read analog channels: kV,I,T...  }
 cm_GetHoursHv     = 21;         { Get total hours HV was ON        }
 cm_ReadStatus     = 22;         { Read status: HV,Interlock,Fault  }
 cm_GetModelId     = 26;         { Get firmware Model ID Number     }
 cm_ResetHours     = 30;         { Reset Run Hours counter          }
 cm_SetInterLk     = 52;         { Set program Interlock ON/OFF     }
 cm_SetRemoteM     = 85;         { Set remote mode ON/OFF           }
 cm_SetHvOnOff     = 99;         { Set HV ON/OFF state              }
 {-------------------------------- CheckCommand flags               }
 cc_Success        = 0;          { Success, data is good to use     }
 cc_BadSumm        = 1;          { Bat status, wrong checksumm      }
 cc_NoneETX        = 2;          { Bad status, last char is not ETX }
 cc_NoneSTX        = 3;          { Bad status, 1-st char is not STX }
 cc_BadLeng        = 4;          { Bad status, small data length    }
 {-------------------------------- Scaling constants                }
 MinCurrentCode    = 0;          { Minimal ADC,DAC code for Current }
 MaxCurrentCode    = 4095;       { Maximal ADC,DAC code for Current }
 MinVoltageCode    = 0;          { Minimal ADC,DAC code for Voltage }
 MaxVoltageCode    = 4095;       { Maximal ADC,DAC code for Voltage }
 FacTempSic        = 0.0732;     { Factor for code --> Temp.S.I.C.  }
 FacTempMul        = 0.0732;     { Factor for code --> Temp.Multip. }
 {-------------------------------- Analog Inputs                    }
 ai_VoltSet        = 0;          { Wanted supply output Voltage,V   }
 ai_CurrSet        = 1;          { Wanted supply output Current,mA  }
 {-------------------------------- Analog Outputs                   }
 ao_VoltGot        = 0;          { Accepted supply output voltage,V }
 ao_CurrGot        = 1;          { Accepted supply output current,mA}
 ao_VoltDac        = 2;          { Accepted supply voltage DAC code }
 ao_CurrDac        = 3;          { Accepted supply current DAC code }
 ao_VoltGet        = 4;          { Real supply output voltage, V    }
 ao_CurrGet        = 5;          { Real supply output current, mA   }
 ao_VoltAdc        = 6;          { Real supply voltage ADC code     }
 ao_CurrAdc        = 7;          { Real supply current ADC code     }
 ao_HvHours        = 8;          { Real run hours in HV ON state    }
 ao_TempSic        = 9;          { Real temperature, S.I.C.,°C      }
 ao_TempMul        = 10;         { Real temperature, multiplier,°C  }
 ao_AnsTime        = 11;         { Request-answer time, ms          }
 {-------------------------------- Digital Inputs                   }
 di_HvOnOff        = 0;          { Wanted HV ON/OFF state           }
 {-------------------------------- Digital Outputs                  }
 do_HvOnOff        = 0;          { HV ON state                      }
 do_InterLk        = 1;          { Interlock state                  }
 do_FaultSt        = 2;          { Fault state                      }
 do_Connect        = 3;          { Connection state                 }
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 errorcodetimeout  : Integer;    { Error code for timeout           }
 errorcodechksumm  : Integer;    { Error code for checksumm         }
 errorcodebadfrmt  : Integer;    { Error code for bad format        }
 errorcodeharderr  : Integer;    { Error code for hardware error    }
 MinVoltageRange   : Real;       { Minimal Voltage Range, V         }
 MaxVoltageRange   : Real;       { Maximal Voltage Range, V         }
 MinCurrentRange   : Real;       { Minimal Current Range, mA        }
 MaxCurrentRange   : Real;       { Maximal Current Range, mA        }
 csCmdAdd          : Integer;    { Check sum Cmd add constant (?)   }
 csAnsAdd          : Integer;    { Check sum Ans add constant (?)   }
 ComPort           : Integer;    { Com port number                  }
 ComReply          : String;     { Com port received reply          }
 ComStatus         : Integer;    { Com port status, see rs_XX const }
 Simulator         : Boolean;    { T/F=Simulation/Driver mode       }
 MinTimeOut        : Integer;    { Minimal Com port TimeOut         }
 MaxTimeOut        : Integer;    { Maximal Com port TimeOut         }
 ComTimeOut        : Integer;    { Com port timeout                 }
 ComTimeGap        : Integer;    { Com port timegap                 }
 SeriaId           : String;     { Seria identifier: SR or SL       }
 kVoltId           : String;     { kVolt identifier by model        }
 PolarId           : String;     { Polarity identifier by model     }
 PowerId           : String;     { Power idntifier by model         }
 ModelId           : String;     { Model signature identifier       }
 ModelSign         : String;     { Model hex signature              }
 WaitQueueStart    : Real;       { Time when rs_WaitQueue started   }
 WaitReplyStart    : Real;       { Time when rs_WaitReply started   }
 Wanted,Actual     : record      { Wanted/actual=accepted parameters}
  HvOnOff          : Integer;    { Wanted/actual HV ON/OFF value    }
  Voltage          : Integer;    { Wanted/actual voltage code       }
  Current          : Integer;    { Wanted/actual current code       }
  RemoteM          : Integer;    { Wanted/actual Remote Mode        }
  InterLk          : Integer;    { Wanted/actual Interlock          }
  ModelId          : Integer;    { Wanted/actual ModelId            }
  FaultSt          : Integer;    { Fault state                      }
  ResetHs          : Integer;    { Wanted/actual Reset Hours value  }
 end;
 CmdTab            : record      { Command table                    }
  Count            : Integer;    { Command counter, 0..MaxCmdNum+1  }
  Current          : Integer;    { Current command index to send    }
  SentCmd          : Integer;    { Last sent command ID             }
  Comment          : array[0..MaxCmdNum] of String;
  Enabled          : array[0..MaxCmdNum] of Boolean;
  CmdId            : array[0..MaxCmdNum] of Integer;
 end;
 Sim               : record      { Simulation data                  }
  Disabled         : Integer;    { Uses to simulate timeouts        }
  CheckBug         : Integer;    { Uses to simulate checksum errors }
  CmdIdBug         : Integer;    { Uses to simulate format errors   }
  Tsic,Tmul        : Integer;    { Temperature codes                }
  HvHours          : Real;       { HV Hours                         }
 end;
 tagHvOnOff        : Integer;    { HV ON state                      }
 tagInterLk        : Integer;    { Interlock state                  }
 tagFaultSt        : Integer;    { Fault state                      }
 tagConnect        : Integer;    { Connection state                 }
 tagAnsTime        : Integer;    { Request-answer time, ms          }
 tagVoltGot        : Integer;    { Accepted voltage,V               }
 tagCurrGot        : Integer;    { Accepted current,mA              }
 tagVoltDac        : Integer;    { Accepted voltage DAC code        }
 tagCurrDac        : Integer;    { Accepted current DAC code        }
 tagVoltGet        : Integer;    { Real voltage,V                   }
 tagCurrGet        : Integer;    { Real current,mA                  }
 tagVoltAdc        : Integer;    { Real voltage ADC code            }
 tagCurrAdc        : Integer;    { Real current ADC code            }
 tagTempSic        : Integer;    { Temperature S.I.C.               }
 tagTempMul        : Integer;    { Temperature muliplier            }
 tagHvHours        : Integer;    { HV run hours                     }
 tagResetHours     : Integer;    { Reset work time hours            }
 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {
 Calculate control check summ of message Data.
 }
 function CheckSumm(Data:String; nAdd:Integer):Char;
 var i,cs:Integer;
 begin
  cs:=nAdd;
  if Length(Data)>0 then begin
   for i:=1 to Length(Data) do cs:=cs+Ord(Data[i]);
   cs:=iOr(64,iAnd(63,256-iAnd(255,cs)));
  end;
  CheckSumm:=Chr(cs);
 end;
 {
 Extract Msg body from Data command.
 }
 function ExtractMsg(Data:String):String;
 begin
  ExtractMsg:=Copy(Data,2,Length(Data)-3);
 end;
 {
 Extract command number from Data command.
 }
 function ExtractCmd(Data:String):String;
 begin
  ExtractCmd:=ExtractWord(1,ExtractMsg(Data));
 end;
 {
 Extract arguments from Data command.
 }
 function ExtractArgs(Data:String):String;
 begin
  ExtractArgs:=SkipWords(1,ExtractMsg(Data));
 end;
 {
 Extract argument n from Data command.
 }
 function ExtractArg(n:Integer;Data:String):String;
 begin
  ExtractArg:=ExtractWord(n+1,ExtractMsg(Data));
 end;
 {
 Check Data command status:
 cc_Sussess - Command looks good to use.
 cc_BadSumm - Bad command, control check sums is wrong.
 cc_NoneETX - Bad command, last char of Data is not ETX.
 cc_NoneSTX - Bad command, 1st char of Data is not STX.
 cc_BadLeng - Bad command, Data length is too small.
 }
 function CheckCommand(Data:String;MinLeng,nAdd:Integer):Integer;
 begin
  if MinLeng<2 then MinLeng:=2;
  if Length(Data)<MinLeng then CheckCommand:=cc_BadLeng else
  if Ord(Data[1])<>_STX then CheckCommand:=cc_NoneSTX else
  if Ord(Data[Length(Data)])<>_ETX then CheckCommand:=cc_NoneETX else
  if CheckSumm(ExtractMsg(Data),nAdd)<>Data[Length(Data)-1] then CheckCommand:=cc_BadSumm else
  CheckCommand:=cc_Success;
 end;
 {
 Compose command by given Cmd and Arg.
 Cmd expected to be 2-char number, like '26'.
 Arg expected to be empty or number as string.
 }
 function ComposeCommand(Cmd,Arg:String;nAdd:Integer):String;
 begin
  if (Length(Cmd)=2) and (Val(Cmd)>0) then begin
   if Length(Arg)=0 then Cmd:=Cmd+',' else Cmd:=Cmd+','+Arg+',';
   ComposeCommand:=Chr(_STX)+Cmd+CheckSumm(Cmd,nAdd)+Chr(_ETX);
  end else ComposeCommand:='';
 end;
 {
 Send data message to COM port.
 }
 procedure ComSendMsg(Data:String);
 begin
  if Length(Data)>0 then begin
   ViewExp('COM: $'+Hex_Encode(Data)+', '+Data);
   if not ComWrite(Data) then Trouble('Could not send "'+Data+'"');
  end;
 end;
 {
 Compose answer and send to COM port.
 Uses in simulation mode.
 }
 procedure ComSendAns(Cmd,Arg:String);
 begin
  ComSendMsg(ComposeCommand(Cmd,Arg,csAnsAdd));
 end;
 {
 Compose command and send to COM port.
 }
 procedure ComSendCmd(Cmd,Arg:String);
 begin
  ComSendMsg(ComposeCommand(Cmd,Arg,csCmdAdd));
 end;
 {
 Initialize model dependent parameters
 }
 procedure InitModel(ModelName:String);
  function CharToSpace(c:Char; Pattern:String; Invert:Boolean):Char;
  begin
   if iXor(Ord(Pos(Dump(c),Pattern)>0),Ord(Invert))>0
   then CharToSpace:=' '
   else CharToSpace:=c;
  end;
  function AlfaToSpace(s:String):String;
  begin
   if Length(s)=0 then AlfaToSpace:='' else
   if Length(s)=1 then AlfaToSpace:=CharToSpace(s[1],'0123456789',True) else
   AlfaToSpace:=AlfaToSpace(Copy(s,1,Length(s) div 2))+AlfaToSpace(Copy(s,(Length(s) div 2)+1));
  end;
  function NumbToSpace(s:String):String;
  begin
   if Length(s)=0 then NumbToSpace:='' else
   if Length(s)=1 then NumbToSpace:=CharToSpace(s[1],'0123456789',False) else
   NumbToSpace:=NumbToSpace(Copy(s,1,Length(s) div 2))+NumbToSpace(Copy(s,(Length(s) div 2)+1));
  end;
 begin
  ModelId:='';
  ModelSign:='';
  MinVoltageRange:=0; MaxVoltageRange:=0;
  MinCurrentRange:=0; MaxCurrentRange:=0;
  SeriaId:=ExtractWord(1,NumbToSpace(ModelName)); 
  kVoltId:=ExtractWord(1,AlfaToSpace(ModelName));
  PolarId:=ExtractWord(2,NumbToSpace(ModelName));
  PowerId:=ExtractWord(2,AlfaToSpace(ModelName));
  Success('Model Name='+ModelName+', Seria='+SeriaId+', kVolt='+kVoltId+', Polar='+PolarId+', Power='+PowerId);
  ModelSign:=ReadIni('ModelSign');
  ModelId:=Hex_Decode(ModelSign);
  if IsSameText(SeriaId,'SR') then begin
   csCmdAdd:=0; csAnsAdd:=_ETX;
   MinVoltageRange:=0; MinCurrentRange:=0;
   MaxVoltageRange:=Round(Val(kVoltId)*1000);
   MaxCurrentRange:=Round(Val(PowerId)*1000*1000/MaxVoltageRange);
   if Length(ModelId+ModelSign)=0 then ModelId:='SR'; 
  end else
  if IsSameText(SeriaId,'SL') then begin
   csCmdAdd:=0; csAnsAdd:=0;
   MinVoltageRange:=0; MinCurrentRange:=0;
   MaxVoltageRange:=Round(Val(kVoltId)*1000);
   MaxCurrentRange:=Round(Val(PowerId)*1000/MaxVoltageRange);
   if Length(ModelId+ModelSign)=0 then ModelId:=URL_Decode('%1E}'); 
  end else
  Trouble('Unknown model '+ModelName);
  Success('ModelId="'+Hex_Encode(ModelId)+'" = "'+URL_Packed(ModelId)+'"');
  if IsNan(MaxVoltageRange-MinVoltageRange)
  or IsInf(MaxVoltageRange-MinVoltageRange)
  or (MaxVoltageRange<=MinVoltageRange)
  then Trouble('Invalid U range ('+Str(MinVoltageRange)+','+Str(MaxVoltageRange)+'), model '+ModelName)
  else Success('Voltage range ('+Str(MinVoltageRange)+','+Str(MaxVoltageRange)+') V');
  if IsNan(MaxCurrentRange-MinCurrentRange)
  or IsInf(MaxCurrentRange-MinCurrentRange)
  or (MaxCurrentRange<=MinCurrentRange)
  then Trouble('Invalid I range ('+Str(MinCurrentRange)+','+Str(MaxCurrentRange)+'), model '+ModelName)
  else Success('Current range ('+Str(MinCurrentRange)+','+Str(MaxCurrentRange)+') mA');
 end;
 {
 Check if device seria is in SeriaList.
 Example: IsSeria('SR,SL')
 }
 function IsSeria(SeriaList:String):Boolean;
 begin
  IsSeria:=WordIndex(SeriaId,SeriaList)>0;
 end;
 {
 Truncate is Trunc replacement to avoid digital noise problem.
 CodeToVoltage(VoltageToCode(Code)) should be equal to Code.
 But it is not, if use simple Trunc instead of Truncate.
 }
 function Truncate(X:Real):Integer;
 begin
  Truncate:=Trunc(X+MachEps*2e3);
 end;
 {
 Convert code to Voltage,V.
 }
 function CodeToVoltage(code:Integer):Real;
 var beta:Real; Voltage:Real;
 begin
  if code<MinVoltageCode then code:=MinVoltageCode;
  if code>MaxVoltageCode then code:=MaxVoltageCode;
  beta:=(code-MinVoltageCode)/(MaxVoltageCode-MinVoltageCode);
  Voltage:=MinVoltageRange+beta*(MaxVoltageRange-MinVoltageRange);
  beta:=(MaxVoltageRange-MinVoltageRange)/(MaxVoltageCode-MinVoltageCode);
  if beta>2 then Voltage:=Truncate(Voltage+0.999999999999);
  if Voltage<MinVoltageRange then Voltage:=MinVoltageRange;
  if Voltage>MaxVoltageRange then Voltage:=MaxVoltageRange;
  CodeToVoltage:=Voltage;
 end;
 {
 Convert Voltage,V to code.
 }
 function VoltageToCode(Voltage:Real):Integer;
 var beta:Real; code:Integer;
 begin
  if Voltage<MinVoltageRange then Voltage:=MinVoltageRange;
  if Voltage>MaxVoltageRange then Voltage:=MaxVoltageRange;
  beta:=(Voltage-MinVoltageRange)/(MaxVoltageRange-MinVoltageRange);
  code:=Truncate(MinVoltageCode+beta*(MaxVoltageCode-MinVoltageCode));
  if code<MinVoltageCode then code:=MinVoltageCode;
  if code>MaxVoltageCode then code:=MaxVoltageCode;
  VoltageToCode:=code;
 end;
 {
 Convert code to Current,mA.
 }
 function CodeToCurrent(code:Integer):Real;
 var beta:Real; Current:Real;
 begin
  if code<MinCurrentCode then code:=MinCurrentCode;
  if code>MaxCurrentCode then code:=MaxCurrentCode;
  beta:=(code-MinCurrentCode)/(MaxCurrentCode-MinCurrentCode);
  Current:=MinCurrentRange+beta*(MaxCurrentRange-MinCurrentRange);
  if Current<MinCurrentRange then Current:=MinCurrentRange;
  if Current>MaxCurrentRange then Current:=MaxCurrentRange;
  CodeToCurrent:=Current;
 end;
 {
 Convert Current,mA to code.
 }
 function CurrentToCode(Current:Real):Integer;
 var beta:Real; code:Integer;
 begin
  if Current<MinCurrentRange then Current:=MinCurrentRange;
  if Current>MaxCurrentRange then Current:=MaxCurrentRange;
  beta:=(Current-MinCurrentRange)/(MaxCurrentRange-MinCurrentRange);
  code:=Truncate(MinCurrentCode+beta*(MaxCurrentCode-MinCurrentCode));
  if code<MinCurrentCode then code:=MinCurrentCode;
  if code>MaxCurrentCode then code:=MaxCurrentCode;
  CurrentToCode:=code;
 end;
 {
 Convert code to temperature SIC,°C.
 }
 function CodeToTempSic(code:Integer):Real;
 begin
  CodeToTempSic:=code*FacTempSic;
 end;
 {
 Convert temperature SIC,°C to code.
 }
 function TempSicToCode(Temp:Real):Integer;
 begin
  TempSicToCode:=Truncate(Temp/FacTempSic);
 end;
 {
 Convert code to temperature Multiplier,°C.
 }
 function CodeToTempMul(code:Integer):Real;
 begin
  CodeToTempMul:=code*FacTempMul;
 end;
 {
 Convert temperature Multiplier,°C to code.
 }
 function TempMulToCode(Temp:Real):Integer;
 begin
  TempMulToCode:=Truncate(Temp/FacTempMul);
 end;
 {
 Test data convertion procedures.
 }
 procedure TestConvertion;
 var c,code,n:Integer; ms:Real;
 begin
  ms:=mkSecNow/1000; n:=0;
  for c:=MinVoltageCode to MaxVoltageCode do begin
   code:=VoltageToCode(CodeToVoltage(c));
   if c<>code then n:=n+1;
  end;
  for c:=MinCurrentCode to MaxCurrentCode do begin
   code:=CurrentToCode(CodeToCurrent(c));
   if c<>code then n:=n+1;
  end;
  for c:=MinVoltageCode to MaxVoltageCode do begin
   code:=TempSicToCode(CodeToTempSic(c));
   if c<>code then n:=n+1;
  end;
  for c:=MinVoltageCode to MaxVoltageCode do begin
   code:=TempMulToCode(CodeToTempMul(c));
   if c<>code then n:=n+1;
  end;
  ms:=Int(mkSecNow/1000-ms);
  Success('TestConvertion: '+Str(n)+' errors, '+Str(ms)+' ms.');
 end;
 {
 Add command to CmdId command table.
 }
 procedure AddCmdId(CmdId:Integer; Enabled:Boolean; Comment:String);
 begin
  if CmdTab.Count<=MaxCmdNum then begin
   CmdTab.Enabled[CmdTab.Count]:=Enabled;
   CmdTab.Comment[CmdTab.Count]:=Comment;
   CmdTab.CmdId[CmdTab.Count]:=CmdId;
   CmdTab.Count:=CmdTab.Count+1;
   Success('CmdTab['+StrFix(CmdTab.Count,2,0)+']: '+Str(CmdId)+', '+Str(Ord(Enabled))+', "'+Comment+'".');
  end;
 end;
 {
 Enable command with given command identifier.
 }
 procedure EnableCmdId(CmdId:Integer; Enabled:Boolean);
 var i:Integer;
 begin
  for i:=0 to CmdTab.Count-1 do
  if CmdTab.CmdId[i]=CmdId then CmdTab.Enabled[i]:=Enabled;
 end;
 {
 Get comment string for given command CmdId.
 }
 function CommentCmdId(CmdId:Integer):String;
 var i,j:Integer;
 begin
  i:=0; j:=-1;
  while (i<CmdTab.Count) and (j<0) do begin
   if CmdTab.CmdId[i]=CmdId then j:=i;
   i:=i+1;
  end;
  if j<0 then CommentCmdId:='' else CommentCmdId:=CmdTab.Comment[j];
 end;
 {
 Set special values to enforce updates of supply state.
 This cause cm_SetHvOnOff,cm_ResetHours,.. etc send.
 We should enforce updates after start or timeout.
 }
 procedure EnforceUpdates(aComStatus:Integer);
 begin
  CmdTab.Current:=0;
  CmdTab.SentCmd:=0;
  Actual.HvOnOff:=-1;
  Actual.Voltage:=-1;
  Actual.Current:=-1;
  Actual.RemoteM:=-1;
  Actual.InterLk:=-1;
  Actual.FaultSt:=-1;
  Actual.ModelId:=-1;
  Actual.ResetHs:=-1;
  ComStatus:=aComStatus;
 end;
 {
 Reset command table and set ComStatus.
 }
 procedure ResetCmd(aComStatus:Integer);
 var IsSR,IsId:Boolean;
 begin
  CmdTab.Count:=0;
  IsSR:=IsSeria('SR');
  EnforceUpdates(aComStatus);
  IsId:=Length(ExtractWord(1,ModelId))>0;
  AddCmdId(cm_GetModelId, IsId, 'Request model ID number');
  AddCmdId(cm_SetRemoteM, True, 'Set remote mode ON, OFF');
  AddCmdId(cm_SetInterLk, IsSR, 'Set program Interlock-1');
  AddCmdId(cm_SetVoltage, True, 'Set DAC code on Voltage');
  AddCmdId(cm_SetCurrent, True, 'Set DAC code on Current');
  AddCmdId(cm_SetHvOnOff, True, 'Set ON/OFF state for HV');
  AddCmdId(cm_ResetHours, True, 'Reset a Hours HV was ON');
  AddCmdId(cm_ReadAnalog, True, 'Request analog channels');
  AddCmdId(cm_GetHoursHv, True, 'Request Hours HV was ON');
  AddCmdId(cm_ReadStatus, True, 'Request HV status flags');  // Must be last command
 end;
 {
 Prepare next command to send.
 Also skips disabled commands. 
 }
 procedure PrepareNextCmd(Step:Integer);
 begin
  CmdTab.Current:=CmdTab.Current+Step;
  if CmdTab.Current<0 then CmdTab.Current:=0;
  if CmdTab.Current>=CmdTab.Count then CmdTab.Current:=0;
  Step:=0;
  while (Step<=MaxCmdNum) and not CmdTab.Enabled[CmdTab.Current] do begin
   CmdTab.Current:=CmdTab.Current+1;
   if CmdTab.Current>=CmdTab.Count then CmdTab.Current:=0;
   Step:=Step+1;
  end;
 end;
 {
 Purge (clear) COM port, special version.
 }
 procedure PurgeComPortSpec;
 var data:String;
 begin
  data:='';
  if ComClear then Details('ComClear');
  while ComCount>0 do begin
   data:=ComRead(MaxComLeng);
   Details('Skip COM: '+Hex_Encode(data)+' '+data);
  end;
  data:='';
 end;
 {
 Handle "ComPort is not available"...
 }
 procedure OnNotAvail;
 begin
  Details('ComPort '+Str(ComPort)+' is not available!');
 end;
 {
 Handle "RS-232 line is ready for new request"...
 If has command to send, try to send the command.
 }
 procedure OnNoRequest;
 var cmd,arg:String;
 begin  
  cmd:='';
  arg:='';
  ComReply:=''; // Clear Com port input buffer
  PrepareNextCmd(0); // Skip disabled commands
  if CmdTab.Enabled[CmdTab.Current] then begin
   CmdTab.SentCmd:=CmdTab.CmdId[CmdTab.Current];
   if CmdTab.SentCmd=cm_GetModelId then begin
    Wanted.ModelId:=1;
    if Wanted.ModelId<>Actual.ModelId then begin
     cmd:=Str(CmdTab.SentCmd);
     arg:='';
    end;
   end else
   if CmdTab.SentCmd=cm_SetRemoteM then begin
    Wanted.RemoteM:=1;
    if Wanted.RemoteM<>Actual.RemoteM then begin
     cmd:=Str(CmdTab.SentCmd);
     arg:=Str(Wanted.RemoteM);
    end;
   end else
   if CmdTab.SentCmd=cm_SetInterLk then begin
    Wanted.InterLk:=1;
    if Wanted.InterLk<>Actual.InterLk then begin
     cmd:=Str(CmdTab.SentCmd);
     arg:=Str(Wanted.InterLk);
    end;
   end else
   if CmdTab.SentCmd=cm_SetVoltage then begin
    if GetAi_N(ai_VoltSet)>0 then begin
     Wanted.Voltage:=VoltageToCode(GetAi_Yn(ai_VoltSet));
     cmd:=Str(CmdTab.SentCmd);
     arg:=Str(Wanted.Voltage);
    end;
   end else
   if CmdTab.SentCmd=cm_SetCurrent then begin
    if GetAi_N(ai_CurrSet)>0 then begin
     Wanted.Current:=CurrentToCode(GetAi_Yn(ai_CurrSet));
     cmd:=Str(CmdTab.SentCmd);
     arg:=Str(Wanted.Current);
    end;
   end else
   if CmdTab.SentCmd=cm_SetHvOnOff then begin
    if GetDi_N(di_HvOnOff)>0 then begin
     Wanted.HvOnOff:=Truncate(DiWord(di_HvOnOff,1));
     if Wanted.HvOnOff<>Actual.HvOnOff then begin
      cmd:=Str(CmdTab.SentCmd);
      arg:=Str(Wanted.HvOnOff);
     end;
    end;
   end else
   if CmdTab.SentCmd=cm_ResetHours then begin
    Wanted.ResetHs:=iGetTag(tagResetHours);
    if Wanted.ResetHs<>Actual.ResetHs then begin
     bNul(iSetTag(tagResetHours,0));
     cmd:=Str(CmdTab.SentCmd);
     arg:='';
    end;
   end else
   if CmdTab.SentCmd=cm_ReadAnalog then begin
    cmd:=Str(CmdTab.SentCmd);
    arg:='';
   end else
   if CmdTab.SentCmd=cm_GetHoursHv then begin
    cmd:=Str(CmdTab.SentCmd);
    arg:='';
   end else
   if CmdTab.SentCmd=cm_ReadStatus then begin
    cmd:=Str(CmdTab.SentCmd);
    arg:='';
   end;
   if Length(cmd)>0 then begin
    PurgeComPortSpec;
    if Length(arg)=0
    then Details(CommentCmdId(CmdTab.SentCmd))
    else Details(CommentCmdId(CmdTab.SentCmd)+' = '+Arg);
    ComSendCmd(cmd,arg);
    ComStatus:=rs_WaitAnswer;
    WaitReplyStart:=mSecNow;
   end else begin
    ComStatus:=rs_NoRequest;
    PrepareNextCmd(1);
   end;
  end;
  cmd:='';
  arg:='';
 end;
 {
 Handle "WaitQueue detected" event
 }
 procedure OnWaitQueue;
 begin
  if mSecNow<WaitQueueStart+ComTimeGap
  then Details('Waiting queue...')
  else ComStatus:=rs_NoRequest;
 end;
 {
 Handle "Request sent, waiting answer" state.
 }
 procedure OnWaitReply;
 var Check:Integer;
 begin
  if Length(ComReply)>MaxStrLeng-MaxComLeng then ComReply:='';
  ComReply:=ComReply+ComRead(MaxComLeng);
  Details('Waiting reply... '+Hex_Encode(ComReply)+' '+ComReply);
  Check:=CheckCommand(ComReply,MinAnsLeng,csAnsAdd);
  if Check=cc_Success then begin
   if ExtractCmd(ComReply)=Str(CmdTab.SentCmd) then begin
    ComTimeOut:=Round(Max(MinTimeOut,Min(MaxTimeOut,ComTimeOut/TimeOutFac)));
    UpdateAo(ao_AnsTime,time,Round(mSecNow-WaitReplyStart));
    bNul(iSetTag(tagAnsTime,Round(mSecNow-WaitReplyStart)));
    ViewImp('COM: $'+Hex_Encode(ComReply)+' '+ComReply);
    UpdateDo(do_Connect,time,1);
    bNul(iSetTag(tagConnect,1));
    ComStatus:=rs_Answer;
   end else begin
    Failure(errorcodebadfrmt,'BadFormat error detected: '+Hex_Encode(ComReply)+' '+ComReply);
    ComStatus:=rs_WaitQueue;
    WaitQueueStart:=mSecNow;
   end;
  end else
  if Check=cc_BadSumm then begin
   Failure(errorcodechksumm,'CheckSumm error detected: '+Hex_Encode(ComReply)+' '+ComReply);
   ComStatus:=rs_WaitQueue;
   WaitQueueStart:=mSecNow;
  end else
  if mSecNow>WaitReplyStart+ComTimeOut
  then ComStatus:=rs_Timeout;
 end;
 {
 Handle "Came Reply, data received"...
 }
 procedure OnCameReply;
 var HvOnOff,InterLk,FaultSt,TSicAdc,VoltAdc,CurrAdc,TMulAdc:Integer;
     TSicGet,VoltSet,VoltGet,CurrSet,CurrGet,TMulGet,HvHours:Real;
     OnOffXor,InterXor,FaultXor:Integer;
  function CheckReply(Expected:String):Boolean;
  begin
   if ExtractArg(1,ComReply)=Expected then begin
    PrepareNextCmd(1);
    CheckReply:=True;
   end else CheckReply:=False;
  end;
 begin
  if CmdTab.SentCmd=cm_GetModelId then begin
   if CheckReply(ExtractWord(1,ModelId)) then begin
    Actual.ModelId:=Wanted.ModelId;
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error GetModelId: '+ExtractArg(1,ComReply));
   end;
  end else
  if CmdTab.SentCmd=cm_SetRemoteM then begin
   if CheckReply(Dump('$')) then begin
    Actual.RemoteM:=Wanted.RemoteM;
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error SetRemoteM: '+ExtractArg(1,ComReply));
   end;
  end else
  if CmdTab.SentCmd=cm_SetInterLk then begin
   if CheckReply(Dump('$')) then begin
    Actual.InterLk:=Wanted.InterLk;
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error SetInterLk: '+ExtractArg(1,ComReply));
   end;
  end else
  if CmdTab.SentCmd=cm_SetVoltage then begin
   if CheckReply(Dump('$')) then begin
    Actual.Voltage:=Wanted.Voltage;
    VoltSet:=CodeToVoltage(Actual.Voltage);
    UpdateAo(ao_VoltGot,time,VoltSet); bNul(rSetTag(tagVoltGot,VoltSet));
    UpdateAo(ao_VoltDac,time,Actual.Voltage); bNul(iSetTag(tagVoltDac,Actual.Voltage));
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error SetVoltage: '+ExtractArg(1,ComReply));
   end;
  end else
  if CmdTab.SentCmd=cm_SetCurrent then begin
   if CheckReply(Dump('$')) then begin
    Actual.Current:=Wanted.Current;
    CurrSet:=CodeToCurrent(Actual.Current);
    UpdateAo(ao_CurrGot,time,CurrSet); bNul(rSetTag(tagCurrGot,CurrSet));
    UpdateAo(ao_CurrDac,time,Actual.Current); bNul(iSetTag(tagCurrDac,Actual.Current));
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error SetCurrent: '+ExtractArg(1,ComReply));
   end;
  end else
  if CmdTab.SentCmd=cm_SetHvOnOff then begin
   if CheckReply(Dump('$')) then begin
    Actual.HvOnOff:=Wanted.HvOnOff;
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error SetHvOnOff: '+ExtractArg(1,ComReply));
    EnforceUpdates(ComStatus);
   end;
  end else
  if CmdTab.SentCmd=cm_ResetHours then begin
   if CheckReply(Dump('$')) then begin
    Actual.ResetHs:=Wanted.ResetHs;
    Details('Ok');
   end else begin
    Failure(errorcodeharderr,'Error ResetHours: '+ExtractArg(1,ComReply));
    EnforceUpdates(ComStatus);
   end;
  end else
  if CmdTab.SentCmd=cm_ReadAnalog then begin
   TSicAdc:=Val(ExtractArg(1,ComReply)); TSicGet:=CodeToTempSic(TSicAdc);
   VoltAdc:=Val(ExtractArg(3,ComReply)); VoltGet:=CodeToVoltage(VoltAdc);
   CurrAdc:=Val(ExtractArg(4,ComReply)); CurrGet:=CodeToCurrent(CurrAdc);
   TMulAdc:=Val(ExtractArg(7,ComReply)); TMulGet:=CodeToTempMul(TMulAdc);
   UpdateAo(ao_TempSic,time,TSicGet); bNul(rSetTag(tagTempSic,TSicGet));
   UpdateAo(ao_VoltAdc,time,VoltAdc); bNul(iSetTag(tagVoltAdc,VoltAdc));
   UpdateAo(ao_CurrAdc,time,CurrAdc); bNul(iSetTag(tagCurrAdc,CurrAdc));
   UpdateAo(ao_VoltGet,time,VoltGet); bNul(rSetTag(tagVoltGet,VoltGet));
   UpdateAo(ao_CurrGet,time,CurrGet); bNul(rSetTag(tagCurrGet,CurrGet));
   UpdateAo(ao_TempMul,time,TMulGet); bNul(rSetTag(tagTempMul,TMulGet));
   PrepareNextCmd(1);
  end else
  if CmdTab.SentCmd=cm_GetHoursHv then begin
   HvHours:=rVal(ExtractArg(1,ComReply));
   UpdateAo(ao_HvHours,time,HvHours); bNul(rSetTag(tagHvHours,HvHours));
   PrepareNextCmd(1);
  end else
  if CmdTab.SentCmd=cm_ReadStatus then begin
   OnOffXor:=Ord(IsSeria('SR'));
   InterXor:=Ord(IsSeria('--'));
   FaultXor:=Ord(IsSeria('SR'));
   HvOnOff:=iXor(OnOffXor,Val(ExtractArg(1,ComReply)));
   InterLk:=iXor(InterXor,Val(ExtractArg(2,ComReply)));
   FaultSt:=iXor(FaultXor,Val(ExtractArg(3,ComReply)));
   UpdateDo(do_HvOnOff,time,HvOnOff); bNul(iSetTag(tagHvOnOff,HvOnOff));
   UpdateDo(do_InterLk,time,InterLk); bNul(iSetTag(tagInterLk,InterLk));
   UpdateDo(do_FaultSt,time,FaultSt); bNul(iSetTag(tagFaultSt,FaultSt));
   PrepareNextCmd(1);
  end;
  ComStatus:=rs_WaitQueue;
  WaitQueueStart:=mSecNow;
 end;
 {
 Handle "TimeOut detected" event
 }
 procedure OnTimeOut;
 begin
  ComReply:='';
  bNul(iSetTag(tagConnect,0));
  UpdateDo(do_Connect,time,0);
  EnforceUpdates(rs_NoRequest);
  bNul(iSetTag(tagAnsTime,Round(mSecNow-WaitReplyStart)));
  UpdateAo(ao_AnsTime,time,Round(mSecNow-WaitReplyStart));
  Failure(errorcodetimeout,'TimeOut('+Str(ComTimeOut)+') detected!');
  ComTimeOut:=Round(Max(MinTimeOut,Min(MaxTimeOut,ComTimeOut*TimeOutFac)));
 end;
 {
 Handle command Cmd with argument Arg coming from Com port.
 Uses in simulation mode.
 }
 procedure Sim_Handle(Cmd,Arg:String);
 var CmdId,csTmp:Integer; OnOffXor,InterXor,FaultXor:Integer;
 begin
  CmdId:=Val(Cmd);
  csTmp:=csAnsAdd;
  csAnsAdd:=csAnsAdd+Sim.CheckBug;
  if Sim.CmdIdBug<>0 then Cmd:=Str(CmdId+Sim.CmdIdBug);
  if Sim.Disabled<>1 then
  if CmdId=cm_SetVoltage then begin
   Details(CommentCmdId(CmdId)+' = '+Arg);
   Wanted.Voltage:=Val(ExtractWord(1,Arg));
   ComSendAns(Cmd,Dump('$'));
  end else
  if CmdId=cm_SetCurrent then begin
   Details(CommentCmdId(CmdId)+' = '+Arg);
   Wanted.Current:=Val(ExtractWord(1,Arg));
   ComSendAns(Cmd,Dump('$'));
  end else
  if CmdId=cm_ReadAnalog then begin
   Details(CommentCmdId(CmdId));
   Sim.Tmul:=TempMulToCode(35+2.5*Random(-1,1)); 
   Sim.Tsic:=TempSicToCode(25+0.5*Random(-1,1));
   ComSendAns(Cmd,Str(Sim.Tsic)+',0,'+Str(Actual.Voltage)+','+Str(Actual.Current)+',0,0,'+Str(Sim.Tmul));
  end else
  if CmdId=cm_GetHoursHv then begin
   Details(CommentCmdId(CmdId));
   ComSendAns(Cmd,StrFix((mSecNow-Sim.HvHours)/3600000,1,2));
  end else
  if CmdId=cm_ReadStatus then begin
   Details(CommentCmdId(CmdId));
   OnOffXor:=Ord(IsSeria('SR'));
   InterXor:=Ord(IsSeria('--'));
   FaultXor:=Ord(IsSeria('SR'));
   ComSendAns(Cmd,Str(iXor(OnOffXor,Actual.HvOnOff))+','+
                  Str(iXor(InterXor,Actual.InterLk))+','+
                  Str(iXor(FaultXor,Actual.FaultSt)));
  end else
  if CmdId=cm_GetModelId then begin
   Details(CommentCmdId(CmdId));
   ComSendAns(Cmd,ModelId);
  end else
  if CmdId=cm_ResetHours then begin
   Details(CommentCmdId(CmdId));
   ComSendAns(Cmd,Dump('$'));
   Sim.HvHours:=mSecNow;
  end else
  if CmdId=cm_SetInterLk then begin
   Details(CommentCmdId(CmdId)+' = '+Arg);
   //Wanted.InterLk:=Val(ExtractWord(1,Arg));
   ComSendAns(Cmd,Dump('$'));
  end else
  if CmdId=cm_SetRemoteM then begin
   Details(CommentCmdId(CmdId)+' = '+Arg);
   Wanted.REMOTEM:=Val(ExtractWord(1,Arg));
   ComSendAns(Cmd,Dump('$'));
  end else
  if CmdId=cm_SetHvOnOff then begin
   Details(CommentCmdId(CmdId)+' = '+Arg);
   Wanted.HvOnOff:=Val(ExtractWord(1,Arg));
   ComSendAns(Cmd,Dump('$'));
  end else
  Trouble('Unknown command: '+Cmd+','+Arg);
  Sim.CmdIdBug:=0;
  Sim.CheckBug:=0;
  csAnsAdd:=csTmp;
 end;
 {
 Process commands coming from COM port.
 Uses in simulation mode.
 }
 procedure Sim_Process(Data:String);
 begin
  if Length(Data)>0 then begin
   ViewImp('COM: $'+Hex_Encode(Data)+' '+Data);
   if CheckCommand(Data,MinCmdLeng,csCmdAdd)=cc_Success then begin
    Details('Received Command="'+ExtractMsg(Data)+'"');
    Sim_Handle(ExtractCmd(Data),ExtractArgs(Data));
   end;
  end;
  Actual.HvOnOff:=Wanted.HvOnOff;
  Actual.InterLk:=Wanted.InterLk;
  Actual.FaultSt:=Wanted.FaultSt;
  Actual.Voltage:=Wanted.Voltage*Ord(Actual.HvOnOff>0);
  if Actual.Voltage<MinVoltageCode then Actual.Voltage:=MinVoltageCode;
  if Actual.Voltage>MaxVoltageCode then Actual.Voltage:=MaxVoltageCode;
  Actual.Current:=CurrentToCode(CodeToVoltage(Actual.Voltage)/SimRLoad);
  if Actual.Current<MinCurrentCode then Actual.Current:=MinCurrentCode;
  if Actual.Current>MaxCurrentCode then Actual.Current:=MaxCurrentCode;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 var i:Integer;
 begin
  ComReply:='';
  SeriaId:=''; kVoltId:=''; PolarId:='';
  PowerId:=''; ModelId:=''; ModelSign:='';
  for i:=0 to MaxCmdNum do CmdTab.Comment[i]:='';
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  {
  Initialize special errors...
  }
  errorcodetimeout:=RegisterErr(DevName+' TimeOut');
  errorcodechksumm:=RegisterErr(DevName+' CheckSumm');
  errorcodebadfrmt:=RegisterErr(DevName+' BadFormat');
  errorcodeharderr:=RegisterErr(DevName+' Hardware error');
  {
  Initialize tags...
  Some tags MUST be co-named with curves!
  }
  InitTag(tagHvOnOff,    CrvName(RefDo(do_HvOnOff)), -1);
  InitTag(tagInterLk,    CrvName(RefDo(do_InterLk)), -1);
  InitTag(tagFaultSt,    CrvName(RefDo(do_FaultSt)), -1);
  InitTag(tagConnect,    CrvName(RefDo(do_Connect)), -1);
  InitTag(tagVoltGot,    CrvName(RefAo(ao_VoltGot)), -2);
  InitTag(tagCurrGot,    CrvName(RefAo(ao_CurrGot)), -2);
  InitTag(tagVoltDac,    CrvName(RefAo(ao_VoltDac)), -1);
  InitTag(tagCurrDac,    CrvName(RefAo(ao_CurrDac)), -1);
  InitTag(tagVoltGet,    CrvName(RefAo(ao_VoltGet)), -2);
  InitTag(tagCurrGet,    CrvName(RefAo(ao_CurrGet)), -2);
  InitTag(tagVoltAdc,    CrvName(RefAo(ao_VoltAdc)), -1);
  InitTag(tagCurrAdc,    CrvName(RefAo(ao_CurrAdc)), -1);
  InitTag(tagHvHours,    CrvName(RefAo(ao_HvHours)), -2);
  InitTag(tagTempSic,    CrvName(RefAo(ao_TempSic)), -2);
  InitTag(tagTempMul,    CrvName(RefAo(ao_TempMul)), -2);
  InitTag(tagAnsTime,    CrvName(RefAo(ao_AnsTime)), -1);
  InitTag(tagResetHours, ReadIni('tagResetHours'),   -1);
  bNul(iSetTag(tagConnect,0));
  {
  Read and open ComPort
  }
  ComPort:=Val(ReadIni('ComPort'));
  if ComPort>0 then begin
   if ComOpen('[SerialPort-COM'+Str(ComPort)+']')
   then Success('ComPort '+Str(ComPort)+' opened') else begin
    Trouble('Could not open ComPort '+Str(ComPort));
    ComPort:=0;
   end;
  end else begin
   Trouble('Invalid ComPort='+Str(ComPort));
   ComPort:=0;
  end;
  {
  Initialize state
  }
  InitModel(ReadIni('ModelName'));
  Simulator:=Val(ReadIni('Simulator'))>0;
  Success('Simulator = '+Str(Ord(Simulator)));
  MinTimeOut:=Val(ReadIni('MinTimeOut'));
  MaxTimeOut:=Val(ReadIni('MaxTimeOut'));
  ComTimeOut:=Val(ReadIni('ComTimeOut'));
  ComTimeGap:=Val(ReadIni('ComTimeGap'));
  if MinTimeOut<=0 then MinTimeOut:=TimeOutMin;
  if MaxTimeOut<=0 then MaxTimeOut:=TimeOutMax;
  if ComTimeOut<=0 then ComTimeOut:=TimeOutDef;
  if ComTimeGap<=0 then ComTimeGap:=TimeGapDef;
  MinTimeOut:=Round(Max(TimeOutMin,Min(TimeOutMax,MinTimeOut)));
  MaxTimeOut:=Round(Max(TimeOutMin,Min(TimeOutMax,MaxTimeOut)));
  ComTimeOut:=Round(Max(MinTimeOut,Min(MaxTimeOut,ComTimeOut)));
  Success('TimeOuts: '+Str(MinTimeOut)+','+Str(ComTimeOut)+','+Str(MaxTimeOut));
  Sim.HvHours:=mSecNow;
  Sim.Disabled:=0;   
  Sim.Tsic:=0;       Sim.Tmul:=0;
  Sim.CmdIdBug:=0;   Sim.CheckBug:=0;
  WaitQueueStart:=0; WaitReplyStart:=0;
  Wanted.ResetHs:=0; Actual.ResetHs:=0;
  Wanted.Voltage:=0; Actual.Voltage:=0;
  Wanted.Current:=0; Actual.Current:=0;
  Wanted.RemoteM:=0; Actual.RemoteM:=0;
  Wanted.InterLk:=0; Actual.InterLk:=0;
  Wanted.HvOnOff:=0; Actual.HvOnOff:=0;
  if ComPort>0
  then ResetCmd(rs_NoRequest)
  else ResetCmd(rs_NotAvail);
  TestConvertion;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if ComPort>0 then bNul(ComClose);
  ComPort:=0;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  {
  Process Com port input...
  }
  if Simulator then begin
   if ComPort>0 then Sim_Process(ComRead(MaxComLeng));
  end else begin
   if ComStatus=rs_NoRequest  then OnNoRequest else
   if ComStatus=rs_WaitQueue  then OnWaitQueue else
   if ComStatus=rs_WaitAnswer then OnWaitReply else
   if ComStatus=rs_Answer    then  OnCameReply else
   if ComStatus=rs_TimeOut   then  OnTimeOut   else OnNotAvail;
  end;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; i:Integer; r:Real;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   Example: @Sim.Disabled=1
   }
   if IsSameText(cmd,'@Sim.Disabled') then begin
    if Simulator and not IsNan(rVal(arg)) then Sim.Disabled:=Ord(rVal(arg)<>0);
    Success(cmd+'='+Str(Sim.Disabled));
    Data:='';
   end else
   {
   Example: @Sim.CheckBug=1
   }
   if IsSameText(cmd,'@Sim.CheckBug') then begin
    if Simulator and not IsNan(rVal(arg)) then Sim.CheckBug:=Ord(rVal(arg)<>0);
    Success(cmd+'='+Str(Sim.CheckBug));
    Data:='';
   end else
   {
   Example: @Sim.CmdIdBug=1
   }
   if IsSameText(cmd,'@Sim.CmdIdBug') then begin
    if Simulator and not IsNan(rVal(arg)) then Sim.CmdIdBug:=Ord(rVal(arg)<>0);
    Success(cmd+'='+Str(Sim.CmdIdBug));
    Data:='';
   end else
   {
   Example: @Sim.InterLk=1
   }
   if IsSameText(cmd,'@Sim.InterLk') then begin
    if Simulator and not IsNan(rVal(arg)) then Wanted.InterLk:=Ord(rVal(arg)<>0);
    Success(cmd+'='+Str(Wanted.InterLk));
    Data:='';
   end else
   {
   Example: @Sim.FaultSt=1
   }
   if IsSameText(cmd,'@Sim.FaultSt') then begin
    if Simulator and not IsNan(rVal(arg)) then Wanted.FaultSt:=Ord(rVal(arg)<>0);
    Success(cmd+'='+Str(Wanted.FaultSt));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
