#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for DEMO_DIM_PING.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

#####################################################
### Tool to Write dim_ping_benchmark DAQ config file.
#####################################################

###############
# InitVariables
###############
readonly devmsg="devPostMsg";            # Method to send/post messages
readonly FACILITY="DIM_PING";            # Facility name
declare PREFIX="DEMO";                   # DIM service prefix
declare -i VERBOSE=0;
declare -i NUMSERV=10;
declare DIM_MODE="UNKNOWN";
declare SERVER_CONFIG="$scriptHOME/${scriptbase:0:8}_dis.cfg";
declare CLIENT_CONFIG="$scriptHOME/${scriptbase:0:8}_dic.cfg";

function call(){
 if [ $# -gt 0 ]; then
  eval "$@";
 fi;
};

function Version(){
 echo "$1 version 1.0";
};

function Help(){
 call Version $1;
 echo "Copyright(c) 2021 Alexey Kuryakin kouriakinemail.ru";
 echo "Tool to generate dim_ping_benchmark configuration";
 echo "Usage :";
 echo " call $1 [options]";
 echo "Options:";
 echo " -h --help      to print help";
 echo " --version      to print version";
 echo " --dns node     to set DIM_DNS_NODE=node";
 echo " --prefix name  to set DIM service prefix [name] like DEMO";
 echo " --numserv  n   to set number of services";
 echo "Examples:";
 echo " call $1 --dns localhost --prefix DEMO --numserv 10";
 echo "";
};

function WriteDimDnsTask(){
 echo "[&DimSrv]";
 echo "DIM_DNS_NODE = $DIM_DNS_NODE";
 echo "DIM_TASK     = DIM_TASK/$PREFIX/$FACILITY/SERVER";
 echo "StdInFifo    = 4096";
 echo "StdOutFifo   = 4096";
 echo "StdInPipe    = 4096";
 echo "StdOutPipe   = 4096";
 echo "[]";
 echo "";
};

function WriteTagList(){
 echo "[TagList]";
 echo "$PREFIX.$FACILITY.DIC_CMND_RATE    = real    0    ; DIM client command     send per second";
 echo "$PREFIX.$FACILITY.DIS_CMND_RATE    = real    0    ; DIM server command     read per second";
 echo "$PREFIX.$FACILITY.DIC_INFO_RATE    = real    0    ; DIM client information read per second";
 echo "$PREFIX.$FACILITY.DIS_INFO_RATE    = real    0    ; DIM server information send per second";
 echo "$PREFIX.$FACILITY.DIC_CMND_COUNT   = real    0    ; DIM client command     send counter";
 echo "$PREFIX.$FACILITY.DIS_CMND_COUNT   = real    0    ; DIM server command     read counter";
 echo "$PREFIX.$FACILITY.DIC_INFO_COUNT   = real    0    ; DIM client information read counter";
 echo "$PREFIX.$FACILITY.DIS_INFO_COUNT   = real    0    ; DIM server information send counter";
 echo "$PREFIX.$FACILITY.DIC_CMND_B_RATE  = real    0    ; DIM client command     send per second";
 echo "$PREFIX.$FACILITY.DIS_CMND_B_RATE  = real    0    ; DIM server command     read per second";
 echo "$PREFIX.$FACILITY.DIC_INFO_B_RATE  = real    0    ; DIM client information read per second";
 echo "$PREFIX.$FACILITY.DIS_INFO_B_RATE  = real    0    ; DIM server information send per second";
 echo "$PREFIX.$FACILITY.DIC_CMND_B_COUNT = real    0    ; DIM client command     send counter";
 echo "$PREFIX.$FACILITY.DIS_CMND_B_COUNT = real    0    ; DIM server command     read counter";
 echo "$PREFIX.$FACILITY.DIC_INFO_B_COUNT = real    0    ; DIM client information read counter";
 echo "$PREFIX.$FACILITY.DIS_INFO_B_COUNT = real    0    ; DIM server information send counter";
 echo "$PREFIX.$FACILITY.PING_SEND_BTN    = integer 0    ; Send ping to DIM server button   [Ping]";
 echo "$PREFIX.$FACILITY.PING_SEND_CYCLE  = integer 0    ; Send ping to DIM server checkbox [Cycle]";
 echo "$PREFIX.$FACILITY.PING_SEND_COUNT  = integer 1    ; Send ping to DIM server [Iteration Count]";
 echo "$PREFIX.$FACILITY.PING_SEND_PERIOD = integer 100  ; Send ping to DIM server [Period ms]";
 echo "$PREFIX.$FACILITY.PING_SERVER_ON   = integer 1    ; Ping server is ON flag";
 echo "$PREFIX.$FACILITY.PING_CLIENT_ON   = integer 1    ; Ping client is ON flag";
 echo "$PREFIX.$FACILITY.PING_TIME        = real    0    ; Ping time, i.e. send/receive response time, [ms]";
 echo "$PREFIX.$FACILITY.PING_TIME_MEAN   = real    0    ; Ping time, mean value (average value), [ms]";
 echo "$PREFIX.$FACILITY.PING_TIME_RMSD   = real    0    ; Ping time, Root Mean Square Deviation, [ms]";
 echo "$PREFIX.$FACILITY.PING_STAT_PERIOD = integer 1000 ; Ping time, statistics collect perod, [ms]";
 echo "$PREFIX.$FACILITY.NUMSERV          = integer $NUMSERV    ; Number of services";
 echo "$PREFIX.$FACILITY.DATALEN          = integer 24   ; Data lenght to ping [bytes]";
 echo "$PREFIX.$FACILITY.DIM_TASK         = string  DIM_TASK/$PREFIX/$FACILITY/SERVER";
 echo "$PREFIX.$FACILITY.DIM_DNS_NODE     = string  $DIM_DNS_NODE";
 echo "$PREFIX.$FACILITY.DIM_MODE         = string  $DIM_MODE";
 echo "$PREFIX.$FACILITY.CPU_LOAD         = real    0    ; CPU Load, Percent";
 for i in $(seq 1 $NUMSERV); do call WriteTagServIter $i; done;
 echo "[]";
 echo "";
};

function WriteTagServIter(){
 if [ -z "$1" ]; then return; fi;
 echo "$PREFIX.$FACILITY.DIC_CMND_$1 = string  ?";
 echo "$PREFIX.$FACILITY.DIS_CMND_$1 = string  ?";
 echo "$PREFIX.$FACILITY.DIC_INFO_$1 = string  ?";
 echo "$PREFIX.$FACILITY.DIS_INFO_$1 = string  ?";
};

function WriteDataStorage(){
 echo "[DataStorage]";
 echo "$PREFIX.$FACILITY.PING_TIME_HISTOGRAM = Curve 1002 0 Black       15 0";
 echo "$PREFIX.$FACILITY.PING_TIME           = Curve 0 1000 Black       15 1";
 echo "$PREFIX.$FACILITY.PING_TIME_MEAN      = Curve 0 1000 Black       15 1";
 echo "$PREFIX.$FACILITY.PING_TIME_RMSD      = Curve 0 1000 Black       15 1";
 echo "$PREFIX.$FACILITY.DIC_CMND_RATE       = Curve 0 1000 LightRed     0 2";
 echo "$PREFIX.$FACILITY.DIS_CMND_RATE       = Curve 0 1000 Red          0 2";
 echo "$PREFIX.$FACILITY.DIC_INFO_RATE       = Curve 0 1000 LightBlue    0 2";
 echo "$PREFIX.$FACILITY.DIS_INFO_RATE       = Curve 0 1000 Blue         0 2";
 echo "$PREFIX.$FACILITY.DIC_CMND_B_RATE     = Curve 0 1000 LightRed     0 2";
 echo "$PREFIX.$FACILITY.DIS_CMND_B_RATE     = Curve 0 1000 Red          0 2";
 echo "$PREFIX.$FACILITY.DIC_INFO_B_RATE     = Curve 0 1000 LightBlue    0 2";
 echo "$PREFIX.$FACILITY.DIS_INFO_B_RATE     = Curve 0 1000 Blue         0 2";
 echo "$PREFIX.$FACILITY.CPU_LOAD            = Curve 0 1000 Black        0 2";
 echo "[]";
 echo "";
};

function WriteMainCtrl(){
 echo "[$FACILITY.MAIN.CTRL]";
 echo "Link sensor DIC_CMND_RATE      with curve $PREFIX.$FACILITY.DIC_CMND_RATE    device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIS_CMND_RATE      with curve $PREFIX.$FACILITY.DIS_CMND_RATE    device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIC_INFO_RATE      with curve $PREFIX.$FACILITY.DIC_INFO_RATE    device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIS_INFO_RATE      with curve $PREFIX.$FACILITY.DIS_INFO_RATE    device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIC_CMND_COUNT     with tag   $PREFIX.$FACILITY.DIC_CMND_COUNT   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIS_CMND_COUNT     with tag   $PREFIX.$FACILITY.DIS_CMND_COUNT   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIC_INFO_COUNT     with tag   $PREFIX.$FACILITY.DIC_INFO_COUNT   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIS_INFO_COUNT     with tag   $PREFIX.$FACILITY.DIS_INFO_COUNT   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_SEND_BTN      with tag   $PREFIX.$FACILITY.PING_SEND_BTN    device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_SEND_CYCLE    with tag   $PREFIX.$FACILITY.PING_SEND_CYCLE  device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_SEND_COUNT    with tag   $PREFIX.$FACILITY.PING_SEND_COUNT  device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_SEND_PERIOD   with tag   $PREFIX.$FACILITY.PING_SEND_PERIOD device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_SERVER_ON     with tag   $PREFIX.$FACILITY.PING_SERVER_ON   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_CLIENT_ON     with tag   $PREFIX.$FACILITY.PING_CLIENT_ON   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_TIME          with curve $PREFIX.$FACILITY.PING_TIME        device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_TIME_MEAN     with curve $PREFIX.$FACILITY.PING_TIME_MEAN   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_TIME_RMSD     with curve $PREFIX.$FACILITY.PING_TIME_RMSD   device &$FACILITY.MAIN.CTRL";
 echo "Link sensor PING_STAT_PERIOD   with tag   $PREFIX.$FACILITY.PING_STAT_PERIOD device &$FACILITY.MAIN.CTRL";
 echo "Link sensor NUMSERV            with tag   $PREFIX.$FACILITY.NUMSERV          device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DATALEN            with tag   $PREFIX.$FACILITY.DATALEN          device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIM_TASK           with tag   $PREFIX.$FACILITY.DIM_TASK         device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIM_DNS_NODE       with tag   $PREFIX.$FACILITY.DIM_DNS_NODE     device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIM_MODE           with tag   $PREFIX.$FACILITY.DIM_MODE         device &$FACILITY.MAIN.CTRL";
 echo "Link sensor CPU_LOAD           with curve $PREFIX.$FACILITY.CPU_LOAD         device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @DIM_GUI                                                    with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @DIM_TREE                                                   with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @RESET_COUNTS                                               with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @PING_CONFIGS                                               with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @PING_REPORT+NEW                                            with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @PING_REPORT+ADD                                            with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @PING_REPORT+EDIT                                           with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor @PING_REPORT+SEE                                            with device &$FACILITY.MAIN.CTRL";
 echo "Link sensor DIM_DATAFLOW_DIAGRAM                                        with device &$FACILITY.MAIN.CTRL";
 echo "[]";
 echo "";
};

function WriteSpectrWindow(){
 echo "[Windows]";
 echo "$FACILITY.PING_TIME_HISTOGRAM = Spectr_Window";
 echo "[$FACILITY.PING_TIME_HISTOGRAM]";
 echo "WindowClass    = SpecManFitWin";
 echo "Spectr         = $PREFIX.$FACILITY.PING_TIME_HISTOGRAM";
 echo "Spectrometr    = &$FACILITY.MAIN.CTRL";
 echo "EnCalibration  = 0";
 echo "HwCalibration  = 1";
 echo "MarkerLabel    = CHN,CNT,PING,*,ROI,*,INT,GND,PIK ; Chan,Count,Ping,HW";
 echo "[]";
 echo "";
};

function WriteCurveWindow(){
 echo "[Windows]";
 echo "$FACILITY.PLOT.PING_TIME = Curve_Window";
 echo "[$FACILITY.PLOT.PING_TIME]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.016666667";
 echo "AxisY = ^CPing_Time^N^L___{ms}, 0, 1000";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME_MEAN";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME_RMSD";
 echo "[]";
 echo "[Windows]";
 echo "$FACILITY.PLOT.PING_RATE = Curve_Window";
 echo "[$FACILITY.PLOT.PING_RATE]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.016666667";
 echo "AxisY = ^CPing_Rate^N^L___{Ping/sec}, 0, 1e3";
 echo "CurveList = $PREFIX.$FACILITY.DIC_CMND_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_CMND_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIC_INFO_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_INFO_RATE";
 echo "[]";
 echo "[Windows]";
 echo "$FACILITY.PLOT.PING_B_RATE = Curve_Window";
 echo "[$FACILITY.PLOT.PING_B_RATE]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.016666667";
 echo "AxisY = ^CData_Rate^N^L___{Byte/sec}, 0, 1e5";
 echo "CurveList = $PREFIX.$FACILITY.DIC_CMND_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_CMND_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIC_INFO_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_INFO_B_RATE";
 echo "[]";
 echo "[Windows]";
 echo "$FACILITY.PLOT.CPU_LOAD = Curve_Window";
 echo "[$FACILITY.PLOT.CPU_LOAD]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.016666667";
 echo "AxisY = ^CCPU_Load^N^L___{Percent}, 0, 100";
 echo "CurveList = $PREFIX.$FACILITY.CPU_LOAD";
 echo "[]";
 echo "";
};

function WriteTabWindow(){
 echo "[Windows]";
 echo "$FACILITY.TAB.PING_PARAM = Tab_Window";
 echo "[$FACILITY.TAB.PING_PARAM]";
 echo "Font      = Name:PT_Mono\Color:Black\Size:14\Style:[Bold]";
 echo "Format    = 16 0 f";
 echo "Colors    = Default, White, Navy";
 echo "TagList   = $PREFIX.$FACILITY.PING_TIME";
 echo "TagList   = $PREFIX.$FACILITY.PING_TIME_MEAN";
 echo "TagList   = $PREFIX.$FACILITY.PING_TIME_RMSD";
 echo "TagList   = $PREFIX.$FACILITY.DIC_CMND_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIS_CMND_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIC_INFO_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIS_INFO_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIC_CMND_B_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIS_CMND_B_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIC_INFO_B_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIS_INFO_B_RATE";
 echo "TagList   = $PREFIX.$FACILITY.DIC_CMND_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIS_CMND_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIC_INFO_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIS_INFO_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIC_CMND_B_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIS_CMND_B_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIC_INFO_B_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.DIS_INFO_B_COUNT";
 echo "TagList   = $PREFIX.$FACILITY.CPU_LOAD";
 echo "[]";
 echo "";
};

function WriteDatSrv(){
 echo "[&DatSrv]";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME_MEAN";
 echo "CurveList = $PREFIX.$FACILITY.PING_TIME_RMSD";
 echo "CurveList = $PREFIX.$FACILITY.DIC_CMND_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_CMND_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIC_INFO_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_INFO_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIC_CMND_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_CMND_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIC_INFO_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.DIS_INFO_B_RATE";
 echo "CurveList = $PREFIX.$FACILITY.CPU_LOAD";
 echo "[]";
 echo "";
};

function WriteCustomParameters(){
 echo "[CustomParameters.TagList]";
 echo "TagList = $PREFIX.$FACILITY.PING_STAT_PERIOD";
 echo "TagList = $PREFIX.$FACILITY.PING_SEND_CYCLE";
 echo "TagList = $PREFIX.$FACILITY.PING_SEND_COUNT";
 echo "TagList = $PREFIX.$FACILITY.PING_SEND_PERIOD";
 echo "TagList = $PREFIX.$FACILITY.PING_SERVER_ON";
 echo "TagList = $PREFIX.$FACILITY.PING_CLIENT_ON";
 echo "TagList = $PREFIX.$FACILITY.DATALEN";
 echo "[]";
 echo "";
};

function WriteTagPrefix(){
 echo "[&$FACILITY.MAIN.CTRL]";
 echo "tagDimPing    = $PREFIX.$FACILITY";
 echo "DIM_MODE      = $DIM_MODE";
 echo "NumServ       = $NUMSERV";
 echo "winHistogram  = $FACILITY.PING_TIME_HISTOGRAM";
 echo "winPingTime   = $FACILITY.PLOT.PING_TIME";
 echo "winPingRate   = $FACILITY.PLOT.PING_RATE";
 echo "winPingTab    = $FACILITY.TAB.PING_PARAM";
 echo "winCpuLoad    = $FACILITY.PLOT.CPU_LOAD";
 echo "winPingBRate  = $FACILITY.PLOT.PING_B_RATE";
 echo "AnalogOutputs = 13";
 echo "Link AnalogOutput 0  with curve $PREFIX.$FACILITY.PING_TIME_HISTOGRAM";
 echo "Link AnalogOutput 1  with curve $PREFIX.$FACILITY.PING_TIME           history 50000";
 echo "Link AnalogOutput 2  with curve $PREFIX.$FACILITY.DIC_CMND_RATE       history 5000";
 echo "Link AnalogOutput 3  with curve $PREFIX.$FACILITY.DIS_CMND_RATE       history 5000";
 echo "Link AnalogOutput 4  with curve $PREFIX.$FACILITY.DIC_INFO_RATE       history 5000";
 echo "Link AnalogOutput 5  with curve $PREFIX.$FACILITY.DIS_INFO_RATE       history 5000";
 echo "Link AnalogOutput 6  with curve $PREFIX.$FACILITY.DIC_CMND_B_RATE     history 5000";
 echo "Link AnalogOutput 7  with curve $PREFIX.$FACILITY.DIS_CMND_B_RATE     history 5000";
 echo "Link AnalogOutput 8  with curve $PREFIX.$FACILITY.DIC_INFO_B_RATE     history 5000";
 echo "Link AnalogOutput 9  with curve $PREFIX.$FACILITY.DIS_INFO_B_RATE     history 5000";
 echo "Link AnalogOutput 10 with curve $PREFIX.$FACILITY.CPU_LOAD            history 5000";
 echo "Link AnalogOutput 11 with curve $PREFIX.$FACILITY.PING_TIME_MEAN      history 5000";
 echo "Link AnalogOutput 12 with curve $PREFIX.$FACILITY.PING_TIME_RMSD      history 5000";
 echo "Calibrations  = 2";
 echo "Calibration#0 = ..\Calibr\PingTime.cal  Chan Ping  * Line Line 0 10000";
 echo "Calibration#1 = ..\Calibr\HalfWidth.cal Chan HW    * Line Line 0 10000";
 echo "[]";
 echo "";
};

function WriteDis(){
 for i in $(seq 1 $NUMSERV); do call WriteDisIter $i; done;
 echo "";
};

function WriteDisIter(){
 if [ -z "$1" ]; then return; fi;
 unix dimcfg \
  -n dis_cmnd $PREFIX/$FACILITY/DIM_CMND_$1 \
  -n tag      $PREFIX.$FACILITY.DIS_CMND_$1 \
  -n $devmsg "&$FACILITY.MAIN.CTRL @DIM_CMND=$1, %**" \
  -n end \
  -n dis_info $PREFIX/$FACILITY/DIM_INFO_$1 \
  -n tag      $PREFIX.$FACILITY.DIS_INFO_$1 \
  -n end \
  -n dic_cmnd $PREFIX/$FACILITY/DIM_CMND_$1 \
  -n tag      $PREFIX.$FACILITY.DIC_CMND_$1 \
  -n end \
  -n dic_info $PREFIX/$FACILITY/DIM_INFO_$1 \
  -n tag      $PREFIX.$FACILITY.DIC_INFO_$1 \
  -n $devmsg "&$FACILITY.MAIN.CTRL @DIM_INFO=$1, %**" \
  -n print   "filling  /////w==  ////  &$FACILITY.MAIN.CTRL" \
  -n ;
};

function WriteDic(){
 for i in $(seq 1 $NUMSERV); do call WriteDicIter $i; done;
 echo "";
};

function WriteDicIter(){
 if [ -z "$1" ]; then return; fi;
 unix dimcfg \
  -n dic_cmnd $PREFIX/$FACILITY/DIM_CMND_$1 \
  -n tag      $PREFIX.$FACILITY.DIC_CMND_$1 \
  -n end \
  -n dic_info $PREFIX/$FACILITY/DIM_INFO_$1 \
  -n tag      $PREFIX.$FACILITY.DIC_INFO_$1 \
  -n $devmsg "&$FACILITY.MAIN.CTRL @DIM_INFO=$1, %**" \
  -n print   "filling  /////w==  ////  &$FACILITY.MAIN.CTRL" \
  -n end \
  -n ;
};

function ValidateVariables(){
 if [ -z "$DIM_DNS_NODE" ]; then export DIM_DNS_NODE="localhost"; fi;
};

function PrintWaitFor(){
 echo "";
 echo "Please wait for generating $1 for $FACILITY";
 echo " with options: --prefix $PREFIX --numserv $NUMSERV --dns $DIM_DNS_NODE";
 echo "";
};

function TypeConfigFile(){
 if [ -n "$1" ] && [ $VERBOSE -gt 0 ] && [ -e "$1" ]; then
  cat "$1";
 fi;
};

function SetEnv(){
 if [ -z "$1" ]; then return 1; fi;
 if [ -z "$2" ]; then return 1; fi;
 export $1="$2";
 return 0;
};

function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

function WriteConfigFile(){
 if [ -z "$DIM_DNS_NODE" ]; then return; fi;
 if [ -z "$FACILITY" ]; then return; fi;
 if [ -z "$PREFIX" ]; then return; fi;
 call WriteDimDnsTask;
 call WriteTagPrefix;
 call WriteTagList;
 call WriteDataStorage;
 call WriteMainCtrl;
 call WriteSpectrWindow;
 call WriteCurveWindow;
 call WriteTabWindow;
 call WriteDatSrv;
 call WriteCustomParameters;
 if [ "$DIM_MODE" = "SERVER" ]; then call WriteDis; fi;
 if [ "$DIM_MODE" = "CLIENT" ]; then call WriteDic; fi;
};

function ParseArguments(){
 if [ -z "$1" ]; then call Help $scriptbase; exit 0; fi;
 while [ -n "$1" ]; do
  case $1 in
   --version)       call Version $scriptbase; exit 0; ;;
   -h|-help|--help) call Help    $scriptbase; exit 0; ;;
   -dns|--dns)         if SetEnv DIM_DNS_NODE "$2"; then  shift; else fatal 1 "Error: Invalid option $1."; fi; ;;
   -numserv|--numserv) if SetEnv NUMSERV      "$2"; then  shift; else fatal 1 "Error: Invalid option $1."; fi; ;;
   -prefix|--prefix)   if SetEnv PREFIX       "$2"; then  shift; else fatal 1 "Error: Invalid option $1."; fi; ;;
   -verbose|--verbose) let "VERBOSE=1"; ;;
   -defaults|--defaults) ;;
   *) ;;
  esac;
  shift;
 done;
};

function MainExecution(){
 ParseArguments "$@";
 call ValidateVariables;
 for a in "$SERVER_CONFIG" "$CLIENT_CONFIG"; do
  if [ "$a" = "$SERVER_CONFIG" ]; then DIM_MODE="SERVER"; fi;
  if [ "$a" = "$CLIENT_CONFIG" ]; then DIM_MODE="CLIENT"; fi;
  call PrintWaitFor         $DIM_MODE;
  call DeleteFiles          $a;
  call WriteConfigFile    > $a;
  call TypeConfigFile       $a;
 done;
 sleep 3;
 return 0;
};

##############
# delete files
##############
function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scriptHOME >/dev/null 2>&1; then
  MainExecution "$@";
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
