{------------------------------------------------------------------------------}
{                                                                              }
{                           Author Yuriy Kopnin                                }
{                            Package VisuaTech                                 }
{                                 LGPL                                         }
{                                                                              }
{------------------------------------------------------------------------------}

unit xFormsBar;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs,
  Themes, LCLIntf, Buttons, ActnList, LCLProc, LCLType, Menus, LMessages,
  ExtCtrls;

type
  TFormItem = class
  public
    Form: TCustomForm;
    ImageIndex: Integer;
    FocusedControl: TWinControl;
  end;

  TFormBarBorder = (fbbLeft, fbbTop, fbbRight, fbbBottom);
  TFormBarBorders = set of TFormBarBorder;

  { TCustomFormsBar }

  TCustomFormsBar = class(TCustomControl)
  private
    FMiniBtnState: TButtonState;
    FormList: TList;
    FHightLightFont: TFont;
    FFormAnderMouse: TCustomForm;
    //FLastActiveFrame: TCustomForm;
    FActiveForm: TCustomForm;
    FCloseBtn: Boolean;
    FSysButton: Boolean;
    FOnCloseBtnDown: TNotifyEvent;
    FOnSysBtnClick: TNotifyEvent;
    FItemDefaultWidth: Integer;
    FMidleLineColor: TColor;
    FImageList: TImageList;
    FWrapText: Boolean;
    FHomeButton: Boolean;
    FHomeButtonState: TButtonState;
    FSysButtonState: TButtonState;
    FOnHomeButtonClick: TNotifyEvent;
    FHomeIconIndex: Integer;
    FSysIconIndex: Integer;
    OrderList: TList;
    FHomeActionLink: TControlActionLink;
    FFormsBarBorders: TFormBarBorders;
    FNative: Boolean;
    FFormsParent: TWinControl;
    FDropDownMenu: TPopupMenu;
    FCloseMiniBtnState: TButtonState;
    CloseBmp: TCustomBitmap;
    ATabList: TFPList;
    FFocusTimer: TTimer;
    function GetDropDownMenu: TPopupmenu;
    function GetFocusTimerEnabled: Boolean;
    function GetFormCount: Integer;
    function GetHomeAction: TBasicAction;
    function GetItems(Index: Integer): TCustomForm;
    function GetLastActiveForm: TCustomForm;
    procedure SetDropDownMenu(AValue: TPopupmenu);
    procedure SetFocusTimerEnabled(AValue: Boolean);
    procedure SetFormsBarBorders(AValue: TFormBarBorders);
    procedure SetFormsParent(AValue: TWinControl);
    procedure SetHighLightFont(AValue: TFont);
    procedure SetFormAnderMouse(AValue: TCustomForm);
    procedure SetHomeAction(AValue: TBasicAction);
    procedure SetHomeButton(AValue: Boolean);
    procedure SetHomeButtonState(AValue: TButtonState);
    procedure SetHomeIconIndex(AValue: Integer);
    procedure SetImageList(AValue: TImageList);
    procedure SetLastActiveForm(AValue: TCustomForm);
    procedure DoHomeActionChange(Sender: TObject);
    procedure SetMiniBtnState(AValue: TButtonState);
    procedure SetNative(AValue: Boolean);
    procedure SetSysButton(AValue: Boolean);
    procedure SetSysButtonState(AValue: TButtonState);
    procedure SetSysIconIndex(AValue: Integer);
    procedure OnFocusTimer(Sender: TObject);
  protected
    CloseBtnWidth: Integer;
    SpaceWidth: Integer;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
       override;
    procedure Paint; override;
    procedure UpdateFormAnderMouse(P: TPoint);
    function AddInList(AForm: TCustomForm; IconIndex: Integer): TFormItem;
    function InsertInList(AForm: TCustomForm; Index: Integer; IconIndex: Integer): TFormItem;
    procedure MouseLeave; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer
       ); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer
       ); override;
    procedure DoCloseBtnDown(AForm: TCustomForm);
    function GetCloseBtnRect(ParentRect: TRect): TRect;
    function GetFormItem(Index: Integer): TFormItem;
    function GetFormItemOfForm(AForm: TCustomForm): TFormItem;
    function GetButtonRect(Index: Integer): TRect; overload;
    function GetButtonRect(AForm: TCustomForm): TRect; overload;
    function GetButtonsRect: TRect;
    function GetSysButtonRect: TRect;
    procedure DoDropDownMenu;
    procedure DoHomeButtonClick;
    procedure DoSysButtonClick;
    procedure HomeActionChange(Sender: TObject);
    procedure Resize; override;

    property HomeButtonState: TButtonState read FHomeButtonState write SetHomeButtonState;
    property SysButtonState: TButtonState read FSysButtonState write SetSysButtonState;
    property FormAnderMouse: TCustomForm read FFormAnderMouse write SetFormAnderMouse;
    property LastActiveForm: TCustomForm read GetLastActiveForm write SetLastActiveForm;
    property HomeActionLink: TControlActionLink read FHomeActionLink write FHomeActionLink;
    property MiniBtnState: TButtonState read FMiniBtnState write SetMiniBtnState;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    procedure CloseAll;
    property Items[Index: Integer]: TCustomForm read GetItems;
    procedure AddForm(AForm: TCustomForm; IconIndex: Integer = -1);
    procedure InsertForm(AForm, AfterForm: TCustomForm; IconIndex: Integer = -1);
    procedure InsertFormWithParent(AForm, ParentForm: TCustomForm);
    procedure ReplaceForm(NewForm, OldForm: TCustomForm; IconIndex: Integer = -1);
    procedure SetActiveControl(AForm: TCustomForm; AWinControl: TWinControl);
    procedure ActivateForm(AForm: TCustomForm);
    procedure RemoveForm(AForm: TCustomForm);
    property FormCount: Integer read GetFormCount;
    property ActiveForm: TCustomForm read FActiveForm;
    property HighLightFont: TFont read FHightLightFont write SetHighLightFont;
    property CloseBtn: Boolean read FCloseBtn write FCloseBtn;
    property SysButton: Boolean read FSysButton write SetSysButton;
    property OnCloseBtnDown: TNotifyEvent read FOnCloseBtnDown write FOnCloseBtnDown;
    property ItemDefaultWidth: Integer read FItemDefaultWidth write FItemDefaultWidth;
    property MildeLineColor: TColor read FMidleLineColor write FMidleLineColor;
    property ImageList: TImageList read FImageList write SetImageList;
    property WrapText: Boolean read FWrapText write FWrapText;
    property HomeButton: Boolean read FHomeButton write SetHomeButton;
    property OnHomeButtonClick: TNotifyEvent read FOnHomeButtonClick write FOnHomeButtonClick;
    property OnSysButtonClick: TNotifyEvent read FOnSysBtnClick write FOnSysBtnClick;
    property HomeIconIndex: Integer read FHomeIconIndex write SetHomeIconIndex;
    property SysIconIndex: Integer read FSysIconIndex write SetSysIconIndex;
    property HomeAction: TBasicAction read GetHomeAction write SetHomeAction;
    property FormsBarBorders: TFormBarBorders read FFormsBarBorders write SetFormsBarBorders default [];
    property Native: Boolean read FNative write SetNative;
    property FormsParent: TWinControl read FFormsParent write SetFormsParent;
    property DropDownMenu: TPopupmenu read GetDropDownMenu write SetDropDownMenu;
    property FocusTimerEnabled: Boolean read GetFocusTimerEnabled write SetFocusTimerEnabled;
  end;

  TFormsBar = class(TCustomFormsBar)
  published
    property Align;
    property Color;
    property Font;
    property ParentColor;
    property ParentFont;
    property HighLightFont;
    property CloseBtn;
    property OnCloseBtnDown;
    property ItemDefaultWidth;
    property MildeLineColor;
    property ImageList;
    property WrapText;
    property HomeButton;
    property OnHomeButtonClick;
    property TabStop;
    property TabOrder;
    property HomeIconIndex;
    property HomeAction;
    property FormsBarBorders;
    property Native;
    property SysButton;
    property SysIconIndex;
    property OnSysButtonClick;
    property FormsParent;
    property DropDownMenu;
  end;

implementation

uses Math, ComCtrls;

{ TCustomFormsBar }

procedure TCustomFormsBar.DoHomeActionChange(Sender: TObject);
begin
  if Sender = HomeAction then HomeActionChange(Sender);
end;

procedure TCustomFormsBar.SetMiniBtnState(AValue: TButtonState);
var
  R: TRect;
begin
  if (FMiniBtnState=AValue) or
     ((AValue = bsHot) and (FMiniBtnState = bsDown)) then Exit;
    FMiniBtnState:= AValue;
    if FFormAnderMouse <> nil then
    begin
      R := GetButtonRect(FFormAnderMouse);
      InvalidateRect(Handle, @R, False);
    end;
    //Invalidate;
end;

procedure TCustomFormsBar.SetNative(AValue: Boolean);
begin
  if FNative=AValue then Exit;
  FNative:=AValue;
  Invalidate;
end;

procedure TCustomFormsBar.SetSysButton(AValue: Boolean);
begin
  if FSysButton <> AValue then
  begin
    FSysButton:= AValue;
    Invalidate;
  end;
end;

procedure TCustomFormsBar.SetSysButtonState(AValue: TButtonState);
var
  R: TRect;
begin
  if FSysButtonState <> AValue then
  begin
    FSysButtonState := AValue;
    R := GetSysButtonRect;
    InvalidateRect(Handle, @R, False);
  end;
end;

procedure TCustomFormsBar.SetSysIconIndex(AValue: Integer);
begin
  if FSysIconIndex <> AValue then
  begin
    FSysIconIndex:=AValue;
    Invalidate;
  end;
end;

procedure TCustomFormsBar.OnFocusTimer(Sender: TObject);
var
  FI: TFormItem;
begin
  if Screen.ActiveControl <> nil then
  begin
    if not (Screen.ActiveControl is TForm) then
    begin
      if Screen.ActiveControl.Owner is TForm then
      begin
        FI := GetFormItemOfForm(TForm(Screen.ActiveControl.Owner));
        if FI <> nil then
          FI.FocusedControl := Screen.ActiveControl;
      end;
    end;
  end;
end;

function TCustomFormsBar.GetFormCount: Integer;
begin
  Result := FormList.Count;
end;

function TCustomFormsBar.GetDropDownMenu: TPopupmenu;
begin
  Result := FDropDownMenu;
end;

function TCustomFormsBar.GetFocusTimerEnabled: Boolean;
begin
  Result := FFocusTimer.Enabled;
end;

function TCustomFormsBar.GetHomeAction: TBasicAction;
begin
  if HomeActionLink <> nil then
    Result := HomeActionLink.Action
  else
    Result := nil;
end;

function TCustomFormsBar.GetItems(Index: Integer): TCustomForm;
begin
  Result := nil;
  if (Index >= 0) and (Index < FormList.Count) then
  begin
    Result := TFormItem(FormList.Items[Index]).Form;
  end;
end;

function TCustomFormsBar.GetLastActiveForm: TCustomForm;
begin
  Result := nil;
  if OrderList.Count > 0 then Result := TCustomForm(OrderList.Items[0]);
end;

procedure TCustomFormsBar.SetDropDownMenu(AValue: TPopupmenu);
begin
  FDropDownMenu := AValue;
  if FDropDownMenu <> nil then
    FDropDownMenu.FreeNotification(Self);
end;

procedure TCustomFormsBar.SetFocusTimerEnabled(AValue: Boolean);
begin
  FFocusTimer.Enabled := AValue;
end;

procedure TCustomFormsBar.SetFormsBarBorders(AValue: TFormBarBorders);
begin
  if FFormsBarBorders=AValue then Exit;
  FFormsBarBorders:=AValue;
  Invalidate;
end;

procedure TCustomFormsBar.SetFormsParent(AValue: TWinControl);
begin
  if FFormsParent=AValue then Exit;
  FFormsParent:=AValue;
  FFormsParent.FreeNotification(Self);
end;

procedure TCustomFormsBar.SetHighLightFont(AValue: TFont);
begin
  if FHightLightFont=AValue then Exit;
  FHightLightFont:=AValue;
  Invalidate;
end;

procedure TCustomFormsBar.SetFormAnderMouse(AValue: TCustomForm);
var
  OldForm: TCustomForm;
  R: TRect;
begin
  if FFormAnderMouse <> AValue then
  begin
    OldForm := FFormAnderMouse;
    FFormAnderMouse := AValue;
    if OldForm <> nil then
    begin
      R := GetButtonRect(OldForm);
      InvalidateRect(Handle, @R, False);
    end;
    if FFormAnderMouse <> nil then
    begin
      R := GetButtonRect(FFormAnderMouse);
      InvalidateRect(Handle, @R, False);
    end;
    //Invalidate;
  end;
end;

procedure TCustomFormsBar.SetHomeAction(AValue: TBasicAction);
begin
  if AValue = nil then
  begin
    HomeActionLink.Free;
    HomeActionLink := nil;
  end
  else
  begin
    if HomeActionLink = nil then
      HomeActionLink := GetActionLinkClass.Create(Self);
    HomeActionLink.Action := AValue;
    HomeActionLink.OnChange:=@DoHomeActionChange;
    HomeActionChange(AValue);
    AValue.FreeNotification(Self);
  end;
end;

procedure TCustomFormsBar.SetHomeButton(AValue: Boolean);
begin
  if FHomeButton <> AValue then
  begin
    FHomeButton:= AValue;
    Invalidate;
  end;
end;

procedure TCustomFormsBar.SetHomeButtonState(AValue: TButtonState);
var
  R: TRect;
begin
  if FHomeButtonState <> AValue then
  begin
    FHomeButtonState := AValue;
    R := GetButtonRect(FormCount);
    InvalidateRect(Handle, @R, False);
  end;
end;

procedure TCustomFormsBar.SetHomeIconIndex(AValue: Integer);
begin
  if FHomeIconIndex <> AValue then
  begin
    FHomeIconIndex:=AValue;
    Invalidate;
  end;
end;

procedure TCustomFormsBar.SetImageList(AValue: TImageList);
begin
  if FImageList=AValue then Exit;
  FImageList:=AValue;
  Invalidate;
end;

procedure TCustomFormsBar.SetLastActiveForm(AValue: TCustomForm);
var
  Find: Boolean;
  I: Integer;
  FI: TFormItem;
begin
  if AValue = nil then Exit;

  if Screen.ActiveControl <> nil then
  begin
    if Screen.ActiveControl.Owner = AValue then
    begin
      FI := GetFormItemOfForm(AValue);
      if FI <> nil then
        FI.FocusedControl := Screen.ActiveControl;
    end;
  end;

  Find := False;
  for I := 0 to OrderList.Count - 1 do
  begin
    if TCustomForm(OrderList.Items[I]) = AValue then
    begin
      Find := True;
      OrderList.Move(I, 0);
      Break;
    end;
  end;

  if not Find then OrderList.Insert(0, AValue);
end;

procedure TCustomFormsBar.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  if (Operation = opRemove) then
  begin
    if AComponent is TCustomForm then RemoveForm(TCustomForm(AComponent))
    else
    if AComponent = FFormsParent then
      FFormsParent := nil
    else
    if AComponent is TBasicAction then
      if AComponent = HomeAction then HomeAction := nil
    else
    if AComponent = DropDownMenu then
      DropDownMenu := nil            ;
  end;
  inherited Notification(AComponent, Operation);
end;

procedure TCustomFormsBar.Paint;

type
  TEdgeStyle = (esNone, esRaised, esLowered);

const
  InnerStyles: array[TEdgeStyle] of Integer = (0, BDR_RAISEDINNER, BDR_SUNKENINNER);
  OuterStyles: array[TEdgeStyle] of Integer = (0, BDR_RAISEDOUTER, BDR_SUNKENOUTER);

var
  AR, CR, TextRect: TRect;
  I, X, Y, TxtOffset: Integer;
  S: string;
  details: TThemedElementDetails;
  txtStyle: TTextStyle;
  AnderMouse: Boolean;
  ImgIndex: Integer;
  DrawForm: TCustomForm;
  FEdgeBorderType : Cardinal;
  Icon: TIcon;
begin
  AR := ClientRect;
  Canvas.Brush.Color := Color;
  if FNative then
  begin
    Details := ThemeServices.GetElementDetails(thHeaderItemNormal);
    ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
  end
  else
    Canvas.FillRect(AR);

  FEdgeBorderType := 0;
  if (fbbTop in FFormsBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_TOP);
  if (fbbBottom in FFormsBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_BOTTOM);
  if (fbbLeft in FFormsBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_LEFT);
  if (fbbRight in FFormsBarBorders)  then
  FEdgeBorderType := FEdgeBorderType or longint(BF_RIGHT);
  DrawEdge(Canvas.Handle, AR,
           InnerStyles[esRaised] or OuterStyles[esLowered], FEdgeBorderType);

  AnderMouse := False;
  for I := 0 to FormCount - 1 do
  begin
    DrawForm := Items[I];
    AR := GetButtonRect(I);

    if I > 0 then
    begin
      Canvas.Pen.Color := MildeLineColor;
      X := AR.Left - SpaceWidth div 2 - 1;
      Canvas.MoveTo(X, AR.Top + 2);
      Canvas.LineTo(X, AR.Bottom - 2);
    end;

    AnderMouse:= False;
    if DrawForm = FormAnderMouse then AnderMouse:= True;

    if DrawForm.Visible then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonHot);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
      Canvas.Font.Assign(HighLightFont);
    end
    else
    begin
      //details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      if AnderMouse then
      begin
        Canvas.Font.Assign(Font);
        details := ThemeServices.GetElementDetails(tbPushButtonNormal);
        ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
      end
      else
      begin
        //Canvas.Brush.Color := Color;
        //Canvas.FillRect(AR);
        Canvas.Font.Assign(Font);
      end;
    end;

    TxtOffset := 2;
    ImgIndex := -1;
    Icon := nil;
    if (ImageList <> nil) then
    begin
      ImgIndex := GetFormItem(I).ImageIndex;

      if ImgIndex >= 0 then
      begin
        TxtOffset := 4 + ImageList.Width + TxtOffset;
        X := AR.Left + 4;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, ImgIndex, DrawForm.Enabled);
      end;
    end;
    if (ImgIndex = -1) then
    begin
      if not Application.MainForm.Icon.Empty then
        Icon := Application.MainForm.Icon
      else
        Icon := Application.Icon;
      TxtOffset := 4 + Icon.Width + TxtOffset;
      X := AR.Left + 4;
      Y := AR.Top + (AR.Bottom - AR.Top - Icon.Height) div 2;
      Canvas.Draw(X, Y, Icon);
    end;
    S := DrawForm.Caption;
    TextRect := AR;
    InflateRect(TextRect, -TxtOffset, -1);
    if TxtOffset <= 2 then TextRect.Right := TextRect.Right - 10;
    txtStyle.SingleLine := not WrapText;
    txtStyle.RightToLeft := False;
    txtStyle.Wordbreak := True;
    txtStyle.Alignment := taCenter;
    txtStyle.Layout := tlCenter;
    txtStyle.EndEllipsis := False;
    txtStyle.SystemFont := False;
    txtStyle.Opaque := False;

    Canvas.TextRect(TextRect, TextRect.Left, TextRect.Top, S, txtStyle);

    if AnderMouse and CloseBtn then
    begin

        AR := GetCloseBtnRect(AR);
        X := AR.Left - 1;
        Y := AR.Top;
        if FMiniBtnState in [bsHot, bsDown] then
        begin
          CR := AR;
          Dec(CR.Left, 1);
          Dec(CR.Right, 1);
          InflateRect(CR, 1, 1);
          details := ThemeServices.GetElementDetails(tbPushButtonHot);
          if FMiniBtnState = bsDown then
            details := ThemeServices.GetElementDetails(tbPushButtonPressed);
          ThemeSErvices.DrawElement(Canvas.Handle, Details, CR, nil);
        end;
        Canvas.Draw(X, Y, CloseBmp);

    end;

  end;

  if HomeButton then
  begin
    AR := GetButtonRect(FormCount);
    if HomeButtonState = bsHot then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end
    else
    if HomeButtonState = bsDown then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonPressed);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end;
    if FormCount > 0 then
    begin
      Canvas.Pen.Color := MildeLineColor;
      X := AR.Left - SpaceWidth div 2 - 1;
      Canvas.MoveTo(X, AR.Top + 2);
      Canvas.LineTo(X, AR.Bottom - 2);
    end;
    if (ImageList <> nil) then
    begin
      if (HomeIconIndex >= 0) and (HomeIconIndex < ImageList.Count) then
      begin
        X := AR.Left + (AR.Right - AR.Left - ImageList.Width) div 2;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, HomeIconIndex);
      end;
    end;
  end;

  if SysButton then
  begin
    AR := GetSysButtonRect;

    if SysButtonState = bsHot then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonNormal);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end
    else
    if SysButtonState = bsDown then
    begin
      details := ThemeServices.GetElementDetails(tbPushButtonPressed);
      ThemeSErvices.DrawElement(Canvas.Handle, Details, AR, nil);
    end;

    if (ImageList <> nil) then
    begin
      if (SysIconIndex >= 0) and (SysIconIndex < ImageList.Count) then
      begin
        X := AR.Left + (AR.Right - AR.Left - ImageList.Width) div 2;
        Y := AR.Top + (AR.Bottom - AR.Top - ImageList.Height) div 2;
        ImageList.Draw(Canvas, X, Y, SysIconIndex);
      end;
    end;
  end;

end;

procedure TCustomFormsBar.UpdateFormAnderMouse(P: TPoint);
var
  I: Integer;
  AR, CR: TRect;
  AForm: TCustomForm;
  B: Boolean;
begin
  AForm := nil;
  B := False;
  for I := 0 to FormCount - 1 do
  begin
    AR := GetButtonRect(I);
    if PtInRect(AR, P) then
    begin
      if Items[I].Enabled then
      begin
        AForm := Items[I];
        CR := GetCloseBtnRect(AR);
        if PtInRect(CR, P) then
          B := True;
      end;
      Break;
    end;
  end;
  FormAnderMouse := AForm;
  if B then MiniBtnState := bsHot
  else MiniBtnState:= bsUp;
end;

function TCustomFormsBar.AddInList(AForm: TCustomForm; IconIndex: Integer
  ): TFormItem;
begin
  Result := TFormItem.Create;
  Result.Form := AForm;
  Result.ImageIndex := IconIndex;

  FormList.Add(Result);
  AForm.FreeNotification(Self);
  ActivateForm(AForm);
end;

function TCustomFormsBar.InsertInList(AForm: TCustomForm; Index: Integer;
  IconIndex: Integer): TFormItem;
begin
  Result := TFormItem.Create;
  Result.Form := AForm;
  Result.ImageIndex := IconIndex;

  FormList.Insert(Index, Result);
  AForm.FreeNotification(Self);
  ActivateForm(AForm);
end;

procedure TCustomFormsBar.MouseLeave;
begin
  inherited MouseLeave;
  FormAnderMouse := nil;
  HomeButtonState := bsUp;
  SysButtonState := bsUp;
end;

procedure TCustomFormsBar.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  P: TPoint;
  AR: TRect;
  CheckHome: Boolean;
  I: Integer;
begin
  if Button = mbLeft then
  begin
    P.x := X;
    P.y := Y;
    CheckHome := True;
    for I := 0 to FormCount - 1 do
    begin
      AR := GetButtonRect(I);
      if PtInRect(AR, P) then
      begin
        if Items[I].Enabled then
        begin
          InflateRect(AR, -2, -1);
          if CloseBtn then
          begin
            AR := GetCloseBtnRect(AR);
            if not PtInRect(AR, P) then
              ActivateForm(Items[I])
            else
              MiniBtnState:= bsDown;
          end
          else ActivateForm(Items[I]);
        end;
        CheckHome := False;
        Break;
      end;
    end;
    if CheckHome and HomeButton then
    begin
      AR := GetButtonRect(FormCount);
      if PtInRect(AR, P) then
      begin
        HomeButtonState := bsDown;
        CheckHome:= False;
      end;
    end;

    if CheckHome and SysButton then
    begin
      AR := GetSysButtonRect;
      if PtInRect(AR, P) then
      begin
        SysButtonState := bsDown;
      end;
    end;
  end;
end;

procedure TCustomFormsBar.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
  AR: TRect;
  B: Boolean;
begin
  P.x := X;
  P.y := Y;
  UpdateFormAnderMouse(P);

  B := True;
  if HomeButton then
  begin
    AR := GetButtonRect(FormCount);
    if PtInRect(AR, P) then
    begin
      B := False;
      if HomeButtonState <> bsDown then
        HomeButtonState := bsHot
    end
    else
    begin
      if HomeButtonState <> bsDown then
        HomeButtonState:= bsUp;
    end;
  end;

  if SysButton and B then
  begin
    AR := GetSysButtonRect;
    if PtInRect(AR, P) then
    begin
      if SysButtonState <> bsDown then
        SysButtonState := bsHot
    end
    else
    begin
      if SysButtonState <> bsDown then
        SysButtonState := bsUp;
    end;
  end;
end;

procedure TCustomFormsBar.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
var
  AR: TRect;
  P: TPoint;
  I: Integer;
begin
  //inherited MouseUp(Button, Shift, X, Y);
  if Button = mbLeft then
  begin
    MiniBtnState := bsUp;
    P.x := X;
    P.y := Y;
    for I := 0 to FormCount - 1 do
    begin
      AR := GetButtonRect(I);
      if PtInRect(AR, P) then
      begin
        if Items[I].Enabled then
        begin
          InflateRect(AR, -2, -1);
          if CloseBtn then
          begin
            AR := GetCloseBtnRect(AR);
            if PtInRect(AR, P) then
            begin
              DoCloseBtnDown(Items[I]);
              Application.ProcessMessages;
              UpdateFormAnderMouse(P);
            end;
          end;
        end;
        Break;
      end;
    end;

    if HomeButton and (HomeButtonState = bsDown) then
    begin
      AR := GetButtonRect(FormCount);
      if PtInRect(AR, P) then
      begin
        HomeButtonState := bsHot;
        DoHomeButtonClick;
        UpdateFormAnderMouse(P);
      end
      else
        HomeButtonState := bsUp;
    end;

    if SysButton and (SysButtonState = bsDown) then
    begin
      AR := GetSysButtonRect;
      if PtInRect(AR, P) then
      begin
        SysButtonState := bsHot;
        DoSysButtonClick;
      end
      else
        SysButtonState := bsUp;
    end;
  end;
end;

procedure TCustomFormsBar.DoCloseBtnDown(AForm: TCustomForm);
begin
  if Assigned(FOnCloseBtnDown) then FOnCloseBtnDown(AForm);
end;

function TCustomFormsBar.GetCloseBtnRect(ParentRect: TRect): TRect;
begin
  Result.Left:= ParentRect.Right - CloseBtnWidth - 4;
  Result.Right:= Result.Left + CloseBtnWidth;
  Result.Top:= ParentRect.Top + (ParentRect.Bottom - ParentRect.Top - CloseBtnWidth) div 2;
  Result.Bottom:= Result.Top + CloseBtnWidth;
end;

function TCustomFormsBar.GetFormItem(Index: Integer): TFormItem;
begin
  Result := nil;
  if (Index >= 0) and (Index < FormList.Count) then
  begin
    Result := TFormItem(FormList.Items[Index]);
  end;
end;

function TCustomFormsBar.GetFormItemOfForm(AForm: TCustomForm): TFormItem;
var
  I: Integer;
  FI: TFormItem;
begin
  Result := nil;
  for I := 0 to FormCount - 1 do
  begin
    FI := GetFormItem(I);
    if FI.Form = AForm then
    begin
      Result := FI;
      Break;
    end;
  end;
end;

function TCustomFormsBar.GetButtonRect(Index: Integer): TRect;
var
  R, AR: TRect;
  W, ClW, I, ABottom, ATop: Integer;
begin
  R := GetButtonsRect;
  ABottom := R.Bottom;
  ATop := R.Top;

  W := ItemDefaultWidth;
  ClW:= R.Right - R.Left - Height;

  if FormCount > 0 then
    W := Min(W, ClW div FormCount);

  R.Right := R.Left + W;
  R.Top := ATop + 1;
  R.Bottom := ABottom - 1;
  for I := 0 to FormCount - 1 do
  begin
    AR := R;
    AR.Right := AR.Right - SpaceWidth;
    if I = Index then
    begin
      Result := AR;
      Break;
    end;
    R.Left:= R.Right;
    R.Right := R.Left + W;
  end;

  if Index = FormCount then
  begin
    //HomeButton
    AR := R;
    AR.Right := AR.Left + (AR.Bottom - AR.Top);
  end;
  Result := AR;
end;

function TCustomFormsBar.GetButtonRect(AForm: TCustomForm): TRect;
var
  R, AR: TRect;
  W, ClW, I, ABottom, ATop: Integer;
  FI: TFormItem;
begin
  R := GetButtonsRect;
  ABottom := R.Bottom;
  ATop := R.Top;

  W := ItemDefaultWidth;
  ClW:= R.Right - R.Left - Height;

  if FormCount > 0 then
    W := Min(W, ClW div FormCount);

  R.Right := R.Left + W;
  R.Top := ATop + 1;
  R.Bottom := ABottom - 1;
  for I := 0 to FormCount - 1 do
  begin
    AR := R;
    AR.Right := AR.Right - SpaceWidth;
    FI := GetFormItem(I);
    if FI.Form = AForm then
    begin
      Result := AR;
      Break;
    end;
    R.Left:= R.Right;
    R.Right := R.Left + W;
  end;

{  if Index = FormCount then
  begin
    //HomeButton
    AR := R;
    AR.Right := AR.Left + (AR.Bottom - AR.Top);
  end;}
  Result := AR;
end;

function TCustomFormsBar.GetButtonsRect: TRect;
begin
  Result := ClientRect;
  Dec(Result.Bottom);
  Dec(Result.Right);
  Inc(Result.Left);
  if fbbTop in FFormsBarBorders then Inc(Result.Top);
  if fbbBottom in FFormsBarBorders then Dec(Result.Bottom);
  if CloseBtn then Dec(Result.Right, Height);
end;

function TCustomFormsBar.GetSysButtonRect: TRect;
begin
  Result := ClientRect;
  Dec(Result.Bottom);
  Dec(Result.Right);
  Inc(Result.Left);
  if fbbTop in FFormsBarBorders then Inc(Result.Top);
  if fbbBottom in FFormsBarBorders then Dec(Result.Bottom);

  Result.Left := Result.Right - (Result.Bottom - Result.Top);
end;

procedure TCustomFormsBar.DoDropDownMenu;
var
  R: TRect;
  P: TPoint;
begin
  R := GetButtonRect(FormCount);
  P.x := R.Left;
  P.y:= R.Bottom;
  P := ClientToScreen(P);
  if P.y > Screen.Height - 10 then
  begin
    P.y:= R.Top;
    P := ClientToScreen(P);
  end;
  DropDownMenu.PopUp(P.x, P.y);
end;

procedure TCustomFormsBar.DoHomeButtonClick;
begin
  if csDesigning in ComponentState then Exit;

  if (HomeActionLink <> nil) and (HomeActionLink.Action <> nil) then
    HomeActionLink.Execute(Self)
  else
  if DropDownMenu <> nil then
    DoDropDownMenu
  else
  if Assigned(FOnHomeButtonClick) then FOnHomeButtonClick(Self);
end;

procedure TCustomFormsBar.DoSysButtonClick;
begin
  if csDesigning in ComponentState then Exit;

  if Assigned(FOnSysBtnClick) then FOnSysBtnClick(ActiveForm);
end;

procedure TCustomFormsBar.HomeActionChange(Sender: TObject);
begin
  if Sender <> nil then
  begin
    if Sender is TAction then
    begin
      HomeIconIndex:= TAction(Sender).ImageIndex;
    end;
  end;
end;

procedure TCustomFormsBar.Resize;
begin
  inherited Resize;
  Invalidate;
end;

constructor TCustomFormsBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ATabList := TFPList.Create;
  FMiniBtnState:= bsUp;
  Width:= 400;
  Height := 20;
  FormList := TList.Create;
  OrderList := TList.Create;
  FHightLightFont := TFont.Create;
  FHightLightFont.Assign(Font);
  FormAnderMouse := nil;
  FActiveForm := nil;
  FCloseBtn := True;
  FSysButton := True;
  CloseBtnWidth := 16;
  SpaceWidth := 5;
  FItemDefaultWidth := 200;
  FMidleLineColor := clGray;
  FWrapText := False;
  FHomeButton := True;
  FHomeButtonState:= bsUp;
  FSysButtonState := bsUp;
  FHomeIconIndex := -1;
  FSysIconIndex := -1;
  TabStop := False;
  FFormsBarBorders := [];
  FNative:= False;
  FFormsParent := nil;
  FFocusTimer := TTimer.Create(Self);
  if not (csDesigning in ComponentState) then
  begin
    FFocusTimer.OnTimer := @OnFocusTimer;
    FFocusTimer.Interval := 500;
  end;
  FFocusTimer.Enabled := False;
  CloseBmp := CreateBitmapFromLazarusResource('fbcloseico');
end;

destructor TCustomFormsBar.Destroy;
begin
  FFocusTimer.Enabled:= False;
  Clear;
  FFocusTimer.Free;
  FormList.Free;
  OrderList.Free;
  FreeThenNil(FHomeActionLink);
  FHightLightFont.Free;
  CloseBmp.Free;
  ATabList.Free;
  inherited Destroy;
end;

procedure TCustomFormsBar.Clear;
var
  I: Integer;
begin
  for I := 0 to FormList.Count - 1 do GetFormItem(I).Free;
  FormList.Clear;
  OrderList.Clear;
  Invalidate;
end;

procedure TCustomFormsBar.CloseAll;
var
  I, N, SI: Integer;
begin
  if FormCount = 0 then Exit;
  N := 3;
  while (N > 0) and (FormCount > 0) do
  begin
    I := FormCount - 1;
    while I >= 0 do
    begin
      if Items[I].Enabled then
      begin
        SI := FormCount;
        Items[I].Close;
        Application.ProcessMessages;
        if SI = FormCount then Exit;
      end;
      Dec(I);
    end;
    Dec(N);
  end;

  {while OrderList.Count > 0 do
  begin
    I := OrderList.Count;

    if not TCustomForm(OrderList.Items[N]).Enabled then
    begin
      if I > 1 then  N := 1;
    end;

    TCustomForm(OrderList.Items[N]).Close;
    if N = 1 then N := 0;
    Application.ProcessMessages;
    if OrderList.Count = I then  Break;
  end;
  if OrderList.Count = 0 then
  begin
    if FormCount = 1 then
    begin
      Items[0].Close;
    end;
  end;}
end;

procedure TCustomFormsBar.AddForm(AForm: TCustomForm; IconIndex: Integer);
var
  I: Integer;
  B: Boolean;
begin
  I := 0;
  B := True;
  while I < FormList.Count do
  begin
    if Items[I] = AForm then
    begin
      B := False;
      Break;
    end;
    Inc(I);
  end;
  if B then
  begin
    AddInList(AForm, IconIndex);
  end;
end;

procedure TCustomFormsBar.InsertForm(AForm, AfterForm: TCustomForm;
  IconIndex: Integer);
var
  I, FindIndex: Integer;
begin
  I := 0;
  FindIndex := -1;
  while I < FormCount do
  begin
    if Items[I] = AfterForm then
    begin
      FindIndex := I + 1;
      Break;
    end;
    Inc(I);
  end;
  if FindIndex >= 0 then InsertInList(AForm, FindIndex, IconIndex)
  else AddForm(AForm, IconIndex);
end;

procedure TCustomFormsBar.InsertFormWithParent(AForm, ParentForm: TCustomForm);
var
  I, FindIndex, IconIndex: Integer;
  FormItem: TFormItem;
begin
  I := 0;
  FindIndex := -1;
  IconIndex := -1;
  while I < FormCount do
  begin
    FormItem := GetFormItem(I);
    if FormItem.Form = ParentForm then
    begin
      FindIndex := I + 1;
      IconIndex:= FormItem.ImageIndex;
      Break;
    end;
    Inc(I);
  end;

  if FindIndex >= 0 then
    InsertInList(AForm, FindIndex, IconIndex)
  else AddForm(AForm, IconIndex);

  ParentForm.Enabled := False;

end;

procedure TCustomFormsBar.ReplaceForm(NewForm, OldForm: TCustomForm;
  IconIndex: Integer);
var
  I, AIndex: Integer;
  FI: TFormItem;
begin
  I := 0;
  AIndex := -1;
  while I < FormCount do
  begin
    if Items[I] = OldForm then
    begin
      AIndex:= I;
      Break;
    end;
    Inc(I);
  end;

  if AIndex >= 0 then
  begin
    FI := GetFormItem(AIndex);
    FI.Form := NewForm;
    FI.ImageIndex := IconIndex;
    FI.FocusedControl := nil;
    NewForm.FreeNotification(Self);
    OldForm.RemoveFreeNotification(Self);
    if LastActiveForm = OldForm then LastActiveForm := nil;
    ActivateForm(NewForm);
  end
  else
    AddForm(NewForm, IconIndex);
end;

procedure TCustomFormsBar.SetActiveControl(AForm: TCustomForm;
  AWinControl: TWinControl);
var
  FormItem: TFormItem;
begin
  FormItem := GetFormItemOfForm(AForm);
  if FormItem <> nil then
    FormItem.FocusedControl := AWinControl;
end;

procedure TCustomFormsBar.ActivateForm(AForm: TCustomForm);
var
  I, N: Integer;
  F: TCustomForm;
  FI: TFormItem;
  AControl: TWinControl;
  R: TRect;
begin
  if (ActiveForm = AForm) or not AForm.Enabled then Exit;

  LastActiveForm := ActiveForm;
  I := 0;
  while I < FormCount do
  begin
    FI := GetFormItem(I);
    F := FI.Form;

    if F <> AForm then
    begin
      if F.Visible then
      begin
        F.Hide;
        Break;
      end;
    end;
    Inc(I);
  end;

  FActiveForm := AForm;
  if (FActiveForm <> nil) and (FActiveForm.Parent = nil) then
  begin
    FActiveForm.BorderStyle:= bsNone;
    FActiveForm.Align:= alClient;
    if FFormsParent = nil then
      AForm.Parent := Self.Parent
    else
      AForm.Parent := FFormsParent;
  end;



  if (FActiveForm <> nil) and not (csDestroying in ComponentState) then
  begin
    FActiveForm.Visible := True;
    FActiveForm.BringToFront;
    Application.ProcessMessages;

    FI := GetFormItemOfForm(FActiveForm);
    if (FI <> nil) and (FI.FocusedControl <> nil) and (FI.FocusedControl.CanFocus) then
    begin
      FI.FocusedControl.SetFocus;
    end
    else
    begin
      ATabList.Clear;
      FActiveForm.GetTabOrderList(ATabList);
      if ATabList.Count > 0 then
      begin
        for I := 0 to ATabList.Count - 1 do
        begin
          AControl := TWinControl(ATabList.Items[I]);
          if (AControl.TabStop) and (AControl.Owner <> nil) and
            not (csAcceptsControls in AControl.ControlStyle)
            and not (csNoFocus in AControl.ControlStyle)
            and AControl.CanFocus then
          begin
            AControl.SetFocus;
            Break;
          end;
        end;
      end;
    end;
  end;


  if FActiveForm <> nil then
  begin
    if LastActiveForm <> nil then
    begin
      R := GetButtonRect(LastActiveForm);
      InvalidateRect(Handle, @R, False);
    end;

    R := GetButtonRect(FActiveForm);
    InvalidateRect(Handle, @R, False);
  end
  else
    Invalidate;
end;

procedure TCustomFormsBar.RemoveForm(AForm: TCustomForm);
var
  I: Integer;
  FI, ParentItem: TFormItem;
  LAForm, IForm: TCustomForm;
  R: TRect;
begin
  for I := 0 to OrderList.Count - 1 do
  begin
    if TCustomForm(OrderList.Items[I]) = AForm then
    begin
      OrderList.Delete(I);
      Break;
    end;
  end;


  for I := 0 to FormCount - 1 do
  begin
      FI := GetFormItem(I);
      IForm := FI.Form;
      if IForm = AForm then
      begin
        if FormAnderMouse = IForm then FormAnderMouse := nil;

        ParentItem := nil;
        if AForm.Owner is TCustomForm then
          ParentItem := GetFormItemOfForm(TCustomForm(AForm.Owner))
        else
        begin
          if AForm.Owner <> nil then
            if AForm.Owner.Owner <> nil then
            begin
              if AForm.Owner.Owner is TCustomForm then
                ParentItem := GetFormItemOfForm(TCustomForm(AForm.Owner.Owner))
            end;
        end;
        if FActiveForm = Items[I] then
        begin
          FActiveForm := nil;
          if ParentItem <> nil then
          begin
            LAForm := ParentItem.Form;
            if LAForm.Enabled = False then LAForm.Enabled := True;
          end
          else
            LAForm := LastActiveForm;
          if LAForm <> nil then ActivateForm(LAForm);
        end
        else
          if ParentItem <> nil then
          begin
            if not ParentItem.Form.Enabled then
              ParentItem.Form.Enabled := True;
          end;
        FI.Free;
        FormList.Delete(I);
        R := ClientRect;
        InvalidateRect(Handle, @R, False);
        Break;
      end;
  end;

end;

initialization

{$I xformbaricon.lrs}

end.
