unit xdbf_reg;

{===============================================================================
||         TxDbf Component         ||         http://TxDbf.sf.net               ||
===============================================================================}
(*
  TxDbf is supplied "AS IS". The author disclaims all warranties,
  expressed or implied, including, without limitation, the warranties of
  merchantability and or fitness for any purpose. The author assumes no
  liability for damages, direct or consequential, which may result from the
  use of TxDbf.

  TxDbf is licensed under the LGPL (lesser general public license).

  You are allowed to use this component in any project free of charge.
  You are
  - NOT allowed to claim that you have created this component.  You are
  - NOT allowed to copy this component's code into your own component and
      claim that the code is your idea.

*)

interface

uses
  PropEdits, SysUtils, Classes, Controls, Forms, Dialogs,
  xdbf_dbffile, xdbf_idxfile, xdbf_fields, xdbf_common, xdbf_str
{$ifdef DELPHI_6}
  ,DesignIntf,DesignEditors
{$endif}
  ,LazarusPackageIntf, LResources;


type
  TTableNameProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TIndexFileNameProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TSortFieldProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TIndexNameProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    procedure SetValue(const Value: string); override;
    function GetValue: string; override;
  end;

  TVersionProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TNativeFieldTypeProperty = class(TCharProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    procedure SetValue(const Value: string); override;
  end;


procedure Register;

implementation

uses
  xdbf;

//==========================================================
//============ DESIGNONLY ==================================
//==========================================================
(*
//==========================================================
//============ TFilePathProperty
//==========================================================
type
  TFilePathProperty = class(TStringProperty)
  public
    function GetValue: string; override;
  end;

function TFilePathProperty.GetValue: string;
begin
  Result := inherited GetValue;
  if Result = EmptyStr then
  begin
    SetValue(ExtractFilePath(ToolServices.GetProjectName));
    Result := inherited GetValue;
  end;
end;
*)

//==========================================================
//============ TTableNameProperty
//==========================================================

procedure TTableNameProperty.Edit; {override;}
var
  FileOpen: TOpenDialog;
  Dbf: TxDbf;
begin
  FileOpen := TOpenDialog.Create(Application);
  try
    with fileopen do begin
      Dbf := GetComponent(0) as TxDbf;
      FileOpen.InitialDir := Dbf.AbsolutePath;
      Filename := GetValue;
      Filter := 'Dbf table|*.dbf';
      if Execute then begin
        SetValue(Filename);
      end;
    end;
  finally
    Fileopen.free;
  end;
end;

function TTableNameProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  Result := [paDialog, paRevertable];
end;

//==========================================================
//============ TIndexFileNameProperty
//==========================================================

procedure TIndexFileNameProperty.Edit; {override;}
var
  FileOpen: TOpenDialog;
  IndexDef: TDbfIndexDef;
  Indexes: TDbfIndexDefs;
  Dbf: TxDbf;
begin
  FileOpen := TOpenDialog.Create(Application);
  try
    with fileopen do begin
      IndexDef := GetComponent(0) as TDbfIndexDef;
      Indexes := TDbfIndexDefs(IndexDef.Collection);
      Dbf := TxDbf(Indexes.FOwner);
      FileOpen.InitialDir := Dbf.AbsolutePath;
      Filename := GetValue;
      Filter := 'Simple index (ndx)|*.ndx'{|Multiple index (mdx)|*.mdx'};
      if Execute then begin
        SetValue(ExtractFileName(Filename));
      end;
    end;
  finally
    Fileopen.free;
  end;
end;

function TIndexFileNameProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  Result := [paDialog, paRevertable];
end;

//==========================================================
//============ TSortFieldProperty
//==========================================================


function TSortFieldProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  Result := [paValueList, paSortList, paRevertable];
end;

procedure TSortFieldProperty.GetValues(Proc: TGetStrProc);
var
  IndexDef: TDbfIndexDef;
  Indexes: TDbfIndexDefs;
  Dbf: TxDbf;
  I: integer;
begin
  IndexDef := GetComponent(0) as TDbfIndexDef;
  Indexes := TDbfIndexDefs(IndexDef.Collection);
  Dbf :=  TxDbf(Indexes.FOwner);
  for I := 0 to Dbf.FieldCount-1 do
  begin
    Proc(Dbf.Fields[i].FieldName);
  end;
end;

//==========================================================
//============ TIndexNameProperty
//==========================================================

function TIndexNameProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  Result := [paValueList, paRevertable];
end;

procedure TIndexNameProperty.GetValues(Proc: TGetStrProc);
var
  Dbf: TxDbf;
  I: Integer;
begin
  Dbf := GetComponent(0) as TxDbf;
  Dbf.UpdateIndexDefs;
  for I := 0 to Dbf.Indexes.Count - 1 do
    Proc(Dbf.Indexes[I].IndexFile);
end;

procedure TIndexNameProperty.SetValue(const Value: string); {override}
var
  Dbf: TxDbf;
begin
  Dbf := GetComponent(0) as TxDbf;
  Dbf.IndexName := Value;
end;

function TIndexNameProperty.GetValue: string; {override;}
var
  Dbf: TxDbf;
begin
  Dbf := GetComponent(0) as TxDbf;
  Result := Dbf.IndexName;
end;

//==========================================================
//============ TVersionProperty
//==========================================================

procedure TVersionProperty.Edit; {override;}
begin
  ShowMessage(
    Format(STRING_VERSION,[TDBF_MAJOR_VERSION, TDBF_MINOR_VERSION]) +
      ' : a dBase component'+#13+
      'for Delphi and c++ builder with no BDE.'+#13+
      #13 +
      'To get the latest version, please visit'+#13+
      'SourceForge: http://TDbf.sf.net' + #13 +
      'Russuan version for cp866 and cp1251: http://visual-t.ru'
      );
end;

function TVersionProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  Result := [paDialog, paReadOnly, paRevertable];
end;

//==========================================================
//============ TNativeFieldTypeProperty
//==========================================================

procedure TNativeFieldTypeProperty.SetValue(const Value: string);
var
  L: Longint;
begin
  if Length(Value) = 0 then L := 0 else
  if Value[1] = '#' then L := StrToInt(Copy(Value, 2, Maxint))
  else L := Ord(Value[1]);
  SetOrdValue(L);
end;

function TNativeFieldTypeProperty.GetAttributes: TPropertyAttributes; {override;}
begin
  result := [paRevertable,paValueList];
end;

procedure TNativeFieldTypeProperty.GetValues(Proc: TGetStrProc);
begin
  Proc('C Character');
  Proc('N Numeric');
  Proc('D Date');
  Proc('L Logical');
  Proc('M Memo');
  Proc('B Blob');
  Proc('F Float');
  Proc('O Double');
  Proc('I Integer');
  Proc('G Graphic');
  Proc('+ AutoIncrement');
  Proc('@ DateTime');
end;

//==========================================================
//============ initialization
//==========================================================
function IDE_DbfDefaultPath:string;
begin
  Result := GetCurrentDir
end;

procedure RegisterUnitDbf;
begin
  xDbf.DbfBasePath := @IDE_DbfDefaultPath;
  RegisterPropertyEditor(TypeInfo(string), TxDbf, 'TableName', TTableNameProperty);
  RegisterPropertyEditor(TypeInfo(string), TxDbf, 'Version', TVersionProperty);
  RegisterPropertyEditor(TypeInfo(string), TxDbf, 'IndexName', TIndexNameProperty);
  RegisterPropertyEditor(TypeInfo(string), TDbfIndexDef, 'IndexFile', TIndexFileNameProperty);
  RegisterPropertyEditor(TypeInfo(string), TDbfIndexDef, 'SortField', TSortFieldProperty);
  RegisterPropertyEditor(TypeInfo(char), TDbfFieldDef, 'NativeFieldType', TNativeFieldTypeProperty);
end;

procedure Register;
begin
  RegisterComponents('Data Access', [TxDbf]);
  RegisterUnit('xDbf', @RegisterUnitDbf);
end;

initialization

  {$i xDBFIco.lrs}

end.
