unit SQLParser;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

const
  msEOF      = AnsiChar(0);
  msSymbol   = AnsiChar(1);
  msString   = AnsiChar(2);
  msInteger  = AnsiChar(3);
  msHexInt   = AnsiChar(4);
  msFloat    = AnsiChar(5);
  msText     = AnsiChar(6);
  msChar     = AnsiChar(7);
  msComent   = AnsiChar(8);
  msUnknSym  = AnsiChar(9);

type
  TIdentSQL = (
    IDS_EOF,
    IDS_Symbol,
    IDS_NewLine,
    IDS_RetCaret,
    IDS_Comment,
    IDS_Identifier,
    IDS_SemiColon,
    IDS_Comma,
    IDS_Period,
    IDS_Colon,
    IDS_OpenRound,
    IDS_CloseRound,
    IDS_GreaterEqual,
    IDS_Greater,
    IDS_Less,
    IDS_LessEqual,
    IDS_OpenBlock,
    IDS_CloseBlock,
    IDS_Dots,
    IDS_NotEqual,
    IDS_Assignment,
    IDS_Plus,
    IDS_Minus,
    IDS_Multiply,
    IDS_Divide,
    IDS_String,
    IDS_QuestionMark,
    IDS_Unknown,

    IDS_ACTIVE,
    IDS_ADD,
    IDS_AFTER,
    IDS_ALL,
    IDS_ALTER,
    IDS_AND,
    IDS_ANY,
    IDS_AS,
    IDS_ASC,
    IDS_ASCENDING,
    IDS_AT,
    IDS_AUTO,
    IDS_AUTODDL,
    IDS_BASED,
    IDS_BASENAME,
    IDS_BASE_NAME,
    IDS_BEFORE,
    IDS_BEGIN,
    IDS_BETWEEN,
    IDS_BLOCK,
    IDS_BLOBEDIT,
    IDS_BUFFER,
    IDS_BY,
    IDS_CACHE,
    IDS_CHARACTER_LENGTH,
    IDS_CHAR_LENGTH,
    IDS_CHECK,
    IDS_CHECK_POINT_LEN,
    IDS_CHECK_POINT_LENGTH,
    IDS_COLLATE,
    IDS_COLLATION,
    IDS_COLUMN,
    IDS_COMMIT,
    IDS_COMMITED,
    IDS_COMPILETIME,
    IDS_COMPUTED,
    IDS_CLOSE,
    IDS_CONDITIONAL,
    IDS_CONNECT,
    IDS_CONSTRAINT,
    IDS_CONTAINING,
    IDS_CONTINUE,
    IDS_CREATE,
    IDS_CURRENT,
    IDS_CURRENT_DATE,
    IDS_CURRENT_TIME,
    IDS_CURRENT_TIMESTAMP,
    IDS_CURSOR,
    IDS_DATABASE,
    IDS_DAY,
    IDS_DB_KEY,
    IDS_DEBUG,
    IDS_DEC,
    IDS_DECLARE,
    IDS_DEFAULT,
    IDS_DELETE,
    IDS_DESC,
    IDS_DESCENDING,
    IDS_DESCRIBE,
    IDS_DESCRIPTOR,
    IDS_DISCONNECT,
    IDS_DISTINCT,
    IDS_DO,
    IDS_DOMAIN,
    IDS_DROP,
    IDS_ECHO,
    IDS_EDIT,
    IDS_ELSE,
    IDS_END,
    IDS_ENTRY_POINT,
    IDS_ESCAPE,
    IDS_EVENT,
    IDS_EXCEPTION,
    IDS_EXECUTE,
    IDS_EXISTS,
    IDS_EXIT,
    IDS_EXTERN,
    IDS_EXTERNAL,
    IDS_EXTRACT,
    IDS_FETCH,
    IDS_FILE,
    IDS_FILTER,
    IDS_FOR,
    IDS_FOREIGN,
    IDS_FOUND,
    IDS_FROM,
    IDS_FULL,
    IDS_FUNCTION,
    IDS_GDSCODE,
    IDS_GENERATOR,
    IDS_GLOBAL,
    IDS_GOTO, IDS_GRANT,
    IDS_GROUP,
    IDS_GROUP_COMMIT_WAIT,
    IDS_GROUP_COMMIT_WAIT_TIME,
    IDS_HAVING,
    IDS_HELP,
    IDS_HOUR,
    IDS_IF,
    IDS_IMMEDIATE,
    IDS_IN,
    IDS_INACTIVE,
    IDS_INDEX,
    IDS_INDICATOR,
    IDS_INIT,
    IDS_INNER,
    IDS_INPUT,
    IDS_INPUT_TYPE,
    IDS_INSERT,
    IDS_INT,
    IDS_INTO,
    IDS_IS,
    IDS_ISOLATION,
    IDS_ISQL,
    IDS_JOIN,
    IDS_KEY,
    IDS_LC_MESSAGES,
    IDS_LC_TYPE,
    IDS_LEFT,
    IDS_LENGTH,
    IDS_LEV,
    IDS_LEVEL,
    IDS_LIKE,
    IDS_LOGFILE,
    IDS_LOG_BUFFER_SIZE,
    IDS_LOG_BUF_SIZE,
    IDS_LONG,
    IDS_MANUAL,
    IDS_MAXIMUM,
    IDS_MAXIMUM_SEGMENT,
    IDS_MAX_SEGMENT,
    IDS_MERGE,
    IDS_MESSAGE,
    IDS_MINIMUM,
    IDS_MINUTE,
    IDS_MODULE_NAME,
    IDS_MONTH,
    IDS_NAMES,
    IDS_NATIONAL,
    IDS_NATURAL,
    IDS_NCHAR,
    IDS_NO,
    IDS_NOAUTO,
    IDS_NOT,
    IDS_NULL,
    IDS_NUM_LOG_BUFFS,
    IDS_NUM_LOG_BUFFERS,
    IDS_OCTET_LENGTH,
    IDS_OF,
    IDS_ON,
    IDS_ONLY,
    IDS_OPEN,
    IDS_OPTION,
    IDS_OR,
    IDS_ORDER,
    IDS_OUTER,
    IDS_OUTPUT,
    IDS_OUTPUT_TYPE,
    IDS_OVERFLOW,
    IDS_PAGE,
    IDS_PAGELENGTH,
    IDS_PAGES,
    IDS_PAGE_SIZE,
    IDS_PARAMETER,
    IDS_PASSWORD,
    IDS_PLAN,
    IDS_POSITION,
    IDS_POST_EVENT,
    IDS_PRECISION,
    IDS_PREPARE,
    IDS_PROCEDURE,
    IDS_PROTECTED,
    IDS_PRIMARY,
    IDS_PRIVILEGES,
    IDS_PUBLIC,
    IDS_QUIT,
    IDS_RAW_PARTITIONS,
    IDS_READ,
    IDS_REAL,
    IDS_RECORD_VERSION,
    IDS_REFERENCES,
    IDS_RELEASE,
    IDS_RESERV,
    IDS_RESERVING,
    IDS_RETAIN,
    IDS_RETURN,
    IDS_RETURNING,
    IDS_RETURNING_VALUES,
    IDS_RETURNS,
    IDS_REVOKE,
    IDS_RIGHT,
    IDS_ROLLBACK,
    IDS_RUNTIME,
    IDS_SCHEMA,
    IDS_SECOND,
    IDS_SEGMENT,
    IDS_SELECT,
    IDS_SET,
    IDS_SHADOW,
    IDS_SHARED,
    IDS_SHELL,
    IDS_SHOW,
    IDS_SINGULAR,
    IDS_SIZE,
    IDS_SNAPSHOT,
    IDS_SOME,
    IDS_SORT,
    IDS_SQL,
    IDS_SQLCODE,
    IDS_SQLERROR,
    IDS_SQLWARNING,
    IDS_STABILITY,
    IDS_STARTING,
    IDS_STARTS,
    IDS_STATEMENT,
    IDS_STATIC,
    IDS_STATISTICS,
    IDS_SUB_TYPE,
    IDS_SUSPEND,
    IDS_TABLE,
    IDS_TERMINATOR,
    IDS_THEN,
    IDS_TO,
    IDS_TRANSACTION,
    IDS_TRANSLATE,
    IDS_TRANSLATION,
    IDS_TRIGGER,
    IDS_TRIM,
    IDS_TYPE,
    IDS_UNCOMMITTED,
    IDS_UNION,
    IDS_UNIQUE,
    IDS_UPDATE,
    IDS_USER,
    IDS_USING,
    IDS_VALUE,
    IDS_VALUES,
    IDS_VARIABLE,
    IDS_VARYING,
    IDS_VERSION,
    IDS_VIEW,
    IDS_WAIT,
    IDS_WEEKDAY,
    IDS_WHEN,
    IDS_WHENEVER,
    IDS_WHERE,
    IDS_WHILE,
    IDS_WITH,
    IDS_WORK,
    IDS_WRITE,
    IDS_YEAR,
    IDS_YEARDAY,

    IDS_BIGINT,
    IDS_BLOB,
    IDS_CHAR,
    IDS_CHARACTER,
    IDS_DATE,
    IDS_DECIMAL,
    IDS_DOUBLE,
    IDS_FLOAT,
    IDS_INTEGER,
    IDS_NUMERIC,
    IDS_SMALLINT,
    IDS_TIME,
    IDS_TIMESTAMP,
    IDS_VARCHAR,

    IDS_AVG,
    IDS_CAST,
    IDS_COUNT,
    IDS_GEN_ID,
    IDS_LOWER,
    IDS_MAX,
    IDS_MIN,
    IDS_SUM,
    IDS_UPPER

  );

  PSQLToken = ^TSQLToken;
  TSQLToken = record
    TokenName: string;
    TokenID: TIdentSQL;
    Col: Integer;
    Row: Integer;
    EndCol: Integer;
    KeyConst: Integer;
    IsEnd: Boolean;
  end;

  { TSQLTokens }

  TSQLTokens = class
  private
    FList: TList;
    FPos: Integer;
    function Get(Index: Integer): PSQLToken;
    function GetToken: PSQLToken;
    function GetPreviewNext: PSQLToken;
  protected
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Count: Integer;
    procedure Next;
    procedure NextWithSkipComment;
    procedure PrevWithSkipComment;
    procedure Prev;
    procedure First;
    procedure Last;
    procedure Add(TokenName: string; TokenID: TIdentSQL;
      Col, Row, EndCol, KeyConst: Integer; IsEnd: Boolean);
    property Pos: Integer read FPos write FPos;
    property Items[Index: Integer]: PSQLToken read Get;
    property Token: PSQLToken read GetToken;
    property PreviewNext: PSQLToken read GetPreviewNext;
  end;

  { THashKeyWord }

  THashKeyWord = class(TObject)
  protected
    fKeyLen: integer;
    fKeyword: string;
    fHashValue: Cardinal;
    fIdnet: TIdentSQL;
  public
    constructor Create(const AKey: string; AIdent: TIdentSQL);
  public
    property Keyword: string read fKeyword;
    property KeywordLen: integer read fKeyLen;
    property HashValue: Cardinal read fHashValue;
    property Idnet: TIdentSQL read fIdnet;
  end;

  { THashKeywordList }

  THashKeywordList = class(TList)
  private
    function Get(Index: Integer): THashKeyWord;
  protected
  public
    procedure Clear; override;
  public
    constructor Create;
    procedure AppendKey(AKey: string; Ident: TIdentSQL);
    function GetKey(AKey: string): THashKeyWord;
    property Items[Index: Integer]: THashKeyWord read Get; default;
  end;

  { TSQLParser }

  TSQLParser = class
  private
    FString: Ansistring;
    FStartPos: PAnsiChar;
    FEndPos: PAnsiChar;
    FRow: Longint;
    FCol: Longint;
    FStartCol: PAnsiChar;
    FTokenPos: Integer;
    FTokenEnd: Integer;
    FCommentOpen: Boolean;
    BlockCount: Integer;
    FInClass: Boolean;
    FLanguage: Integer;
    FPropOpen: Boolean;
    FKeyWordList: THashKeywordList;
    function GetCol: Integer;
    function GetColEnd: Integer;
  protected

    function TokenString: string;
    function TokenText: string;
    function TokenFloat: Extended;
    function TokenInt: Integer;
  public
    //function PreviewToken: AnsiChar;
    UseTranscription: Boolean;
    constructor Create;
    destructor Destroy; override;
    function NextToken: AnsiChar; virtual;
    procedure SetParseStr(S: string);
    procedure GetTokensList(Tokens: TSQLTokens);
    property Row: LongInt read FRow;
    property Col: LongInt read GetCol;
    property ColEnd: LongInt read GetColEnd;
    property TokenPos: Integer read FTokenPos;
    property TokenEnd: Integer read FTokenEnd;
    property AsString: string read TokenString;
    property AsFloat: Extended read TokenFloat;
    property AsInteger: Integer read TokenInt;
    property AsText: string read TokenText;
    property KeyWordList: THashKeywordList read FKeyWordList;
  end;

  TSQLInstrucitonType = (sqliCreate, sqliAlter, sqliCreateAlter, sqliSelect,
    sqliUpdate, sqliDelete);

  { TSQLInstruction }

  TSQLInstruction = class
  private
    FList: TList;
    function Get(Index: Integer): TSQLInstruction;
  public
    iType: TSQLInstrucitonType;
    StartPos: TPoint;
    EndPos: TPoint;
    RelationList: TStringList;
    constructor Create;
    destructor Destroy; override;
    function Count: Integer;
    function Add(Instruction: TSQLInstruction): Integer;
    property Items[Index: Integer]: TSQLInstruction read Get; default;
  end;

  { TSQLInstructionList }

  TSQLInstructionList = class
  private
    FList: TList;

    function Get(Index: Integer): TSQLInstruction;
  protected
  public
    RelationList: TStringList;
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Add(AInstruction: TSQLInstruction): Integer;
    property Items[Index: Integer]: TSQLInstruction read Get;
  end;


implementation

uses CodeProcs, dpUtils;

type
  TSQLIdentChars = set of char;

var
  SQLIdent: TSQLIdentChars;

{ TSQLInstructionList }

function TSQLInstructionList.Get(Index: Integer): TSQLInstruction;
begin
  Result := TSQLInstruction(FList.Items[Index]);
end;

constructor TSQLInstructionList.Create;
begin
  inherited Create;
  FList := TList.Create;
  RelationList := TStringList.Create;
end;

destructor TSQLInstructionList.Destroy;
begin
  Clear;
  FList.Free;
  RelationList.Free;
  inherited Destroy;
end;

procedure TSQLInstructionList.Clear;
var
  I: Integer;
begin
  for I := 0 to FList.Count - 1 do
  begin
    TSQLInstruction(FList.Items[I]).Free;
    FList.Items[I] := nil;
  end;
  FList.Clear;
end;

function TSQLInstructionList.Add(AInstruction: TSQLInstruction): Integer;
begin
  Result := FList.Add(AInstruction);
end;



{ TSQLInstruction }

function TSQLInstruction.Get(Index: Integer): TSQLInstruction;
begin
  Result := TSQLInstruction(FList.Items[Index]);
end;

constructor TSQLInstruction.Create;
begin
  inherited Create;
  RelationList := TStringList.Create;
end;

destructor TSQLInstruction.Destroy;
var
  I: Integer;
begin
  for I := 0 to FList.Count - 1 do
  begin
    TSQLInstruction(FList.Items[I]).Free;
    FList.Items[I] := nil;
  end;
  FList.Clear;
  RelationList.Free;
  inherited Destroy;
end;

function TSQLInstruction.Count: Integer;
begin
  Result := FList.Count;
end;

function TSQLInstruction.Add(Instruction: TSQLInstruction): Integer;
begin
  Result := FList.Add(Instruction);
end;

{ TSQLParser }

function TSQLParser.GetCol: Integer;
begin
  Result := FStartPos - FStartCol;
end;

function TSQLParser.GetColEnd: Integer;
begin
  Result := FEndPos - FStartCol;
end;

function TSQLParser.TokenString: string;
var
  I: Integer;
begin
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(Result, FStartPos, I);
  end;
end;

function TSQLParser.TokenText: string;
var
  I: Integer;
begin
  if (FStartPos <> nil) or (FEndPos <> nil) then
  begin
    I := FEndPos - FStartPos;
    FTokenPos := FTokenEnd;
    FTokenEnd := FTokenPos + I;
    SetString(Result, FStartPos + 1, I - 2);
    I := Length(Result);
    if I > 0 then
    begin
      if (Result[I] = '''') or (Result[I] = '"') then Delete(Result, I, 1);
      if (Result[1] = '''') or (Result[1] = '"') then Delete(Result, 1, 1);
    end;
  end;
end;

function TSQLParser.TokenFloat: Extended;
begin
  Result := StrToFloat(AsString);
end;

function TSQLParser.TokenInt: Integer;
begin
  Result := StrToInt(AsString);
end;

constructor TSQLParser.Create;
begin
  inherited Create;
  FKeyWordList := THashKeywordList.Create;

end;

destructor TSQLParser.Destroy;
begin
  FKeyWordList.Free;
  inherited Destroy;
end;

function TSQLParser.NextToken: AnsiChar;
var
  P: PAnsiChar;
begin
  if FStartPos = nil then
  begin
    Result := msEOF;
    Exit;
  end;
  FStartPos := FEndPos;
  P := FStartPos;

  if P^ = #0 then
  begin
    Result := msEOF;
    Exit;
  end;

  while (P^ = ' ') or (P^ = #8) or (P^ = #13) or (P^ = #10) do
  begin
    case P^ of
      ' ', #8:
      begin
        while (P^ = ' ') or (P^ = #8) do
        begin
          Inc(P);
        end;
      end;

      #13:
      begin
        Inc(P);
      end;

      #10:
      begin
        Inc(P);
        FStartCol := P;
        Inc(FRow);
      end;
    end;

  end;
  FStartPos := P;

  if ((P^ = '/') and ((P + 1)^ = '*')) or FCommentOpen then
  begin
    FCommentOpen := True;
    while True do
    begin
      if (P^ = '*') and ((P + 1)^ = '/') then
      begin
        Inc(P, 2);
        FCommentOpen := False;
        Break;
      end
      else
      if CharInSet(P^, [#13, #10, #0]) then Break
      else Inc(P);
    end;
    Result := msComent;
    FEndPos := P;
    Exit;
  end;

  if (P^ = '-') and ((P + 1)^ = '-') then
  begin
    while True do
    begin
      Inc(P);
      if CharInSet(P^, [#13, #10, #0]) then Break;
    end;
    Result := msComent;
    FEndPos := P;
    Exit;
  end;

  case P^ of
    #0:
    begin
      Result := msEOF;
    end;

    'A'..'Z', 'a'..'z', '_', #128..#255:
    begin
      Inc(P);
      while CharInSet(P^, ['A'..'Z', 'a'..'z', '0'..'9', '_', #128..#255]) do Inc(P);
        Result := msString;
    end;

    '0'..'9':
    begin
      Result := msInteger;

      if (P^ = '0') and ((P + 1)^ = 'x') then
      begin
        Result := msHexInt;
        Inc(P);
        while CharInSet(P^, ['0'..'9', 'A'..'F', 'a'..'f']) do Inc(P);
      end
      else
      begin
        Inc(P);
        while CharInSet(P^, ['0'..'9', '.']) do
        begin
          if (P^ = '.') then
          begin
            if CharInSet((P + 1)^, ['0'..'9']) then
              Result := msFloat
            else Break;
          end;
          Inc(P);
        end;
      end;
    end;

    '''':
    begin
      Inc(P);
      while (P^ <> #0) and (P^ <> #10) and (P^ <> #13) do
      begin
        if P^ = '''' then
        begin
          if (P + 1)^ = '''' then
          begin
            Inc(P, 2);
            Continue;
          end
          else
          begin
            Inc(P);
            Break;
          end;
        end;
        Inc(P);
      end;
      Result := msText;
    end;

    '"':
    begin
      Inc(P);
      while (P^ <> #0) and (P^ <> #10) and (P^ <> #13) do
      begin
        if P^ = '"' then
        begin
          if (P + 1)^ = '"' then
          begin
            Inc(P, 2);
            Continue;
          end
          else
          begin
            Inc(P);
            Break;
          end;
        end;
        Inc(P);
      end;
      Result := msText;
    end;

    else
    begin
      if (P^ = '<') and ((P + 1)^ = '=')  then Inc(P)
      else
      if (P^ = '>') and ((P + 1)^ = '=')  then Inc(P)
      else
      if (P^ = '<') and ((P + 1)^ = '>') then Inc(P);
      Inc(P);
      Result := msSymbol;
    end;
  end;
  FEndPos := P;
end;

procedure TSQLParser.SetParseStr(S: string);
var
  I: Integer;
begin
  I := Length(S);
  FString := '';
  FRow := 0;
  FCol := 0;
  FTokenPos := 0;
  FTokenEnd := 0;

  if I > 0 then
  begin
    FString := dp_UTF8ToAnsi(S);
    I := Length(FString);
    Inc(I);
    SetLength(FString, I);
    FString[I] := #0;
    FStartPos := @FString[1];
    FEndPos := FStartPos;
    FStartCol := FStartPos;
  end
  else
  begin
    FStartPos := nil;
    FEndPos := nil;
    FStartCol := nil;
  end;
end;

procedure TSQLParser.GetTokensList(Tokens: TSQLTokens);
var
  C: AnsiChar;
  S: string;
  AnsStr: AnsiString;
  HKW: THashKeyWord;
begin
  Tokens.Clear;
  BlockCount := 0;
  FInClass := False;
  FRow := 0;
  if FString = '' then
  begin
    Tokens.Add('', IDS_EOF, 0, 0, 0, 0, False);
    Exit;
  end;
  C := NextToken;
  while C <> msEOF do
  begin
    case C of

      msSymbol:
      begin
        S := string(AsString);
        if S = '>=' then Tokens.Add(S, IDS_GreaterEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '>' then Tokens.Add(S, IDS_Greater, Col, Row, ColEnd, 2, False)
        else
        if S = '<' then Tokens.Add(S, IDS_Less, Col, Row, ColEnd, 2, False)
        else
        if S = '<=' then Tokens.Add(S, IDS_LessEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '<>' then Tokens.Add(S, IDS_NotEqual, Col, Row, ColEnd, 2, False)
        else
        if S = '(' then Tokens.Add(S, IDS_OpenRound, Col, Row, ColEnd, 2, False)
        else
        if S = ')' then Tokens.Add(S, IDS_CloseRound, Col, Row, ColEnd, 2, False)
        else
        if S = '[' then Tokens.Add(S, IDS_OpenBlock, Col, Row, ColEnd, 2, False)
        else
        if S = ']' then Tokens.Add(S, IDS_CloseBlock, Col, Row, ColEnd, 2, False)
        else
        if S = ',' then Tokens.Add(S, IDS_Comma, Col, Row, ColEnd, 2, False)
        else
        if S = '.' then Tokens.Add(S, IDS_Period, Col, Row, ColEnd, 2, False)
        else
        if S = '..' then Tokens.Add(S, IDS_Dots, Col, Row, ColEnd, 2, False)
        else
        if S = ';' then Tokens.Add(S, IDS_SemiColon, Col, Row, ColEnd, 2, False)
        else
        if S = ':' then Tokens.Add(S, IDS_Colon, Col, Row, ColEnd, 2, False)
        else
        if S = '=' then Tokens.Add(S, IDS_Assignment, Col, Row, ColEnd, 2, False)
        else
        if S = '+' then Tokens.Add(S, IDS_Plus, Col, Row, ColEnd, 2, False)
        else
        if S = '-' then Tokens.Add(S, IDS_Minus, Col, Row, ColEnd, 2, False)
        else
        if S = '*' then Tokens.Add(S, IDS_Multiply, Col, Row, ColEnd, 2, False)
        else
        if S = '/' then Tokens.Add(S, IDS_Divide, Col, Row, ColEnd, 2, False)
        else
        if S = '?' then Tokens.Add(S, IDS_QuestionMark, Col, Row, ColEnd, 2, False)
        else
          Tokens.Add(S, IDS_Unknown, Col, Row, ColEnd, 2, False);
      end;

      msString:
      begin
        S := AsString;
        AnsStr := LowerCase(S);
        HKW := KeyWordList.GetKey(AnsStr);
        if HKW <> nil then
          Tokens.Add(S, HKW.Idnet, Col, Row, ColEnd, 0, False)
        else
          Tokens.Add(S, IDS_Identifier, Col, Row, ColEnd, 0, False)

      end;
      msComent:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_Comment, Col, Row, ColEnd, 0, False);
      end;
      msInteger:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_Integer, Col, Row, ColEnd, 0, False);
      end;
      msHexInt:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_Integer, Col, Row, ColEnd, 0, False);
      end;

      msFloat:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_Float, Col, Row, ColEnd, 0, False);
      end;

      msText:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_String, Col, Row, ColEnd, 0, False);
      end;

      msUnknSym:
      begin
        S := string(AsString);
        Tokens.Add(S, IDS_Unknown, Col, Row, ColEnd, 0, False);
      end;
    end;
    C := NextToken;
  end;
  Tokens.Add('', IDS_EOF, Col, Row, ColEnd, 0, False);
end;

{ THashKeywordList }

function THashKeywordList.Get(Index: Integer): THashKeyWord;
begin
  Result := THashKeyWord(inherited Items[Index]);
end;

procedure THashKeywordList.Clear;
var
  i: integer;
begin
  for i := 0 to Count - 1 do
    THashKeyWord(Items[i]).Free;
  inherited Clear;
end;

constructor THashKeywordList.Create;
begin
  inherited Create;
  AppendKey('active', IDS_ACTIVE);
  AppendKey('add', IDS_ADD);
  AppendKey('after', IDS_AFTER);
  AppendKey('all', IDS_ALL);
  AppendKey('alter', IDS_ALTER);
  AppendKey('and', IDS_AND);
  AppendKey('any', IDS_ANY);
  AppendKey('as', IDS_AS);
  AppendKey('asc', IDS_ASC);
  AppendKey('ascending', IDS_ASCENDING);
  AppendKey('at', IDS_AT);
  AppendKey('auto', IDS_AUTO);
  AppendKey('autoddl', IDS_AUTODDL);
  AppendKey('based', IDS_BASED);
  AppendKey('basename', IDS_BASENAME);
  AppendKey('base_name', IDS_BASE_NAME);
  AppendKey('before', IDS_BEFORE);
  AppendKey('begin', IDS_BEGIN);
  AppendKey('between', IDS_BETWEEN);
  AppendKey('block', IDS_BLOCK);
  AppendKey('blobedit', IDS_BLOBEDIT);
  AppendKey('buffer', IDS_BUFFER);
  AppendKey('by', IDS_BY);
  AppendKey('cache', IDS_CACHE);
  AppendKey('character_length', IDS_CHARACTER_LENGTH);
  AppendKey('char_length', IDS_CHAR_LENGTH);
  AppendKey('check', IDS_CHECK);
  AppendKey('check_point_len', IDS_CHECK_POINT_LEN);
  AppendKey('check_point_length', IDS_CHECK_POINT_LENGTH);
  AppendKey('collate', IDS_COLLATE);
  AppendKey('collation', IDS_COLLATION);
  AppendKey('column', IDS_COLUMN);
  AppendKey('commit', IDS_COMMIT);
  AppendKey('commited', IDS_COMMITED);
  AppendKey('compiletime', IDS_COMPILETIME);
  AppendKey('computed', IDS_COMPUTED);
  AppendKey('close', IDS_CLOSE);
  AppendKey('conditional', IDS_CONDITIONAL);
  AppendKey('connect', IDS_CONNECT);
  AppendKey('constraint', IDS_CONSTRAINT);
  AppendKey('containing', IDS_CONTAINING);
  AppendKey('continue', IDS_CONTINUE);
  AppendKey('create', IDS_CREATE);
  AppendKey('current', IDS_CURRENT);
  AppendKey('current_date', IDS_CURRENT_DATE);
  AppendKey('current_time', IDS_CURRENT_TIME);
  AppendKey('current_timestamp', IDS_CURRENT_TIMESTAMP);
  AppendKey('cursor', IDS_CURSOR);
  AppendKey('database', IDS_DATABASE);
  AppendKey('day', IDS_DAY);
  AppendKey('key', IDS_DB_KEY);
  AppendKey('debug', IDS_DEBUG);
  AppendKey('dec', IDS_DEC);
  AppendKey('declare', IDS_DECLARE);
  AppendKey('default', IDS_DEFAULT);
  AppendKey('delete', IDS_DELETE);
  AppendKey('desc', IDS_DESC);
  AppendKey('descending', IDS_DESCENDING);
  AppendKey('describe', IDS_DESCRIBE);
  AppendKey('descriptor', IDS_DESCRIPTOR);
  AppendKey('disconnect', IDS_DISCONNECT);
  AppendKey('distinct', IDS_DISTINCT);
  AppendKey('do', IDS_DO);
  AppendKey('domain', IDS_DOMAIN);
  AppendKey('drop', IDS_DROP);
  AppendKey('echo', IDS_ECHO);
  AppendKey('edit', IDS_EDIT);
  AppendKey('else', IDS_ELSE);
  AppendKey('end', IDS_END);
  AppendKey('entry_point', IDS_ENTRY_POINT);
  AppendKey('escape', IDS_ESCAPE);
  AppendKey('event', IDS_EVENT);
  AppendKey('exception', IDS_EXCEPTION);
  AppendKey('execute', IDS_EXECUTE);
  AppendKey('exists', IDS_EXISTS);
  AppendKey('exit', IDS_EXIT);
  AppendKey('extern', IDS_EXTERN);
  AppendKey('external', IDS_EXTERNAL);
  AppendKey('extract', IDS_EXTRACT);
  AppendKey('fetch', IDS_FETCH);
  AppendKey('file', IDS_FILE);
  AppendKey('filter', IDS_FILTER);
  AppendKey('for', IDS_FOR);
  AppendKey('foreign', IDS_FOREIGN);
  AppendKey('found', IDS_FOUND);
  AppendKey('from', IDS_FROM);
  AppendKey('full', IDS_FULL);
  AppendKey('function', IDS_FUNCTION);
  AppendKey('gdscode', IDS_GDSCODE);
  AppendKey('generator', IDS_GENERATOR);
  AppendKey('global', IDS_GLOBAL);
  AppendKey('goto', IDS_GOTO);
  AppendKey('grant', IDS_GRANT);
  AppendKey('group', IDS_GROUP);
  AppendKey('group_commit_wait', IDS_GROUP_COMMIT_WAIT);
  AppendKey('group_commit_wait_time', IDS_GROUP_COMMIT_WAIT_TIME);
  AppendKey('having', IDS_HAVING);
  AppendKey('help', IDS_HELP);
  AppendKey('hour', IDS_HOUR);
  AppendKey('if', IDS_IF);
  AppendKey('immediate', IDS_IMMEDIATE);
  AppendKey('in', IDS_IN);
  AppendKey('inactive', IDS_INACTIVE);
  AppendKey('index', IDS_INDEX);
  AppendKey('indicator', IDS_INDICATOR);
  AppendKey('init', IDS_INIT);
  AppendKey('inner', IDS_INNER);
  AppendKey('input', IDS_INPUT);
  AppendKey('input_type', IDS_INPUT_TYPE);
  AppendKey('insert', IDS_INSERT);
  AppendKey('int', IDS_INT);
  AppendKey('into', IDS_INTO);
  AppendKey('is', IDS_IS);
  AppendKey('isolation', IDS_ISOLATION);
  AppendKey('isql', IDS_ISQL);
  AppendKey('join', IDS_JOIN);
  AppendKey('key', IDS_KEY);
  AppendKey('message', IDS_LC_MESSAGES);
  AppendKey('lc_type', IDS_LC_TYPE);
  AppendKey('left', IDS_LEFT);
  AppendKey('length', IDS_LENGTH);
  AppendKey('lev', IDS_LEV);
  AppendKey('level', IDS_LEVEL);
  AppendKey('like', IDS_LIKE);
  AppendKey('logfile', IDS_LOGFILE);
  AppendKey('buffer_size', IDS_LOG_BUFFER_SIZE);
  AppendKey('log_buf_size', IDS_LOG_BUF_SIZE);
  AppendKey('long', IDS_LONG);
  AppendKey('manual', IDS_MANUAL);
  AppendKey('maximum', IDS_MAXIMUM);
  AppendKey('maximum_segment', IDS_MAXIMUM_SEGMENT);
  AppendKey('max_segment', IDS_MAX_SEGMENT);
  AppendKey('merge', IDS_MERGE);
  AppendKey('message', IDS_MESSAGE);
  AppendKey('minimum', IDS_MINIMUM);
  AppendKey('minute', IDS_MINUTE);
  AppendKey('module_name', IDS_MODULE_NAME);
  AppendKey('month', IDS_MONTH);
  AppendKey('names', IDS_NAMES);
  AppendKey('national', IDS_NATIONAL);
  AppendKey('natural', IDS_NATURAL);
  AppendKey('nchar', IDS_NCHAR);
  AppendKey('no', IDS_NO);
  AppendKey('noauto', IDS_NOAUTO);
  AppendKey('not', IDS_NOT);
  AppendKey('null', IDS_NULL);
  AppendKey('num_log_buffs', IDS_NUM_LOG_BUFFS);
  AppendKey('num_log_buffers', IDS_NUM_LOG_BUFFERS);
  AppendKey('octet_length', IDS_OCTET_LENGTH);
  AppendKey('of', IDS_OF);
  AppendKey('on', IDS_ON);
  AppendKey('only', IDS_ONLY);
  AppendKey('open', IDS_OPEN);
  AppendKey('option', IDS_OPTION);
  AppendKey('or', IDS_OR);
  AppendKey('order', IDS_ORDER);
  AppendKey('outer', IDS_OUTER);
  AppendKey('output', IDS_OUTPUT);
  AppendKey('output_tye', IDS_OUTPUT_TYPE);
  AppendKey('overflow', IDS_OVERFLOW);
  AppendKey('page', IDS_PAGE);
  AppendKey('pagelength', IDS_PAGELENGTH);
  AppendKey('pages', IDS_PAGES);
  AppendKey('page_size', IDS_PAGE_SIZE);
  AppendKey('parameter', IDS_PARAMETER);
  AppendKey('password', IDS_PASSWORD);
  AppendKey('plan', IDS_PLAN);
  AppendKey('position', IDS_POSITION);
  AppendKey('post_event', IDS_POST_EVENT);
  AppendKey('precision', IDS_PRECISION);
  AppendKey('prepare', IDS_PREPARE);
  AppendKey('procedure', IDS_PROCEDURE);
  AppendKey('protected', IDS_PROTECTED);
  AppendKey('primaty', IDS_PRIMARY);
  AppendKey('privileges', IDS_PRIVILEGES);
  AppendKey('public', IDS_PUBLIC);
  AppendKey('quit', IDS_QUIT);
  AppendKey('raw_partitions', IDS_RAW_PARTITIONS);
  AppendKey('read', IDS_READ);
  AppendKey('real', IDS_REAL);
  AppendKey('record_version', IDS_RECORD_VERSION);
  AppendKey('references', IDS_REFERENCES);
  AppendKey('release', IDS_RELEASE);
  AppendKey('reserv', IDS_RESERV);
  AppendKey('reserving', IDS_RESERVING);
  AppendKey('retain', IDS_RETAIN);
  AppendKey('return', IDS_RETURN);
  AppendKey('returning', IDS_RETURNING);
  AppendKey('returning_values', IDS_RETURNING_VALUES);
  AppendKey('returns', IDS_RETURNS);
  AppendKey('revoke', IDS_REVOKE);
  AppendKey('right', IDS_RIGHT);
  AppendKey('rollback', IDS_ROLLBACK);
  AppendKey('runtime', IDS_RUNTIME);
  AppendKey('schema', IDS_SCHEMA);
  AppendKey('second', IDS_SECOND);
  AppendKey('segment', IDS_SEGMENT);
  AppendKey('select', IDS_SELECT);
  AppendKey('set', IDS_SET);
  AppendKey('shadow', IDS_SHADOW);
  AppendKey('shared', IDS_SHARED);
  AppendKey('shell', IDS_SHELL);
  AppendKey('show', IDS_SHOW);
  AppendKey('singular', IDS_SINGULAR);
  AppendKey('size', IDS_SIZE);
  AppendKey('snapshot', IDS_SNAPSHOT);
  AppendKey('some', IDS_SOME);
  AppendKey('sort', IDS_SORT);
  AppendKey('sql', IDS_SQL);
  AppendKey('sqlcode', IDS_SQLCODE);
  AppendKey('sqlerror', IDS_SQLERROR);
  AppendKey('sqlwarning', IDS_SQLWARNING);
  AppendKey('stability', IDS_STABILITY);
  AppendKey('starting', IDS_STARTING);
  AppendKey('starts', IDS_STARTS);
  AppendKey('statement', IDS_STATEMENT);
  AppendKey('static', IDS_STATIC);
  AppendKey('statistics', IDS_STATISTICS);
  AppendKey('sub_type', IDS_SUB_TYPE);
  AppendKey('suspend', IDS_SUSPEND);
  AppendKey('table', IDS_TABLE);
  AppendKey('terminator', IDS_TERMINATOR);
  AppendKey('then', IDS_THEN);
  AppendKey('to', IDS_TO);
  AppendKey('transaction', IDS_TRANSACTION);
  AppendKey('translate', IDS_TRANSLATE);
  AppendKey('translation', IDS_TRANSLATION);
  AppendKey('trigger', IDS_TRIGGER);
  AppendKey('trim', IDS_TRIM);
  AppendKey('type', IDS_TYPE);
  AppendKey('uncommitted', IDS_UNCOMMITTED);
  AppendKey('union', IDS_UNION);
  AppendKey('unique', IDS_UNIQUE);
  AppendKey('update', IDS_UPDATE);
  AppendKey('user', IDS_USER);
  AppendKey('using', IDS_USING);
  AppendKey('value', IDS_VALUE);
  AppendKey('values', IDS_VALUES);
  AppendKey('variable', IDS_VARIABLE);
  AppendKey('varying', IDS_VARYING);
  AppendKey('version', IDS_VERSION);
  AppendKey('view', IDS_VIEW);
  AppendKey('wait', IDS_WAIT);
  AppendKey('weekday', IDS_WEEKDAY);
  AppendKey('when', IDS_WHEN);
  AppendKey('whenever', IDS_WHENEVER);
  AppendKey('where', IDS_WHERE);
  AppendKey('while', IDS_WHILE);
  AppendKey('with', IDS_WITH);
  AppendKey('work', IDS_WORK);
  AppendKey('write', IDS_WRITE);
  AppendKey('year', IDS_YEAR);
  AppendKey('yearday', IDS_YEARDAY);
  AppendKey('bigint', IDS_BIGINT);
  AppendKey('blob', IDS_BLOB);
  AppendKey('char', IDS_CHAR);
  AppendKey('character', IDS_CHARACTER);
  AppendKey('date', IDS_DATE);
  AppendKey('decimal', IDS_DECIMAL);
  AppendKey('double', IDS_DOUBLE);
  AppendKey('float', IDS_FLOAT);
  AppendKey('integer', IDS_INTEGER);
  AppendKey('numeric', IDS_NUMERIC);
  AppendKey('smallint', IDS_SMALLINT);
  AppendKey('time', IDS_TIME);
  AppendKey('timestamp', IDS_TIMESTAMP);
  AppendKey('varchar', IDS_VARCHAR);
  AppendKey('avg', IDS_AVG);
  AppendKey('cast', IDS_CAST);
  AppendKey('count', IDS_COUNT);
  AppendKey('gen_id', IDS_GEN_ID);
  AppendKey('lower', IDS_LOWER);
  AppendKey('max', IDS_MAX);
  AppendKey('min', IDS_MIN);
  AppendKey('sum', IDS_SUM);
  AppendKey('upper', IDS_UPPER);
end;

procedure THashKeywordList.AppendKey(AKey: string; Ident: TIdentSQL);
var
  HKV: THashKeyWord;
begin
  if GetKey(AKey) = nil then
  begin
    HKV := THashKeyWord.Create(AKey, Ident);
    Add(HKV);
  end;
end;

function THashKeywordList.GetKey(AKey: string): THashKeyWord;
var
  AHash: Cardinal;
  HKW: THashKeyWord;
  I: Integer;
begin
  Result := nil;
  AKey := LowerCase(AKey);
  AHash := HashName(PAnsiChar(AKey));
  for I := 0 to Count - 1 do
  begin
    HKW := Items[I];
    if (HKW.HashValue = AHash) and (HKW.Keyword = AKey) then
    begin
      Result := HKW;
      Break;
    end;
  end;
end;

{ THashKeyWord }

constructor THashKeyWord.Create(const AKey: string; AIdent: TIdentSQL);
begin
  inherited Create;
  fKeyLen := Length(AKey);
  fKeyword := AKey;
  fHashValue := HashName(PAnsiChar(AKey));
  fIdnet := AIdent;
end;

{ TSQLTokens }

function TSQLTokens.Get(Index: Integer): PSQLToken;
begin
  if (Index < 0) or (Index >= FList.Count) then Index := FList.Count  - 1;
  Result := PSQLToken(FList.Items[Index]);
end;

function TSQLTokens.GetToken: PSQLToken;
begin
  Result := Get(FPos);
end;

function TSQLTokens.GetPreviewNext: PSQLToken;
begin
  Result := Get(FPos + 1);
end;

constructor TSQLTokens.Create;
begin
  inherited Create;
  FList := TList.Create;
  FPos := 0;
end;

destructor TSQLTokens.Destroy;
begin
  Clear;
  FList.Free;
  inherited Destroy;
end;

procedure TSQLTokens.Clear;
var
  PR: PSQLToken;
  I: Integer;
begin
  FPos := 0;
  I := 0;
  while I < FList.Count do
  begin
    PR := PSQLToken(FList.Items[I]);
    Dispose(PR);
    FList.Items[I] := nil;
    Inc(I);
  end;
  FList.Clear;
end;

function TSQLTokens.Count: Integer;
begin
  Result := FList.Count;
end;

procedure TSQLTokens.Next;
begin
  Inc(FPos);
end;

procedure TSQLTokens.NextWithSkipComment;
begin
  Inc(FPos);
  if Token^.TokenID = IDS_Comment then
    while Token^.TokenID = IDS_Comment do Inc(FPos);
end;

procedure TSQLTokens.PrevWithSkipComment;
begin
  Dec(FPos);
  if Token^.TokenID = IDS_Comment then
    while Token^.TokenID = IDS_Comment do Dec(FPos);
end;

procedure TSQLTokens.Prev;
begin
  Dec(FPos);
end;

procedure TSQLTokens.First;
begin
  FPos := 0;
end;

procedure TSQLTokens.Last;
begin
  FPos := Count - 1;
end;

procedure TSQLTokens.Add(TokenName: string; TokenID: TIdentSQL; Col, Row,
  EndCol, KeyConst: Integer; IsEnd: Boolean);
var
  PR: PSQLToken;
begin
  New(PR);
  PR^.TokenName := TokenName;
  PR^.TokenID := TokenID;
  PR^.Col := Col;
  PR^.EndCol := EndCol;
  PR^.Row := Row;
  PR^.KeyConst := KeyConst;
  PR^.IsEnd := IsEnd;
  FList.Add(PR);
end;

end.

