{------------------------------------------------------------------------------}
{                                                                              }
{                               Alexey Kuryakin                                }
{                                    LGPL                                      }
{                                                                              }
{------------------------------------------------------------------------------}

unit expandenvvar; // Expand environment variables in Windows or Unix style

{$IFDEF FPC}{$mode objfpc}{$ENDIF}{$H+}
{$IFDEF FPC}{$inline on}{$ENDIF}

interface

uses
  Classes,SysUtils,Dialogs,Process,StrUtils,Math{$IFDEF WINDOWS},Windows{$ENDIF};

const                   // Expand modes for ExpandEnvironmentVariables
  emExpandLikeWinC = $01; // Windows Classic  - %Variable%
  emExpandLikeWinE = $02; // Windows Expanded - !Variable!
  emExpandLikeUnix = $04; // Unix classic     - $Variable
  emExpandLikeBash = $08; // Bash style       - ${Variable}
  emExpandBashCall = $10; // Bash call        - $(Command)
  emExpandBackTick = $20; // Bash call        - `Command`

const // Default value of ExpEnv Mode (when passed Mode=0)
  emExpandDefaults : Integer = emExpandLikeWinC + emExpandLikeWinE
                             + emExpandLikeUnix + emExpandLikeBash
                             + emExpandBashCall + emExpandBackTick;

const // Max.number of iterations for ExpEnv substitutions
  MaxExpEnvIterations : Integer = 10;

// Expand environment variables from Expression.
// Supports Unix-like $Name (for example, $HOME\Demo).
// Supports Unix-like ${Name} (for example, ${HOME}\Demo).
// Supports Unix-like `Command` (for example, $(pwd)).
// Supports Unix-like $(Command) (for example, $(pwd)).
// Supports Windows-like !Name! (for example, !ProgramFiles!\Demo).
// Supports Windows-like %Name% (for example, %ProgramFiles%\Demo).
// Note: arguments (%0,%1..%9 or $0,$1..$9) is NOT suported, it's not real environment variables.
function ExpandEnvironmentVariables(const Expression:String; Mode:Integer=0):String;

// Get/Set environment variables.
function GetEnvironmentVariable(Name:String):String;
function SetEnvironmentVariable(Name,Value:String):Boolean;

// Uses to set PChar arguments from String.
function PCharArg(const S:String):PChar; {$IFDEF FPC}inline;{$ENDIF}
function PWideCharArg(const S:WideString):PWideChar; {$IFDEF FPC}inline;{$ENDIF}

// Just for testing purposes.
function TestExpandEnvironmentVariables:String;

type
 LongString = String;

implementation

type
 TEnvSubstitutor=function(Name:LongString):LongString;

function PCharArg(const S:String):PChar; {$IFDEF FPC}inline;{$ENDIF}
begin
  if (S='') then Result:=nil else Result:=PChar(S);
end;

function PWideCharArg(const S:WideString):PWideChar; {$IFDEF FPC}inline;{$ENDIF}
begin
  if (S='') then Result:=nil else Result:=PWideChar(S);
end;

{$IFDEF WINDOWS}
function GetEnvironmentVariable(Name:String):String;
begin
  Result:='';
  if (Name='') then Exit;
  SetLength(Result,1024*64);
  SetLength(Result,Windows.GetEnvironmentVariable(PChar(Name),PChar(Result),Length(Result)));
end;
function SetEnvironmentVariable(Name,Value:String):Boolean;
begin
 Result:=Windows.SetEnvironmentVariable(PCharArg(Name),PCharArg(Value));
end;
{$ELSE}
function getenv(__name:Pchar):Pchar;cdecl;external 'c' name 'getenv';
function putenv(__string:Pchar):longint;cdecl;external 'c' name 'putenv';
function setenv(__name:Pchar; __value:Pchar; __replace:longint):longint;cdecl;external 'c' name 'setenv';
function unsetenv(__name:Pchar):longint;cdecl;external 'c' name 'unsetenv';
function GetEnvironmentVariable(Name:String):String;
begin
 if (Name='') then Result:='' else Result:=getenv(PCharArg(Name));
end;
function SetEnvironmentVariable(Name,Value:String):Boolean;
begin
 Result:=false;
 if (Name='') then Exit;
 if (Value='') then begin Result:=(unsetenv(PCharArg(Name))<>-1); Exit; end;
 Result:=(setenv(PCharArg(Name),PCharArg(Value),1)<>-1);
end;
{$ENDIF}

// Internal utility routines
////////////////////////////
procedure BookMark(i:Integer; var iStart,Count:Integer); {$IFDEF FPC}inline;{$ENDIF}
begin
  iStart:=i; Count:=0;
end;

procedure Accumulate(iStart:Integer; var Count:Integer); {$IFDEF FPC}inline;{$ENDIF}
begin
  if (iStart>0) then inc(Count);
end;

function TrySubstitution(var S:LongString; var iStart,Count:Integer; LeftOff,RightOff:Integer; Substitutor:TEnvSubstitutor=nil):Boolean;
var sName,sValue:LongString;
begin
  Result:=false;
  if (iStart>0) then begin
    if (Count>0) then begin
      sName:=Copy(S,iStart+LeftOff,Count);
      if Assigned(Substitutor)
      then sValue:=Substitutor(sName)
      else sValue:=GetEnvironmentVariable(sName);
      if not Assigned(Substitutor) then // For environment variables only
      if (sValue='') then Exit; // Environment variable is not defined, skip it.
      S:=Copy(S,1,iStart-1)+sValue+Copy(S,iStart+Count+LeftOff+RightOff,MaxInt);
      Result:=true;
    end;
    BookMark(0,iStart,Count);
  end;
end;

// Environment substitution in Windows Cmd Classic style.
// Replace %Name% expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeWinC(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeWinC)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '%':  if (iStart=0) then BookMark(i,iStart,Count) else
          if TrySubstitution(S,iStart,Count,1,1) then begin Result:=true; Exit; end else iStart:=0;
    else  Accumulate(iStart,Count);
  end;
end;

// Environment substitution in Windows Cmd Expanded style.
// Replace !Name! expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeWinE(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeWinE)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '!':  if (iStart=0) then BookMark(i,iStart,Count) else
          if TrySubstitution(S,iStart,Count,1,1) then begin Result:=true; Exit; end else iStart:=0;
    else  Accumulate(iStart,Count);
  end;
end;

// Environment substitution in Unix Shell style.
// Replace $Name expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeUnix(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeUnix)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '$':  if (iStart=0) then BookMark(i,iStart,Count);
    'A'..'Z',
    'a'..'z',
    '0'..'9',
    '_':  Accumulate(iStart,Count);
    else  if TrySubstitution(S,iStart,Count,1,0) then begin Result:=true; Exit; end else iStart:=0;
  end;
  if TrySubstitution(S,iStart,Count,1,0) then begin Result:=true; Exit; end;
end;

// Environment substitution in Unix Bash style.
// Replace ${Name} expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeBash(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeBash)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '$':  if (iStart=0) then BookMark(i,iStart,Count);
    '{':  if (iStart>0) then if (i<>iStart+1) then BookMark(0,iStart,Count);
    '}':  if TrySubstitution(S,iStart,Count,2,1) then begin Result:=true; Exit; end else iStart:=0;
    else  Accumulate(iStart,Count);
  end;
end;

function RunOut(Cmd:LongString):LongString;
var outstr:LongString;
begin
 Result:=''; outstr:='';
 if RunCommand(cmd,outstr)
 then Result:=TrimRightSet(outstr,[#13,#10]);
end;

// Environment substitution in Bash call style.
// Replace $(Name) expression to RunCommand out, return true if succeded.
function ExpandBashCall(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
 Result:=false;
 if ((Mode and emExpandBashCall)=0) then Exit;
 iStart:=0; Count:=0;
 for i:=1 to Length(S) do
 case S[i] of
  '$': if (iStart=0) then BookMark(i,iStart,Count);
  '(': if (iStart>0) then if (i<>iStart+1) then BookMark(0,iStart,Count);
  ')': if TrySubstitution(S,iStart,Count,2,1,@RunOut) then begin Result:=true; Exit; end else iStart:=0;
  else Accumulate(iStart,Count);
 end;
end;

// Environment substitution in Bash call with backticks style.
// Replace `Command` expression to RunCommand output, return true if succeded.
function ExpandBackTick(var S:LongString; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
 Result:=false;
 if ((Mode and emExpandBackTick)=0) then Exit;
 iStart:=0; Count:=0;
 for i:=1 to Length(S) do
 case S[i] of
  '`': if (iStart=0) then BookMark(i,iStart,Count) else
       if TrySubstitution(S,iStart,Count,1,1,@RunOut) then begin Result:=true; Exit; end else iStart:=0;
  else Accumulate(iStart,Count);
 end;
end;

function ExpandEnvironmentVariables(const Expression:String; Mode:Integer=0):String;
var i,MaxIterations:Integer;
begin
 Result:=Expression; if (Result='') then Exit;
 if (Mode=0) then Mode:=emExpandDefaults; if (Mode=0) then Exit;
 MaxIterations:=Max(0,MaxExpEnvIterations);
 i:=0; while (ExpandLikeWinE(Result,Mode) or ExpandLikeWinC(Result,Mode)) and (i<MaxIterations) do inc(i);
 i:=0; while (ExpandLikeBash(Result,Mode) or ExpandLikeUnix(Result,Mode)) and (i<MaxIterations) do inc(i);
 i:=0; while (ExpandBashCall(Result,Mode) or ExpandBackTick(Result,Mode)) and (i<MaxIterations) do inc(i);
end;

function TestExpandEnvironmentVariables:String;
begin
 Result:='Test ExpandEnvironmentVariables:'+LineEnding
        +ExpandEnvironmentVariables('$HOME')+LineEnding
        +ExpandEnvironmentVariables('%HOME%')+LineEnding
        +ExpandEnvironmentVariables('!HOME!')+LineEnding
        +ExpandEnvironmentVariables('${HOME}')+LineEnding
        +ExpandEnvironmentVariables('$HOME\Demo')+LineEnding
        +ExpandEnvironmentVariables('!HOME!\Demo')+LineEnding
        +ExpandEnvironmentVariables('%HOME%\Demo')+LineEnding
        +ExpandEnvironmentVariables('${HOME}\Demo')+LineEnding
        +ExpandEnvironmentVariables('$UserProfile')+LineEnding
        +ExpandEnvironmentVariables('!UserProfile!')+LineEnding
        +ExpandEnvironmentVariables('%UserProfile%')+LineEnding
        +ExpandEnvironmentVariables('${UserProfile}')+LineEnding
        +ExpandEnvironmentVariables('$UserProfile\Demo')+LineEnding
        +ExpandEnvironmentVariables('!UserProfile!\Demo')+LineEnding
        +ExpandEnvironmentVariables('%UserProfile%\Demo')+LineEnding
        +ExpandEnvironmentVariables('${UserProfile}\Demo')+LineEnding;
end;

initialization

  // Just for testing:
  // ShowMessage(TestExpandEnvironmentVariables);

end.

