{------------------------------------------------------------------------------}
{                                                                              }
{                               Yuriy Kopnin                                   }
{                                   LGPL                                       }
{                                                                              }
{------------------------------------------------------------------------------}
unit Editor;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, SynEdit, synhighlighterpas_r, SynHighlighterAny,
  SynHighlighterMulti, SynCompletion, Forms, Controls, Graphics,
  Dialogs, ComCtrls, ExtCtrls, StdCtrls, types, LCLType, Menus, ActnList,
  dpi_parser, SurCodeMap, SynEditTypes, FormDesig, synhighlighterdpas,
  ComponentExport, dpCompil, dpUtils, typinfo, TreeViewPropEdit,
  SynEditMarkupWordGroup, SynEditMarkup, SynEditMouseCmds,
  LazSynEditMouseCmdsTypes, SynEditMarkupCtrlMouseLink, SynEditMarks,
  SynEditHighlighter;
  //SynEditMarkupHighAll;

type
  TPageModule = class;

  { TCompletionItem }

  TCompletionItem = class
  private
  protected
  public
    ItemType: TCompletionItemType;
    ItemParam: string;
    ItemResult: string;
    destructor Destroy; override;
  end;

  { TSuSynEdit }

  TOnFindDeclaration = procedure (DeclUnit, ObjName: string; ACol, ARow: Integer) of Object;
  TOnSetDelBreakPoint = procedure (ALint: Integer);

  TSuSynEdit = class(TSynEdit)
  private
    FPageModule: TPageModule;
    FOnFindDeclaration: TOnFindDeclaration;
  protected
    CynMarkupMouseLink: TSynEditMarkupCtrlMouseLink;
    MA: TSynEditMouseAction;
    KeyMarkUp: TSynEditMarkup;
    MouseActionWord: string;

    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    function FindDeclaration(S: string; X, Y: Integer; GoDeclaration: Boolean = True): Boolean;
    procedure MouseLink(Sender: TObject; X, Y: Integer; var AllowMouseLink: Boolean);
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer
       ); override;
    procedure SetDelBreakPoint;
    function GetReadOnly: boolean; override;
  public
    MarkImages: TImageList;
    ExecutMark: TSynEditMark;
    ExMarkIndex: Integer;
    //MarkCaret: TSynEditMarkupHighlightAllCaret;
    constructor Create(AOwner: TComponent; PM: TPageModule);
    procedure CenterCaret;
    procedure DoSetComment;
    procedure DoImplementMethodCur;
    procedure DoFindMethodDef;
    procedure AddComponent(AOwner, AComponent: TComponent);
    procedure DelComponent(AOwner, AComponent: TComponent);
    procedure CreateMethod(AMethodName, AClassName, DeclStr, ImplStr: string);
    procedure RanameComponent(AComponent: TComponent; const NewName, OldName: string);
    procedure SetUnitName(UName: string);
    procedure ShowMethod(AMethodName, AClassName: string);
    procedure SetExecutePos(P: TPoint);
    procedure Block_Indent;
    procedure Block_UnIndent;
    procedure GetUnitList(SList: TStrings; Tokens: TMemoTokens);
    procedure GetUnitList(SList: TStrings);
    procedure UsesUnits(AUnitName: string);
    procedure Select(P1, P2: TPoint);
    function SelectionExist: Boolean;
    procedure ReplaceSelect(NewText: string);
    property PageModule: TPageModule read FPageModule write FPageModule;
    property OnFindDeclaration: TOnFindDeclaration
             read FOnFindDeclaration write FOnFindDeclaration;
  end;

  { TPageModule }

  TPageModule = class(TTabSheet)
  private
    FSynEdit: TSuSynEdit;
    FDesignObject: TComponent;
    FDesignForm: TCustomForm;
    FReadOnly: Boolean;
    function GetReadOnly: Boolean;
    procedure SetReadOnly(AValue: Boolean);
  protected
    AltCurStr: string;
    CompletionList: TStringList;
    WordOfCurPos: TStringList;
    procedure ExecuteCompletion(Sender: TObject);
    function PaintCompletionItem(const AKey: string; ACanvas: TCanvas;
      X, Y: integer; Selected: boolean; Index: integer): boolean;
    procedure SetDesignObject(AValue: TComponent);
    function GetModified: Boolean;
    procedure SetModified(Value: Boolean);
    procedure SynEditStatChange(Sender: TObject; Changes: TSynStatusChanges);
    procedure FindDeclarationObject(DeclUnit, ObjName: string; ACol, ARow: Integer);
    procedure OnSynPaste(Sender: TObject; var AText: String;
      var AMode: TSynSelectionMode; ALogStartPos: TPoint;
      var AnAction: TSynCopyPasteAction);
  public
    CodeLanguage: string;

    Completion: TSynCompletion;
    RootName: string;
    ExtModule: Boolean;
    constructor Create(TheOwner: TComponent); override;
    destructor Destroy; override;
    procedure ClearCompletion;
    procedure CaretPosInStatusBar;
    procedure EditModeInStatusBar;
    procedure CompletionSearchPosition(var Position :integer);
    procedure AddCompletionItem(AType: TCompletionItemType; AName, AParam, AResult: string);
    procedure OnCompletion(var Value: string;
                                   SourceValue: string;
                                   var SourceStart, SourceEnd: TPoint;
                                   KeyChar: TUTF8Char;
                                   Shift: TShiftState);
    procedure OnPrevChar(Sender: TObject);
    procedure OnNextChar(Sender: TObject);
    procedure ComplUTF8KeyPress(Sender: TObject; var UTF8Key: TUTF8Char);
    property SynEdit: TSuSynEdit read FSynEdit;
    property DesignObject: TComponent read FDesignObject write SetDesignObject;
    property DesignForm: TCustomForm read FDesignForm write FDesignForm;
    property Modified: Boolean read GetModified write SetModified;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly;
  end;

  { TEditorForm }

  TEditorForm = class(TForm)
    ActCloseModule: TAction;
    ActFindDialog: TAction;
    ActFindNext: TAction;
    ActBreakPoint: TAction;
    ActBlockIndent: TAction;
    ActBlockUnIndent: TAction;
    ActChangeReadOnly: TAction;
    ActFindDeclaration: TAction;
    ActSetComment: TAction;
    ActPaste: TAction;
    ActSelectCut: TAction;
    ActSelectCopy: TAction;
    ActReplace: TAction;
    ActionList1: TActionList;
    FindDialog1: TFindDialog;
    ImageList1: TImageList;
    imgBookMarks: TImageList;
    ListBox1: TListBox;
    MenuItem1: TMenuItem;
    MenuItem10: TMenuItem;
    MenuItem11: TMenuItem;
    MenuItem12: TMenuItem;
    MenuItem13: TMenuItem;
    MenuItem14: TMenuItem;
    MenuItem15: TMenuItem;
    MenuItem16: TMenuItem;
    MenuItem17: TMenuItem;
    MenuItem2: TMenuItem;
    MenuItem3: TMenuItem;
    MenuItem4: TMenuItem;
    MenuItem5: TMenuItem;
    MenuItem6: TMenuItem;
    MenuItem7: TMenuItem;
    MenuItem8: TMenuItem;
    MenuItem9: TMenuItem;
    PageControl1: TPageControl;
    PagePopupMenu: TPopupMenu;
    EditPopup: TPopupMenu;
    ReplaceDialog1: TReplaceDialog;
    Splitter1: TSplitter;
    StatusBar1: TStatusBar;
    procedure ActBlockIndentExecute(Sender: TObject);
    procedure ActBlockUnIndentExecute(Sender: TObject);
    procedure ActBreakPointExecute(Sender: TObject);
    procedure ActChangeReadOnlyExecute(Sender: TObject);
    procedure ActChangeReadOnlyUpdate(Sender: TObject);
    procedure ActCloseModuleExecute(Sender: TObject);
    procedure ActFindDeclarationExecute(Sender: TObject);
    procedure ActFindDialogExecute(Sender: TObject);
    procedure ActFindNextExecute(Sender: TObject);
    procedure ActPasteExecute(Sender: TObject);
    procedure ActPasteUpdate(Sender: TObject);
    procedure ActReplaceExecute(Sender: TObject);
    procedure ActSelectCopyExecute(Sender: TObject);
    procedure ActSelectCopyUpdate(Sender: TObject);
    procedure ActSelectCutExecute(Sender: TObject);
    procedure ActSetCommentExecute(Sender: TObject);
    procedure FindDialog1Find(Sender: TObject);
    procedure FindDialog1Show(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ListBox1DrawItem(Control: TWinControl; Index: Integer;
      ARect: TRect; State: TOwnerDrawState);
    procedure ListBox1KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState
      );
    procedure ListBox1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ListBox1MouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure PageControl1Change(Sender: TObject);
    procedure PageControl1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ReplaceDialog1Replace(Sender: TObject);
    procedure ReplaceDialog1Show(Sender: TObject);
  private
    DoubleClicked: Boolean;
    function GetPageModule(Index: Integer): TPageModule;
    procedure ParseMessage;
  public
    SynDPasSyn1: TSynDPasSyn;
    SynRPasSyn: TSynRPasSyn;
    DisableSynDPasSyn: TSynDPasSyn;

    procedure HideExecMark;
    procedure AssignDPascalHighlight;
    function GetUnitMap(AUnitName: string; var BaseProc: TSuProcItem): TSuCodeMap;
    function GetPageModuleOfRoot(ARoot: TComponent): TPageModule;
    function ActivePageModule: TPageModule;
    function ModuleCount: Integer;
    function GetModuleByName(AUnitName: string): TPageModule;
    procedure CloseAll;
    function CloseModule(PM: TPageModule): Boolean;
    procedure ActivateByRoot(ARoot: TComponent);
   // procedure InsertClassComponent(ARoot, AComponent: TComponent);
    function Modified: Boolean;
    function NewModule(DesignObj: TComponent; DesignForm: TCustomForm;
      ACaption, ACode, ALang: string): TPageModule;
    procedure WriteDebugMessage(S: string);
    procedure GoDebugPos(S: string);
    procedure GoModulePos(AModule: string; ARow, ACol: Integer);
    procedure UpdateEditorFont;
    property Modules[Index: Integer]: TPageModule read GetPageModule;
  end;

procedure CreateEditor;

var
  EditorForm: TEditorForm;
  UserControlShift: TShiftState = [ssCtrl, ssShift];
  EditorDefaultFontName: string;
  EditorDefaultFontSize: Integer;
  ReadOnlyEditorColor: TColor;
  ReadOnlyGrayFont: Boolean;

implementation

{$R *.lfm}

uses Main, ProjectManager, dpRTTIutils, SuFrame, LazUTF8, Clipbrd,
  SynEditKeyCmds, EnvOptions, LazFileUtils, appconst, LConvEncoding,
  LangTransCode;

{}

procedure CreateEditor;
begin
  if EditorForm = nil then
  begin
    EditorForm := TEditorForm.Create(Application);
    EditorForm.Left := MainIDEDesForm.Inspector.Left + MainIDEDesForm.Inspector.Width + 16;
    EditorForm.Top := MainIDEDesForm.Top + MainIDEDesForm.Height + 38;
    EditorForm.Show;
  end;
end;

{ TCompletionItem }

destructor TCompletionItem.Destroy;
begin
  ItemParam := '';
  ItemResult := '';
  inherited Destroy;
end;

{ TSuSynEdit }

procedure TSuSynEdit.KeyDown(var Key: Word; Shift: TShiftState);
var
  S: string;
  I, BX, BY: Integer;
begin
  if ((Key = 67) or (Key = 83)) and (Shift = UserControlShift) then
  begin
    Key := 0;
    DoImplementMethodCur;
  end
  else
  if (Key = VK_UP) and (Shift = UserControlShift) then
  begin
    Key := 0;
    DoFindMethodDef;
  end
  else
  if ((Key = VK_RETURN) and (Shift = [ssCtrl])) or
     ((Key = VK_UP) and (Shift = [ssAlt])) then
  begin
    Key := 0;
    S := GetWordAtRowCol(CaretXY);
    if S <> '' then FindDeclaration(S, CaretX, CaretY);
  end
  else
  if (Key = VK_F5) and (Shift = []) then
    SetDelBreakPoint
  else
  if (Key in [VK_0..VK_9]) and (Shift = UserControlShift) then
  begin
    I := Key - VK_0;
    if GetBookMark(I, BX, BY) then
    begin
      if (BX = CaretX) and (BY = CaretY) then
        ClearBookMark(I)
      else
        SetBookMark(I, CaretX, CaretY);
    end
    else
      SetBookMark(I, CaretX, CaretY);
  end
  else
  if (Key in [VK_0..VK_9]) and (Shift = [ssCtrl]) then
  begin
    I := Key - VK_0;
    if IsBookmark(I) then
    begin
      GotoBookMark(I);
      CenterCaret;
    end;
  end
  else
  if ((Key = 191) or (Key = VK_DIVIDE)) and (Shift = [ssCtrl]) then
  begin
    DoSetComment;
    Key := 0;
  end
  else
  if (Key = 77) and (Shift = [ssCtrl]) then
  begin
    if not ReadOnly then
      PasteFromClipboard;
  end
  else
  if (Key = 83) and (Shift = [ssCtrl]) then
    CopyToClipboard
  else
  if (Key = 88) and (Shift = [ssCtrl]) then
  begin
    if not ReadOnly then CutToClipboard;
  end
  else
  if (Key = 81) and (Shift = [ssCtrl]) then
    Undo
  else
    inherited KeyDown(Key, Shift);
end;

function TSuSynEdit.FindDeclaration(S: string; X, Y: Integer; GoDeclaration: Boolean): Boolean;
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CodeMap, ExtCodeMap: TSuCodeMap;
  SList, ExtUses: TStringList;
  BaseProc, Proc, AProc, ExtProc: TSuProcItem;
  ExtList: TList;
  IsDecl: Boolean;
  AObj: TSuObject;
  AObjClass: TSuObjectClass;
  AFindName, SS, Str: string;
  StartX, EndX, I: Integer;

  function FindByName(AName: string): TSuObject;
  var
    M: Integer;
    ACodeMap: TSuCodeMap;
    AProc: TSuProcItem;
  begin

    for M := 0 to ExtList.Count - 1 do
    begin
      ACodeMap := TSuCodeMap(ExtList.Items[M]);
      AProc := ACodeMap.MainProc;
      Result := AProc.FindObject(AName);
      if Result = nil then
        Result := AProc.FindClass(AName);

      if Result <> nil then Break;
    end;
  end;

  function FindClassByName(AName: string): TSuObjectClass;
  var
    M: Integer;
    ACodeMap: TSuCodeMap;
    AProc: TSuProcItem;
  begin
    Result := nil;
    for M := 0 to ExtList.Count - 1 do
    begin
      ACodeMap := TSuCodeMap(ExtList.Items[M]);
      AProc := ACodeMap.MainProc;
      Result := AProc.FindClass(AName);

      if Result <> nil then Break;
    end;
  end;

begin
  Result := False;

  if FOnFindDeclaration = nil then Exit;
  AFindName:= UTF8LowerCase(S);
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CodeMap := TSuCodeMap.Create;
  ExtUses := TStringList.Create;
  ExtList := TList.Create;
  SList := TStringList.Create;
  try
    GetWordBoundsAtRowCol(Point(X, Y), StartX, EndX);
    Dec(StartX);
    while StartX > 0 do
    begin
      Str := Lines.Strings[Y - 1];
      I := Length(Str);
      if (StartX <= I) and (Str[StartX] = '.') then
      begin
        AFindName:= GetWordAtRowCol(Point(StartX, Y) );
        SList.Insert(0, AFindName);
        GetWordBoundsAtRowCol(Point(StartX, Y), StartX, EndX);
        Dec(StartX);
      end
      else Break;
    end;
    SList.Add(S);

    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    BaseProc := CodeMap.GetCodeMap(Tokens);
    BaseProc.GetUnitList(ExtUses);
    ExtList.Add(CodeMap);
    for I := 0 to ExtUses.Count - 1 do
    begin
      SS := ExtUses.Strings[I];
      ExtCodeMap := EditorForm.GetUnitMap(SS, ExtProc);
      if ExtCodeMap <> nil then
        ExtList.Add(ExtCodeMap);
    end;

    AObj := nil;

    for I := 0 to SList.Count - 1 do
    begin
      AFindName := UTF8LowerCase(SList.Strings[I]);
      if I = 0 then
      begin
        AObj := FindByName(AFindName);

        if (AObj = nil) then
        begin
          Proc := BaseProc.GetProcOfPos(X, Y - 1, IsDecl);
          if IsDecl then
          begin
            Proc := nil;
            if GoDeclaration then
              DoFindMethodDef;
            Result := True;
          end
          else
          if Proc <> nil then
          begin
            AObj := Proc.FindObject(AFindName);
            if (AObj = nil) and (Proc.Proc <> nil) and (Proc.Proc <> BaseProc) then
              AObj := Proc.Proc.FindObject(AFindName);

            if AObj = nil then
            begin
              if Proc.ClassOwner <> nil then
              begin
                if Proc.ClassOwner.Name = AFindName then
                  AObj := Proc.ClassOwner
                else
                begin
                  AObj := Proc.ClassOwner.Find(AFindName);
                end;
              end;
            end;
          end;
        end;

        if AObj = nil then Break;
      end
      else
      begin
        if AObj <> nil then
        begin
          AObjClass := nil;
          if AObj is TSuObjectClass then
            AObjClass := TSuObjectClass(AObj)
          else
            AObjClass := FindClassByName(AObj.ValueType);

          if AObjClass <> nil then
             AObj := AObjClass.Find(AFindName)
          else
          begin
            AObj := nil;
            Break;
          end;
        end
      end;

    end;

    if AObj = nil then
    begin
      //Ищем во внешних модулях
      for I := 0 to ExtUses.Count - 1 do
      begin
        SS := UTF8LowerCase(ExtUses.Strings[I]);
        if SS = AFindName then
        begin
          Result := True;
          if GoDeclaration then
           ProjManager.OpenModuleByName(S, False);
        end;
      end;
    end;

    if AObj <> nil then
    begin
      Result := True;
      if GoDeclaration then
      begin
        if AObj is TSuProcItem then
        begin
          AProc := TSuProcItem(AObj);
          FOnFindDeclaration(AProc.DeclUnit, AProc.Name, AProc.DeclPos.x, AProc.DeclPos.y);
        end
        else
          FOnFindDeclaration(AObj.DeclUnit, AObj.Name, AObj.Col, AObj.Row);
      end;
    end;
  finally
    Parser.Free;
    CodeMap.Free;
    Tokens.Free;
    SList.Free;
    ExtUses.Free;
    //Очищаем с первого элемента, элемент 0 = CodeMap и уже разрушен
    for I:= 1 to ExtList.Count - 1 do
      TSuCodeMap(ExtList.Items[I]).Free;
    ExtList.Free;
  end;
end;

procedure TSuSynEdit.MouseLink(Sender: TObject; X, Y: Integer;
  var AllowMouseLink: Boolean);
begin
  AllowMouseLink := False;
  MouseActionWord := GetWordAtRowCol(Point(X, Y));
  if MouseActionWord <> '' then
  begin
    if Highlighter <> nil then
    begin
      if not Highlighter.IsKeyword(MouseActionWord) then
        AllowMouseLink := FindDeclaration(MouseActionWord, X, Y, False);
    end
    else
      AllowMouseLink := FindDeclaration(MouseActionWord, X, Y, False);
  end;
  if not AllowMouseLink then MouseActionWord := '';
end;

procedure TSuSynEdit.MouseUp(Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);
var
  P: TPoint;
begin
  if (Button = TMouseButton.mbLeft) and (Shift = [ssCtrl]) and (MouseActionWord <> '') then
  begin
    //P := PixelsToLogicalPos(Point(X, Y));
    P := CaretXY;
    if FindDeclaration(MouseActionWord, P.x, P.y) then
      MouseActionWord := '';
  end;
  //else
   inherited MouseUp(Button, Shift, X, Y);
end;

procedure TSuSynEdit.SetDelBreakPoint;
var
  ML: TSynEditMarkLine;
  AMark: TSynEditMark;
  I: Integer;
  AList: TList;
  IsDel: Boolean;
  S: string;
begin
  IsDel := False;
  ML := Marks.Line[CaretY];
  if ML <> nil then
  begin
    AList := TList.Create;
    try
      for I := 0 to ML.Count - 1 do
      begin
        if not ML.Items[I].IsBookmark then
        begin
          if ML.Items[I].ImageIndex = 0 then
            AList.Add(ML.Items[I]);
        end;
      end;
      if AList.Count > 0 then
      begin
        //Delete BreakPoint
        IsDel := True;
        for I := 0 to AList.Count - 1 do
        begin
          AMark := TSynEditMark(AList.Items[I]);
          ML.Remove(AMark);
          AMark.Free;
          S := EditorForm.PageControl1.ActivePage.Caption + '=' + IntToStr(CaretY - 1);
          MainIDEDesForm.DelBreakPoint(S);
        end;
      end;
    finally
      AList.Free;
    end;
  end;
  if not IsDel then
  begin
    //Set BreakPoint
    I := CaretY - 1;
    S := '';
    if I > 0 then S := Lines.Strings[I];
    if S <> '' then
    begin
      AMark := TSynEditMark.Create(Self);
      AMark.Line := CaretY;
      AMark.ImageList := MarkImages;
      AMark.ImageIndex := 0;
      AMark.Visible := true;
      Marks.Add(AMark);
      S := EditorForm.PageControl1.ActivePage.Caption + '=' + IntToStr(CaretY - 1);
      MainIDEDesForm.AddBreakPoint(S);
    end;
  end;
end;

function TSuSynEdit.GetReadOnly: boolean;
begin
  Result:= inherited GetReadOnly;
  if not Result then
     Result := MainIDEDesForm.Process1.Running;

  if not Result then
    Result := ProjManager.ReadOnly;
end;

constructor TSuSynEdit.Create(AOwner: TComponent; PM: TPageModule);
begin
  inherited Create(AOwner);
  Font.Quality:= fqDefault;
  TabWidth := 2;
  Options := Options - [eoSmartTabs];
  Options := Options + [eoTabIndent];
  MouseOptions := MouseOptions + [emShowCtrlMouseLinks];
  FPageModule := PM;

  Gutter.AutoSize:= True;

  Gutter.MarksPart(0).AutoSize := False;
  Gutter.MarksPart(0).Width := 30;
  MarkImages := nil;
  KeyMarkUp := MarkupByClass[TSynEditMarkupWordGroup];
  KeyMarkUp.MarkupInfo.Background := clCream;
  KeyMarkUp.MarkupInfo.Foreground := MainIDEDesForm.SynPasSyn1.KeyAttri.Foreground;

  MouseActionWord := '';

  MA := MouseActions.Add;
  MA.Shift := [ssCtrl];
  MA.ShiftMask := [ssCtrl];
  MA.Command := emcMouseLink;
  MA.Option := emcoMouseLinkShow;
  MA.Button:= mbLeft;
  MA.ClickCount := ccSingle;
  MA.ClickDir:= cdUp;
  MA.IgnoreUpClick := False;

  CynMarkupMouseLink := TSynEditMarkupCtrlMouseLink(MarkupByClass[TSynEditMarkupCtrlMouseLink]);
  CynMarkupMouseLink.MarkupInfo.Foreground := clBlue;
  CynMarkupMouseLink.MarkupInfo.Style:= [fsUnderline];
  OnMouseLink := @MouseLink;
  MouseLinkColor.Foreground := clBlue;

  ExecutMark := TSynEditMark.Create(Self);
  ExecutMark.ImageIndex := 1;
  ExecutMark.ImageList := EditorForm.ImageList1;
  ExecutMark.Visible := False;
  ExMarkIndex := Marks.Add(ExecutMark);

  //MarkCaret := TSynEditMarkupHighlightAllCaret.Create(Self);
  //MarkCaret.WaitTime := 1000;
  //MarkCaret.FullWordMaxLen := 3;
  //Marks.Add(MarkCaret);
  //OnReplaceText := @ReplaceText;
end;

procedure TSuSynEdit.CenterCaret;
var
  LineCenterCount, I: Integer;
begin
  MA.MoveCaret := False;
  LineCenterCount:= LinesInWindow div 2;
  I := CaretY - LineCenterCount;
  if I < 1 then I := 1;
  TopLine := I;
end;

procedure TSuSynEdit.DoSetComment;
var
  I, N, L, M, ComPos: Integer;
  S: string;
  P, PEnd: TPoint;
  Rascoment, AllRowComment: Boolean;
  SStart, SEnd: Integer;
begin
  if ReadOnly then Exit;

  if SelectionExist then
  begin
    SStart := SelStart;
    SEnd := SelEnd;
    I := BlockBegin.Y;
    M := BlockEnd.Y;
    S := Lines.Strings[I - 1];
    L := UTF8Length(S);
    for N := 1 to L do
        if S[N] <> #32 then Break;
    ComPos := N;
    Inc(I);
    AllRowComment := True;
    Rascoment := False;
    while I <= M do
    begin
      S := Lines.Strings[I - 1];
      L := UTF8Length(S);
      for N := 1 to L do
        if S[N] <> #32 then
        begin
          if N < L then
          begin
            if (S[N] <> '/') or (S[N + 1] <> '/') then
               AllRowComment := False;
          end;
          Break;
        end;
      if (N < ComPos) and (L > 0) then ComPos := N;
      Inc(I);
    end;
    if AllRowComment then Rascoment := True;

    I := BlockBegin.Y;
    BeginUndoBlock;
    while I <= M do
    begin
      S := Lines.Strings[I - 1];
      L := UTF8Length(S);
      if Rascoment then
      begin
       for N := 1 to L do
          if S[N] <> #32 then Break;

        if N < L then
        begin
          if (S[N] = '/') and (S[N + 1] = '/') then
          begin
            P.Y := I;
            P.X := N;
            PEnd.Y := P.Y;
            PEnd.X := P.X + 2;
            P := PhysicalToLogicalPos(P);
            PEnd := PhysicalToLogicalPos(PEnd);
            TextBetweenPoints[P, PEnd] := '';
            Dec(SEnd, 2);
          end;
        end;
      end
      else
      begin
        P.Y := I;
        P.X := ComPos;
        P := PhysicalToLogicalPos(P);
        TextBetweenPoints[P, P] := '//';
        Inc(SEnd, 2);
      end;
      Inc(I);
    end;
    SelStart := SStart;
    SelEnd := SEnd;
    EndUndoBlock;
  end
  else
  begin
    S := Lines.Strings[CaretY - 1];
    L := UTF8Length(S);
    N := 1;
    if L = 0 then
    begin
      Rascoment := False;
      N := CaretX;
    end
    else
    begin
      Rascoment := True;
      for N := 1 to L do
        if S[N] <> #32 then
        begin
          if N < L then
          begin
            if (S[N] <> '/') or (S[N + 1] <> '/') then
               Rascoment := False;
          end;
          Break;
        end;
    end;
    P.Y := CaretY;
    P.X := N;
    if Rascoment then
    begin
      PEnd.Y := P.Y;
      PEnd.X := P.X + 2;
      P := PhysicalToLogicalPos(P);
      PEnd := PhysicalToLogicalPos(PEnd);
      TextBetweenPointsEx[P, PEnd, scamAdjust] := '';
      //if CaretX >= PEnd.X then CaretX := CaretX - 2;
    end
    else
    begin
      P := PhysicalToLogicalPos(P);
      TextBetweenPointsEx[P, P, scamAdjust] := '//';
    end;
  end;
end;

procedure TSuSynEdit.DoImplementMethodCur;
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CodeMap: TSuCodeMap;
  BaseProc: TSuProcItem;
  ProcItem: TSuProcItem;
  S, MethodStr: string;
  P: TPoint;
  I, NewPos: Integer;
  IsDecl: Boolean;
begin
  if ReadOnly then Exit;
  S := GetWordAtRowCol(CaretXY);
  if S <> '' then
  begin
    Parser := TSuMemoParser.Create;
    Tokens := TMemoTokens.Create;
    CodeMap := TSuCodeMap.Create;
    try
      Parser.SetParseStr(Text, FPageModule.CodeLanguage);
      Parser.GetTokensList(Tokens);
      BaseProc := CodeMap.GetCodeMap(Tokens);
      NewPos := CodeMap.GetNewPosition(Tokens);

      if (NewPos > 0) and (BaseProc <> nil) then
      begin
        ProcItem := BaseProc.GetProcOfPos(CaretX - 1, CaretY - 1, IsDecl);
        if BaseProc.LastViewProc <> nil then
        begin
          if BaseProc.LastViewProc.BodyRowEnd > 0 then
            NewPos := BaseProc.LastViewProc.BodyRowEnd + 2;
        end;

        if not IsDecl then ProcItem := nil;

        if (ProcItem <> nil) then
        begin
          if ProcItem.BodyRowEnd = 0 then
          begin

            MethodStr:= #13#10 + ProcItem.MethodDef + ' ';
            if ProcItem.ClassOwner <> nil then
              MethodStr:= MethodStr + ProcItem.ClassOwner.OriginalName + '.';
            MethodStr := MethodStr + ProcItem.OriginalName;
            if ProcItem.Params.Count > 0 then
            begin
              MethodStr:= MethodStr + '(';
              I := 0;
              while I < ProcItem.ParamCount do
              begin
                if I > 0 then MethodStr:= MethodStr + '; ';
                if ProcItem.Param[I].AsOut then
                  MethodStr:= MethodStr + GetKeyOfLangID(ID_out) + ' '
                else
                if ProcItem.Param[I].AsVar then
                  MethodStr:= MethodStr + GetKeyOfLangID(ID_var) + ' '
                else
                if ProcItem.Param[I].AsConst then
                  MethodStr:= MethodStr + GetKeyOfLangID(ID_const) + ' ';


                while I < ProcItem.ParamCount do
                begin
                  if ((I + 1) < ProcItem.ParamCount) and
                    (ProcItem.Param[I + 1].ValueType = ProcItem.Param[I].ValueType) and
                     (ProcItem.Param[I + 1].AsVar = ProcItem.Param[I].AsVar) and
                     (ProcItem.Param[I + 1].AsConst = ProcItem.Param[I].AsConst) and
                     (ProcItem.Param[I + 1].AsOut = ProcItem.Param[I].AsOut) then
                  begin
                    MethodStr:= MethodStr + ProcItem.Param[I].OriginalName + ', ';
                    Inc(I);
                  end
                  else
                  begin
                    MethodStr:= MethodStr + ProcItem.Param[I].OriginalName;
                    Break;
                  end;
                end;
                MethodStr:= MethodStr + ': ' + ProcItem.Param[I].ValueType;
                Inc(I);
              end;
              MethodStr:= MethodStr + ')';
            end;
            if ProcItem.ValueType <> '' then
              MethodStr:= MethodStr + ': ' + ProcItem.ValueType;

            MethodStr:= MethodStr + ';';
            MethodStr:= MethodStr + #13#10 + GetKeyOfLangID(ID_begin);
            if (ProcItem.MethodDef = 'constructor') or (UTF8CompareText(ProcItem.MethodDef, 'конструктор') = 0) then
            begin
              if (ProcItem.MethodDef = 'constructor') then
                  S := 'inherited'
              else
                  S := GetKeyOfLangID(ID_inherited);

              if (ProcItem.ParamCount > 0) and ((ProcItem.Param[0].Name = 'aowner') or (ProcItem.Param[0].Name = 'авладелец')) then
                MethodStr := MethodStr +  #13#10'  ' + S + ' ' + ProcItem.OriginalName + '(' + ProcItem.Param[0].OriginalName + ');'#13#10#13#10 + GetKeyOfLangID(ID_end) + ';'#13#10
              else
                MethodStr := MethodStr +  #13#10'  ' + S + ' ' + ProcItem.OriginalName +';'#13#10#13#10 + GetKeyOfLangID(ID_end) + ';'#13#10;
            end
            else
            if (UTF8CompareText(ProcItem.MethodDef, 'destructor') = 0) or (UTF8CompareText(ProcItem.MethodDef, 'деструктор') = 0) then
              MethodStr := MethodStr +  #13#10#13#10'  ' + GetKeyOfLangID(ID_inherited) + ' ' + GetTransName('Destroy') +';'#13#10 + GetKeyOfLangID(ID_end) + ';'#13#10
            else
              MethodStr := MethodStr + #13#10#13#10 + GetKeyOfLangID(ID_end) + ';'#13#10;
            p.x := 1; P.y := NewPos;
            P := PhysicalToLogicalPos(P);
            TextBetweenPointsEx[P, P, scamAdjust] := MethodStr;
            CaretX := 3;
            CaretY := P.y + 3;
            SetFocus;
            CenterCaret;
          end
          else
          begin
            CaretX:= 2;
            CaretY := ProcItem.BodyRowBegin + 1;
            SetFocus;
            CenterCaret;
          end;
        end;
        {else
        begin

        end;}
        //if AObj = nil then BaseProc.FindClassMethod();
      end;
    finally
      Parser.Free;
      CodeMap.Free;
      Tokens.Free;
    end;

  end;
end;

procedure TSuSynEdit.DoFindMethodDef;
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CodeMap: TSuCodeMap;
  BaseProc, FndProc: TSuProcItem;
  IsDecl: Boolean;
begin
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CodeMap := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    BaseProc := CodeMap.GetCodeMap(Tokens);
    if (BaseProc <> nil) then
    begin
      FndProc := BaseProc.GetProcOfPos(CaretX - 1, CaretY - 1, IsDecl);
      if FndProc <> nil then
      begin
        if IsDecl then
        begin
          //Если курсов в объявлении переходим к реалихации
          CaretX := FndProc.BodyColBegin + 1;
          CaretY := FndProc.BodyRowBegin + 1;;
          SetFocus;
          CenterCaret;
        end
        else
        begin
          CaretX := FndProc.DeclPos.x + 1;
          CaretY := FndProc.DeclPos.y + 1;;
          SetFocus;
          CenterCaret;
        end;
      end;
    end;
  finally
    Parser.Free;
    CodeMap.Free;
    Tokens.Free;
  end;
end;

{procedure TSuSynEdit.ReplaceText(Sender: TObject; const ASearch, AReplace:
      string; Line, Column: integer; var ReplaceAction: TSynReplaceAction);
begin

end;}

procedure TSuSynEdit.AddComponent(AOwner, AComponent: TComponent);
var
  AClassName, ACompName, ACompClassName: string;
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CodeMap: TSuCodeMap;
  BaseProc: TSuProcItem;
  OC: TSuObjectClass;
  Point: TPoint;
  S: string;
begin
  AClassName := 'T' + AOwner.Name;
  ACompName := AComponent.Name;
  if AComponent is TSuFrame then
    ACompClassName := TSuFrame(AComponent).VirtualClassName
  else
    ACompClassName := AComponent.ClassName;
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CodeMap := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    BaseProc := CodeMap.GetCodeMap(Tokens);
    if BaseProc <> nil then
    begin
      OC := BaseProc.FindClass(AClassName);
      if OC <> nil then
      begin
        if not OC.ComponentExist(ACompName) then
        begin
          OC.GetPositionForComponent(Point);
          S := ACompName + ': ' + ACompClassName + ';';
          Lines.Insert(Point.Y, '');
          Inc(Point.y); Inc(Point.x);
          TextBetweenPoints[Point, Point] := S;
          if CaretY >= Point.y then CaretY := CaretY + 1;
        end;
      end;
    end;
  finally
    Parser.Free;
    CodeMap.Free;
    Tokens.Free;
  end;
end;

procedure TSuSynEdit.DelComponent(AOwner, AComponent: TComponent);
var
  AClassName, ACompName: string;
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CM: TSuCodeMap;
  OC: TSuObjectClass;
  P: TSuProcItem;
  Point: TPoint;
begin
  AClassName := 'T' + AOwner.Name;
  ACompName := AComponent.Name;
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CM := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    P := CM.GetCodeMap(Tokens);
    P.FindClass(AClassName);
    if P <> nil then
    begin
      OC := P.FindClass(AClassName);
      if OC <> nil then
      begin
        if OC.GetCompoenentPos(ACompName, Point) then
        begin
          Lines.Delete(Point.Y);
          if CaretY > Point.y then CaretY := CaretY - 1;
        end;
      end;
    end;

  finally
    CM.Free;
    Tokens.Free;
    Parser.Free;
  end;
end;

procedure TSuSynEdit.ShowMethod(AMethodName, AClassName: string);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CM: TSuCodeMap;
  P, M: TSuProcItem;
  Point: TPoint;
begin
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CM := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    P := CM.GetCodeMap(Tokens);
    P.FindClass(AClassName);
    if P <> nil then
    begin
      M := P.FindClassMethod(AClassName, AMethodName);
      if M <> nil then
      begin
        Point.x := M.BodyColBegin + 1;
        Point.y := M.BodyRowBegin + 1;
        CaretXY := Point;
        EditorForm.Show;
        SetFocus;
        CenterCaret;
      end
    end;

  finally
    CM.Free;
    Tokens.Free;
    Parser.Free;
  end;
end;

procedure TSuSynEdit.SetExecutePos(P: TPoint);
var
  I: Integer;
begin
  CaretXY := P;
  I := TopLine + LinesInWindow  - 2;
  ExecutMark.Line:= CaretY;
  ExecutMark.Visible := True;
  if CaretY >= I then CenterCaret;
  if not EditorForm.Active then EditorForm.Show;
  SetFocus;
end;

procedure TSuSynEdit.Block_Indent;
begin
  ExecuteCommand(ecBlockIndent, '', nil);
end;

procedure TSuSynEdit.Block_UnIndent;
begin
  ExecuteCommand(ecBlockUnindent, '', nil);
end;

procedure TSuSynEdit.GetUnitList(SList: TStrings; Tokens: TMemoTokens);
begin
  Tokens.First;
  while Tokens.Token^.TokenID <> ID_EOF do
  begin
      if Tokens.Token^.TokenID = ID_uses then
      begin
        Tokens.Next;
        while True do
        begin
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            SList.Add(Tokens.Token^.TokenName);
            Tokens.Next;
            if Tokens.Token^.TokenID = ID_Comma then
            begin
              Tokens.Next;
              Continue;
            end
            else Break;
          end
          else Exit;

          Tokens.Next;
        end;
      end;
      Tokens.Next;
  end;
end;

procedure TSuSynEdit.GetUnitList(SList: TStrings);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
begin
  Tokens := TMemoTokens.Create;
  Parser := TSuMemoParser.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    GetUnitList(SList, Tokens);
  finally
    Tokens.Free;
    Parser.Free;
  end;
end;

procedure TSuSynEdit.UsesUnits(AUnitName: string);
var
  Pos: Integer;
  S, TokName: string;
  B: Boolean;
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  //CM: TSuCodeMap;
  Point: TPoint;
begin
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  //CM := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    //CM.GetCodeMap(Tokens);

    Tokens.First;
    B := False;
    Pos := 0;
    while Tokens.Token^.TokenID <> ID_EOF do
    begin
      if Tokens.Token^.TokenID = ID_program then
      begin
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Identifier then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_SemiColon then
          begin
            Tokens.Next;
            Pos := Tokens.Pos;
            B := TRue;
          end;
        end;
      end
      else
      if Tokens.Token^.TokenID = ID_Implementation then
      begin
        Tokens.Next;
        Pos := Tokens.Pos;
        B := TRue;
        Break;
      end;
      Tokens.Next;
    end;

    if B then
    begin
      Tokens.Pos := Pos;
      if Tokens.Token^.TokenID = ID_uses then
      begin
        Tokens.Next;
        while Tokens.Token^.TokenID <> ID_EOF do
        begin
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            TokName := Tokens.Token^.TokenName;
            if SameText(TokName, AUnitName) then Exit;
            Tokens.Next;
            if Tokens.Token^.TokenID = ID_Comma then
            begin
              Tokens.Next;
              Continue;
            end;

            if Tokens.Token^.TokenID = ID_SemiColon then
            begin
              Modified := True;
              Point.Y := Tokens.Token^.Row + 1;
              Point.X := Tokens.Token^.Col + 1;

              if (Point.X + UTF8Length(AUnitName)) >= 80 then
              begin
                AUnitName := ','#13#10'  ' + AUnitName;
                if CaretY > Point.Y then CaretY := CaretY + 1;
              end
              else
                AUnitName := ', ' + AUnitName;

              S := UTF8Copy(Lines.Strings[Tokens.Token^.Row], 1, Point.X);
              Point.X := UTF8Length(S);
              Point := PhysicalToLogicalPos(Point);
              TextBetweenPointsEx[Point, Point, scamIgnore] := AUnitName;
               Exit;
            end;

          end
          else Exit;

          Tokens.Next;
        end;
      end
      else
      begin
        Point.Y := Tokens.Token^.Row + 1;
        Point.x := 1;
        AUnitName := GetKeyOfLangID(ID_uses) + ' ' + AUnitName + ';'#13#10;
        if Lines.Strings[Tokens.Token^.Row] <> '' then
          AUnitName := AUnitName + #13#10;
        Point := PhysicalToLogicalPos(Point);
        TextBetweenPointsEx[Point, Point, scamAdjust] := AUnitName;
      end;
    end;

  finally
    //CM.Free;
    Tokens.Free;
    Parser.Free;
  end;

end;

procedure TSuSynEdit.Select(P1, P2: TPoint);
begin
  FBlockSelection.StartLineBytePos := P1;
  FBlockSelection.EndLineBytePos := P2;
  FBlockSelection.ActiveSelectionMode := smNormal;
end;

function TSuSynEdit.SelectionExist: Boolean;
begin
  Result := FBlockSelection.SelText <> '';
end;

procedure TSuSynEdit.ReplaceSelect(NewText: string);
begin
  if FBlockSelection.SelText <> '' then
  begin
    TextBetweenPoints[FBlockSelection.StartLineBytePos, FBlockSelection.EndLineBytePos] := NewText;
  end;
end;

procedure TSuSynEdit.CreateMethod(AMethodName, AClassName, DeclStr, ImplStr: string);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CM: TSuCodeMap;
  OC: TSuObjectClass;
  P, M: TSuProcItem;
  Point: TPoint;
  N: Integer;
begin
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CM := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    P := CM.GetCodeMap(Tokens);
    P.FindClass(AClassName);
    if P <> nil then
    begin
      M := P.FindClassMethod(AClassName, AMethodName);
      if M <> nil then
      begin
        Point.x := M.BodyColBegin + 1;
        Point.y := M.BodyRowBegin + 1;
        CaretXY := Point;
        EditorForm.Show;
        SetFocus;
      end
      else
      begin
        OC := P.FindClass(AClassName);
        if OC <> nil then
        begin
          OC.GetPositionForClassMethod(Point, N);
          Lines.Insert(Point.y, '');
          Inc(Point.y); Inc(Point.x);
          TextBetweenPoints[Point, Point] := DeclStr;
          if N = 0 then
            N := CM.GetNewPosition(Tokens);
          Lines.Insert(N, '');
          Lines.Insert(N, '');
          Point.y := N + 2;
          Point.x := 1;
          TextBetweenPoints[Point, Point] := ImplStr + #13#10 +
                                      GetKeyOfLangID(ID_begin) + #13#10#13#10 + GetKeyOfLangID(ID_end) + ';';
          Inc(Point.y, 2);
          Inc(Point.x, 2);
          CaretXY := Point;
          EditorForm.Show;
          SetFocus;
          CenterCaret;
        end;
      end;
    end;

  finally
    CM.Free;
    Tokens.Free;
    Parser.Free;
  end;
end;

procedure TSuSynEdit.RanameComponent(AComponent: TComponent; const NewName, OldName: string);
var
  NewTypeName, OldTypeName, S, AOldName: string;
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  PBegin, PEnd: TPoint;
  B: Boolean;
begin
  AOldName:= LowerCase(OldName);
  if AComponent.Owner = nil then
  begin
    OldTypeName := 't' + AOldName;
    NewTypeName := 'T' + NewName;
  end;
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    Tokens.Last;
    BeginUndoBlock;
    while Tokens.Pos > 0 do
    begin
        if Tokens.Token^.TokenID = ID_Identifier then
        begin
          S := LowerCase(Tokens.Token^.TokenName);
          if S = AOldName then
          begin
            PBegin.y:= Tokens.Token^.Row + 1;
            PBegin.x:= Tokens.Token^.Col + 1;
            PEnd.y := PBegin.y;
            PEnd.x := PBegin.x + UTF8Length(OldName);

            B := True;
            Tokens.PrevWithSkipComment;
            if Tokens.Token^.TokenID = ID_Period then
            begin
              B := False;
              Tokens.PrevWithSkipComment;
              if (Tokens.Token^.TokenID = ID_Identifier) and
                SameText(Tokens.Token^.TokenName, 'self') then B := True;
              Tokens.NextWithSkipComment;
            end;
            Tokens.NextWithSkipComment;

            if B then
            begin
              PBegin := PhysicalToLogicalPos(PBegin);
              PEnd := PhysicalToLogicalPos(PEnd);
              TextBetweenPointsEx[PBegin, PEnd, scamAdjust] := NewName;
            end;

          end
          else
          if (AComponent.Owner = nil) and
            (S = OldTypeName) then
          begin
            PBegin.y:= Tokens.Token^.Row + 1;
            PBegin.x:= Tokens.Token^.Col + 1;
            PEnd.y := PBegin.y;
            PEnd.x := PBegin.x + UTF8Length(OldTypeName);
            PBegin := PhysicalToLogicalPos(PBegin);
            PEnd := PhysicalToLogicalPos(PEnd);
            TextBetweenPoints[PBegin, PEnd] := NewTypeName;
          end;
        end;

      Tokens.PrevWithSkipComment;
    end;
    EndUndoBlock;
  finally
    Tokens.Free;
    Parser.Free;
  end;
end;

procedure TSuSynEdit.SetUnitName(UName: string);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CM: TSuCodeMap;
  P, P2: TPoint;
  L: Integer;
begin
  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CM := TSuCodeMap.Create;
  try
    Parser.SetParseStr(Text, FPageModule.CodeLanguage);
    Parser.GetTokensList(Tokens);
    Tokens.First;
     if Tokens.Token^.TokenID = ID_Comment then Tokens.NextWithSkipComment;
     if Tokens.Token^.TokenID = ID_Unit then
     begin
       Tokens.NextWithSkipComment;
       if (Tokens.Token^.TokenID = ID_Identifier) and not
         (UTF8CompareText(Tokens.Token^.TokenName, UName) = 0) then
       begin
         P.y := Tokens.Token^.Row + 1;
         P.x := Tokens.Token^.Col + 1;
         L := UTF8Length(Tokens.Token^.TokenName);
         P2 := P;
         P2.x := P2.x + L;
         P := PhysicalToLogicalPos(P);
         P2 := PhysicalToLogicalPos(P2);
         TextBetweenPoints[P, P2] := UName;
       end;
     end;
  finally
    CM.Free;
    Tokens.Free;
    Parser.Free;
  end;
end;

{TPageModule}
constructor TPageModule.Create(TheOwner: TComponent);
begin
  inherited Create(TheOwner);

  RootName:='';
  CodeLanguage := 'PASCAL';
  PageControl := TEditorForm(TheOwner).PageControl1;
  FSynEdit := TSuSynEdit.Create(Self, Self);
  FSynEdit.Font.Name:= EditorDefaultFontName;
  FSynEdit.Font.Size := EditorDefaultFontSize;
  FSynEdit.Parent := Self;
  FSynEdit.Align := alClient;
  FSynEdit.OnStatusChange:= @SynEditStatChange;
  FSynEdit.OnFindDeclaration := @FindDeclarationObject;
  FSynEdit.BookMarkOptions.BookmarkImages := TEditorForm(TheOwner).imgBookMarks;
  FSynEdit.MarkImages := TEditorForm(TheOwner).ImageList1;
  FSynEdit.Options2 := FSynEdit.Options2 + [eoFoldedCopyPaste];
  FSynEdit.OnPaste := @OnSynPaste;
  CompletionList := TStringList.Create;
  Completion := TSynCompletion.Create(Self);
  Completion.SelectedColor := 10841427;
  Completion.Editor := FSynEdit;
  Completion.TheForm.Font.Assign(FSynEdit.Font);
  Completion.Width := 500;
  Completion.OnExecute := @ExecuteCompletion;
  Completion.OnSearchPosition:=@CompletionSearchPosition;
  Completion.OnPaintItem := @PaintCompletionItem;
  Completion.OnCodeCompletion := @OnCompletion;
  Completion.OnKeyNextChar := @OnNextChar;
  Completion.OnKeyPrevChar := @OnPrevChar;
  Completion.OnUTF8KeyPress:= @ComplUTF8KeyPress;
  WordOfCurPos := TStringList.Create;
end;

destructor TPageModule.Destroy;
begin
  ClearCompletion;
  CompletionList.Free;
  WordOfCurPos.Free;
  FSynEdit.Free;
  inherited Destroy;
end;

procedure TPageModule.ClearCompletion;
var
  I: Integer;
begin
  Completion.ItemList.Clear;
  for I := 0 to CompletionList.Count - 1 do
  begin
    CompletionList.Objects[I].Free;
    CompletionList.Objects[I] := nil;
  end;
  CompletionList.Clear;
end;

procedure TPageModule.CaretPosInStatusBar;
begin
  EditorForm.StatusBar1.Panels[0].Text := '   ' + IntToStr(SynEdit.CaretY) + ' : ' + IntToStr(SynEdit.CaretX);
end;

procedure TPageModule.EditModeInStatusBar;
begin
  if FSynEdit.InsertMode then
    EditorForm.StatusBar1.Panels[1].Text := 'ВСТ'
  else
    EditorForm.StatusBar1.Panels[1].Text := 'ЗАМ';
end;

procedure TPageModule.CompletionSearchPosition(var Position: integer);
var
  I: Integer;

  procedure Add(AIndex: Integer);
  var
    S, CS: string;
    CI: TCompletionItem;
    //N: Integer;
  begin
    S := CompletionList.Strings[AIndex];
    CS := Completion.CurrentString;
    if CS <> '' then
    begin
      if CS[1] = '@' then Delete(CS, 1, 1);
    end;
    {N := Pos(CS, ',');
    if N > 0 then Delete(CS, N, 1);}
    if (CS = '') or
      (UTF8Pos(UTF8LowerCase(CS), UTF8LowerCase(s)) = 1) then
    begin
      CI := TCompletionItem(CompletionList.Objects[AIndex]);
      Completion.ItemList.AddObject(S, CI);
    end;
  end;

begin
  Completion.ItemList.Clear;
  for I := 0 to CompletionList.Count - 1 do
  begin
    Add(I);
  end;
  if Completion.ItemList.Count > 0 then
    Position := 0
  else
    Position := -1;
end;

procedure TPageModule.AddCompletionItem(AType: TCompletionItemType; AName, AParam,
  AResult: string);
var
  CI: TCompletionItem;
  S: string;
begin
  S := AltCurStr; //Trim(Completion.CurrentString);
  if (S = '') or
    (UTF8Pos(UTF8LowerCase(S), UTF8LowerCase(AName)) = 1) then
  begin
      CI := TCompletionItem.Create;
      CI.ItemType := AType;
      CI.ItemParam := AParam;
      CI.ItemResult := AResult;
      CompletionList.AddObject(AName, CI);
      Completion.ItemList.AddObject(AName, CI);
  end;
end;

procedure TPageModule.OnCompletion(var Value: string; SourceValue: string;
  var SourceStart, SourceEnd: TPoint; KeyChar: TUTF8Char; Shift: TShiftState);
var
  CI: TCompletionItem;
  S: string;
  P, P2: TPoint;
begin
  if Completion.ItemList.Count = 0 then Exit;
  CI := TCompletionItem(Completion.ItemList.Objects[Completion.Position]);
  if CI.ItemType in [citProc, citFunc] then
  begin
    if CI.ItemParam <> '' then
    begin
      P := SourceStart;
      Dec(P.x);
      P2 := SourceStart;
      S := SynEdit.TextBetweenPoints[P, P2];
      if S <> '@' then
      begin
        P2 := SourceEnd;
        Inc(P2.x);
        P := SourceEnd;
        S := SynEdit.TextBetweenPoints[P, P2];
        if S <> '(' then
        begin
          S := '(' + CI.ItemParam + ')';
          Value := Value + S;
        end;
      end;
    end;
  end
  else
  if CI.ItemType = citIndexProp then
  begin
    S := '[' + CI.ItemParam + ']';
    Value := Value + S;
  end;
  if (KeyChar = '.') or (KeyChar = ';') or (KeyChar = '(') or (KeyChar = ')') then
  begin
    Value := Value + KeyChar;
  end;
end;

procedure TPageModule.OnPrevChar(Sender: TObject);
var
  XY: TPoint;
  StartX: integer;
  EndX: integer;
  Line: string;
  I: Integer;
begin
   XY := SynEdit.LogicalCaretXY;
  if XY.Y > SynEdit.Lines.Count then exit;
  SynEdit.GetWordBoundsAtRowCol(XY, StartX, EndX);
  Line := SynEdit.GetWordAtRowCol(XY);
  Dec(XY.X, 1);
  if (EndX < XY.X) then exit;
  if XY.X = StartX then Line := ''
  else
  begin
    I := XY.X - StartX;
    Line:= UTF8Copy(Line, 1, I);
  end;
  SynEdit.LogicalCaretXY := XY;
  Completion.CurrentString := Line;
end;

procedure TPageModule.OnNextChar(Sender: TObject);
var
  XY: TPoint;
  StartX: integer;
  EndX: integer;
  I: Integer;
  Line: string;
begin
  XY := SynEdit.LogicalCaretXY;
  if XY.Y > SynEdit.Lines.Count then exit;
  SynEdit.GetWordBoundsAtRowCol(XY, StartX, EndX);
  Line := SynEdit.GetWordAtRowCol(XY);
  Inc(XY.X, 1);
  if (EndX < XY.X) then exit;
  if XY.X = StartX then Line := ''
  else
  begin
    I := XY.X - StartX;
    Line:= UTF8Copy(Line, 1, I);
  end;
  SynEdit.LogicalCaretXY := XY;
  Completion.CurrentString := Line;
end;

procedure TPageModule.ComplUTF8KeyPress(Sender: TObject; var UTF8Key: TUTF8Char
  );
var
  p: TPoint;
begin
  if UTF8Key = '.' then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit.InsertTextAtCaret(UTF8Key);
    UTF8Key:= '';
    Application.ProcessMessages;
    p := SynEdit.ClientToScreen(Point(SynEdit.CaretXPix, SynEdit.CaretYPix + SynEdit.LineHeight + 1));
    Completion.Execute('', P.x, P.y);
  end
  else
  if (UTF8Key = '=') or (UTF8Key = ':') or (UTF8Key = '+')
    or (UTF8Key = '-') or (UTF8Key = '*') or (UTF8Key = '/') then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit.InsertTextAtCaret(' ' + UTF8Key + ' ');
    UTF8Key:= '';
  end
  else
  if (UTF8Key = '(') or (UTF8Key = ')') or (UTF8Key = '[') or (UTF8Key = ']') then
  begin
    Completion.OnValidate(Completion.TheForm, '', []);
    SynEdit.InsertTextAtCaret(UTF8Key);
    UTF8Key:= '';
  end;
end;

function TPageModule.GetReadOnly: Boolean;
begin
  Result := ProjManager.ReadOnly;
  if not Result then Result := FReadOnly;
end;

procedure TPageModule.SetReadOnly(AValue: Boolean);
begin
  if ProjManager.ReadOnly then AValue := ReadOnly;

  if FReadOnly <> AValue then
  begin
    FReadOnly := AValue;
    FSynEdit.ReadOnly := FReadOnly;
    EditorForm.UpdateEditorFont;
  end;
end;

procedure TPageModule.ExecuteCompletion(Sender: TObject);
var
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
  CodeMap, ExtCodeMap: TSuCodeMap;
  SuObj: TSuObject;
  SuClass, InhClass: TSuObjectClass;
  SuType: TDpType;
  IO: TIdentObject;
  BaseProc, Proc, AProc, ExtProc: TSuProcItem;
  I, N, R: Integer;
  S, ParamList, ValType: string;
  CI: TCompletionItemType;
  ExtUses: TStringList;
  ExtList: TList;
  PI: PPropInfo;
  RoundOpen, CaretX, StartX, EndX: Integer;
  B, IsDecl: Boolean;

  procedure AddPublicProperty(AClass: TClass);
  var
    PropList: TPropList;
  begin
    if AClass <> nil then
    begin
      PropList := TPropList.Create(AClass, tkAny);
      try
        I := 0;
        while I < PropList.Count do
        begin
          CI := citProp;
          S := PropList.Items[I]^.Name;
          ValType :=  PropList.Items[I]^.PropType^.Name;
          AddCompletionItem(CI, S, '', ValType);
          Inc(I);
        end;
      finally
        PropList.Free;
      end;
    end;
  end;

  procedure AddItemFromSuClass;
  var
    M, F: Integer;
    InhClass: TSuObjectClass;
  begin
    for M := 0 to SuClass.Vars.Count -1 do
    begin
      SuObj := TSuObject(SuClass.Vars.Items[M]);
      CI := citVariable;
      if SuObj.ObjType = itClassComponent then CI := citObject;
      AddCompletionItem(CI, SuObj.OriginalName, '', SuObj.ValueType);
    end;

    for M := 0 to SuClass.Methods.Count - 1 do
    begin
      AProc := TSuProcItem(SuClass.Methods.Items[M]);
      ParamList := '';
      for F := 0 to AProc.Params.Count - 1 do
      begin
        if ParamList <> '' then ParamList := ParamList + ', ';
        SuObj := TSuObject(AProc.Params.Items[F]);
        ParamList := ParamList + SuObj.OriginalName + ': ' + SuObj.ValueType;
      end;
      AddCompletionItem(citProc, AProc.OriginalName, ParamList, AProc.ValueType);
    end;

    if SuClass.InhFrom <> '' then
    begin
      InhClass := BaseProc.FindClass(SuClass.InhFrom);
      if InhClass <> Nil then
      begin
        SuClass := InhClass;
        AddItemFromSuClass;
      end;
    end;
  end;

  procedure AddItemFromSuType;
  var
    Obj: TIdentObject;

    SuProc: TSuProc;
    M: Integer;
    RetValue: string;
    SaveSuType: TDpType;
  begin
    SaveSuType := SuType;
    while SuType <> nil do
    begin
      N := 0;

      while N < SuType.ClassObjects.Count do
      begin
        Obj := SuType.ClassObjects.Items[N];
        CI := citVariable;
        RetValue:= Obj.ValueType.OriginalName;;
        case Obj.IdentType of
          itConst: CI:= citConst;
          itVar: CI := citVariable;
          itParam: CI := citVariable;
          itProc:
          begin
            CI := citProc;
             if TSuProc(Obj).FIsFunction then
               CI := citFunc
             else RetValue := '';
          end;

          itType: CI := citType;
          itClassVar: CI := citVariable;
          itClassComponent: CI := citObject;
          itProp: CI := citProp;
        end;

        ParamList := '';
        if Obj is TClassIndexedProp then
        begin
          CI := citIndexProp;
          for M := 0 to TClassIndexedProp(Obj).Params.Count - 1 do
          begin
            if M > 0 then
              ParamList := ParamList + ', ';
            ParamList := ParamList + TDpType(TClassIndexedProp(Obj).Params.Items[M]).Name;
          end;
        end
        else
        if Obj.IdentType = itProc then
        begin
          SuProc := TSuProc(Obj);
          for M := 0 to SuProc.ParamCount - 1 do
          begin
            if ParamList <> '' then ParamList:= ParamList + ', ';
            ParamList := ParamList + SuProc.Param[M].OriginalName + ': ' + SuProc.Param[M].ValueType.OriginalName;
          end;
        end;
        AddCompletionItem(CI, Obj.OriginalName, ParamList, RetValue);
        Inc(N);
      end;
      SuType := SuType.Parent;
    end;
    if SaveSuType.ValueClassType <> nil then AddPublicProperty(SaveSuType.ValueClassType);
  end;

  procedure LoadItemFromSuProc(AProc: TSuProcItem);
  var M, F: Integer;
     LProc: TSuProcItem;
     AllowAddClassItem: Boolean;
  begin
    for M := 0 to AProc.Vars.Count -1 do
    begin
      SuObj := TSuObject(AProc.Vars.Items[M]);
      CI := citVariable;
      if SuObj.ObjType = itConst then CI := citConst;
      AddCompletionItem(CI, SuObj.OriginalName, '', SuObj.ValueType);
    end;

    for M := 0 to AProc.Procs.Count - 1 do
    begin
      LProc := TSuProcItem(AProc.Procs.Items[M]);
      ParamList := '';
      for F := 0 to LProc.Params.Count - 1 do
      begin
        if ParamList <> '' then ParamList := ParamList + ', ';
        SuObj := TSuObject(LProc.Params.Items[F]);
        ParamList := ParamList + SuObj.OriginalName + ': ' + SuObj.ValueType;
      end;
      AddCompletionItem(citProc, LProc.OriginalName, ParamList, LProc.ValueType);
    end;

    if AProc.ClassOwner <> nil then
    begin
      AllowAddClassItem := True;
      if (AProc.Proc <> nil) and (AProc.Proc.ClassOwner = AProc.ClassOwner) then AllowAddClassItem := False;
      if AllowAddClassItem then
      begin
        SuClass := Proc.ClassOwner;
        AddItemFromSuClass;
      end;
    end;

    for M := 0 to AProc.Classes.Count - 1 do
    begin
      SuClass := TSuObjectClass(AProc.Classes.Items[M]);
      AddCompletionItem(citType, SuClass.OriginalName, '', '');
    end;

    for M := 0 to AProc.Types.Count - 1 do
    begin
      SuObj := TSuObject(AProc.Types.Items[M]);
      AddCompletionItem(citType, SuObj.OriginalName, '', '');
    end;
  end;

begin
  ClearCompletion;
  WordOfCurPos.Clear;

  AltCurStr := '';

  Parser := TSuMemoParser.Create;
  Tokens := TMemoTokens.Create;
  CodeMap := TSuCodeMap.Create;
  ExtUses := TStringList.Create;
  ExtList := TList.Create;
  try
    Parser.SetParseStr(FSynEdit.Text, CodeLanguage);
    Parser.GetTokensList(Tokens);
    BaseProc := CodeMap.GetCodeMap(Tokens);
    BaseProc.GetUnitList(ExtUses);
    for N := 0 to ExtUses.Count - 1 do
    begin
      ExtCodeMap := EditorForm.GetUnitMap(ExtUses.Strings[N], ExtProc);
      ExtList.Add(ExtCodeMap);
    end;
    Tokens.First;
    while Tokens.Token^.TokenID <> ID_EOF do
    begin
      R := Tokens.Token^.Row + 1;
      if R = SynEdit.CaretY then
      begin
        StartX:= Tokens.Token^.Col + 1;
        EndX := Tokens.Token^.EndCol;
        CaretX := SynEdit.CaretX - 1;
        if (StartX <= CaretX)
          and (EndX >= CaretX) then
        begin
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            AltCurStr := UTF8Copy(Tokens.Token^.TokenName, 1, CaretX - StartX + 1);
            Tokens.PrevWithSkipComment;

          end;
          if Tokens.Token^.TokenID = ID_Period then
            Tokens.PrevWithSkipComment;
          if Tokens.Token^.TokenID = ID_CloseRound then
          begin
            RoundOpen := 1;
            while Tokens.Pos > 0 do
            begin
              Tokens.PrevWithSkipComment;
              if Tokens.Pos <= 0 then Break;
              if Tokens.Token^.TokenID = ID_CloseRound then Inc(RoundOpen)
              else
              if Tokens.Token^.TokenID = ID_OpenRound then
              begin
                Dec(RoundOpen);
                if RoundOpen = 0 then Break;
              end;
            end;
            if Tokens.Token^.TokenID = ID_OpenRound then Tokens.PrevWithSkipComment;
          end;
          if Tokens.Token^.TokenID = ID_CloseBlock then
          begin
            RoundOpen := 1;
            while Tokens.Pos > 0 do
            begin
              Tokens.PrevWithSkipComment;
              if Tokens.Pos <= 0 then Break;
              if Tokens.Token^.TokenID = ID_CloseBlock then Inc(RoundOpen)
              else
              if Tokens.Token^.TokenID = ID_OpenBlock then
              begin
                Dec(RoundOpen);
                if RoundOpen = 0 then Break;
              end;
            end;
            if Tokens.Token^.TokenID = ID_OpenBlock then Tokens.PrevWithSkipComment;
          end;

          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            WordOfCurPos.Add(Tokens.Token^.TokenName);
            Tokens.PrevWithSkipComment;
            if Tokens.Token^.TokenID = ID_Period then
            begin
              while Tokens.Token^.TokenID = ID_Period do
              begin
                Tokens.PrevWithSkipComment;
                if Tokens.Token^.TokenID = ID_Identifier then
                begin
                  WordOfCurPos.Insert(0, Tokens.Token^.TokenName);
                  Tokens.PrevWithSkipComment;
                end
                else
                if Tokens.Token^.TokenID = ID_CloseRound then
                begin
                  RoundOpen := 1;
                  while Tokens.Pos > 0 do
                  begin
                    Tokens.PrevWithSkipComment;
                    if Tokens.Pos <= 0 then Break;
                    if Tokens.Token^.TokenID = ID_CloseRound then Inc(RoundOpen)
                    else
                    if Tokens.Token^.TokenID = ID_OpenRound then
                    begin
                      Dec(RoundOpen);
                      if RoundOpen = 0 then Break;
                    end;
                  end;
                end
                else
                if Tokens.Token^.TokenID = ID_CloseBlock then
                begin
                  RoundOpen := 1;
                  while Tokens.Pos > 0 do
                  begin
                    Tokens.PrevWithSkipComment;
                    if Tokens.Pos <= 0 then Break;
                    if Tokens.Token^.TokenID = ID_CloseBlock then Inc(RoundOpen)
                    else
                    if Tokens.Token^.TokenID = ID_OpenBlock then
                    begin
                      Dec(RoundOpen);
                      if RoundOpen = 0 then Break;
                    end;
                  end;
                end
                else Break;
              end;
            end;
          end;
          Break;
        end;
      end
      else
      if Tokens.Token^.Row > SynEdit.CaretY then Break;
      Tokens.NextWithSkipComment;
    end;

    Proc := BaseProc.GetProcOfPos(FSynEdit.CaretX, FSynEdit.CaretY, IsDecl);
    if IsDecl then Proc := nil;
    I := 0;
    ValType := '';
    SuType := nil;
    SuClass := nil;
    while I < WordOfCurPos.Count do
    begin
      if I = 0 then
      begin
        SuClass := BaseProc.FindClass(WordOfCurPos.Strings[0]);
        if SuClass = nil then
        begin
          SuType := Compil.FindType(WordOfCurPos.Strings[0], nil, False);
          if SuType = nil then
          begin
            IO := Compil.ExternalObject.Find(WordOfCurPos.Strings[0], 0);
            if IO <> nil then
            begin
              SuType := IO.ValueType;
              B := True;
            end;
          end;
        end
        else
        if SuClass <> nil then
        begin
          SuType := Compil.FindType(SuClass.InhFrom, nil, False);
        end;

        if (SuClass = nil) and (SuType = nil) then
        begin
          SuObj := BaseProc.FindObject(WordOfCurPos.Strings[0]);
          if SuObj = nil then
          if (Proc <> nil) then
          begin
            SuObj := Proc.FindObject(WordOfCurPos.Strings[0]);
            if (SuObj = nil) and (Proc.Proc <> nil) and (Proc.Proc <> BaseProc) then
              SuObj := Proc.Proc.FindObject(WordOfCurPos.Strings[0]);
            if (SuObj = nil) and (Proc.ClassOwner <> nil) then
            begin
              SuObj := Proc.ClassOwner.Find(WordOfCurPos.Strings[0]);
              if SuObj = nil then
              begin
                if Proc.ClassOwner.InhFrom <> '' then
                begin
                  InhClass := BaseProc.FindClass(Proc.ClassOwner.InhFrom);
                  while InhClass <> nil do
                  begin
                    SuObj := InhClass.Find(WordOfCurPos.Strings[0]);
                    if SuObj <> nil then Break;
                    InhClass := BaseProc.FindClass(InhClass.InhFrom);
                  end;
                end;
              end;
            end;
          end;

          if SuObj = nil then
          begin
            N := 0;
            while N < ExtList.Count do
            begin
              ExtCodeMap := TSuCodeMap(ExtList.Items[N]);
              ExtProc := ExtCodeMap.MainProc;
              SuObj :=  ExtProc.FindObject(WordOfCurPos.Strings[0]);
              if SuObj = nil then
              begin
                SuClass := ExtProc.FindClass(WordOfCurPos.Strings[0]);
                if SuClass <> nil then
                   SuType := Compil.FindType(SuClass.InhFrom, nil, False);
              end;
              if (SuObj <> nil) or (SuClass <> nil) then
              begin
                BaseProc := ExtProc;
                Break;
              end;
              Inc(N);
            end;
          end;

          if (SuType = nil) and (SuClass = nil) then
          if SuObj <> nil then
          begin
            SuClass := BaseProc.FindClass(SuObj.ValueType);
            if SuClass <> nil then
            begin
              SuType := Compil.FindType(SuClass.InhFrom, nil, False);
            end
            else
            begin
              SuType := Compil.FindType(SuObj.ValueType, nil, False);
              if SuType = nil then
              begin
                //Поиск типа во внешнем модуле
                N := 0;
                while N < ExtList.Count do
                begin
                  ExtCodeMap := TSuCodeMap(ExtList.Items[N]);
                  ExtProc := ExtCodeMap.MainProc;
                  SuClass := ExtProc.FindClass(SuObj.ValueType);
                  if SuClass <> nil then
                        SuType := Compil.FindType(SuClass.InhFrom, nil, False);
                  if (SuClass <> nil) then
                  begin
                    BaseProc := ExtProc;
                    Break;
                  end;
                  Inc(N);
                end;
              end;
            end;
          end
          else
          begin
            if (Proc <> nil) and (Proc.ClassOwner <> nil) then
            begin
              S := WordOfCurPos.Strings[I];
              SuClass := BaseProc.FindClass(Proc.ClassOwner.ValueType);
              if SuClass <> nil then
              begin
                SuType := Compil.FindType(SuClass.InhFrom, nil, False);
                if SuType <> nil then
                begin
                  IO := SuType.FindClassObject(S, False);
                  if IO <> nil then
                    SuType := IO.ValueType
                  else
                  if SuType.ValueClassType <> nil then
                  begin
                    PI := GetPropInfo(SuType.ValueClassType, S);
                    if PI <> nil then
                    begin
                      S := PI^.PropType^.Name;
                      SuType := Compil.FindType(S, nil, False);
                      SuClass := nil;
                    end;
                  end
                  else
                  begin
                     SuType := nil;
                     SuClass := nil;
                  end;
                end;
              end;
            end;
          end;
        end;
      end
      else
      begin
        S := WordOfCurPos.Strings[I];
        B := False;
        if SuClass <> nil then
        begin
          SuObj := SuClass.Find(S);
          if SuObj <> nil then
          begin
            SuClass := BaseProc.FindClass(SuObj.ValueType);
            if SuClass = nil then
            begin
              SuType := Compil.FindType(SuObj.ValueType, nil, False);
              //B := True;
            end;
          end;
        end;

        if (SuType <> nil) and (not B) then
        begin
          IO := SuType.FindClassObject(S, False);
          if IO <> nil then
            SuType := IO.ValueType
          else
          begin
            if SuType.ValueClassType <> nil then
            begin
              PI := GetPropInfo(SuType.ValueClassType, S);
              if PI <> nil then
              begin
                S := PI^.PropType^.Name;
                SuType := Compil.FindType(S, nil, False);
              end;
            end
            else
             SuType := nil;
          end;
        end
        else Break;
      end;
      Inc(I);
    end;

    if SuType <> nil then AddItemFromSuType;
    if SuClass <> nil then AddItemFromSuClass;

    //----------------------------------

    if WordOfCurPos.Count = 0 then
    begin
       Proc := BaseProc.GetProcOfPos(FSynEdit.CaretX, FSynEdit.CaretY, IsDecl);
       if IsDecl then Proc := nil;
       if Proc <> nil then
       begin
         LoadItemFromSuProc(Proc);
         if Proc.ClassOwner <> nil then
         begin
           SuType := Compil.FindType(Proc.ClassOwner.InhFrom, nil, False);
           if SuType <> nil then AddItemFromSuType;
         end;

         if (Proc.Proc <> nil) and (Proc.Proc <> BaseProc) then
         begin
           Proc := Proc.Proc;
           while Proc <> nil do
           begin
             LoadItemFromSuProc(Proc);
             Proc := Proc.Proc;
             if (Proc = nil) or (Proc = BaseProc) then Break;
           end;
         end;
       end;

       LoadItemFromSuProc(BaseProc);
       for I := 0 to ExtList.Count - 1 do
       begin
         ExtCodeMap := TSuCodeMap(ExtList.Items[I]);
         if ExtCodeMap <> nil then
         begin
           ExtProc := ExtCodeMap.MainProc;
           LoadItemFromSuProc(ExtProc);
         end;
       end;
       GetDefaultContextList(@AddCompletionItem);
    end;
  finally
    for I := 0 to ExtList.Count - 1 do
    begin
      TSuCodeMap(ExtList.Items[I]).Free;
    end;
    Parser.Free;
    CodeMap.Free;
    Tokens.Free;
    ExtUses.Free;
    ExtList.Free;
  end;


end;

function TPageModule.PaintCompletionItem(const AKey: string; ACanvas: TCanvas;
  X, Y: integer; Selected: boolean; Index: integer): boolean;
var
  S: string;
  CI: TCompletionItem;
  L: Integer;
begin
  CI := TCompletionItem(Completion.ItemList.Objects[Index]);
  S := '';

  if Selected then
    ACanvas.Font.Color := clYellow
  else
    ACanvas.Font.Color := clNavy;

  case CI.ItemType of
    citProp:
    begin
      S := 'prop    ';
      if Selected then
        ACanvas.Font.Color := clFuchsia
      else
        ACanvas.Font.Color := clPurple;
    end;
    citIndexProp:
    begin
      S := 'indprop ';
      if Selected then
        ACanvas.Font.Color := clFuchsia
      else
        ACanvas.Font.Color := clPurple;
    end;
    citObject: S := 'object  ';
    citVariable:
    begin
      S := 'var     ';
      if Selected then
        ACanvas.Font.Color := clRed
      else
        ACanvas.Font.Color := clMaroon;
    end;
    citProc: S := 'proc    ';
    citFunc: S := 'func    ';
    citField: S := 'field   ';
    citKwd:
    begin
      S := 'keyword ';
      if Selected then
        ACanvas.Font.Color := clWhite
      else
        ACanvas.Font.Color := clBlack;
    end;
    citTable: S := 'table   ';
    citInParam:
    begin
      S := 'InParam ';
      ACanvas.Font.Color := clMaroon;
    end;
    citOutParam:
    begin
     S := 'OutPara ';
     ACanvas.Font.Color := clMaroon;
    end;
    citConst:
    begin
      S := 'const   ';
      if Selected then
        ACanvas.Font.Color := clYellow
      else
        ACanvas.Font.Color := clBlue;
    end;
    citType:
    begin
      S := 'type    ';
      if Selected then
        ACanvas.Font.Color := clLime
      else
        ACanvas.Font.Color := clGreen;
    end;
  end;
  ACanvas.Font.Bold := False;

  ACanvas.TextOut(X, Y, S);
  L := ACanvas.TextWidth(S);
  Inc(X, L);

  S := AKey;
  if Selected then
    ACanvas.Font.Color := clWhite
  else
    ACanvas.Font.Color := clBlack;
  ACanvas.Font.Bold := True;
  ACanvas.TextOut(X, Y, S);
  L := ACanvas.TextWidth(S);
  Inc(X, L);

  if CI.ItemParam <> '' then
  begin
    ACanvas.Font.Bold := False;
    if CI.ItemType = citIndexProp then
      S := '[' + CI.ItemParam + ']'
    else
      S := '(' + CI.ItemParam + ')';
    ACanvas.TextOut(X, Y, S);
    L := ACanvas.TextWidth(S);
    Inc(X, L);
  end;
  if CI.ItemResult <> '' then
  begin
    ACanvas.Font.Bold := False;
    S := ': ' + CI.ItemResult;
    ACanvas.TextOut(X, Y, S);
  end;
  Result := True;
end;

procedure TPageModule.SetDesignObject(AValue: TComponent);
begin
  FDesignObject := AValue;
  if AValue <> nil then RootName := AValue.Name;
end;

function TPageModule.GetModified: Boolean;
begin
  Result := False;
  Result := SynEdit.Modified;
  if not Result then
  begin
    if DesignForm <> nil then
    begin
      Result := TDesignerHook(DesignForm.Designer).IsModified;
    end;
  end;
end;

procedure TPageModule.SetModified(Value: Boolean);
begin
  SynEdit.Modified := Value;
  if DesignForm <> nil then
  begin
    if Value then
      TDesignerHook(DesignForm.Designer).Modified
    else
      TDesignerHook(DesignForm.Designer).ResetModified;
  end;
end;

procedure TPageModule.SynEditStatChange(Sender: TObject;
  Changes: TSynStatusChanges);
begin
  if PageControl.ActivePage = Self then
  begin
    if (scCaretX in Changes) or (scCaretY in Changes) then
      CaretPosInStatusBar
    else
    if (scInsertMode in Changes) then
      EditModeInStatusBar;
  end;
end;

procedure TPageModule.FindDeclarationObject(DeclUnit, ObjName: string; ACol,
  ARow: Integer);
var
  PM: TPageModule;
begin
  PM := nil;
  if DeclUnit <> '' then
    PM := ProjManager.OpenModuleByName(DeclUnit, False);

  if PM <> nil then
  begin
    PM.SynEdit.SetFocus;
    PM.SynEdit.CaretY:= ARow + 1;
    PM.SynEdit.CaretX:= ACol + 1;
    PM.SynEdit.CenterCaret;
  end;
end;

procedure TPageModule.OnSynPaste(Sender: TObject; var AText: String;
  var AMode: TSynSelectionMode; ALogStartPos: TPoint;
  var AnAction: TSynCopyPasteAction);
begin
  if AMode > smCurrent then
    AMode := smNormal;
  AnAction := scaContinue;
end;

{ TEditorForm }

procedure TEditorForm.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  CloseAction := caFree;
end;

procedure TEditorForm.FormCreate(Sender: TObject);
begin
  DoubleClicked := False;
  SynDPasSyn1 := TSynDPasSyn.Create(Self);
  SynRPasSyn := TSynRPasSyn.Create(Self);
  DisableSynDPasSyn := TSynDPasSyn.Create(Self);
  AssignDPascalHighlight;
end;

procedure TEditorForm.FormActivate(Sender: TObject);
begin
  MainIDEDesForm.LastActiveForm := Self;
end;

procedure TEditorForm.ActCloseModuleExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  CloseModule(PM);
end;

procedure TEditorForm.ActFindDeclarationExecute(Sender: TObject);
var
  S: string;
  PM: TPageModule;
begin
  PM := ActivePageModule;
  S := PM.SynEdit.GetWordAtRowCol(PM.SynEdit.CaretXY);
  if S <> '' then
    PM.SynEdit.FindDeclaration(S, PM.SynEdit.CaretX, PM.SynEdit.CaretY, True);
end;

procedure TEditorForm.ActBreakPointExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
    PM.SynEdit.SetDelBreakPoint;
end;

procedure TEditorForm.ActChangeReadOnlyExecute(Sender: TObject);

  function ModuleIsWriteable(ShowMode:Boolean):Boolean;
  var ExtPath,ExtFileName,CodeExt:String;
  SL: TStringList;
  I: Integer;
  begin
    Result:=false;
    SL := TStringList.Create;
    SL.Delimiter:= ';';
    SL.DelimitedText:= ProjManager.ProjectOption.ExternalPath;
    for I := 0 to SL.Count - 1 do
    begin
      ExtPath := Trim(ProjManager.ProjectOption.ExpandPath(SL.Strings[I]));
      if (ExtPath<>'') and (Pos('ftp://',ExtPath)<>1) and (Pos('http://',ExtPath)<>1) and (Pos('https://',ExtPath)<>1) then
      begin
        if SameText(ProjManager.ProjectOption.Language,ALterLanguage) then
           CodeExt:='.dp' else CodeExt := '.pas';
        ExtFileName:=AppendPathDelim(ExtPath)+ActivePageModule.Caption+CodeExt;
        Result := FileIsWritable(ExtFileName);
        if Result then Exit;
      end;
    end;


    //ExtPath:=Trim(ProjManager.ProjectOption.ExpandExternalPath); ExtFileName:='';
    //if (ExtPath<>'') and (Pos('ftp://',ExtPath)<>1) and (Pos('http://',ExtPath)<>1) and (Pos('https://',ExtPath)<>1) then begin
    //  if SameText(ProjManager.ProjectOption.Language,ALterLanguage) then CodeExt:='.dp' else CodeExt := '.pas';
    //  ExtFileName:=AppendPathDelim(ExtPath)+ActivePageModule.Caption+CodeExt;
    //  Result:=FileIsWritable(ExtFileName);
    //end;
    if not Result and ShowMode and (ExtFileName<>'')
    then ShowMessage('Файл '+ExtFileName+' защищен от записи.'+LineEnding+'Редактирование недоступно.');
  end;

begin
  // Check/confirm external modules to switch edit state.
  if ActivePageModule.ReadOnly and ActivePageModule.ExtModule then begin
    if not ModuleIsWriteable(true) then Exit;
    if MessageDlg('Предупреждение: внешний модуль',
                  'Модуль '+ActivePageModule.Caption+' является внешним (External).'+LineEnding
                 +'Его редактирование может повлиять на работу других программ DieselPascal.'+LineEnding
                 +'Вы уверены, что хотите перейти в режим редактирования?',
                  mtConfirmation, [mbYes, mbNo, mbCancel], '') <> mrYes then Exit;
  end;
  ActivePageModule.ReadOnly := not ActivePageModule.ReadOnly;
  if MainIDEDesForm.Inspector <> nil then
    MainIDEDesForm.Inspector.Enabled := ActivePageModule.ReadOnly = False;

end;

procedure TEditorForm.ActChangeReadOnlyUpdate(Sender: TObject);
begin
  if ActivePageModule.ReadOnly then TAction(Sender).Checked := True
  else TAction(Sender).Checked := False;
end;

procedure TEditorForm.ActBlockIndentExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
    PM.SynEdit.Block_Indent;
end;

procedure TEditorForm.ActBlockUnIndentExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
    PM.SynEdit.Block_UnIndent;
end;

procedure TEditorForm.ActFindDialogExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    FindDialog1.FindText := PM.SynEdit.GetWordAtRowCol(PM.SynEdit.CaretXY);
    //FindDialog1.Top := (Screen.Height - 100) div 2;
    FindDialog1.Execute;
  end;
end;

procedure TEditorForm.ActFindNextExecute(Sender: TObject);
begin
  if FindDialog1.FindText <> '' then
    FindDialog1.OnFind(FindDialog1);
end;

procedure TEditorForm.ActPasteExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if not PM.ReadOnly then
      PM.SynEdit.PasteFromClipboard;
  end;
end;

procedure TEditorForm.ActPasteUpdate(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if (PM <> nil) then
  begin
    if PM.SynEdit.CanPaste then
      TAction(Sender).Enabled := True
    else
      TAction(Sender).Enabled := False
  end;
end;

procedure TEditorForm.ActReplaceExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if not PM.ReadOnly then
    begin
      ReplaceDialog1.FindText := PM.SynEdit.GetWordAtRowCol(PM.SynEdit.CaretXY);
      ReplaceDialog1.Execute;
    end
    else
      ShowMessage('Module is ReadOnly');
  end;
end;

procedure TEditorForm.ActSelectCopyExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if PM.SynEdit.SelectionExist then
      PM.SynEdit.CopyToClipboard;
  end;
end;

procedure TEditorForm.ActSelectCopyUpdate(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if PM.SynEdit.SelectionExist and not PM.ReadOnly then
      TAction(Sender).Enabled := True
    else
      TAction(Sender).Enabled := False;
  end;
end;

procedure TEditorForm.ActSelectCutExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if PM.SynEdit.SelectionExist then
      PM.SynEdit.CutToClipboard;
  end;
end;

procedure TEditorForm.ActSetCommentExecute(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
      PM.SynEdit.DoSetComment;
end;

procedure TEditorForm.FindDialog1Find(Sender: TObject);
var
  PM: TPageModule;
  SynEdit: TSuSynEdit;
  S, S2, FindStr: string;
  I, L, EndPos, aStep: Integer;
  P1, P2: TPoint;

  procedure SelectFindText(AX, AY: Integer);
  begin
    P1.y := AY;
    P1.x := AX;
    P2.y := AY;
    P2.X := AX + UTF8Length(TFindDialog(Sender).FindText);
    SynEdit.Select(P1, P2);
    if aStep < 0 then
      SynEdit.CaretXY := P1
    else
      SynEdit.CaretXY := P2;
    SynEdit.CenterCaret;
  end;

  function ACompareText(AFindText: string): Boolean;
  begin
    Result := False;
    if frMatchCase in TFindDialog(Sender).Options then
      Result := FindStr = AFindText
    else
      Result := SameText(FindStr, AFindText);

  end;

begin
  FindStr := TFindDialog(Sender).FindText;
  if not (frMatchCase in TFindDialog(Sender).Options) then
    FindStr := UpperCase(FindStr);

  PM := ActivePageModule;
  SynEdit := PM.SynEdit;

  aStep := 1;
  EndPos:= SynEdit.Lines.Count - 1;

  if not (frDown in TFindDialog(Sender).Options) then
  begin
    aStep := - 1;
    EndPos := 1;
  end;
  I := SynEdit.CaretY - 1;

  while I <> EndPos do
  begin
    if not (frMatchCase in TFindDialog(Sender).Options) then
      S := UpperCase(SynEdit.Lines.Strings[I])
    else
      S := SynEdit.Lines.Strings[I];

    if aStep < 0 then
    begin
      if I = SynEdit.CaretY - 1 then
         S := UTF8Copy(S, 1, SynEdit.CaretX);
    end;

    if (I = SynEdit.CaretY - 1) and (aStep > 0) then
    begin
      L := UTF8Length(S);
      if L > SynEdit.CaretX then
      begin
        S := UTF8Copy(S, SynEdit.CaretX, L - SynEdit.CaretX);
        L := UTF8Pos(FindStr, S);
        if L > 0 then
        begin
          if frWholeWord in TFindDialog(Sender).Options then
          begin
            P1.X := SynEdit.CaretX + L;
            P1.Y := I + 1;
            S2 := SynEdit.GetWordAtRowCol(P1);
            if ACompareText(S2) then
            begin
              SelectFindText(SynEdit.CaretX + L - 1, I + 1);
              Break;
            end;
          end
          else
          begin
            SelectFindText(SynEdit.CaretX + L - 1, I + 1);
            Break;
          end;
        end;
      end;
    end
    else
    begin
      L := UTF8Pos(FindStr, S);
      if L > 0 then
      begin
        if frWholeWord in TFindDialog(Sender).Options then
        begin
          P1.X := L;
          P1.Y := I + 1;
          S2 := SynEdit.GetWordAtRowCol(P1);
          if ACompareText(S2) then
          begin
            SelectFindText(L, I + 1);
            Break;
          end;
        end
        else
        begin
          SelectFindText(L, I +1);
          Break;
        end;
      end;
    end;
    I := I + aStep;
  end;
  SynEdit.Lines;
end;

procedure TEditorForm.FindDialog1Show(Sender: TObject);
begin
  FindDialog1.Top := (Screen.Height - FindDialog1.Height) div 2;
end;

procedure TEditorForm.FormDestroy(Sender: TObject);
begin
  EditorForm := nil;
end;

procedure TEditorForm.FormShow(Sender: TObject);
begin
  ActCloseModule.Caption:= 'Закрыть';
  Left := MainIDEDesForm.EditorDefPos.Left;
  Top := MainIDEDesForm.EditorDefPos.Top;
  Height := MainIDEDesForm.EditorDefPos.Bottom;
  Width := MainIDEDesForm.EditorDefPos.Right;
end;

procedure TEditorForm.ListBox1DrawItem(Control: TWinControl; Index: Integer;
  ARect: TRect; State: TOwnerDrawState);
var
  S: string;
  X, Y, N: Integer;
begin
  ListBox1.Canvas.Font.Color := clBlack;
  ListBox1.Canvas.Brush.Color := clWhite;
  if odSelected in State then
  begin
    ListBox1.Canvas.Font.Color := clNavy;
    ListBox1.Canvas.Brush.Color := clSkyBlue;
  end;

  S := ListBox1.Items.Strings[Index];
  if Pos('[Подсказка]', S) = 1 then
    ListBox1.Canvas.Font.Color := clBlue
  else
  if Pos('[Предупреждение]', S) = 1 then
    ListBox1.Canvas.Font.Color := clMaroon
  else
  if Pos('[Ошибка]', S) = 1 then
    ListBox1.Canvas.Font.Color := clRed;

  N := ListBox1.Canvas.TextHeight('W');
  Y := (ARect.Bottom - ARect.Top - N) div 2;
  X := Y; if X < 2 then X := 2;
  ListBox1.Canvas.FillRect(ARect);
  ListBox1.Canvas.TextRect(ARect, ARect.Left + X, ARect.Top + Y, S);
end;

procedure TEditorForm.ListBox1KeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_RETURN then ParseMessage;
end;

procedure TEditorForm.ListBox1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if ssDouble in Shift then DoubleClicked := True;
end;

procedure TEditorForm.ListBox1MouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if DoubleClicked then
  begin
    DoubleClicked := False;
    ParseMessage;
  end;
end;

procedure TEditorForm.PageControl1Change(Sender: TObject);
begin
  TPageModule(PageControl1.ActivePage).CaretPosInStatusBar;
  TPageModule(PageControl1.ActivePage).EditModeInStatusBar;

  if MainIDEDesForm.Inspector <> nil then
    MainIDEDesForm.Inspector.Enabled := TPageModule(PageControl1.ActivePage).ReadOnly = False;

  if MainIDEDesForm.Process1.Running then
    HideExecMark;
end;

procedure TEditorForm.PageControl1MouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then PM.SynEdit.SetFocus;
end;

procedure TEditorForm.ReplaceDialog1Replace(Sender: TObject);
var
  PM: TPageModule;
begin
  PM := ActivePageModule;
  if PM <> nil then
  begin
    if frReplaceAll in ReplaceDialog1.Options then
    begin
      if not PM.SynEdit.SelectionExist then
        ReplaceDialog1.OnFind(ReplaceDialog1);
      while PM.SynEdit.SelectionExist do
      begin
        PM.SynEdit.ReplaceSelect(TReplaceDialog(Sender).ReplaceText);
        ReplaceDialog1.OnFind(ReplaceDialog1);
      end;
    end
    else
      PM.SynEdit.ReplaceSelect(TReplaceDialog(Sender).ReplaceText);
  end;
end;

procedure TEditorForm.ReplaceDialog1Show(Sender: TObject);
begin
  ReplaceDialog1.Top := (Screen.Height - ReplaceDialog1.Height) div 2;
end;

function TEditorForm.NewModule(DesignObj: TComponent; DesignForm: TCustomForm;
  ACaption, ACode, ALang: string): TPageModule;
begin
  Result := TPageModule.Create(Self);
  Result.CodeLanguage := ALang;
  Result.DesignObject := DesignObj;
  Result.DesignForm := DesignForm;
  Result.Caption := ACaption;
  Result.SynEdit.PopupMenu := EditPopup;
  if SameText(ALang, 'PASCAL') then
    Result.SynEdit.Highlighter := SynRPasSyn //MainIDEDesForm.SynPasSyn1
  else
    Result.SynEdit.Highlighter := SynDPasSyn1;
  Result.SynEdit.Text := ACode;
  //Result.Modified := True;
  PageControl1.ActivePage := Result;
  ActivePageModule.EditModeInStatusBar;
  ActivePageModule.CaretPosInStatusBar;
end;

procedure TEditorForm.WriteDebugMessage(S: string);
begin
  ListBox1.Items.Add(S);
  ListBox1.ItemIndex:= ListBox1.Count - 1;
end;

procedure TEditorForm.GoDebugPos(S: string);
var
  I, L: Integer;
  AModule, APos, PosX, PosY: string;
  PM: TPageModule;
  P: TPoint;
begin
  I := LastDelimiter('=', S);
  L := Length(S);
  if I > 0 then
  begin
    AModule := Copy(S, 1, I - 1);
    APos := Copy(S, I + 1, L - I);
    I := LastDelimiter(',', APos);
    L := Length(APos);
    if I > 0 then
    begin
      PosX := Copy(APos, 1, I - 1);
      PosY := Copy(APos, I + 1, L - I);
      P.x := StrToInt(PosX);
      P.y := StrToInt(PosY);
      PM := ProjManager.OpenModuleByName(AModule, False);
      if PM <> nil then
      begin
        PageControl1.ActivePage := PM;
        PM.SynEdit.SetExecutePos(P);
      end;
    end;

  end;
end;

procedure TEditorForm.GoModulePos(AModule: string; ARow, ACol: Integer);
var
  P: TPoint;
  PM: TPageModule;
begin
  PM := ProjManager.OpenModuleByName(AModule, False);
  if PM <> nil then
  begin
    P.x:= ACol;
    P.y:= ARow;
    PageControl1.ActivePage := PM;
    PM.SynEdit.CaretXY := P;
    PM.SynEdit.SetFocus;
    PM.SynEdit.CenterCaret;
  end;
end;

procedure TEditorForm.UpdateEditorFont;
var
  I: Integer;
begin
  for I := 0 to ModuleCount - 1 do
  begin
    Modules[I].SynEdit.Font.Name := EditorDefaultFontName;
    Modules[I].SynEdit.Font.Size := EditorDefaultFontSize;
    Modules[I].SynEdit.KeyMarkUp.MarkupInfo.Foreground := MainIDEDesForm.SynPasSyn1.KeyAttri.Foreground;
    if Modules[I].ReadOnly then
    begin
      Modules[I].FSynEdit.Color := ReadOnlyEditorColor;

      if ReadOnlyGrayFont then
      begin
        Modules[I].FSynEdit.KeyMarkUp.MarkupInfo.Foreground := clBlack;
        if SameText(ProjManager.ProjectOption.Language, 'PASCAL') then
          Modules[I].FSynEdit.Highlighter := MainIDEDesForm.SynPasSyn2
        else
          Modules[I].FSynEdit.Highlighter := EditorForm.DisableSynDPasSyn;
      end
      else
      begin
        if SameText(ProjManager.ProjectOption.Language, 'PASCAL') then
        Modules[I].FSynEdit.Highlighter := EditorForm.SynRPasSyn
      else
        Modules[I].FSynEdit.Highlighter := EditorForm.SynDPasSyn1;
      end;
    end
    else
    begin
      Modules[I].FSynEdit.Color := clWhite;
      Modules[I].FSynEdit.KeyMarkUp.MarkupInfo.Foreground := MainIDEDesForm.SynPasSyn1.KeyAttri.Foreground;

      if SameText(ProjManager.ProjectOption.Language, 'PASCAL') then
        Modules[I].FSynEdit.Highlighter := EditorForm.SynRPasSyn
      else
        Modules[I].FSynEdit.Highlighter := EditorForm.SynDPasSyn1;
    end;
  end;
end;

function TEditorForm.GetPageModuleOfRoot(ARoot: TComponent): TPageModule;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to PageControl1.PageCount - 1 do
  begin
    if TPageModule(PageControl1.Pages[I]).DesignObject = ARoot then
    begin
      Result := TPageModule(PageControl1.Pages[I]);
      Break;
    end;
  end;
end;

function TEditorForm.GetPageModule(Index: Integer): TPageModule;
begin
  Result := TPageModule(PageControl1.Pages[Index]);
end;

procedure TEditorForm.ParseMessage;
var
  S, AModule, ColRowStr: string;
  I, L, StartI, ARow, ACol: Integer;
  B: Boolean;
  P, PStart: PChar;
  C: Char;

  function GetRowCol(out AR, AC: Integer): Boolean;
  var
    N: Integer;
    ColStr: string;
  begin
    Result := False;
    N := LastDelimiter(',', ColRowStr);
    if N > 0 then
    begin
      ColStr := UTF8Copy(ColRowStr, 1, N - 1);
      AR := StrToInt(ColStr);
      Delete(ColRowStr, 1, N);
      AC := StrToInt(ColRowStr);
      Result := True;
      //AR := IntToStr();
    end;
  end;
begin
  if ListBox1.ItemIndex < 0 then Exit;
  S := ListBox1.Items.Strings[ListBox1.ItemIndex];
  if S = '' then Exit;
  L := Length(S);
  if L = 0 then Exit;
  Inc(L);
  SetLength(S, L);
  S[L] := #0;
  P := @S[1];
  if P^ <> '[' then Exit;
  B := False;
  I := UTF8CharacterLength(p);
  Inc(P, I);
  while P^ <> #0 do
  begin
    if P^ = ']' then
    begin
     B := True;
     Break;
    end;
    I := UTF8CharacterLength(p);
    Inc(P, I);
  end;
  if not B then Exit;
  Inc(P, 2);
  PStart:= P;
  B := False;
  while P <> #0 do
  begin
    if P^ = '(' then
    begin
      B := True;
      Break;
    end;
    I := UTF8CharacterLength(p);
    Inc(P, I);
  end;
  if B then
  begin
    B := False;
    SetString(AModule, PStart, P - PStart);
    I := UTF8CharacterLength(p);
    Inc(P, I);
    PStart := P;
    while P <> #0 do
    begin
      if P^ = ')' then
      begin
        B := True;
        Break;
      end;
      I := UTF8CharacterLength(p);
      Inc(P, I);
    end;
    if B then
    begin
      SetString(ColRowStr, PStart, P - PStart);
      if ColRowStr <> '' then
      begin
        if GetRowCol(ARow, ACol) then
        begin
          GoModulePos(AModule, ARow, ACol);
        end;
      end;
    end;
  end;
end;

procedure TEditorForm.AssignDPascalHighlight;
begin
  SynDPasSyn1.Assign(MainIDEDesForm.SynPasSyn1);
  SynRPasSyn.Assign(MainIDEDesForm.SynPasSyn1);
  DisableSynDPasSyn.Assign(MainIDEDesForm.SynPasSyn2);
end;

procedure TEditorForm.HideExecMark;
var
  I: Integer;
begin
  for I := 0 to ModuleCount - 1 do
  begin
    if Modules[I].SynEdit.ExecutMark <> nil then
      Modules[I].SynEdit.ExecutMark.Visible := False;
  end;
end;

function TEditorForm.GetUnitMap(AUnitName: string; var BaseProc: TSuProcItem
  ): TSuCodeMap;
var
  I: Integer;
  Code, CodeLanguage: string;
  Tokens: TMemoTokens;
  Parser: TSuMemoParser;
begin
  Result := nil;
  CodeLanguage := ProjManager.ProjectOption.Language;
  for I := 0 to ModuleCount - 1 do
  begin
    if Modules[I].Caption = AUnitName then
    begin
      Code := Modules[I].SynEdit.Text;
      CodeLanguage := Modules[I].CodeLanguage;
      Break;
    end;
  end;
  if Code = '' then
    Code := ProjManager.GetModuleCode(AUnitName);

  if Code <> '' then
  begin
    Parser := TSuMemoParser.Create;
    Tokens := TMemoTokens.Create;
    Result := TSuCodeMap.Create;
    try
      Parser.SetParseStr(Code, CodeLanguage);
      Parser.GetTokensList(Tokens);
      BaseProc := Result.GetCodeMap(Tokens);
    finally
      Parser.Free;
      Tokens.Free;
    end;

  end;
end;

procedure TEditorForm.ActivateByRoot(ARoot: TComponent);
var
  PM: TPageModule;
begin
  PM := GetPageModuleOfRoot(ARoot);
  if PM <> nil then  PageControl1.ActivePage := PM;
end;

function TEditorForm.ActivePageModule: TPageModule;
begin
  if PageControl1.PageCount = 0 then
    Result := nil
  else
    Result := TPageModule(PageControl1.ActivePage);
end;

function TEditorForm.ModuleCount: Integer;
begin
  Result := PageControl1.PageCount;
end;

function TEditorForm.GetModuleByName(AUnitName: string): TPageModule;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to ModuleCount - 1 do
  begin
    if SameText(Modules[I].Caption, AUnitName) then
    begin
      Result := Modules[I];
      Break;
    end;
  end;
end;

procedure TEditorForm.CloseAll;
var
  I: Integer;
begin
  I := ModuleCount - 1;
  while I >= 0 do
  begin
    if not CloseModule(Modules[I]) then Break;
    Dec(I);
    if EditorForm = nil then Exit;
  end;
end;

function TEditorForm.CloseModule(PM: TPageModule): Boolean;
var
  I: Integer;
  Node: TTreeNode;
begin
  Result := False;
  if PM.Modified then
  begin

    I := Application.MessageBox(PChar('Сохранить ' + PM.Caption + '?'), PChar(PM.Caption), MB_YESNOCANCEL or MB_ICONQUESTION);
    case I of
      IDYES: ProjManager.SaveModule(PM);
      IDCANCEL: Exit;
    end;
  end;

  if MainIDEDesForm.LastActiveForm = PM.DesignForm then
    MainIDEDesForm.LastActiveForm := Self;

  PM.FSynEdit.ExecutMark.Visible := False;
  PM.FSynEdit.Marks.Delete(PM.FSynEdit.ExMarkIndex);
  PM.FSynEdit.ExecutMark.Free;
  PM.FSynEdit.ExecutMark := nil;

  Node := ProjManager.GetModuleNode(PM.Caption);
  if (PM.DesignObject is TDataModule) and (Node <> nil) then
  begin
    TDesignerHook(PM.DesignForm.Designer).DesignClose;
    PM.DesignForm.Visible := True;
    PM.DesignForm.Hide;
  end
  else
  begin
    if Node <> nil then PNodeObject(Node.Data)^.ADesignHook := nil;
    MainIDEDesForm.DManager.DeleteAndFreeByRoot(PM.DesignObject);
  end;
  PM.Visible:= False;
  PM.Free;
  Result := True;
  if ModuleCount = 0 then Free;
end;

{procedure TEditorForm.InsertClassComponent(ARoot, AComponent: TComponent);
var
  PM: TPageModule;
begin
  PM := GetPageModuleOfRoot(ARoot);
  if PM <> nil then
  begin
  end;
end;}

function TEditorForm.Modified: Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to ModuleCount - 1 do
  begin
    if Modules[I].Modified then
    begin
      Result := True;
      Break;
    end;
  end;
end;

initialization

ReadOnlyEditorColor := clWhite;
ReadOnlyGrayFont := True;

//TSynValidStringChars

end.

