#!/bin/bash

###########################################################
## Copyright (c) 2002-2025 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Diesel Pascal wrapper tool.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

function print_version(){
 echo "$scriptname version 1.0";
};

function print_help(){
 print_version;
 echo "Copyright (c) 2002-2025 Alexey Kuryakin daqgroup@mail.ru";
 echo "Utility call/configure Diesel Pascal interpreter/designer.";
 echo "Usage: $scriptname [option] [script] [arguments]";
 echo "Options:";
 echo " --version    - print version";
 echo " -h,--help    - print help screen";
 echo " -r,--run     - run script with CrossMachine (default)";
 echo " -d,--design  - run CrossDesigner IDE to edit script";
 echo " -a,--assoc   - associate *.lm9 with CrossMachine";
 echo " -q,--query   - query MIME type application/x-diesel";
 echo " -u,--update  - update database of MIME associations";
 echo "Script:";
 echo " [script] is *.lm9 file with Diesel Pascal code.";
 echo "Arguments:";
 echo " [arguments] is any command line arguments for script.";
};

function get_mime_assoc(){
 gio mime application/x-diesel;
};

function set_mime_assoc(){
 unix sudoit gio mime application/x-diesel CrossMachine.desktop;
         gio mime application/x-diesel CrossMachine.desktop;
};

function update_mime_assoc(){
 echo "$(langstr ru "Выполняется" en "Executing") update-mime-database …";
 unix sudoit update-mime-database /usr/share/mime;
 set_mime_assoc;
 echo "$(langstr ru "Выполняется" en "Executing") update-desktop-database …";
 unix sudoit update-desktop-database;
 echo "$(langstr ru "Выполняется" en "Executing") update-menus …";
 unix sudoit update-menus;
 get_mime_assoc;
};

function error_no_arguments(){
 local ru="Ошибка: не заданы аргументы. Смотрите $scriptname -h для справки.";
 local en="Error: no arguments. Call $scriptname -h for help.";
 fatal 1 "$(langstr ru "$ru" en "$en")";
};

function error_bad_arguments(){
 local ru="Ошибка: неверные аргументы ($*). Смотрите $scriptname -h для справки.";
 local en="Error: invalid arguments ($*). Call $scriptname -h for help.";
 fatal 1 "$(langstr ru "$ru" en "$en")";
};

function error_not_installed(){
 local ru="Ошибка: Diesel Pascal не установлен.";
 local en="Error: Diesel Pascal is not installed.";
 fatal 1 "$(langstr ru "$ru" en "$en")";
};

function error_no_script(){
 local ru="Ошибка: не задан сценарий для выполнения.";
 local en="Error: no script to run.";
 fatal 1 "$(langstr ru "$ru" en "$en")";
};

function error_bad_script(){
 local ru="Ошибка: $1 - это не *.lm9 сценарий.";
 local en="Error: $1 is not *.lm9 script.";
 fatal 1 "$(langstr ru "$ru" en "$en")";
};

function run_cross_machine(){
 case $1 in
  '')    error_no_script; ;;
  *.lm9) unix grun CrossMachine "$@"; ;;
  *)     error_bad_script "$@"; ;;
 esac;
};

function run_cross_designer(){
 case $1 in
  '')    unix grun CrossDesigner "$@"; ;;
  *.lm9) unix grun CrossDesigner "$@"; ;;
  *)     error_bad_script "$@"; ;;
 esac;
};

function diesel_pascal(){
 case $1 in
 '')                   error_no_arguments; ;;
 --version)            print_version; return 0; ;;
  -h|-help|--help)     print_help; return 0; ;;
  -r|-run|--run)       shift; run_cross_machine "$@"; ;;
  -d|-design|--design) shift; run_cross_designer "$@"; ;;
  -a|-assoc|--assoc)   set_mime_assoc; ;;
  -q|-query|--query)   get_mime_assoc; ;;
  -u|-update|--update) update_mime_assoc; ;;
  *.lm9)               run_cross_machine "$@"; ;;
  *)                   error_bad_arguments "$@"; ;;
 esac;
};

function main(){
 if which CrossMachine >/dev/null 2>&1; then
  diesel_pascal "$@";
 else
  error_not_installed;
 fi;
};

main "$@";

#############
# END OF FILE
#############
