////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// RTD and ThermoCouple calculation unit: convert mV to °C and vise versa.    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20161016 - Creation & test                                                 //
// 20161021 - modify *_ttc to have better performance & robustness            //
// 20230529 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_rtdtc; // RTD & TC = resistive thermodetectors & thermocouples.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 _crw_alloc, _crw_fifo, _crw_str, _crw_fio, _crw_fpu, _crw_zm;
 
const
 // thermocouple_x  = Ord('x'),  x=R,S,B,J,T,E,K,N,L,M
 // thermocouple_Ax = Ord('A')*256+Ord('x'),   x=1,2,3
 thermocouple_R   = 82;      { Thermocouple type R   }
 thermocouple_S   = 83;      { Thermocouple type S   }
 thermocouple_B   = 66;      { Thermocouple type B   }
 thermocouple_J   = 74;      { Thermocouple type J   }
 thermocouple_T   = 84;      { Thermocouple type T   }
 thermocouple_E   = 69;      { Thermocouple type E   }
 thermocouple_K   = 75;      { Thermocouple type K   }
 thermocouple_N   = 78;      { Thermocouple type N   }
 thermocouple_A1  = 16689;   { Thermocouple type A1  }
 thermocouple_A2  = 16690;   { Thermocouple type A2  }
 thermocouple_A3  = 16691;   { Thermocouple type A3  }
 thermocouple_L   = 76;      { Thermocouple type L   }
 thermocouple_M   = 77;      { Thermocouple type M   }
 rtd_id_pt100_385 = 385;     { RTD Pt100 a=0.00385   }
 rtd_id_pt100_391 = 391;     { RTD Pt100 a=0.00391   }
 rtd_id_cu100_428 = 428;     { RTD Cu100 a=0.00428   }
 rtd_id_cu100_426 = 426;     { RTD Cu100 a=0.00426   }
 rtd_id_ni100_617 = 617;     { RTD Ni100 a=0.00617   }
 
type Real = Double;

var
 thermocouple_cnt : Real;    { TC  call counter      }
 thermocouple_eps : Real;    { TC  calc epsilon      }
 thermocouple_tol : Real;    { TC  calc tolerance    }
 thermocouple_big : Real;    { TC  big range value   }
 thermocouple_del : Real;    { TC  line ext. delta   }
 thermocouple_fac : Real;    { TC  srch grow factor  }
 thermocouple_met : Integer; { TC  calc method       }
 thermocouple_mit : Integer; { TC  max.iterations    }
 restemperdet_cnt : Real;    { RTD call counter      }
 restemperdet_eps : Real;    { RTD calc epsilon      }
 restemperdet_tol : Real;    { RTD calc tolerance    }
 restemperdet_big : Real;    { RTD big range value   }
 restemperdet_del : Real;    { RTD line ext. delta   }
 restemperdet_fac : Real;    { RTD srch grow factor  }
 restemperdet_met : Integer; { RTD calc method       }
 restemperdet_mit : Integer; { RTD max.iterations    }

function thermocouple_emf(t:Real; idn:Integer):Real;
function thermocouple_lot(idn:Integer):Real;
function thermocouple_hit(idn:Integer):Real;
function thermocouple_idn(ids:LongString):Integer;
function thermocouple_ids(idn:Integer):LongString;
function thermocouple_ttc(emf:Real; idn:Integer):Real;
function thermocouple_tst:Integer;

function restemperdet_ohm(t:Real; idn:Integer):Real;
function restemperdet_lot(idn:Integer):Real;
function restemperdet_hit(idn:Integer):Real;
function restemperdet_idn(ids:LongString):Integer;
function restemperdet_ids(idn:Integer):LongString;
function restemperdet_ttc(ohm:Real; idn:Integer):Real;
function restemperdet_tst:Integer;

implementation

function StrFix(x:Real;w,d:Integer):LongString;
begin
 Result:=Format('%*.*f',[w,d,x]);
end;

//////////////////////////////////////////////////////////////////////////////
// RTD and ThermoCouple calculation unit: convert mV to °C and vise versa.
//////////////////////////////////////////////////////////////////////////////
// Reference list:
// 1) GOST R 8.585-2001                              - thermocouple tables
// 2) http://srdata.nist.gov/its90/download/all.tab  - thermocouple tables
// 3) GOST R 50431-92                                - thermocouple tables
// 4) GOST 6651-2009                                 - RTD tables+formulas
// 5) GOST R 8.625-2006                              - RTD tables+formulas
// 6) GOST 6651-94                                   - RTD tables+formulas
// Note: GOST R 8.585-2001, GOST 6651-2009 are most actual, others is obsolete
//////////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_emf(t:Real; idn:Integer):Real;
// Return:
//  Thermocouple EMF (electromotive force), mV. Return NaN on invalid idn.
// Arguments:
//  t   - temperature, °C. Cold junction is at 0°C.
//  idn - thermocouple type identifier number:
//   R  = $0052 =    82 = Ord('R')
//   S  = $0053 =    83 = Ord('S')
//   B  = $0042 =    66 = Ord('B')
//   J  = $004A =    74 = Ord('J')
//   T  = $0054 =    84 = Ord('T')
//   E  = $0045 =    69 = Ord('E')
//   K  = $004B =    75 = Ord('K')
//   N  = $004E =    78 = Ord('N')
//   A1 = $4131 = 16689 = Ord('A')*256+Ord('1')
//   A2 = $4132 = 16690 = Ord('A')*256+Ord('2')
//   A3 = $4133 = 16691 = Ord('A')*256+Ord('3')
//   L  = $004C =    76 = Ord('L')
//   M  = $004D =    77 = Ord('M')
//////////////////////////////////////////////////////////////////////////////
function thermocouple_emf(t:Real; idn:Integer):Real;
const correct_L:Boolean=false; // Apply GOST corrections?
var emf:Real; absq:Boolean;
 // pn(t,L,H,..) define a polynom of n-th degree
 // use linear extrapolation outside (L,H) range
 // Ci coefficients taken from NIST, GOST tables
 function p3(t,L,H,c0,c1,c2,c3:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c3;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=3*c3;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p3:=f;
 end;
 function p4(t,L,H,c0,c1,c2,c3,c4:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c4;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=4*c4;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p4:=f;
 end;
 function p5(t,L,H,c0,c1,c2,c3,c4,c5:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c5;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=5*c5;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p5:=f;
 end;
 function p6(t,L,H,c0,c1,c2,c3,c4,c5,c6:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c6;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=6*c6;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p6:=f;
 end;
 function p8(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c8;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=8*c8;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p8:=f;
 end;
 function p9(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8,c9:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c9;f:=c8+f*x;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=9*c9;q:=8*c8+q*x;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p9:=f;
 end;
 function p9e(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8,c9,a0,a1,a2:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c9;f:=c8+f*x;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  f:=f+a0*exp(a1*sqr(x-a2));
  if s<>0 then begin // linear extrapolation
   q:=9*c9;q:=8*c8+q*x;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   q:=q+a0*exp(a1*sqr(x-a2))*a1*2*(x-a2);
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p9e:=f;
 end;
 function p10(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8,c9,c10:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c10;f:=c9+f*x;f:=c8+f*x;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=10*c10;q:=9*c9+q*x;q:=8*c8+q*x;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p10:=f;
 end;
 function p13(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8,c9,c10,c11,c12,c13:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c13;f:=c12+f*x;f:=c11+f*x;f:=c10+f*x;
  f:=c9+f*x;f:=c8+f*x;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=13*c13;q:=12*c12+q*x;q:=11*c11+q*x;q:=10*c10+q*x;
   q:=9*c9+q*x;q:=8*c8+q*x;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p13:=f;
 end;
 function p14(t,L,H,c0,c1,c2,c3,c4,c5,c6,c7,c8,c9,c10,c11,c12,c13,c14:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(H,t)); s:=t-x;
  f:=c14;f:=c13+f*x;f:=c12+f*x;f:=c11+f*x;f:=c10+f*x;
  f:=c9+f*x;f:=c8+f*x;f:=c7+f*x;f:=c6+f*x;f:=c5+f*x;f:=c4+f*x;f:=c3+f*x;f:=c2+f*x;f:=c1+f*x;f:=c0+f*x;
  if s<>0 then begin // linear extrapolation
   q:=14*c14;q:=13*c13+q*x;q:=12*c12+q*x;q:=11*c11+q*x;q:=10*c10+q*x;
   q:=9*c9+q*x;q:=8*c8+q*x;q:=7*c7+q*x;q:=6*c6+q*x;q:=5*c5+q*x;q:=4*c4+q*x;q:=3*c3+q*x;q:=2*c2+q*x;q:=c1+q*x;
   if absq then q:=abs(q);
   f:=f+q*s;
  end;
  p14:=f;
 end;
begin
 absq:=False;
 if (idn=Ord('B')) then begin
  absq:=(t<=0); // Special case because B thermocouple is not monotonous near 0
  if t < 630.615 then
  emf:=p6(t, 0.000, 630.615,
   0.000000000000E+00,
  -0.246508183460E-03,
   0.590404211710E-05,
  -0.132579316360E-08,
   0.156682919010E-11,
  -0.169445292400E-14,
   0.629903470940E-18)
  else
  emf:=p8(t, 630.615, 1820.000,
  -0.389381686210E+01,
   0.285717474700E-01,
  -0.848851047850E-04,
   0.157852801640E-06,
  -0.168353448640E-09,
   0.111097940130E-12,
  -0.445154310330E-16,
   0.989756408210E-20,
  -0.937913302890E-24);
 end else
 if (idn=Ord('E')) then begin
  if t < 0.000 then
  emf:=p13(t, -270.000, 0.000,
   0.000000000000E+00,
   0.586655087080E-01,
   0.454109771240E-04,
  -0.779980486860E-06,
  -0.258001608430E-07,
  -0.594525830570E-09,
  -0.932140586670E-11,
  -0.102876055340E-12,
  -0.803701236210E-15,
  -0.439794973910E-17,
  -0.164147763550E-19,
  -0.396736195160E-22,
  -0.558273287210E-25,
  -0.346578420130E-28)
  else
  emf:=p10(t, 0.000, 1000.000,
   0.000000000000E+00,
   0.586655087100E-01,
   0.450322755820E-04,
   0.289084072120E-07,
  -0.330568966520E-09,
   0.650244032700E-12,
  -0.191974955040E-15,
  -0.125366004970E-17,
   0.214892175690E-20,
  -0.143880417820E-23,
   0.359608994810E-27);
 end else
 if (idn=Ord('J')) then begin
  if t < 760.000 then
  emf:=p8(t, -210.000, 760.000,
   0.000000000000E+00,
   0.503811878150E-01,
   0.304758369300E-04,
  -0.856810657200E-07,
   0.132281952950E-09,
  -0.170529583370E-12,
   0.209480906970E-15,
  -0.125383953360E-18,
   0.156317256970E-22)
  else
  emf:=p5(t, 760.000, 1200.000,
   0.296456256810E+03,
  -0.149761277860E+01,
   0.317871039240E-02,
  -0.318476867010E-05,
   0.157208190040E-08,
  -0.306913690560E-12);
 end else
 if (idn=Ord('K')) then begin
  if t < 0.000 then
  emf:=p10(t, -270.000, 0.000,
   0.000000000000E+00,
   0.394501280250E-01,
   0.236223735980E-04,
  -0.328589067840E-06,
  -0.499048287770E-08,
  -0.675090591730E-10,
  -0.574103274280E-12,
  -0.310888728940E-14,
  -0.104516093650E-16,
  -0.198892668780E-19,
  -0.163226974860E-22)
  else
  emf:=p9e(t, 0.000, 1372.000,
  -0.176004136860E-01,
   0.389212049750E-01,
   0.185587700320E-04,
  -0.994575928740E-07,
   0.318409457190E-09,
  -0.560728448890E-12,
   0.560750590590E-15,
  -0.320207200030E-18,
   0.971511471520E-22,
  -0.121047212750E-25,
   0.118597600000E+00,
  -0.118343200000E-03,
   0.126968600000E+03);
 end else
 if (idn=Ord('N')) then begin
  if t < 0.000 then
  emf:=p8(t, -270.000, 0.000,
   0.000000000000E+00,
   0.261591059620E-01,
   0.109574842280E-04,
  -0.938411115540E-07,
  -0.464120397590E-10,
  -0.263033577160E-11,
  -0.226534380030E-13,
  -0.760893007910E-16,
  -0.934196678350E-19)
  else
  emf:=p10(t, 0.000, 1300.000,
   0.000000000000E+00,
   0.259293946010E-01,
   0.157101418800E-04,
   0.438256272370E-07,
  -0.252611697940E-09,
   0.643118193390E-12,
  -0.100634715190E-14,
   0.997453389920E-18,
  -0.608632456070E-21,
   0.208492293390E-24,
  -0.306821961510E-28);
 end else
 if (idn=Ord('R')) then begin
  if t < 1064.180 then
  emf:=p9(t, -50.000, 1064.180,
   0.000000000000E+00,
   0.528961729765E-02,
   0.139166589782E-04,
  -0.238855693017E-07,
   0.356916001063E-10,
  -0.462347666298E-13,
   0.500777441034E-16,
  -0.373105886191E-19,
   0.157716482367E-22,
  -0.281038625251E-26)
  else
  if t < 1664.500 then
  emf:=p5(t, 1064.180, 1664.500,
   0.295157925316E+01,
  -0.252061251332E-02,
   0.159564501865E-04,
  -0.764085947576E-08,
   0.205305291024E-11,
  -0.293359668173E-15)
  else
  emf:=p4(t, 1664.500, 1768.100,
   0.152232118209E+03,
  -0.268819888545E+00,
   0.171280280471E-03,
  -0.345895706453E-07,
  -0.934633971046E-14);
 end else
 if (idn=Ord('S')) then begin
  if t < 1064.180 then
  emf:=p8(t, -50.000, 1064.180,
   0.000000000000E+00,
   0.540313308631E-02,
   0.125934289740E-04,
  -0.232477968689E-07,
   0.322028823036E-10,
  -0.331465196389E-13,
   0.255744251786E-16,
  -0.125068871393E-19,
   0.271443176145E-23)
  else
  if t < 1664.500 then
  emf:=p4(t, 1064.180, 1664.500,
   0.132900444085E+01,
   0.334509311344E-02,
   0.654805192818E-05,
  -0.164856259209E-08,
   0.129989605174E-13)
  else
  emf:=p4(t, 1664.500, 1768.100,
   0.146628232636E+03,
  -0.258430516752E+00,
   0.163693574641E-03,
  -0.330439046987E-07,
  -0.943223690612E-14);
 end else
 if (idn=Ord('T')) then begin
  if t < 0.000 then
  emf:=p14(t, -270.000, 0.000,
   0.000000000000E+00,
   0.387481063640E-01,
   0.441944343470E-04,
   0.118443231050E-06,
   0.200329735540E-07,
   0.901380195590E-09,
   0.226511565930E-10,
   0.360711542050E-12,
   0.384939398830E-14,
   0.282135219250E-16,
   0.142515947790E-18,
   0.487686622860E-21,
   0.107955392700E-23,
   0.139450270620E-26,
   0.797951539270E-30)
  else
  emf:=p8(t, 0.000, 400.000,
   0.000000000000E+00,
   0.387481063640E-01,
   0.332922278800E-04,
   0.206182434040E-06,
  -0.218822568460E-08,
   0.109968809280E-10,
  -0.308157587720E-13,
   0.454791352900E-16,
  -0.275129016730E-19);
 end else
 if (idn=Ord('A')*256+Ord('1')) then begin
  emf:=p8(t, 0.000, 2500.000,
   7.1564735E-04,
   1.1951905E-02,
   1.6672625E-05,
  -2.8287807E-08,
   2.8397839E-11,
  -1.8505007E-14,
   7.3632123E-18,
  -1.6148878E-21,
   1.4901679E-25);
 end else
 if (idn=Ord('A')*256+Ord('2')) then begin
  emf:=p8(t, 0.000, 1800.000,
  -1.0850558E-04,
   1.1642292E-02,
   2.1280289E-05,
  -4.4258402E-08,
   5.5652058E-11,
  -4.3801310E-14,
   2.0228390E-17,
  -4.9354041E-21,
   4.8119846E-25);
 end else
 if (idn=Ord('A')*256+Ord('3')) then begin
  emf:=p8(t, 0.000, 1800.000,
  -1.0649133E-04,
   1.1686475E-02,
   1.8022157E-05,
  -3.3436998E-08,
   3.7081688E-11,
  -2.5748444E-14,
   1.0301893E-17,
  -2.0735944E-21,
   1.4678450E-25);
 end else
 if (idn=Ord('L')) then begin
  if correct_L then
  if t < 0.000 then
  emf:=p8(t, -200.000, 0.000,
  -5.8952244E-05,
   6.3391502E-02,
   6.7592964E-05,
   2.0672566E-07,
   5.5720884E-09,
   5.7133860E-11,
   3.2995593E-13,
   9.9232420E-16,
   1.2079584E-18)
  else
  emf:=p8(t, 0.000, 800.000,
  -1.8656953E-05,
   6.3310975E-02,
   6.0153091E-05,
  -8.0073134E-08,
   9.6946071E-11,
  -3.6047289E-14,
  -2.4694775E-16,
   4.2880341E-19,
  -2.0725297E-22)
  else
  emf:=p8(t, -200.000, 800.000,
  -4.1626930E-06,
   6.3310880E-02,
   6.0118088E-05,
  -7.9469796E-08,
   9.3101891E-11,
  -2.4299630E-14,
  -2.6547176E-16,
   4.4332477E-19,
  -2.1172626E-22);
 end else
 if (idn=Ord('M')) then begin
  emf:=p3(t, -200.000, 100.000,
   2.4455560E-06,
   4.2638917E-02,
   5.0348392E-05,
  -4.4974485E-08);
 end else
 emf:=_NaN;
 thermocouple_cnt:=thermocouple_cnt+1;
 thermocouple_emf:=emf;
end;
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_lot(idn:Integer):Real;
// Return:
//  Thermocouple low temperature range, °C. Return NaN on invalid idn.
// Arguments:
//  idn - thermocouple type identifier number (see thermocouple_emf).
//////////////////////////////////////////////////////////////////////////////
function thermocouple_lot(idn:Integer):Real;
var t:Real;
begin
 if (idn=Ord('B'))              then t:=   0.000 else
 if (idn=Ord('E'))              then t:=-270.000 else
 if (idn=Ord('J'))              then t:=-210.000 else
 if (idn=Ord('K'))              then t:=-270.000 else
 if (idn=Ord('N'))              then t:=-270.000 else
 if (idn=Ord('R'))              then t:= -50.000 else
 if (idn=Ord('S'))              then t:= -50.000 else
 if (idn=Ord('T'))              then t:=-270.000 else
 if (idn=Ord('A')*256+Ord('1')) then t:=   0.000 else
 if (idn=Ord('A')*256+Ord('2')) then t:=   0.000 else
 if (idn=Ord('A')*256+Ord('3')) then t:=   0.000 else
 if (idn=Ord('L'))              then t:=-200.000 else
 if (idn=Ord('M'))              then t:=-200.000 else
 t:=_NaN;
 thermocouple_lot:=t;
end;
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_hit(idn:Integer):Real;
// Return:
//  Thermocouple high temperature range, °C. Return NaN on invalid idn.
// Arguments:
//  idn - thermocouple type identifier number (see thermocouple_emf).
//////////////////////////////////////////////////////////////////////////////
function thermocouple_hit(idn:Integer):Real;
var t:Real;
begin
 if (idn=Ord('B'))              then t:=1820.000 else
 if (idn=Ord('E'))              then t:=1000.000 else
 if (idn=Ord('J'))              then t:=1200.000 else
 if (idn=Ord('K'))              then t:=1372.000 else
 if (idn=Ord('N'))              then t:=1300.000 else
 if (idn=Ord('R'))              then t:=1768.100 else
 if (idn=Ord('S'))              then t:=1768.100 else
 if (idn=Ord('T'))              then t:= 400.000 else
 if (idn=Ord('A')*256+Ord('1')) then t:=2500.000 else
 if (idn=Ord('A')*256+Ord('2')) then t:=1800.000 else
 if (idn=Ord('A')*256+Ord('3')) then t:=1800.000 else
 if (idn=Ord('L'))              then t:= 800.000 else
 if (idn=Ord('M'))              then t:= 100.000 else
 t:=_NaN;
 thermocouple_hit:=t;
end;
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_idn(ids:String):Integer;
// Return:
//  Thermocouple type identification number. Return NaN on invalid ids.
// Arguments:
//  ids - thermocouple type identifier string (see thermocouple_emf).
//////////////////////////////////////////////////////////////////////////////
function thermocouple_idn(ids:LongString):Integer;
var idn:Integer;
begin
 ids:=Trim(ids);
 if (ids='B')  then idn:=Ord('B')              else
 if (ids='E')  then idn:=Ord('E')              else
 if (ids='J')  then idn:=Ord('J')              else
 if (ids='K')  then idn:=Ord('K')              else
 if (ids='N')  then idn:=Ord('N')              else
 if (ids='R')  then idn:=Ord('R')              else
 if (ids='S')  then idn:=Ord('S')              else
 if (ids='T')  then idn:=Ord('T')              else
 if (ids='A1') then idn:=Ord('A')*256+Ord('1') else
 if (ids='A2') then idn:=Ord('A')*256+Ord('2') else
 if (ids='A3') then idn:=Ord('A')*256+Ord('3') else
 if (ids='L')  then idn:=Ord('L')              else
 if (ids='M')  then idn:=Ord('M')              else
 idn:=0;
 thermocouple_idn:=idn;
end;
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_ids(idn:Integer):String;
// Return:
//  Thermocouple type identifier string. Return empty string on invalid idn.
// Arguments:
//  idn - thermocouple type identifier number (see thermocouple_emf).
//////////////////////////////////////////////////////////////////////////////
function thermocouple_ids(idn:Integer):LongString;
var ids:LongString;
begin
 if (idn=Ord('B'))              then ids:=Chr(idn)                     else
 if (idn=Ord('E'))              then ids:=Chr(idn)                     else
 if (idn=Ord('J'))              then ids:=Chr(idn)                     else
 if (idn=Ord('K'))              then ids:=Chr(idn)                     else
 if (idn=Ord('N'))              then ids:=Chr(idn)                     else
 if (idn=Ord('R'))              then ids:=Chr(idn)                     else
 if (idn=Ord('S'))              then ids:=Chr(idn)                     else
 if (idn=Ord('T'))              then ids:=Chr(idn)                     else
 if (idn=Ord('A')*256+Ord('1')) then ids:=Chr(idn div 256)+''+Chr(idn) else
 if (idn=Ord('A')*256+Ord('2')) then ids:=Chr(idn div 256)+''+Chr(idn) else
 if (idn=Ord('A')*256+Ord('3')) then ids:=Chr(idn div 256)+''+Chr(idn) else
 if (idn=Ord('L'))              then ids:=Chr(idn)                     else
 if (idn=Ord('M'))              then ids:=Chr(idn)                     else
 ids:='';
 thermocouple_ids:=ids; ids:='';
end;
//////////////////////////////////////////////////////////////////////////////
// function thermocouple_ttc(emf:Real; idn:Integer):Real;
// Return:
//  Thermocouple temperature, °C. Return NAN on error or invalid idn.
// Arguments:
//  emf - thermocouple EMF, mv. Cold junction is at 0, °C.
//  idn - thermocouple type identifier number (see thermocouple_emf).
//////////////////////////////////////////////////////////////////////////////
function thermocouple_ttc(emf:Real; idn:Integer):Real;
var a,b,c,d,e,fa,fb,fc,eps,tol,tol1,xm,p,q,r,s,big,del,fac:Real;
 itn,met,mit:Integer; Terminate,Failure:Boolean;
 function FSign(a,b:Real):Real;
 begin
  if b<=0 then FSign:=-abs(a) else FSign:=abs(a);
 end;
 function Sign(x:Real):Integer;
 begin
  if x<0 then Sign:=-1 else if x>0 then Sign:=+1 else Sign:=0;
 end;
 function F(t:Real):Real;
 var goal:Real;
 begin
  goal:=thermocouple_emf(t,idn)-emf;
  if isNaN(goal) then Failure:=True;
  if isInf(goal) then Failure:=True;
  F:=goal;
 end;
begin
 c:=0; itn:=0;
 Failure:=False; Terminate:=False;
 met:=thermocouple_met; if met<=0 then met:=1;     // zeroin method
 mit:=thermocouple_mit; if mit<=0 then mit:=128;   // max.iteration
 eps:=thermocouple_eps; if eps<=0 then eps:=1E-15; // mach. epsilon
 tol:=thermocouple_tol; if tol<=0 then tol:=1E-14; // the tolerance
 big:=thermocouple_big; if big<=0 then big:=1E+15; // big range
 del:=thermocouple_del; if del<=0 then del:=0;     // delta
 fac:=thermocouple_fac; if fac<=1 then fac:=1;     // factor
 if (idn=Ord('B')) and (abs(emf)<3.0E-3) then begin
  c:=0.0; // Special case because B thermocouple is not monotonous near 0
 end else
 if not Failure then begin
  a:=thermocouple_lot(idn); b:=thermocouple_hit(idn);
  if a>b then begin c:=a; a:=b; b:=c; end;
  fa:=F(a); fb:=F(b);
  if del>0 then begin // Use linear extrapolation outside (a,b)
   if fa<fb then begin
    if Sign(fa)>0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end else
    if Sign(fb)<0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end;
   end else
   if fa>fb then begin
    if Sign(fb)>0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end else
    if Sign(fa)<0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end;
   end;
  end else
  if Sign(fa)*Sign(fb)>0 then begin
   d:=b-a;
   if Sign(fa)*Sign(fb-fa)>0 then if fac<=1 then a:=-big else begin
    while (itn<mit) and (abs(a-d)<big) and not Failure do begin
     if Sign(F(a-d))*Sign(fb)<=0 then begin a:=a-d; itn:=mit; end;
     d:=d*2; itn:=itn+1;
    end;
   end;
   if Sign(fb)*Sign(fb-fa)<0 then if fac<=1 then b:=+big else begin
    while (itn<mit) and (abs(b+d)<big) and not Failure do begin
     if Sign(fa)*Sign(F(b+d))<=0 then begin b:=b+d; itn:=mit; end;
     d:=d*2; itn:=itn+1;
    end;
   end;
   itn:=0;
  end;
  if not Terminate then
  if met=1 then begin // Method: ZeroIn
   fa:=F(a);
   if fa=0.0 then c:=a else begin
    fb:=F(b);
    if fb=0.0 then c:=b else begin
     c:=a; fc:=fa; d:=b-a; e:=d;
     if Sign(fa)*Sign(fb)>0 then Failure:=True;
     while not Terminate and not Failure do begin
      if abs(fc)<abs(fb) then begin a:=b; b:=c; c:=a; fa:=fb; fb:=fc; fc:=fa; end;
      tol1:=2.0*eps*abs(b)+0.5*tol; xm:=0.5*(c-b);
      if (abs(xm)<=tol1) or (fb=0.0) or (itn>=mit) then Terminate:=True else begin
       if (abs(e)<tol1) or (abs(fa)<=abs(fb)) then begin
        d:=xm; e:=d;
       end else begin
        if a<>c then begin
         q:=fa/fc; r:=fb/fc; s:=fb/fa; p:=s*(2.0*xm*q*(q-r)-(b-a)*(r-1.0)); q:=(q-1.0)*(r-1.0)*(s-1.0);
        end else begin
         s:=fb/fa; p:=2.0*xm*s; q:=1.0-s;
        end;
        if p>0.0 then q:=-q; p:=abs(p);
        if (2.0*p>=3.0*xm*q-abs(tol1*q)) or (p>=abs(e*q*0.5)) then begin
         d:=xm; e:=d;
        end else begin
         e:=d; d:=p/q;
        end;
       end;
       a:=b; fa:=fb;
       if abs(d)> tol1 then b:=b+d else b:=b+FSign(tol1,xm);
       fb:=F(b); if fb*(fc/abs(fc))>0 then begin c:=a; fc:=fa; d:=b-a; e:=d; end;
      end;
      itn:=itn+1;
     end;
     c:=b;
    end;
   end;
  end else begin // Method: Bisection
   fa:=F(a);
   if fa=0.0 then c:=a else begin
    fb:=F(b);
    if fb=0.0 then c:=b else begin
     if Sign(fa)*Sign(fb)>0 then Failure:=True;
     while not Terminate and not Failure do begin
      c:=a+(b-a)*0.5; fc:=F(c);
      if (fc=0.0) then Terminate:=True else
      if (itn>=mit) then Terminate:=True else
      if (abs(b-a)<tol) then Terminate:=True else
      if Sign(fc)*Sign(fa)<0 then begin b:=c; fb:=fc; end else
      if Sign(fc)*Sign(fb)<0 then begin a:=c; fa:=fc; end else Failure:=True;
      itn:=itn+1;
     end;
    end;
   end;
  end;
 end;
 if Failure then c:=_NaN;
 thermocouple_ttc:=c;
end;
//////////////////////////////////////////////////////////////////////////////
// Test termocouple conversions; return error count; 0 means OK.
//////////////////////////////////////////////////////////////////////////////
function thermocouple_tst:Integer;
var errors:Integer;
 procedure TestEmf(t:Real; idn:Integer; s:LongString);
 var emf,ttc:Real; fine:Boolean; q:LongString;
 begin
  fine:=true;
  emf:=thermocouple_emf(t,idn);
  ttc:=thermocouple_ttc(emf,idn);
  q:=StrFix(emf,1,3);
  if q='-0.000' then q:='0.000';
  if s<>'' then fine:=(q=s); q:='';
  if (abs(t-ttc)>1E-6) then fine:=false;
  writeln('Test ',thermocouple_ids(idn):2,':  t=',t:4:0,'  emf=',emf:7:3,'  ttc=',ttc:9:3,'  ',fine);
  if not fine then errors:=errors+1;
 end;
begin
 errors:=0;
 TestEmf( -273,  Ord('B'),  '');
 TestEmf(    0,  Ord('B'),  '0.000');
 TestEmf(  300,  Ord('B'),  '0.431');
 TestEmf(  600,  Ord('B'),  '1.792');
 TestEmf(  900,  Ord('B'),  '3.957');
 TestEmf( 1820,  Ord('B'), '13.820');
 TestEmf( 1823,  Ord('B'),  '');
 TestEmf( -273,  Ord('E'),  '');
 TestEmf( -270,  Ord('E'), '-9.835');
 TestEmf( -100,  Ord('E'), '-5.237');
 TestEmf(    0,  Ord('E'),  '0.000');
 TestEmf(  300,  Ord('E'), '21.036');
 TestEmf(  600,  Ord('E'), '45.093');
 TestEmf(  900,  Ord('E'), '68.787');
 TestEmf( 1000,  Ord('E'), '76.373');
 TestEmf( 1003,  Ord('E'),  '');
 TestEmf( -273,  Ord('J'),  '');
 TestEmf( -210,  Ord('J'), '-8.095');
 TestEmf( -100,  Ord('J'), '-4.633');
 TestEmf(    0,  Ord('J'),  '0.000');
 TestEmf(  300,  Ord('J'), '16.327');
 TestEmf(  600,  Ord('J'), '33.102');
 TestEmf(  900,  Ord('J'), '51.877');
 TestEmf( 1200,  Ord('J'), '69.553');
 TestEmf( 1203,  Ord('J'),  '');
 TestEmf( -273,  Ord('K'),  '');
 TestEmf( -270,  Ord('K'), '-6.458');
 TestEmf( -100,  Ord('K'), '-3.554');
 TestEmf(    0,  Ord('K'),  '0.000');
 TestEmf(  100,  Ord('K'),  '4.096');
 TestEmf(  300,  Ord('K'), '12.209');
 TestEmf(  600,  Ord('K'), '24.905');
 TestEmf(  900,  Ord('K'), '37.326');
 TestEmf( 1372,  Ord('K'), '54.886');
 TestEmf( 1375,  Ord('K'),  '');
 TestEmf( -273,  Ord('N'),  '');
 TestEmf( -270,  Ord('N'), '-4.345');
 TestEmf( -100,  Ord('N'), '-2.407');
 TestEmf(    0,  Ord('N'),  '0.000');
 TestEmf(  300,  Ord('N'),  '9.341');
 TestEmf(  600,  Ord('N'), '20.613');
 TestEmf(  900,  Ord('N'), '32.371');
 TestEmf( 1300,  Ord('N'), '47.513');
 TestEmf( 1303,  Ord('N'),  '');
 TestEmf(  -53,  Ord('R'),  '');
 TestEmf(  -50,  Ord('R'), '-0.226');
 TestEmf(    0,  Ord('R'),  '0.000');
 TestEmf(  300,  Ord('R'),  '2.401');
 TestEmf(  600,  Ord('R'),  '5.583');
 TestEmf(  900,  Ord('R'),  '9.205');
 TestEmf( 1768,  Ord('R'), '21.101');
 TestEmf( 1771,  Ord('R'),  '');
 TestEmf(  -53,  Ord('S'),  '');
 TestEmf(  -50,  Ord('S'), '-0.236');
 TestEmf(    0,  Ord('S'),  '0.000');
 TestEmf(  300,  Ord('S'),  '2.323');
 TestEmf(  600,  Ord('S'),  '5.239');
 TestEmf(  900,  Ord('S'),  '8.449');
 TestEmf( 1768,  Ord('S'),  '18.693');
 TestEmf( 1771,  Ord('S'),  '');
 TestEmf( -273,  Ord('T'),  '');
 TestEmf( -270,  Ord('T'), '-6.258');
 TestEmf( -100,  Ord('T'), '-3.379');
 TestEmf(    0,  Ord('T'),  '0.000');
 TestEmf(  300,  Ord('T'), '14.862');
 TestEmf(  400,  Ord('T'), '20.872');
 TestEmf(  403,  Ord('T'),  '');
 TestEmf(   -3,  Ord('A')*256+Ord('1'),  '');
 TestEmf(    0,  Ord('A')*256+Ord('1'),  '0.001');
 TestEmf(  600,  Ord('A')*256+Ord('1'),  '9.606');
 TestEmf( 1000,  Ord('A')*256+Ord('1'), '16.128');
 TestEmf( 1500,  Ord('A')*256+Ord('1'), '23.311');
 TestEmf( 2500,  Ord('A')*256+Ord('1'), '33.640');
 TestEmf( 2503,  Ord('A')*256+Ord('1'),  '');
 TestEmf(   -3,  Ord('A')*256+Ord('2'),  '');
 TestEmf(    0,  Ord('A')*256+Ord('2'),  '0.000');
 TestEmf(  600,  Ord('A')*256+Ord('2'),  '9.707');
 TestEmf( 1000,  Ord('A')*256+Ord('2'), '16.289');
 TestEmf( 1500,  Ord('A')*256+Ord('2'), '23.515');
 TestEmf( 1800,  Ord('A')*256+Ord('2'), '27.232');
 TestEmf( 1803,  Ord('A')*256+Ord('2'),  '');
 TestEmf(   -3,  Ord('A')*256+Ord('3'),  '');
 TestEmf(    0,  Ord('A')*256+Ord('3'),  '0.000');
 TestEmf(  600,  Ord('A')*256+Ord('3'),  '9.506');
 TestEmf( 1000,  Ord('A')*256+Ord('3'), '15.980');
 TestEmf( 1500,  Ord('A')*256+Ord('3'), '23.106');
 TestEmf( 1800,  Ord('A')*256+Ord('3'), '26.773');
 TestEmf( 1803,  Ord('A')*256+Ord('3'),  '');
 TestEmf( -203,  Ord('L'),  '');
 TestEmf( -200,  Ord('L'), '-9.488');
 TestEmf( -100,  Ord('L'), '-5.641');
 TestEmf(    0,  Ord('L'),  '0.000');
 TestEmf(  300,  Ord('L'), '22.843');
 TestEmf(  600,  Ord('L'), '49.108');
 TestEmf(  800,  Ord('L'), '66.466');
 TestEmf(  803,  Ord('L'),  '');
 TestEmf( -203,  Ord('M'),  '');
 TestEmf( -200,  Ord('M'), '-6.154');
 TestEmf( -100,  Ord('M'), '-3.715');
 TestEmf(    0,  Ord('M'),  '0.000');
 TestEmf(   50,  Ord('M'),  '2.252');
 TestEmf(  100,  Ord('M'),  '4.722');
 TestEmf(  103,  Ord('M'),  '');
 thermocouple_tst:=errors;
end;
//////////////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_ohm(t:Real; idn:Integer):Real;
// Return:
//  RTD resistance, Ohms. Return NaN on invalid idn.
// Arguments:
//  t    - temperature, °C.
//  idn  - RTD type identifier number:
//   385 = RTD Pt100 platinum платина R0=100 alfa=0.00385
//   391 = RTD Pt100 platinum платина R0=100 alfa=0.00391
//   428 = RTD Cu100 copper   медь    R0=100 alfa=0.00428
//   426 = RTD Cu100 copper   медь    R0=100 alfa=0.00426
//   617 = RTD Ni100 nikel    никель  R0=100 alfa=0.00617
//////////////////////////////////////////////////////////////////////////////
function restemperdet_ohm(t:Real; idn:Integer):Real;
var ohm:Real;
 // R0AiBjCkDn(t,L,H,..) define polynom function
 // use linear extrapolation outside (L,H) range
 // A,B,C,D: coefficients taken from GOST tables
 function R0A1B2(t,L,R,R0,A,B:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(R,t)); s:=t-x;
  f:=R0*(1+x*(A+x*B));
  if s<>0 then begin // linear extrapolation
   q:=R0*(A+x*2*B);
   f:=f+q*s;
  end;
  R0A1B2:=f;
 end;
 function R0A1B2C4D3(t,L,R,R0,A,B,C,D:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(R,t)); s:=t-x;
  f:=R0*(1+x*(A+x*(B+x*C*(x+D))));
  if s<>0 then begin // linear extrapolation
   q:=R0*(A+x*(2*B+x*C*(4*x+3*D)));
   f:=f+q*s;
  end;
  R0A1B2C4D3:=f;
 end;
 function R0A1B2C3D1(t,L,R,R0,A,B,C,D:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(R,t)); s:=t-x;
  f:=R0*(1+x*(A+B*D+x*(B+C*x)));
  if s<>0 then begin // linear extrapolation
   q:=R0*(A+B*D+x*(2*B+C*3*x));
   f:=f+q*s;
  end;
  R0A1B2C3D1:=f;
 end;
 function R0A1B2C3D2(t,L,R,R0,A,B,C,D:Real):Real;
 var x,s,f,q:Real;
 begin
  x:=max(L,min(R,t)); s:=t-x;
  f:=R0*(1+x*(A+x*(B+C*(x+D))));
  if s<>0 then begin // linear extrapolation
   q:=R0*(A+x*(2*B+C*(3*x+2*D)));
   f:=f+q*s;
  end;
  R0A1B2C3D2:=f;
 end;
begin
 if (idn=385) then begin
  if t < 0.000
  then ohm:=R0A1B2C4D3(t, -200.000,   0.000, 100.0, 3.9083E-3, -5.775E-7,  -4.183E-12, -100.00)
  else ohm:=R0A1B2(    t,    0.000, 850.000, 100.0, 3.9083E-3, -5.775E-7);
 end else
 if (idn=391) then begin
  if t < 0.000
  then ohm:=R0A1B2C4D3(t, -200.000,   0.000, 100.0, 3.9690E-3, -5.841E-7,  -4.330E-12, -100.00)
  else ohm:=R0A1B2(    t,    0.000, 850.000, 100.0, 3.9690E-3, -5.841E-7);
 end else
 if (idn=428) then begin
  if t < 0.000
  then ohm:=R0A1B2C3D1(t, -180.000,   0.000, 100.0, 4.28E-3,   -6.2032E-7, 8.5154E-10, 6.7)
  else ohm:=R0A1B2(    t,    0.000, 200.000, 100.0, 4.28E-3,   0);
 end else
 if (idn=426) then begin
  ohm:=R0A1B2(         t,  -50.000, 200.000, 100.0, 4.26E-3,   0);
 end else
 if (idn=617) then begin
  if t < 100.000
  then ohm:=R0A1B2(    t,  -60.000, 100.000, 100.0, 5.4963E-3,  6.7556E-6)
  else ohm:=R0A1B2C3D2(t,  100.000, 180.000, 100.0, 5.4963E-3,  6.7556E-6, 9.2004E-09, -100.00);
 end else
 ohm:=_NaN;
 restemperdet_cnt:=restemperdet_cnt+1;
 restemperdet_ohm:=ohm;
end;
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_lot(idn:Integer):Real;
// Return:
//  RTD low temperature range, °C. Return NaN on invalid idn.
// Arguments:
//  idn - RTD type identifier number (see restemperdet_ohm).
//////////////////////////////////////////////////////////////////////////////
function restemperdet_lot(idn:Integer):Real;
var t:Real;
begin
 if (idn=385) then t:=-200 else
 if (idn=391) then t:=-200 else
 if (idn=428) then t:=-180 else
 if (idn=426) then t:=-50  else
 if (idn=617) then t:=-60  else
 t:=_NaN;
 restemperdet_lot:=t;
end;
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_hit(idn:Integer):Real;
// Return:
//  RTD high temperature range, °C. Return NaN on invalid idn.
// Arguments:
//  idn - RTD type identifier number (see restemperdet_ohm).
//////////////////////////////////////////////////////////////////////////////
function restemperdet_hit(idn:Integer):Real;
var t:Real;
begin
 if (idn=385) then t:=850 else
 if (idn=391) then t:=850 else
 if (idn=428) then t:=200 else
 if (idn=426) then t:=200 else
 if (idn=617) then t:=180 else
 t:=_NaN;
 restemperdet_hit:=t;
end;
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_idn(ids:String):Integer;
// Return:
//  RTD type identification number. Return NaN on invalid ids.
// Arguments:
//  ids - RTD type identifier string (see restemperdet_ohm).
//////////////////////////////////////////////////////////////////////////////
function restemperdet_idn(ids:LongString):Integer;
var idn:Integer;
begin
 ids:=Trim(ids);
 if (ids='PT100_385') then idn:=385 else
 if (ids='PT100_391') then idn:=391 else
 if (ids='CU100_428') then idn:=428 else
 if (ids='CU100_426') then idn:=426 else
 if (ids='NI100_617') then idn:=617 else
 idn:=0;
 restemperdet_idn:=idn;
end;
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_ids(idn:Integer):String;
// Return:
//  RTD type identifier string. Return empty string on invalid idn.
// Arguments:
//  idn - RTD type identifier number (see restemperdet_ohm).
//////////////////////////////////////////////////////////////////////////////
function restemperdet_ids(idn:Integer):LongString;
var ids:LongString;
begin
 if (idn=385) then ids:='PT100_385' else
 if (idn=391) then ids:='PT100_391' else
 if (idn=428) then ids:='CU100_428' else
 if (idn=426) then ids:='CU100_426' else
 if (idn=617) then ids:='NI100_617' else
 ids:='';
 restemperdet_ids:=ids; ids:='';
end;
//////////////////////////////////////////////////////////////////////////////
// function restemperdet_ttc(ohm:Real; idn:Integer):Real;
// Return:
//  RTD temperature, °C. Return NAN on error or invalid idn.
// Arguments:
//  ohm - RTD resistance, ohms.
//  idn - RTD type: 385,391, 428, 426, 617
//////////////////////////////////////////////////////////////////////////////
function restemperdet_ttc(ohm:Real; idn:Integer):Real;
var a,b,c,d,e,fa,fb,fc,eps,tol,tol1,xm,p,q,r,s,big,del,fac:Real;
 itn,met,mit:Integer; Terminate,Failure:Boolean;
 function FSign(a,b:Real):Real;
 begin
  if b<=0 then FSign:=-abs(a) else FSign:=abs(a);
 end;
 function Sign(x:Real):Integer;
 begin
  if x<0 then Sign:=-1 else if x>0 then Sign:=+1 else Sign:=0;
 end;
 function F(t:Real):Real;
 var goal:Real;
 begin
  goal:=restemperdet_ohm(t,idn)-ohm;
  if isNaN(goal) then Failure:=True;
  if isInf(goal) then Failure:=True;
  F:=goal;
 end;
begin
 c:=0; itn:=0;
 Failure:=False; Terminate:=False;
 met:=restemperdet_met; if met<=0 then met:=1;     // zeroin method
 mit:=restemperdet_mit; if mit<=0 then mit:=128;   // max.iteration
 eps:=restemperdet_eps; if eps<=0 then eps:=1E-15; // mach. epsilon
 tol:=restemperdet_tol; if tol<=0 then tol:=1E-14; // the tolerance
 big:=restemperdet_big; if big<=0 then big:=1E+15; // big range
 del:=restemperdet_del; if del<=0 then del:=0;     // delta
 fac:=restemperdet_fac; if fac<=1 then fac:=1;     // factor
 if not Failure then begin
  a:=restemperdet_lot(idn); b:=restemperdet_hit(idn);
  if a>b then begin c:=a; a:=b; b:=c; end;
  fa:=F(a); fb:=F(b);
  if del>0 then begin // Use linear extrapolation outside (a,b)
   if fa<fb then begin
    if Sign(fa)>0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end else
    if Sign(fb)<0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end;
   end else
   if fa>fb then begin
    if Sign(fb)>0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end else
    if Sign(fa)<0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end;
   end;
  end else
  if Sign(fa)*Sign(fb)>0 then begin
   d:=b-a;
   if Sign(fa)*Sign(fb-fa)>0 then if fac<=1 then a:=-big else begin
    while (itn<mit) and (abs(a-d)<big) and not Failure do begin
     if Sign(F(a-d))*Sign(fb)<=0 then begin a:=a-d; itn:=mit; end;
     d:=d*2; itn:=itn+1;
    end;
   end;
   if Sign(fb)*Sign(fb-fa)<0 then if fac<=1 then b:=+big else begin
    while (itn<mit) and (abs(b+d)<big) and not Failure do begin
     if Sign(fa)*Sign(F(b+d))<=0 then begin b:=b+d; itn:=mit; end;
     d:=d*2; itn:=itn+1;
    end;
   end;
   itn:=0;
  end;
  if not Terminate then
  if met=1 then begin // Method: ZeroIn
   fa:=F(a);
   if fa=0.0 then c:=a else begin
    fb:=F(b);
    if fb=0.0 then c:=b else begin
     c:=a; fc:=fa; d:=b-a; e:=d;
     if Sign(fa)*Sign(fb)>0 then Failure:=True;
     while not Terminate and not Failure do begin
      if abs(fc)<abs(fb) then begin a:=b; b:=c; c:=a; fa:=fb; fb:=fc; fc:=fa; end;
      tol1:=2.0*eps*abs(b)+0.5*tol; xm:=0.5*(c-b);
      if (abs(xm)<=tol1) or (fb=0.0) or (itn>=mit) then Terminate:=True else begin
       if (abs(e)<tol1) or (abs(fa)<=abs(fb)) then begin
        d:=xm; e:=d;
       end else begin
        if a<>c then begin
         q:=fa/fc; r:=fb/fc; s:=fb/fa; p:=s*(2.0*xm*q*(q-r)-(b-a)*(r-1.0)); q:=(q-1.0)*(r-1.0)*(s-1.0);
        end else begin
         s:=fb/fa; p:=2.0*xm*s; q:=1.0-s;
        end;
        if p>0.0 then q:=-q; p:=abs(p);
        if (2.0*p>=3.0*xm*q-abs(tol1*q)) or (p>=abs(e*q*0.5)) then begin
         d:=xm; e:=d;
        end else begin
         e:=d; d:=p/q;
        end;
       end;
       a:=b; fa:=fb;
       if abs(d)> tol1 then b:=b+d else b:=b+FSign(tol1,xm);
       fb:=F(b); if fb*(fc/abs(fc))>0 then begin c:=a; fc:=fa; d:=b-a; e:=d; end;
      end;
      itn:=itn+1;
     end;
     c:=b;
    end;
   end;
  end else begin // Method: Bisection
   fa:=F(a);
   if fa=0.0 then c:=a else begin
    fb:=F(b);
    if fb=0.0 then c:=b else begin
     if Sign(fa)*Sign(fb)>0 then Failure:=True;
     while not Terminate and not Failure do begin
      c:=a+(b-a)*0.5; fc:=F(c);
      if (fc=0.0) then Terminate:=True else
      if (itn>=mit) then Terminate:=True else
      if (abs(b-a)<tol) then Terminate:=True else
      if Sign(fc)*Sign(fa)<0 then begin b:=c; fb:=fc; end else
      if Sign(fc)*Sign(fb)<0 then begin a:=c; fa:=fc; end else Failure:=True;
      itn:=itn+1;
     end;
    end;
   end;
  end;
 end;
 if Failure then c:=_NaN;
 restemperdet_ttc:=c;
end;
//////////////////////////////////////////////////////////////////////////////
// Test RTD conversions; return error count; 0 means OK.
//////////////////////////////////////////////////////////////////////////////
function restemperdet_tst:Integer;
var errors:Integer;
 procedure TestOhm(t:Real; idn:Integer; s:LongString);
 var ohm,ttc:Real; fine:Boolean; q:LongString;
 begin
  fine:=true;
  ohm:=restemperdet_ohm(t,idn);
  ttc:=restemperdet_ttc(ohm,idn);
  q:=StrFix(ohm,1,2);
  if q='-0.00' then q:='0.00';
  if s<>'' then fine:=(q=s); q:='';
  if (abs(t-ttc)>1E-6) then fine:=false;
  writeln('Test ',restemperdet_ids(idn):2,':  t=',t:4:0,'  ohm=',ohm:7:3,'  ttc=',ttc:9:3,'  ',fine);
  if not fine then errors:=errors+1;
 end;
begin
 errors:=0;
 TestOhm( -203,  385,   '');
 TestOhm( -200,  385,   '18.52');
 TestOhm(  -50,  385,   '80.31');
 TestOhm(    0,  385,  '100.00');
 TestOhm(   50,  385,  '119.40');
 TestOhm(  100,  385,  '138.51');
 TestOhm(  150,  385,  '157.33');
 TestOhm(  850,  385,  '390.48');
 TestOhm(  853,  385,  '');
 TestOhm( -203,  391,   '');
 TestOhm( -200,  391,   '17.24');
 TestOhm(  -50,  391,   '80.00');
 TestOhm(    0,  391,  '100.00');
 TestOhm(   50,  391,  '119.70');
 TestOhm(  100,  391,  '139.11');
 TestOhm(  150,  391,  '158.22');
 TestOhm(  850,  391,  '395.16');
 TestOhm(  853,  391,  '');
 TestOhm(  -53,  426,   '');
 TestOhm(  -50,  426,   '78.70');
 TestOhm(  -20,  426,   '91.48');
 TestOhm(    0,  426,  '100.00');
 TestOhm(   50,  426,  '121.30');
 TestOhm(  100,  426,  '142.60');
 TestOhm(  150,  426,  '163.90');
 TestOhm(  200,  426,  '185.20');
 TestOhm(  203,  426,  '');
 TestOhm( -183,  428,   '');
 TestOhm( -180,  428,   '20.53');
 TestOhm(  -50,  428,   '78.46');
 TestOhm(    0,  428,  '100.00');
 TestOhm(   50,  428,  '121.40');
 TestOhm(  100,  428,  '142.80');
 TestOhm(  150,  428,  '164.20');
 TestOhm(  200,  428,  '185.60');
 TestOhm(  203,  428,  '');
 TestOhm(  -73,  617,   '');
 TestOhm(  -60,  617,   '69.45');
 TestOhm(  -50,  617,   '74.21');
 TestOhm(    0,  617,  '100.00');
 TestOhm(   50,  617,  '129.17');
 TestOhm(  100,  617,  '161.72');
 TestOhm(  150,  617,  '198.68');
 TestOhm(  180,  617,  '223.21');
 TestOhm(  183,  617,  '');
 restemperdet_tst:=errors;
end;
//////////////////////////////////////////////////////////////////////////////

procedure InitRtdTc;
begin
 thermocouple_cnt:=0;
 thermocouple_eps:=macheps*1;
 thermocouple_tol:=macheps*10;
 thermocouple_big:=1E+15;
 thermocouple_del:=1;
 thermocouple_fac:=2;
 thermocouple_met:=1;
 thermocouple_mit:=128;
 restemperdet_cnt:=0;
 restemperdet_eps:=macheps*1;
 restemperdet_tol:=macheps*10;
 restemperdet_big:=1E+15;
 restemperdet_del:=1;
 restemperdet_fac:=2;
 restemperdet_met:=1;
 restemperdet_mit:=128;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_rtdtc;
begin
 InitRtdTc;
end;

procedure Free_crw_rtdtc;
begin
end;

initialization

 Init_crw_rtdtc;

finalization

 Free_crw_rtdtc;

end.

//////////////
// END OF FILE
//////////////

