#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

function note(){ return; };

function fatal(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
 if which notify-send; then notify-send -t 30000 -i dialog-error "$date: $script" "$1"; fi;
 echo -ne "\n$1\n\n";
 exit 1;
};

note "Script identification.";
note "######################";
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

note "Extract target package name from script name.";
note "#############################################";
readonly targetname="$(echo "$scriptbase" | sed 's/^[a-z]*-//')";

note "###########################";
note "First argument is a nimber?";
note "###########################";
function is_number(){
 case $1 in
  ''|*[!0-9]*)  return 1; ;;
  *)            return 0; ;;
 esac;
};

note "################################";
note "Uses language dependent strings?";
note "################################";
let uses_langstr=1; note "# 1/0=ON/OFF";

note "##############################";
note "Language dependent string echo";
note "langstr en 'Hello' ru 'Привет'";
note "##############################";
function langstr(){
 local lng="en"; local msg="";
 if [ "$uses_langstr" = "1" ]; then lng="${LANG:0:2}"; fi;
 while [ -n "$lng" ] && [ -n "$1" ] && [ -n "$2" ]; do
  if [ "$lng" = "$1" ]; then msg="$2"; break; fi;
  if [ "$lng" = "en" ]; then msg="$2"; fi;
  if [ "$lng" = "us" ]; then msg="$2"; fi;
  if [ "$lng" = "uk" ]; then msg="$2"; fi;
  shift; shift;
 done;
 if [ -n "$msg" ]; then echo "$msg"; fi;
};

note "###################################";
note "Notification messages with timeout.";
note "Example: notify_ok title message 10";
note "Example: notify_ok 10 title message";
note "###################################";
declare -i default_notify_timeout=5;
function notify_info(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u low -t $timeout -i dialog-information "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Information: $2\n\n";
};
function notify_ok(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u normal -t $timeout -i dialog-ok "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Ok: $2\n\n";
};
function notify_ok_apply(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u normal -t $timeout -i dialog-ok-apply "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Ok-apply: $2\n\n";
};
function notify_warning(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u normal -t $timeout -i dialog-warning "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Warning: $2\n\n";
};
function notify_cancel(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u critical -t $timeout -i dialog-cancel "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Cancel: $2\n\n";
};
function notify_error(){
 if which notify-send >/dev/null 2>&1; then
  local timeout="$default_notify_timeout";
  if is_number "$1"; then timeout="$1"; shift; fi;
  if is_number "$3"; then timeout="$3"; fi; timeout="$(echo "$timeout*1000" | bc)";
  notify-send -u critical -t $timeout -i dialog-error "$(date +%Y.%m.%d-%H:%M:%S) - $1" "$2";
 fi;
 echo -ne "\n$1 - Error: $2\n\n";
};

function write_release_txt(){
 local releasetxt="$scriptHOME/crwdaq/release.txt";
 local date="$(date +%Y%m%d-%H%M%S)";
 echo "$date" > $releasetxt;
};

function strip_bin(){
 if [ -n "$1" ] && [ -e "$1" ]; then
  local dir="$HOME/.local/share/daqgroup/strip";
  if [ ! -e "$dir" ]; then mkdir -p $dir; fi;
  local tmp="$dir/$(basename $1)";
  if [ -e $tmp ]; then rm -fv $tmp; fi;
  if strip $1 -o $tmp; then
   if cmp -s $1 $tmp; then
    echo "strip: skipping $1";
   else
    echo "strip: handling $1";
    strip $1;
   fi;
  fi;
  if [ -e $tmp ]; then rm -fv $tmp; fi;
 fi;
};

function strip_bin_files(){
 strip_bin $scriptHOME/crwdaq/crwdaq;
 strip_bin $scriptHOME/crwdaq/resource/fpquibrk/fpquibrk;
 strip_bin $scriptHOME/crwdaq/resource/daqsite/webserver/websrv;
 strip_bin $scriptHOME/crwdaq/resource/daqsite/cronserver/crongrd;
 strip_bin $scriptHOME/crwdaq/resource/daqsite/speakserver/speaksrv;
 strip_bin $scriptHOME/crwdaq/demo/demo_shm/utility/demo_shm_client;
 strip_bin $scriptHOME/crwdaq/resource/sample/lpr/demo_shm_client/demo_shm_client;
 strip_bin $scriptHOME/crwdaq/resource/sample/lpr/daq_user_task/daq_user_task;
 strip_bin $scriptHOME/crwdaq/resource/sample/lpr/hello_world/hello_world;
 strip_bin $scriptHOME/crwdaq/resource/sample/lpr/hello_plugin/libhello_plugin.so;
 strip_bin $scriptHOME/crwdaq/resource/sample/lpr/test_plugin/libtest_plugin.so;
 strip_bin $scriptHOME/crwdaq/resource/daqtool/lcard/e140/daqpas/e140_dll/libe140_dll.so;
 strip_bin $scriptHOME/crwdaq/resource/daqtool/lcard/e140/daqpas/e140_drv/e140_drv;
};

function clear_empty_dirs(){
 for dir in $(find "$scriptHOME/crwdaq" -type d -empty -name backup); do rmdir -v "$dir"; done;
 for dir in $(find "$scriptHOME/crwdaq" -type d -empty -name lib);    do rmdir -v "$dir"; done;
};

function checksum_in_dirs(){
 local flist="$*";
 for dir in $flist; do
  if [ -d "$dir" ]; then
   if pushd $dir >/dev/null; then
    unix checksum -r -mchecksum "*";
   fi;
   popd >/dev/null;
  fi;
 done;
};

function checksum_secure(){
 if [ -z "$CRW_DAQ_VAR_TMP_DIR" ]; then
  # fake environ variable to pass test
  export CRW_DAQ_VAR_TMP_DIR="$TMPDIR";
 fi;
 $scriptHOME/crwdaq/resource/crwdaq.secure.sh --secure-checksum;
};

function crwdaq_demo_clear(){
 if pushd $scriptHOME/crwdaq/demo; then $scriptHOME/crwdaq/demo/clear.sh; fi;
 popd;
};

function copy_needed_files(){
 # smisite -> shared:
 cp -fvu $scriptHOME/crwkit/add/src/smiserver/smiproxy.exe $scriptHOME/crwdaq/resource/daqsite/smiserver/;
 cp -fvu $scriptHOME/crwkit/add/src/smiserver/smiuisrv.exe $scriptHOME/crwdaq/resource/daqsite/smiserver/;
 cp -fvu $scriptHOME/crwdaq/resource/smisite/smi/bin/*.manifest $scriptHOME/crwdaq/resource/shared/;
 cp -fvu $scriptHOME/crwdaq/resource/smisite/smi/bin/*.dll $scriptHOME/crwdaq/resource/shared/;
 # dimsite -> shared:
 cp -fvu $scriptHOME/crwkit/add/src/dimserver/dimsrv.exe $scriptHOME/crwdaq/resource/daqsite/dimserver/;
 cp -fvu $scriptHOME/crwdaq/resource/dimsite/dim/bin/*.manifest $scriptHOME/crwdaq/resource/shared/;
 cp -fvu $scriptHOME/crwdaq/resource/dimsite/dim/bin/*.dll $scriptHOME/crwdaq/resource/shared/;
};

function pack_daqgroup_suite_crwkit(){
 function deletebackuplib(){
  while [ -n "$1" ]; do
   if [ -d "$1" ]; then
    local arch="$(uname -m)";
    local bckfiles="$(find $1 -type d | grep "/backup$")";
    local libfiles="$(find $1 -type d | grep "/lib/$arch-linux$")";
    for f in $bckfiles $libfiles; do rm -fvr $f; done;
   fi;
   shift;
  done;
 };
 if [ -z "$XZ_OPT" ]; then export XZ_OPT="-9"; fi;
 if pushd $scriptHOME; then
  local flist="";
  for d in $(ls -1 .); do
   if [ -d $d ]; then
    flist="$flist $d";
   fi;
  done;
  flist="crwdaq";
  strip_bin_files;
  clear_empty_dirs;
  copy_needed_files;
  crwdaq_demo_clear;
  write_release_txt;
  deletebackuplib $flist;
  clear_empty_dirs;
  checksum_secure;
  checksum_in_dirs $flist;
  echo "XZ_OPT=$XZ_OPT";
  local date="$(date +%Y%m%d)";
  local arch="$date-$targetname.tar.xz";
  if [ -e $arch ]; then rm -fv $arch; fi;
  local command="tar -cpJf $arch $flist $scriptname";
  echo -ne "\n Homedir: $(pwd)\n Execute: $command\n";
  if $command; then
   notify_ok "$scriptname" "$(langstr en 'Created archive' ru 'Создан архив') $arch.";
  else
   notify_error "$scriptname" "$(langstr en 'Failed achive' ru 'Сбой создания архива') $arch.";
  fi;
 fi;
 popd;
 echo -e "\n$(date +%Y.%m.%d-%H:%M:%S) - $scriptname done.\n";
}
pack_daqgroup_suite_crwkit;

##############
## END OF FILE
##############
