#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CrwKit script to run useful DaqGroup's "cherry" scripts.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

#############################################################################################################################
# https://stackoverflow.com/questions/1763891/is-it-possible-to-make-stdout-and-stderr-output-be-of-different-colors-in-xterm
#############################################################################################################################
function colorize_norm()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[31m&\e[m,'   >&2 ) 3>&1 | sed $'s,.*,\e[32m&\e[m,'   );
function colorize_bold()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[1;31m&\e[m,' >&2 ) 3>&1 | sed $'s,.*,\e[1;32m&\e[m,' );
function colorize_head()( set -o pipefail; ( "$@" 2>&1>&3 | sed $'s,.*,\e[1;35m&\e[m,' >&2 ) 3>&1 | sed $'s,.*,\e[1;37m&\e[m,' );
function colorize_none()( "$@"; );

################################
# print $1 to stderr without EOL
################################
function print_stderr(){
 1>&2 echo -ne "$1";
};

##############################
# fatal error message and exit
##############################
function fatal(){
 if [[ -t 1 ]] && [[ -t 2 ]]; then
  colorize_bold print_stderr "\n\n$2\n\n";
 else
  print_stderr "\n\n$2\n\n";
 fi;
 exit $1;
};

function do_version(){
 echo "$scriptname version 1.0";
};

function do_help(){
 echo "########################################################";
 echo "Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru";
 echo "Script to get current/primary screen(s)/monitor(s) size.";
 echo "Usage:";
 echo " $scriptname [-options]";
 echo "Options:";
 echo " -h,--help      - show help";
 echo " --version      - show version";
 echo " -a,--all       - all information";
 echo " -l,--list      - list monitor(s) size";
 echo " -s,--screen    - current screen  size";
 echo " -p,--primary   - primary monitor size";
 echo "Examples:";
 echo " crwkit $scriptname";
 echo "########################################################";
};

function primary_size(){
 local size="$(xrandr -q | grep ' connected primary ' | sed -e 's/.* primary\s*//' -e 's/[+\-\s].*//' -e 's/x/ /')";
 if [[ -n "$size" ]]; then echo "Primary: $size"; fi;
};

function screen_size(){
 local size="$(xrandr -q | grep "^Screen\s.*:" | sed -e 's/^Screen\s*/Screen/' -e 's/:.*, current \s*/: /' -e 's/[,].*//' -e 's/ x / /')";
 if [[ -n "$size" ]]; then echo "$size"; fi;
};

function monitor_list(){
 xrandr -q | grep '*+' | awk '{ print $1 }' | sed 's/x/ /' | cat -n | tr '\t' ' ' | sed -E 's/^\s*/Monitor/' | sed 's/ /: /';
};

function monitor_count(){
 echo "Monitors: $(xrandr -q | grep '*+' | wc -l)";
};

function display_name(){
 echo "DISPLAY: $DISPLAY";
};

function monitor_info(){
 screen_size;
 primary_size;
 display_name;
 monitor_list;
 monitor_count;
};

######################
# main function to run
######################
function main(){
 case $1 in
  '')                    fatal 1 "Error: no arguments. See $scriptfile -h for help."; ;;
  -h|-help|--help)       do_help; ;;
  -version|--version)    do_version; ;;
  -a|-all|-all)          monitor_info; ;;
  -l|-list|--list)       monitor_list; ;;
  -p|-prim|--prim)       primary_size; ;;
  -primary|--primary)    primary_size; ;;
  -s|-screen|--screen)   screen_size;  ;;
  *)                     fatal 1 "Error: bad arguments. See $scriptfile -h for help."; ;;
 esac;
};

main "$@";

#############
# END OF FILE
#############
