#!/bin/bash

function about(){
 echo "$(basename ${BASH_SOURCE[0]}) version 20240731";
 echo "##############################################################";
 echo "# Install Builder to produce *.run installation files.       #";
 echo "# Copyright (c) 2023 Alexey Kuryakin kouriakine@mail.ru      #";
 echo "##############################################################";
};

function note(){ return; };

function fatal(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
 if which notify-send; then notify-send -t 30000 -i dialog-error "$date: $script" "$1"; fi;
 echo -ne "\n$1\n\n";
 exit 1;
};

note "Script identification.";
note "######################";
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

note "Extract target package name from script name.";
note "#############################################";
readonly targetname="$(echo "$scriptbase" | sed 's/^[a-z]*-//')";

note "The payload directory contains user installer + data archive(s).";
note "################################################################";
readonly payloaddir="$scriptHOME/payload-$targetname";

note "Include the install-library.sh library source.";
note "##############################################";
readonly installlib="$payloaddir/install-library.sh";
source $installlib || fatal "Error: not found $installlib.";

readonly nowait=0; note "Flag (0/1) of Installer to don't wait Enter.";

note "##################";
note "Build target file.";
note "##################";
function build_target(){
 if [ ! -e "$payloaddir/" ]; then
  local bug="<big>$(langstr en 'Could not find directory' ru 'Не найден каталог'):\n<b>$(basename $payloaddir)</b></big>";
  ask_zenity 0 --error --text "$bug" --timeout 180 >/dev/null 2>&1;
  fatal "Error: Could not find: $(basename $payloaddir)";
 fi;
 if [ ! -e "$payloaddir/install-$targetname.sh" ]; then
  local bug="<big>$(langstr en 'Could not find script' ru 'Не найден скрипт'):\n<b>install-$targetname.sh</b></big>";
  ask_zenity 0 --error --text "$bug" --timeout 180 >/dev/null 2>&1;
  fatal "Error: Could not find: install-$targetname.sh";
 fi;
 local date="$(date +%Y%m%d)";
 local timestamp="$(date +%Y%m%d-%H%M%S)";
 local arch=$date-build-$targetname.tar.xz;
 echo "$timestamp" > $payloaddir/$releasefile;
 echo; echo "$releasefile:"; echo "$timestamp";
 rm -f install-$targetname.run; rm -f $arch; sleep 1;
 set_max_tar_compression_mode $arch;
 if tar -cpJf $arch build-$targetname.* payload-$targetname/*; then
  notify_ok "$scriptname" "$(langstr en 'Created archive' ru 'Создан архив') $arch.";
 else
  notify_error "$scriptname" "$(langstr en 'Failed achive' ru 'Сбой создания архива') $arch.";
 fi;
 local opt="--xz"; export OPT_XZ="-9";
 if [ "$nowait" = "1" ]; then opt="$opt --nowait"; fi;
 cp -fv $scriptbase.* $payloaddir/; mkdir payload-$targetname/payload-$targetname;
 if makeself $opt $payloaddir install-$targetname.run $targetname ./install-$targetname.sh; then
  notify_ok_apply "$scriptname" "$(langstr en 'Built installer' ru 'Создан инсталлятор ') install-$targetname.run.";
 else
  notify_error "$scriptname" "$(langstr en 'Failed build' ru 'Сбой создания инсталлятора') install-$targetname.run.";
 fi;
 rm -fv $payloaddir/$scriptbase.*; rmdir payload-$targetname/payload-$targetname;
};

note "########";
note "# Main #";
note "########";
function main(){
 about; print_identification; sleep 1;
 check_sudo_is_avail || fatal "Error: sudo is not available.";
 check_install_package zenity makeself zip unzip xz:xz-utils 7za:p7zip;
 local ask="$(zen_hello_user)\n\n$(zen_do_you_want_to $(langstr en 'Build' ru 'Создать'))\n";
 ask="$ask\n<big><b><span fgcolor='green'>install-$targetname.run</span></b>?</big>";
 if ask_zenity 0 --question --text "$ask" --timeout 180 >/dev/null 2>&1; then
  pushd $scriptHOME && build_target; popd;
  echo -ne "\nDone\n\n";
 else
  echo -ne "\nCancel\n\n";
 fi;
};

main "$@";

#############
# END OF FILE
#############
