#!/bin/bash

###########################################################
## Copyright (c) 2023-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## colorcode - print color code(s) by given color name(s).
###########################################################

# Script Identification
#######################
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

function print_version(){
 echo "$scriptname version 1.0";
};

function print_help(){
 print_version;
 echo "Copyright (c) 2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Print color code(s) by given color name(s).";
 echo "";
 echo "Syntax:     $scriptname [options] [arguments]";
 echo "";
 echo "Options:    --               => end options";
 echo "            --version        => show version";
 echo "            -h,--help        => show help screen";
 echo "            -d,--dec         => print color as decimal (default)";
 echo "            -w,--web         => print color in WEB format #rrggbb";
 echo "            -x,--hex         => print color in HEX format 0xbbggrr";
 echo "            -p,--pas         => print color in HEX format \$bbggrr";
 echo "            -l,--list        => print list of colors as 'name ggbbrr'";
 echo "";
 echo "Arguments:  list of color names to convert";
 echo "";
 echo "Notes:";
 echo " 1) If color name not found, print color name and return error code";
 echo " 2) The (rr,gg,bb) are (red,green,blue) codes in HEX (%2.2x) format";
 echo " 3) Options (-opt,--opt) are both valid, so (-dec,--dec) are equals";
 echo "";
 echo "Examples:   $scriptname --help";
 echo "            $scriptname --version";
 echo "            $scriptname --list";
 echo "            $scriptname black white";
 echo "            $scriptname -x maroon lime navy";
 echo "            $scriptname -web red green blue";
};

########################
# application variables:
########################
readonly color_table_txt="$scriptHOME/color_table.txt";
declare -i argn=0; # argument counter
declare -i nerr=0; # number of errors
declare formatid="d";  # d/w/x/p=dec/#web/0xhex/$hex

###############################
# handler for fatal errors:
# print $2 to stderr & exit $1.
###############################
function fatal(){
 1>&2 echo -ne "\n\n$2\n\n";
 exit $1;
};

#####################
# check $1 is option.
#####################
function is_option(){
 case $1 in
  -*) return 0; ;;
  *)  return 1; ;;
 esac;
};

######################
# print list of colors
# format: name ggbbrr
######################
function list_colors(){
 if [ -e $color_table_txt ]; then
  cat $color_table_txt;
 else
  fatal 1 "Error: not found $color_table_txt.";
 fi;
};

#############################
# print color code by name $1
#############################
function print_color(){
 local name="$1";
 if [ -n "$name" ]; then
  if [ ! -e $color_table_txt ]; then fatal 1 "Error: not found $color_table_txt."; fi;
  local hex="$(cat $color_table_txt | grep -i -P "^$name " | tail -n 1 | cut -d ' ' -f 2)";
  if [ ${#hex} -eq 6 ]; then
   local web="${hex:4:2}${hex:2:2}${hex:0:2}";
   local code=0; let code=0x$hex;
   case $formatid in
    d) echo "$code"; ;;
    w) echo "#$web"; ;;
    x) echo "0x$hex"; ;;
    p) echo "\$$hex"; ;;
    *) echo "$name"; let nerr++; ;;
   esac;
  else
   echo "$name"; let nerr++;
  fi;
 fi;
};

#############################
# handle positional arguments
#############################
function handle_args(){
 local arg="$1";
 if [ -n "$arg" ]; then
  print_color "$arg";
  let argn++;
 fi;
};

###############
# main function
###############
function main(){
 if [ $# -le 0 ]; then fatal 1 "Error: No arguments. See --help."; fi;
 local opt=""; local arg=""; local val=""; local isopt=1;
 while [ -n "$1" ]; do
  arg="$1"; shift;
  if is_option "$arg" && [ $isopt -gt 0 ]; then
   case $arg in
    --)                     let isopt=0; ;;
    -h|-help|--help)        print_help; exit 0; ;;
    -version|--version)     print_version; exit 0; ;;
    -d|-dec|--dec)          formatid="d"; ;;
    -p|-pas|--pas)          formatid="p"; ;;
    -x|-hex|--hex)          formatid="x"; ;;
    -w|-web|--web)          formatid="w"; ;;
    -l|-list|--list)        list_colors; exit 0; ;;
    *)                      fatal 1 "Error: unknown option $opt. See --help."; ;;
   esac;
  else
   case $opt in
    '')                     handle_args "$arg"; ;;
    *)                      fatal 1 "Error: unknown option $opt. See --help."; ;;
   esac;
   opt="";
   val="";
  fi;
 done;
 if [[ $nerr -eq 0 ]]; then return 0; else return 1; fi;
};

main "$@";

##############
## END OF FILE
##############
