#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
# Utility to find application handler (Desktop Entry).
# References:
# 1) XDG Desktop Menu Specification.
# 2) XDG Base Directory Specification.
# 3) XDG Icon Theme Specification.
# 4) XDG Icon Naming Specification.
# 5) XDG Shared MIME-info Database.
# 6) XDG Desktop Entry Specification.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

source "$scriptHOME/ftype.in";

function do_version(){
 echo "$scriptname version 1.0";
};

function do_print_help(){
 do_version;
cat<<EOF
$scriptname Copyright (c) 2025 Alexey Kuryakin daqgroup@mail.ru
Utility to find application handler by short name like 'app.desktop'.
Usage:
 $scriptname [-options] [app.desktop [params]]
Options:
 --version   - print version
 -h,--help   - print help screen
 -e,--exec   - print Exec parameter of .desktop file
 -i,--icon   - print Icon parameter of .desktop file
 -L,--Link   - follow i.e. dereference .desktop file link
 -c,--cmd    - print command line with substituted parameters
Parameters:
 app.desktop - short name of application handler, must be like *.desktop
 params      - arguments for command line substitution with option --cmd
Examples:
 $scriptname firefox.desktop         # print full file path for .desktop file
 $scriptname -i firefox.desktop      # print Icon parameter for .desktop file 
 $scriptname -e firefox.desktop      # print Exec parameter for .desktop file
 $scriptname -c firefox.desktop site # print command line to open web (site)
 $scriptname --help                  # print help screen
 $scriptname --version               # print version
EOF
};

function find_app(){
 local line="";
 local app="$1";
 if [[ -n $app ]]; then
  local fapp="$(find_app_handler $app)";
  if [[ -e $fapp ]]; then
   line="$fapp";
   if [[ $opt_exec -eq 1 ]]; then
    local exe="$(unix $readini "$fapp" "[Desktop Entry]" "Exec" | drop_key_equal)";
    if [[ -n $exe ]]; then line="$exe"; else line=""; fi;
    if [[ $opt_cmd -eq 1 ]] && [[ -n $exe ]]; then
     shift; subst_exec_args "$exe" "$@"; return;
    fi;
   fi;
   if [[ $opt_icon -eq 1 ]]; then
    local ico="$(unix $readini "$fapp" "[Desktop Entry]" "Icon" | drop_key_equal)";
    if [[ -n $ico ]]; then line="$ico"; else line=""; fi;
   fi;
  fi;
 else
  fatal 1 "Error: application is not specified.";
 fi;
 if [[ -n $line ]]; then
  echo "$line";
 else
  fatal 1 "Error: missed $app";
 fi;
};

function find_apps(){
 if [[ -z $1 ]]; then
  fatal 1 "Error: no arguments specified. Use -h option for help.";
  return;
 fi;
 while [[ -n $1 ]]; do
  case $1 in
   *.desktop) if [[ $opt_cmd -eq 0 ]]; then find_app "$1"; else find_app "$@"; break; fi; ;;
   *)         fatal 1 "Error: Bad argument: expect *.desktop but \"$1\" found."; ;;
  esac;
  shift;
 done;
};

######
# MAIN
######

function main(){
 check_readini;
 while [[ -n $1 ]]; do
  case $1 in
   --version)       do_version; return 0; ;;
   -h|-help|--help) do_print_help; return 0; ;;
   -e|-exec|--exec) let opt_exec=1; ;;
   -i|-icon|--icon) let opt_icon=1; ;;
   -L|-Link|--Link) let opt_link=1; ;;
   -c|-cmd|--cmd)   let opt_exec=1; let opt_cmd=1; ;;
   -*)              fatal 1 "Error: bad option $1"; ;;
   *)               break; ;;
  esac;
  shift;
 done;
 find_apps "$@";
};

main "$@";

##############
## END OF FILE
##############
