#!/bin/sh

function about(){
 echo "$(basename ${BASH_SOURCE[0]}) version 20230329";
 echo "##############################################################";
 echo "# Utiluty to validate (CRLF->LF) files of Diesel Pascal.     #";
 echo "# Copyright (c) 2023 Alexey Kuryakin kouriakine@mail.ru      #";
 echo "##############################################################";
 echo "# Usage: diesel-validate-eol.sh [options] [files]            #";
 echo "# Options: -d|--default to convert {../external,../examples} #";
 echo "##############################################################";
};

function note(){ return; };
function fatal(){ echo -ne "\n$1\n\n"; exit 1; };

note "Script identification.";
note "######################";
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(readlink -f $(dirname $scriptfile))";
readonly scriptFILE="$(readlink -f $scriptfile)";
readonly scriptHOME="$(readlink -f $(dirname $scriptFILE))";

note "#######################";
note "Print user/script info.";
note "#######################";
function print_identification(){
 echo "UserID: $USER@$HOSTNAME on $(lsb_release -ds)";
 echo "RunDir: $startupdir";
 echo "Locate: $scripthome";
 echo "Script: $scriptname";
};

note "############################";
note "Check packages is installed.";
note "############################";
function check_install_package(){
 while [ -n "$1" ]; do
  local program="$(echo "$1" | cut -d : -f 1)";
  local package="$(echo "$1" | cut -d : -f 2)";
  echo -n "Check $program from package $package: "
  if [ -z "$(which $program)" ]; then sudo apt install $1; fi;
  if [ -n "$(which $program)" ]; then echo "OK"; else echo "FAIL"; fatal "Error: missed package $package."; fi;
  shift;
 done;
};

note "########################";
note "Check sudo is available.";
note "########################";
function check_sudo_is_avail(){
 if sudo true; then return 0; else return 1; fi;
};

note "####################################";
note "Validate text file(s) eol (CRLF->LF)";
note "####################################";
declare -i numdone=0; declare -i numbugs=0;
function validate_textfile_eol(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then
   local f="$(readlink -f "$1")";
   local fprop="$(dos2unix -i "$f")";
   local ncrlf="$(echo "$fprop" | awk '{print $1}')";
   local numlf="$(echo "$fprop" | awk '{print $2}')";
   local numcr="$(echo "$fprop" | awk '{print $3}')";
   local isbom="$(echo "$fprop" | awk '{print $4}')";
   local ftype="$(echo "$fprop" | awk '{print $5}')";
   if [ "$ftype" = "text" ] && [ $(($ncrlf+$numcr)) -ne 0 ]; then
    if sudo dos2unix -s -q -o "$f"; then
     echo "Done:   $f";
     let numdone+=1;
    else
     echo "Failed: $f";
     let numbugs+=1;
    fi;
   else
    echo "Skip:   $f";
   fi;
  else
   echo "Missed: $f";
   let numbugs+=1;
  fi;
  shift;
 done;
 echo "$numdone file(s) done, $numbugs error(s) found.";
 if [ $numbugs -gt 0 ]; then fatal "Error: bugs found."; fi;
};

note "########";
note "# Main #";
note "########";
function main(){
 about; print_identification;
 check_sudo_is_avail || fatal "Error: sudo is not available.";
 check_install_package dos2unix;
 if [ "$1" = "-d" ] || [ "$1" = "--default" ]; then
  local list="";
  list="$list $(find $scriptHOME/../external/*.pas -type f -print)";
  list="$list $(find $scriptHOME/../external/*.md  -type f -print)";
  list="$list $(find $scriptHOME/../examples/*.lm9 -type f -print)";
  validate_textfile_eol $list;
 else
  validate_textfile_eol "$@";
 fi;
 echo; echo "Done $scriptname"; echo;
};

main "$@";

#############
# END OF FILE
#############
