////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Surf Window to plot surfaces.                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231125 - Modified for FPC (A.K.)                                         //
// 20240626 - PaintBoxPosParams                                               //
////////////////////////////////////////////////////////////////////////////////

unit form_surfwindow; // Form Surf Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, Printers,
 Form_ListBoxSelection, Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,_crw_sort,
 _crw_ef, _crw_ee, _crw_zm, _crw_curves,
 _crw_riff, _crw_delauna, _crw_colors,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSurfWindow = class(TFormCrwDaqSysChild)
    PaintBox: TPaintBox;
    ScrollBarPsi: TScrollBar;
    ScrollBarPhi: TScrollBar;
    ActionSurfToolsCloneWindow: TAction;
    ActionSurfToolsEditComment: TAction;
    ActionSurfToolsEditWindowStyle: TAction;
    ActionSurfToolsEditWindowClip: TAction;
    ActionSurfToolsExtractSlicesX: TAction;
    ActionSurfToolsExtractSlicesY: TAction;
    ActionSurfToolsSmoothing: TAction;
    MenuSurfTools: TMenuItem;
    MenuSurfToolsCloneWindow: TMenuItem;
    MenuSurfToolsEditComment: TMenuItem;
    MenuSurfToolsEditWindowStyle: TMenuItem;
    MenuSurfToolsEditWindowClip: TMenuItem;
    MenuSurfToolsExtractSlicesX: TMenuItem;
    MenuSurfToolsExtractSlicesY: TMenuItem;
    MenuSurfToolsSmoothing: TMenuItem;
    ToolButtonSurfToolsCloneWindow: TToolButton;
    ToolButtonSurfToolsEditComment: TToolButton;
    ToolButtonSurfToolsEditWindowStyle: TToolButton;
    ToolButtonSurfToolsEditWindowClip: TToolButton;
    ToolButtonSurfToolsExtractSliceX: TToolButton;
    ToolButtonSurfToolsExtractSliceY: TToolButton;
    ToolButtonSurfToolsSmoothing: TToolButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure PaintBoxPaint(Sender: TObject);
    procedure ActionSurfToolsCloneWindowExecute(Sender: TObject);
    procedure ActionSurfToolsEditCommentExecute(Sender: TObject);
    procedure ActionSurfToolsEditWindowStyleExecute(Sender: TObject);
    procedure ActionSurfToolsEditWindowClipExecute(Sender: TObject);
    procedure ActionSurfToolsExtractSlicesXExecute(Sender: TObject);
    procedure ActionSurfToolsExtractSlicesYExecute(Sender: TObject);
    procedure ActionSurfToolsSmoothingExecute(Sender: TObject);
    procedure ScrollBarChange(Sender: TObject);
  private
    { Private declarations }
    myZmatrix      : TDoubleMatrix;
    myLimits       : TRect2D;
    myClip         : TRect2D;
    myScale        : TPoint3D;
    mySliceMode    : Cardinal;
    myGroundColor  : TColor;
    myUpColor      : TColor;
    myDownColor    : TColor;
    myAxisColor    : TColor;
    myGridColor    : TColor;
    myTitleColor   : TColor;
    myLegendColor  : TColor;
    myTitle        : LongString;
    myLegend       : LongString;
    myComment      : TText;
    myPlotParams   : TPlot3DParams;
    myNClip        : TRect2I;
    myCurrNum      : TPoint2I;
    mySmartDrawer  : TSmartDrawer;
    function  GetNumPoints:TPoint2I;
    procedure SetNumPoints(const aNumPoints:TPoint2I);
    function  GetX(ix:LongInt):Double;
    function  GetY(iy:LongInt):Double;
    function  GetZ(ix,iy:LongInt):Double;
    procedure SetZ(ix,iy:LongInt; NewZ:Double);
    function  GetLimits:TRect2D;
    procedure SetLimits(const aLimits:TRect2D);
    function  GetClip:TRect2D;
    procedure SetClip(const aClip:TRect2D);
    procedure CheckClip;
    function  GetScale:TPoint3D;
    procedure SetScale(const aScale:TPoint3D);
    function  GetPsi:Double;
    procedure SetPsi(Angle:Double);
    function  GetPhi:Double;
    procedure SetPhi(Angle:Double);
    function  GetSliceMode:Cardinal;
    procedure SetSliceMode(aSliceMode:Cardinal);
    function  GetGroundColor:TColor;
    procedure SetGroundColor(aGroundColor:TColor);
    function  GetUpColor:TColor;
    procedure SetUpColor(aUpColor:TColor);
    function  GetDownColor:TColor;
    procedure SetDownColor(aDownColor:TColor);
    function  GetAxisColor:TColor;
    procedure SetAxisColor(aAxisColor:TColor);
    function  GetGridColor:TColor;
    procedure SetGridColor(aGridColor:TColor);
    function  GetTitleColor:TColor;
    procedure SetTitleColor(aTitleColor:TColor);
    function  GetLegendColor:TColor;
    procedure SetLegendColor(aLegendColor:TColor);
    function  GetTitle:LongString;
    procedure SetTitle(const aTitle:LongString);
    function  GetLegend:LongString;
    procedure SetLegend(const aLegend:LongString);
    function  GetComment:TText;
    function  GetPlotArea:TRect2I;
    function  GetIndexShift:TPoint2I;
    function  GetPointShift:TPoint2I;
    function  GetMargin:TRect2I;
    function  GetViewPointText:LongString;
  public
    { Public declarations }
    property  NumPoints:TPoint2I      read GetNumPoints   write SetNumPoints;
    property  X[ix:LongInt]:Double    read GetX;
    property  Y[iy:LongInt]:Double    read GetY;
    property  Z[ix,iy:LongInt]:Double read GetZ           write SetZ;  default;
    property  Limits:TRect2D          read GetLimits      write SetLimits;
    property  Clip:TRect2D            read GetClip        write SetClip;
    property  Scale:TPoint3D          read GetScale       write SetScale;
    property  Psi:Double              read GetPsi         write SetPsi;
    property  Phi:Double              read GetPhi         write SetPhi;
    property  SliceMode:Cardinal      read GetSliceMode   write SetSliceMode;
    property  GroundColor:TColor      read GetGroundColor write SetGroundColor;
    property  UpColor:TColor          read GetUpColor     write SetUpColor;
    property  DownColor:TColor        read GetDownColor   write SetDownColor;
    property  AxisColor:TColor        read GetAxisColor   write SetAxisColor;
    property  GridColor:TColor        read GetGridColor   write SetGridColor;
    property  TitleColor:TColor       read GetTitleColor  write SetTitleColor;
    property  LegendColor:TColor      read GetLegendColor write SetLegendColor;
    property  Title:LongString        read GetTitle       write SetTitle;
    property  Legend:LongString       read GetLegend      write SetLegend;
    property  Comment:TText           read GetComment;
    property  PlotArea:TRect2I        read GetPlotArea;
    property  IndexShift:TPoint2I     read GetIndexShift;
    property  PointShift:TPoint2I     read GetPointShift;
    property  Margin:TRect2I          read GetMargin;
    function  Convert(ax,ay,az:Double):TPoint2I;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure FileSave; override;
    procedure FileSaveAs; override;
    procedure DrawView; override;
    procedure PrintView; override;
    function  GetImage(BlackAndWhite:Boolean):TBitmap;
    function  Clone:TFormSurfWindow;
    function  PaintBoxPosParams:LongString;
  end;
  TSurfWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormSurfWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormSurfWindow);
  public
    property   Window[i:Integer]:TFormSurfWindow read GetWindow write SetWindow; default;
  end;

const
  FullSurfWindowList : TSurfWindowList = nil;
  smNumberUp                           = $0004;
  MaxSurfN                             = 256;
  Margins            : TRect2I         = (A:(X:5; Y:10); B:(X:5; Y:5));
  DefGroundColor     : TColor          = clLtGray;
  DefUpColor         : TColor          = clBlue;
  DefDownColor       : TColor          = clAqua;
  DefAxisColor       : TColor          = clBlack;
  DefGridColor       : TColor          = clBlack;
  DefTitleColor      : TColor          = clBlack;
  DefLegendColor     : TColor          = clBlack;

function  NewSurfWindow(aRows    : Integer = 60;
                        aColumns : Integer = 60;
                  const aCaption : LongString = '???';
                  const aTitle   : LongString = '???';
                  const aLegend  : LongString = '???'):TFormSurfWindow;
procedure Kill(var TheObject:TFormSurfWindow); overload;

function  NewSurfWindowList(aOwnsObjects : Boolean = true;
                            aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                            aStep        : LongInt = DefaultTObjectStorageStep
                                       ) : TSurfWindowList;
procedure Kill(var TheObject:TSurfWindowList); overload;

function  ActiveSurfWindow:TFormSurfWindow;
function  FindSurfWindow(const Caption:LongString; CreateNew:Boolean=false):TFormSurfWindow;
function  NewSurfWindowByPlotter(const NumPoints : TPoint2I;
                                 const Limits    : TRect2D;
                                 const Plotter   : TPlotter3d;
                                 const Caption   : LongString;
                                 const Title     : LongString;
                                 const Legend    : LongString):TFormSurfWindow;
function  NewSurfWindowByFormula(const NumPoints : TPoint2I;
                                 const Limits    : TRect2D;
                                 const Script    : LongString):TFormSurfWindow;
function  NewSurfWindowComposedFromCurves(Form:TForm):TFormSurfWindow;
function  SaveSurfWindowToCRW(FileName:LongString; ITEM:TFormSurfWindow):Boolean;
function  LoadSurfWindowFromCRW(FileName:LongString; const Chunk:TRiffChunk):TFormSurfWindow;

procedure ReadSurfaceWindowsPalette(const TheIniFile:LongString);

implementation

uses
 Form_CrwDaq,
 Form_SurfToolsSmoothingDialog,
 Form_SurfWindowStyleDialog,
 Form_SurfWindowClipDialog,
 Form_TextEditDialog,
 Form_CurveWindow,
 Form_SurfByFormula;

{$R *.lfm}

 {
 ******************************
 TSurfWindowList implementation
 ******************************
 }
function   TSurfWindowList.GetWindow(i:Integer):TFormSurfWindow;
begin
 Result:=TFormSurfWindow(Items[i]);
end;

procedure  TSurfWindowList.SetWindow(i:Integer; aWindow:TFormSurfWindow);
begin
 Items[i]:=aWindow;
end;

function  NewSurfWindowList(aOwnsObjects : Boolean = true;
                            aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                            aStep        : LongInt = DefaultTObjectStorageStep
                                       ) : TSurfWindowList;
begin
 Result:=nil;
 try
  Result:=TSurfWindowList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewSurfWindowList');
 end;
end;

procedure Kill(var TheObject:TSurfWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 ************************
 General purpose routines
 ************************
 }
function  NewSurfWindow(aRows    : Integer = 60;
                        aColumns : Integer = 60;
                  const aCaption : LongString = '???';
                  const aTitle   : LongString = '???';
                  const aLegend  : LongString = '???'):TFormSurfWindow;
const WinNum:Integer=0;
begin
 Result:=nil;
 try
  Application.CreateForm(TFormSurfWindow, Result);
  with Result do begin
   try
    LockDraw;
    Inc(WinNum);
    NumPoints:=Point2I(aRows,aColumns);
    if aCaption='???'
    then Caption:=RusEng('Поверхность z(x,y) ','Surface z(x,y) ')+IntToStr(WinNum)
    else Caption:=aCaption;
    if aTitle='???'
    then Title:=RusEng('Заголовок','Title')
    else Title:=aTitle;
    if aLegend='???'
    then Legend:=RusEng('Легенда','Legend')
    else Legend:=aLegend;
    StatusBar.Hide;
    ScrollBarPhi.Hide;
    StatusBar.Show;
    ScrollBarPhi.Show;
   finally
    UnlockDraw;
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure Kill(var TheObject:TFormSurfWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end;
end;

function ActiveSurfWindow:TFormSurfWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormSurfWindow) then Result:=TFormSurfWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveSurfWindow');
 end;
end;

type
 TFindRec = packed record
  Match   : TFormSurfWindow;
  Caption : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TFindRec(Custom^) do
 if (Form is TFormSurfWindow) and SameText(Form.Caption,Caption) then begin
  Match:=(Form as TFormSurfWindow);
  Terminate:=true;
 end;
end;

function FindSurfWindow(const Caption:LongString; CreateNew:Boolean=false):TFormSurfWindow;
var FindRec:TFindRec;
begin
 FindRec:=Default(TFindRec);
 try
  FindRec.Match:=nil;
  FindRec.Caption:=StringBuffer(Caption);
  if (Caption<>'') then SdiMan.ForEachChild(FindForm,@FindRec);
  Result:=FindRec.Match;
  if CreateNew and not Assigned(Result) then Result:=NewSurfWindow;
 finally
  FindRec.Caption:='';
 end;
end;

function NewSurfWindowByFormula(const NumPoints : TPoint2I;
                                const Limits    : TRect2D;
                                const Script    : LongString):TFormSurfWindow;
var ee:TExpressionEvaluator; i,j:LongInt; x,y,z:Double; Success:Boolean;
const WinNum:Integer=0;
begin
 Result:=nil;
 try
  Inc(WinNum);
  Result:=NewSurfWindow(NumPoints.X,NumPoints.Y,
                        RusEng('Поверхность z(x,y) по формуле ','Surface z(x,y) by formula ')+IntToStr(WinNum),
                        RusEng('Заголовок','Title'),
                        RusEng('Легенда','Legend'));
  if Result.Ok then                        
  try
   Result.LockDraw;
   Result.Limits:=Limits;
   Result.Comment.Text:=RusEng('Источник:скрипт','Source:Script')+EOL+Script;
   ee:=NewExpressionEvaluator;
   try
    ee.Script:=Script;
    Success:=true;
    for i:=0 to NumPoints.X-1 do begin
     x:=Limits.A.X + i * RectSizeX(Limits) / (NumPoints.X-1);
     for j:=0 to NumPoints.Y-1 do begin
      y:=Limits.A.Y + j * RectSizeY(Limits) / (NumPoints.Y-1);
      Success:=Success and ee.SetValue('x',x);
      Success:=Success and ee.SetValue('y',y);
      Success:=Success and (ee.RunScript=ee_Ok);
      Success:=Success and ee.GetValue('z',z);
      Success:=Success and not isNanOrInf(z);
      if Success then Result[i,j]:=z else begin
       Kill(Result);
       break;
      end;
     end;
     if not Success then break;
    end;
   finally
    Kill(ee);
   end;
  finally
   Result.UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function NewSurfWindowByPlotter(const NumPoints : TPoint2I;
                                const Limits    : TRect2D;
                                const Plotter   : TPlotter3d;
                                const Caption   : LongString;
                                const Title     : LongString;
                                const Legend    : LongString):TFormSurfWindow;
var
 i,j   : LongInt;
 x,y,z : Double;
begin
 Result:=nil;
 if Plotter.Ready then
 try
  Result:=NewSurfWindow(NumPoints.X,NumPoints.Y,Caption,Title,Legend);
  if Result.Ok then
  try
   Result.LockDraw;
   Result.Limits:=Limits;
   for i:=0 to NumPoints.X-1 do begin
    x:=Limits.A.X + i * RectSizeX(Limits) / (NumPoints.X-1);
    for j:=0 to NumPoints.Y-1 do begin
     y:=Limits.A.Y + j * RectSizeY(Limits) / (NumPoints.Y-1);
     z:=Plotter.Interpolate(x,y);
     if isNanOrInf(z) then Kill(Result) else Result[i,j]:=z;
     if not Result.Ok then break;
    end;
    if not Result.Ok then break;
   end;
  finally
   Result.UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function NewSurfWindowComposedFromCurves(Form:TForm):TFormSurfWindow;
const FailsFlags = cfInvalid+cfNanInf+cfTooSmall+cfNotSortedX+cfDuplicatesX;
var
 First,Last:TCurve; Lim:TRect2D; Count,i,j,m:LongInt;
 buf:TParsingBuffer; CurveWindow:TFormCurveWindow;
begin
 Result:=nil;
 try
  if Form is TFormCurveWindow then CurveWindow:=Form as TFormCurveWindow else CurveWindow:=nil;
  if CurveWindow.Ok and not CurveWindow.CheckIsDataProtected then begin
   Count:=CurveWindow.Curves.Count;
   First:=CurveWindow.Curves[0];
   Last:=CurveWindow.Curves[Count-1];
   if Trouble(Count<2,RusEng('Слишком мало кривых!','No enough curves!')) then Exit;
   for i:=0 to Count-1 do
   if CurveWindow.Curves[i].Flags and FailsFlags <> 0 then begin
    Error(RusEng('Недопустимые входные данные!','Invalid input data!'));
    Exit;
   end;
   m:=0;
   Lim:=Rect2D(0,0,0,0);
   if (ScanVarDouble(svUpCase,StrCopyBuff(buf,UnifyAlias(First.Name)),'SLICEX#%f',Lim.A.X)<>nil) and
      (ScanVarDouble(svUpCase,StrCopyBuff(buf,UnifyAlias(Last.Name )),'SLICEX#%f',Lim.B.X)<>nil)
   then begin
    m:=smSliceX;
    with Last.Limits do begin
     Lim.A.Y:=A.X;
     Lim.B.Y:=B.X;
    end;
   end;
   if (ScanVarDouble(svUpCase,StrCopyBuff(buf,UnifyAlias(First.Name)),'SLICEY#%f',Lim.A.Y)<>nil) and
      (ScanVarDouble(svUpCase,StrCopyBuff(buf,UnifyAlias(Last.Name )),'SLICEY#%f',Lim.B.Y)<>nil)
   then begin
    m:=smSliceY;
    with Last.Limits do begin
     Lim.A.X:=A.X;
     Lim.B.X:=B.X;
    end;
   end;
   if (m=0) or RectIsEmpty(Lim) then begin
    Error(RusEng('Недопустимые входные данные!','Invalid input data!'));
    Exit;
   end;
   case m of
    smSliceX: begin
               Result:=NewSurfWindow(Count,Last.Count,RusEng('Поверхность из X-сечений','Surface composed from X-slices'));
               Result.Limits:=Lim;
               for i:=0 to Result.NumPoints.X-1 do
               for j:=0 to Result.NumPoints.Y-1 do
               Result[i,j]:=CurveWindow.Curves[i].Interpolate(Lim.A.Y+j*RectSizeY(Lim)/(Result.NumPoints.Y-1));
              end;
    smSliceY: begin
               Result:=NewSurfWindow(Last.Count,Count,RusEng('Поверхность из Y-сечений','Surface composed from Y-slices'));
               Result.Limits:=Lim;
               for i:=0 to Result.NumPoints.X-1 do
               for j:=0 to Result.NumPoints.Y-1 do
               Result[i,j]:=CurveWindow.Curves[j].Interpolate(Lim.A.X+i*RectSizeX(Lim)/(Result.NumPoints.X-1));
              end;
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function SaveSurfWindowToCRW(FileName:LongString; ITEM:TFormSurfWindow):Boolean;
var Filer:TRiffFiler; Form,Chunk:TRiffChunk; i,j:LongInt;
begin
 Result:=false;
 FileName:=UnifyFileAlias(FileName,ua_FileDefLow);
 Filer:=NewRiffFiler(FileName,fmOpenReadWrite,sig_CRWF);
 if Filer.Ok then
 try
  {
  seek to end of file and create form ITEM CRVW
  }
  Filer.Validate;
  Filer.SeekChunkNext(Filer.RootForm);
  Form:=Filer.CreateForm(sig_ITEM,sig_SURW);
   {
   Save ITEM name - window name
   }
   Chunk:=Filer.CreateChunk(sig_name);
   Filer.WriteString(ITEM.Caption);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM time - save time
   }
   Chunk:=Filer.CreateChunk(sig_time);
   Filer.WriteDouble(msecnow);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM vers - version
   }
   Chunk:=Filer.CreateChunk(sig_vers);
   Filer.WriteLongInt(sig_0001);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM ltwh - left,top,width,height
   }
   Chunk:=Filer.CreateChunk(sig_ltwh);
   Filer.WriteLongInt(ITEM.Left);
   Filer.WriteLongInt(ITEM.Top);
   Filer.WriteLongInt(ITEM.Width);
   Filer.WriteLongInt(ITEM.Height);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM titl - Title
   }
   Chunk:=Filer.CreateChunk(sig_titl);
   Filer.WriteString(ITEM.Title);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM lgnd - Legend
   }
   Chunk:=Filer.CreateChunk(sig_lgnd);
   Filer.WriteString(ITEM.Legend);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM text - Comment
   }
   Chunk:=Filer.CreateChunk(sig_text);
   Filer.WriteString(ITEM.Comment.Text);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM clrp - Color palette
   }
   Chunk:=Filer.CreateChunk(sig_clrp);
   Filer.WriteLongInt(ITEM.GroundColor);
   Filer.WriteLongInt(ITEM.UpColor);
   Filer.WriteLongInt(ITEM.DownColor);
   Filer.WriteLongInt(ITEM.AxisColor);
   Filer.WriteLongInt(ITEM.GridColor);
   Filer.WriteLongInt(ITEM.TitleColor);
   Filer.WriteLongInt(ITEM.LegendColor);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM lims - Limits
   }
   Chunk:=Filer.CreateChunk(sig_lims);
   Filer.WriteRect2D(ITEM.Limits);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM clip  - Clip
   }
   Chunk:=Filer.CreateChunk(sig_clip);
   Filer.WriteRect2D(ITEM.Clip);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM data  - NumPoints + Z[i,j] array
   }
   Chunk:=Filer.CreateChunk(sig_data);
   Filer.WritePoint2I(ITEM.NumPoints);
   for i:=0 to ITEM.NumPoints.X-1 do
   for j:=0 to ITEM.NumPoints.Y-1 do Filer.WriteDouble(ITEM[i,j]);
   Filer.FlushChunk(Chunk);
   {
   Save ITEM view  - Scale,Phi,Psi,SliceMode
   }
   Chunk:=Filer.CreateChunk(sig_view);
   Filer.WriteDouble(ITEM.Scale.X);
   Filer.WriteDouble(ITEM.Scale.Y);
   Filer.WriteDouble(ITEM.Scale.Z);
   Filer.WriteDouble(ITEM.Phi);
   Filer.WriteDouble(ITEM.Psi);
   Filer.WriteLongInt(ITEM.SliceMode);
   Filer.FlushChunk(Chunk);
  {
  В конце записи формы надо обновить ее заголовок вызовом FlushChunk.
  }
  Filer.FlushChunk(Form);
  Filer.Flush;
  Result:=true;
  SendToMainConsole('@silent @integrity save.crw '+FileName+EOL);
 except
  on EReadError  do begin
   Filer.Modified:=false;
   Echo(RusEng('Ошибка чтения файла ','Error reading file ')+FileName);
  end;
  on EWriteError do begin
   Filer.Modified:=false;
   Echo(RusEng('Ошибка записи файла ','Error writing file ')+FileName);
  end;
  on E:Exception do begin
   Filer.Modified:=false;
   BugReport(E);
  end;
 end else Echo(RusEng('Ошибка открытия файла ','Error open file ')+FileName);
 Kill(Filer);
end;

type
 TScanRec  = packed record
  Template : TRiffChunk;
  Found    : Boolean;
 end;

procedure ScanChunk(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
begin
 with TScanRec(Custom^) do
 if (Template.dwSign   = Chunk.dwSign) and
    (Template.dwSize   = Chunk.dwSize) and
    (Template.dwFormID = Chunk.dwFormID) and
    (Template.dwOffset = Chunk.dwOffset)
 then begin
  Found:=true;
  Terminate:=true;
 end;
end;

procedure LoadSurfWindow(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
var
 Window : TFormSurfWindow absolute Custom;
 r,r1   : TRect2I;
 i,j,left,top,width,height : LongInt;
 Scale  : TPoint3D;
begin
 if Window.Ok then
 case Chunk.dwSign of
  sig_name : Window.Caption:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_ltwh : begin
              left:=Filer.ReadLongInt;
              top:=Filer.ReadLongInt;
              width:=Filer.ReadLongInt;
              height:=Filer.ReadLongInt;
              r:=Rect2I(left,top,left+width,top+height);
              if Assigned(FormCrwDaq)
              then r1:=TRect2I(FormCrwDaq.ClientRect)
              else r1:=r;
              if RectIsEqual(r1,RectUnion(r1,r)) then begin
               Window.Left:=Left;
               Window.Top:=top;
               Window.Width:=width;
               Window.Height:=height;
              end;
             end; 
  sig_titl : Window.Title:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_lgnd : Window.Legend:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_text : Window.Comment.Text:=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  sig_clrp : begin
              Window.GroundColor:=Filer.ReadLongInt;
              Window.UpColor:=Filer.ReadLongInt;
              Window.DownColor:=Filer.ReadLongInt;
              Window.AxisColor:=Filer.ReadLongInt;
              Window.GridColor:=Filer.ReadLongInt;
              Window.TitleColor:=Filer.ReadLongInt;
              Window.LegendColor:=Filer.ReadLongInt;
             end;
  sig_lims : Window.Limits:=Filer.ReadRect2D;
  sig_clip : Window.Clip:=Filer.ReadRect2D;
  sig_data : begin
              Window.NumPoints:=Filer.ReadPoint2I;
              for i:=0 to Window.NumPoints.X-1 do
              for j:=0 to Window.NumPoints.Y-1 do Window[i,j]:=Filer.ReadDouble;
             end;
  sig_view : begin
              Scale.X:=Filer.ReadDouble;
              Scale.Y:=Filer.ReadDouble;
              Scale.Z:=Filer.ReadDouble;
              Window.Scale:=Scale;
              Window.Phi:=Filer.ReadDouble;
              Window.Psi:=Filer.ReadDouble;
              Window.SliceMode:=Filer.ReadLongInt;
             end;
 end;
end;

function LoadSurfWindowFromCRW(FileName:LongString; const Chunk:TRiffChunk):TFormSurfWindow;
var
 Filer   : TRiffFiler;
 ScanRec : TScanRec;
begin
 Result:=nil;
 FileName:=UnifyFileAlias(FileName);
 if FileExists(FileName) and (Chunk.dwSign=sig_ITEM) and (Chunk.dwFormID=sig_SURW) then begin
  Filer:=NewRiffFiler(FileName,fmOpenReadWrite,sig_CRWF);
  if Filer.Ok then
  try
   Filer.Validate;
   ScanRec.Template:=Chunk;
   ScanRec.Found:=false;
   Filer.ForEach(ScanChunk,Filer.RootForm,@ScanRec);
   if ScanRec.Found then begin
    Result:=NewSurfWindow;
    try
     Result.LockDraw;
     Filer.ForEach(LoadSurfWindow,Chunk,Result);
    finally
     Result.UnlockDraw;
    end;
   end;
   SendToMainConsole('@silent @integrity load.crw '+FileName+EOL);
  except
   on EReadError  do begin
    Kill(Result);
    Filer.Modified:=false;
    Echo(RusEng('Ошибка чтения файла ','Error reading file ')+FileName);
   end;
   on EWriteError do begin
    Kill(Result);
    Filer.Modified:=false;
    Echo(RusEng('Ошибка записи файла ','Error writing file ')+FileName);
   end;
   on E:Exception do begin
    Kill(Result);
    Filer.Modified:=false;
    BugReport(E);
   end;
  end else Echo(RusEng('Ошибка открытия файла ','Error open file ')+FileName);
  Kill(Filer);
 end;
end;

procedure ReadSurfaceWindowsPalette(const TheIniFile:LongString);
 function ReadColor(const ColName:LongString; DefColor:TColor):TColor;
 var s:LongString;
 begin
  Result:=DefColor;
  try
   s:='';
   if ReadIniFileAlpha(TheIniFile,'[Surface Windows Palette]',ColName+'%a',s)
   or ReadIniFileAlpha(SysIniFile,'[Surface Windows Palette]',ColName+'%a',s)
   then Result:=StringToColor(s);
  except
   on E:Exception do Result:=DefColor;
  end;
 end;
begin
 DefGroundColor := ReadColor('GroundColor', DefGroundColor);
 DefUpColor     := ReadColor('UpColor',     DefUpColor);
 DefDownColor   := ReadColor('DownColor',   DefDownColor);
 DefAxisColor   := ReadColor('AxisColor',   DefAxisColor);
 DefGridColor   := ReadColor('GridColor',   DefGridColor);
 DefTitleColor  := ReadColor('TitleColor',  DefTitleColor);
 DefLegendColor := ReadColor('LegendColor', DefLegendColor);
end;

 {
 *******************************************************************************
 TFormSurfWindow implementation
 *******************************************************************************
 }
function TFormSurfWindow.GetNumPoints:TPoint2I;
begin
 if Assigned(Self)
 then Result:=Point2I(myZmatrix.Rows, myZmatrix.Columns)
 else Result:=Point2I(0,0);
end;

procedure TFormSurfWindow.SetNumPoints(const aNumPoints:TPoint2I);
begin
 if Assigned(Self) then begin
  try
   LockDraw;
   myZmatrix.Rows:=aNumPoints.X;
   myZmatrix.Columns:=aNumPoints.Y;
   CheckClip;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetX(ix:LongInt):Double;
begin
 Result:=0;
 if Assigned(Self)
 then with Limits do Result:=A.X+ix*(B.X-A.X)/(NumPoints.X-1);
end;

function  TFormSurfWindow.GetY(iy:LongInt):Double;
begin
 Result:=0;
 if Assigned(Self)
 then with Limits do Result:=A.Y+iy*(B.Y-A.Y)/(NumPoints.Y-1);
end;

function  TFormSurfWindow.GetZ(ix,iy:LongInt):Double;
begin
 if Assigned(Self) then Result:=myZmatrix[ix,iy] else Result:=0;
end;

procedure TFormSurfWindow.SetZ(ix,iy:LongInt; NewZ:Double);
begin
 if Assigned(Self) then myZmatrix[ix,iy]:=NewZ;
end;

function TFormSurfWindow.GetLimits:TRect2D;
begin
 if Ok then Result:=myLimits else Result:=Rect2D(0,0,0,0);
end;

procedure TFormSurfWindow.SetLimits(const aLimits:TRect2D);
begin
 if Ok then begin
  try
   LockDraw;
   myLimits:=RectValidate(aLimits);
   if myLimits.B.X=myLimits.A.X then myLimits.B.X:=myLimits.A.X+1;
   if myLimits.B.Y=myLimits.A.Y then myLimits.B.Y:=myLimits.A.Y+1;
   myClip:=myLimits;
   CheckClip;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormSurfWindow.GetClip:TRect2D;
begin
 if Ok then Result:=myClip else Result:=Rect2D(0,0,0,0);
end;

procedure TFormSurfWindow.SetClip(const aClip:TRect2D);
begin
 if Ok then begin
  try
   LockDraw;
   myClip:=aClip;
   CheckClip;
  finally
   UnlockDraw;
  end;
 end;
end;

procedure TFormSurfWindow.CheckClip;
var D:TPoint2D;
begin
 if Ok then begin
  D:=RectSize(Limits);
  myNClip:=Rect2I(max(0,             round((NumPoints.X-1)*((myClip.A.X-myLimits.A.X)/D.X))),
                  max(0,             round((NumPoints.Y-1)*((myClip.A.Y-myLimits.A.Y)/D.Y))),
                  min(NumPoints.X-1, round((NumPoints.X-1)*((myClip.B.X-myLimits.A.X)/D.X))),
                  min(NumPoints.Y-1, round((NumPoints.Y-1)*((myClip.B.Y-myLimits.A.Y)/D.Y))));
  myCurrNum:=Point2I(RectSizeX(myNClip)+1,RectSizeY(myNClip)+1);
  if (myCurrNum.X<2) or (myCurrNum.Y<2) or (myCurrNum.X>MaxSurfN) or (myCurrNum.Y>MaxSurfN) then begin
   myClip:=myLimits;
   myNClip:=Rect2I(0, 0, NumPoints.X-1, NumPoints.Y-1);
   myCurrNum:=NumPoints;
  end else begin
   myClip:=Rect2D(myLimits.A.X+(myNClip.A.X*D.X)/(NumPoints.X-1),
                  myLimits.A.Y+(myNClip.A.Y*D.Y)/(NumPoints.Y-1),
                  myLimits.A.X+(myNClip.B.X*D.X)/(NumPoints.X-1),
                  myLimits.A.Y+(myNClip.B.y*D.Y)/(NumPoints.Y-1));
  end;
 end;
end;

function TFormSurfWindow.GetScale:TPoint3D;
begin
 if Ok then Result:=myScale else Result:=Point3D(1,1,1);
end;

procedure TFormSurfWindow.SetScale(const aScale:TPoint3D);
begin
 if Ok then begin
  try
   LockDraw;
   myScale:=aScale;
   if myScale.X<=0 then myScale.X:=1;
   if myScale.Y<=0 then myScale.Y:=1;
   if myScale.Z<=0 then myScale.Z:=1;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetPsi:double;
begin
 if Ok then Result:=DegToRad(ScrollBarPsi.Position/60) else Result:=0;
end;

procedure TFormSurfWindow.SetPsi(Angle:double);
begin
 if Ok then begin
  try
   LockDraw;
   ScrollBarPsi.Position:=round(RadToDeg(Angle)*60);
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetPhi:double;
begin
 if Ok then Result:=DegToRad(ScrollBarPhi.Position/60) else Result:=0;
end;

procedure TFormSurfWindow.SetPhi(Angle:double);
begin
 if Ok then begin
  try
   LockDraw;
   ScrollBarPhi.Position:=round(RadToDeg(Angle)*60);
  finally
   UnlockDraw;
  end;
 end; 
end;

function  TFormSurfWindow.GetSliceMode:Cardinal;
begin
 if Ok then Result:=mySliceMode else Result:=0;
end;

procedure TFormSurfWindow.SetSliceMode(aSliceMode:Cardinal);
begin
 if Ok then begin
  try
   LockDraw;
   mySliceMode:=aSliceMode;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetGroundColor:TColor;
begin
 if Ok then Result:=myGroundColor else Result:=0;
end;

procedure TFormSurfWindow.SetGroundColor(aGroundColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myGroundColor:=aGroundColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetUpColor:TColor;
begin
 if Ok then Result:=myUpColor else Result:=0;
end;

procedure TFormSurfWindow.SetUpColor(aUpColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myUpColor:=aUpColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetDownColor:TColor;
begin
 if Ok then Result:=myDownColor else Result:=0;
end;

procedure TFormSurfWindow.SetDownColor(aDownColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myDownColor:=aDownColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetAxisColor:TColor;
begin
 if Ok then Result:=myAxisColor else Result:=0;
end;

procedure TFormSurfWindow.SetAxisColor(aAxisColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myAxisColor:=aAxisColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetGridColor:TColor;
begin
 if Ok then Result:=myGridColor else Result:=0;
end;

procedure TFormSurfWindow.SetGridColor(aGridColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myGridColor:=aGridColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetTitleColor:TColor;
begin
 if Ok then Result:=myTitleColor else Result:=0;
end;

procedure TFormSurfWindow.SetTitleColor(aTitleColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myTitleColor:=aTitleColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetLegendColor:TColor;
begin
 if Ok then Result:=myLegendColor else Result:=0;
end;

procedure TFormSurfWindow.SetLegendColor(aLegendColor:TColor);
begin
 if Ok then begin
  try
   LockDraw;
   myLegendColor:=aLegendColor;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetTitle:LongString;
begin
 if Ok then Result:=myTitle else Result:='';
end;

procedure TFormSurfWindow.SetTitle(const aTitle:LongString);
begin
 if Ok then begin
  try
   LockDraw;
   myTitle:=aTitle;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetLegend:LongString;
begin
 if Ok then Result:=myLegend else Result:='';
end;

procedure TFormSurfWindow.SetLegend(const aLegend:LongString);
begin
 if Ok then begin
  try
   LockDraw;
   myLegend:=aLegend;
  finally
   UnlockDraw;
  end;
 end;
end;

function  TFormSurfWindow.GetComment:TText;
begin
 if Ok then Result:=myComment else Result:=nil;
end;

function TFormSurfWindow.GetViewPointText:LongString;
var PsiGrad,PsiMin,PhiGrad,PhiMin:LongInt;
begin
 if Ok then begin
  PsiGrad:=ScrollBarPsi.Position div 60;
  PsiMin :=ScrollBarPsi.Position mod 60;
  PhiGrad:=ScrollBarPhi.Position div 60;
  PhiMin :=ScrollBarPhi.Position mod 60;
  Result:=Format(RusEng(
   ' Угол:%3.3d°%2.2d'',%3.3d°%2.2d''  Диап: %.8g<=X<=%.8g; %.8g<=Y<=%.8g ',
   ' View:%3.3d°%2.2d'',%3.3d°%2.2d''  Clip: %.8g<=X<=%.8g; %.8g<=Y<=%.8g '),
   [PsiGrad, PsiMin, PhiGrad, PhiMin, myClip.A.X, myClip.B.X, myClip.A.Y, myClip.B.Y]);
 end else Result:='';
end;

function  TFormSurfWindow.GetPlotArea:TRect2I;
begin
 if Ok then Result:=Rect2I(0, 0, PaintBox.Width, PaintBox.Height) else Result:=Rect2I(0,0,0,0);
end;

function  TFormSurfWindow.GetIndexShift:TPoint2I;
begin
 if Assigned(Self) then Result:=myNClip.A else Result:=Point2I(0,0);
end;

function  TFormSurfWindow.GetPointShift:TPoint2I;
begin
 if Assigned(Self) then begin
  Result.X:=0;
  Result.Y:=PaintBox.Height-RectSizeY(Margin);
 end else Result:=Point2I(0,0); 
end;

function  TFormSurfWindow.GetMargin:TRect2I;
var H:LongInt;
begin
 if Assigned(Self) then begin
  H:=abs(PaintBox.Canvas.Font.Height);
  Result:=Rect2I(Margins.A.X+1, Margins.A.Y+1+H*3, Margins.B.X+1, Margins.B.Y+1+H*3);
 end else Result:=Rect2I(0,0,0,0); 
end;

function TFormSurfWindow.Convert(ax,ay,az:Double):TPoint2I;
var EyeX,EyeY:double;
begin
 if Assigned(Self) then begin
  ax:=ax*Scale.X;
  ay:=ay*Scale.Y;
  az:=az*Scale.Z;
  EyeX:=ax*myPlotParams.e[1][1]+ay*myPlotParams.e[1][2];
  EyeY:=ax*myPlotParams.e[2][1]+ay*myPlotParams.e[2][2]+az*myPlotParams.e[2][3];
  Result.X:=round(myPlotParams.a.x+myPlotParams.b.x*EyeX);
  Result.Y:=round(myPlotParams.a.y+myPlotParams.b.y*EyeY);
  Result.X:=PointShift.X+Result.X;
  Result.Y:=PointShift.Y-Result.Y;
 end else Result:=Point2I(0,0);
end;

procedure TFormSurfWindow.UpdateCommands;
var Exposed:Boolean;
begin
 inherited UpdateCommands;
 Exposed:=FormIsExposed(Self);
 ActionFileSave.Enabled:=Exposed;
 ActionFileSaveAs.Enabled:=Exposed;
 ActionFilePrint.Enabled:=Exposed;
 ActionSurfToolsCloneWindow.Enabled:=Exposed;
 ActionSurfToolsEditComment.Enabled:=Exposed;
 ActionSurfToolsEditWindowStyle.Enabled:=Exposed;
 ActionSurfToolsEditWindowClip.Enabled:=Exposed;
 ActionSurfToolsExtractSlicesX.Enabled:=Exposed;
 ActionSurfToolsExtractSlicesY.Enabled:=Exposed;
 ActionSurfToolsSmoothing.Enabled:=Exposed;
end;

function TheSurfMatrix(ix,iy:Integer; Custom:Pointer):double;
begin
 with TFormSurfWindow(Custom) do Result:=Z[ix+IndexShift.X, iy+IndexShift.Y]
end;

procedure TheSurfPutPixel(ix,iy,aColor:Integer; Custom:Pointer);
begin
 with TFormSurfWindow(Custom) do
 PaintBox.Canvas.Pixels[PointShift.X+ix, PointShift.Y-iy]:=aColor;
end;

procedure TheEmptyPutPixel(ix,iy,aColor:Integer; Custom:Pointer);
begin
 // this stub uses to calculate limits
end;

procedure TFormSurfWindow.DrawView;
const
 am_X     = $01;
 am_Y     = $02;
 am_Z     = $04;
 am_XN    = $08;
 am_YN    = $10;
 am_ZN    = $20;
 am_First = $40;
 am_Last  = $80;
 procedure DrawAxis(Flags:Cardinal);
 const
  d1=8; d2=4; mind=8; RadX=2; RadY=2; dpx=3; dpy=3; dpz=2;
  Spc:TPoint2I=(X:1;Y:1);
 var
  R2:TRect2I;
  Msg:LongString;
  p,ex,ey,ez,e0:TPoint2I;
  x,y,z,x1,x2,y1,y2,z1,z2:double;
  i:LongInt;
  xAxis,yAxis,zAxis:TAxisGrid;
  ShiftX,ShiftY:boolean;
  {check text justification}
  procedure  ShiftCorrection;
  begin
   if ShiftX
   then inc(p.x,RadX+Spc.X+1)
   else dec(p.x,RadX+Spc.X+1+PaintBox.Canvas.TextWidth(Msg));
   if ShiftY
   then inc(p.y,RadY+Spc.Y+1)
   else dec(p.y,RadY+Spc.Y+1+abs(PaintBox.Canvas.TextHeight(Msg)));
 end;
 begin
  if Flags and am_First <> 0 then mySmartDrawer.StartDraw;
  with myPlotParams do begin
   e0:=Convert(Lim.a.x,Lim.a.y,zmin);
   ex:=Convert(Lim.b.x,Lim.a.y,zmin);
   ey:=Convert(Lim.a.x,Lim.b.y,zmin);
   ez:=Convert(Lim.a.x,Lim.a.y,zmax);
   x1:=Lim.a.x;
   x2:=Lim.b.x;
   y1:=Lim.a.y;
   y2:=Lim.b.y;
   z1:=zmin;
   z2:=zmax;
  end;
  if Flags and (am_Z+am_ZN) <> 0 then begin
   if AxisColor<>GroundColor then
   if Flags and am_Z <> 0 then DrawArrow(PaintBox.Canvas,e0,ez,AxisColor,psSolid,pmCopy,1,d1,d2);
   if Flags and am_ZN <> 0 then
   if (abs(ez.x-e0.x)>mind) or (abs(ez.y-e0.y)>mind) then begin
    zAxis:=AxisGrid(z1,z2);
    Msg:='Z';
    if zAxis.Pow<>0 then Msg:=Msg+'*1e'+d2s(zAxis.Pow);
    p.x:=ez.x-PaintBox.Canvas.TextWidth(Msg) shr 1;
    p.x:=max(p.x,Spc.X+Margins.A.X);
    p.x:=min(p.x,PaintBox.Width-PaintBox.Canvas.TextWidth(Msg)-Spc.X-Margins.B.X);
    p.y:=ez.y-abs(PaintBox.Canvas.TextHeight(Msg))-Spc.Y-dpz;
    if AxisColor<>GroundColor then
    mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,AxisColor,GroundColor,Spc.X,Spc.Y);
    if GridColor<>GroundColor then begin
     for i:=0 to zAxis.Num do begin
      x:=x1;
      y:=y1;
      z:=zAxis.Start+i*zAxis.Step;
      if (z<=z1+zAxis.Step*0.5) or (z>=z2-zAxis.Step*0.5) then continue;
      p:=Convert(x,y,z);
      R2.A.X:=p.x-RadX;
      R2.B.X:=p.x+RadX;
      R2.A.Y:=p.y-RadY;
      R2.B.Y:=p.y+RadY;
      DrawBar(PaintBox.Canvas,R2,GridColor);
      Msg:=Format('%.8g',[z/zAxis.Scale]);
      if p.x<PaintBox.Width shr 1
      then inc(p.x,RadX+Spc.X+1)
      else dec(p.x,RadX+Spc.X+1+PaintBox.Canvas.TextWidth(Msg));
      dec(p.y,abs(PaintBox.Canvas.TextHeight(Msg)) shr 1);
      mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,GridColor,GroundColor,Spc.X,Spc.Y);
     end;
    end;
   end;
  end;
  if Flags and (am_X+am_XN) <>0 then begin
   if AxisColor<>GroundColor then
   if Flags and am_X <> 0 then DrawArrow(PaintBox.Canvas,e0,ex,AxisColor,psSolid,pmCopy,1,d1,d2);
   if Flags and am_XN <> 0 then
   if (abs(ex.x-e0.x)>mind) or (abs(ex.y-e0.y)>mind) then begin
    xAxis:=AxisGrid(x1,x2);
    Msg:='X';
    if xAxis.Pow<>0 then Msg:=Msg+'*1e'+d2s(xAxis.Pow);
    p.x:=ex.x-PaintBox.Canvas.TextWidth(Msg) shr 1;
    p.x:=max(p.x,Spc.X+Margins.A.X);
    p.x:=min(p.x,PaintBox.Width-PaintBox.Canvas.TextWidth(Msg)-Spc.X-Margins.B.X);
    if ex.y>e0.y
     then p.y:=ex.y+Spc.Y+dpx
     else p.y:=ex.y-abs(PaintBox.Canvas.TextHeight(Msg))-Spc.Y-dpx;
    if AxisColor<>GroundColor then
    mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,AxisColor,GroundColor,Spc.X,Spc.Y);
    if GridColor<>GroundColor then begin
     for i:=0 to xAxis.Num do begin
      x:=xAxis.Start+i*xAxis.Step;
      y:=y1;
      z:=z1;
      if (x<=x1+xAxis.Step*0.5) or (x>=x2-xAxis.Step*0.5) then continue;
      P:=Convert(x,y,z);
      R2.A.X:=p.x-RadX;
      R2.B.X:=p.x+RadX;
      R2.A.Y:=p.y-RadY;
      R2.B.Y:=p.y+RadY;
      DrawBar(PaintBox.Canvas,R2,GridColor);
      Msg:=Format('%.8g',[x/xAxis.Scale]);
      ShiftX:=(myPlotParams.SinPhi>0);
      if ShiftX then ShiftY:=(ex.y-e0.y)*(ex.x-e0.x)<0
                else ShiftY:=(ex.y-e0.y)*(ex.x-e0.x)>0;
      ShiftCorrection;
      mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,GridColor,GroundColor,Spc.X,Spc.Y);
     end;
    end;
   end;
  end;
  if Flags and (am_Y+am_YN) <>0 then begin
   if AxisColor<>GroundColor then
   if Flags and am_Y <> 0 then DrawArrow(PaintBox.Canvas,e0,ey,AxisColor,psSolid,pmCopy,1,d1,d2);
   if Flags and am_YN <> 0 then
   if (abs(ey.x-e0.x)>mind) or (abs(ey.y-e0.y)>mind) then begin
    yAxis:=AxisGrid(y1,y2);
    Msg:='Y';
    if yAxis.Pow<>0 then Msg:=Msg+'*1e'+d2s(yAxis.Pow);
    p.x:=ey.x-PaintBox.Canvas.TextWidth(Msg) shr 1;
    p.x:=max(p.x,Spc.X+Margins.A.X);
    p.x:=min(p.x,PaintBox.Width-PaintBox.Canvas.TextWidth(Msg)-Spc.X-Margins.B.X);
    if ey.y>e0.y
     then p.y:=ey.y+Spc.Y+dpy
     else p.y:=ey.y-PaintBox.Canvas.TextHeight(Msg)-Spc.Y-dpy;
    if AxisColor<>GroundColor then
    mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,AxisColor,GroundColor,Spc.X,Spc.Y);
    if GridColor<>GroundColor then begin
     for i:=0 to yAxis.Num do begin
      x:=x1;
      y:=yAxis.Start+i*yAxis.Step;
      z:=z1;
      if (y<=y1+yAxis.Step*0.5) or (y>=y2-yAxis.Step*0.5) then continue;
      p:=Convert(x,y,z);
      R2.A.X:=p.x-RadX;
      R2.B.X:=p.x+RadX;
      R2.A.Y:=p.y-RadY;
      R2.B.Y:=p.y+RadY;
      DrawBar(PaintBox.Canvas,R2,GridColor);
      Msg:=Format('%.8g',[y/yAxis.Scale]);
      ShiftX:=not (myPlotParams.CosPhi>0);
      if ShiftX then ShiftY:=(ey.y-e0.y)*(ey.x-e0.x)<0
                else ShiftY:=(ey.y-e0.y)*(ey.x-e0.x)>0;
      ShiftCorrection;
      mySmartDrawer.DrawText(PaintBox.Canvas,p,msg,GridColor,GroundColor,Spc.X,Spc.Y);
     end;
    end;
   end;
  end;
  if Flags and am_Last <> 0 then mySmartDrawer.StopDraw;
 end;
var
 Org,Size,p:TPoint2I; Marg:TRect2I; s:LongString; Befor,After:Cardinal;
begin
 if Ok and IsFormViewable then
 try
  DebugLogReport_DrawView;
  {Clear the bitmap}
  with PaintBox do DrawBar(Canvas,Rect2I(0,0,Width,Height),GroundColor);
  {now prepare params to plot}
  CheckClip;
  Marg:=Margin;
  Org:=Marg.A;
  with PaintBox do Size:=Point2I(Width-(Marg.A.X+Marg.B.X),Height-(Marg.A.Y+Marg.B.Y));
  {
  plot the surface to null-to calc myPlotParams
  }
  PlotSurface(Clip.a.x, Clip.a.y, Clip.b.x, Clip.b.y,
              TheSurfMatrix, myCurrNum.x, myCurrNum.y,
              Phi, Psi, Org.X, Org.Y, Size.X, Size.Y,
              TheEmptyPutPixel, UpColor, DownColor,
              myScale, myPlotParams, 0, Self);
  Befor:=0;
  with myPlotParams do begin
   if sinphi<0 then Befor:=Befor or (am_Y + am_YN);
   if cosphi<0 then Befor:=Befor or (am_X + am_XN);
   if Befor and (am_X+am_Y) <> 0 then Befor:=Befor or (am_Z + am_ZN);
   if SliceMode and smNumberUp <> 0
   then Befor:=Befor and not (am_XN+am_YN+am_ZN);
   After:=not Befor and (am_X+am_Y+am_Z+am_XN+am_YN+am_ZN);
  end;
  {
  plot the surface to PaintBox
  }
  DrawAxis(Befor+am_First);
  PlotSurface(Clip.a.x, Clip.a.y, Clip.b.x, Clip.b.y,
              TheSurfMatrix, myCurrNum.x, myCurrNum.y,
              Phi, Psi, Org.X, Org.Y, Size.X, Size.Y,
              TheSurfPutPixel, UpColor, DownColor,
              myScale, myPlotParams, SliceMode and (smSliceX+smSliceY),Self);
  DrawAxis(After+am_Last);
  {plot the text message on top of the plot}
  s:=Title;
  with PaintBox do begin
   p:=Point2I((Width-Canvas.TextWidth(s)) div 2,1);
   DrawText(Canvas,p,s,TitleColor,GroundColor);
  end;
  {plot the legend message on bottom of the plot}
  s:=Legend;
  with PaintBox do begin
   p:=Point2I((Width-Canvas.TextWidth(s)) div 2,Height-1-abs(Canvas.TextHeight(s)));
   DrawText(Canvas,p,s,LegendColor,GroundColor);
  end;
  StatusBar.SimpleText:=GetViewPointText;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure TFormSurfWindow.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu; tb,sb,sx,sy:Boolean;
var Params:LongString;
 function GetTable(Mode:Integer):LongString;
 var List:TStringList; s:LongString; i,j:Integer;
 begin
  Result:='';
  List:=TStringList.Create;
  try
   case Mode of
    0 : for i:=0 to NumPoints.X-1 do begin
         s:='';
         for j:=0 to NumPoints.Y-1 do begin
          if Length(s)>0 then s:=s+#9;
          s:=s+Format('%g',[Z[i,j]]);
         end;
         List.Add(s);
        end;
    1 : for j:=0 to NumPoints.Y-1 do begin
         s:='';
         for i:=0 to NumPoints.X-1 do begin
          if Length(s)>0 then s:=s+#9;
          s:=s+Format('%g',[Z[i,j]]);
         end;
         List.Add(s);
        end;
   end;
   Result:=List.Text;
   Echo(RusEng('Таблица скопирована в буфер обмена!','Table copied to clipboard!'));
  finally
   List.Free;
   Finalize(s);
  end;
 end;
begin
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +PaintBoxPosParams;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать как таблицу Z(x,y) в Буфер Обмена'+EOL+
                         'Скопировать как таблицу Z(y,x) в Буфер Обмена'+EOL+
                         'Скопировать изображение в Буфер Обмена (цвет.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (ч./б.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (серый)'+EOL+
                         'Напечатать на Принтере '+Printer.PrinterName,
                         'Copy as Z(x,y) table to Clipboard'+EOL+
                         'Copy as Z(y,x) table to Clipboard'+EOL+
                         'Copy bitmap to Clipboard (color)'+EOL+
                         'Copy bitmap to Clipboard (b./w.)'+EOL+
                         'Copy bitmap to Clipboard (gray)'+EOL+
                         'Print hardcopy on Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  tb:=ToolBar.Visible;
  sb:=StatusBar.Visible;
  sx:=ScrollBarPhi.Visible;
  sy:=ScrollBarPsi.Visible;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarPhi.Visible:=False;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarPsi.Visible:=False;
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0,1:  Clipboard.AsText:=GetTable(Key);
    2..4: begin
           Bmp:=GetPrintableImage;
           if Assigned(Bmp) then
           try
            if (Key=3) then ConvertBmpToBlackAndWhite(Bmp,GroundColor,GroundColor,1);
            if (Key=4) then ConvertBmpToBlackAndWhite(Bmp,GroundColor,GroundColor,3);
            CopyFormBmpToClipboard(Bmp,true);
           finally
            Kill(Bmp);
           end;
          end;
    5:    if HasPrintersDialog then Print;
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=tb;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=sb;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarPhi.Visible:=sx;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarPsi.Visible:=sy;
  end;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

function TFormSurfWindow.GetImage(BlackAndWhite:Boolean):TBitmap;
begin
 Result:=nil;
 if Ok then
 try
  Toolbar.Hide;
  StatusBar.Hide;
  ScrollBarPhi.Hide;
  ScrollBarPsi.Hide;
  DrawView;
  Result:=GetFormImage;
  if BlackAndWhite then ConvertBmpToBlackAndWhite(Result,GroundColor,GroundColor);
  Toolbar.Show;
  StatusBar.Show;
  ScrollBarPhi.Show;
  ScrollBarPsi.Show;
  DrawView;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TFormSurfWindow.Clone:TFormSurfWindow;
var i,j:LongInt;
begin
 Result:=nil;
 if Ok then
 try
  Application.CreateForm(TFormSurfWindow,Result);
  if Result.Ok then
  try
   Result.LockDraw;
   Result.Caption:=RusEng('КОПИЯ:','COPY:')+Caption;
   Result.Width:=Width;
   Result.Height:=Height;
   Result.NumPoints:=NumPoints;
   for i:=0 to NumPoints.X-1 do
   for j:=0 to NumPoints.Y-1 do Result[i,j]:=Self[i,j];
   Result.Limits:=Limits;
   Result.Clip:=Clip;
   Result.Scale:=Scale;
   Result.Phi:=Phi;
   Result.Psi:=Psi;
   Result.Title:=Title;
   Result.Legend:=Legend;
   Result.Comment.Text:=Comment.Text;
   Result.GroundColor:=GroundColor;
   Result.UpColor:=UpColor;
   Result.DownColor:=DownColor;
   Result.AxisColor:=AxisColor;
   Result.GridColor:=GridColor;
   Result.TitleColor:=TitleColor;
   Result.LegendColor:=LegendColor;
   Result.SliceMode:=SliceMode;
   Result.StatusBar.Hide;
   Result.ScrollBarPhi.Hide;
   Result.StatusBar.Show;
   Result.ScrollBarPhi.Show;
  finally
   Result.UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TFormSurfWindow.PaintBoxPosParams:LongString;
begin
 if Ok then Result:=ControlPosParams(PaintBox) else Result:='';
end;

procedure TFormSurfWindow.FormCreate(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
  myZmatrix:=NewDoubleMatrix(60,60);
  myZmatrix.Master:=@myZmatrix;
  myZMatrix.Origin:=0;
  myZmatrix.Exceptions:=false;
  myComment:=NewText;
  myComment.Master:=@myComment;
  mySmartDrawer:=NewSmartDrawer;
  myScale:=Point3D(1,1,1);
  mySliceMode:=smSliceX+smSliceY+smNumberUp;
  myGroundColor:=DefGroundColor;
  myUpColor:=DefUpColor;
  myDownColor:=DefDownColor;
  myAxisColor:=DefAxisColor;
  myGridColor:=DefGridColor;
  myTitleColor:=DefTitleColor;
  myLegendColor:=DefLegendColor;
  myTitle:='';
  myLegend:='';
  Limits:=Rect2D(0,0,1,1);
  AutoScroll:=false;
  SetStandardFont(Self);
  SetAllButtonsCursor(Self,crHandPoint);
  with ScrollBarPhi do begin
   Min:=0;
   Max:=360*60;
   SmallChange:=15;
   LargeChange:=60*5;
   Position:=45*60;
  end;
  with ScrollBarPsi do begin
   Min:=0;
   Max:=180*60;
   SmallChange:=15;
   LargeChange:=60*5;
   Position:=45*60;
  end;
 finally
  UnlockDraw;
 end;
 UpdateMenu(MenuSurfToolsCloneWindow,
            RusEng('Клонировать окно','Clone window'),
            RusEng('Создать точную копию окна','Create exact copy of window'),
            ShortCut(Word('D'),[ssCtrl]));
 UpdateMenu(MenuSurfTools,
            RusEng('Поверхность','Surface')+MenuRightSpace,
            RusEng('Функции работы с поверхностью','Surface window routines'),
            0);
 UpdateMenu(MenuSurfToolsEditComment,
            RusEng('Изменить легенду','Edit legend'),
            RusEng('Редактировать легенду (паспорт) окна с поверхностью','Edit surface window legend (comment)'),
            ShortCut(Word('L'),[ssCtrl]));
 UpdateMenu(MenuSurfToolsEditWindowStyle,
            RusEng('Изменить стиль окна','Edit window style'),
            RusEng('Изменить имя, цвет и другие уставки окна','Edit window name, title, color etc.'),
            ShortCut(Word('W'),[ssCtrl]));
 UpdateMenu(MenuSurfToolsEditWindowClip,
            RusEng('Выбор фрагмента','Select window clip'),
            RusEng('Выбрать видимый фрагмент поверхности','Select visible window region'),
            ShortCut(Word('F'),[ssCtrl]));
 UpdateMenu(MenuSurfToolsExtractSlicesX,
            RusEng('Сечения X=const','Extract slices X=const'),
            RusEng('Строит кривые сечений поверхности X=const','Extract slices X=const as curves'),
            ShortCut(Word('X'),[ssCtrl]));
 UpdateMenu(MenuSurfToolsExtractSlicesY,
            RusEng('Сечения Y=const','Extract slices Y=const'),
            RusEng('Строит кривые сечений поверхности Y=const','Extract slices Y=const as curves'),
            ShortCut(Word('Y'),[ssCtrl]));
 UpdateMenu(MenuSurfToolsSmoothing,
            RusEng('Сгладить поверхность','Surface smoothing'),
            RusEng('Сгладить поверхность, убрать "шум"','Smooth surface, kill off noise'),
            ShortCut(Word('S'),[ssCtrl]));
end;

procedure TFormSurfWindow.FormDestroy(Sender: TObject);
begin
 myTitle:='';
 myLegend:='';
 Kill(myZmatrix);
 Kill(myComment);
 Kill(mySmartDrawer);
 inherited;
end;

procedure TFormSurfWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullSurfWindowList.Add(Self);
 AddonSdiFlags(sf_SdiSurfWin);
end;

procedure TFormSurfWindow.BeforeDestruction;
begin
 FullSurfWindowList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormSurfWindow.PaintBoxPaint(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormSurfWindow.ScrollBarChange(Sender: TObject);
begin
 inherited;
 if Ok then begin
  try
   LockDraw;
  finally
   UnlockDraw;
  end;
 end;
end;

procedure TFormSurfWindow.FileSave;
const FName:LongString='';
var Ext,Params:LongString;
begin
 try
  SaveDialog.Title:=RusEng('Файл\Сохранить','File\Save');
  SaveDialog.Filter:=RusEng('Архивные файлы CRW (*.crw)|*.crw|'+
                            'Архивные файлы DAQ (*.daq)|*.daq|'+
                            'Все остальные файлы  (*.*)|*.*|',
                            'CRW archive files (*.crw)|*.crw|'+
                            'DAQ archive files (*.daq)|*.daq|'+
                            'All other files     (*.*)|*.*|');
  if IsEmptyStr(SaveDialog.FileName) then SaveDialog.FileName:=FName;
  if IsEmptyStr(SaveDialog.FileName)
  then OpenDialogSelectType(SaveDialog,'*.crw')
  else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
  Params:=ControlPosParams(ToolBar,'LB');
  if ExecuteFileDialog(GuardOpenDialog(SaveDialog),Params) then begin
   FName:=UnifyFileAlias(SaveDialog.FileName);
   Ext:=UnifyAlias(SaveDialog.DefaultExt);
   if IsEmptyStr(Ext) or IsWildCard(Ext)
   then FName:=UnifyFileAlias(FName,ua_FileDefLow)
   else FName:=UnifyFileAlias(ForceExtension(FName,Ext),ua_FileDefLow);
   SaveSurfWindowToCrw(FName,Self);
  end;
 except
  on E:Exception do BugReport(E,Self,'FileSave');
 end; 
end;

procedure TFormSurfWindow.FileSaveAs;
var Params:LongString;
begin
 if Ok then begin
  Params:=IfThen(Visible,ControlPosParams(PaintBox,'LT'),'');
  FormSurfWindowStyleDialogExecute(Self,Params);
  FileSave;
 end;
end;

procedure TFormSurfWindow.ActionSurfToolsEditCommentExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsEditComment)<0 then Exit;
 inherited;
 TextEditDialog(RusEng('Редактирование паспорта окна','Edit window comment'),
                RusEng('Отредактируйте паспорт окна "','Please, edit comment of window "')+Caption+'"',
                Comment,PaintBoxPosParams);
end;

procedure TFormSurfWindow.ActionSurfToolsEditWindowStyleExecute( Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsEditWindowStyle)<0 then Exit;
 inherited;
 FormSurfWindowStyleDialogExecute(Self,PaintBoxPosParams);
end;

procedure TFormSurfWindow.ActionSurfToolsEditWindowClipExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsEditWindowClip)<0 then Exit;
 inherited;
 FormSurfToolsWindowClipDialogExecute(Self,PaintBoxPosParams);
end;

procedure TFormSurfWindow.ActionSurfToolsExtractSlicesXExecute(Sender: TObject);
var CurveWindow:TFormCurveWindow; Curve:TCurve; x,y:Double; i,j:LongInt;
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsExtractSlicesX)<0 then Exit;
 inherited;
 if Ok then
 try
  CurveWindow:=NewCurveWindow(RusEng('Сечения X=const','X=const slices'));
  for i:=0 to NumPoints.X-1 do begin
   x:=Limits.A.X + i * RectSizeX(Limits) / (NumPoints.X-1);
   Curve:=NewCurve(0,Format('SliceX#%g',[x]),clBlack,$1F);
   for j:=0 to NumPoints.Y-1 do begin
    y:=Limits.A.Y + j * RectSizeY(Limits) / (NumPoints.Y-1);
    Curve.AddPoint(y,Z[i,j]);
   end;
   CurveWindow.AddCurve(Curve);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormSurfWindow.ActionSurfToolsExtractSlicesYExecute(Sender: TObject);
var CurveWindow:TFormCurveWindow; Curve:TCurve; x,y:Double; i,j:LongInt;
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsExtractSlicesY)<0 then Exit;
 inherited;
 if Ok then
 try
  CurveWindow:=NewCurveWindow(RusEng('Сечения Y=const','Y=const slices'));
  for j:=0 to NumPoints.Y-1 do begin
   y:=Limits.A.Y + j * RectSizeY(Limits) / (NumPoints.Y-1);
   Curve:=NewCurve(0,Format('SliceY#%g',[y]),clBlack,$1F);
   for i:=0 to NumPoints.X-1 do begin
    x:=Limits.A.X + i * RectSizeX(Limits) / (NumPoints.X-1);
    Curve.AddPoint(x,Z[i,j]);
   end;
   CurveWindow.AddCurve(Curve);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TFormSurfWindow.ActionSurfToolsCloneWindowExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsCloneWindow)<0 then Exit;
 inherited;
 if Ok then Clone;
end;

procedure TFormSurfWindow.ActionSurfToolsSmoothingExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionSurfToolsSmoothing)<0 then Exit;
 inherited;
 FormSurfToolsSmoothingDialogExecute(Self,PaintBoxPosParams);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_surfwindow;
begin
 FullSurfWindowList:=NewSurfWindowList(false);
 FullSurfWindowList.Master:=@FullSurfWindowList;
end;

procedure Free_form_surfwindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullSurfWindowList.Count', FullSurfWindowList.Count]));
 Kill(FullSurfWindowList);
end;

initialization

 Init_form_surfwindow;

finalization

 Free_form_surfwindow;

end.

//////////////
// END OF FILE
//////////////

