////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Common crwdaq constants, variables, functions.                             //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20240819 - Created by A.K.                                                 //
////////////////////////////////////////////////////////////////////////////////

unit unit_crwdaq_common; //  unit for crwdaq common resources.

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, strutils,
 Graphics, Interfaces, LMessages,
 Forms, Controls, ComCtrls, Dialogs,
 lcltype, lclintf, lclproc, typinfo,
 ActnList, Menus, ToolWin, ImgList, Printers, Clipbrd,
 _crw_alloc, _crw_rtc, _crw_str, _crw_ef, _crw_dbglog,
 _crw_eldraw, _crw_guard, _crw_wmctrl,
 _crw_appforms, _crw_apptools, _crw_apputils;

 // Return [0,1,2] if aItem.Name [not,equal,starts] with aName.
function MenuItemNameMatch(aItem:TMenuItem; aName:LongString):Integer;

 // Prepare SDI menu.
procedure PrepareCrwDaqSdiMenu(Form:TForm);

 // DebugLog channel '_MenuGroup'.
function dlc_MenuGroup:Integer; inline;

implementation

uses form_crwdaqsyschild;

const
 dlc_MenuGroup_:Integer=0;

function dlc_MenuGroup:Integer;
begin
 Result:=dlc_MenuGroup_;
end;

function MenuItemNameMatch(aItem:TMenuItem; aName:LongString):Integer;
begin
 Result:=0;
 if (aItem=nil) or (aName='') then Exit;
 if SameText(aName,aItem.Name) then Exit(1);
 if StartsText(aName,aItem.Name) then Exit(2);
end;

function UpdateMenuItemGroupIndex(MenuItem:TMenuItem;
                                   BaseIndex:Integer;
                                   const Name:LongString):Boolean;
var n,cgi,ngi:Integer;
begin
 Result:=false;
 if Assigned(MenuItem) and (Name<>'') then begin
  cgi:=IfThen((MenuItem.Owner is TFormCrwDaqSysChild),1,0);
  n:=MenuItemNameMatch(MenuItem,Name);
  if (n>0) then begin
   Result:=true; ngi:=MenuItem.GroupIndex;
   if IsWindows then ngi:=BaseIndex+(n-1);
   if IsUnix then ngi:=BaseIndex+(n-1)+cgi*2;
   if (MenuItem.GroupIndex<>ngi) then MenuItem.GroupIndex:=ngi;
   if DebugLogEnabled(dlc_MenuGroup)
   then DebugLog(dlc_MenuGroup,Format('%s.%s.GroupIndex = %d',
        [MenuItem.Owner.ClassName,MenuItem.Name,MenuItem.GroupIndex]));
  end;
 end;
end;

procedure umigiAction(Component:TComponent; Index:Integer; var Terminate:Boolean; Custom:Pointer);
var MenuItem:TMenuItem;
begin
 if (Component is TMenuItem) then begin
  MenuItem:=TMenuItem(Component);
  if UpdateMenuItemGroupIndex(MenuItem, 250, 'MenuWindowsShowActive' ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 240, 'MenuWindowsSelector'   ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 230, 'MenuCopyrightProof'    ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 200, 'MenuHelp'              ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 210, 'MenuLink'              ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 120, 'MenuWindows'           ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 110, 'MenuTools'             ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 100, 'MenuDaq'               ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 60,  'MenuSpectr'            ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 50,  'MenuSurfTools'         ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 40,  'MenuCurve'             ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 30,  'MenuView'              ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 20,  'MenuEdit'              ) then Exit;
  if UpdateMenuItemGroupIndex(MenuItem, 10,  'MenuFile'              ) then Exit;
 end;
end;

procedure PrepareCrwDaqSdiMenu(Form:TForm);
begin
 if SdiMan.EnableSdiFormMenuMerge then
 if (Form is TMasterForm) then
 try
  ForEachComponent(Form,umigiAction,nil);
  if (Form is TFormCrwDaqSysChild) and SdiMan.HideChildSdiFormMenu then Form.Menu:=nil;
 except
  on E:Exception do BugReport(E,nil,'PrepareCrwDaqSdiMenu');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_unit_crwdaq_common;
begin
 dlc_MenuGroup_:=RegisterDebugLogChannel('_MenuGroup');
end;

procedure Free_unit_crwdaq_common;
begin
end;

initialization

 Init_unit_crwdaq_common;

finalization

 Free_unit_crwdaq_common;

end.

//////////////
// END OF FILE
//////////////

