 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2003, <kouriakine@mail.ru>
 Shared memory access class.
 Modifications:
 20030113 - Creation & test
 20030330 - Struggle for safety (add some try/except checks)...
 20040424 - FileMappingResult,MapViewResult
 ****************************************************************************
 }

unit _sharm;

{$I _sysdef}

interface

uses
 SysUtils, Windows, Math, _alloc, _str;

 ///////////////////////////////////////////////////////////////////////////////
 // Shared memory is simple and fast way for Inter-Process Communication (IPC).
 // Class TSharedMemory gives you shared memory access, you may use it for IPC.
 // Notes:
 //  1) To access shared memory block, processes must use the same NAME and SIZE.
 //     Blocks with diffrenent NAMEs will not be shared.
 //  2) Memory will be allocated by 4K pages, so allocated size will be adjusted
 //     to 4096 bytes. For example, if you want allocate 4097 bytes (4096+1),
 //     8192 (4096*2) bytes will be really allocated. Function CAPACITY returns
 //     size, adjusted by page size.
 //  3) Use ASINTEGER,ASDOUBLE for protected data access.
 //     Use ASPTR,ASCHAR for common access with exception handling.
 //     For example:
 //      buf:=NewSharedMemory('SharedBufer',4096);
 //      buf.AsInteger[1024]:=1;  // No exception, neither assignment.
 //      Integer(buf[4096]^):=1;  // Exception will be raised.
 //  4) Data access benchmark result on P4-2000:
 //      p:=PIntegerArray(buf[0]); p[0]^:=p[0]^+1;  -> 140 millions op/sec
 //      Integer(buf[0]^):=Integer(buf[0]^)+1;      -> 90  millions op/sec
 //      buf.AsInteger[0]:=buf.AsInteger[0]+1;      -> 33  millions op/sec
 //  5) TSharedMemory designed for inter-process communications. No reason
 //     to use this one for inter-thread communications, because threads
 //     has the same address space.
 // Example:
 //  Process A:
 //   buf:=NewSharedMemory('SharedBufer',5000); // Create shared memory block
 //   Echo(Format('Buffer name = %s',[buf.Name]));
 //   Echo(Format('Buffer size = %s',[buf.Size]));
 //   Echo(Format('Allocated size = %d',[buf.Capacity]));
 //   buf.AsInteger[0]:=1;
 //   buf.AsDouble[1]:=pi;
 //   Double(buf[12]^):=exp(1);
 //   ..etc
 //   Kill(buf);
 //  Process B:
 //   buf:=NewSharedMemory('SharedBufer',5000); // Access the same memory block
 //   if buf.AsInteger[0]=1 then begin ...
 //   ..etc
 //   Kill(buf);
 ///////////////////////////////////////////////////////////////////////////////
type
 TSharedMemory = class(TMasterObject)
 private
  myFile     : THandle;
  myData     : Pointer;
  mySize     : Integer;
  myCapacity : Integer;
  myName     : packed array[0..255] of Char;
  myFMR      : Integer;
  myMVR      : Integer;
  function  GetSize:Integer;
  function  GetData:Pointer;
  function  GetName:ShortString;
  function  GetCapacity:Integer;
  function  GetAsPtr(i:Integer):Pointer;
  function  GetAsPChar(i:Integer):PChar;
  function  GetAsInteger(i:Integer):Integer;
  procedure SetAsInteger(i:Integer; v:Integer);
  function  GetAsDouble(i:Integer):Double;
  procedure SetAsDouble(i:Integer; v:Double);
  function  GetFileMappingResult:Integer;
  function  GetMapViewResult:Integer;
 protected
  function  CheckOk:Boolean; override;
 public
  constructor Create(const aName:ShortString; aSize:Integer);
  destructor  Destroy; override;
  property    Size                 : Integer     read GetSize;
  property    Data                 : Pointer     read GetData;
  property    Name                 : ShortString read GetName;
  property    Capacity             : Integer     read GetCapacity;
  property    AsPtr[i:Integer]     : Pointer     read GetAsPtr;     default;
  property    AsPChar[i:Integer]   : PChar       read GetAsPChar;
  property    AsInteger[i:Integer] : Integer     read GetAsInteger  write SetAsInteger;
  property    AsDouble[i:Integer]  : Double      read GetAsDouble   write SetAsDouble;
  property    FileMappingResult    : Integer     read GetFileMappingResult;
  property    MapViewResult        : Integer     read GetMapViewResult;
 public
  class function SysPageSize(Def:Integer=4096):Integer;
 end;

function  NewSharedMemory(const aName:ShortString; aSize:Integer):TSharedMemory;
procedure Kill(var TheObject:TSharedMemory); overload;

 ///////////////////////////////
 // Simplified API for DaqPascal
 ///////////////////////////////

function shm_init(name:LongString; size,mode:Integer):Integer;
function shm_ref(ref:Integer):TSharedMemory;
function shm_free(ref:Integer):Boolean;
function shm_delink(name:LongString):Boolean;
function shm_iop(ref:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
function shm_rop(ref:Integer; offset:Integer; op:Char; data:Double):Double;
function shm_fop(ref:Integer; offset:Integer; op:Char; data:Single):Single;
function shm_sop(ref:Integer; offset:Integer; op:Char; data:LongString):LongString;
function shm_ctrl(ref:Integer; arg:LongString):LongString;
function shm_ioresult(code:Integer=0):Integer;
const shm_total_errors:SizeInt=0;
threadvar shm_errno:Integer;

const
 shm_err_ok  = 0; // Success
 shm_err_op  = 1; // Bad operation code
 shm_err_ref = 2; // Bad object reference
 shm_err_arg = 3; // Bad arguments found
 shm_err_bug = 4; // Exception raised

implementation

constructor TSharedMemory.Create(const aName:ShortString; aSize:Integer);
begin
 inherited Create;
 Exceptions:=false;
 myFile:=0;
 myData:=nil;
 mySize:=Max(0,aSize);
 myCapacity:=AdjustBufferSize(mySize,SysPageSize);
 myName:='';
 if mySize>0 then StrPCopy(myName,Trim(aName));
 if myName<>'' then begin
  myFile:=CreateFileMapping(INVALID_HANDLE_VALUE, nil, PAGE_READWRITE+SEC_COMMIT, 0, myCapacity, myName);
  myFMR:=GetLastError;
 end else myFMR:=0;
 if myFile<>0 then begin
  myData:=MapViewOfFile(myFile, FILE_MAP_ALL_ACCESS, 0, 0, 0);
  myMVR:=GetLastError;
 end else myMVR:=0;
end;

destructor  TSharedMemory.Destroy;
begin
 if myData<>nil then UnmapViewOfFile(myData);
 if myFile<>0 then CloseHandle(myFile);
 inherited Destroy;
end;

function TSharedMemory.GetSize:Integer;
begin
 if Assigned(Self) then Result:=mySize else Result:=0;
end;

function TSharedMemory.GetCapacity:Integer;
begin
 if Assigned(Self) then Result:=myCapacity else Result:=0;
end;

function TSharedMemory.GetData:Pointer;
begin
 if Assigned(Self) then Result:=myData else Result:=nil;
end;

function TSharedMemory.GetName:ShortString;
begin
 if Assigned(Self) then Result:=StrPas(myName) else Result:='';
end;

function TSharedMemory.GetAsPtr(i:Integer):Pointer;
begin
 if Assigned(Self) and Assigned(myData) then Result:=@(PChar(myData)[i]) else Result:=nil;
end;

function TSharedMemory.GetAsPChar(i:Integer):PChar;
begin
 if Assigned(Self) and Assigned(myData) then Result:=@(PChar(myData)[i]) else Result:=nil;
end;

function TSharedMemory.GetAsInteger(i:Integer):Integer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Result:=PIntegerArray(myData)[i];
 except
  on E:Exception do ErrorFound(E);
 end;
end;

procedure TSharedMemory.SetAsInteger(i:Integer;v:Integer);
begin
 if Assigned(Self) then
 try
  PIntegerArray(myData)[i]:=v;
 except
  on E:Exception do ErrorFound(E);
 end;
end;

function TSharedMemory.GetAsDouble(i:Integer):Double;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Result:=PDoubleArray(myData)[i];
 except
  on E:Exception do ErrorFound(E);
 end;
end;

procedure TSharedMemory.SetAsDouble(i:Integer; v:Double);
begin
 if Assigned(Self) then
 try
  PDoubleArray(myData)[i]:=v;
 except
  on E:Exception do ErrorFound(E);
 end;
end;

function  TSharedMemory.GetFileMappingResult:Integer;
begin
 if Assigned(Self) then Result:=myFMR else Result:=0;
end;

function  TSharedMemory.GetMapViewResult:Integer;
begin
 if Assigned(Self) then Result:=myMVR else Result:=0;
end;

function TSharedMemory.CheckOk:Boolean;
begin
 Result:=Assigned(myData);
end;

class function TSharedMemory.SysPageSize(Def:Integer=4096):Integer;
const ThePageSize:Integer=0; var SysInfo:TSystemInfo;
begin
 if (ThePageSize=0) then begin
  GetSystemInfo(SysInfo);
  if (SysInfo.dwPageSize>0)
  then ThePageSize:=SysInfo.dwPageSize;
  end;
 if (ThePageSize>0)
 then Result:=ThePageSize
 else Result:=Def;
end;

function  NewSharedMemory(const aName:ShortString; aSize:Integer):TSharedMemory;
begin
 Result:=nil;
 try
  Result:=TSharedMemory.Create(aName,aSize);
  if not Result.Ok then Kill(Result);
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure Kill(var TheObject:TSharedMemory); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end;
end;

 ///////////////////////////////
 // Simplified API for DaqPascal
 ///////////////////////////////

function shm_init(name:LongString; size,mode:Integer):Integer;
var shm:TSharedMemory;
begin
 Result:=0;
 if (name<>'') and (size>0) then
 try
  shm:=NewSharedMemory(name,size);
  if shm.Ok then Result:=shm.Ref else Kill(shm);
 except
  on E:Exception do BugReport(E,nil,'shm_init');
 end;
end;

function shm_ref(ref:Integer):TSharedMemory;
var obj:TObject;
begin
 Result:=nil;
 if (ref=0) then Exit;
 obj:=ObjectRegistry[ref];
 if (obj is TSharedMemory)
 then Result:=TSharedMemory(obj);
end;

function shm_free(ref:Integer):Boolean;
var shm:TSharedMemory;
begin
 shm:=shm_ref(ref);
 Result:=Assigned(shm);
 if Result then Kill(shm);
end;

function shm_delink(name:LongString):Boolean;
begin
 Result:=false;
end;

function shm_fail(cond:Boolean; err:Integer):Boolean;
begin
 Result:=cond;
 if Result then shm_errno:=err;
 if Result then LockedInc(shm_total_errors);
end;

function shm_iop(ref:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
var shm:TSharedMemory; Ptr:Pointer;
begin
 Result:=0;
 if shm_fail(ref=0,shm_err_ref) then Exit;
 if shm_fail(offset<0,shm_err_arg) then Exit;
 try
  shm:=shm_ref(ref);
  if shm_fail(shm=nil,shm_err_ref) then Exit;
  if shm_fail(offset>shm.Size-SizeOf(data),shm_err_arg) then Exit;
  case op of
   'r','R': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=LongInt(Ptr^);
   end;
   'w','W': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=LongInt(Ptr^);
    LongInt(Ptr^):=data;
   end;
   '+': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=LockedAdd(LongInt(Ptr^),data);
   end;
   '-': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=LockedSub(LongInt(Ptr^),data);
   end;
   'e','E': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=LockedExchange(LongInt(Ptr^),data);
   end;
   else shm_errno:=shm_err_op;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'shm_iop');
   shm_errno:=shm_err_bug;
  end;
 end;
end;

function shm_rop(ref:Integer; offset:Integer; op:Char; data:Double):Double;
var shm:TSharedMemory; Ptr:Pointer;
begin
 Result:=0;
 if shm_fail(ref=0,shm_err_ref) then Exit;
 if shm_fail(offset<0,shm_err_arg) then Exit;
 try
  shm:=shm_ref(ref);
  if shm_fail(shm=nil,shm_err_ref) then Exit;
  if shm_fail(offset>shm.Size-SizeOf(data),shm_err_arg) then Exit;
  Result:=data;
  case op of
   'r','R': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=Double(Ptr^);
   end;
   'w','W': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=Double(Ptr^);
    Double(Ptr^):=data;
   end;
   else shm_errno:=shm_err_op;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'shm_rop');
   shm_errno:=shm_err_bug;
  end;
 end;
end;

function shm_fop(ref:Integer; offset:Integer; op:Char; data:Single):Single;
var shm:TSharedMemory; Ptr:Pointer;
begin
 Result:=0;
 if shm_fail(ref=0,shm_err_ref) then Exit;
 if shm_fail(offset<0,shm_err_arg) then Exit;
 try
  shm:=shm_ref(ref);
  if shm_fail(shm=nil,shm_err_ref) then Exit;
  if shm_fail(offset>shm.Size-SizeOf(data),shm_err_arg) then Exit;
  Result:=data;
  case op of
   'r','R': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=Single(Ptr^);
   end;
   'w','W': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=Single(Ptr^);
    Single(Ptr^):=data;
   end;
   else shm_errno:=shm_err_op;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'shm_fop');
   shm_errno:=shm_err_bug;
  end;
 end;
end;

function shm_sop(ref:Integer; offset:Integer; op:Char; data:LongString):LongString;
var shm:TSharedMemory; Ptr:Pointer;
begin
 Result:='';
 if shm_fail(ref=0,shm_err_ref) then Exit;
 if shm_fail(data='',shm_err_arg) then Exit;
 if shm_fail(offset<0,shm_err_arg) then Exit;
 try
  shm:=shm_ref(ref);
  if shm_fail(shm=nil,shm_err_ref) then Exit;
  if shm_fail(offset>shm.Size-Length(data),shm_err_arg) then Exit;
  Result:=data;
  case op of
   'r','R': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=StringBuffer(Ptr,Length(data));
   end;
   'w','W': begin
    Ptr:=shm.AsPtr[offset];
    if shm_fail(Ptr=nil,shm_err_arg) then Exit;
    Result:=StringBuffer(Ptr,Length(data));
    SafeMove(PChar(data)^,Ptr^,Length(data));
   end;
   else shm_errno:=shm_err_op;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'shm_sop');
   shm_errno:=shm_err_bug;
  end;
 end;
end;

function shm_ioresult(code:Integer=0):Integer;
begin
 Result:=shm_errno;
 shm_errno:=code;
end;

function shm_ctrl(ref:Integer; arg:LongString):LongString;
var shm:TSharedMemory; sn,sv,par,pars:LongString; p,i,iv:Integer; qv:Int64;
const rParams='Name,Size,Mode,Capacity,SysPageSize';
const wParams='ioresult,shm_total_errors';
begin
 Result:='';
 p:=ExtractNameValuePair(arg,sn,sv);
 // Get/Set common system Parameters
 if SameText(sn,'SysPageSize') then begin
  Result:=IntToStr(TSharedMemory.SysPageSize);
  Exit;
 end;
 if SameText(sn,'shm_total_errors') then begin
  if (p>0) and (sv<>'') and TryStrToInt64(sv,qv)
  then Result:=IntToStr(LockedExchange(shm_total_errors,qv))
  else Result:=IntToStr(shm_total_errors);
  Exit;
 end;
 // Get/Set object (ref) Parameters
 if shm_fail(ref=0,shm_err_ref) then Exit;
 if shm_fail(arg='',shm_err_arg) then Exit;
 try
  shm:=shm_ref(ref);
  if shm_fail(shm=nil,shm_err_ref) then Exit;
  if SameText(sn,'Name') then begin
   Result:=shm.Name;
  end else
  if SameText(sn,'Size') then begin
   Result:=IntToStr(shm.Size);
  end else
  if SameText(sn,'Mode') then begin
   Result:=IntToStr(0);
  end else
  if SameText(sn,'Capacity') then begin
   Result:=IntToStr(shm.Capacity);
  end else
  if SameText(sn,'ioresult') or SameText(sn,'shm_ioresult') then begin
   if (p>0) and (sv<>'') and TryStrToInt(sv,iv)
   then Result:=IntToStr(shm_ioresult(iv))
   else Result:=IntToStr(shm_ioresult(0));
  end else
  if SameText(sn,'*') or SameText(sn,'?') then begin
   if (p=0) then pars:=rParams+','+wParams else pars:=wParams;
   for i:=1 to WordCount(pars,ScanSpaces) do begin
    par:=ExtractWord(i,pars,ScanSpaces);
    if (sn='*') then Result:=Result+par+EOL;
    if (sn='?') then Result:=Result+par+'='+shm_ctrl(ref,par)+EOL
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'shm_ctrl');
   shm_errno:=shm_err_bug;
  end;
 end;
end;

end.

