////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для вычисления БПФ(Быстрого Фурье Преобразования)
или FFT(Fast Fourier Transform).
Для каждой кривой вычисляется:
Real - реальная часть комплексного FFT преобразования,
Image - мнимая часть комплексного FFT преобразования,
Amplitude - амплитуда комплексного FFT преобразования,
Decibell - амплитуда комплексного FFT преобразования в Децибеллах (в логарифмической шкале),
Phase - фаза (угол arctan(Im/Re)) комплексного FFT преобразования.
Надо также задать:
NumberOfPoints - число точек, 0 означает автоматический выбор.
Inverse - 0/1 - прямое/обратное преобразование.
ShiftIn - 0/1 - входной сигнал задан на 0..2 или -1..1
ShiftOut - 0/1 - входной сигнал задан на 0..2 или -1..1
[]
[Manual.Eng]
This plugin uses to make FFT - Fast Fourier Transform.
Calculate:
Real - real part of complex FFT,
Image - image part of complex FFT,
Amplitude - amplitude (module) of complex FFT,
Decibell - amplitude in Decibells, i.e. amplitude in Log scale,
Phase - phase, i.e. arctan(Im/Re) of complex FFT.
[]
[Arguments.Rus]
Имя окна = Результат:"Fourier Transform"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
NumberOfPoints = 0
Inverse = 0
ShiftIn = 0
ShiftOut = 1
[]
[Arguments.Eng]
Caption = Result:"Fourier Transform"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
NumberOfPoints = 0
Inverse = 0
ShiftIn = 0
ShiftOut = 1
[]
}

library _curve_fft;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_ef, _crw_fft,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
var
 Re, Im:PDoubleArray;
 x, Ave, fNyquist : Double;
 i, cIn, cRe, cIm, cAm, cDb, cPh, cFirst, cLast, NumberOfPoints, Inverse, ShiftIn, ShiftOut : Integer;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  if not GetArgumentAsInteger('Inverse',Inverse)
  then Raise EDanApi.Create(RusEng('Не задан параметр Inverse!',
                                   'Uncertain parameter Inverse!'));
  if not GetArgumentAsInteger('ShiftIn',ShiftIn)
  then Raise EDanApi.Create(RusEng('Не задан параметр ShiftIn!',
                                   'Uncertain parameter ShiftOut!'));
  if not GetArgumentAsInteger('ShiftOut',ShiftOut)
  then Raise EDanApi.Create(RusEng('Не задан параметр ShiftOut!',
                                   'Uncertain parameter ShiftOut!'));
  for cIn:=cFirst to cLast do begin
   if CurveLength[cIn]<1 then Continue;
   if not GetArgumentAsInteger('NumberOfPoints',NumberOfPoints)
   then Raise EDanApi.Create(RusEng('Не задан параметр NumberOfPoints!',
                                    'Uncertain parameter NumberOfPoints!'));
   if NumberOfPoints<1 then begin
    NumberOfPoints:=1;
    while NumberOfPoints<CurveLength[cIn] do NumberOfPoints:=NumberOfPoints*2;
   end;
   Re:=Allocate(NumberOfPoints*SizeOf(Re[0]));
   Im:=Allocate(NumberOfPoints*SizeOf(Im[0]));
   try
    for i:=0 to NumberOfPoints-1 do begin
     with CurveLimits[cIn] do
     Re[i]:=CurveInterpolateAt(cIn,A.X+(B.X-A.X)*i/NumberOfPoints);
     Im[i]:=0;
    end;
    with CurveLimits[cIn] do fNyquist:=0.5*NumberOfPoints/(B.X-A.X);
    if not ShiftedFFT(NumberOfPoints,@Re[0],@Im[0],Inverse<>0,ShiftIn<>0,ShiftOut<>0)
    then Raise EDanApi.Create(RusEng('Не могу сделать FFT!',
                                     'Could not evaluate FFT!'));
    Ave:=0;
    for i:=0 to NumberOfPoints-1 do Ave:=Max(Max(Ave,Hypot(Re[i],Im[i])),MachEps);
    cRe:=CreateCurve(twin,CurveName[cIn]+'.FFT.Real',CurveColor[cIn],CurveStyle[cIn],CurveComment[cIn]);
    cIm:=CreateCurve(twin,CurveName[cIn]+'.FFT.Image',CurveColor[cIn],CurveStyle[cIn],CurveComment[cIn]);
    cAm:=CreateCurve(twin,CurveName[cIn]+'.FFT.Amplitude',CurveColor[cIn],CurveStyle[cIn],CurveComment[cIn]);
    cDb:=CreateCurve(twin,CurveName[cIn]+'.FFT.Decibell',CurveColor[cIn],CurveStyle[cIn],CurveComment[cIn]);
    cPh:=CreateCurve(twin,CurveName[cIn]+'.FFT.Phase',CurveColor[cIn],CurveStyle[cIn],CurveComment[cIn]);
    for i:=0 to NumberOfPoints-1 do begin
     if ShiftOut<>0
     then x:=fNyquist*(i*2/NumberOfPoints-1)
     else x:=fNyquist*i*2/(NumberOfPoints-1);
     CurveAddPoint(cRe,Point2d(x,Re[i]));
     CurveAddPoint(cIm,Point2d(x,Im[i]));
     CurveAddPoint(cAm,Point2d(x,Hypot(Re[i],Im[i])));
     CurveAddPoint(cDb,Point2d(x,Decibell(Re[i],Im[i],Ave,1E6)));
     CurveAddPoint(cPh,Point2d(x,Phase(Re[i],Im[i])));
     Refreshment(100);
    end;
   finally
    Deallocate(Pointer(Re));
    Deallocate(Pointer(Im));
   end;
  end;
  SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
