#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

function fatal(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
 if which notify-send; then notify-send -t 30000 -i dialog-error "$date: $script" "$1"; fi;
 1>&2 echo "-ne ";"\n$1\n\n";
 exit 1;
};

# Script identification.
########################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

function uniheat_main(){
 add_uniheat_table 8  > "$scriptHOME/uh_8.cfg";
 add_uniheat_table 16 > "$scriptHOME/uh_16.cfg";
 add_uniheat_table 24 > "$scriptHOME/uh_24.cfg";
 add_uniheat_table 32 > "$scriptHOME/uh_32.cfg";
 add_uniheat_table 10 > "$scriptHOME/uh_10.cfg";
 add_uniheat_table 20 > "$scriptHOME/uh_20.cfg";
 add_uniheat_table 30 > "$scriptHOME/uh_30.cfg";
 add_uniheat_common   > "$scriptHOME/uniheat.cfg";
};

function add_uniheat_table(){
 add_generalmap $1;
 add_uniheat_datastorage $1;
 add_uniheat_taglist $1;
 add_uniheat_winctrl $1;
 add_uniheat_winprop;
 add_uniheat_ctrl $1;
 add_custom_param $1;
 add_startupscript $1;
 add_uniheat_pulse $1;
 add_uniheat_redir_t $1;
}

function add_generalmap(){
 echo ";***********************************************************";
 echo ";*** КОНФИГУРАЦИОННЫЙ ФАЙЛ CRW-DAQ ДЛЯ СОЗДАНИЯ ПРОГРАММ  **";
 echo ";*** УПРАВЛЕНИЯ ПЕЧАМИ (ЗАГОТОВКА НА $1 ПЕЧЕЙ)            **";
 echo ";*** Подсистема управления печами UniHeat                 **";
 echo ";***********************************************************";
 echo "";
 echo ";******************************************************************************";
 echo "; Описание подключения к общей системе: (здесь i-номер печи)";
 echo "; 1) Обеспечить отсутствие конфликта имен кривых и тегов.";
 echo ";    Пересечений имен не будет, если учесть, что все используемые";
 echo ";    в UniHeat объекты имеют префикс \"UH_\" от Universal Heaters.";
 echo "; 2) Скопировать все файлы UH_*.BMP -> Bitmaps, UH_*.PAS -> DaqPas,";
 echo ";    UH_*.CFG -> Config, UH_*.CRC -> Circuits.";
 echo "; 3) Отредактировать файл UH_TP.CFG - селектор сигналов.";
 echo ";    При помощи тегов-селекторов UH_TNUMi,UH_PNUMi этот модуль должен";
 echo ";    обеспечить запись каналов давления и температуры в кривые";
 echo ";    UH_TREALi,UH_PREALi. Используется программа UH_REDIR.PAS.";
 echo ";    Задаются также списки имен каналов UH_TNAMES, UH_PNAMES.";
 echo ";    Имя каждого канала не длиннее 4 символов. Список разделяется";
 echo ";    слешем, например, T1/T2/T3...";
 echo "; 4) Выходные кривые UH_GATEi,UH_BLKi надо использовать для управления";
 echo ";    печами как сигнал включения печи и блокировки соответственно.";
 echo "; 5) При необходимости контроля по току надо обеспечить запись";
 echo ";    в тег UH_ICTRLi: 0=Ok, 1=Обрыв. По умолчанию не используется (всегда 0).";
 echo "; 6) При сложной логике блокировки внешняя программа может выставлять внешнюю";
 echo ";    блокировку UH_EBLKi, за которую эта программа должна целиком отвечать.";
 echo "; 7) Если часть каналов не используется, их при старте надо запретить";
 echo ";    записью 0 в тег UH_HENABL. Запрещенная печь \"умирает\",  то есть";
 echo ";    не реагирует на нажатие кнопки включения и на блокировки.";
 echo ";    Каналы UH_GATEi,UH_BLKi можно просто не использовать.";
 echo ";    Каналы UH_TNUMi,UH_PNUMi надо завести на выходы UH_REDIR, но входы";
 echo ";    можно не подключать.";
 echo "; 8) При необходимости внешнего управления (нагрев по графику)";
 echo ";    надо просто периодически записывать в тег UH_TGOALi нужную";
 echo ";    целевую температуру.";
 echo "; 9) Переменная [&UH_CTRL] UsesSpeakSrv = 1 включает речевые сообщения о";
 echo ";    блокировке и обрыве через синтезатор &SpeakSrv, вместо звуковых файлов.";
 echo ";******************************************************************************";
 echo "; 20000830 Alexey Kuryakin";
 echo "; 20131013 Alexey Kuryakin";
 echo "; 20140825 Alexey Kuryakin";
 echo "; 20240403 Alexey Kuryakin";
 echo ";******************************************************************************";
 echo "";
};

function add_uniheat_datastorage(){
 echo ";**************************";
 echo ";*** База данных кривых ***";
 echo ";**************************";
 echo "";
 for n in $(seq 1 $1); do add_uniheat_curve $n; done;
};

function add_uniheat_curve(){
 echo "[DataStorage]                             ;------ Печь  $1 ----------------";
 echo "UH_GATE$1   = Curve  0 150  Black  15    1 ;управление регулятором мощности";
 echo "UH_BLK$1    = Curve  0 150  Black  15    1 ;блокировка регулятора  мощности";
 echo "UH_TREAL$1  = Curve  0 150  Black  15    1 ;текущая температура печи";
 echo "UH_PREAL$1  = Curve  0 150  Black  15    1 ;текущее давление печи";
 echo "UH_QX$1     = Curve  0 150  Black  15    1 ;текущая скважность печи";
 echo "[]";
 echo "";
};

function add_uniheat_taglist(){
 echo ";**************************************";
 echo ";*** Описание глобальных переменных ***";
 echo ";**************************************";
 echo "";
 for n in $(seq 1 $1); do add_uniheat_tags $n; done;
 add_uniheat_tl_setting;
};

function add_uniheat_tags(){
 echo "[TagList]                    ;----- Печь  $1 --------------------------";
 echo "UH_HNAME$1   = string HEAT-$1 ;имя печи";
 echo "UH_TNUM$1    = integer $1     ;номер термопары печи";
 echo "UH_PNUM$1    = integer $1     ;номер датчика давления";
 echo "UH_HENABL$1  = integer 1     ;разрешение печи";
 echo "UH_TENABL$1  = integer 1     ;разрешение блокировки по T";
 echo "UH_PENABL$1  = integer 1     ;разрешение блокировки по P";
 echo "UH_WENABL$1  = integer 1     ;разрешение блокировки по WDT";
 echo "UH_BT$1      = integer 0     ;состояние кнопки печи";
 echo "UH_ST$1      = integer 0     ;статус печи";
 echo "UH_Q1$1      = real    50    ;скважность 1";
 echo "UH_Q2$1      = real    20    ;скважность 2";
 echo "UH_Q3$1      = real    0     ;скважность 3";
 echo "UH_TGOAL$1   = real    0     ;целевая температура";
 echo "UH_DT1$1     = real    5     ;зазор Q1/Q2";
 echo "UH_DT2$1     = real    2     ;зазор Q3/0";
 echo "UH_BLK$1     = integer 0     ;блокировка включена/выключена 1/0";
 echo "UH_ICTRL$1   = integer 0     ;контроль тока";
 echo "UH_PBLK$1    = real    1600  ;давление блокировки";
 echo "UH_TBLK$1    = real    1600  ;температура блокировки";
 echo "UH_WDT$1     = real    5     ;WatchDogTime в секундах";
 echo "UH_EBLK$1    = integer 0     ;внешняя блокировка";
 echo "UH_PERIOD$1  = real    3000  ;период генератора скважности в миллисекундах";
 echo "UH_TNAME$1   = string        ;имя сенсора текущей термопары (по селектору)";
 echo "UH_PNAME$1   = string        ;имя сенсора текущего давления (по селектору)";
 echo "UH_MODE$1    = integer 0     ;режим работы программы (нагрев/охлаждение)";
 echo "[]";
 echo "";
};

function add_uniheat_tl_setting(){
 echo "[TagList]                   ;----- Настройка печей -------------------";
 echo "UH_HNAMEX   = string  0     ;имя печи";
 echo "UH_Q1X      = real    0     ;скважность 1";
 echo "UH_Q2X      = real    0     ;скважность 2";
 echo "UH_Q3X      = real    0     ;скважность 3";
 echo "UH_DT1X     = real    0     ;зазор Q1/Q2";
 echo "UH_DT2X     = real    0     ;зазор Q3/0";
 echo "UH_PNUMX    = integer 0     ;номер датчика давления";
 echo "UH_TNUMX    = integer 0     ;номер датчика температуры";
 echo "UH_PBLKX    = real    0     ;давление блокировки";
 echo "UH_TBLKX    = real    0     ;температура блокировки";
 echo "UH_PREALX   = real    0     ;реальное давление";
 echo "UH_TREALX   = real    0     ;реальная температура";
 echo "UH_WDTX     = real    0     ;WatchDogTime в секундах";
 echo "UH_SOUNDX   = integer 1     ;разрешение звука";
 echo "UH_HENABLX  = integer 1     ;разрешение печи";
 echo "UH_TENABLX  = integer 1     ;разрешение блокировки по T";
 echo "UH_PENABLX  = integer 1     ;разрешение блокировки по P";
 echo "UH_WENABLX  = integer 1     ;разрешение блокировки по WDT";
 echo "UH_TNAMEX   = string        ;имя сенсора текущей температуры (по селектору)";
 echo "UH_PNAMEX   = string        ;имя сенсора текущего давления   (по селектору)";
 echo "UH_TGOALX   = real    0     ;целевая температура стабилизации";
 echo "UH_MODEX    = integer 0     ;режим работы программы (нагрев/охлаждение)";
 echo "[]";
 echo "";
};

function add_uniheat_winctrl(){
 echo ";************************************";
 echo ";*** Мнемосхема управления печами ***";
 echo ";************************************";
 echo "";
 echo "[Windows]";
 echo "UH_Control  = Circuit_Window";
 echo "[UH_Control]";
 echo "Name = ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "Circuit = ~~\resource\daqsite\uniheat\circuits\uh_$1.crc";
 for n in $(seq 1 $1); do add_uniheat_winctrl_linksens $n; done;
};

function add_uniheat_winctrl_linksens(){
 echo ";--------------- Печь  $1 --------------------------";
 echo "Link sensor   UH_HENABL$1  with   tag   UH_HENABL$1 device &UH_CTRL";
 echo "Link sensor   UH_BT$1      with   tag   UH_ST$1     device &UH_CTRL";
 echo "Link sensor   UH_TNUM$1    with   tag   UH_TNUM$1   device &UH_CTRL";
 echo "Link sensor   UH_TNAME$1   with   tag   UH_TNAME$1  device &UH_CTRL";
 echo "Link sensor   UH_TGOAL$1   with   tag   UH_TGOAL$1  device &UH_CTRL";
 echo "Link sensor   UH_TREAL$1   with   curve UH_TREAL$1  device &UH_CTRL";
 echo "Link sensor   UH_GATE$1    with   curve UH_GATE$1   device &UH_CTRL";
 echo "Link sensor   UH_BLK$1     with   curve UH_BLK$1    device &UH_CTRL";
 echo "Link sensor   UH_PROP$1    with   tag   UH_HNAME$1  device &UH_CTRL";
};

function add_uniheat_winprop(){
 echo "[]";
 echo "";
 echo ";*****************************";
 echo ";*** Диалог настройки печи ***";
 echo ";*****************************";
 echo "[Windows]";
 echo "UH_Property = Circuit_Window";
 echo "[UH_Property]";
 echo "Name = ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "Circuit = ~~\resource\daqsite\uniheat\circuits\uh_prop.crc";
 echo "Link sensor   UH_HNAMEX   with tag UH_HNAMEX  device &UH_CTRL";
 echo "Link sensor   UH_HENABLX  with tag UH_HENABLX device &UH_CTRL";
 echo "Link sensor   UH_PBLKX    with tag UH_PBLKX   device &UH_CTRL";
 echo "Link sensor   UH_PENABLX  with tag UH_PENABLX device &UH_CTRL";
 echo "Link sensor   UH_PREALX   with tag UH_PREALX  device &UH_CTRL";
 echo "Link sensor   UH_PNAMEX   with tag UH_PNAMEX  device &UH_CTRL";
 echo "Link sensor   UH_PNUMX    with tag UH_PNUMX   device &UH_CTRL";
 echo "Link sensor   UH_WDTX     with tag UH_WDTX    device &UH_CTRL";
 echo "Link sensor   UH_WENABLX  with tag UH_WENABLX device &UH_CTRL";
 echo "Link sensor   UH_TBLKX    with tag UH_TBLKX   device &UH_CTRL";
 echo "Link sensor   UH_TENABLX  with tag UH_TENABLX device &UH_CTRL";
 echo "Link sensor   UH_TREALX   with tag UH_TREALX  device &UH_CTRL";
 echo "Link sensor   UH_TNAMEX   with tag UH_TNAMEX  device &UH_CTRL";
 echo "Link sensor   UH_TNUMX    with tag UH_TNUMX   device &UH_CTRL";
 echo "Link sensor   UH_Q1X      with tag UH_Q1X     device &UH_CTRL";
 echo "Link sensor   UH_Q2X      with tag UH_Q2X     device &UH_CTRL";
 echo "Link sensor   UH_Q3X      with tag UH_Q3X     device &UH_CTRL";
 echo "Link sensor   UH_DT1X     with tag UH_DT1X    device &UH_CTRL";
 echo "Link sensor   UH_DT2X     with tag UH_DT2X    device &UH_CTRL";
 echo "Link sensor   UH_TGOALX   with tag UH_TGOALX  device &UH_CTRL";
 echo "Link sensor   UH_SOUNDX   with tag UH_SOUNDX  device &UH_CTRL";
 echo "Link sensor   UH_CHECKX   with                device &UH_CTRL";
 echo "Link sensor   UH_CANCELX  with                device &UH_CTRL";
 echo "Link sensor   UH_SAVEINIX with                device &UH_CTRL";
 echo "Link sensor   UH_LOADINIX with                device &UH_CTRL";
 echo "Link sensor   UH_MODE     with tag UH_MODEX   device &UH_CTRL";
 echo "Link sensor   UH_MODE_N   with tag UH_MODEX   device &UH_CTRL";
 echo "[]";
 echo "";
};

function add_uniheat_ctrl(){
 let "ai=$1+1";
 echo ";**********************************************************";
 echo ";*** Основная программа регулирования и настройки печей ***";
 echo ";**********************************************************";
 echo "[DeviceList]";
 echo "&UH_CTRL = device software program";
 echo "[&UH_CTRL]";
 echo "Comment        = Стабилизатор температуры печей + Блокировка по давлению и WatchDog";
 echo "InquiryPeriod  = 1";
 echo "DevicePolling  = 10, tpHighest";
 echo "ProgramSource  = ~~\resource\daqsite\uniheat\daqpas\uh_ctrl.pas";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "StartupScript  = [&UH_CTRL.StartupScript]";
 echo "FinallyScript  = [&UH_CTRL.FinallyScript]";
 echo "AnalogFifo     = 1000";
 echo "DigitalFifo    = 1000";
 echo "AnalogInputs   = $ai  ; температуры печей";
 for n in $(seq 1 $1); do add_uniheat_ctrl_ai $n; done;
 echo "DigitalInputs  = $ai  ; давления печей";
 for n in $(seq 1 $1); do add_uniheat_ctrl_di $n; done
 echo "AnalogOutputs  = $ai  ; скважности для управления печами";
 for n in $(seq 1 $1); do add_uniheat_ctrl_ao $n; done;
 echo "DigitalOutputs  = $ai ; блокировки для управления печами";
 for n in $(seq 1 $1); do add_uniheat_ctrl_do $n; done;
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [UniHeat.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ..\data\uniheat.ini";
 echo "CustomIniSection  = [UniHeat.CustomParameters]";
 echo "CustomIniBackups  = ..\data\custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "";
 echo "[ConfigFileList]";
 echo "ConfigFile = ..\data\uniheat.ini";
 echo "[]";
 echo "";
};
function add_uniheat_ctrl_ai(){
 echo "Link AnalogInput   $1  with curve UH_TREAL$1";
};
function add_uniheat_ctrl_di(){
 echo "Link DigitalInput  $1  with curve UH_PREAL$1  bit 0";
};
function add_uniheat_ctrl_ao(){
 echo "Link AnalogOutput  $1  with curve UH_QX$1  history 50";
};
function add_uniheat_ctrl_do(){
 echo "Link DigitalOutput $1  with curve UH_BLK$1  history 50";
};

function add_custom_param(){
 echo "[UniHeat.CustomParameters.TagList]";
 for n in $(seq  1 $1); do add_custom_params $n; done;
 echo "[]";
 echo "";
};

function add_custom_params(){
 echo ";------- Heat $1, custom parameters -----------------------------------------------------------";
 echo "TagList = UH_HNAME$1, UH_MODE$1";
 echo "TagList = UH_HENABL$1, UH_BT$1, UH_TGOAL$1, UH_TNUM$1, UH_TENABL$1, UH_TBLK$1, UH_PENABL$1, UH_PNUM$1,";
 echo "TagList = UH_WENABL$1, UH_WDT$1, UH_Q1$1, UH_Q2$1, UH_Q3$1, UH_DT1$1, UH_DT2$1, UH_PBLK$1, UH_PERIOD$1,";
};

function add_startupscript(){
 case $1 in
  8)  let x=1010; let y=170; ;;
  10) let x=1265; let y=170; ;;
  16) let x=1010; let y=290; ;;
  20) let x=1265; let y=290; ;;
  24) let x=1010; let y=410; ;;
  30) let x=1265; let y=410; ;;
  32) let x=1010; let y=525; ;;
  *)  ;;
 esac;
 echo "[&UH_CTRL.StartupScript]";
 echo "@WinShow   ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinDraw   ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Top=0|Left=0|Width=$x|Height=$y";
 echo "@WinDraw   ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Options=-Min,-Max,+Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo "@WinSelect ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinHide   ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinShow   ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinDraw   ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Top=170|Left=167|Width=920|Height=380";
 echo "@WinDraw   ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Options=-Min,-Max,-Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo "@WinSelect ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinHide   ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "[]";
 echo "";
 echo "[&UH_CTRL.FinallyScript]";
 echo "@WinHide ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@WinHide ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "[]";
 echo "";
 echo ";****************************************************************";
 echo ";*** Создать задания CRON для рисования окон системы UniHeat  *** ";
 echo ";****************************************************************";
 echo "[&CronSrv.StartupScript]";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_CONTROL.HOME 0 0 0";
 echo "@cron.job UH_CONTROL.HOME @WinShow ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@cron.job UH_CONTROL.HOME @WinDraw ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Top=0|Left=0|Width=$x|Height=$y";
 echo "@cron.job UH_CONTROL.HOME @WinDraw ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Options=-Min,-Max,+Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_CONTROL.SHOW 0 0 0";
 echo "@cron.job UH_CONTROL.SHOW @WinShow ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_CONTROL.HIDE 0 0 0";
 echo "@cron.job UH_CONTROL.HIDE @WinHide ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_CONTROL.SELECT 0 0 0";
 echo "@cron.job UH_CONTROL.SELECT @WinSelect ***_КОНТРОЛЬ_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_PROPERTY.HOME 0 0 0";
 echo "@cron.job UH_PROPERTY.HOME @WinShow ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo "@cron.job UH_PROPERTY.HOME @WinDraw ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Top=170|Left=167|Width=920|Height=380";
 echo "@cron.job UH_PROPERTY.HOME @WinDraw ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***|Options=-Min,-Max,-Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_PROPERTY.SHOW 0 0 0";
 echo "@cron.job UH_PROPERTY.SHOW @WinShow ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_PROPERTY.HIDE 0 0 0";
 echo "@cron.job UH_PROPERTY.HIDE @WinHide ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab UH_PROPERTY.SELECT 0 0 0";
 echo "@cron.job UH_PROPERTY.SELECT @WinSelect ***_НАСТРОЙКА_НАГРЕВАТЕЛЕЙ_UNIHEAT_***";
 echo ";---------------------------------------------------------------------";
 echo "[]";
 echo "";
};

function add_uniheat_pulse(){
 let "ai=$1+1";
 echo ";**************************************";
 echo ";*** Генератор скважности для печей ***";
 echo ";**************************************";
 echo "[DeviceList]";
 echo "&UH_PULSE = device software program";
 echo "[&UH_PULSE]";
 echo "Comment        = Генератор импульсов скважности для печей";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 4, tpTimeCritical";
 echo "ProgramSource  = ~~\resource\daqsite\uniheat\daqpas\uh_pulse.pas";
 echo "DigitalFifo    = 10000";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogInputs   = $ai ; скважности для управления печами";
 for n in $(seq 1 $1); do add_uniheat_pulse_ai $n; done;
 echo "DigitalOutputs = $ai ; сигналы включения печи";
 for n in $(seq 1 $1); do add_uniheat_pulse_do $n; done;
 echo "[]";
 echo "";
};
function add_uniheat_pulse_ai(){
 echo "Link AnalogInput   $1  with curve UH_QX$1";
};
function add_uniheat_pulse_do(){
 echo "Link DigitalOutput $1  with curve UH_GATE$1  history 50";
};

function add_uniheat_redir_t(){
 let "ao=$1+1";
 echo ";************************************";
 echo ";*** Селектор каналов температуры ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_TNAMES - имена датчиков T ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: T1/T2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[DeviceList]";
 echo "&UH_REDIR_T = device software program";
 echo "[&UH_REDIR_T]";
 echo "Comment        = Модуль для назначения термопар для печей";
 echo "InquiryPeriod  = 1";
 echo "DevicePolling  = 10, tpHighest";
 echo "ProgramSource  = ~~\Resource\DaqSite\UniHeat\DaqPas\UH_REDIR.PAS";
 echo "SelectorPrefix = UH_TNUM";
 echo "AnalogFifo     = 10000";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogOutputs  = $ao";
 for n in $(seq 1 $1); do add_uniheat_redir_t_ao $n; done;
 echo "[]";
 echo "";
 echo ";************************************";
 echo ";*** Селектор каналов давлений    ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_PNAMES - имена датчиков P ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: P1/P2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[DeviceList]";
 echo "&UH_REDIR_P = device software program";
 echo "[&UH_REDIR_P]";
 echo "Comment        = Модуль для назначения блокировок по давлению для печей";
 echo "InquiryPeriod  = 1";
 echo "DevicePolling  = 10, tpHighest";
 echo "ProgramSource  = ~~\resource\daqsite\uniheat\daqpas\uh_redir.pas";
 echo "SelectorPrefix = UH_PNUM";
 echo "AnalogFifo     = 10000";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogOutputs  = $ao";
 for n in $(seq 1 $1); do add_uniheat_redir_p_ao $n; done;
 echo "[]";
 echo "";
};
function add_uniheat_redir_t_ao(){
 echo "Link AnalogOutput  $1  with curve UH_TREAL$1  history 50";
};
function add_uniheat_redir_p_ao(){
 echo "Link AnalogOutput  $1  with curve UH_PREAL$1  history 50";
};

function add_uniheat_common(){
 echo ";***********************************************************";
 echo ";*** КОНФИГУРАЦИОННЫЙ ФАЙЛ CRW-DAQ ДЛЯ СОЗДАНИЯ ПРОГРАММ  **";
 echo ";*** УПРАВЛЕНИЯ ПЕЧАМИ (ЗАГОТОВКА НА 8,16,24 или 32 ПЕЧИ) **";
 echo ";*** Подсистема селектора каналов для управления UniHeat  **";
 echo ";*** Этот файл надо настраивать индивидуально для каждой  **";
 echo ";*** системы, т.к. кривые источников данных меняются.     **";
 echo ";*** Для задействования 8,16,24 или 32 печей используйте  **";
 echo ";*** конфигурации UH_8,UH_16,UH_24,UH_32 соответственно.  **";
 echo ";***********************************************************";
 echo "[ConfigFileList] ; Подключение конфигурации UniHeat";
 echo "ConfigFile = ~~\resource\daqsite\uniheat\config\uh_8.cfg";
 echo "[]";
 echo "";
 echo ";************************************";
 echo ";*** Селектор каналов температуры ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_TNAMES - имена датчиков T ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: T1/T2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[&UH_REDIR_T]";
 echo "UsesSmoothing  = 1";
 echo "AnalogInputs   = 11";
 for n in $(seq 1 10); do add_uniheat_comredir_tai $n; done;
 echo "[]";
 echo "[TagList]";
 echo "UH_TNAMES = string SENT1/SENT2/SENT3/SENT4/SENT5/SENT6/SENT7/SENT8/SENT9/SENT10";
 echo "[]";
 echo "";
 echo ";************************************";
 echo ";*** Селектор каналов давлений    ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_PNAMES - имена датчиков P ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: P1/P2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[&UH_REDIR_P]";
 echo "UsesSmoothing  = 1";
 echo "AnalogInputs   = 11";
 for n in $(seq 1 10); do add_uniheat_comredir_pai $n; done;
 echo "[]";
 echo "[TagList]";
 echo "UH_PNAMES = string SENP1/SENP2/SENP3/SENP4/SENP5/SENP6/SENP7/SENP8/SENP9/SENP10";
 echo "[]";
};
function add_uniheat_comredir_tai(){
 echo "Link AnalogInput $1  with curve SENSOR.T$1  smoothing 1e-4 0 2 2";
};
function add_uniheat_comredir_pai(){
 echo "Link AnalogInput $1  with curve SENSOR.P$1  smoothing 1e-4 0 2 2";
};

function main(){
 uniheat_main "$@";
};

main "$@";

##############
## END OF FILE
##############
