 {
 Functions for NetRS485.
 function RS485_get_tag(tag:Integer;def:Real):Real;
 function RS485_set_tag(tag:Integer; val:Real):Boolean;
 function RS485_inc_tag(tag:Integer; inc:Real):Boolean;
 procedure RS485_fixerror(tag,code:Integer; msg:String);
 function RS485_is_except(fid:Integer):Boolean;
 function RS485_un_except(fid:Integer):Integer;
 function RS485_as_except(fid:Integer; isOn:Boolean):Integer;
 function RS485_addr_ok(addr:Integer):Boolean;
 function RS485_insert_le(dat,le:String):String;
 function RS485_proxy_poll(cmd:String; ref,cid,tot,port,uid,fid:Integer; dat:String):String;
 function RS485_proxy_nice(cmd:String; ref,cid,tim,port,uid,fid:Integer; dat:String; wid:Integer):String;
 function RS485_proxy_reply(cmd,arg:String; var ref,cid,tim,port,uid,fid:Integer; var dat:String):Boolean;
 function RS485_errmsg(errno:Integer):String;
 procedure ClearNetRS485;
 procedure InitNetRS485;
 procedure FreeNetRS485;
 procedure PollNetRS485;
 }
 //
 // Get integer or real tag value or default.
 //
 function RS485_get_tag(tag:Integer; def:Real):Real;
 begin
  if TypeTag(tag)=1 then RS485_get_tag:=iGetTag(tag) else
  if TypeTag(tag)=2 then RS485_get_tag:=rGetTag(tag) else
  RS485_get_tag:=def;
 end;
 //
 // Set integer or real tag value.
 //
 function RS485_set_tag(tag:Integer; val:Real):Boolean;
 begin
  if TypeTag(tag)=1 then RS485_set_tag:=iSetTag(tag,Round(val)) else
  if TypeTag(tag)=2 then RS485_set_tag:=rSetTag(tag,val) else
  RS485_set_tag:=False;
 end;
 //
 // Increment integer or real tag value. Uses for error counting.
 //
 function RS485_inc_tag(tag:Integer; inc:Real):Boolean;
 begin
  if TypeTag(tag)=1 then RS485_inc_tag:=iSetTag(tag,iGetTag(tag)+Round(inc)) else
  if TypeTag(tag)=2 then RS485_inc_tag:=rSetTag(tag,rGetTag(tag)+inc) else
  RS485_inc_tag:=False;
 end;
 //
 // Fix RS485 error: increment tag & code & print message.
 //
 procedure RS485_fixerror(tag,code:Integer; msg:String);
 begin
  if not RS485_inc_tag(tag,1) then
  if code<>0 then bNul(FixError(code));
  if Length(msg)>0 then Problem(msg);
 end;
 //
 // Return true only if RS485 function id (fid) contains exception bit.
 //
 function RS485_is_except(fid:Integer):Boolean;
 begin
  RS485_is_except:=HasFlags(fid,RS485_er_Except);
 end;
 //
 // Unset exception bit, i.e. return RS485 function id (fid) without exception bit.
 //
 function RS485_un_except(fid:Integer):Integer;
 begin
  RS485_un_except:=iAnd(fid,iNot(RS485_er_Except));
 end;
 //
 // Assign exception bit, i.e. return RS485 function id (fid) with exception bit set ON.
 //
 function RS485_as_except(fid:Integer; isOn:Boolean):Integer;
 begin
  if isOn
  then RS485_as_except:=iOr(fid,RS485_er_Except)
  else RS485_as_except:=iAnd(fid,iNot(RS485_er_Except));
 end;
 //
 // Check if device address is OK.
 //
 function RS485_addr_ok(addr:Integer):Boolean;
 begin
  RS485_addr_ok:=((addr>=RS485_MinUnitId) and (addr<=RS485_MaxUnitId));
 end;
 //
 // Insert Line Ending In Data
 //
 function RS485_insert_le(dat,le:String):String;
 var s:string;
 begin
  s:='';
  if length(le)>0 then begin 
   if Frac(Length(le)/2)<>0 then s:=dat else s:=dat+'//'+le;
  end else s:=dat;
  RS485_insert_le:=s;
  s:='';
 end;
 //
 // Encode command to send to &RS485Proxy or reply to driver.
 // cmd  - (in)  command name, expected @RS485.Poll, @RS485.Reply, @RS485.Refuse, @RS485.Timeout or empty string.
 // ref  - (in)  sender device reference, usually driver's devMySelf
 // cid  - (in)  command id, any user defined number
 // tot  - (in)  timeout, ms
 // port - (in)  logical port
 // uid  - (in)  unit id
 // dat  - (in)  PDU data; PDU=(fid+dat)
 // Return formatted data ready to send to &RS485Proxy.
 //
 function RS485_proxy_poll(cmd:String; ref,cid,tot,port,uid:Integer; dat:String):String;
 begin
  if Length(cmd)=0 then cmd:='@RS485.Poll';
  RS485_proxy_poll:=cmd+' '+Str(ref)+' '+Str(cid)+' '+Str(tot)+' '
                           +Str(port)+' '+Str(uid)+' $$'+dat+EOL;
 end;
 //
 // Nice (human-readable) format to print &RS485Proxy messages.
 // If wid>0, use $HEX_ENCODE(dat)...; if wid<0 use Trim(dat)...
 //
 function RS485_proxy_nice(cmd:String; ref,cid,tim,port,uid:Integer; dat:String; wid:Integer):String;
 begin
  if wid<0 then if length(dat)>abs(wid)
  then dat:=Copy(dat,1,abs(wid))+'...';
  if wid>0 then if length(dat)<=wid
  then dat:='$$'+dat
  else dat:='$$'+Copy(dat,1,wid div 2)+'...';
  RS485_proxy_nice:=cmd+' '+RefInfo(ref,'Name')+' '+Str(cid)+' '+Str(tim)
                        +' '+Str(port)+' '+Str(uid)+' '+dat;
 end;
 //
 // Decode Reply/Timeout/Refuse message received from &RS485Proxy.
 // cmd  - (in)  Should be @RS485.Poll or @RS485.Timeout or @RS485.Refuse
 // arg  - (in)  command argument list[7]: ref cid tim port uid $$dat
 // ref  - (out) sender device reference, expected &RS485Proxy
 // cid  - (out) command id, any user defined number
 // tim  - (out) response time, ms
 // port - (out) logical port
 // uid  - (out) unit id
 // dat  - (out) data;
 // Return true if message looks acceptable to use.
 // Also sets RS485_errno value on errors.
 //
 function RS485_proxy_reply(cmd,arg:String; var ref,cid,tim,port,uid:Integer; var dat:String):Boolean;
 var p,cmdid:Integer;
 begin
  ref:=0;
  cmdid:=HashList_GetLink(StdIn_CmdHashTab,cmd);
  if (cmdid=cmd_NetRS485Poll) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then RS485_errno:=RS485_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<=0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not RS485_addr_ok(uid) then begin ref:=0; RS485_errno:=RS485_er_BADUID; end;
    if Length(dat)=0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetRS485Reply) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then RS485_errno:=RS485_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not RS485_addr_ok(uid) then begin ref:=0; RS485_errno:=RS485_er_BADUID; end;
    if Length(dat)=0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetRS485Timeout) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then RS485_errno:=RS485_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not RS485_addr_ok(uid) then begin ref:=0; RS485_errno:=RS485_er_BADUID; end;
    if Length(dat)=0 then begin ref:=0; RS485_errno:=RS485_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetRS485Refuse) then begin
   ref:=Val(ExtractWord(1,arg));   if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
   cid:=Val(ExtractWord(2,arg));   if ref=0 then RS485_errno:=RS485_er_BADREF;
   tim:=Val(ExtractWord(3,arg));
   port:=Val(ExtractWord(4,arg));
   uid:=Val(ExtractWord(5,arg));
   dat:=Trim(SkipWords(5,arg));
  end else
  RS485_errno:=RS485_er_BADCMD;
  if not IsRefDevice(ref) then begin ref:=0; RS485_errno:=RS485_er_BADREF; end;
  if ref=0  then begin cid:=0; tim:=0; port:=0; uid:=0; dat:=''; end;
  RS485_proxy_reply:=(ref<>0);
 end;
 //
 // Get RS485 error message by error code (errno).
 //
 function RS485_errmsg(errno:Integer):String;
 begin
  if errno=RS485_er_OK     then RS485_errmsg:='SUCCESS('+Str(errno)+')' else
  if errno=RS485_er_ILLADR then RS485_errmsg:='ILLEGAL_ADDRESS('+Str(errno)+')' else
  if errno=RS485_er_BADUID then RS485_errmsg:='BAD_UNIT('+Str(errno)+')' else
  if errno=RS485_er_BADREF then RS485_errmsg:='BAD_REFERENCE('+Str(errno)+')' else
  RS485_errmsg:='FAILURE('+Str(errno)+')'; 
 end;
 //
 // Clear NetRS485.
 //
 procedure ClearNetRS485;
 begin
 end;
 //
 // Initialize NetRS485.
 //
 procedure InitNetRS485;
 begin
  RS485_errno:=RS485_er_OK;
  InitDevice(devRS485Proxy, RS485Proxy, 1);
  cmd_NetRS485Poll    := RegisterStdInCmd('@RS485.Poll',    '');
  cmd_NetRS485Reply   := RegisterStdInCmd('@RS485.Reply',   '');
  cmd_NetRS485Timeout := RegisterStdInCmd('@RS485.Timeout', '');
  cmd_NetRS485Refuse  := RegisterStdInCmd('@RS485.Refuse',  '');
 end;
 //
 // Finalize NetRS485.
 //
 procedure FreeNetRS485;
 begin
 end;
 //
 // Poll NetRS485.
 //
 procedure PollNetRS485;
 begin
 end;
 
