 {
 Functions for DbLibrary.
 function  db_engine_list:String;
 function  db_engine_uses:Integer;
 function  db_engine_uses_name:String;
 procedure db_engine_uses_assign(aEngine:Integer);
 function  db_AdoTypeToString(FieldType:Integer):String;
 function  db_SqlDbTypeToString(FieldType:Integer):String;
 function  db_EngineFieldTypeToString(Engine,FieldType:Integer):String;
 function  db_FieldTypeToString(FieldType:Integer):String;
 function  db_AdoTypeToTagType(FieldType:Integer):Integer;
 function  db_SqlDbTypeToTagType(FieldType:Integer):Integer;
 function  db_EngineTypeToTagType(Engine,FieldType:Integer):Integer;
 function  db_FieldTypeToTagType(FieldType:Integer):Integer;
 function  db_AdoTypeIsDateTime(FieldType:Integer):Boolean;
 function  db_SqlDbTypeIsDateTime(FieldType:Integer):Boolean;
 function  db_EngineTypeIsDateTime(Engine,FieldType:Integer):Boolean;
 function  db_FieldTypeIsDateTime(FieldType:Integer):Boolean;
 function  db_EngineTypeIsBlob(Engine,FieldType:Integer):Boolean;
 function  db_FieldTypeIsBlob(FieldType:Integer):Boolean;
 function  db_DetectBlobImageType(blob:String):String;
 function  db_AdoDateTimeToMs(FieldData:Real; FieldType:Integer):Real;
 function  db_MsToAdoDateTime(ms:Real; FieldType:Integer):Real;
 function  db_SqlDbDateTimeToMs(FieldData:Real; FieldType:Integer):Real;
 function  db_MsToSqlDbDateTime(ms:Real; FieldType:Integer):Real;
 function  db_EngineDateTimeToMs(FieldData:Real; Engine,FieldType:Integer):Real;
 function  db_MsToEngineDateTime(ms:Real; Engine,FieldType:Integer):Real;
 function  db_FieldDateTimeToMs(FieldData:Real; Engine,FieldType:Integer):Real;
 function  db_MsToFieldDateTime(ms:Real; FieldType:Integer):Real; 
 function  db_query_connectionstring(cs,id:String):String;
 function  db_subst_connectionstring(cs,id,sv:String):String;
 function  db_sample_connectionstring(arg:String):String;
 function  db_query_dbapimasterkey:String;
 function  db_build_connectionstring(keys,server,dbname,uid,pwd,opt:String; mode:Integer):String;
 function  db_validate_adapt_dbname(constr:String):String;
 function  db_validate_known_providers(constr:String):String;
 function  db_build_selectalltables(dbtype,database:String):String;
 function  db_supporteddbtypes:String;
 procedure db_bugreport_mode(con,mode:Integer);
 function  db_just_sql_query(con:Integer; sql:String):Boolean;
 function  db_easy_sql_query(cs,sql:String):Boolean;
 function  db_default_uid(dbtype:String):String;
 function  db_default_pwd(dbtype:String):String;
 function  db_sample_database_file(arg:String):String;
 function  db_detect_dbtype(con:Integer; constr:String):String;
 function  db_query_selectalltables(con:Integer; dbtype:String):String;
 function  db_query_showtables(constr,dbtype:String):String;
 procedure db_read_connectionstring_samples;
 procedure db_read_selectalltables_samples;
 procedure db_read_database_file_samples;
 function  db_is_firebird_exe_running:Boolean;
 function  db_create_local_fdb(fdb,uid,pwd,opt,constr:String; mode:Integer):Boolean;
 procedure ClearDbLibrary;
 procedure InitDbLibrary;
 procedure FreeDbLibrary;
 procedure PollDbLibrary;
 }
 {
 Get available engine list.
 }
 function db_engine_list:String;
 begin
  db_engine_list:=TrimDef(_DbLibrary_Internals_.engine_list,'ADO,SQLDB,ZEOS');
 end;
 {
 Get/set uses engine ID and name.
 }
 function db_engine_uses:Integer;
 begin
  db_engine_uses:=_DbLibrary_Internals_.engine_uses;
 end;
 function db_engine_uses_name:String;
 begin
  db_engine_uses_name:=ExtractWord(db_engine_uses,db_engine_list);
 end;
 procedure db_engine_uses_assign(aEngine:Integer);
 begin
  case aEngine of
   db_engine_ado,db_engine_sqldb,db_engine_zeos: begin
    _DbLibrary_Internals_.engine_uses:=aEngine;
    _DbLibrary_Internals_.SupportedDbTypes:='';
   end;
  end;
 end;
 {
 Converts an ADO native types into string related.
 }
 function db_AdoTypeToString(FieldType:Integer):String;
 var s:String;
 begin
  s := 'adUnknown';
  case FieldType of
   adChar             : s := 'adChar';
   adVarChar          : s := 'adVarChar';
   adBSTR             : s := 'adBSTR';
   adWChar            : s := 'adWChar';
   adVarWChar         : s := 'adVarWChar';
   adBoolean          : s := 'adBoolean';
   adTinyInt          : s := 'adTinyInt';
   adUnsignedTinyInt  : s := 'adUnsignedTinyInt';
   adSmallInt         : s := 'adSmallInt';
   adUnsignedSmallInt : s := 'adUnsignedSmallInt';
   adInteger          : s := 'adInteger';
   adUnsignedInt      : s := 'adUnsignedInt';
   adBigInt           : s := 'adBigInt';
   adUnsignedBigInt   : s := 'adUnsignedBigInt';
   adSingle           : s := 'adSingle';
   adDouble           : s := 'adDouble';
   adDecimal          : s := 'adDecimal';
   adNumeric          : s := 'adNumeric';
   adVarNumeric       : s := 'adVarNumeric';
   adCurrency         : s := 'adCurrency';
   adDBDate           : s := 'adDBDate';
   adDBTime           : s := 'adDBTime';
   adDate             : s := 'adDate';
   adDBTimeStamp      : s := 'adDBTimeStamp';
   adFileTime         : s := 'adFileTime';
   adLongVarChar      : s := 'adLongVarChar';
   adLongVarWChar     : s := 'adLongVarWChar';
   adBinary           : s := 'adBinary';
   adVarBinary        : s := 'adVarBinary';
   adLongVarBinary    : s := 'adLongVarBinary';
   adGUID             : s := 'adGUID';
   adEmpty            : s := 'adEmpty';
   adError            : s := 'adError';
   adArray            : s := 'adArray';
   adChapter          : s := 'adChapter';
   adIDispatch        : s := 'adIDispatch';
   adIUnknown         : s := 'adIUnknown';
   adPropVariant      : s := 'adPropVariant';
   adUserDefined      : s := 'adUserDefined';
   adVariant          : s := 'adVariant';
  end;
  db_AdoTypeToString:=s; s:='';
 end;
 {
 Converts an SQLDB native types into string related.
 }
 function db_SqlDbTypeToString(FieldType:Integer):String;
 var s:String;
 begin
  s := 'ftUnknown';
  case FieldType of
   ftUnknown       : s:='ftUnknown';       // 0  Unknown data type
   ftString        : s:='ftString';        // 1  String data value (ansistring)
   ftSmallint      : s:='ftSmallint';      // 2  Small integer value(1 byte, signed)
   ftInteger       : s:='ftInteger';       // 3  Regular integer value (4 bytes, signed)
   ftWord          : s:='ftWord';          // 4  Word-sized value(2 bytes, unsigned)
   ftBoolean       : s:='ftBoolean';       // 5  Boolean value
   ftFloat         : s:='ftFloat';         // 6  Floating point value (double)
   ftCurrency      : s:='ftCurrency';      // 7  Currency value (4 decimal points)
   ftBCD           : s:='ftBCD';           // 8  Binary Coded Decimal value (DECIMAL and NUMERIC SQL types)
   ftDate          : s:='ftDate';          // 9  Date value
   ftTime          : s:='ftTime';          // 10 Time value
   ftDateTime      : s:='ftDateTime';      // 11 Date/Time (timestamp) value
   ftBytes         : s:='ftBytes';         // 12 Array of bytes value, fixed size (unytped)
   ftVarBytes      : s:='ftVarBytes';      // 13 Array of bytes value, variable size (untyped)
   ftAutoInc       : s:='ftAutoInc';       // 14 Auto-increment integer value (4 bytes)
   ftBlob          : s:='ftBlob';          // 15 Binary data value (no type, no size)
   ftMemo          : s:='ftMemo';          // 16 Binary text data (no size)
   ftGraphic       : s:='ftGraphic';       // 17 Graphical data value (no size)
   ftFmtMemo       : s:='ftFmtMemo';       // 18 Formatted memo data value (no size)
   ftParadoxOle    : s:='ftParadoxOle';    // 19 Paradox OLE field data (no size)
   ftDBaseOle      : s:='ftDBaseOle';      // 20 Paradox OLE field data
   ftTypedBinary   : s:='ftTypedBinary';   // 21 Binary typed data (no size)
   ftCursor        : s:='ftCursor';        // 22 Cursor data value (no size)
   ftFixedChar     : s:='ftFixedChar';     // 23 Fixed character array (string)
   ftWideString    : s:='ftWideString';    // 24 Widestring (2 bytes per character)
   ftLargeint      : s:='ftLargeint';      // 25 Large integer value (8-byte)
   ftADT           : s:='ftADT';           // 26 ADT value
   ftArray         : s:='ftArray';         // 27 Array data
   ftReference     : s:='ftReference';     // 28 Reference data
   ftDataSet       : s:='ftDataSet';       // 29 Dataset data (blob)
   ftOraBlob       : s:='ftOraBlob';       // 30 Oracle BLOB data
   ftOraClob       : s:='ftOraClob';       // 31 Oracle CLOB data
   ftVariant       : s:='ftVariant';       // 32 Variant data value
   ftInterface     : s:='ftInterface';     // 33 interface data value
   ftIDispatch     : s:='ftIDispatch';     // 34 Dispatch data value
   ftGuid          : s:='ftGuid';          // 35 GUID data value
   ftTimeStamp     : s:='ftTimeStamp';     // 36 Timestamp data value
   ftFMTBcd        : s:='ftFMTBcd';        // 37 Formatted BCD (Binary Coded Decimal) value.
   ftFixedWideChar : s:='ftFixedWideChar'; // 38 Fixed wide character date (2 bytes per character)
   ftWideMemo      : s:='ftWideMemo';      // 39 Widestring memo data
  end;
  db_SqlDbTypeToString:=s; s:='';
 end;
 {
 Converts an Engine native types into string related.
 }
 function db_EngineFieldTypeToString(Engine,FieldType:Integer):String;
 var s:String;
 begin
  s := '';
  case Engine of
   db_engine_ado:   s:=db_AdoTypeToString(FieldType);
   db_engine_sqldb: s:=db_SqlDbTypeToString(FieldType);
   db_engine_zeos:  s:=db_SqlDbTypeToString(FieldType);
  end;
  db_EngineFieldTypeToString:=s; s:='';
 end;
 {
 Converts an db_engine_uses native types into string related.
 }
 function db_FieldTypeToString(FieldType:Integer):String;
 begin
  db_FieldTypeToString:=db_EngineFieldTypeToString(db_engine_uses,FieldType);
 end;
 {
 Converts an ADO native types into tag types.
 }
 function db_AdoTypeToTagType(FieldType:Integer):Integer;
 var TagType:Integer;
 begin
  TagType := tag_type_nil;
  case FieldType of
   adChar             : TagType := tag_type_string;
   adVarChar          : TagType := tag_type_string;
   adBSTR             : TagType := tag_type_string;
   adWChar            : TagType := tag_type_string;
   adVarWChar         : TagType := tag_type_string;
   adBoolean          : TagType := tag_type_int;
   adTinyInt          : TagType := tag_type_int;
   adUnsignedTinyInt  : TagType := tag_type_int;
   adSmallInt         : TagType := tag_type_int;
   adUnsignedSmallInt : TagType := tag_type_int;
   adInteger          : TagType := tag_type_int;
   adUnsignedInt      : TagType := tag_type_int;
   adBigInt           : TagType := tag_type_real;
   adUnsignedBigInt   : TagType := tag_type_real;
   adSingle           : TagType := tag_type_real;
   adDouble           : TagType := tag_type_real;
   adDecimal          : TagType := tag_type_real;
   adNumeric          : TagType := tag_type_real;
   adVarNumeric       : TagType := tag_type_real;
   adCurrency         : TagType := tag_type_real;
   adDBDate           : TagType := tag_type_real;
   adDBTime           : TagType := tag_type_real;
   adDate             : TagType := tag_type_real;
   adDBTimeStamp      : TagType := tag_type_real;
   adFileTime         : TagType := tag_type_real;
   adLongVarChar      : TagType := tag_type_string;
   adLongVarWChar     : TagType := tag_type_string;
   adBinary           : TagType := tag_type_nil;
   adVarBinary        : TagType := tag_type_nil;
   adLongVarBinary    : TagType := tag_type_nil;
   adGUID             : TagType := tag_type_string;
   adEmpty            : TagType := tag_type_nil;
   adError            : TagType := tag_type_int;
   adArray            : TagType := tag_type_nil;
   adChapter          : TagType := tag_type_nil;
   adIDispatch        : TagType := tag_type_nil;
   adIUnknown         : TagType := tag_type_nil;
   adPropVariant      : TagType := tag_type_string;
   adUserDefined      : TagType := tag_type_nil;
   adVariant          : TagType := tag_type_string;
  end;
  db_AdoTypeToTagType:=TagType;
 end;
 {
 Converts an SQLDB native types into tag types.
 }
 function db_SqlDbTypeToTagType(FieldType:Integer):Integer;
 var TagType:Integer;
 begin
  TagType := tag_type_nil;
  case FieldType of
   ftUnknown       : TagType := tag_type_nil;
   ftString        : TagType := tag_type_string;
   ftSmallint      : TagType := tag_type_int;
   ftInteger       : TagType := tag_type_int;
   ftWord          : TagType := tag_type_int;
   ftBoolean       : TagType := tag_type_int;
   ftFloat         : TagType := tag_type_real;
   ftCurrency      : TagType := tag_type_real;
   ftBCD           : TagType := tag_type_real;
   ftDate          : TagType := tag_type_real;
   ftTime          : TagType := tag_type_real;
   ftDateTime      : TagType := tag_type_real;
   ftBytes         : TagType := tag_type_string;
   ftVarBytes      : TagType := tag_type_string;
   ftAutoInc       : TagType := tag_type_int;
   ftBlob          : TagType := tag_type_string;
   ftMemo          : TagType := tag_type_string;
   ftGraphic       : TagType := tag_type_string;
   ftFmtMemo       : TagType := tag_type_string;
   ftParadoxOle    : TagType := tag_type_string;
   ftDBaseOle      : TagType := tag_type_string;
   ftTypedBinary   : TagType := tag_type_string;
   ftCursor        : TagType := tag_type_string;
   ftFixedChar     : TagType := tag_type_string;
   ftWideString    : TagType := tag_type_string;
   ftLargeint      : TagType := tag_type_real;
   ftADT           : TagType := tag_type_string;
   ftArray         : TagType := tag_type_string;
   ftReference     : TagType := tag_type_string;
   ftDataSet       : TagType := tag_type_string;
   ftOraBlob       : TagType := tag_type_string;
   ftOraClob       : TagType := tag_type_string;
   ftVariant       : TagType := tag_type_string;
   ftInterface     : TagType := tag_type_string;
   ftIDispatch     : TagType := tag_type_string;
   ftGuid          : TagType := tag_type_string;
   ftTimeStamp     : TagType := tag_type_real;
   ftFMTBcd        : TagType := tag_type_real;
   ftFixedWideChar : TagType := tag_type_string;
   ftWideMemo      : TagType := tag_type_string;
  end;
  db_SqlDbTypeToTagType:=TagType;
 end;
 {
 Converts an Engine native types into tag types.
 }
 function db_EngineTypeToTagType(Engine,FieldType:Integer):Integer;
 var TagType:Integer;
 begin
  TagType := tag_type_nil;
  case Engine of
   db_engine_ado:   TagType := db_AdoTypeToTagType(FieldType);
   db_engine_sqldb: TagType := db_SqlDbTypeToTagType(FieldType);
   db_engine_zeos:  TagType := db_SqlDbTypeToTagType(FieldType);
  end;
  db_EngineTypeToTagType:=TagType;
 end;
 {
 Converts an db_engine_uses native types into tag types.
 }
 function db_FieldTypeToTagType(FieldType:Integer):Integer;
 begin
  db_FieldTypeToTagType:=db_EngineTypeToTagType(db_engine_uses,FieldType);
 end; 
 {
 ADO check FieldType is one of possible DateTime formats.
 }
 function db_AdoTypeIsDateTime(FieldType:Integer):Boolean;
 var flag:Boolean;
 begin
  flag:=false;
  case FieldType of
   adDBDate           : flag := true;
   adDBTime           : flag := true;
   adDate             : flag := true;
   adDBTimeStamp      : flag := true;
   adFileTime         : flag := true;
  end;
  db_AdoTypeIsDateTime:=flag;
 end;
 {
 SQLDB check FieldType is one of possible DateTime formats.
 }
 function db_SqlDbTypeIsDateTime(FieldType:Integer):Boolean;
 var flag:Boolean;
 begin
  flag:=false;
  case FieldType of
   ftDate          : flag := true;
   ftTime          : flag := true;
   ftDateTime      : flag := true;
   ftTimeStamp     : flag := true;
  end;
  db_SqlDbTypeIsDateTime:=flag;
 end;
 {
 Engine check FieldType is one of possible DateTime formats.
 }
 function db_EngineTypeIsDateTime(Engine,FieldType:Integer):Boolean;
 var flag:Boolean;
 begin
  flag:=false;
  case Engine of
   db_engine_ado   : flag := db_AdoTypeIsDateTime(FieldType);
   db_engine_sqldb : flag := db_SqlDbTypeIsDateTime(FieldType);
   db_engine_zeos  : flag := db_SqlDbTypeIsDateTime(FieldType);
  end;
  db_EngineTypeIsDateTime:=flag;
 end;
 {
 db_engine_uses check FieldType is one of possible DateTime formats.
 }
 function db_FieldTypeIsDateTime(FieldType:Integer):Boolean;
 begin
  db_FieldTypeIsDateTime:=db_EngineTypeIsDateTime(db_engine_uses,FieldType);
 end;
 {
 Engine check FieldType is one of possible BLOB formats.
 }
 function db_EngineTypeIsBlob(Engine,FieldType:Integer):Boolean;
 var flag:Boolean;
 begin
  flag:=false;
  case Engine of
   db_engine_ado:
   case FieldType of
    adBinary,adVarBinary,adLongVarBinary: flag:=true;
   end;
   db_engine_sqldb,db_engine_zeos:
   case FieldType of
    ftBlob,ftGraphic,ftFmtMemo,ftParadoxOle,ftDBaseOle,
    ftTypedBinary,ftOraBlob,ftOraClob,ftBytes,ftVarBytes: flag:=true;
   end;
  end;
  db_EngineTypeIsBlob:=flag;
 end;
 {
 db_engine_uses check FieldType is one of possible BLOB formats.
 }
 function db_FieldTypeIsBlob(FieldType:Integer):Boolean;
 begin
  db_FieldTypeIsBlob:=db_EngineTypeIsBlob(db_engine_uses,FieldType);
 end;
 {
 Detect blob is (possible) a picture (bmp,gif,png,pbm,pgm,ppm,jpg,xpm,tif,pcx).
 }
 function db_DetectBlobImageType(blob:String):String;
 begin
  if (blob='')
  then db_DetectBlobImageType:=''
  else db_DetectBlobImageType:=ParamStr('DetectBlobImageType b64 '+base64_encode(blob));
 end;
 {
 ADO convert FieldData with FieldType of any DateTime type to (ms since Xmas).
 }
 function db_AdoDateTimeToMs(FieldData:Real; FieldType:Integer):Real;
 var ms:Real;
 begin
  ms:=FieldData;
  case FieldType of
   adDBDate           : ms := OleTimeToMs(FieldData);
   adDBTime           : ms := OleTimeToMs(FieldData);
   adDate             : ms := OleTimeToMs(FieldData);
   adDBTimeStamp      : ms := OleTimeToMs(FieldData);
   adFileTime         : ms := FileTimeToMs(FieldData);
  end;
  db_AdoDateTimeToMs:=ms;
 end;
 {
 ADO convert (ms since Xmas) to DateTime data of given FieldType.
 }
 function db_MsToAdoDateTime(ms:Real; FieldType:Integer):Real;
 var FieldData:Real;
 begin
  FieldData:=ms;
  case FieldType of
   adDBDate           : FieldData := MsToOleTime(ms);
   adDBTime           : FieldData := MsToOleTime(ms);
   adDate             : FieldData := MsToOleTime(ms);
   adDBTimeStamp      : FieldData := MsToOleTime(ms);
   adFileTime         : FieldData := MsToFileTime(ms);
  end;
  db_MsToAdoDateTime:=FieldData;
 end;
 {
 SQLDB convert FieldData with FieldType of any DateTime type to (ms since Xmas).
 }
 function db_SqlDbDateTimeToMs(FieldData:Real; FieldType:Integer):Real;
 var ms:Real;
 begin
  ms:=FieldData;
  case FieldType of
   ftDate           : ms := OleTimeToMs(FieldData);
   ftTime           : ms := OleTimeToMs(FieldData);
   ftDateTime       : ms := OleTimeToMs(FieldData);
   ftTimeStamp      : ms := OleTimeToMs(FieldData);
  end;
  db_SqlDbDateTimeToMs:=ms;
 end;
 {
 SQLDB convert (ms since Xmas) to DateTime data of given FieldType.
 }
 function db_MsToSqlDbDateTime(ms:Real; FieldType:Integer):Real;
 var FieldData:Real;
 begin
  FieldData:=ms;
  case FieldType of
   ftDate           : FieldData := MsToOleTime(ms);
   ftTime           : FieldData := MsToOleTime(ms);
   ftDateTime       : FieldData := MsToOleTime(ms);
   ftTimeStamp      : FieldData := MsToOleTime(ms);
  end;
  db_MsToSqlDbDateTime:=FieldData;
 end;
 {
 Engine convert FieldData with FieldType of any DateTime type to (ms since Xmas).
 }
 function db_EngineDateTimeToMs(FieldData:Real; Engine,FieldType:Integer):Real;
 var ms:Real;
 begin
  ms:=FieldData;
  case Engine of
   db_engine_ado    : ms := db_AdoDateTimeToMs(FieldData,FieldType);
   db_engine_sqldb  : ms := db_SqlDbDateTimeToMs(FieldData,FieldType);
   db_engine_zeos   : ms := db_SqlDbDateTimeToMs(FieldData,FieldType);
  end;
  db_EngineDateTimeToMs:=ms;
 end;
 {
 Engine convert (ms since Xmas) to DateTime data of given FieldType.
 }
 function db_MsToEngineDateTime(ms:Real; Engine,FieldType:Integer):Real;
 var FieldData:Real;
 begin
  FieldData:=ms;
  case Engine of
   db_engine_ado    : FieldData := db_MsToAdoDateTime(ms,FieldType);
   db_engine_sqldb  : FieldData := db_MsToSqlDbDateTime(ms,FieldType);
   db_engine_zeos   : FieldData := db_MsToSqlDbDateTime(ms,FieldType);
  end;
  db_MsToEngineDateTime:=FieldData;
 end;
 {
 db_engine_uses FieldData with FieldType of any DateTime type to (ms since Xmas).
 }
 function db_FieldDateTimeToMs(FieldData:Real; Engine,FieldType:Integer):Real;
 begin
  db_FieldDateTimeToMs:=db_EngineDateTimeToMs(FieldData,db_engine_uses,FieldType);
 end;
 {
 db_engine_uses convert (ms since Xmas) to DateTime data of given FieldType.
 }
 function db_MsToFieldDateTime(ms:Real; FieldType:Integer):Real;
 begin
  db_MsToFieldDateTime:=db_MsToEngineDateTime(ms,db_engine_uses,FieldType);
 end;
 {
 Multiname query of ConnectionString parameters because
 same parameters may have different names (User Id,User,UID):
 cs:=db_query_connectionstring(cs,'User Id;User;UID');
 }
 function db_query_connectionstring(cs,id:String):String;
 var i:Integer; rt,dl:String;
  procedure Cleanup;
  begin
   rt:=''; dl:='';
  end;
 begin
  Cleanup;
  dl:=WordDelims(db_sym_punctchars);
  for i:=1 to WordCount(id) do if (rt='') then begin
   rt:=CookieScan(cs,ExtractWord(i,id),Ord(';'));
   rt:=Trim(rt);
  end;
  sNul(WordDelims(dl));
  db_query_connectionstring:=rt;
  Cleanup;
 end;
 {
 Multiname substitution for ConnectionString uses because
 same parameters may have different names (User Id,User,UID):
 cs:=db_subst_connectionstring(cs,'User Id;User;UID','SYSDBA');
 }
 function db_subst_connectionstring(cs,id,sv:String):String;
 var i,pe,nc,Lines:Integer; rt,sl,sn,dl:String;
  procedure Cleanup;
  begin
   rt:=''; sl:=''; sn:=''; dl:='';
  end;
 begin
  Cleanup;
  rt:=Trim(cs); nc:=0;
  cs:=Trim(cs); id:=Trim(id); sv:=Trim(sv);
  if (cs<>'') and (id<>'') {and (sv<>'')} then begin
   Lines:=StringToText(StringReplace(cs,';',EOL,rfReplaceAll));
   for i:=0 to text_numln(Lines)-1 do begin
    sl:=text_getln(Lines,i);
    pe:=Pos('=',sl);
    if (pe>0) then begin
     sn:=Trim(Copy(sl,1,pe-1));
     dl:=WordDelims(db_sym_punctchars);
     if (WordIndex(sn,id)>0) then begin
      if text_putln(Lines,i,sn+'='+sv)
      then nc:=nc+1;
     end;
     sNul(WordDelims(dl));
    end;
   end;
   for i:=text_numln(Lines)-1 downto 0 do begin
    sl:=text_getln(Lines,i);
    if IsEmptyStr(sl) or IsSameText(StrFetch(sl,length(sl)),'=')
    then bNul(text_delln(Lines,i));
   end;
   if (nc>0) then rt:=StringReplace(TextToString(Lines),EOL,';',rfReplaceAll);
   bNul(text_free(Lines));
  end;
  db_subst_connectionstring:=rt;
  Cleanup;
 end;
 {
 Get sample connection string from library of samples.
 Examples:
 s:=db_sample_connectionstring('MSDASQL;SQLite3');
 s:=db_sample_connectionstring('MSDASQL;Firebird');
 }
 function db_sample_connectionstring(arg:String):String;
 var i,j,n,wc,txt:Integer; sl,cs,dl:String;
  procedure Cleanup;
  begin
   sl:=''; cs:=''; dl:='';
  end;
 begin
  Cleanup;
  dl:=WordDelims('');
  sNul(WordDelims(dl+db_sym_punctchars));
  txt:=_DbLibrary_Internals_.connectionstring_samples;
  for i:=0 to text_numln(txt)-1 do if (cs='') then begin
   sl:=Trim(text_getln(txt,i)); wc:=WordCount(arg); n:=0;
   for j:=1 to wc do if (WordIndex(ExtractWord(j,arg),sl)>0) then n:=n+1;
   if (wc>0) and (n=wc) then cs:=sl;
  end;
  db_sample_connectionstring:=cs;
  sNul(WordDelims(dl));
  Cleanup;
 end;
 {
 Query DbApi Master Key for internal encryption.
 Key padded to be 32 byte length (GOST required).
 }
 function db_query_dbapimasterkey:String;
 const KeyLen=32;
 begin
  if (_DbLibrary_Internals_.DbApiMasterKey='')
  then _DbLibrary_Internals_.DbApiMasterKey:=Trim(ReadIniStr(4+8+16,ParamStr(db_sym_SysIniFile),'[DbApi.Defaults]',db_sym_DbApiMasterKey));
  if (_DbLibrary_Internals_.DbApiMasterKey='') then _DbLibrary_Internals_.DbApiMasterKey:=db_sym_DbApiMasterKey; // Fallback value
  if (Length(_DbLibrary_Internals_.DbApiMasterKey)<KeyLen) then _DbLibrary_Internals_.DbApiMasterKey:=RightPad(_DbLibrary_Internals_.DbApiMasterKey,KeyLen,' ');
  if (Length(_DbLibrary_Internals_.DbApiMasterKey)>KeyLen) then _DbLibrary_Internals_.DbApiMasterKey:=Copy(_DbLibrary_Internals_.DbApiMasterKey,1,KeyLen);
  db_query_dbapimasterkey:=(_DbLibrary_Internals_.DbApiMasterKey);
 end;
 {
 For internal use only.
 }
 procedure db_decode_pwd(var s:String; m:Integer);
 var key:String;
 begin
  key:='';
  m:=iAnd(m,15);
  if (s<>'') then
  case m of
   1,2,3,4,5,6,7,8: begin
    key:=db_query_dbapimasterkey;
    Cryptographer(s,key,false,m);
   end;
  end;
  key:='';
 end;
 {
 Build ConnectionString from library of samples.
 Agruments:
  1. keys   - List of keywords for sample search like 'MSDASQL;Firebird'
  2. server - Hostname or IP address of Database server (if it is required)
  3. dbname - Database name - may be as (host:filename, filename, database alias, data source name)
  4. uid    - User account name
  5. pwd    - User password (may be passed as encrypted string, see mode description)
  6. opt    - Extra options, as list or Name=Value;... like 'ctype=win1251;dialect=3;pagesize=8192;'
  7. mode   - build mode as bit mask
              bit[0..3]    - reserved for encryption method:
              mask = 15      0 - password is not encrypted, on 1..8 see Cryptographer(...) help
                             1=Blowfish,2=Gost,3=RC2,4=RC4,5=RC5,6=RC6,7=BASE64,8=HEX,9..15-none
              16 = bit[4]  - don`t compose server:dbname in database name

 Example:
  cs:=db_build_connectionstring('MSDASQL;Firebird','localhost','employee','SYSDBA','masterkey','',0);
 }
 function db_build_connectionstring(keys,server,dbname,uid,pwd,opt:String; mode:Integer):String;
 var cs,tmp:String; useff:Boolean;
  procedure Cleanup;
  begin
   cs:=''; tmp:='';
  end;
  procedure ValidateTailSemicolon(var s:String);
  begin
   s:=Trim(s); if (s<>'') then if (StrFetch(s,Length(s))<>';') then s:=s+';';
  end;
  // ?string has server:dbname format?
  function HasFullDBNameFormat(s:String):Boolean;
  var has:Boolean; p1,p2:Integer;
  begin
   has:=false; p1:=Pos(':',s); p2:=Pos(':\',s);
   if (p1>1) and (p1<length(s)) and (p1<>p2) then has:=true;
   HasFullDBNameFormat:=has;
  end;
  procedure ValidateDbname;
  var p,t:Integer;
  begin
   p:=Pos(':',dbname);
   if (p>2) then begin
    tmp:=Copy(dbname,1,p-1);
    dbname:=Copy(dbname,p+1);
    t:=Pos('\',dbname)+Pos('/',dbname)+Pos(':',dbname);
    if (t>0) then dbname:=AdaptFileName(dbname);
    dbname:=tmp+':'+dbname;
   end else begin
    t:=Pos('\',dbname)+Pos('/',dbname)+Pos(':',dbname);
    if (t>0) then dbname:=AdaptFileName(dbname);
   end;
  end;
 begin
  Cleanup;
  keys:=Trim(keys); // Remove excess spaces
  server:=Trim(server); dbname:=Trim(dbname);
  uid:=Trim(uid); pwd:=Trim(pwd); opt:=Trim(opt);
  if (WordCount(keys)>0) then begin
   cs:=db_sample_connectionstring(keys);
   if (cs<>'') then begin
    useff:=not HasFlags(mode,16); // ?use server:dbname format of dbname?
    if (db_engine_uses=db_engine_sqldb) then useff:=false;
    if (db_engine_uses=db_engine_zeos) then useff:=false;
    if useff then useff:=(server<>'') and not HasFullDBNameFormat(dbname);
    if useff then tmp:=db_query_connectionstring(cs,db_idns_dbname);
    if useff then useff:=HasFullDBNameFormat(tmp);
    if useff then dbname:=server+':'+dbname;
    ValidateDbname;
    ValidateTailSemicolon(cs);
    if (pwd<>'') then db_decode_pwd(pwd,mode);
    if (uid<>'') then cs:=db_subst_connectionstring(cs,db_idns_uid,uid);
    if (pwd<>'') then cs:=db_subst_connectionstring(cs,db_idns_pwd,pwd);
    if (server<>'') then cs:=db_subst_connectionstring(cs,db_idns_server,server);
    if (dbname<>'') then cs:=db_subst_connectionstring(cs,db_idns_dbname,dbname);
    if (cs<>'') and (Pos('=',opt)>0) then begin
     ValidateTailSemicolon(cs);
     cs:=cs+opt;
    end;
    ValidateTailSemicolon(cs);
   end;
  end;
  db_build_connectionstring:=cs;
  Cleanup;
 end;
 {
 Validate connectionstring Database name by using AdaptFileName.
 }
 function db_validate_adapt_dbname(constr:String):String;
 var cs,tmp,dbname,dbn:String; p:Integer;
  procedure Cleanup;
  begin
   cs:=''; tmp:=''; dbname:=''; dbn:='';
  end;
  procedure ValidateDbname;
  var p,t:Integer;
  begin
   p:=Pos(':',dbname);
   if (p>2) then begin
    tmp:=Copy(dbname,1,p-1);
    dbname:=Copy(dbname,p+1);
    t:=Pos('\',dbname)+Pos('/',dbname)+Pos(':',dbname);
    if (t>0) then dbname:=AdaptFileName(dbname);
    dbname:=tmp+':'+dbname;
   end else begin
    t:=Pos('\',dbname)+Pos('/',dbname)+Pos(':',dbname);
    if (t>0) then dbname:=AdaptFileName(dbname);
   end;
  end;
  procedure ValidateDatabaseName;
  var i:Integer;
  begin
   for i:=1 to WordCount(db_idns_dbname) do begin
    dbn:=ExtractWord(i,db_idns_dbname);
    dbname:=Trim(db_query_connectionstring(cs,dbn));
    if (dbn<>'') then begin
     ValidateDbname;
     cs:=db_subst_connectionstring(cs,dbn,dbname);
    end;
   end;
  end;
 begin
  Cleanup;
  cs:=constr;
  ValidateDatabaseName;
  db_validate_adapt_dbname:=cs;
  Cleanup;
 end;
 {
 Make some replacements for known providers.
 }
 function db_validate_known_providers(constr:String):String;
 var cs,tmp,prv,drv:String; p:Integer;
  procedure Cleanup;
  begin
   cs:=''; tmp:=''; prv:=''; drv:='';
  end;
 begin
  Cleanup;
  cs:=constr;
  if (db_engine_uses=db_engine_sqldb) then begin
   drv:=db_query_connectionstring(cs,db_sym_Driver);
   prv:=db_query_connectionstring(cs,db_sym_Provider);
   // Provider IB for Firebird or Interbase
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_Firebird),UpCaseStr(drv))
     +Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_Interbase),UpCaseStr(drv))
     +Pos(UpCaseStr(db_sym_IBProvider),UpCaseStr(prv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_IB);
   end;
   // Provider PQ for PostgreSQL
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_PostgreSQL),UpCaseStr(drv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_PostgreSQL);
   end;
   // Provider SQLite3 for SQLite3
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_SQLite3),UpCaseStr(drv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_SQLite3);
   end;
  end;
  if (db_engine_uses=db_engine_zeos) then begin
   drv:=db_query_connectionstring(cs,db_sym_Driver);
   prv:=db_query_connectionstring(cs,db_sym_Provider);
   // Provider Firebird for Firebird or Interbase
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_Firebird),UpCaseStr(drv))
     +Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_Interbase),UpCaseStr(drv))
     +Pos(UpCaseStr(db_sym_IBProvider),UpCaseStr(prv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_Firebird);
   end;
   // Provider PQ for PostgreSQL
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_PostgreSQL),UpCaseStr(drv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_PostgreSQL);
   end;
   // Provider SQLite3 for SQLite3
   p:=Pos(UpCaseStr(db_sym_MSDASQL),UpCaseStr(prv))
     *Pos(UpCaseStr(db_sym_SQLite3),UpCaseStr(drv));
   if (p>0) then begin
    cs:=db_subst_connectionstring(cs,db_sym_Driver,'');
    cs:=db_subst_connectionstring(cs,db_sym_Provider,db_sym_SQLite);
   end;
  end;
  cs:=db_validate_adapt_dbname(cs);
  db_validate_known_providers:=cs;
  Cleanup;
 end;
 {
 Build SQL query to select all tables of Database.
 Use samples located in SysIniFile [DbApi.SQL.Query.SelectAllTables].
 Database type (dbtype) is one of: (SQLite3,Firebird,MySQL,PostgreSQL).
 }
 function db_build_selectalltables(dbtype,database:String):String;
 var sql:String;
  procedure Cleanup;
  begin
   sql:='';
  end;
 begin
  Cleanup;
  dbtype:=Trim(dbtype);
  database:=Trim(database);
  if not IsEmptyStr(dbtype) then begin
   if GetStringVar(_DbLibrary_Internals_.selectalltables_samples,dbtype,sql) then sql:=Trim(sql) else sql:='';
   if (sql<>'') and (database<>'') then sql:=StringReplace(sql,'${Database}',database,rfReplaceAll+rfIgnoreCase);
  end;
  db_build_selectalltables:=sql;
  Cleanup;
 end;
 {
 Return DbApi supported DB types of engine.
 }
 function db_supporteddbtypesof(eid:Integer):String;
 var s,fb:String;
 begin
  s:=''; fb:='';
  case eid of
   db_engine_ado:   fb:='SQLite3,Firebird,MySQL,PostgreSQL';
   db_engine_sqldb: fb:='MSSQL,Sybase,PQ,Oracle,ODBC,MySQL40,MySQL41,MySQL50,MySQL51,MySQL55,MySQL56,MySQL57,MySQL80,SQLite3,IB';
   db_engine_zeos:  fb:='ado,ASA,asa_capi,firebird,interbase,mariadb,mssql,mysql,odbc_a,odbc_w,OleDb,oracle,pooled.*,postgresql,sqlite,sybase,WebServiceProxy';
  end;
  s:=Trim(ExtractWord(eid,db_engine_list));
  s:=TrimDef(ReadIniStr(4+8+16,ParamStr(db_sym_SysIniFile),'[DbApi.Defaults.'+s+']','SupportedDbTypes'),fb);
  db_supporteddbtypesof:=s;
  s:=''; fb:='';
 end;
 {
 Return list of DbApi supported DB types.
 }
 function db_supporteddbtypes:String;
 begin
  if (_DbLibrary_Internals_.SupportedDbTypes='')
  then _DbLibrary_Internals_.SupportedDbTypes:=db_supporteddbtypesof(db_engine_uses);
  db_supporteddbtypes:=_DbLibrary_Internals_.SupportedDbTypes;
 end;
 {
 Return list of DbApi supported DB types for all engines.
 }
 function db_allsupporteddbtypes:String;
 var s:String; i:Integer;
 begin
  s:='';
  for i:=1 to WordCount(db_engine_list) do s:=s+','+db_supporteddbtypesof(i);
  db_allsupporteddbtypes:=Copy(s,2);
  s:='';
 end;
 {
 Set bug report mode=(0,1,2,3)=(Hide,Echo,Soft,Hard) for connection (con).
 }
 procedure db_bugreport_mode(con,mode:Integer);
 begin
  if (con<>0) then
  case mode of
   db_brm_hide: sNul(db_ctrl(con,'BugReportPrefix=@!EHideException: '));
   db_brm_echo: sNul(db_ctrl(con,'BugReportPrefix=@!EEchoException: '));
   db_brm_soft: sNul(db_ctrl(con,'BugReportPrefix=@!ESoftException: '));
   db_brm_hard: sNul(db_ctrl(con,'BugReportPrefix='));
  end;
 end;
 {
 Just do SQL query with connection (con) and SQL command (sql).
 }
 function db_just_sql_query(con:Integer; sql:String):Boolean;
 var rst:Integer; done:Boolean;
 begin
  done:=false; sql:=Trim(sql);
  if (con<>0) and (sql<>'') then
  if db_active(con) then
  if (db_begintrans(con)>0) then begin
   rst:=db_execute(con,sql,adCmdText);
   if (rst<>0)
   then done:=db_committrans(con)
   else bNul(db_rollbacktrans(con));
   FreeAndZero(rst);
  end;
  db_just_sql_query:=done;
 end;
 {
 Do easy SQL query with connection string (cs) and SQL command (sql).
 }
 function db_easy_sql_query(cs,sql:String):Boolean;
 var con:Integer; done:Boolean;
 begin
  done:=false;
  cs:=Trim(cs); sql:=Trim(sql);
  if (cs<>'') and (sql<>'') then begin
   con:=db_connection(db_engine_uses,cs);
   if (con<>0) then begin
    db_bugreport_mode(con,db_brm_uses);
    if db_open(con,adConnectUnspecified)
    then done:=db_just_sql_query(con,sql);
   end;
   FreeAndZero(con);
  end;
  db_easy_sql_query:=done;
 end;
 {
 Get default user id and password for DB of type (dbtype).
 }
 function db_default_uid(dbtype:String):String;
 var uid:String;
  procedure Cleanup;
  begin
   uid:='';
  end;
 begin
  Cleanup;
  if IsSameText(dbtype,db_sym_IB) then uid:=db_sym_SYSDBA;
  if IsSameText(dbtype,db_sym_Firebird) then uid:=db_sym_SYSDBA;
  db_default_uid:=uid;
  Cleanup;
 end;
 function db_default_pwd(dbtype:String):String;
 var pwd:String;
  procedure Cleanup;
  begin
   pwd:='';
  end;
 begin
  Cleanup;
  if IsSameText(dbtype,db_sym_IB) then pwd:=db_sym_masterkey;
  if IsSameText(dbtype,db_sym_Firebird) then pwd:=db_sym_masterkey;
  db_default_pwd:=pwd;
  Cleanup;
 end;
 {
 Get sample database file name from library of samples.
 Examples:
 s:=db_sample_database_file('SQLite3');
 s:=db_sample_database_file('Firebird');
 }
 function db_sample_database_file(arg:String):String;
 var txt:Integer; s:String;
  procedure Cleanup;
  begin
   s:='';
  end;
 begin
  Cleanup;
  arg:=Trim(arg);
  if (arg<>'') then begin
   txt:=_DbLibrary_Internals_.database_file_samples;
   if not GetStringVar(txt,arg,s) then s:='';
   if not FileExists(s) then s:='';
  end;
  db_sample_database_file:=s;
  Cleanup;
 end;
 {
 Detect database type by connection (con) or connection string (constr).
 Supported database types enumerated in db_supporteddbtypes list.
 }
 function db_detect_dbtype(con:Integer; constr:String):String;
 var st,sw,sc,sl,dl:String; iw:Integer;
  procedure Cleanup;
  begin
   st:=''; sw:=''; sc:=''; sl:=''; dl:='';
  end;
 begin
  Cleanup;
  if (con=0)
  then constr:=Trim(constr)
  else constr:=Trim(db_ctrl(con,db_sym_ConnectionStringInit)+EOL+constr);
  if not IsEmptyStr(constr) then begin
   dl:=WordDelims('');
   sNul(WordDelims(dl+db_sym_punctchars));
   sl:=db_supporteddbtypes+','+db_sym_IBProvider;
   sc:=db_query_connectionstring(constr,db_sym_Provider)+','+db_query_connectionstring(constr,db_sym_Driver);
   for iw:=1 to WordCount(sl) do if (st='') then begin
    sw:=ExtractWord(iw,sl); if (WordIndex(sw,sc)>0) then st:=sw;
    if IsSameText(st,db_sym_IBProvider) then st:=db_sym_Firebird;
   end;
   sNul(WordDelims(dl));
  end;
  db_detect_dbtype:=st;
  Cleanup;
 end;
 {
 Make SQL query command to show (select all) tables for connection (con) and database type (dbtype).
 Connection (con) should be initialized for wanted database and opened as well.
 Example: ListOfTables:=db_query_selectalltables(con,db_sym_Firebird);
 }
 function db_query_selectalltables(con:Integer; dbtype:String):String;
 var sqlcmd,id,sv:String; tp,rst,lines:Integer;
  procedure Cleanup(how:Integer);
  begin
   sqlcmd:=''; id:=''; sv:='';
   if (how<>0) then lines:=text_new else FreeAndZero(lines);
  end;
 begin
  Cleanup(1);
  if (con<>0) then
  if (db_state(con)=adStateOpen) then begin
   if IsEmptyStr(dbtype) then dbtype:=db_detect_dbtype(con,'');
   sqlcmd:=db_build_selectalltables(dbtype,'');
   if not IsEmptyStr(sqlcmd) then begin
    if (db_begintrans(con)>0) then begin
     rst:=db_execute(con,sqlcmd,adCmdText);
     if (rst<>0) then begin
      if (db_fieldscount(rst)=1) then begin
       while not db_eof(rst) do begin
        id:=db_fieldsNames(rst,0);
        tp:=db_fieldsTypes(rst,id);
        if (db_AdoTypeToTagType(tp)=tag_type_string)
        then sv:=Trim(db_fieldsAsString(rst,id,'r',''))
        else sv:='';
        if (sv<>'') then bNul(text_addln(lines,sv));
        bNul(db_movenext(rst));
       end;
      end else Problem('Bad data on query: '+sqlcmd);
     end else Problem('Failed db_execute: '+sqlcmd);
     if not db_committrans(con) then Problem('Failed Commit Transaction.');
    end else Problem('Could not start transaction.');
   end else Problem('Could not build SelectAllTables command.');
  end else Problem('Could not read closed or busy Database.');
  db_query_selectalltables:=Text_ToString(lines);
  Cleanup(0);
 end;
 {
 Query list of tables for database by connection string (constr) and database type (dbtype).
 }
 function db_query_showtables(constr,dbtype:String):String;
 var s:String; con:Integer;
  procedure Cleanup;
  begin
   s:='';
  end;
 begin
  Cleanup;
  constr:=Trim(constr); dbtype:=Trim(dbtype);
  if (dbtype='') then dbtype:=db_detect_dbtype(0,constr);
  if (constr<>'') and (dbtype<>'') then begin
   con:=db_connection(db_engine_uses,constr);
   if (con<>0) then begin
    if db_open(con,adConnectUnspecified) then begin
     s:=db_query_selectalltables(con,dbtype);
    end else Problem('Failed db_open: '+constr);
   end else Problem('Failed db_connection: '+constr);
   FreeAndZero(con);
  end else Problem('Empty args of showtables.');
  db_query_showtables:=s;
  Cleanup;
 end;
 {
 Read ConnectionString samples from system INI file to internal buffer.
 }
 procedure db_read_connectionstring_samples;
 var i,txt:Integer; sl:String;
  procedure Cleanup;
  begin
   sl:='';
  end;
 begin
  Cleanup;
  txt:=ReadIniSection(text_new,0,ParamStr(db_sym_SysIniFile),'[DbApi.ConnectionString.Samples]');
  FreeAndZero(_DbLibrary_Internals_.connectionstring_samples);
  _DbLibrary_Internals_.connectionstring_samples:=txt;
  for i:=text_numln(txt)-1 downto 0 do begin
   sl:=Trim(text_getln(txt,i)); // Read lines as ConnectionString = ...
   if (StrFetch(sl,1)=';') then sl:=''; if (Pos('=',sl)=0) then sl:='';
   if not IsSameText(ExtractWord(1,sl),'ConnectionString') then sl:='';
   if (WordIndex('Provider',sl)=0) then sl:=''; // Must have Provider?
   sl:=SkipWords(1,sl); // Skip ConnectionString = ...
   if (sl='')
   then bNul(text_delln(txt,i))
   else bNul(text_putln(txt,i,sl));
  end;
  // Add fallback ConnectionStrings if nothing found
  bNul(text_addln(txt,'Provider=MSDASQL;DRIVER=SQLite3 ODBC Driver;Database=c:\mydb.db;'));
  bNul(text_addln(txt,'Provider=LCPI.IBProvider.5.Free;Location=localhost:employee;User ID=SYSDBA;Password=masterkey;'));
  bNul(text_addln(txt,'Provider=MSDASQL;DRIVER=Firebird/InterBase(r) driver;DBNAME=localhost:employee;UID=SYSDBA;PWD=masterkey;'));
  bNul(text_addln(txt,'Provider=MSDASQL;Driver=PostgreSQL ANSI;Server=localhost;Database=myDataBase;Uid=myUsername;Pwd=myPassword;'));
  bNul(text_addln(txt,'Provider=MSDASQL;Driver=PostgreSQL Unicode;Server=localhost;Database=myDataBase;Uid=myUsername;Pwd=myPassword;'));
  bNul(text_addln(txt,'Provider=MSDASQL;Driver=MySQL ODBC 8.0 ANSI Driver;Server=localhost;Database=myDataBase;User=myUsername;Password=myPassword;Option=3;'));
  bNul(text_addln(txt,'Provider=MSDASQL;Driver=MySQL ODBC 8.0 Unicode Driver;Server=localhost;Database=myDataBase;User=myUsername;Password=myPassword;Option=3;'));
  Cleanup;
 end;
 {
 Read samples of "select all tables" SQL query from SysIniFile [DbApi.SQL.Query.SelectAllTables].
 }
 procedure db_read_selectalltables_samples;
 var i,txt:Integer; sl:String;
  procedure Cleanup;
  begin
   sl:='';
  end;
 begin
  Cleanup;
  txt:=ReadIniSection(text_new,0,ParamStr(db_sym_SysIniFile),'[DbApi.SQL.Query.SelectAllTables]');
  FreeAndZero(_DbLibrary_Internals_.selectalltables_samples);
  _DbLibrary_Internals_.selectalltables_samples:=txt;
  // Add fallback values if ini-file failed
  bNul(text_addln(txt,'MySQL = show tables;'));
  bNul(text_addln(txt,'SQLite3 = select name from sqlite_master;'));
  bNul(text_addln(txt,'Firebird = select RDB$RELATION_NAME from RDB$RELATIONS R where R.RDB$SYSTEM_FLAG = 0;'));
  bNul(text_addln(txt,'PostgreSQL = select table_name from information_schema.tables where table_schema=''public'';'));
  for i:=text_numln(txt)-1 downto 0 do begin
   sl:=Trim(text_getln(txt,i));
   if (Pos('=',sl)=0) then sl:=''; if (StrFetch(sl,1)=';') then sl:='';
   if (WordIndex(ExtractWord(1,sl),db_allsupporteddbtypes)=0) then sl:='';
   if (Pos(';',sl)>0) then sl:=Copy(sl,1,Pos(';',sl)-1); // Drop tail ;
   // GetStringVar require Name=Value without spaces around '=' sign.
   if (Pos('=',sl)>0) then sl:=ExtractWord(1,sl)+'='+SkipWords(1,sl);
   if (sl='')
   then bNul(text_delln(txt,i))
   else bNul(text_putln(txt,i,sl));
  end;
  Cleanup;
 end;
 {
 Read database file samples from SysIniFile [DbApi.Database.File.Samples].
 }
 procedure db_read_database_file_samples;
 var i,txt:Integer; sl:String;
  procedure Cleanup;
  begin
   sl:='';
  end;
 begin
  Cleanup;
  txt:=ReadIniSection(text_new,4+8+16,ParamStr(db_sym_SysIniFile),'[DbApi.Database.File.Samples]');
  FreeAndZero(_DbLibrary_Internals_.database_file_samples);
  _DbLibrary_Internals_.database_file_samples:=txt;
  // Add fallback values if ini-file failed
  if not GetStringVar(txt,db_sym_SQLite3,sl)  then bNul(text_addln(txt,'SQLite3 = %CRW_DAQ_SYS_HOME_DIR%\Resource\DbSamples\sample.db'));
  if not GetStringVar(txt,db_sym_Firebird,sl) then bNul(text_addln(txt,'Firebird = %CRW_DAQ_SYS_HOME_DIR%\Resource\DbSamples\sample.fdb'));
  if not GetStringVar(txt,db_sym_IB,sl) then bNul(text_addln(txt,'IB = %CRW_DAQ_SYS_HOME_DIR%\Resource\DbSamples\sample.fdb'));
  for i:=text_numln(txt)-1 downto 0 do begin
   sl:=Trim(text_getln(txt,i));
   if (Pos('=',sl)=0) then sl:=''; if (StrFetch(sl,1)=';') then sl:='';
   if (WordIndex(ExtractWord(1,sl),db_allsupporteddbtypes)=0) then sl:='';
   if (Pos(';',sl)>0) then sl:=Copy(sl,1,Pos(';',sl)-1); // Drop tail ;
   sl:=ExpEnv(sl); // Expand env.variables like %CRW_DAQ_SYS_HOME_DIR%
   // GetStringVar require Name=Value without spaces around '=' sign.
   if (Pos('=',sl)>0) then sl:=ExtractWord(1,sl)+'='+AdaptFileName(Trim(SkipWords(1,sl)));
   if (sl='')
   then bNul(text_delln(txt,i))
   else bNul(text_putln(txt,i,sl));
  end;
  Cleanup;
 end;
 {
 Check Firebird process is running.
 }
 function db_is_firebird_exe_running:Boolean;
 begin
  db_is_firebird_exe_running:=(ParamStr('ListOf pids where name='+AdaptExeFileName('firebird.exe'))<>'');
 end;
 {
 Create local Firebird database (fdb) with user (uid) password (pwd) options (opt).
 Use constr for master connection (use empty string for defaults).
 Example:
 flag:=db_create_local_fdb('c:\test.fdb','','','PageSize = 8192','',0);
 }
 function db_create_local_fdb(fdb,uid,pwd,opt,constr:String; mode:Integer):Boolean;
 var sample,sqlcmd,dbhost,dbname,dbfile,server,cmd,ans:String;
     done,fbRunning,rcm:Boolean; tmout:Integer;
  procedure Cleanup;
  begin
   sample:=''; sqlcmd:=''; done:=false;
   dbhost:=''; dbname:=''; dbfile:=''; server:='';
   cmd:=''; ans:=''; fbRunning:=False; rcm:=False;
   tmout:=0;
  end;
  function SysSample:String;
  begin
   SysSample:=GetEnv('SystemDrive')+Copy(sample,3,MaxInt);
  end;
  procedure EnsureLocalhostServer;
  begin
   server:=db_query_connectionstring(constr,db_sym_Server);
   if not IsEmptyStr(server) then begin
    server:=IfThenStr(fbRunning,db_sym_localhost,'');
    constr:=db_subst_connectionstring(constr,db_sym_Server,server);
   end;
   dbname:=db_query_connectionstring(constr,db_sym_DBName);
   if not IsEmptyStr(dbname) then begin
    if (ExtractNameValuePair(dbname,dbhost,dbfile,'=',3)>2) then begin
     dbhost:=IfThenStr(fbRunning,db_sym_localhost,'');
    end else begin dbfile:=dbname; dbhost:=''; end;
    if fbRunning and IsEmptyStr(server) then dbhost:=db_sym_localhost;
    dbname:=IfThenStr(dbhost='',dbfile,dbhost+':'+dbfile);
    constr:=db_subst_connectionstring(constr,db_sym_DBName,dbname);
   end;
  end;
 begin
  Cleanup;
  fdb:=Trim(fdb); tmout:=5000;
  uid:=Trim(uid); pwd:=Trim(pwd);
  constr:=Trim(constr); opt:=Trim(opt);
  if (fdb<>'') then fdb:=DefaultExtension(fdb,'.fdb');
  fbRunning:=db_is_firebird_exe_running;
  Success(IfThenStr(fbRunning,'Firebird is RUNNING','Firebird is STOPPED'));
  if (fdb<>'') and not FileExists(fdb) then begin
   if (constr='') then begin
    sample:=DefaultPath(db_sample_database_file(db_sym_Firebird),ParamStr('HOMEDIR'));
    if IsEmptyStr(sample) or not FileExists(sample) then sample:='employee.fdb';
    if (Pos(':',sample)=2) and FileExists(SysSample) then sample:=SysSample;
    constr:=db_build_connectionstring(db_sym_Firebird,db_sym_localhost,sample,db_sym_SYSDBA,db_sym_masterkey,'',0);
    EnsureLocalhostServer;
    if (db_engine_uses=db_engine_sqldb) then begin
     constr:=db_subst_connectionstring(constr,db_sym_Driver,'');
     constr:=db_subst_connectionstring(constr,db_sym_Provider,db_sym_IB);
     if not fbRunning then constr:=StringReplace(constr,db_sym_localhost+':','',rfIgnoreCase);
    end;
    if (db_engine_uses=db_engine_zeos) then begin
     constr:=db_subst_connectionstring(constr,db_sym_Driver,'');
     constr:=db_subst_connectionstring(constr,db_sym_Provider,db_sym_Firebird);
     if not fbRunning then constr:=StringReplace(constr,db_sym_localhost+':','',rfIgnoreCase);
    end;
   end;
   db_decode_pwd(pwd,mode);
   if (uid='') then uid:=db_default_uid(db_sym_Firebird);
   if (pwd='') then pwd:=db_default_pwd(db_sym_Firebird);
   sqlcmd:='create database '+AnsiQuotedStr(fdb,Apostrophe);
   sqlcmd:=sqlcmd+' user '+AnsiQuotedStr(uid,Apostrophe);
   sqlcmd:=sqlcmd+' password '+AnsiQuotedStr(pwd,Apostrophe);
   if not IsEmptyStr(opt) then sqlcmd:=sqlcmd+' '+opt;
   if (db_engine_uses=db_engine_ado) then begin
    done:=db_easy_sql_query(constr,sqlcmd);
   end else
   if (db_engine_uses=db_engine_sqldb) then begin
    dbname:=IfThenStr(fbRunning,db_sym_localhost+':','')+fdb;
    //db_engine_sqldb: done:=db_easy_sql_query(constr,sqlcmd);
    constr:=db_subst_connectionstring(constr,db_sym_Driver,'');
    constr:=db_subst_connectionstring(constr,db_sym_DBName,dbname);
    writeln('CREATEDB ',constr,' ',sqlcmd);
    done:=db_create('Engine=SQLDB;'+constr);
   end;
   if (db_engine_uses=db_engine_zeos) then begin
    dbname:=IfThenStr(fbRunning,db_sym_localhost+':','')+fdb;
    //db_engine_zeos: done:=db_easy_sql_query(constr,sqlcmd);
    constr:=db_subst_connectionstring(constr,db_sym_Driver,'');
    constr:=db_subst_connectionstring(constr,db_sym_DBName,dbname);
    writeln('CREATEDB ',constr,' ',sqlcmd);
    done:=db_create('Engine=ZEOS;'+constr);
   end;
  end;
  // When Unix client mode: make file read/writable for all
  if IsUnix and FileExists(fdb) and not fbRunning then begin
   cmd:='chmod -c ugo+rw '+fdb;
   rcm:=(RunSysCommand(cmd,'','',ans,tmout)=0);
   if not rcm then rcm:=(RunSysCommand('sudo -n '+cmd,'','',ans,tmout)=0);
   if rcm then Success('Luck chmod '+fdb) else Problem('Fail chmod '+fdb);
   if not IsEmptyStr(ans) then Success(ans);
  end;
  if FileExists(fdb) then done:=true;
  db_create_local_fdb:=done;
  Cleanup;
 end;
 {
 Clear DbLibrary.
 }
 procedure ClearDbLibrary;
 begin
  _DbLibrary_Internals_.connectionstring_samples:=0;
  _DbLibrary_Internals_.selectalltables_samples:=0;
  _DbLibrary_Internals_.database_file_samples:=0;
  _DbLibrary_Internals_.AllSupportedDbTypes:='';
  _DbLibrary_Internals_.SupportedDbTypes:='';
  _DbLibrary_Internals_.DbApiMasterKey:='';
  _DbLibrary_Internals_.engine_list:='';
 end;
 {
 Initialize DbLibrary.
 }
 procedure InitDbLibrary;
 begin
  ShouldPollDbLibrary:=false;
  // Uses "message pump" for COM objects
  // with COM STA (single thread apartment)
  sNul(ParamStr('Polling.UseMsgPump 1'));
  _DbLibrary_Internals_.engine_uses:=iValDef(ReadIni('[DbApi.Defaults] EngineId'),db_engine_ado);
  _DbLibrary_Internals_.engine_list:=TrimDef(ReadIni('[DbApi.Defaults] EngineList'),'ADO,SQLDB,ZEOS');
  db_read_connectionstring_samples;
  db_read_selectalltables_samples;
  db_read_database_file_samples;
  db_brm_uses:=db_brm_echo;
 end;
 {
 Finalize DbLibrary.
 }
 procedure FreeDbLibrary;
 begin
  FreeAndZero(_DbLibrary_Internals_.connectionstring_samples);
  FreeAndZero(_DbLibrary_Internals_.selectalltables_samples);
  FreeAndZero(_DbLibrary_Internals_.database_file_samples);
 end;
 {
 Poll DbLibrary.
 }
 procedure PollDbLibrary;
 begin
 end;
