#!/bin/bash

###########################################################
## Copyright (c) 2002-2025 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Watcher for crwdaq: report result after crwdaq exit,  ##
## make analysis of leakage.log file etc.                ##
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

readonly crwpid="$CRW_DAQ_SYS_EXE_PID";
readonly logdir="$CRW_DAQ_VAR_TMP_DIR";
readonly leaklog="$logdir/leakage.log";
readonly log="$logdir/$scriptbase.log";
readonly arguments="$*";
readonly maxline=100;

declare -i numleaks=0;
declare -i numsuccs=0;
declare -i numlines=0;
declare -i numleave=0;
declare -i numenter=0;

#########################################
# check command $1 with PID $2 is running
#########################################
function cmd_pid_running(){
 if [ $# -ne 2 ]; then return 1; fi;
 if [ -n "$(pgrep -x $1 | grep "\b$2\b")" ]; then return 0; fi;
 return 1;
};

########################
# analyse standard input
# to detect exit status
# and to find leaks
########################
function leakage_analysis(){
 local line=""; local value="";
 while [ $numlines -lt $maxline ]; do
  read line; let "numlines+=1";
  if [ -z "$line" ]; then break; fi; 1>&2 echo "$line";
  if [ "${line:0:6}" = "Enter " ]; then let "numenter+=1"; break; fi;
  if [ "${line:0:6}" = "Leave " ]; then let "numleave+=1"; break; fi;
  if [ "${line:0:6}" = "value=" ]; then let "$line"; else continue; fi;
  if [ "$value" = "0" ]; then let "numsuccs+=1"; continue; fi;
  if [ -n "$value" ]; then let "numleaks+=1"; fi;
 done;
 echo "let numlines=$numlines;";
 echo "let numenter=$numenter;";
 echo "let numleave=$numleave;";
 echo "let numleaks=$numleaks;";
 echo "let numsuccs=$numsuccs;";
};

##################################
# prepare leakage.log for analysys
##################################
function tac_leakage_log(){
 tail -n $maxline "$1" | tac \
 | grep -ive 'ObjectRegistry.Space' \
 | grep -ive 'ObjectRegistry.Capacity' \
 | grep -ive 'Balance of GetMem/FreeMem' \
 | grep -ive 'TTask.DetachedPidList.Count' \
 | sed -re 's/\s*(\w[^=]*\w)\s*=\s*(\S*)\s*/value=\2/';
};

######################
# Write note to syslog
######################
function syslog_note(){
 $scriptHOME/crwdaq.syslog.sh "$@";
};

###########################################
# Print session report message with header.
# $1 must be as SEVERITY: SENDER - MESSAGE.
###########################################
function syslog_report(){
 local arg="$1"; # Argument must have format like: SEVERITY: SENDER - MESSAGE
 local severity="$(echo "$arg" | cut -d: -f1 | xargs)"; if [[ -z "$severity" ]]; then return; fi;
 local sender="$(echo "$arg" | cut -d' ' -f2 | xargs)"; if [[ -z "$sender" ]];   then return; fi;
 local msg="$(echo "$arg" | sed 's/^.* - //' | xargs)"; if [[ -z "$msg" ]];      then return; fi;
 syslog_note                                  \
      "$severity: $sender - ________________" \
      "$severity: $sender - Session Outcome." \
      "$severity: $sender - ^^^^^^^^^^^^^^^^" \
      "$severity: $sender - $msg.";
};

#####################################
# Analyse leakage.log and make report
#####################################
function report_result(){
 local expr="$(tac_leakage_log "$1" | leakage_analysis | xargs)";
 if [ -n "$expr" ]; then eval "$expr"; fi;
 echo "numlines = $numlines";
 echo "numenter = $numenter";
 echo "numleave = $numleave";
 echo "numleaks = $numleaks";
 echo "numsuccs = $numsuccs";
 local msg=""; local delay=15000;
 local prog="$CRW_DAQ_SYS_TITLE";
 local time="$(date +%Y.%m.%d-%H:%M:%S)";
 local sndr="$(hostname)/$CRW_DAQ_SYS_SESSION_HEAD/System";
 if [ $numleave -gt 0 ]; then
  if [ $numleaks -gt 0 ]; then
   let "delay=60*60*1000";
   msg="$(langstr ru "Процесс $crwpid завершен успешно. Однако найдены утечки памяти" en "Process $crwpid stop succeed. But memory leaks found")";
   unix tooltip-notifier text "$time $prog: $msg ($numleaks)." preset stdWarning delay $delay;
   echo "$time $prog: PROBLEM - $msg ($numleaks)." | tee -a "$log";
   syslog_report "WARN/PROBLEM: $sndr - $msg ($numleaks).";
   return 1;
  fi;
  if [ $numsuccs -gt 0 ]; then
   let "delay=60*1000";
   msg="$(langstr ru "Процесс $crwpid завершен успешно. Проблем не обнаружено" en "Process $crwpid stop succeed. No problems found")";
   unix tooltip-notifier text "$time $prog: $msg." preset stdSuccess delay $delay;
   echo "$time $prog: SUCCESS - $msg." | tee -a "$log";
   syslog_report "INFO/SUCCESS: $sndr - $msg.";
   return 0;
  fi;
 fi;
 if [ $numenter -gt 0 ]; then
  let "delay=24*60*60*1000";
  msg="$(langstr ru "Процесс $crwpid завершен аварийно" en "Process $crwpid abnormal termination")";
  unix tooltip-notifier text "$time $prog: $msg." preset stdError delay $delay;
  echo "$time $prog: FAILURE - $msg." | tee -a "$log";
  syslog_report "FATAL/FAILURE: $sndr - $msg.";
  return 1;
 fi;
};

function add_log(){
 local prog="$CRW_DAQ_SYS_TITLE";
 local time="$(date +%Y.%m.%d-%H:%M:%S)";
 echo "$time $prog: $*" |  tee -a "$log";
};

function check_log(){
 touch "$log";
 add_log "$scriptname $arguments";
};

declare -i logsec=0;

function run_crwdaqlogroll(){
 if is_number "$1"; then
  local -i sec=$1; let sec-=$logsec;
  if [[ $sec -lt 60 ]]; then return; fi;
  $scriptHOME/crwdaqlogroll.sh;
  let logsec=$1;
 fi;
};

######
# Main
######
function main(){
 if [ -z "$crwpid" ]; then fatal 1 "Error: crwdaq environ (PID)."; fi;
 if [ -z "$logdir" ]; then fatal 1 "Error: crwdaq environ (dir)."; fi;
 if [ ! -e "$leaklog" ]; then fatal 1 "Error: not found leakage.log."; fi;
 check_log;
 if cmd_pid_running crwdaq $crwpid; then
  local ms0="$(unix msecnow)";
  local ms="0"; local sec="0";
  while cmd_pid_running crwdaq $crwpid; do
   ms="$(unix msecnow)"; let "sec=(ms-ms0)/1000";
   echo "crwdaq running $sec second(s).";
   sleep 5; run_crwdaqlogroll $sec;
  done;
  echo "crwdaq stopped.";
  report_result "$leaklog";
 else
  fatal 1 "Error: crwdaq PID $crwpid is not running.";
 fi;
};

main "$@";

##############
## END OF FILE
##############
