 //////////////////////////////////////////////////////////////////////////////
 // E140 driver for CRW-DAQ
 // Last release: 20101125 by Kuruakin Alexey <kouriakine@mail.ru>
 //////////////////////////////////////////////////////////////////////////////
LIBRARY E140_DLL;

{$I _sysdef}

uses ShareMem, SysUtils, Windows, Math, Classes, mmsystem, ShellApi, _CrwApi,
     Lusbapi, E140_SHM;

 //////////////////////////////////////////////////////////////////////////////
 // Utilities for E140 driver.
 //////////////////////////////////////////////////////////////////////////////
const
 AdcMaxChans  = 32;    // Max. ADC channels number

type
 TStatSum = packed record // Statistic accumulator
  N       : Integer;      // Number of points
  sx      : Extended;     // Sum(x)
  sy      : Extended;     // Sum(y)
 end;

procedure ZeroStatSum(var Stat:TStatSum);
begin
 with Stat do begin N:=0; sx:=0; sy:=0; end;
end;

procedure AddStatSum(var Stat:TStatSum; const x,y:Double);
begin
 with Stat do begin N:=N+1; sx:=sx+x; sy:=sy+y; end;
end;

function FitStatSum(const Stat:TStatSum; var x,y:Double):Boolean;
begin
 Result:=False;
 with Stat do if N>0 then begin Result:=True; x:=sx/N; y:=sy/N; end;
end;

 //////////////////////////////////////////////////////////////////////////////
 // General E140 driver object.
 // To be placed in DaqDataSheet area.
 //////////////////////////////////////////////////////////////////////////////
type
 PE140 = ^TE140;						     // Pointer to E140 driver
 TE140 = object							// General E140 driver object
 public
  CrwApi        : TCrwApi;					// Points to CrwApi interface
  tag           : record					// All tags in this record
   Start        : Integer;					// i Tag of START button
   CadjP        : Integer;                      // i Clock adjustment period, sec
   CadjC        : Integer;                      // r Clock adjustment coefficient
   GateAo       : Integer;                      // i Tag to enable AO, i.e. DAC
   Readout      : Integer;                      // s Tag of Readout data
   Fir          : array[0..AdcMaxChans-1] of Integer;  // i FIR filter(s) order
  end;
  StatSum       : array[0..AdcMaxChans-1] of TStatSum; // Statistic summs
  shmFile       : THandle;                      // Handle  of Shared Memory
  shmName       : LongString;                   // Name    of Shared Memory
  shmData       : PE140_SHARED_BUFFER;          // Pointer of Shared Memory
  TimeBase      : Double;                       // Time base, ms
  TimeStart     : Double;                       // Time of start
  RateFact      : Double;                       // Rate scale factor
  TimeAdju      : Double;                       // Time adjustment
  TimeBuff      : Double;                       // Time of block
  TimeDiff      : TStatSum;                     // Time difference
  MaxFact       : Double;                       // Maximal RateFact
  MinFact       : Double;                       // Minimal RateFact
  DevRef        : Integer;                      // Device reference
 public
  procedure DAQ_CMD_INIT(TheCrwApi:TCrwApi);
  procedure DAQ_CMD_FREE;
  procedure DAQ_CMD_POLL;
  procedure DAQ_CMD_FAIL;
  procedure InitSharedMemoryBuffer(aName:LongString);
  procedure FreeSharedMemoryBuffer;
  procedure Failure(const msg:LongString);
 end;

 //////////////////////////////////////////////////////////////////////////////
 // E140 driver initialization
 //////////////////////////////////////////////////////////////////////////////
procedure TE140.DAQ_CMD_INIT(TheCrwApi:TCrwApi);
var
 i : Integer;
 //
 // Initialize tag with given name and type
 //
 procedure InitTag(var tag:integer; const name:ShortString; typ:integer);
 begin
  with CrwApi,SysApi,DaqApi do begin
   tag:=findtag(name);
   if (typ>0) and (typetag(tag)<>typ)
   then Failure(Format('Wrong tag "%s".',[name]));
  end;
 end;
 //
 // Get TimeBase
 //
 function GetTimeBase:Double;
 var i:Integer; tb:Double;
 begin
  with CrwApi,SysApi,GuiApi,DaqApi do begin
   Result:=Int(mSecNow-Time*TimeUnits);
   for i:=1 to 1024 do begin
    tb:=Int(mSecNow-Time*TimeUnits);
    if Result = tb then Break;
    Result:=tb;
   end;
  end;
 end;
begin
 FillChar(Self,SizeOf(Self),0);
 CrwApi:=TheCrwApi;
 if Assigned(CrwApi) then 
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  RedirectStdIn(Input);
  RedirectStdOut(Output);
  TimeBase:=GetTimeBase();
  DevRef:=FindObjectRef('Device','');
  InitTag(tag.Start,   ReadIni('tagStart'),	   1);
  InitTag(tag.CadjP,   ReadIni('tagCadjP'),   1);
  InitTag(tag.CadjC,   ReadIni('tagCadjC'),   2);
  InitTag(tag.GateAo,  ReadIni('tagGateAo'),  1);
  InitTag(tag.Readout, ReadIni('tagReadout'), 3);
  for i:=0 to AdcMaxChans-1 do begin
   InitTag(tag.Fir[i], StringReplace(ReadIni('tagFir##'),'##',IntToStr(i),[]), 1);
   ZeroStatSum(StatSum[i]);
  end;
  ZeroStatSum(TimeDiff);
  TimeStart:=0;
  RateFact:=1;
  TimeAdju:=0;
  TimeBuff:=0;
  if StrToIntDef(Trim(ReadIni('Simulation')),0)=1
  then MaxFact:=1.5 else MaxFact:=1.05;
  MinFact:=1/MaxFact;
 end;
end;

 //////////////////////////////////////////////////////////////////////////////
 // E140 driver finalization, free DaqDataSheet
 //////////////////////////////////////////////////////////////////////////////
procedure TE140.DAQ_CMD_FREE;
begin
 if Assigned(CrwApi) then
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  FreeSharedMemoryBuffer;
  DaqDataSheet(0);
 end;
end;

 //////////////////////////////////////////////////////////////////////////////
 // E140 driver main polling loop
 //////////////////////////////////////////////////////////////////////////////
procedure TE140.DAQ_CMD_POLL;
var adOfs,adAmp,adFSR,adCod,adVal,adTim,x,y,x1,y1,x2,y2,mSecs,cTime,KRate,ARate:Double;
    daVal,daCod,tKadr,tSamp,tBuff,tOffs,tCorr,tDiff,cAdjP,cAdjC,aFact,pFact,sFact:Double;
    i,nWords,Offset,Index,nRequest,nChans,iKadr,nKadr,iCtrl,iChan,iGain,iBuff:Integer;
    nBlocks,nSamples,nBuff,nFact,iFact:Int64; BuffCode:SmallInt;
    FirTab:array[0..AdcMaxChans-1] of Integer;
    OfsCal,AmpCal:array[0..3] of Double;
    Readout,SharmName:LongString;
begin
 if Assigned(CrwApi) then
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  //
  // Extract ADC block parameters passed via Readout string tag.
  // Readout = SharmName,nWords,Offset,Index,nRequest,nBlock,nSamples
  //
  Readout:=sGetTag(tag.Readout);                            // Comma separated list of parameters
  if WordCount(Readout,ScanSpaces)<7 then Exit;             // Must contain at least seven words:
  SharmName:=ExtractWord(1,Readout,ScanSpaces);             // 1 : Shared memory buffer name;
  nWords:=StrToInt(ExtractWord(2,Readout,ScanSpaces));      // 2 : Data block size, words;
  Offset:=StrToInt(ExtractWord(3,Readout,ScanSpaces));      // 3 : Data block offset, bytes;
  Index:=StrToInt(ExtractWord(4,Readout,ScanSpaces));       // 4 : Data block index in Buffer;
  nRequest:=StrToInt(ExtractWord(5,Readout,ScanSpaces));    // 5 : Request number, 0..E140NumBuffers-1;
  nBlocks:=StrToInt64(ExtractWord(6,Readout,ScanSpaces));   // 6 : Data blocks counter;
  nSamples:=StrToInt64(ExtractWord(7,Readout,ScanSpaces));  // 7 : ADC samples counter.
  //
  // Check shared memory buffer, (re)-initialize if not initialized or name changed.
  //
  if not Assigned(shmData)                    // If shared memory is not initialized
  or not SameText(shmName,SharmName)          // Or shared memory name changed, i.e. driver restarted
  then InitSharedMemoryBuffer(SharmName);     // Then (re)-initialize shared memory buffer.
  //
  // Process ADC data passed in shared memory buffer, if one exists.
  //
  if Assigned(shmData) then with shmData^,AdcActual,Description.Adc do begin
   //
   // Get current time by computer clock.
   //
   cTime:=Time;                               // Get current time, TimeUnits since TimeBase
   mSecs:=mSecNow;                            // Get current time, ms since Xmas 0001:01:01
   //
   // On first data block initialize time of ADC start.
   //
   if (nBlocks=0) or (TimeStart=0) then begin // On first buffer income
    TimeStart:=mSecs                          // find time of ADC start
     -(nWords div ChannelsQuantity)/KadrRate  // as current time, minus
     -(nWords mod ChannelsQuantity)/AdcRate;  // time of buffer readout
    TimeStart:=Int(TimeStart);                // and save it for future
    TimeAdju:=TimeStart;                      // Init time adjustment
    RateFact:=1;                              // Rate scale factor
    TimeBuff:=0;                              // Clear buffer time
   end;
   //
   // Initialize uses variables and tables:
   // current ADC rate, buffer time offset, Fir table, ADC calibration.
   //
   adTim:=cTime;                              // To avoid compiler hints
   ARate:=AdcRate*RateFact;                   // Apply AdcRate  correction
   KRate:=KadrRate*RateFact;                  // Apply KadrRate correction
   tOffs:=TimeStart-TimeBase+TimeBuff;        // ADC buffer time offset,ms
   for iChan:=0 to AdcMaxChans-1 do FirTab[iChan]:=iGetTag(tag.Fir[iChan]);
   for iGain:=Low(OfsCal) to High(OfsCal) do begin
    if Active then begin                      // If ADC description Active
     OfsCal[iGain]:=OffsetCalibration[iGain]; // ADC offset for correction
     AmpCal[iGain]:=ScaleCalibration[iGain];  // ADC amplification (scale)
    end else begin                            // Otherwise uses defaults
     OfsCal[iGain]:=0;                        // ADC offset for correction
     AmpCal[iGain]:=1;                        // ADC amplification (scale)
    end;
    if AmpCal[iGain]=0 then begin             // ADC calibration not good?
     OfsCal[iGain]:=0;                        // ADC offset for correction
     AmpCal[iGain]:=1;                        // ADC amplification (scale)
    end;
   end;
   //
   // Start cycle for each ADC samples.
   //
   for i:=0 to nWords-1 do begin              // Cycle by buffer nWords
    //
    // Find time for this sample.
    //
    iKadr:=i mod ChannelsQuantity;            // Sample number inside frame
    nKadr:=i div ChannelsQuantity;            // Frame  number inside block
    tSamp:=nKadr/KRate+iKadr/ARate;           // Sample time since buffer start, ms
    adTim:=(tOffs+tSamp)/TimeUnits;           // Sample time since TimeBase in TimeUnits
    //
    // Extract channel and gain from ControlTable for this sample.
    // Skip channel if one not connected to AO curve or has Fir<1.
    // Extract ADC full scale range and ADC calibration (by gain).
    //
    iCtrl:=ControlTable[iKadr];               // Control word of sample
    iChan:=iCtrl and $1F;                     // Channel number, 0..31
    iGain:=(iCtrl shr 6) and 3;               // Gain code in 6,7 bits
    if RefAo(iChan)=0 then Continue;          // Skip empty AnalogOutputs
    if FirTab[iChan]<1 then Continue;         // Skip channels with FIR<1
    adFSR:=ADC_INPUT_RANGES_E140[iGain];      // ADC full scale range, Volts
    adOfs:=OfsCal[iGain];                     // ADC offset for correction
    adAmp:=AmpCal[iGain];                     // ADC amplification (scale)
    //
    // Extract ADC data from shared buffer and convert to milliVolts.
    //
    BuffCode:=Buffer[Index+i];                // Buffered ADC code, not corrected
    adCod:=(BuffCode+adOfs)*adAmp;            // Corrected ADC code; FSR=8000 ADC
    adVal:=adCod*adFSR*(1000/8000);           // Convert corrected ADC code to mV
    //
    // Apply Fir filter. Put filtered data to AO curve.
    //
    AddStatSum(StatSum[iChan],adTim,adVal);   // Add the point to StatSum
    if StatSum[iChan].N>=FirTab[iChan] then begin  // If StatSum filled enough
     if not FitStatSum(StatSum[iChan],x,y)    // Fit data collected in StatSum
     then Failure('Data fit fails!');         // Raise exception on error?
     if not PutAo(iChan,x,y)                  // Put data point to AnalogOutput
     then {Failure('PutAO fails!')};          // Raise exception on error?
     ZeroStatSum(StatSum[iChan]);             // Clear StatSum for future
    end;
   end;
   //
   // Prepare time offset for next data buffer.
   //
   nBuff:=nWords div ChannelsQuantity;        // Number of frames in nWords
   iBuff:=nWords mod ChannelsQuantity;        // Number of steps  in nWords
   tBuff:=nBuff/KRate+iBuff/ARate;            // Buffer filling time, ms
   TimeBuff:=TimeBuff+tBuff;                  // Next buffer since ADC Start, ms
   //
   // Calculate time difference: ADC vs PC clock.
   //
   tDiff:=(cTime-adTim)*TimeUnits;            // Calculate ADC vs PC time difference, ms
   AddStatSum(TimeDiff,cTime,tDiff);          // Calculate time difference stat. summ
   if RefDo(1)<>0 then PutDo(1,cTime,tDiff);  // Put difference to curve if one exist
   if RefDo(2)<>0 then PutDo(2,cTime,ARate);  // Put ADC rate   to curve if one exist
   //
   // Clock adjustment procedure. RateFact uses to adjust ADC clock rate.
   // Procedure should calculate new RateFact for next adjustment period.
   //
   cAdjC:=rGetTag(tag.CadjC);                 // Clock adjustment coefficient
   cAdjP:=iGetTag(tag.CadjP)*1e3;             // Clock adjustment period, ms
   if cAdjP<=0 then RateFact:=1 else          // Check adjustment period
   if cAdjC<=0 then RateFact:=1 else          // Check adjustment coeff.
   if cAdjC>1  then RateFact:=1 else          // Check adjustment coeff.
   if mSecs-TimeAdju>cAdjP then begin         // Apply clock adjustment
    nFact:=(nSamples+nWords) div ChannelsQuantity; // Frame  counter
    iFact:=(nSamples+nWords) mod ChannelsQuantity; // Sample counter
    aFact:=nFact/KadrRate+iFact/AdcRate;      // ADC time since start
    pFact:=mSecs-TimeStart;                   // PC  time since start
    if FitStatSum(TimeDiff,tCorr,tDiff) then pFact:=pFact+2*tDiff;
    sFact:=aFact/pFact;                       // Sampling rate factor estimation
    RateFact:=cAdjC*sFact+(1-cAdjC)*RateFact; // Rate scale factor fit+smoothing
    if IsNan(RateFact)  then RateFact:=1;     // Check rate scale factor is good
    if IsInf(RateFact)  then RateFact:=1;     // Check rate scale factor is good
    if RateFact>MaxFact then RateFact:=1;     // Check rate scale factor is good
    if RateFact<MinFact then RateFact:=1;     // Check rate scale factor is good
    //Echo(Format('%15g, %15g, %15g',[tDiff, RateFact, sFact])); // Debug notes
    ZeroStatSum(TimeDiff);                    // Add time stat. summs
    TimeAdju:=mSecs;                          // Save adjustment time
   end;
   //
   // Send new DAC value to driver if one changed.
   //
   for i:=0 to 1 do                           // DAC 0,1 output
   if RefAi(i)<>0 then                        // If curve connected
   if GetAi_N(i)>0 then                       // If curve has points
   if iGetTag(tag.GateAo)<>0 then             // If DAC gate enabled
   if shmData.Description.Dac.Active then     // If DAC is installed
   with shmData.Description.Dac do begin      // Digital to Analog
    daVal:=GetAi_Yn(i)/1000;                  // Get last Y, Volts
    x1:=-DAC_OUTPUT_RANGE_E140; y1:=-2048;    // 1-st endpoint
    x2:=+DAC_OUTPUT_RANGE_E140; y2:=+2047;    // 2-nd endpoint
    daCod:=y1+(daVal-x1)*(y2-y1)/(x2-x1);     // Interpolation
    daCod:=(daCod+OffsetCalibration[i])*ScaleCalibration[i];
    daCod:=Max(y1,Min(y2,daCod));             // Limits
    DeviceMessage(DevRef,Format('@E140.AO=%d,%d'#13#10,[i,Round(daCod)]));
   end;
  end;
 end;
end;

 //////////////////////////////////////////////////////////////////////////////
 // E140 driver failure if unrecognized command detected
 //////////////////////////////////////////////////////////////////////////////
procedure TE140.DAQ_CMD_FAIL;
begin
 if Assigned(CrwApi) then
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  RAISE EDaqApi.Create(Format('Wrong DaqCommand=%d!',[DaqCommand]));
 end;
end;

procedure TE140.InitSharedMemoryBuffer(aName:LongString);
begin
 if Assigned(CrwApi) then
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  if Assigned(shmData) or (shmFile<>0) then FreeSharedMemoryBuffer;
  shmFile:=OpenFileMapping(FILE_MAP_READ, False, PChar(aName));
  shmData:=MapViewOfFile(shmFile, FILE_MAP_READ, 0, 0, 0);
  shmName:=aName;
 end;
end;

procedure TE140.FreeSharedMemoryBuffer;
begin
 if Assigned(CrwApi) then
 with CrwApi,SysApi,GuiApi,DaqApi do begin
  if Assigned(shmData) then UnmapViewOfFile(shmData);
  if shmFile<>0 then CloseHandle(shmFile);
  shmData:=nil;
  shmName:='';
  shmFile:=0;
 end;
end;

procedure TE140.Failure(const msg:LongString);
begin
 RAISE EDaqApi.Create(msg);
end;

 //////////////////////////////////////////////////////////////////////////////
 // MAIN PLUGIN FUNCTION
 //////////////////////////////////////////////////////////////////////////////
function CRW32_PLUGIN(TheCrwApi:TCrwApi):Integer; StdCall;
var E140:PE140;
begin
 Result:=0;
 with TheCrwApi,SysApi,GuiApi,DaqApi do
 try
  if Target <> ForDataAcquisition then RAISE EDaqApi.Create('Invalid Target!');
  E140:=DaqDataSheet(sizeof(E140^));
  if not Assigned(E140)
  then RAISE EDaqApi.Create('Out of memory!');
  if Assigned(E140.CrwApi) and (E140.CrwApi<>TheCrwApi)
  then RAISE EDaqApi.Create('Invalid CrwApi.');
  case DaqCommand of
   DAQ_CMD_INIT: E140.DAQ_CMD_INIT(TheCrwApi);	// Initialize E140 driver
   DAQ_CMD_FREE: E140.DAQ_CMD_FREE;			// Finalize   E140 driver
   DAQ_CMD_POLL: E140.DAQ_CMD_POLL;			// General    E140 data acquisition loop
   else          E140.DAQ_CMD_FAIL;			// Detected   E140 driver failure
  end;
 except
  on E:Exception do begin
   Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
   if UsesBlaster then Voice('EXCEPTION');
   Result:=-1;
  end;
 end;			
end;

exports CRW32_PLUGIN name CRW32_PLUGIN_ID;
begin
end.
 
 