 {                                                                            
 ***********************************************************************
 Daq Pascal application program GENDC_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @PrintCmdTable - Print Command Table state.
| @Simulate id v - Set simulation data v to id=(MV?,PV?,
|                  MC?,PC?,RMT?,OUT?,FLD?,AST?,OVP?,UVL?,
|                  MODE?,FLT?,FENA?,FEVE?,STAT?,SENA?,SEVE?,
|                  OHM,NLP).
| @Setting id v  - Send command id with data v, id=(ADR,IDN,
|                  CLS,RST,RMT,PV,PC,OUT,FLD,AST,OVP,UVL,OVM,
|                  SAV,RCL,FENA,SENA,RATE_A,RATE_V).
| @Ramping id v  - Start ramping id=(PVn,PCn) to v value.
| @Remote @cmd   - run @cmd remotely: send to DIM server
| @Edit id       - edit id (PV,PC,OVP,UVL,FENA,SENA)
| @MenuToolsOpen - open Menu Tools dialog.
| @LoadIni       - load params from INI file.
| @SaveIni       - save params to   INI file.
| @DimGuiClick   - handle remote click from DIM
|********************************************************
[]
 }
{
;***************************************************
;** Table of TDK-Lambda Genesys series DC params. **
;***************************************************
[TDK-LAMBDA-GENESYS-SERIES-TABLE];******************
;***************************************************
;GEN-1U-1500W series. Document: IA575-04-01-Rev. X  
;Ident          Vmax    Imax    OVP         UVL     
;***************************************************
 GEN6-100       6.000   100.00  0.5 7.50    5.70    
 GEN6-200       6.000   200.00  0.5 7.50    5.70    
 GEN8-90        8.000   90.00   0.5 10.0    7.60    
 GEN8-180       8.000   180.00  0.5 10.0    7.60    
 GEN12.5-60     12.500  60.000  1.0 15.0    11.9    
 GEN12.5-120    12.500  120.00  1.0 15.0    11.9    
 GEN20-38       20.000  38.000  1.0 24.0    19.0    
 GEN20-76       20.000  76.00   1.0 24.0    19.0    
 GEN30-25       30.000  25.000  2.0 36.0    28.5    
 GEN30-50       30.000  50.000  2.0 36.0    28.5    
 GEN40-19       40.000  19.000  2.0 44.0    38.0    
 GEN40-38       40.000  38.000  2.0 44.0    38.0    
 GEN50-30       50.000  30.000  5.0 57.0    47.5    
 GEN60-12.5     60.000  12.500  5.0 66.0    57.0    
 GEN60-25       60.000  25.000  5.0 66.0    57.0    
 GEN80-9.5      80.00   9.500   5.0 88.0    76.0    
 GEN80-19       80.00   19.000  5.0 88.0    76.0    
 GEN100-7.5     100.00  7.500   5.0 110     95.0    
 GEN100-15      100.00  15.000  5.0 110     95.0    
 GEN150-5       150.00  5.000   5.0 165     142     
 GEN150-10      150.00  10.000  5.0 165     142     
 GEN300-2.5     300.00  2.500   5.0 330     285     
 GEN300-5       300.00  5.000   5.0 330     285     
 GEN600-1.3     600.00  1.300   5.0 660     570     
 GEN600-2.6     600.00  2.600   5.0 660     570     
;***************************************************
;GEN-1U-2400W series. Document: IA669-04-01-Rev. F  
;Ident          Vmax    Imax    OVP         UVL     
;***************************************************
 GEN8-300       8.000   300.00  0.5 10.0    7.60    
 GEN10-240      10.000  240.00  0.5 12.0    9.50    
 GEN16-150      16.000  150.00  1.0 19.0    15.2    
 GEN20-120      20.000  120.00  1.0 24.0    19.0    
 GEN30-80       30.000  80.000  2.0 36.0    28.5    
 GEN40-60       40.000  60.000  2.0 44.0    38.0    
 GEN60-40       60.000  40.000  5.0 66.0    57.0    
 GEN80-30       80.00   30.000  5.0 88.0    76.0    
 GEN100-24      100.00  24.000  5.0 110.0   95.0    
 GEN150-16      150.00  16.000  5.0 165     142     
 GEN300-8       300.00  8.0000  5.0 330     285     
 GEN600-4       600.00  4.0000  5.0 660     570     
;***************************************************
;GEN-2U-3300W series. Document: IA626-04-01-Rev. Q  
;Ident          Vmax    Imax    OVP         UVL     
;***************************************************
 GEN8-400       8.000   400.00  0.5 10.0    7.60    
 GEN10-330      10.000  330.00  0.5 12.0    9.50    
 GEN15-220      15.000  220.00  1.0 18.0    14.3    
 GEN20-165      20.000  165.00  1.0 24.0    19.0    
 GEN30-110      30.000  110.00  2.0 36.0    28.5    
 GEN40-85       40.000  85.00   2.0 44.0    38.0    
 GEN60-55       60.000  55.000  5.0 66.0    57.0    
 GEN80-42       80.00   42.000  5.0 88.0    76.0    
 GEN100-33      100.00  33.000  5.0 110     95.0    
 GEN150-22      150.00  22.000  5.0 165     142     
 GEN200-16.5    200.00  16.500  5.0 220     190     
 GEN300-11      300.00  11.000  5.0 330     285     
 GEN600-5.5     600.00  5.500   5.0 660     570     
;***************************************************
;GEN-2U-5000W series. Document: IA657-04-01-Rev. M  
;Ident          Vmax    Imax    OVP         UVL     
;***************************************************
 GEN8-600       8.000   600.00  0.5 10.0    7.60    
 GEN10-500      10.000  500.00  0.5 12.0    9.50    
 GEN16-310      16.000  310.00  1.0 19.0    15.2    
 GEN20-250      20.000  250.00  1.0 24.0    19.0    
 GEN30-170      30.000  170.00  2.0 36.0    28.5    
 GEN40-125      40.000  125.00  2.0 44.0    38.0    
 GEN60-85       60.000  85.000  5.0 66.0    57.0    
 GEN80-65       80.00   65.000  5.0 88.0    76.0    
 GEN100-50      100.00  50.000  5.0 110     95.0    
 GEN150-34      150.00  34.000  5.0 165     142     
 GEN200-25      200.00  25.000  5.0 220     190     
 GEN300-17      300.00  17.000  5.0 330     285     
 GEN400-13      400.00  13.000  5.0 440     380     
 GEN500-10      500.00  10.000  5.0 550     475     
 GEN600-8.5     600.00  8.500   5.0 660     570     
;***************************************************
;GEN-3U-15kW series.  Document: 83532002 Rev E      
;Ident          Vmax    Imax    OVP         UVL     
;***************************************************
 GEN7.5-1000    7.5000  1000.0  0.75 7.88   7.125   
 GEN10-1000     10.000  1000.0  1.0  10.5   9.5     
 GEN12.5-800    12.500  800.00  1.25 13.13  11.875  
 GEN20-500      20.000  500.00  2.0  21.0   19.0    
 GEN25-400      25.000  400.00  2.5  26.25  23.75   
 GEN30-333      30.000  333.00  3.0  31.5   28.5    
 GEN40-250      40.000  250.00  4.0  42.0   38.0    
 GEN50-200      50.000  200.00  5.0  52.5   47.5    
 GEN60-167      60.000  167.00  6.0  63.0   57.0    
 GEN80-125      80.000  125.00  8.0  84.0   76.0    
 GEN100-100     100.00  100.00  10.0 105.0  95.0    
 GEN125-80      125.00  80.000  12.5 131.25 118.75  
 GEN150-66      150.00  66.000  15.0 157.5  142.5   
 GEN200-50      200.00  50.000  20.0 210.0  190.0   
 GEN250-40      250.00  40.000  25.0 262.5  237.5   
 GEN300-33      300.00  33.000  30.0 315.0  285.0   
 GEN400-25      400.00  25.000  40.0 420.0  380.0   
 GEN500-20      500.00  20.000  50.0 525.0  475.0   
 GEN600-17      600.00  17.000  60.0 630.0  570.0   
;15kW
 GEN30-500      30.000  500.00  3.0  31.5   28.5    
 GEN40-375      40.000  375.00  4.0  42.0   38.0    
 GEN50-300      50.000  300.00  5.0  52.5   47.5    
 GEN60-250      60.000  250.00  6.0  63.0   57.0    
 GEN80-187.5    80.000  187.50  8.0  84.0   76.0    
 GEN100-150     100.00  150.00  10.0 105.0  95.0    
 GEN125-120     125.00  120.00  12.5 131.25 118.75  
 GEN150-100     150.00  100.00  15.0 157.5  142.5   
 GEN200-75      200.00  75.000  20.0 210.0  190.0   
 GEN250-60      250.00  60.000  25.0 262.5  237.5   
 GEN300-50      300.00  50.000  30.0 315.0  285.0   
 GEN400-37.5    400.00  37.500  40.0 420.0  380.0   
 GEN500-30      500.00  30.000  50.0 525.0  475.0   
 GEN600-25      600.00  25.000  60.0 630.0  570.0   
;***************************************************
[];*************************************************
;***************************************************
}
program GENDC_DRV;               { TDK Lambda Genesys DC driver     }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 st_NoReq           = 0;         { Status : No request sent         }
 st_WaitAns         = 1;         { Status : Waiting answer          }
 st_WaitGap         = 2;         { Status : Waiting time gap        }
 st_TimeOut         = 3;         { Status : TimeOut found           }
 cm_ADRn            = 1;         { ADR n                            }
 cm_CLS             = 2;         { CLS                              }
 cm_RST             = 3;         { RST                              }
 cm_RMTn            = 4;         { RMT n                            }
 cm_RMTa            = 5;         { RMT?                             }
 cm_IDNa            = 6;         { IDN?                             }
 cm_REVa            = 7;         { REV?                             }
 cm_SNa             = 8;         { SN?                              }
 cm_DATEa           = 9;         { DATE?                            }
 cm_PVn             = 10;        { PV n                             }
 cm_PVa             = 11;        { PV?                              }
 cm_MVa             = 12;        { MV?                              }
 cm_PCn             = 13;        { PC n                             }
 cm_PCa             = 14;        { PC?                              }
 cm_MCa             = 15;        { MC?                              }
 cm_DVCa            = 16;        { DVC?                             }
 cm_OUTn            = 17;        { OUT n                            }
 cm_OUTa            = 18;        { OUT?                             }
 cm_FLDn            = 19;        { FLD n                            }
 cm_FLDa            = 20;        { FLD?                             }
 cm_OVPn            = 21;        { OVP n                            }
 cm_OVPa            = 22;        { OVP?                             }
 cm_OVM             = 23;        { OVM                              }
 cm_UVLn            = 24;        { UVL n                            }
 cm_UVLa            = 25;        { UVL?                             }
 cm_ASTn            = 26;        { AST n                            }
 cm_ASTa            = 27;        { AST?                             }
 cm_SAV             = 28;        { SAV                              }
 cm_RCL             = 29;        { RCL                              }
 cm_MODEa           = 30;        { MODE?                            }
 cm_STTa            = 31;        { STT?                             }
 cm_FLTa            = 32;        { FLT?                             }
 cm_FENAn           = 33;        { FENA n                           }
 cm_FENAa           = 34;        { FENA?                            }
 cm_FEVEa           = 35;        { FEVE?                            }
 cm_STATa           = 36;        { STAT?                            }
 cm_SENAn           = 37;        { SENA n                           }
 cm_SENAa           = 38;        { SENA?                            }
 cm_SEVEa           = 39;        { SEVE?                            }
 MaxCmdNum          = 39;        { Max command id number            }
 MaxTabNum          = 128;       { Max model table counter          }
 ao_POLL_RATE       = 0;         { Analog outputs...                }
 ao_ERROR_CNT       = 1;         {                                  }
 ao_PAR_MC          = 2;         {                                  }
 ao_PAR_MV          = 3;         {                                  }
 ao_PAR_MW          = 4;         {                                  }
 ao_PAR_PC          = 5;         {                                  }
 ao_PAR_PV          = 6;         {                                  }
 ao_PAR_SR          = 7;         {                                  }
 ao_PAR_FR          = 8;         {                                  }
 ao_BT_OUT          = 9;         {                                  }
 ao_BT_FLD          = 10;        {                                  }
 ao_BT_AST          = 11;        {                                  }
 ao_PAR_OVP         = 12;        {                                  }
 ao_PAR_UVL         = 13;        {                                  }
 ao_PAR_RMT         = 14;        {                                  }
 ao_PAR_MODE        = 15;        {                                  }
 ao_PAR_FENA        = 16;        {                                  }
 ao_PAR_FEVE        = 17;        {                                  }
 ao_PAR_SENA        = 18;        {                                  }
 ao_PAR_SEVE        = 19;        {                                  }
 ao_PAR_SRQ         = 20;        {                                  }
 si_OFFLINE         = 'OFFLINE'; { To identify offline state        }
 RS_232             = 232;       {                                  }
 RS_422             = 422;       {                                  }
 RS_485             = 485;       {                                  }
 DelayOnStart       = 1000;      { Delay before start polling       }
 UseCmdHash         = true;      { Use CmdHash for fast search      }

type
 TTagRef  = record tag,nai,ndi,nao,ndo:Integer; val:Real; end;
 TRamping = record t0,t1,v0,v1:real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 GENDC             : record      { GEN DC data                      }
  Simulator        : Boolean;    { Simulator mode                   }
  ecTimeOut        : Integer;    { Error code: TimeOut              }
  Com              : record      { COM port data                    }
   Port            : Integer;    {  Port number                     }
   Addr            : Integer;    {  Address                         }
   Status          : Integer;    {  Status flag                     }
   TimeOut         : Integer;    {  TimeOut, ms                     }
   TimeGap         : Integer;    {  Time delay after poll, ms       }
   PollPeriod      : Integer;    { Poll period, ms                  }
   Req,Ans,Buf     : String;     {  Request, Answer                 }
   ReqTime,AnsTime : Real;       {  Request, Answer time, ms        }       
   LastPoll        : Real;       { Time of last poll, ms            }
   PollRate        : Integer;    { Poll rate, poll/sec              }
   Interface       : Integer;    { 232/422/485                      }
   UseCheckSum     : Boolean;    { Use checksum                     }
  end;                           {                                  }
  Cmd              : record      {  Command cycle data              }
   Num             : Integer;    {   Current command number         }
   Enabled         : array [1..MaxCmdNum] of Boolean; {             }
   Acronym         : array [1..MaxCmdNum] of String;  {             }
   AnsPref         : array [1..MaxCmdNum] of String;  {             }
   DataFmt         : array [1..MaxCmdNum] of String;  {             }
   Comment         : array [1..MaxCmdNum] of String;  {             }
   OpData          : array [1..MaxCmdNum] of Real;    {             }
   OpBuff          : array [1..MaxCmdNum] of Real;    {             }
   SimDat          : array [1..MaxCmdNum] of Real;    {             }
   AoNum           : array [1..MaxCmdNum] of Integer; {             }
   Flags           : array [1..MaxCmdNum] of Integer; {             }
   Tag             : array [1..MaxCmdNum] of Integer; {             }
   Min             : array [1..MaxCmdNum] of Real;    {             }
   Max             : array [1..MaxCmdNum] of Real;    {             }
  end;                           {                                  }
  Tab              : record      {  Table of model parameters       }
   Count           : Integer;    {   Table counter                  }
   IdName          : array [1..MaxTabNum] of String;  {             }
   VolFmt          : array [1..MaxTabNum] of String;  {             }
   CurFmt          : array [1..MaxTabNum] of String;  {             }
   OvpFmt          : array [1..MaxTabNum] of String;  {             }
   UvlFmt          : array [1..MaxTabNum] of String;  {             }
   VolMax          : array [1..MaxTabNum] of Real;    {             }
   CurMax          : array [1..MaxTabNum] of Real;    {             }
   OvpMin          : array [1..MaxTabNum] of Real;    {             }
   OvpMax          : array [1..MaxTabNum] of Real;    {             }
   UvlMax          : array [1..MaxTabNum] of Real;    {             }
  end;                           {                                  }
  CmdHash          : Integer;    { Hash list ref. for fast search   }
  POLL_ENABLE      : TTagRef;    {                                  }
  POLL_RATE        : TTagRef;    {                                  }
  ERROR_CNT        : TTagRef;    {                                  }
  ID_NAME          : TTagRef;    {                                  }
  ID_IDN           : TTagRef;    {                                  }
  ID_REV           : TTagRef;    {                                  }
  ID_SN            : TTagRef;    {                                  }
  ID_DATE          : TTagRef;    {                                  }
  PAR_MC           : TTagRef;    {                                  }
  PAR_MV           : TTagRef;    {                                  }
  PAR_MW           : TTagRef;    {                                  }
  PAR_PC           : TTagRef;    {                                  }
  PAR_PV           : TTagRef;    {                                  }
  PAR_SR           : TTagRef;    {                                  }
  PAR_FR           : TTagRef;    {                                  }
  BT_FLD           : TTagRef;    {                                  }
  BT_AST           : TTagRef;    {                                  }
  BT_OUT           : TTagRef;    {                                  }
  PAR_OVP          : TTagRef;    {                                  }
  PAR_UVL          : TTagRef;    {                                  }
  PAR_RMT          : TTagRef;    {                                  }
  PAR_MODE         : TTagRef;    {                                  }
  PAR_FENA         : TTagRef;    {                                  }
  PAR_FEVE         : TTagRef;    {                                  }
  PAR_SENA         : TTagRef;    {                                  }
  PAR_SEVE         : TTagRef;    {                                  }
  PAR_SRQ          : TTagRef;    {                                  }
  RATE_A           : TTagRef;    {                                  }
  RATE_V           : TTagRef;    {                                  }
  RAMP_A           : TTagRef;    {                                  } 
  RAMP_V           : TTagRef;    {                                  } 
  BT_FLD_GUI       : TTagRef;    {                                  }
  BT_AST_GUI       : TTagRef;    {                                  }
  BT_OUT_GUI       : TTagRef;    {                                  }
  RampingA         : TRamping;   {                                  }
  RampingV         : TRamping;   {                                  }
  RampingTime      : Real;       {                                  }
  ModelCode        : Integer;    { Encoded model ranges (Tab index) }
  ModelName        : String;     { Model name like LAMBDA,GEN6-100  }
  SimOhm           : Real;       { Simulate output resistance, Ohm  }
  SimNLP           : Real;       { Simulate noise level, percent    }
 end;                            {                                  }
 cmd_DimTagUpdate  : Integer;    { @DimTagUpdate                    }
 cmd_Simulate      : Integer;    { @Simulate                        }
 cmd_PrintCmdTable : Integer;    { @PrintCmdTable                   }
 cmd_Ramping       : Integer;    { @Ramping                         }
 cmd_Setting       : Integer;    { @Setting                         }
 cmd_Remote        : Integer;    { @Remote                          }
 cmd_Edit          : Integer;    { @Edit                            }
 cmd_MenuToolsOpen : Integer;    { @MenuToolsOpen                   }
 cmd_LoadIni       : Integer;    { @LoadIni                         }
 cmd_SaveIni       : Integer;    { @SaveIni                         }
 MenuToolsSelected : Integer;    { MenuTools selected item index    }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 //
 // Calculate checksum - simple sum of bytes, method=38(LOSER), 8 bits.
 //
 function CalcCheckSum(s:String):Integer;
 begin
  CalcCheckSum:=HashIndexOf(s,256,38);
 end;
 //
 // Append checksum ($xx hex) to data string s.
 // Example: STT? -> STT?$3A, STAT? -> STAT?$7B
 //
 function AppendCheckSum(s:String):String;
 var len,cs:Integer;
 begin
  len:=Length(s); cs:=0;
  if len>0 then cs:=CalcCheckSum(s);
  if len>0 then s:=s+'$'+HexB(cs);
  AppendCheckSum:=s;
 end;
 //
 // Check if data string s has valid checksum.
 //
 function HasValidCheckSum(s:String):Boolean;
 var p,c1,c2:Integer;
 begin
  c1:=0; c2:=0;
  p:=Pos('$',s);
  if (p>0) then begin
   c1:=CalcCheckSum(Copy(s,1,p-1));
   c2:=Val(Copy(s,p));
  end;
  HasValidCheckSum:=(c1=c2);
 end;
 //
 // Check if data string s has a checksum.
 //
 function HasCheckSum(s:String):Boolean;
 var p:Integer;
 begin
  p:=Pos('$',s);
  HasCheckSum:=(p>0);
 end;
 //
 // Drop checksum from string. STT?$3A -> STT?
 //
 function DropCheckSum(s:String):String;
 var p:Integer;
 begin
  p:=Pos('$',s);
  if (p>0) then s:=Copy(s,1,p-1); 
  DropCheckSum:=s;
 end;
 //
 // Test check sum algorithm.
 //
 procedure TestCheckSum;
  procedure Test(s1,s2:String);
  var m:Integer;
  begin
   m:=Ord(HasValidCheckSum(AppendCheckSum(s1)));
   Success(StrFmt('Test %-5s ',s1)+StrFmt(' %-8s ',s2)
          +StrFmt(' %-8s ',AppendCheckSum(s1))+StrFmt(' %d ',m));
  end;
 begin
  Test('STT?',  'STT?$3A');
  Test('STAT?', 'STAT?$7B');
 end;
 //
 // Ramping routines.
 //
 procedure RampingClear(var R:TRamping);
 begin
  R.t0:=0; R.t1:=0; R.v0:=0; R.v1:=0;
 end;
 procedure RampingStart(var R:TRamping; v0,v1,rate:Real);
 begin
  RampingClear(R);
  R.t0:=mSecNow; R.t1:=R.t0; R.v0:=v0; R.v1:=v1;
  if (rate>0) then R.t1:=R.t1+1000*abs((v1-v0)/rate);
 end;
 function RampingGoing(var R:TRamping):Boolean;
 begin
  RampingGoing:=(R.t0>0);
 end;
 function RampingComplete(var R:TRamping):Boolean;
 begin
  RampingComplete:=(mSecNow>R.t1);
 end;
 function RampingValue(var R:TRamping):Real;
 var v,ms:Real;
 begin
  ms:=mSecNow;
  if (ms>R.t1) or (R.t0=0) then v:=R.v1
  else v:=R.v0+(R.v1-R.v0)*(ms-R.t0)/(R.t1-R.t0);
  RampingValue:=v;
 end;
 //
 // Get format by sample: 60.00 -> %5.2f
 //
 function GetFormatBySample(Sample:String):String;
 var w,d:Integer;
 begin
  w:=Length(Sample); d:=w-Pos('.',Sample);
  if (d=w) then begin d:=1; w:=w+2; end;
  GetFormatBySample:='%'+Str(w)+'.'+Str(d)+'f';
 end;
 //
 // Find table index by string model IDN, like 'GEN6-100'.
 //
 function GENDC_Tab_Find(IDN:String):Integer;
 var i,n:Integer;
 begin
  n:=0; i:=0;
  if (GENDC.Tab.Count>0) then
  if not IsEmptyStr(IDN) then
  repeat
   i:=i+1;
   if IsSameText(IDN,GENDC.Tab.IdName[i]) then n:=i;
  until (i>=MaxTabNum) or (i>=GENDC.Tab.Count) or (n>0);
  GENDC_Tab_Find:=n;
 end;
 //
 // Clear GENDC series table.
 //
 procedure GENDC_Tab_Clear;
 var i:Integer;
 begin
  GENDC.Tab.Count:=0;
  for i:=1 to MaxTabNum do GENDC.Tab.IdName[i]:='';
  for i:=1 to MaxTabNum do GENDC.Tab.VolFmt[i]:='';
  for i:=1 to MaxTabNum do GENDC.Tab.CurFmt[i]:='';
  for i:=1 to MaxTabNum do GENDC.Tab.OvpFmt[i]:='';
  for i:=1 to MaxTabNum do GENDC.Tab.UvlFmt[i]:='';
  for i:=1 to MaxTabNum do GENDC.Tab.VolMax[i]:=0;
  for i:=1 to MaxTabNum do GENDC.Tab.CurMax[i]:=0;
  for i:=1 to MaxTabNum do GENDC.Tab.OvpMin[i]:=0;
  for i:=1 to MaxTabNum do GENDC.Tab.OvpMax[i]:=0;
  for i:=1 to MaxTabNum do GENDC.Tab.UvlMax[i]:=0;
 end;
 //
 // Read GENDC series table.
 //
 procedure GENDC_Tab_Read;
 var sect,i,n,e:Integer;
 begin
  GENDC_Tab_Clear; e:=0;
  sect:=ReadIniSection(text_New,28,ProgramSourcePas,'[TDK-LAMBDA-GENESYS-SERIES-TABLE]');
  for i:=0 to text_NumLn(sect)-1 do begin
   if IsSameText('GEN',Copy(text_GetLn(sect,i),1,3)) then
   if (WordCount(text_GetLn(sect,i))=6) then begin
    n:=GENDC.Tab.Count+1;
    if n<=MaxTabNum then begin
     GENDC.Tab.IdName[n]:=ExtractWord(1,text_GetLn(sect,i));
     GENDC.Tab.VolFmt[n]:=GetFormatBySample(ExtractWord(2,text_GetLn(sect,i)));
     GENDC.Tab.VolMax[n]:=rVal(ExtractWord(2,text_GetLn(sect,i)));
     GENDC.Tab.CurFmt[n]:=GetFormatBySample(ExtractWord(3,text_GetLn(sect,i)));
     GENDC.Tab.CurMax[n]:=rVal(ExtractWord(3,text_GetLn(sect,i)));
     GENDC.Tab.OvpFmt[n]:=GetFormatBySample(ExtractWord(5,text_GetLn(sect,i)));
     GENDC.Tab.OvpMin[n]:=rVal(ExtractWord(4,text_GetLn(sect,i)));
     GENDC.Tab.OvpMax[n]:=rVal(ExtractWord(5,text_GetLn(sect,i)));
     GENDC.Tab.UvlFmt[n]:=GetFormatBySample(ExtractWord(6,text_GetLn(sect,i)));
     GENDC.Tab.UvlMax[n]:=rVal(ExtractWord(6,text_GetLn(sect,i)));
     GENDC.Tab.Count:=n;
     if not IsSameText(GENDC.Tab.IdName[n],'GEN'+Str(GENDC.Tab.VolMax[n])+'-'+Str(GENDC.Tab.CurMax[n])) then begin
      Problem(text_GetLn(sect,i));
      e:=e+1;
     end;
     {
     write(StrFmt('%-14s ', GENDC.Tab.IdName[n]));
     write(StrFmt('%-7s ',  GENDC.Tab.VolFmt[n]));
     write(StrFmt('%-7s ',  GENDC.Tab.CurFmt[n]));
     write(StrFmt('%-11s ', GENDC.Tab.OvpFmt[n]));
     write(StrFmt('%-7s ',  GENDC.Tab.UvlFmt[n]));
     writeln;
     }
     writeln(text_GetLn(sect,i));
    end;
   end;
  end;
  Success(StrFmt('Read %d items.',n)+StrFmt(' Found %d error(s).',e));
  bNul(text_Free(sect));
  TestCheckSum;
 end;
 //
 // Init GEN series table.
 //
 procedure GENDC_Tab_Init;
 begin
  GENDC_Tab_Clear;
  GENDC_Tab_Read;
 end;
 //
 // Check ModelCode is valid or return zero.
 //
 function CheckModelCode(ModelCode:Integer):Integer;
 begin
  if (ModelCode<1) then ModelCode:=0 else
  if (ModelCode>MaxTabNum) then ModelCode:=0 else
  if (ModelCode>GENDC.Tab.Count) then ModelCode:=0;
  CheckModelCode:=ModelCode;
 end;
 //
 // Check & return GENDC ModelCode, nonzero if IDN detection was successful.
 //
 function GENDC_ModelCode:Integer;
 begin
  GENDC_ModelCode:=CheckModelCode(GENDC.ModelCode);
 end;
 //
 // Parse IDN string like 'LAMBDA,GEN6-100', find ModelCode.
 //
 function GENDC_ParseIDN(IDN:String; silent:Boolean):Integer;
 var i,ModelCode:Integer;
 begin
  ModelCode:=0; IDN:=Trim(IDN);
  for i:=1 to WordCount(IDN) do if (ModelCode=0) then
  if IsSameText(Copy(ExtractWord(i,IDN),1,3),'GEN') then
  ModelCode:=CheckModelCode(GENDC_Tab_Find(ExtractWord(i,IDN)));
  if not silent then
  if CheckModelCode(ModelCode)>0 
  then Success('Detected IDN: '+GENDC.Tab.IdName[ModelCode])
  else Problem('Fail detect IDN: '+IDN);
  GENDC_ParseIDN:=ModelCode;
 end;
 //
 // Extract GEN data by prefix (MV,PV,MC,PC,SR,FR)
 // GENDC_ExtractData('MV(00.006),PV(10.000),MC(000.00),PC(3.000),SR(04),FR(00)','PV'); -> 10.000
 //
 function GENDC_ExtractData(Data,Prefix:String):String;
 var p:Integer;
 begin
  p:=Pos(Prefix,Data);
  if (p=0) then Data:='' else begin
   Data:=Copy(Data,p+Length(Prefix));
   p:=1; while Pos(StrFetch(Data,p),'0123456789.-+()')>0 do p:=p+1;
   Data:=StrRemoveBrackets(Copy(Data,1,p-1),'(',')');
  end;
  GENDC_ExtractData:=Data;
 end;
 //
 // Format GENDC data: GENDC_Fmt('%6.2f',pi); -> 003.14
 //                    GENDC_Fmt('%2.2X',15); -> 0F
 //
 function GENDC_Fmt(Fmt:String; Data:Real):String;
 var i,n:Integer; c:Char;
 begin
  if Pos('%',Fmt)>0 then begin
   c:=StrFetch(LoCaseStr(Fmt),Length(Fmt));
   if (c='x') or (c='d') then Fmt:=StrFmt(Fmt,Round(Data)) else
   if (c='f') or (c='g') or (c='e') then Fmt:=StrFmt(Fmt,Data) else Fmt:='';
   if Length(Fmt)>0 then Fmt:=StrReplace(Fmt,Dump(' '),Dump('0'),3);
  end else Fmt:='';
  GENDC_Fmt:=Fmt;
 end;
 //
 // Find command by id (Acronym).
 //
 function GENDC_Cmd_Find(id:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(id)>0) then
  if UseCmdHash and (GENDC.CmdHash<>0) then begin
   n:=HashList_GetLink(GENDC.CmdHash,id);
  end else begin
   for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(id,GENDC.Cmd.Acronym[i])
   then n:=i;
  end;
  GENDC_Cmd_Find:=n;
 end;
 //
 // Nice tooltip notifier.
 //
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  if Length(aText)>0 then begin
   ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
  end;
 end;
 //
 // Post command to local/remote server.
 //
 procedure PostCmdLocal(cmd:String);
 begin
  DevPostCmdLocal(cmd);
 end;
 procedure PostCmdRemote(cmd:String);
 begin
  Dim_GuiConsoleSend(DevName,cmd);
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс Статусных Регистров        (CLS)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting CLS');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сброс Регистров Устройства       (RST)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting RST');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Включить   Защиту (foldback)     (FLD)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting FLD 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Защиту (foldback)     (FLD)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting FLD 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Включить   Авто-(ре)Старт        (AST)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting AST 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Авто-(ре)Старт        (AST)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting AST 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Подключить Выход                 (OUT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting OUT 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Отключить  Выход                 (OUT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting OUT 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('OverVoltageProtection to Maximum (OVM)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting OVM');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Remote mode - LOC                (RMT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting RMT 0');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Remote mode - REM                (RMT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting RMT 1');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Remote mode - LLO                (RMT)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting RMT 2');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Save cuttent settings            (SAV)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting SAV');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Recall saved settings            (RCL)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Remote @Setting RCL');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка по Току, А                (PC)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit PC');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка по Напряжению, В          (PV)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit PV');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Защита по переНапряжению, В      (OVP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit OVP');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Защита по недоНапряжению, В      (UVL)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit UVL');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка Status Enable Register  (SENA)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit SENA');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Уставка Fault  Enable Register  (FENA)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Edit FENA');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Печать состояния таблицы команд');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@PrintCmdTable');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 450 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  235 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 {
 Menu Sena Starter to start editing.
 }
 procedure MenuSenaStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Уставка Status Enable Register');
   n:=n+EditAddInputLn('Задайте Status Enable Register:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Бит 0 - CV    - Const Voltage    - Стабилизация напряжения ');
   n:=n+EditAddInputLn('Бит 1 - CC    - Const Current    - Стабилизация тока       ');
   n:=n+EditAddInputLn('Бит 2 - NFLT  - NO FAULT flag    - Нет ошибок/сбоев        ');
   n:=n+EditAddInputLn('Бит 3 - FLT   - FAULT flag       - Есть ошибка/сбой        ');
   n:=n+EditAddInputLn('Бит 4 - AST   - AutoStart mode   - Режим авто-старта       ');
   n:=n+EditAddInputLn('Бит 5 - FDE   - Foldback enabled - Разрешена защита по току');
   n:=n+EditAddInputLn('Бит 6 - SPARE - Fixed to zero    - Не используется         ');
   n:=n+EditAddInputLn('Бит 7 - LCL   - Local mode       - Включен локальный режим ');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 230 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  107 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('CheckBoxList',EditGetUID('SEL_'+NameTag(GENDC.PAR_SENA.tag)),'0 '+Str(iGetTag(GENDC.PAR_SENA.tag)));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Fena Starter to start editing.
 }
 procedure MenuFenaStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Уставка Fault Enable Register');
   n:=n+EditAddInputLn('Задайте Fault Enable Register:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Бит 0 - SPARE - Fixed to zero             - Не используется                  ');
   n:=n+EditAddInputLn('Бит 1 - AC    - AC fail has occurred      - Сбой по питанию                  ');
   n:=n+EditAddInputLn('Бит 2 - OTP   - OverTemperatureProtection - Защита по температуре            ');
   n:=n+EditAddInputLn('Бит 3 - FOLD  - Foldback Protection       - Защита по току                   ');
   n:=n+EditAddInputLn('Бит 4 - OVP   - OverVoltage Protection    - Защита по напряжению             ');
   n:=n+EditAddInputLn('Бит 5 - SO    - Rear panel Shut Off       - Выключение (задняя панель)       ');
   n:=n+EditAddInputLn('Бит 6 - OFF   - Front panel Output Off    - Откл. нагрузки (передняя панель) ');
   n:=n+EditAddInputLn('Бит 7 - ENA   - Rear panel J1 Closed      - Запуск размыкания (задняя панель)');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[Bold]');
   n:=n+EditAddSetting('@set Form.Left 270 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  107 relative '+Copy(DevName,2)+' PaintBox');
   //////////////////////////////////////////
   n:=n+EditAddClosing('CheckBoxList',EditGetUID('SEL_'+NameTag(GENDC.PAR_FENA.tag)),'0 '+Str(iGetTag(GENDC.PAR_FENA.tag)));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 //
 // Initialize tag refreshment value.
 //
 procedure GENDC_FillTag(tag:Integer; InitVal:Real);
  procedure Process(var R:TTagRef);
  begin
   if (R.tag=tag) then bNul(ShouldRefresh(R.val,InitVal)>0);
  end;
 begin
  if (TypeTag(tag)>0) then begin
   Process(GENDC.POLL_ENABLE);
   Process(GENDC.POLL_RATE);
   Process(GENDC.ERROR_CNT);
   Process(GENDC.ID_NAME);
   Process(GENDC.ID_IDN);
   Process(GENDC.ID_REV);
   Process(GENDC.ID_SN);
   Process(GENDC.ID_DATE);
   Process(GENDC.PAR_MC);
   Process(GENDC.PAR_MV);
   Process(GENDC.PAR_MW);
   Process(GENDC.PAR_PC);
   Process(GENDC.PAR_PV);
   Process(GENDC.PAR_SR);
   Process(GENDC.PAR_FR);
   Process(GENDC.PAR_OVP);
   Process(GENDC.PAR_UVL);
   Process(GENDC.BT_FLD);
   Process(GENDC.BT_AST);
   Process(GENDC.BT_OUT);
   Process(GENDC.PAR_RMT);
   Process(GENDC.PAR_MODE);
   Process(GENDC.PAR_FENA);
   Process(GENDC.PAR_FEVE);
   Process(GENDC.PAR_SENA);
   Process(GENDC.PAR_SEVE);
   Process(GENDC.PAR_SRQ);
   Process(GENDC.RATE_A);
   Process(GENDC.RATE_V);
   Process(GENDC.RAMP_A);
   Process(GENDC.RAMP_V);
  end;
 end;
 //
 // Initialize tags.
 //
 procedure GENDC_FillTags(InitVal:Real);
 begin
  GENDC.POLL_ENABLE.val:=InitVal;
  GENDC.POLL_RATE.val:=InitVal;
  GENDC.ERROR_CNT.val:=InitVal;
  GENDC.ID_NAME.val:=InitVal;
  GENDC.ID_IDN.val:=InitVal;
  GENDC.ID_REV.val:=InitVal;
  GENDC.ID_SN.val:=InitVal;
  GENDC.ID_DATE.val:=InitVal;
  GENDC.PAR_MC.val:=InitVal;
  GENDC.PAR_MV.val:=InitVal;
  GENDC.PAR_MW.val:=InitVal;
  GENDC.PAR_PC.val:=InitVal;
  GENDC.PAR_PV.val:=InitVal;
  GENDC.PAR_SR.val:=InitVal;
  GENDC.PAR_FR.val:=InitVal;
  GENDC.PAR_OVP.val:=InitVal;
  GENDC.PAR_UVL.val:=InitVal;
  GENDC.BT_FLD.val:=InitVal;
  GENDC.BT_AST.val:=InitVal;
  GENDC.BT_OUT.val:=InitVal;
  GENDC.PAR_RMT.val:=InitVal;
  GENDC.PAR_MODE.val:=InitVal;
  GENDC.PAR_FENA.val:=InitVal;
  GENDC.PAR_FEVE.val:=InitVal;
  GENDC.PAR_SENA.val:=InitVal;
  GENDC.PAR_SEVE.val:=InitVal;
  GENDC.PAR_SRQ.val:=InitVal;
  GENDC.RATE_A.val:=InitVal;
  GENDC.RATE_V.val:=InitVal;
  GENDC.RAMP_A.val:=InitVal;
  GENDC.RAMP_V.val:=InitVal;
 end;
 procedure GENDC_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
   InitTag(GENDC.POLL_ENABLE.tag, Prefix+'.POLL_ENABLE',    1); 
   InitTag(GENDC.POLL_RATE.tag,   Prefix+'.POLL_RATE',      2);
   InitTag(GENDC.ERROR_CNT.tag,   Prefix+'.ERROR_CNT',      2);
   InitTag(GENDC.ID_NAME.tag,     Prefix+'.ID_NAME',        3);
   InitTag(GENDC.ID_IDN.tag,      Prefix+'.ID_IDN',         3);
   InitTag(GENDC.ID_REV.tag,      Prefix+'.ID_REV',         3);
   InitTag(GENDC.ID_SN.tag,       Prefix+'.ID_SN',          3);
   InitTag(GENDC.ID_DATE.tag,     Prefix+'.ID_DATE',        3);
   InitTag(GENDC.PAR_MC.tag,      Prefix+'.PAR_MC',         2);
   InitTag(GENDC.PAR_MV.tag,      Prefix+'.PAR_MV',         2);
   InitTag(GENDC.PAR_MW.tag,      Prefix+'.PAR_MW',         2);
   InitTag(GENDC.PAR_PC.tag,      Prefix+'.PAR_PC',         2);
   InitTag(GENDC.PAR_PV.tag,      Prefix+'.PAR_PV',         2);
   InitTag(GENDC.PAR_SR.tag,      Prefix+'.PAR_SR',         1);
   InitTag(GENDC.PAR_FR.tag,      Prefix+'.PAR_FR',         1);
   InitTag(GENDC.BT_FLD.tag,      Prefix+'.BT_FLD',         1);
   InitTag(GENDC.BT_AST.tag,      Prefix+'.BT_AST',         1);
   InitTag(GENDC.BT_OUT.tag,      Prefix+'.BT_OUT',         1);
   InitTag(GENDC.PAR_OVP.tag,     Prefix+'.PAR_OVP',        2);
   InitTag(GENDC.PAR_UVL.tag,     Prefix+'.PAR_UVL',        2);
   InitTag(GENDC.PAR_RMT.tag,     Prefix+'.PAR_RMT',        1);
   InitTag(GENDC.PAR_MODE.tag,    Prefix+'.PAR_MODE',       1);
   InitTag(GENDC.PAR_FENA.tag,    Prefix+'.PAR_FENA',       1);
   InitTag(GENDC.PAR_FEVE.tag,    Prefix+'.PAR_FEVE',       1);
   InitTag(GENDC.PAR_SENA.tag,    Prefix+'.PAR_SENA',       1);
   InitTag(GENDC.PAR_SEVE.tag,    Prefix+'.PAR_SEVE',       1);
   InitTag(GENDC.PAR_SRQ.tag,     Prefix+'.PAR_SRQ',        2);
   InitTag(GENDC.RATE_A.tag,      Prefix+'.RATE_A',         2);
   InitTag(GENDC.RATE_V.tag,      Prefix+'.RATE_V',         2);
   InitTag(GENDC.RAMP_A.tag,      Prefix+'.RAMP_A',         1);
   InitTag(GENDC.RAMP_V.tag,      Prefix+'.RAMP_V',         1);
   GENDC.BT_FLD_GUI.tag:=GENDC.BT_FLD.tag; GENDC.BT_FLD_GUI.val:=iAnd(iGetTag(GENDC.BT_FLD.tag),1);
   GENDC.BT_AST_GUI.tag:=GENDC.BT_AST.tag; GENDC.BT_AST_GUI.val:=iAnd(iGetTag(GENDC.BT_AST.tag),1);
   GENDC.BT_OUT_GUI.tag:=GENDC.BT_OUT.tag; GENDC.BT_OUT_GUI.val:=iAnd(iGetTag(GENDC.BT_OUT.tag),1);
   GENDC_FillTags(-MaxReal);
  end;
 end;
 //
 // Update DIM tag in server mode.
 //
 procedure UpdateDimTag(tag:Integer);
 var v:Real;
 begin
  if DIM_IsServerMode then
  if (TypeTag(tag)>0) then begin
   v:=GetStampOfTag(tag,_NaN);
   if not IsNan(v) then GENDC_FillTag(tag,v);
   DIM_UpdateTag(tag,'');
  end;
 end;
 //
 // Check if COM port opened.
 //
 function IsComPortOpened:Boolean;
 begin
  IsComPortOpened:=(ComSpace>=0);
 end;
 //
 // Check command number is valid. 
 //
 function IsValidCmdNum(n:Integer):Boolean;
 begin
  IsValidCmdNum:=(n>=1) and (n<=MaxCmdNum);
 end;
 //
 // Check if command (cmd) is query (?), i.e. should wait number reply.
 //
 function IsQueryCmd(cmd:String):Boolean;
 begin
  IsQueryCmd:=(Pos('?',cmd)>0);
 end;
 function IsQueryCmdNum(n:Integer):Boolean;
 begin
  if IsValidCmdNum(n)
  then IsQueryCmdNum:=IsQueryCmd(GENDC.Cmd.Acronym[n])
  else IsQueryCmdNum:=false;
 end;
 //
 // Detect SRQ !nn message. 
 //
 function IsSRQ(cmd:String):Boolean;
 begin
  IsSRQ:=(StrFetch(cmd,1)='!') and (Length(cmd)>1);
 end;
 //
 // Enable/disable command n.
 //
 procedure EnableCmdNum(n:Integer; Enabled:Boolean);
 begin
  if IsValidCmdNum(n) then GENDC.Cmd.Enabled[n]:=Enabled;
 end;
 //
 // Check if GENDC polling enabled, with warning or not.
 //
 function GENDC_CheckPolling(Warn:Integer):Boolean;
 var flag:Boolean; msg:String;
 begin
  msg:='';
  flag:=iGetTag(GENDC.POLL_ENABLE.tag)>0;
  if (Warn<>0) and not flag then begin
   msg:=StrFmt('Для изменения параметров %s надо включить Опрос.',sGetTag(GENDC.ID_NAME.tag));
   if iAnd(Warn,1)>0 then NiceNotify(msg,15000);
   if iAnd(Warn,2)>0 then UnixWBox('Предупреждение',msg,600,20,15);
   if iAnd(Warn,4)>0 then Cron('@run /sw7 unix messagebox "Предупреждение" "'+msg+'" -t 15');
   if iAnd(Warn,8)>0 then Warning(msg);
  end;
  GENDC_CheckPolling:=flag;
  msg:='';
 end;
 //
 // Clear commands.
 //
 procedure GENDC_Clear_Cmd;
 var i:Integer;
 begin
  for i:=1 to MaxCmdNum do begin
   GENDC.Cmd.Acronym[i]:='';
   GENDC.Cmd.AnsPref[i]:='';
   GENDC.Cmd.DataFmt[i]:='';
   GENDC.Cmd.Comment[i]:='';
   GENDC.Cmd.Enabled[i]:=False;
   GENDC.Cmd.OpData[i]:=_NAN;
   GENDC.Cmd.OpBuff[i]:=_NAN;
   GENDC.Cmd.SimDat[i]:=_NAN;
   GENDC.Cmd.AoNum[i]:=-1;
   GENDC.Cmd.Flags[i]:=0;
   GENDC.Cmd.Tag[i]:=0;
   GENDC.Cmd.Min[i]:=0;
   GENDC.Cmd.Max[i]:=0;
  end;
 end;
 //
 // Initialize commands.
 //
 procedure GENDC_Init_Cmd;
  procedure AddHash(n:Integer; key:String);
  begin
   key:=Trim(key);
   if Length(key)>0 then
   if IsValidCmdNum(n) then
   if GENDC.CmdHash<>0 then
   bNul(HashList_SetLink(GENDC.CmdHash,key,n));
  end;
  procedure Add(n,AoNum:Integer; Info:String);
  begin
   if IsValidCmdNum(n) then begin
    GENDC.Cmd.Acronym[n]:=ExtractWord(1,Info);
    GENDC.Cmd.AnsPref[n]:=StrReplace(ExtractWord(2,Info),Dump('*'),'',3);
    GENDC.Cmd.DataFmt[n]:=StrReplace(ExtractWord(3,Info),Dump('*'),'',3);
    GENDC.Cmd.Comment[n]:=ExtractWord(4,Info);
    GENDC.Cmd.Enabled[n]:=Val(ExtractWord(5,Info))>0;
    GENDC.Cmd.OpData[n]:=rVal(ExtractWord(6,Info));
    GENDC.Cmd.SimDat[n]:=rVal(ExtractWord(7,Info));
    GENDC.Cmd.AoNum[n]:=AoNum;
    GENDC.Cmd.Tag[n]:=FindTag(CrvName(RefAo(AoNum)));
    GENDC.Cmd.Min[n]:=rVal(ExtractWord(8,Info));
    GENDC.Cmd.Max[n]:=rVal(ExtractWord(9,Info));
    AddHash(n,GENDC.Cmd.Acronym[n]);
   end;
  end;
 begin
  GENDC.Cmd.Num:=1;
  //  Command   AoNum        Acron  Ans Fmt   Comment          En OpData SimDat  Min Max
  Add(cm_ADRn,  -1,          'ADR   **  %2.0f Set_Address       1      6      6    0  30');
  Add(cm_CLS,   -1,          'CLS   **  ***** Clear_Status      0      0      0    0   0');
  Add(cm_RST,   -1,          'RST   **  ***** Reset_Device      0      0      0    0   0');
  Add(cm_RMTn,  ao_PAR_RMT,  'RMT   **  %1.0f Set_Remote_Mode   0      1      1    0   2');
  Add(cm_RMTa,  ao_PAR_RMT,  'RMT?  **  ***** Get_Remote_Mode   1      1      1    1   2');
  Add(cm_IDNa,  -1,          'IDN?  **  ***** Get_Ident_Name    1      0      0    0   0');
  Add(cm_REVa,  -1,          'REV?  **  ***** Get_Revision      1      0      0    0   0');
  Add(cm_SNa,   -1,          'SN?   **  ***** Get_Serial_Number 1      0      0    0   0');
  Add(cm_DATEa, -1,          'DATE? **  ***** Get_Date_Of_Test  1      0      0    0   0');
  Add(cm_PVn,   ao_PAR_PV,   'PV    **  %6.2f Set_Program_Volt  0      0      0    0 120');
  Add(cm_PVa,   ao_PAR_PV,   'PV?   PV  ***** Get_Program_Volt  1      0    6.0    0 120');
  Add(cm_MVa,   ao_PAR_MV,   'MV?   MV  ***** Get_Measure_Volt  1      0    5.9    0 120');
  Add(cm_PCn,   ao_PAR_PC,   'PC    **  %6.2f Set_Program_Curr  0      0      0    0 132');
  Add(cm_PCa,   ao_PAR_PC,   'PC?   PC  ***** Get_Program_Curr  1      0    5.0    0 132');
  Add(cm_MCa,   ao_PAR_MC,   'MC?   MC  ***** Get_Measure_Curr  1      0    4.9    0 132');
  Add(cm_DVCa,  -1,          'DVC?  **  ***** Display_Vol_Curr  0      0      0    0   0');
  Add(cm_OUTn,  ao_BT_OUT,   'OUT   **  %1.0f Set_Output_State  0      0      0    0   1');
  Add(cm_OUTa,  ao_BT_OUT,   'OUT?  **  ***** Get_Output_State  1      0      0    0   1');
  Add(cm_FLDn,  ao_BT_FLD,   'FLD   **  %1.0f Set_Foldback_Prot 0      0      0    0   2');
  Add(cm_FLDa,  ao_BT_FLD,   'FLD?  **  ***** Get_Foldback_Prot 1      0      0    0   1');
  Add(cm_OVPn,  ao_PAR_OVP,  'OVP   **  %4.1f Set_OverVolt_Prot 0      0      0    0 132');
  Add(cm_OVPa,  ao_PAR_OVP,  'OVP?  **  ***** Get_OverVolt_Prot 1      0      0    0 132');
  Add(cm_OVM,   -1,          'OVM   **  ***** Set_OVP_To_Max    0      0      0    0   0');
  Add(cm_UVLn,  ao_PAR_UVL,  'UVL   **  %4.1f Set_UnderVolt_Lim 0      0      0    0 120');
  Add(cm_UVLa,  ao_PAR_UVL,  'UVL?  **  ***** Get_UnderVolt_Lim 1      0      0    0 120');
  Add(cm_ASTn,  ao_BT_AST,   'AST   **  %1.0f Set_AutoStartMode 0      0      0    0   1');
  Add(cm_ASTa,  ao_BT_AST,   'AST?  **  ***** Get_AutoStartMode 1      0      0    0   1');
  Add(cm_SAV,   -1,          'SAV   **  ***** Save_Actual_Value 0      0      0    0   0');
  Add(cm_RCL,   -1,          'RCL   **  ***** Recall_saved_vals 0      0      0    0   0');
  Add(cm_MODEa, ao_PAR_MODE, 'MODE? **  ***** Get_OperationMode 1      0      0    0   2');
  Add(cm_STTa,  -1,          'STT?  **  ***** Get_Status_Total  0      0      0    0   0');
  Add(cm_FLTa,  ao_PAR_FR,   'FLT?  FR  ***** Get_Fault_Registr 1      0      0    0 255');
  Add(cm_FENAn, ao_PAR_FENA, 'FENA  **  %2.2X Set_Fault_Enable  0      0      0    0 255');
  Add(cm_FENAa, ao_PAR_FENA, 'FENA? **  ***** Get_Fault_Enable  1      0      0    0 255');
  Add(cm_FEVEa, ao_PAR_FEVE, 'FEVE? **  ***** Get_Fault_Event   1      0      0    0 255');
  Add(cm_STATa, ao_PAR_SR,   'STAT? SR  ***** Get_Status_Regist 1      0      0    0 255');
  Add(cm_SENAn, ao_PAR_SENA, 'SENA  **  %2.2X Set_Status_Enable 0      0      0    0 255');
  Add(cm_SENAa, ao_PAR_SENA, 'SENA? **  ***** Get_Status_Enable 1      0      0    0 255');
  Add(cm_SEVEa, ao_PAR_SEVE, 'SEVE? **  ***** Get_Status_Event  1      0      0    0 255');
  GENDC.Cmd.OpData[cm_ADRn]:=GENDC.Com.Addr;
 end;
 //
 // Print command table
 //
 procedure PrintCmdTable;
 var n:Integer;
 begin
  Success('Command table:');
  Success(StrFmt('%-2s ','Cm')
         +StrFmt('%-2s ','Ao')
         +StrFmt('%-5s ','Acro')
         +StrFmt('%-3s ','Ans')
         +StrFmt('%-5s ','Fmt')
         +StrFmt('%-17s ','Comment')
         +StrFmt('%-2s ','En')
         +StrFmt('%6s ','OpData')
         +StrFmt('%6s ','SimDat')
         +StrFmt('%6s ','Min')
         +StrFmt('%6s ','Max')
         );
  for n:=1 to MaxCmdNum do
  Success(StrFmt('%-2d ',n)
         +StrFmt('%-2d ',GENDC.Cmd.AoNum[n])
         +StrFmt('%-5s ',GENDC.Cmd.Acronym[n])
         +StrFmt('%-3s ',GENDC.Cmd.AnsPref[n])
         +StrFmt('%-5s ',GENDC.Cmd.DataFmt[n])
         +StrFmt('%-17.17s ',GENDC.Cmd.Comment[n])
         +StrFmt('%-2d ',Ord(GENDC.Cmd.Enabled[n]))
         +StrFmt('%6g ',GENDC.Cmd.OpData[n])
         +StrFmt('%6g ',GENDC.Cmd.SimDat[n])
         +StrFmt('%6g ',GENDC.Cmd.Min[n])
         +StrFmt('%6g ',GENDC.Cmd.Max[n])
         );
 end;
 //
 // Update ranges/formats by ModelCode.
 //
 function GENDC_UpdateModel(ModelCode:Integer):Integer;
 begin
  ModelCode:=CheckModelCode(ModelCode);
  if ModelCode>0 then begin
   GENDC.Cmd.DataFmt[cm_PVn]:=GENDC.Tab.VolFmt[ModelCode];
   GENDC.Cmd.Max[cm_PVn]:=GENDC.Tab.VolMax[ModelCode];
   GENDC.Cmd.Max[cm_PVa]:=GENDC.Tab.VolMax[ModelCode];
   GENDC.Cmd.Max[cm_MVa]:=GENDC.Tab.VolMax[ModelCode];
   GENDC.Cmd.DataFmt[cm_PCn]:=GENDC.Tab.CurFmt[ModelCode];
   GENDC.Cmd.Max[cm_PCn]:=GENDC.Tab.CurMax[ModelCode];
   GENDC.Cmd.Max[cm_PCa]:=GENDC.Tab.CurMax[ModelCode];
   GENDC.Cmd.Max[cm_MCa]:=GENDC.Tab.CurMax[ModelCode];
   GENDC.Cmd.DataFmt[cm_OVPn]:=GENDC.Tab.OvpFmt[ModelCode];
   GENDC.Cmd.Max[cm_OVPn]:=GENDC.Tab.OvpMax[ModelCode];
   GENDC.Cmd.Max[cm_OVPa]:=GENDC.Tab.OvpMax[ModelCode];
   GENDC.Cmd.Min[cm_OVPn]:=GENDC.Tab.OvpMin[ModelCode];
   GENDC.Cmd.Min[cm_OVPa]:=GENDC.Tab.OvpMin[ModelCode];
   GENDC.Cmd.DataFmt[cm_UVLn]:=GENDC.Tab.UvlFmt[ModelCode];
   GENDC.Cmd.Max[cm_UVLn]:=GENDC.Tab.UvlMax[ModelCode];
   GENDC.Cmd.Max[cm_UVLa]:=GENDC.Tab.UvlMax[ModelCode];
  end;
  GENDC_UpdateModel:=ModelCode;
 end;
 //
 // Main command cycle.
 //
 procedure GENDC_CMD_CYCLE;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if IsValidCmdNum(Num)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if GENDC.Cmd.Enabled[Num]
    then i:=MaxCmdNum
    else i:=i+1;
   end; 
   NextCmdNum:=Num;
  end;
  procedure ClearRequest(Num,Status:Integer);
  begin
   GENDC.Com.Req:='';
   GENDC.Com.Ans:='';
   GENDC.Com.Buf:='';
   GENDC.Com.ReqTime:=0;
   GENDC.Com.AnsTime:=0;
   GENDC.Com.Status:=Status;
   GENDC.Cmd.Num:=ValidateCmdNum(Num);
  end;
  function ReqCmd(n:Integer):String;
  var s:String; OpData:Real;
  begin
   s:='';
   if IsValidCmdNum(n) then begin
    s:=GENDC.Cmd.Acronym[n];
    if (Length(s)>0) and not IsQueryCmd(s) then begin
     OpData:=GENDC.Cmd.OpData[n];
     if IsNan(OpData) then s:='' else
     s:=s+' '+GENDC_Fmt(GENDC.Cmd.DataFmt[n],OpData);
    end;
    if (Length(s)>0) and GENDC.Com.UseCheckSum then s:=AppendCheckSum(s);
   end;
   ReqCmd:=s;
   s:='';
  end;
  procedure PrepareRequest(n:Integer);
  begin
   if IsValidCmdNum(n) then begin
    GENDC.Com.Req:=ReqCmd(n);
    if (Length(GENDC.Com.Req)>0) then begin
     GENDC.Com.Req:=GENDC.Com.Req+CRLF;
    end;
   end;
  end;
  procedure UpdateIDN(data:String);
  var v:Real;
  begin
   if not IsEmptyStr(data) then begin
    GENDC.ModelCode:=GENDC_UpdateModel(GENDC_ParseIDN(data,iGetTag(GENDC.POLL_ENABLE.tag)=0));
    if (GENDC_ModelCode=0) then data:=si_OFFLINE;
    if IsEmptyStr(data) then data:=si_OFFLINE;
    bNul(sSetTag(GENDC.ID_IDN.tag,data));
    if ShouldRefresh(GENDC.ID_IDN.val,GetStampOfTag(GENDC.ID_IDN.tag,_NaN))>0
    then UpdateDimTag(GENDC.ID_IDN.tag);
   end;
  end;
  procedure UpdateCmdValue(n:Integer; value:Real);
  var aonum,tag:Integer;
  begin
   if IsValidCmdNum(n) and not IsNan(value) and not IsInf(value) then begin
    tag:=GENDC.Cmd.Tag[n]; aonum:=GENDC.Cmd.AoNum[n];
    if TypeTag(tag)=1 then begin
      bNul(iSetTag(tag,Round(value)));
      UpdateDimTag(tag);
    end;
    if TypeTag(tag)=2 then begin
     bNul(rSetTag(tag,value));
     UpdateDimTag(tag);
    end;
    if aonum>=0 then UpdateAo(aonum,time,value);
    GENDC.Cmd.OpData[n]:=value;
   end;
  end;  
  procedure Update_PAR_MW;
  var volt,amper,watt:Real;
  begin
   volt:=GENDC.Cmd.OpData[cm_MVa];
   amper:=GENDC.Cmd.OpData[cm_MCa];
   watt:=volt*amper;
   bNul(rSetTag(GENDC.PAR_MW.tag,watt));
   UpdateDimTag(GENDC.PAR_MW.tag);
   UpdateAo(ao_PAR_MW,time,watt);
  end;
  procedure UpdateSTT(n:Integer; data:String; Hex:Boolean);
  var value:Real;
  begin
   if Length(data)>0 then
   if IsValidCmdNum(n) then begin
    data:=GENDC_ExtractData(data,GENDC.Cmd.AnsPref[n]);
    if Length(data)>0 then begin
     if Hex then value:=rVal('$'+data) else value:=rVal(data);
     if not IsNan(value) and not IsInf(value) then begin
      UpdateCmdValue(n,value);
      if (n=cm_MCa) then Update_PAR_MW; // Update Watts
     end;
    end;
   end;
  end;
  procedure CheckReplyStatus(s:String);
   procedure Fine(msg:String); begin Details(s+' - '+msg); end;
   procedure Fail(msg:String); begin Trouble(s+' - '+msg); end;
  begin
   if IsSameText(s,'OK')  then Fine('Success.')              else
   if IsSameText(s,'E01') then Fail('PV over range.')        else
   if IsSameText(s,'E02') then Fail('PV under range.')       else
   if IsSameText(s,'E04') then Fail('OVP range fail.')       else
   if IsSameText(s,'E06') then Fail('UVL range fail.')       else
   if IsSameText(s,'E07') then Fail('OUT ON failed.')        else
   if IsSameText(s,'C01') then Fail('Illegal command.')      else
   if IsSameText(s,'C02') then Fail('Missing parameter.')    else
   if IsSameText(s,'C03') then Fail('Illegal parameter.')    else
   if IsSameText(s,'C04') then Fail('Checksum error.')       else
   if IsSameText(s,'C05') then Fail('Setting out of range.') else
   Fail('Unknown reply.');
  end;
  procedure DecodeData(n:Integer; data:String);
  var v:Real; i:Integer;
  begin
   if IsValidCmdNum(n) then begin
    if HasCheckSum(data) then begin
     if HasValidCheckSum(data) then begin
      data:=DropCheckSum(data);
     end else begin
      Trouble('Checksum error - '+data);
      data:='';
     end;
    end;
    if Length(data)>0 then begin
     if (n=cm_ADRn) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_CLS) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_RST) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_RMTn) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_RMTa) then begin
      i:=WordIndex(data,'LOC REM LLO')-1;
      if (i>=0) then v:=i else v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_IDNa) then begin
      Success('Received IDN  = '+data);
      UpdateIDN(data);
      if GENDC_ModelCode>0 then EnableCmdNum(n,false);
     end else
     if (n=cm_REVa) then begin
      Success('Received REV  = '+data);
      bNul(sSetTag(GENDC.ID_REV.tag,data));
      UpdateDimTag(GENDC.ID_REV.tag);
      EnableCmdNum(n,false);
     end else
     if (n=cm_SNa) then begin
      Success('Received SN   = '+data);
      bNul(sSetTag(GENDC.ID_SN.tag,data));
      UpdateDimTag(GENDC.ID_SN.tag);
      EnableCmdNum(n,false);
     end else
     if (n=cm_DATEa) then begin
      Success('Received DATE = '+data);
      bNul(sSetTag(GENDC.ID_DATE.tag,data));
      UpdateDimTag(GENDC.ID_DATE.tag);
      EnableCmdNum(n,false);
     end else
     if (n=cm_PVn) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_PVa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_MVa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_PCn) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_PCa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_MCa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
      if not IsNan(v) then Update_PAR_MW; // Update Watts
     end else
     if (n=cm_DVCa) then begin
      v:=rVal(ExtractWord(1,data));
      if not IsNan(v) then UpdateCmdValue(cm_MVa,v);
      v:=rVal(ExtractWord(2,data));
      if not IsNan(v) then UpdateCmdValue(cm_PVa,v);
      v:=rVal(ExtractWord(3,data));
      if not IsNan(v) then UpdateCmdValue(cm_MCa,v);
      if not IsNan(v) then Update_PAR_MW; // Update Watts
      v:=rVal(ExtractWord(4,data));
      if not IsNan(v) then UpdateCmdValue(cm_PCa,v);
      v:=rVal(ExtractWord(5,data));
      if not IsNan(v) then UpdateCmdValue(cm_OVPa,v);
      v:=rVal(ExtractWord(6,data));
      if not IsNan(v) then UpdateCmdValue(cm_UVLa,v);
     end else
     if (n=cm_OUTn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_OUTa) then begin
      i:=WordIndex(data,'OFF ON')-1;
      if (i>=0) then v:=i else v:=rVal(data);
      if not IsNan(v) then v:=iSetBitState(iGetTag(GENDC.BT_OUT.tag),1,v<>0);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_FLDn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_FLDa) then begin
      i:=WordIndex(data,'OFF ON')-1;
      if (i>=0) then v:=i else v:=rVal(data);
      if not IsNan(v) then v:=iSetBitState(iGetTag(GENDC.BT_FLD.tag),1,v<>0);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_OVPn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_OVPa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_OVM) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_UVLn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_UVLa) then begin
      v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_ASTn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_ASTa) then begin
      i:=WordIndex(data,'OFF ON')-1;
      if (i>=0) then v:=i else v:=rVal(data);
      if not IsNan(v) then v:=iSetBitState(iGetTag(GENDC.BT_AST.tag),1,v<>0);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_SAV) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_RCL) then begin
      CheckReplyStatus(data);
     end else
     if (n=cm_MODEa) then begin
      i:=WordIndex(data,'OFF CV CC')-1;
      if (i>=0) then v:=i else v:=rVal(data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_STTa) then begin
      UpdateSTT(cm_MVa,data,false);
      UpdateSTT(cm_PVa,data,false);
      UpdateSTT(cm_MCa,data,false);
      UpdateSTT(cm_PCa,data,false);
      UpdateSTT(cm_STATa,data,true);
      UpdateSTT(cm_FLTa,data,true);
     end else
     if (n=cm_FLTa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_FENAn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_FENAa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_FEVEa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_STATa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_SENAn) then begin
      CheckReplyStatus(data);
      EnableCmdNum(n,false);
     end else
     if (n=cm_SENAa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end else
     if (n=cm_SEVEa) then begin
      v:=rVal('$'+data);
      if not IsNan(v) then UpdateCmdValue(n,v);
     end;
    end;
   end;
  end;
  function HandleRequest(n:Integer; Ans:String):Boolean;
  var Handled:Boolean;
  begin
   Handled:=false;
   if IsValidCmdNum(n) and (Length(Ans)>0) then begin
    ViewImp('COM < '+Ans);
    DecodeData(n,Ans);
    Handled:=True;
   end;
   HandleRequest:=Handled;
  end;
  procedure PrintDetails;
  begin
   if iAnd(DebugFlags,dfDetails)>0 then Details('Run '+Str(RunCount)
    +', Cmd '+GENDC.Cmd.Acronym[GENDC.Cmd.Num]
    +', Enb '+Str(Ord(GENDC.Cmd.Enabled[GENDC.Cmd.Num]))
    +', Sta '+Str(GENDC.Com.Status)
    +', Req '+Trim(GENDC.Com.Req)
    +', Ans '+Trim(GENDC.Com.Ans)
    +', Buf '+Trim(GENDC.Com.Buf)
    );
  end;
  procedure RampingHandler;
  var v:Real;
  begin
   if mSecNow>GENDC.RampingTime+GENDC.Com.PollPeriod then begin
    if RampingGoing(GENDC.RampingA) then begin
     v:=RampingValue(GENDC.RampingA);
     DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[cm_PCn]+' '+Str(v));
     if RampingComplete(GENDC.RampingA) then RampingClear(GENDC.RampingA);
    end;
    if RampingGoing(GENDC.RampingV) then begin
     v:=RampingValue(GENDC.RampingV);
     DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[cm_PVn]+' '+Str(v));
     if RampingComplete(GENDC.RampingV) then RampingClear(GENDC.RampingV);
    end;
    bNul(iSetTag(GENDC.RAMP_A.tag,Ord(RampingGoing(GENDC.RampingA))));
    bNul(iSetTag(GENDC.RAMP_V.tag,Ord(RampingGoing(GENDC.RampingV))));
    GENDC.RampingTime:=mSecNow;
   end;
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsNan(GENDC.Cmd.OpBuff[n]) then begin
    if not IsQueryCmdNum(n) then begin
     GENDC.Cmd.OpData[n]:=GENDC.Cmd.OpBuff[n];
     EnableCmdNum(n,true);
    end;
    GENDC.Cmd.OpBuff[n]:=_NAN;
   end;
  end;
  procedure EnableIDN;
  begin
   EnableCmdNum(cm_ADRn,true);
   EnableCmdNum(cm_IDNa,true);
   EnableCmdNum(cm_REVa,true);
   EnableCmdNum(cm_SNa,true);
   EnableCmdNum(cm_DATEa,true);
  end;
  procedure ExecuteIdleActions;
  begin
   UpdateOpData;
   RampingHandler;
   if (GENDC_ModelCode=0) then EnableIDN;
  end;
  procedure DimRefreshTag(var R:TTagRef);
  var tag:Integer; v:Real;
  begin
   tag:=R.tag; v:=GetStampOfTag(tag,_NaN);
   if not IsNaN(v) then if ShouldRefresh(R.val,v)>0 then UpdateDimTag(tag);
  end;
 begin
  //
  // COM port communication handler
  //
  if not DIM_IsClientMode then begin
   if IsComPortOpened then begin
    if iGetTag(GENDC.POLL_ENABLE.tag)>0 then begin
     GENDC.Cmd.Num:=ValidateCmdNum(GENDC.Cmd.Num);
     if iAnd(DebugFlags,dfDetails)>0 then PrintDetails;
     if GENDC.Com.Status=st_NoReq then begin
      ExecuteIdleActions;
      ClearRequest(GENDC.Cmd.Num,st_NoReq);
      if mSecNow>=GENDC.Com.LastPoll+GENDC.Com.PollPeriod then begin
       if GENDC.Cmd.Enabled[GENDC.Cmd.Num] then PrepareRequest(GENDC.Cmd.Num);
       if Length(GENDC.Com.Req)>0 then begin
        PurgeComPort;
        if ComWrite(GENDC.Com.Req) then begin
         ViewExp('COM > '+Trim(GENDC.Com.Req));
         GENDC.Com.Status:=st_WaitAns;
         GENDC.Com.ReqTime:=mSecNow;
        end else begin
         Trouble('Could not send request '+Trim(GENDC.Com.Req));
         ClearRequest(NextCmdNum(GENDC.Cmd.Num),st_NoReq);
        end;
        if GENDC.Com.Status=st_WaitAns
        then GENDC.Com.LastPoll:=mSecNow;
       end else ClearRequest(NextCmdNum(GENDC.Cmd.Num),st_NoReq);
      end;
     end else
     if GENDC.Com.Status=st_WaitAns then begin
      if Com_Readln(GENDC.Com.Ans,GENDC.Com.Buf) then begin
       if IsSRQ(GENDC.Com.Ans) then begin
        // Skip SRQ messages
        Success('SRQ detected.');
        bNul(rSetTag(GENDC.PAR_SRQ.tag,rGetTag(GENDC.PAR_SRQ.tag)+1));
        UpdateAo(ao_PAR_SRQ,time,rGetTag(GENDC.PAR_SRQ.tag));
        DimRefreshTag(GENDC.PAR_SRQ);
       end else begin // Handle normal answer
        bNul(HandleRequest(GENDC.Cmd.Num,Trim(GENDC.Com.Ans)));
        GENDC.Com.PollRate:=GENDC.Com.PollRate+1;
        GENDC.Com.Status:=st_WaitGap;
        GENDC.Com.AnsTime:=mSecNow;
       end;
      end else
      if mSecNow>GENDC.Com.ReqTime+GENDC.Com.TimeOut then begin
       GENDC.Com.Status:=st_TimeOut;
      end;
     end;
     if GENDC.Com.Status=st_WaitGap then begin
      if mSecNow>=GENDC.Com.AnsTime+GENDC.Com.TimeGap then begin
       if not IsQueryCmdNum(GENDC.Cmd.Num) then EnableCmdNum(GENDC.Cmd.Num,false);
       ClearRequest(NextCmdNum(GENDC.Cmd.Num),st_NoReq);
      end;
     end;
     if GENDC.Com.Status=st_TimeOut then begin
      if GENDC.Cmd.Num=cm_IDNa then UpdateIDN(si_OFFLINE);
      Failure(GENDC.ecTimeOut,'TimeOut on command '+GENDC.Cmd.Acronym[GENDC.Cmd.Num]+' request '+Trim(GENDC.Com.Req));
      ClearRequest(NextCmdNum(GENDC.Cmd.Num),st_NoReq);
      EnableIDN;
     end;
    end else begin
     ClearRequest(NextCmdNum(0),st_NoReq);
     UpdateIDN(si_OFFLINE);
     EnableIDN;
    end;
    if SysTimer_Pulse(1000)>0 then begin
     UpdateAo(ao_PAR_SRQ,time,rGetTag(GENDC.PAR_SRQ.tag));
    end;
   end;
  end;
  //
  // Update DIM services
  //
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then GENDC_FillTags(-MaxReal);
   DimRefreshTag(GENDC.POLL_ENABLE);
   DimRefreshTag(GENDC.POLL_RATE);
   DimRefreshTag(GENDC.ERROR_CNT);
   DimRefreshTag(GENDC.ID_IDN);
   DimRefreshTag(GENDC.ID_REV);
   DimRefreshTag(GENDC.ID_SN);
   DimRefreshTag(GENDC.ID_DATE);
   DimRefreshTag(GENDC.PAR_MC);
   DimRefreshTag(GENDC.PAR_MV);
   DimRefreshTag(GENDC.PAR_MW);
   DimRefreshTag(GENDC.PAR_PC);
   DimRefreshTag(GENDC.PAR_PV);
   DimRefreshTag(GENDC.PAR_OVP);
   DimRefreshTag(GENDC.PAR_UVL);
   DimRefreshTag(GENDC.PAR_SR);
   DimRefreshTag(GENDC.PAR_FR);
   DimRefreshTag(GENDC.BT_FLD);
   DimRefreshTag(GENDC.BT_AST);
   DimRefreshTag(GENDC.BT_OUT);
   DimRefreshTag(GENDC.PAR_RMT);
   DimRefreshTag(GENDC.PAR_MODE);
   DimRefreshTag(GENDC.PAR_FENA);
   DimRefreshTag(GENDC.PAR_FEVE);
   DimRefreshTag(GENDC.PAR_SENA);
   DimRefreshTag(GENDC.PAR_SEVE);
   DimRefreshTag(GENDC.PAR_SRQ);
   DimRefreshTag(GENDC.RATE_A);
   DimRefreshTag(GENDC.RATE_V);
   DimRefreshTag(GENDC.RAMP_A);
   DimRefreshTag(GENDC.RAMP_V);
  end;
 end;
 //
 // Xor bit on click (local version).
 //
 procedure ClickTagXorLocal(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // Xor bit on click (remote version).
 //
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DIM_GuiClickSend(DIM_GuiClickBuff+'NewValue='+Str(iXor(iGetTag(tag),XorMask)));
   bNul(Voice(snd_Click));
  end;
 end;
 //
 // GUI Handler to process user input...
 //
 procedure GENDC_GUI_POLL;
 var ClickCurve:Integer; s,slb:String;
  procedure ClickEditGendcCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if ClickTag=GENDC.Cmd.Tag[n] then begin
    DevPostCmdLocal('@Edit '+GENDC.Cmd.Acronym[n]+' '+mime_encode(slb));
    bNul(Voice(snd_Click));
   end;
  end;
  procedure DevSendGendcCmd(n:Integer; prefix,data:String; min,max:Real);
  var v:Real;
  begin
   if IsValidCmdNum(n) then
   if not IsEmptyStr(data) then
   if TypeTag(GENDC.Cmd.Tag[n])=1 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmdLocal(prefix+GENDC.Cmd.Acronym[n]+' '+Str(Round(v)));
   end else
   if TypeTag(GENDC.Cmd.Tag[n])=2 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevPostCmdLocal(prefix+GENDC.Cmd.Acronym[n]+' '+Str(v));
   end else
   if TypeTag(GENDC.Cmd.Tag[n])=3 then begin
    DevPostCmdLocal(prefix+GENDC.Cmd.Acronym[n]+' '+Trim(data));
   end;
  end;
  procedure CheckEditGendcCmd(n:Integer; prefix:String);
  var s:String;
  begin
   s:='';
   if IsValidCmdNum(n) then
   if CheckEditTag(GENDC.Cmd.Tag[n],s) then begin
    if ((n=cm_PVn) and (rGetTag(GENDC.RATE_V.tag)>0))
    or ((n=cm_PCn) and (rGetTag(GENDC.RATE_A.tag)>0))
    then DevSendGendcCmd(n,prefix+'@Ramping ',s,GENDC.Cmd.Min[n],GENDC.Cmd.Max[n])
    else DevSendGendcCmd(n,prefix+'@Setting ',s,GENDC.Cmd.Min[n],GENDC.Cmd.Max[n]);
    EditReset;
   end;
   s:='';
  end;
  procedure CheckEditGendcSel(n:Integer; prefix:String);
  var i,j,tag:Integer;
  begin
   if IsValidCmdNum(n) then begin
    tag:=GENDC.Cmd.Tag[n];
    if TypeTag(tag)>0 then
    if EditTestResultName(EditGetUID('SEL_'+NameTag(tag))) then begin
     if EditTestResultCode(mr_OK) then begin
      j:=0;
      for i:=1 to EditGetAnswerCount-1 do
      j:=iSetBit(j,Val(EditGetAnswer(i)),1);
      DevPostCmdLocal(prefix+'@Setting '+GENDC.Cmd.Acronym[n]+' '+Str(j));
     end;
     EditReset;
    end;
   end;
  end;
  procedure RefreshSetting(n:Integer; var S:TTagRef);
  begin
   if IsValidCmdNum(n) then begin
    if TypeTag(S.tag)=1 then
    if ShouldRefresh(S.val,iGetTag(S.tag))>0 then
    DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[n]+' '+Str(iGetTag(S.tag)));
    if TypeTag(S.tag)=2 then
    if ShouldRefresh(S.val,rGetTag(S.tag))>0 then
    DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[n]+' '+Str(rGetTag(S.tag)));
   end;
  end;
  procedure RefreshBtn(n:Integer; var S:TTagRef);
  begin
   if IsValidCmdNum(n) then begin
    if TypeTag(S.tag)=1 then
    if ShouldRefresh(S.val,iAnd(iGetTag(S.tag),1))>0 then
    DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[n]+' '+Str(iAnd(iGetTag(S.tag),1)));
   end;
  end;
 begin
  s:=''; slb:='';
  DIM_GuiClickBuff:='';
  //
  // Handle commands...
  //
  if DIM_IsServerMode then begin
   RefreshBtn(cm_FLDn,GENDC.BT_FLD_GUI);
   RefreshBtn(cm_ASTn,GENDC.BT_AST_GUI);
   RefreshBtn(cm_OUTn,GENDC.BT_OUT_GUI);
  end;
  //
  // Handle user mouse/keyboard clicks...
  // ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,KeyUp)
  // ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  //
  if ClickWhat<>0 then
  repeat
   //
   // Copy GUI click to DIM buffer for remote execution.
   //
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   //
   // Handle MouseDown/KeyDown
   //
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    //
    // Handle Left mouse button click
    //
    if (ClickButton=VK_LBUTTON) then begin
     //
     // Handle local clicks
     //
     if ClickIsLocal then begin
      // Sensor left bottom settings for editing
      slb:= SetFormUnderSensorLeftBottom(ClickParams(''));
      //
      // Handle sensor clicks...
      //
      if IsSameText(ClickSensor,'HELP') then begin
       Cron('@Browse '+DaqFileRef(AdaptFileName(ReadIni('[DAQ] HelpFile')),'.htm'));
       bNul(Voice(snd_Click));
      end;
      if (ClickTag<>0) then begin
       if (ClickTag=GENDC.ID_IDN.tag) then begin
        DevPostCmdLocal('@Remote @Setting IDN');
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=DIM_GuiClickFallBackModeTag) then begin
        if DIM_IsServerMode
        then ClickTagXorLocal(DIM_GuiClickFallBackModeTag,1)
        else ShowTooltip('text "Режим DIM FallBack применим только к серверу."  preset stdNotify delay 15000');
       end;
       ClickTagXorRemote(GENDC.POLL_ENABLE.tag,1);
       if (ClickTag=GENDC.RATE_A.tag) then begin
        StartEditTagEx(GENDC.RATE_A.tag,'Темп роста Тока, Ампер/сек',slb);
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=GENDC.RATE_V.tag) then begin
        StartEditTagEx(GENDC.RATE_V.tag,'Темп роста Напряжения, Вольт/сек',slb);
        bNul(Voice(snd_Click));
       end;
       if (ClickTag=GENDC.BT_FLD.tag)
       or (ClickTag=GENDC.BT_AST.tag)
       or (ClickTag=GENDC.BT_OUT.tag)
       or (ClickTag=GENDC.Cmd.Tag[cm_PCn])
       or (ClickTag=GENDC.Cmd.Tag[cm_PVn])
       or (ClickTag=GENDC.Cmd.Tag[cm_OVPn])
       or (ClickTag=GENDC.Cmd.Tag[cm_UVLn])
       or (ClickTag=GENDC.Cmd.Tag[cm_FENAn])
       or (ClickTag=GENDC.Cmd.Tag[cm_SENAn])
       then begin
        if GENDC_CheckPolling(1) then begin
         ClickTagXorRemote(GENDC.BT_FLD.tag,1);
         ClickTagXorRemote(GENDC.BT_AST.tag,1);
         ClickTagXorRemote(GENDC.BT_OUT.tag,1);
         ClickEditGendcCmd(cm_PCn);
         ClickEditGendcCmd(cm_PVn);
         ClickEditGendcCmd(cm_OVPn);
         ClickEditGendcCmd(cm_UVLn);
         ClickEditGendcCmd(cm_FENAn);
         ClickEditGendcCmd(cm_SENAn);
        end;
       end;
      end;
      //
      // Select Plot & Tab windows by curve...
      //
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      //
      // Console commands: @url_encoded_sensor ...
      //
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     //
     // Handle remote clicks comes from DIM via @DimGuiClick message.
     // @DimGuiClick default handler decode and write events to FIFO,
     // so we can find it as clicks and can handle it in usual way.
     //
     if ClickIsRemote then begin
      //
      // Show time difference.
      //
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      //
      // Handle remote console commands...
      //
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevPostCmdLocal(s);
      //
      // Handle remote sensor clicks...
      //
      if TypeTag(ClickTag)>0 then begin
       s:=ClickParams('NewValue');
       if Length(s)>0 then begin
        if ClickTag=GENDC.POLL_ENABLE.tag then UpdateTag(ClickTag,s,0,1);
        if ClickTag=GENDC.BT_FLD.tag then UpdateTag(ClickTag,s,0,3);
        if ClickTag=GENDC.BT_AST.tag then UpdateTag(ClickTag,s,0,3);
        if ClickTag=GENDC.BT_OUT.tag then UpdateTag(ClickTag,s,0,3);
       end;
      end;
     end;
    end;
   end;
  until (ClickRead=0);
  //
  // Edit handling...
  //
  if EditStateDone then begin
   if CheckEditTag(GENDC.RATE_A.tag,s) then DevPostCmdLocal('@Remote @Setting RATE_A '+s);
   if CheckEditTag(GENDC.RATE_V.tag,s) then DevPostCmdLocal('@Remote @Setting RATE_V '+s);
   CheckEditGendcCmd(cm_PCn,'@Remote ');
   CheckEditGendcCmd(cm_PVn,'@Remote ');
   CheckEditGendcCmd(cm_OVPn,'@Remote ');
   CheckEditGendcCmd(cm_UVLn,'@Remote ');
   CheckEditGendcSel(cm_FENAn,'@Remote ');
   CheckEditGendcSel(cm_SENAn,'@Remote ');
   //
   // Menu TOOLS.
   //
   MenuToolsHandler;
   //
   // Warning,Information.
   //
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  DIM_GuiClickBuff:='';
  s:=''; slb:='';
 end;
 //
 // Handle data requests in simulation mode. See GEN manual.
 //
 procedure GENDC_Sim(Req:String);
 var cmd,arg:String; v,u,rndv,rnda,rm,pv,pc,mv,mc,op,up,sr,fr,ot,as,fd,ohm,nlp:Real;
  procedure CopyLocal(Local:Boolean);
  begin
   if Local then begin
    pv:=GENDC.Cmd.SimDat[cm_PVa];   pc:=GENDC.Cmd.SimDat[cm_PCa];
    mv:=GENDC.Cmd.SimDat[cm_MVa];   mc:=GENDC.Cmd.SimDat[cm_MCa];
    op:=GENDC.Cmd.SimDat[cm_OVPa];  up:=GENDC.Cmd.SimDat[cm_UVLa];
    sr:=GENDC.Cmd.SimDat[cm_STATa]; fr:=GENDC.Cmd.SimDat[cm_FLTa];
    rm:=GENDC.Cmd.SimDat[cm_RMTa];  ot:=GENDC.Cmd.SimDat[cm_OUTa];
    fd:=GENDC.Cmd.SimDat[cm_FLDa];  as:=GENDC.Cmd.SimDat[cm_ASTa];
    ohm:=GENDC.SimOhm; nlp:=GENDC.SimNLP;
   end else begin
    GENDC.Cmd.SimDat[cm_PVa]:=pv;   GENDC.Cmd.SimDat[cm_PCa]:=pc;
    GENDC.Cmd.SimDat[cm_MVa]:=mv;   GENDC.Cmd.SimDat[cm_MCa]:=mc;
    GENDC.Cmd.SimDat[cm_OVPa]:=op;  GENDC.Cmd.SimDat[cm_UVLa]:=up;
    GENDC.Cmd.SimDat[cm_STATa]:=sr; GENDC.Cmd.SimDat[cm_FLTa]:=fr;
    GENDC.Cmd.SimDat[cm_RMTa]:=rm;  GENDC.Cmd.SimDat[cm_OUTa]:=ot;
    GENDC.Cmd.SimDat[cm_FLDa]:=fd;  GENDC.Cmd.SimDat[cm_ASTa]:=as;
   end;
  end;
 begin
  cmd:=''; arg:='';
  if Length(Req)>0 then begin
   ViewImp('COM < '+Trim(Req));
   if Length(Req)>0 then begin
    CopyLocal(true);
    cmd:=ExtractWord(1,DropCheckSum(Req));
    arg:=ExtractWord(2,DropCheckSum(Req));
    v:=rVal(arg);
    rndv:=nlp/100*Random(-1,1)*pv;
    rnda:=nlp/100*Random(-1,1)*pc;
    if (ohm>0) then begin
     mv:=pv; mc:=pc*ot;
     if mv/ohm*ot<pc then begin
      sr:=iSetBit(Round(sr),0,Ord(ot>0));
      sr:=iSetBit(Round(sr),1,0);
      mc:=mv/ohm*ot;
     end else begin
      sr:=iSetBit(Round(sr),0,0);
      sr:=iSetBit(Round(sr),1,Ord(ot>0));
      mv:=pc*ohm;
     end;
     if (fd>0) and (mc>=pc) then begin
      fr:=iSetBit(Round(fr),3,1);
     end;
     if (fr>0) then begin
      sr:=iSetBit(Round(sr),3,1);
     end;
     sr:=iSetBit(Round(sr),2,Ord(fr=0));
     sr:=iSetBit(Round(sr),4,Ord(as<>0));
     sr:=iSetBit(Round(sr),5,Ord(fd<>0));
     sr:=iSetBit(Round(sr),7,Ord(rm=0));
    end;
    if IsSameText('ADR',cmd) then begin
     if not IsNan(v) then GENDC.Cmd.SimDat[cm_ADRn]:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('CLS',cmd) then begin
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('RST',cmd) then begin
     GENDC.Com.Ans:='OK';
     sr:=0;
     fr:=0;
    end;
    if IsSameText('RMT',cmd) then begin
     if not IsNan(v) then rm:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('RMT?',cmd) then begin
     GENDC.Com.Ans:=ExtractWord(Round(rm)+1,'LOC REM LLO');
    end;
    if IsSameText('IDN?',cmd) then begin
     if IsEmptyStr(GENDC.ModelName) then GENDC.ModelName:='LAMBDA,GEN6-100';
     GENDC.Com.Ans:=GENDC.ModelName+' SIM';
    end;
    if IsSameText('REV?',cmd) then begin
     GENDC.Com.Ans:='1U1K:5.1.3';
    end;
    if IsSameText('SN?',cmd) then begin
     GENDC.Com.Ans:='011B431-0003';
    end;
    if IsSameText('DATE?',cmd) then begin
     GENDC.Com.Ans:='2019/06/25';
    end;
    if IsSameText('PV',cmd) then begin
     if not IsNan(v) then pv:=v;
     if not IsNan(v) then mv:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('PV?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',pv);
    end;
    if IsSameText('MV?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',max(0,mv+rndv));
    end;    
    if IsSameText('PC',cmd) then begin
     if not IsNan(v) then pc:=v;
     if not IsNan(v) then mc:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('PC?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',pc);
    end;
    if IsSameText('MC?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',max(0,mc+rnda));
    end;
    if IsSameText('DVC?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',max(0,mv+rndv))+','
                   +GENDC_Fmt('%6.2f',pv)+','
                   +GENDC_Fmt('%6.2f',max(0,mc+rnda))+','
                   +GENDC_Fmt('%6.2f',pc)+','
                   +GENDC_Fmt('%6.2f',op)+','
                   +GENDC_Fmt('%6.2f',up);
    end;
    if IsSameText('OUT',cmd) then begin
     u:=WordIndex(arg,'OFF ON')-1; if u>=0 then v:=u;
     if not IsNan(v) then ot:=Ord(v<>0);
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('OUT?',cmd) then begin
     GENDC.Com.Ans:=ExtractWord(Round(ot)+1,'OFF ON');
    end;
    if IsSameText('FLD',cmd) then begin
     u:=WordIndex(arg,'OFF ON')-1; if u>=0 then v:=u;
     if not IsNan(v) then fd:=Ord(v<>0);
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('FLD?',cmd) then begin
     GENDC.Com.Ans:=ExtractWord(Round(fd)+1,'OFF ON');
    end;
    if IsSameText('AST',cmd) then begin
     u:=WordIndex(arg,'OFF ON')-1; if u>=0 then v:=u;
     if not IsNan(v) then as:=Ord(v<>0);
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('AST?',cmd) then begin
     GENDC.Com.Ans:=ExtractWord(Round(as)+1,'OFF ON');
    end;
    if IsSameText('SAV',cmd) then begin
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('RCL',cmd) then begin
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('MODE?',cmd) then begin
     GENDC.Com.Ans:=ExtractWord(1+iAnd(Round(sr),3)*Ord(ot>0),'OFF CV CC ERR');
    end;
    if IsSameText('SENA',cmd) then begin
     v:=rVal('$'+arg);
     if not IsNan(v) then GENDC.Cmd.SimDat[cm_SENAa]:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('SENA?',cmd) then begin
     GENDC.Com.Ans:=HexB(Round(GENDC.Cmd.SimDat[cm_SENAa]));
    end;
    if IsSameText('SEVE?',cmd) then begin
     GENDC.Com.Ans:=HexB(Round(GENDC.Cmd.SimDat[cm_SEVEa]));
    end;
    if IsSameText('FENA',cmd) then begin
     v:=rVal('$'+arg);
     if not IsNan(v) then GENDC.Cmd.SimDat[cm_FENAa]:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('FENA?',cmd) then begin
     GENDC.Com.Ans:=HexB(Round(GENDC.Cmd.SimDat[cm_FENAa]));
    end;
    if IsSameText('FEVE?',cmd) then begin
     GENDC.Com.Ans:=HexB(Round(GENDC.Cmd.SimDat[cm_FEVEa]));
    end;
    if IsSameText('OVP',cmd) then begin
     if not IsNan(v) then op:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('OVP?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',op);
    end;
    if IsSameText('OVM',cmd) then begin
     op:=GENDC.Cmd.Max[cm_OVPn];
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('UVL',cmd) then begin
     if not IsNan(v) then up:=v;
     GENDC.Com.Ans:='OK';
    end;
    if IsSameText('UVL?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%6.2f',up);
    end;
    if IsSameText('STAT?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%2.2X',sr);
    end;
    if IsSameText('FLT?',cmd) then begin
     GENDC.Com.Ans:=GENDC_Fmt('%2.2X',fr);
    end;
    if IsSameText('STT?',cmd) then begin
     GENDC.Com.Ans:='MV('+GENDC_Fmt('%6.2f',max(0,mv+rndv))+'),'
                   +'PV('+GENDC_Fmt('%6.2f',pv)+'),'
                   +'MC('+GENDC_Fmt('%6.2f',max(0,mc+rnda))+'),'
                   +'PC('+GENDC_Fmt('%6.2f',pc)+'),'
                   +'SR('+GENDC_Fmt('%2.2X',sr)+'),'
                   +'FR('+GENDC_Fmt('%2.2X',fr)+')';
    end;
    CopyLocal(false);
    if Length(GENDC.Com.Ans)>0 then begin
     if HasCheckSum(GENDC.Com.Req)
     then GENDC.Com.Ans:=AppendCheckSum(GENDC.Com.Ans);
     if ComWrite(GENDC.Com.Ans+CRLF)
     then ViewExp('COM > '+Trim(GENDC.Com.Ans))
     else Trouble('Could not send '+Trim(GENDC.Com.Ans));
    end;
   end;
  end;
  GENDC.Com.Req:='';
  GENDC.Com.Ans:='';
  cmd:=''; arg:='';
 end;
 //
 // Self Test procedures.
 //
 procedure GENDC_SelfTest;
 begin
  Writeln(GENDC_ExtractData('MV(5.010),PV(5.010),MC(00.00),PC(24.31),SR(04),FR(00)','PV'));
 end;
 //
 // GENDC cleanup.
 //
 procedure GENDC_Clear;
 begin
  GENDC_Clear_Cmd;
  GENDC_Tab_Clear;
  GENDC.CmdHash:=0;
  GENDC.Com.Req:='';
  GENDC.Com.Ans:='';
  GENDC.Com.Buf:='';
  GENDC.ModelCode:=0;
  GENDC.ModelName:='';
  GENDC.RampingTime:=0;
  RampingClear(GENDC.RampingA);
  RampingClear(GENDC.RampingV);
  GENDC.SimOhm:=0;
  GENDC.SimNLP:=0;
 end;
 //
 // GENDC initialization.
 //
 procedure GENDC_Init;
 var i:Integer;
 begin
  //
  // Register error codes
  //
  GENDC.ecTimeOut:=RegisterErr(progname+': TimeOut');
  //
  // Initialize variables
  //
  MenuToolsSelected:=-1;
  GENDC.Com.ReqTime:=0;
  GENDC.Com.AnsTime:=0;
  GENDC.Com.LastPoll:=0;
  GENDC.Com.Status:=st_NoReq;
  GENDC.Com.Interface:=RS_232;
  GENDC.Com.UseCheckSum:=false;
  //
  // Initialize command table.
  //
  GENDC.CmdHash:=HashList_Init(0);
  GENDC_Init_Cmd;
  GENDC_Tab_Init;
  //
  // Read ini file variables
  //
  GENDC.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  if GENDC.Simulator
  then Success('Run simulator mode.')
  else Success('Run as driver mode.');
  GENDC.Com.Port:=iValDef(ReadIni('ComPort'),0);
  Success('ComPort = '+Str(GENDC.Com.Port));
  GENDC.Com.Addr:=iValDef(ReadIni('ComAddr'),1);
  Success('ComAddr = '+Str(GENDC.Com.Addr));
  GENDC.Com.TimeOut:=iValDef(ReadIni('ComTimeOut'),0);
  Success('ComTimeOut = '+Str(GENDC.Com.TimeOut));
  GENDC.Com.TimeGap:=iValDef(ReadIni('ComTimeGap'),0);
  Success('ComTimeGap = '+Str(GENDC.Com.TimeGap));
  GENDC.Com.PollPeriod:=iValDef(ReadIni('PollPeriod'),100);
  Success('PollPeriod = '+Str(GENDC.Com.PollPeriod));
  GENDC.ModelName:=ReadIniVar('ModelName',28);
  Success('ModelName = '+GENDC.ModelName);
  GENDC.ModelCode:=GENDC_UpdateModel(GENDC_ParseIDN(GENDC.ModelName,false));
  GENDC.Cmd.OpData[cm_ADRn]:=GENDC.Com.Addr;
  GENDC.Com.Interface:=iValDef(GENDC_ExtractData(ReadIni('ComInterface'),'RS'),RS_232);
  Success('ComInterface = RS'+Str(GENDC.Com.Interface));
  GENDC.Com.UseCheckSum:=(iValDef(ReadIni('ComUseCheckSum'),1)<>0);
  Success('ComUseCheckSum = '+Str(Ord(GENDC.Com.UseCheckSum)));
  //
  // Initialize tags...
  //
  if not GENDC.Simulator then begin
   GENDC_InitTags(ReadIni('tagPrefix'));
   bNul(sSetTag(GENDC.ID_IDN.tag,GENDC.ModelName));
  end;
  //
  // Initialize COM port
  //
  if not DIM_IsClientMode then
  if ComOpen('[SerialPort-COM'+Str(GENDC.Com.Port)+']')
  then Success('COM port initialized.')
  else Trouble('COM port failed.');
 end;
 //
 // GENDC finalization.
 //
 procedure GENDC_Free;
 begin
  bNul(ComClose);
  if (GENDC.CmdHash<>0) then begin
   bNul(HashList_Free(GENDC.CmdHash));
   GENDC.CmdHash:=0;
  end;
  GENDC_Tab_Clear;
 end;
 //
 // GENDC polling.
 //
 procedure GENDC_Poll;
 begin
  if GENDC.Simulator then begin
   if IsComPortOpened then begin
    //if (Pos(';',GENDC.Com.Buf)>0) and (Pos(Copy(CRLF,1,1),GENDC.Com.Buf)=0)
    //then GENDC.Com.Buf:=StrReplace(GENDC.Com.Buf,Dump(';'),';'+CRLF,3);
    while Com_Readln(GENDC.Com.Req,GENDC.Com.Buf) do GENDC_Sim(GENDC.Com.Req);
   end;
  end else begin
   GENDC_GUI_POLL;
   if not DIM_IsClientMode then begin
    if SysTimer_Pulse(1000)>0 then begin
     bNul(rSetTag(GENDC.ERROR_CNT.tag,GetErrCount(-1)));
     UpdateDimTag(GENDC.ERROR_CNT.tag);
     UpdateAo(ao_ERROR_CNT,time,GetErrCount(-1));
     bNul(rSetTag(GENDC.POLL_RATE.tag,GENDC.Com.PollRate));
     UpdateDimTag(GENDC.POLL_RATE.tag);
     UpdateAo(ao_POLL_RATE,time,GENDC.Com.PollRate);
     GENDC.Com.PollRate:=0;
    end;
    if mSecNow-FixmSecNow>DelayOnStart then
    GENDC_CMD_CYCLE;
   end;
  end;
 end;
 //
 // Clear user application strings...
 //
 procedure ClearApplication;
 begin
  GENDC_Clear;
 end;
 //
 // User application Initialization...
 //
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  GENDC_Init;
  RunStartupScript;
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  cmd_DimTagUpdate          := RegisterStdInCmd('@DimTagUpdate',          '');
  cmd_Simulate              := RegisterStdInCmd('@Simulate',              '');
  cmd_PrintCmdTable         := RegisterStdInCmd('@PrintCmdTable',         '');
  cmd_Ramping               := RegisterStdInCmd('@Ramping',               '');
  cmd_Setting               := RegisterStdInCmd('@Setting',               '');
  cmd_Remote                := RegisterStdInCmd('@Remote',                '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',                  '');
  cmd_MenuToolsOpen         := RegisterStdInCmd('@MenuToolsOpen',         '');
  cmd_LoadIni               := RegisterStdInCmd('@LoadIni',               '');
  cmd_SaveIni               := RegisterStdInCmd('@SaveIni',               '');
  GENDC_SelfTest;
 end;
 //
 // User application Finalization...
 //
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  GENDC_Free;
 end;
 //
 // User application Polling...
 //
 procedure PollApplication;
 begin
  GENDC_Poll;
 end;
 //
 // Process data coming from standard input...
 //
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid,tag,n,m:Integer; v:Real; slb:String;
  procedure SettingCmd(n:Integer; v:Real);
  begin
   if not IsNan(v) then
   if IsValidCmdNum(n) then begin
    Success(cmd+' '+GENDC.Cmd.Acronym[n]+' '+Str(v));
    GENDC.Cmd.OpBuff[n]:=v;
    EnableCmdNum(n,true);
    if (n=cm_FLDn) then bNul(iSetTag(GENDC.BT_FLD.tag,Ord(v<>0)));
    if (n=cm_ASTn) then bNul(iSetTag(GENDC.BT_AST.tag,Ord(v<>0)));
    if (n=cm_OUTn) then bNul(iSetTag(GENDC.BT_OUT.tag,Ord(v<>0)));
   end;
  end;
  procedure CheckEditCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(GENDC.Cmd.Tag[n])>=1 then
   if TypeTag(GENDC.Cmd.Tag[n])<=2 then
   if IsSameText(ExtractWord(1,arg),GENDC.Cmd.Acronym[n])
   or IsSameText(ExtractWord(1,arg),GENDC.Cmd.AnsPref[n]) then
   StartEditTagEx(GENDC.Cmd.Tag[n],GENDC.Cmd.Acronym[n]+': '+GENDC.Cmd.Comment[n],slb);
  end;
  procedure UpdateGenCmd(n:Integer);
  begin
   if IsValidCmdNum(n) then
   if TypeTag(GENDC.Cmd.Tag[n])>=1 then
   if TypeTag(GENDC.Cmd.Tag[n])<=2 then
   DevPostCmdLocal('@Setting '+GENDC.Cmd.Acronym[n]+' '+ TagAsText(GENDC.Cmd.Tag[n]));
  end;
  procedure UpdateAoByTag(AoNum,tag:Integer);
  begin
   if (AoNum>=0) then begin
    if TypeTag(tag)=1 then UpdateAo(AoNum,time,iGetTag(tag));
    if TypeTag(tag)=2 then UpdateAo(AoNum,time,rGetTag(tag));
   end;
  end;
  procedure UpdateSettingsAfterLoadIni;
  begin
   if (iGetTag(GENDC.POLL_ENABLE.tag)<>0) then begin
    DevPostCmdLocal('@Setting OVP '+TagAsText(GENDC.PAR_OVP.tag));
    DevPostCmdLocal('@Setting UVL '+TagAsText(GENDC.PAR_UVL.tag));
    DevPostCmdLocal('@Setting FENA $'+HexB(iGetTag(GENDC.PAR_FENA.tag)));
    DevPostCmdLocal('@Setting SENA $'+HexB(iGetTag(GENDC.PAR_SENA.tag)));
    DevPostCmdLocal('@Setting OUT '+Str(iAnd(iGetTag(GENDC.BT_OUT.tag),1)));
    DevPostCmdLocal('@Setting FLD '+Str(iAnd(iGetTag(GENDC.BT_FLD.tag),1)));
    DevPostCmdLocal('@Setting AST '+Str(iAnd(iGetTag(GENDC.BT_AST.tag),1)));
    DevPostCmdLocal('@Ramping PV '+TagAsText(GENDC.PAR_PV.tag));
    DevPostCmdLocal('@Ramping PC '+TagAsText(GENDC.PAR_PC.tag));
   end;
  end;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  slb:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Remote
   }
   if (cmdid=cmd_Remote) then begin
    if not IsEmptyStr(arg) then PostCmdRemote(Trim(arg));
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if TypeTag(tag)>0 then begin
      if tag=GENDC.POLL_RATE.tag then UpdateAoByTag(ao_POLL_RATE,tag);
      if tag=GENDC.ERROR_CNT.tag then UpdateAoByTag(ao_ERROR_CNT,tag);
      if tag=GENDC.PAR_MC.tag    then UpdateAoByTag(ao_PAR_MC,tag);
      if tag=GENDC.PAR_MV.tag    then UpdateAoByTag(ao_PAR_MV,tag);
      if tag=GENDC.PAR_MW.tag    then UpdateAoByTag(ao_PAR_MW,tag);
      if tag=GENDC.PAR_PC.tag    then UpdateAoByTag(ao_PAR_PC,tag);
      if tag=GENDC.PAR_PV.tag    then UpdateAoByTag(ao_PAR_PV,tag);
      if tag=GENDC.PAR_SR.tag    then UpdateAoByTag(ao_PAR_SR,tag);
      if tag=GENDC.PAR_FR.tag    then UpdateAoByTag(ao_PAR_FR,tag);
      if tag=GENDC.BT_OUT.tag    then UpdateAoByTag(ao_BT_OUT,tag);
      if tag=GENDC.BT_FLD.tag    then UpdateAoByTag(ao_BT_FLD,tag);
      if tag=GENDC.BT_AST.tag    then UpdateAoByTag(ao_BT_AST,tag);
      if tag=GENDC.PAR_OVP.tag   then UpdateAoByTag(ao_PAR_OVP,tag);
      if tag=GENDC.PAR_UVL.tag   then UpdateAoByTag(ao_PAR_UVL,tag);
      if tag=GENDC.PAR_RMT.tag   then UpdateAoByTag(ao_PAR_RMT,tag);
      if tag=GENDC.PAR_MODE.tag  then UpdateAoByTag(ao_PAR_MODE,tag);
      if tag=GENDC.PAR_FENA.tag  then UpdateAoByTag(ao_PAR_FENA,tag);
      if tag=GENDC.PAR_FEVE.tag  then UpdateAoByTag(ao_PAR_FEVE,tag);
      if tag=GENDC.PAR_SENA.tag  then UpdateAoByTag(ao_PAR_SENA,tag);
      if tag=GENDC.PAR_SEVE.tag  then UpdateAoByTag(ao_PAR_SEVE,tag);
      if tag=GENDC.PAR_SRQ.tag   then UpdateAoByTag(ao_PAR_SRQ,tag);
      if tag=GENDC.ID_IDN.tag then begin
       GENDC.ModelCode:=GENDC_UpdateModel(GENDC_ParseIDN(sGetTag(GENDC.ID_IDN.tag),false));
      end;
     end;
    end;
   end else
   {
   @Setting PV   1.0     @Setting PV?   1.0
   @Setting PC   2.0     @Setting PC?   2.0
   @Setting OVP  3.0     @Setting OVP?  3.0
   @Setting UVL  4.0     @Setting UVL?  4.0
   @Setting OUT  1       @Setting OUT?  1
   @Setting FLD  1       @Setting FLD?  1
   @Setting AST  1       @Setting AST?  1
   @Setting SENA 255     @Setting SENA? 255
   @Setting FENA 255     @Setting FENA? 255
   @Setting ADR  6
   @Setting RMT  1
   @Setting CLS
   @Setting RST
   @Setting OVM
   @Setting SAV
   @Setting RCL
   @Setting IDN
   @Setting RATE_A 1.0
   @Setting RATE_V 1.0
   }
   if (cmdid=cmd_Setting) then begin
    if not GENDC.Simulator and not DIM_IsClientMode then begin
     n:=GENDC_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=rVal(ExtractWord(2,arg));
      if (n=cm_PVa)   then n:=cm_PVn;
      if (n=cm_PCa)   then n:=cm_PCn;
      if (n=cm_OVPa)  then n:=cm_OVPn;
      if (n=cm_UVLa)  then n:=cm_UVLn;
      if (n=cm_OUTa)  then n:=cm_OUTn;
      if (n=cm_FLDa)  then n:=cm_FLDn;
      if (n=cm_ASTa)  then n:=cm_ASTn;
      if (n=cm_SENAa) then n:=cm_SENAn;
      if (n=cm_FENAa) then n:=cm_FENAn;
      if (n=cm_PVn)   then SettingCmd(n,v);
      if (n=cm_PCn)   then SettingCmd(n,v);
      if (n=cm_OVPn)  then SettingCmd(n,v);
      if (n=cm_UVLn)  then SettingCmd(n,v);
      if (n=cm_OUTn)  then SettingCmd(n,v);
      if (n=cm_FLDn)  then SettingCmd(n,v);
      if (n=cm_ASTn)  then SettingCmd(n,v);
      if (n=cm_SENAn) then SettingCmd(n,v);
      if (n=cm_FENAn) then SettingCmd(n,v);
      if (n=cm_ADRn)  then SettingCmd(n,v);
      if (n=cm_RMTn)  then SettingCmd(n,v);
      if (n=cm_CLS)   then SettingCmd(n,0);
      if (n=cm_RST)   then SettingCmd(n,0);
      if (n=cm_RST)   then bNul(iSetTag(GENDC.BT_FLD.tag,0));
      if (n=cm_RST)   then bNul(iSetTag(GENDC.BT_AST.tag,0));
      if (n=cm_RST)   then bNul(iSetTag(GENDC.BT_OUT.tag,0));
      if (n=cm_RST)   then bNul(rSetTag(GENDC.PAR_SRQ.tag,0));
      if (n=cm_OVM)   then SettingCmd(n,0);
      if (n=cm_SAV)   then SettingCmd(n,0);
      if (n=cm_RCL)   then SettingCmd(n,0);
     end else
     if IsSameText(ExtractWord(1,arg),'RATE_A') then begin
      UpdateTag(GENDC.RATE_A.tag,ExtractWord(2,arg),0,100);
     end else
     if IsSameText(ExtractWord(1,arg),'RATE_V') then begin
      UpdateTag(GENDC.RATE_V.tag,ExtractWord(2,arg),0,100);
     end else
     if IsSameText(ExtractWord(1,arg),'IDN') then begin
      bNul(sSetTag(GENDC.ID_IDN.tag,'?'));
      UpdateDimTag(GENDC.ID_IDN.tag);
      EnableCmdNum(cm_IDNa,true);
     end;
    end;
    Data:='';
   end else 
   {
   @Ramping PV 1.0
   @Ramping PC 2.0
   }
   if (cmdid=cmd_Ramping) then begin
    if not GENDC.Simulator and not DIM_IsClientMode then begin
     n:=GENDC_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=rVal(ExtractWord(2,arg));
      if (n=cm_PVa) then n:=cm_PVn;
      if (n=cm_PCa) then n:=cm_PCn;
      if (n=cm_PVn) then begin
       if (v>=GENDC.Cmd.Min[cm_PVn]) and (v<=GENDC.Cmd.Max[cm_PVn]) then begin
        RampingStart(GENDC.RampingV,GENDC.Cmd.OpData[cm_PVa],v,rGetTag(GENDC.RATE_V.tag));
        Success(cmd+' '+GENDC.Cmd.Acronym[cm_PVn]+' '+Str(v));
       end else Problem('VOL ramping value out of range.');
      end;
      if (n=cm_PCn) then begin
       if (v>=GENDC.Cmd.Min[cm_PCn]) and (v<=GENDC.Cmd.Max[cm_PCn]) then begin
        RampingStart(GENDC.RampingA,GENDC.Cmd.OpData[cm_PCa],v,rGetTag(GENDC.RATE_A.tag));
        Success(cmd+' '+GENDC.Cmd.Acronym[cm_PCn]+' '+Str(v));
       end else Problem('CUR ramping value out of range.');
      end;
     end;
    end;
    Data:='';
   end else 
   {
   @Edit PV <mime_encoded_sensor_left_bottom_settings>
   @Edit PC
   @Edit OVP
   @Edit UVL
   @Edit FENA
   @Edit SENA
   }
   if (cmdid=cmd_Edit) then begin
    if not GENDC.Simulator then begin
     n:=GENDC_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      if GENDC_CheckPolling(1) then begin
       slb:=mime_decode(ExtractWord(2,arg));
       CheckEditCmd(cm_PCn);
       CheckEditCmd(cm_PVn);
       CheckEditCmd(cm_OVPn);
       CheckEditCmd(cm_UVLn);
       if (n=cm_FENAn) then MenuFenaStarter;
       if (n=cm_SENAn) then MenuSenaStarter;
      end;
     end;
    end;
    Data:='';
   end else
   {
   @MenuToolsOpen
   }
   if (cmdid=cmd_MenuToolsOpen) then begin
    MenuToolsStarter;
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    if not GENDC.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
     UpdateSettingsAfterLoadIni;
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    if not GENDC.Simulator and not DIM_IsClientMode then begin
     iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    end;
    Data:='';
   end else
   {
   @Simulate PV? 1.0
   @Simulate MV? 2.0
   @Simulate PC? 3.0
   @Simulate MC? 4.0
   @Simulate RMT? 1
   @Simulate OUT? 1
   @Simulate FLD? 1
   @Simulate AST? 1
   @Simulate OVP? 5.0
   @Simulate UVL? 0.5
   @Simulate MODE? 1
   @Simulate FLT? 1
   @Simulate FENA? 255
   @Simulate FEVE? 255
   @Simulate STAT? 1
   @Simulate SENA? 255
   @Simulate SEVE? 255
   @Simulate OHM 1.5
   @Simulate NLP 0.1
   }
   if (cmdid=cmd_Simulate) then begin
    if GENDC.Simulator then begin
     n:=GENDC_Cmd_Find(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      GENDC.Cmd.SimDat[n]:=rValDef(ExtractWord(2,arg),GENDC.Cmd.SimDat[n]);
      Success(cmd+' '+GENDC.Cmd.Acronym[n]+' '+Str(GENDC.Cmd.SimDat[n]));
     end;
     if IsSameText(ExtractWord(1,arg),'OHM') then begin
      GENDC.SimOhm:=rValDef(ExtractWord(2,arg),GENDC.SimOhm);
      Success(cmd+' OHM '+Str(GENDC.SimOhm));
     end;
     if IsSameText(ExtractWord(1,arg),'NLP') then begin
      GENDC.SimNLP:=rValDef(ExtractWord(2,arg),GENDC.SimNLP);
      Success(cmd+' NLP '+Str(GENDC.SimNLP));
     end;
    end;
    Data:='';
   end else
   {
   @PrintCmdTable
   }
   if (cmdid=cmd_PrintCmdTable) then begin
    PrintCmdTable;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  slb:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
