////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpCmdArgs; // Diesel Pascal command line arguments parsing.

interface

type
 TCmdArgs = class(TStringList)
  private
   FEngineFile, FEngineName, FEngineBase, FEngineDir, FEngineExt : String;
   FScriptFile, FScriptName, FScriptBase, FScriptDir, FScriptExt : String;
  protected
   procedure Update;
   procedure ClearAll;
  public
   constructor Create;
   destructor  Destroy;
   property    EngineFile : String read FEngineFile; // Engine file dir\base.ext
   property    EngineName : String read FEngineName; // Engine file base.ext
   property    EngineBase : String read FEngineBase; // Engine file basename
   property    EngineDir  : String read FEngineDir;  // Engine file dir
   property    EngineExt  : String read FEngineExt;  // Engine file ext
   property    ScriptFile : String read FScriptFile; // Script file dir\base.ext
   property    ScriptName : String read FScriptName; // Script file base.ext
   property    ScriptBase : String read FScriptBase; // Script file basename
   property    ScriptDir  : String read FScriptDir;  // Script file dir
   property    ScriptExt  : String read FScriptExt;  // Script file ext
   function    IsOption(S:String):Boolean;           // Option start with [-,/]
   function    HasOption(Opt:String):Boolean;        // Check option Opt exists
   function    GetOptionValue(Opt:String):String;    // -Opt Value or -Opt=Value
  end;

function CmdArgs:TCmdArgs; // The only one command line parser

// Usage is like:
// if CmdArgs.HasOption('--test')
// then ShowMessage('--test = '+CmdArgs.GetOptionValue('--test'));
// for i:=0 to CmdArgs.Count-1 do ShowMessage(CmdArgs.Strings[i]);

implementation

var TheCmdArgs:TCmdArgs=nil;

function CmdArgs:TCmdArgs;
begin
 if (TheCmdArgs=nil) then TheCmdArgs:=TCmdArgs.Create;
 Result:=TheCmdArgs;
end;

constructor TCmdArgs.Create;
begin
 inherited Create;
 if (TheCmdArgs=nil) then TheCmdArgs:=Self;
 Update;
end;

destructor TCmdArgs.Destroy;
begin
 if (TheCmdArgs=Self) then TheCmdArgs:=nil;
 ClearAll;
 inherited Destroy;
end;

procedure TCmdArgs.ClearAll;
begin
 Clear;
 FEngineFile:=''; FEngineName:=''; FEngineBase:=''; FEngineDir:=''; FEngineExt:='';
 FScriptFile:=''; FScriptName:=''; FScriptBase:=''; FScriptDir:=''; FScriptExt:='';
end;

procedure TCmdArgs.Update;
var i:Integer;
begin
 ClearAll;
 FEngineFile:=ExpandFileName(ParamStr(0));   FScriptFile:=ExpandFileName(ParamStr(1));
 FEngineName:=ExtractFileName(EngineFile);   FScriptName:=ExtractFileName(ScriptFile);
 FEngineBase:=ReplaceFileExt(EngineName,''); FScriptBase:=ReplaceFileExt(ScriptName,'');
 FEngineDir:=ExtractFileDir(EngineFile);     FScriptDir:=ExtractFileDir(ScriptFile);
 FEngineExt:=ExtractFileExt(EngineFile);     FScriptExt:=ExtractFileExt(ScriptFile);
 for i:=0 to ParamCount do Add(ParamStr(i));
end;

function TCmdArgs.IsOption(S:String):Boolean;
begin
 Result:=false;
 if (S='') then Exit;
 if (S[1]='-') then Result:=true else
 if (S[1]='/') then Result:=WINDOWS;
end;

function TCmdArgs.HasOption(Opt:String):Boolean;
begin
 Result:=false;
 if (Opt='') then Exit;
 if not IsOption(Opt) then Exit;
 if (IndexOf(Opt)<0) and (IndexOfName(Opt)<0) then Exit;
 Result:=true;
end;

function TCmdArgs.GetOptionValue(Opt:String):String;
var i:Integer;
begin
 Result:='';
 if (Opt='') then Exit;
 if not IsOption(Opt) then Exit;
 i:=IndexOf(Opt);
 if (i>=0) and (i<Count-1) then begin
  if not IsOption(Strings[i+1])
  then Result:=Strings[i+1];
  Exit;
 end;
 i:=IndexOfName(Opt);
 if (i>=0) then begin
  Result:=Values[Opt];
  Exit;
 end;
end;

initialization

 CmdArgs;

end.
