////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Surf Window Style Dialog.                                             //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231125 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_surfwindowstyledialog; // Form Surf Window Style Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons,
 lcltype, lclintf, ColorBox,
 Form_SurfWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormSurfWindowStyleDialog }

  TFormSurfWindowStyleDialog = class(TMasterForm)
    ColorBox: TColorBox;
    PanelCaption: TPanel;
    LabelCaption: TLabel;
    EditCaption: TEdit;
    LabelTitle: TLabel;
    EditTitle: TEdit;
    LabelLegend: TLabel;
    EditLegend: TEdit;
    LabelScale: TLabel;
    LabelScaleX: TLabel;
    EditScaleX: TEdit;
    LabelScaleY: TLabel;
    EditScaleY: TEdit;
    LabelScaleZ: TLabel;
    EditScaleZ: TEdit;
    LabelSliceMode: TLabel;
    CheckBoxSliceX: TCheckBox;
    CheckBoxSliceY: TCheckBox;
    CheckBoxNumberUp: TCheckBox;
    RadioGroupColor: TRadioGroup;
    GroupBoxAngles: TGroupBox;
    LabelPsi: TLabel;
    EditPsi: TEdit;
    LabelPhi: TLabel;
    EditPhi: TEdit;
    GroupBoxLimits: TGroupBox;
    LabelX1: TLabel;
    EditX1: TEdit;
    LabelY1: TLabel;
    EditY1: TEdit;
    LabelX2: TLabel;
    EditX2: TEdit;
    LabelY2: TLabel;
    EditY2: TEdit;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    ButtonPreview: TButton;
    ButtonSetAsDefault: TButton;
    procedure ColorBoxChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure ButtonSetAsDefaultClick(Sender: TObject);
    procedure ButtonPreviewClick(Sender: TObject);
    procedure LabelCaptionClick(Sender: TObject);
    procedure LabelTitleClick(Sender: TObject);
    procedure LabelLegendClick(Sender: TObject);
    procedure LabelScaleXClick(Sender: TObject);
    procedure LabelScaleYClick(Sender: TObject);
    procedure LabelScaleZClick(Sender: TObject);
    procedure LabelPsiClick(Sender: TObject);
    procedure LabelPhiClick(Sender: TObject);
    procedure LabelX1Click(Sender: TObject);
    procedure LabelY1Click(Sender: TObject);
    procedure LabelX2Click(Sender: TObject);
    procedure LabelY2Click(Sender: TObject);
    procedure RadioGroupColorClick(Sender: TObject);
  private
    myActiveSurfWindow : TFormSurfWindow;
    myColorBoxLock : Boolean;
  public
    procedure Setup;
    function  Apply:Boolean;
    procedure ColorBoxUpdate(Direction:Integer);
  end;

procedure FormSurfWindowStyleDialogExecute(aForm:TForm; const aParams:LongString='');

implementation

{$R *.lfm}

const
  FormSurfWindowStyleDialog : TFormSurfWindowStyleDialog = nil;

function ActiveSurfWindow(aForm:TForm=nil):TFormSurfWindow;
begin
 Result:=nil;
 if Assigned(FormSurfWindowStyleDialog) and not Assigned(aForm)
 then aForm:=FormSurfWindowStyleDialog.myActiveSurfWindow;
 if FormExists(aForm) and (aForm is TFormSurfWindow)
 then Result:=TFormSurfWindow(aForm);
 if not Assigned(Result) then Result:=Form_SurfWindow.ActiveSurfWindow;
end;

var
 Save:record
  Caption,Title,Legend:PureString;
  Clip:TRect2D;
  Scale:TPoint3D;
  Phi,Psi:Double;
  SliceMode:LongInt;
  GroundColor,UpColor,DownColor,AxisColor,GridColor,TitleColor,LegendColor:TColor;
 end;

procedure SaveParams;
begin
 with ActiveSurfWindow do
 if Ok then begin
  Save.Caption:=Caption;
  Save.Title:=Title;
  Save.Legend:=Legend;
  Save.Clip:=Clip;
  Save.Scale:=Scale;
  Save.Phi:=Phi;
  Save.Psi:=Psi;
  Save.SliceMode:=SliceMode;
  Save.GroundColor:=GroundColor;
  Save.UpColor:=UpColor;
  Save.DownColor:=DownColor;
  Save.AxisColor:=AxisColor;
  Save.GridColor:=GridColor;
  Save.TitleColor:=TitleColor;
  Save.LegendColor:=LegendColor;
 end;
end;

procedure RestoreParams;
begin
 with ActiveSurfWindow do
 if Ok then begin
  try
   LockDraw;
   Caption:=Save.Caption;
   Title:=Save.Title;
   Legend:=Save.Legend;
   Clip:=Save.Clip;
   Scale:=Save.Scale;
   Phi:=Save.Phi;
   Psi:=Save.Psi;
   SliceMode:=Save.SliceMode;
   GroundColor:=Save.GroundColor;
   UpColor:=Save.UpColor;
   DownColor:=Save.DownColor;
   AxisColor:=Save.AxisColor;
   GridColor:=Save.GridColor;
   TitleColor:=Save.TitleColor;
   LegendColor:=Save.LegendColor;
  finally
   UnlockDraw;
  end;
 end;
end;

procedure FormSurfWindowStyleDialogExecute(aForm:TForm; const aParams:LongString='');
var apFlags:Integer;
begin
 if Assigned(aForm) then
 if ActiveSurfWindow(aForm).Ok then
 if CanShowModal(FormSurfWindowStyleDialog) then
 try
  if not Assigned(FormSurfWindowStyleDialog)  then begin
   Application.CreateForm(TFormSurfWindowStyleDialog, FormSurfWindowStyleDialog);
   FormSurfWindowStyleDialog.Master:=@FormSurfWindowStyleDialog;
  end;
  if Assigned(FormSurfWindowStyleDialog) then begin
   FormSurfWindowStyleDialog.myActiveSurfWindow:=ActiveSurfWindow(aForm);
   apFlags:=FormSurfWindowStyleDialog.ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then LocateFormToCenterOfScreen(FormSurfWindowStyleDialog);
   with ActiveSurfWindow do begin
    SaveParams;
    FormSurfWindowStyleDialog.Setup;
    if mrVoice(FormSurfWindowStyleDialog.ShowModal)=mrOk then begin
     if not FormSurfWindowStyleDialog.Apply then RestoreParams;
    end else begin
     RestoreParams;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormSurfWindowStyleDialogExecute'); 
 end;
end;

procedure TFormSurfWindowStyleDialog.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Диалог выбора стиля окна поверхности','Surface window style selection dialog');
 SmartUpdate(LabelCaption,RusEng('Имя окна','Window name'));
 SmartUpdate(LabelTitle,RusEng('Заголовок окна','Window title'));
 SmartUpdate(LabelLegend,RusEng('Легенда окна','Window legend'));
 SmartUpdate(LabelScale,RusEng('Масштаб','Axis scale'));
 SmartUpdate(LabelSliceMode,RusEng('Рисовать','Plot flags'));
 SmartUpdate(CheckBoxSliceX,RusEng('X-сечения','X-slices'));
 SmartUpdate(CheckBoxSliceY,RusEng('Y-сечения','Y-slices'));
 SmartUpdate(CheckBoxNumberUp,RusEng('ЦифрыСверху','Number up'));
 SmartUpdate(RadioGroupColor,RusEng('Цвет','Color'));
 RadioGroupColor.Items.Text:=RusEng('Фон'+EOL+
                                    'Cверху'+EOL+
                                    'Cнизу'+EOL+
                                    'Оси'+EOL+
                                    'Цифры'+EOL+
                                    'Заголовок'+EOL+
                                    'Легенда',
                                    'Ground'+EOL+
                                    'Upper'+EOL+
                                    'Lower'+EOL+
                                    'Axis'+EOL+
                                    'Grid'+EOL+
                                    'Title'+EOL+
                                    'Legend');
 RadioGroupColor.ItemIndex:=0;
 SmartUpdate(GroupBoxAngles,RusEng('Угол зрения','View angle'));
 SmartUpdate(LabelPsi,RusEng('Сферический,°','Spheric,°'));
 SmartUpdate(LabelPhi,RusEng('Аксиальный,°','Axial,°'));
 SmartUpdate(GroupBoxLimits,RusEng('Видимый регион','View limits'));
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 SmartUpdate(ButtonPreview,RusEng('Применить','Apply now'));;
 SmartUpdate(ButtonSetAsDefault,RusEng('Сделать основными','Save as default'));
end;

procedure TFormSurfWindowStyleDialog.ButtonSetAsDefaultClick(Sender: TObject);
begin
 with ActiveSurfWindow do
 if Ok then begin
  DefGroundColor:=GroundColor;
  DefUpColor:=UpColor;
  DefDownColor:=DownColor;
  DefAxisColor:=AxisColor;
  DefGridColor:=GridColor;
  DefTitleColor:=TitleColor;
  DefLegendColor:=LegendColor;
 end;
end;

procedure TFormSurfWindowStyleDialog.ButtonPreviewClick(Sender: TObject);
begin
 Apply;
 Setup;
end;

procedure TFormSurfWindowStyleDialog.Setup;
begin
 with ActiveSurfWindow do
 if Ok then begin
  EditCaption.Text:=Caption;
  EditTitle.Text:=Title;
  EditLegend.Text:=Legend;
  EditScaleX.Text:=Format('%g',[Scale.X]);
  EditScaleY.Text:=Format('%g',[Scale.Y]);
  EditScaleZ.Text:=Format('%g',[Scale.Z]);
  EditX1.Text:=Format('%g',[Clip.A.X]);
  EditY1.Text:=Format('%g',[Clip.A.Y]);
  EditX2.Text:=Format('%g',[Clip.B.X]);
  EditY2.Text:=Format('%g',[Clip.B.Y]);
  EditPhi.Text:=Format('%g',[RadToDeg(Phi)]);
  EditPsi.Text:=Format('%g',[RadToDeg(Psi)]);
  CheckBoxSliceX.Checked:=SliceMode and smSliceX <> 0;
  CheckBoxSliceY.Checked:=SliceMode and smSliceY <> 0;
  CheckBoxNumberUp.Checked:=SliceMode and smNumberUp <> 0;
  ColorBoxUpdate(-1);
 end;
end;

function TFormSurfWindowStyleDialog.Apply:Boolean;
var aClip:TRect2D; aScale:TPoint3D; aPhi,aPsi:Double;
begin
 Result:=false;
 with ActiveSurfWindow do
 if Ok then begin
  if Str2Real(EditX1.Text,aClip.A.X) and
     Str2Real(EditY1.Text,aClip.A.Y) and
     Str2Real(EditX2.Text,aClip.B.X) and
     Str2Real(EditY2.Text,aClip.B.Y) and
     Str2Real(EditScaleX.Text,aScale.X) and
     Str2Real(EditScaleY.Text,aScale.Y) and
     Str2Real(EditScaleZ.Text,aScale.Z) and
     Str2Real(EditPhi.Text,aPhi) and
     Str2Real(EditPsi.Text,aPsi) and
     not isNanOrInf(aClip) and
     not RectIsEmpty(aClip) and
     not isNanOrInf(Phi) and
     not isNanOrInf(Psi) and
     not isNanOrInf(Scale.X) and
     not isNanOrInf(Scale.Y) and
     not isNanOrInf(Scale.Z) and
     not (Scale.X=0) and
     not (Scale.Y=0) and
     not (Scale.Z=0)
  then begin
   try
    LockDraw;
    Caption:=EditCaption.Text;
    Title:=EditTitle.Text;
    Legend:=EditLegend.Text;
    Clip:=aClip;
    Scale:=aScale;
    Phi:=DegToRad(aPhi);
    Psi:=DegTorad(aPsi);
    SliceMode:=smSliceX*ord(CheckBoxSliceX.Checked)+
               smSliceY*ord(CheckBoxSliceY.Checked)+
               smNumberUp*ord(CheckBoxNumberUp.Checked);
    ColorBoxUpdate(1);
   finally
    UnlockDraw;
   end;
   Result:=true;
  end else Warning(RusEng('Заданы неверные параметры!','Invalid parameters!'));
 end;
end;

procedure TFormSurfWindowStyleDialog.LabelCaptionClick(Sender: TObject);
begin
 SmartFocus(EditCaption);
end;

procedure TFormSurfWindowStyleDialog.LabelTitleClick(Sender: TObject);
begin
 SmartFocus(EditTitle);
end;

procedure TFormSurfWindowStyleDialog.LabelLegendClick(Sender: TObject);
begin
 SmartFocus(EditLegend);
end;

procedure TFormSurfWindowStyleDialog.LabelScaleXClick(Sender: TObject);
begin
 SmartFocus(EditScaleX);
end;

procedure TFormSurfWindowStyleDialog.LabelScaleYClick(Sender: TObject);
begin
 SmartFocus(EditScaleY);
end;

procedure TFormSurfWindowStyleDialog.LabelScaleZClick(Sender: TObject);
begin
 SmartFocus(EditScaleZ);
end;

procedure TFormSurfWindowStyleDialog.LabelPsiClick(Sender: TObject);
begin
 SmartFocus(EditPsi);
end;

procedure TFormSurfWindowStyleDialog.LabelPhiClick(Sender: TObject);
begin
 SmartFocus(EditPhi);
end;

procedure TFormSurfWindowStyleDialog.LabelX1Click(Sender: TObject);
begin
 SmartFocus(EditX1);
end;

procedure TFormSurfWindowStyleDialog.LabelY1Click(Sender: TObject);
begin
 SmartFocus(EditY1);
end;

procedure TFormSurfWindowStyleDialog.LabelX2Click(Sender: TObject);
begin
 SmartFocus(EditX2);
end;

procedure TFormSurfWindowStyleDialog.LabelY2Click(Sender: TObject);
begin
 SmartFocus(EditY2);
end;

procedure TFormSurfWindowStyleDialog.ColorBoxUpdate(Direction:Integer);
begin
 with ActiveSurfWindow do
 if Ok then begin
  if (Direction>0) then begin
   if not myColorBoxLock then
   case RadioGroupColor.ItemIndex of
    0 : GroundColor:=ColorBox.Selected;
    1 : UpColor:=ColorBox.Selected;
    2 : DownColor:=ColorBox.Selected;
    3 : AxisColor:=ColorBox.Selected;
    4 : GridColor:=ColorBox.Selected;
    5 : TitleColor:=ColorBox.Selected;
    6 : LegendColor:=ColorBox.Selected;
   end;
  end;
  if (Direction<0) then begin
   case RadioGroupColor.ItemIndex of
    0 : ColorBox.Selected:=GroundColor;
    1 : ColorBox.Selected:=UpColor;
    2 : ColorBox.Selected:=DownColor;
    3 : ColorBox.Selected:=AxisColor;
    4 : ColorBox.Selected:=GridColor;
    5 : ColorBox.Selected:=TitleColor;
    6 : ColorBox.Selected:=LegendColor;
   end;
  end;
 end;
end;

procedure TFormSurfWindowStyleDialog.ColorBoxChange(Sender: TObject);
begin
 ColorBoxUpdate(1);
end;

procedure TFormSurfWindowStyleDialog.RadioGroupColorClick(Sender: TObject);
begin
 try
  myColorBoxLock:=true;
  ColorBoxUpdate(-1);
 finally
  myColorBoxLock:=false;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_surfwindowstyledialog;
begin
end;

procedure Free_form_surfwindowstyledialog;
begin
end;

initialization

 Init_form_surfwindowstyledialog;

finalization

 Free_form_surfwindowstyledialog;

end.

//////////////
// END OF FILE
//////////////


