////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Модуль содержит программы вычисления торможения ускоренных ионов в         //
// веществе, заимствованные из книги                                          //
//                                                                            //
// J.F.Ziegler, J.P.Biersack and U.Littmark. "The stopping and range of       //
// ions in solids", vol.1 of The stopping and ranges of ions in matter,       //
// Pergamon Press, 1985                                                       //
//                                                                            //
// Вычисление страгглинга ускоренных ионов в веществе производится с          //
// помощью полуэмпирического метода, предложенного в работе                   //
//                                                                            //
// Q.Yang, D.J. O'Connor and Z.Wang.                                          //
// "Empirical formulae for energy loss straggling of ions in matter",         //
// Nucl. Instr. and Meth. B 61 (1991) 149-155.                                //
//                                                                            //
// Автор:      Романихин В.П.                                                 //
// Переделки : Курякин А.В. - мелкие изменения всвязи с другой структурой     //
//             таблиц                                                         //
//                                                                            //
// Единицы измерений:                                                         //
// dE/dX[эВ/(1e15*атом/кв.см.)] = 1e23/N*dEdX[эв/ангстрем]                    //
// = 1e21/N*dE/dX[эВ*кв.см./мкг;кэВ*кв.см./мг;МэВ*кв.см./г]                   //
// N [1/куб.см.] = p[г/куб.см.] * Nav / A [А,Е.М.]                            //
// Nav = 6.02e23                                                              //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231127 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_stopion; // Stop Ion

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 _crw_alloc, _crw_ef, _crw_str, _crw_zm,
 _crw_fio, _crw_rfadata;

 {
 Вычисление сечения электронного торможения ионов.
 Сечение вычисляется в единицах эВ/(10е15 атом/кв.см).
 Z1:    атомный номер иона,
 Z2:    атомный номер мишени,
 E:     энергия иона [кэВ]
 }
function EStop(Z1,Z2:integer; E:double):double;

 {
 Вычисление сечения ядерного торможения ионов
 Сечение вычисляется в единицах эВ/(10е15 атом/кв.см).
 Z1:    атомный номер иона,
 Z2:    атомный номер мишени,
 E:     энергия иона [кэВ]
 }
function NStop(Z1,Z2:integer; E:double):double;


 {
 Универсальная процедура вычисления сечения торможения для ускоренных
 ионов в веществе. Вычисляются сечения электронного и ядерного торможения.
 Выбор единиц измерения зависит от значения параметра Units.
 Z1,Z2:   атомные номера иона и мишени,
 EE:       энергия ионов, кэв
 Units    = 0 или 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
          = 4        единицы LSS.
 SE,SN:   сечения электронного и ядерного торможения ионов
 }
procedure UniStop(Z1,Z2,Units:integer; EE:double; var SE,SN:double);

 {
 Процедура вычисляет относительный страгглинг ускоренных ионов
 в веществе мишени. Рассматриваются только конденсированные мишени.
 Результатом работы процедуры является отношение диперсии энергетических
 потерь ионов при торможении к дисперсии энергетического разброса Бора.
 Z1,Z2:   атомные номера иона и мишени,
 EE:      энергия ионов, кэв
 }
function Stragg(Z1,Z2:integer; EE:double):double;

 {
 Используя правило Брэгга, вычислить тормозную способности мишени
 состоящей из нескольких элементов в единицах [эВ/(10**15 атом/кв.см]
 }
function BraggRule(Energy : double;           { энергия пучка [кэВ]          }
                   ZBeam  : integer;          { ион пучка                    }
               var ZTarg  : array of integer; { массив атомов мишени         }
               var DTarg  : array of double;  { отн. атом.доли атомов мишени }
                   NTarg  : integer           { число атомов мишени          }
                   ):double;

 {
 Вспомогательная процедура вычисляет среднюю атомную массу смеси
 }
function AverageAtomMass(
                var ZTarg  : array of integer; { из каких атомов мишень      }
                var DTarg  : array of double;  { относ.атом.доли элементов   }
                    NTarg  : integer           { число элементов в мишени    }
                   ):double;

 {
 Вычисление тормозной способности вещества мишени для иона при произвольной
 энергии Energy [кэВ]
 Единицы зависят от величины Units:
 Units    = 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
 ZBeam
 }
function dE_dX(Energy : double;           { энергия пучка [кэВ]             }
               Units  : integer;          { единицы измерения 1..3          }
               ZBeam  : integer;          { из каких ионов пучек            }
           var ZTarg  : array of integer; { из каких атомов мишень          }
           var DTarg  : array of double;  { относ. атом.доли элем. в мишени }
               NTarg  : integer;          { число элементов в мишени        }
               Rho    : double            { плотность мишени                }
               ):double;

 {
 Вычисление пробега иона от энергии E1 до E2. Энергия - [кэВ], E1 >= E2.
 Единицы зависят от величины Units:
 Units    = 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
 }
function GetIonTrack(
               E1,E2  : double;           { интервал энергий,[кэВ],E1>=E2   }
               Units  : integer;          { единицы измерения 1..3          }
               ZBeam  : integer;          { из каких ионов пучек            }
           var ZTarg  : array of integer; { из каких атомов мишень          }
           var DTarg  : array of double;  { относ. атом.доли элем. в мишени }
               NTarg  : integer;          { число элементов в мишени        }
               RTarg  : double            { плотность мишени                }
                    ) : double;

const
  BeamEmin = 0.01;     { минимальная энергия иона, кэВ  }
  BeamEmax = 150000.0; { максимальная энергия иона, кэВ }

 {
 Функция рассчитывает сечение ионизации для K и L оболочек "сигма эн"
 Отчет 8/5435, 1979, стр.7
 Для справки
  Mn = 1.008982   AEM
  Mp = 1.0075957  AEM  <-> Lambda = 1836.132
  Me = 5.48760E-4 AEM
 }
function RfaWhat_Sn(Energy : double;    { энергия ионизирующего излучения,[кэВ]}
                    Lambda : double;    { отношение массы частицы к массе e    }
                    AtomId : byte;      { идентификатор атома мишени           }
                    LineId : TRfaFields { идентификатор линии мишени           }
                   )       : double;    { сечение ионизации [кв.см.]           }

 {
 Найти коэффициент флюоресцентного выхода "омега" (фактор эффективности)
 8/5435, 1979, стр.7
 }
function RfaWhat_Omega(AtomId:byte; LineId:TRfaFields; Energy:double):double;

 {
 Найти относительную интенсивность K характеристической рентгеновской линии
 8/5435, 1979, стр.7
 }
function RfaWhat_K(AtomId : integer;   { атомный номер       }
                   LineId : TRfaFields { идентификатор линии }
                  ) : double;

 {
 Найти эффективный коэффициент ослабления для многокомпонентной матрицы.
 Массовые доли получаем пересчетом атомных долей через атомные веса и
 атомные доли.
 Отчет 8/5435, 1979, стр.10
 }
function RfaWhat_Mu(Energy : double;           { энергия пучка, [кэВ]       }
                var ZTarg  : array of integer; { заряды элементов матрицы   }
                var DTarg  : array of double;  { атомные доли элементов     }
                    NTarg  : integer           { число элементов матрицы    }
                   ) : double;                 { в единицах [кв.см./грамм]  }

 {
 }
function Rfa_Gamma(
          {описание линии ХРИ для которой ищется гамма}
          AtomId : byte;             { атом для которого ищется выход ХРИ   }
          LineId : TRfaFields;       { линия для которой ищется выход ХРИ   }
          {описание пучка}
          ZBeam  : byte;             { ион пучка                            }
          E2Beam : double;           { энергия пучка                        }
          E1Beam : double;           { нижний предел                        }
          {описание геометрии}
          Alfa   : double;           { угол падения пучка от нормали, град  }
          Beta   : double;           { угол на детектор от нормали, град    }
          Thick  : double;           { толщина мишени                       }
          {описание матрицы мишени}
          ZTarg  : array of integer; { элементы входящие в состав матрицы   }
          DTarg  : array of double;  { относительные доли элементов матрицы }
          NTarg  : integer;          { число элементов в матрице            }
          RTarg  : double            { плотность матрицы                    }
                  ) : double;

implementation

const
 Nav=6.0222e23;
 TheRelTol=1e-4;

 {
 Вычисление тормозной способности протонов в веществе
 в единицах эВ/(10е15 атом/кв.см).
 Z:     зарядовое число атома мишени,
 E:     (удельная) энергия протонов, кэВ(/аем),
 P:     коэффициенты для вычисления тормозной способности
 }
function StopP(Z:integer; E:double):double;
var PE,PE0,SL,SH,SE,VP,P1,P2,P3,P4,P5,P6,P7,P8:double;
begin
 StopP:=0;
 if Rfa.Accessible[Z] then begin
  if E < 1.0e-6 then E:=1.0e-6;
  PE0:=25.0;
  PE:=Max(E,PE0);
  P1:=Rfa[Z,rf_PSTOP_P1];
  P2:=Rfa[Z,rf_PSTOP_P2];
  P3:=Rfa[Z,rf_PSTOP_P3];
  P4:=Rfa[Z,rf_PSTOP_P4];
  P5:=Rfa[Z,rf_PSTOP_P5];
  P6:=Rfa[Z,rf_PSTOP_P6];
  P7:=Rfa[Z,rf_PSTOP_P7];
  P8:=Rfa[Z,rf_PSTOP_P8];
  SL:=P1*Power(PE,P2)+P3*Power(PE,P4);
  SH:=P5/Power(PE,P6)*Ln(P7/PE+P8*PE);
  SE:=SL*SH/(SL+SH);
  if E <= PE0 then begin
   if Z > 6 then VP:=0.45 else VP:=0.25;
   SE:=SE*Power(E/PE0,VP);
  end;
  StopP:=SE;
 end;
end;

 {
 Вычисление масштабирующего множителя для нахождения тормозной
 способности ионов гелия в веществе, используя при расчете
 тормозную способность протонов.
 Z:     зарядовое число атома мишени,
 E:     удельная энергия ионов гелия, кэВ/аем,
 G:     искомый множитель к тормозной способности протонов,
 HE:    значение удельной энергии, которое нужно подставить в
        процедуру вычисления торможения протонов.
 }
procedure StopHe(Z:integer; E:double; out G,HE:double);
var A,B,HEH,HE0:double;
begin
 HE0:=1.0;
 HE:=Max(HE0,E);
 B:=Ln(HE);
 A:=0.2865+B*(0.1266+B*(-0.001429+B*(0.02402+B*(-0.01135+B*0.001475))));
 HEH:=1.0-exp(-Min(30.0, A));
 A:=1.0+(0.007+0.00005*Z)*exp(-sqr(7.6-Max(0.0,B)));
 HEH:=HEH*sqr(A);
 G:=HEH;
 if E <= HE0 then G:=G*sqrt(E/HE0);
end;

 {
 Вычисление масштабирующего множителя для нахождения тормозной
 способности тяжелых ионов в веществе, используя при расчете
 тормозную способность протонов.
 Z1,Z2:     зарядовые числа иона и атома мишени,
 E:         удельная энергия ионов, кэВ/аем,
 G:     искомый множитель к тормозной способности протонов,
 HE:    значение удельной энергии, которое нужно подставить в
        процедуру вычисления торможения протонов.
 }
procedure StopHI(Z1,Z2:integer; E:double; var G,HE:double);
var YR,YRmin,VR,VRmin,A,B,Q,L,L0,L1,Tmp,V,V2,V4,Vmin,VF,Z13,Z23,Zeta:double;
begin
 G:=0;
 HE:=0;
 if Rfa.Accessible[Z1] and Rfa.Accessible[Z2] then begin
  YRmin:=0.13;
  VRmin:=1.0;
  VF:=Rfa[Z2,rf_ATOMS_VFERMI];
  V:=sqrt(E/25.0)/VF;
  V2:=sqr(V);
  V4:=sqr(V2);
  if V < 1.0 then VR:=0.75*VF*(1.0+(0.6666667*V2)-V4/15.0)
  else VR:=V*VF*(1.0+0.2/V2);
  Z13:=Power(Z1,0.33333);
  Z23:=sqr(Z13);
  YR:=Max(YRmin,VR/Z23);
  YR:=Max(YR,VRmin/Z23);
  Tmp:=Power(YR,0.3);
  A:=-0.803*Tmp+1.3167*sqr(Tmp)+0.38157*YR+0.008983*sqr(YR);
  Q:=Min(1.0,Max(0.0,1.0-exp(-Min(A,50.0))));
  {Q - уровень ионизации иона при скорости YR.
   Преобразуем это в эффективный заряд иона}
  B:=(Min(0.43,Max(0.32,0.12+0.025*Z1)))/Z13;
  L0:=(0.8-Q*(Min(1.2,0.6+Z1/30.0)))/Z13;
  if Q >= 0.2 then
  if Q >= Max(0.0,0.9-0.025*Z1) then begin
   if Q >= Max(0.0,1.0-0.025*Min(16.0,Z1))
   then L1:=B*(1.0-Q)/(0.025*Min(16.0,Z1))
   else L1:=B;
  end else begin
   L1:=B*(Q-0.2)/abs(Max(0.0,0.9-0.025*Z1)-0.2000001);
  end
  else L1:=0.0;
  L:=Max(L1,L0*Rfa[Z1,rf_ATOMS_LFCTR]);
  Zeta:=Q+(0.5/sqr(VF))*(1.0-Q)*Ln(1.0+sqr(4.0*L*VF/1.919));
  A:=-sqr(7.6-Max(0.0,Ln(E)));
  Zeta:=Zeta*(1.0+(1.0/sqr(Z1))*(0.18+0.0015*Z2)*exp(A));
  if YR > Max(YRmin,VRmin/Z23) then begin
   G:=sqr(Zeta);
   HE:=E;
  end else begin
   VRmin:=Max(VRmin,YRmin*Z23);
   Vmin:=0.5*(VRmin+sqrt(Max(0.0,sqr(VRmin)-0.8*sqr(VF))));
   Tmp:=25.0*sqr(Vmin);
   if (Z2 = 6) or (((Z2 = 14) or (Z2 = 32) and (Z1 <= 19)))
   then Q:=0.375
   else Q:=0.5;
   G:=sqr(Zeta)*Power(E/Tmp,Q);
   HE:=Tmp;
  end;
 end;
end;

 {
 Вычисление сечения электронного торможения ионов.
 Сечение вычисляется в единицах эВ/(10е15 атом/кв.см).
 Z1:    атомный номер иона,
 Z2:    атомный номер мишени,
 E:     энергия иона [кэВ]
 }
function EStop(Z1,Z2:integer; E:double):double;
var
  Gamma:double;   {масшт. коэфф. для тяжелых ионов}
  EM:double;      {удельная энергия иона}
  SE,Y:double;
begin
 EStop:=0;
 if Rfa.Accessible[Z1] and Rfa.Accessible[Z2] then begin
  if E < 1.0e-6 then begin
   EStop:=0.0;
   Exit;
  end;
  EM:=E/Rfa[Z1,rf_ATOMS_M2];
  case Z1 of
   1  : SE:=StopP(Z2,EM);
   2  : begin
         StopHe(Z2,EM,Gamma,Y);
         SE:=4.0*Gamma*StopP(Z2,Y);
        end;
   else begin
         StopHI(Z1,Z2,EM,Gamma,Y);
         SE:=sqr(Z1)*Gamma*StopP(Z2,Y);
        end;
  end;
  EStop:=SE;
 end;
end;

 {
 Вычисление сечения ядерного торможения ионов
 Сечение вычисляется в единицах эВ/(10е15 атом/кв.см).
 Z1:    атомный номер иона,
 Z2:    атомный номер мишени,
 E:     энергия иона [кэВ]
 }
function NStop(Z1,Z2:integer; E:double):double;
var
  Eps:double;     {энергия в единицах [LSS]}
  Y,FEps,NCoef,SN:double;
begin
 NStop:=0;
 if Rfa.Accessible[Z1] and Rfa.Accessible[Z2] then begin
  if E < 1.0e-6 then begin
   NStop:=0.0;
   Exit;
  end;
  Y:=Power(Z1,0.23)+Power(Z2,0.23);
  FEps:=32.53*Rfa[Z2,rf_ATOMS_M2]/(Z1*Z2*(Rfa[Z1,rf_ATOMS_M2]+Rfa[Z2,rf_ATOMS_M2])*Y);
  NCoef:=Z1*Z2*Rfa[Z1,rf_ATOMS_M2]*8.462*10.0/((Rfa[Z1,rf_ATOMS_M2]+Rfa[Z2,rf_ATOMS_M2])*Y);
  Eps:=FEps*E;
  if Eps < 30.0 then begin
   Y:=0.01321*Power(Eps,0.21226)+0.19593*sqrt(Eps);
   SN:=0.5*Ln(1.0+1.1383*Eps)/(Eps+Y);
  end else SN:=Ln(Eps)/(2.0*Eps);
  NStop:=SN*NCoef/10.0;
 end;
end;

 {
 Процедура вычисляет относительный страгглинг ускоренных ионов
 в веществе мишени. Рассматриваются только конденсированные мишени.
 Результатом работы процедуры является отношение диперсии энергетических
 потерь ионов при торможении к дисперсии энергетического разброса Бора.
 Z1,Z2:   атомные номера иона и мишени,
 EE:      энергия ионов, кэв
 }
function Stragg(Z1,Z2:integer; EE:double):double;
const
  {коэффициенты для учета флуктуаций зарядового
  состояния протонов при прохождении их через вещество}
  B1 = 0.1955;   B2 = 0.6941;   B3 = 2.522;  B4 = 1.040;
  {коэффициенты для учета флуктуаций зарядового состояния
  более тяжелых ионов при прохождении их через вещество}
  C1 = 1.273e-2; C2 = 3.458e-2; C3 = 0.3931; C4 = 3.812;
var
  E,En,Em,G,SChu,XChu,Gamma:double;
begin
 Stragg:=0;
 if Rfa.Accessible[Z1] and Rfa.Accessible[Z2] then begin
  E:=EE/Rfa[Z1,rf_ATOMS_M2];
  Em:=E/1000.0; {удельная энергия в МэВ/аем}
  {вычисление масштабирующих величин}
  case Z1 of
    1 :  begin
          G:=1.0;
          En:=E;
         end;
    2 :  StopHe(Z2,E,G,En);
    else StopHI(Z1,Z2,E,G,En);
  end;
  {вычисление столкновительного страгглинга}
  SChu:=G/(1.0+Rfa[Z1,rf_STRAG_S1]*Power(Em,Rfa[Z1,rf_STRAG_S2])+
        Rfa[Z1,rf_STRAG_S3]*Power(Em,Rfa[Z1,rf_STRAG_S4]));
  {вычисление зарядообменного страгглинга}
  if Z1 = 1 then begin
   {случай протонов}
   Gamma:=B3*(1.0-exp(-B4*Em));
   XChu:=B1*Gamma/(sqr(Em-B2)+sqr(Gamma));
  end else begin
   {случай более тяжелых ионов}
   En:=Em/(Z1*sqrt(Z1*Z2));
   Gamma:=C3*(1.0-exp(-C4*En));
   XChu:=Power(Z1,1.33333)/Power(Z2,0.33333)*
   C1*Gamma/(sqr(En-C2)+sqr(Gamma));
  end;
  Stragg:=SChu+XChu;
 end;
end;

 {
 Универсальная процедура вычисления сечения торможения для ускоренных
 ионов в веществе. Вычисляются сечения электронного и ядерного торможения.
 Выбор единиц измерения зависит от значения параметра Units.
 Z1,Z2:   атомные номера иона и мишени,
 EE:       энергия ионов, кэв
 Units    = 0 или 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
          = 4        единицы LSS.
 SE,SN:   сечения электронного и ядерного торможения ионов
 }
procedure UniStop(Z1,Z2,Units:integer; EE:double; var SE,SN:double);
var Tmp:double;
begin
 SE:=0;
 SN:=0;
 if Rfa.Accessible[Z1] and Rfa.Accessible[Z2] then begin
  if EE < 1.0e-10 then begin
   SE:=0.0;
   SN:=0.0;
   Exit;
  end;
  {вычисление электронного торможения}
  SE:=EStop(Z1,Z2,EE);
  {вычисление ядерного торможения}
  SN:=NStop(Z1,Z2,EE);
  {выбрать требуемые единицы измерения}
  case Units of
   2 : begin
        SE:=SE*(Nav*1e-24)/Rfa[Z2,rf_ATOMS_M2];
        SN:=SN*(Nav*1e-24)/Rfa[Z2,rf_ATOMS_M2];
       end;
   3 : begin
        SE:=SE*Rfa[Z2,rf_ATOMS_ATRHO]*1.0e-23;
        SN:=SN*Rfa[Z2,rf_ATOMS_ATRHO]*1.0e-23;
       end;
   4 : begin
        Tmp:=(Rfa[Z1,rf_ATOMS_M2]+Rfa[Z2,rf_ATOMS_M2])*sqrt(Power(Z1,0.66667)+
        Power(Z2,0.66667))/(Z1*Z2*Rfa[Z1,rf_ATOMS_M2]*8.462);
        SE:=SE*Tmp;
        SN:=SN*Tmp;
       end;
  end;
 end;
end;

 {
 Используя правило Брэгга, вычислить тормозную способности мишени
 состоящей из нескольких элементов в единицах [эВ/(10**15 атом/кв.см]
 }
function BraggRule(Energy : double;           { энергия пучка [кэВ]          }
                   ZBeam  : integer;          { ион пучка                    }
               var ZTarg  : array of integer; { массив атомов мишени         }
               var DTarg  : array of double;  { отн. атом.доли атомов мишени }
                   NTarg  : integer           { число атомов мишени          }
                   ):double;
var j:integer; SE,SN,StopI:double;
begin
 StopI:=0.0;
 for j:=0 to NTarg-1 do begin
  SE:=EStop(ZBeam,ZTarg[j],Energy);
  SN:=NStop(ZBeam,ZTarg[j],Energy);
  StopI:=StopI+(SE+SN)*DTarg[j];
 end;
 BraggRule:=StopI;
end;

 {
 Вспомогательная процедура вычисляет среднюю атомную массу смеси
 }
function AverageAtomMass(
                var ZTarg  : array of integer; { из каких атомов мишень      }
                var DTarg  : array of double;  { относ.атом.доли элементов   }
                    NTarg  : integer           { число элементов в мишени    }
                   ):double;
var i:integer; M2:double;
begin
 M2:=0;
 for i:=0 to NTarg-1 do begin
  if Rfa.Accessible[ZTarg[i]] then M2:=M2+DTarg[i]*Rfa[ZTarg[i],rf_ATOMS_M2];
 end;
 AverageAtomMass:=M2;
end;

 {
 Коэфф. перевода тормозной способности из эВ/(10**15 атом/кв.см)
 в единицу измерения, зависящую от значения параметра Units.
 M:   массовые числа иона и мишени,
 AtRho:   атомная плотность мишени,
 Units    = 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
 S:       сечения торможения ионов в эВ/(10**15 атом/кв.см)
 }
function TranCoef(M,AtRho:double; Units:integer):double;
begin
 case Units of
  1  : TranCoef:=1;
  2  : TranCoef:=(Nav*1e-24)/M;
  3  : TranCoef:=AtRho*1.0e-23;
  else TranCoef:=1;
 end;
end;

 {
 Вспомогательная процедура вычисляет средний заряд Z2, массу M2, атомную
 плотность AtRho
 }
procedure Z2M2AtRho(
                var ZTarg  : array of integer; { из каких атомов мишень      }
                var DTarg  : array of double;  { относ. доли элем. в мишени  }
                    NTarg  : integer;          { число ат.элементов в мишени }
                    Rho    : double;           { плотность мишени            }
                out Z2,M2,AtRho:double         { искомые параметры мишени    }
                   );
var i:integer;
begin
 Z2:=0;
 M2:=0;
 for i:=0 to NTarg-1 do begin
  if Rfa.Accessible[ZTarg[i]] then begin
   M2:=M2+DTarg[i]*Rfa[ZTarg[i],rf_ATOMS_M2];
   Z2:=Z2+DTarg[i]*ZTarg[i];
  end;
 end;
 AtRho:=Rho*0.6022/M2*1.0e22;
end;

 {
 Вычисление тормозной способности вещества мишени для иона при произвольной
 энергии Energy [кэВ]
 Единицы зависят от величины Units:
 Units    = 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
 ZBeam
 }
function dE_dX(Energy : double;           { энергия пучка [кэВ]             }
               Units  : integer;          { единицы измерения 1..3          }
               ZBeam  : integer;          { из каких ионов пучек            }
           var ZTarg  : array of integer; { из каких атомов мишень          }
           var DTarg  : array of double;  { относ. атом.доли элем. в мишени }
               NTarg  : integer;          { число элементов в мишени        }
               Rho    : double            { плотность мишени                }
               ):double;
var Z2,M2,AtRho:double;
begin
 Z2M2AtRho(ZTarg,DTarg,NTarg,Rho,Z2,M2,AtRho);
 dE_dX:=BraggRule(Energy,ZBeam,ZTarg,DTarg,NTarg)*TranCoef(M2,AtRho,Units);
end;

 {
 Внутренний буфер для функции dX_dE
 }
type
 TdXdETemp=packed record
  ZBeam : integer;
  ZTarg : packed array[0..RfaMaxAtomId-1] of integer;
  DTarg : packed array[0..RfaMaxAtomId-1] of double;
  NTarg : integer;
  RTarg : double;
 end;

 {
 Подинтегральная функция для GetIonTrack
 Вычисление величины, обратной тормозной способности мишени
 [эВ/(10**15 атом/кв.см)] при произвольной энергии E [кэВ]
 }
function dX_dE(E:double; Custom:Pointer):double;
var dEdX:double;
begin
 if Assigned(Custom) then with TdXdETemp(Custom^) do begin
  dEdX:=BraggRule(E,ZBeam,ZTarg,DTarg,NTarg);
  if dEdX<>0 then dX_dE:=1/dEdX else dX_dE:=0;  {обходим деление на 0}
 end else begin
  dX_dE:=0;
 end;
end;

 {
 Вычисление пробега иона от энергии E1 до E2. Энергия - [кэВ], E1 >= E2.
 Единицы зависят от величины Units:
 Units    = 1  эВ/(10**15 атом/кв.см)
          = 2        МэВ/(мг/кв.см)
          = 3        эВ/ангстрем
 }
function GetIonTrack(
               E1,E2  : double;           { интервал энергий,[кэВ],E1>=E2 }
               Units  : integer;          { единицы измерения 1..3        }
               ZBeam  : integer;          { из каких ионов пучек          }
           var ZTarg  : array of integer; { из каких атомов мишень        }
           var DTarg  : array of double;  { относ. ат.доли элем. в мишени }
               NTarg  : integer;          { число элементов в мишени      }
               RTarg  : double            { плотность мишени              }
                    ) : double;
var Z2,M2,AtRho,AbsTol,RelTol,Res,ErrFlag:double; i,NoFun:integer; dXdETemp:^TdXdETemp;
begin
 GetIonTrack:=0;
 E2:=max(E2,0);
 if E1<E2 then exit;
 New(dXdETemp);
 if Assigned(dXdETemp) and (NTarg in [1..RfaMaxAtomId]) then begin
  dXdETemp.ZBeam:=ZBeam;
  for i:=0 to NTarg-1 do dXdETemp.ZTarg[i]:=ZTarg[i];
  for i:=0 to NTarg-1 do dXdETemp.DTarg[i]:=DTarg[i];
  dXdETemp.NTarg:=NTarg;
  dXdETemp.RTarg:=RTarg;
  AbsTol:=0;
  RelTol:=TheRelTol;
  Res:=0; NoFun:=0; ErrFlag:=0;
  Quanc8(dX_dE,E2,E1,AbsTol,RelTol,Res,NoFun,ErrFlag,dXdETemp);
  Res:=Res*1e-3; {так как энергия в кэВ, а dEdX в эВ}
  Z2M2AtRho(ZTarg,DTarg,NTarg,RTarg,Z2,M2,AtRho);
  Res:=Res/TranCoef(M2,AtRho,Units);
  GetIonTrack:=Res;
 end;
 if Assigned(dXdETemp) then Dispose(dXdETemp);
end;

 {
 Функция рассчитывает сечение ионизации для K и L оболочек "сигма эн"
 Отчет 8/5435, 1979, стр.7
 Для справки
  Mn = 1.008982   AEM
  Mp = 1.0075957  AEM  <-> Lambda = 1836.132
  Me = 5.48760E-4 AEM
 }
function RfaWhat_Sn(Energy : double;    { энергия ионизирующего излучения,[кэВ]}
                    Lambda : double;    { отношение массы частицы к массе e    }
                    AtomId : byte;      { идентификатор атома мишени           }
                    LineId : TRfaFields { идентификатор линии мишени           }
                   )       : double;    { сечение ионизации барн=1e-24[кв.см.] }
var u,x,p:double;
begin
 Result:=0;
 if Rfa.Accessible[AtomId] then begin
  if LineId in rf_K_Series then begin
   u:=Rfa[AtomID,rf_EKAB];
   x:=Ln(Energy/(Lambda*u));
   p:=2.0471+x*(-0.0065906+x*(-0.47448+x*(0.099190+x*(0.046063+x*0.0060853))));
   Result:=Exp(p)/sqr(u)*1e4;
  end else
  if LineId in rf_L_Series then begin
   u:=(Rfa[AtomID,rf_EL1AB]+Rfa[AtomID,rf_EL2AB]+2*Rfa[AtomID,rf_EL3AB])/4;
   x:=Ln(Energy/(Lambda*u));
   p:=3.6082+x*(0.37123+x*(-0.36971+x*(-0.78593E-4+x*(0.0025063+x*0.0012613))));
   Result:=Exp(p)/sqr(u)*1e4;
  end;
 end;
end;

 {
 Найти коэффициент флюоресцентного выхода "омега" (фактор эффективности)
 8/5435, 1979, стр.7
 }
function RfaWhat_Omega(AtomId:byte;LineId:TRfaFields; Energy:double):double;
begin
 Result:=0;
 if Rfa.Accessible[AtomId] then begin
  if LineId in rf_K_Series then begin
   if Energy >= Rfa[AtomId,rf_EKAB] then Result:=Rfa[AtomId,rf_KEMISSION];
  end else
  if LineId in rf_L2_Series then begin
   if (Rfa[AtomId,rf_EL2AB] <= Energy) and (Energy < Rfa[AtomId,rf_EL1AB]) then Result:=Rfa[AtomId,rf_L22EMISSION];
   if (Rfa[AtomId,rf_EL1AB] <= Energy) then Result:=Rfa[AtomId,rf_L21EMISSION];
  end else
  if LineId in rf_L3_Series then begin
   if (Rfa[AtomId,rf_EL3AB] <= Energy) and (Energy < Rfa[AtomId,rf_EL2AB]) then Result:=Rfa[AtomId,rf_L33EMISSION];
   if (Rfa[AtomId,rf_EL2AB] <= Energy) and (Energy < Rfa[AtomId,rf_EL1AB]) then Result:=Rfa[AtomId,rf_L32EMISSION];
   if (Rfa[AtomId,rf_EL1AB] <= Energy) then Result:=Rfa[AtomId,rf_L31EMISSION];
  end;
 end;
end;

 {
 Найти относительную интенсивность K характеристической рентгеновской линии
 8/5435, 1979, стр.7
 }
function RfaWhat_K(AtomId : integer;     { атомный номер       }
                   LineId : TRfaFields   { идентификатор линии }
                  ) : double;
begin
 if Rfa.Accessible[AtomId] and (LineId in rf_All_Series)
 then Result:=Rfa[AtomID,Succ(LineId)]
 else Result:=0;
end;

 {
 Найти эффективный коэффициент ослабления для многокомпонентной матрицы.
 Массовые доли получаем пересчетом атомных долей через атомные веса и
 атомные доли.
 Отчет 8/5435, 1979, стр.10
 }
function RfaWhat_Mu(Energy : double;           { энергия пучка, [кэВ]       }
                var ZTarg  : array of integer; { заряды элементов матрицы   }
                var DTarg  : array of double;  { атомные доли элементов     }
                    NTarg  : integer           { число элементов матрицы    }
                   ) : double;                 { в единицах [кв.см./грамм]  }
var i,AtomId:integer;
    AtomDose,AtomMass,MassDose,MassSum,MuSum,CrossSection:double;
begin
 {MassSum-средний атомный вес матрицы}
 MassSum:=0;
 for i:=0 to NTarg-1 do begin
  AtomId:=ZTarg[i];
  if Rfa.Accessible[AtomId] then begin
   AtomDose:=DTarg[i];
   AtomMass:=Rfa[AtomID,rf_ATOMS_M2];
   MassSum:=MassSum+AtomDose*AtomMass;
  end;
 end;
 {MuSum-взвешенная по массовым долям}
 MuSum:=0;
 for i:=0 to NTarg-1 do begin
  AtomId:=ZTarg[i];
  if Rfa.Accessible[AtomId] then begin
   AtomDose:=DTarg[i];
   AtomMass:=Rfa[AtomID,rf_ATOMS_M2];
   MassDose:=AtomDose*AtomMass/MassSum;
   CrossSection:=Rfa.FindCrossSection(AtomId,rf_TOTAL,Energy);
   MuSum:=MuSum+MassDose*CrossSection;
  end;
 end;
 RfaWhat_Mu:=MuSum;
end;

 {
 Внутренний буфер для dGdE
 }
type
 TdGdETemp=record
  AtomId : byte;
  LineId : TRfaFields;
  ZBeam  : byte;
  E2Beam : double;
  E1Beam : double;
  Alfa   : double;
  Beta   : double;
  Thick  : double;
  ZTarg  : array[0..RfaMaxAtomId-1] of integer;
  DTarg  : array[0..RfaMaxAtomId-1] of double;
  NTarg  : integer;
  RTarg  : double;
 end;

 {
 Подинтегральная функция для расчета ХРИ в толстых образцах
 }
function dGdE(Ep:double; Custom:Pointer):double;
var Res,K,Omega,X,Mu,IonTrack,Track,Exponent,SigmaX,SigmaN,dEdX:double;
    Lambda,AtomMass,Coef:double;
    Units:integer;
 function Exp(x:extended):extended;
 begin
  if x<-300 then Exp:=0 else Exp:=system.Exp(x);
 end;
begin
 Res:=0;
 if Assigned(Custom) then with TdGdETemp(Custom^) do begin
  if Rfa.Accessible[AtomId] then begin
   {атомная масса примеси}
   AtomMass:=Rfa[AtomID,rf_ATOMS_M2];
   {торможение в единицах эВ/(10**15 атом/кв.см)}
   Units:=1;
   {коефф для перевода в кэВ*кв.см./г}
   Coef:=1e-18*Nav/AverageAtomMass(ZTarg,DTarg,NTarg);
   {отношение массы иона к массе электрона}
   Lambda:=1836.132;
   {пробег в единицах эВ/(10**15 атом/кв.см)}
   IonTrack:=GetIonTrack(E2Beam,Ep,Units,ZBeam,ZTarg,DTarg,NTarg,RTarg);
   {перевод пробега в см}
   Track:=IonTrack/Coef*1e3;
   {учет угла падения}
   X:=cos(Alfa)*Track;
   {найти коэфф ослабления}
   Mu:=RfaWhat_Mu(Rfa[AtomID,LineId],ZTarg,DTarg,NTarg);
   {найти ослабление}
   Exponent:=Exp(-Mu*X/cos(Beta));
   {учет толщины}
   if X>Thick then Exponent:=0;
   {относит интенсивность ХРИ}
   K:=RfaWhat_K(AtomId,LineId);
   {коэфф флуоресцентного выхода безразмерный}
   Omega:=RfaWhat_Omega(AtomId,LineId,Ep);
   {тормозная способность в единицах эВ/(10**15 атом/кв.см)}
   dEdX:=dE_dX(Ep,Units,ZBeam,ZTarg,DTarg,NTarg,RTarg);
   {перевод в кеВ/(атом/кв.см.)}
   dEdX:=dEdX*Coef;
   {сечение ионизации в кв.см.}
   SigmaN:=RfaWhat_Sn(Ep,Lambda,AtomId,LineId);
   {сечение флюоресценции в кв.см.}
   SigmaX:=SigmaN*Omega*K;
   {готово}
   Res:=SigmaX*Exponent/dEdX/RTarg/AtomMass;
   debugout(stdfDebug,'************');
   debugout(stdfDebug,'ep='+f2s(ep));
   debugout(stdfDebug,'atommass='+f2s(atommass));
   debugout(stdfDebug,'RTarg='+f2s(RTarg));
   debugout(stdfDebug,'coef='+f2s(coef));
   debugout(stdfDebug,'track='+f2s(track));
   debugout(stdfDebug,'x='+f2s(x));
   debugout(stdfDebug,'mu='+f2s(mu));
   debugout(stdfDebug,'-Mu*X/cos(Beta)='+f2s(-Mu*X/cos(Beta)));
   debugout(stdfDebug,'exp='+f2s(exponent));
   debugout(stdfDebug,'sigman='+f2s(sigman));
   debugout(stdfDebug,'k='+f2s(k));
   debugout(stdfDebug,'omega='+f2s(omega));
   debugout(stdfDebug,'sigmax='+f2s(sigmax));
   debugout(stdfDebug,'res='+f2s(res));
  end;
 end;
 dGdE:=Res;
end;

 {
 }
function Rfa_Gamma(
          {описание линии ХРИ для которой ищется гамма}
          AtomId : byte;             { атом для которого ищется выход ХРИ   }
          LineId : TRfaFields;       { линия для которой ищется выход ХРИ   }
          {описание пучка}
          ZBeam  : byte;             { ион пучка                            }
          E2Beam : double;           { энергия пучка                        }
          E1Beam : double;           { нижний предел                        }
          {описание геометрии}
          Alfa   : double;           { угол падения пучка от нормали, град  }
          Beta   : double;           { угол на детектор от нормали, град    }
          Thick  : double;           { толщина мишени                       }
          {описание матрицы мишени}
          ZTarg  : array of integer; { элементы входящие в состав матрицы   }
          DTarg  : array of double;  { относительные доли элементов матрицы }
          NTarg  : integer;          { число элементов в матрице            }
          RTarg  : double            { плотность матрицы                    }
                  ) : double;
var Res,AbsTol,RelTol,Flag:double; i,NoFun:integer; dGdETemp:^TdGdETemp;
begin
 Res:=0; NoFun:=0; Flag:=0;
 New(dGdETemp);
 if Assigned(dGdETemp) and (NTarg in [1..RfaMaxAtomId]) then begin
  dGdETemp.AtomId:=AtomId;
  dGdETemp.LineId:=LineId;
  dGdETemp.ZBeam:=ZBeam;
  dGdETemp.E2Beam:=E2Beam;
  dGdETemp.E1Beam:=E1Beam;
  dGdETemp.Alfa:=DegToRad(Alfa);
  dGdETemp.Beta:=DegToRad(Beta);
  dGdETemp.Thick:=Thick;
  for i:=0 to NTarg-1 do dGdETemp.ZTarg[i]:=ZTarg[i];
  for i:=0 to NTarg-1 do dGdETemp.DTarg[i]:=DTarg[i];
  dGdETemp.NTarg:=NTarg;
  dGdETemp.RTarg:=RTarg;
  AbsTol:=0;
  RelTol:=TheRelTol;
  Quanc8(dGdE,E1Beam,E2Beam,AbsTol,RelTol,Res,NoFun,Flag,dGdETemp);
 end;
 if Assigned(dGdETemp) then Dispose(dGdETemp);
 Rfa_Gamma:=Res; {ткак как сечение в барнах}
end;

{
procedure Test;
const
 z:array[0..1] of integer=(47,79);
 d:array[0..1] of double =(2,2);
begin
 if RfaInit('c:\crwexe\config\rfadata.ini') then begin
  writeln(memavail);
  writeln('dEdX=',dE_dX(10000,1,2,z,d,2,0));
  writeln('Len=',GetIonTrack(10000,0,1,2,z,d,2,0));
  writeln(memavail);
  readln;
  halt;
 end;
end;
begin
 Test;
end;
}

(*
const
 Energy:double=4500; Lambda:double=1836; AtomId:byte=13; LineId:byte=idEKa;
begin
 if not RfaInit('c:\crwexe\config\rfadata.ini') then halt;
 AtomId:=RfaFindAtom('zn');
 LineId:=RfaFindLine('la1');
 writeln('SigmaN=',
         RfaWhat_Sn(Energy,           { энергия ионизирующего излучения kev }
                    Lambda,           { отношение массы частицы к массе e   }
                    AtomId,           { идентификатор атома мишени          }
                    LineId            { идентификатор линии мишени          }
                   ),
         EOL,'Omega=',
         RfaWhat_Omega(AtomId,LineId,Energy)
                   );

 readln;
 halt;
*)

(*
const
 Energy:double=1000; Zbeam=1; atomid=29{13};
 ztarg:array[0..1] of integer=(atomid,0);
 dtarg:array[0..1] of double=(1,0);
 NTarg=1;
 rho:double=0;
 coef:double=0;
begin
 if not RfaInit('c:\crwexe\config\rfadata.ini') then halt;
 rho:=RfaFind(atomid)^.atom_densiti;
 {коефф для перевода в кэВ*кв.см./г}
 Coef:=1e-18*Nav/RfaFind(atomid)^.atoms.m2;
 writeln('rho=',rho,EOL,
         'A=',RfaFind(atomid)^.atoms.m2,EOL,
         'coef=',coef,EOL,
         'dE/dX=',
         dE_dX(Energy,
               2,
               ZBeam,
               ZTarg,
               DTarg,
               NTarg,
               rho
               )*coef,EOL,
          'mu=',
          RfaWhat_Mu(3,ZTarg,DTarg,NTarg),EOL,
          'x=',
          GetIonTrack(
               4500,
               beamemin,
               2,
               ZBeam,
               ZTarg,
               DTarg,
               NTarg,
               rho)

          );
 readln;
 halt;
*)

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_stopion;
begin
end;

procedure Free_crw_stopion;
begin
end;

initialization

 Init_crw_stopion;

finalization

 Free_crw_stopion;

end.

//////////////
// END OF FILE
//////////////

