////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq plugin for demo_dll_sin DAQ config - how to use plugins in DAQ.     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 202411120 - Sample created by A.K.                                         //
////////////////////////////////////////////////////////////////////////////////

library demo_dll_sin_plugin;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
 ////////////////////////////
 // Comment on TDaqDataSheet:
 // Hold all DAQ data there!!
 // Init all on DAQ_CMD_INIT.
 // Free all on DAQ_CMD_FREE.
 // Use data on DAQ_CMD_POLL.
 ////////////////////////////
type ////////////////////////
 TDaqDataSheet = packed record
  tagStart : Integer;
  refOmega : Integer;
  refSinus : Integer;
  devOmega : Integer;
  Sinus    : Double;
  Omega    : Double;
  timeSec  : Double;
  dllName  : LongString;
 end; //// TDaqDataSheet ////
 ////////////////////////////
 procedure InitTag(var tag:Integer; const name:ShortString; typ:Integer);
 begin
  with CrwApi,SysApi,DaqApi do begin
   tag:=findtag(name);
   if (typ>0) and (typetag(tag)<>typ) 
   then RAISE EDaqApi.Create(Format('Could not init tag "%s".',[name]));
  end;
 end;
 procedure InitRef(var ref:Integer; value:Integer);
 begin
  ref:=value;
  if ref=0 then RAISE EDaqApi.Create('Invalid reference.');
 end;
begin
 Result:=0;
 with CrwApi,SysApi,GuiApi,DaqApi do
 try
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAcquisition);
  ///////////////////////////////////////////////////////////////
  // Nota Bene: all driver data must be located in DaqDataSheet!!
  ///////////////////////////////////////////////////////////////
  with TDaqDataSheet(DaqDataSheet(SizeOf(TDaqDataSheet))^) do begin
   case DaqCommand of
    ////////////////////
    // Initialize driver
    ////////////////////
    DAQ_CMD_INIT: begin
     RedirectStdIn(Input);
     RedirectStdOut(Output);
     //////////////////////////////////////////////////
     // Initialize USER data located in DaqDataSheet //
     //////////////////////////////////////////////////
     dllName:=ExtractFileNameExt(PluginBinaryFileName);
     GuiApi.Echo(Format('Starting %s …',[dllName]));
     InitTag(tagStart, 'START', 1);
     InitRef(refOmega, refAI(0));
     InitRef(refSinus, refAI(1));
     InitRef(devOmega, FindObjectRef('Device','&Omega'));
     iSetTag(tagStart,0);
     CurveLength[refSinus]:=0;
     CurveLength[refOmega]:=0;
     //////////////////
     GuiApi.Echo('Ok');
    end;
    //////////////////
    // Finalize driver
    //////////////////
    DAQ_CMD_FREE: begin
     GuiApi.Echo(Format('Stopping %s …',[dllName]));
     iSetTag(tagStart,0);
     //////////////////
     // Don`t forget to
     // Clear all data:
     //////////////////
     tagStart:=0;
     refOmega:=0;
     refSinus:=0;
     devOmega:=0;
     dllName:='';
     //////////////////
     // Free DataSheet:
     //////////////////
     DaqDataSheet(0);//
     //////////////////
     GuiApi.Echo('Ok');
    end;
    ////////////////////
    // Main polling loop
    ////////////////////
    DAQ_CMD_POLL: begin
     /////////////////////////////
     // USER data processing code:
     /////////////////////////////
     if igettag(tagStart)>0 then begin
      Omega:=CurveLastPoint[refOmega].Y;
      timeSec:=time*timeunits*1e-3;
      Sinus:=sin(Omega*timeSec);
      putao(0,time,Sinus);
      writeln(Format('sin(%11.4f)=%10.6f',[getai_xn(1),getai_yn(1)]));
      Echo(Format('RunCount = %g',[RunCount]));
     end;
     if (clickbutton=VK_LBUTTON)  then begin
      if clicksensor='START' then begin
       isettag(tagSTART,ord(igettag(tagSTART)=0));
       voice('Чик');
      end;
      if clicksensor='SINUS' then begin
       winselect('DEMO_DLL_SIN_PLOT');
       voice('Чик');
      end;
      if clicksensor='OMEGA' then begin
       DeviceAction(devOmega);
       voice('Чик');
      end;
      if clicksensor='WDT' then begin
       voice('Чик');
       sleep(10000);
      end;
     end;
     if (clickbutton=VK_RBUTTON)  then begin
      if clicksensor='SINUS' then begin
       winhide('DEMO_DLL_SIN_PLOT');
       voice('Чик');
      end;
     end;
     /////////////////////////////
     // USER data processing done.
     /////////////////////////////
    end;
   end; // case
  end; // with
 except
  on E:Exception do begin
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
