unit mcSaturationVaporPressure;
{
  MicroCline SaturationVaporPressure Component

  This is an adaption of Holger Voemel's vp program
    Holger.Voemel@Colorado.edu

  Made from code in
    http://cires.colorado.edu/~voemel/vp.html
    http://cires.colorado.edu/~voemel/vp.pro
      25 saturation vapor pressure equations

  local copies
    C:\Delphi\ScienceComponents\mcWaterVapor\Help\vp.html
    C:\Delphi\ScienceComponents\mcWaterVapor\Help\vp_pro.txt

  Vapor pressure over liquid water below 0C
  Vapor pressure over ice

  At 21C, Fukuta goes negative (is worthless), calibrated from -20C to -35C
  The rest appear to be reasonable above 0C

  Delphi does not have an exponent operator. Instead
     T^3 -> IntPower(T_K,3)

     Psat =  ->  Psat :=
     delete the dollar signs (line continuation characters)
     remove decimal points after integers (in Fortran, this make the constant real)
     T    -> T_K
     temp -> T_C
     10.^()  ->  power(10, ())

  Written by Robert Clemenzi 05-29-2012 (c)
  05-29-12 Start of file

  06-02-12 Added code for Seinfeld_Pandis and GISS_AR4
  06-11-12 Added string to enum conversion
  02-25-13 Added routine to populate a stringlist (comboBox) - mc_VaporPressure_Algorithms
  02-26-13 Renamed from mcVaporPressure_Vomel and made it a class
           Stopped vaious exceptions at low temperatures
            by limiting lowest temperature to 150 K
            my test form - Saturation_vapor_pressure_frm - limits this to 130 K
  05-05-14 Added Antoine and August equations - http://en.wikipedia.org/wiki/Antoine_equation
}

interface

uses classes, stdctrls;

type
    // equation numbers are from http://cires.colorado.edu/~voemel/vp.html
    // 5 equations are from the program, but not the reference
  Tvp_Formulas = (
    vpf_MartiMauersberger,    //            18 ice
    vpf_MagnusTetens,         //  8 liquid  15 ice
    vpf_GoffGratch,           //  1 liquid  12 ice
    vpf_Buck_original,        //  6 liquid  17 ice
    vpf_Buck_manual,          //  5 liquid  16 ice  Buck Research Manuals, 1996
    vpf_CIMO,                 //  2 liquid  14 ice
    vpf_Vaisala,              //  just a name, no formulas provided
    vpf_WMO_Goff,             //  3 liquid     ice in program
    vpf_WMO2000,              //  3 liquid  has an error
    vpf_Wexler,               //    program liquid
    vpf_Sonntag,              //  7 liquid     ice in program
    vpf_Bolton,               //  9 liquid
    vpf_Fukuta,               //    program liquid
    vpf_HylandWexler,         //  4 liquid  13 ice
    vpf_IAPWS,                // 11 liquid
    vpf_Preining,             //    program liquid
    vpf_MurphyKoop,           // 10 liquid  19 ice
    vpf_Seinfeld_Pandis,
    vpf_GISS_AR4,
    vpf_Antoine_equation,
    vpf_August_equation,
    vpf_Unknown
   );

  Tvp_Phase = (vp_liquid, vp_ice);

  // I decided to convert this to a class because the number of subroutines
  // was beginning to increase
{
function mc_VaporPressure(Temperature_K: Real; Formula: string; Phase:Tvp_Phase): extended;
function mc_VaporPressure_Voemel(Temperature_K: Real; Formula: Tvp_Formulas; Phase:Tvp_Phase): extended;
function mc_VaporPressure_ConvertString(Formula: string; Phase:Tvp_Phase=vp_liquid): Tvp_Formulas;
}
type T_mc_SaturationVaporPressure = class(TObject)
  private
  protected
  public
    constructor Create; virtual; // not needed
    destructor Destroy; override;

    procedure Get_Algorithms(Phase:Tvp_Phase; Strings: TStrings);
    procedure Configure_ComboBox(Phase:Tvp_Phase; ComboBox: TComboBox);
    function  VaporPressure(Temperature_K: Real; Formula: Tvp_Formulas; Phase:Tvp_Phase): extended; overload;
    function  VaporPressure(Temperature_K: Real; Formula: string; Phase:Tvp_Phase): extended; overload;
    function  VaporPressure(Temperature_K: Real; Formula: string; Phase: string = 'liquid'): extended; overload;

  end;

function mc_SaturationVaporPressure : T_mc_SaturationVaporPressure;

implementation

uses sysutils, math;

var Fmc_SaturationVaporPressure : T_mc_SaturationVaporPressure;

function  mc_SaturationVaporPressure : T_mc_SaturationVaporPressure;
begin
  if Fmc_SaturationVaporPressure=nil then begin
    Fmc_SaturationVaporPressure := T_mc_SaturationVaporPressure.Create;
  end;

  result := Fmc_SaturationVaporPressure;
end;

procedure T_mc_SaturationVaporPressure.Configure_ComboBox(Phase: Tvp_Phase;
  ComboBox: TComboBox);
var
  i: integer;
  str:string;
begin
  str := ComboBox.Text; // save the current selection

  // This loads new values with nothing selected
  Get_Algorithms(Phase, ComboBox.Items);

  // If the original selection is still available, select it
  // Otherwise, select the first item
  i := ComboBox.Items.IndexOf(str);
  if i = -1 then i := 0; // in case it is not found
  ComboBox.ItemIndex := i;
end;

constructor T_mc_SaturationVaporPressure.Create;
begin

end;

destructor T_mc_SaturationVaporPressure.Destroy;
begin
  inherited;

end;

function ALOG(x:extended):extended;
// This is just the Fortran name for ln(x)
// 'A' was used to make the result a "real" value
// because functions starting with 'i' thru 'n' return intergers
// Defining this means I don't have to change the code
begin
  result := ln(x);
end;

function ALOG10(x:extended):extended;
// This is just the Fortran base-10 log
// 'A' was used to make the result a "real" value
// because functions starting with 'i' thru 'n' return intergers
// Defining this means I don't have to change the code
begin
  result := log10(x);
end;

function mc_VaporPressure_Voemel(Temperature_K: Real; Formula: Tvp_Formulas; Phase:Tvp_Phase): extended;
var
  Psat     : extended;
  T_K, T_C : extended;
  Ts       : extended; // steam point temperature in K
  ews      : extended; // saturation pressure at steam point temperature, normal atmosphere

  // used for IAPWS
  Tc : extended; // K   :    Temperature at the critical point
  Pc : extended; // hPa : Vapor pressure at the critical point
  nu,a1,a2,a3,a4,a5,a6 : extended;

  x: extended; // vpf_Fukuta

  ei0, T0: extended; // vpf_GoffGratch, vpf_IAPWS

  a: extended; // vpf_Seinfeld_Pandis

begin
    // Without this escape, many of the formulas generate exceptions
    // Only one is calibrated to -35C over water
    // One may be good to -100C (173 K) over ice
    // At 150 K, the larget result is 8E-5 over water, 6E-8 over ice
  if Temperature_K < 150  then begin
    result := 0; // close enough
    exit;
  end;

  T_K := Temperature_K; // Most formulas use T in [K]
  T_C := T_K - 273.15 ; // For formulas using [C]
 try
  if Phase = vp_liquid then begin
    // Calculate saturation pressure over liquid water
    case Formula of

      vpf_MagnusTetens: begin
        // Source: Murray, F. W., On the computation of saturation vapor pressure,
        // J. Appl. Meteorol., 6, 203-204, 1967.
        // Psat = 10.^(7.5*(Temp)/(Temp+237.5) + 0.7858)         ; // Murray quotes this as the original formula and
         Psat := 6.1078 * EXP(17.269388 * (T_K-273.16) / (T_K-35.86)) ; // this as the mathematical equivalent in the form of base e.
        end;

      vpf_GoffGratch: begin
        // Goff Gratch formulation
        // Source : Smithsonian Meteorological Tables, 5th edition, p. 350, 1984
        // From original source: Goff and Gratch (1946), p. 107.
         Ts    := 373.16       ; // steam point temperature in K
         ews   := 1013.246     ; // saturation pressure at steam point temperature, normal atmosphere

         Psat := power(10, (- 7.90298*(Ts/T_K-1) + 5.02808 * ALOG10(Ts/T_K)
                     - 1.3816E-7 * (power(10, (11.344*(1-T_K/Ts)))-1)
                     + 8.1328E-3*(power(10, (-3.49149*(Ts/T_K-1))) -1) + ALOG10(ews))
                     )
        end;

      vpf_Buck_original: begin
        // Buck's vapor pressure formulation based on Tetens formula
	// Source: Buck, A. L., New equations for computing vapor pressure and
        // enhancement factor, J. Appl. Meteorol., 20, 1527-1532, 1981.
         Psat := 6.1121 * EXP(17.502 * T_C / (240.97+T_C))
        end;

      vpf_Buck_manual: begin
        // Buck's vapor pressure formulation based on Tetens formula
      	// Source: Buck Research, Model CR-1A Hygrometer Operating Manual, Sep 2001
         Psat := 6.1121 * EXP((18.678 - (T_C) / 234.5) * (T_C) / (257.14+T_C))
        end;

      vpf_CIMO: begin
        // Source: Annex 4B, Guide to Meteorological Instruments and
        // Methods of Observation, WMO Publication No 8, 7th edition,
        // Geneva, 2008. (CIMO Guide)
         Psat := 6.112 * EXP(17.62 * T_C/(243.12 + T_C))
        end;

      vpf_WMO_Goff: begin
        // Intended WMO formulation, originally published by Goff (1957)
        // incorrectly referenced by WMO technical regulations, WMO-NO 49,
        // Vol I, General Meteorological Standards and Recommended Practices,
        // App. A, Corrigendum Aug 2000. and incorrectly referenced by
        // WMO technical regulations, WMO-NO 49, Vol I,
        // General Meteorological Standards and Recommended Practices, App. A, 1988.

        // this formula is wrong - it contains a typographical error in the original

        Ts   := 273.16       ; // triple point temperature in K

        Psat := power(10, (10.79574*(1-Ts/T_K)
                   - 5.02800 * ALOG10(T_K/Ts)
                   + 1.50475E-4 * (1-power(10, (-8.2969*(T_K/Ts-1))))
                   + 0.42873E-3 * (power(10, (+4.76955*(1-Ts/T_K)))-1)
                   + 0.78614)
                   )
        end;

      vpf_WMO2000: begin
        // WMO formulation, which is very similar to Goff Gratch
        // Source : WMO technical regulations, WMO-NO 49, Vol I,
        // General Meteorological Standards and Recommended Practices,
        // App. A, Corrigendum Aug 2000.
        //  10.^(-4.76955 vs 10.^(+4.76955    the +4 is wrong

         Ts   := 273.16       ; // triple point temperature in K

         Psat := power(10, (10.79574*(1-Ts/T_K)
                     - 5.02800 * ALOG10(T_K/Ts)
                     + 1.50475E-4 * (1-power(10, (-8.2969*(T_K/Ts-1))))
                     + 0.42873E-3 * (power(10, (-4.76955*(1-Ts/T_K)))-1)
                     + 0.78614)
                     )
        end;

      vpf_Wexler: begin
        // Wexler, A., Vapor pressure formulation for ice,
        // Journal of Research of the National Bureau of Standards-A.
        // 81A, 5-20, 1977.
        Psat := EXP( -2.9912729E3   / IntPower(T_K,2)// T_K^-2
                    - 6.0170128E3   / T_K            // T_K^-1
                    + 1.887643854E1                  // T_K^0
                    - 2.8354721E-2  * T_K
                    + 1.7838301E-5  * IntPower(T_K,2)
                    - 8.4150417E-10 * IntPower(T_K,3)
                    - 4.4412543E-13 * IntPower(T_K,4)
                    + 2.858487 * ALOG(T_K)) / 100
        end;

      vpf_Sonntag: begin
        // Source: Sonntag, D., Advancements in the field of hygrometry,
        // Meteorol. Z., N. F., 3, 51-66, 1994.
        Psat := EXP( -6096.9385  / T_K            // T_K^-1
                   + 16.635794
                   - 2.711193E-2 * T_K
                   + 1.673952E-5 * IntPower(T_K,2)
                   + 2.433502 * ALOG(T_K))
        end;

      vpf_Bolton: begin
        // Source: Bolton, D., The computation of equivalent potential temperature,
        // Monthly Weather Report, 108, 1046-1053, 1980. equation (10)
         Psat := 6.112 * EXP(17.67 * T_C / (T_C+243.5))
        end;

      vpf_Fukuta: begin
        // Source: Fukuta, N. and C. M. Gramada,
        // Vapor pressure measurement of supercooled water,
        // J. Atmos. Sci., 60, 1871-1875, 2003.

        // This paper does not give a vapor pressure formulation,
        // but rather a correction over the Smithsonian Tables.
        // Thus calculate the table value first, then use the correciton
        // to get to the measured value.

        // The calibration is from -20C to -35C - I checked the paper

         Ts    := 373.16   ;// steam point temperature in K
         ews   := 1013.246 ;// saturation pressure at steam point temperature, normal atmosphere

         Psat := power(10, (-7.90298*(Ts/T_K-1) + 5.02808 * ALOG10(Ts/T_K)
                     - 1.3816E-7 * (power(10, (11.344*(1-T_K/Ts)))-1)
                     + 8.1328E-3*(power(10, (-3.49149*(Ts/T_K-1))) -1) + ALOG10(ews))
                     ) ;

         x:= T_C + 19 ;
         Psat := Psat * (0.9992
                       + 7.113E-4*x
                       - 1.847E-4*intpower(x,2)
                       + 1.189E-5*intpower(x,3)
                       + 1.130E-7*intpower(x,4)
                       - 1.743E-8*intpower(x,5))

         // I have no clue what the following means - unrecognized syntax
         // the write up says that the formula is valid only down to -39C
         // Therefore, I think this must set an out of range value
        // s = WHERE(Temp LT -39)
        // Psat(s) = 9.99E99
        end;

     // Marti and Mauersberger don't have a vapor pressure curve over liquid.
//    vpf_HylandWexler, vpf_MartiMauersberger: begin
      vpf_HylandWexler: begin
         // Source Hyland, R. W. and A. Wexler, Formulations
         // for the Thermodynamic Properties of the saturated Phases of H2O
         // from 173.15K to 473.15K, ASHRAE Trans, 89(2A), 500-519, 1983.
         Psat := EXP(-0.58002206E4 / T_K
                    + 0.13914993E1
                    - 0.48640239E-1 * T_K
                    + 0.41764768E-4 * IntPower(T_K,2)
                    - 0.14452093E-7 * IntPower(T_K,3)
                    + 0.65459673E1  * ALOG(T_K)) / 100
        end;

      vpf_IAPWS: begin
        // Source: Wagner W. and A. Pru? (2002),
        // The IAPWS formulation 1995 for the thermodynamic properties
        // of ordinary water substance for general and scientific use,
        // J. Phys. Chem. Ref. Data, 31(2), 387-535.

        // This is the 'official' formulation from the
        // International Association for the Properties of Water and Steam
        // The valid range of this formulation is 273.16 <= T <= 647.096 K
        // and is based on the ITS90 temperature scale.

         Tc:= 647.096  ; // K   : Temperature at the critical point
         Pc:= 22.064E4 ; // hPa : Vapor pressure at the critical point
         nu:= (1-T_K/Tc)    ;
         a1:=  -7.85951783  ;
         a2:=   1.84408259  ;
         a3:= -11.7866497   ;
         a4:=  22.6807411   ;
         a5:= -15.9618719   ;
         a6:=   1.80122502  ;
         Psat := Pc * EXP(Tc/T_K *
            (a1*nu
           + a2*power(nu,1.5 )
           + a3*power(nu,3.  )
           + a4*power(nu,3.5 )
           + a5*power(nu,4.  )
           + a6*power(nu,7.5 )
           ))
        end;

      vpf_Preining: begin
         // Source : Vehkamaeki, H., M. Kulmala, I. Napari, K. E. J. Lehtinen,
         // C.Timmreck, M. Noppel, and A. Laaksonen (2002), J. Geophys.
         // Res., 107, doi:10.1029/2002JD002184.
        Psat := EXP(-7235.424651/T_K
                   + 77.34491296
                   + 5.7113E-3 * T_K
                   - 8.2 * ALOG(T_K)) / 100
        end;

      vpf_MurphyKoop: begin
        // Source : Murphy and Koop, Review of the vapour pressure of ice
        // and supercooled water for atmospheric applications,
        // Q. J. R. Meteorol. Soc (2005), 131, pp. 1539-1565.

        Psat := exp(54.842763 - 6763.22 / T_K - 4.210 * ALOG(T_K)
         + 0.000367 * T_K
         + TANH(0.0415 * (T_K - 218.8)) * (53.878 - 1331.22 / T_K - 9.44523 * ALOG(T_K) + 0.014025 * T_K)) / 100.
        end;

      vpf_GISS_AR4: begin
        // Source : GISS Model-E for IPCC AR4
        // Latent_heat = 2.5E6 water
        //  6.108 mbar * 0.62197 (mwat/mair) = 3.797915
    //   Psat := 3.797915 * exp(2.5E6 * (7.93252E-6 - 2.166847E-3/T_K))
         Psat := 6.108 * exp(2.5E6 * (7.93252E-6 - 2.166847E-3/T_K))
        end;

      vpf_Seinfeld_Pandis: begin
        // Source : Atmospheric Chemistry and Physics by Seinfeld and Pandis
         a    := 1 - (373.15/T_K);
         Psat := 1013.25 * exp((13.3185 - (1.97 + (0.6445 + (0.1299*a))*a)*a)*a);
        end;


      vpf_Antoine_equation: begin
        // Source : http://en.wikipedia.org/wiki/Antoine_equation
         Psat := 1013.25/760 * power(10,(8.07131 - 1730.63/(233.426 + T_C)));
        end;

      vpf_August_equation: begin
        // Source : http://en.wikipedia.org/wiki/Antoine_equation
        // Source : Physics Related to Anesthesia
        //            By John D. Current, M.D.
        //            http://books.google.com/books?id=RgcpOQ444vgC&pg=PA184&lpg=PA184
        //              the coefficients are on page 189
        //            http://en.wikipedia.org/wiki/Vapour_pressure_of_water   05-05-14

         Psat := 1013.25/760 * exp(20.386 - 5132 / T_K);
        end;


{*    vpf_Vaisala: begin  // not in the reference program
        end;

      vpf_MartiMauersberger: begin
        end;

*}
    else
      Psat := 0; // should never be called
    end;

  end else begin
    // Calculate saturation pressure over ice
    case Formula of

      vpf_MartiMauersberger: begin
        // Source : Marti, J. and K Mauersberger,
        // A survey and new measurements of ice vapor pressure
        // at temperatures between 170 and 250 K, GRL 20, 363-366, 1993.
         Psat := power(10,(-2663.5 / T_K + 12.537)) / 100.
        end;

      vpf_MagnusTetens: begin
        // Source: Murray, F. W., On the computation of saturation vapor pressure,
        // J. Appl. Meteorol., 6, 203-204, 1967.

        // Murray quotes this as the original formula 
        // Psat := 10.^(9.5 * Temp/(265.5+Temp) + 0.7858) ;

        // and this as the mathematical aquivalent in the form of base e.
         Psat := 6.1078 * EXP(21.8745584 * (T_K-273.16) / (T_K-7.66)) ;
        end;

        // This is the suggested standard / default
        // IAPWS does not provide a vapor pressure formulation over ice
//    vpf_GoffGratch, vpf_IAPWS: begin
      vpf_GoffGratch: begin
        // Source : Smithsonian Meteorological Tables, 5th edition, p. 350, 1984

         ei0   := 6.1071       ; // mbar
         T0    := 273.16       ; // triple point in K

         Psat := power(10, (-9.09718 * (T0/T_K - 1)
                  - 3.56654 * ALOG10(T0/T_K)
                  + 0.876793 * (1. - T_K/T0) + ALOG10(ei0))
                     )
        end;

      vpf_Buck_original: begin
        // Buck's vapor pressure formulation based on Tetens formula
        // Source: Buck, A. L., New equations for computing vapor pressure
        // and enhancement factor, J. Appl. Meteorol., 20, 1527-1532, 1981.
         Psat := 6.1115 * EXP(22.452 * T_C / (272.55+T_C))
        end;

      vpf_Buck_manual: begin
        // Buck's vapor pressure formulation based on Tetens formula
        // Source: Buck Research, Model CR-1A Hygrometer Operating Manual, Sep 2001
         Psat := 6.1115 * EXP((23.036 - T_C / 333.7) * T_C / (279.82+T_C))
        end;

      vpf_CIMO: begin
        // Source: Annex 4B, Guide to Meteorological Instruments and Methods
        // of Observation,
        // WMO Publication No 8, 7th edition, Geneva, 2008. (CIMO Guide)
         Psat := 6.112 * EXP(22.46 * T_C/(272.62 + T_C))
        end;

//    vpf_WMO_Goff, vpf_WMO2000: begin
      vpf_WMO_Goff: begin
        // these are different over liquid water, but the same over ice

        // WMO formulation, which is very similar to Goff Gratch
        // Source : WMO technical regulations, WMO-NO 49, Vol I,
        // General Meteorological Standards and Recommended Practices,
        // Aug 2000, App. A.

         T0   := 273.16       ; // triple point temperature in K

         Psat := power(10, (-9.09685 * (T0/T_K - 1) - 3.56654 * ALOG10(T0/T_K)
                     + 0.87682 * (1 - T_K/T0) + 0.78614)
                     )
        end;

      vpf_Sonntag: begin
        // Source: Sonntag, D., Advancements in the field of hygrometry,
        // Meteorol. Z., N. F., 3, 51-66, 1994.
         Psat := EXP( -6024.5282 / T_K
	             + 24.721994
		     + 1.0613868E-2 * T_K
		     - 1.3198825E-5 * IntPower(T_K,2)
		     - 0.49382577 * ALOG(T_K))
        end;

      vpf_HylandWexler: begin
        // Source Hyland, R. W. and A. Wexler, Formulations for
        // the Thermodynamic Properties of the saturated Phases
        // of H2O from 173.15K to 473.15K, ASHRAE Trans, 89(2A), 500-519, 1983.
         Psat := EXP( -0.56745359E4 / T_K
                     + 0.63925247E1
                     - 0.96778430E-2  * T_K
                     + 0.62215701E-6  * IntPower(T_K,2)
                     + 0.20747825E-8  * IntPower(T_K,3)
                     - 0.94840240E-12 * IntPower(T_K,4)
                     + 0.41635019E1 * ALOG(T_K)) / 100.
        end;

      vpf_MurphyKoop: begin
        // Source : Murphy and Koop, Review of the vapour pressure of ice
        // and supercooled water for atmospheric applications,
        // Q. J. R. Meteorol. Soc (2005), 131, pp. 1539-1565.
         Psat := exp(9.550426 - 5723.265/T_K + 3.53068 * ALOG(T_K) - 0.00728332 * T_K) / 100
        end;

      vpf_GISS_AR4: begin
        // Source : GISS Model-E for IPCC AR4
        // Latent_heat = 2.834E6 ice
        //  6.108 mbar * 0.62197 (mwat/mair) = 3.797915
    //   Psat := 3.797915 * exp(2.834E6*(7.93252E-6 - 2.166847E-3/T_K))
         Psat := 6.108 * exp(2.834E6*(7.93252E-6 - 2.166847E-3/T_K))
        end;

{*        these are not in the reference program
      vpf_WMO2000
      vpf_Vaisala: begin
        end;

      vpf_Wexler: begin
        end;

      vpf_Bolton: begin
        end;

      vpf_Fukuta: begin
        end;

      vpf_Preining: begin
        end;
*}

    else
      Psat := 0; // should never be called
    end;

    // At this point, the referenced code checks the temperature and uses the
    // GoffGratch (water) for temperatures above freezing.

    // I have decided not to add that
    // If I was going to implement that, I would modify the code
    // to NOT repeat a formula already used above

  end;
 except
   on EInvalidOp do Psat := -99; // error
   on EOverflow  do Psat := -99; // error - vpf_WMO2000 with T_k < 140k
 end;

  result := Psat;
end;

function mc_VaporPressure_ConvertString(Formula: string; Phase:Tvp_Phase=vp_liquid): Tvp_Formulas;
// this converts the strings to constants, it makes comboboxes easier to implement
var
  vpf_Formula : Tvp_Formulas;
begin
          if  Formula='MartiMauersberger' then vpf_Formula := vpf_MartiMauersberger
     else if  Formula='MagnusTetens'      then vpf_Formula := vpf_MagnusTetens
     else if  Formula='GoffGratch'        then vpf_Formula := vpf_GoffGratch
     else if  Formula='Buck_original'     then vpf_Formula := vpf_Buck_original
     else if  Formula='Buck_manual'       then vpf_Formula := vpf_Buck_manual
     else if  Formula='CIMO'              then vpf_Formula := vpf_CIMO
     else if  Formula='WMO_Goff'          then vpf_Formula := vpf_WMO_Goff
     else if  Formula='WMO2000'           then vpf_Formula := vpf_WMO2000
     else if  Formula='Wexler'            then vpf_Formula := vpf_Wexler
     else if  Formula='Sonntag'           then vpf_Formula := vpf_Sonntag
     else if  Formula='Bolton'            then vpf_Formula := vpf_Bolton
     else if  Formula='Fukuta'            then vpf_Formula := vpf_Fukuta
     else if  Formula='HylandWexler'      then vpf_Formula := vpf_HylandWexler
     else if  Formula='IAPWS'             then vpf_Formula := vpf_IAPWS
     else if  Formula='Preining'          then vpf_Formula := vpf_Preining
     else if  Formula='MurphyKoop'        then vpf_Formula := vpf_MurphyKoop
     else if  Formula='Seinfeld_Pandis'   then vpf_Formula := vpf_Seinfeld_Pandis
     else if  Formula='GISS_AR4'          then vpf_Formula := vpf_GISS_AR4
     else if  Formula='Antoine equation'  then vpf_Formula := vpf_Antoine_equation
     else if  Formula='August equation'   then vpf_Formula := vpf_August_equation

     else                                      vpf_Formula := vpf_Unknown;

  if Phase = vp_liquid then begin
    // No formula for saturation pressure over liquid water
          if  Formula='MartiMauersberger' then vpf_Formula := vpf_Unknown;

  end else begin
    // No formula for saturation pressure over ice

          if  Formula='WMO2000'           then vpf_Formula := vpf_Unknown
     else if  Formula='Wexler'            then vpf_Formula := vpf_Unknown
     else if  Formula='Bolton'            then vpf_Formula := vpf_Unknown
     else if  Formula='Fukuta'            then vpf_Formula := vpf_Unknown
     else if  Formula='IAPWS'             then vpf_Formula := vpf_Unknown
     else if  Formula='Preining'          then vpf_Formula := vpf_Unknown
     else if  Formula='Seinfeld_Pandis'   then vpf_Formula := vpf_Unknown
     else if  Formula='Antoine equation'  then vpf_Formula := vpf_Unknown
     else if  Formula='August equation'   then vpf_Formula := vpf_Unknown
      ;
  end;

  result := vpf_Formula;
end;

{ T_mc_SaturationVaporPressure }

procedure T_mc_SaturationVaporPressure.Get_Algorithms(Phase: Tvp_Phase;
  Strings: TStrings);
begin
  if Phase = vp_liquid then begin
    Strings.CommaText := 'GoffGratch,CIMO,HylandWexler,'
      + 'MagnusTetens,Buck_original,Buck_manual,WMO_Goff,WMO2000,'
      + 'Wexler,Sonntag,Bolton,Fukuta,IAPWS,Preining,MurphyKoop,'
      + 'Seinfeld_Pandis,GISS_AR4,"Antoine equation","August equation"';

  end else begin
    Strings.CommaText := 'GoffGratch,CIMO,HylandWexler,'
      + 'MartiMauersberger,MagnusTetens,Buck_original,Buck_manual,'
      + 'WMO_Goff,Sonntag,MurphyKoop,GISS_AR4';

  end;
end;

function T_mc_SaturationVaporPressure.VaporPressure(Temperature_K: Real;
  Formula: string; Phase: Tvp_Phase): extended;
// this uses strings to make comboboxes easier to implement
var
  vpf_Formula : Tvp_Formulas;
begin
  vpf_Formula := mc_VaporPressure_ConvertString(Formula, Phase);
  result      := mc_VaporPressure_Voemel(Temperature_K, vpf_Formula, Phase);
end;

function T_mc_SaturationVaporPressure.VaporPressure(Temperature_K: Real;
  Formula: string; Phase: string = 'liquid'): extended;
begin
  if  SameText(Phase, 'ice') then
    result := VaporPressure(Temperature_K, Formula, vp_ice)
  else
    result := VaporPressure(Temperature_K, Formula, vp_liquid);
end;

function T_mc_SaturationVaporPressure.VaporPressure(Temperature_K: Real;
  Formula: Tvp_Formulas; Phase: Tvp_Phase): extended;
begin
  result := mc_VaporPressure_Voemel(Temperature_K, Formula, Phase);
end;

initialization  // required so that *finalization* can be defined

finalization
  Fmc_SaturationVaporPressure.Free;

end.

