////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// DAQ System base.                                                           //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
// 20240602 - OutFifoCleanDelay                                               //
// 20240618 - OutFifoCleanRunCount                                            //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqsys; // DAQ System base

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow, Form_SpectrWindow,
 Form_Calculator, Form_ListBoxSelection, Form_UartTerminal,
 Form_CalibDialog, Form_DaqPoint2DEdit,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_daqtags, _crw_ef, _crw_curves, _crw_proc,
 _crw_pio, _crw_couple, _crw_riff, _crw_polling,
 _crw_calib, _crw_task, _crw_syscal, _crw_hl,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sesman,
 _crw_sect, _crw_flgrd, _crw_syslog,
 _crw_appforms, _crw_apptools, _crw_apputils;

const
 DefPreProcessorTimeOut   = 60; // Sec
 DefPostProcessorTimeOut  = 60; // Sec
 MinPreProcessorTimeOut   = 15; // Sec
 MinPostProcessorTimeOut  = 15; // Sec
 DefPreProcessorPipeSize  = 64; // kb
 DefPostProcessorPipeSize = 64; // kb
 MinPreProcessorPipeSize  = 16; // kb
 MinPostProcessorPipeSize = 16; // kb

type
 TPreProcessorRec = packed record
  CmdLine         : PureString;
  TimeOut         : Integer;
  Display         : Integer;
  PipeSize        : Integer;
  Priority        : PureString;
 end;
 TPostProcessorRec = TPreProcessorRec;
 
type
 TDaqSystem = class(TMasterObject)
 private
  myCurves        : TCurveList;
  myPolling       : TPolling;
  myWatchdog      : TPolling;
  myDispatcher    : TPolling;
  myTimer         : TIntervalTimer;
  myAcqTimer      : TIntervalTimer;
  mySaveTimer     : TIntervalTimer;
  myHistTimer     : TIntervalTimer;
  mySpecTimer     : TIntervalTimer;
  myCurWinList    : TCurveWindowList;
  myTabWinList    : TTabWindowList;
  myCirWinList    : TCircuitWindowList;
  mySpeWinList    : TSpectrWindowList;
  myConsoleFifo   : TFifo;
  myWarningList   : TText;
  myHistList      : TText;
  myConfigFile    : LongString;
  mySitePath      : LongString;
  myDataPath      : LongString;
  myDataFile      : LongString;
  myBackupFile    : LongString;
  myLogFile       : LongString;
  myHelpFile      : LongString;
  myMergeFile     : LongString;
  myConsoleFile   : LongString;
  myDaqDataWindow : TFormCurveWindow;
  myLogBook       : TFormTextEditor;
  myHistLimits    : packed array[0..2] of LongInt;
  mySoundOn       : Boolean;
  myVoiceEcho     : Boolean;
  myConsoleMode   : Word;
  mySaveOnStop    : Boolean;
  myAutoStart     : Boolean;
  myLogoDelay     : Integer;
  myDaqDesktop    : LongString;
  myNavigator     : LongString;
  myStopConfirm   : LongString;
  myErrMsgTable   : packed array[Byte] of LongString;
  myCurWinPeriod  : LongInt;
  myTabWinPeriod  : LongInt;
  myCirWinPeriod  : LongInt;
  mySpeWinPeriod  : LongInt;
  myFixPriority   : packed record Wanted,Period:DWORD; end;
  mySetPriority   : packed record Wanted,Period:DWORD; end;
  myPreProcessor  : TPreProcessorRec;
  myPostProcessor : TPostProcessorRec;
  myHelpFileFormats : LongString;
  mySysLogSignBase : LongString;
  mySysLogSignConf : LongString;
  function    GetCurves:TCurveList;
  function    GetPolling:TPolling;
  function    GetWatchdog:TPolling;
  function    GetDispatcher:TPolling;
  function    GetCurvesNameList:LongString;
  function    GetTagsNameList:LongString;
  function    GetTimer:TIntervalTimer;
  function    GetAcqTimer:TIntervalTimer;
  function    GetSaveTimer:TIntervalTimer;
  function    GetHistTimer:TIntervalTimer;
  function    GetSpecTimer:TIntervalTimer;
  function    GetCurWinList:TCurveWindowList;
  function    GetTabWinList:TTabWindowList;
  function    GetCirWinList:TCircuitWindowList;
  function    GetSpeWinList:TSpectrWindowList;
  function    GetWarningList:TText;
  function    GetHistList:TText;
  function    GetConfigFile:LongString;
  function    GetConfigPath:LongString;
  function    GetSitePath:LongString;
  function    GetDataPath:LongString;
  function    GetDataFile:LongString;
  procedure   SetDataFile(const aName:LongString);
  function    GetBackupFile:LongString;
  function    GetTempPath:LongString;
  function    GetLogFile:LongString;
  function    GetHelpFile:LongString;
  function    GetConsoleFile:LongString;
  function    GetDaqDataWindow:TFormCurveWindow;
  function    GetSoundOn:Boolean;
  procedure   SetSoundOn(OnOff:Boolean);
  procedure   InitErrMsgTable;
  procedure   FreeErrMsgTable;
  function    GetErrMsg(ErrorCode:Integer):LongString;
  function    GetMonitorPeriod(wt:LongInt):LongInt;
  procedure   SetMonitorPeriod(wt,ms:LongInt);
  function    GetPreProcessor:TPreProcessorRec;
  function    GetPostProcessor:TPostProcessorRec;
 public
  {
  Создание / уничтожение DAQ. DAQ существует в единственном экземпляре.
  }
  constructor Create;
  destructor  Destroy; override;
 public
  {
  Свойства DAQ
  }
  property    Curves            : TCurveList         read GetCurves;
  property    Polling           : TPolling           read GetPolling;
  property    Watchdog          : TPolling           read GetWatchdog;
  property    Dispatcher        : TPolling           read GetDispatcher;
  property    TagsNameList      : LongString         read GetTagsNameList;
  property    CurvesNameList    : LongString         read GetCurvesNameList;
  property    Timer             : TIntervalTimer     read GetTimer;
  property    AcqTimer          : TIntervalTimer     read GetAcqTimer;
  property    SaveTimer         : TIntervalTimer     read GetSaveTimer;
  property    HistTimer         : TIntervalTimer     read GetHistTimer;
  property    SpecTimer         : TIntervalTimer     read GetSpecTimer;
  property    CurWinList        : TCurveWindowList   read GetCurWinList;
  property    TabWinList        : TTabWindowList     read GetTabWinList;
  property    CirWinList        : TCircuitWindowList read GetCirWinList;
  property    SpeWinList        : TSpectrWindowList  read GetSpeWinList;
  property    WarningList       : TText              read GetWarningList;
  property    HistList          : TText              read GetHistList;
  property    ConfigFile        : LongString         read GetConfigFile;
  property    ConfigPath        : LongString         read GetConfigPath;
  property    SitePath          : LongString         read GetSitePath;
  property    DataPath          : LongString         read GetDataPath;
  property    DataFile          : LongString         read GetDataFile write SetDataFile;
  property    BackupFile        : LongString         read GetBackupFile;
  property    TempPath          : LongString         read GetTempPath;
  property    LogFile           : LongString         read GetLogFile;
  property    HelpFile          : LongString         read GetHelpFile;
  property    ConsoleFile       : LongString         read GetConsoleFile;
  property    DaqDataWindow     : TFormCurveWindow   read GetDaqDataWindow;
  property    SoundOn           : Boolean            read GetSoundOn write SetSoundOn;
  property    ErrMsg[i:Integer] : LongString         read GetErrMsg;
  property    MonitorPeriod[wt:LongInt] : LongInt    read GetMonitorPeriod write SetMonitorPeriod;
  property    PreProcessor      : TPreProcessorRec   read GetPreProcessor;
  property    PostProcessor     : TPostProcessorRec  read GetPostProcessor;
 public
  { Configuration parsing }
  function    FileRef(const FileName : LongString;
                      const DefExt   : LongString = '';
                      const BaseFile : LongString = '*'):LongString;
  function    FileRel(const FileName : LongString;
                      const BaseFile : LongString = '*'):LongString;
  procedure   ReadConfig; virtual;
  procedure   ReadDaqSection;
  procedure   ReadDataStorageSection;
  procedure   ReadWindowsSection(wt:word; DaqDataWin:TFormCurveWindow);
  procedure   UpdateWindowsTimeBaseTimeUnits;
  procedure   SaveWinLocations(wt:word);
  procedure   LoadWinLocations(wt:word);
  procedure   Animate; virtual;
  procedure   ShowLogo(const aLogo:LongString);
  procedure   ShowLogoDelay(aDelay:Integer);
  { General methods }
  procedure   InitSession(CfgFile:LongString='');
  procedure   DoneSession;
  procedure   InitEnvironment; virtual;
  procedure   FreeEnvironment; virtual;
  procedure   RunPreProcessor; virtual;
  procedure   RunPostProcessor; virtual;
  procedure   StartSession; virtual;
  procedure   StopSession; virtual;
  function    Start:Boolean; virtual;
  function    Stop:Boolean; virtual;
  function    Clear:Boolean; virtual;
  function    CustomClear(const Params:LongString=''):Boolean;
  procedure   Idle; virtual;
  procedure   Poll; virtual;
  procedure   DispatchEvents; virtual;
  function    CommonProperty(P:TText):TText; virtual;
  function    BriefDaqSysInfoCounters:String;
  function    BriefDaqSysBugsCounters:String;
  procedure   DaqInfo; virtual;
  { Data file I/0 }
  procedure   SaveData(FileName:LongString);
  procedure   SaveDataCustom(FileName,Caption,Title,Legend,CurveList:LongString);
  procedure   LoadData(FileName:LongString);
  procedure   SaveBackup;
  procedure   LoadBackup;
  procedure   Remedy(const Params:LongString='');
  { Window operations }
  procedure   WinOpen(wt:word);
  procedure   WinClose(wt:word);
  procedure   WinHide(wt:word);
  procedure   WinShow(wt:word);
  procedure   WinDraw(wt:word);
  procedure   WindowsOperations(const Params:LongString='');
  procedure   OpenDynamicWindow(const Params:LongString='');
  procedure   UpdateSpectralWindows;
  { Service windows }
  procedure   OpenDaqDataWindow;
  procedure   OpenLogBook(ws:TWindowState=wsNormal);
  procedure   ShowLogBook(ws:TWindowState);
  procedure   TouchLogBook;
  procedure   SaveLogBook;
  procedure   OpenHelp;
  function    HelpFileFormats:LongString;
  { DAQ console routines }
  procedure   OpenConsole(Hidden:Boolean=false);
  procedure   ConsoleEcho(const s:LongString);
  procedure   Report(const msg:LongString);
  procedure   ConsoleFlush;
  procedure   UpdateSysLogSign;
  function    SysLogSign(Mode:Integer; Sub:LongString=''):LongString;
  {Some service dialogs}
  function    SelectCurve(const aCaption,aTitle:LongString; const Params:LongString=''):TCurve;
  function    SelectCurveList(const Params:LongString=''):LongString;
  function    SelectTagList(const Params:LongString=''):LongString;
  function    CheckSessionStarted:Boolean;
  function    CheckAcquisitionStarted:Boolean;
  function    ConfirmExit(const Params:LongString=''):Boolean;
  function    ConfirmStop(const Params:LongString=''):Boolean;
  {History length control routines}
  procedure   CheckHistoryLength;
  procedure   LimitHistoryLength(MaxHistLen:LongInt);
  {Miscellaneous utilites}
  procedure   Voice(const WhatToSay:LongString);
  function    RegisterErrMsg(const Msg:LongString):Integer;
  procedure   AddWarning(const Msg:LongString);
  procedure   ViewData(const Params:LongString='');
  procedure   Navigator(const Params:LongString='');
  procedure   Integral(const Params:LongString='');
  procedure   MedianFiltr(const Params:LongString='');
  procedure   CutRegion(const Params:LongString='');
  procedure   SetUserMarker(const Params:LongString='');
  procedure   EditCurvePoint(const Params:LongString='');
  procedure   CurveSmoothing(const Params:LongString='');
  procedure   OpenCalculator(const Params:LongString='');
  procedure   TipWarning(const Msg:LongString; const Opt:LongString='');
 end;

const
 {
 Единственный экземпляр DAQ.
 }
 Daq : TDaqSystem           = nil;
 DefaultDaqPollDelay        = 4;
 DefaultDaqPollPriority     = tpHigher;
 DefaultDaqDispDelay        = 4;
 DefaultDaqDispPriority     = tpHighest;
 DefaultDaqWatchdogDelay    = 1000;
 DefaultDaqWatchdogPriority = tpTimeCritical;
 DefaultDaqWatchdogDeadline = 60000;
 DefaultDaqTimeOut          = INFINITE;

 {
 Флаги типов окон
 }
 wtCurWin                = $0001;
 wtTabWin                = $0002;
 wtCirWin                = $0004;
 wtSpeWin                = $0008;
 wtAll                   = wtCurWin+wtTabWin+wtCirWin+wtSpeWin;
 wtAllWin                = wtAll+1;
 {
 Флаги режимов консоли
 }
 cmfInit                 = $0001;
 cmfDone                 = $0002;
 cmfStart                = $0004;
 cmfStop                 = $0008;
 cmfLoad                 = $0010;
 cmfSave                 = $0020;
 cmfInteg                = $0040;
 cmfMarker               = $0080;
 cmfDefault              = cmfInit+cmfDone+cmfInteg;
 {
 Коды счетчика ошибок
 }
 ecAnalogFifoLost        = 0;
 ecDigitalFifoLost       = 1;
 ecAnalogDispatch        = 2;
 ecDigitalDispatch       = 3;
 ecFormulaSyntax         = 4;
 ecAdamWatchDogFail      = 5;
 ecHardwareStart         = 6;
 ecSpectrChan            = 7;
 ecDaqPascalStart        = 8;
 ecDaqPascalHalt         = 9;
 ecDialog                = 10;
 ecPipeIO                = 11;
 ecPipeOver              = 12;
 ecTagBug                = 13;
 ecWarningList           = 14;
 ecWatchdogDeadline      = 15;
 ecSensorTagEval         = 16;
 ecSensorLedEval         = 17;
 ecSensorPainter         = 18;
 ecSensorClickFifoLost   = 19;
 ecFormatErrorInStrFmt   = 20;
 ecAdamFormat            = 100;
 ecAdamTimeOut           = 101;
 ecAdamRequest           = 102;
 ecAdamStart             = 103;
 ecAdamCheckSum          = 104;
 ecDmaInit               = 105;
 ecDmaLost               = 106;
 ecADCRead               = 107;
 ecCamacIRQError         = 108;
 ecCounterOverflow       = 109;
 ecADCOverflow           = 110;
 ecMuxError              = 111;
 ecAdamStop              = 112;
 ecUnknown               = 255;

const
 DaqShowLogoDeadline:Int64=0;

procedure Timer_DaqShowLogoHide;

 {
 *******************************************************************************
 Utilites
 *******************************************************************************
 }
procedure DaqAlignStr(var Str:PureString); overload;
procedure DaqAlignStr(var Str:LongString); overload;
procedure DaqClearCurve(Curve:TCurve; HistLen:LongInt);
function  DaqIntegral(Curve:TCurve; a,b:Double):Double;

 {
 ***********************************************************************
 DAQ Top List - list of latest window toplist to stay open while another
 windows to be minimized. Call it periodically to minimize CPU overhead.
 ***********************************************************************
 }
const
 CurveTopListLimitDefault   = 0; // Counter of Curve_Window   toplist
 TableTopListLimitDefault   = 0; // Counter of Tab_Window     toplist
 CircuitTopListLimitDefault = 0; // Counter of Circuit_Window toplist
 SpectrTopListLimitDefault  = 0; // Counter of Spectr_Window  toplist
 DaqTopListPeriodDefault    = 5; // Poll period of DAQ window toplist, sec

 CurveTopListLimit   : Integer = CurveTopListLimitDefault;
 TableTopListLimit   : Integer = TableTopListLimitDefault;
 CircuitTopListLimit : Integer = CircuitTopListLimitDefault;
 SpectrTopListLimit  : Integer = SpectrTopListLimitDefault;
 DaqTopListPeriod    : Integer = DaqTopListPeriodDefault;

procedure DaqTopListMonitor_Timer;
procedure DaqTopListMonitor_Start;
procedure DaqTopListMonitor_Stop;
procedure DaqTopListMonitor_Send;

function GetDaqSysInfo(const What:LongString):LongString;

implementation

uses
 Form_CrwDaq,
 Form_CrwDaqLogo,
 Form_DaqHelpViewer,
 Form_DaqHistLenDialog,
 Form_DaqHistoryWarning,
 Form_DaqEditTagDialog,
 Form_DaqDateTimeCalculator,
 Form_DaqOpenDynamicWindowDialog,
 Form_DaqMedianFilterDialog,
 Form_DaqCurveSmoothing,
 Form_DaqWinOperDialog,
 Form_DaqDeviceControl,
 Form_DaqControlDialog,
 _crw_DaqPascalDevice,
 _crw_daqdev;

 {
 Internal use only
 }
procedure DefaultDaqPollAction(aPolling:TPolling; var Terminate:Boolean);
begin
 if Daq.Ok then Daq.Poll else Terminate:=true;
end;

procedure DefaultDaqDispAction(aPolling:TPolling; var Terminate:Boolean);
begin
 if Daq.Ok then Daq.DispatchEvents else Terminate:=true;
end;

procedure DefaultDaqWatchdogAction(aPolling:TPolling; var Terminate:Boolean);
begin
 if Daq. Ok then begin
  if Daq.AcqTimer.IsStart then FullDaqDeviceList.Watchdog;
 end else Terminate:=true;
end;

function ScreenWorkAreaRect:TRect;
begin
 if SdiMan.IsMdiMode and (Application.MainForm.FormStyle=fsMdiForm)
 then Result:=Application.MainForm.ClientRect
 else Result:=Screen.WorkAreaRect;
end;

 {
 *******************************************************************************
 TDaqSystem implementation
 *******************************************************************************
 }
function TDaqSystem.GetCurves:TCurveList;
begin
 if Assigned(Self) then Result:=myCurves else Result:=nil;
end;

function TDaqSystem.GetPolling:TPolling;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

function TDaqSystem.GetWatchdog:TPolling;
begin
 if Assigned(Self) then Result:=myWatchdog else Result:=nil;
end;

function TDaqSystem.GetDispatcher:TPolling;
begin
 if Assigned(Self) then Result:=myDispatcher else Result:=nil;
end;

function TDaqSystem.GetCurvesNameList:LongString;
var p:TText;
begin
 Result:='';
 try
  p:=NewText;
  try
   Curves.GetList(p);
   Result:=p.Text;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetCurvesNameList');
 end;
end;

function TDaqSystem.GetTagsNameList:LongString;
var p:TText; tag:Integer;
begin
 Result:='';
 try
  p:=NewText;
  try
   for tag:=tag_ref_min to tag_ref_max do
   if (TypeTag(tag)>0) then p.AddLn(NameTag(tag));
   Result:=p.Text;
  finally
   Kill(p);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetTagsNameList');
 end;
end;

function TDaqSystem.GetTimer:TIntervalTimer;
begin
 if Assigned(Self) then Result:=myTimer else Result:=nil;
end;

function TDaqSystem.GetAcqTimer:TIntervalTimer;
begin
 if Assigned(Self) then Result:=myAcqTimer else Result:=nil;
end;

function TDaqSystem.GetSaveTimer:TIntervalTimer;
begin
 if Assigned(Self) then Result:=mySaveTimer else Result:=nil;
end;

function TDaqSystem.GetHistTimer:TIntervalTimer;
begin
 if Assigned(Self) then Result:=myHistTimer else Result:=nil;
end;

function TDaqSystem.GetSpecTimer:TIntervalTimer;
begin
 if Assigned(Self) then Result:=mySpecTimer else Result:=nil;
end;

function TDaqSystem.GetCurWinList:TCurveWindowList;
begin
 if Assigned(Self) then Result:=myCurWinList else Result:=nil;
end;

function TDaqSystem.GetTabWinList:TTabWindowList;
begin
 if Assigned(Self) then Result:=myTabWinList else Result:=nil;
end;

function TDaqSystem.GetCirWinList:TCircuitWindowList;
begin
 if Assigned(Self) then Result:=myCirWinList else Result:=nil;
end;

function TDaqSystem.GetSpeWinList:TSpectrWindowList;
begin
 if Assigned(Self) then Result:=mySpeWinList else Result:=nil;
end;

function TDaqSystem.GetWarningList:TText;
begin
 if Assigned(Self) then Result:=myWarningList else Result:=nil;
end;

function TDaqSystem.GetHistList:TText;
begin
 if Assigned(Self) then Result:=myHistList else Result:=nil;
end;

function TDaqSystem.GetConfigFile:LongString;
begin
 if Assigned(Self) then Result:=myConfigFile else Result:='';
end;

function TDaqSystem.GetConfigPath:LongString;
begin
 if Assigned(Self) then Result:=ExtractFilePath(myConfigFile) else Result:='';
end;

function TDaqSystem.GetSitePath:LongString;
begin
 if Assigned(Self) then Result:=mySitePath else Result:='';
end;

function TDaqSystem.GetDataPath:LongString;
begin
 if Assigned(Self) then Result:=myDataPath else Result:='';
end;

function TDaqSystem.GetDataFile:LongString;
begin
 if Assigned(Self) then Result:=myDataFile else Result:='';
end;

procedure TDaqSystem.SetDataFile(const aName:LongString);
begin
 if Assigned(Self) then myDataFile:=aName;
end;

function TDaqSystem.GetBackupFile:LongString;
begin
 if Assigned(Self) then Result:=myBackupFile else Result:='';
end;

function TDaqSystem.GetTempPath:LongString;
begin
 Result:=ExtractFilePath(BackupFile);
end;

function TDaqSystem.GetLogFile:LongString;
begin
 if Assigned(Self) then Result:=myLogFile else Result:='';
end;

function TDaqSystem.GetHelpFile:LongString;
begin
 if Assigned(Self) then Result:=myHelpFile else Result:='';
end;

function TDaqSystem.GetConsoleFile:LongString;
begin
 if Assigned(Self) then Result:=myConsoleFile else Result:='';
end;

function TDaqSystem.GetDaqDataWindow:TFormCurveWindow;
begin
 if Assigned(Self) then Result:=myDaqDataWindow else Result:=nil;
end;

function TDaqSystem.GetSoundOn:Boolean;
begin
 if Assigned(Self) then Result:=mySoundOn else Result:=false;
end;

procedure TDaqSystem.SetSoundOn(OnOff:Boolean);
begin
 if Assigned(Self) then mySoundOn:=OnOff;
end;

procedure TDaqSystem.InitErrMsgTable;
var i:Integer;
begin
 if Ok then begin
  for i:=Low(myErrMsgTable) to High(myErrMsgTable) do myErrMsgTable[i]:='';
  myErrMsgTable[ecAnalogFifoLost]:=RusEng('Переполнение FIFO аналоговых событий','Analog event FIFO overflow');
  myErrMsgTable[ecDigitalFifoLost]:=RusEng('Переполнение FIFO цифровых событий','Digital event FIFO overflow');
  myErrMsgTable[ecAnalogDispatch]:=RusEng('Ошибка диспетчера аналоговых событий','Analog event dispatcher error');
  myErrMsgTable[ecDigitalDispatch]:=RusEng('Ошибка диспетчера цифровых событий','Digital event dispatcher error');
  myErrMsgTable[ecFormulaSyntax]:=RusEng('Синтаксическая ошибка в формуле','Syntax error in formula');
  myErrMsgTable[ecAdamWatchDogFail]:=RusEng('Сторожевой таймер: сбой устройства ADAM','Watchdog: ADAM device fail');
  myErrMsgTable[ecHardwareStart]:=RusEng('Аппаратная ошибка при старте','Hardware start error');
  myErrMsgTable[ecSpectrChan]:=RusEng('Неверный спектрометрический канал','Wrong spectrometry channel');
  myErrMsgTable[ecDaqPascalStart]:=RusEng('Ошибка при старте программы DAQ PASCAL','Error starting DAQ PASCAL program');
  myErrMsgTable[ecDaqPascalHalt]:=RusEng('Ошибка выполнения программы DAQ PASCAL','DAQ PASCAL program runtime error');
  myErrMsgTable[ecDialog]:=RusEng('Ошибка устройства-диалога','Dialog device error');
  myErrMsgTable[ecPipeIO]:=RusEng('Ошибка ввода-вывода pipe','Pipe in/out error');
  myErrMsgTable[ecPipeOver]:=RusEng('Переполнение FIFO pipe','Pipe fifo overflow');
  myErrMsgTable[ecTagBug]:=RusEng('Неверный тип тега Get/SetTag','Wrong Get/SetTag type');
  myErrMsgTable[ecWarningList]:=RusEng('Проблемы при загрузке конфигурации','Configuration loading problems');
  myErrMsgTable[ecWatchdogDeadline]:=RusEng('Сторожевой таймер: зафиксирован DEADLINE','System watchdog: DEADLINE detected');
  myErrMsgTable[ecSensorTagEval]:=RusEng('Ошибка исполнения сценария TagEval(v)','Error executing TagEval(v) script');
  myErrMsgTable[ecSensorLedEval]:=RusEng('Ошибка исполнения сценария LedEval(v)','Error executing LedEval(v) script');
  myErrMsgTable[ecSensorPainter]:=RusEng('Ошибка исполнения сценария Painter(v)','Error executing Painter(v) script');
  myErrMsgTable[ecSensorClickFifoLost]:=RusEng('Переполнение FIFO пользовательских событий.','GUI events FIFO overflow.');
  myErrMsgTable[ecFormatErrorInStrFmt]:=RusEng('Ошибка форматирования в StrFmt()','Format error in StrFmt()');
  myErrMsgTable[ecAdamFormat]:=RusEng('Ошибка формата сообщений ADAM','ADAM format error');
  myErrMsgTable[ecAdamTimeOut]:=RusEng('TimeOut при опросе ADAM','ADAM timeout');
  myErrMsgTable[ecAdamRequest]:=RusEng('Неверный запрос ADAM','Wrong ADAM request');
  myErrMsgTable[ecAdamStart]:=RusEng('Ошибка при старте ADAM','ADAM start error');
  myErrMsgTable[ecAdamStop]:=RusEng('Ошибка при останове ADAM','ADAM stop error');
  myErrMsgTable[ecAdamCheckSum]:=RusEng('Ошибка контрольной суммы ADAM','ADAM checksum error');
  myErrMsgTable[ecDmaInit]:=RusEng('Ошибка инициализации DMA','DMA initialization fails');
  myErrMsgTable[ecDmaLost]:=RusEng('Переполнение буферов DMA','DMA buffers overflow');
  myErrMsgTable[ecADCRead]:=RusEng('Ошибка чтения кода ADC','ADC readout error');
  myErrMsgTable[ecCamacIRQError]:=RusEng('Перегрузка CAMAC по прерываниям','CAMAC bus overload');
  myErrMsgTable[ecCounterOverflow]:=RusEng('Переполнение счетчика','Counter overflow');
  myErrMsgTable[ecADCOverflow]:=RusEng('Перегрузка АЦП по входу','ADC overrange');
  myErrMsgTable[ecMuxError]:=RusEng('Ошибка мультиплексора','Muliplexor error');
  myErrMsgTable[ecUnknown]:=RusEng('Неизвестная ошибка','Unknown bug');
 end;
end;

procedure TDaqSystem.FreeErrMsgTable;
var i:Integer;
begin
 if Ok then begin
  for i:=Low(myErrMsgTable) to High(myErrMsgTable) do myErrMsgTable[i]:='';
 end;
end;

function TDaqSystem.GetErrMsg(ErrorCode:Integer):LongString;
begin
 Result:='';
 if Ok  then
 if InRange(ErrorCode,Low(myErrMsgTable),High(myErrMsgTable))
 then Result:=myErrMsgTable[ErrorCode]
 else Result:=myErrMsgTable[ecUnknown];
end;

function  TDaqSystem.GetMonitorPeriod(wt:LongInt):LongInt;
begin
 Result:=0;
 if Ok then
 case wt of
  wtCurWin : Result:=myCurWinPeriod;
  wtTabWin : Result:=myTabWinPeriod;
  wtCirWin : Result:=myCirWinPeriod;
  wtSpeWin : Result:=mySpeWinPeriod;
 end;
end;

procedure SetupMonitorPeriod(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if (aObject is TFormCrwDaqSysChild) then TFormCrwDaqSysChild(aObject).MonitorPeriod:=PointerToPtrInt(CustomData);
end;

procedure TDaqSystem.SetMonitorPeriod(wt,ms:LongInt);
begin
 if Ok then begin
  ms:=max(0,ms);
  if HasFlags(wt,wtCurWin) then begin CurWinList.ForEach(SetupMonitorPeriod,PtrIntToPointer(ms)); myCurWinPeriod:=ms; end;
  if HasFlags(wt,wtTabWin) then begin TabWinList.ForEach(SetupMonitorPeriod,PtrIntToPointer(ms)); myTabWinPeriod:=ms; end;
  if HasFlags(wt,wtCirWin) then begin CirWinList.ForEach(SetupMonitorPeriod,PtrIntToPointer(ms)); myCirWinPeriod:=ms; end;
  if HasFlags(wt,wtSpeWin) then begin SpecTimer.IntervalMs[0]:=ms;                                mySpeWinPeriod:=ms; end;
 end;
end;

function TDaqSystem.GetPreProcessor:TPreProcessorRec;
begin
 if Ok then with myPreProcessor do begin
  Result.CmdLine:=CmdLine;
  Result.TimeOut:=TimeOut;
  Result.Display:=Display;
  Result.PipeSize:=PipeSize;
  Result.Priority:=Priority;
 end else begin
  Result.CmdLine:='';
  Result.TimeOut:=0;
  Result.Display:=0;
  Result.PipeSize:=0;
  Result.Priority:='';
 end;
end;

function TDaqSystem.GetPostProcessor:TPostProcessorRec;
begin
 if Ok then with myPostProcessor do begin
  Result.CmdLine:=CmdLine;
  Result.TimeOut:=TimeOut;
  Result.Display:=Display;
  Result.PipeSize:=PipeSize;
  Result.Priority:=Priority;
 end else begin
  Result.CmdLine:='';
  Result.TimeOut:=0;
  Result.Display:=0;
  Result.PipeSize:=0;
  Result.Priority:='';
 end;
end;

 {
 Инициализация DAQ
 }
constructor TDaqSystem.Create;
var FifoSize:Integer;
begin
 Kill(TObject(Daq));
 inherited Create;
 Daq:=Self;
 Daq.Master:=@Daq;
 myCurves:=NewCurveList(true);
 myCurves.Master:=@myCurves;
 myPolling:=NewPolling(DefaultDaqPollAction,
                       DefaultDaqPollDelay,  DefaultDaqPollPriority,
                       false, 'Daq.Polling');
 myPolling.Master:=@myPolling;
 myPolling.LinkObject:=Self;
 myWatchdog:=NewPolling(DefaultDaqWatchdogAction,
                        DefaultDaqWatchdogDelay,  DefaultDaqWatchdogPriority,
                        false, 'Daq.Watchdog');
 myWatchdog.Master:=@myWatchdog;
 myWatchdog.LinkObject:=Self;
 myDispatcher:=NewPolling(DefaultDaqDispAction,
                          DefaultDaqDispDelay,  DefaultDaqDispPriority,
                          false, 'Daq.Dispatcher');
 myDispatcher.Master:=@myDispatcher;
 myDispatcher.LinkObject:=Self;
 myTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(150,1,nil));
 myTimer.Master:=@myTimer;
 myTimer.LocalTimeUnits:=1000;                                          { 1  sec }
 myAcqTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(500,1,nil));
 myAcqTimer.Master:=@myAcqTimer;
 mySaveTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(600000,1,nil));   { 10 min }
 mySaveTimer.Master:=@mySaveTimer;
 myHistTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(60000,1,nil));    { 1  min }
 myHistTimer.Master:=@myHistTimer;
 mySpecTimer:=NewIntervalTimer(tmCyclic,NewIntervalMs(10000,1,nil));    { 10 sec }
 mySpecTimer.Master:=@mySpecTimer;
 myCurWinList:=NewCurveWindowList(true);
 myCurWinList.Master:=@myCurWinList;
 myTabWinList:=NewTabWindowList(true);
 myTabWinList.Master:=@myTabWinList;
 myCirWinList:=NewCircuitWindowList(true);
 myCirWinList.Master:=@myCirWinList;
 mySpeWinList:=NewSpectrWindowList(true);
 mySpeWinList.Master:=@mySpeWinList;
 FifoSize:=64;
 if ReadIniFileInteger(SysIniFile,SectDaqSys,'DaqConsoleFifoSize%i',FifoSize)
 then FifoSize:=Max(16,FifoSize) else FifoSize:=64;
 myConsoleFifo:=NewFifo(FifoSize*1024);
 myConsoleFifo.Master:=@myConsoleFifo;
 myWarningList:=NewText;
 myWarningList.Master:=@myWarningList;
 myHistList:=NewText;
 myHistList.Master:=@myHistList;
 myConfigFile:=UnifyFileAlias(SubSystemIniFile(SectDaqSys),ua_FileDefLow);
 mySitePath:=UnifyFileAlias(AddPathDelim(AddPathDelim(HomeDir)+'resource')+'daqsite',ua_FileDefLow);
 myDataPath:=UnifyFileAlias(SubSystemDataPath(SectDaqSys),ua_FileDefLow);
 myDataFile:=UnifyFileAlias(ForcePath(myDataPath,'*.daq'),ua_FileDefLow);
 myMergeFile:=UnifyFileAlias(ForcePath(myDataPath,'*.daq'),ua_FileDefLow);
 myBackupFile:=UnifyFileAlias(ForcePath(TempDir,'__backup.daq'),ua_FileDefLow);
 myLogFile:='';
 myHelpFile:='';
 myDaqDataWindow:=nil;
 myLogBook:=nil;
 myConsoleMode:=cmfDefault;
 mySoundOn:=false;
 mySaveOnStop:=false;
 myAutoStart:=false;
 myHistLimits[0]:=0;
 myHistLimits[1]:=0;
 myHistLimits[2]:=0;
 myLogoDelay:=0;
 myDaqDesktop:='';
 myNavigator:='';
 myStopConfirm:='';
 InitErrMsgTable;
 myVoiceEcho:=false;
 myConsoleFile:='';
 OpenConsole(true);
 myCurWinPeriod:=0;
 myTabWinPeriod:=0;
 myCirWinPeriod:=0;
 mySpeWinPeriod:=Round(SpecTimer.IntervalMs[0]);
 myFixPriority.Wanted:=0;
 myFixPriority.Period:=0;
 mySetPriority.Wanted:=0;
 mySetPriority.Period:=0;
 myHelpFileFormats:='';
 mySysLogSignBase:='';
 mySysLogSignConf:='';
 UpdateSysLogSign;
end;

 {
 Уничтожение DAQ
 }
destructor  TDaqSystem.Destroy;
begin
 ConsoleFlush;
 KillFormCalibDialog;
 KillFormDaqHelpViewer;
 KillFormDaqEditTagDialog;
 KillFormDaqHistLenDialog;
 KillFormDaqHistoryWarning;
 KillFormDaqDateTimeCalculator;
 KillFormDaqOpenDynamicWindowDialog;
 KillFormDaqMedianFilterDialog;
 KillFormDaqWinOperDialog;
 KillFormDaqDeviceControl;
 KillFormDaqControlDialog;
 Kill(myDaqDataWindow);
 Kill(myLogBook);
 Kill(myHistList);
 Kill(myWarningList);
 Kill(myConsoleFifo);
 Kill(mySpeWinList);
 Kill(myCirWinList);
 Kill(myTabWinList);
 Kill(myCurWinList);
 Kill(mySpecTimer);
 Kill(myHistTimer);
 Kill(mySaveTimer);
 Kill(myAcqTimer);
 Kill(myTimer);
 Kill(myDispatcher);
 Kill(myWatchdog);
 Kill(myPolling);
 Kill(myCurves);
 myConfigFile:='';
 mySitePath:='';
 myDataPath:='';
 myDataFile:='';
 myBackupFile:='';
 myLogFile:='';
 myHelpFile:='';
 myMergeFile:='';
 myConsoleFile:='';
 myDaqDesktop:='';
 myNavigator:='';
 myStopConfirm:='';
 myHelpFileFormats:='';
 mySysLogSignBase:='';
 mySysLogSignConf:='';
 FreeErrMsgTable;
 inherited Destroy;
end;

function TDaqSystem.FileRef(const FileName,DefExt,BaseFile:LongString):LongString;
begin
 Result:='';
 if Ok then
 if (BaseFile='*')
 then Result:=SmartFileRef(FileName,DefExt,ConfigFile)
 else Result:=SmartFileRef(FileName,DefExt,BaseFile);
end;

function TDaqSystem.FileRel(const FileName,BaseFile:LongString):LongString;
begin
 Result:='';
 if Ok then
 if (BaseFile='*')
 then Result:=SmartFileRel(FileName,ConfigFile)
 else Result:=SmartFileRel(FileName,BaseFile);
end;

 {
 Метод должен загружать необходимые данные для инициализации DAQ-системы
 из конфигурационного файла и подготавливать данные для начала сеанса DAQ
 }
procedure TDaqSystem.ReadConfig;
var w:Word;
begin
 if Ok then begin
  Timer.Start;
  ReadDaqSection;
  InitEnvironment;
  RunPreProcessor;
  ReadDataStorageSection;
  ReadCurveWindowsPalette(ConfigFile);
  ReadSpectrWindowsPalette(ConfigFile);
  ReadSurfaceWindowsPalette(ConfigFile);
  ReadWindowsSection(wtAll,nil);
  UpdateWindowsTimeBaseTimeUnits;
  w:=0;
  if ReadIniFileWord(ConfigFile,SectDaq,'AutoLoadWinPos%w',w)
  then LoadWinLocations(w and wtAll);
  MonitorPeriod[wtCurWin]:=myCurWinPeriod;
  MonitorPeriod[wtTabWin]:=myTabWinPeriod;
  MonitorPeriod[wtCirWin]:=myCirWinPeriod;
  MonitorPeriod[wtSpeWin]:=mySpeWinPeriod;
 end;
end;

 {
 Чтение из файла конфигурации секции [DAQ]
 }
procedure TDaqSystem.ReadDaqSection;
var f:Double; b:Boolean; s:LongString; d:LongInt; p:TThreadPriority;
begin
 if Ok then begin
  f:=0; b:=false; s:=''; d:=0; p:=tpNormal;
  {}
  if ReadIniFilePriorityClass(ConfigFile,SectDaq,'ProcessPriorityClass',mySetPriority.Wanted,mySetPriority.Period)
  then ForcePriorityClass(mySetPriority.Wanted,mySetPriority.Period) else
  if ReadIniFilePriorityClass(SysIniFile,SectDaqSys,'ProcessPriorityClass',mySetPriority.Wanted,mySetPriority.Period)
  then ForcePriorityClass(mySetPriority.Wanted,mySetPriority.Period) else
  if ReadIniFilePriorityClass(SysIniFile,SectSystem,'ProcessPriorityClass',mySetPriority.Wanted,mySetPriority.Period)
  then ForcePriorityClass(mySetPriority.Wanted,mySetPriority.Period) else begin
   mySetPriority.Wanted:=0;
   mySetPriority.Period:=0;
  end;
  {}
  if ReadIniFilePath(ConfigFile,SectDaq,'BackFile', ConfigPath, s)
  then myBackupFile:=UnifyFileAlias(DefaultExtension(s,'.daq'),ua_FileDefLow)
  else myBackupFile:=UnifyFileAlias(ForcePath(TempDir,'__backup.daq'),ua_FileDefLow);
  myBackupFile:=UnifyFileAlias(FileGuard.ProtectHomeDir(myBackupFile),ua_FileDefLow);
  if not DirExists(TempPath) then MkDir(TempPath);
  {}
  if ReadIniFilePath(SysIniFile,SectSystem,'DaqSite',HomeDir,s)
  then mySitePath:=UnifyFileAlias(s,ua_FileDefLow)
  else mySitePath:=UnifyFileAlias(AddPathDelim(AddPathDelim(HomeDir)+'resource')+'daqsite',ua_FileDefLow);
  {}
  if ReadIniFilePath(ConfigFile, SectDaq, 'DataPath', ConfigPath, s)
  then myDataPath:=UnifyFileAlias(s,ua_FileDefLow)
  else myDataPath:=UnifyFileAlias(SubSystemDataPath(SectDaqSys),ua_FileDefLow);
  myDataPath:=UnifyFileAlias(FileGuard.ProtectHomeDir(myDataPath),ua_FileDefLow);
  if not DirExists(myDataPath) then MkDir(myDataPath);
  myDataFile:=ForcePath(myDataPath,'*.daq');
  myMergeFile:=ForcePath(myDataPath,'*.daq');
  {}
  if ReadIniFilePath(ConfigFile, SectDaq, 'LogFile', ConfigPath, myLogFile) then begin
   myLogFile:=UnifyFileAlias(DefaultExtension(myLogFile, '.log'),ua_FileDefLow);
   myLogFile:=UnifyFileAlias(FileGuard.ProtectHomeDir(myLogFile),ua_FileDefLow);
   MkDirByFileName(myLogFile);
  end else myLogFile:='';
  {}
  if ReadIniFilePath(ConfigFile, SectDaq, 'HelpFile', ConfigPath, myHelpFile) then begin
   myHelpFile:=UnifyAlias(DefaultExtension(myHelpFile,'.hlp'),ua_FileDefLow);
  end else myHelpFile:='';
  {}
  f:=1;
  if ReadIniFileDouble(ConfigFile, SectDaq,    'TimeUnit%f', f)
  or ReadIniFileDouble(SysIniFile, SectDaqSys, 'TimeUnit%f', f)
  then Timer.LocalTimeUnits:=f*1000
  else Timer.LocalTimeUnits:=1000;
  {}
  f:=0;
  ReadIniFileDouble(ConfigFile, SectDaq, 'Save%f', f);
  SaveTimer.IntervalMs[0]:=f*Timer.LocalTimeUnits;
  {}
  if ReadIniFilePath(ConfigFile, SectDaq, 'SoundLibrary', ConfigPath, s)
  then AddSoundLibrary(s);
  {}
  AlphaStayFree:=0.7;
  ReadIniFileDouble(ConfigFile, SectDaq, 'AlphaStayFree%f', AlphaStayFree);
  {}
  if ReadIniFileBoolean(ConfigFile, SectDaq, 'UseBlaster%b', b) and b and not UsesBlaster then begin
   if InitBlaster then begin
    if ReadIniFilePath(SysIniFile,SectSystem,'SoundLibrary',HomeDir,s) then AddSoundLibrary(s);
    if ReadIniFileLongInt(SysIniFile,SectSystem,'BlasterCheckDelay%d',d) then BlasterMemoPollPeriod:=d*1000;
    if ReadIniFileLongInt(SysIniFile,SectSystem,'MaxBlasterDataSize%d',d) then BlasterMaxDataSize:=d*1024;
    if ReadIniFileBoolean(SysIniFile,SectSystem,'UseSystemVoice%b',b) then EnableSystemVoice:=b;
    if ReadIniFileLongInt(SysIniFile,SectSystem,'SayTimeInterval%d',d) then SayTimeInterval:=d;
    Echo(RusEng('Звуковая подсистема : Включена.','Sound subsystem '+ASCII_Tab+': Ok.'));
   end else begin
    Echo(RusEng('Звуковая подсистема : Сбой.','Sound subsystem : Fails.'));
   end;
  end;
  if UsesBlaster then begin
   if ReadIniFileDouble(ConfigFile, SectDaq, 'BlasterVolume%f', f)
   then SetBlasterVolume(f,f);
   if ReadIniFileDouble(ConfigFile, SectDaq, 'BlasterSpeed%f', f)
   then SetBlasterSpeed(f);
   ReadIniFileBoolean(ConfigFile, SectDaq, 'SoundOn%b', mySoundOn);
  end;
  {}
  if ReadIniFileDouble(ConfigFile, SectDaq, 'InquiryPeriod%f', f)
  then AcqTimer.IntervalMs[0]:=f;
  {}
  mySaveOnStop:=false;
  ReadIniFileBoolean(ConfigFile, SectDaq, 'SaveOnStop%b', mySaveOnStop);
  {}
  myAutoStart:=false;
  ReadIniFileBoolean(ConfigFile, SectDaq, 'AutoStart%b', myAutoStart);
  {}
  if ReadIniFileRecord(ConfigFile, SectDaq, 'HistoryLimits%d;%d;%d', myHistLimits)
  then begin
   myHistLimits[0]:=max(myHistLimits[0], 0);
   myHistLimits[1]:=max(myHistLimits[1], myHistLimits[0]);
   myHistLimits[2]:=max(myHistLimits[2], myHistLimits[1]);
  end else begin
   myHistLimits[0]:=0;
   myHistLimits[1]:=0;
   myHistLimits[2]:=0;
  end;
  {}
  f:=1;
  ReadIniFileDouble(ConfigFile, SectDaq, 'HistoryCheckPeriod%f', f);
  HistTimer.IntervalMs[0]:=f*60000;
  {}
  f:=10;
  if ReadIniFileDouble(ConfigFile, SectDaq,    'SpectrUpdatePeriod%f', f)
  or ReadIniFileDouble(SysIniFile, SectDaqSys, 'SpectrUpdatePeriod%f', f)
  then SpecTimer.IntervalMs[0]:=f*1000
  else SpecTimer.IntervalMs[0]:=10000;
  {}
  if ReadIniFilePath(ConfigFile, SectDaq, 'Desktop', ConfigPath, myDaqDesktop)
  then myDaqDesktop:=DefaultExtension(myDaqDesktop,'.bmp');
  if not FileExists(myDaqDesktop) then myDaqDeskTop:=FormCrwDaq.DesktopLogoFile;
  if not FileExists(myDaqDesktop) then myDaqDeskTop:='';
  ShowLogo(RusEng('Подождите, идет загрузка CRW-DAQ...','Wait while loading CRW-DAQ...'));
  {}
  d:=0;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'ShowLogoDelay%d', d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'ShowLogoDelay%d', d)
  then myLogoDelay:=d*1000
  else myLogoDelay:=0;
  {}
  myConsoleMode:=cmfDefault;
  ReadIniFileWord(ConfigFile, SectDaq, 'ConsoleMode%w', myConsoleMode);
  {}
  myNavigator:='';
  ReadIniFileAlpha(ConfigFile, SectDaq, 'Navigator%a', myNavigator);
  {}
  myStopConfirm:='';
  if ReadIniFileAlpha(ConfigFile, SectDaq, 'StopConfirm%a', myStopConfirm)
  then DaqAlignStr(myStopConfirm);
  {}
  myVoiceEcho:=false;
  ReadIniFileBoolean(ConfigFile, SectDaq, 'VoiceEcho%b', myVoiceEcho);
  {}
  if not ReadIniFilePath(ConfigFile, SectDaq, 'ConsoleFile',
                         ConfigPath, myConsoleFile)
  then myConsoleFile:='';
  if IsNonEmptyStr(myConsoleFile) then begin
   myConsoleFile:=UnifyFileAlias(FileGuard.ProtectHomeDir(myConsoleFile),ua_FileDefLow);
   MkDirByFileName(myConsoleFile);
  end;
  {}
  if ReadIniFilePolling(ConfigFile, SectDaq,    'DevicePolling', d, p)
  or ReadIniFilePolling(SysIniFile, SectDaqSys, 'DevicePolling', d, p)
  then begin
   Polling.Delay:=d;
   Polling.Priority:=p;
  end;
  if ReadIniFilePolling(ConfigFile, SectDaq,    'DispatcherPolling', d, p)
  or ReadIniFilePolling(SysIniFile, SectDaqSys, 'DispatcherPolling', d, p)
  then begin
   Dispatcher.Delay:=d;
   Dispatcher.Priority:=p;
  end;
  if ReadIniFilePolling(ConfigFile, SectDaq,    'WatchdogPolling', d, p)
  or ReadIniFilePolling(SysIniFile, SectDaqSys, 'WatchdogPolling', d, p)
  then begin
   Watchdog.Delay:=d;
   Watchdog.Priority:=p;
  end;
  {}
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'CurveMonitorPeriod%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'CurveMonitorPeriod%d',d)
  then myCurWinPeriod:=d else myCurWinPeriod:=0;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'TableMonitorPeriod%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'TableMonitorPeriod%d',d)
  then myTabWinPeriod:=d else myTabWinPeriod:=0;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'CircuitMonitorPeriod%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'CircuitMonitorPeriod%d',d)
  then myCirWinPeriod:=d else myCirWinPeriod:=0;
  mySpeWinPeriod:=Round(SpecTimer.IntervalMs[0]);
  {}
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'CurveTopListLimit%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'CurveTopListLimit%d',d)
  then CurveTopListLimit:=d else CurveTopListLimit:=CurveTopListLimitDefault;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'TableTopListLimit%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'TableTopListLimit%d',d)
  then TableTopListLimit:=d else TableTopListLimit:=TableTopListLimitDefault;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'CircuitTopListLimit%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'CircuitTopListLimit%d',d)
  then CircuitTopListLimit:=d else CircuitTopListLimit:=CircuitTopListLimitDefault;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'SpectrTopListLimit%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'SpectrTopListLimit%d',d)
  then SpectrTopListLimit:=d else SpectrTopListLimit:=SpectrTopListLimitDefault;
  if ReadIniFileLongInt(ConfigFile, SectDaq,    'DaqTopListPeriod%d',d)
  or ReadIniFileLongInt(SysIniFile, SectDaqSys, 'DaqTopListPeriod%d',d)
  then DaqTopListPeriod:=d else DaqTopListPeriod:=DaqTopListPeriodDefault;
  {}
  with myPreProcessor do begin
   CmdLine:=''; TimeOut:=0; Display:=0; PipeSize:=0; Priority:='';
   if ReadIniFileString(ConfigFile,SectDaq,'PreProcessor%s',CmdLine) then begin
    CmdLine:=AdaptExeFileName(CmdLine);
    if IsNonEmptyStr(CmdLine) then begin
     if ReadIniFileInteger(ConfigFile,SectDaq,'PreProcessorTimeOut%i',TimeOut)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PreProcessorTimeOut%i',TimeOut)
     then TimeOut:=Max(MinPreProcessorTimeOut,TimeOut) else TimeOut:=DefPreProcessorTimeOut;
     if ReadIniFileInteger(ConfigFile,SectDaq,'PreProcessorDisplay%i',Display)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PreProcessorDisplay%i',Display)
     then Display:=Max(SW_HIDE,Display) else Display:=SW_HIDE;
     if ReadIniFileInteger(ConfigFile,SectDaq,'PreProcessorPipeSize%i',PipeSize)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PreProcessorPipeSize%i',PipeSize)
     then PipeSize:=Max(MinPreProcessorPipeSize,PipeSize) else PipeSize:=DefPreProcessorPipeSize;
     if ReadIniFileString(ConfigFile,SectDaq,'PreProcessorPriority%s',Priority)
     or ReadIniFileString(SysIniFile,SectDaqSys,'PreProcessorPriority%s',Priority)
     then Priority:=Trim(Priority) else Priority:='Normal,tpNormal,tpNormal,tpNormal';
    end;
   end;
  end;
  {}
  with myPostProcessor do begin
   CmdLine:=''; TimeOut:=0; Display:=0; PipeSize:=0; Priority:='';
   if ReadIniFileString(ConfigFile,SectDaq,'PostProcessor%s',CmdLine) then begin
    CmdLine:=AdaptExeFileName(CmdLine);
    if IsNonEmptyStr(CmdLine) then begin
     if ReadIniFileInteger(ConfigFile,SectDaq,'PostProcessorTimeOut%i',TimeOut)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PostProcessorTimeOut%i',TimeOut)
     then TimeOut:=Max(MinPostProcessorTimeOut,TimeOut) else TimeOut:=DefPostProcessorTimeOut;
     if ReadIniFileInteger(ConfigFile,SectDaq,'PostProcessorDisplay%i',Display)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PostProcessorDisplay%i',Display)
     then Display:=Max(SW_HIDE,Display) else Display:=SW_HIDE;
     if ReadIniFileInteger(ConfigFile,SectDaq,'PostProcessorPipeSize%i',PipeSize)
     or ReadIniFileInteger(SysIniFile,SectDaqSys,'PostProcessorPipeSize%i',PipeSize)
     then PipeSize:=Max(MinPostProcessorPipeSize,PipeSize) else PipeSize:=DefPostProcessorPipeSize;
     if ReadIniFileString(ConfigFile,SectDaq,'PostProcessorPriority%s',Priority)
     or ReadIniFileString(SysIniFile,SectDaqSys,'PostProcessorPriority%s',Priority)
     then Priority:=Trim(Priority) else Priority:='Normal,tpNormal,tpNormal,tpNormal';
    end;
   end;
  end;
  {}
  if ReadIniFileInteger(ConfigFile,SectDaq,'CalibDialogMode%i',CalibDialogMode)
  or ReadIniFileInteger(SysIniFile,SectDaqSys,'CalibDialogMode%i',CalibDialogMode)
  then {Ok} else CalibDialogMode:=cdm_Default;
  { [DAQ] OutFifoCleanDelay = 10000 }
  if ReadIniFileInteger(ConfigFile,SectDaq,'OutFifoCleanDelay%i',d)
  or ReadIniFileInteger(SysIniFile,SectDaqSys,'OutFifoCleanDelay%i',d)
  then if (d>=0) then TProgramDevice.OutFifoCleanDelay:=d;
  { [DAQ] OutFifoCleanRunCount = 1e5 }
  if ReadIniFileDouble(ConfigFile,SectDaq,'OutFifoCleanRunCount%f',f)
  or ReadIniFileDouble(SysIniFile,SectDaqSys,'OutFifoCleanRunCount%f',f)
  then if (f>=0) then TProgramDevice.OutFifoCleanRunCount:=f;
 end;
end;

 {
 Разбор секции [DataStorage] и [TagList]. Создание списка кривых и тегов.
 Начальная инициализация статических кривых - спектров.
 }
procedure TDaqSystem.ReadDataStorageSection;
var i,j:Integer; p:TText;
begin
 if Ok then begin
  Curves.Count:=0;
  ReadCurveList(Curves, ConfigFile, '[DataStorage]');
  for i:=0 to Curves.Count-1 do
  if Curves[i].Ok and Curves[i].IsStatic then
  for j:=0 to Curves[i].Count-1 do Curves[i][j]:=Point2D(j,0);
  ClearTags;
  p:=NewText;
  ReadTags(ConfigFile, '[TagList]', p);
  for i:=0 to p.Count-1 do AddWarning(p[i]);
  Kill(p);
 end;
end;

 {
 Чтение конфигурации окон из секции [Windows].
 Если DaqDataWin = nil, берем кривые из базы данных DAQ, при этом окна не владеют
 данными и защищены от закрытия.
 Если DaqDataWin <> nil, копируем кривые из окна, при этом окна владеют данными
 и не защищены от закрытия.
 }
procedure TDaqSystem.ReadWindowsSection(wt:word; DaqDataWin:TFormCurveWindow);
var i,Num:Integer; NameList:TText;
 {
 Разбор конфигурации окна типа TFormCurveWindow с данным именем
 }
 procedure InsertCurWin(const WinName:LongString);
 var
  i       : LongInt;
  WinType : LongString;
  aName   : LongString;
  Win     : TFormCurveWindow;
  AxisX   : packed record Name:PureString; MinV,MaxV:Double end;
  AxisY   : packed record Name:PureString; MinV,MaxV:Double end;
  Curve   : TCurve;
  List    : TText;
  WTAX    : Boolean;
  UsesDS  : Boolean;
 begin
  WinType:=''; WTAX:=false;
  {Получить и проверить тип окна}
  if IsNonEmptyStr(WinName) then
  if ReadIniFileAlpha(ConfigFile,'[Windows]',WinName+'%a',WinType) then
  if SameText(WinType,'CURVE_WINDOW') then begin
   {Задать значения по умолчанию}
   aName:=WinName;
   with AxisX do begin Name:=' '; MinV:=0; MaxV:=1; end;
   with AxisY do begin Name:=' '; MinV:=0; MaxV:=1; end;
   UsesDS:=true;
   {Чтение значений из файла}
   ReadIniFileString(ConfigFile,'['+WinName+']','Name%s',aName);
   ReadIniFileRecord(ConfigFile,'['+WinName+']','AxisX%a;%f;%f',AxisX,efConfigNC);
   ReadIniFileRecord(ConfigFile,'['+WinName+']','AxisY%a;%f;%f',AxisY,efConfigNC);
   ReadIniFileBoolean(ConfigFile,'['+WinName+']','UsesDownSampling%b',UsesDS);
   {Коррекция параметров окна}
   aName:=TrimChars(aName,ScanSpaces,ScanSpaces);
   with AxisY do if (Pos('^',Name)>0) then DaqAlignStr(Name) else Name:=^C' '+ASCII_CR+^L+Name;
   with AxisX do if (Pos('^',Name)>0) then DaqAlignStr(Name) else Name:=^R+Name+ASCII_CR+^C' ';
   {Чтение WantTimeAxisX }
   if ReadIniFileBoolean(ConfigFile,'['+WinName+']','WantTimeAxisX%b',WTAX) then {ok} else
   if ReadIniFileBoolean(ConfigFile,SectDaq,        'WantTimeAxisX%b',WTAX) then {ok} else
   if ReadIniFileBoolean(SysIniFile,SectDaqSys,     'WantTimeAxisX%b',WTAX) then {ok} else WTAX:=false;
   {Создание окна}
   Win:=NewCurveWindow(aName, AxisY.Name, AxisX.Name);
   if Win.Ok then begin
    {Чтение списка имен кривых CurveList=c1, c2, c3,...}
    List:=ExtractEnumWordList(ConfigFile,'['+WinName+']','CurveList',efConfig);
    {Задание списка кривых и параметров окна}
    if (DaqDataWin=nil) then begin
     {
     В случае реальной конфигурации, при вызове с DaqDataWin = nil, кривые
     берутся как ссылки из списка Curves. Кривые при этом не принадлежат окну,
     поэтому надо защитить окно и данные. Окно также вставляется в список окон
     DAQ и включается в список мониторинга для отображения в динамике.
     }
     if List.Ok and (List.Count>0) then begin
      for i:=0 to List.Count-1 do begin
       Curve:=Curves.Find(List[i]);
       if Curve.Ok
       then Win.AddCurve(Curve)
       else AddWarning('['+WinName+'] -> Not found curve '+List[i]);
      end;
     end else AddWarning('['+WinName+'] -> CurveList empty in '+Win.Caption);
     Win.CloseAction:=caMinimize;
     Win.IsDataProtected:=true;
     Win.StartMonitoring;
     CurWinList.Add(Win);
     Win.DaqRef:=Daq.Ref;
    end else begin
     {
     В случае загрузки данных без DAQ, то есть DaqDataWin <> nil, кривые
     копируются из окна. При этом кривые принадлежат окну, поэтому не надо
     защищать окна и данные, также не надо вставлять в окна в список DAQ.
     }
     if List.Ok and (List.Count>0) then begin
      for i:=0 to List.Count-1 do begin
       Curve:=NewCurveCopy(DaqDataWin.Curves.Find(List[i]));
       if Curve.Ok
       then Win.AddCurve(Curve)
       else AddWarning('['+WinName+'] -> Not found curve '+List[i]);
      end;
     end else AddWarning('['+WinName+'] -> CurveList empty in '+Win.Caption);
    end;
    {Удалить список имен}
    Kill(List);
    {Коррекция пределов и размеров и прорисовка}
    Win.World:=Rect2D(AxisX.MinV, AxisY.MinV, AxisX.MaxV, AxisY.MaxV);
    Win.Width:=350;
    Win.Height:=250;
    Win.Left:=20*Num;
    Win.Top:=20*Num;
    Win.DefCurveNum:=-1;
    Win.WantTimeAxisX:=WTAX;
    Win.UsesDownSampling:=UsesDS;
    Win.Update;
    inc(Num);
   end;
  end;
 end;
 {
 Разбор конфигурации окна типа TFormTabWindow с данным именем
 }
 procedure InsertTabWin(const WinName:LongString);
 var
  i       : LongInt;
  aName   : LongString;
  WinType : LongString;
  Win     : TFormTabWindow;
  Curve   : TCurve;
  Tag     : Integer;
  List    : TText;
 begin
  WinType:='';
  {Получить и проверить тип окна}
  if IsNonEmptyStr(WinName) then
  if ReadIniFileAlpha(ConfigFile,'[Windows]',WinName+'%a',WinType) then
  if SameText(WinType,'TAB_WINDOW') then begin
   {Задать значения по умолчанию}
   aName:=WinName;
   {Чтение значений из файла}
   ReadIniFileString(ConfigFile,'['+WinName+']','Name%s',aName);
   {Коррекция параметров окна}
   aName:=TrimChars(aName,ScanSpaces,ScanSpaces);
   {Cоздание окна. Только в случае реальной конфигурации.}
   if DaqDataWin=nil then Win:=NewTabWindow(aName) else Win:=nil;
   if Win.Ok then begin
    {Чтение списка кривых CurveList=c1,c2,c3,...}
    List:=ExtractEnumWordList(ConfigFile,'['+WinName+']','CurveList',efConfig);
    try
     for i:=0 to List.Count-1 do begin
      Curve:=Curves.Find(List[i]);
      if Curve.Ok
      then Win.AddItem(Curve,0,'')
      else AddWarning('['+WinName+'] -> Not found curve '+List[i]);
     end;
    finally
     Kill(List);
    end;
    {Чтение списка тегов TagList=t1,t2,t3,...}
    List:=ExtractEnumWordList(ConfigFile,'['+WinName+']','TagList',efConfig);
    try
     for i:=0 to List.Count-1 do begin
      Tag:=FindTag(List[i]);
      if (Tag<>0)
      then Win.AddItem(nil,Tag,'')
      else AddWarning('['+WinName+'] -> Not found tag '+List[i]);
     end;
    finally
     Kill(List);
    end;
    {Конфигурирование и добавление окна в список}
    if Win.Items.Count=0 then AddWarning('['+WinName+'] -> CurveList and TagList empty in '+Win.Caption);
    Win.Config(ConfigFile,'['+WinName+']');
    Win.CloseAction:=caMinimize;
    Win.StartMonitoring;
    TabWinList.Add(Win);
    Win.DaqRef:=Daq.Ref;
    {Обновить таблицу}
    Win.ShouldResize;
    Win.UpdateGrid;
    Win.UpdateSize;
    {Коррекция пределов и размеров и прорисовка}
    Win.Left:=ScreenWorkAreaRect.Right-Win.Width-20*Num-(ScreenWorkAreaRect.Width div 5);
    Win.Top:=20*Num;
    Win.Update;
    inc(Num);
   end;
  end;
 end;
 {
 Разбор конфигурации окна типа TFormCircuitWindow с данным именем
 }
 procedure InsertCirWin(const WinName:LongString);
 var Win:TFormCircuitWindow; WinType:LongString;
 begin
  WinType:='';
  if IsNonEmptyStr(WinName) then
  if ReadIniFileAlpha(ConfigFile,'[Windows]',WinName+'%a',WinType) then
  if SameText(WinType,'CIRCUIT_WINDOW') then begin
   Win:=NewCircuitWindowFromCfgFile(ConfigFile,'['+WinName+']','Circuit');
   if Win.Ok then begin
    Win.CloseAction:=caMinimize;
    Win.StartMonitoring;
    CirWinList.Add(Win);
    Win.DaqRef:=Daq.Ref;
    Win.Left:=ScreenWorkAreaRect.Right-Win.Width-20*Num;
    Win.Top:=ScreenWorkAreaRect.Bottom-Win.Height-20*Num;
    Win.Update;
    inc(Num);
   end;
  end;
 end;
 {
 Разбор конфигурации окна типа TFormSpectrWindow с данным именем
 }
 procedure InsertSpeWin(const WinName:LongString);
 var WinType,aName,Crv,wClass:LongString;
 var Curve:TCurve; Win:TFormSpectrWindow;
 begin
  WinType:=''; wClass:='';
  {получить и проверить тип окна}
  if IsNonEmptyStr(WinName) then
  if ReadIniFileAlpha(ConfigFile,'[Windows]',WinName+'%a',WinType) then
  if SameText(WinType,'SPECTR_WINDOW') then
  if ReadIniFileAlpha(ConfigFile,'['+WinName+']','WindowClass%a',wClass) then begin
   {чтение кривой}
   Curve:=nil;
   Crv:='';
   if ReadIniFileAlpha(ConfigFile,'['+WinName+']','Spectr%a',Crv) then
   if DaqDataWin=nil
   then Curve:=Curves.Find(Crv)
   else Curve:=NewCurveCopy(DaqDataWin.Curves.Find(Crv));
   {чтение имени окна}
   aName:=WinName;
   ReadIniFileString(ConfigFile,'['+WinName+']','Name%s',aName);
   aName:=TrimChars(aName,ScanSpaces,ScanSpaces);
   {создание окна}
   Win:=NewSpectrWindow(wClass, aName, Curve, DaqDataWin<>nil);
   if Win.Ok then begin
    {Задание параметров окна}
    if (DaqDataWin=nil) then begin
     Win.CloseAction:=caMinimize;
     SpeWinList.Add(Win);
     Win.DaqRef:=Daq.Ref;
    end;
    {Конфигурация окна}
    Win.Config(ConfigFile,'['+WinName+']');
    {Коррекция пределов, размеров и прорисовка}
    Win.Left:=20*Num;
    Win.Top:=20*Num;
    Win.Update;
    inc(Num);
   end else begin
    AddWarning('['+WinName+'] -> Not found or invalid spectr '+crv);
    if (DaqDataWin<>nil) then Kill(Curve);
   end;
  end;
 end;
begin
 if Ok then
 try
  NameList:=ExtractWordList(ConfigFile, '[Windows]', efConfig, 1, ScanSpaces);
  try
   if NameList.Ok then begin
    WinClose(wt);
    if HasFlags(wt,wtCurWin) and Assigned(CurWinList) then begin
     Num:=0;
     for i:=0 to NameList.Count-1 do InsertCurWin(NameList[i]);
    end;
    if HasFlags(wt,wtTabWin) and Assigned(TabWinList) then begin
     Num:=0;
     for i:=0 to NameList.Count-1 do InsertTabWin(NameList[i]);
    end;
    if HasFlags(wt,wtCirWin) and Assigned(CirWinList) then begin
     Num:=0;
     for i:=0 to NameList.Count-1 do InsertCirWin(NameList[i]);
    end;
    if HasFlags(wt,wtSpeWin) and Assigned(SpeWinList) then begin
     Num:=0;
     for i:=0 to NameList.Count-1 do InsertSpeWin(NameList[i]);
    end;
   end;
  finally
   Kill(NameList);
  end;
 except
  on E:Exception do BugReport(E,Self,'ReadWindowsSection'); 
 end;
end;

procedure _UpdateWindowTimes(Index:LongInt; const aObject:TObject; var Terminate:Boolean; Custom:Pointer);
begin
 if (aObject is TFormCurveWindow) then
 if (TObject(Custom) is TDaqSystem) then begin
  TFormCurveWindow(aObject).TimeBase:=TDaqSystem(TObject(Custom)).Timer.StartTime;
  TFormCurveWindow(aObject).TimeUnits:=TDaqSystem(TObject(Custom)).Timer.LocalTimeUnits;
 end;
end;

 {
 Обновить поля TimeBase, TimeUnits
 }
procedure TDaqSystem.UpdateWindowsTimeBaseTimeUnits;
begin
 if Ok then CurWinList.ForEach(_UpdateWindowTimes,Self);
end;

 {
 Сохранить положение, размер и другие параметры окон в *.dsk - файле
 }
procedure TDaqSystem.SaveWinLocations(wt:word);
var p:TText; i:Integer;
 function WinAlias(Form:TForm):LongString;
 begin
  if (Form is TForm) then Result:=ReplaceString(Form.Caption,' ','_') else Result:='?';
 end;
begin
 if Ok then begin
  p:=NewText;
  if HasFlags(wt,wtCurWin) then
  for i:=0 to CurWinList.Count-1 do
  if CurWinList[i].Ok then begin
   p.Addln('[Windows]');
   p.Addln(Format('%s = %s', [WinAlias(CurWinList[i]), 'CURVE_WINDOW']));
   p.Addln('['+WinAlias(CurWinList[i])+']');
   with CurWinList[i] do
   p.Addln(Format('Bounds = %d, %d, %d, %d',[Left, Top, Left+Width, Top+Height]));
   with CurWinList[i].World do
   p.Addln(Format('Limits = %g, %g, %g, %g',[A.X, A.Y, B.X, B.Y]));
  end;
  if HasFlags(wt,wtTabWin) then
  for i:=0 to TabWinList.Count-1 do
  if TabWinList[i].Ok then begin
   p.Addln('[Windows]');
   p.Addln(Format('%s = %s', [WinAlias(TabWinList[i]), 'TAB_WINDOW']));
   p.Addln('['+WinAlias(TabWinList[i])+']');
   with TabWinList[i] do
   p.Addln(Format('Bounds = %d, %d, %d, %d',[Left, Top, Left+Width, Top+Height]));
  end;
  if HasFlags(wt,wtCirWin) then
  for i:=0 to CirWinList.Count-1 do
  if CirWinList[i].Ok then begin
   p.Addln('[Windows]');
   p.Addln(Format('%s = %s', [WinAlias(CirWinList[i]), 'CIRCUIT_WINDOW']));
   p.Addln('['+WinAlias(CirWinList[i])+']');
   with CirWinList[i] do
   p.Addln(Format('Bounds = %d, %d, %d, %d',[Left, Top, Left+Width, Top+Height]));
  end;
  if HasFlags(wt,wtSpeWin) then
  for i:=0 to SpeWinList.Count-1 do
  if SpeWinList[i].Ok then begin
   p.Addln('[Windows]');
   p.Addln(Format('%s = %s', [WinAlias(SpeWinList[i]), 'SPECTR_WINDOW']));
   p.Addln('['+WinAlias(SpeWinList[i])+']');
   with SpeWinList[i] do
   p.Addln(Format('Bounds = %d, %d, %d, %d',[Left, Top, Left+Width, Top+Height]));
  end;
  i:=p.WriteFile(ForceExtension(ForcePath(TempPath,ConfigFile),'.dsk'));
  Kill(p);
  Trouble(i<>0,RusEng('Ошибка записи!','Write error!'));
 end;
end;

 {
 Восстановить параметры окон из файла
 }
procedure TDaqSystem.LoadWinLocations(wt:word);
var
 i     : Integer;
 R,D   : TRect2I;
 L     : TRect2D;
 fname : LongString;
 wtype : LongString;
 function WinAlias(Form:TForm):LongString;
 begin
  if (Form is TForm) then Result:=ReplaceString(Form.Caption,' ','_') else Result:='?';
 end;
 procedure LoadCurWin(W:TFormCurveWindow);
 begin
  if W.Ok then
  if ReadIniFileAlpha(fname,'[Windows]',WinAlias(W)+'%a',wtype) then
  if SameText(wtype,'CURVE_WINDOW') then begin
   if ReadIniFileRecord(fname,'['+WinAlias(W)+']','Bounds%i;%i;%i;%i',R) then begin
    D:=Rect2I(ScreenWorkAreaRect);
    if RectIsEqual(R,RectIntersection(R,D)) and not RectIsEmpty(R) then begin
     W.Left:=R.A.X;
     W.Top:=R.A.Y;
     W.Width:=R.B.X-R.A.X;
     W.Height:=R.B.Y-R.A.Y;
     W.Update;
    end;
   end;
   if ReadIniFileRecord(fname,'['+WinAlias(W)+']','Limits%f;%f;%f;%f',L) then
   if not RectIsEmpty(L) then W.World:=L;
  end;
 end;
 procedure LoadTabWin(W:TFormTabWindow);
 begin
  if W.Ok then
  if ReadIniFileAlpha(fname,'[Windows]',WinAlias(W)+'%a',wtype) then
  if SameText(wtype,'TAB_WINDOW') then begin
   if ReadIniFileRecord(fname,'['+WinAlias(W)+']','Bounds%i;%i;%i;%i',R) then begin
    D:=Rect2I(ScreenWorkAreaRect);
    if RectIsEqual(R,RectIntersection(R,D)) and not RectIsEmpty(R) then begin
     W.Left:=R.A.X;
     W.Top:=R.A.Y;
     W.Width:=R.B.X-R.A.X;
     W.Height:=R.B.Y-R.A.Y;
     W.Update;
    end;
   end;
  end;
 end;
 procedure LoadCirWin(W:TFormCircuitWindow);
 begin
  if W.Ok then
  if ReadIniFileAlpha(fname,'[Windows]',WinAlias(W)+'%a',wtype) then
  if SameText(wtype,'CIRCUIT_WINDOW') then begin
   if ReadIniFileRecord(fname,'['+WinAlias(W)+']','Bounds%i;%i;%i;%i',R) then begin
    D:=Rect2I(ScreenWorkAreaRect);
    if RectIsEqual(R,RectIntersection(R,D)) and not RectIsEmpty(R) then begin
     W.Left:=R.A.X;
     W.Top:=R.A.Y;
     W.Width:=R.B.X-R.A.X;
     W.Height:=R.B.Y-R.A.Y;
     W.Update;
    end;
   end;
  end;
 end;
 procedure LoadSpeWin(W:TFormSpectrWindow);
 begin
  if W.Ok then
  if ReadIniFileAlpha(fname,'[Windows]',WinAlias(W)+'%a',wtype) then
  if SameText(wtype,'SPECTR_WINDOW') then begin
   if ReadIniFileRecord(fname,'['+WinAlias(W)+']','Bounds%i;%i;%i;%i',R) then begin
    D:=Rect2I(ScreenWorkAreaRect);
    if RectIsEqual(R,RectIntersection(R,D)) and not RectIsEmpty(R) then begin
     W.Left:=R.A.X;
     W.Top:=R.A.Y;
     W.Width:=R.B.X-R.A.X;
     W.Height:=R.B.Y-R.A.Y;
     W.Update;
    end;
   end;
  end;
 end;
begin
 if Ok then begin
  wtype:='';
  fname:=ForceExtension(ForcePath(TempPath,ConfigFile),'.dsk');
  if FileExists(fname) then begin
   if HasFlags(wt,wtCurWin) then for i:=0 to CurWinList.Count-1 do LoadCurWin(CurWinList[i]);
   if HasFlags(wt,wtTabWin) then for i:=0 to TabWinList.Count-1 do LoadTabWin(TabWinList[i]);
   if HasFlags(wt,wtCirWin) then for i:=0 to CirWinList.Count-1 do LoadCirWin(CirWinList[i]);
   if HasFlags(wt,wtSpeWin) then for i:=0 to SpeWinList.Count-1 do LoadSpeWin(SpeWinList[i]);
  end;
 end;
end;

 {
 Метод должен связывать и оживлять загруженные методом ReadConfig
 обьекты - схемы, блоки, кривые ...
 }
procedure TDaqSystem.Animate;
var i:Integer;
begin
 if Ok then
 for i:=0 to SpeWinList.Count-1 do if SpeWinList[i].Ok then SpeWinList[i].Animate;
end;

 {
 Метод должен показывать/скрывать логотип при загрузке DAQ системы.
 }
procedure TDaqSystem.ShowLogo(const aLogo:LongString);
begin
 if Ok then
 try
  if IsNonEmptyStr(aLogo) and FileExists(myDaqDesktop)
  then ShowFormCrwDaqLogo('CRW-DAQ LOGO',aLogo,myDaqDesktop,true)
  else HideFormCrwDaqLogo;
 except
  on E:Exception do BugReport(E,Self,'ShowLogo');
 end;
end;

procedure Timer_DaqShowLogoHide;
begin
 if (DaqShowLogoDeadline>0) then
 if (IntMSecNow>DaqShowLogoDeadline) then begin
  DaqShowLogoDeadline:=0;
  Daq.ShowLogo('');
 end;
end;

procedure TDaqSystem.ShowLogoDelay(aDelay:Integer);
begin
 if Ok then
 if (aDelay>0) then begin
  SecondActions.Add(Timer_DaqShowLogoHide);
  DaqShowLogoDeadline:=IntMSecNow+aDelay;
 end else begin
  DaqShowLogoDeadline:=0;
 end;
end;

 {
 Процедура вызывается перед загрузкой сессии для установки окружения.
 }
procedure TDaqSystem.InitEnvironment;
var p:TText; i:Integer;
begin
 SetEnv('CRW_DAQ_CONFIG_FILE',GetRealFilePathName(ConfigFile));
 SetEnv('CRW_DAQ_CONFIG_HOME_DIR',GetRealFilePathName(ConfigPath));
 SetEnv('CRW_DAQ_CONFIG_DATA_PATH',GetRealFilePathName(DataPath));
 SetEnv('CRW_DAQ_CONFIG_TEMP_PATH',GetRealFilePathName(TempPath));
 SetEnv('CRW_DAQ_CONFIG_LOAD_TIME',Format('%g',[mSecNow]));
 SetEnv('CRW_DAQ_CONFIG_BASE_TIME',Format('%g',[Timer.StartTime]));
 SetEnv('CRW_DAQ_CONFIG_TIME_UNITS',Format('%g',[Timer.LocalTimeUnits]));
 SetEnv('CRW_DAQ_CONFIG_START_TIME','0');
 SetEnv('CRW_DAQ_CONFIG_PATH',GetRealFilePathName(ConfigPath)+PathSep);
 //AddSearchPath('CRW_DAQ_CONFIG_PATH',DataPath);
 //AddSearchPath('CRW_DAQ_CONFIG_PATH',TempPath);
 p:=ExtractEnumWordList(ConfigFile,SectDaq,'SearchPath',efConfig);
 for i:=0 to p.Count-1 do if MaybeEnvStr(p[i]) then p[i]:=ExpEnv(p[i]);
 for i:=0 to p.Count-1 do AddSearchPath('CRW_DAQ_CONFIG_PATH',GetRealFilePathName(FileRef(AdaptFileName(p[i]))));
 Kill(p);
 p:=ExtractEnumWordList(SysIniFile,SectDaqSys,'IncludePath',efConfig);
 for i:=0 to p.Count-1 do if MaybeEnvStr(p[i]) then p[i]:=ExpEnv(p[i]);
 for i:=0 to p.Count-1 do AddSearchPath('CRW_DAQ_INCLUDE_PATH',GetRealFilePathName(FileRef(AdaptFileName(p[i]),'',ProgName)));
 Kill(p);
 ValidateEnvPathList('CRW_DAQ_CONFIG_PATH,CRW_DAQ_INCLUDE_PATH');
end;

 {
 Процедура вызывается после завершения сессии для очистки окружения.
 }
procedure TDaqSystem.FreeEnvironment;
begin
 SetEnv('CRW_DAQ_CONFIG_FILE','');
 SetEnv('CRW_DAQ_CONFIG_HOME_DIR','');
 SetEnv('CRW_DAQ_CONFIG_DATA_PATH','');
 SetEnv('CRW_DAQ_CONFIG_TEMP_PATH','');
 SetEnv('CRW_DAQ_CONFIG_LOAD_TIME','');
 SetEnv('CRW_DAQ_CONFIG_BASE_TIME','');
 SetEnv('CRW_DAQ_CONFIG_TIME_UNITS','');
 SetEnv('CRW_DAQ_CONFIG_START_TIME','');
 SetEnv('CRW_DAQ_INCLUDE_PATH','');
 SetEnv('CRW_DAQ_CONFIG_PATH','');
end;

 {
 Процедура вызывается перед чтением конфигурации для запуска препроцессора.
 }
procedure TDaqSystem.RunPreProcessor;
begin
end;

 {
 Процедура вызывается после завершения сессии для запуска постпроцессора.
 }
procedure TDaqSystem.RunPostProcessor;
begin
end;

 {
 Процедура загружает конфигурацию DAQ и начинает сеанс работы DAQ.
 }
procedure TDaqSystem.InitSession(CfgFile:LongString='');
var Params:LongString;
begin
 if Guard.Level<ga_Root then
 if not Guard.Trusted(CfgFile) then begin
  Echo(RusEng('Guard: Конфигурация не найдена в списке разрешенных.'+EOL+
              '       '+CfgFile,
              'Guard: Configuration is not trusted, access denied.'+EOL+
              '       '+CfgFile));
  Guard.Log:='Deny '+Guard.LevelName[Guard.Level]+', '+CfgFile;
  _crw_snd.Voice(Guard.Sound);
  SystemConsole.Activate;
  Exit;
 end;
 Params:='';
 if FormDaqControlDialog.Ok and FormDaqControlDialog.Active
 then Params:=ControlPosParams(FormDaqControlDialog.ToolButtonDaqInit,'RT');
 if Ok then
 if not Trouble(Timer.IsStart,RusEng('DAQ: Система уже загружена!','DAQ is already loaded!')) then begin
  if IsNonEmptyStr(CfgFile) then myConfigFile:=UnifyFileAlias(CfgFile,ua_FileDefLow);
  UpdateSysLogSign; // Update sign of DAQ system for SysLog identification
  if not IsWildCard(ConfigFile) then begin
   if NoProblem(FileExists(ConfigFile),RusEng('Нет файла ','Could not find')+ConfigFile,Params) then begin
    try
     myFixPriority.Wanted:=WantedPriorityClass;
     myFixPriority.Period:=PeriodPriorityClass;
     InitErrMsgTable;
     TCircuitSensorTagEvalErrorCount:=0;
     TCircuitSensorLedEvalErrorCount:=0;
     TCircuitSensorPainterErrorCount:=0;
     DynamicCurWinNumber:=0;
     DynamicTabWinNumber:=0;
     WarningList.Count:=0;
     SetTagBugs(0);
     ReadConfig;
     Animate;
     OpenDaqDataWindow;
     TouchLogBook;
     StartSession;
     Dispatcher.Enable(true,DefaultDaqTimeOut);
     if myAutoStart or (SystemCalculator.Eval('_Daq_Force_Start_')=1) then Start;
    finally
     if (DaqShowLogoDeadline<=0)
     then ShowLogo('');
    end;
   end;
  end;
 end;
end;

 {
 Завершение сеанса DAQ
 }
procedure TDaqSystem.DoneSession;
begin
 if Ok then
 if CheckSessionStarted and not CheckAcquisitionStarted then begin
  if SolidCrwDaqLogo then
  ShowLogo(RusEng('Подождите, идет завершение CRW-DAQ...','Wait while finalizing CRW-DAQ...'));
  try
   Dispatcher.Enable(false,DefaultDaqTimeOut);
   Timer.Stop;
   WinClose(wtSpeWin); {???!!!}
   CalibDialogMode:=ReadCalibDialogMode(SysIniFile,SectDaqSys);
   KillFormCalibDialog;
   KillFormDaqHelpViewer;
   KillFormDaqHistoryWarning;
   KillFormDaqDateTimeCalculator;
   StopSession;
   Kill(myDaqDataWindow);
   SaveLogBook;
   Kill(myLogBook);
   WinClose(wtAll);
   ReadCurveWindowsPalette(SysIniFile);
   ReadSpectrWindowsPalette(SysIniFile);
   ReadSurfaceWindowsPalette(SysIniFile);
   Curves.Count:=0;
   ClearTags;
   WarningList.Count:=0;
   ForcePriorityClass(myFixPriority.Wanted,myFixPriority.Period);
   RunPostProcessor;
   FreeEnvironment;
  finally
   ShowLogo('');
  end;
 end;
end;

 {
 Действия по началу сеанса DAQ после загрузки и анимации.
 Read date-time as Day.Month.Year-Hour:Minut:Second, 05.04.1999-18:48:11
 }
procedure TDaqSystem.StartSession;
type
 DD_MM_YY_HH_MM_SS = record day,month,year,hour,min,sec:word; end;
var
 s    : LongString;
 Data : DD_MM_YY_HH_MM_SS;
 t0   : Double;
 function ReadDateTime(const s:LongString; var d:DD_MM_YY_HH_MM_SS):Boolean;
 const Delims:TCharSet=[';','-','.',' ',':'];
 var sday,smonth,syear,shour,smin,ssec:LongString;
 begin
  sday:=extractword(1,s,Delims);
  smonth:=extractword(2,s,Delims);
  syear:=extractword(3,s,Delims);
  shour:=extractword(4,s,Delims);
  smin:=extractword(5,s,Delims);
  ssec:=extractword(6,s,Delims);
  with mSecToNativeTime(mSecNow) do begin
   if IsSameText(sday,'*') then sday:=IntToStr(day);
   if IsSameText(smonth,'*') then smonth:=IntToStr(month);
   if IsSameText(syear,'*') then syear:=IntToStr(year);
   if IsSameText(shour,'*') then shour:=IntToStr(hour);
   if IsSameText(smin,'*') then smin:=IntToStr(minute);
   if IsSameText(ssec,'*') then ssec:=IntToStr(second);
  end;
  with d do
  Result:=Str2Word(sday,   day)   and
          Str2Word(smonth, month) and
          Str2Word(syear,  year)  and
          Str2Word(shour,  hour)  and
          Str2Word(smin,   min)   and
          Str2Word(ssec,   sec);
 end;
begin
 if Ok then begin
  s:=''; SafeFillChar(Data,SizeOf(Data),0);
  if ReadIniFileString(ConfigFile,SectDaq,'TimeBase%s',s) then begin
   s:=TrimChars(s,[' ','='],[' ','=']);
   if ReadDateTime(s,Data) then begin
    with Data do t0:=DateTimeToMsec(year,month,day,hour,min,sec,0);
    if IsNanOrInf(t0)
    then AddWarning('Invalid date/time: '+s)
    else Timer.StartAt(t0);
    UpdateWindowsTimeBaseTimeUnits;
   end else AddWarning('Could not read date/time: '+s);
  end;
  DaqTopListMonitor_Start;
  CurveWindowsProfiler_Start;
  TableWindowsProfiler_Start;
  CircuitWindowsProfiler_Start;
  HistTimer.Start;
  SpecTimer.Start;
  OpenConsole(myConsoleMode and cmfInit=0);
  ConsoleEcho('');
  ConsoleEcho(StdDateTimePrompt+RusEng('Начало сеанса CRW-DAQ.','Start CRW-DAQ session.'));
  ConsoleEcho(RusEng('Файл ','File ')+ConfigFile);
  ConsoleEcho(BriefDaqSysInfoCounters);
  ConsoleEcho(BriefDaqSysBugsCounters);
  if WarningList.Count>0 then ConsoleEcho(RusEng('Обнаружены проблемы!','Problems found!'));
  SetEnv('CRW_DAQ_CONFIG_BASE_TIME',Format('%g',[Timer.StartTime]));
  SetEnv('CRW_DAQ_CONFIG_TIME_UNITS',Format('%g',[Timer.LocalTimeUnits]));
  SendToMainConsole('@silent @integrity enter.daq '+ConfigFile+EOL);
 end;
end;

 {
 Действия по завершению сеанса DAQ
 }
procedure TDaqSystem.StopSession;
begin
 if Ok then begin
  SpecTimer.Stop;
  HistTimer.Stop;
  CircuitWindowsProfiler_Stop;
  TableWindowsProfiler_Stop;
  CurveWindowsProfiler_Stop;
  DaqTopListMonitor_Stop;
  KillFormDaqHistoryWarning;
  myDaqDesktop:='';
  OpenConsole(myConsoleMode and cmfDone=0);
  ConsoleEcho(StdDateTimePrompt+RusEng('Конец сеанса CRW-DAQ.','Stop CRW-DAQ session.'));
  ConsoleEcho(RusEng('Файл ','Files ')+ConfigFile);
  ConsoleEcho('');
  SendToMainConsole('@silent @integrity leave.daq '+ConfigFile+EOL);
  myConsoleMode:=cmfDefault;
  myNavigator:='';
  myStopConfirm:='';
  myVoiceEcho:=false;
  myConsoleFile:='';
 end;
end;

 {
 Процедура пуска измерений
 }
function TDaqSystem.Start:Boolean;
begin
 Result:=false;
 SystemCalculator.Eval('_Crw_Force_Exit_=0');
 SystemCalculator.Eval('_Daq_Force_Exit_=0');
 SystemCalculator.Eval('_Daq_Force_Stop_=0');
 SystemCalculator.Eval('_Daq_Force_Start_=0');
 if Ok then
 if CheckSessionStarted and not AcqTimer.IsStart then begin
  AcqTimer.Start;
  if (SaveTimer.IntervalMs[0]>0) then SaveTimer.Start;
  OpenConsole(myConsoleMode and cmfStart=0);
  ConsoleEcho(StdDateTimePrompt+RusEng('Старт DAQ.','Start DAQ.'));
  //Polling.Enable(true,DefaultDaqTimeOut); - see TCrwDaq.Start
  //Watchdog.Enable(true,DefaultDaqTimeOut); - see TCrwDaq.Start
  ShowLogoDelay(myLogoDelay);
  Result:=true;
 end;
end;

 {
 Процедура остановки измерений
 }
function TDaqSystem.Stop:Boolean;
 function Confirmed:Boolean;
 var Params:LongString;
 begin
  Params:='';
  if FormDaqControlDialog.Ok then
  if FormDaqControlDialog.Active then
  Params:=ControlPosParams(FormDaqControlDialog.ToolButtonDaqStop,'RT');
  Result:=ConfirmStop(Params);
 end;
begin
 Result:=false;
 if Ok then
 if CheckSessionStarted and AcqTimer.IsStart and Confirmed then begin
  Polling.Enable(false,DefaultDaqTimeOut);
  Watchdog.Enable(false,DefaultDaqTimeOut);
  OpenConsole(myConsoleMode and cmfStop=0);
  ConsoleEcho(StdDateTimePrompt+RusEng('Останов DAQ.','Stop DAQ'));
  if mySaveOnStop and FormDaqControlDialog.Ok
  then FormDaqControlDialog.DaqSaveExecute;
  AcqTimer.Stop;
  SaveTimer.Stop;
  Result:=true;
 end;
end;

 {
 Процедура очистки данных
 }
function TDaqSystem.Clear:Boolean;
var
 i : Integer;
begin
 Clear:=false;
 if Ok then
 if CheckSessionStarted and not CheckAcquisitionStarted then
 if YesNo(RusEng('Вы уверены, что хотите стереть все данные?',
                 'Do you really want to clear all data curves?'))=mrYes
 then begin
  for i:=0 to Curves.Count-1 do DaqClearCurve(Curves[i],0);
  KillFormDaqHistoryWarning;
  WinDraw(wtCurWin+wtTabWin+wtSpeWin);
  Clear:=true;
 end;
end;

 {
 Процедура выборочной очистки данных
 }
function TDaqSystem.CustomClear(const Params:LongString=''):Boolean;
var i,HistLen:Integer; List:TText;
begin
 Result:=false;
 if Ok then
 if CheckSessionStarted then begin
  List:=NewText;
  List.Text:=SelectCurveList(Params);
  if List.Count>0 then begin
   HistLen:=myHistLimits[0];
   if FormDaqHistLenDialogExecute(HistLen,Params)=mrOk then begin
    for i:=0 to List.Count-1 do DaqClearCurve(Curves.Find(List[i]),HistLen);
    KillFormDaqHistoryWarning;
    WinDraw(wtCurWin+wtTabWin+wtSpeWin);
    Result:=true;
   end;
  end;
  Kill(List);
 end;
end;

 {
 Действия в фоновом цикле таймера
 }
procedure TDaqSystem.Idle;
begin
 if Ok then begin
  if DaqDataWindow.Ok then DaqDataWindow.WindowState:=wsMinimized;
  ConsoleFlush;
  CheckHistoryLength;
  if Timer.IsStart then begin
   if AcqTimer.IsStart then begin
    if SaveTimer.Event then SaveBackup;
    if SpecTimer.Event then UpdateSpectralWindows;
   end;
  end;
 end;
end;

 {
 Действия в основном цикле опроса, который выполняется в другом потоке (!!!!).
 }
procedure TDaqSystem.Poll;
begin
end;

 {
 Действия по диспетчеризации событий, выполняется в другом потоке (!!!!).
 }
procedure TDaqSystem.DispatchEvents;
begin
end;

 {
 Общие свойства системы
 }
function TDaqSystem.CommonProperty(P:TText):TText;
begin
 Result:=P;
 if Ok then begin
  P.Addln(SectDaq);
  P.Addln('ConfigFile = '+ConfigFile);
  with PreProcessor do
  if IsNonEmptyStr(CmdLine) then begin
   P.Addln('PreProcessor = '+CmdLine);
   P.Addln('PreProcessorTimeOut = '+IntToStr(TimeOut));
   P.Addln('PreProcessorDisplay = '+IntToStr(Display));
   P.Addln('PreProcessorPipeSize = '+IntToStr(PipeSize));
   P.Addln('PreProcessorPriority = '+Priority);
  end;
  with PostProcessor do
  if IsNonEmptyStr(CmdLine) then begin
   P.Addln('PostProcessor = '+CmdLine);
   P.Addln('PostProcessorTimeOut = '+IntToStr(TimeOut));
   P.Addln('PostProcessorDisplay = '+IntToStr(Display));
   P.Addln('PostProcessorPipeSize = '+IntToStr(PipeSize));
   P.Addln('PostProcessorPriority = '+Priority);
  end;
  P.Addln('ProcessPriorityClass = '+d2s(GetPriorityClassLevel(mySetPriority.Wanted))+', '+d2s(mySetPriority.Period));
  P.Addln('BackFile = '+FileRel(BackupFile));
  P.Addln('DaqSite = '+FileRel(SitePath));
  P.Addln('DataPath = '+FileRel(DataPath));
  P.Addln('LogFile = '+FileRel(LogFile));
  P.Addln('HelpFile = '+FileRel(HelpFile));
  P.Addln('TimeUnit = '+f2s(Timer.LocalTimeUnits/1000)+' sec');
  P.Addln('Save = '+f2s(SaveTimer.IntervalMs[0]/Timer.LocalTimeUnits)+' units');
  P.Addln('AlphaStayFree = '+f2s(AlphaStayFree));
  P.Addln('UseBlaster = '+d2s(ord(UsesBlaster)));
  P.Addln('SoundOn = '+d2s(ord(SoundOn)));
  P.Addln('InquiryPeriod = '+f2s(AcqTimer.IntervalMs[0]));
  P.Addln('SaveOnStop = '+d2s(ord(mySaveOnStop)));
  P.Addln('AutoStart = '+d2s(ord(myAutoStart)));
  P.Addln('HistoryLimits = '+d2s(myHistLimits[0])+', '+d2s(myHistLimits[1])+', '+d2s(myHistLimits[2]));
  P.Addln('HistoryCheckPeriod = '+f2s(HistTimer.IntervalMs[0]/60000)+' min');
  P.Addln('SpectrUpdatePeriod = '+f2s(SpecTimer.IntervalMs[0]/1000)+' sec');
  P.Addln('Desktop = '+FileRel(myDaqDesktop));
  P.Addln('ConsoleMode = $'+HexL(myConsoleMode));
  P.Addln('Navigator = '+myNavigator);
  P.Addln('StopConfirm = '+myStopConfirm);
  P.Addln('VoiceEcho = '+d2s(ord(myVoiceEcho)));
  P.Addln('ConsoleFile = '+FileRel(ConsoleFile));
  P.Addln(Format('DevicePolling = %d, %s',[Polling.Delay,GetPriorityName(Polling.Priority)]));
  P.Addln(Format('DispatcherPolling = %d, %s',[Dispatcher.Delay,GetPriorityName(Dispatcher.Priority)]));
  P.Addln(Format('CurveMonitorPeriod = %d ms',   [myCurWinPeriod]));
  P.Addln(Format('TableMonitorPeriod = %d ms',   [myTabWinPeriod]));
  P.Addln(Format('CircuitMonitorPeriod = %d ms', [myCirWinPeriod]));
  P.Addln(Format('SpectrMonitorPeriod = %d ms',  [mySpeWinPeriod]));
  P.Addln(Format('CurveTopListLimit = %d',    [CurveTopListLimit]));
  P.Addln(Format('TableTopListLimit = %d',    [TableTopListLimit]));
  P.Addln(Format('CircuitTopListLimit = %d',  [CircuitTopListLimit]));
  P.Addln(Format('SpectrTopListLimit = %d',   [SpectrTopListLimit]));
  P.Addln(Format('DaqTopListPeriod = %d sec', [DaqTopListPeriod]));
 end;
end;

 {
 Краткая сводка счетчиков объектов.
 }
function TDaqSystem.BriefDaqSysInfoCounters:String;
begin
 Result:='';
 if not Ok then Exit;
 if not Timer.isStart then Exit;
 Result:=Format(RusEng('Найдено тегов:%s, кривых:%s, устройств:%s, графиков:%s, мнемосхем:%s, сенсоров:%s.',
                       'Found tags:%s, curves:%s, devices:%s, charts:%s, circuits:%s, mnemonic sensors:%s.'),
   [GetDaqSysInfo('Tag.Count'),GetDaqSysInfo('Curve.Count'),GetDaqSysInfo('Device.Count'),
    GetDaqSysInfo('Curve_Window.Count'),GetDaqSysInfo('Circuit_Window.Count'),GetDaqSysInfo('Sensor.Count')]);
end;

 {
 Краткая сводка счетчиков ошибок.
 }
function TDaqSystem.BriefDaqSysBugsCounters:String;
var en,wn:Int64; i:Integer;
begin
 Result:='';
 if not Ok then Exit;
 if not Timer.isStart then Exit;
 en:=0; for i:=0 to 255 do inc(en,FullDaqDeviceList.ErrorsCount[i]); wn:=Daq.WarningList.Count;
 Result:=Format(RusEng('Найдено ошибок работы DAQ: %d, проблем конфигурации: %d.','Found DAQ runtime errors: %d, config problems: %d.'),[en,wn]);
end;

 {
 Вызвать окно справки
 }
procedure TDaqSystem.DaqInfo;
begin
end;

 {
 Сохранение данных DAQ с записью окна отката DAQ по таким правилам:
  1 - Сохранение идет в файл CRW в окно с именем 'DAQ DATA WINDOW'
  2 - Комментарий окна содержит строку времени в формате
      _DAQ_START_TIME_ = TIME ( YYYY.MM.DD-HH:MM:SS )
      затем имя конфигурационного файла в формате
      _DAQ_CONFIG_FILE_ = FILENAME
      затем содержимое конфигурационного файла
  3 - Коллекция кривых содержит все кривые DAQ. Кривые идентифицируются по имени.
 }
procedure TDaqSystem.SaveData(FileName:LongString);
var i:Integer; List:TText; Back,Temp:LongString;
begin
 if Ok then
 try
  FileName:=UnifyFileAlias(FileName);
  if Timer.IsStart and DaqDataWindow.Ok then begin
   OpenConsole(myConsoleMode and cmfSave=0);
   ConsoleEcho(StdDateTimePrompt+RusEng('Сохранение данных.','Save data.'));
   ConsoleEcho(RusEng('Файл ','File ')+FileName);
   DaqDataWindow.Curves.Count:=0;
   DaqDataWindow.Comment.Count:=0;
   DaqDataWindow.Caption:='DAQ DATA WINDOW';
   DaqDataWindow.Title:='';
   DaqDataWindow.Legend:='';
   for i:=0 to Curves.Count-1 do DaqDataWindow.Curves.Add(Curves[i]);
   DaqDataWindow.Comment.Addln(Format('_DAQ_START_TIME_ = %14.0f ( %s )',
                         [Timer.StartTime, StdDateTimeStr(Timer.StartTime)]));
   DaqDataWindow.Comment.Addln('_DAQ_CONFIG_FILE_ = '+ConfigFile);
   List:=ConstructFullConfig(ConfigFile);
   DaqDataWindow.Comment.Concat(List);
   Kill(List);
   Back:=ForceExtension(ForcePath(SessionManager.VarTmpDir,FileName),'.$$$');
   Temp:=ForceExtension(ForcePath(SessionManager.VarTmpDir,FileName),'.###');
   if SaveCurveWindowToCRW(Temp, DaqDataWindow) then begin
    FileErase(Back);
    FileRename(FileName,Back);
    FileRename(Temp,FileName);
   end else begin
    FileErase(Temp);
    ConsoleEcho(RusEng('Ошибка записи ','Error write ')+FileName);
   end;
   DaqDataWindow.Curves.Count:=0;
   DaqDataWindow.Comment.Count:=0;
  end;
 except
  ConsoleEcho(RusEng('Сбой при записи ','Fails on write ')+FileName);
 end;
end;

 {
 Выборочное сохранение данных
 }
procedure TDaqSystem.SaveDataCustom(FileName,Caption,Title,Legend,CurveList:LongString);
var i:Integer; scap,stit,sleg,scrv:LongString; curv:TCurve; List:TText;
begin
 with Daq do
 if Ok then
 try
  FileName:=UnifyFileAlias(FileName);
  if Timer.IsStart and DaqDataWindow.Ok then begin
   ConsoleEcho(StdDateTimePrompt+RusEng('Сохранение данных.','Save data.'));
   ConsoleEcho(RusEng('Файл ','File ')+FileName);
   DaqDataWindow.Curves.Count:=0;
   DaqDataWindow.Comment.Count:=0;
   scap:=DaqDataWindow.Caption;
   stit:=DaqDataWindow.Title;
   sleg:=DaqDataWindow.Legend;
   DaqDataWindow.Caption:=Caption;
   DaqDataWindow.Title:=Title;
   DaqDataWindow.Legend:=Legend;
   for i:=1 to WordCount(CurveList,ScanSpaces) do begin
    scrv:=ExtractWord(i,CurveList,ScanSpaces);
    if IsSameText(scrv,UnifySection(scrv)) then begin
     List:=ExtractEnumWordList(ConfigFile,scrv,'CurveList',efConfig);
     try
      while List.Count>0 do begin
       scrv:=List[0];
       if IsNonEmptyStr(scrv) then begin
        curv:=Curves.Find(scrv);
        if curv.Ok and (DaqDataWindow.Curves.IndexOf(curv)<0)
        then DaqDataWindow.Curves.Add(curv);
       end;
       List.DelLn(0);
      end;
     finally
      Kill(List);
     end;
    end else begin
     curv:=Curves.Find(scrv);
     if curv.Ok and (DaqDataWindow.Curves.IndexOf(curv)<0)
     then DaqDataWindow.Curves.Add(curv);
    end;
   end;
   if not SaveCurveWindowToCRW(FileName, DaqDataWindow)
   then ConsoleEcho(RusEng('Ошибка записи ','Error write ')+FileName);
   DaqDataWindow.Curves.Count:=0;
   DaqDataWindow.Comment.Count:=0;
   DaqDataWindow.Caption:=scap;
   DaqDataWindow.Title:=stit;
   DaqDataWindow.Legend:=sleg;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'SaveDataCustom');
   ConsoleEcho(E.Message);
  end;
 end;
end;

 {
 Восстановление данных,сохраненных процедурой DaqSaveData:
  1 - Читаем в файле DAQ окно с именем 'DAQ DATA WINDOW'
  2 - Читаем в комментарии окна базовое время в формате _DAQ_START_TIME_%f
  3 - Копируем данные из коллекции кривых окна в список Curves.
 Окно сохранения DAQ читается и обычными средствами CRW, поэтому при
 проблемах с восстановлением можно прочитать кривые и так. Поскольку
 комментарий содержит конфигурацию, можно восстановить и ее.
 }
procedure ItemFindName(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
begin
 if Chunk.dwSign=sig_name then begin
  PureString(Custom^):=Filer.ReadString(Chunk.dwSize,rsmf_FixUtf8);
  Terminate:=true;
 end;
end;

procedure ItemFindWindow(Filer:TRiffFiler; const Chunk:TRiffChunk; var Terminate:Boolean; Custom:Pointer);
var Name:PureString;
begin
 if (Chunk.dwSign=sig_ITEM) and (Chunk.dwFormID=sig_CRVW) then begin
  Name:='';
  Filer.ForEach(ItemFindName,Chunk,@Name);
  if Name='DAQ DATA WINDOW' then TRiffChunk(Custom^):=Chunk;
 end;
end;

function FindDaqDataWindowChunk(const FileName:LongString):TRiffChunk;
var Filer : TRiffFiler;
begin
 Result:=RiffChunk(0,0,0,0);
 Filer:=NewRiffFiler(FileName,fmOpenRead,sig_CRWF);
 if Filer.Ok then
 try
  Filer.ForEach(ItemFindWindow,Filer.RootForm,@Result);
 except
  on EReadError  do begin
   Filer.Modified:=false;
   Error(RusEng('Ошибка чтения файла ','Error reading file ')+FileName);
  end;
  on EWriteError do begin
   Filer.Modified:=false;
   Error(RusEng('Ошибка записи файла ','Error writing file ')+FileName);
  end;
  else RAISE;
 end else Error(RusEng('Ошибка открытия файла ','Error open file ')+FileName);
 Kill(Filer);
end;

function LoadDaqDataWindow(const FileName:LongString):TFormCurveWindow;
var Chunk:TRiffChunk;
begin
 Result:=nil;
 if FileExists(FileName) then begin
  Chunk:=FindDaqDataWindowChunk(FileName);
  if (Chunk.dwSign=sig_ITEM) and (Chunk.dwFormID=sig_CRVW)
  then Result:=LoadCurveWindowFromCRW(FileName,Chunk);
 end;
end;

procedure TDaqSystem.LoadData(FileName:LongString);
var
 i              : Integer;
 Win            : TFormCurveWindow;
 Buff           : TParsingBuffer;
 SaveConfigFile : LongString;
 procedure RestoreCurve(Source:TCurve);
 var
  Dest : TCurve;
 begin
  if Source.Ok and IsNonEmptyStr(Source.Name) then begin
   Dest:=Curves.Find(Source.Name);
   if Dest.Ok then begin
    Dest.AssignData(Source.PX[0],Source.PY[0],Source.Count);
    Dest.AssignComment(Source.Comment);
   end;
  end;
 end;
 function ResetTime(Win:TFormCurveWindow):Boolean;
 var i:Integer; t:Double;
 begin
  Result:=False; t:=0;
  if Win.Ok then
  for i:=0 to Win.Comment.Count-1 do
  if IsNonEmptyStr(Win.Comment[i]) then
  if pos('_DAQ',Win.Comment[i])=1 then
  if ScanVarDouble(svConfig,StrCopyBuff(Buff,Win.Comment[i]),'_DAQ_START_TIME_%f',t)<>nil then begin
   SetEnv('CRW_DAQ_CONFIG_BASE_TIME',Format('%g',[t]));
   Timer.StartAt(t);
   Result:=true;
   break;
  end;
 end;
begin
 if Ok then
 if Timer.IsStart then begin
  {
  если загружен DAQ, читаем данные из окна в имеющиеся аппаратурные буферы
  }
  FileName:=UnifyFileAlias(FileName);
  OpenConsole(myConsoleMode and cmfLoad=0);
  ConsoleEcho(StdDateTimePrompt+RusEng('Загрузка данных.','Load data.'));
  ConsoleEcho(RusEng('Файл ','File ')+FileName);
  Win:=LoadDaqDataWindow(FileName);
  if Win.Ok then begin
   if not ResetTime(Win) then ConsoleEcho(RusEng('Не могу восстановить время пуска!','Could not read start time!'));
   for i:=0 to Win.Curves.Count-1 do RestoreCurve(Win.Curves[i]);
   UpdateWindowsTimeBaseTimeUnits;
  end else ConsoleEcho(RusEng('Не смог прочитать ','Could not read ')+FileName);
  Kill(Win);
 end else begin
  {
  если не загружен DAQ, создаем временный файл конфигурации,
  подменяем ConfigFile, по данным из временного файла и восстанавливаем
  список окон, удаляем временный файл конфигурации и возвращаем ConfigFile
  }
  Win:=LoadDaqDataWindow(FileName);
  if Win.Ok then begin
   if Win.Comment.Count>0 then begin
    SaveConfigFile:=ConfigFile;
    myConfigFile:=ForcePath(ConfigPath,'$$$$$$$$.cfg');
    Win.Comment.WriteFile(ConfigFile);
    ReadWindowsSection(wtCurWin+wtSpeWin,Win);
    FileErase(ConfigFile);
    myConfigFile:=SaveConfigFile;
   end;
  end else ConsoleEcho(RusEng('Не смог прочитать ','Could not read ')+FileName);
 end;
end;

 {
 Автосохранение
 }
procedure TDaqSystem.SaveBackup;
begin
 if Ok then begin
  if IsNonEmptyStr(BackupFile) then SaveData(BackupFile);
  SaveLogBook;
 end;
end;

 {
 Восстановление после автосохранения
 }
procedure TDaqSystem.LoadBackup;
begin
 if Ok then begin
  if IsNonEmptyStr(BackupFile) then LoadData(BackupFile);
  WinDraw(wtAll);
 end;
end;

 {
 После запроса - восстановление данных из файла автосохранения
 }
procedure TDaqSystem.Remedy(const Params:LongString='');
begin
 if Ok then
 if CheckSessionStarted and not CheckAcquisitionStarted then begin
  if YesNo(RusEng('Восстановить данные из файла автосохранения?',
                  'Restore data from backup file?'),Params)=mrYes
  then LoadBackup;
 end;
end;

 {
 Открыть (прочитать из конфигурации) окна данного типа.
 }
procedure TDaqSystem.WinOpen(wt:word);
begin
 if Ok then ReadWindowsSection(wt,nil);
end;

 {
 Закрыть (уничтожить) окна заданного типа.
 }
procedure TDaqSystem.WinClose(wt:word);
begin
 if Ok then begin
  if HasFlags(wt,wtCurWin) then CurWinList.Count:=0;
  if HasFlags(wt,wtTabWin) then TabWinList.Count:=0;
  if HasFlags(wt,wtCirWin) then CirWinList.Count:=0;
  if HasFlags(wt,wtSpeWin) then SpeWinList.Count:=0;
 end;
end;

 {
 Спрятать (минимизировать) окна заданного типа.
 }
procedure TDaqSystem.WinHide(wt:word);
var i:Integer;
begin
 if Ok then begin
  if HasFlags(wt,wtCurWin) then
  for i:=0 to CurWinList.Count-1 do
  if CurWinList[i].Ok then CurWinList[i].WindowState:=wsMinimized;
  if HasFlags(wt,wtTabWin) then
  for i:=0 to TabWinList.Count-1 do
  if TabWinList[i].Ok then TabWinList[i].WindowState:=wsMinimized;
  if HasFlags(wt,wtCirWin) then
  for i:=0 to CirWinList.Count-1 do
  if CirWinList[i].Ok then CirWinList[i].WindowState:=wsMinimized;
  if HasFlags(wt,wtSpeWin) then
  for i:=0 to SpeWinList.Count-1 do
  if SpeWinList[i].Ok then SpeWinList[i].WindowState:=wsMinimized;
 end;
end;

 {
 Показать (развернуть) окна заданного типа.
 }
procedure TDaqSystem.WinShow(wt:word);
var i:Integer;
begin
 if Ok then begin
  if HasFlags(wt,wtCurWin) then
  for i:=0 to CurWinList.Count-1 do
  if CurWinList[i].Ok then CurWinList[i].WindowState:=wsNormal;
  if HasFlags(wt,wtTabWin) then
  for i:=0 to TabWinList.Count-1 do
  if TabWinList[i].Ok then TabWinList[i].WindowState:=wsNormal;
  if HasFlags(wt,wtCirWin) then
  for i:=0 to CirWinList.Count-1 do
  if CirWinList[i].Ok then CirWinList[i].WindowState:=wsNormal;
  if HasFlags(wt,wtSpeWin) then
  for i:=0 to SpeWinList.Count-1 do
  if SpeWinList[i].Ok then SpeWinList[i].WindowState:=wsNormal;
 end;
end;

 {
 Перерисовать окна заданного типа.
 }
procedure TDaqSystem.WinDraw(wt:word);
var i:Integer;
begin
 if Ok then begin
  if HasFlags(wt,wtCurWin) then
  for i:=0 to CurWinList.Count-1 do
  if CurWinList[i].Ok then CurWinList[i].Repaint;
  if HasFlags(wt,wtTabWin) then
  for i:=0 to TabWinList.Count-1 do
  if TabWinList[i].Ok then TabWinList[i].Repaint;
  if HasFlags(wt,wtCirWin) then
  for i:=0 to CirWinList.Count-1 do
  if CirWinList[i].Ok then CirWinList[i].Repaint;
  if HasFlags(wt,wtSpeWin) then
  for i:=0 to SpeWinList.Count-1 do
  if SpeWinList[i].Ok then SpeWinList[i].Repaint;
 end;
end;

 {
 Операции с окнами через диалог.
 Мнемосхемы и спектры нельзя закрывать после загрузки DAQ, поэтому вместо
 открытия/закрытия они сворачиваются/разворачиваются.
 }
procedure TDaqSystem.WindowsOperations(const Params:LongString='');
var
 What : Integer;
 Who  : Integer;
begin
 if Ok then
 if CheckSessionStarted then begin
  What:=0;
  Who:=0;
  if (FormDaqWinOperDialogExecute(What,Who,Params)=mrOk) then
  case What of
   0: WinShow(Who);
   1: begin
       WinOpen(Who and not (wtCirWin+wtSpeWin));
       WinShow(Who and     (wtCirWin+wtSpeWin));
      end;
   2: LoadWinLocations(Who);
   3: WinHide(Who);
   4: begin
       WinClose(Who and not (wtCirWin+wtSpeWin));
       WinHide(Who  and     (wtCirWin+wtSpeWin));
      end;
   5: SaveWinLocations(Who);
  end;
 end;
end;

 {
 Открыть динамическое окно
 }
procedure TDaqSystem.OpenDynamicWindow(const Params:LongString='');
var i,tag,dcwn,dtwn,aType:Integer; List:TText; Curve:TCurve;
var crvList,tagList,aCaption,aTitle,aLegend,aOpt:LongString;
var CurWin:TFormCurveWindow; TabWin:TFormTabWindow;
begin
 if Ok then
 if CheckSessionStarted then
 try
  List:=NewText;
  try
   dcwn:=DynamicCurWinNumber;
   dtwn:=DynamicTabWinNumber;
   aCaption:=RusEng('Новое окно DAQ','New DAQ Window');
   aTitle:='^C'+RusEng('Заголовок','Title')+'^N^L   [Y]';
   aLegend:='[?]';
   if Timer.LocalTimeUnits=1.0 then aLegend:=RusEng('[мСек]','[mSec]');
   if Timer.LocalTimeUnits=1000.0 then aLegend:=RusEng('[Сек]','[Sec]');
   if Timer.LocalTimeUnits=1000.0*60 then aLegend:=RusEng('[Мин]','[Min]');
   if Timer.LocalTimeUnits=1000.0*60*60 then aLegend:=RusEng('[Час]','[Hour]');
   if Timer.LocalTimeUnits=1000.0*60*60*24 then aLegend:=RusEng('[Сут]','[Day]');
   aLegend:='^R'+RusEng('Время ','Time ')+aLegend+'   ^N^C'+RusEng('Легенда','Legend');
   aType:=0;
   aOpt:='preset stdInformation delay 15000';
   if FormDaqOpenDynamicWindowDialogExecute(aCaption, aTitle, aLegend, aType, Params)=mrOk then begin
    DaqAlignStr(aTitle);
    DaqAlignStr(aLegend);
    case aType of
     0 :
      begin
       crvList:=SelectCurveList(Params);
       if (Length(crvList)>0) then begin
        CurWin:=NewCurveWindow(aCaption, aTitle, aLegend);
        if CurWin.Ok then begin
         List.Text:=crvList;
         for i:=0 to List.Count-1 do begin
          Curve:=Curves.Find(List[i]);
          if Curve.Ok then CurWin.AddCurve(Curve);
         end;
         CurWin.TimeBase:=Timer.StartTime;
         CurWin.TimeUnits:=Timer.LocalTimeUnits;
         CurWin.CloseAction:=caMinimize;
         CurWin.IsDataProtected:=true;
         CurWin.StartMonitoring;
         CurWinList.Add(CurWin);
         CurWin.DaqRef:=Daq.Ref;
         CurWin.DefCurveNum:=-1;
         CurWin.AutoRange;
        end;
       end else begin
        TipWarning(RusEng('Не выбраны данные для отображения.','No data selected to display.'),aOpt);
        DynamicCurWinNumber:=dcwn;
       end;
      end;
     1 :
      begin
       crvList:=SelectCurveList(Params);
       tagList:=SelectTagList(Params);
       if (Length(crvList)+Length(tagList)>0) then begin
        TabWin:=NewTabWindow(aCaption);
        if TabWin.Ok then begin
         List.Text:=crvList;
         for i:=0 to List.Count-1 do begin
          Curve:=Curves.Find(List[i]);
          if Curve.Ok then TabWin.AddItem(Curve,0,'');
         end;
         List.Text:=tagList;
         for i:=0 to List.Count-1 do begin
          tag:=FindTag(List[i]);
          if (TypeTag(tag)>0) then TabWin.AddItem(nil,tag,'');
         end;
         TabWin.UpdateGrid;
         TabWin.UpdateSize;
         TabWin.ShouldResize;
         TabWin.CloseAction:=caMinimize;
         TabWin.StartMonitoring;
         TabWinList.Add(TabWin);
         TabWin.DaqRef:=Daq.Ref;
        end;
       end else begin
        TipWarning(RusEng('Не выбраны данные для отображения.','No data selected to display.'),aOpt);
        DynamicTabWinNumber:=dtwn;
       end;
      end;
    end;
   end;
  finally
   Kill(List);
   crvList:='';
   tagList:='';
  end;
 except
  on E:Exception do BugReport(E,Self,'OpenDynamicWindow');
 end;
end;

 {
 Перерисовка спектрометрических окон. В данном случае рисует все окна с кривыми,
 в которых есть спектры - то есть статические кривые.
 }
procedure TDaqSystem.UpdateSpectralWindows;
var
 i : Integer;
 j : Integer;
begin
 if Ok then begin
  for i:=0 to CurWinList.Count-1 do
  if CurWinList[i].Ok then
  for j:=0 to CurWinList[i].Curves.Count-1 do
  if CurWinList[i].Curves[j].IsStatic then begin
   CurWinList[i].DrawView;
   break;
  end;
  for i:=0 to SpeWinList.Count-1 do
  if SpeWinList[i].Ok then SpeWinList[i].DrawView;
 end;
end;

 {
 Открыть окно, используемое для внутренних целей при сохранении данных.
 }
procedure TDaqSystem.OpenDaqDataWindow;
begin
 if Ok then begin
  if not Assigned(myDaqDataWindow) then begin
   myDaqDataWindow:=NewCurveWindow('DAQ DATA WINDOW','','',wsMinimized);
   myDaqDataWindow.Master:=@myDaqDataWindow;
   myDaqDataWindow.TimeBase:=Timer.StartTime;
   myDaqDataWindow.TimeUnits:=Timer.LocalTimeUnits;
   myDaqDataWindow.CloseAction:=caMinimize;
   myDaqDataWindow.IsDataProtected:=true;
   myDaqDataWindow.BorderIcons:=[biSystemMenu];
   myDaqDataWindow.AddonSdiFlags(sf_SdiNoRestore);
  end;
 end;
end;

 {
 Открыть окно для редактирования журнала.
 }
procedure TDaqSystem.OpenLogBook(ws:TWindowState=wsNormal);
begin
 if Ok then
 try
  if CheckSessionStarted then
  if IsNonEmptyStr(LogFile) then
  if myLogBook.Ok then begin
   ShowLogBook(ws);
  end else begin
   TouchLogBook;
   myLogBook:=NewTextEditor(LogFile);
   if myLogBook.Ok then begin
    myLogBook.Master:=@myLogBook;
    myLogBook.CloseAction:=caFree;
    myLogBook.PerformGoToEnd;
    myLogBook.AddonSdiFlags(sf_SdiNoRestore);
    ShowLogBook(ws);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'OpenLogBook');
 end;
end;

procedure TDaqSystem.ShowLogBook(ws:TWindowState);
begin
 if Ok then
 try
  if CheckSessionStarted then
  if IsNonEmptyStr(LogFile) then
  if Assigned(myLogBook) then begin
   myLogBook.WindowState:=ws;
   if (ws=wsNormal) then myLogBook.BringToFront;
  end;
 except
  on E:Exception do BugReport(E,Self,'ShowLogBook');
 end;
end;

procedure TDaqSystem.SaveLogBook;
begin
 if Ok then
 try
  if myLogBook.Ok then
  if myLogBook.PerformModified then begin
   if myLogBook.IsNoNamePath(myLogBook.PathName) then Exit;
   myLogBook.FileSave;
  end;
 except
  on E:Exception do BugReport(E,Self,'SaveLogBook');
 end;
end;

procedure TDaqSystem.TouchLogBook;
var p:TText;
begin
 if Ok then
 try
  if CheckSessionStarted then
  if IsNonEmptyStr(LogFile) then
  if not FileExists(LogFile) then begin
   p:=NewText;
   try
    p.Addln(StdDateTimePrompt+RusEng('Журнал CRW-DAQ создан.','CRW-DAQ LogBook created.'));
    if p.WriteFile(LogFile)<>0 then ConsoleEcho(RusEng('Ошибка создания ','Error create ')+LogFile);
   finally
    Kill(p);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'TouchLogBook');
 end;
end;


 {
 Read [DaqSys] HelpFileFormats or return fallback Help File Formats list.
 }
function TDaqSystem.HelpFileFormats:LongString;
const FallbackHelpFileFormats='.htm|.html|.doc|.odt|.odp|.rtf|.pdf|.ps|.chm|.md|.mkd|.markdown|.ppt|.pps';
begin
 Result:='';
 if Ok then begin
  if (myHelpFileFormats='') then begin
   if ReadIniFileAlpha(SysIniFile,SectDaqSys,'HelpFileFormats%a',myHelpFileFormats)
   then myHelpFileFormats:=Trim(myHelpFileFormats) else myHelpFileFormats:='';
   if (myHelpFileFormats='') then myHelpFileFormats:=FallbackHelpFileFormats;
   if IsUnix then myHelpFileFormats:=LowerCase(myHelpFileFormats);
  end;
  Result:=myHelpFileFormats;
 end;
end;

 {
 Открыть окно справки DAQ.
 }
procedure TDaqSystem.OpenHelp;
begin
 if Ok and CheckSessionStarted then
 try
  if WordIndex(UnifyAlias(ExtractFileExt(HelpFile)),HelpFileFormats,['|'])>0
  then SmartExecute(HelpFile,SW_SHOWNORMAL,'open')
  else OpenFormDaqHelpViewer(HelpFile);
 except
  on E:Exception do BugReport(E,Self,'OpenHelp');
 end;
end;

 {
 Открыть консольное окно для вывода системных сообщений.
 }
procedure TDaqSystem.OpenConsole(Hidden:Boolean=false);
begin
 if SystemConsole.Form.Ok then begin
  if not Hidden then SystemConsole.Activate;
 end else Init_System_Console;
end;

 {
 Записать строку в FIFO консоли. Потокобезопасно.
 }
procedure TDaqSystem.ConsoleEcho(const s:LongString);
begin
 if Ok then myConsoleFifo.PutText(s+EOL) else Exit;
 if SysLog.Notable(SeverityOfDaqPrint)
 then SysLog.Note(0,SeverityOfDaqPrint,SysLogSign(1),s);
end;

 {
 Вывод сообщения и открытие консоли
 }
procedure TDaqSystem.Report(const msg:LongString);
begin
 ConsoleEcho(msg);
 if GetCurrentThreadID=MainThreadID then OpenConsole;
end;

 {
 Вытолкнуть FIFO консоли в окно и в файл. Выполняется в основном потоке в Idle.
 }
procedure TDaqSystem.ConsoleFlush;
var
 f : Text;
 s : LongString;
begin
 if Ok then
 if myConsoleFifo.Count>0 then begin
  s:=myConsoleFifo.GetText;
  Echo(s,'');
  if IsNonEmptyStr(ConsoleFile) then begin
   if FileExists(ConsoleFile) then begin
    if IOResult<>0 then;
    System.Assign(f,ConsoleFile);
    System.Append(f);
    System.Write(f,s);
    System.Close(f);
    if IOResult<>0 then;
   end else begin
    if IOResult<>0 then;
    System.Assign(f,ConsoleFile);
    System.Rewrite(f);
    System.Write(f,s);
    System.Close(f);
    if IOResult<>0 then;
   end;
  end;
 end;
end;

procedure TDaqSystem.UpdateSysLogSign;
var fn:LongString;
begin
 if Assigned(Self) then begin
  mySysLogSignBase:=sdr_SysDAQ;
  mySysLogSignConf:=mySysLogSignBase;
  fn:=Trim(ExtractFileNameExt(myConfigFile));
  if IsNonEmptyStr(ExtractBaseName(fn)) and not IsWildCard(fn)
  then mySysLogSignConf:=mySysLogSignConf+SysLog.SenderSep+fn;
 end;
end;

function TDaqSystem.SysLogSign(Mode:Integer; Sub:LongString=''):LongString;
begin
 Result:='';
 if Assigned(Self) then begin
  if HasFlags(Mode,1)
  then Result:=mySysLogSignConf
  else Result:=mySysLogSignBase;
  if (Sub<>'') then Result:=Result+SysLog.SenderSep+Sub;
 end;
end;

 {
 Диалог выбора кривой из базы данных DAQ.
 }
function TDaqSystem.SelectCurve(const aCaption,aTitle:LongString; const Params:LongString=''):TCurve;
begin
 if Ok and (Curves.Count>0)
 then Result:=Curves[ListBoxMenu(aCaption,aTitle,CurvesNameList,0,Params)]
 else Result:=nil;
end;

 {
 Диалог выбора множества кривых из базы данных DAQ, возвращает список имен.
 }
function TDaqSystem.SelectCurveList(const Params:LongString=''):LongString;
begin
 if Ok and (Curves.Count>0)
 then Result:=ListBoxMultiSelection(RusEng('Выбрать список кривых',
                                           'Select curves from list'),
                                    RusEng('Отметьте нужные кривые',
                                           'Select curves you need'),
                                           CurvesNameList,
                                           false,false,0,0,Params)
 else Result:='';
end;

 {
 Диалог выбора множества тегов из базы данных DAQ, возвращает список имен.
 }
function TDaqSystem.SelectTagList(const Params:LongString=''):LongString;
var tagList:LongString;
begin
 tagList:=TagsNameList;
 if (Length(tagList)>0)
 then Result:=ListBoxMultiSelection(RusEng('Выбрать список тегов',
                                           'Select tags from list'),
                                    RusEng('Отметьте нужные теги',
                                           'Select tags you need'),
                                           tagList,
                                           false,false,0,0,Params)
 else Result:='';
end;

 {
 Диалог проверки того, что DAQ загружен с выдачей ругательства, если это не так.
 }
function TDaqSystem.CheckSessionStarted:Boolean;
begin
 Result:=NoProblem(Timer.IsStart,RusEng('DAQ: система не загружена!','DAQ: does not loaded.'));
end;

 {
 Диалог проверки того, что DAQ запущен с выдачей ругательства, если это так.
 }
function TDaqSystem.CheckAcquisitionStarted:Boolean;
var Params:LongString;
begin
 Params:='';
 if FormCrwDaq.Ok then Params:=ControlPosParams(FormCrwDaq.ToolButtonDaq,'LB');
 Result:=Trouble(AcqTimer.IsStart,RusEng('DAQ: идет набор данных!','DAQ: measure in progress!'),Params);
end;

 {
 Диалог разрешения завершения сеанса DAQ.
 }
function TDaqSystem.ConfirmExit(const Params:LongString=''):Boolean;
begin
 if Ok
 then Result:=(SystemCalculator.Eval('_Daq_Force_Exit_')=1) or
              (YesNo(RusEng('DAQ: завершить сеанс?','DAQ: exit session?'),
              Params)=mrYes)
 else Result:=true;
end;

 {
 Диалог разрешения остановки сеанса DAQ.
 }
function TDaqSystem.ConfirmStop(const Params:LongString=''):Boolean;
begin
 if Ok
 then Result:=(SystemCalculator.Eval('_Daq_Force_Exit_')=1) or
              (SystemCalculator.Eval('_Daq_Force_Stop_')=1) or
              IsEmptyStr(myStopConfirm) or
              (YesNo(RusEng('Вы уверены, что можно остановить DAQ?',
              'You really want stop DAQ?')+EOL+EOL+myStopConfirm,
              Params)=mrYes)
 else Result:=true;
end;


 {
 Процедура проверяет длину истории динамических кривых и выдает диагностику
 если история слишком велика.
 }
procedure TDaqSystem.CheckHistoryLength;
var
 i     : Integer;
 MaxN  : Integer;
 Curve : TCurve;
begin
 if Ok then
 if (myHistLimits[1]>0) and (myHistLimits[2]>0) and HistTimer.Event then begin
  MaxN:=0;
  myHistList.Count:=0;
  for i:=0 to Curves.Count-1 do begin
   Curve:=Curves[i];
   if Curve.Ok and Curve.IsDynamic then begin
    MaxN:=max(Curve.Count,MaxN);
    if Curve.Count>myHistLimits[1] then myHistList.Addln(Curve.Name);
   end;
  end;
  if MaxN>myHistLimits[1] then begin
   if MaxN>myHistLimits[2] then begin
    LimitHistoryLength(myHistLimits[0]);
    KillFormDaqHistoryWarning;
   end else begin
    OpenFormDaqHistoryWarning(myHistLimits[0],
                              myHistLimits[1],
                              myHistLimits[2],
                              myHistList.Text);
    Voice('очистпам');
   end;
  end else KillFormDaqHistoryWarning;
  myHistList.Count:=0;
 end;
end;

 {
 Ограничить длину истории
 }
procedure TDaqSystem.LimitHistoryLength(MaxHistLen:LongInt);
var
 i     : Integer;
 Curve : TCurve;
begin
 if Ok then begin
  for i:=0 to Curves.Count-1 do begin
   Curve:=Curves[i];
   if Curve.Ok and Curve.IsDynamic then Curve.ClearHistory(MaxHistLen);
  end;
  WinDraw(wtCurWin+wtTabWin);
 end;
end;

 {
 Выдать звуковое сообщение, если оно разрешено.
 }
procedure TDaqSystem.Voice(const WhatToSay:LongString);
begin
 if Ok and SoundOn then begin
  _crw_snd.Voice(WhatToSay);
  if myVoiceEcho
  then ConsoleEcho(StdDateTimePrompt+RusEng('Голос: ','Voice: ')+WhatToSay);
 end;
end;

 {
 Зарегистрировать класс ошибок DAQ. Возвращает код ошибки.
 Этот код используется для идентификации ошибок DAQ.
 }
function TDaqSystem.RegisterErrMsg(const Msg:LongString):Integer;
var i:Integer;
begin
 Result:=ecUnknown;
 if Ok and IsNonEmptyStr(Msg) then begin
  for i:=Low(myErrMsgTable) to High(myErrMsgTable) do
  if SameText(UnifyAlias(myErrMsgTable[i]),UnifyAlias(Msg)) then begin
   Result:=i;
   break;
  end;
  if Result=ecUnknown then
  for i:=Low(myErrMsgTable) to High(myErrMsgTable) do
  if IsEmptyStr(myErrMsgTable[i]) then begin
   myErrMsgTable[i]:=Trim(Msg);
   Result:=i;
   break;
  end;
 end;
end;

 {
 Занести сообщение в список предупреждений DAQ.
 }
procedure TDaqSystem.AddWarning(const Msg:LongString);
begin
 if Ok then begin
  WarningList.Addln(Msg);
  DebugOut(stdfDebug,Msg);
 end;
end;

 {
 Послать всплывающее уведомление.
 }
procedure TDaqSystem.TipWarning(const Msg:LongString; const Opt:LongString='');
var Txt,Line:LongString;
begin
 if Msg<>'' then
 if Ok then begin
  Txt:=GetEnv('CRW_DAQ_SYS_SESSION_HEAD');
  Txt:=IfThen(IsEmptyStr(Txt),Msg,Txt+': '+Msg);
  Line:=Trim(Format('@silent @tooltip text %s %s',[AnsiQuotedStr(Txt,QuoteMark),Opt]));
  SendToMainConsole(Line+EOL);
 end;
end;

 {
 Просмотр кривых DAQ в табличном виде
 }
procedure TDaqSystem.ViewData(const Params:LongString='');
var p:TText; i:Integer; Curve:TCurve;
const Limit=1024*100;
begin
 if Ok then
 if CheckSessionStarted then begin
  Curve:=SelectCurve(RusEng('Посмотреть кривую','View curve'),RusEng('Имя кривой','Curve name'),Params);
  if Curve.Ok then begin
   p:=NewText;
   try
    Curve.Lock;
    try
     if Curve.Count = 0 then p.AddLn(RusEng('Нет данных','No data')) else
     if Curve.Count >= Limit then p.Addln(RusEng('Слишком большая кривая!','Too long curve!')) else
     for i:=0 to Curve.Count-1 do  with Curve[i] do p.Addln(Format('%-20g  %-20g', [X, Y]));
    finally
     Curve.Unlock;
    end;
    ListBoxMenu(RusEng('Таблица ','Table ')+Curve.Name,Format('%-20s  %-20s',['X','Y']),p.Text,0,Params);
   finally
    Kill(p);
   end;
  end;
 end;
end;

 {
 Вызвать окно навигатора
 }
type
 TNavRec = packed record
   Win : TForm;
   Cap : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TNavRec(Custom^) do
 if (Form is TForm) then
 if SameText(UnifyAlias(Form.Caption),UnifyAlias(Cap)) then begin
  Win:=Form;
  Terminate:=true;
 end;
end;

procedure TDaqSystem.Navigator(const Params:LongString='');
var NavRec:TNavRec; Items:LongString; i:Integer;
begin
 if Ok then
 try
  NavRec:=Default(TNavRec);
  try
   NavRec.Win:=nil;
   NavRec.Cap:='';
   if (myNavigator<>'') then NavRec.Cap:=StringBuffer(sGetTag(FindTag(myNavigator)));
   if (NavRec.Cap<>'') then SdiMan.ForEachChild(FindForm,@NavRec);
   if not Assigned(NavRec.Win) then begin
    if CirWinList.Count=0
    then NavRec.Win:=SdiMan.SelectChildDialog(RusEng('Открыть окно','Open window'),
         RusEng('Какое окно хотите открыть?','Which window you want to open?'),
         sm_ZOrder,Params)
    else
    if CirWinList.Count=1 then NavRec.Win:=CirWinList[0] else begin
     Items:='';
     for i:=0 to CirWinList.Count-1 do
     if CirWinList[i].Ok then begin
      if Length(Items)>0 then Items:=Items+EOL;
      Items:=Items+CirWinList[i].Caption;
     end;
     i:=ListBoxMenu(RusEng('Открыть мнемосхему','Open mnemonic window'),
                    RusEng('Какое окно открыть?','Which window you want?'),Items);
     if i>=0 then NavRec.Win:=CirWinList[i];
     Items:='';
    end;
   end;
   if Assigned(NavRec.Win) then begin
    NavRec.Win.WindowState:=wsNormal;
    NavRec.Win.BringToFront;
   end;
  finally
   NavRec.Cap:='';
  end;
 except
  on E:Exception do BugReport(E,Self,'Navigator');
 end;
end;

 {
 Сосчитать интеграл. Пределы заданы РОИ окна.
 }
procedure TDaqSystem.Integral(const Params:LongString='');
var W:TFormCurveWindow; C:TCurve; L:TRect2D; sum1,sum2,sum3:Double;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  C:=W.DefCurve;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(C.Ok,RusEng('Не выбрана кривая в окне!','No curve selected!'),Params) and
     NoProblem(C.Count>0,RusEng('Нет точек на кривой!','Curve has no points!'),Params) and
     NoProblem(C.Flags and (cfNotSortedX+cfDuplicatesX)=0,RusEng('Кривая не отсортирована!','Curve is not sorted!'),Params) and
     NoProblem(not IsNanOrInf(L),RusEng('Не выбраны пределы (РОИ).','No ROI selected!'),Params)
  then begin
   L:=RectValidate(L);
   sum1:=DaqIntegral(C,L.a.x,L.b.x);
   sum2:=Trapezium(L.a.x, C.Interpolate(L.a.x), L.b.x, C.Interpolate(L.b.x));
   sum3:=sum1-sum2;
   SystemCalculator.Eval(Format('int_a=%g',[L.a.x]));
   SystemCalculator.Eval(Format('int_b=%g',[L.b.x]));
   SystemCalculator.Eval(Format('int_full=%g',[sum1]));
   SystemCalculator.Eval(Format('int_line=%g',[sum2]));
   SystemCalculator.Eval(Format('int_mean=%g',[sum3]));
   OpenConsole(myConsoleMode and cmfInteg=0);
   ConsoleEcho(StdDateTimePrompt+RusEng('Интеграл ','Integral ')+C.Name);
   ConsoleEcho('  Нижний предел     int_a    = '+f2s(L.a.x));
   ConsoleEcho('  Верхний предел    int_b    = '+f2s(L.b.x));
   ConsoleEcho('  Полный интеграл   int_full = '+f2s(sum1));
   ConsoleEcho('  Линейный фон      int_line = '+f2s(sum2));
   ConsoleEcho('  Интеграл без фона int_mean = '+f2s(sum3));
  end;
 end;
end;

 {
 Медианный фильтр.  Пределы заданы РОИ окна.
 }
procedure TDaqSystem.MedianFiltr(const Params:LongString='');
var i,Width:Integer; L:TRect2D; C,Temp:TCurve; W:TFormCurveWindow;
var AbsEps,RelEps:Double;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  C:=W.DefCurve;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(C.Ok,RusEng('Не выбрана кривая в окне!','No curve selected!'),Params) and
     NoProblem(C.Count>0,RusEng('Нет точек на кривой!','Curve has no points!'),Params) and
     NoProblem(C.IsDynamic,RusEng('Операция недоступна для спектра!','Operation not available for spectral data!'),Params) and
     NoProblem(C.Flags and (cfNotSortedX+cfDuplicatesX)=0,RusEng('Кривая не отсортирована!','Curve is not sorted!'),Params) and
     NoProblem(not IsNanOrInf(L),RusEng('Не выбраны пределы (РОИ).','No ROI selected!'),Params)
  then begin
   L:=RectValidate(L); AbsEps:=0; RelEps:=0; Width:=0;
   if FormDaqMedianFilterDialogExecute(Width,AbsEps,RelEps,Params)=mrOk then
   try
    C.Lock;
    Temp:=NewCurveMedianFiltr(C,Width,AbsEps,RelEps);
    if Temp.Ok then
    for i:=C.GetIndexAt(L.A.X) to C.GetIndexAt(L.B.X) do C[i]:=Temp[i];
   finally
    Kill(Temp);
    C.Unlock;
   end;
   W.Repaint;
  end;
 end;
end;

 {
 Удалить точки из выбранной области
 }
procedure TDaqSystem.CutRegion(const Params:LongString='');
var i:Integer; L:TRect2D; C,Temp:TCurve; W:TFormCurveWindow;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  C:=W.DefCurve;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(C.Ok,RusEng('Не выбрана кривая в окне!','No curve selected!'),Params) and
     NoProblem(C.Count>0,RusEng('Нет точек на кривой!','Curve has no points!'),Params) and
     NoProblem(C.IsDynamic,RusEng('Операция недоступна для спектра!','Operation not available for spectral data!'),Params) and
     NoProblem(not IsNanOrInf(L),RusEng('Не выбраны пределы (РОИ).','No ROI selected!'),Params)
  then begin
   L:=RectValidate(L);
   try
    Temp:=NewCurve;
    C.Lock;
    for i:=0 to C.Count-1 do
    if not RectContainsPoint(L,C[i]) then with C[i] do Temp.AddPoint(x,y);
    C.AssignData(Temp.PX[0], Temp.PY[0], Temp.Count);
   finally
    C.Unlock;
    Kill(Temp);
   end;
   W.Repaint;
  end;
 end;
end;

 {
 Поставить пользовательский маркер
 }
procedure TDaqSystem.SetUserMarker(const Params:LongString='');
var L:TRect2D; W:TFormCurveWindow;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(not IsNanOrInf(L.A),RusEng('Не установлен левый маркер РОИ.','No left ROI marker selected!'),Params)
  then begin
   SystemCalculator.Eval(Format('user_marker_x=%g',[L.A.X]));
   SystemCalculator.Eval(Format('user_marker_y=%g',[L.A.Y]));
   OpenConsole(myConsoleMode and cmfMarker=0);
   ConsoleEcho(StdDateTimePrompt+RusEng('Пользовательский маркер','User marker'));
   ConsoleEcho(Format(' user_marker_x = %g',[L.A.X]));
   ConsoleEcho(Format(' user_marker_y = %g',[L.A.Y]));
  end;
 end;
end;

 {
 Редактировать точку
 }
procedure TDaqSystem.EditCurvePoint(const Params:LongString='');
var i:Integer; P:TPoint2D; L:TRect2D; C:TCurve; W:TFormCurveWindow;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  C:=W.DefCurve;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(C.Ok,RusEng('Не выбрана кривая в окне!','No curve selected!'),Params) and
     NoProblem(C.Count>0,RusEng('Нет точек на кривой!','Curve has no points!'),Params) and
     NoProblem(C.IsDynamic,RusEng('Операция недоступна для спектра!','Operation not available for spectral data!'),Params) and
     NoProblem(C.Flags and (cfNotSortedX+cfDuplicatesX)=0,RusEng('Кривая не отсортирована!','Curve is not sorted!'),Params) and
     NoProblem(not IsNanOrInf(L.A),RusEng('Не установлен левый маркер РОИ.','No left ROI marker selected!'),Params)
  then begin
   try
    C.Lock;
    i:=C.GetIndexAt(L.A.X);
    P:=C[i];
   finally
    C.UnLock;
   end;
   if PointIsEqual(P,L.A) then begin
    FormDaqPoint2DEditExecute(P,Params);
    if not PointIsEqual(P,L.A) then begin
     try
      C.Lock;
      i:=C.GetIndexAt(L.A.X);
      if PointIsEqual(L.A,C[i]) then C[i]:=P;
     finally
      C.UnLock;
     end;
     try
      W.LockDraw;
      if not RectContainsPoint(W.World,P) then W.AutoRange;
      W.Roi:=Rect2D(P,L.B);
     finally
      W.UnlockDraw;
     end;
    end;
   end else begin
    OpenConsole(false);
    Daq.ConsoleEcho(RusEng('Нет такой точки на кривой!'+EOL+
                              'Используйте Shift+Mouse при выборе левого маркера РОИ'+EOL+
                              'для выбора редактируемой точки кривой.',
                              'Not found this point!'+EOL+
                              'Use Shift+Mouse to choose left ROI marker at curve point to edit.'));
   end;
  end;
 end;
end;

 {
 Сглаживание.  Пределы заданы РОИ окна.
 }
procedure TDaqSystem.CurveSmoothing(const Params:LongString='');
var L:TRect2D; C:TCurve; W:TFormCurveWindow; i1,i2:Integer; Win:TFormCurveWindow;
const RelEps=1e-10;
begin
 if Ok then begin
  W:=ActiveCurveWindow;
  C:=W.DefCurve;
  L:=W.Roi;
  if NoProblem(W.Ok,RusEng('Не выбрано окно с данными!','No data window selected!'),Params) and
     NoProblem(C.Ok,RusEng('Не выбрана кривая в окне!','No curve selected!'),Params) and
     NoProblem(C.IsDynamic,RusEng('Операция недоступна для спектра!','Operation not available for spectral data!'),Params) and
     NoProblem(C.Flags and (cfNotSortedX+cfDuplicatesX)=0,RusEng('Кривая не отсортирована!','Curve is not sorted!'),Params) and
     NoProblem(not IsNanOrInf(L),RusEng('Не выбраны пределы (РОИ).','No ROI selected!'),Params)
  then begin
   L:=RectValidate(L);
   {open widow for origin and smoothed curve and copy ROI to this one}
   Win:=NewCurveWindow(RusEng('Сглаживание ','Smoothing ')+C.Name,'','',wsMinimized);
   if Win.Ok then begin
    try
     C.Lock;
     i1:=C.GetIndexAt(L.A.X);
     i2:=C.GetIndexAt(L.B.X);
     while (i1-1>0) and (C[i1-1].X>=L.A.X) do dec(i1);
     while (i2+1<C.Count) and (C[i2+1].X<=L.B.X) do inc(i2);
     Win.AddCurve(NewCurveCopy(C,i1,i2));
    finally
     C.Unlock;
    end;
    if Win.Curves[0].Count>3 then begin
     Win.AddCurve(NewCurveCopy(Win.Curves[0]));
     Win.Curves[0].Color:=clBlack;
     Win.Curves[0].Style:=$1F;
     Win.Curves[1].Color:=clYellow;
     Win.Curves[1].Style:=$10;
     Win.DefCurve:=nil;
     if IsNonEmptyStr(Params) then Win.ApplyParams(Params);
     {smooth and replace ROI points to smoothed one}
     if (FormDaqCurveSmoothingExecute(Win,Params)=mrOk) then
     if (Win.Curves[1].Count>3) then
     try
      C.Lock;
      i1:=C.GetIndexAt(L.A.X);
      i2:=C.GetIndexAt(L.B.X);
      while (i1-1>0) and (C[i1-1].X>=L.A.X) do dec(i1);
      while (i2+1<C.Count) and (C[i2+1].X<=L.B.X) do inc(i2);
      if (i1=0) or (Win.Curves[1][0].X>=C[i1].X-RectSizeX(L)*RelEps) then
      if (i2=C.Count-1) or (Win.Curves[1].LastPoint.X<=C[i2].X+RectSizeX(L)*RelEps) then begin
       C.DeletePoints(i1,i2-i1+1);
       C.InsertPoints(i1,Win.Curves[1].PX[0],Win.Curves[1].PY[0],Win.Curves[1].Count);
      end else begin
       ConsoleEcho(RusEng('Не могу вставить точки в кривую!','Cannot insert points to curve!'));
      end;
     finally
      C.Unlock;
     end;
    end;
   end;
   Kill(Win);
   W.DrawView;
  end;
 end;
end;

 {
 Открыть калькулятор
 }
procedure TDaqSystem.OpenCalculator(const Params:LongString='');
begin
 ExecuteCalculator(Params);
end;

 {
 *******************************************************************************
 Utilites implementation
 *******************************************************************************
 }
procedure DaqAlignStr(var Str:PureString);
begin
 Str:=ReplaceString(Str,'_',' ');
 Str:=ReplaceString(Str,'^C',^C);
 Str:=ReplaceString(Str,'^L',^L);
 Str:=ReplaceString(Str,'^R',^R);
 Str:=ReplaceString(Str,'^N',ASCII_CR);
end;

procedure DaqAlignStr(var Str:LongString);
var S:PureString;
begin
 S:=Str;
 DaqAlignStr(S);
 Str:=S;
end;

procedure DaqClearCurve(Curve:TCurve; HistLen:LongInt);
var i:longint;
begin
 if Curve is TCurve then
 try
  Curve.Lock;
  if Curve.IsDynamic then begin
   if HistLen>0 then Curve.ClearHistory(HistLen) else Curve.Count:=0;
  end else begin
   for i:=0 to Curve.Count-1 do Curve[i]:=Point2D(Curve[i].X, 0);
  end;
 finally
  Curve.Unlock;
 end;
end;

function DaqIntegral(Curve:TCurve; a,b:Double):Double;
var i,i1,i2:LongInt; sum:Extended;
begin
 sum:=0;
 if Curve.Ok then
 try
  Curve.Lock;
  i1:=Curve.GetIndexAt(a);
  i2:=Curve.GetIndexAt(b);
  sum:=0;
  sum:=sum+Trapezium(a,Curve.Interpolate(a),Curve[i1].X,Curve[i1].Y);
  for i:=i1 to i2-1 do sum:=sum+Trapezium(Curve[i].X,   Curve[i].Y,
                                          Curve[i+1].X, Curve[i+1].Y);
  sum:=sum+Trapezium(Curve[i2].X, Curve[i2].Y, b, Curve.Interpolate(b));
 finally
  Curve.Unlock;
 end;
 Result:=sum;
end;

 {
 *******************************************************************************
 DaqTopList implementation
 *******************************************************************************
 }
procedure DaqTopListMonitor_Send;
var TopList:LongString;
begin
 TopList:='';
 if (CurveTopListLimit>0)   then TopList:=Format('%s Curve_Window %d',[TopList,CurveTopListLimit]);
 if (TableTopListLimit>0)   then TopList:=Format('%s Tab_Window %d',[TopList,TableTopListLimit]);
 if (CircuitTopListLimit>0) then TopList:=Format('%s Circuit_Window %d',[TopList,CircuitTopListLimit]);
 if (SpectrTopListLimit>0)  then TopList:=Format('%s Spectr_Window %d',[TopList,SpectrTopListLimit]);
 if IsNonEmptyStr(TopList)  then SendToMainConsole(Format('@silent @daq toplist %s',[Trim(TopList)])+EOL);
end;

procedure DaqTopListMonitor_Timer;
const Cycle:Integer=0; var Period:Integer;
begin
 Period:=DaqTopListPeriod;
 if (Period>0) then begin
  if (Cycle=0) then DaqTopListMonitor_Send;
  Inc(Cycle); if (Cycle>=Period) then Cycle:=0;
 end;
end;

procedure DaqTopListMonitor_Start;
begin
 SecondActions.Add(DaqTopListMonitor_Timer);
end;

procedure DaqTopListMonitor_Stop;
begin
 SecondActions.Remove(DaqTopListMonitor_Timer);
end;

 {
 ******************************************
 GetDaqSysInfo - Get DAQ System Information
 ******************************************
 }
const
 DaqSysInfoHasher:THashList=nil;

function GetDaqSysInfo(const What:LongString):LongString;
 type
  TStringIdentifier = (
   sid_Unknown,
   sid_TagCount,
   sid_CurveCount,
   sid_DeviceCount,
   sid_SensorCount,
   sid_Tab_WindowCount,
   sid_Tab_WindowDrawViewCount,
   sid_Tab_WindowMonitorCallCount,
   sid_Tab_WindowMonitorDrawCount,
   sid_Curve_WindowCount,
   sid_Curve_WindowDrawViewCount,
   sid_Curve_WindowDrawCurveCount,
   sid_Curve_WindowDrawPointCount,
   sid_Curve_WindowMonitorCallCount,
   sid_Curve_WindowMonitorDrawCount,
   sid_Curve_WindowMonitorCurveCount,
   sid_Curve_WindowMonitorPointCount,
   sid_Spectr_WindowCount,
   sid_Circuit_WindowCount,
   sid_Circuit_WindowDrawViewCount,
   sid_Circuit_WindowMakeBitmapCount,
   sid_Circuit_WindowDrawSensorCount,
   sid_Circuit_WindowPollSensorCount,
   sid_Circuit_WindowTagEvalCallCount,
   sid_Circuit_WindowLedEvalCallCount,
   sid_Circuit_WindowPainterCallCount,
   sid_Circuit_WindowPainterApiCallCount,
   sid_Circuit_WindowPainterApiDrawCount,
   sid_Circuit_WindowMonitorCallCount,
   sid_Circuit_WindowMonitorDrawCount,
   sid_Unused
  );
 procedure InitDaqSysInfoHasher;
 procedure AddSid(const key:LongString; sid:TStringIdentifier);
 begin
  DaqSysInfoHasher.KeyedLinks[key]:=Ord(sid);
 end;
 begin
  if (DaqSysInfoHasher<>nil) then Exit;
  DaqSysInfoHasher:=NewHashList(false,HashList_DefaultHasher);
  DaqSysInfoHasher.Master:=@DaqSysInfoHasher;
  AddSid( 'Tag.Count'                           , sid_TagCount);                          // Счетчик всех тегов
  AddSid( 'Curve.Count'                         , sid_CurveCount);                        // Счетчик всех кривых
  AddSid( 'Device.Count'                        , sid_DeviceCount);                       // Счетчик всех устройств
  AddSid( 'Sensor.Count'                        , sid_SensorCount);                       // Счетчик всех сенсоров
  AddSid( 'Tab_Window.Count'                    , sid_Tab_WindowCount);                   // Счетчик всех окон Tab_Window
  AddSid( 'Tab_Window.DrawView.Count'           , sid_Tab_WindowDrawViewCount);           // Счетчик рисований окна
  AddSid( 'Tab_Window.MonitorCall.Count'        , sid_Tab_WindowMonitorCallCount);        // Счетчик вызовов   монитора окон
  AddSid( 'Tab_Window.MonitorDraw.Count'        , sid_Tab_WindowMonitorDrawCount);        // Счетчик рисований в мониторе окон
  AddSid( 'Curve_Window.Count'                  , sid_Curve_WindowCount);                 // Счетчик всех окон Curve_Window
  AddSid( 'Curve_Window.DrawView.Count'         , sid_Curve_WindowDrawViewCount);         // Счетчик рисований окна
  AddSid( 'Curve_Window.DrawCurve.Count'        , sid_Curve_WindowDrawCurveCount);        // Счетчик рисований кривых
  AddSid( 'Curve_Window.DrawPoint.Count'        , sid_Curve_WindowDrawPointCount);        // Счетчик рисований точек
  AddSid( 'Curve_Window.MonitorCall.Count'      , sid_Curve_WindowMonitorCallCount);      // Счетчик вызовов   монитора окон
  AddSid( 'Curve_Window.MonitorDraw.Count'      , sid_Curve_WindowMonitorDrawCount);      // Счетчик рисований в мониторе окон
  AddSid( 'Curve_Window.MonitorCurve.Count'     , sid_Curve_WindowMonitorCurveCount);     // Счетчик рисований кривых в мониторе
  AddSid( 'Curve_Window.MonitorPoint.Count'     , sid_Curve_WindowMonitorPointCount);     // Счетчик рисований точек  в мониторе
  AddSid( 'Spectr_Window.Count'                 , sid_Spectr_WindowCount);                // Счетчик всех окон Spectr_Window
  AddSid( 'Circuit_Window.Count'                , sid_Circuit_WindowCount);               // Счетчик всех окон Circuit_Window
  AddSid( 'Circuit_Window.DrawView.Count'       , sid_Circuit_WindowDrawViewCount);       // Счетчик рисований окна
  AddSid( 'Circuit_Window.MakeBitmap.Count'     , sid_Circuit_WindowMakeBitmapCount);     // Счетчик создания  изображений  сенсора
  AddSid( 'Circuit_Window.DrawSensor.Count'     , sid_Circuit_WindowDrawSensorCount);     // Счетчик рисований (реальных)   сенсора
  AddSid( 'Circuit_Window.PollSensor.Count'     , sid_Circuit_WindowPollSensorCount);     // Счетчик опросов для обновления сенсора
  AddSid( 'Circuit_Window.TagEvalCall.Count'    , sid_Circuit_WindowTagEvalCallCount);    // Счетчик вызовов формулы   TagEval(v)
  AddSid( 'Circuit_Window.LedEvalCall.Count'    , sid_Circuit_WindowLedEvalCallCount);    // Счетчик вызовов формулы   LedEval(v)
  AddSid( 'Circuit_Window.PainterCall.Count'    , sid_Circuit_WindowPainterCallCount);    // Счетчик вызовов сценариев Painter(v)
  AddSid( 'Circuit_Window.PainterApiCall.Count' , sid_Circuit_WindowPainterApiCallCount); // Счетчик вызовов функций   Painter
  AddSid( 'Circuit_Window.PainterApiDraw.Count' , sid_Circuit_WindowPainterApiDrawCount); // Счетчик рисований внутри  Painter
  AddSid( 'Circuit_Window.MonitorCall.Count'    , sid_Circuit_WindowMonitorCallCount);    // Счетчик вызовов монитора окон
  AddSid( 'Circuit_Window.MonitorDraw.Count'    , sid_Circuit_WindowMonitorDrawCount);    // Счетчик рисований в мониторе
 end;
 function Identify(const key:LongString):TStringIdentifier;
 var sid:Integer;
 begin
  if (DaqSysInfoHasher=nil) then InitDaqSysInfoHasher;
  sid:=DaqSysInfoHasher.KeyedLinks[key];
  if (sid>=Ord(Low(TStringIdentifier))) and (sid<=Ord(High(TStringIdentifier)))
  then Result:=TStringIdentifier(sid)
  else Result:=sid_Unknown;
 end;
begin
 Result:='';
 if (What<>'') then
 try
  case Identify(What) of
   sid_TagCount                          : Result:=IntToStr(CountTags);
   sid_CurveCount                        : Result:=IntToStr(Daq.Curves.Count);
   sid_DeviceCount                       : Result:=IntToStr(FullDaqDeviceList.Count);
   sid_SensorCount                       : Result:=IntToStr(FullCircuitWindowList.SensorsCount);
   sid_Tab_WindowCount                   : Result:=IntToStr(Daq.TabWinList.Count);
   sid_Tab_WindowDrawViewCount           : Result:=IntToStr(TableWindowsProfiler.Curr.DrawView);
   sid_Tab_WindowMonitorCallCount        : Result:=IntToStr(TableWindowsProfiler.Curr.MonitorCall);
   sid_Tab_WindowMonitorDrawCount        : Result:=IntToStr(TableWindowsProfiler.Curr.MonitorDraw);
   sid_Curve_WindowCount                 : Result:=IntToStr(Daq.CurWinList.Count);
   sid_Curve_WindowDrawViewCount         : Result:=IntToStr(CurveWindowsProfiler.Curr.DrawView);
   sid_Curve_WindowDrawCurveCount        : Result:=IntToStr(CurveWindowsProfiler.Curr.DrawCurve);
   sid_Curve_WindowDrawPointCount        : Result:=IntToStr(CurveWindowsProfiler.Curr.DrawPoint);
   sid_Curve_WindowMonitorCallCount      : Result:=IntToStr(CurveWindowsProfiler.Curr.MonitorCall);
   sid_Curve_WindowMonitorDrawCount      : Result:=IntToStr(CurveWindowsProfiler.Curr.MonitorDraw);
   sid_Curve_WindowMonitorCurveCount     : Result:=IntToStr(CurveWindowsProfiler.Curr.MonitorCurve);
   sid_Curve_WindowMonitorPointCount     : Result:=IntToStr(CurveWindowsProfiler.Curr.MonitorPoint);
   sid_Spectr_WindowCount                : Result:=IntToStr(Daq.SpeWinList.Count);
   sid_Circuit_WindowCount               : Result:=IntToStr(Daq.CirWinList.Count);
   sid_Circuit_WindowDrawViewCount       : Result:=IntToStr(CircuitWindowsProfiler.Curr.DrawView);
   sid_Circuit_WindowMakeBitmapCount     : Result:=IntToStr(CircuitWindowsProfiler.Curr.MakeBitmap);
   sid_Circuit_WindowDrawSensorCount     : Result:=IntToStr(CircuitWindowsProfiler.Curr.DrawSensor);
   sid_Circuit_WindowPollSensorCount     : Result:=IntToStr(CircuitWindowsProfiler.Curr.PollSensor);
   sid_Circuit_WindowTagEvalCallCount    : Result:=IntToStr(CircuitWindowsProfiler.Curr.TagEvalCall);
   sid_Circuit_WindowLedEvalCallCount    : Result:=IntToStr(CircuitWindowsProfiler.Curr.LedEvalCall);
   sid_Circuit_WindowPainterCallCount    : Result:=IntToStr(CircuitWindowsProfiler.Curr.PainterCall);
   sid_Circuit_WindowPainterApiCallCount : Result:=IntToStr(CircuitWindowsProfiler.Curr.PainterApiCall);
   sid_Circuit_WindowPainterApiDrawCount : Result:=IntToStr(CircuitWindowsProfiler.Curr.PainterApiDraw);
   sid_Circuit_WindowMonitorCallCount    : Result:=IntToStr(CircuitWindowsProfiler.Curr.MonitorCall);
   sid_Circuit_WindowMonitorDrawCount    : Result:=IntToStr(CircuitWindowsProfiler.Curr.MonitorDraw);
  end;
 except
  on E:Exception do BugReport(E,nil,'GetDaqSysInfo');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqsys;
begin
end;

procedure Free_crw_daqsys;
begin
 Kill(DaqSysInfoHasher);
end;

initialization

 Init_crw_daqsys;

finalization

 Free_crw_daqsys;

end.

//////////////
// END OF FILE
//////////////

