////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// DAQ PASCAL P-code runtime interpreter.                                     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20020308 - Modified for Delphi (A.K.)                                      //
// 20230501 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqpascalruntime; // CRW DAQ Pascal Runtime

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 _crw_alloc, _crw_str, _crw_fio, _crw_fifo, _crw_fpu,
 _crw_pio, _crw_ef, _crw_zm, _crw_rtc, _crw_daqtags,
 _crw_daqsys, _crw_daqdev, _crw_utf8, _crw_adamdev,
 _crw_daqpascalcompiler;

 {
 **************************
 P-code runtime interpreter
 **************************
 }
function RtmInterpreter(var TheDaqPascal:TDaqPascal):boolean;

implementation

 {
 **************************
 P-code runtime interpreter
 **************************
 }
function RtmInterpreter(var TheDaqPascal:TDaqPascal):boolean;
const
 fldi     = 8;                        { default integer width  }
 fldr     = 20;                       { default real    width  }
 fldb     = 8;                        { default boolean width  }
 fldc     = 1;                        { default char    width  }
type
 pstype   = (ps_Running,              { program running now    }
             ps_Finalized,            { successfully finalized }
             ps_StackOverflow,        { stack overflow detect  }
             ps_CaseCheck,            { error in case statement}
             ps_DivByZero,            { divizion by zero detect}
             ps_IndexCheck,           { out of index range     }
             ps_StringManager,        { string manager error   }
             ps_UnexpectedBreak,      { unknown exception      }
             ps_ExternalBreak,        { break on external flag }
             ps_StackUnbalanced,      { stack balance error    }
             ps_CrvLockUnbalanced,    { crvlock balance error  }
             ps_TagLockUnbalanced,    { taglock balance error  }
             ps_SystemCheck);         { invalid opcode/operand }
var
 opcnt  : integer;                    { local operation counter}
 opcode : integer;                    { current operation code }
 xoper  : integer;                    { x operand              }
 yoper  : integer;                    { y operand              }
 prgcnt : integer;                    { program counter        }
 prgsts : pstype;                     { program status         }
 t      : integer;                    { stack pointer          }
 tstart : integer;                    { start value of t       }
 b      : integer;                    { base pointer of block  }
 b0     : integer;                    { string table pointer   }
 i      : integer;                    { temporary }
 h1     : integer; memsh1 : strngptr; { temporary }
 h2     : integer; memsh2 : strngptr; { temporary }
 h3     : integer; memsh3 : strngptr; { temporary }
 h4     : integer; memsh4 : strngptr; { temporary }
 h5     : integer; memsh5 : strngptr; { temporary }
 h6     : integer; memsh6 : strngptr; { temporary }
 pch1   : PChar;                      { temporary }
 pch2   : PChar;                      { temporary }
 r1     : real;                       { temporary }
 blkcnt : integer;                    { temporary }
 brkflg : integer;                    { break flag }
 {
 procedure raise exception with given error code
 }
 procedure RunTimeError(errno:pstype; trap:PChar=nil);
 var msg:LongString;
 begin
  with TheDaqPascal do begin
   prgsts:=errno;
   ExceptResult:=ord(errno);
   msg:=Format('Daq Pascal Runtime Error %d, opcode %d',[ExceptResult, opcode]);
   if Assigned(trap) then msg:=msg+', trap '+StrPas(trap);
   if (opcode=8) then msg:=msg+', fun '+IntToStr(yoper);
   RAISE EDaqPascalRuntimeError.Create(msg);
  end;
 end;
 {
 procedure moves size bytes from src to dst
 }
 procedure move(const src; var dst; size:Integer);
 begin
  if size>0 then system.move(src,dst,size);
 end;
 {
 nil-protected string buffer pointer of string s
 }
 function memstrng(s:Integer):strngptr;
 begin
  with TheDaqPascal do begin
   if s<1 then RunTimeError(ps_StringManager,'memstrng:1');
   if s>stabmax then RunTimeError(ps_StringManager,'memstrng:2');
   Result:=stab.table[s];
   if Result=nil then RunTimeError(ps_StringManager,'memstrng:3');
  end;
 end;
 {
 stack dump for debug
 }
 procedure dump;
 var p,h3:integer;
 begin
  with TheDaqPascal do
  with dtab^ do begin
   SmartFileClose(LstOut);
   Assign(LstOut,LstFile);
   Append(LstOut);
   h3:=itab[h2].lev;
   writeln(LstOut);
   writeln(LstOut);
   writeln(LstOut,'        calling ',itab[h2].name);
   writeln(LstOut,'          level ',h3:4);
   writeln(LstOut,' start of  code ',prgcnt:4);
   writeln(LstOut);
   writeln(LstOut);
   writeln(LstOut,' contents of display ');
   writeln(LstOut);
   for p:=h3+1 downto 1 do writeln(LstOut,p:4,display[p]:6);
   writeln(LstOut);
   writeln(LstOut);
   writeln(LstOut,' top of stack   ',t:4,' frame base ':14,b:4);
   writeln(LstOut);
   writeln(LstOut);
   writeln(LstOut,'stack contents':20);
   writeln(LstOut);
   for  p:=t  downto  1  do writeln(LstOut,p:14,s[p].i:8);
   writeln(LstOut,'< = = = >':22);
   SmartFileClose(LstOut);
   if IOResult<>0 then;
  end;
 end;
 {
 procedure allocates memory for dynamic string of length t (uses alignment to minimize number of memory reallocations)
 }
 function sget(out s:integer; t:integer):strngptr;
 var v:integer;
 begin
  with TheDaqPascal do begin
   v := StrAlignLeng(t);  { get size aligned to strngalign and check size is valid }
   if (v<t+sizeof(strnghdr)) or (t<0) or (v>slenmax) then RunTimeError(ps_StringManager,'sget:1');
   s := StrAlloc(v);
   if s<1 then RunTimeError(ps_StringManager,'sget:2');
   if s>stabmax then RunTimeError(ps_StringManager,'sget:3');
   Result:=stab.table[s];
   if Result=nil then RunTimeError(ps_StringManager,'sget:4');
   Result.head.leng := t;
   Result.head.dyna := v-sizeof(strnghdr);
  end;
 end;
 {
 procedure free memory under dynamic (!) string p
 }
 procedure sfree(p:integer);
 begin
  with TheDaqPascal do begin
   if memstrng(p).head.dyna=0 then exit; { static string ! }
   StrFree(p);
  end;
 end;
 {
 Block frame s[b+5].i contains entry to list of temporary strings for current
 block (procedure or function) with base pointer b.
 This list uses to release temporary strings on procedure/function exit.
 This procedure uses to link string s[j].i to this list.
 Also assign b0 pointer to parent block's entry for future.
 }
 procedure link(j:integer);
 var i: integer;
 begin
  with TheDaqPascal do
  with dtab^ do begin
   b0 := b;
   i := itab[s[b0+4].i].lev;
   while j<b0 do begin
    b0 := display[i];
    i := i-1;
   end;
   s[j].p := s[b0+5].i;
   s[b0+5].i := j;
   s[j].cn := strngs;
   // Set b0 pointer to parent block's list entry
   for i:=b0_count-1 downto 0 do
   if b0_stack[i]<b+5 then begin
    b0:=b0_stack[i];
    Break;
   end;
  end;
 end;
 {
 copy string rt to lf
 }
 procedure scopy(lf,rt:integer);
 var h1,h2,h3,h4: integer; memsh1,memsh3:strngptr;
 begin
  with TheDaqPascal do
  with dtab^ do begin
   h1 := s[lf].s; if h1=0 then memsh1:=nil else memsh1:=memstrng(h1);
   if h1=0 then h2:=0 else h2 := memsh1.head.dyna;
   h3 := s[rt].s; memsh3:=memstrng(h3);
   h4 := memsh3.head.leng;
   if (h1 = 0) or (h2 < h4) or (h2 >= h4+strngalign) then begin
    if h1=0 then link(lf) else if h2<>0 then sfree(h1);
    memsh1:=sget(h1,h4);
    s[lf].s := h1;
   end else memsh1.head.leng := h4;
   if prgsts = ps_Running then move(memsh3.data.buff,memsh1.data.buff,h4);
  end;
 end;
 {
 extract string for given dynamic string handle h1
 mode = 0 - string, 1 - temporary string, 2 - char
 }
 procedure extract(h1:integer; var sbuff:PureString; mode:integer);
 var h2:integer; memsh1:strngptr;
 begin
  case mode of
   0,1: { argument is string or temporary string }
    begin
     memsh1:=memstrng(h1);
     h2:=memsh1.head.leng;
     if h2>255 then h2:=255;
     SetString(sbuff,memsh1.data.buff,h2);
     if mode=1 then sfree(h1);         { argument is temporary string }
    end;
   2: { argument is char }
    sbuff:=chr(h1);
   else RunTimeError(ps_SystemCheck,'extract');
  end;
 end;
 {
 create new dynamic string and copy Pascal string s to this dynamic string
 }
 function pas2dynstr(var s:PureString):integer;
 var h1,h3:integer; memsh3:strngptr;
 begin
  with TheDaqPascal do begin
   h1:=length(s);
   if h1=0 then h3:=nul else begin
    memsh3:=sget(h3,h1);
    move(s[1],memsh3.data.buff,h1);
   end;
   pas2dynstr:=h3;
  end;
 end;
 {
 extract long string for given dynamic string handle h1
 mode = 0 - string, 1 - temporary string, 2 - char
 }
 procedure extractlong(h1:integer; var lbuff:LongString; mode:integer);
 var h2:integer; memsh1:strngptr;
 begin
  case mode of
   0,1: { argument is string or temporary string }
    begin
     memsh1:=memstrng(h1);
     h2:=memsh1.head.leng;
     SetString(lbuff,memsh1.data.buff,h2);
     if mode=1 then sfree(h1);         { argument is temporary string }
    end;
   2: { argument is char }
    lbuff:=chr(h1);
   else RunTimeError(ps_SystemCheck,'extractlong');
  end;
 end;
 {
 create new dynamic string and copy long string s to this dynamic string
 }
 function pas2dynstrlong(var s:LongString):integer;
 var h1,h3:integer; memsh3:strngptr;
 begin
  with TheDaqPascal do begin
   h1:=length(s);
   if h1=0 then h3:=nul else begin
    memsh3:=sget(h3,h1);
    move(s[1],memsh3.data.buff,h1);
   end;
   pas2dynstrlong:=h3;
  end;
 end;
 {
 fetch char with given index from string handle h1
 mode = 0 - string, 1 - temporary string, 2 - char
 }
 function strfetch(h1,index,mode:integer):integer;
 var h2:integer; memsh1:strngptr;
 begin
  Result:=0;
  case mode of
   0,1: { argument is string or temporary string }
    begin
     memsh1:=memstrng(h1);
     h2:=memsh1.head.leng;
     if (index>=1) and (index<=h2) then Result:=memsh1.data.bytes[index-1];
     if mode=1 then sfree(h1);         { argument is temporary string }
    end;
   2: { argument is char }
    if index=1 then Result:=ord(chr(h1));
   else RunTimeError(ps_SystemCheck,'strfetch');
  end;
 end;
 {
 convert string a to integer value i
 if first symbol is $, convert as hex-value
 }
 function atoi(a:PChar; var i:integer):boolean;
 var errpos,len,p:integer; L:longint;
 begin
  atoi:=false;
  if a^='$' then begin          {hex-value}
   inc(a);
   L:=0;
   len:=0;
   while a^<>#0 do begin
    p:=pos(locase(a^),'0123456789abcdef');
    if p>0 then L:=(L shl 4)+(p-1) else exit;
    inc(a);
    inc(len);
    if len>sizeof(i)*2 then exit;
   end;
   if len=0 then exit;
   i:=L;
   atoi:=true;
  end else begin                {decimal value}
   val(a,i,errpos);
   atoi:=(errpos=0);
  end;
 end;
 {
 convert string a to real value f
 if first symbol is $, convert as hex-value
 }
 function atof(a:PChar; var f:real):boolean;
 var errpos,len,p:integer; L:longint;
 begin
  atof:=false;
  if a^='$' then begin          {hex-value}
   inc(a);
   L:=0;
   len:=0;
   while a^<>#0 do begin
    p:=pos(locase(a^),'0123456789abcdef');
    if p>0 then L:=(L shl 4)+(p-1) else exit;
    inc(a);
    inc(len);
    if len>sizeof(L)*2 then exit;
   end;
   if len=0 then exit;
   f:=L;
   atof:=true;
  end else begin                {decimal value}
   val(a,f,errpos);
   atof:=(errpos=0);
  end;
 end;
 {
 convert real and integer to string
 }
 function f2s(f:real):LongString;
 begin
  f2s:=Format('%g',[f]);
 end;
 function d2s(d:integer):LongString;
 begin
  d2s:=Format('%d',[d]);
 end;
 {
 }
begin { ***************** Interpret *************************************** }
 {
 }
 brkflg:=0;
 RtmInterpreter:=false;
 with TheDaqPascal do
 if CompileOk  then
 with dtab^ do begin
  {
  After two Runtime Errors sleep some time to avoid system overload.
  }
  if (MortalCount>1) then
  if (MortalTime<>0) then begin
   if (IntMSecNow>=MortalTime+MortalDelay) or (MortalTime<0) or (MortalDelay<=0) then begin
    MortalTime:=0; // Reset postmortal timer
   end else begin
    Result:=true;  // Sleep some time after
    Exit;          // two runtime errors
   end;
  end;
  {
  this interpreter could not be call recursively!
  }
  inc(EntryCount);
  if EntryCount>1 then begin
   RtmInterpreter:=true;
   dec(EntryCount);
   exit;
  end;
  {
  Cleanup temporary strings
  }
  if (lbuf1<>'') then lbuf1:='';
  if (lbuf2<>'') then lbuf2:='';
  if (lbuf3<>'') then lbuf3:='';
  {
  protected interpreter entry
  }
  opcnt:=0;
  try
   {
   **************************
   interpreter initialization
   **************************
   }
   s[1].i := 0;
   s[2].i := 0;
   s[3].i := -1;
   s[4].i := btab[1].last;
   display[1] := 0;
   t := btab[2].vsize - 1;
   b := 0;
   prgcnt := itab[s[4].i].adr;
   prgsts := ps_Running;
   if t > stacksize then RunTimeError(ps_StackOverflow,'init');
   runcount:=runcount+1;
   tstart:=t;
   b0:=0; b0_count:=0; { initialize stack of temporary string lists }
   {
   ********************************************
   here starts the main loop of the interpreter
   ********************************************
   }
   while true do begin
    with ctab^[prgcnt] do begin
     opcode:=f;                  { get opcode            }
     xoper:=x;                   { get x operand         }
     yoper:=y;                   { get y operand         }
    end;
    inc(opcnt);                  { inc operation counter }
    inc(prgcnt);                 { inc program counter   }
    case opcode of               { P-Code implementation }
     {
     Load address
     }
     0:
     begin
      t := t+1;
      if t > stacksize then RunTimeError(ps_StackOverflow);
      s[t].i := display[xoper] + yoper;
     end;
     {
     Load value
     }
     1:
     begin
      t := t+1;
      if t > stacksize then RunTimeError(ps_StackOverflow);
      s[t] := s[display[xoper] + yoper];
     end;
     {
     Load indirect
     }
     2:
     begin
      t := t+1;
      if t > stacksize then RunTimeError(ps_StackOverflow);
      s[t] := s[s[display[xoper] + yoper].i];
     end;
     {
     Update DISPLAY
     }
     3:
     begin
      h1 := yoper;
      h2 := xoper;
      h3 := b;
      repeat
       display[h1] := h3;
       h1 := h1-1;
       h3 := s[h3+2].i
      until h1 = h2;
     end;
     {
     not used
     }
     4,
     5,
     6:
     RunTimeError(ps_SystemCheck,'opcode_4');
     {
     Concatenate
     }
     7:
     case yoper and 3 of
      0: begin   {char+char}
          memsh1:=sget(h1,2);
          memsh1.data.bytes[0] := s[t-1].i;
          memsh1.data.bytes[1] := s[t].i;
          t := t-1;
          s[t].i := h1;
         end;
      1: begin   {string+char}
          h1 := s[t-1].i; memsh1:=memstrng(h1);
          h2 := memsh1.head.leng;
          memsh3:=sget(h3,h2+1);
          move(memsh1.data.buff,memsh3.data.buff,h2);
          if (yoper and 4) = 4 then sfree(h1);
          memsh3.data.bytes[h2] := s[t].i;
          t := t-1;
          s[t].i := h3;
         end;
      2: begin   {char+string}
          h1 := s[t].i; memsh1:=memstrng(h1);
          h2 := memsh1.head.leng;
          memsh4:=sget(h4,h2+1);
          move(memsh1.data.bytes,memsh4.data.bytes[1],h2);
          memsh4.data.bytes[0] := s[t-1].i;
          if (yoper and 8) = 8 then sfree(h1);
          t := t-1;
          s[t].i := h4;
         end;
      3: begin   {string+string}
          h5 := s[t-1].i; memsh5:=memstrng(h5);
          h6 := s[t].i;   memsh6:=memstrng(h6);
          h3 := memsh5.head.leng;
          h4 := memsh6.head.leng;
          memsh2:=sget(h2,h3+h4);
          move(memsh5.data.bytes,memsh2.data.bytes,h3);
          move(memsh6.data.bytes,memsh2.data.bytes[h3],h4);
          if (yoper and 4) = 4 then sfree(h5);
          if (yoper and 8) = 8 then sfree(h6);
          t := t-1;
          s[t].i := h2;
         end;
     end;
     {
     Standard functions ********************************************************
     }
     8:
     case yoper of
      {
      abs(integer):integer
      }
      000:
      s[t].i := abs(s[t].i);
      {
      abs(real):real
      }
      001:
      s[t].r := abs(s[t].r);
      {
      sqr(integer):integer
      }
      002:
      s[t].i := sqr(s[t].i);
      {
      sqr(real):real
      }
      003:
      s[t].r := sqr(s[t].r);
      {
      odd(integer):boolean
      }
      004:
      s[t].b := odd(s[t].i);
      {
      chr(integer):char
      }
      005:
      s[t].c := chr(s[t].i);
      {
      ord(char):integer
      }
      006:
      s[t].i := ord(s[t].c);
      {
      succ(char):char
      }
      007:
      s[t].c := succ(s[t].c);
      {
      pred(char):char
      }
      008:
      s[t].c := pred(s[t].c);
      {
      round(real):integer
      }
      009:
      s[t].i := round(s[t].r);
      {
      trunc(real):integer
      }
      010:
      s[t].i := trunc(s[t].r);
      {
      sin(real):real
      }
      011:
      s[t].r := sin(s[t].r);
      {
      cos(real):real
      }
      012:
      s[t].r := cos(s[t].r);
      {
      exp(real):real
      }
      013:
      s[t].r := exp(s[t].r);
      {
      ln(real):real
      }
      014:
      s[t].r := ln(s[t].r);
      {
      sqrt(real):real
      }
      015:
      s[t].r := sqrt(s[t].r);
      {
      arctan(real):real
      }
      016:
      s[t].r := arctan(s[t].r);
      {
      eof:boolean
      }
      017:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       if (IORes=0) and not IsFileClosed(StdInp) then begin
        SetInOutRes(0);
        {$IFDEF SKIP_DRAFT}
        if SameText(StrPas(TTextRec(StdInp).Name),'FIFO:') and eof(StdInp) then begin close(StdInp); reset(StdInp); end;
        {$ENDIF ~SKIP_DRAFT}
        s[t].b := eof(StdInp);
        IORes:=IOResult;
       end else begin
        s[t].b:=false;
       end;
      end;
      {
      eoln:boolean
      }
      018:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       if (IORes=0) and not IsFileClosed(StdInp) then begin
        SetInOutRes(0);
        s[t].b := eoln(StdInp);
        IORes:=IOResult;
       end else begin
        s[t].b:=false;
       end;
      end;
      {
      maxavail:integer
      }
      019:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _maxavail;
      end;
      {
      length(string):integer
      }
      020:
      s[t].i := memstrng(s[t].i).head.leng;
      {
      length(string temporary):integer
      }
      021:
      begin
       h1 := s[t].i;
       s[t].i := memstrng(h1).head.leng;
       sfree(h1);
      end;
      {
      length(char):integer
      }
      022:
      s[t].i := 1;
      {
      copy(string,integer,integer):string
      }
      023:
      begin
       h1 := s[t-2].i; memsh1:=memstrng(h1);
       h4 := memsh1.head.leng;
       h2 := s[t-1].i;
       if (h2 < 1) or (h2 > h4) then begin h4 := 0; h2 := 2; end;
       h3 := s[t].i;
       if h3 > h4-h2+1 then h3 := h4-h2+1;
       if h3 < 0 then h3 := 0;
       memsh5:=sget(h5,h3);
       move(memsh1.data.bytes[h2-1],memsh5.data.bytes[0],h3);
       s[t-2].i := h5;
       t := t-2;
      end;
      {
      copy string temporary
      }
      024:
      begin
       h1 := s[t-2].i; memsh1:=memstrng(h1);
       h4 := memsh1.head.leng;
       h2 := s[t-1].i;
       if (h2 < 1) or (h2 > h4)
       then memsh1.head.leng := 0
       else begin
        h3 := s[t].i;
        if h3 > h4-h2+1 then h3 := h4-h2+1;
        if h3 < 0 then h3 := 0;
        move(memsh1.data.bytes[h2-1],memsh1.data.bytes[0],h3);
        memsh1.head.leng := h3;
       end;
       t := t-2;
      end;
      {
      copy char
      }
      025:
      begin
       memsh1:=sget(h1,1);
       if (s[t-1].i = 1) and (s[t].i > 0)
       then memsh1.data.bytes[0] := s[t-2].i
       else memsh1.head.leng := 0;
       s[t-2].i := h1;
       t := t-2;
      end;
      {
      pos(string,string):integer
      }
      026,
      027,
      030,
      031:
      begin
       h1 := s[t-1].i; memsh1:=memstrng(h1);
       h2 := s[t].i;   memsh2:=memstrng(h2);
       t := t-1;
       h6 := memsh1.head.leng;
       h3 := memsh2.head.leng+1-h6;
       if (h3<=0) or (h6=0)
       then s[t].i := 0
       else begin
        h4 := 0;
        while h4<h3 do begin
         h5 := 0;
         while (h5<h6) and (memsh1.data.bytes[h5]=memsh2.data.bytes[h4+h5]) do h5 := h5+1;
         if h5=h6 then h3:=h4-1 else h4 := h4+1;
        end;
        if h3=h4 then s[t].i := 0 else s[t].i := h4+1;
       end;
       if odd(yoper) then sfree(h1);
       if yoper > 29 then sfree(h2);
      end;
      {
      not executable
      }
      029:
      RunTimeError(ps_SystemCheck,'opcode_29');
      {
      pos(char,string)
      }
      028,
      032:
      begin
       h1 := s[t-1].i;
       h2 := s[t].i; memsh2:=memstrng(h2);
       h3 := memsh2.head.leng;
       h4 := 0;
       while (h4<h3) and (memsh2.data.bytes[h4]<>h1) do h4 := h4+1;
       if yoper=32 then sfree(h2); (**** was sfree(h3), that is bug! ****)
       t := t-1;
       if h4<h3 then s[t].i := h4+1 else s[t].i := 0;
      end;
      {
      str(integer):string, str(real):string
      }
      033,
      034:
      begin
       if yoper=34 then lbuf1:=f2s(s[t].r) else lbuf1:=d2s(s[t].i);
       s[t].i := pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      val(string):integer,rval(string):real
      }
      035,
      036,
      037,
      038:
      begin
       h1 := s[t].i; memsh1:=memstrng(h1);
       h2 := memsh1.head.leng;
       SetString(lbuf1,memsh1.data.buff,h2);
       lbuf1:=SysUtils.Trim(lbuf1);
       if yoper < 37 then begin
        if (Length(lbuf1)=0) or not atoi(PChar(lbuf1),s[t].i) then s[t].i:=0;
       end else begin
        if (Length(lbuf1)=0) or not atof(PChar(lbuf1),s[t].r) then s[t].r:=_nan;
       end;
       if not odd(yoper) then sfree(h1);
       lbuf1:='';
      end;
      {
      rand:real
      }
      039:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := random;
      end;
      {
      msecnow:real
      }
      040:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _msecnow;
      end;
      {
      secnow:real
      }
      041:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _secnow;
      end;
      {
      getticks:real
      }
      042:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _getticks;
      end;
      {
      time:real
      }
      043:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _time;
      end;
      {
      timeunits:real
      }
      044:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _timeunits;
      end;
      {
      numais:integer
      }
      045:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _numais;
      end;
      {
      numdis:integer
      }
      046:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _numdis;
      end;
      {
      numaos:integer
      }
      047:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _numaos;
      end;
      {
      numdos:integer
      }
      048:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _numdos;
      end;
      {
      numcals:integer
      }
      049:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _numcals;
      end;
      {
      pi:real
      }
      050:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := pi;
      end;
      {
      deg(real):real
      }
      051:
      s[t].r := RadToDeg(s[t].r);
      {
      rad(real):real
      }
      052:
      s[t].r := DegToRad(s[t].r);
      {
      tan(real):real
      }
      053:
      s[t].r := tan(s[t].r);
      {
      asin(real):real
      }
      054:
      s[t].r := arcsin(s[t].r);
      {
      acos(real):real
      }
      055:
      s[t].r := arccos(s[t].r);
      {
      atan(real):real
      }
      056:
      s[t].r := arctan(s[t].r);
      {
      sinh(real):real
      }
      057:
      s[t].r := sinh(s[t].r);
      {
      cosh(real):real
      }
      058:
      s[t].r := cosh(s[t].r);
      {
      tanh(real):real
      }
      059:
      s[t].r := tanh(s[t].r);
      {
      gamma(real):real
      }
      060:
      s[t].r := gamma(s[t].r);
      {
      isnan(real):boolean
      }
      061:
      s[t].b := isnan(s[t].r);
      {
      isinf(real):boolean
      }
      062:
      s[t].b := isinf(s[t].r);
      {
      sign(real):integer
      }
      063:
      s[t].i := round(sign(s[t].r));
      {
      int(real):real
      }
      064:
      s[t].r := int(s[t].r);
      {
      frac(real):real
      }
      065:
      s[t].r := frac(s[t].r);
      {
      floor(real):real
      }
      066:
      s[t].r := floor(s[t].r);
      {
      ceil(real):real
      }
      067:
      s[t].r := ceil(s[t].r);
      {
      log(real,real):real
      }
      068:
      begin
       s[t-1].r:=logN(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      hypot(real,real):real
      }
      069:
      begin
       s[t-1].r:=hypot(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      random(real,real):real
      }
      070:
      begin
       s[t-1].r:=s[t-1].r+random*(s[t].r-s[t-1].r);
       t:=t-1;
      end;
      {
      max(real,real):real
      }
      071:
      begin
       s[t-1].r:=max(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      min(real,real):real
      }
      072:
      begin
       s[t-1].r:=min(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      eq(real,real):boolean
      }
      073:
      begin
       s[t-1].b:=(s[t-1].r=s[t].r);
       t:=t-1;
      end;
      {
      ne(real,real):boolean
      }
      074:
      begin
       s[t-1].b:=(s[t-1].r<>s[t].r);
       t:=t-1;
      end;
      {
      lt(real,real):boolean
      }
      075:
      begin
       s[t-1].b:=(s[t-1].r<s[t].r);
       t:=t-1;
      end;
      {
      gt(real,real):real
      }
      076:
      begin
       s[t-1].b:=(s[t-1].r>s[t].r);
       t:=t-1;
      end;
      {
      le(real,real):boolean
      }
      077:
      begin
       s[t-1].b:=(s[t-1].r<=s[t].r);
       t:=t-1;
      end;
      {
      ge(real,real):boolean
      }
      078:
      begin
       s[t-1].b:=(s[t-1].r>=s[t].r);
       t:=t-1;
      end;
      {
      bitor(real,real):real
      }
      079:
      begin
       s[t-1].r:=round(s[t-1].r) or round(s[t].r);
       t:=t-1;
      end;
      {
      bitxor(real,real):real
      }
      080:
      begin
       s[t-1].r:=round(s[t-1].r) xor round(s[t].r);
       t:=t-1;
      end;
      {
      bitand(real,real):real
      }
      081:
      begin
       s[t-1].r:=round(s[t-1].r) and round(s[t].r);
       t:=t-1;
      end;
      {
      refai(integer):integer
      }
      082:
      s[t].i:=_refai(s[t].i);
      {
      refdi(integer):integer
      }
      083:
      s[t].i:=_refdi(s[t].i);
      {
      refao(integer):integer
      }
      084:
      s[t].i:=_refao(s[t].i);
      {
      refdo(integer):integer
      }
      085:
      s[t].i:=_refdo(s[t].i);
      {
      refcalibr(integer):integer
      }
      086:
      s[t].i:=_refcalibr(s[t].i);
      {
      getai_n(integet):real
      }
      087:
      s[t].r:=_getai_n(s[t].i);
      {
      getdi_n(integer):real
      }
      088:
      s[t].r:=_getdi_n(s[t].i);
      {
      crvlen(real):real
      }
      089:
      s[t].r:=_crvlen(s[t].r);
      {
      crvx(real,real):real
      }
      090:
      begin
       s[t-1].r:=_crvx(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      crvx(real,real):real
      }
      091:
      begin
       s[t-1].r:=_crvy(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      getai_xn(integer):real
      }
      092:
      s[t].r:=_getai_xn(s[t].i);
      {
      getai_yn(integer):real
      }
      093:
      s[t].r:=_getai_yn(s[t].i);
      {
      getdi_xn(integer):real
      }
      094:
      s[t].r:=_getdi_xn(s[t].i);
      {
      getdi_yn(integer):real
      }
      095:
      s[t].r:=_getdi_yn(s[t].i);
      {
      getai(integer,real):real
      }
      096:
      begin
       s[t-1].r:=_getai(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      getdi(integer,real):real
      }
      097:
      begin
       s[t-1].r:=_getdi(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      getai_xi(integer,real):real
      }
      098:
      begin
       s[t-1].r:=_getai_xi(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      getai_yi(integer,real):real
      }
      099:
      begin
       s[t-1].r:=_getai_yi(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      getdi_xi(integer,real):real
      }
      100:
      begin
       s[t-1].r:=_getdi_xi(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      getdi_yi(integer,real):real
      }
      101:
      begin
       s[t-1].r:=_getdi_yi(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      crvput(real,real,real,real):boolean
      }
      102:
      begin
       s[t-3].b:=_crvput(s[t-3].r,s[t-2].r,s[t-1].r,s[t].r);
       t:=t-3;
      end;
      {
      crvinteg(real,real,real):real
      }
      103:
      begin
       s[t-2].r:=_crvinteg(s[t-2].r,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      putev(integer,integer,real,real,real):boolean
      }
      104:
      begin
       s[t-4].b:=_putev(s[t-4].i,s[t-3].i,s[t-2].r,s[t-1].r,s[t].r);
       t:=t-4;
      end;
      {
      putao(integer,real,real):boolean
      }
      105:
      begin
       s[t-2].b:=_putao(s[t-2].i,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      putdo(integer,real,real)
      }
      106:
      begin
       s[t-2].b:=_putdo(s[t-2].i,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      calibr(integer,real,real):real
      }
      107:
      begin
       s[t-2].r:=_calibr(s[t-2].i,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      power(real,real):real
      }
      108:
      begin
       if frac(s[t].r)=0 then begin
        if s[t-1].r=2
        then s[t-1].r:=fscale(1.0,round(s[t].r))
        else s[t-1].r:=intpower(s[t-1].r,round(s[t].r))
       end else s[t-1].r:=power(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      bitnot(real):real
      }
      109:
      s[t].r:=not round(s[t].r);
      {
      getbitmask(integer):real
      }
      110:
      s[t].r:=fscale(1.0,s[t].i);
      {
      fixerror(integer):boolean
      }
      111:
      s[t].b:=_fixerror(s[t].i);
      {
      inportb(integer):integer
      }
      112:
      s[t].i:=Port[s[t].i];
      {
      outportb(integer,integer):integer
      }
      113:
      begin
       Port[s[t-1].i]:=s[t].i;
       s[t-1].i:=s[t].i;
       t:=t-1;
      end;
      {
      isbit(real,integer):boolean
      }
      114:
      begin
       s[t-1].b:=round(s[t-1].r) and GetBitMask(s[t].i) <> 0;
       t:=t-1;
      end;
      {
      reset(string):integer
      }
      115,
      116,
      117:
      begin
       extractlong(s[t].i,lbuf1,yoper-115);
       s[t].i:=_reset(lbuf1);
       lbuf1:='';
      end;
      {
      rewrite(string):integer
      }
      118,
      119,
      120:
      begin
       extractlong(s[t].i,lbuf1,yoper-118);
       s[t].i:=_rewrite(lbuf1);
       lbuf1:='';
      end;
      {
      append(string):integer
      }
      121,
      122,
      123:
      begin
       extractlong(s[t].i,lbuf1,yoper-121);
       s[t].i:=_append(lbuf1);
       lbuf1:='';
      end;
      {
      ioresult:integer
      }
      124:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _ioresult;
      end;
      {
      fileexists(string):boolean
      }
      125,
      126,
      127:
      begin
       extractlong(s[t].i,lbuf1,yoper-125);
       s[t].b:=fileexists(lbuf1);
       lbuf1:='';
      end;
      {
      runcount:real
      }
      128:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := runcount;
      end;
      {
      voice(string):boolean
      }
      129,
      130,
      131:
      begin
       extractlong(s[t].i,lbuf1,yoper-129);
       s[t].b:=_voice(lbuf1);
       lbuf1:='';
      end;
      {
      action(string):boolean
      }
      132,
      133,
      134:
      begin
       extractlong(s[t].i,lbuf1,yoper-132);
       s[t].b:=_action(lbuf1);
       lbuf1:='';
      end;
      {
      clear(string):boolean
      }
      135,
      136,
      137:
      begin
       extractlong(s[t].i,lbuf1,yoper-135);
       s[t].b:=_clear(lbuf1);
       lbuf1:='';
      end;
      {
      cleardev(string):boolean
      }
      138,
      139,
      140:
      begin
       extractlong(s[t].i,lbuf1,yoper-138);
       s[t].b:=_cleardevice(lbuf1);
       lbuf1:='';
      end;
      {
      start(string):boolean
      }
      141,
      142,
      143:
      begin
       extractlong(s[t].i,lbuf1,yoper-141);
       s[t].b:=_start(lbuf1);
       lbuf1:='';
      end;
      {
      stop(string):boolean
      }
      144,
      145,
      146:
      begin
       extractlong(s[t].i,lbuf1,yoper-144);
       s[t].b:=_stop(lbuf1);
       lbuf1:='';
      end;
      {
      devmsg(string):real
      devsendmsg(string):real
      }
      147,
      148,
      149:
      begin
       extractlong(s[t].i,lbuf1,yoper-147);
       s[t].r:=_devsendmsg(lbuf1);
       lbuf1:='';
      end;
      {
      debugout(string):boolean
      }
      150,
      151,
      152:
      begin
       extractlong(s[t].i,lbuf1,yoper-150);
       s[t].b:=_debugout(lbuf1);
       lbuf1:='';
      end;
      {
      clearcurve(string):boolean
      }
      153,
      154,
      155:
      begin
       extractlong(s[t].i,lbuf1,yoper-153);
       s[t].b:=_clearcurve(lbuf1);
       lbuf1:='';
      end;
      {
      savecrw(string):boolean
      }
      156,
      157,
      158:
      begin
       extractlong(s[t].i,lbuf1,yoper-156);
       s[t].b:=_savecrw(lbuf1);
       lbuf1:='';
      end;
      {
      specmark(string):real
      }
      159,
      160,
      161:
      begin
       extractlong(s[t].i,lbuf1,yoper-159);
       s[t].r:=_specmarker(lbuf1);
       lbuf1:='';
      end;
      {
      specmarkl(string):real
      }
      162,
      163,
      164:
      begin
       extractlong(s[t].i,lbuf1,yoper-162);
       s[t].r:=_specmarkerl(lbuf1);
       lbuf1:='';
      end;
      {
      specmarkr(string):real
      }
      165,
      166,
      167:
      begin
       extractlong(s[t].i,lbuf1,yoper-165);
       s[t].r:=_specmarkerr(lbuf1);
       lbuf1:='';
      end;
      {
      specroil(string):real
      }
      168,
      169,
      170:
      begin
       extractlong(s[t].i,lbuf1,yoper-168);
       s[t].r:=_specroil(lbuf1);
       lbuf1:='';
      end;
      {
      specroir(string):real
      }
      171,
      172,
      173:
      begin
       extractlong(s[t].i,lbuf1,yoper-171);
       s[t].r:=_specroir(lbuf1);
       lbuf1:='';
      end;
      {
      windraw(string):boolean
      }
      174,
      175,
      176:
      begin
       extractlong(s[t].i,lbuf1,yoper-174);
       s[t].b:=_windraw(lbuf1);
       lbuf1:='';
      end;
      {
      winshow(string):boolean
      }
      177,
      178,
      179:
      begin
       extractlong(s[t].i,lbuf1,yoper-177);
       s[t].b:=_winshow(lbuf1);
       lbuf1:='';
      end;
      {
      winhide(string):boolean
      }
      180,
      181,
      182:
      begin
       extractlong(s[t].i,lbuf1,yoper-180);
       s[t].b:=_winhide(lbuf1);
       lbuf1:='';
      end;
      {
      winselect(string):boolean
      }
      183,
      184,
      185:
      begin
       extractlong(s[t].i,lbuf1,yoper-183);
       s[t].b:=_winselect(lbuf1);
       lbuf1:='';
      end;
      {
      global(string):real
      }
      186,
      187,
      188:
      begin
       extractlong(s[t].i,lbuf1,yoper-186);
       s[t].r:=_global(lbuf1);
       lbuf1:='';
      end;
      {
      tm_new:integer
      }
      189:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _tm_new;
      end;
      {
      tm_free(real):boolean
      }
      190:
      s[t].b:=_tm_free(s[t].r);
      {
      tm_gettime(real):real
      }
      191:
      s[t].r:=_tm_gettime(s[t].r);
      {
      tm_start(real):boolean
      }
      192:
      s[t].b:=_tm_start(s[t].r);
      {
      tm_stop(real):boolean
      }
      193:
      s[t].b:=_tm_stop(s[t].r);
      {
      tm_isstart(real):boolean
      }
      194:
      s[t].b:=_tm_isstart(s[t].r);
      {
      tm_event(real):boolean
      }
      195:
      s[t].b:=_tm_event(s[t].r);
      {
      tm_curint(real):integer
      }
      196:
      s[t].i:=_tm_curint(s[t].r);
      {
      tm_numint(real):integer
      }
      197:
      s[t].i:=_tm_numint(s[t].r);
      {
      tm_addint(real,real):boolean
      }
      198:
      begin
       s[t-1].b:=_tm_addint(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      tm_getint(real,integer):real
      }
      199:
      begin
       s[t-1].r:=_tm_getint(s[t-1].r,s[t].i);
       t:=t-1;
      end;
      {
      tm_setint(real,integer,real):boolean
      }
      200:
      begin
       s[t-2].b:=_tm_setint(s[t-2].r,s[t-1].i,s[t].r);
       t:=t-2;
      end;
      {
      _nan:real
      }
      201:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _nan;
      end;
      {
      _plusinf:real
      }
      202:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _plusinf;
      end;
      {
      _minusinf:real
      }
      203:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _minusinf;
      end;
      {
      _nil:integer
      }
      204:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := 0;
      end;
      {
      macheps:real
      }
      205:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := macheps;
      end;
      {
      maxint:integer
      }
      206:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := maxint;
      end;
      {
      comopen(string):boolean
      }
      207,
      208,
      209:
      begin
       extractlong(s[t].i,lbuf1,yoper-207);
       s[t].b:=_comopen(lbuf1);
       lbuf1:='';
      end;
      {
      comclose:boolean
      }
      210:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _comclose;
      end;
      {
      comcount:integer
      }
      211:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _comcount;
      end;
      {
      comwrite(string):boolean
      }
      212,
      213,
      214:
      begin
       extractlong(s[t].i,lbuf1,yoper-212);
       s[t].b:=_comwrite(lbuf1);
       lbuf1:='';
      end;
      {
      comread(integer):string
      }
      215:
      begin
       lbuf1:=_comread(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      aimap(integer,integer):real
      }
      216:
      begin
       s[t-1].r:=_aimap(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      dimap(integer,integer):real
      }
      217:
      begin
       s[t-1].r:=_dimap(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      aomap(integer,integer):real
      }
      218:
      begin
       s[t-1].r:=_aomap(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      domap(integer,integer):real
      }
      219:
      begin
       s[t-1].r:=_domap(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      diword(integer,integer):real
      }
      220:
      begin
       s[t-1].r:=_diword(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      irqinit(integer):boolean
      }
      221:
      s[t].b:=_irqinit(s[t].i);
      {
      irqfree:boolean
      }
      222:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _irqfree;
      end;
      {
      isisrnow:boolean
      }
      223:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _isisrnow;
      end;
      {
      readini(string):string
      }
      224,
      225,
      226:
      begin
       extractlong(s[t].i,lbuf1,yoper-224);
       lbuf1:=_readini(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      function caminitpkk(integer,integer,integer,integer):boolean
      }
      227:
      begin
       s[t-3].b:=_caminitpkk(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      function camdonepkk:boolean
      }
      228:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _camdonepkk;
      end;
      {
      function camtypepkk:integer
      }
      229:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _camtypepkk;
      end;
      {
      function camnaf(integer,integer,integer,integer):boolean
      }
      230:
      begin
       s[t-3].b:=_camnaf(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      function camwrite(integer,integer,integer,integer,integer):boolean
      }
      231:
      begin
       s[t-4].b:=_camwrite(s[t-4].i,s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-4;
      end;
      {
      function camwritelong(integer,integer,integer,integer,real):boolean
      }
      232:
      begin
       s[t-4].b:=_camwritelong(s[t-4].i,s[t-3].i,s[t-2].i,s[t-1].i,s[t].r);
       t:=t-4;
      end;
      {
      function camread(integer,integer,integer,integer):integer
      }
      233:
      begin
       s[t-3].i:=_camread(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      function camreadlong(integer,integer,integer,integer):real
      }
      234:
      begin
       s[t-3].r:=_camreadlong(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      function camdeclare(integer):boolean
      }
      235:
      s[t].b:=_camdeclare(s[t].i);
      {
      function camzero(integer):boolean
      }
      236:
      s[t].b:=_camzero(s[t].i);
      {
      function camclear(integer):boolean
      }
      237:
      s[t].b:=_camclear(s[t].i);
      {
      function camsetinhib(integer):boolean
      }
      238:
      s[t].b:=_camsetinhib(s[t].i);
      {
      function camclrinhib(integer):boolean
      }
      239:
      s[t].b:=_camclrinhib(s[t].i);
      {
      function camsetlmask(integer,integer):boolean
      }
      240:
      begin
       s[t-1].b:=_camsetlmask(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function camsetrmask(integer,integer):boolean
      }
      241:
      begin
       s[t-1].b:=_camsetrmask(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function camgetlmask(integer):integer
      }
      242:
      s[t].i:=_camgetlmask(s[t].i);
      {
      function camgetrword(integer):integer
      }
      243:
      s[t].i:=_camgetrword(s[t].i);
      {
      function camgetbcard(integer):integer
      }
      244:
      s[t].i:=_camgetbcard(s[t].i);
      {
      function camisx(integer):boolean
      }
      245:
      s[t].b:=_camisx(s[t].i);
      {
      function camisxq(integer):boolean
      }
      246:
      s[t].b:=_camisxq(s[t].i);
      {
      function camgetcsr(integer):integer
      }
      247:
      s[t].i:=_camgetcsr(s[t].i);
      {
      function camenabint(integer,integer):boolean
      }
      248:
      begin
       s[t-1].b:=_camenabint(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function camdisabint(integer,integer):boolean
      }
      249:
      begin
       s[t-1].b:=_camdisabint(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function camismyreq(integer):boolean
      }
      250:
      s[t].b:=_camismyreq(s[t].i);
      {
      function camgetrcode(integer):integer
      }
      251:
      s[t].i:=_camgetrcode(s[t].i);
      {
      function camclrreq(integer,integer):boolean
      }
      252:
      begin
       s[t-1].b:=_camclrreq(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function camsaveregs(integer):boolean
      }
      253:
      s[t].b:=_camsaveregs(s[t].i);
      {
      function camrestregs(integer):boolean
      }
      254:
      s[t].b:=_camrestregs(s[t].i);
      {
      function cammaskofst(integer):integer
      }
      255:
      s[t].i:=_cammaskofst(s[t].i);
      {
      function camstoflam(integer):integer
      }
      256:
      s[t].i:=_camstoflam(s[t].i);
      {
      function camgetrmask(c:integer):integer
      }
      257:
      s[t].i:=_camgetrmask(s[t].i);
      {
      function pkkclrreq(integer):boolean
      }
      258:
      s[t].b:=_pkkclrreq(s[t].i);
      {
      function pkkenabint(integer):boolean
      }
      259:
      s[t].b:=_pkkenabint(s[t].i);
      {
      function pkkdisabint(integer):boolean
      }
      260:
      s[t].b:=_pkkdisabint(s[t].i);
      {
      function findtag(string):integer
      }
      261,
      262,
      263:
      begin
       extractlong(s[t].i,lbuf1,yoper-261);
       s[t].i:=findtag(lbuf1);
       lbuf1:='';
      end;
      {
      function inittag(string,integer):integer
      }
      264,
      265,
      266:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-264);
       s[t-1].i:=inittag(lbuf1,s[t].i); lbuf1:='';
       t:=t-1;
      end;
      {
      function freetag(integer):boolean
      }
      267:
      s[t].b:=freetag(s[t].i);
      {
      function typetag(integer):integer
      }
      268:
      s[t].i:=typetag(s[t].i);
      {
      function nametag(integer):string
      }
      269:
      begin
       lbuf1:=nametag(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      function igettag(integer):integer
      }
      270:
      s[t].i:=igettag(s[t].i);
      {
      function rgettag(integer):real
      }
      271:
      s[t].r:=rgettag(s[t].i);
      {
      function sgettag(integer):string
      }
      272:
      begin
       lbuf1:=sgettag(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      function isettag(integer,integer):boolean
      }
      273:
      begin
       s[t-1].b:=isettag(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function rsettag(integer,real):boolean
      }
      274:
      begin
       s[t-1].b:=rsettag(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      function ssettag(integer,string):boolean
      }
      275,
      276,
      277:
      begin
       extractlong(s[t].i,lbuf1,yoper-275);
       s[t-1].b:=ssettag(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      function clickbutton:integer
      }
      278:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _clickbutton;
      end;
      {
      function clicksensor:string
      }
      279:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=_clicksensor;
       s[t].i := pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      function clicktag:integer
      }
      280:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _clicktag;
      end;
      {
      function crvfind(string):real
      }
      281,
      282,
      283:
      begin
       extractlong(s[t].i,lbuf1,yoper-281);
       s[t].r:=_crvfind(lbuf1);
       lbuf1:='';
      end;
      {
      function crvget(real,real):real
      }
      284:
      begin
       s[t-1].r:=_crvget(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      function crvwhere(real,real):real
      }
      285:
      begin
       s[t-1].r:=_crvwhere(s[t-1].r,s[t].r);
       t:=t-1;
      end;
      {
      function ms2year(real):integer
      }
      286:
      s[t].i:=_ms2year(s[t].r);
      {
      function ms2month(real):integer
      }
      287:
      s[t].i:=_ms2month(s[t].r);
      {
      function ms2day(real):integer
      }
      288:
      s[t].i:=_ms2day(s[t].r);
      {
      function ms2hour(real):integer
      }
      289:
      s[t].i:=_ms2hour(s[t].r);
      {
      function ms2min(real):integer
      }
      290:
      s[t].i:=_ms2min(s[t].r);
      {
      function ms2sec(real):integer
      }
      291:
      s[t].i:=_ms2sec(s[t].r);
      {
      fileerase(string):boolean
      }
      292,
      293,
      294:
      begin
       extractlong(s[t].i,lbuf1,yoper-292);
       s[t].b:=fileerase(lbuf1);
       lbuf1:='';
      end;
      {
      filerename(string):boolean
      }
      295,
      296,
      297:
      begin
       extractlong(s[t].i,lbuf1,yoper-295);
       s[t].b:=_filerename(lbuf1);
       lbuf1:='';
      end;
      {
      filecopy(string):boolean
      }
      298,
      299,
      300:
      begin
       extractlong(s[t].i,lbuf1,yoper-298);
       s[t].b:=_filecopy(lbuf1);
       lbuf1:='';
      end;
      {
      inot(integer):integer
      }
      301:
      s[t].i:=not s[t].i;
      {
      ior(integer,integer):integer
      }
      302:
      begin
       s[t-1].i:=s[t-1].i or s[t].i;
       t:=t-1;
      end;
      {
      ixor(integer,integer):integer
      }
      303:
      begin
       s[t-1].i:=s[t-1].i xor s[t].i;
       t:=t-1;
      end;
      {
      iand(integer,integer):integer
      }
      304:
      begin
       s[t-1].i:=s[t-1].i and s[t].i;
       t:=t-1;
      end;
      {
      ishift(integer,integer):integer
      }
      305:
      begin
       if s[t].i>=0
       then s[t-1].i:=s[t-1].i shl  s[t].i
       else s[t-1].i:=s[t-1].i shr -s[t].i;
       t:=t-1;
      end;
      {
      rshift(real,integer):real
      }
      306:
      begin
       s[t-1].r:=fscale(s[t-1].r,s[t].i);
       t:=t-1;
      end;
      {
      upcasestr(string):string
      }
      307,
      308,
      309:
      begin
       extractlong(s[t].i,lbuf1,yoper-307);
       lbuf1:=upcasestr(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      locasestr(string):string
      }
      310,
      311,
      312:
      begin
       extractlong(s[t].i,lbuf1,yoper-310);
       lbuf1:=locasestr(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      worddelims(string):string
      }
      313,
      314,
      315:
      begin
       extractlong(s[t].i,lbuf1,yoper-313);
       lbuf1:=_worddelims(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      wordcount(string):integer
      }
      316,
      317,
      318:
      begin
       extractlong(s[t].i,lbuf1,yoper-316);
       s[t].i:=wordcount(lbuf1,worddelims);
       lbuf1:='';
      end;
      {
      extractword(integer,string):string
      }
      319,
      320,
      321:
      begin
       extractlong(s[t].i,lbuf1,yoper-319);
       lbuf1:=extractword(s[t-1].i,lbuf1,worddelims);
       s[t-1].i:=pas2dynstrlong(lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      getfattr(string):integer
      }
      322,
      323,
      324:
      begin
       extractlong(s[t].i,lbuf1,yoper-322);
       s[t].i:=_getfattr(lbuf1);
       lbuf1:='';
      end;
      {
      setfattr(string,integer):integer
      }
      325,
      326,
      327:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-325);
       s[t-1].i:=_setfattr(lbuf1,s[t].i); lbuf1:='';
       t:=t-1;
      end;
      {
      doserror:integer
      }
      328:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _DosError;
      end;
      {
      daqdllinit(string):integer
      }
      329,
      330,
      331:
      begin
       extractlong(s[t].i,lbuf1,yoper-329);
       lbuf1:=Daq.FileRef(lbuf1,'.dll');
       s[t].i:=_daqdllinit(lbuf1);
       lbuf1:='';
      end;
      {
      daqdllfree(integer):boolean
      }
      332:
      s[t].b:=_daqdllfree(s[t].i);
      {
      daqdllcall(integer,integer):boolean
      }
      333:
      begin
       s[t-1].b:=_daqdllcall(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      paramstr(string):string
      }
      334,
      335,
      336:
      begin
       extractlong(s[t].i,lbuf1,yoper-334);
       lbuf1:=_paramstr(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      registererr(string):integer
      }
      337,
      338,
      339:
      begin
       extractlong(s[t].i,lbuf1,yoper-337);
       s[t].i:=Daq.RegisterErrMsg(lbuf1);
       lbuf1:='';
      end;
      {
      geterrcount(integer):real
      }
      340:
      case s[t].i of
       -2:        s[t].r:=FullDaqDeviceList.ErrorsTotal;
       -1:        s[t].r:=TheDevice.ErrorsTotal;
       0..255:    s[t].r:=TheDevice.ErrorsCount[s[t].i];
       256..511:  s[t].r:=FullDaqDeviceList.ErrorsCount[s[t].i-256];
       else       s[t].r:=0;
      end;
      {
      adam_get(string):string
      }
      341,
      342,
      343:
      begin
       extractlong(s[t].i,lbuf1,yoper-341);
       lbuf1:=_adam_get(Adam_Slot,lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adam_request(string,integer):boolean
      }
      344,
      345,
      346:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-344);
       s[t-1].b:=_adam_request(Adam_Slot,lbuf1,s[t].i); lbuf1:='';
       t:=t-1;
      end;
      {
      adam_status:integer
      }
      347:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _adam_status(Adam_Slot);
      end;
      {
      adam_reqtime:real
      }
      348:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _adam_reqtime(Adam_Slot);
      end;
      {
      hexb(integer):string
      }
      349:
      begin
       lbuf1:=hexb(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      hexw(integer):string
      }
      350:
      begin
       lbuf1:=hexw(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      hexl(real):string
      }
      351:
      begin
       lbuf1:=hexl(round(s[t].r));
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      strfix(real,integer,integer):string
      }
      352:
      begin
       str(s[t-2].r:s[t-1].i:s[t].i,sbuff);
       s[t-2].i:=pas2dynstr(sbuff);
       t:=t-2;
      end;
      {
      f_reset(string,integer):integer
      }
      353,
      354,
      355:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-353);
       s[t-1].i:=_f_reset(lbuf1,s[t].i); lbuf1:='';
       t:=t-1;
      end;
      {
      f_rewrite(string,integer):integer
      }
      356,
      357,
      358:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-356);
       s[t-1].i:=_f_rewrite(lbuf1,s[t].i); lbuf1:='';
       t:=t-1;
      end;
      {
      f_read(integer):string
      }
      359:
      begin
       lbuf1:=_f_read(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      f_write(string):integer
      }
      360,
      361,
      362:
      begin
       extractlong(s[t].i,lbuf1,yoper-360);
       s[t].i:=_f_write(lbuf1);
       lbuf1:='';
      end;
      {
      f_size:real
      }
      363:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _f_size;
      end;
      {
      f_seek(real):real
      }
      364:
      s[t].r:=_f_seek(round(s[t].r));
      {
      f_close:boolean
      }
      365:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _f_close;
      end;
      {
      dump(boolean):string
      dump(char):string
      dump(integer):string
      dump(real):string
      }
      366,
      367,
      368,
      369:
      begin
       case yoper of
        366 : sbuff:=char(s[t].b);
        367 : sbuff:=s[t].c;
        368 : begin
               sbuff[0]:=chr(sizeof(s[t].i));
               move(s[t].i,sbuff[1],sizeof(s[t].i));
              end;
        369 : begin
               sbuff[0]:=chr(sizeof(s[t].r));
               move(s[t].r,sbuff[1],sizeof(s[t].r));
              end;
        else sbuff:='';
       end;
       s[t].i:=pas2dynstr(sbuff);
      end;
      {
      dump2b(string):boolean
      }
      370,
      371,
      372:
      begin
       extract(s[t].i,sbuff,yoper-370);
       if sbuff='' then sbuff[1]:=#0;
       s[t].b:=boolean(sbuff[1]);
      end;
      {
      dump2c(string):char
      }
      373,
      374,
      375:
      begin
       extract(s[t].i,sbuff,yoper-373);
       if sbuff='' then sbuff[1]:=#0;
       s[t].c:=sbuff[1];
      end;
      {
      dump2i(string):integer
      }
      376,
      377,
      378:
      begin
       extract(s[t].i,sbuff,yoper-376);
       for i:=length(sbuff)+1 to sizeof(s[t].i) do sbuff[i]:=#0;
       move(sbuff[1],s[t].i,sizeof(s[t].i));
      end;
      {
      dump2r(string):real
      }
      379,
      380,
      381:
      begin
       extract(s[t].i,sbuff,yoper-379);
       for i:=length(sbuff)+1 to sizeof(s[t].r) do sbuff[i]:=#0;
       move(sbuff[1],s[t].r,sizeof(s[t].r));
      end;
      {
      mkdir(string):boolean
      }
      382,
      383,
      384:
      begin
       extractlong(s[t].i,lbuf1,yoper-382);
       s[t].b:=mkdir(lbuf1);
       lbuf1:='';
      end;
      {
      sleep(integer):boolean
      }
      385:
      s[t].b:=_sleep(s[t].i);
      {
      mksecnow:real
      }
      386:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _mksecnow;
      end;
      {
      shellexecute(string):integer
      }
      387,
      388,
      389:
      begin
       extractlong(s[t].i,lbuf1,yoper-387);
       s[t].i:=_shellexecute(lbuf1);
       lbuf1:='';
      end;
      {
      crlf:string
      }
      390:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=_crlf;
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      progname:string
      }
      391:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=_progname;
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      devname:string
      }
      392:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=_devname;
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      crvname(ref:real):string
      }
      393:
      begin
       lbuf1:=_crvname(s[t].r);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      crvlock(real):boolean
      }
      394:
      s[t].b:=_crvlock(s[t].r);
      {
      crvunlock(real):boolean
      }
      395:
      s[t].b:=_crvunlock(s[t].r);
      {
      crvins(real,real,real,real):real
      }
      396:
      begin
       s[t-3].r:=_crvins(s[t-3].r,s[t-2].r,s[t-1].r,s[t].r);
       t:=t-3;
      end;
      {
      crvdel(real,real,real):real
      }
      397:
      begin
       s[t-2].r:=_crvdel(s[t-2].r,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      inportw(integer):integer
      }
      398:
      s[t].i:=_inportw(s[t].i);
      {
      outportw(integer,integer):integer
      }
      399:
      begin
       s[t-1].i:=_outportw(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      inportl(integer):integer
      }
      400:
      s[t].i:=_inportl(s[t].i);
      {
      outportl(integer,integer):integer
      }
      401:
      begin
       s[t-1].i:=_outportl(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      edit(string):string
      }
      402,
      403,
      404:
      begin
       extractlong(s[t].i,lbuf1,yoper-402);
       lbuf1:=_edit(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      editstate:integer
      }
      405:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _editstate;
      end;
      {
      echo(string):boolean
      }
      406,
      407,
      408:
      begin
       extractlong(s[t].i,lbuf1,yoper-406);
       s[t].b:=_echo(lbuf1);
       lbuf1:='';
      end;
      {
      eval(string):real
      }
      409,
      410,
      411:
      begin
       extractlong(s[t].i,lbuf1,yoper-409);
       s[t].r:=_eval(lbuf1);
       lbuf1:='';
      end;
      {
      evar(string,real):boolean
      }
      412,
      413,
      414:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-412);
       s[t-1].b:=_evar(lbuf1,s[t].r); lbuf1:='';
       t:=t-1;
      end;
      {
      comspace:integer
      }
      415:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _comspace;
      end;
      {
      comclear:boolean
      }
      416:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _comclear;
      end;
      {
      add3d(real,real,real):boolean
      }
      417:
      begin
       s[t-2].b:=_add3d(s[t-2].r,s[t-1].r,s[t].r);
       t:=t-2;
      end;
      {
      plot3d(integer,integer,real,real,real,real,string):boolean
      }
      418,
      419,
      420:
      begin
       extractlong(s[t].i,lbuf1,yoper-418);
       s[t-6].b:=_plot3d(s[t-6].i,s[t-5].i,s[t-4].r,s[t-3].r,s[t-2].r,s[t-1].r,lbuf1); lbuf1:='';
       t:=t-6;
      end;
      {
      task_init(string):integer
      }
      421,
      422,
      423:
      begin
       extractlong(s[t].i,lbuf1,yoper-421);
       s[t].i:=_task_init(lbuf1);
       lbuf1:='';
      end;
      {
      task_free(integer):boolean
      }
      424:
      s[t].b:=_task_free(s[t].i);
      {
      task_ref(integer):integer
      }
      425:
      s[t].i:=_task_ref(s[t].i);
      {
      task_pid(integer):integer
      }
      426:
      s[t].i:=_task_pid(s[t].i);
      {
      task_run(integer):boolean
      }
      427:
      s[t].b:=_task_run(s[t].i);
      {
      task_wait(integer,integer):boolean
      }
      428:
      begin
       s[t-1].b:=_task_wait(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      task_send(integer,string):integer
      }
      429,
      430,
      431:
      begin
       extractlong(s[t].i,lbuf1,yoper-429);
       s[t-1].i:=_task_send(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      task_recv(integer,integer):string
      }
      432:
      begin
       lbuf1:=_task_recv(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      task_txcount(integer):integer
      }
      433:
      s[t].i:=_task_txcount(s[t].i);
      {
      task_rxcount(integer):integer
      }
      434:
      s[t].i:=_task_rxcount(s[t].i);
      {
      task_txspace(integer):integer
      }
      435:
      s[t].i:=_task_txspace(s[t].i);
      {
      task_rxspace(integer):integer
      }
      436:
      s[t].i:=_task_rxspace(s[t].i);
      {
      task_result(integer):integer
      }
      437:
      s[t].i:=_task_result(s[t].i);
      {
      task_kill(integer,integer,integer,integer):boolean
      }
      438:
      begin
       s[t-3].b:=_task_kill(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      task_ctrl(integer,string):string
      }
      439,
      440,
      441:
      begin
       extractlong(s[t].i,lbuf1,yoper-439);
       lbuf1:=_task_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      getcomspec:string
      }
      442:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=_getcomspec;
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      mime_encode(string):string
      }
      443,
      444,
      445:
      begin
       extractlong(s[t].i,lbuf1,yoper-443);
       lbuf1:=_mime_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      mime_decode(string):string
      }
      446,
      447,
      448:
      begin
       extractlong(s[t].i,lbuf1,yoper-446);
       lbuf1:=_mime_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      clickparams(string):string
      }
      449,
      450,
      451:
      begin
       extractlong(s[t].i,lbuf1,yoper-449);
       lbuf1:=_clickparams(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      str2shortcut(string):integer
      }
      452,
      453,
      454:
      begin
       extractlong(s[t].i,lbuf1,yoper-452);
       s[t].i:=_str2shortcut(lbuf1);
       lbuf1:='';
      end;
      {
      shortcut2str(integer):string
      }
      455:
      begin
       lbuf1:=_shortcut2str(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      crvgetln(real,integer):string
      }
      456:
      begin
       lbuf1:=_crvgetln(s[t-1].r,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      crvputln(real,integer,string):boolean
      }
      457,
      458,
      459:
      begin
       extractlong(s[t].i,lbuf1,yoper-457);
       s[t-2].b:=_crvputln(s[t-2].r,s[t-1].i,lbuf1); lbuf1:='';
       t:=t-2;
      end;
      {
      crvinsln(real,integer,string):boolean
      }
      460,
      461,
      462:
      begin
       extractlong(s[t].i,lbuf1,yoper-460);
       s[t-2].b:=_crvinsln(s[t-2].r,s[t-1].i,lbuf1); lbuf1:='';
       t:=t-2;
      end;
      {
      crvaddln(real,string):boolean
      }
      463,
      464,
      465:
      begin
       extractlong(s[t].i,lbuf1,yoper-463);
       s[t-1].b:=_crvaddln(s[t-1].r,lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      crvdelln(real,integer):boolean
      }
      466:
      begin
       s[t-1].b:=_crvdelln(s[t-1].r,s[t].i);
       t:=t-1;
      end;
      {
      crvnumln(real):integer
      }
      467:
      s[t].i:=_crvnumln(s[t].r);
      {
      awakeflag:boolean
      }
      468:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _awakeflag;
      end;
      {
      reffind(string):integer
      }
      469,
      470,
      471:
      begin
       extractlong(s[t].i,lbuf1,yoper-469);
       s[t].i:=_reffind(lbuf1);
       lbuf1:='';
      end;
      {
      refinfo(real,string):string
      }
      472,
      473,
      474:
      begin
       extractlong(s[t].i,lbuf1,yoper-472);
       lbuf1:=_refinfo(s[t-1].r,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      devsend(integer,string):real
      }
      475,
      476,
      477:
      begin
       extractlong(s[t].i,lbuf1,yoper-475);
       s[t-1].r:=_devsend(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      issametext(string,string):boolean
      }
      478,
      479,
      480,
      481,
      482,
      483,
      484,
      485,
      486:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-478) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-478) mod 3);
       s[t-1].b:=_issametext(lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      defaultextension(string,string):string
      }
      487,
      488,
      489,
      490,
      491,
      492,
      493,
      494,
      495:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-487) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-487) mod 3);
       lbuf1:=_defaultextension(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      forceextension(string,string):string
      }
      496,
      497,
      498,
      499,
      500,
      501,
      502,
      503,
      504:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-496) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-496) mod 3);
       lbuf1:=_forceextension(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      defaultpath(string,string):string
      }
      505,
      506,
      507,
      508,
      509,
      510,
      511,
      512,
      513:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-505) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-505) mod 3);
       lbuf1:=_defaultpath(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      forcepath(string,string):string
      }
      514,
      515,
      516,
      517,
      518,
      519,
      520,
      521,
      522:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-514) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-514) mod 3);
       lbuf1:=_forcepath(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      makerelativepath(string,string):string
      }
      523,
      524,
      525,
      526,
      527,
      528,
      529,
      530,
      531:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-523) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-523) mod 3);
       lbuf1:=_makerelativepath(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      trim(string):string
      }
      532,
      533,
      534:
      begin
       extractlong(s[t].i,lbuf1,yoper-532);
       lbuf1:=_trim(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      trimleft(string):string
      }
      535,
      536,
      537:
      begin
       extractlong(s[t].i,lbuf1,yoper-535);
       lbuf1:=_trimleft(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      trimright(string):string
      }
      538,
      539,
      540:
      begin
       extractlong(s[t].i,lbuf1,yoper-538);
       lbuf1:=_trimright(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      fexpand(string):string
      }
      541,
      542,
      543:
      begin
       extractlong(s[t].i,lbuf1,yoper-541);
       lbuf1:=_fexpand(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      addbackslash(string):string
      }
      544,
      545,
      546:
      begin
       extractlong(s[t].i,lbuf1,yoper-544);
       lbuf1:=_addbackslash(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      dropbackslash(string):string
      }
      547,
      548,
      549:
      begin
       extractlong(s[t].i,lbuf1,yoper-547);
       lbuf1:=_dropbackslash(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      extractfilepath(string):string
      }
      550,
      551,
      552:
      begin
       extractlong(s[t].i,lbuf1,yoper-550);
       lbuf1:=_extractfilepath(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      extractfilename(string):string
      }
      553,
      554,
      555:
      begin
       extractlong(s[t].i,lbuf1,yoper-553);
       lbuf1:=_extractfilename(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      extractfileext(string):string
      }
      556,
      557,
      558:
      begin
       extractlong(s[t].i,lbuf1,yoper-556);
       lbuf1:=_extractfileext(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      iswildcard(string):boolean
      }
      559,
      560,
      561:
      begin
       extractlong(s[t].i,lbuf1,yoper-559);
       s[t].b:=_iswildcard(lbuf1);
       lbuf1:='';
      end;
      {
      isrelativepath(string):boolean
      }
      562,
      563,
      564:
      begin
       extractlong(s[t].i,lbuf1,yoper-562);
       s[t].b:=_isrelativepath(lbuf1);
       lbuf1:='';
      end;
      {
      hasextension(string):boolean
      }
      565,
      566,
      567:
      begin
       extractlong(s[t].i,lbuf1,yoper-565);
       s[t].b:=_hasextension(lbuf1);
       lbuf1:='';
      end;
      {
      direxists(string):boolean
      }
      568,
      569,
      570:
      begin
       extractlong(s[t].i,lbuf1,yoper-568);
       s[t].b:=_direxists(lbuf1);
       lbuf1:='';
      end;
      {
      text_new:integer
      }
      571:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _text_new;
      end;
      {
      text_free(integer):boolean
      }
      572:
      s[t].b:=_text_free(s[t].i);
      {
      text_getln(integer,integer):string
      }
      573:
      begin
       lbuf1:=_text_getln(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      text_putln(integer,integer,string):boolean
      }
      574,
      575,
      576:
      begin
       extractlong(s[t].i,lbuf1,yoper-574);
       s[t-2].b:=_text_putln(s[t-2].i,s[t-1].i,lbuf1); lbuf1:='';
       t:=t-2;
      end;
      {
      text_insln(integer,integer,string):boolean
      }
      577,
      578,
      579:
      begin
       extractlong(s[t].i,lbuf1,yoper-577);
       s[t-2].b:=_text_insln(s[t-2].i,s[t-1].i,lbuf1); lbuf1:='';
       t:=t-2;
      end;
      {
      text_addln(integer,string):boolean
      }
      580,
      581,
      582:
      begin
       extractlong(s[t].i,lbuf1,yoper-580);
       s[t-1].b:=_text_addln(s[t-1].i,lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      text_delln(integer,integer):boolean
      }
      583:
      begin
       s[t-1].b:=_text_delln(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      text_numln(integer):integer
      }
      584:
      s[t].i:=_text_numln(s[t].i);
      {
      readinisection(integer,integer,string,string):integer
      }
      585,
      586,
      587,
      588,
      589,
      590,
      591,
      592,
      593:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-585) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-585) mod 3);
       s[t-3].i:=_readinisection(s[t-3].i,s[t-2].i,lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-3;
      end;
      {
      getenv(string):string
      }
      594,
      595,
      596:
      begin
       extractlong(s[t].i,lbuf1,yoper-594);
       lbuf1:=_getenv(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      setenv(string,string):boolean
      }
      597,
      598,
      599,
      600,
      601,
      602,
      603,
      604,
      605:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-597) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-597) mod 3);
       s[t-1].b:=_setenv(lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      vdpm_opcount:real
      }
      606:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := opcount+opcnt;
      end;
      {
      hex_encode(string):string
      }
      607,
      608,
      609:
      begin
       extractlong(s[t].i,lbuf1,yoper-607);
       lbuf1:=_hex_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      hex_decode(string):string
      }
      610,
      611,
      612:
      begin
       extractlong(s[t].i,lbuf1,yoper-610);
       lbuf1:=_hex_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      crypt_ctrl(string):string
      }
      613,
      614,
      615:
      begin
       extractlong(s[t].i,lbuf1,yoper-613);
       lbuf1:=_crypt_ctrl(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      crypt_encode(string,string):string
      }
      616,
      617,
      618,
      619,
      620,
      621,
      622,
      623,
      624:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-616) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-616) mod 3);
       lbuf1:=_crypt_encode(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      crypt_decode(string,string):string
      }
      625,
      626,
      627,
      628,
      629,
      630,
      631,
      632,
      633:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-625) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-625) mod 3);
       lbuf1:=_crypt_decode(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      getmd5fromstr(string):string
      }
      634,
      635,
      636:
      begin
       extractlong(s[t].i,lbuf1,yoper-634);
       lbuf1:=_getmd5fromstr(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      getmd5fromfile(string):string
      }
      637,
      638,
      639:
      begin
       extractlong(s[t].i,lbuf1,yoper-637);
       lbuf1:=_getmd5fromfile(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      getmd5fromtext(integer):string
      }
      640:
      begin
       lbuf1:=_getmd5fromtext(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      wdt_reset(boolean):real
      }
      641:
      s[t].r := _wdt_reset(s[t].b);
      {
      guard_check(string):integer
      }
      642,
      643,
      644:
      begin
       extractlong(s[t].i,lbuf1,yoper-642);
       s[t].i:=_guard_check(lbuf1);
       lbuf1:='';
      end;
      {
      dirlist(integer,integer,string,string):integer
      }
      645,
      646,
      647,
      648,
      649,
      650,
      651,
      652,
      653:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-645) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-645) mod 3);
       s[t-3].i:=_dirlist(s[t-3].i,s[t-2].i,lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-3;
      end;
      {
      pidlist(integer):integer
      }
      654:
      s[t].i := _pidlist(s[t].i);
      {
      pidkill(integer,integer,integer):integer
      }
      655:
      begin
       s[t-2].i:=_pidkill(s[t-2].i,s[t-1].i,s[t].i);
       t:=t-2;
      end;
      {
      enumerate(integer,string):integer
      }
      656,
      657,
      658:
      begin
       extractlong(s[t].i,lbuf1,yoper-656);
       s[t-1].i:=_enumerate(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      timebase:real
      }
      659:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _timebase;
      end;
      {
      daqfileref(string,string):string
      }
      660,
      661,
      662,
      663,
      664,
      665,
      666,
      667,
      668:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-660) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-660) mod 3);
       lbuf1:=_daqfileref(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      getclockres(integer):real
      }
      669:
      s[t].r := _GetClockRes(s[t].i);
      {
      setclockres(real):real
      }
      670:
      s[t].r := _SetClockRes(s[t].r);
      {
      url_encode(string):string
      }
      671,
      672,
      673:
      begin
       extractlong(s[t].i,lbuf1,yoper-671);
       lbuf1:=_url_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      url_decode(string):string
      }
      674,
      675,
      676:
      begin
       extractlong(s[t].i,lbuf1,yoper-674);
       lbuf1:=_url_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      datetime2ms(integer,integer,integer,integer,integer,integer,integer):real
      }
      677:
      begin
       s[t-6].r:=_datetime2ms(s[t-6].i,s[t-5].i,s[t-4].i,s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-6;
      end;
      {
      stackavail:integer
      }
      678:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := stacksize-t;
      end;
      {
      cpu_count:integer
      }
      679:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _cpu_count;
      end;
      {
      cpu_start:boolean
      }
      680:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].b := _cpu_start;
      end;
      {
      cpu_clock:real
      }
      681:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _cpu_clock;
      end;
      {
      cpu_mhz:real
      }
      682:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].r := _cpu_mhz;
      end;
      {
      pidaffinity(integer,integer):integer
      }
      683:
      begin
       s[t-1].i:=_pidaffinity(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      devaffinity(integer,integer):integer
      }
      684:
      begin
       s[t-1].i:=_devaffinity(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      pipe_init(string):integer
      }
      685,
      686,
      687:
      begin
       extractlong(s[t].i,lbuf1,yoper-685);
       s[t].i:=_pipe_init(lbuf1);
       lbuf1:='';
      end;
      {
      pipe_free(integer):boolean
      }
      688:
      s[t].b:=_pipe_free(s[t].i);
      {
      pipe_ref(integer):integer
      }
      689:
      s[t].i:=_pipe_ref(s[t].i);
      {
      pipe_pid(integer):integer
      }
      690:
      s[t].i:=_pipe_pid(s[t].i);
      {
      pipe_run(integer):boolean
      }
      691:
      s[t].b:=_pipe_run(s[t].i);
      {
      pipe_wait(integer,integer):boolean
      }
      692:
      begin
       s[t-1].b:=_pipe_wait(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      pipe_send(integer,string):integer
      }
      693,
      694,
      695:
      begin
       extractlong(s[t].i,lbuf1,yoper-693);
       s[t-1].i:=_pipe_send(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      pipe_recv(integer,integer):string
      }
      696:
      begin
       lbuf1:=_pipe_recv(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      pipe_txcount(integer):integer
      }
      697:
      s[t].i:=_pipe_txcount(s[t].i);
      {
      pipe_rxcount(integer):integer
      }
      698:
      s[t].i:=_pipe_rxcount(s[t].i);
      {
      pipe_txspace(integer):integer
      }
      699:
      s[t].i:=_pipe_txspace(s[t].i);
      {
      pipe_rxspace(integer):integer
      }
      700:
      s[t].i:=_pipe_rxspace(s[t].i);
      {
      pipe_result(integer):integer
      }
      701:
      s[t].i:=_pipe_result(s[t].i);
      {
      pipe_kill(integer,integer,integer,integer):boolean
      }
      702:
      begin
       s[t-3].b:=_pipe_kill(s[t-3].i,s[t-2].i,s[t-1].i,s[t].i);
       t:=t-3;
      end;
      {
      pipe_ctrl(integer,string):string
      }
      703,
      704,
      705:
      begin
       extractlong(s[t].i,lbuf1,yoper-703);
       lbuf1:=_pipe_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1); lbuf1:='';
       t:=t-1;
      end;
      {
      pipe_txclear(integer):boolean
      }
      706:
      s[t].b:=_pipe_txclear(s[t].i);
      {
      pipe_rxclear(integer):boolean
      }
      707:
      s[t].b:=_pipe_rxclear(s[t].i);
      {
      pipe_count(integer):integer
      }
      708:
      s[t].i:=_pipe_count(s[t].i);
      {
      pipe_stream(integer,integer):integer
      }
      709:
      begin
       s[t-1].i:=_pipe_stream(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      pipe_connected(integer):integer
      }
      710:
      s[t].i:=_pipe_connected(s[t].i);
      {
      strconv(string,string):string
      }
      711,
      712,
      713,
      714,
      715,
      716,
      717,
      718,
      719:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-711) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-711) mod 3);
       lbuf1:=_strconv(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      dump2f(string):real
      }
      720,
      721,
      722:
      begin
       extractlong(s[t].i,lbuf1,yoper-720);
       s[t].r:=_dump2f(lbuf1);
       lbuf1:='';
      end;
      {
      dumpf(real):string
      }
      723:
      begin
       lbuf1:=_dumpf(s[t].r);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      url_packed(string):string
      }
      724,
      725,
      726:
      begin
       extractlong(s[t].i,lbuf1,yoper-724);
       lbuf1:=_url_packed(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      base32_encode(string):string
      }
      727,
      728,
      729:
      begin
       extractlong(s[t].i,lbuf1,yoper-727);
       lbuf1:=_base32_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      base32_decode(string):string
      }
      730,
      731,
      732:
      begin
       extractlong(s[t].i,lbuf1,yoper-730);
       lbuf1:=_base32_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      base32_alphabet(string):string
      }
      733,
      734,
      735:
      begin
       extractlong(s[t].i,lbuf1,yoper-733);
       lbuf1:=_base32_alphabet(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      hashindexof(string,integer,integer):integer
      }
      736,
      737,
      738:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-736);
       s[t-2].i:=_hashindexof(lbuf1,s[t-1].i,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      getpid:integer
      }
      739:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _getpid;
      end;
      {
      hashlist_init(integer):integer
      }
      740:
      s[t].i:=_hashlist_init(s[t].i);
      {
      hashlist_free(integer):boolean
      }
      741:
      s[t].b:=_hashlist_free(s[t].i);
      {
      hashlist_count(integer):integer
      }
      742:
      s[t].i:=_hashlist_count(s[t].i);
      {
      hashlist_getkey(integer,integer):string
      }
      743:
      begin
       lbuf1:=_hashlist_getkey(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_delete(integer,string):boolean
      }
      744,
      745,
      746:
      begin
       extractlong(s[t].i,lbuf1,yoper-744);
       s[t-1].b:=_hashlist_delete(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_indexof(integer,string):integer
      }
      747,
      748,
      749:
      begin
       extractlong(s[t].i,lbuf1,yoper-747);
       s[t-1].i:=_hashlist_indexof(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_getdata(integer,string):real
      }
      750,
      751,
      752:
      begin
       extractlong(s[t].i,lbuf1,yoper-750);
       s[t-1].r:=_hashlist_getdata(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_setdata(integer,string,real):boolean
      }
      753,
      754,
      755:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-753);
       s[t-2].b:=_hashlist_setdata(s[t-2].i,lbuf1,s[t].r);
       lbuf1:='';
       t:=t-2;
      end;
      {
      hashlist_getlink(integer,string):integer
      }
      756,
      757,
      758:
      begin
       extractlong(s[t].i,lbuf1,yoper-756);
       s[t-1].i:=_hashlist_getlink(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_setlink(integer,string,integer):boolean
      }
      759,
      760,
      761:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-759);
       s[t-2].b:=_hashlist_setlink(s[t-2].i,lbuf1,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      hashlist_getpara(integer,string):string
      }
      762,
      763,
      764:
      begin
       extractlong(s[t].i,lbuf1,yoper-762);
       lbuf1:=_hashlist_getpara(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      hashlist_setpara(integer,string,string):boolean
      }
      765,
      766,
      767,
      768,
      769,
      770,
      771,
      772,
      773:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-765) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-765) mod 3);
       s[t-2].b:=_hashlist_setpara(s[t-2].i,lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-2;
      end;
      {
      devpost(integer,string):real
      }
      774,
      775,
      776:
      begin
       extractlong(s[t].i,lbuf1,yoper-774);
       s[t-1].r:=_devpost(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      devpostmsg(string):real
      }
      777,
      778,
      779:
      begin
       extractlong(s[t].i,lbuf1,yoper-777);
       s[t].r:=_devpostmsg(lbuf1);
       lbuf1:='';
      end;
      {
      clickwrite(string):integer
      }
      780,
      781,
      782:
      begin
       extractlong(s[t].i,lbuf1,yoper-780);
       s[t].i:=_clickwrite(lbuf1);
      end;
      {
      function clickread:integer
      }
      783:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _clickread;
      end;
      {
      function clickwhat:integer
      }
      784:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _clickwhat;
      end;
      {
      function clickwrote:integer
      }
      785:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _clickwrote;
      end;
      {
      clickfilter(integer):integer
      }
      786:
      s[t].i:=_clickfilter(s[t].i);
      {
      clickawaker(integer):integer
      }
      787:
      s[t].i:=_clickawaker(s[t].i);
      {
      strfetch(string,integer):char
      }
      788,
      789,
      790:
      begin
       s[t-1].i:=strfetch(s[t-1].i,s[t].i,yoper-788);
       t:=t-1;
      end;
      {
      cookiescan(string,string,integer):string
      }
      791,
      792,
      793,
      794,
      795,
      796,
      797,
      798,
      799:
      begin
       extractlong(s[t-1].i, lbuf2, (yoper-791) div 3);
////// extractlong(s[t-2].i, lbuf1, (yoper-791) mod 3); <-- replacement to avoid big data copy:
       h1:=s[t-2].i;            h3:=(yoper-791) mod 3;
       if h3 in [0,1] then begin // string or temporary string argument
        memsh1:=memstrng(h1);    // get string record pointer
        h2:=memsh1.head.leng;    // use buffer/length to avoid data copy
        lbuf2:=CookieScan(memsh1.data.buff,h2,lbuf2,s[t].i);
        if h3=1 then sfree(h1);  // temporary string
       end else
       if h3=2 then begin
        lbuf1:=chr(h1);          // char argument
        lbuf2:=CookieScan(lbuf1,lbuf2,s[t].i);
       end else RunTimeError(ps_SystemCheck,'cookiescan');
       s[t-2].i:=pas2dynstrlong(lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-2;
      end;
      {
      taglock(boolean):integer
      }
      800:
      s[t].i := _taglock(s[t].b);
      {
      iatomictagop(integer,char,integer):boolean
      }
      801:
      begin
       s[t-2].b:=iAtomicTagOp(s[t-2].i,s[t-1].c,s[t].i);
       t:=t-2;
      end;
      {
      ratomictagop(integer,char,real):boolean
      }
      802:
      begin
       s[t-2].b:=rAtomicTagOp(s[t-2].i,s[t-1].c,s[t].r);
       t:=t-2;
      end;
      {
      stringofchar(char,integer):string
      }
      803:
      begin
       h2:=max(s[t].i,0);
       memsh1:=sget(h1,h2);
       if h2>0 then fillchar(memsh1.data.buff,h2,s[t-1].c);
       s[t-1].i:=h1;
       t:=t-1;
      end;
      {
      igetdump(var string,integer):integer
      }
      804:
      begin
       h1:=s[t-1].i; memsh1:=memstrng(h1);
       h2:=memsh1.head.leng;
       h3:=s[t].i;
       if (h3>=0) and (h3<=h2-sizeof(h4)) then h4:=Integer(Pointer(@memsh1.data.bytes[h3])^) else h4:=0;
       s[t-1].i:=h4;
       t:=t-1;
      end;
      {
      isetdump(var string,integer,integer):boolean
      }
      805:
      begin
       h1:=s[t-2].i; memsh1:=memstrng(h1);
       h2:=memsh1.head.leng;
       h3:=s[t-1].i;
       h4:=s[t].i;
       h5:=1;
       if (h3>=0) and (h3<=h2-sizeof(h4)) then Integer(Pointer(@memsh1.data.bytes[h3])^):=h4 else h5:=0;
       s[t-2].b:=(h5>0);
       t:=t-2;
      end;
      {
      rgetdump(var string,integer):real
      }
      806:
      begin
       h1:=s[t-1].i; memsh1:=memstrng(h1);
       h2:=memsh1.head.leng;
       h3:=s[t].i;
       if (h3>=0) and (h3<=h2-sizeof(r1)) then r1:=Real(Pointer(@memsh1.data.bytes[h3])^) else r1:=0;
       s[t-1].r:=r1;
       t:=t-1;
      end;
      {
      rsetdump(var string,integer,real):boolean
      }
      807:
      begin
       h1:=s[t-2].i; memsh1:=memstrng(h1);
       h2:=memsh1.head.leng;
       h3:=s[t-1].i;
       r1:=s[t].r;
       h5:=1;
       if (h3>=0) and (h3<=h2-sizeof(r1)) then Real(Pointer(@memsh1.data.bytes[h3])^):=r1 else h5:=0;
       s[t-2].b:=(h5>0);
       t:=t-2;
      end;
      {
      getai_par(integer,integer):real
      }
      808:
      begin
       s[t-1].r:=_getai_par(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      getao_par(integer,integer):real
      }
      809:
      begin
       s[t-1].r:=_getao_par(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      getdi_par(integer,integer):real
      }
      810:
      begin
       s[t-1].r:=_getdi_par(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      getdo_par(integer,integer):real
      }
      811:
      begin
       s[t-1].r:=_getdo_par(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      strfmt(string,variant):string
      }
      812,
      813,
      814,
      815,
      816,
      817,
      818,
      819,
      820,
      821:
      begin
       h1:=(yoper-812) mod 2; // string/temporary format
       h2:=(yoper-812) div 2; // string/temporary/char/int/real data
       extractlong(s[t-1].i, lbuf1, h1); // get format as string/temporary
       case h2 of                        // get data   as string/temporary/char/int/real
        0,1 : begin
               extractlong(s[t].i,  lbuf2, h2);
               lbuf2:=_strfmt(lbuf1,[lbuf2]);
              end;
        2   : lbuf2:=_strfmt(lbuf1,[s[t].c]);
        3   : lbuf2:=_strfmt(lbuf1,[s[t].i]);
        4   : lbuf2:=_strfmt(lbuf1,[s[t].r]);
        else RunTimeError(ps_SystemCheck,'strfmt');
       end;
       s[t-1].i:=pas2dynstrlong(lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      utf8_encode_ansi(string):string
      }
      822,
      823,
      824:
      begin
       extractlong(s[t].i,lbuf1,yoper-822);
       lbuf1:=utf8_encode_ansi(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_decode_ansi(string):string
      }
      825,
      826,
      827:
      begin
       extractlong(s[t].i,lbuf1,yoper-825);
       lbuf1:=utf8_decode_ansi(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_uppercase(string):string
      }
      828,
      829,
      830:
      begin
       extractlong(s[t].i,lbuf1,yoper-828);
       lbuf1:=utf8_uppercase(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_lowercase(string):string
      }
      831,
      832,
      833:
      begin
       extractlong(s[t].i,lbuf1,yoper-831);
       lbuf1:=utf8_lowercase(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_length(string):integer
      }
      834,
      835,
      836:
      begin
       extractlong(s[t].i,lbuf1,yoper-834);
       s[t].i:=utf8_length(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_copy(string,integer,integer):string
      }
      837,
      838,
      839:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-837);
       lbuf1:=utf8_copy(lbuf1,s[t-1].i,s[t].i);
       s[t-2].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      utf8_ord(string,integer):integer
      }
      840,
      841,
      842:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-840);
       s[t-1].i:=utf8_ord(lbuf1,s[t].i);
       lbuf1:='';
       t:=t-1;
      end;
      {
      utf8_chr(integer):string
      }
      843:
      begin
       lbuf1:=utf8_chr(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      utf8_bom:string
      }
      844:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       lbuf1:=utf8_bom;
       s[t].i := pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adjustlinebreaks(string):string
      }
      845,
      846,
      847:
      begin
       extractlong(s[t].i,lbuf1,yoper-845);
       lbuf1:=SysUtils.AdjustLineBreaks(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      eol:string
      lineending:string
      slinebreak:string
      }
      848:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       sbuff:=LineEnding;
       s[t].i := pas2dynstr(sbuff);
      end;
      {
      directoryseparator:char
      pathdelim:char
      }
      849:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].c := DirectorySeparator;
      end;
      {
      pathseparator:char
      pathsep:char
      }
      850:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].c := PathSeparator;
      end;
      {
      easyipc_init(string,string):integer
      }
      851,
      852,
      853,
      854,
      855,
      856,
      857,
      858,
      859:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-851) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-851) mod 3);
       s[t-1].i:=_easyipc_init(lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      easyipc_free(integer):boolean
      }
      860:
      s[t].b:=_easyipc_free(s[t].i);
      {
      easyipc_poll(integer):boolean
      }
      861:
      s[t].b:=_easyipc_poll(s[t].i);
      {
      easyipc_send(integer,string):boolean
      }
      862,
      863,
      864:
      begin
       extractlong(s[t].i,lbuf1,yoper-862);
       s[t-1].b:=_easyipc_send(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      easyipc_recv(integer,integer):string
      }
      865:
      begin
       lbuf1:=_easyipc_recv(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      easyipc_ctrl(integer,string):string
      }
      866,
      867,
      868:
      begin
       extractlong(s[t].i,lbuf1,yoper-866);
       lbuf1:=_easyipc_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      expenv(string):string
      }
      869,
      870,
      871:
      begin
       extractlong(s[t].i,lbuf1,yoper-869);
       lbuf1:=_expenv(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      createtempfile(string):string
      }
      872,
      873,
      874:
      begin
       extractlong(s[t].i,lbuf1,yoper-872);
       lbuf1:=_createtempfile(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      stringreplace(string,string,string,integer):string
      }
      875,
      876,
      877,
      878,
      879,
      880,
      881,
      882,
      883,
      884,
      885,
      886,
      887,
      888,
      889,
      890,
      891,
      892,
      893,
      894,
      895,
      896,
      897,
      898,
      899,
      900,
      901:
      begin
       h4:=s[t].i;
       extractlong(s[t-1].i, lbuf3,  (yoper-875) div 9);
       extractlong(s[t-2].i, lbuf2, ((yoper-875) mod 9) div 3);
       extractlong(s[t-3].i, lbuf1, ((yoper-875) mod 9) mod 3);
       lbuf1:=_stringreplace(lbuf1,lbuf2,lbuf3,h4);
       s[t-3].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       lbuf3:='';
       t:=t-3;
      end;
      {
      ansiquotedstr(string,char):string
      }
      902,
      903,
      904:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-902);
       lbuf1:=_ansiquotedstr(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      ansidequotedstr(string,char):string
      }
      905,
      906,
      907:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-905);
       lbuf1:=_ansidequotedstr(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      ansiskipquotedstr(string,char):string
      }
      908,
      909,
      910:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-908);
       lbuf1:=_ansiskipquotedstr(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      posex(string,string,integer):integer
      }
      911,
      912,
      913,
      914,
      915,
      916,
      917,
      918,
      919:
      begin
       h2:=s[t-1].i; h4:=(yoper-911) div 3; h6:=1; pch2:=PChar(@s[t-1].c);
       h1:=s[t-2].i; h3:=(yoper-911) mod 3; h5:=1; pch1:=PChar(@s[t-2].c);
       case h3 of
        0,1: begin memsh1:=memstrng(h1); pch1:=memsh1.data.buff; h5:=memsh1.head.leng; end;
        2  : ; // already assigned above
        else RunTimeError(ps_SystemCheck,'posex');
       end;
       case h4 of
        0,1: begin memsh2:=memstrng(h2); pch2:=memsh2.data.buff; h6:=memsh2.head.leng; end;
        2  : ; // already assigned above
        else RunTimeError(ps_SystemCheck,'posex');
       end;
       s[t-2].i:=posex(pch1,h5,pch2,h6,s[t].i-1)+1;
       if h3=1 then sfree(h1); // temporary string
       if h4=1 then sfree(h2); // temporary string
       t:=t-2;
      end;
      {
      text_tostring(integer):string
      }
      920:
      begin
       lbuf1:=_text_tostring(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      imax(integer,integer):integer
      }
      921:
      begin
       s[t-1].i:=Max(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      imin(integer,integer):integer
      }
      922:
      begin
       s[t-1].i:=Min(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      hasflags(integer,integer):boolean
      }
      923:
      begin
       s[t-1].b:=HasFlags(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function gettagcolor(integer):integer
      }
      924:
      s[t].i:=gettagcolor(s[t].i);
      {
      function settagcolor(integer,integer):boolean
      }
      925:
      begin
       s[t-1].b:=settagcolor(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      function gettagparam(integer):real
      }
      926:
      s[t].r:=gettagparam(s[t].i);
      {
      function settagparam(integer,real):boolean
      }
      927:
      begin
       s[t-1].b:=settagparam(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      function gettagtimer(integer):real
      }
      928:
      s[t].r:=gettagtimer(s[t].i);
      {
      function settagtimer(integer,real):boolean
      }
      929:
      begin
       s[t-1].b:=settagtimer(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      extractfirstparam(string,char):string
      }
      930,
      931,
      932:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-930);
       lbuf1:=_extractfirstparam(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      skipfirstparam(string,char):string
      }
      933,
      934,
      935:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-933);
       lbuf1:=_skipfirstparam(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      isoption(string,string):boolean
      }
      936,
      937,
      938,
      939,
      940,
      941,
      942,
      943,
      944:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-936) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-936) mod 3);
       s[t-1].b:=_isoption(lbuf1,lbuf2);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      getoptionvalue(string):string
      }
      945,
      946,
      947:
      begin
       extractlong(s[t].i,lbuf1,yoper-945);
       lbuf1:=_getoptionvalue(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      ansiquotedifneed(string,char):string
      }
      948,
      949,
      950:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-948);
       lbuf1:=_ansiquotedifneed(lbuf1,s[t].c);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      islexeme(string,integer):boolean
      }
      951,
      952,
      953:
      begin
       h1:=s[t-1].i; h2:=s[t].i; h3:=yoper-951; pch1:=PChar(@s[t-1].c); h4:=1;
       case h3 of
        0,1 : begin memsh1:=memstrng(h1); pch1:=memsh1.data.buff; h4:=memsh1.head.leng; end;
        2  : ; // already assigned above
        else RunTimeError(ps_SystemCheck,'islexeme');
       end;
       s[t-1].b:=IsLexeme(pch1,h4,h2);
       if h3=1 then sfree(h1); // temporary string
       t:=t-1;
      end;
      {
      regexp_init(integer,string):integer
      }
      954,
      955,
      956:
      begin
       extractlong(s[t].i,lbuf1,yoper-954);
       s[t-1].i:=_regexp_init(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      regexp_free(integer):boolean
      }
      957:
      s[t].b:=_regexp_free(s[t].i);
      {
      regexp_ref(integer):integer
      }
      958:
      s[t].i:=_regexp_ref(s[t].i);
      {
      regexp_ctrl(integer,string):string
      }
      959,
      960,
      961:
      begin
       extractlong(s[t].i,lbuf1,yoper-959);
       lbuf1:=_regexp_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      regexp_test(integer,string):boolean
      }
      962,
      963,
      964:
      begin
       extractlong(s[t].i,lbuf1,yoper-962);
       s[t-1].b:=_regexp_test(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      regexp_exec(integer,string):integer
      }
      965,
      966,
      967:
      begin
       extractlong(s[t].i,lbuf1,yoper-965);
       s[t-1].i:=_regexp_exec(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      regexp_replace(integer,string,string):string
      }
      968,
      969,
      970,
      971,
      972,
      973,
      974,
      975,
      976:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-968) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-968) mod 3);
       lbuf1:=_regexp_replace(s[t-2].i,lbuf1,lbuf2);
       s[t-2].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-2;
      end;
      {
      regexp_matchnum(integer,integer):integer
      }
      977:
      begin
       s[t-1].i:=_regexp_matchnum(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      regexp_matchpos(integer,integer,integer):integer
      }
      978:
      begin
       s[t-2].i:=_regexp_matchpos(s[t-2].i,s[t-1].i,s[t].i);
       t:=t-2;
      end;
      {
      regexp_matchlen(integer,integer,integer):integer
      }
      979:
      begin
       s[t-2].i:=_regexp_matchlen(s[t-2].i,s[t-1].i,s[t].i);
       t:=t-2;
      end;
      {
      regexp_matchstr(integer,integer,integer):string
      }
      980:
      begin
       lbuf1:=_regexp_matchstr(s[t-2].i,s[t-1].i,s[t].i);
       s[t-2].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      regexp_escape(string):string
      }
      981,
      982,
      983:
      begin
       extractlong(s[t].i,lbuf1,yoper-981);
       lbuf1:=_regexp_escape(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      backslash_encode(string):string
      }
      984,
      985,
      986:
      begin
       extractlong(s[t].i,lbuf1,yoper-984);
       lbuf1:=_backslash_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      backslash_decode(string):string
      }
      987,
      988,
      989:
      begin
       extractlong(s[t].i,lbuf1,yoper-987);
       lbuf1:=_backslash_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      backslash_encoder_ctrl(string):string
      }
      990,
      991,
      992:
      begin
       extractlong(s[t].i,lbuf1,yoper-990);
       lbuf1:=_backslash_encoder_ctrl(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      fsm_new:integer
      }
      993:
      begin
       t := t+1;
       if t > stacksize then RunTimeError(ps_StackOverflow);
       s[t].i := _fsm_new;
      end;
      {
      fsm_free(integer):boolean
      }
      994:
      s[t].b:=_fsm_free(s[t].i);
      {
      fsm_ref(integer):integer
      }
      995:
      s[t].i:=_fsm_ref(s[t].i);
      {
      fsm_root(integer):integer
      }
      996:
      s[t].i:=_fsm_root(s[t].i);
      {
      fsm_type(integer):integer
      }
      997:
      s[t].i:=_fsm_type(s[t].i);
      {
      fsm_parent(integer):integer
      }
      998:
      s[t].i:=_fsm_parent(s[t].i);
      {
      fsm_name(integer):string
      }
      999:
      begin
       lbuf1:=_fsm_name(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      fsm_path(integer):string
      }
      1000:
      begin
       lbuf1:=_fsm_path(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      fsm_ctrl(integer,string):string
      }
      1001,
      1002,
      1003:
      begin
       extractlong(s[t].i,lbuf1,yoper-1001);
       lbuf1:=_fsm_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      fsm_count(integer,integer):integer
      }
      1004:
      begin
       s[t-1].i:=_fsm_count(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      fsm_items(integer,integer,integer):integer
      }
      1005:
      begin
       s[t-2].i:=_fsm_items(s[t-2].i,s[t-1].i,s[t].i);
       t:=t-2;
      end;
      {
      fsm_get_iparam(integer):integer
      }
      1006:
      s[t].i:=_fsm_get_iparam(s[t].i);
      {
      fsm_set_iparam(integer,integer):boolean
      }
      1007:
      begin
       s[t-1].b:=_fsm_set_iparam(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      fsm_get_fparam(integer):real
      }
      1008:
      s[t].r:=_fsm_get_fparam(s[t].i);
      {
      fsm_set_fparam(integer,real):boolean
      }
      1009:
      begin
       s[t-1].b:=_fsm_set_fparam(s[t-1].i,s[t].r);
       t:=t-1;
      end;
      {
      fsm_get_sparam(integer):string
      }
      1010:
      begin
       lbuf1:=_fsm_get_sparam(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      fsm_set_sparam(integer,string):boolean
      }
      1011,
      1012,
      1013:
      begin
       extractlong(s[t].i,lbuf1,yoper-1011);
       s[t-1].b:=_fsm_set_sparam(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      fsm_add(integer,integer,string):integer
      }
      1014,
      1015,
      1016:
      begin
       extractlong(s[t].i,lbuf1,yoper-1014);
       s[t-2].i:=_fsm_add(s[t-2].i,s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      fsm_find(integer,integer,string):integer
      }
      1017,
      1018,
      1019:
      begin
       extractlong(s[t].i,lbuf1,yoper-1017);
       s[t-2].i:=_fsm_find(s[t-2].i,s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      fsm_get_state(integer):integer
      }
      1020:
      s[t].i:=_fsm_get_state(s[t].i);
      {
      fsm_set_state(integer,string):integer
      }
      1021,
      1022,
      1023:
      begin
       extractlong(s[t].i,lbuf1,yoper-1021);
       s[t-1].i:=_fsm_set_state(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      fsm_set_state(integer,integer):integer
      }
      1024:
      begin
       s[t-1].i:=_fsm_set_state(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      fsm_link(integer,string):integer
      }
      1025,
      1026,
      1027:
      begin
       extractlong(s[t].i,lbuf1,yoper-1025);
       s[t-1].i:=_fsm_link(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      fsm_modified(integer,integer):integer
      }
      1028:
      begin
       s[t-1].i:=_fsm_modified(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      fsm_name_rule(integer):integer
      }
      1029:
      s[t].i:=_fsm_name_rule(s[t].i);
      {
      upcase(char):char
      }
      1030:
      s[t].c:=_upcase(s[t].c);
      {
      locase(char):char
      }
      1031:
      s[t].c:=_locase(s[t].c);
      {
      htonl(integer):integer
      }
      1032:
      s[t].i:=_htonl(s[t].i);
      {
      ntohl(integer):integer
      }
      1033:
      s[t].i:=_ntohl(s[t].i);
      {
      htons(integer):integer
      }
      1034:
      s[t].i:=_htons(s[t].i);
      {
      ntohs(integer):integer
      }
      1035:
      s[t].i:=_ntohs(s[t].i);
      {
      db_create(string):boolean
      }
      1036,
      1037,
      1038:
      begin
       extractlong(s[t].i,lbuf1,yoper-1036);
       s[t].b:=_db_create(lbuf1);
       lbuf1:='';
      end;
      {
      db_connection(integer,string):integer
      }
      1039,
      1040,
      1041:
      begin
       extractlong(s[t].i,lbuf1,yoper-1039);
       s[t-1].i:=_db_connection(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_recordset(integer,string):integer
      }
      1042,
      1043,
      1044:
      begin
       extractlong(s[t].i,lbuf1,yoper-1042);
       s[t-1].i:=_db_recordset(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_command(integer,string):integer
      }
      1045,
      1046,
      1047:
      begin
       extractlong(s[t].i,lbuf1,yoper-1045);
       s[t-1].i:=_db_command(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_free(integer):boolean
      }
      1048:
      s[t].b:=_db_free(s[t].i);
      {
      db_ref(integer):integer
      }
      1049:
      s[t].i:=_db_ref(s[t].i);
      {
      db_root(integer):integer
      }
      1050:
      s[t].i:=_db_root(s[t].i);
      {
      db_type(integer):integer
      }
      1051:
      s[t].i:=_db_type(s[t].i);
      {
      db_parent(integer):integer
      }
      1052:
      s[t].i:=_db_parent(s[t].i);
      {
      db_state(integer):integer
      }
      1053:
      s[t].i:=_db_state(s[t].i);
      {
      db_close(integer):boolean
      }
      1054:
      s[t].b:=_db_close(s[t].i);
      {
      db_open(integer,integer):boolean
      }
      1055:
      begin
       s[t-1].b:=_db_open(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      db_ctrl(integer,string):string
      }
      1056,
      1057,
      1058:
      begin
       extractlong(s[t].i,lbuf1,yoper-1056);
       lbuf1:=_db_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_bugscount(integer):integer
      }
      1059:
      s[t].i:=_db_bugscount(s[t].i);
      {
      db_bugsclear(integer):integer
      }
      1060:
      s[t].i:=_db_bugsclear(s[t].i);
      {
      db_errors(integer):string
      }
      1061:
      begin
       lbuf1:=_db_errors(s[t].i);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      db_errorscount(integer):integer
      }
      1062:
      s[t].i:=_db_errorscount(s[t].i);
      {
      db_errorsclear(integer):integer
      }
      1063:
      s[t].i:=_db_errorsclear(s[t].i);
      {
      db_execute(integer,string,integer):integer
      }
      1064,
      1065,
      1066:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-1064);
       s[t-2].i:=_db_execute(s[t-2].i,lbuf1,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      db_cancel(integer):boolean
      }
      1067:
      s[t].b:=_db_cancel(s[t].i);
      {
      db_update(integer):boolean
      }
      1068:
      s[t].b:=_db_update(s[t].i);
      {
      db_(integer):boolean
      }
      1069:
      s[t].b:=_db_cancelupdate(s[t].i);
      {
      db_begintrans(integer):integer
      }
      1070:
      s[t].i:=_db_begintrans(s[t].i);
      {
      db_(integer):boolean
      }
      1071:
      s[t].b:=_db_committrans(s[t].i);
      {
      db_rollbacktrans(integer):boolean
      }
      1072:
      s[t].b:=_db_rollbacktrans(s[t].i);
      {
      db_bof(integer):boolean
      }
      1073:
      s[t].b:=_db_bof(s[t].i);
      {
      db_eof(integer):boolean
      }
      1074:
      s[t].b:=_db_eof(s[t].i);
      {
      db_movefirst(integer):boolean
      }
      1075:
      s[t].b:=_db_movefirst(s[t].i);
      {
      db_movelast(integer):boolean
      }
      1076:
      s[t].b:=_db_movelast(s[t].i);
      {
      db_movenext(integer):boolean
      }
      1077:
      s[t].b:=_db_movenext(s[t].i);
      {
      db_moveprevious(integer):boolean
      }
      1078:
      s[t].b:=_db_moveprevious(s[t].i);
      {
      db_fieldscount(integer):integer
      }
      1079:
      s[t].i:=_db_fieldscount(s[t].i);
      {
      db_fieldsnames(integer,integer):string
      }
      1080:
      begin
       lbuf1:=_db_fieldsnames(s[t-1].i,s[t].i);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_fieldstypes(integer,string):integer
      }
      1081,
      1082,
      1083:
      begin
       extractlong(s[t].i,lbuf1,yoper-1081);
       s[t-1].i:=_db_fieldstypes(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_fieldsasint(integer,string,char,integer):integer
      }
      1084,
      1085,
      1086:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-1084);
       s[t-3].i:=_db_fieldsasint(s[t-3].i,lbuf1,s[t-1].c,s[t].i);
       lbuf1:='';
       t:=t-3;
      end;
      {
      db_fieldsasfloat(integer,string,char,real):integer
      }
      1087,
      1088,
      1089:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-1087);
       s[t-3].r:=_db_fieldsasfloat(s[t-3].i,lbuf1,s[t-1].c,s[t].r);
       lbuf1:='';
       t:=t-3;
      end;
      {
      db_fieldsasstring(integer,string,char,string):string
      }
      1090,
      1091,
      1092,
      1093,
      1094,
      1095,
      1096,
      1097,
      1098:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-1090) div 3);
       extractlong(s[t-2].i, lbuf1, (yoper-1090) mod 3);
       lbuf1:=_db_fieldsasstring(s[t-3].i,lbuf1,s[t-1].c,lbuf2);
       s[t-3].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-3;
      end;
      {
      db_addnew(integer,string):boolean
      }
      1099,
      1100,
      1101:
      begin
       extractlong(s[t].i,lbuf1,yoper-1099);
       s[t-1].b:=_db_addnew(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      db_delete(integer,integer):boolean
      }
      1102:
      begin
       s[t-1].b:=_db_delete(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      db_requery(integer,integer):boolean
      }
      1103:
      begin
       s[t-1].b:=_db_requery(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      db_resync(integer,integer,integer):boolean
      }
      1104:
      begin
       s[t-2].b:=_db_resync(s[t-2].i,s[t-1].i,s[t].i);
       t:=t-2;
      end;
      {
      db_supports(integer,integer):boolean
      }
      1105:
      begin
       s[t-1].b:=_db_supports(s[t-1].i,s[t].i);
       t:=t-1;
      end;
      {
      db_save(integer,string,integer):boolean
      }
      1106,
      1107,
      1108:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-1106);
       s[t-2].b:=_db_save(s[t-2].i,lbuf1,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      poseol(string,integer,integer):integer
      }
      1109,
      1110,
      1111:
      begin
       h1:=s[t-2].i; h2:=s[t-1].i; h3:=s[t].i;
       h4:=(yoper-1109); h5:=0;
       case h4 of
        0,1: begin { argument is string or temporary string }
              memsh1:=memstrng(h1);
              pch1:=memsh1.data.buff;
              h5:=poseol(pch1,memsh1.head.leng,h2,h3);
              if (h4=1) then sfree(h1); { argument is temporary string }
             end;
        2:   begin { argument is char }
              pch1:=PChar(@h1);
              h5:=poseol(pch1,1,h2,h3);
             end;
        else RunTimeError(ps_SystemCheck,'poseol');
       end;
       s[t-2].i:=h5;
       t:=t-2;
      end;
      {
      strtimefmt(string,real):string
      }
      1112,
      1113,
      1114:
      begin
       extractlong(s[t-1].i,lbuf1,yoper-1112);
       lbuf1:=_strtimefmt(lbuf1,s[t].r);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      shm_init(string,integer,integer):integer
      }
      1115,
      1116,
      1117:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-1115);
       s[t-2].i:=_shm_init(lbuf1,s[t-1].i,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      shm_ref(integer):integer
      }
      1118:
      s[t].i:=_shm_ref(s[t].i);
      {
      shm_free(integer):boolean
      }
      1119:
      s[t].b:=_shm_free(s[t].i);
      {
      shm_delink(string):boolean
      }
      1120,
      1121,
      1122:
      begin
       extractlong(s[t].i,lbuf1,yoper-1120);
       s[t].b:=_shm_delink(lbuf1);
       lbuf1:='';
      end;
      {
      shm_iop(integer,integer,char,integer):integer
      }
      1123:
      begin
       s[t-3].i:=_shm_iop(s[t-3].i,s[t-2].i,s[t-1].c,s[t].i);
       t:=t-3;
      end;
      {
      shm_rop(integer,integer,char,real):real
      }
      1124:
      begin
       s[t-3].r:=_shm_rop(s[t-3].i,s[t-2].i,s[t-1].c,s[t].r);
       t:=t-3;
      end;
      {
      shm_fop(integer,integer,char,real):real
      }
      1125:
      begin
       s[t-3].r:=_shm_fop(s[t-3].i,s[t-2].i,s[t-1].c,s[t].r);
       t:=t-3;
      end;
      {
      shm_sop(integer,integer,char,string):string
      }
      1126,
      1127,
      1128:
      begin
       extractlong(s[t].i,lbuf1,yoper-1126);
       lbuf1:=_shm_sop(s[t-3].i,s[t-2].i,s[t-1].c,lbuf1);
       s[t-3].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-3;
      end;
      {
      shm_ctrl(integer,string):string
      }
      1129,
      1130,
      1131:
      begin
       extractlong(s[t].i,lbuf1,yoper-1129);
       lbuf1:=_shm_ctrl(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      shm_ioresult(integer):integer
      }
      1132:
      s[t].i:=_shm_ioresult(s[t].i);
      {
      strtointbase(string,integer,integer):integer
      }
      1133,
      1134,
      1135:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-1133);
       s[t-2].i:=_strtointbase(lbuf1,s[t-1].i,s[t].i);
       lbuf1:='';
       t:=t-2;
      end;
      {
      inttostrbase(integer,integer,integer):string
      }
      1136:
      begin
       lbuf1:=_inttostrbase(s[t-2].i,s[t-1].i,s[t].i);
       s[t-2].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      adaptfilename(string):string
      }
      1137,
      1138,
      1139:
      begin
       extractlong(s[t].i,lbuf1,yoper-1137);
       lbuf1:=_adaptfilename(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adaptexefilename(string):string
      }
      1140,
      1141,
      1142:
      begin
       extractlong(s[t].i,lbuf1,yoper-1140);
       lbuf1:=_adaptexefilename(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adaptfilenamemode(integer):integer
      }
      1143:
      s[t].i:=_adaptfilenamemode(s[t].i);
      {
      skipwords(integer,string):string
      }
      1144,
      1145,
      1146:
      begin
       extractlong(s[t].i,lbuf1,yoper-1144);
       lbuf1:=_skipwords(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      phrasecount(string):integer
      }
      1147,
      1148,
      1149:
      begin
       extractlong(s[t].i,lbuf1,yoper-1147);
       s[t].i:=_phrasecount(lbuf1);
       lbuf1:='';
      end;
      {
      extractphrase(integer,string):string
      }
      1150,
      1151,
      1152:
      begin
       extractlong(s[t].i,lbuf1,yoper-1150);
       lbuf1:=_extractphrase(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      skipphrases(integer,string):string
      }
      1153,
      1154,
      1155:
      begin
       extractlong(s[t].i,lbuf1,yoper-1153);
       lbuf1:=_skipphrases(s[t-1].i,lbuf1);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      phraselisttotextlines(string):string
      }
      1156,
      1157,
      1158:
      begin
       extractlong(s[t].i,lbuf1,yoper-1156);
       lbuf1:=_phraselisttotextlines(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      phrasequotes(string):string
      }
      1159,
      1160,
      1161:
      begin
       extractlong(s[t].i,lbuf1,yoper-1159);
       lbuf1:=_phrasequotes(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adaptdllfilename(string):string
      }
      1162,
      1163,
      1164:
      begin
       extractlong(s[t].i,lbuf1,yoper-1162);
       lbuf1:=_adaptdllfilename(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      adaptlnkfilename(string):string
      }
      1165,
      1166,
      1167:
      begin
       extractlong(s[t].i,lbuf1,yoper-1165);
       lbuf1:=_adaptlnkfilename(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      db_engineid(integer):integer
      }
      1168:
      s[t].i:=_db_engineid(s[t].i);
      {
      db_active(integer):boolean
      }
      1169:
      s[t].b:=_db_active(s[t].i);
      {
      pct_encode(string):string
      }
      1170,
      1171,
      1172:
      begin
       extractlong(s[t].i,lbuf1,yoper-1170);
       lbuf1:=_pct_encode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      pct_decode(string):string
      }
      1173,
      1174,
      1175:
      begin
       extractlong(s[t].i,lbuf1,yoper-1173);
       lbuf1:=_pct_decode(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      pct_encoder_ctrl(string):string
      }
      1176,
      1177,
      1178:
      begin
       extractlong(s[t].i,lbuf1,yoper-1176);
       lbuf1:=_pct_encoder_ctrl(lbuf1);
       s[t].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
      end;
      {
      readfiletobuffer(string,integer,integer):string
      }
      1179,
      1180,
      1181:
      begin
       extractlong(s[t-2].i,lbuf1,yoper-1179);
       lbuf1:=_ReadFileToBuffer(lbuf1,s[t-1].i,s[t].i);
       s[t-2].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       t:=t-2;
      end;
      {
      writebuffertofile(string,string,integer):integer
      }
      1182,
      1183,
      1184,
      1185,
      1186,
      1187,
      1188,
      1189,
      1190:
      begin
       extractlong(s[t-1].i, lbuf2, (yoper-1182) div 3);
       extractlong(s[t-2].i, lbuf1, (yoper-1182) mod 3);
       s[t-2].i:=_writebuffertofile(lbuf1,lbuf2,s[t].i);
       lbuf1:='';
       lbuf2:='';
       t:=t-2;
      end;
      {
      getfileproperties(string,string):string
      }
      1191,
      1192,
      1193,
      1194,
      1195,
      1196,
      1197,
      1198,
      1199:
      begin
       extractlong(s[t].i,   lbuf2, (yoper-1191) div 3);
       extractlong(s[t-1].i, lbuf1, (yoper-1191) mod 3);
       lbuf1:=_getfileproperties(lbuf1,lbuf2);
       s[t-1].i:=pas2dynstrlong(lbuf1);
       lbuf1:='';
       lbuf2:='';
       t:=t-1;
      end;
      {
      syslognotable(integer):boolean
      }
      1200:
      s[t].b := _SysLogNotable(s[t].i);
      {
      syslognote(integer,string):integer
      }
      1201,
      1202,
      1203:
      begin
       extractlong(s[t].i,lbuf1,yoper-1201);
       s[t-1].i:=_SysLogNote(s[t-1].i,lbuf1);
       lbuf1:='';
       t:=t-1;
      end;
      {
      wdt_timeout(integer):integer
      }
      1204:
      s[t].i:=_wdt_timeout(s[t].i);
      {
      not used yoper codes:
      }
      else
      RunTimeError(ps_SystemCheck,'bad_yoper');
     end; { StandardFunctions case yoper **************************************}
     {
     Add y to the element on top of the stack
     }
     9:
     s[t].i := s[t].i + yoper;
     {
     Jump to y (unconditional)
     }
     10:
     prgcnt := yoper;
     {
     Jump to y if stack top false
     }
     11:
     begin
      if not s[t].b then prgcnt := yoper;
      t := t-1;
     end;
     {
     Jump to y (case table) and select entry
     }
     12:
     begin
      h1 := s[t].i;
      t := t-1;
      h2 := yoper;
      h3 := 0;
      repeat
       if ctab^[h2].f = 10 then begin
        if ctab^[h2].y <> h2+1 then RunTimeError(ps_SystemCheck,'case');
        prgcnt := ctab^[h2].y; // ending case table when no one selected
        break;
       end;
       if ctab^[h2].f <> 13 then RunTimeError(ps_CaseCheck);
       if ctab^[h2].y = h1 then begin
        h3 := 1;
        prgcnt := ctab^[h2+1].y
       end else h2 := h2 + 2
      until h3 <> 0;
     end;
     {
     Entry in case table ... NOT EXECUTABLE
     }
     13:
     RunTimeError(ps_SystemCheck,'opcode_13');
     {
     For loop entry test - UP
     }
     14:
     begin
      h1 := s[t-1].i;
      if h1 <= s[t].i then s[s[t-2].i].i := h1 else begin
       t := t-3;
       prgcnt := yoper;
      end;
     end;
     {
     For loop retry test - UP
     }
     15:
     begin
      h2 := s[t-2].i;
      h1 := s[h2].i +1;
      if h1 <= s[t].i then begin
       s[h2].i := h1;
       prgcnt := yoper;
      end else t := t-3;
     end;
     {
     For loop entry test - DOWN
     }
     16:
     begin
      h1 := s[t-1].i;
      if h1 >= s[t].i then s[s[t-2].i].i := h1 else begin
       prgcnt := yoper;
       t := t-3;
      end;
     end;
     {
     For loop retry test - DOWN
     }
     17:
     begin
      h2 := s[t-2].i;
      h1 := s[h2].i - 1;
      if h1 >= s[t].i then begin
       s[h2].i := h1;
       prgcnt := yoper;
      end else t := t-3;
     end;
     {
     Mark stack
     }
     18:
     begin
      h1 := btab[itab[yoper].ref].vsize;
      if t+h1 > stacksize then RunTimeError(ps_StackOverflow);
      t := t+6;
      b0 := t;
      s[b0].i := 0;
      s[b0].p := 0;
      s[t-2].i := h1-1;
      s[t-1].i := yoper;
      {add b0 to stack of string tables}
      if b0_count>=b0_capacity then begin
       Reallocate(Pointer(b0_stack),
                  Max(8 * sizeof(b0_stack[0]),
                      2 * AllocSize(b0_stack)));
       b0_capacity:=AllocSize(b0_stack) div sizeof(b0_stack[0]);
       if b0_count>=b0_capacity then RunTimeError(ps_StringManager);
      end;
      b0_stack[b0_count]:=b0;
      Inc(b0_count);
     end;
     {
     Call user procedure
     }
     19:
     begin
      h1 := t - yoper;            { h1 points to base }
      h2 := s[h1+4].i;            { h2 points to tab }
      h3 := itab[h2].lev;
      display[h3+1] := h1;
      h4 := s[h1+3].i + h1;
      s[h1+0].i := 0;              {clear future function result}
      s[h1+0].p := 0;              {clear future string link}
      s[h1+1].i := prgcnt;
      s[h1+2].i := display[h3];
      s[h1+3].i := b;
      fillchar(s[t+1],(h4-t)*sizeof(s[1]),0);
      b := h1;
      t := h4;
      prgcnt := itab[h2].adr;
      if not _isISRnow then
      if stackdump and DebugMode then dump;
     end;
     {
     Indexed fetch (element size > 1)
     }
     20:
     begin
      h1 := yoper;      { h1 points to atab }
      h2 := atab[h1].low;
      h3 := s[t].i;
      if h3 < h2 then RunTimeError(ps_IndexCheck);
      if h3 > atab[h1].high then RunTimeError(ps_IndexCheck);
      t := t-1;
      s[t].i := s[t].i + (h3-h2);
     end;
     {
     Indexed fetch
     }
     21:
     begin
      h1 := yoper;      { h1 points to atab }
      h2 := atab[h1].low;
      h3 := s[t].i;
      if h3 < h2 then RunTimeError(ps_IndexCheck);
      if h3 > atab[h1].high then RunTimeError(ps_IndexCheck);
      t := t-1;
      s[t].i := s[t].i + (h3-h2)*atab[h1].elsize;
     end;
     {
     Load block
     }
     22:
     begin
      h1 := s[t].i;
      t := t-1;
      h2 := yoper + t;
      if h2 > stacksize then RunTimeError(ps_StackOverflow);
      while t < h2 do begin
       t := t+1;
       if s[h1].cn = strngs then begin
        s[t].s := 0;
        scopy(t,h1);
       end
       else s[t] := s[h1];
       h1 := h1+1
      end;
     end;
     {
     Copy block
     }
     23:
     begin
      h1 := s[t-1].i;
      h2 := s[t].i;
      h3 := h1 + yoper;
      while h1 < h3 do begin
       if s[h2].cn = strngs then begin
        s[h1].s := 0;
        scopy(h1,h2);
       end
       else s[h1] := s[h2];
       h1 := h1+1;
       h2 := h2+1
      end;
      t := t-2;
     end;
     {
     Load literal
     }
     24:
     begin
      t := t+1;
      if t > stacksize then RunTimeError(ps_StackOverflow);
      s[t].i := yoper;
     end;
     {
     Load real
     }
     25:
     begin
      t := t+1;
      if t > stacksize then RunTimeError(ps_StackOverflow);
      s[t].r := rtab[yoper];
     end;
     {
     Float
     }
     26:
     begin
      h1 := t - yoper;
      s[h1].r := s[h1].i;
     end;
     {
     Read (y denotes type .. 1 integer, 2 real, 4 char)
     }
     27:
     begin
      case yoper of
       1: if IORes=0 then begin
           SetInOutRes(0);
           read(StdInp,s[s[t].i].i);
           IORes:=IOResult;
          end;
       2: if IORes=0 then begin
           SetInOutRes(0);
           read(StdInp,s[s[t].i].r);
           IORes:=IOResult;
          end;
       4: if IORes=0 then begin
           SetInOutRes(0);
           read(StdInp,s[s[t].i].c);
           IORes:=IOResult;
          end;
       5: begin
           lbuf1:='';
           if IORes=0 then begin
            SetInOutRes(0);
            read(StdInp,lbuf1);
            IORes:=IOResult;
           end;
           h1 := length(lbuf1);
           if h1>slenmax then RunTimeError(ps_StringManager);
           if h1=0
           then h3 := nul
           else begin
            memsh3:=sget(h3,h1);
            move(lbuf1[1],memsh3.data.buff,h1);
           end;
           h4 := s[t].i;
           h5 := s[h4].i;
           if h5 = 0 then link(h4) else sfree(h5);
           s[h4].i := h3;
           lbuf1:='';
          end;
      end;
      t := t-1;
     end;
     {
     Write string...not executable
     }
     28:
     RunTimeError(ps_SystemCheck,'opcode_28');
     {
     Write - default field widths
     }
     29:
     begin
      case yoper of
       1: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,s[t].i:fldi);
           IORes:=IOResult;
          end;
       2: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,s[t].r:fldr);
           IORes:=IOResult;
          end;
       3: if IORes=0 then begin
           SetInOutRes(0);
           if s[t].b then write(StdOut,'true':fldb)
                     else write(StdOut,'false':fldb);
           IORes:=IOResult;
          end;
       4: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,chr(s[t].i):fldc);
           IORes:=IOResult;
          end;
      end;
      t := t-1;
     end;
     {
     Write - defined field widths
     }
     30:
     begin
      case yoper of
       1: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,s[t-1].i:max(s[t].i,1));
           IORes:=IOResult;
          end;
       2: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,s[t-1].r:max(s[t].i,1));
           IORes:=IOResult;
          end;
       3: if IORes=0 then begin
           SetInOutRes(0);
           if s[t-1].b then write(StdOut,'true':max(s[t].i,1))
                       else write(StdOut,'false':max(s[t].i,1));
           IORes:=IOResult;
          end;
       4: if IORes=0 then begin
           SetInOutRes(0);
           write(StdOut,chr(s[t-1].i):max(s[t].i,1));
           IORes:=IOResult;
          end;
      end ;
      t := t-2;
     end;
     {
     Assign string (of length 1) to char
     }
     31:
     begin
      h1 := s[t].i; memsh1:=memstrng(h1);
      if memsh1.head.leng <> 1 then RunTimeError(ps_StringManager);
      s[s[t-1].i].i := memsh1.data.bytes[0];
      if (yoper and 8) = 8 then sfree(h1);
      t := t-2;
     end;
     {
     Relation operators for strings
     }
     32:
     begin
      h2 := s[t-1].i;
      h3 := s[t].i;
      case yoper and 3 of
       1: begin  {strngs~chars}
           memsh2:=memstrng(h2);
           h4 := memsh2.head.leng;
           if h4=0 then h5 := 64
           else if h3>memsh2.data.bytes[0] then h5 := 64
           else if h3<memsh2.data.bytes[0] then h5 := 32
           else if h4=1 then h5 := 16
           else h5 := 32;
          end;
       2: begin  {chars~strngs}
           memsh3:=memstrng(h3);
           h4 := memsh3.head.leng;
           if h4=0 then h5 := 32
           else if h2>memsh3.data.bytes[0] then h5 := 32
           else if h2<memsh3.data.bytes[0] then h5 := 64
           else if h4=1 then h5 := 16
           else h5 := 64;
          end;
       3: begin  {strngs~strngs}
           memsh2:=memstrng(h2);
           memsh3:=memstrng(h3);
           h4 := memsh2.head.leng;
           h1 :=0;
           h5 := memsh3.head.leng;
           if h5<h4 then h4 := h5 else h5 := h4;
           while h1<h4 do begin
            if memsh2.data.bytes[h1] <> memsh3.data.bytes[h1]
            then h4 := h1
            else h1 := h1+1;
           end;
           if h4=h5
           then if memsh2.head.leng=memsh3.head.leng then h5 := 16
           else if memsh2.head.leng<memsh3.head.leng then h5 := 64 else h5 := 32
           else if memsh2.data.bytes[h1]<memsh3.data.bytes[h1] then h5 := 64
           else h5 := 32;
          end;
      end;
      if (yoper and 5) = 5 then sfree(h2);
      if (yoper and 10) = 10 then sfree(h3);
      t := t-1;
      s[t].b := (yoper and h5) > 0;
     end;
     {
     not used
     }
     33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,
     58,59,60,61,62,63,64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,
     83,84,85,86,87,88,89,90,91,92,93,94,95,96,97,98,99,100,101,102,103,104,105,
     106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,
     125,126,127,128,129,130:
     RunTimeError(ps_SystemCheck,'opcode_unused');
     {
     Halt
     }
     131:
     begin
      prgsts := ps_Finalized;
      break;
     end;
     {
     Exit procedure
     }
     132:
     begin
      h1 := s[b+5].i;
      while h1 <> 0 do begin
       sfree(s[h1].i);
       h1 := s[h1].p;
      end;
      t := b-1;
      prgcnt := s[b+1].i;
      b := s[b+3].i;
      {Release stack of temporary strings}
      if b0_count>0 then begin
       Dec(b0_count);
       b0_stack[b0_count]:=0;
      end else RunTimeError(ps_StringManager);
     end;
     {
     Exit function
     }
     133:
     begin
      h1 := s[b+5].i;
      while h1 <> 0 do begin
       if h1<>b then sfree(s[h1].i); {do not free string function result!}
       h1 := s[h1].p;
      end;
      t := b;
      prgcnt := s[b+1].i;
      b := s[b+3].i;
      {Release stack of temporary strings}
      if b0_count>0 then begin
       Dec(b0_count);
       b0_stack[b0_count]:=0;
      end else RunTimeError(ps_StringManager);
     end;
     {
     Fetch
     }
     134:
     s[t] := s[s[t].i];
     {
     Not
     }
     135:
     s[t].b := not s[t].b;
     {
     - integer (unary)
     }
     136:
     s[t].i := - s[t].i;
     {
     Write real - defined field
     }
     137:
     begin
      if IORes=0 then begin
       SetInOutRes(0);
       write(StdOut,s[t-2].r:max(s[t-1].i,1):max(s[t].i,0));
       IORes:=IOResult;
      end;
      t := t-3;
     end;
     {
     Store
     }
     138:
     begin
      s[s[t-1].i] := s[t];
      t := t-2;
     end;
     {
     Real =
     }
     139:
     begin
      t := t-1;
      s[t].b := s[t].r = s[t+1].r;
     end;
     {
     Real <>
     }
     140:
     begin
      t := t-1;
      s[t].b := s[t].r <> s[t+1].r;
     end;
     {
     Real <
     }
     141:
     begin
      t := t-1;
      s[t].b := s[t].r < s[t+1].r;
     end;
     {
     Real <=
     }
     142:
     begin
      t := t-1;
      s[t].b := s[t].r <= s[t+1].r;
     end;
     {
     Real >
     }
     143:
     begin
      t := t-1;
      s[t].b := s[t].r > s[t+1].r;
     end;
     {
     Real >=
     }
     144:
     begin
      t := t-1;
      s[t].b := s[t].r >= s[t+1].r;
     end;
     {
     Integer =
     }
     145:
     begin
      t := t-1;
      s[t].b := s[t].i = s[t+1].i;
     end;
     {
     Integer <>
     }
     146:
     begin
      t := t-1;
      s[t].b := s[t].i <> s[t+1].i;
     end;
     {
     Integer <
     }
     147:
     begin
      t := t-1;
      s[t].b := s[t].i < s[t+1].i;
     end;
     {
     Integer <=
     }
     148:
     begin
      t := t-1;
      s[t].b := s[t].i <= s[t+1].i;
     end;
     {
     Integer >
     }
     149:
     begin
      t := t-1;
      s[t].b := s[t].i > s[t+1].i;
     end;
     {
     Integer >=
     }
     150:
     begin
      t := t-1;
      s[t].b := s[t].i >= s[t+1].i;
     end;
     {
     Or
     }
     151:
     begin
      t := t-1;
      s[t].b := s[t].b or s[t+1].b;
     end;
     {
     + integer
     }
     152:
     begin
      t := t-1;
      s[t].i := s[t].i + s[t+1].i;
     end;
     {
     - integer
     }
     153:
     begin
      t := t-1;
      s[t].i := s[t].i - s[t+1].i;
     end;
     {
     + real
     }
     154:
     begin
      t := t-1;
      s[t].r := s[t].r + s[t+1].r;
     end;
     {
     - real
     }
     155:
     begin
      t := t-1;
      s[t].r := s[t].r - s[t+1].r;
     end;
     {
     And
     }
     156:
     begin
      t := t-1;
      s[t].b := s[t].b and s[t+1].b;
     end;
     {
     * integer
     }
     157:
     begin
      t := t-1;
      s[t].i := s[t].i * s[t+1].i;
     end;
     {
     Div
     }
     158:
     begin
      t := t-1;
      if s[t+1].i = 0 then RunTimeError(ps_DivByZero);
      s[t].i := s[t].i div s[t+1].i;
     end;
     {
     Mod
     }
     159:
     begin
      t := t-1;
      if s[t+1].i = 0 then RunTimeError(ps_DivByZero);
      s[t].i := s[t].i mod s[t+1].i;
     end;
     {
     * real
     }
     160:
     begin
      t := t-1;
      s[t].r := s[t].r * s[t+1].r;
     end;
     {
     /
     }
     161:
     begin
      t := t-1;
      s[t].r := s[t].r / s[t+1].r;
     end;
     {
     Readln
     }
     162:
     if IORes=0 then begin
      SetInOutRes(0);
      readln(StdInp);
      IORes:=IOResult;
     end;
     {
     Writeln
     }
     163:
     if IORes=0 then begin
      SetInOutRes(0);
      writeln(StdOut);
      IORes:=IOResult;
     end;
     {
     - real (unary)
     }
     164:
     s[t].r := - s[t].r;
     {
     Index string
     }
     165:
     begin
      h1 := s[t-1].i; memsh1:=memstrng(h1);
      h2 := s[t].i;
      if (h2 <= 0) or (h2 > memsh1.head.leng) then RunTimeError(ps_IndexCheck);
      t := t-1;
      s[t].i := memsh1.data.bytes[h2-1];
     end;
     {
     Assign string temporary to string
     }
     166:
     begin
      h2 := s[t-1].i;
      h1 := s[h2].i;
      if h1=0 then link(h2) else sfree(h1);
      s[h2].i := s[t].i;
      t := t-2;
     end;
     {
     Convert array to string
     }
     167:
     begin
      h1 := s[t].i;
      memsh3:=sget(h3,yoper);
      for h4 := 0 to yoper-1 do memsh3.data.bytes[h4] := ord(s[h1+h4].c);
      s[t].i := h3;
     end;
     {
     Assign character to string
     }
     168:
     begin
      h2 := s[s[t-1].i].i; if h2=0 then memsh2:=nil else memsh2:=memstrng(h2);
      if memsh2=nil then h4:=0 else h4:=memsh2.head.dyna;
      if (h2=0) or (h4 > strngalign-sizeof(strnghdr)) then begin
       memsh3:=sget(h3,1);
       if h2=0 then link(s[t-1].i) else sfree(h2);
       h2 := h3; memsh2:=memsh3;
       s[s[t-1].i].i := h2;
      end;
      memsh2.data.bytes[0] := s[t].i;
      memsh2.head.leng := 1;
      t := t-2;
     end;
     {
     Assign string to string
     }
     169:
     begin
      scopy(s[t-1].i, t);
      t := t-2;
     end;
     {
     Write string
     Write string temporary
     }
     170,
     171:
     begin
      h3 := s[t].i; memsh3:=memstrng(h3);
      t := t-1;
      h2 := memsh3.head.leng;
      if h2>0 then begin
       SetString(lbuf1,memsh3.data.buff,h2);
       if IORes=0 then begin
        SetInOutRes(0);
        write(StdOut,lbuf1);
        IORes:=IOResult;
       end;
       lbuf1:='';
      end;
      if opcode = 171 then sfree(h3);
     end;
     {
     Value parameter - string
     }
     172:
     begin
      h1 := s[t].i; memsh1:=memstrng(h1);
      h4 := memsh1.head.leng;
      memsh2:=sget(h2,h4);
      move(memsh1.data.buff,memsh2.data.buff,h4);
      s[t].i := h2;
      s[t].p := s[b0].i;
      s[b0].i := t;
     end;
     {
     Value parameter - string temporary
     }
     173:
     begin
      s[t].p := s[b0].i;
      s[b0].i := t;
     end;
     {
     Convert string to array
     Convert string temporary to array
     }
     174,
     175:
     begin
      h1 := s[t].i; memsh1:=memstrng(h1);
      h2 := memsh1.head.leng;
      h4 := s[t-1].i;
      if h2>=yoper
      then for h3 := 0 to yoper-1 do s[h4+h3].c := memsh1.data.buff[h3]
      else begin
       for h3 := 0 to h2-1 do s[h4+h3].c := memsh1.data.buff[h3];
       for h3 := h4+h2 to h4+yoper-1 do s[h3].c := ' '
      end;
      if opcode=175 then sfree(h1);
      t := t-2;
     end;
     {
     Write string - defined field
     Write string temporary - defined field
     }
     176,
     177:
     begin
      h4 := s[t].i;
      h3 := s[t-1].i; memsh3:=memstrng(h3);
      h2 := memsh3.head.leng;
      if h2>=h4 then h2 := h4 else begin
       lbuf1:=StringOfChar(' ',h4-h2);
       if IORes=0 then begin
        SetInOutRes(0);
        write(StdOut,lbuf1);
        IORes:=IOResult;
       end;
       lbuf1:='';
      end;
      if h2>0 then begin
       SetString(lbuf1,memsh3.data.buff,h2);
       if IORes=0 then begin
        SetInOutRes(0);
        write(StdOut,lbuf1);
        IORes:=IOResult;
       end;
       lbuf1:='';
      end;
      if opcode=177 then sfree(h3);
      t := t-2;
     end;
     {
     not used
     }
     else
     RunTimeError(ps_SystemCheck,'opcode_invalid');
    end; {case opcode }
    {
    Check external break flag
    }
    if BreakFlag<>0 then begin
     brkflg:=BreakFlag;
     _fixerror(brkflg);
     BreakFlag:=0;
     RunTimeError(ps_ExternalBreak);
    end;
   end; {while}
   if prgsts <> ps_Finalized then RunTimeError(ps_UnexpectedBreak);
   if t <> tstart then RunTimeError(ps_StackUnbalanced);
   if crvlocks.Count>0 then RunTimeError(ps_CrvLockUnbalanced);
   if taglocks.Count>0 then RunTimeError(ps_TagLockUnbalanced);
   {
   It's Ok.
   }
   MortalCount:=0;
   RtmInterpreter:=true;
  except
   on E:Exception do begin
    {
    Runtime errors handller:
    }
    while crvlocks.Count>0 do begin
     Dec(crvlocks.Count);
     crvlocks.Items[crvlocks.Count].Unlock;
    end;
    while taglocks.Count>0 do _taglock(false);
    case prgsts of
     ps_CaseCheck         : ErrorMessage:=' because of undefined case. ';
     ps_DivByZero         : ErrorMessage:=' because of division by 0. ';
     ps_IndexCheck        : ErrorMessage:=' because of invalid index. ';
     ps_StackOverflow     : ErrorMessage:=' because of storage overflow. ';
     ps_StringManager     : ErrorMessage:=' because of string manager error. ';
     ps_UnexpectedBreak   : ErrorMessage:=' because of unexpected break. ';
     ps_ExternalBreak     : ErrorMessage:=' because of external break '+IntToStr(brkflg)+'.';
     ps_StackUnbalanced   : ErrorMessage:=' because of unbalanced stack. ';
     ps_CrvLockUnbalanced : ErrorMessage:=' because of unbalanced crvlock/unlock. ';
     ps_TagLockUnbalanced : ErrorMessage:=' because of unbalanced taglock/unlock. ';
     ps_SystemCheck       : ErrorMessage:=' because of bug in compiler. ';
     else                   ErrorMessage:=' because of unknown reason. ';
    end;
    ErrorMessage:=TheDevice.Name+': Halted at '+d2s(prgcnt-1)+ErrorMessage;
    BugReport(E,TheDaqPascal,Trim(ErrorMessage));
    if SysLogNotable(SeverityOfDaqRunError) then begin
     ErrorMessage:=Trim(StrAfterOf(ErrorMessage,':'));
     SysLogNote(0,SeverityOfDaqRunError,TheDevice.SysLogSign,ErrorMessage);
    end;
    MortalTime:=IntMSecNow;
    inc(MortalCount);
    if not _isISRnow then
    if DebugMode and (IORes=0) and not IsFileClosed(StdOut) then begin
     SetInOutRes(0);
     writeln(StdOut);
     writeln(StdOut,ErrorMessage);
     h1 := b;
     blkcnt := 10;   { post mortem dump }
     repeat
      writeln(StdOut);
      blkcnt := blkcnt - 1;
      if blkcnt = 0 then h1 := 0;
      h2 := s[h1+4].i;
      if h1<>0 then writeln(StdOut,' ',itab[h2].name,' called at',s[h1+1].i:5);
      h2 := btab[itab[h2].ref].last;
      while h2 <> 0 do with itab[h2] do begin
       if obj = vvariable then if typ in stantyps then begin
        write(StdOut,'    ',name,' = ');
        if normal then h3 := h1+adr else h3 := s[h1+adr].i;
        case typ of
         ints : write(StdOut,s[h3].i);
         reals: write(StdOut,s[h3].r);
         bools: if s[h3].b then write(StdOut,'true') else write(StdOut,'false');
         chars: write(StdOut,chr(s[h3].i mod 64));
        end;
        writeln(StdOut);
       end ;
       h2 := link
      end;
      h1 := s[h1+3].i
     until h1 < 0;
     writeln(StdOut);
     IORes:=IOResult;
    end;
    lbuf1:='';
    lbuf2:='';
   end;
  end;
  opcount:=opcount+opcnt;
  {
  Cleanup temporary strings
  }
  if (lbuf1<>'') then lbuf1:='';
  if (lbuf2<>'') then lbuf2:='';
  if (lbuf3<>'') then lbuf3:='';
  {
  ****************
  interpreter exit
  ****************
  }
  dec(EntryCount);
 end;
end; { RtmInterpreter }

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqpascalruntime;
begin
end;

procedure Free_crw_daqpascalruntime;
begin
end;

initialization

 Init_crw_daqpascalruntime;

finalization

 Free_crw_daqpascalruntime;

end.

//////////////
// END OF FILE
//////////////

