////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Console Window - fast console for text output.                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231117 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_consolewindow; // Form Console Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_gloss, _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

const
  ConsoleInterLineMargin = 2;
  ConsoleNumBits         = 10;

type
  TFormConsoleWindow  = class;
  TConsolePollingProc = procedure(aConsole : TFormConsoleWindow);
  TConsoleFilterProc  = procedure(aConsole : TFormConsoleWindow;
                              var aText    : LongString);

  { TFormConsoleWindow }

  TFormConsoleWindow = class(TFormCrwDaqSysChild)
    GroupBoxInput: TGroupBox;
    CheckBoxInput: TCheckBox;
    BitBtnInputToggle: TBitBtn;
    ComboBoxInput: TComboBox;
    GroupBoxOutput: TGroupBox;
    PanelInputCheckBox: TPanel;
    PanelInputComboBox: TPanel;
    PanelOutput: TPanel;
    ScrollBarV: TScrollBar;
    ScrollBarH: TScrollBar;
    PaintBox: TPaintBox;
    ActionEditDelete: TAction;
    ActionEditCut: TAction;
    ActionEditCopy: TAction;
    ActionEditPaste: TAction;
    ActionEditInputEnable: TAction;
    ActionEditInputDisable: TAction;
    ActionEditInputToggle: TAction;
    MenuEdit: TMenuItem;
    MenuEditDelete: TMenuItem;
    MenuEditCut: TMenuItem;
    MenuEditCopy: TMenuItem;
    MenuEditPaste: TMenuItem;
    MenuInputSeparator: TMenuItem;
    MenuEditInputToggle: TMenuItem;
    PopupMenuFilePrint: TMenuItem;
    PopupMenuFilePrintSeparator: TMenuItem;
    PopupMenuEditDelete: TMenuItem;
    PopupMenuEditCut: TMenuItem;
    PopupMenuEditCopy: TMenuItem;
    PopupMenuEditPaste: TMenuItem;
    PopupMenuEditInputSeparator: TMenuItem;
    PopupMenuEditInputToggle: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure GroupBoxInputDblClick(Sender: TObject);
    procedure GroupBoxOutputDblClick(Sender: TObject);
    procedure PaintBoxPaint(Sender: TObject);
    procedure PanelOutputDblClick(Sender: TObject);
    procedure ScrollBarVChange(Sender: TObject);
    procedure ScrollBarHChange(Sender: TObject);
    procedure ComboBoxInputDblClick(Sender: TObject);
    procedure ComboBoxInputKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormActivate(Sender: TObject);
    procedure ActionEditDeleteExecute(Sender: TObject);
    procedure ActionEditCutExecute(Sender: TObject);
    procedure ActionEditCopyExecute(Sender: TObject);
    procedure ActionEditPasteExecute(Sender: TObject);
    procedure ActionEditInputDisableExecute(Sender: TObject);
    procedure ActionEditInputEnableExecute(Sender: TObject);
    procedure ActionEditInputToggleExecute(Sender: TObject);
    procedure CheckBoxInputChange(Sender: TObject);
  private
    { Private declarations }
    myCursor      : Integer;
    myLines       : array[0..(1 shl ConsoleNumBits)-1] of ShortString;
    myTextColor   : TColor;
    myInpFifo     : TFifo;
    myOutFifo     : TFifo;
    myInpOwns     : Boolean;
    myOutOwns     : Boolean;
    myInpFilter   : TConsoleFilterProc;
    myOutFilter   : TConsoleFilterProc;
    myPolling     : TConsolePollingProc;
    myCustom      : Pointer;
    myGuardInput  : Cardinal;
    function  LineToIndex(i:Integer):Integer;
    function  GetLine(i:Integer):ShortString;
    procedure SetLine(i:Integer; const aLine:ShortString);
    function  LinesPerPage:Integer;
    function  LineRect(i:Integer):TRect2I;
    function  GetTextColor:TColor;
    procedure SetTextColor(aColor:TColor);
    function  GetGroundColor:TColor;
    procedure SetGroundColor(aColor:TColor);
    function  GetInpFifo:TFifo;
    function  GetOutFifo:TFifo;
    function  GetInpOwns:Boolean;
    function  GetOutOwns:Boolean;
    function  GetInpFilter:TConsoleFilterProc;
    function  GetOutFilter:TConsoleFilterProc;
    function  GetPolling:TConsolePollingProc;
    procedure SetPolling(aPolling:TConsolePollingProc);
    function  GetCustom:Pointer;
    procedure SetCustom(aCustom:Pointer);
    function  GetGuardInput:Cardinal;
    procedure SetGuardInput(aGuard:Cardinal);
  public
    { Public declarations }
    property  Lines[i:Integer]:ShortString read GetLine        write SetLine;
    property  TextColor:TColor             read GetTextColor   write SetTextColor;
    property  GroundColor:TColor           read GetGroundColor write SetGroundColor;
    property  InpFifo:TFifo                read GetInpFifo;
    property  OutFifo:TFifo                read GetOutFifo;
    property  InpOwns:Boolean              read GetInpOwns;
    property  OutOwns:Boolean              read GetOutOwns;
    property  InpFilter:TConsoleFilterProc read GetInpFilter;
    property  OutFilter:TConsoleFilterProc read GetOutFilter;
    property  Polling:TConsolePollingProc  read GetPolling     write SetPolling;
    property  Custom:Pointer               read GetCustom      write SetCustom;
    property  GuardInput:Cardinal          read GetGuardInput  write SetGuardInput;
    procedure UpdateLines(const aText:LongString);
  public
    procedure AssignFifo(aInpFifo:TFifo; aInpOwns:Boolean; aInpFilter:TConsoleFilterProc;
                         aOutFifo:TFifo; aOutOwns:Boolean; aOutFilter:TConsoleFilterProc);
    procedure UpdateScrollBars;
    procedure UpdateInputBox;
    procedure ClearOutput;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure DrawView; override;
    procedure PutText(const aText:LongString);  // to output fifo
    procedure StartMonitoring;
    procedure StopMonitoring;
    procedure Monitoring;
  public
    class var ConsoleAutoFocusInput:Boolean;
    class var ConsoleAutoEnableInput:Boolean;
  end;
  TConsoleWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormConsoleWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormConsoleWindow);
  public
    property   Window[i:Integer]:TFormConsoleWindow read GetWindow write SetWindow; default;
  end;

const
  FullConsoleWindowList : TConsoleWindowList = nil;
  ConsoleWindowsMonitor : TConsoleWindowList = nil;

function  NewConsoleWindow(const aCaption   : LongString;
                                 aInpFifo   : TFifo;
                                 aInpOwns   : Boolean;
                                 aInpFilter : TConsoleFilterProc;
                                 aOutFifo   : TFifo;
                                 aOutOwns   : Boolean;
                                 aOutFilter : TConsoleFilterProc):TFormConsoleWindow;
procedure Kill(var TheObject:TFormConsoleWindow); overload;
function  ActiveConsoleWindow:TFormConsoleWindow;

function  NewConsoleWindowList(aOwnsObjects : Boolean = true;
                               aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                               aStep        : LongInt = DefaultTObjectStorageStep
                                          ) : TConsoleWindowList;
procedure Kill(var TheObject:TConsoleWindowList); overload;

implementation

{$R *.lfm}

 {
 *********************************
 TConsoleWindowList implementation
 *********************************
 }
function TConsoleWindowList.GetWindow(i:Integer):TFormConsoleWindow;
begin
 Result:=TFormConsoleWindow(Items[i]);
end;

procedure TConsoleWindowList.SetWindow(i:Integer; aWindow:TFormConsoleWindow);
begin
 Items[i]:=aWindow;
end;

function NewConsoleWindowList(aOwnsObjects : Boolean = true;
                              aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                              aStep        : LongInt = DefaultTObjectStorageStep
                                         ) : TConsoleWindowList;
begin
 Result:=nil;
 try
  Result:=TConsoleWindowList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewConsoleWindowList');
 end;
end;

procedure Kill(var TheObject:TConsoleWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 *******************************************************************************
 General purpose routines
 *******************************************************************************
 }
function  NewConsoleWindow(const aCaption   : LongString;
                                 aInpFifo   : TFifo;
                                 aInpOwns   : Boolean;
                                 aInpFilter : TConsoleFilterProc;
                                 aOutFifo   : TFifo;
                                 aOutOwns   : Boolean;
                                 aOutFilter : TConsoleFilterProc):TFormConsoleWindow;
begin
 Result:=nil;
 try
  Application.CreateForm(TFormConsoleWindow,Result);
  with Result do if Ok then begin
   try
    LockDraw;
    Caption:=aCaption;
    AssignFifo(aInpFifo, aInpOwns, aInpFilter, aOutFifo, aOutOwns, aOutFilter);
    ComboBoxInput.Width:=GroupBoxInput.Width-20;
    UpdateScrollBars;
   finally
    UnlockDraw;
   end;
  end else Kill(Result);
 except
  on E:Exception do BugReport(E,nil,'NewConsoleWindow');
 end;
end;

procedure Kill(var TheObject:TFormConsoleWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function ActiveConsoleWindow:TFormConsoleWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if Sdiman.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormConsoleWindow) then Result:=TFormConsoleWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveConsoleWindow');
 end;
end;

 {
 *******************************************************************************
 TFormConsoleWindow implementation
 *******************************************************************************
 }
function TFormConsoleWindow.LineToIndex(i:Integer):Integer;
begin { Note that High(myLines) is power of two minus 1 }
 if Assigned(Self)
 then Result:=(myCursor+i) and High(myLines)
 else Result:=0;
end;

function  TFormConsoleWindow.GetLine(i:Integer):ShortString;
begin
 if Assigned(Self) and InRange(i,Low(myLines),High(myLines))
 then Result:=myLines[LineToIndex(i)]
 else Result:='';
end;

procedure TFormConsoleWindow.SetLine(i:Integer; const aLine:ShortString);
begin
 if Assigned(Self) and InRange(i,Low(myLines),High(myLines))
 then myLines[LineToIndex(i)]:=aLine;
end;

function TFormConsoleWindow.LinesPerPage:Integer;
begin
 if Assigned(Self)
 then Result:=Max(1,PaintBox.Height div Max(1,RectSizeY(LineRect(0))))
 else Result:=0;
end;

function TFormConsoleWindow.LineRect(i:Integer):TRect2I;
var h:Integer;
begin
 if Assigned(Self) then begin
  h:=abs(PaintBox.Font.Height)+ConsoleInterLineMargin;
  Result.A.X:=0;
  Result.A.Y:=i*h;
  Result.B.X:=PaintBox.Width;
  Result.B.Y:=Result.A.Y+h;
 end else Result:=Rect2I(0,0,0,0);
end;

function  TFormConsoleWindow.GetTextColor:TColor;
begin
 if Ok then Result:=myTextColor else Result:=0;
end;

procedure TFormConsoleWindow.SetTextColor(aColor:TColor);
begin
 if Ok then
 try
  LockDraw;
  myTextColor:=aColor;
 finally
  UnlockDraw;
 end;
end;

function  TFormConsoleWindow.GetGroundColor:TColor;
begin
 if Ok then Result:=PaintBox.Color else Result:=0;
end;

procedure TFormConsoleWindow.SetGroundColor(aColor:TColor);
begin
 if Ok then
 try
  LockDraw;
  PaintBox.Color:=aColor;
 finally
  UnlockDraw;
 end;
end;

function  TFormConsoleWindow.GetInpFifo:TFifo;
begin
 if Assigned(Self) then Result:=myInpFifo else Result:=nil;
end;

function  TFormConsoleWindow.GetOutFifo:TFifo;
begin
 if Assigned(Self) then Result:=myOutFifo else Result:=nil;
end;

function  TFormConsoleWindow.GetInpOwns:Boolean;
begin
 if Assigned(Self) then Result:=myInpOwns else Result:=false;
end;

function  TFormConsoleWindow.GetOutOwns:Boolean;
begin
 if Assigned(Self) then Result:=myOutOwns else Result:=false;
end;

function  TFormConsoleWindow.GetInpFilter:TConsoleFilterProc;
begin
 if Assigned(Self) then Result:=myInpFilter else Result:=nil;
end;

function  TFormConsoleWindow.GetOutFilter:TConsoleFilterProc;
begin
 if Assigned(Self) then Result:=myOutFilter else Result:=nil;
end;

function TFormConsoleWindow.GetPolling:TConsolePollingProc;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

procedure TFormConsoleWindow.SetPolling(aPolling:TConsolePollingProc);
begin
 if Assigned(Self) then myPolling:=aPolling;
end;

function  TFormConsoleWindow.GetCustom;
begin
 if Assigned(Self) then Result:=myCustom else Result:=nil;
end;

procedure TFormConsoleWindow.SetCustom(aCustom:Pointer);
begin
 if Assigned(Self) then myCustom:=aCustom;
end;

function TFormConsoleWindow.GetGuardInput:Cardinal;
begin
 if Assigned(Self) then Result:=myGuardInput else Result:=ga_Lock;
end;

procedure TFormConsoleWindow.SetGuardInput(aGuard:Cardinal);
begin
 if Assigned(Self) then myGuardInput:=Max(ga_Lock,Min(ga_Root,aGuard));
end;

procedure TFormConsoleWindow.UpdateLines(const aText:LongString);
var i,Index,Leng:Integer; ch:Char;
const TabLeng=4;
begin
 if Ok and (Length(aText)>0) then
 try
  LockDraw;
  try
   Leng:=Length(aText);
   Index:=LineToIndex(High(myLines));
   for i:=0 to Leng-1 do begin
    ch:=PChar(aText)[i];
    case ch of
     ASCII_CR, ASCII_LF : begin
      if (ch=ASCII_CR) and (i<Leng-1) then           // if CR,LF found,
      if (PChar(aText)[i+1]=ASCII_LF) then continue; // then ignore CR
      inc(myCursor);
      Index:=LineToIndex(High(myLines));
      myLines[Index]:='';
     end;
     ASCII_TAB : begin
      myLines[Index]:=myLines[Index]+' ';
      while (Length(myLines[Index])<sizeof(myLines[0])-1) and
            (Length(myLines[Index]) mod TabLeng <> 0)
      do myLines[Index]:=myLines[Index]+' ';
     end;
     else  myLines[Index]:=myLines[Index]+ch;
    end;
   end;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateLines');
 end;
end;

procedure TFormConsoleWindow.AssignFifo(aInpFifo:TFifo; aInpOwns:Boolean; aInpFilter:TConsoleFilterProc;
                                        aOutFifo:TFifo; aOutOwns:Boolean; aOutFilter:TConsoleFilterProc);
begin
 if Ok then
 try
  LockDraw;
  try
   if myInpOwns then Kill(myInpFifo);
   if myOutOwns then Kill(myOutFifo);
   myInpFifo:=aInpFifo;
   myInpOwns:=aInpOwns;
   myInpFilter:=aInpFilter;
   myOutFifo:=aOutFifo;
   myOutOwns:=aOutOwns;
   myOutFilter:=aOutFilter;
   GroupBoxInput.Visible:=InpFifo.Ok;
   GroupBoxOutput.Visible:=OutFifo.Ok;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'AssignFifo');
 end;
end;

procedure TFormConsoleWindow.UpdateScrollBars;
begin
 if Ok then
 try
  LockDraw;
  try
   with ScrollBarV do begin
    Min:=Low(myLines);
    Max:=math.Max(Min,High(myLines)-(LinesPerPage-1));
    SmallChange:=1;
    LargeChange:=10*SmallChange;
    Position:=Max;
   end;
   with ScrollBarH do begin
    Min:=0;
    Max:=(sizeof(myLines[0])-20)*PaintBox.Canvas.TextWidth('O');
    SmallChange:=PaintBox.Canvas.TextWidth('O');
    LargeChange:=10*SmallChange;
    Position:=Min;
   end;
   UpdateInputBox;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateScrollBars');
 end;
end;

procedure TFormConsoleWindow.UpdateInputBox;
begin
 if Ok then
 try
  // Update EditInput state
  if CheckBoxInput.Checked then begin
   SmartUpdate(ActionEditInputToggle,RusEng('Запретить Ввод','Disable Input'));
   ActionEditInputToggle.ImageIndex:=177;
  end else begin
   SmartUpdate(ActionEditInputToggle,RusEng('Разрешить Ввод','Enable Input'));
   ActionEditInputToggle.ImageIndex:=176;
  end;
  PopupMenuEditInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  MenuEditInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  BitBtnInputToggle.ImageIndex:=ActionEditInputToggle.ImageIndex;
  // Validate ComboBoxInput width ...
  ComboBoxInput.Width:=PanelInputComboBox.Width-ComboBoxInput.Left*2-2;
 except
  on E:Exception do BugReport(E,Self,'UpdateInputBox');
 end;
end;

procedure TFormConsoleWindow.ClearOutput;
begin
 if Ok then
 try
  LockDraw;
  SafeFillChar(myLines,sizeof(myLines),0);
  myCursor:=0;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.UpdateCommands;
var Exposed,CanPaste,CanInput:Boolean;
begin
 try
  inherited UpdateCommands;
  Exposed:=FormIsExposed(Self);
  CanPaste:=Exposed and Clipboard.HasFormat(CF_TEXT);
  CanInput:=CheckBoxInput.Checked;
  ActionFileSave.Enabled:=false;
  ActionFileSaveAs.Enabled:=false;
  ActionFilePrint.Enabled:=Exposed;
  ActionEditDelete.Enabled:=Exposed;
  ActionEditCut.Enabled:=Exposed;
  ActionEditCopy.Enabled:=Exposed;
  ActionEditPaste.Enabled:=CanPaste;
  ComboBoxInput.Enabled:=CanInput;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormConsoleWindow.DrawView;
var i:Integer; R:TRect2I;
begin
 try
  if Ok and IsFormViewable then begin
   DebugLogReport_DrawView;
   for i:=0 to LinesPerPage-1 do begin
    R:=LineRect(i);
    DrawBar(PaintBox.Canvas,R,GroundColor);
    RectMove(R,-ScrollBarH.Position,0);
    DrawText(PaintBox.Canvas,R.A,Lines[ScrollBarV.Position+i],TextColor,GroundColor);
   end;
   DrawBar(PaintBox.Canvas,LineRect(LinesPerPage),GroundColor);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawView');
 end;
end;

procedure TFormConsoleWindow.PutText(const aText:LongString);
begin
 OutFifo.PutText(aText);
end;

procedure ConsoleWindowMonitor(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if (aObject is TFormConsoleWindow) then
 with TFormConsoleWindow(aObject) do if MonitorEvent then Monitoring;
end;

procedure ConsoleWindowsMonitoring;
begin
 ConsoleWindowsMonitor.ForEach(ConsoleWindowMonitor,nil);
end;

procedure TFormConsoleWindow.StartMonitoring;
begin
 try
  if Ok and ConsoleWindowsMonitor.Ok then begin
   if (ConsoleWindowsMonitor.IndexOf(Self)<0) then begin
    ConsoleWindowsMonitor.Add(Self);
    Tick55Actions.Add(ConsoleWindowsMonitoring);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'StartMonitoring');
 end;
end;

procedure TFormConsoleWindow.StopMonitoring;
begin
 try
  if Ok and ConsoleWindowsMonitor.Ok then begin
   if (ConsoleWindowsMonitor.IndexOf(Self)>=0) then begin
    ConsoleWindowsMonitor.Remove(Self);
    if ConsoleWindowsMonitor.Count=0 then
    Tick55Actions.Remove(ConsoleWindowsMonitoring);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'StopMonitoring');
 end;
end;

procedure TFormConsoleWindow.Monitoring;
var aText:LongString;
begin
 if Ok then
 try
  if (InpFifo.Count>0) then begin
   aText:=InpFifo.GetText;
   if Assigned(myInpFilter) then myInpFilter(Self,aText);
   aText:='';
  end;
  if (OutFifo.Count>0) then begin
   aText:=OutFifo.GetText;
   if Assigned(myOutFilter) then myOutFilter(Self,aText);
   if (Length(aText)>0) then UpdateLines(aText);
   aText:='';
  end;
  if Assigned(myPolling) then myPolling(Self);
  if PendingTrigger[ptrg_FormActivate] then begin
   PendingTrigger[ptrg_FormActivate]:=false;
   if ConsoleAutoFocusInput then PendingTrigger[ptrg_FocusInput]:=true;;
  end;
  if PendingTrigger[ptrg_FocusInput] then begin
   PendingTrigger[ptrg_FocusInput]:=false;
   if Active and not ComboBoxInput.Focused then
   if ComboBoxInput.CanSetFocus then ComboBoxInput.SetFocus;
  end;
 except
  on E:Exception do BugReport(E,Self,'Monitoring');
 end;
end;

procedure TFormConsoleWindow.FormCreate(Sender: TObject);
begin
 inherited;
 ToolBar.Hide;
 StatusBar.Hide;
 myInpFifo:=nil;
 myInpOwns:=false;
 myInpFilter:=nil;
 myOutFifo:=nil;
 myOutOwns:=false;
 myOutFilter:=nil;
 myCustom:=nil;
 myGuardInput:=ga_Guest;
 try
  LockDraw;
  TextColor:=clBtnText;
  GroundColor:=clBtnFace;
  ClearOutput;
  UpdateScrollBars;
 finally
  UnlockDraw;
 end;
 if SdiMan.IsSdiMode then GroupBoxInput.Cursor:=SdiMan.CursorActivateMainSdiForm;
 if SdiMan.IsSdiMode then GroupBoxOutput.Cursor:=SdiMan.CursorActivateMainSdiForm;
 UpdateMenu(MenuEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню для редактирования.','Editor operations.'),
            0);
 UpdateMenu(MenuEditDelete,
            RusEng('Удалить','Delete'),
            RusEng('Удалить выделенный фрагмент.','Delete selected region.'),
            ShortCut(VK_DELETE,[ssCtrl]));
 UpdateMenu(MenuEditCut,
            RusEng('Вырезать','Cut'),
            RusEng('Вырезать в буфер обмена.','Cut to clipboard.'),
            ShortCut(Word('X'),[ssCtrl]));
 UpdateMenu(MenuEditCopy,
            RusEng('Копировать','Copy'),
            RusEng('Копировать в буфер обмена.','Copy to clipboard.'),
            ShortCut(Word('C'),[ssCtrl]));
 UpdateMenu(MenuEditPaste,
            RusEng('Вставить','Paste'),
            RusEng('Вставить из буфера обмена.','Paste from clipboard.'),
            ShortCut(Word('V'),[ssCtrl]));
 UpdateMenu(MenuEditInputToggle,
            RusEng('Ввод','Input'),
            RusEng('Разрежить/Запретить Консольный Ввод.','Enable/Disable Console Input.'),
            ShortCut(Word('I'),[ssCtrl]));
 GroupBoxInput.Caption:=RusEng('Ввод:','Input:');
 GroupBoxOutput.Caption:=RusEng('Вывод:','Output:');
 GroupBoxInput.Hint:=RusEng('Консольный Ввод.','Console Input.');
 GroupBoxOutput.Hint:=RusEng('Консольный Вывод.','Console Output.');
 CheckBoxInput.Hint:=RusEng('Разрешить/Запретить Ввод.','Enable/Disable Input.');
 CheckBoxInput.Checked:=ConsoleAutoEnableInput;
 BitBtnInputToggle.Hint:=CheckBoxInput.Hint;
 BitBtnInputToggle.Caption:='';
 UpdateInputBox;
end;

procedure TFormConsoleWindow.FormDestroy(Sender: TObject);
begin
 ClearOutput;
 AssignFifo(nil,false,nil,nil,false,nil);
 inherited;
end;

procedure TFormConsoleWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullConsoleWindowList.Add(Self);
 AddonSdiFlags(sf_SdiConsole);
end;

procedure TFormConsoleWindow.BeforeDestruction;
begin
 StopMonitoring;
 FullConsoleWindowList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormConsoleWindow.FormResize(Sender: TObject);
begin
 inherited;
 UpdateScrollBars;
end;

procedure TFormConsoleWindow.GroupBoxInputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.GroupBoxOutputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.PanelOutputDblClick(Sender: TObject);
begin
 SdiMan.ActivateMainForm;
end;

procedure TFormConsoleWindow.PaintBoxPaint(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ScrollBarVChange(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ScrollBarHChange(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormConsoleWindow.ComboBoxInputDblClick(Sender: TObject);
var History:Integer;
begin
 if Guard.CheckAction(GuardInput,'TFormConsoleWindow.Input')<0 then Exit;
 if Ok then
 try
  inherited;
  History:=SysGlossary.ReadIniLinkDef(SysIniFile,SectSystem,'ConsoleInputHistory',50);
  RecordComboBoxHistory(ComboBoxInput,History);
  InpFifo.PutText(ComboBoxInput.Text+EOL);
  ComboBoxInput.Text:='';
 except
  on E:Exception do BugReport(E,Self,'ComboBoxInputDblClick');
 end;
end;

procedure TFormConsoleWindow.ComboBoxInputKeyDown(Sender: TObject;  var Key: Word; Shift: TShiftState);
 procedure UpDown(step:Integer);
 var i:Integer;
 begin
  if (ComboBoxInput.Items.Count>0) then begin
   i:=EnsureRange(ComboBoxInput.ItemIndex+Step,0,ComboBoxInput.Items.Count-1);
   ComboBoxInput.ItemIndex:=i;
   Key:=0;
  end;
 end;
begin
 inherited;
 case Key of
  VK_RETURN : ComboBoxInputDblClick(Sender);
  VK_DOWN   : UpDown(-1);
  VK_UP     : UpDown(+1);
 end;
end;

procedure TFormConsoleWindow.CheckBoxInputChange(Sender: TObject);
begin
 UpdateInputBox;
 PendingTrigger[ptrg_FocusInput]:=true;
end;

procedure TFormConsoleWindow.ActionEditInputEnableExecute(Sender: TObject);
begin
 if Ok then begin
  CheckBoxInput.Checked:=true;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.ActionEditInputToggleExecute(Sender: TObject);
begin
 if Ok then begin
  if Guard.CheckAction(ga_Guest,ActionEditInputToggle)<0 then Exit;
  CheckBoxInput.Checked:=not CheckBoxInput.Checked;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.ActionEditInputDisableExecute(Sender: TObject);
begin
 if Ok then begin
  CheckBoxInput.Checked:=false;
  UpdateInputBox;
 end;
end;

procedure TFormConsoleWindow.FormActivate(Sender: TObject);
begin
 inherited;
 UpdateInputBox;
end;

procedure TFormConsoleWindow.ActionEditDeleteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditDelete)<0 then Exit;
 inherited;
 ClearOutput;
end;

procedure TFormConsoleWindow.ActionEditCutExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCut)<0 then Exit;
 inherited;
 ActionEditCopyExecute(Sender);
 ClearOutput;
end;

procedure TFormConsoleWindow.ActionEditCopyExecute(Sender: TObject);
var i,j:Integer; s:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionEditCopy)<0 then Exit;
 if Ok then
 try
  inherited;
  j:=-1;
  s:='';
  for i:=Low(myLines) to High(myLines) do begin
   if Length(Lines[i])>0 then j:=i;
   if j>=0 then s:=s+Lines[i]+EOL;
  end;
  Clipboard.AsText:=s;
  s:='';
 except
  on E:Exception do BugReport(E,Self,'ActionEditCopyExecute');
 end;
end;

procedure TFormConsoleWindow.ActionEditPasteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditPaste)<0 then Exit;
 if Ok then
 try
  inherited;
  if Clipboard.HasFormat(CF_TEXT) then PutText(Clipboard.AsText);
 except
  on E:Exception do BugReport(E,Self,'ActionEditPasteExecute');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_consolewindow;
begin
 TFormConsoleWindow.ConsoleAutoFocusInput:=false;
 TFormConsoleWindow.ConsoleAutoEnableInput:=false;
 FullConsoleWindowList:=NewConsoleWindowList(false);
 FullConsoleWindowList.Master:=@FullConsoleWindowList;
 ConsoleWindowsMonitor:=NewConsoleWindowList(false);
 ConsoleWindowsMonitor.Master:=@ConsoleWindowsMonitor;
end;

procedure Free_form_consolewindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullConsoleWindowList.Count', FullConsoleWindowList.Count]));
 ResourceLeakageLog(Format('%-60s = %d',['ConsoleWindowsMonitor.Count', ConsoleWindowsMonitor.Count]));
 Kill(FullConsoleWindowList);
 Kill(ConsoleWindowsMonitor);
end;

initialization

 Init_form_consolewindow;

finalization

 Free_form_consolewindow;

end.

//////////////
// END OF FILE
//////////////

