'****************************************************************
'** Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru      **
'** Under MIT License, see https://opensource.org/licenses/MIT **
'****************************************************************
'** Command to find (registered) application path.             **
'** Both EXE filename or file type or file extension           **
'** association may be used to identify application.           **
'** A list of arguments can be used, and first found           **
'** of them will be returned as application filepath.          **
'** Usage: GetAppPath firefox.exe opera.exe .htm .html         **
'** Use CSCRIPT //I //NOLOGO GetAppPath.vbs .. to call         **
'****************************************************************

Option Explicit

'******* System objects
'**********************

dim StdIn  : set StdIn  = WScript.StdIn
dim StdOut : set StdOut = WScript.StdOut
dim StdErr : set StdErr = WScript.StdErr
dim Shell  : set Shell  = WScript.CreateObject("WScript.Shell")
dim FSO    : set FSO    = WScript.CreateObject("Scripting.FileSystemObject")

'******* Utility routines
'************************

sub WriteLn(line)
 StdOut.WriteLine(line)
end sub

sub PrintStdErr(line)
 StdErr.WriteLine(line)
end sub

sub Abort(ExitCode,ErrorMessage)
 PrintStdErr(ErrorMessage)
 WScript.Quit(ExitCode)
end sub

sub Assert(Condition,ExitCode,ErrorMessage)
 if not cbool(Condition) then
  call Abort(ExitCode,ErrorMessage)
 end if
end sub

function atoldef(s,def)
 On Error Resume Next
 dim l : l=clng(s)
 if (Err.Number<>0) then : l=def : end if : Err.Clear
 On Error Goto 0
 atoldef=l
end function

function IsEmptyStr(s)
 IsEmptyStr=not(trim(s)<>"")
end function

function SameText(a,b)
 SameText=not(ucase(cstr(a))<>ucase(cstr(b)))
end function

function GetEnv(VarName)
 GetEnv = Shell.Environment("PROCESS").Item(VarName)
end function

function ExpandEnvironmentStrings(str)
 ExpandEnvironmentStrings = Shell.ExpandEnvironmentStrings(str)
end function

function ShellRun(cmd,show,wait)
 ShellRun = Shell.Run(cmd,show,wait)
end function

function ShellRegRead(key)
 dim result : result = ""
 On Error Resume Next
 result = Shell.RegRead(key)
 if (Err.Number<>0) then : result = "" : end if : Err.Clear
 On Error Goto 0
 ShellRegRead = result
end function

function GetBaseName(path)
 GetBaseName = FSO.GetBaseName(path)
end function

function GetFileName(path)
 GetFileName = FSO.GetFileName(path)
end function

function GetExtensionName(path)
 GetExtensionName = FSO.GetExtensionName(path)
end function

function HasExtensionName(path)
 HasExtensionName = not IsEmptyStr(path) and not IsEmptyStr(GetExtensionName(path))
end function

function IsExtension(arg) '*** is arg looks like extension i.e. ".html"?
 IsExtension = not IsEmptyStr(arg) and HasExtensionName(arg) and (SameText(arg,"."+GetExtensionName(arg)))
end function

function GetParentFolderName(path)
 GetParentFolderName = FSO.GetParentFolderName(path)
end function

function HasParentFolderName(path)
 HasParentFolderName = not IsEmptyStr(path) and not IsEmptyStr(GetParentFolderName(path))
end function

function FileExists(path)
 FileExists = FSO.FileExists(path)
end function

function BuildPath(dir,name)
 BuildPath = FSO.BuildPath(dir,name)
end function

function GetAbsolutePathName(path)
 GetAbsolutePathName = FSO.GetAbsolutePathName(path)
end function

function SpecialFolders(name)
 SpecialFolders = Shell.SpecialFolders(name)
end function

function GetTempName
 GetTempName = FSO.GetTempName
end function

function GetEnvTEMP
 dim path : path=trim(GetEnv("TEMP"))
 const fallbackTEMP = "%SystemRoot%\TEMP"
 if IsEmptyStr(path) then path=ExpandEnvironmentStrings(fallbackTEMP) end if
 GetEnvTEMP=path
end function

function GetEnvPATH
 dim path : path=trim(GetEnv("PATH"))
 const fallbackPATH = "%SystemRoot%;%SystemRoot%\System32"
 if IsEmptyStr(path) then path=ExpandEnvironmentStrings(fallbackPATH) end if
 GetEnvPATH=path
end function

function GetEnvPATHEXT
 dim path : path=trim(GetEnv("PATHEXT"))
 const fallbackPATHEXT = ".COM;.EXE;.BAT;.CMD"
 if IsEmptyStr(path) then path=fallbackPATHEXT end if
 GetEnvPATHEXT=path
end function

function GetTempFileName(dir,ext)
 dim name : name=BuildPath(trim(dir),GetTempName)
 if IsExtension(ext) then name=BuildPath(GetParentFolderName(name),GetBaseName(name))+trim(ext) end if
 GetTempFileName=name
end function

function GetLongFileName(name) '*** Short to long names: "C:\PROGRA~1" => "C:\Program Files"
 if not IsEmptyStr(name) and FileExists(name) and (InStr(name,"~")>0) then
  dim scut : set scut=Shell.CreateShortcut(GetTempFileName(GetEnvTEMP,".lnk"))
  scut.TargetPath=FSO.GetFile(name)
  name=scut.TargetPath
 end if
 GetLongFileName=name
end function

function SearchFile(name,dirlist,extlist,sep)
 dim result : result=""
 name=trim(name) : sep=trim(sep)
 if (name<>"") and (sep<>"") then
  dim dirs : dirs=split(sep+trim(dirlist),sep) : dim idir : idir=0
  dim exts : exts=split(sep+trim(extlist),sep) : dim iext : iext=0
  for iext=0 to UBound(exts)
   dim ext : ext=trim(exts(iext)) : ext=GetExtensionName(ext)
   for idir=0 to UBound(dirs)
    dim dir : dir=trim(dirs(idir)) : dir=GetAbsolutePathName(dir)
    dim fqn : fqn=BuildPath(dir,name) : if (ext<>"") then fqn=fqn+"."+ext end if
    if (fqn<>"") and FileExists(fqn) then result=GetAbsolutePathName(trim(fqn)) end if
    if (result<>"") then exit for end if
   next
   if (result<>"") then exit for end if
  next
 end if
 SearchFile=result
end function

const HKCR="HKEY_CLASSES_ROOT"
const HKCU="HKEY_CURRENT_USER"
const HKLM="HKEY_LOCAL_MACHINE"

function GetRegAppPath(app)
 dim s : s=""
 const keyAppPaths="\SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\"
 if IsEmptyStr(s) then s=ShellRegRead(HKCU+keyAppPaths+app+"\") end if
 if IsEmptyStr(s) then s=ShellRegRead(HKLM+keyAppPaths+app+"\") end if
 GetRegAppPath=s
end function

function GetSystemAssoc(assoc)
 dim s : s="" : assoc=trim(assoc)
 if (len(assoc)>1) and SameText(left(assoc,1),".") then
  if IsEmptyStr(s) then s=ShellRegRead(HKCR+"\"+assoc+"\") end if
  if IsEmptyStr(s) then s=ShellRegRead(HKCU+"\SOFTWARE\Classes\"+assoc+"\") end if
  if IsEmptyStr(s) then s=ShellRegRead(HKLM+"\SOFTWARE\Classes\"+assoc+"\") end if
 end if
 GetSystemAssoc=trim(s)
end function

function GetSystemFType(ftype)
 dim s : s="" : ftype=trim(ftype)
 if (len(ftype)>0) then
  if IsEmptyStr(s) then s=ShellRegRead(HKCR+"\"+ftype+"\Shell\Open\Command\") end if
  if IsEmptyStr(s) then s=ShellRegRead(HKCU+"\SOFTWARE\Classes\"+ftype+"\Shell\Open\Command\") end if
  if IsEmptyStr(s) then s=ShellRegRead(HKLM+"\SOFTWARE\Classes\"+ftype+"\Shell\Open\Command\") end if
 end if
 GetSystemFType=trim(s)
end function

function ExtractFirstWord(s)
 dim words : s=trim(s)
 if not IsEmptyStr(s) then
  if (left(s,1)="""") then
   words=split(s,"""") : if (UBound(words)>1) then s=words(1) else s="" end if
  else
   words=split(s," ") : if (UBound(words)>0) then s=words(0) end if
  end if
 end if
 ExtractFirstWord=s
end function

function GetSystemFTypeExe(ftype)
 dim exe : exe="" : ftype=trim(ftype)
 if not IsEmptyStr(ftype) then exe=ExtractFirstWord(GetSystemFType(ftype)) end if
 GetSystemFTypeExe=trim(exe)
end function

function GetSystemAssocExe(assoc)
 dim exe : exe="" : assoc=trim(assoc)
 if not IsEmptyStr(assoc) then exe=GetSystemFTypeExe(GetSystemAssoc(assoc)) end if
 GetSystemAssocExe=trim(exe)
end function

function IsTypeExe(arg) '*** is arg looks like executable i.e. "name.exe" or "name.cmd", see PATHEXT
 dim result : result=false
 if HasExtensionName(arg) then
  dim aext : aext=GetExtensionName(arg)
  dim exts : exts=split(GetEnvPATHEXT,";") : dim iext : iext=0
  for iext=0 to UBound(exts)
   dim ext: ext= trim(exts(iext)): ext=GetExtensionName(ext)
   if (ext<>"") and SameText(ext,aext) then result=true end if
   if result then exit for end if
  next
 end if
 IsTypeExe=result
end function

function IsCmdName(arg) '*** is arg looks like command (just a simple word) i.e. "name"?
 IsCmdName = not IsEmptyStr(arg) and not HasParentFolderName(arg) and not HasExtensionName(arg)
end function

function IsFullExe(arg) '*** is arg looks like full EXE name i.e. "path\name.exe"?
 IsFullExe = not IsEmptyStr(arg) and HasParentFolderName(arg) and IsTypeExe(arg)
end function

function IsExeName(arg) '*** is arg looks like short EXE name i.e. "name.exe"
 IsExeName = not IsEmptyStr(arg) and not HasParentFolderName(arg) and IsTypeExe(arg)
end function

function IsItAssoc(arg)   '*** is arg looks like association by extension i.e. ".html"?
 IsItAssoc = not IsEmptyStr(arg) and not HasParentFolderName(arg) and IsExtension(arg)
end function

function IsItFType(arg)   '*** is arg looks like file type i.e. "htmlfile" or "TIFImage.Document"?
 IsItFType = not IsEmptyStr(arg) and not HasParentFolderName(arg) and not IsItAssoc(arg) and not IsExeName(arg)
end function

function DoubleQuotedIfNeed(arg) '*** apply double quotes to string which contains spaces
 if (InStr(arg," ")>0) or (InStr(arg,vbTab)>0) or (InStr(arg,vbCr)>0) or (InStr(arg,vbLf)>0) then
  arg=""""+arg+""""
 end if
 DoubleQuotedIfNeed=arg
end function

'******* Print Version
'*********************

sub PrintVersion(name)
 WriteLn(ucase(name)+" version 1.0")
end sub

'******* Print help screen
'*************************

sub PrintHelp(name)
 PrintVersion(name)
 WriteLn("Find registered application, print path or run it.")
 WriteLn("Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru")
 WriteLn("Under MIT License, see https://opensource.org/licenses/MIT")
 WriteLn("Help on "+ucase(name)+":")
 WriteLn(" =================> Syntax:")
 WriteLn("  "+ucase(name)+" [Options] [Arguments] [--run] [Options] [Params]")
 WriteLn(" =================> Options:")
 WriteLn("   --            => options ended, next is params")
 WriteLn("   --version     => print program version and exit")
 WriteLn("   -h,--help     => print this help screen and exit")
 WriteLn("   -f,--full     => next argument expect to be full path\name.ext")
 WriteLn("   -e,--exe      => next is short file name.exe to search in PATH")
 WriteLn("   -b,--base     => next is base name to search with PATH/PATHEXT")
 WriteLn("   -r,--reg      => next is name.exe to search registry App Paths")
 WriteLn("   -a,--assoc    => next argument expect to be association (.ext)")
 WriteLn("   -t,--ftype    => next argument expect to be filetype specifier")
 WriteLn("   -c,--cmd      => set --filter feb (full+exe+base) for commands")
 WriteLn("   --filter f    => set filter f (check types)  for next argument")
 WriteLn("   -        f    => by default f=febrat & filter reset to default")
 WriteLn("   -             => after following argument processed by filter.")
 WriteLn("   -             => for example --filter feb equals to -c option.")
 WriteLn("   --run,--start => run (start) application with following Params")
 WriteLn("   --wait        => with --run option wait application until ends")
 WriteLn("   --test        => with --run option don't run but print cmdline")
 WriteLn("   --show n      => with --run option set application show mode n")
 WriteLn("          n = 0  => SW_HIDE to run application with hidden window")
 WriteLn("              1  => SW_SHOWNORMAL  to activate and display window")
 WriteLn("              2  => SW_SHOWMINIMIZED  activate and show minimized")
 WriteLn("              3  => SW_SHOWMAXIMIZED  activate and show maximized")
 WriteLn("              4  => SW_SHOWNOACTIVATE display it but not activate")
 WriteLn("              5  => SW_SHOW activate, display in current position")
 WriteLn("              6  => SW_MINIMIZE minimize, activate another window")
 WriteLn("              7  => SW_SHOWMINNOACTIVE run minimized, keeps focus")
 WriteLn("              8  => SW_SHOWNA show in normal pos, do not activate")
 WriteLn("              9  => SW_RESTORE activate and display normal window")
 WriteLn("              10 => SW_SHOWDEFAULT display it in default position")
 WriteLn(" =================> Arguments:")
 WriteLn("   file.exe      => find application by EXE file  like firefox.exe")
 WriteLn("   file          => find application by base name like firefox")
 WriteLn("   ftype         => find application by file type like htmlfile")
 WriteLn("   .ext          => find application by extension like .html")
 WriteLn("   Params        => parameters to pass for application on --run")
 WriteLn(" =================> Exit Code:")
 WriteLn("   0             => specified application found")
 WriteLn("   1             => specified application is not found ")
 WriteLn("   2             => some error found (bad arguments/options)")
 WriteLn("   else          => some error found (internal script error)")
 WriteLn("   with --run --wait options return application's exit code.")
 WriteLn(" =================> Note:")
 WriteLn("   Search application full path\name.ext registered in system")
 WriteLn("   and specified by argument EXE file name (like firefox.exe)")
 WriteLn("   or file type (like htmlfile) or file extension (like .htm)")
 WriteLn("   The list of argument(s) can be passed to find application.")
 WriteLn("   If any specified applications found, return first of them.")
 WriteLn("   By default just print found applicaton full path\name.ext.")
 WriteLn("   Option --run or --start  will run application with Params.")
 WriteLn("   Option --show uses to specify application showwindow mode.")
 WriteLn("   Option --wait uses to run application and wait until ends.")
 WriteLn("   Option --test uses to print a cmd line to run application.")
 WriteLn("   By default try to detect argument by type (febrat), where:")
 WriteLn("   f = full path\name.ext (like c:\windows\system32\cmd.exe);")
 WriteLn("   e = exe  file name.ext to search with PATH (like cmd.exe);")
 WriteLn("   b = base file name to search with PATH/PATHEXT (like cmd);")
 WriteLn("   r = reg  name.ext in registry App Paths (like chrome.exe);")
 WriteLn("   a = assoc is file extension name association (like .html);")
 WriteLn("   t = ftype is registry file type specifier (like htmlfile);")
 WriteLn("   c = cmd is feb (full+exe+base) for command (like cmd.exe).")
 WriteLn("   By default all argument types checking but you can specify")
 WriteLn("   following argument type by options (-f,-e,-b,-r,-a,-t,-c).")
 WriteLn("   You can specify types by --filter option (default=febrat).")
 WriteLn(" =================> Examples:")
 WriteLn("  call "+name+" --help")
 WriteLn("  call "+name+" --version")
 WriteLn("  echo FIND APPLICATION AND PRINT HIS PATH:")
 WriteLn("  call "+name+" -e akelpad.exe -r notepad++.exe -t txtfile -a .txt")
 WriteLn("  call "+name+" firefox.exe chrome.exe opera.exe iexplore.exe .html")
 WriteLn("  call "+name+" -t htmlfile .html .htm -t shtmlfile .shtml -t urlfile .url")
 WriteLn("  echo FIND APPLICATION AND RUN (START) WITH PARAMETERS:")
 WriteLn("  call "+name+" .txt --run --test c:\Crw32exe\Crw32.ini")
 WriteLn("  call "+name+" -t textfile --run --test c:\Crw32exe\Crw32.ini")
 WriteLn("  call "+name+" -c %ComSpec% -c cmd -c cmd.exe --run /k echo Run CMD is OK")
 WriteLn("  call "+name+" notepad.exe .txt --run --wait --show 3 c:\Crw32exe\Crw32.ini")
 WriteLn("  call "+name+" firefox.exe .html --start c:\Crw32exe\Resource\Manual\crw-daq.htm")
 WriteLn("  call "+name+" SumatraPDF.exe -t SumatraPDF AcroRd32.exe Acrobat.exe -t acrobat .pdf --run")
end sub

'******* Program data and options
'********************************

dim argnum    : argnum    = 0
dim AppPath   : AppPath   = ""
dim Params    : Params    = ""
dim optRun    : optRun    = false
dim optWait   : optWait   = false
dim optTest   : optTest   = false
dim optShow   : optShow   = 1
const DefFilter           = "febrat"
dim optFilter : optFilter = DefFilter

sub SetOptRun(opt)
 optRun=opt
 if optRun then
  optWait=false
  optTest=false
  optShow=1
 end if
end sub

sub SetOptWait(opt)
 optWait=opt
end sub

sub SetOptTest(opt)
 optTest=opt
end sub

sub SetOptShow(opt)
 dim sw : sw=atoldef(opt,-1)
 call Assert((sw>=0) and (sw<=10),2,"Error: invalid option --show "+opt+". Valid option is --show n, where n=0..10.")
 optShow=sw
end sub

sub SetOptFilter(opt)
 optFilter=lcase(opt)
end sub

sub AddParams(arg)
 if optRun then
  if not IsEmptyStr(arg) then Params=Params+" "+DoubleQuotedIfNeed(arg) end if
 end if
end sub

'******* Handle arguments
'************************
sub HandleArgs(arg)
 argnum=argnum+1
 if not optRun then
  if IsEmptyStr(AppPath) and (InStr(optFilter,"f")>0) and IsFullExe(arg) then
   AppPath=trim(ExtractFirstWord(arg))
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  if IsEmptyStr(AppPath) and (InStr(optFilter,"e")>0) and IsExeName(arg) then
   AppPath=SearchFile(arg,GetEnvPATH,"",";")
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  if IsEmptyStr(AppPath) and (InStr(optFilter,"b")>0) and IsCmdName(arg) then
   AppPath=SearchFile(arg,GetEnvPATH,GetEnvPATHEXT,";")
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  if IsEmptyStr(AppPath) and (InStr(optFilter,"r")>0) and IsExeName(arg) then
   AppPath=ExpandEnvironmentStrings(GetRegAppPath(arg))
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  if IsEmptyStr(AppPath) and (InStr(optFilter,"a")>0) and IsItAssoc(arg) then
   AppPath=ExpandEnvironmentStrings(GetSystemAssocExe(arg))
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  if IsEmptyStr(AppPath) and (InStr(optFilter,"t")>0) and IsItFType(arg) then
   AppPath=ExpandEnvironmentStrings(GetSystemFTypeExe(arg))
   if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
  end if
  optFilter=DefFilter
 end if
 if optRun then
  call AddParams(arg)
 end if
end sub

'******* Command line parsing
'****************************

dim i,arg,opt,isopt : arg="" : opt="" : isopt=true
for i=0 to WScript.Arguments.Count-1
 arg=WScript.Arguments(i)
 if (left(arg,1)="-") and isopt and (opt="") then
  select case arg
   case "--"                    : isopt=false ' end of options, interpret next arguments as params
   case "--version"             : PrintVersion(GetBaseName(WScript.ScriptName)) : WScript.Quit(0)
   case "-h","--help"           : PrintHelp(GetBaseName(WScript.ScriptName))    : WScript.Quit(0)
   case "--run","--start"       : call SetOptRun(true)      '*** start application, if found
   case "--wait"                : call SetOptWait(true)     '*** wait started process ending
   case "--test"                : call SetOptTest(true)     '*** test: do not run, but print
   case "-f","--full"           : call SetOptFilter("f")    '*** full path\name.ext expected
   case "-e","--exe"            : call SetOptFilter("e")    '*** exe  file name.exe expected
   case "-b","--base"           : call SetOptFilter("b")    '*** base file name     expected
   case "-r","--reg"            : call SetOptFilter("r")    '*** uses registry   (App Paths)
   case "-a","--assoc"          : call SetOptFilter("a")    '*** uses associations   (assoc)
   case "-t","--ftype"          : call SetOptFilter("t")    '*** uses file type name (ftype)
   case "-c","--cmd"            : call SetOptFilter("feb")  '*** full+exe+base  for commands
   case "--filter"              : opt=arg                   '*** or set another types filter
   case "--show"                : opt=arg                   '*** ShowWindow modes to --start
   case else                    : call Abort(2,"Error: unknown option "+arg+". See --help.")
  end select
 else
  select case opt
   case ""                      : call HandleArgs(arg)
   case "--show"                : call SetOptShow(arg)
   case "--filter"              : call SetOptFilter(arg)
   case else                    : call Abort(2,"Error: unknown option "+opt+". See --help.")
  end select
  opt=""
 end if
next

'******* Task execution
'**********************

if (WScript.Arguments.Count=0) then : PrintHelp(GetBaseName(WScript.ScriptName)) : WScript.Quit(0) : end if
if not IsEmptyStr(AppPath) and not FileExists(AppPath) then AppPath="" end if
call Assert(AppPath<>"",1,"Error: specified application is not found.")
AppPath=GetLongFileName(GetAbsolutePathName(AppPath))
if optRun then
 dim cmdline : cmdline=DoubleQuotedIfNeed(AppPath)+Params : Writeln(cmdline)
 if not optTest then WScript.Quit(ShellRun(cmdline,optShow,optWait)) end if
else
 Writeln(AppPath)
end if
WScript.Quit(0)

'******* Done
'************
