////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020-2023 Alexey Kuryakin kouriakine@mail.ru - LGPL license. //
////////////////////////////////////////////////////////////////////////////////

unit dpSysUtils; // Diesel Pascal system utilities.

interface

uses dpCmdArgs,dpSystem,dpEasyIpc;

 // Application Folders
function  InitSpecFolders:Integer;             // Initialize special folders
function  InitAppFolders:Integer;              // Create local Data & Config
const     DefaultVendorName = 'DaqGroup';      // Default Vendor Name
function  VendorName:String;                   // Current Vendor Name
function  ApplicationName:String;              // Application Name
procedure SetVendorName(aVendor:String);       // Set new Vendor Name
procedure SetApplicationName(aName:String);    // Set new Application Name
function  AddVendorAppName(Dir:String):String; // Result is Dir/Vendor/AppName

 // Get Application Data & Config directories.
function GetAppDataDir(Global:Boolean; SubDir:Boolean):String;
function GetAppConfigDir(Global:Boolean; SubDir:Boolean):String;

 // Create directory tree (recursive).
function ForceDirectories(Dir:String):Boolean;

// Language support.
const lng_UNKNOWN = 0;
const lng_ENGLISH = 1;
const lng_RUSSIAN = 2;
var LanguageId : Integer = 0;
function RusEng(Rus,Eng:String):String;

// Bit operations.
function GetBitMask(BitNum:Integer):Integer;
function IsBit(Data,BitNum:Integer):Boolean;
function HasFlags(Mode,Flags:Integer):Boolean;

// Math operations.
function iMin(a,b:Integer):Integer;
function iMax(a,b:Integer):Integer;
function fMin(a,b:Double):Double;
function fMax(a,b:Double):Double;

// String routines.
function StrFetch(const S:String; i:Integer):Char;
function AddPathDelim(S:String):String;
function DropPathDelim(S:String):String;

// Text routines.
function  StrLineFetch(Lines:TStrings; i:Integer):String;
procedure StringsAppendFromStrings(List,Strings:TStrings);
procedure StringsAppendFromText(List:TStrings; Strings:String);

// Miscellaneous routines.
function AdjustBufferSize(Size:LongInt; Step:LongInt):LongInt;

 // Stream Seek Options.
const soFromBeginning = 0;
const soFromCurrent   = 1;
const soFromEnd       = 2;

// Environment variables.     // Unix      or Windows
function GetShell:String;     // $SHELL    or %ComSpec%
function GetComSpec:String;   // $SHELL    or %ComSpec%
function GetHostName:String;  // $HOSTNAME or %ComputerName%
function GetUserName:String;  // $USER     or %UserName%
function GetTempDir:String;   // $TMPDIR   or %Temp%
function GetHomeDir:String;   // $HOME     or %UserProfile%

// Processes.
function GetPidPriority(Pid:Cardinal):Integer;

// Files and directories.
function MakeDir(Dir:String):Boolean;
function FileWriteAsText(FileName,TextLines:String; ioMode:Cardinal):Boolean;
function FileReadAsText(FileName:String; ioMode:Cardinal):String;
function FileReadAsTextFromCmdArgsOption(Opt:String; ioMode:Cardinal):String;
const ioModeAsIs = $00000000; // Read as is, i.e. as default Utf8.
const ioModeAnsi = $00000001; // Read as Ansi, i.e.convert Utf8->Ansi.
const ioModeDrop = $20000000; // Drop file after read (use for TMP files).
const ioModeAuto = $40000000; // Auto drop TMP files, detect --ansi option.

// Default IPC channel from command line.
// call demo.lm9 --easyipc=.\test,TimeOut=0,LogsHistory=100
function defIpc:THandle;      // Default EasyIPC pipe reference.
function defIpcInit:Boolean;  // Default EasyIPC initialization.
function defIpcFree:Boolean;  // Default EasyIPC finalization.

// Default params read from file specified by command line.
// call demo.lm9 --ansi --params=c:\Crw32exe\Temp\params.tmp
function defParams:TStringList; // Default params as list.
function defParamsText:String;  // Default params as text.
function defParamsInit:Boolean; // Default params initialization.
function defParamsFree:Boolean; // Default params finalization.
var defParamsIoMode:Integer=0;  // Default params ioMode.

// Application activate.
function AppActivateByHWnd(Wnd:HWND):Boolean;
function AppActivateByTitle(aTitle:String):Boolean;
function AppActivateByClassTitle(aClass,aTitle:String):Boolean;
function AppActivateByPidClassTitle(Pid:Cardinal; aClass,aTitle:String):Boolean;

// DateTime routines.
function msecnow:Double;                    // Milliseconds since 1/1/0001
function DateTimeToMs(dt:TDateTime):Double; // DateTime to milliseconds
function MsToDateTime(ms:Double):TDateTime; // Milliseconds to datetime
const DateTimeDelta = 693593.0; // Days between 1/1/0001 and 12/30/1899
const MSecsPerDay   = 86400000; // Milliseconds per day = 24*60*60*1000

const
 QuoteMark = '"';

 // Find index of word s in list of string items.
 // For example, WordIndex('3','1 2 3 4 5',' ')=3.
function WordIndex(s,list,delims:String):Integer;

procedure NiceFormatByPattern(Lines:TStrings; Pattern:String; LineBeg:Integer=0; LineEnd:Integer=MaxInt);

implementation

uses dpLinux;

function InitSpecFolders:Integer;
var unix,cmd,txt,line,sn,sv:String; Lines:TStringList; i,p:Integer;
begin
 Result:=0;
 if IsWindows then unix:='unix.exe' else unix:='unix';
 if IsEmptyStr(FileSearch(unix,GetEnv('PATH'),false)) then Exit;
 cmd:='unix getspecialfolderpath';
 if IsWindows then cmd:='cmd /c unix -n getspecialfolderpath | unix -n recode cp'+IntToStr(WinApi.GetOEMCP)+'..utf-8';
 if RunCommand(cmd,txt) then
 try
  Lines:=TStringList.Create;
  try
   Lines.Text:=txt;
   for i:=0 to Lines.Count-1 do begin
    line:=Trim(Lines.Strings[i]);
    if (Pos('CSIDL_',line)<>1) then continue;
    p:=Pos('=',line); if (p=0) then continue;
    sn:=Trim(Copy(line,1,p-1)); sv:=Copy(line,p+1,MaxInt);
    if SetEnv(sn,sv) then Inc(Result);
   end;
  finally
   Lines.Free;
  end;
 except
  on E:Exception do BugReport(E,nil,'InitSpecFolders');
 end;
end;

function InitAppFolders:Integer;
begin
 Result:=0;
 if ForceDirectories(GetAppDataDir(false,true)) then Inc(Result);
 if ForceDirectories(GetAppConfigDir(false,true)) then Inc(Result);
end;

var
 TheApplicationName:String='';
 TheVendorName:String=DefaultVendorName;

procedure SetVendorName(aVendor:String);
begin
 TheVendorName:=Trim(aVendor);
end;

function VendorName:String;
begin
 Result:=TheVendorName;
end;

procedure SetApplicationName(aName:String);
begin
 TheApplicationName:=Trim(aName);
end;

function ApplicationName:String;
begin
 if (TheApplicationName='') then SetApplicationName(GetScriptBaseName);
 Result:=TheApplicationName;
end;

function AddVendorAppName(Dir:String):String;
var Vendor,AppName:String;
begin
 Result:=Trim(Dir);
 if (Result='') then Exit;
 Vendor:=Trim(VendorName);
 AppName:=Trim(ApplicationName);
 if IsUnix then Vendor:=LowerCase(Vendor);
 if IsUnix then AppName:=LowerCase(AppName);
 if (Vendor<>'') then Result:=AddPathDelim(Result)+Vendor;
 if (AppName<>'') then Result:=AddPathDelim(Result)+AppName;
end;

function GetAppDataDir(Global:Boolean; SubDir:Boolean):String;
begin
 Result:='';
 if IsUnix then begin
  if Global
  then Result:='/usr/share'
  else Result:=ExpEnv('$HOME/.local/share');
 end else
 if IsWindows then begin
  if Global then begin
   Result:=GetEnv('ProgramData');
   if (Result='') then Result:=GetEnv('CSIDL_COMMON_APPDATA');
   if (Result='') then Result:=ExpEnv('%AllUsersProfile%\Application Data');
  end else begin
   Result:=GetEnv('AppData');
   if (Result='') then Result:=GetEnv('CSIDL_APPDATA');
   if (Result='') then Result:=ExpEnv('%UserProfile%\Application Data');
  end;
 end;
 if not DirectoryExists(Result) then Result:='';
 if SubDir and (Result<>'') then Result:=AddVendorAppName(Result);
end;

function GetAppConfigDir(Global:Boolean; SubDir:Boolean):String;
begin
 Result:='';
 if IsUnix then begin
  if Global
  then Result:='/etc'
  else Result:=ExpEnv('$HOME/.config');
 end else
 if IsWindows then begin
  if Global then begin
   Result:=GetEnv('AllUsersProfile');
  end else begin
   Result:=GetEnv('UserProfile');
   if (Result='') then Result:=GetEnv('CSIDL_PROFILE');
  end;
 end;
 if not DirectoryExists(Result) then Result:='';
 if SubDir and (Result<>'') then Result:=AddVendorAppName(Result);
end;

function ForceDirectories(Dir:String):Boolean;
begin
 Result:=MakeDir(Dir);
end;

function RusEng(Rus,Eng:String):String;
begin
 Result:='';
 case LanguageId of
  lng_RUSSIAN: if (Rus<>'') then Result:=Rus else if (Eng<>'') then Result:=Eng;
  lng_ENGLISH: if (Eng<>'') then Result:=Eng else if (Rus<>'') then Result:=Rus;
  else         if (Eng<>'') then Result:=Eng else if (Rus<>'') then Result:=Rus;
 end;
end;

function iMin(a,b:Integer):Integer; begin if (a<b) then Result:=a else Result:=b; end;
function iMax(a,b:Integer):Integer; begin if (a>b) then Result:=a else Result:=b; end;
function fMin(a,b:Double):Double;   begin if (a<b) then Result:=a else Result:=b; end;
function fMax(a,b:Double):Double;   begin if (a>b) then Result:=a else Result:=b; end;

function GetBitMask(BitNum:Integer):Integer;
begin
 Result:=(1 shl BitNum);
end;

function IsBit(Data,BitNum:Integer):Boolean;
begin
 Result:=HasFlags(Data,GetBitMask(BitNum));
end;

function HasFlags(Mode,Flags:Integer):Boolean;
begin
 Result:=((Mode and Flags)<>0);
end;

function StrFetch(const S:String; i:Integer):Char;
begin
 if (i<1) or (i>Length(S)) then Result:=#0 else Result:=S[i];
end;

function AddPathDelim(S:String):String;
begin
 Result:=S;
 if (S='') then Exit;
 if (LastDelimiter(PathDelim,S)<Length(S)) then Result:=S+PathDelim;
end;

function DropPathDelim(S:String):String;
begin
 Result:=S;
 if (S='') then Exit;
 if (LastDelimiter(PathDelim,S)=Length(S)) then Result:=Copy(S,1,Length(S)-1);
end;

function StrLineFetch(Lines:TStrings; i:Integer):String;
begin
 Result:='';
 if (i<0) then Exit;
 if (Lines=nil) then Exit;
 if (i>=Lines.Count) then Exit;
 Result:=Lines.Strings[i];
end;

procedure StringsAppendFromStrings(List,Strings:TStrings);
var i:Integer;
begin
 if (List=nil) then Exit;
 if (Strings=nil) then Exit;
 try
  for i:=0 to Strings.Count-1 do
  List.Add(Strings.Strings[i]);
 except
  on E:Exception do BugReport(E,nil,'StringsAppendFromStrings');
 end;
end;

procedure StringsAppendFromText(List:TStrings; Strings:String);
var Temp:TStringList;
begin
 if (List=nil) then Exit;
 if (Strings='') then Exit;
 try
  Temp:=TStringList.Create;
  try
   Temp.Text:=Strings;
   StringsAppendFromStrings(List,Temp);
  finally
   Temp.Free;
  end;
 except
  on E:Exception do BugReport(E,nil,'StringsAppendFromText');
 end;
end;

function AdjustBufferSize(Size:LongInt; Step:LongInt):LongInt;
begin
 if Size<0 then Size:=0;
 if Step<1 then Step:=1;
 Result:=iMax(Size,((Size+Step-1) div Step)*Step);
end;

function GetComSpec:String;
begin
 Result:=GetShell;
end;

function GetShell:String;
begin
 Result:='';
 if IsWindows then begin
  Result:=GetEnv('ComSpec');
  if (Result='') or not FileExists(Result)
  then Result:=FileSearch('cmd.exe',GetEnv('PATH'),false);
  if (Result='') or not FileExists(Result) then Result:='cmd';
 end else
 if IsUnix then begin
  Result:=GetEnv('SHELL');
  if (Result='') or not FileExists(Result)
  then Result:=FileSearch('bash',GetEnv('PATH'),false);
  if (Result='') or not FileExists(Result) then Result:='/bin/sh';
 end;
end;

function GetHostName:String;
var p,e:Integer;
begin
 Result:='';
 if IsUnix then Result:=Trim(execute_command_as_text('hostname',p,e)) else
 if IsWindows then Result:=AnsiLowerCase(Trim(GetEnv('ComputerName')));
end;

function GetUserName:String;
begin
 Result:='';
 if IsUnix then Result:=GetEnv('USER') else
 if IsWindows then Result:=GetEnv('UserName');
end;

function GetTempDir:String;
begin
 Result:='';
 if IsUnix then Result:=GetEnv('TMPDIR') else
 if IsWindows then Result:=GetEnv('Temp');
end;

function GetHomeDir:String;
begin
 Result:='';
 if IsUnix then Result:=GetEnv('HOME') else
 if IsWindows then Result:=GetEnv('UserProfile');
end;

function GetPidPriority(Pid:Cardinal):Integer;
begin
 if (Pid=0) then Pid:=GetProcessId;
 Result:=StrToIntDef(ExtractWord(3,GetListOfProcesses(Pid,0,''),', '),0)
end;

function MakeDir(Dir:String):Boolean;
var i:Integer;
begin
 Result:=false;
 Dir:=Trim(Dir);
 if (Dir<>'') then
 try
  for i:=1 to Length(Dir) do if Dir[i] in ['\','/'] then
  if not DirectoryExists(Copy(Dir,1,i-1)) then
  if not CreateDir(Copy(Dir,1,i-1)) then break;
  if not DirectoryExists(Dir) then CreateDir(Dir);
  Result:=DirectoryExists(Dir);
 except
  on E:Exception do begin
   BugReport(E,nil,'MakeDir');
   Result:=false;
  end;
 end;
end;

function FileWriteAsText(FileName,TextLines:String; ioMode:Cardinal):Boolean;
var List:TStringList;
begin
 Result:=false;
 if (FileName<>'') and (TextLines<>'') then
 try
  List:=TStringList.Create;
  try
   List.Text:=TextLines;
   if HasFlags(ioMode,ioModeAnsi) then List.ConvertUTF8ToAnsi;
   List.SaveToFile(FileName);
   Result:=true;
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'FileWriteAsText');
   Result:=false;
  end;
 end;
end;

function FileReadAsText(FileName:String; ioMode:Cardinal):String;
var List:TStringList;
begin
 Result:='';
 if (FileName<>'') and FileExists(FileName) then
 try
  List:=TStringList.Create;
  try
   List.LoadFromFile(FileName);
   if HasFlags(ioMode,ioModeAnsi) then List.ConvertAnsiToUTF8;
   if HasFlags(ioMode,ioModeDrop) then DeleteFile(FileName);
   Result:=List.Text;
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,nil,'FileReadAsText');
   Result:='';
  end;
 end;
end;

function FileReadAsTextFromCmdArgsOption(Opt:String; ioMode:Cardinal):String;
var FileName:String;
begin
 Result:='';
 if (Opt='') then Exit;
 if not CmdArgs.HasOption(Opt) then Exit;
 FileName:=CmdArgs.GetOptionValue(Opt); if (FileName='') then Exit;
 FileName:=ExpandFileName(FileName); if not FileExists(FileName) then Exit;
 if HasFlags(ioMode,ioModeAuto) then begin
  if SameText(ExtractFileExt(FileName),'.TMP') then ioMode:=ioMode or ioModeDrop;
  if (Pos('ansi',Lowercase(Opt))>0) then ioMode:=ioMode or ioModeAnsi;
  if CmdArgs.HasOption('--ansi') then ioMode:=ioMode or ioModeAnsi;
 end;
 Result:=FileReadAsText(FileName,ioMode);
end;

var theIpc:THandle=0;

function defIpc:THandle;
begin
 Result:=theIpc;
end;

function defIpcInit:Boolean;
begin
 if (theIpc=0) then theIpc:=EasyIpc_InitFromCmdArgs('--easyipc');
 Result:=EasyIpc_Valid(theIpc);
end;

function defIpcFree:Boolean;
begin
 Result:=true;
 if (theIpc<>0) then Result:=EasyIpc_Free(theIpc);
 theIpc:=0;
end;

var theParamsText:String='';
var theParams:TStringList=nil;

function defParams:TStringList;
begin
 if (theParams=nil) then theParams:=TStringList.Create;
 Result:=theParams;
end;

function defParamsText:String;
begin
 Result:=theParamsText;
end;

function defParamsInit:Boolean;
begin
 theParamsText:=FileReadAsTextFromCmdArgsOption('--params',defParamsIoMode)
               +FileReadAsTextFromCmdArgsOption('--params-utf8',defParamsIoMode)
               +FileReadAsTextFromCmdArgsOption('--params-ansi',defParamsIoMode);
 defParams.Text:=theParamsText;
 Result:=(theParamsText<>'');
end;

function defParamsFree:Boolean;
begin
 if (theParams<>nil) then theParams.Free; theParams:=nil;
 theParamsText:='';
 Result:=true;
end;

function AppActivateByHWnd(Wnd:HWND):Boolean;
begin
 Result:=false;
 if not IsWindows then Exit;
 if (Wnd=0) or not WinApi.IsWindow(Wnd) then Exit;
 if not WinApi.ShowWindow(Wnd,SW_SHOW) then Exit;
 if not WinApi.BringWindowToTop(Wnd) then Exit;
 if not WinApi.SetForegroundWindow(Wnd) then Exit;
 Result:=true;
end;

function AppActivateByTitle(aTitle:String):Boolean;
begin
 Result:=false; if (aTitle='') then Exit;
 if IsWindows then Result:=AppActivateByHWnd(WinApi.FindWindow('',aTitle));
 if IsUnix then begin
  execute_wmctrl(FormatVar('-a "%s" -F',aTitle));
  Result:=Pos(aTitle,execute_wmctrl('-l'))>0;
 end;
end;

function AppActivateByClassTitle(aClass,aTitle:String):Boolean;
begin
 Result:=false; if (aTitle='') then Exit;
 if IsWindows then Result:=AppActivateByHWnd(WinApi.FindWindow(aClass,aTitle));
 if IsUnix then begin
  execute_wmctrl(FormatVar('-a "%s" -F',aTitle));
  Result:=Pos(aTitle,execute_wmctrl('-l'))>0;
 end;
end;

function AppActivateByPidClassTitle(Pid:Cardinal; aClass,aTitle:String):Boolean;
begin
 Result:=false;
 if (aTitle='') then Exit; if not IsWindows then Exit;
 Result:=AppActivateByHWnd(StrToIntDef(ExtractWord(1,GetListOfWindows(Pid,aClass,aTitle),', '),0));
end;

function DateTimeToMs(dt:TDateTime):Double;
begin
 Result:=(DateTimeDelta+dt)*MSecsPerDay;
end;

function MsToDateTime(ms:Double):TDateTime;
begin
 Result:=ms/MSecsPerDay-DateTimeDelta;
end;

function msecnow:Double;
begin
 Result:=DateTimeToMs(Now);
end;

function WordIndex(s,list,delims:String):Integer;
var i,j,n:Integer;
begin
 i:=1;
 j:=0;
 n:=WordCount(list,delims);
 while i<=n do begin
  if SameText(s,ExtractWord(i,list,delims)) then begin
   j:=i;
   i:=n;
  end;
  i:=i+1;
 end;
 WordIndex:=j;
end;

procedure NiceFormatByPattern(Lines:TStrings; Pattern:String; LineBeg:Integer=0; LineEnd:Integer=MaxInt);
var i,p,m:Integer;
begin
 if (Lines<>nil) and (pattern<>'') then
 try
  m:=0;
  for i:=0 to Lines.Count-1 do begin
   if (i<LineBeg) then Continue;
   if (i>LineEnd) then Continue;
   p:=Pos(pattern,Lines.Strings[i]);
   m:=imax(m,p);
  end;
  for i:=0 to Lines.Count-1 do begin
   if (i<LineBeg) then Continue;
   if (i>LineEnd) then Continue;
   p:=Pos(pattern,Lines.Strings[i]);
   if (p>0) and (m-p>0)
   then Lines.Strings[i]:=Copy(Lines.Strings[i],1,p-1)+StringOfChar(' ',m-p)+Copy(Lines.Strings[i],p,MaxInt);
  end;
 except
  on E:Exception do BugReport(E,nil,'NiceFormatByPattern');
 end;
end;

initialization

 // InitAppFolders;

end.
