////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpDaqConfig; // Diesel Pascal DAQ system config files reader.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpVbs,dpCrwDaqSys;

type
 TDaqConfig = class(TObject) // Config files reader
 protected
  FRecursion  : Integer;
  FTotalList  : TStringList;
 public
  constructor Create;
  destructor  Destroy;
 public
  // Drop comment after ; char
  function DropComment(S:String):String;
  // Detect 0:not a section, 1:any section, 2:given section
  function DetectSectionHeader(S,Section:String):Integer;
  // Get variable value from Line contains 'Name = Value' expression
  function GetVarValue(Line,Name:String; var Value:String):Boolean;
  // Extract section from List/Text/File with given mode
  // Extraction from file uses [ConfigFileList] features
  function ExtractSectionFromList(List:TStringList; Section:String; efMode:Integer):String;
  function ExtractSectionFromText(TextLines:String; Section:String; efMode:Integer):String;
  function ExtractSectionFromFile(FileName:String; Section:String; efMode:Integer):String;
 end;

function DaqConfig:TDaqConfig; // General TDaqConfig class instance

const efDrop     = 1;    // flag for efMode - Drop comments
const efTrim     = 2;    // flag for efMode - Trim
const efLower    = 4;    // flag for efMode - Lowercase
const efUpper    = 8;    // flag for efMode - Uppercase
const efAsAnsi   = 16;   // flag for efMode - I/O as Ansi format
const efConfig   = efDrop+efTrim+efAsAnsi+efUpper;  // Default for configs
const efConfigNC = efDrop+efTrim+efAsAnsi;          // Config with case sense

implementation

var TheDaqConfig : TDaqConfig = nil;

function DaqConfig:TDaqConfig;
begin
 if (TheDaqConfig=nil) then TheDaqConfig:=TDaqConfig.Create;
 Result:=TheDaqConfig;
end;

constructor TDaqConfig.Create;
begin
 inherited Create;
 FRecursion:=0;
 FTotalList:=TStringList.Create;
 if (TheDaqConfig=nil) then TheDaqConfig:=Self;
end;

destructor TDaqConfig.Destroy;
begin
 if (TheDaqConfig=Self) then TheDaqConfig:=nil;
 FTotalList.Free;
 FRecursion:=0;
 inherited Destroy;
end;

function TDaqConfig.DropComment(S:String):String;
var p:Integer;
begin
 p:=Pos(';',S);
 if (p>0) then S:=Copy(S,1,p-1);
 Result:=S;
end;

function TDaqConfig.DetectSectionHeader(S,Section:String):Integer;
begin
 Result:=0; // It' not section header
 S:=Trim(DropComment(S)); if (S='') then Exit;
 if (S[1]<>'[') then Exit; if (S[Length(S)]<>']') then Exit;
 Result:=1; // It's some section header found
 if (Section<>'') and not SameText(S,Section) then Exit;
 Result:=2; // It's section specified found
end;

function TDaqConfig.GetVarValue(Line,Name:String; var Value:String):Boolean;
var p:Integer;
begin
 Result:=false; Value:='';
 Line:=Trim(Line); Name:=Trim(Name);
 if (Line='') or (Name='') then Exit;
 p:=Pos('=',Line); if (p<2) then Exit;
 if not SameText(Trim(Copy(Line,1,p-1)),Name) then Exit;
 Value:=Trim(Copy(Line,p+1,Length(Line)-p));
 Result:=true;
end;

function TDaqConfig.ExtractSectionFromList(List:TStringList; Section:String; efMode:Integer):String;
var i:Integer; Temp:TStringList; InSection:Boolean; S:String;
begin
 Result:='';
 Section:=Trim(Section);
 if (List<>nil) and (List.Count>0) then
 if (DetectSectionHeader(Section,'')>0) then
 try
  Temp:=TStringList.Create;
  try
   InSection:=false;
   for i:=0 to List.Count-1 do begin
    S:=List.Strings[i];
    case DetectSectionHeader(S,Section) of
     0: if InSection then begin
         if ((efMode and efDrop)  <> 0) then S:=DropComment(S);
         if ((efMode and efTrim)  <> 0) then S:=Trim(S);
         if ((efMode and efLower) <> 0) then S:=Lowercase(S);
         if ((efMode and efUpper) <> 0) then S:=Uppercase(S);
         if (S<>'') then Temp.Add(S);
        end;
     1: InSection:=false;
     2: InSection:=true;
    end;
   end;
   Result:=Temp.Text;
  finally
   Temp.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromList');
   Result:='';
  end;
 end;
end;

function TDaqConfig.ExtractSectionFromText(TextLines:String; Section:String; efMode:Integer):String;
var List:TStringList;
begin
 Result:='';
 if (TextLines<>'') and (Section<>'') then
 try
  List:=TStringList.Create;
  try
   List.Text:=TextLines;
   Result:=ExtractSectionFromList(List,Section,efMode);
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromText');
   Result:='';
  end;
 end;
end;

function TDaqConfig.ExtractSectionFromFile(FileName:String; Section:String; efMode:Integer):String;
var Lines,Sect:String; List:TStringList; i,ioMode:Integer; S,FN,UserProfile,HomeDir:String;
begin
 Result:='';
 Section:=Trim(Section);
 FileName:=Trim(FileName);
 if (DetectSectionHeader(Section,'')>0) then
 if (FileName<>'') and FileExists(FileName) then
 try
  ioMode:=ioModeAsIs;
  if (FRecursion=0) then FTotalList.Clear;
  if ((efMode and efAsAnsi)<>0) then ioMode:=ioMode or ioModeAnsi;
  Lines:=FileReadAsText(FileName,ioMode);
  Result:=ExtractSectionFromText(Lines,Section,efMode);
  Sect:=ExtractSectionFromText(Lines,'[ConfigFileList]',efConfigNC);
  List:=TStringList.Create;
  try
   List.Text:=Sect;
   for i:=0 to List.Count-1 do begin
    if GetVarValue(List.Strings[i],'ConfigFile',S) then begin
     FN:=Trim(S);
     if (FN<>'') then
     if (Pos(':\',FN)=0) then
     if (Pos('\\',FN)<>1) then begin
      if (Pos('~\',FN)=1) then begin
       UserProfile:=GetEnv('USERPROFILE');
       if (UserProfile<>'') and DirectoryExists(UserProfile)
       then FN:=AddPathDelim(UserProfile)+Copy(FN,2,Length(FN)-1)
       else FN:='';
      end else
      if (Pos('~~\',FN)=1) then begin
       HomeDir:=GetEnv('CRW_DAQ_SYS_HOME_DIR');
       if (HomeDir<>'') and DirectoryExists(HomeDir)
       then FN:=AddPathDelim(HomeDir)+Copy(FN,3,Length(FN)-2)
       else FN:='';
      end else
      FN:=AddPathDelim(ExtractFileDir(FileName))+FN;
     end;
     if (FN<>'') and FileExists(FN) then
     if (FTotalList.IndexOf(FN)<0) then begin
      FTotalList.Add(FN);
      try
       Inc(FRecursion);
       Result:=Result+ExtractSectionFromFile(FN,Section,efMode);
      finally
       Dec(FRecursion);
       if FTotalList.IndexOf(FN)>=0 then
       FTotalList.Delete(FTotalList.IndexOf(FN));
      end;
     end;
    end;
   end;
  finally
   List.Free;
  end;
 except
  on E:Exception do begin
   BugReport(E,Self,'ExtractSectionFromFile');
   Result:='';
  end;
 end;
end;

initialization

 DaqConfig;

end.
