////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Simple IPC client to send string messages via TSimpleIpcClient.            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231018 - Created by A.K.                                                 //
////////////////////////////////////////////////////////////////////////////////

program simpleipcsend;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, forms, Interfaces,
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_fio, _crw_sesman;

{$R *.res}

procedure PrintVersion;
var ver:LongString;
begin
 ver:=GetFileVersionInfoAsText(ProgName); if IsEmptyStr(ver) then Exit;
 writeln(CookieScan(ver,'ProductName'),' version ',CookieScan(ver,'ProductVersion'));
end;

procedure PrintHelp;
var ver,bn:LongString;
begin
 ver:=GetFileVersionInfoAsText(ProgName); if IsEmptyStr(ver) then Exit;
 writeln(CookieScan(ver,'ProductName'),' version ',CookieScan(ver,'ProductVersion'));
 writeln(CookieScan(ver,'FileDescription'));
 writeln(CookieScan(ver,'LegalCopyright'));
 writeln(CookieScan(ver,'Comments'));
 bn:=ExtractBaseName(ProgName);
 writeln('Usage:');
 writeln(' ',bn,' [-opt] ServerID ["Message1" ["Message2"] ...]');
 writeln(' where:');
 writeln('  -opt     - options {-h,-help;-version}.');
 writeln('  ServerID - mandatory IPC server identifier');
 writeln('  Message1 - Message 1 to send to IPC server');
 writeln('  Message2 - Message 2 to send, etc');
 writeln('Note:');
 writeln('  Messages may be passed via command line,');
 writeln('  or also may comes from StdIn pipe.');
 writeln('Exit code:');
 writeln('  0 - succeed');
 writeln('  1 - error');
 writeln('Example:');
 writeln('  ',bn,' -help');
 writeln('  ',bn,' ipc_daqgroup_crwdaq_pid_4012 "@Test Command"');
 writeln('  echo "@Test Command" | ',bn,' ipc_daqgroup_crwdaq_pid_4012');
end;

procedure Main;
var ServerId,Msg,Line:LongString; i,nsent,nfail,iores:Integer;
begin
 try
  if CmdArgs.HasOption('-h')
  or CmdArgs.HasOption('-help')
  or CmdArgs.HasOption('--help') then begin
   PrintHelp;
   Exit;
  end;
  if CmdArgs.HasOption('-version')
  or CmdArgs.HasOption('--version') then begin
   PrintVersion;
   Exit;
  end;
  ServerID:=CmdArgs.GetParam(1);
  if IsEmptyStr(ServerID) then begin
   writeln(StdErr,'Error: ServerID is not specified.');
   ExitCode:=1;
   Exit;
  end;
  nsent:=0; nfail:=0; line:='';
  for i:=2 to CmdArgs.NumParams do begin
   Msg:=CmdArgs.GetParam(i);
   if SessionManager.SimpleIpcSendMessage(ServerID,Msg)
   then Inc(nsent) else Inc(nfail);
  end;
  // Pipe StdIn handling
  if (GetFileType(Input)=FILE_TYPE_PIPE) then begin
   while not Eof(Input) do begin
    Readln(Input,Line); iores:=IoResult;
    if (iores=0) then Msg:=Line else begin
     writeln(StdErr,'Error: '+IoResultToString(iores));
     ExitCode:=1;
     Break;
    end;
    if SessionManager.SimpleIpcSendMessage(ServerID,Msg) then begin
     writeln(StdOut,'Sent: '+Msg);
     Inc(nsent);
    end else begin
     writeln(StdErr,'Fail: '+Msg);
     Inc(nfail);
    end;
   end;
  end;
  if (nsent>0) then writeln(StdErr,nsent,' message(s) sent');
  if (nfail>0) then writeln(StdErr,nfail,' message(s) fail');
  if (nsent=0) then ExitCode:=1;
  if (nfail>0) then ExitCode:=1;
 except
  on E:Exception do BugReport(E,nil,'Main');
 end;
end;

begin
 Main;
end.

//////////////
// END OF FILE
//////////////

