#!/bin/sh

#######################################################
# Copyright (c) 2023 Alexey Kuryakin kouriakine@mail.ru
# Utiluty to set default file handler for Diesel Pascal
# files (*.lm9) + to set default Designer IDE settings.
#######################################################

notify(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)";
 local name="$(basename ${BASH_SOURCE[0]}).sh";
 local icon="/usr/share/pixmaps/CrossDesigner.png";
 notify-send -i CrossDesigner -t 15000 -i $icon "$date: $name" "$1";
};

extract_diesel_settings(){
 local dir="/opt/diesel/settings";
 zip="$dir/diesel-defaults.zip";
 echo "Extract $zip";
 if 7za x -y -o$HOME $zip; then
  notify "Начальные настройки Diesel Pascal установлены.";
 else
  zenity --error --text "Сбой устновки настроек Diesel Pascal." >/dev/null 2>&1;
 fi;
};

ask_extract_diesel_settings(){
 if zenity --question --text "Задать начальные настройки\nдля Diesel Pascal Designer IDE?" >/dev/null 2>&1; then
  extract_diesel_settings;
 fi;
};

see_default_diesel(){
 local app="$(xdg-mime query default application/x-diesel)";
 if [ -n "$(echo "$app" | grep '.desktop')" ]; then
  notify "Для Diesel Pascal задан обработчик файлов $app.";
 else
  zenity --error --text "Сбой задания обработчика файлов Diesel Pascal." >/dev/null 2>&1;
 fi;
};

set_default_diesel(){
 local mime="application/x-diesel";
 local app="$(zenity --title 'Обработчик Diesel Pascal' --text 'Обработчик файлов для <b>Diesel Pascal</b>' --list --column 'Программа обработки' 'CrossMachine.desktop' 'CrossDesigner.desktop' 2>/dev/null)";
 [ -z "$app" ] && { return 0; };
 if which xdg-mime >/dev/null 2>&1; then
  xdg-mime default $app $mime;
  see_default_diesel;
  return;
 fi;
 if which gio >/dev/null 2>&1; then
  gio mime $mime $app;
  return;
 fi;
};

main(){
 set_default_diesel;
 ask_extract_diesel_settings;
};

main "$@";

#############
# END OF FILE
#############
