unit SurCodeMap;

//{$mode delphi}
{$mode objfpc}{$H+}

interface

uses
  SysUtils, Classes, dpUtils, dpi_parser;

type
  TSuObject = class;

  TSuObjectList = class(TObjectList)
  private
  protected
    function GetObjects(Index: Integer): TSuObject;
  public
    property Objects[Index: Integer]: TSuObject read GetObjects;
  end;

  { TSuObject }

  TSuObject = class
  private
    FName: string;
    FOriginalName: string;
    FHashValue: Cardinal;
    FDeclUnit: string;
    FCol: Integer;
    FEndCol: Integer;
    FRow: Integer;
    FValueType: string;
    FObjType: TsuIdentType;
    FValue: String;
    FVars: TSuObjectList;
  protected
    procedure SetName(Value: string);
  public
    constructor Create;
    destructor Destroy; override;
    property Name: string read FName write SetName;
    property OriginalName: string read FOriginalName;
    property HashValue: Cardinal read FHashValue;
    property Col: Integer read FCol;
    property Row: Integer read FRow;
    property EndCol: Integer read FEndCol;
    property ValueType: string read FValueType;
    property ObjType: TsuIdentType read FObjType;
    property DeclUnit: string read FDeclUnit write FDeclUnit;
    property Vars: TSuObjectList read FVars;
  end;

  TSuProcParam = class(TSuObject)
  public
    AsVar: Boolean;
    AsConst: Boolean;
    AsOut: Boolean;
  end;

  TSuCodeMap = class;

  TSuObjectClass = class(TSuObject)
  private
    FObjects: TList;
    FMethods: TSuObjectList;
    FObjectMethods: TList;
    FCodeMap: TSuCodeMap;
    FInheritsFrom: string;
  protected
  public
    constructor Create(ACodeMap: TSuCodeMap);
    destructor Destroy; override;
    procedure GetPositionForClassMethod(var P: TPoint; var CodeRow: Integer);
    procedure GetPositionForComponent(var P: TPoint);
    function GetCompoenentPos(ACompName: string; var P: TPoint): Boolean;
    function ComponentExist(AComponentName: string): Boolean;
    function Find(AObjName: string): TSuObject;
    property Methods: TSuObjectList read FMethods;
    property InhFrom: string read FInheritsFrom;
  end;

  { TSuProcItem }

  TSuProcItem = class(TSuObject)
  private
    FUnits: TStringList;
    FClasses: TSuObjectList;
    FTyps: TSuObjectList;
    FParams: TList;
    FProcs: TSuObjectList;
    FProc: TSuProcItem;
    FClassOwner: TSuObjectClass;
    FIsFunction: Boolean;

    FBodyRowBegin: Integer;
    FBodyColBegin: Integer;
    FBodyRowEnd: Integer;
    FMethodDef: string;
    function GetProcParam(Index: Integer): TSuProcParam;

  protected
    procedure AddVar(AVar: TSuObject);
    procedure AddParam(AParam: TSuObject);
    procedure AddType(AType: TSuObject);
    procedure Addproc(AProc: TSuProcItem);
    function GetParamCount: Integer;
  public
    DeclPos: TPoint;
    ImplPos: TPoint;
    LastViewProc: TSuProcItem;
    constructor Create;
    destructor Destroy; override;
    procedure GetUnitList(SL: TStrings);
    function FindClass(AClassName: string): TSuObjectClass;
    function FindClassMethod(AClassName, AMethodName: string): TSuProcItem;
    function FindObject(AObjName: string): TSuObject;
    function GetProcOfPos(X, Y: Integer; out IsDecl: Boolean): TSuProcItem;
    property Proc: TSuProcItem read FProc write FProc;
    property ClassOwner: TSuObjectClass read FClassOwner;
    property IsFunction: Boolean read FIsFunction;
    property BodyRowBegin: Integer read FBodyRowBegin;
    property BodyColBegin: Integer read FBodyColBegin;
    property BodyRowEnd: Integer read FBodyRowEnd;
    property Types: TSuObjectList read FTyps;
    property Classes: TSuObjectList read FClasses;
    property Procs: TSuObjectList read FProcs;
    property Params: TList read FParams;
    property ParamCount: Integer read GetParamCount;
    property Param[Index: Integer]: TSuProcParam read GetProcParam;
    property MethodDef: string read FMethodDef write FMethodDef;
  end;

  { TSuCodeMap }

  TSuCodeMap = class
  private
    FTokens: TMemoTokens;
    FMainProc: TSuProcItem;
  protected
    IsImplementation: Boolean;
    procedure DoProgram(Proc: TSuProcItem);
    function DoBlock(Proc: TSuProcItem): Boolean;
    function DoUses(Proc: TSuProcItem): Boolean;
    function DoType(Proc: TSuProcItem): Boolean;
    function DoProcParam(Proc: TSuProcItem): Boolean;
    function DoCheckParam(Proc: TSuProcItem): Boolean;
    function DoConst(Proc: TSuProcItem): Boolean;
    function DoVar(Proc: TSuProcItem): Boolean;
    function DoSetDeclaration(var MinVal, MaxVal, RefType: string;
      Proc: TSuProcItem): Boolean;
    function DoArray(Proc: TSuProcItem; var AValueType: string): Boolean;
    function DoMethod(Proc: TSuProcItem): Boolean;
    function DoCompoundStat(Proc: TSuProcItem): Boolean;
    function DoEnumConst(Proc: TSuProcItem; aType: TSuObject): Boolean;
    procedure TokenPosToSuObject(AObject: TSuObject);
  public
    destructor Destroy; override;
    function GetCodeMap(ATokens: TMemoTokens): TSuProcItem;
    function GetNewPosition(ATokens: TMemoTokens): Integer;
    property Tokens: TMemoTokens read FTokens;
    property MainProc: TSuProcItem read FMainProc;
  end;

implementation

uses LazUTF8;

{TSuObjectList}
function TSuObjectList.GetObjects(Index: Integer): TSuObject;
begin
  Result := TSuObject(Items[Index]);
end;

{TSuObject}

constructor TSuObject.Create;
begin
  inherited Create;
  FHashValue := 0;
  FName := '';
  FOriginalName := '';
  FVars := TSuObjectList.Create;
end;

destructor TSuObject.Destroy;
begin
  FVars.Free;
  inherited Destroy;
end;

procedure TSuObject.SetName(Value: string);
begin
  FOriginalName := Value;
  FName := UTF8LowerCase(Value);
  FHashValue := Hash(FName);
end;

{TSuObjectClass}

constructor TSuObjectClass.Create(ACodeMap: TSuCodeMap);
begin
  inherited Create;
  FCodeMap := ACodeMap;
  FObjects := TList.Create;
  FMethods := TSuObjectList.Create;
  FObjectMethods := TList.Create;
end;

destructor TSuObjectClass.Destroy;
begin
  FMethods.Free;
  FObjects.Free;
  FObjectMethods.Free;
  inherited Destroy;
end;

procedure TSuObjectClass.GetPositionForClassMethod(var P: TPoint; var CodeRow: Integer);
var
  I: Integer;
  PName: string;
begin
  CodeRow := FCodeMap.FMainProc.BodyRowBegin;
  if FObjectMethods.Count = 0 then
  begin
    GetPositionForComponent(P);
  end
  else
  begin
    for I := 0 to FObjectMethods.Count - 1 do
    begin
      PName:= TSuProcItem(FObjectMethods.Items[I]).Name;

      P.X := TSuObject(FObjectMethods.Items[I]).Col;
      P.Y := TSuObject(FObjectMethods.Items[I]).Row + 1;
      if TSuProcItem(FObjectMethods.Items[I]).BodyRowEnd > 0 then
        CodeRow := TSuProcItem(FObjectMethods.Items[I]).BodyRowEnd + 2;
    end;
  end;
end;

procedure TSuObjectClass.GetPositionForComponent(var P: TPoint);
var
  I: Integer;
begin
  P.X := Col + 2;
  P.Y := Row + 1;
  for I := 0 to FObjects.Count - 1 do
  begin
    P.X := TSuObject(FObjects.Items[I]).Col;
    P.Y := TSuObject(FObjects.Items[I]).Row + 1;
  end;
end;

function TSuObjectClass.GetCompoenentPos(ACompName: string; var P: TPoint): Boolean;
var
  I: Integer;
  LCCompName: string;
begin
  Result := False;
  LCCompName:= UTF8LowerCase(ACompName);
  for I := 0 to FObjects.Count - 1 do
  begin
    if TSuObject(FObjects.Items[I]).Name = LCCompName then
    begin
      Result := True;
      P.X := TSuObject(FObjects.Items[I]).Col;
      P.Y := TSuObject(FObjects.Items[I]).Row;
    end;
  end;
end;

function TSuObjectClass.ComponentExist(AComponentName: string): Boolean;
var
  I: Integer;
  LCCOmponentName: string;
begin
  Result := False;
  LCCOmponentName:= UTF8LowerCase(AComponentName);
  for I := 0 to FObjects.Count - 1 do
  begin
    if TSuObject(FObjects.Items[I]).FName = AComponentName then
    begin
      Result := True;
      Exit;
    end;
  end;
end;

function TSuObjectClass.Find(AObjName: string): TSuObject;
var
  I: Integer;
  H: Cardinal;
begin
  AObjName := UTF8LowerCase(AObjName);
  H := Hash(AObjName);
  Result := nil;
  for I := 0 to FVars.Count - 1 do
  begin
    if (TSuObject(FVars.Items[I]).HashValue = H) and
     (TSuObject(FVars.Items[I]).Name = AObjName) then
    begin
      Result := TSuObject(FVars.Items[I]);
      Exit;
    end;
  end;
  for I := 0 to FMethods.Count - 1 do
  begin
    if (TSuObject(FMethods.Items[I]).HashValue = H) and
     (TSuObject(FMethods.Items[I]).Name = AObjName) then
    begin
      Result := TSuObject(FMethods.Items[I]);
      Break;
    end;
  end;
end;

{TSuProcItem}

constructor TSuProcItem.Create;
begin
  inherited Create;
  FUnits := TStringList.Create;
  FTyps := TSuObjectList.Create;
  FProcs := TSuObjectList.Create;
  FClasses := TSuObjectList.Create;
  FParams := TList.Create;
  FObjType := itProc;
  DeclPos.x := 0;
  DeclPos.y := 0;
  ImplPos.y := 0;
  ImplPos.x := 0;
  FMethodDef := '';
end;

destructor TSuProcItem.Destroy;
begin
  FUnits.Free;
  FTyps.Free;
  FProcs.Free;
  FClasses.Free;
  FParams.Free;
  inherited Destroy;
end;

function TSuProcItem.GetParamCount: Integer;
begin
  Result := FParams.Count;
end;

function TSuProcItem.GetProcParam(Index: Integer): TSuProcParam;
begin
  Result := TSuProcParam(Params.Items[Index]);
end;

procedure TSuProcItem.AddVar(AVar: TSuObject);
begin
  Vars.Add(AVar);
end;

procedure TSuProcItem.AddParam(AParam: TSuObject);
begin
  AddVar(AParam);
  FParams.Add(AParam);
end;

procedure TSuProcItem.AddType(AType: TSuObject);
begin
  FTyps.Add(AType);
end;

procedure TSuProcItem.Addproc(AProc: TSuProcItem);
begin
  if FClassOwner <> nil then AProc.FClassOwner := FClassOwner;
  FProcs.Add(AProc);
end;

procedure TSuProcItem.GetUnitList(SL: TStrings);
begin
  SL.Assign(FUnits);
end;

function TSuProcItem.FindClass(AClassName: string): TSuObjectClass;
var
  I: Integer;
  H: Cardinal;
  S: string;
begin
  Result := nil;
  AClassName := UTF8LowerCase(AClassName);
  H := Hash(AClassName);
  for I := 0 to FClasses.Count - 1 do
  begin
    if (TSuObjectClass(FClasses.Items[I]).HashValue = H) and
      (TSuObjectClass(FClasses.Items[I]).Name = AClassName) then
    begin
      Result := TSuObjectClass(FClasses.Items[I]);
      Break;
    end;
  end;
end;

function TSuProcItem.FindObject(AObjName: string): TSuObject;
var
  I: Integer;
  HashVal: Cardinal;
begin
  Result := nil;
  AObjName := UTF8LowerCase(AObjName);
  HashVal := Hash(AObjName);
  for I := 0 to FVars.Count - 1 do
  begin
    if (TSuObject(Vars.Items[I]).HashValue = HashVal) and
      (TSuObject(Vars.Items[I]).Name = AObjName) then
    begin
      Result := TSuObject(Vars.Items[I]);
      Break;
    end;
  end;
  if Result = nil then
  begin
    for I := 0 to FProcs.Count - 1 do
    begin
      if (TSuObject(FProcs.Items[I]).HashValue = HashVal) and
        (TSuObject(FProcs.Items[I]).Name = AObjName) then
      begin
        Result := TSuObject(FProcs.Items[I]);
        Break;
      end;
    end;
  end;
  //if Result = nil then
  //  if ((Proc <> nil) and (Proc.Proc <> Self)) then Result := Proc.FindObject(AObjName);
end;

function TSuProcItem.GetProcOfPos(X, Y: Integer; out IsDecl: Boolean
  ): TSuProcItem;
var
  I, N, M: Integer;
  C: TSuObjectClass;

  function GetProcOfProc(AProc: TSuProcItem): TSuProcItem;
  var
    ResProc: TSuProcItem;
    ChP: Integer;
  begin
    Result := nil;
    if AProc.Procs.Count > 0 then
    begin
      for ChP := 0 to Proc.Procs.Count - 1 do
      begin
        ResProc := TSuProcItem(AProc.Procs.Items[ChP] );
        if (ResProc.ImplPos.y <= Y) and
           (ResProc.BodyRowEnd >= Y) then
        begin
          Result := ResProc;
          Exit;
        end;
      end;
    end;
  end;

begin
  Result := nil;

  IsDecl := False;
  LastViewProc := nil;
  for I := 0 to FProcs.Count - 1 do
  begin
    Proc := TSuProcItem(FProcs.Items[I]);
    if (Proc.ImplPos.y <= Y) and
      (Proc.BodyRowEnd >= Y) then
    begin
      Result := Proc;
      Exit;
    end
    else
    if (Proc.DeclPos.Y = Y) and (Proc.DeclPos.x <= X) then
    begin
      M := Proc.DeclPos.x + Length(Proc.OriginalName);
      if M >= X then
      begin
        IsDecl := True;
        Result := Proc;
        Proc := GetProcOfProc(Proc);
        if Proc <> nil then Result := Proc;;
        Exit;
      end;
    end
    else
      LastViewProc := Proc;
  end;

  LastViewProc := nil;

  for I := 0 to FClasses.Count - 1 do
  begin
    C := TSuObjectClass(FClasses.Items[I]);

    for N := 0 to C.FMethods.Count - 1 do
    begin
      Proc := TSuProcItem(C.FMethods.Items[N]);
      if (Proc.ImplPos.y <= Y) and
      (Proc.BodyRowEnd >= Y) then
      begin
        Result := Proc;
        Proc := GetProcOfProc(Proc);
        if Proc <> nil then Result := Proc;;
        Exit;
      end
      else
      if (Proc.DeclPos.Y = Y) and (Proc.DeclPos.x <= X) then
      begin
        M := Proc.DeclPos.x + Length(Proc.OriginalName);
        if M >= X then
        begin
          IsDecl := True;
          Result := Proc;
          Proc := GetProcOfProc(Proc);
          if Proc <> nil then Result := Proc;;
          Exit;
        end;
      end
      else
        LastViewProc := Proc;
    end;
  end;
end;

function TSuProcItem.FindClassMethod(AClassName, AMethodName: string): TSuProcItem;
var
  N: Integer;
  OC: TSuObjectClass;
  LCMethodName: string;
begin
  Result := nil;
  OC := FindClass(AClassName);
  LCMethodName:= UTF8LowerCase(AMethodName);
  if OC <> nil then
  begin
    for N := 0 to OC.FMethods.Count - 1 do
    begin
        if TSuProcItem(OC.FMethods.Items[N]).Name = LCMethodName then
        begin
          Result := TSuProcItem(OC.FMethods.Items[N]);
          Exit;
        end;
    end;
  end;
end;

{TSuCodeMap}

function TSuCodeMap.GetCodeMap(ATokens: TMemoTokens): TSuProcItem;
begin
  FTokens := ATokens;
  FMainProc := TSuProcItem.Create;
  Result := FMainProc;
  Tokens.First;
  DoProgram(FMainProc);
end;

function TSuCodeMap.GetNewPosition(ATokens: TMemoTokens): Integer;
var
  I, N: Integer;
begin
  Result := -1;
  FTokens := ATokens;
  Tokens.First;
  I := -1;
  while Tokens.Token^.TokenID <> ID_EOF do
  begin
    if Tokens.Token^.TokenID = ID_Implementation then
      I := Tokens.Token^.Row + 2
    else
    if Tokens.Token^.TokenID = ID_initialization then
    begin
      if I <= Tokens.Token^.Row then Result := I
      else
        Result := -1;
      Break;
    end
    else
    if Tokens.Token^.TokenID = ID_end then
    begin
      N := Tokens.Token^.Row + 2;
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Period then
      begin
        Result := I;
        Break;
      end
      else
        I := N;
    end;
    Tokens.NextWithSkipComment;
  end;
end;

procedure TSuCodeMap.DoProgram(Proc: TSuProcItem);
begin
  IsImplementation := False;
  if Tokens.Token^.TokenID = ID_Comment then Tokens.NextWithSkipComment;
  
  if Tokens.Token^.TokenID in [ID_program, ID_unit] then
  begin
    Tokens.NextWithSkipComment;
    if Tokens.Token^.TokenID = ID_Identifier then
    begin
      Proc.Name := Tokens.Token^.TokenName;
      Proc.DeclUnit := Tokens.Token^.TokenName;
      Proc.MethodDef:= 'programma';
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_SemiColon then
      begin
        Tokens.NextWithSkipComment;
        DoBlock(Proc);
      end;
    end;
  end;
end;

function TSuCodeMap.DoBlock(Proc: TSuProcItem): Boolean;
begin
  Result := True;
  while Tokens.Token^.TokenID <> ID_EOF do
  begin
    case Tokens.Token^.TokenID of
      ID_uses:
      begin
        Result := DoUses(Proc);
        if not Result then Exit;
      end;

      ID_type:
      begin
        Result := DoType(Proc);
        if not Result then Exit;
      end;

      ID_const:
      begin
        Result := DoConst(Proc);
        if not Result then Exit;
      end;

      ID_Implementation:
      begin
        Tokens.NextWithSkipComment;
        IsImplementation := True;
      end;

      ID_Interface:
      begin
        Tokens.NextWithSkipComment;
      end;

      ID_var:
      begin
        Result := DoVar(Proc);
        if not Result then Exit;
      end;

      ID_method, ID_constructor, ID_destructor, ID_function, ID_procedure:
      begin
        Result := DoMethod(Proc);
        if not Result then Exit;
      end;

      else
      begin
        Break;
      end;
    end;
  end;

  if Result then
  begin
    if Tokens.Token^.TokenID in [ID_begin, ID_initialization] then
    begin
     Proc.FBodyRowBegin := Tokens.Token^.Row + 1;
     Proc.FBodyColBegin := Tokens.Token^.Col + 2;
      Result := DoCompoundStat(Proc);
      Proc.FBodyRowEnd := Tokens.Token^.Row;
    end
    else
    if Tokens.Token^.TokenID = ID_End then
    begin
      Proc.FBodyRowBegin := Tokens.Token^.Row;
      Proc.FBodyColBegin := 2;
    end
    else
    begin
      Result := False;
    end;
  end;
end;

function TSuCodeMap.DoUses(Proc: TSuProcItem): Boolean;
begin
  Result := True;
  Tokens.NextWithSkipComment;
  while True do
  begin
    if Tokens.Token^.TokenID = ID_Identifier then
    begin
      Proc.FUnits.Add(string(Tokens.Token^.TokenName));
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Comma then Tokens.NextWithSkipComment
      else
      if Tokens.Token^.TokenID = ID_SemiColon then
      begin
        Tokens.NextWithSkipComment;
        Break;
      end
      else  Break;
    end
    else
    begin
      Result := False;
      Break;
    end;
  end;
end;

function TSuCodeMap.DoType(Proc: TSuProcItem): Boolean;
var
  AName: TDpString;
  ClassCol, ClassRow: Integer;

  function DoClassType: Boolean;
  var
    AClassOfName: String;
    AClass: TSuObjectClass;
    Status: Integer;

    function DoClassObject: Boolean;
    var
      IO: TSuObject;
      IOName: TDpString;
      IOValType: string;
      ObjCol, ObjRow: Integer;
      NameList: TStringList;
      I: Integer;
    begin
      Result := True;
      NameList := TStringList.Create;
      ObjCol := 0;
      ObjRow := 0;
      try
      while Tokens.Token^.TokenID <> ID_EOF do
      begin
        IOName := Tokens.Token^.TokenName;
        NameList.Append(string(IOName));
        ObjCol := Tokens.Token^.Col;
        ObjRow := Tokens.Token^.Row;
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Comma then
        begin
          Tokens.NextWithSkipComment;
          Continue;
        end;

        if Tokens.Token^.TokenID = ID_Colon then
        begin
          Tokens.NextWithSkipComment;
          Break;
        end
        else
        begin
          Result := False;
          Break;
        end;
      end;

      if Tokens.Token^.TokenID = ID_array then
      begin
        Result := DoArray(Proc, IOValType);
      end
      else
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        IOValType := string(Tokens.Token^.TokenName);
        Tokens.NextWithSkipComment;
      end
      else
      begin
        Result := False;
        Tokens.NextWithSkipComment;
      end;


      for I := 0 to NameList.Count - 1 do
      begin
        IO := TSuObject.Create;
        IO.FValueType := IOValType;
        IO.Name := NameList.Strings[I];
        IO.FObjType := itVar;
        IO.FCol := ObjCol;
        IO.FRow := ObjRow;
        IO.DeclUnit := Proc.DeclUnit;
        AClass.FVars.Add(IO);
        if Status = 0 then
          AClass.FObjects.Add(IO);
      end;
      NameList.Clear;
      finally
        NameList.Free;
      end;
    end;

    function DoScripProp: Boolean;
    var
      SP: TSuObject;
    begin
      Result := True;
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        SP := TSuObject.Create;
        SP.Name := Tokens.Token^.TokenName;
        SP.FCol := Tokens.Token^.Col;
        SP.FRow := Tokens.Token^.Row;
        SP.FObjType := itProp;
        SP.DeclUnit := Proc.DeclUnit;
        AClass.FVars.Add(SP);
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Colon then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            SP.FValueType := Tokens.Token^.TokenName;
            Tokens.NextWithSkipComment;
            if Tokens.Token^.TokenID in [ID_read, ID_write] then
            begin
              if Tokens.Token^.TokenID = ID_read then
              begin
                Tokens.NextWithSkipComment;
                if Tokens.Token^.TokenID in [ID_nil, ID_Identifier] then
                begin
                  Tokens.NextWithSkipComment;
                end
                else
                begin
                  Result := False;
                  Tokens.NextWithSkipComment;
                end;

              end;

              if Tokens.Token^.TokenID = ID_write then
              begin
                Tokens.NextWithSkipComment;
                if Tokens.Token^.TokenID in [ID_nil, ID_Identifier] then
                begin
                  Tokens.NextWithSkipComment;
                end
                else Result := False;
              end;
            end
            else
            begin
              Result := False;
            end;
          end
          else
          begin
            Result := False;
          end;
        end
        else
        begin
          Result := False;
        end;
      end
      else
      begin
        Result := False;
      end;
    end;

    function DoClassMethod: Boolean;
    var
      ClassProc: TSuProcItem;
      Param: TSuObject;
      PrCol, PrRow: Integer;
      MethodDef: string;
    begin
      Result := True;
      MethodDef := Tokens.Token^.TokenName;
      PrCol := Tokens.Token^.Col;
      PrRow := Tokens.Token^.Row;
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Identifier then
        begin
          ClassProc := TSuProcItem.Create;
          ClassProc.Proc := Proc;
          ClassProc.FObjType := itProc;
          ClassProc.FClassOwner := AClass;
          ClassProc.Name := string(Tokens.Token^.TokenName);
          ClassProc.DeclPos.x:= Tokens.Token^.Col;
          ClassProc.DeclPos.y:= Tokens.Token^.Row;
          ClassProc.DeclUnit := Proc.DeclUnit;
          ClassProc.FValueType := '';
          ClassProc.FCol := PrCol;
          ClassProc.FRow := PrRow;
          ClassProc.MethodDef := MethodDef;
          Param := TSuObject.Create;
          Param.Name := 'Self';
          Param.FObjType := itVar;
          Param.FValueType := AClass.OriginalName;
          ClassProc.AddVar(Param);
          AClass.FMethods.Add(ClassProc);
          if Status = 0 then
            AClass.FObjectMethods.Add(ClassProc);

          Tokens.NextWithSkipComment;
          Result := DoProcParam(ClassProc);
          if Result then
          begin
            if  Tokens.Token^.TokenID = ID_Colon then
            begin
                ClassProc.FIsFunction := True;
                Tokens.NextWithSkipComment;
                if Tokens.Token^.TokenID = ID_Identifier then
                begin
                  Param := TSuObject.Create;
                  Param.Name := 'Result';
                  Param.FValueType := string(Tokens.Token^.TokenName);
                  ClassProc.AddVar(Param);
                  ClassProc.FValueType := Param.ValueType;
                  Tokens.NextWithSkipComment;
                end;
            end;
          end;

        end
    end;

  begin
    Result := True;
    Status := 0;
    if Tokens.Token^.TokenID = ID_record then Status := 3;
    
    Tokens.NextWithSkipComment;
    AClassOfName := 'TObject';
    if Tokens.Token^.TokenID = ID_OpenRound then
    begin
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        AClassOfName := string(Tokens.Token^.TokenName);
        Tokens.NextWithSkipComment;
      end;
      if Tokens.Token^.TokenID = ID_CloseRound then Tokens.NextWithSkipComment
      else
      begin
        Result := False;
        Exit;
      end;
    end;
    AClass := TSuObjectClass.Create(Self);
    AClass.Name := string(AName);
    AClass.FRow := ClassRow;
    AClass.FCol := ClassCol;
    AClass.DeclUnit := Proc.DeclUnit;
    AClass.FInheritsFrom := AClassOfName;
    AClass.FValueType := AClass.OriginalName;
    Proc.FClasses.Add(AClass);
    while Tokens.Token^.TokenID <> ID_end do
    begin
      if Tokens.Token^.TokenID = ID_private then
      begin
        Status := 1;
        Tokens.NextWithSkipComment;
        Continue;
      end
      else
      if Tokens.Token^.TokenID = ID_protected then
      begin
        Status := 2;
        Tokens.NextWithSkipComment;
        Continue;
      end
      else
      if Tokens.Token^.TokenID = ID_public then
      begin
        Status := 3;
        Tokens.NextWithSkipComment;
        Continue;
      end;
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        Result := DoClassObject;
      end
      else
      if Tokens.Token^.TokenID = ID_property then
      begin
        Result := DoScripProp;
      end
      else
      if Tokens.Token^.TokenID in [ID_procedure, ID_function, ID_method,
        ID_constructor, ID_destructor] then
      begin
        Result := DoClassMethod;
      end
      else
      begin
        Result := False;
        Break;
      end;

      if Tokens.Token^.TokenID = ID_SemiColon then
      begin
        Tokens.NextWithSkipComment;
        Continue;
      end
      else
      begin
        Result := False;
        Break;
      end;

    end;

    if Tokens.Token^.TokenID = ID_end then Tokens.NextWithSkipComment
    else
    begin
      Result := False;
    end;
  end;

  function DoEnumType: Boolean;
  var
    AType: TSuObject;
    IO: TSuObject;
    I: Integer;
  begin
    Result := True;
    AType := TSuObject.Create;
    AType.Name := string(AName);
    Proc.FTyps.Add(AType);
    Result := DoEnumConst(Proc, AType);
    //Tokens.NextWithSkipComment;
    //I := 0;
    //while True do
    //begin
    //  if Tokens.Token^.TokenID = ID_Identifier then
    //  begin
    //    IO := TSuObject.Create;
    //    IO.Name := string(Tokens.Token^.TokenName);
    //    IO.FValue := IntToStr(I);
    //    Inc(I);
    //    IO.FValueType := AType.OriginalName;
    //    Proc.AddVar(IO);
    //    Tokens.NextWithSkipComment;
    //    if Tokens.Token^.TokenID = ID_Comma then
    //    begin
    //      Tokens.NextWithSkipComment;
    //      Continue;
    //    end
    //    else Break;
    //  end
    //  else
    //    Break;
    //end;
    //
    //if Tokens.Token^.TokenID = ID_CloseRound then
    //begin
    //  Tokens.NextWithSkipComment;
    //end
    //else
    //begin
    //  Result := False;
    //end;
  end;

  function DoSimpleType: Boolean;
  var
    AParent: TSuObject;
  begin
    Result := True;
    AParent := TSuObject.Create;
    AParent.Name := string(AName);
    Proc.FTyps.Add(AParent);
  end;

  function DoSetType: Boolean;
  var
    AType: TSuObject;
    ATypeOf: string;
  begin
    Result := True;
    AType := TSuObject.Create;
    AType.Name := string(AName);
    Proc.FTyps.Add(AType);
    Tokens.NextWithSkipComment;
    if Tokens.Token^.TokenID = ID_of then Tokens.NextWithSkipComment
    else
    begin
      Result := False;
    end;

    if Tokens.Token^.TokenID = ID_Identifier then
    begin
      ATypeOf := string(Tokens.Token^.TokenName);
      AType.FValueType := ATypeOf;
      Tokens.NextWithSkipComment;
    end
    else
    if Tokens.Token^.TokenID = ID_OpenRound then
    begin
      Result := DoEnumConst(Proc, AType);
    end
    else
    if Tokens.Token^.TokenID = ID_Integer then
    begin
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Dots then
      begin
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Integer then
        begin
          Tokens.NextWithSkipComment;
        end;
      end
      else
      begin
        Result := False;
      end;
    end;
  end;

  function DoProcType: Boolean;
  var
    DeclProc: TSuProcItem;
    Param: TSuObject;
    AType: TSuObject;
  begin
    Result := True;
    AType := TSuObject.Create;
    AType.Name := string(AName);
    Proc.FTyps.Add(AType);

    DeclProc := TSuProcItem.Create;
    DeclProc.Name:= 'DeclProc';
    DeclProc.DeclUnit := Proc.DeclUnit;
    DeclProc.Proc := Proc;
    DeclProc.FValueType := '';
    DeclProc.Name := string(Tokens.Token^.TokenName);
    DeclProc.DeclPos.x := Tokens.Token^.Col;
    DeclProc.DeclPos.y := Tokens.Token^.Row;
    AType.FVars.Add(DeclProc);
    Tokens.NextWithSkipComment;
    if Tokens.Token^.TokenID = ID_OpenRound then
    begin
      Result := DoProcParam(DeclProc);
      if Result then
      begin
        if Tokens.Token^.TokenID = ID_Colon then
        begin
          DeclProc.FIsFunction := True;
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            Param := TSuObject.Create;
            Param.Name := 'Result';
            Param.FValueType := string(Tokens.Token^.TokenName);
            DeclProc.AddVar(Param);
            DeclProc.FValueType := Param.ValueType;
            Tokens.NextWithSkipComment;
          end
          else
          begin
            Result := False;
          end;
        end;
      end;
    end;

    if Tokens.Token^.TokenID <> ID_SemiColon then
    begin
      Result := False;
    end;
  end;

begin
  Result := True;
  Tokens.NextWithSkipComment;
  if Tokens.Token^.TokenID = ID_Identifier then
  begin
    while Tokens.Token^.TokenID = ID_Identifier do
    begin
      AName := Tokens.Token^.TokenName;
      ClassCol := Tokens.Token^.Col;
      ClassRow := Tokens.Token^.Row;
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Assignment then
      begin
        Tokens.NextWithSkipComment;
        case Tokens.Token^.TokenID of
          ID_class, ID_record:
          begin
            Result := DoClassType;
            if not Result then Exit;
          end;

          ID_OpenRound:
          begin
            Result := DoEnumType;
            if not Result then Exit;
          end;

          ID_Identifier:
          begin
            Result := DoSimpleType;
            if not Result then Exit;
          end;

          ID_set:
          begin
            Result := DoSetType;
            if not Result then Exit;
          end;

          ID_procedure, ID_function, ID_method:
          begin
            Result := DoProcType;
            if not Result then Exit;
          end;
        end;
      end;

      if Tokens.Token^.TokenID = ID_SemiColon then Tokens.NextWithSkipComment
      else
      begin
        Result := False;
        Break;
      end;
    end;
  end
  else
  begin
    Result := False;
  end;
end;

function TSuCodeMap.DoCheckParam(Proc: TSuProcItem): Boolean;
var
   Status: Integer;
begin
  Result := True;
  if Tokens.Token^.TokenID <> ID_OpenRound then
  begin
    Exit;
  end;
  Tokens.NextWithSkipComment;
  if Tokens.Token^.TokenID = ID_CloseRound then
  begin
     Exit;
  end;
  Status := 0;

  while Tokens.Pos < Tokens.Count do
  begin
      case Status of
        0:
        begin
          //   
          if Tokens.Token^.TokenID = ID_var then
          begin
              Tokens.NextWithSkipComment;
          end
          else
          if Tokens.Token^.TokenID = ID_out then
          begin
              Tokens.NextWithSkipComment;
          end
          else
          if Tokens.Token^.TokenID = ID_const then
          begin
              Tokens.NextWithSkipComment;
          end;

          Status := 1;
          Continue;
        end;

        1:
        begin
          //    )
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            Status := 2;
          end
          else
          begin
            Result := False;
            Break;
          end;
        end;

        2:
        begin
          // ,  :
          if Tokens.Token^.TokenID = ID_Comma then Status := 1
          else
          if Tokens.Token^.TokenID = ID_Colon then Status := 3;
        end;

        3:
        begin
          //  
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            Status := 4;
          end
          else
          begin
            Result := False;
            Break;
          end;
        end;

        4:
        begin
          if Tokens.Token^.TokenID = ID_CloseRound then
          begin
            Tokens.NextWithSkipComment;
            Break;
          end
          else
          if Tokens.Token^.TokenID = ID_SemiColon then Status := 0;
        end;
      end;

      Tokens.NextWithSkipComment;
    end;
end;

function TSuCodeMap.DoProcParam(Proc: TSuProcItem): Boolean;
var
   Status: Integer;
   SList: TStringList;
   PP: TSuProcParam;
   AType: string;
   AsVar: Boolean;
   AsConst: Boolean;
   AsOut: Boolean;
   I: Integer;
   T: TIdentToken;
begin
  Result := True;
  if Tokens.Token^.TokenID <> ID_OpenRound then Exit;
  Tokens.NextWithSkipComment;
  if Tokens.Token^.TokenID = ID_CloseRound then Exit;
  Status := 0;
  AsVar := False;
  AsConst := False;
  AsOut := False;
  SList := TStringList.Create;
  try
    while Tokens.Token^.TokenID <> ID_EOF do
    begin

      case Status of
        0:
        begin
          //   
          if Tokens.Token^.TokenID = ID_var then
          begin
            AsVar := True;
            Tokens.NextWithSkipComment;
          end
          else
          if Tokens.Token^.TokenID = ID_out then
          begin
            AsOut := True;
            Tokens.NextWithSkipComment;
          end
          else
          if Tokens.Token^.TokenID = ID_const then
          begin
            AsConst:= True;
            Tokens.NextWithSkipComment;
          end;

          Status := 1;
          Continue;
        end;

        1:
        begin
          //    )
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            SList.Add(Tokens.Token^.TokenName);
            Status := 2;
          end
          else
          begin
            Result := False;
            Break;
          end;
        end;

        2:
        begin
          // ,  :
          if Tokens.Token^.TokenID = ID_Comma then Status := 1
          else
          if Tokens.Token^.TokenID = ID_Colon then Status := 3
          else
          begin
            Result := False;
            Break;
          end;
        end;

        3:
        begin
          //  
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            I := 0;
            while I < SList.Count do
            begin
              AType := Tokens.Token^.TokenName;
              PP := TSuProcParam.Create;
              PP.DeclUnit := Proc.DeclUnit;
              PP.Name := SList.Strings[I];
              PP.FValueType := AType;
              PP.AsVar := AsVar;
              PP.AsOut := AsOut;
              PP.AsConst := AsConst;
              TokenPosToSuObject(PP);
              //if PP.FValueType = nil then WriteError(Proc.DeclUnit, ceUnknownType, '');
              Proc.AddParam(PP);
              Inc(I);
            end;
            AsVar := False;
            AsConst := False;
            AsOut := False;
            T := Tokens.PreviewNext^.TokenID;
            if (T = ID_Equal) then T := ID_Assignment;
            if T = ID_Assignment then
            begin
              while not (Tokens.Token^.TokenID in [ID_EOF, ID_SemiColon, ID_CloseRound]) do
                Tokens.NextWithSkipComment;
              Tokens.Prev;
            end;
            SList.Clear;
            Status := 4;
          end
          else
          begin
            Result := False;
            Break;
          end;
        end;

        4:
        begin
          if Tokens.Token^.TokenID = ID_CloseRound then
          begin
            Tokens.NextWithSkipComment;
            Break;
          end
          else
          if Tokens.Token^.TokenID = ID_SemiColon then Status := 0;
        end;
      end;

      Tokens.NextWithSkipComment;
    end;
  finally
    SList.Free;
  end;
end;

function TSuCodeMap.DoConst(Proc: TSuProcItem): Boolean;
var
  CName: string;
  DSC: TSuObject;
begin
  Result := True;
  Tokens.NextWithSkipComment;
  if Tokens.Token^.TokenID = ID_Identifier then
  begin
    while Tokens.Token^.TokenID = ID_Identifier do
    begin
      CName := string(Tokens.Token^.TokenName);
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Assignment then
      begin
        Tokens.NextWithSkipComment;
        DSC := TSuObject.Create;
        DSC.DeclUnit:= Proc.DeclUnit;
        DSC.Name := CName;
        DSC.FCol := Tokens.Token^.Col;
        DSC.FRow := Tokens.Token^.Row;
        DSC.FObjType := itConst;
        while Tokens.Token^.TokenID <> ID_EOF do
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_SemiColon then Break;
        end;
        Proc.AddVar(DSC);

        if Tokens.Token^.TokenID = ID_SemiColon then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_Identifier then Continue
          else Break;
        end
        else
        begin
          Result := False;
          Break;
        end;

      end
      else
      begin
        Result := False;
        Break;
      end;
      Tokens.NextWithSkipComment;
    end;
  end
  else
  begin
    Result := False;
  end;
end;

function TSuCodeMap.DoVar(Proc: TSuProcItem): Boolean;
var
  VarList: TStringList;
  PosList: TStringList;
  I: Integer;
  V: TSuObject;
  Ar: TSuObject;
  St: TSuObject;
  AValType, MinVal, MaxVal, S: string;
  DeclRow, DeclCol: Integer;

  procedure PosStrToPos(AStr: string);
  var
    AM, AL: Integer;
    AC: string;
  begin
    AM := LastDelimiter(',', AStr);
    if AM > 0 then
    begin
      AC := Copy(AStr, 1, AM - 1);
      DeclCol:= StrToInt(AC);
      AL := Length(AStr);
      AC := Copy(AStr, AM + 1, AL - AM);
      DeclRow:= StrToInt(AC);
    end;
  end;

begin
  Result := True;
  Tokens.NextWithSkipComment;
  VarList := TStringList.Create;
  PosList := TStringList.Create;
  try
    while Tokens.Token^.TokenID <> ID_SemiColon do
    begin
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        // 
        DeclRow := Tokens.Token^.Row;
        DeclCol:= Tokens.Token^.Col;
        S := Tokens.Token^.TokenName;
        VarList.Add(Tokens.Token^.TokenName);
        PosList.Add(IntToStr(DeclCol) + ',' + IntToStr(DeclRow));
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Comma then
        begin
          // 
          Tokens.NextWithSkipComment;
          Continue;
        end
        else
        if Tokens.Token^.TokenID = ID_Colon then
        begin
          //  
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID in [ID_Identifier, ID_array, ID_set] then
          begin
            if Tokens.Token^.TokenID = ID_set then
            begin
              Result := DoSetDeclaration(MinVal, MaxVal, AValType, Proc);
              I := 0;
              while I < VarList.Count do
              begin
                S := VarList.Strings[I];
                PosStrToPos(PosList.Strings[I]);
                St := TSuObject.Create;
                St.FCol:= DeclCol;
                St.FRow := DeclRow;
                St.DeclUnit := Proc.DeclUnit;
                St.Name := VarList.Strings[I];
                St.FValueType := 'set ' + MinVal + '..' + MaxVal + ' of ' + AValType;
                Proc.AddVar(St);
                Inc(I);
              end;
              VarList.Clear;
              PosList.Clear;
            end
            else
            if Tokens.Token^.TokenID = ID_array then
            begin
              I := 0;
              Result := DoArray(Proc, AValType);

               while I < VarList.Count do
               begin
                 S := VarList.Strings[I];
                 PosStrToPos(PosList.Strings[I]);
                 Ar := TSuObject.Create;
                 Ar.DeclUnit:= Proc.DeclUnit;
                 Ar.FCol := DeclCol;
                 Ar.FRow := DeclRow;
                 Ar.Name := VarList.Strings[I];
                 Ar.FValueType := AValType;
                 Proc.AddVar(Ar);
                 Inc(I);
               end;
                VarList.Clear;
                PosList.Clear;
            end
            else
            begin
              AValType := string(Tokens.Token^.TokenName);
              if Tokens.PreviewNext^.TokenID = ID_Assignment then
              begin
                while Tokens.Token^.TokenID <> ID_SemiColon do
                  Tokens.NextWithSkipComment;
                Tokens.PrevWithSkipComment;
              end;
              I := 0;
              while I < VarList.Count do
              begin
                S := VarList.Strings[I];
                PosStrToPos(PosList.Strings[I]);
                V := TSuObject.Create;
                V.DeclUnit:= Proc.DeclUnit;
                V.FRow := DeclRow;
                V.FCol := DeclCol;
                V.Name := VarList.Strings[I];
                V.FValueType := AValType;

                Proc.AddVar(V);
                Inc(I);
              end;
              Tokens.NextWithSkipComment;
            end;
            VarList.Clear;
            PosList.Clear;
            if Tokens.Token^.TokenID = ID_SemiColon then
            begin
              // 
              //   - ,   
              Tokens.NextWithSkipComment;
              if Tokens.Token^.TokenID = ID_Identifier then Continue
              else Break;
            end
            else
            begin
              Result := False;
              Break;
            end;
          end
          else
          begin
            Result := False;
            Break;
          end;
        end
        else
        begin
          Result := False;
          Break;
        end;
      end
      else
      begin
        Result := False;
        Break;
      end;
      Tokens.NextWithSkipComment;
    end;
  finally
    VarList.Free;
    PosList.Free;
  end;
end;

function TSuCodeMap.DoSetDeclaration(var MinVal, MaxVal, RefType: string;
  Proc: TSuProcItem): Boolean;
begin
  Result := True;
  Tokens.NextWithSkipComment;
  MinVal := '0';
  MaxVal := '255';
  if Tokens.Token^.TokenID = ID_of then
  begin
    Tokens.NextWithSkipComment;
    if Tokens.Token^.TokenID = ID_Identifier then
    begin
      RefType := string(Tokens.Token^.TokenName);
      Tokens.NextWithSkipComment;
    end
    else
    if Tokens.Token^.TokenID = ID_OpenRound then
    begin
      Result := DoEnumConst(Proc, nil);
    end
    else
    if Tokens.Token^.TokenID = ID_Integer then
    begin
      MinVal := string(Tokens.Token^.TokenName);
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Dots then
      begin
        Tokens.NextWithSkipComment;
        if Tokens.Token^.TokenID = ID_Integer then
          MaxVal := string(Tokens.Token^.TokenName);
      end;
    end;
  end
  else
  begin
    Result := False;
  end;
end;

function TSuCodeMap.DoArray(Proc: TSuProcItem; var AValueType: string): Boolean;
var
  Status: Integer;
begin
  Tokens.NextWithSkipComment;
  Result := True;
  AValueType := 'array';
  if Tokens.Token^.TokenID = ID_OpenBlock then
  begin
    AValueType := AValueType + ' [';
    Status := 0;
    Tokens.NextWithSkipComment;
    if Tokens.Token^.TokenID = ID_CloseBlock then
    begin
      AValueType := AValueType + ']';
      Tokens.NextWithSkipComment;
    end
    else
    begin
      while True do
      begin
        case Status of
          0: //   
          begin
            if Tokens.Token^.TokenID = ID_Integer then
            begin
              AValueType := AValueType + Tokens.Token^.TokenName;
              //List.Add(Tokens.Token^.TokenName);
              Tokens.NextWithSkipComment;
              if Tokens.Token^.TokenID = ID_Dots then
              begin
                AValueType := AValueType + '..';
                Tokens.NextWithSkipComment;
                if Tokens.Token^.TokenID = ID_Integer then
                begin
                  AValueType := AValueType + Tokens.Token^.TokenName;
                  Status := 1;
                end
                else
                begin
                  Break;
                end;
              end
              else
              begin
                Result := False;
                Break;
              end;
            end
            else
            begin
              Result := False;
              Break;
            end;

          end;

          1: // ]  ,
          begin
            if Tokens.Token^.TokenID = ID_Comma then
            begin
              Status := 0;
              AValueType := AValueType + ',';
            end
            else
            if Tokens.Token^.TokenID = ID_CloseBlock then
            begin
              AValueType := AValueType + ']';
              Tokens.NextWithSkipComment;
              Break;
            end
            else
            begin
              Result := False;
              Break;
            end;
          end;
        end;
        Tokens.NextWithSkipComment;
      end;

    end;
  end;

  if Tokens.Token^.TokenID = ID_of then
  begin
    Tokens.NextWithSkipComment;
    AValueType := AValueType + ' of ';
    if Tokens.Token^.TokenID in [ID_Identifier, ID_const] then
    begin
      AValueType := AValueType + Tokens.Token^.TokenName;
      Tokens.NextWithSkipComment;
    end;
  end;
end;

function TSuCodeMap.DoMethod(Proc: TSuProcItem): Boolean;
var
  DSP: TSuProcItem;
  Param, SO: TSuObject;
  AType, MethodDef: string;
  ImplBegin: TPoint;
begin
  Result := True;
  MethodDef:= Tokens.Token^.TokenName;
  ImplBegin.y := Tokens.Token^.Row;
  ImplBegin.x := Tokens.Token^.Col;
  Tokens.NextWithSkipComment;
  DSP := nil;
  if Tokens.Token^.TokenID = ID_Identifier then
  begin
    if Tokens.PreviewNext^.TokenID = ID_Period then
    begin
      AType := string(Tokens.Token^.TokenName);
      Tokens.NextWithSkipComment;
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Identifier then
      begin
        DSP := FMainProc.FindClassMethod(AType, string(Tokens.Token^.TokenName));
        Tokens.NextWithSkipComment;
        DoCheckParam(DSP);
      end
      else
      begin
        Result := False;
        Exit;
      end;
    end
    else
    begin
      SO := Proc.FindObject(Tokens.Token^.TokenName);
      if (SO <> nil) and (SO.FObjType = itProc) then
      begin
        DSP := TSuProcItem(SO);
        Tokens.NextWithSkipComment;
        DoCheckParam(DSP);
      end;
    end;
    if DSP = nil then
    begin
      DSP := TSuProcItem.Create;
      DSP.DeclUnit := Proc.DeclUnit;
      DSP.Proc := Proc;
      DSP.FValueType := '';
      DSP.MethodDef := MethodDef;
      // 
      DSP.Name := string(Tokens.Token^.TokenName);
      DSP.DeclPos.x := Tokens.Token^.Col;
      DSP.DeclPos.y := Tokens.Token^.Row;
      Proc.AddProc(DSP);
      Tokens.NextWithSkipComment;
      //
      Result := DoProcParam(DSP);
    end;//if result
    if Result then
    begin
        //  
        if Tokens.Token^.TokenID = ID_Colon then
        begin
          DSP.FIsFunction := True;
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_Identifier then
          begin
            Param := TSuObject.Create;
            Param.Name := 'Result';
            Param.FValueType := string(Tokens.Token^.TokenName);
            DSP.AddVar(Param);
            DSP.FValueType := Param.ValueType;
            Tokens.NextWithSkipComment;
          end
          else
          begin
            Result := False;
          end;
        end;
    end;
    if Tokens.Token^.TokenID = ID_SemiColon then
    begin
      Tokens.NextWithSkipComment;
      if IsImplementation then
      begin
        if Tokens.Token^.TokenID = ID_Forward then
        begin
          Tokens.NextWithSkipComment;
          if Tokens.Token^.TokenID = ID_SemiColon then Tokens.NextWithSkipComment
          else
          begin
            Result := False;
          end;
          Exit;
        end;
        DSP.ImplPos := ImplBegin;
        Result := DoBlock(DSP);
        if Result then
        begin
          if Tokens.Token^.TokenID = ID_SemiColon then Tokens.NextWithSkipComment
          else
          begin
            Result := False;
          end;
        end;
      end;
    end
    else
    begin
      Result := False;
    end;
  end
  else
  begin
    Result := False;
  end;
end;

function TSuCodeMap.DoCompoundStat(Proc: TSuProcItem): Boolean;
begin
  Result := True;
  Tokens.NextWithSkipComment;
  while Tokens.Token^.TokenID <> ID_EOF do
  begin
    if Tokens.Token^.IsEnd then
    begin
      Tokens.NextWithSkipComment;
      Break;
    end;
    Tokens.NextWithSkipComment;
  end;
end;

function TSuCodeMap.DoEnumConst(Proc: TSuProcItem; aType: TSuObject): Boolean;
var
  IO: TSuObject;
  I: Integer;
begin
  Result := True;
  Tokens.NextWithSkipComment;
  I := 0;
  while True do
  begin
    if Tokens.Token^.TokenID = ID_Identifier then
    begin
      IO := TSuObject.Create;
      IO.Name := string(Tokens.Token^.TokenName);
      IO.FValue := IntToStr(I);
      Inc(I);
      if aType = nil then
        IO.FValueType := 'Enum'
      else
        IO.FValueType := AType.OriginalName;
      Proc.AddVar(IO);
      Tokens.NextWithSkipComment;
      if Tokens.Token^.TokenID = ID_Comma then
      begin
        Tokens.NextWithSkipComment;
        Continue;
      end
      else Break;
    end
    else
      Break;
  end;

  if Tokens.Token^.TokenID = ID_CloseRound then
  begin
    Tokens.NextWithSkipComment;
  end
  else
  begin
    Result := False;
  end;
end;

procedure TSuCodeMap.TokenPosToSuObject(AObject: TSuObject);
begin
  AObject.FCol := Tokens.Token^.Col;
  AObject.FRow := Tokens.Token^.Row;
  AObject.FEndCol := Tokens.Token^.EndCol;
end;

destructor TSuCodeMap.Destroy;
begin
  MainProc.Free;
  inherited Destroy;
end;

end.
