{------------------------------------------------------------------------------}
{                                                                              }
{                               DieselPascal:                                  }
{                               Yuriy Kopnin                                   }
{                                    LGPL                                      }
{                                                                              }
{                             winapiwrapper.pas:                               }
{                      Contributed by Alexey Kuryakin                          }
{                                    LGPL                                      }
{                                                                              }
{------------------------------------------------------------------------------}
unit winapiwrapper; // Wrapper for Windows API

{$mode objfpc}{$H+}
{$inline on}

interface

uses
  Classes, SysUtils, LCLProc, LCLType;

type
  TWinApi = class(TObject)
    constructor Create;
    destructor  Destroy; override;
  public
    function FindWindow(aClassName, aWindowName: WideString): HWND;
    function FindWindowEx(hWndParent, hWndChildAfter: HWND; aClass, aWindow: WideString): HWND;
    function SendMessage(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM): LRESULT;
    function PostMessage(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM): Boolean;
    function BringWindowToTop(Wnd: HWND): Boolean;
    function CloseWindow(Wnd: HWND): Boolean;
    function GetClassName(Wnd: HWND): WideString;
    function GetClientRectAsText(Wnd: HWND): String;
    function GetAncestor(Wnd: HWND; gaFlags: UINT): HWND;
    function GetDesktopWindow: HWND;
    function GetForegroundWindow: HWND;
    function GetParent(Wnd: HWND): HWND;
    function GetShellWindow: HWND;
    function GetTopWindow(Wnd: HWND): HWND;
    function GetWindow(Wnd: HWND; uCmd: UINT): HWND;
    function GetWindowProcessId(Wnd: HWND): DWORD;
    function GetWindowRectAsText(Wnd: HWND): String;
    function GetWindowText(Wnd: HWND): WideString;
    function IsChild(WndParent, Wnd: HWND): Boolean;
    function IsIconic(Wnd: HWND): Boolean;
    function IsWindow(Wnd: HWND): Boolean;
    function IsWindowVisible(Wnd: HWND): Boolean;
    function IsZoomed(Wnd: HWND): Boolean;
    function MoveWindow(Wnd: HWND; X, Y, W, H: Integer; Repaint:Boolean): Boolean;
    function SendMessageTimeOut(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM; Flags, TimeOut: UINT; lpdwResult: PPtrUInt): LRESULT;
    function SetForegroundWindow(Wnd: HWND): Boolean;
    function SetWindowPos(Wnd, WndInsertAfter: HWND; x, y, cx, cy: Integer; Flags: UINT): Boolean;
    function SetWindowText(Wnd: HWND; const Caption:WideString): Boolean;
    function ShowWindow(Wnd: HWND; nCmdShow: Integer): Boolean;
    function ShowWindowAsync(Wnd: HWND; nCmdShow: Integer): Boolean;
    function WindowFromPoint(x, y: Integer): HWND;
    function SendWmCopyData(Target,Sender:HWND; dwData:PtrUInt; const Data:String; Hex:Boolean):LRESULT;
    function GetListOfProcesses(PID,PPID:Cardinal; Exe:String):String;
    function GetListOfModules(PID:Cardinal):String;
    function GetListOfWindows(Pid:Cardinal; lpClass,lpTitle:String):String;
    function GetProcessId(hProcess:THandle):DWORD;
    function GetCurrentProcess:THandle;
    function GetCurrentProcessId:DWORD;
    function CloseHandle(hObject:THandle):Boolean;
    function OpenProcess(dwDesiredAccess:DWORD; bInheritHandle:Boolean;  dwProcessId:DWORD):THandle;
    function GetExitCodeProcess(hProcess:THandle; var lpExitCode:DWord):Boolean;
    function GetProcessImageFileName(hProcess:THandle):String;
    function TerminateProcess(hProcess:THandle; uExitCode:UInt):Boolean;
    function GetPriorityClass(hProcess:THandle):DWORD;
    function SetPriorityClass(hProcess:THandle; dwPriorityClass:DWORD):Boolean;
    function GetProcessTimesAsText(hProcess:THandle):String;
    function GetSystemTimesAsText:String;
    function GetProcessAffinityMask(hProcess:THandle):QWord;
    function GetSystemAffinityMask:QWord;
    function SetProcessAffinityMask(hProcess:THandle; Affinity:QWord):Boolean;
    function WinExec(lpCmdLine:String; uCmdShow:UINT):UINT;
    function GetWindowLong(Wnd:HWND; nIndex:Integer):LongInt;
    function SetWindowLong(Wnd:HWND; nIndex:Integer; dwNewLong:LongInt):LongInt;
    function GetWindowLongPtr(Wnd:HWND; nIndex:Integer):PtrInt;
    function SetWindowLongPtr(Wnd:HWND; nIndex:Integer; dwNewLong:PtrInt):PtrInt;
    function ShellExecuteEx(fMask:DWORD; wnd:HWND; lpVerb:WideString; lpFile:WideString; lpParameters:WideString; lpDirectory:WideString; nShow:Integer; lpClass:WideString; hkeyClass:THandle; dwHotKey:DWORD; hIcon:THandle; var hInstApp:THandle; var hProcess:THandle):Boolean;
    function WaitForSingleObject(hHandle:THandle; dwMilliseconds:DWORD):DWORD;
    function GetACP:UInt;
    function GetOEMCP:UInt;
  end;

function WinApi:TWinApi; inline;

implementation

uses LConvEncoding, LazFileUtils, LazUTF8,
  {$IFDEF WINDOWS}
   windows, ShellApi, JwaWinUser, JwaTlHelp32, JwaPsApi, JwaWinBase, easyipc,
  {$ENDIF}
  expandenvvar;

const TheWinApi:TWinApi=nil;

function WinApi:TWinApi; inline;
begin
  if (TheWinApi=nil) then TheWinApi:=TWinApi.Create;
  Result:=TheWinApi;
end;

function PCharArg(const S:String):PChar; inline;
begin
  if (S='') then Result:=nil else Result:=PChar(S);
end;

function PWideCharArg(const S:WideString):PWideChar; inline;
begin
  if (S='') then Result:=nil else Result:=PWideChar(S);
end;

function StrToHex(const Str:String):String;
begin
  Result:='';
  if (Str='') then exit;
  SetLength(Result,Length(Str)*2);
  BinToHex(PChar(Str),PChar(Result),Length(Str));
end;

function HexToStr(const Hex:String):String;
begin
  Result:='';
  if (Hex='') then exit;
  SetLength(Result,Length(Hex) div 2);
  SetLength(Result,HexToBin(PChar(Hex),PChar(Result),Length(Result)));
end;

constructor TWinApi.Create;
begin
  inherited Create;
  if (TheWinApi=nil) then TheWinApi:=Self;
end;

destructor  TWinApi.Destroy;
begin
  if (Self=TheWinApi) then TheWinApi:=nil;
  inherited Destroy;
end;

function TWinApi.FindWindow(aClassName, aWindowName: WideString): HWND;
begin
  Result := 0;
  {$IFDEF WINDOWS}
  Result := Windows.FindWindowW(PWideCharArg(aClassName), PWideCharArg(aWindowName));
  {$ENDIF}
end;

function TWinApi.FindWindowEx(hWndParent, hWndChildAfter: HWND; aClass, aWindow: WideString): HWND;
begin
  Result := 0;
  {$IFDEF WINDOWS}
  Result := Windows.FindWindowExW(hWndParent, hWndChildAfter, PWideCharArg(aClass), PWideCharArg(aWindow));
  {$ENDIF}
end;

function TWinApi.SendMessage(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM): LRESULT;
begin
  Result := 0;
  {$IFDEF WINDOWS}
  Result := Windows.SendMessage(Wnd, Msg, WPAR, LPAR);
  {$ENDIF}
end;

function TWinApi.PostMessage(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM): Boolean;
begin
  Result := false;
  {$IFDEF WINDOWS}
  Result := Windows.PostMessage(Wnd, Msg, WPAR, LPAR);
  {$ENDIF}
end;

function TWinApi.BringWindowToTop(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.BringWindowToTop(Wnd);
  {$ENDIF}
end;

function TWinApi.CloseWindow(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.CloseWindow(Wnd);
  {$ENDIF}
end;

function TWinApi.GetAncestor(Wnd: HWND; gaFlags: UINT): HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetAncestor(Wnd,gaFlags);
  {$ENDIF}
end;

function TWinApi.GetClassName(Wnd: HWND): WideString;
begin
  Result:='';
  {$IFDEF WINDOWS}
  SetLength(Result,1024*4);
  SetLength(Result,Windows.GetClassNameW(Wnd,PWideChar(Result),Length(Result)));
  {$ENDIF}
end;

function TWinApi.GetClientRectAsText(Wnd: HWND): String;
var R:TRect;
begin
  Result:='';
  {$IFDEF WINDOWS}
  if Windows.GetClientRect(Wnd,R) then Result:=Format('%d, %d, %d, %d',[R.Left,R.Top,R.Right,R.Bottom]);
  {$ENDIF}
end;

function TWinApi.GetDesktopWindow: HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetDesktopWindow;
  {$ENDIF}
end;

function TWinApi.GetForegroundWindow: HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetForegroundWindow;
  {$ENDIF}
end;

function TWinApi.GetParent(Wnd: HWND): HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetParent(Wnd);
  {$ENDIF}
end;

function TWinApi.GetShellWindow: HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=JwaWinUser.GetShellWindow;
  {$ENDIF}
end;

function TWinApi.GetTopWindow(Wnd: HWND): HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetTopWindow(Wnd);
  {$ENDIF}
end;

function TWinApi.GetWindow(Wnd: HWND; uCmd: UINT): HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.GetWindow(Wnd,uCmd);
  {$ENDIF}
end;

function TWinApi.GetWindowProcessId(Wnd: HWND): DWORD;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  if Windows.GetWindowThreadProcessId(Wnd,Result)=0 then Result:=0;
  {$ENDIF}
end;

function TWinApi.GetWindowRectAsText(Wnd: HWND): String;
var R:TRect;
begin
  Result:='';
  {$IFDEF WINDOWS}
  if Windows.GetWindowRect(Wnd,R) then Result:=Format('%d, %d, %d, %d',[R.Left,R.Top,R.Right,R.Bottom]);
  {$ENDIF}
end;

function TWinApi.GetWindowText(Wnd: HWND): WideString;
begin
  Result:='';
  {$IFDEF WINDOWS}
  SetLength(Result,1024*4);
  SetLength(Result,Windows.GetWindowTextW(Wnd,PWideChar(Result),Length(Result)));
  {$ENDIF}
end;

function TWinApi.IsChild(WndParent, Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.IsChild(WndParent,Wnd);
  {$ENDIF}
end;

function TWinApi.IsIconic(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.IsIconic(Wnd);
  {$ENDIF}
end;

function TWinApi.IsWindow(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.IsWindow(Wnd);
  {$ENDIF}
end;

function TWinApi.IsWindowVisible(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.IsWindowVisible(Wnd);
  {$ENDIF}
end;

function TWinApi.IsZoomed(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.IsZoomed(Wnd);
  {$ENDIF}
end;

function TWinApi.MoveWindow(Wnd: HWND; X, Y, W, H: Integer; Repaint:Boolean): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.MoveWindow(Wnd,X,Y,W,H,Repaint);
  {$ENDIF}
end;

function TWinApi.SendMessageTimeOut(Wnd: HWND; Msg: Cardinal; WPAR: WPARAM; LPAR: LPARAM; Flags, TimeOut: UINT; lpdwResult:PPtrUInt): LRESULT;
begin
  Result := 0;
  {$IFDEF WINDOWS}
  Result := Windows.SendMessageTimeOut(Wnd, Msg, WPAR, LPAR, Flags, TimeOut, lpdwResult);
  {$ENDIF}
end;

function TWinApi.SetForegroundWindow(Wnd: HWND): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.SetForegroundWindow(Wnd);
  {$ENDIF}
end;

function TWinApi.SetWindowPos(Wnd, WndInsertAfter: HWND; x, y, cx, cy: Integer; Flags: UINT): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.SetWindowPos(Wnd,WndInsertAfter,x,y,cx,cy,Flags);
  {$ENDIF}
end;

function TWinApi.SetWindowText(Wnd: HWND; const Caption:WideString): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.SetWindowTextW(Wnd,PWideCharArg(Caption));
  {$ENDIF}
end;

function TWinApi.ShowWindow(Wnd: HWND; nCmdShow: Integer): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.ShowWindow(Wnd,nCmdShow);
  {$ENDIF}
end;

function TWinApi.ShowWindowAsync(Wnd: HWND; nCmdShow: Integer): Boolean;
begin
  Result:=false;
  {$IFDEF WINDOWS}
  Result:=Windows.ShowWindowAsync(Wnd,nCmdShow);
  {$ENDIF}
end;

function TWinApi.WindowFromPoint(x, y: Integer): HWND;
begin
  Result:=0;
  {$IFDEF WINDOWS}
  Result:=Windows.WindowFromPoint(Point.Create(x,y));
  {$ENDIF}
end;

function TWinApi.SendWmCopyData(Target,Sender:HWND; dwData:PtrUInt; const Data:String; Hex:Boolean):LRESULT;
{$IFDEF WINDOWS} var cds:TCopyDataStruct; Buff:String; {$ENDIF}
begin
  Result:=0;
  {$IFDEF WINDOWS}
  if Hex then Buff:=HexToStr(Data) else Buff:=Data;
  cds.dwData:=dwData;
  cds.cbData:=Length(Buff);
  cds.lpData:=PCharArg(Buff);
  Result:=WinApi.SendMessage(Target,WM_COPYDATA,WPARAM(Sender),LPARAM(@cds));
  {$ENDIF}
end;

// List processes as PID, PPID, Priority, EXE
function TWinApi.GetListOfProcesses(PID,PPID:Cardinal; Exe:String):String;
{$IFDEF WINDOWS} var hProcessSnap:THandle; pe32:PROCESSENTRY32; List:TStringList; {$ENDIF}
begin
 Result:='';
 {$IFDEF WINDOWS}
 List:=TStringList.Create;
 try
  hProcessSnap:=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if(hProcessSnap<>0) and (hProcessSnap<>INVALID_HANDLE_VALUE) then
  try
   ZeroMemory(@pe32,sizeof(pe32));
   pe32.dwSize:=sizeof(pe32);
   if Process32First(hProcessSnap,pe32) then
   repeat
     if (PID=0) or (PID=pe32.th32ProcessID) then
     if (PPID=0) or (PPID=pe32.th32ParentProcessID) then
     if (Exe='') or SameText(Exe,pe32.szExeFile) then
     List.Add(Format('%d, %d, %d, %s',[pe32.th32ProcessID,pe32.th32ParentProcessID,pe32.pcPriClassBase,pe32.szExeFile]));
   until not Process32Next(hProcessSnap,pe32);
  finally
   CloseHandle(hProcessSnap);
  end;
  if (List.Count>0) then Result:=List.Text;
 finally
  List.Free;
 end;
 {$ENDIF}
end;

// List exe/dll modules uses by PID
function TWinApi.GetListOfModules(PID:Cardinal):String;
{$IFDEF WINDOWS} var hModuleSnap:THandle; me32:MODULEENTRY32; List:TStringList; {$ENDIF}
begin
 Result:='';
 {$IFDEF WINDOWS}
 List:=TStringList.Create;
 try
  hModuleSnap:=CreateToolhelp32Snapshot(TH32CS_SNAPMODULE,PID);
  if(hModuleSnap<>0) and (hModuleSnap<>INVALID_HANDLE_VALUE) then
  try
   ZeroMemory(@me32,sizeof(me32));
   me32.dwSize:=sizeof(me32);
   if Module32First(hModuleSnap,me32) then
   repeat
    if (PID=0) or (PID=me32.th32ProcessID) then
    List.Add(me32.szExePath);
   until not Module32Next(hModuleSnap,me32);
  finally
   CloseHandle(hModuleSnap);
  end;
  if (List.Count>0) then Result:=List.Text;
 finally
  List.Free;
 end;
 {$ENDIF}
end;

{$IFDEF WINDOWS}
type TEnumWinRec = record List:TStringList; lpClass,lpTitle:String; Pid:Cardinal; end;
function EnumWinProc(Wnd:HWND; Param:LPARAM):BOOL; stdcall;
var winPid:DWORD; aClass,aTitle:String;
const BuffSize=1024*32;
begin
 Result:=true;
 if IsWindow(Wnd) and (Param<>0) then with TEnumWinRec(Pointer(Param)^) do begin
  winPid:=0; GetWindowThreadProcessId(Wnd,@winPid);
  SetLength(aClass,BuffSize); SetLength(aClass,GetClassName(Wnd,PChar(aClass),Length(aClass)));
  SetLength(aTitle,BuffSize); SetLength(aTitle,GetWindowText(Wnd,PChar(aTitle),Length(aTitle)));
  if (Pid=0) or (winPid=Pid) then
  if (lpClass='') or SameText(lpClass,aClass) then
  if (lpTitle='') or SameText(lpTitle,aTitle) then
  List.Add(Format('$%X, %d, %s, %s',[Wnd,winPid,aClass,aTitle]));
  aClass:=''; aTitle:='';
 end;
end;
{$ENDIF}

// List windows as $HWND, PID, Class, Title
function TWinApi.GetListOfWindows(Pid:Cardinal; lpClass,lpTitle:String):String;
{$IFDEF WINDOWS} var List:TStringList; R:TEnumWinRec; {$ENDIF}
begin
 Result:='';
 {$IFDEF WINDOWS}
 ZeroMemory(@R,SizeOf(R));
 List:=TStringList.Create;
 try
  R.List:=List;
  R.lpClass:=lpClass;
  R.lpTitle:=lpTitle;
  R.Pid:=Pid;
  EnumWindows(@EnumWinProc,LPARAM(@R));
  if (List.Count>0) then Result:=List.Text;
 finally
  List.Free;
  R.lpClass:='';
  R.lpTitle:='';
 end;
 {$ENDIF}
end;

function TWinApi.GetProcessId(hProcess:THandle):DWORD;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=JwaWinBase.GetProcessId(hProcess);
 {$ENDIF}
end;

function TWinApi.GetCurrentProcess:THandle;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetCurrentProcess;
 {$ENDIF}
end;

function TWinApi.GetCurrentProcessId:DWORD;
begin
 {$IFDEF WINDOWS}
 Result:=Windows.GetCurrentProcessId;
 {$ELSE}
 Result:=System.GetProcessId;
 {$ENDIF}
end;

function TWinApi.CloseHandle(hObject:THandle):Boolean;
begin
 Result:=false;
 {$IFDEF WINDOWS}
 Result:=Windows.CloseHandle(hObject);
 {$ENDIF}
end;

function TWinApi.OpenProcess(dwDesiredAccess:DWORD; bInheritHandle:Boolean;  dwProcessId:DWORD):THandle;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.OpenProcess(dwDesiredAccess,bInheritHandle,dwProcessId);
 {$ENDIF}
end;

function TWinApi.GetExitCodeProcess(hProcess:THandle; var lpExitCode:DWord):Boolean;
begin
 Result:=false;
 {$IFDEF WINDOWS}
 Result:=Windows.GetExitCodeProcess(hProcess,lpExitCode);
 {$ENDIF}
end;

function TWinApi.GetProcessImageFileName(hProcess:THandle):String;
begin
 Result:='';
 {$IFDEF WINDOWS}
 SetLength(Result,MAX_PATH);
 SetLength(Result,JwaPsApi.GetProcessImageFileName(hProcess,PChar(Result),Length(Result)));
 {$ENDIF}
end;

function TWinApi.TerminateProcess(hProcess:THandle; uExitCode:UInt):Boolean;
begin
 Result:=false;
 {$IFDEF WINDOWS}
 Result:=Windows.TerminateProcess(hProcess,uExitCode);
 {$ENDIF}
end;

function TWinApi.GetPriorityClass(hProcess:THandle):DWORD;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetPriorityClass(hProcess);
 {$ENDIF}
end;

function TWinApi.SetPriorityClass(hProcess:THandle; dwPriorityClass:DWORD):Boolean;
begin
 Result:=false;
 {$IFDEF WINDOWS}
 Result:=Windows.SetPriorityClass(hProcess,dwPriorityClass);
 {$ENDIF}
end;

function TWinApi.GetProcessTimesAsText(hProcess:THandle):String;
{$IFDEF WINDOWS}var lpCreationTime,lpExitTime,lpKernelTime,lpUserTime:JwaWinBase.TFileTime;{$ENDIF}
begin
 Result:='';
 {$IFDEF WINDOWS}
 if JwaWinBase.GetProcessTimes(hProcess,lpCreationTime,lpExitTime,lpKernelTime,lpUserTime)
 then Result:=Format('%d, %d, %d, %d',[QWord(lpCreationTime),QWord(lpExitTime),QWord(lpKernelTime),QWord(lpUserTime)]);
 {$ENDIF}
end;

function TWinApi.GetSystemTimesAsText:String;
{$IFDEF WINDOWS}var lpIdleTime,lpKernelTime,lpUserTime:JwaWinBase.TFileTime;{$ENDIF}
begin
 Result:='';
 {$IFDEF WINDOWS}
 if JwaWinBase.GetSystemTimes(@lpIdleTime,@lpKernelTime,@lpUserTime)
 then Result:=Format('%d, %d, %d',[QWord(lpIdleTime),QWord(lpKernelTime),QWord(lpUserTime)]);
 {$ENDIF}
end;

function TWinApi.GetProcessAffinityMask(hProcess:THandle):QWord;
{$IFDEF WINDOWS}var lpProcessAffinityMask,lpSystemAffinityMask:DWORD_PTR;{$ENDIF}
begin
 Result:=0;
 {$IFDEF WINDOWS}
 if JwaWinBase.GetProcessAffinityMask(hProcess,lpProcessAffinityMask,lpSystemAffinityMask)
 then Result:=lpProcessAffinityMask;
 {$ENDIF}
end;

function TWinApi.GetSystemAffinityMask:QWord;
{$IFDEF WINDOWS}var lpProcessAffinityMask,lpSystemAffinityMask:DWORD_PTR;{$ENDIF}
begin
 Result:=0;
 {$IFDEF WINDOWS}
 if JwaWinBase.GetProcessAffinityMask(Windows.GetCurrentProcess,lpProcessAffinityMask,lpSystemAffinityMask)
 then Result:=lpSystemAffinityMask;
 {$ENDIF}
end;

function TWinApi.SetProcessAffinityMask(hProcess:THandle; Affinity:QWord):Boolean;
{$IFDEF WINDOWS}var lpProcessAffinityMask:DWORD_PTR;{$ENDIF}
begin
 Result:=false;
 {$IFDEF WINDOWS}
 lpProcessAffinityMask:=Affinity;
 Result:=JwaWinBase.SetProcessAffinityMask(hProcess,lpProcessAffinityMask);
 {$ENDIF}
end;

function TWinApi.WinExec(lpCmdLine:String; uCmdShow:UINT):UINT;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.WinExec(PCharArg(lpCmdLine),uCmdShow);
 {$ENDIF}
end;

function TWinApi.GetWindowLong(Wnd:HWND; nIndex:Integer):LongInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetWindowLong(Wnd,nIndex);
 {$ENDIF}
end;

function TWinApi.SetWindowLong(Wnd:HWND; nIndex:Integer; dwNewLong:LongInt):LongInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.SetWindowLong(Wnd,nIndex,dwNewLong);
 {$ENDIF}
end;

function TWinApi.GetWindowLongPtr(Wnd:HWND; nIndex:Integer):PtrInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetWindowLongPtr(Wnd,nIndex);
 {$ENDIF}
end;

function TWinApi.SetWindowLongPtr(Wnd:HWND; nIndex:Integer; dwNewLong:PtrInt):PtrInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.SetWindowLongPtr(Wnd,nIndex,dwNewLong);
 {$ENDIF}
end;

 // see https://docs.microsoft.com/en-us/windows/win32/api/shellapi/ns-shellapi-shellexecuteinfow
function TWinApi.ShellExecuteEx(fMask:DWORD; wnd:HWND; lpVerb:WideString; lpFile:WideString; lpParameters:WideString; lpDirectory:WideString; nShow:Integer; lpClass:WideString; hkeyClass:THandle; dwHotKey:DWORD; hIcon:THandle; var hInstApp:THandle; var hProcess:THandle):Boolean;
{$IFDEF WINDOWS}var se:SHELLEXECUTEINFOW;{$ENDIF}
begin
 Result:=false;
 {$IFDEF WINDOWS}
 ZeroMemory(@se,sizeof(se)); se.cbSize:=SizeOf(se);
 se.fMask:=fMask; se.wnd:=Wnd; se.lpVerb:=PWideCharArg(lpVerb); se.lpFile:=PWideCharArg(lpFile);
 se.lpParameters:=PWideCharArg(lpParameters); se.lpDirectory:=PWideCharArg(lpDirectory);
 se.nShow:=nShow; se.lpClass:=PWideCharArg(lpClass); se.hkeyClass:=hkeyClass;
 se.dwHotKey:=dwHotKey; se.DUMMYUNIONNAME.hIcon:=hIcon;
 Result:=ShellExecuteExW(@se);
 hInstApp:=se.hInstApp;
 hProcess:=se.hProcess;
 {$ENDIF}
end;

 // see https://docs.microsoft.com/en-us/windows/win32/api/synchapi/nf-synchapi-waitforsingleobject
function TWinApi.WaitForSingleObject(hHandle:THandle; dwMilliseconds:DWORD):DWORD;
begin
 Result:=High(DWORD);
 {$IFDEF WINDOWS}
 Result:=Windows.WaitForSingleObject(hHandle,dwMilliseconds);
 {$ENDIF}
end;

function TWinApi.GetACP:UInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetACP;
 {$ENDIF}
end;

function TWinApi.GetOEMCP:UInt;
begin
 Result:=0;
 {$IFDEF WINDOWS}
 Result:=Windows.GetOEMCP;
 {$ENDIF}
end;

initialization

 WinApi;

finalization

 WinApi.Free;

end.

