{------------------------------------------------------------------------------}
{                                                                              }
{                               Alexey Kuryakin                                }
{                                    LGPL                                      }
{                                                                              }
{------------------------------------------------------------------------------}

unit expandenvvar; // Expand environment variables in Windows or Unix style

{$IFDEF FPC}{$mode objfpc}{$ENDIF}{$H+}
{$IFDEF FPC}{$inline on}{$ENDIF}

interface

uses
  Classes,SysUtils,Dialogs{$IFDEF WINDOWS},Windows{$ENDIF};

const                   // Expand modes for ExpandEnvironmentVariables
  emExpandLikeWinC = 1; // Windows Classic  - %Variable%
  emExpandLikeWinE = 2; // Windows Expanded - !Variable!
  emExpandLikeUnix = 4; // Unix classic     - $Variable
  emExpandLikeBash = 8; // Bash style       - ${Variable}
  emExpandDefaults = emExpandLikeWinC + emExpandLikeWinE + emExpandLikeUnix + emExpandLikeBash;

// Expand environment variables from Expression.
// Supports Unix-like $Name (for example, $HOME\Demo).
// Supports Unix-like ${Name} (for example, ${HOME}\Demo).
// Supports Windows-like !Name! (for example, !ProgramFiles!\Demo).
// Supports Windows-like %Name% (for example, %ProgramFiles%\Demo).
// Note: arguments (%0,%1..%9 or $0,$1..$9) is NOT suported, it's not real environment variables.
function ExpandEnvironmentVariables(const Expression:String; Mode:Integer=emExpandDefaults):String;

// Get/Set environment variables.
function GetEnvironmentVariable(Name:String):String;
function SetEnvironmentVariable(Name,Value:String):Boolean;

// Uses to set PChar arguments from String.
function PCharArg(const S:String):PChar; {$IFDEF FPC}inline;{$ENDIF}
function PWideCharArg(const S:WideString):PWideChar; {$IFDEF FPC}inline;{$ENDIF}

// Just for testing purposes.
function TestExpandEnvironmentVariables:String;

implementation

function PCharArg(const S:String):PChar; {$IFDEF FPC}inline;{$ENDIF}
begin
  if (S='') then Result:=nil else Result:=PChar(S);
end;

function PWideCharArg(const S:WideString):PWideChar; {$IFDEF FPC}inline;{$ENDIF}
begin
  if (S='') then Result:=nil else Result:=PWideChar(S);
end;

{$IFDEF WINDOWS}
function GetEnvironmentVariable(Name:String):String;
begin
  Result:='';
  if (Name='') then Exit;
  SetLength(Result,1024*64);
  SetLength(Result,Windows.GetEnvironmentVariable(PChar(Name),PChar(Result),Length(Result)));
end;
function SetEnvironmentVariable(Name,Value:String):Boolean;
begin
 Result:=Windows.SetEnvironmentVariable(PCharArg(Name),PCharArg(Value));
end;
{$ELSE}
function getenv(__name:Pchar):Pchar;cdecl;external 'c' name 'getenv';
function putenv(__string:Pchar):longint;cdecl;external 'c' name 'putenv';
function setenv(__name:Pchar; __value:Pchar; __replace:longint):longint;cdecl;external 'c' name 'setenv';
function unsetenv(__name:Pchar):longint;cdecl;external 'c' name 'unsetenv';
function GetEnvironmentVariable(Name:String):String;
begin
 if (Name='') then Result:='' else Result:=getenv(PCharArg(Name));
end;
function SetEnvironmentVariable(Name,Value:String):Boolean;
begin
 Result:=false;
 if (Name='') then Exit;
 if (Value='') then begin Result:=(unsetenv(PCharArg(Name))<>-1); Exit; end;
 Result:=(setenv(PCharArg(Name),PCharArg(Value),1)<>-1);
end;
{$ENDIF}

// Internal utility routines
////////////////////////////
procedure BookMark(i:Integer; var iStart,Count:Integer); {$IFDEF FPC}inline;{$ENDIF}
begin
  iStart:=i; Count:=0;
end;

procedure Accumulate(iStart:Integer; var Count:Integer); {$IFDEF FPC}inline;{$ENDIF}
begin
  if (iStart>0) then inc(Count);
end;

function TrySubstitution(var S:String; var iStart,Count:Integer; LeftOff,RightOff:Integer):Boolean;
var sName,sValue:String;
begin
  Result:=false;
  if (iStart>0) then begin
    if (Count>0) then begin
      sName:=Copy(S,iStart+LeftOff,Count);
      sValue:=GetEnvironmentVariable(sName);
      if (sValue='') then Exit; // Environment variable is not defined, skip it.
      S:=Copy(S,1,iStart-1)+sValue+Copy(S,iStart+Count+LeftOff+RightOff,MaxInt);
      Result:=true;
    end;
    BookMark(0,iStart,Count);
  end;
end;

// Environment substitution in Windows Cmd Classic style.
// Replace %Name% expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeWinC(var S:String; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeWinC)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '%':  if (iStart=0) then BookMark(i,iStart,Count) else
          if TrySubstitution(S,iStart,Count,1,1) then begin Result:=true; Exit; end;
    else  Accumulate(iStart,Count);
  end;
end;

// Environment substitution in Windows Cmd Expanded style.
// Replace !Name! expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeWinE(var S:String; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeWinE)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '!':  if (iStart=0) then BookMark(i,iStart,Count) else
          if TrySubstitution(S,iStart,Count,1,1) then begin Result:=true; Exit; end;
    else  Accumulate(iStart,Count);
  end;
end;

// Environment substitution in Unix Shell style.
// Replace $Name expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeUnix(var S:String; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeUnix)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '$':  if (iStart=0) then BookMark(i,iStart,Count);
    'A'..'Z',
    'a'..'z',
    '0'..'9',
    '_':  Accumulate(iStart,Count);
    else  if TrySubstitution(S,iStart,Count,1,0) then begin Result:=true; Exit; end;
  end;
  if TrySubstitution(S,iStart,Count,1,0) then begin Result:=true; Exit; end;
end;

// Environment substitution in Unix Bash style.
// Replace ${Name} expression to GetEnv(Name), return true if one found.
// Do not replace anything if environment variable Name is not defined.
function ExpandLikeBash(var S:String; Mode:Integer):Boolean;
var i,iStart,Count:Integer;
begin
  Result:=false;
  if ((Mode and emExpandLikeBash)=0) then Exit;
  iStart:=0; Count:=0;
  for i:=1 to Length(S) do
  case S[i] of
    '$':  if (iStart=0) then BookMark(i,iStart,Count);
    '{':  if (iStart>0) then if (i<>iStart+1) then BookMark(0,iStart,Count);
    '}':  if TrySubstitution(S,iStart,Count,2,1) then begin Result:=true; Exit; end;
    else  Accumulate(iStart,Count);
  end;
end;

function ExpandEnvironmentVariables(const Expression:String; Mode:Integer=emExpandDefaults):String;
var i:Integer; const MaxIterations=10;
begin
  Result:=Expression;
  if (Result='') then Exit; if (Mode=0) then Exit;
  i:=0; while (ExpandLikeWinE(Result,Mode) or ExpandLikeWinC(Result,Mode)) and (i<MaxIterations) do inc(i);
  i:=0; while (ExpandLikeBash(Result,Mode) or ExpandLikeUnix(Result,Mode)) and (i<MaxIterations) do inc(i);
end;

function TestExpandEnvironmentVariables:String;
begin
 Result:='Test ExpandEnvironmentVariables:'+LineEnding
        +ExpandEnvironmentVariables('$HOME')+LineEnding
        +ExpandEnvironmentVariables('%HOME%')+LineEnding
        +ExpandEnvironmentVariables('!HOME!')+LineEnding
        +ExpandEnvironmentVariables('${HOME}')+LineEnding
        +ExpandEnvironmentVariables('$HOME\Demo')+LineEnding
        +ExpandEnvironmentVariables('!HOME!\Demo')+LineEnding
        +ExpandEnvironmentVariables('%HOME%\Demo')+LineEnding
        +ExpandEnvironmentVariables('${HOME}\Demo')+LineEnding
        +ExpandEnvironmentVariables('$UserProfile')+LineEnding
        +ExpandEnvironmentVariables('!UserProfile!')+LineEnding
        +ExpandEnvironmentVariables('%UserProfile%')+LineEnding
        +ExpandEnvironmentVariables('${UserProfile}')+LineEnding
        +ExpandEnvironmentVariables('$UserProfile\Demo')+LineEnding
        +ExpandEnvironmentVariables('!UserProfile!\Demo')+LineEnding
        +ExpandEnvironmentVariables('%UserProfile%\Demo')+LineEnding
        +ExpandEnvironmentVariables('${UserProfile}\Demo')+LineEnding;
end;

initialization

  // Just for testing:
  // ShowMessage(TestExpandEnvironmentVariables);

end.

