unit XButtons;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Buttons, Controls;

type
  TCustomXSpeedButtton = class(TGraphicControl)
  private
    FGroupIndex: integer;
    FLastDrawDetails: TThemedElementDetails;
    FLayout: TButtonLayout;
    FMargin: integer;
    FSpacing: integer;
    FShortcut: TShortCut;
    FShowAccelChar: boolean;
    FShowCaption: boolean;
    FAllowAllUp: boolean;
    FDown: boolean;
    FDownLoaded: boolean;// value of Down set during loading
    FDragging: boolean;
    FFlat: boolean;
    FMouseInControl: boolean;
    FImageList: TImageList;
    FImageIndex: integer;
    FDownImageIndex: integer;
    procedure SetShowCaption(const AValue: boolean);
    procedure UpdateExclusive;
    procedure SetAllowAllUp(Value: boolean);
    procedure SetLayout(const Value: TButtonLayout);
    procedure SetShowAccelChar(Value: boolean);
    procedure CMButtonPressed(var Message: TLMessage); message CM_BUTTONPRESSED;
    procedure CMEnabledChanged(var Message: TLMessage); message CM_ENABLEDCHANGED;
  private
    procedure DoMouseUp(var Message: TLMMouse; Button: TMouseButton);
    function GetDownImageIndex: integer;
    function GetImageIndex: integer; virtual;
    function GetImageList: TImageList;
    procedure SetDownImageIndex(AValue: integer);
    procedure SetImageIndex(AValue: integer); virtual;
    procedure SetImageList(AValue: TImageList);
    procedure WMLButtonDown(var Message: TLMLButtonDown); message LM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TLMLButtonUp); message LM_LBUTTONUP;
    procedure WMLButtonDBLCLK(var Message: TLMLButtonDblClk); message LM_LBUTTONDBLCLK;
  protected
    FState: TButtonState;
    class procedure WSRegisterClass; override;
    function DialogChar(var Message: TLMKey): boolean; override;
    procedure CalculatePreferredSize(var PreferredWidth, PreferredHeight: integer;
      WithThemeSpace: boolean); override;
    procedure MeasureDraw(Draw: boolean; PaintRect: TRect;
      out PreferredWidth, PreferredHeight: integer);
    procedure MouseEnter; override;
    procedure MouseLeave; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: integer); override;
    procedure Paint; override;
    procedure PaintBackground(var PaintRect: TRect); virtual;
    procedure SetDown(Value: boolean);
    procedure SetGroupIndex(const Value: integer);
    procedure SetFlat(const Value: boolean);
    procedure SetMargin(const Value: integer);
    procedure SetSpacing(const Value: integer);
    procedure RealSetText(const Value: TCaption); override;
    procedure UpdateState(InvalidateOnChange: boolean); virtual;
    function GetDrawDetails: TThemedElementDetails; virtual;
    property MouseInControl: boolean read FMouseInControl;
    procedure ActionChange(Sender: TObject; CheckDefaults: boolean); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    class function GetControlClassDefaultSize: TSize; override;
    procedure Loaded; override;
  protected
    function GetTextSize(Drawing: boolean; PaintRect: TRect): TSize; virtual;
    function DrawGlyph(ACanvas: TCanvas; const AClient: TRect;
      const AOffset: TPoint; AState: TButtonState; BiDiFlags: longint): TRect; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function FindDownButton: TEmbededButton;
    procedure Click; override; // make Click public
  public
    property AllowAllUp: boolean read FAllowAllUp write SetAllowAllUp default False;
    property Color default clBtnFace;
    property Down: boolean read FDown write SetDown default False;
    property Flat: boolean read FFlat write SetFlat default False;
    property GroupIndex: integer read FGroupIndex write SetGroupIndex default 0;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property Margin: integer read FMargin write SetMargin default -1;
    property ShowAccelChar: boolean
      read FShowAccelChar write SetShowAccelChar default True;
    property ShowCaption: boolean read FShowCaption write SetShowCaption default True;
    property Spacing: integer read FSpacing write SetSpacing default 4;
    property ImageList: TImageList read GetImageList write SetImageList;
    property ImageIndex: integer read GetImageIndex write SetImageIndex;
    property DownImageIndex: integer read GetDownImageIndex write SetDownImageIndex;
  end;

implementation

uses Dialogs;

constructor TCustomXSpeedButtton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  with GetControlClassDefaultSize do
    SetInitialBounds(0, 0, CX, CY);
  ControlStyle := ControlStyle + [csCaptureMouse] -
    [csSetCaption, csClickEvents, csOpaque];
  FLayout := blGlyphLeft;
  FAllowAllUp := False;
  FMouseInControl := False;
  FDragging := False;
  FShowAccelChar := True;
  FSpacing := 4;
  FMargin := -1;
  Color := clBtnFace;
  FShowCaption := True;
  FImageIndex := -1;
  FDownImageIndex := -1;
end;

destructor TCustomXSpeedButtton.Destroy;
begin
  FImageList := nil;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.FindDownButton: TCustomXSpeedButtton;

  Searches the speed button with Down=true and the same GroupIndex.
 ------------------------------------------------------------------------------}
function TCustomXSpeedButtton.FindDownButton: TCustomXSpeedButtton;

  function FindDown(AWinControl: TWinControl): TCustomXSpeedButtton;
  var
    i: integer;
    Child: TControl;
    Button: TCustomXSpeedButtton;
  begin
    if AWinControl = nil then
      Exit(nil);
    for i := 0 to AWinControl.ControlCount - 1 do
    begin
      Child := AWinControl.Controls[i];
      if Child is TCustomXSpeedButtton then
      begin
        Button := TCustomXSpeedButtton(Child);
        if (Button.GroupIndex = GroupIndex) and (Button.Down) then
          Exit(Button);
      end;
      if Child is TWinControl then
      begin
        Result := FindDown(TWinControl(Child));
        if Result <> nil then
          Exit;
      end;
    end;
    Result := nil;
  end;

begin
  if Down or (GroupIndex = 0) then
    exit(Self);
  Result := FindDown(GetFirstParentForm(Self));
end;

procedure TCustomXSpeedButtton.Click;
begin
  inherited Click;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetAllowAllUp
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetAllowAllUp(Value: boolean);
begin
  if FAllowAllUp <> Value then
  begin
    FAllowAllUp := Value;
    UpdateExclusive;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetDown
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetDown(Value: boolean);
var
  OldState: TButtonState;
  OldDown: boolean;
begin
  //since Down needs GroupIndex, then we need to wait that all properties
  //loaded before we continue
  if (csLoading in ComponentState) then
  begin
    FDownLoaded := Value;
    exit;
  end
  else
  begin
    if FGroupIndex = 0 then
      Value := False;
    if FDown <> Value then
    begin
      if FDown and not FAllowAllUp then
        Exit;
      OldDown := FDown;
      FDown := Value;
      OldState := FState;
      if FDown then
        FState := bsExclusive
      else
        FState := UpState[FMouseInControl];
      if (OldDown <> FDown) or (OldState <> FState) then
        Invalidate;
      if Value then
        UpdateExclusive;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetFlat
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetFlat(const Value: boolean);
begin
  if FFlat <> Value then
  begin
    FFlat := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetGroupIndex
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetGroupIndex(const Value: integer);
begin
  if FGroupIndex <> Value then
  begin
    FGroupIndex := Value;
    UpdateExclusive;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetMargin
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetMargin(const Value: integer);
begin
  if FMargin <> Value then
  begin
    FMargin := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetSpacing
  Params: Value:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetSpacing(const Value: integer);
begin
  if FSpacing <> Value then
  begin
    FSpacing := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  procedure TCustomXSpeedButtton.RealSetText(const Value: TCaption);
 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.RealSetText(const Value: TCaption);
begin
  if Caption = Value then
    Exit;
  inherited RealSetText(Value);

  Invalidate;
end;

{------------------------------------------------------------------------------
  procedure TCustomXSpeedButtton.UpdateState(InvalidateOnChange: boolean);
 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.UpdateState(InvalidateOnChange: boolean);
var
  OldState: TButtonState;
begin
  OldState := FState;
  if not IsEnabled then
  begin
    FState := bsDisabled;
    FDragging := False;
  end
  else
  begin
    if FState = bsDisabled then
    begin
      if FDown and (GroupIndex <> 0) then
        FState := bsExclusive
      else
        FState := UpState[FMouseInControl];
    end
    else
    if (FState in [bsHot, bsDown]) and (not FMouseInControl) and
      (not FDragging) and (not FDown) then
    begin
      // return to normal
      FState := bsUp;
    end
    else
    if (FState = bsUp) and FMouseInControl then
      FState := bsHot;
  end;
  if FState <> OldState then
    if (Action is TCustomAction) then
      TCustomAction(Action).Checked := FState = bsDown;
  //if InvalidateOnChange then DebugLn(['TxCustomSpeedButton.UpdateState ',DbgSName(Self),' InvalidateOnChange=',InvalidateOnChange,' StateChange=',FState<>OldState]);
  if InvalidateOnChange and ((FState <> OldState) or not
    ThemedElementDetailsEqual(FLastDrawDetails, GetDrawDetails)) then
    Invalidate;
end;

{------------------------------------------------------------------------------
  function TCustomXSpeedButtton.GetDrawDetails: TThemedElementDetails;
 ------------------------------------------------------------------------------}
function TCustomXSpeedButtton.GetDrawDetails: TThemedElementDetails;

  function ButtonPart: TThemedButton;
  begin
    // tbPushButtonNormal, tbPushButtonHot, tbPushButtonPressed,
    // tbPushButtonDisabled, tbPushButtonDefaulted

    // no check states available
    Result := tbPushButtonNormal;
    if not IsEnabled then
      Result := tbPushButtonDisabled
    else
    if FState in [bsDown, bsExclusive] then
      Result := tbPushButtonPressed
    else
    if FState = bsHot then
      Result := tbPushButtonHot
    else
      Result := tbPushButtonNormal;
  end;

  function ToolButtonPart: TThemedToolBar;
  begin
    // ttbButtonNormal, ttbButtonHot, ttbButtonPressed, ttbButtonDisabled
    // ttbButtonChecked, ttbButtonCheckedHot
    if not IsEnabled then
      Result := ttbButtonDisabled
    else
    begin
      if Down then
      begin // checked states
        if FMouseInControl then
          Result := ttbButtonCheckedHot
        else
          Result := ttbButtonChecked;
      end
      else
      begin
        if FState in [bsDown, bsExclusive] then
          Result := ttbButtonPressed
        else
        if FState = bsHot then
          Result := ttbButtonHot
        else
          Result := ttbButtonNormal;
      end;
    end;
  end;

begin
  if Flat then
    Result := ThemeServices.GetElementDetails(ToolButtonPart)
  else
    Result := ThemeServices.GetElementDetails(ButtonPart);
end;

procedure TCustomXSpeedButtton.ActionChange(Sender: TObject; CheckDefaults: boolean);
begin
  inherited ActionChange(Sender, CheckDefaults);
  if Sender is TCustomAction then
  begin
    with TCustomAction(Sender) do
    begin
      if CheckDefaults or (Self.GroupIndex = 0) then
        Self.GroupIndex := GroupIndex;
      {if (Glyph.Empty) and (ActionList <> nil) and (ActionList.Images <> nil) and
         (ImageIndex >= 0) and (ImageIndex < ActionList.Images.Count) then
        ActionList.Images.GetBitmap(ImageIndex, Glyph);}
    end;
  end;
end;

function TCustomXSpeedButtton.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TSpeedButtonActionLink;
end;

class function TCustomXSpeedButtton.GetControlClassDefaultSize: TSize;
begin
  Result.CX := 23;
  Result.CY := 22;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.UpdateExclusive
  Params: none
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.UpdateExclusive;
var
  msg: TLMessage;
begin
  if (FGroupIndex <> 0) and (Parent <> nil) and (not (csLoading in ComponentState)) then
  begin
    Msg.Msg := CM_ButtonPressed;
    Msg.WParam := FGroupIndex;
    Msg.LParam := PtrInt(Self);
    Msg.Result := 0;
    Parent.Broadcast(Msg);
  end;
end;

{------------------------------------------------------------------------------
  Function: TCustomXSpeedButtton.GetGlyph
  Params: none
  Returns:  The bitmap

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetShowCaption(const AValue: boolean);
begin
  if FShowCaption = AValue then
    exit;
  FShowCaption := AValue;
  invalidate;
end;

function TCustomXSpeedButtton.DialogChar(var Message: TLMKey): boolean;
begin
  Result := False;
  // Sometimes LM_CHAR is received instead of LM_SYSCHAR, maybe intentionally
  // (LCL handles it) or maybe sent by mistake. In either case exit.
  if (Message.Msg <> LM_SYSCHAR) or not FShowAccelChar then
    Exit;
  if IsAccel(Message.CharCode, Caption) then
  begin
    Result := True;
    if GroupIndex <> 0 then
      SetDown(not FDown);
    Click;
  end
  else
    Result := inherited DialogChar(Message);
end;

procedure TCustomXSpeedButtton.CalculatePreferredSize(
  var PreferredWidth, PreferredHeight: integer; WithThemeSpace: boolean);
var
  r: TRect;
begin
  r := Rect(0, 0, 0, 0);
  MeasureDraw(False, r, PreferredWidth, PreferredHeight);
end;

procedure TCustomXSpeedButtton.MeasureDraw(Draw: boolean; PaintRect: TRect;
  out PreferredWidth, PreferredHeight: integer);
var
  GlyphWidth, GlyphHeight: integer;
  Offset, OffsetCap: TPoint;
  ClientSize, TotalSize, TextSize, GlyphSize: TSize;
  M, S: integer;
  SIndex: longint;
  TMP: string;
  TextFlags: integer;
  DrawDetails: TThemedElementDetails;
  FixedWidth: boolean;
  FixedHeight: boolean;
  TextRect: TRect;
  HasGlyph: boolean;
  HasText: boolean;
  CurLayout: TButtonLayout;
begin
  DrawDetails := GetDrawDetails;

  PreferredWidth := 0;
  PreferredHeight := 0;
  HasGlyph := False;

  if Draw then
  begin
    FLastDrawDetails := DrawDetails;
    PaintBackground(PaintRect);
    FixedWidth := True;
    FixedHeight := True;
  end
  else
  begin
    FixedWidth := WidthIsAnchored;
    FixedHeight := HeightIsAnchored;
  end;
  ClientSize.cx := PaintRect.Right - PaintRect.Left;
  ClientSize.cy := PaintRect.Bottom - PaintRect.Top;

  GlyphSize.cx := 16;
  GlyphSize.cy := 16;

  if ImageList <> nil then
  begin
    GlyphSize.cx := ImageList.Width;
    GlyphSize.cy := ImageList.Height;
    if (ImageIndex >= 0) and (ImageIndex < ImageList.Count) then HasGlyph := True ;
  end;
  GlyphWidth := GlyphSize.CX;
  GlyphHeight := GlyphSize.CY;

  CurLayout := BidiAdjustButtonLayout(UseRightToLeftReading, Layout);
  if ShowCaption and (Caption <> '') then
  begin
    TextRect := PaintRect;
    // for wordbreak compute the maximum size for the text
    if Margin > 0 then
      InflateRect(TextRect, -Margin, -Margin);
    if HasGlyph then
    begin
      if (Spacing >= 0) then
        if CurLayout in [blGlyphLeft, blGlyphRight] then
          Dec(TextRect.Right, Spacing)
        else
          Dec(TextRect.Bottom, Spacing);
      if CurLayout in [blGlyphLeft, blGlyphRight] then
        Dec(TextRect.Right, GlyphWidth)
      else
        Dec(TextRect.Bottom, GlyphHeight);
    end;
    if not FixedWidth then
    begin
      TextRect.Left := 0;
      TextRect.Right := High(TextRect.Right) div 2;
    end;
    if not FixedHeight then
    begin
      TextRect.Top := 0;
      TextRect.Bottom := High(TextRect.Bottom) div 2;
    end;
    TextSize := GetTextSize(Draw, TextRect);
  end
  else
  begin
    TextSize.cx := 0;
    TextSize.cy := 0;
  end;
  HasText := (TextSize.cx <> 0) or (TextSize.cy <> 0);

  if Caption <> '' then
  begin
    TMP := Caption;
    SIndex := DeleteAmpersands(TMP);
    if SIndex > 0 then
      if SIndex <= Length(TMP) then
      begin
        FShortcut := Ord(TMP[SIndex]);
      end;
  end;

  if HasGlyph and HasText then
    S := Spacing
  else
    S := 0;

  M := Margin;
  if not Draw then
  begin
    if M < 0 then
      M := 2;
    if S < 0 then
      S := M;
  end;

  // Calculate caption and glyph layout
  if M = -1 then
  begin
    // auto compute margin to center content
    if S = -1 then
    begin
      // use the same value for Spacing and Margin
      TotalSize.cx := TextSize.cx + GlyphWidth;
      TotalSize.cy := TextSize.cy + GlyphHeight;
      if Layout in [blGlyphLeft, blGlyphRight] then
        M := (ClientSize.cx - TotalSize.cx) div 3
      else
        M := (ClientSize.cy - TotalSize.cy) div 3;
      S := M;
    end
    else
    begin
      // fixed Spacing and center content
      TotalSize.cx := GlyphWidth + S + TextSize.cx;
      TotalSize.cy := GlyphHeight + S + TextSize.cy;
      if Layout in [blGlyphLeft, blGlyphRight] then
        M := (ClientSize.cx - TotalSize.cx) div 2
      else
        M := (ClientSize.cy - TotalSize.cy) div 2;
    end;
  end
  else
  begin
    // fixed Margin
    if S = -1 then
    begin
      // use the rest for Spacing between Glyph and Caption
      TotalSize.cx := ClientSize.cx - (Margin + GlyphWidth);
      TotalSize.cy := ClientSize.cy - (Margin + GlyphHeight);
      if Layout in [blGlyphLeft, blGlyphRight] then
        S := (TotalSize.cx - TextSize.cx) div 2
      else
        S := (TotalSize.cy - TextSize.cy) div 2;
    end;
  end;

  //debugln(['TxCustomSpeedButton.MeasureDraw AAA3 ',DbgSName(Self),' PaintRect=',dbgs(PaintRect),' GlyphSize=',GlyphWidth,'x',GlyphHeight,' TextSize=',TextSize.cx,'x',TextSize.cy,' S=',S,' M=',M]);

  if Draw then
  begin
    case CurLayout of
      blGlyphLeft:
      begin
        Offset.X := M;
        Offset.Y := (ClientSize.cy - GlyphHeight) div 2;
        OffsetCap.X := Offset.X + GlyphWidth + S;
        OffsetCap.Y := (ClientSize.cy - TextSize.cy) div 2 - 1;
        if (FState = bsDown) or (Down = true) then
        begin
          OffsetCap.Y := OffsetCap.Y + 1;
          OffsetCap.X := OffsetCap.X + 1;
        end;
      end;
      blGlyphRight:
      begin
        Offset.X := ClientSize.cx - M - GlyphWidth;
        Offset.Y := (ClientSize.cy - GlyphHeight) div 2;
        OffsetCap.X := Offset.X - S - TextSize.cx;
        OffsetCap.Y := (ClientSize.cy - TextSize.cy) div 2 - 1;
        if (FState = bsDown) or (Down = true) then
        begin
          OffsetCap.Y := OffsetCap.Y + 1;
          OffsetCap.X := OffsetCap.X + 1;
        end;
      end;
      blGlyphTop:
      begin
        Offset.X := (ClientSize.cx - GlyphWidth) div 2;
        Offset.Y := M;
        OffsetCap.X := (ClientSize.cx - TextSize.cx) div 2;
        OffsetCap.Y := Offset.Y + GlyphHeight + S;
      end;
      blGlyphBottom:
      begin
        Offset.X := (ClientSize.cx - GlyphWidth) div 2;
        Offset.Y := ClientSize.cy - M - GlyphHeight;
        OffsetCap.X := (ClientSize.cx - TextSize.cx) div 2;
        OffsetCap.Y := Offset.Y - S - TextSize.cy;
      end;
    end;

    DrawGlyph(Canvas, PaintRect, Offset, FState, 0);

    if FShowCaption and (Caption <> '') then
    begin
      with PaintRect, OffsetCap do
      begin
        Left := Left + X;
        Top := Top + Y;

      end;

      TextFlags := DT_LEFT or DT_TOP;
      if UseRightToLeftReading then
        TextFlags := TextFlags or DT_RTLREADING;

      if Draw then
        ThemeServices.DrawText(Canvas, DrawDetails, Caption, PaintRect,
          TextFlags, 0);
    end;
  end
  else
  begin
    // measuring, not drawing
    case CurLayout of
      blGlyphLeft, blGlyphRight:
      begin
        PreferredWidth := 2 * M + S + GlyphWidth + TextSize.cx;
        PreferredHeight := 2 * M + Max(GlyphHeight, TextSize.cy);
      end;
      blGlyphTop, blGlyphBottom:
      begin
        PreferredWidth := 2 * M + Max(GlyphWidth, TextSize.cx);
        PreferredHeight := 2 * M + S + GlyphHeight + TextSize.cy;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.Paint
  Params: none
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.Paint;
var
  PaintRect: TRect;
  PreferredWidth: integer;
  PreferredHeight: integer;
begin
  UpdateState(False);
  //if FGlyph = nil then exit;

  PaintRect := ClientRect;
  MeasureDraw(True, PaintRect, PreferredWidth, PreferredHeight);

  inherited Paint;
end;

procedure TCustomXSpeedButtton.PaintBackground(var PaintRect: TRect);
begin
  if ThemeServices.HasTransparentParts(FLastDrawDetails) then
  begin
    Canvas.Brush.Color := Color;
    Canvas.FillRect(PaintRect);
  end;
  ThemeServices.DrawElement(Canvas.Handle, FLastDrawDetails, PaintRect);
  //PaintRect := ThemeServices.ContentRect(Canvas.Handle, FLastDrawDetails, PaintRect);
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.MouseDown
  Params: Button:
          Shift:
          X, Y:
  Returns:  nothing
 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if csDesigning in ComponentState then
    exit;

  if (Button = mbLeft) and IsEnabled then
  begin
    if not FDown then
    begin
      FState := bsDown;
      if (Action is TCustomAction) then
        TCustomAction(Action).Checked := False;
      Invalidate;
    end;
    FDragging := True;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.MouseMove
  Params: Shift:
          X, Y:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.MouseMove(Shift: TShiftState; X, Y: integer);
var
  NewState: TButtonState;
begin
  inherited MouseMove(Shift, X, Y);
  if csDesigning in ComponentState then
    exit;

  if FDragging then
  begin
    //DebugLn('Trace:FDragging is true');
    if FDown then
      NewState := bsExclusive
    else
    begin
      if (X >= 0) and (X < Width) and (Y >= 0) and (Y < Height) then
        NewState := bsDown
      else
        NewState := UpState[FMouseInControl];
    end;

    if NewState <> FState then
    begin
      //debugln(['TxCustomSpeedButton.MouseMove ',DbgSName(Self),' fState=',ord(fstate),' NewState=',ord(NewState)]);
      FState := NewState;
      Invalidate;
    end;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.MouseUp
  Params: Button:
          Shift:
          X, Y:
  Returns:  nothing
 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
end;

{------------------------------------------------------------------------------
       TCustomXSpeedButtton DoMouseUp  "Event Handler"
------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.DoMouseUp(var Message: TLMMouse; Button: TMouseButton);
begin
  if not (csNoStdEvents in ControlStyle) then
    with Message do
      MouseUp(Button, KeysToShiftState(Keys), XPos, YPos);
end;

function TCustomXSpeedButtton.GetDownImageIndex: integer;
begin
  Result := FDownImageIndex;
end;

function TCustomXSpeedButtton.GetImageIndex: integer;
begin
  Result := FImageIndex;
end;

function TCustomXSpeedButtton.GetImageList: TImageList;
begin
  Result := FImageList;
end;

procedure TCustomXSpeedButtton.SetDownImageIndex(AValue: integer);
begin
  if FDownImageIndex <> AValue then
    FDownImageIndex := AValue;
end;

procedure TCustomXSpeedButtton.SetImageIndex(AValue: integer);
begin
  if FImageIndex = AValue then
    Exit;
  FImageIndex := AValue;
  Invalidate;
end;

procedure TCustomXSpeedButtton.SetImageList(AValue: TImageList);
begin
  if FImageList <> AValue then
  begin
    FImageList := AValue;
    Invalidate;
  end;
end;

procedure TCustomXSpeedButtton.WMLButtonDown(var Message: TLMLButtonDown);
begin
  inherited;

  // because csClickEvents is not set no csClicked is set in the inherited method
  Include(FControlState, csClicked);
end;

procedure TCustomXSpeedButtton.WMLButtonDBLCLK(var Message: TLMLButtonDblClk);
begin
  inherited;
  // if in a group, raise dblclick event, otherwise translate to click event
  if Down then
    DblClick
  else
    Click;
end;

class procedure TCustomXSpeedButtton.WSRegisterClass;
begin
  inherited WSRegisterClass;
  //RegisterCustomXSpeedButton;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.WMLButtonUp
  Params: Message
  Returns: Nothing

  Mouse event handler
 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.WMLButtonUp(var Message: TLMLButtonUp);
var
  OldState: TButtonState;
  NeedClick: boolean;
begin
  //DebugLn('TxCustomSpeedButton.WMLButtonUp A ',DbgSName(Self),' csCaptureMouse=',DbgS(csCaptureMouse in ControlStyle),' csClicked=',DbgS(csClicked in ControlState));
  if (csCaptureMouse in ControlStyle) and (mbLeft in CaptureMouseButtons) then
  begin
    {$IFDEF VerboseMouseCapture}
    DebugLn('TxCustomSpeedButton.WMLButtonUp ', Name, ':', ClassName);
    {$ENDIF}
    MouseCapture := False;
  end;

  NeedClick := False;

  if not (csDesigning in ComponentState) and FDragging then
  begin
    OldState := FState;
    FDragging := False;

    if FGroupIndex = 0 then
    begin
      FState := UpState[FMouseInControl];
      if OldState <> FState then
        Invalidate;
    end
    else
    if (Message.XPos >= 0) and (Message.XPos < Width) and (Message.YPos >= 0) and
      (Message.YPos < Height) then
    begin
      SetDown(not FDown);
      NeedClick := True;
    end;
  end;

  DoMouseUp(Message, mbLeft);

  if csClicked in ControlState then
  begin
    Exclude(FControlState, csClicked);
    //DebugLn('TxCustomSpeedButton.WMLButtonUp B ',dbgs(ClientRect.Left),',',dbgs(ClientRect.Top),',',dbgs(ClientRect.Right),',',dbgs(ClientRect.Bottom),' ',dbgs(Message.Pos.X),',',dbgs(Message.Pos.Y));
    if PtInRect(ClientRect, SmallPointToPoint(Message.Pos)) then
    begin
      //DebugLn('TxCustomSpeedButton.WMLButtonUp C');
      // Important: Calling Click can invoke modal dialogs, so call this as last
      NeedClick := False;
      Click;
    end;
  end;

  if NeedClick then
    Click;
  //DebugLn('TxCustomSpeedButton.WMLButtonUp END');
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.SetLayout
  Params: Value: new layout value
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.SetLayout(const Value: TButtonLayout);
begin
  if Value <> FLayout then
  begin
    FLayout := Value;
    Invalidate;
  end;
end;

procedure TCustomXSpeedButtton.SetShowAccelChar(Value: boolean);
begin
  if FShowAccelChar <> Value then
  begin
    FShowAccelChar := Value;
    Invalidate;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.CMButtonPressed
  Params: Message:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.CMButtonPressed(var Message: TLMessage);
var
  Sender: TCustomXSpeedButtton;
begin
  if csDestroying in ComponentState then
    exit;
  if Message.WParam = WParam(FGroupIndex) then
  begin
    Sender := TCustomXSpeedButtton(Message.LParam);
    if Sender <> Self then
    begin
      if Sender.Down and FDown then
      begin
        FDown := False;
        FState := UpState[FMouseInControl];
        Invalidate;
      end;
      FAllowAllUp := Sender.AllowAllUp;
    end;
  end;
end;

procedure TCustomXSpeedButtton.Loaded;
begin
  inherited Loaded;
  UpdateExclusive;
  if FDownLoaded then
    SetDown(FDownLoaded);
end;

function TCustomXSpeedButtton.GetTextSize(Drawing: boolean; PaintRect: TRect): TSize;
var
  TMP: string;
  TXTStyle: TTextStyle;
  Flags: cardinal;
begin
  if FShowCaption and (Caption <> '') then
  begin
    TMP := Caption;
    TXTStyle := Canvas.TextStyle;
    TXTStyle.Opaque := False;
    TXTStyle.Clipping := True;
    TXTStyle.ShowPrefix := ShowAccelChar;
    TXTStyle.Alignment := taLeftJustify;
    TXTStyle.Layout := tlTop;
    TXTStyle.RightToLeft := UseRightToLeftReading;
    TXTStyle.SystemFont := Canvas.Font.IsDefault;//Match System Default Style
    DeleteAmpersands(TMP);

    Flags := DT_CalcRect;
    if not TXTStyle.SingleLine then
      Inc(Flags, DT_WordBreak);

    DrawText(Canvas.Handle, PChar(TMP), Length(TMP), PaintRect, Flags);
    Result.CY := PaintRect.Bottom - PaintRect.Top;
    Result.CX := PaintRect.Right - PaintRect.Left;
  end
  else
  begin
    Result.CY := 0;
    Result.CX := 0;
  end;
end;

function TCustomXSpeedButtton.DrawGlyph(ACanvas: TCanvas; const AClient: TRect;
  const AOffset: TPoint; AState: TButtonState; BiDiFlags: longint): TRect;
var
  ImgIndex: integer;
begin
  Result := AClient;
  if ImageList = nil then
    Exit;

  if FState = bsDown then
  begin
    if (DownImageIndex >= 0) and (DownImageIndex < ImageList.Count) then
      ImgIndex := DownImageIndex
    else
      ImgIndex := ImageIndex;
  end
  else
    ImgIndex := ImageIndex;

  if (ImgIndex >= 0) and (ImgIndex < ImageList.Count) then
  begin
    if (AState = bsDown) or (Down = true) then
      ImageList.Draw(ACanvas, AOffset.x + 1, AOffset.y + 1, ImgIndex, Enabled)
    else
     ImageList.Draw(ACanvas, AOffset.x, AOffset.y, ImgIndex, Enabled);
  end;
  {if Assigned(FGlyph) then
  begin
    if (AState = bsDown) or (Down = true) then
      Result := FGlyph.Draw(ACanvas, AClient, point(AOffset.x + 1, AOffset.y + 1), AState, ATransparent, BiDiFlags)
    else
      Result := FGlyph.Draw(ACanvas, AClient, AOffset, AState, ATransparent, BiDiFlags);
  end;}
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.CMEnabledChanged
  Params: Message:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.CMEnabledChanged(var Message: TLMEssage);
begin
  //Should create a new glyph based on the new state
  UpdateState(True);
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.MouseEnter
  Params:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.MouseEnter;
begin
  if csDesigning in ComponentState then
    exit;
  if not FMouseInControl and IsEnabled and (GetCapture = 0) then
  begin
    FMouseInControl := True;
    UpdateState(True);
    inherited MouseEnter;
  end;
end;

{------------------------------------------------------------------------------
  Method: TCustomXSpeedButtton.MouseLeave
  Params:
  Returns:  nothing

 ------------------------------------------------------------------------------}
procedure TCustomXSpeedButtton.MouseLeave;
begin
  if csDesigning in ComponentState then
    exit;
  ///DebugLn(['TxCustomSpeedButton.MouseLeave ',DbgSName(Self),' FMouseInControl=',FMouseInControl,' FDragging=',FDragging]);
  if FMouseInControl then
  begin
    FMouseInControl := False;
    if IsEnabled then
    begin
      if FDragging and (not MouseCapture) then
      begin
        // something fetched our mouse capture
        FDragging := False;
      end;
      UpdateState(True);
      inherited MouseLeave;
    end;
  end;
end;


end.
